// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 23432 $
//  $Date: 2008-06-24 16:25:52 +0300 (Tue, 24 Jun 2008) $
//  $Author: yab $

// Rosetta Headers
#include "initialize.h"
#include "after_opts.h"
#include "abrelax.h"
#include "antibody_modeling.h"
#include "barcode_stats.h"
#include "constraints.h"
#include "counters.h"
#include "crankshaft.h"
#include "design.h"
#include "decoy_features.h"
#include "decoystats.h"
#include "dipolar.h"
#include "disulfides.h"
#include "docking.h"
#include "dock_loop_ensemble.h"
#include "docking_ns.h"
#include "dummy_model.h"
#include "evolve.h"
#include "files_paths.h"
#include "filters.h"
#include "force_barcode.h"
#include "fragments.h"
#include "fullatom.h"
#include "hbonds.h"
#include "homolog_distances.h"
#include "input_pdb.h"
#include "jumping_minimize.h"
#include "jumping_pairings.h" // temporary hack
#include "knots.h"
#include "ligand.h"
#include "ligand_ns.h"
#include "loops.h"
#include "loop_relax.h"
#include "maps.h"
#include "maps_ns.h"
#include "map_sequence.h"
#include "minimize.h"
#include "misc.h"
#include "monte_carlo.h"
#include "native.h"
#include "options.h"
#include "orient_rms.h"
#include "output_decoy.h"
#include "pack_fwd.h"
#include "param.h"
#include "pdb.h"
#include "pH_main.h"
#include "pKa_mode.h"
#include "pose.h"
#include "pose_io.h"
#include "prof.h"
#include "DomainInsertionMode.h"
#include "read_aa_ss.h"
#include "read_aaproperties.h"
#include "recover.h"
#include "refold.h"
#include "relax_structure.h"
#include "saxs_model.h"
#include "score.h"
#include "start.h"
#include "status.h"
#include "structure.h"
#include "taboo_search.h"
#include "termini.h"
#include "trajectory.h"
#include "vdw.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/formatted.i.hh>
#include <ObjexxFCL/string.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/icstream.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>

// C++ Headers
#include <algorithm>
#include <cassert>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <string>
#include <vector>

#ifdef BOINC
#include "boinc_rosetta_util.h"
#include "boinc_rosetta_graphics.h"
#endif

// Namespaces
namespace fml_common {
	int filename_max_length = { 21 };
}



///////////////////////////////////////////////////////////////////////////////
/// @begin initialize_query
///
/// @brief carry out initializations specific to the query (ie independent of a
/// particular conformation); read and score native structure
///
/// @detailed
///
/// @global_read
///     io_options_bool   block files_paths.h
///     query_defined     files_paths.h
///     file_path_int     block files_paths.h
///     file_path_char    block files_paths.h
///     protein_name      misc.h
///     protein_chain     misc.h
///     score             misc.h
///
/// @global_write
///
/// @remarks
/// many indirect changes to global variables
///
/// many functions that self-initialize should re-initialize if the
/// query changes:  filters, output_compressed
/// There is currently no mechanism to indicate that the query has
/// changed, so only one query initialization should occur
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
void
initialize_query()
{
	using namespace files_paths;
	using namespace misc;
	using namespace param;

	if (create_domins_fasta) return;

	if ( multi_chain && !design::dna_interface && !antibody_modeler)
		get_size_from_docking_map();

	if (get_dummy_model_flag()) read_dummy_model();
	initialize_saxs_profile();

//------ read query info -----------------------------------
	bool seq_undefined = true;

	if ( map_start_sequence ) {
		// this is tricky: we want to prevent redimensioning MAX_RES
		// during the simulation b/c then fragment info gets obliterated
		// so try to figure out what the max possible length is:
		MAX_RES_assign_res( get_map_sequence_alignment_length() );
	}

	classical_constraints::BOUNDARY::init_module();

	if ( query_defined ) read_aa( seq_undefined ); // sequence
	setup_protein_sstype( seq_undefined ); //chu ss_type

	if ( seq_undefined ) { // Get total_residue from first pdb file
		if ( use_fasta ) {
			std::cerr << "ERROR:: Unable to obtain sequence information.\n";
			std::cerr << "        fasta file must be provided.\n";
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		get_size_from_first_pdb( require_start, silent_input );
	}

	// yab:
	// The following line is FORCED.  This actually should not happen!
	// Object instantiation should take care of this!!  But due to current
	// initialization pipeline there's no way to get around this!!!
	// See notes at top of constraints.cc.
	classical_constraints::BOUNDARY::reset_cst_res_wt();

	// Check that MAX_RES is set (not an assert since bad inputs can cause this)
	if ( ! MAX_RES().initialized() ) {
		std::cerr << "ERROR:: MAX_RES dimension cannot be determined from inputs\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	structure::BOUNDARY::init_module(); // setup SecondaryStructureEval instance this needs to happen AFTER total_residue is obtained
	if ( use_homolog_env_stats ) structure::BOUNDARY::read_homolog_env_stats_file();
	if ( use_homolog_pair_stats ) structure::BOUNDARY::read_homolog_pair_stats_file();
	if ( use_homolog_cendist_stats ) {
		//std::string distance_pdf_filename = stringafteroption("distance_pdf_filename");
		//read_distance_pdf( distance_pdf_filename );
		//initialize_distance_pdf();
	}
	//barcode_initialize_start(total_residue); // 2005-10-14 murphp: moved this call to after read_fragments()

	if ( use_constraints ) {
		if ( seq_undefined) {
			std::cout << "WARNING:: Protein sequence not defined by fasta" << std::endl;
			std::cout << "          Unable to process constraints" << std::endl;
			std::cout << "To use constraints:  " << std::endl;
			std::cout << "  Define PDB ID and chain and provide fasta file" << std::endl;
		} else {
			classical_constraints::BOUNDARY::load_constraints_from_file(); // distance constraints
			std::string const fullname( constraints_path + protein_name_prefix + protein_name + protein_chain + '.' + dpl_ext );
			// dipolar
			bool dipolar_exist;
			read_dipolar( fullname, total_residue, res, dipolar_exist );
		}
	}

	if ( require_frags ) {
		read_fragments();
		fragment_diversity();
	}

	barcode_initialize_start(total_residue); // 2005-10-14 murphp: moved this call from before read_fragments()

	setup_filters();

	if ( !query_defined ) return;

	docking_init_unboundrot_query();

// ------ native structure-----------------------
	std::string filename( pdb_path_1 + protein_name_prefix + protein_name + ".pdb" );
	std::cout << "Searching for pdb...: " << filename << std::endl;
	pdb_x.clear();
	pdb_x.open( filename );
	if ( !pdb_x ) {
		filename = pdb_path_2 + protein_name_prefix + protein_name + ".pdb";
		std::cout << "Searching for pdb...: " << filename << std::endl;
		pdb_x.clear();
		pdb_x.open( filename );
		if ( !pdb_x ) {
			std::cout << "no pdb file for " << protein_name_prefix << protein_name << std::endl;
			pdb_x.clear();
			return;
		}
	}

//car read coordinates,add centroids,angles,CB
	reset_secstruct();
	bool fail;
	input_pdb( pdb_x, use_fasta, input_fa, fail );
	pdb_x.close();
	pdb_x.clear();
	if ( fail ) {
		std::cout << "Failed to input native: " << pdb_x.filename() << std::endl;
		 // cbs
		return;
	}

	read_ss( "native" ); // define native ss (if not defined already)
	docking_init_query();

	// this sets native as current_pose if pose_flag()
	store_native(input_fa || get_use_native_centroid_flag() , output_fa);

//---------- score native --------------
	structure::BOUNDARY::setup_SS_default_weight(); // Use user-specified parallel or antiparallel weights to score native.
	bool const use_barcode_to_score = get_use_barcode_to_score();
	if (use_barcode_to_score) barcode_initialize_decoy();
	if (score_get_vary_omega()) {
		setup_omega_tether();
	}
	if ( get_looprlx_exist() ) reset_loop_chain_breaks();

	if ( pose_flag() && !domain_insertion ) {
		// doesnt make scorefile entry or anything, improve this later
		// note the early return; nothing after this will happen
		score_set_default_function(output_fa);
		score_set_evaluate_all_terms(true);
		initialize_get_native_pose().score( scorefxn );
		score_set_evaluate_all_terms(false);
		return;
	}

	initialize_maps();
	score_set_new_pose(); // ensure complete calculation
	dipolar_set_verbose(true);
	score_set_default_function(output_fa);
	score_set_evaluate_all_terms(true);
	filename = protein_name_prefix + protein_name + ".pdb";
	select_rotamer_set( "large" );

	if ( !scorefile_entry_present( filename, "native", output_fa ) ) {
		if ( output_fa ) {
			fullatom_score_position(!input_fa,include_inputchi);
			decoystats_store_native();
			decoy_features_ns::decoy_features_store_native();
		} else {
			classical_constraints::BOUNDARY::constraints_set_verbose(true);
			mc_global_track::mc_score::score = scorefxn();
		}
		scorefile_output( filename, "native", output_fa );
	}

	if ( repack_input ) {
		if ( !scorefile_entry_present( filename, "nat_repacked", output_fa ) ) {
			fullatom_score_position(repack_input,include_inputchi);
			scorefile_output( filename, "nat_repacked", output_fa );
		}
	}

	docking_score_input( filename, "native" );
	convert_docking_nat_contact_map();

	select_rotamer_set( "default" );
	score_set_evaluate_all_terms( false );

}

////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_start
///
/// @brief carry out initializations specific to the start (ie a particular conformation of the query)
///
/// @detailed
///
/// @param[in]   mode - in - main rosetta protocol
/// @param[in]   startnum - in -number of current starting structure
/// @param[out]   outputnum - in - number of decoys to be built for this start
/// @param[out]   status - out - return status of  initialization
///
/// @global_read
///     io_options_bool   block files_paths.h
///     query_defined     files_paths.h
///     file_path_int     block files_paths.h
///     file_path_char    block files_paths.h
///     protein_name      misc.h
///     protein_chain     misc.h
///     total_residue     misc.h
///     score             misc.h
///
/// @global_write
///     current_pose      block misc.h
///
/// @remarks
///car many indirect changes to global variables
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
initialize_start(
	std::string const & mode,
	int const startnum,
	int const outputnum,
	std::string & status, // return status
  std::stringstream *pdbdata    // if == NULL, behaves normally, if != NULL will read PDB from the memory block
)
{
	using namespace files_paths;
	using namespace misc;
	using namespace param;

// status return values:
//  "complete" all outputs done for this start
//  "fail" unable to initialize this start
//  "okay" all good

// local variables
	bool angle_fail,coord_fail,file_exists;
	std::string filename;

//-------------Initialize starting structure ----------------------

	status = "fail"; // initialize

	barcode_initialize_start( total_residue );

	if ( pKa_mode::get_pKa_flag() ) {
		// Setup pH titration without hijacking the start list
		// Must be called before check_decoy since pH_titration filename modification depends on pH_ns::pH being set
		pKa_mode::setup_titration_list();
	}

//car check to see if the last decoy exists for this starting structure.
	if ( !overwrite_pdbs ) {
		check_decoy_exists( mode, startnum, outputnum, file_exists, filename );
		if ( file_exists ) {
			status = "complete";
			return;
		}
	}

	// jk if we only want one output file, create an empty pdb
	// jk so that other instances of rosetta don't even bother
	// jk reading the input pdb
	if ( output_coord && !overwrite_pdbs && ( outputnum == 1)) {
		touch_before_reading=true;
		std::string fullname;
		bool file_exists;
		check_decoy_exists( mode, startnum, 1, file_exists, filename );
		fullname = pdb_out_path + filename;
		utility::io::ozstream pdb_out_xstream;
		pdb_out_xstream.open( fullname+".in_progress" );
		pdb_out_xstream.close();
		pdb_out_xstream.clear();
	} else {
		touch_before_reading=false;
	}

	reset_timer();  //start timing operations for this structure

//car extended chain start:
	if ( start_file == "none" && !create_domins_fasta && !antibody_modeler) {
		assert( ( MAX_RES().initialized() ) );
		reset_hetero(); //pb + mj
		for ( int i = 1; i <= total_residue; ++i ) {
			phi(i) = init_phi;
			psi(i) = init_psi;
			omega(i) = init_omega;
			secstruct(i) = 'L';
			name(i) = "EXTD";
		}
		initialize_maps();

		if ( pose_flag() && !domain_insertion ) { // monica temp find a better fix
			pose_ns::Pose & start_pose(  *(new pose_ns::Pose ));
			initialize_set_start_pose( start_pose );
			bool const fullatom ( false );
			bool const ideal_pose ( true );
			bool const coords_init ( false );
			pose_from_misc( start_pose, fullatom, ideal_pose, coords_init );
			//start_pose.refold();
			status = "okay";
			return;
		}
		// Monica Berrondo: for domain_insertion mode, read in the host protein
		// and insertion protein names to read in coordinates later
		if( domain_insertion ) {
			pose_ns::Pose & start_pose( *(new pose_ns::Pose) );
			initialize_set_start_pose( start_pose );
			domins_start( start_pose );
			status = "okay";
			save_START();
			return;
		}
		refold(1,total_residue);
		save_START();
		status = "okay";
		return;
	}

	if (homolog_to_query_mapping) map_start_sequence = false;

	if (pdbdata == NULL){
		// Monica Berrondo: for domain_insertion mode, read in the host protein
		// and insertion protein names to read in coordinates later
		if( domain_insertion ) {
			pose_ns::Pose & start_pose( *(new pose_ns::Pose) );
			initialize_set_start_pose( start_pose );
			domins_start( start_pose );
			if (mode != "score") {
				return;
			}
		}
		// aroop: Antibody Modeling mode
		if( antibody_modeler || docking::antibody_modeling_dock_n_snug_flag ) {
			pose_ns::Pose & start_pose( *(new pose_ns::Pose) );
			initialize_set_start_pose( start_pose );
			antibody_modeling_initialize_pose( start_pose, mode );
			status = "okay";
			if (!docking::antibody_modeling_dock_n_snug_flag && (mode != "score"))
				return;
		}

		if ( silent_input ) {
			input_compressed(angle_fail,coord_fail);
			if ( angle_fail ) return;
			//coord_fail = true; // must refold before coordinates exist
		} else {
			utility::io::izstream pdb_stream;
			filename = open_start_pdb( pdb_stream );
			std::cout << "reading starting structure: " << pdb_stream.filename() << std::endl;
	//car read coordinates and get seqeuence if needed:
			reset_secstruct();
			input_pdb(pdb_stream, (use_fasta&&!map_start_sequence), input_fa,coord_fail);
	//car read torsion angles and secstruct if available
	//car overwrites angles measured from coordinates
			read_pdb_phipsi(pdb_stream,angle_fail);
			pdb_stream.close();
			pdb_stream.clear();
			if ( coord_fail && angle_fail ) return; // no coords or angles
			read_ss( "start" );
		}
  }else {  // mtyka addition - reads from *pdbdata provided pdbdata != NULL

    utility::io::icstream tmpstream(*pdbdata);  // convert string stream to icstream
		// decide if the data is silent or PDB
		std::string firstline;
	  tmpstream >> bite( firstline ) >> skip; // Read the whole line
		if ( firstline.substr(0,9) == "SEQUENCE:" ){
		   std::cout << "Silent Input file \n";
		 	 silent_input=true;
		}else{
		   std::cout << "PDB Input file \n";
		   silent_input=false;
		}

		tmpstream.seek_beg();
		if ( silent_input ) {
		  std::string start_file_sav = start_file;
			start_file = "";
			input_compressed_istream(tmpstream,angle_fail);
			start_file = start_file_sav;
			output_file = start_file_sav;
			if ( angle_fail ) return;
			coord_fail = true; // must refold before coordinates exist
		} else {
      std::cout << "Reading PDB file from string stream\n";
      reset_secstruct();
      input_pdb(tmpstream,use_fasta,input_fa,coord_fail);

      read_pdb_phipsi(tmpstream,angle_fail);
      if ( coord_fail && angle_fail ) return; // no coords or angles
      read_ss( "start" );
		}
  }

	if ( map_start_sequence ) { // in files_paths namespace
		// transform input structure to fasta-defined sequence
		// should leave the transformed coords,ss in misc arrays
		// note that the data currently in misc is taken from the start
		// pdb and/or silent-file and might not agree with the target
		// sequence. We have disabled sequence checking in silent-io
		// and input_pdb if map_start_sequence is true
		//
		// assumes that we have the query defined and a fasta-file available
		// to define the target sequence of the mapping
		const bool save_pose_flag( pose_flag() );
		const bool ok ( map_misc_coords( input_fa, !coord_fail,
																		 files_paths::idealized_structure ) );
		set_pose_flag( save_pose_flag ); // ensure return to proper state
		if (!ok ) {
			status = "fail";
			return;
		}
	}

	if ( pose_flag() ) {
		////////////////////////////////////////////////
		// pose-specific initialization
		// note: return statement at the end of this block!
		if ( coord_fail && ( angle_fail || !idealized_structure ) ) {
			// cant make coords structure w/o refolding:
			status = "fail";
			return;
		}
		pose_ns::Pose & start_pose( *(new pose_ns::Pose ));
		initialize_set_start_pose( start_pose );
		bool ideal_pose( idealized_structure );
		bool coords_init ( !coord_fail );
		pose_from_misc( start_pose, input_fa, ideal_pose, coords_init);
		//if ( coord_fail ) {
		//	start_pose.refold();
		//}
		status = "okay";
		if ( output_fa && !input_fa ) {
			// add sidechains if necessary
			select_rotamer_set("large");
			bool const repack_rotamers( true ); // they dont exist yet
			start_pose.set_fullatom_flag( true, repack_rotamers );
			select_rotamer_set("default");
		}

		// score:
		score_set_default_function(output_fa);
		score_set_evaluate_all_terms(true);
		start_pose.score( scorefxn );
		score_set_evaluate_all_terms(false);
		return;
	}

//chu save bond length and bond angles from starting structure if desired
//bqian moved. should be before setup_variable_regions.
	initialize_start_bond_info(coord_fail,total_residue,res,Eposition,
	 phi,psi,omega);

	initialize_maps();
	setup_variable_regions();
	initialize_start_loops();

	initialize_docking_start();
	docking_initialize_silent_input_file();

//bs	initialize_disulfide_globals();

	// this bool == if nucleic acids present, valid given mode?
	if ( !nucleic_acids_check( mode ) ) {
		status = "dna_fail";
		return;
	}

	// In "-score" mode, we would sometimes like to apply filters.
	// That requires setting up the filters with each structure,
	// not just at the beginning in initialize_query.
	if (get_apply_filters_flag()){
		setup_protein_sstype_from_input(); //Uses DSSP or whatever was read in.
		setup_filters();
		monte_carlo_reset();
		bool accepted ( true );
		apply_filters_wrapper( false /*fullatom*/, accepted );
	}


//car refold if desired and possible
	if ( refold_input && !( angle_fail && coord_fail ) ) {
		monte_carlo_accept_best(); // must initialize best before refolding
		refold(1,total_residue);
		coord_fail = false;
	}
	if ( coord_fail ) return; // do not have coordinates, do not proceed
	if ( !knot_filter() ) std::cout << "WARNING:: Starting structure " <<
	 filename << " is knotted!!" << std::endl;
	status = "okay";
	save_START(); // save changes from mode-specific initializations

	//chu for silent_output of a structure with non-ideal geometry
	silent_out_save_start_pose( input_fa, false, true );

//car score starting structure:
	select_rotamer_set( "large" );
	score_set_new_pose();
	dipolar_set_verbose( true );
	calc_rms();
	filename = start_file + ".pdb";
	score_set_default_function(output_fa);
	score_set_evaluate_all_terms(true);
	structure::BOUNDARY::setup_rsd_weight();  //TEMPORARY
	structure::BOUNDARY::setup_SS_default_weight();
	structure::BOUNDARY::setup_SS_weight();
	classical_constraints::BOUNDARY::setup_cst_res_weight();
	bool const use_barcode_to_score = get_use_barcode_to_score();
	if (use_barcode_to_score) barcode_initialize_decoy();
	if (score_get_vary_omega()) {
		setup_omega_tether();
	}
	if ( get_looprlx_exist() ) reset_loop_chain_breaks();

	if (  skip_scorefile_check || !scorefile_entry_present(filename,"input",output_fa) ) {
		if ( output_fa ) {
			fullatom_score_position(!input_fa,include_inputchi);
			decoystats_store_decoy();
			decoy_features_ns::decoy_features_store_decoy();
		} else {
			classical_constraints::BOUNDARY::constraints_set_verbose(true);
			mc_global_track::mc_score::score = scorefxn();
		}
		scorefile_output( filename, "input", output_fa );
	}

	if ( repack_input ) {
		if ( !scorefile_entry_present( filename, "inp_repacked", output_fa ) ) {
			fullatom_score_position(true,include_inputchi);
			scorefile_output( filename, "inp_repacked", output_fa );
			save_START(); // save repacked coords as START
		}
	}

	docking_score_input(filename,"input");
	loop_score_start(filename,output_fa,include_inputchi);

	select_rotamer_set( "default" );
	score_set_evaluate_all_terms( false );

//bqian initialize taboo search for new starting structure
	initialize_taboo_start();
	setup_termini();

// aroop: fragment insertion whilst docking rigid body moves
	if(docking::dle_flag)
		dle_initialize();

}

//////////////////////////////////////////////////////////////////////////////
/// @begin reset_secstruct
///
/// @brief
//  this routine signals that secstruct is no longer defined;
//  required by the logic that the dssp reader uses to decide
//  whether or not to read a dssp file
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void reset_secstruct()
{
	using namespace param;
	using namespace misc;

	secstruct = ' ';
}

////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_decoy
///
/// @brief carry out initializations specific to a decoy structure
///  (ie one simulation starting from a particular start conformation)
///
/// @detailed
///
/// @param
///
/// @global_read
///     start_pose              block start.h
///     ligand_flag             ligand.h
///     start_hetero_atom_coord ligand.h
///     start_allow_insert      start.h
/// @global_write
///     current_pose            block misc.h
///     hetero_atom_coord       ligand.h
///     allow_insert            maps_ns.h
///     n_low_accept            counters.h
///     naccept                 counters.h
///     ntrials                 counters.h
///
/// @remarks
///car arbitrarily calls with score0; normally a recover_LOW is
///car called shortly after this function
///car
///car many indirect changes to global variables
///
/// @references
///
/// @authors  car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
initialize_decoy()
{
	using namespace counters;
	using namespace ligand;
	using namespace misc;
	using namespace param;
	using namespace protein_maps;
	using namespace start;

	mc_global_track::diagnose::rms_min = 100.0;
	naccept = 0;
	ntrials = 0;
	n_low_accept = 0;

	if ( pose_flag() ) {
		// allocate space for this simulation's decoy:
		pose_ns::Pose & pose( *( new pose_ns::Pose ));
		initialize_set_decoy_pose( pose ); // ensures destruction
		// recover starting pose
		pose = initialize_get_start_pose();
		//set_current_pose( pose ); // copies to misc
		//update_sequence();
		// pointers to native and start for rmsds, etc
		pose.set_start_pose( initialize_get_start_pose() );
		if ( get_native_exists() ) {
			pose.set_native_pose( initialize_get_native_pose() );
		}
		// temporary hack for beta-sheet stuff:
		if ( truefalseoption("pairing_file") ) {
			pose_from_random_pairings( pose );
			//pose.refold();
		}
		return;
	}

	if ( files_paths::antibody_modeler) {
		set_pose_flag( true );
		// allocate space for this simulation's decoy:
		pose_ns::Pose & pose( *( new pose_ns::Pose ));
		initialize_set_decoy_pose( pose ); // ensures destruction
		// recover starting pose
		pose = initialize_get_start_pose();
		pose.copy_to_misc();
		update_sequence();
		pose.set_start_pose( initialize_get_start_pose() );
		return;
	}

	phi = start_phi;
	psi = start_psi;
	omega = start_omega;
	secstruct = start_secstruct;
	name = start_name;
	res = start_res;
	res_variant = start_res_variant;
	allow_insert = start_allow_insert;
	allow_repack = true; // repack unless we hear otherwise
	allow_rottrial = true;
	update_sequence();

	position = start_position;

	centroid = start_centroid;

	full_coord = start_fullcoord;

//mj initialize  to starting coordinates
	if ( get_ligand_flag() ) {
    	ligand_one->recover_start_coordinates(
	);
  	}

	reset_insert_map();
	initialize_docking_decoy();
	initialize_decoy_loops();
	score_set_new_pose();
	mc_global_track::mc_score::score = score0();
	save_status_info( "initialize", 0, 0 );
	monte_carlo_reset();
	reset_movie();
//bqian initalize search trace and taboo_map for new decoy
	initialize_taboo_decoy();
	//jack resetting trajectory info.  Do not remove.
	clear_trajectory();
	//	trajectory::rmsd_vector.clear();
	//	trajectory::energy_vector.clear();
	if ( get_pH_packing_flag() ) calc_dGprotonation_table();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin store_native
///
/// @brief save info about native structure
///
/// @detailed
///
/// @param[in]   input_fa - in - where fullatom coordinates read in?
///
/// @global_read
///     current_pose        block start.h
///     ligand_flag         ligand.h
///     hetero_atom_count   ligand.h
///     hetero_atom_coord   ligand.h
///
/// @global_write
///     native        block native.h
///     native_hetero_atom_coord       ligand.h
///
/// @remarks
///car may have secstruct and angles even if no pdb file
///
///car  native coords are translated to origin except in docking and loop modes
///
/// @references
///
/// @authors 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
store_native( bool const input_fa, bool const output_fa )
{
	using namespace ligand;
	using namespace misc;
	using namespace native;
	using namespace param;

	native_exists = true;

	if ( pose_flag() && !files_paths::domain_insertion ) { // monica temp find a better fix
		pose_ns::Pose & native_pose( *(new pose_ns::Pose ));
		initialize_set_native_pose( native_pose );
		native_pose.simple_fold_tree( total_residue );
		bool const ideal_pose( false ); // allow_bb_move ==> false
		bool const coords_init( true ); // new_*_refold ==> false
		pose_from_misc( native_pose, input_fa, ideal_pose, coords_init );
		if ( output_fa && !input_fa ) {
			select_rotamer_set("large");
			bool const repack_rotamers( true ); // they dont exist yet
			native_pose.set_fullatom_flag( true, repack_rotamers );
			select_rotamer_set("default");
		}
		//set_current_pose( native_pose );
		return;
	}
	if ( files_paths::domain_insertion ) { // monica temp find a better fix
		pose_ns::Pose & native_pose( *(new pose_ns::Pose ));
		initialize_set_native_pose( native_pose );
		domins_native( native_pose );
		return;
	}

// copy current location into native_ block
	for ( int i = 1; i <= total_residue; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			native_ca(j,i) = Eposition(j,2,i);
		}
	}
	nat_secstruct = secstruct;
	native_phi = phi;
	native_psi = psi;
	native_omega = omega;
	native_Eposition = Eposition;
	native_centroid = centroid;
	native_occupancy = pdb::occupancy;

// save fullatom coordinates (at least backbone, sc if known)
	if ( !input_fa ) {
		initialize_fullcoord_array(Eposition,native_coord,total_residue,res,
		 res_variant);
	} else {
		native_coord = full_coord;
	}

	if (get_use_native_centroid_flag()) {
		initialize_fullcoord_array(Eposition,native_coord,total_residue,res,
		 res_variant);
		native_centroid_parm_eachres = compute_centroid_parm_eachres( res, res_variant, native_coord);
	}


//mj save ligand coordinates
  if ( get_ligand_flag() ) {
    native_hetero_atom_coord.clear();
    native_hetero_atom_coord.reserve(ligand::ligand_one->atom_vector.size());
    for ( size_t i = 0; i < ligand::ligand_one->atom_vector.size(); ++i ) {
      native_hetero_atom_coord.push_back(ligand::ligand_one->atom_vector[i]->get_coordinates());
    }
  }

//     translate molecule to center
	if (( ! files_paths::multi_chain || design::dna_interface || files_paths::antibody_modeler ) && !get_loop_flag() ) {
		for ( int i = 1; i <= 3; ++i ) {
			float sum = 0.0;
			for ( int j = 1; j <= total_residue; ++j ) {
				sum += native_ca(i,j);
			}
			sum /= float(total_residue);
			for ( int j = 1; j <= total_residue; ++j ) {
				native_ca(i,j) -= sum;
			}
		}
	}

	docking_store_native();

}

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
/// @begin retrieve_native_centroid_pose
///
/// @brief recover centroid-level info about native
///
/// @detailed sufficient for centroid re-scoring
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors pb
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
retrieve_native_centroid_pose()
{
	using namespace misc;
	using namespace native;
	using namespace param;

	if ( !get_native_exists() ) {
		std::cout << "native doesnt exist!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

// copy into current location from native_ block
	for ( int i = 1; i <= total_residue; ++i ) {
		secstruct(i) = nat_secstruct(i);
		phi(i) = native_phi(i);
		psi(i) = native_psi(i);
		omega(i) = native_omega(i);
		for ( int j = 1; j <= MAX_POS; ++j ) {
			for ( int k = 1; k <= 3; ++k ) {
				Eposition(k,j,i) = native_Eposition(k,j,i);
			}
		}
		for ( int k=1; k<= 3; ++k ) {
			centroid(k,i) = native_centroid(k,i);
		}
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin get_native_exists
///
/// @brief return value of native_exist global
///
/// @detailed
///
/// @return
///
/// @global_read
///  native_exists static
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_native_exists()
{
	return native::native_exists;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin save_START
///
/// @brief record all information that must be restored when the next decoy is initialized
///
/// @detailed
///
/// @global_read
///     current_pose            block misc.h
///     hetero_atom_coord       ligand.h
///     allow_insert            maps_ns.h
///
/// @global_write
///     start_pose              block start.h
///     ligand_flag             ligand.h
///     hetero_atom_count       ligand.h
///     start_hetero_atom_coord ligand.h
///     start_allow_insert      start.h
///
/// @remarks
///car current_pose, including coordinates should be current before calling
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
save_START()
{
	using namespace ligand;
	using namespace misc;
	using namespace param;
	using namespace protein_maps;
	using namespace start;

	start_phi = phi;
	start_psi = psi;
	start_omega = omega;
	start_secstruct = secstruct;
	start_name = name;
	start_res = res;
	start_res_variant = res_variant;
	start_allow_insert = allow_insert;
	start_position = position;
	start_centroid = centroid;
	start_fullcoord = full_coord;

//mj save starting coordinates
  if ( get_ligand_flag() ) {
    ligand_one->set_start_coordinates();
  }

	start_initialized = true;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin delete_START
///
/// @brief eliminate the current 'start' pose to prepare for a new starting pose
///
/// @detailed
///c
/// @global_read
///
/// @global_write
///  start_initialized start.h
///
/// @remarks
///
/// @references
///
/// @authors car 3/17/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
delete_START()
{
	using namespace start;

	start_initialized = false;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin setup_start_list
///
/// @brief setup list of starting structures on which to operate
///
/// @detailed
///
/// @param[out]   nstartnm - out - number of starting structures
/// @param[out]   startnm - out - names of starting structure
/// @param[out]   outnm - out - names of output structure
/// @param[out]   startch - out - protein chain of starting structure
///
/// @global_read
///  silent_input      files_paths.h
///  protein_chain     misc.h
///
/// @global_write
///  silent_input_file files_paths.h
///
/// @remarks
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
setup_start_list(
	int & nstartnm, // number of starting pdb names
	FArray1D_string & startnm, // starting pdb files
	FArray1D_string & outnm, // output pdb files
	FArray1D_char & startch // starting chain id
)
{
	using namespace files_paths;
	using namespace param;

//car local
	std::string tmpc;
	std::string listname;

	stringafteroption( "s", "none", tmpc );
	if ( tmpc != "none" ) {
		if ( silent_input ) {
			if ( ! has_suffix( tmpc, ".out" ) ) tmpc += ".out";
			silent_input_file = tmpc;
			if ( truefalseoption( "all" ) ) {
				std::cout << "Using all files in " << silent_input_file << std::endl;
				make_start_list( nstartnm, startnm, outnm, startch );
			} else {
				stringafteroption( "l", "none", listname );
				if ( listname == "none" ) goto L999;
				std::cout << "Reading in labels/indices from list: " << listname << std::endl;
				read_start_list( listname, nstartnm, startnm, outnm, startch );
			}
		} else { //cj put single structure at top of list
			if ( has_suffix( tmpc, ".pdb" ) ) tmpc.erase( tmpc.length() - 4 ); // strip terminal .pdb
			std::cout << "Starting structure: " << tmpc << std::endl;
			nstartnm = 1;
			startnm( nstartnm ) = tmpc;
			strip_path( tmpc, outnm( nstartnm ) );
			startch( nstartnm ) = protein_chain;
		}
	} else {
		if ( silent_input ) goto L999;
		stringafteroption( "l", "none", listname );
		if ( listname == "none" ) {
			if ( require_start ) goto L998;
			nstartnm = 1;
			startch( nstartnm ) = protein_chain;
			startnm( nstartnm ) = "none";
			outnm( nstartnm ) = "none";
			calc_filename_max_length( nstartnm, outnm );
			return;
		}

		std::cout << "Reading in structure names from list: " << listname << std::endl;

		read_start_list( listname, nstartnm, startnm, outnm, startch );
	}

	calc_filename_max_length( nstartnm, outnm );

	// Set start file globals for getting first pdb size (input_pdb_get_size)
	start_file = startnm(1);
	output_file = outnm(1);
	protein_chain = startch(1);

	return;

L999:
	std::cout << "ERROR:: starting structure required" << std::endl;
	std::cout << "With -silent_input must specify: " << std::endl;
	std::cout << "             -s <silent format .out file>" << std::endl;
	std::cout << "AND          -l <label/index list file>" << std::endl;
	utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
L998:
	std::cout << "ERROR:: starting structure required" << std::endl;
	std::cout << "Must specify -s <start structure pdbfile>" << std::endl;
	std::cout << "OR           -l <start structures pdb list file>" << std::endl;
	utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin read_start_list
///
/// @brief read list of starting structures to operate on from a file
///
/// @detailed
///
/// @param[in]   listname - in - filename to read from
/// @param[out]   nstartnm - out - number of starting structures
/// @param[out]   startnm - out - names of starting structures
/// @param[out]   outnm - out - names of output structures
/// @param[out]   startch - out - protein chain of starting structures
///
/// @global_read
///  protein_chain     misc.h
///  start_path        files_paths.h
///  start_x           files_paths.h
///  chain_last_char   files_paths.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
read_start_list(
	std::string const & listname,
	int & nstartnm, // number of starting pdb names
	FArray1D_string & startnm, // starting pdb files
	FArray1D_string & outnm, // output pdb files
	FArray1D_char & startch // starting chain id
)
{
	using namespace files_paths;
	using namespace param;

	start_x.clear();
	start_x.open( listname );
	if ( !start_x ) {
		std::cout << "ERROR:: " << start_x.filename() <<
		 " does not exist ... exiting   !" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	std::string inpline;
	std::string tmp_start;
	std::string tmp_output;
	nstartnm = 0;

	while ( getline( start_x, inpline ) ) {
    std::istringstream line_stream ( inpline,std::istringstream::in );
    line_stream >> tmp_start;
		ObjexxFCL::strip_whitespace( tmp_start ); // Trim off any leading or trailing whitespace
		if ( has_suffix( tmp_start, ".pdb" ) ) tmp_start.erase( tmp_start.length() - 4 ); // strip terminal .pdb

		++nstartnm;
		if ( nstartnm > MAX_START ) {
			std::cout << "Error::number of pdbs in " <<
				listname << " exceeds MAX_START (" <<
				MAX_START << ')' << std::endl;
			std::cout << "exiting..." << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		//bk retrieve chain id
		if ( chain_last_char ) {
			int const nd = tmp_start.length() - 1;
			startch( nstartnm ) = tmp_start[nd];
			tmp_start.erase( nd );
		} else {
			startch( nstartnm ) = protein_chain;
		}

		// read the output file if specified, otherwise get it from the start file
    if ( line_stream >> tmp_output ) {
			ObjexxFCL::strip_whitespace( tmp_output ); // Trim off any leading or trailing whitespace
			if ( has_suffix( tmp_output, ".pdb" ) ) tmp_output.erase( tmp_output.length() - 4 ); // strip terminal .pdb
			// if reading directly, DON'T strip path!
			outnm( nstartnm ) = tmp_output;
		} else {
			// if taking from the start file, strip path
			strip_path( tmp_start, outnm( nstartnm ) );
		}

		startnm( nstartnm ) = tmp_start;
	}

	start_x.close();
	start_x.clear();
	std::cout << "Number of starting structures: " << nstartnm << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin make_start_list
///
/// @brief create list of starting names from a silent input file
///
/// @detailed
///
/// @param[out]   nstartnm - out - number of starting structures
/// @param[out]   startnm - out - names of starting structure
/// @param[out]   outnm - out - names of output structures
/// @param[out]   startch - out - protein chain of starting structures
///
/// @global_read
///  protein_chain     misc.h
///  start_path        files_paths.h
///  start_x           files_paths.h
///
/// @global_write
///
/// @remarks
///car implements -all option for silent input
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
make_start_list(
	int & nstartnm, // number of starting pdb names
	FArray1D_string & startnm, // starting pdb files
	FArray1D_string & outnm, // output pdb files
	FArray1D_char & startch // starting chain id
)
{
	using namespace files_paths;
	using namespace param;

//car local
	std::string const filename( start_path + silent_input_file );

	start_x.clear();
	start_x.open( filename );
	if ( !start_x ) {
		std::cout << "ERROR:: Unable to open silent_input file: " <<
		 start_x.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	nstartnm = 0;
L30:
	std::string line, tag, tmp_tag;
	while ( start_x ) {
		start_x >> bite( line ) >> skip;
		if ( start_x ) {
			if ( line.substr(0,6) != "SCORE:" ) goto L30;
			if ( line.substr(12,5) == "score" ) goto L30; // formatting assumption

			// get the last thing on the line:
			std::istringstream line_stream( line );
			line_stream >> tmp_tag >> tag;
			assert( tmp_tag == "SCORE:" );
			if ( tag == "score" ) goto L30;
			while ( !line_stream.fail() ) {
				tag = tmp_tag;
				line_stream >> tmp_tag;
			}
			if ( tag == "description" ) goto L30;

			// add this tag to the list
			++nstartnm;
			startnm( nstartnm ) = tag;
			strip_path( tag, outnm( nstartnm ) );
			startch( nstartnm ) = protein_chain;
		}
	}
	start_x.close();
	start_x.clear();
	std::cout << "Number of starting structures: " << nstartnm << std::endl;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin calc_filename_max_length
///
/// @brief
///
/// @detailed
///
/// @param  nstartnm - [in/out]? -
/// @param  outnm - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
calc_filename_max_length(
	int const nstartnm,
	FArray1D_string const & outnm
)
{
	using namespace param;
	using namespace files_paths;
	using namespace fml_common;

	std::string dummy;

	fml_common::filename_max_length = 21; // the old default
	for ( int i = 1; i <= nstartnm; ++i ) {
		strip_path( outnm(i), dummy );
		int extension = 2 + 9; // xx + _0001.pdb
		if ( output_pdb_gz ){
			extension += 3; // .gz
		}

		filename_max_length = std::max( filename_max_length,
		 static_cast< int >( dummy.length() ) + extension );

	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin retrieve_filename_max_length
///
/// @brief
///
/// @detailed
///
/// @param[out]  fml -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
retrieve_filename_max_length( int & fml )
{
	using namespace fml_common;

	fml = fml_common::filename_max_length;
}

/////////////////////////////////////////////////////////////////////////////
//// store native, start, decoy poses ///////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

namespace initialize_pose_ns {
	pose_ns::Pose * native_pose( 0 );
	pose_ns::Pose * start_pose( 0 );
	pose_ns::Pose * decoy_pose( 0 );
}

void
initialize_set_native_pose(
	pose_ns::Pose & pose
)
{
	using namespace initialize_pose_ns;
	if ( native_pose != &pose && native_pose != 0 ) {
		std::cout << "cant re-assign native poses!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	native_pose = &pose;
}

pose_ns::Pose &
initialize_get_native_pose()
{
	using namespace initialize_pose_ns;
	if ( native_pose == 0 ) {
		std::cout << "STOP: native_pose does not exist!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	return *native_pose;
}

//
void
initialize_set_start_pose(
	pose_ns::Pose & pose
)
{
	using namespace initialize_pose_ns;
	if ( start_pose != &pose && start_pose != 0 ) delete start_pose;
	start_pose = &pose;
}

pose_ns::Pose &
initialize_get_start_pose()
{
	return *initialize_pose_ns::start_pose;
}

//
void
initialize_set_decoy_pose(
	pose_ns::Pose & pose
)
{
	using namespace initialize_pose_ns;
	if ( decoy_pose != &pose && decoy_pose != 0 ) delete decoy_pose;
	decoy_pose = &pose;
}

pose_ns::Pose &
initialize_get_decoy_pose() {
	return *initialize_pose_ns::decoy_pose;
}

bool
get_use_barcode_to_score()
{
	static bool init = { false };
	static bool use_barcode_to_score = { false };

	if ( !init ) {
		//If false, just add centroid pack score to cb_Score.
		use_barcode_to_score = truefalseoption("use_barcode_to_score");
		init = true;
	}

	return use_barcode_to_score;
}

bool get_apply_filters_flag()
{
	static bool init ( false );
	static bool apply_filters_flag ( false );

	if (!init){
		apply_filters_flag = truefalseoption( "apply_filters" );
		init = true;
	}

	return apply_filters_flag;
}

///////////////////////////////////////////////////////////////////////////////
void
read_checkpoint_decoy(
	std::string filename,
	bool fullatom
){

	using namespace misc;

	utility::io::izstream pdb_stream;

	pdb_stream.clear();
	pdb_stream.open( filename );
	if ( !pdb_stream ) {
		std::cout << "ERROR:: Can't find checkpoint file: " << pdb_stream.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	bool coord_fail( false );
	bool angle_fail( false );
	std::cout << "reading starting structure: " << pdb_stream.filename() << std::endl;
	reset_secstruct();
	input_pdb(pdb_stream, true /*use_fasta*/, fullatom/*input_fa*/, coord_fail );
	read_pdb_phipsi(pdb_stream,angle_fail);
	pdb_stream.close();
	pdb_stream.clear();
	if ( coord_fail && angle_fail ) {
		std::cout << "ERROR:: Read check point file error: " << pdb_stream.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	set_checkpoint_status( fullatom );

}

////////////////////////////////////////////////////////////////////////////////
/// @begin nucleic_acids_check
////////////////////////////////////////////////////////////////////////////////
bool nucleic_acids_check( std::string const mode )
{
	design::dna_interface = false;
	for ( int pos = 1; pos <= misc::total_residue; ++pos ) {
		if ( param_aa::is_NA(misc::res(pos)) ) {
			design::dna_interface = true;
			break;
		}
	}
	if ( ( mode != "design" && mode != "pdbstats" && mode != "score" ) &&
	 design::dna_interface ) {
		std::cout << "ERROR: Starting structure contains nucleic acids, but " <<
		"Nucleic acids are not allowed in this mode!" << std::endl;
		return false;
	}
	return true;
}
////////////////////////////////////////////////////////////////////////////////


#ifdef BOINC_GRAPHICS

// In principle the following function belongs in boinc_rosetta_graphics.cc or
// boinc_rosetta_util.cc, but there's a strange conflict that occurs when you
// use streams and the BOINC WIN32 libraries.

void get_wu_desc() {
	using namespace graphics;
	static bool init = {false};

	if (!init){
		// read description file if one exists
		// keep row format
		std::string description_file;
		stringafteroption("description_file","rosetta_description.txt", description_file);
		utility::io::izstream desc_stream( description_file );
		if ( desc_stream ) {
			std::string tmpstr;
			while (desc_stream && !desc_stream.eof()) {
				desc_stream.getline( tmpstr );
				wu_desc_rows.push_back( tmpstr );
			}
		}
		desc_stream.close();
		desc_stream.clear();

		if (wu_desc_rows.size() > 0) {
			wu_text_box_height = float(wu_desc_rows.size()+1)/wu_desc_rows_per_small_box;
			aspect_height += wu_text_box_height;
		}
		init = true;
	}
}
#endif

