// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 23432 $
//  $Date: 2008-06-24 16:25:52 +0300 (Tue, 24 Jun 2008) $
//  $Author: yab $


// Rosetta Headers
#include "input_pdb.h"
#include "aa_name_conversion.h"
#include "aaproperties_pack.h"
#include "angles.h"
#include "after_opts.h"
#include "design.h"
#include "diagnostics_rosetta.h"
#include "dna_ns.h"
#include "docking.h"
#include "docking_ns.h"
#include "enzyme_ns.h"
#include "files_paths.h"
#include "fullatom.h"
#include "fullatom_setup.h"
#include "jumping_refold.h"
#include "ligand.h"
#include "maps.h"
#include "misc.h"
#include "monte_carlo.h"
#include "misc_removal.h" //Quick hack.
#include "namespace_fullatom_flag.h"
#include "output_decoy.h"
#include "pack.h"
#include "pack_geom_inline.h"
#include "PackerTask.h"
#include "param.h"
#include "param_aa.h"
#include "pdb.h"
#include "pose_ligand.h"
#include "current_pose.h"
#include "pose_io.h"
#include "random_numbers.h"
#include "read_aa_ss.h"
#include "read_aaproperties.h"
#include "recover.h"
#include "refold.h"
#include "rotamer_functions.h"
#include "runlevel.h"
#include "silent_input.h"
#include "symmetric_design.h"
#include "template_pack.h"
#include "termini.h"
#include "util_vector.h"
#include "water.h"
#include "enzyme.h"

//KMa phospho_ser
#include "add_pser.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/char.functions.hh>
#include <ObjexxFCL/string.functions.hh>

// Numeric Headers
#include <numeric/conversions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/irstream.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <cmath>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <string>
#include <vector>


// Namespaces
namespace centroid_private {
	FArray2D_float new_res( 3, 4 );
	FArray2D_float old_res( 3, 4 );
}


////////////////////////////////////////////////////////////////////////////////
/// @begin input_pdb
///
/// @brief
///car read coordinates from an open file
///car coordinates put in position array, and calpha etc arrays
///
/// @detailed
///car option dependencies:
///car   read_all_chains  ignore chain info, read past TER, treat as single chain/domain
///car   allow_missing    allow missing backbone atoms
///car   skip_missing     skip residues with missing atoms
///car   multi_chain      read multiple chains; TER defines chains/domains
///
/// @param  iunit - [in/out]? -
/// @param  seq_defined - [in/out]? - sequence already defined by fasta/dat etc
/// @param  fullatom - [in/out]? - read fullatom coordinates?
/// @param  fail - [in/out]? - read failure?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
input_pdb(
	utility::io::irstream & pdb_stream,
	bool const seq_defined, // sequence already defined by fasta/dat etc
	bool const fullatom, // read fullatom coordinates?
	bool & fail // read failure?
)
{

//------------------------------------------------
//  THIS READS PDB FILES
//car revised Sept 28,2000
//     last revised December 4, 1996
//     written by Kim T Simons
//------------------------------------------------

	using namespace docking;
	using namespace files_paths;
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace pdb;
	using namespace runlevel_ns;

//car local
	float dist2;
	int iatom; // current res and atom (iatom is fullcoord # scheme)
	int idomain; // number of the current domain
	char ichain; // current chain being read
	char last_insert_let;
	int last_res_num; // detecting new residue
	int nmissing, nmissing_res; // counters for missing atoms
	FArray1D_bool missing_atom( MAX_ATOM()(), true );  // missing atoms in current residue
	bool new_domain, is_missing = false;
	int start, length;

//car input fields from pdbfile
	std::string pdb_card;
	int atom_num; // atom number
	std::string atom_name;
	char res_conf;
	std::string pdb_resname, prev_pdb_resname( 3, ' ' );
	char chain_id, prev_chain_id = ' ';
	int res_num, prev_res_num = 0; // residue number
	char insert_let;
	float x,y,z; // coordinates
	float bfactor; // bfactor
	float occ; // occupancy
	float dis; // tmp

	bool const start_Nterm = termini_ns::use_N_terminus;
	bool const start_Cterm = termini_ns::use_C_terminus;

//car-----------------------
//car initialize
//car-----------------------
	Eposition = 0.0f;
	full_coord = 0.0f;

	int nres = input_pdb_get_size( pdb_stream );
	//lin when treat ligand as amino acid, we do need extra residue
	//if( get_read_ligaa_flag() ) nres = nres + 2 ;
	pdb_stream.seek_beg(); // Reposition to top of pdb file

	if ( nres != total_residue ) {
		if ( seq_defined ) {
			std::cout << "Number of residues in pdb file " << nres
			 << " disagrees with total_residue: " << total_residue << std::endl;
			fail = true;
			return;
		} else {
			total_residue = nres;
			if ( total_residue <= 0  ) {
				if( !get_read_ligaa_flag() ) {
					std::cerr << "ERROR:: Unable to determine sequence length from pdb file\n";
					fail = true;
					return;
				}
				//utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			} else {
				// For symmetrical pose we need to add extra residues
				if ( docking_pose_symm_subsystem && docking_pose_symm_subsystem &&
						!monomer_input && docking_pose_symm_subsystem_size == 0 ) docking_pose_symm_subsystem_size = pose_symm_n_monomers;
				int const num_simulated_monomers = docking_pose_symm_subsystem ? docking_pose_symm_subsystem_size : pose_symm_n_monomers;
				int tot_res (  docking_pose_symmetry ? (monomer_input ? num_simulated_monomers*total_residue + num_simulated_monomers : total_residue + num_simulated_monomers ) : total_residue );
//        if( get_enable_ligaa_flag() && get_docking_flag() ) // SJF
//          tot_res += 2;
				MAX_RES_assign_res( tot_res ); // Set MAX_RES
			}
		}
	}
	if( get_enable_ligaa_flag() && get_docking_flag() ) {
      int const tot_res = nres + 2;
			MAX_RES_assign_res( tot_res ); // Set MAX_RES
	}


	fail = false;

	int ires = 0;
	idomain = 1;
	new_domain = false; // first res begins domain 1
	ichain = protein_chain;
	if ( ichain == '_' ) {
		ichain = ' ';
	}

	last_res_num = -999;
	last_insert_let = 'Z';


//bk intialize array which keeps track of atoms missing from pdb
//bk and initialize array that contains bvalues
	nmissing = 0;
	nmissing_res = 0;
	missing_input_atom = true;
	bvalue = 80.0; // default is high (poor) b value
	occupancy = -1.0; // default is missing occupancy


//car-----------------------
//car find start of pdb file:
//car-----------------------

L40:
	pdb_stream >> bite( 6, pdb_card ) >> bite( 5, atom_num ) >> skip( 1 ) >>
	 bite( 4, atom_name ) >> bite( res_conf ) >> bite( 3, pdb_resname ) >>
	 skip( 1 ) >> bite( chain_id ) >> bite( 4, res_num ) >>
	 bite( insert_let ) >> skip( 3 ) >>
	 bite( 8, x ) >> bite( 8, y ) >> bite( 8, z ) >>
	 bite( 6, occ ) >> bite( 6, bfactor ) >> skip;
	if ( pdb_stream.eof() ) {
		pdb_stream.clear();
		if( get_read_ligaa_flag() ) {// allow to read only ligand
			goto L500;
		} else {
			goto L50;
		}
	} else if ( pdb_stream.fail() ) {
		pdb_stream.clear();
		pdb_stream >> skip;
		goto L40;
	}
	if ( runlevel > inform ) std::cout << pdb_card << "--" << chain_id << "--" <<
	 atom_name << std::endl;

	if ( pdb_card != "ATOM  " ) goto L40;

//car if the chain isn't defined yet, define as the first chain we find
	if ( ichain == '-' ) {
		ichain = chain_id;
		protein_chain = ichain;
		if ( protein_chain == ' ' ) protein_chain = '_';
	}
//car decide if its the correct chain:
	if ( !read_all_chains && ichain != chain_id ) goto L40;


//car-----------------------
//car read the coordinates:
//car-----------------------
	while ( true ) {

//car check termination conditions:
//car check for TER to signal new domain
		if ( pdb_card.substr(0,3) == "TER" ) {
			if ( !read_all_chains && !multi_chain ) goto L500;
			new_domain = true; // next res begins new_domain
		}
//car different chain
		if ( !read_all_chains && chain_id != ichain ) goto L500;

//bk skip alternate configurations
		if ( res_conf == 'B' || res_conf == 'C' || res_conf == 'D' ) goto L300;

//car read ATOM records:
		if ( pdb_card == "ATOM  " ) {

//car detect new residue:
			if ( last_res_num != res_num || last_insert_let != insert_let ) {

				last_res_num = res_num;
				last_insert_let = insert_let;

// before starting new residue, check if previous residue is complete:
				if ( ires > 0 ) {
					check_missing_residue( missing_atom, ires, nmissing, nmissing_res,
					 is_missing, prev_res_num, prev_chain_id, prev_pdb_resname );
					save_missing_atoms( missing_atom, ires );
					if ( is_missing ) {
						if ( skip_missing || runlevel > standard ) {
							std::cout << "residue " << ires << " (PDB " << res_chain(ires) <<
								' ' << pdb_res_num(ires) << ' ' << residue3(ires) << ") ";
							if ( skip_missing ) {
								std::cout << "is excluded" << std::endl;
							} else {
								std::cout << "is missing backbone atoms" << std::endl;
							}
						}
					}
				}
				if ( !is_missing || !skip_missing ) {
					prev_res_num = res_num;
					prev_chain_id = chain_id;
					prev_pdb_resname = pdb_resname;
					++ires;
				}
				missing_atom = true;

				if ( new_domain ) {
					domain_end(idomain) = ires-1;
					++idomain; // next domain
					if ( idomain > MAX_SEGMENTS ) {
						std::cout << "Increase MAX_SEGMENTS in param.cc" << std::endl;
						utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
					}
					new_domain = false;
				}

				pdb_res_num(ires) = res_num; // res info
				pdb_insert_let(ires) = insert_let;
				res_chain(ires) = chain_id;
				if ( seq_defined ) {
					//jg !!! this is not working for unbound docking structure with a different sequence
					if ( pdb_resname != residue3(ires) && ires <= total_residue ) {
						std::cout << "WARNING!  Sequence in pdb file " <<
						 "disagrees with fasta or dat file" << std::endl;
						std::cout << "position " << ires << "  pdb_resname " <<
						 pdb_resname << " residue3 " << residue3(ires) << std::endl;
						fail = true;
						return;
					}
				} else {
					residue3(ires) = pdb_resname;
					num_from_name( pdb_resname, res(ires) );
					if ( res(ires) == 0 ) {
						std::cout << "unrecognized residue: " << pdb_resname << std::endl;
						std::cout << "current res: " << ires << " pdb res " << res_num << std::endl;
						fail = true;
						return;
					}
				}
			}

			if ( atom_name == " N  " ) {
				Eposition(1,1,ires) = x;
				Eposition(2,1,ires) = y;
				Eposition(3,1,ires) = z;
				if ( occ > 0.0 ) missing_atom(1) = false;
				else if( randomize_missing ){
					Eposition(1,1,ires) = ran3()*9999;
					Eposition(2,1,ires) = ran3()*9999;
					Eposition(3,1,ires) = ran3()*9999;
				}
				bvalue(1,ires) = bfactor;
				occupancy(1,ires) = occ;
			} else if ( atom_name == " CA " ) {
				Eposition(1,2,ires) = x;
				Eposition(2,2,ires) = y;
				Eposition(3,2,ires) = z;
				if ( occ > 0.0 ) missing_atom(2) = false;
				else if( randomize_missing ){
					Eposition(1,2,ires) = ran3()*9999;
					Eposition(2,2,ires) = ran3()*9999;
					Eposition(3,2,ires) = ran3()*9999;
				}
				bvalue(2,ires) = bfactor;
				occupancy(2,ires) = occ;
			} else if ( atom_name == " C  " ) {
				Eposition(1,4,ires) = x;
				Eposition(2,4,ires) = y;
				Eposition(3,4,ires) = z;
				if ( occ > 0.0 ) missing_atom(3) = false;
				else if( randomize_missing ){
					Eposition(1,4,ires) = ran3()*9999;
					Eposition(2,4,ires) = ran3()*9999;
					Eposition(3,4,ires) = ran3()*9999;
				}

				bvalue(3,ires) = bfactor;
				occupancy(3,ires) = occ;
			} else if ( atom_name == " CB " ) {
				Eposition(1,3,ires) = x;
				Eposition(2,3,ires) = y;
				Eposition(3,3,ires) = z;
				if ( occ > 0.0 ) missing_atom(5) = false;
				else if( randomize_missing ){
					Eposition(1,3,ires) = ran3()*9999;
					Eposition(2,3,ires) = ran3()*9999;
					Eposition(3,3,ires) = ran3()*9999;
				}
				bvalue(5,ires) = bfactor;
				occupancy(5,ires) = occ;
			} else if ( ( atom_name == " O  " ) || ( atom_name == " OT1" ) ) {
				Eposition(1,5,ires) = x;
				Eposition(2,5,ires) = y;
				Eposition(3,5,ires) = z;
				if ( occ > 0.0 ) missing_atom(4) = false;
				else if( randomize_missing ){
					Eposition(1,5,ires) = ran3()*9999;
					Eposition(2,5,ires) = ran3()*9999;
					Eposition(3,5,ires) = ran3()*9999;
				}
				bvalue(4,ires) = bfactor;
				occupancy(4,ires) = occ;
			}
			if ( fullatom || get_use_native_centroid_flag() ) {
				if( design::try_both_his_tautomers && res(ires)==aa_his ){
					atom_num_from_atom_name(atom_name,res(ires),1,iatom);
					if(iatom==0) atom_num_from_atom_name(atom_name,res(ires),2,iatom);
				} else {
					atom_num_from_atom_name( atom_name, res(ires), 1, iatom );
				}

				if ( iatom != 0 ) {
					full_coord(1,iatom,ires) = x;
					full_coord(2,iatom,ires) = y;
					full_coord(3,iatom,ires) = z;
					if ( occ > 0.0 ) missing_atom(iatom) = false;
					bvalue(iatom,ires) = bfactor;
					occupancy(iatom,ires) = occ;
				} else {
//ora: report atom name that was not recognized
					if ( runlevel > inform ) std::cout << "Ignored " << atom_name << " in " <<
					 residue3(ires) << ires << std::endl;
				}
			}    // different atom names
		} // pdbcard == "ATOM  "

L300:;
		// Read the next line
		pdb_stream >> bite( 6, pdb_card ) >> bite( 5, atom_num ) >> skip( 1 ) >>
		 bite( 4, atom_name ) >> bite( res_conf ) >> bite( 3, pdb_resname ) >>
		 skip( 1 ) >> bite( chain_id ) >> bite( 4, res_num ) >>
		 bite( insert_let ) >> skip( 3 ) >>
		 bite( 8, x ) >> bite( 8, y ) >> bite( 8, z ) >>
		 bite( 6, occ ) >> bite( 6, bfactor ) >> skip;
		if ( pdb_stream.eof() ) {
			pdb_stream.clear();
			goto L500;
		} else if ( pdb_stream.fail() ) {
			pdb_stream.clear();
			pdb_stream >> skip;
			goto L300;
		}
	}

L500:; // exit do while loop

//car check last residue for missing atoms
	if ( ires > 0 ) {
		check_missing_residue( missing_atom, ires, nmissing, nmissing_res,
		 is_missing, prev_res_num, prev_chain_id, prev_pdb_resname );
		save_missing_atoms( missing_atom, ires );
		if ( is_missing ) {
			if ( skip_missing || runlevel > standard ) {
				std::cout << "residue " << SS( ires ) << " (PDB " <<
					res_chain(ires) << ' ' << SS( pdb_res_num(ires) ) <<
					residue3(ires) << ") ";
				if ( skip_missing ) {
					std::cout << "is excluded" << std::endl;
					--ires;
				} else {
					std::cout << "is missing residues" << std::endl;
				}
			}
		}
	}


//car define end of last domain
	domain_end(idomain) = ires;
	total_domains = idomain;

//car-----------------------
//car error checking
//car-----------------------
	if ( nmissing > 0 ) {
		std::cout << "WARNING: " << nmissing << " missing backbone atoms at " <<
		 nmissing_res << " residues in pdb file" << std::endl;
		std::cout << "allow_missing= " << allow_missing << std::endl;
		std::cout << "skip_missing= " << skip_missing << std::endl;
		if ( !allow_missing ) fail = true;
	}

	if ( fail ) return;

//  CONNECTIVITY CHECK BETWEEN NUMERICALLY ADJACENT RESIDUES
	if ( runlevel >= inform ) {
		for ( int i = 1; i < total_residue; ++i ) {
			float const Edif1 = Eposition(1,4,i) - Eposition(1,1,i+1);
			float const Edif2 = Eposition(2,4,i) - Eposition(2,1,i+1);
			float const Edif3 = Eposition(3,4,i) - Eposition(3,1,i+1);
			dist2 = ( Edif1 * Edif1 ) + ( Edif2 * Edif2 ) + ( Edif3 * Edif3 );
			if ( dist2 > 4.0 ) {
				std::cout << "chain break: " << i << '-' << i+1 << ' ' <<
				 SS( std::sqrt(dist2) ) << std::endl;
				write_res( std::cout, i );
				std::cout << " C(i)";
				for ( int j = 1; j <= 3; ++j ) {
					std::cout << SS( Eposition(j,4,i) );
				} std::cout << std::endl;
				write_res( std::cout, i+1 );
				std::cout << " N(i)";
				for ( int j = 1; j <= 3; ++j ) {
					std::cout << SS( Eposition(j,1,i+1) );
				} std::cout << std::endl;
			} else if ( dist2 < 001 ) {
				std::cout << "zero length peptide bond: " << i << '-' << i+1 << std::endl;
				write_res( std::cout, i );
				std::cout << " C(i)";
				for ( int j = 1; j <= 3; ++j ) {
					std::cout << SS( Eposition(j,4,i) );
				} std::cout << std::endl;
				write_res( std::cout, i+1 );
				std::cout << " N(i)";
				for ( int j = 1; j <= 3; ++j ) {
					std::cout << SS( Eposition(j,1,i+1) );
				} std::cout << std::endl;
			}
		}
	}

// RESIDUE SEQUENCE
	if ( runlevel >= verbose ) {
		std::cout << "PDB input sequence:" << std::endl;
		for ( int i = 1; i <= total_residue; ++i ) {
			write_res_coords( std::cout, i );
		}
	}

//car-----------------------
//car complete structure
//car-----------------------
//car atom types, residue names
	if ( !seq_defined ) {
		convert3_num(); // generate res
		for ( int i = 1; i <= total_residue; ++i ) {
			res_variant(i) = 1;
			if( design::try_both_his_tautomers && res(i)==aa_his ){
				distance_bk(full_coord(1,12,i),full_coord(1,6,i),dis);
				if( dis < 2.5 && dis >2.0 ) res_variant(i)=2;
				else res_variant(i)=1;
			}
		}
		misc_removal::init_modules_TEMPORARY();
		update_sequence();
	}

	if ( ! quiettruefalseoption( "no_clean" ) ) { //SGM Option leaves GLY CB alone for testing
		for ( int i = 1; i <= total_residue; ++i ) {
			name(i) = "INPT"; // may be overwritten
			if ( res(i) == aa_gly ) compute_CB_position(Eposition(1,1,i)); // Cbeta for glycines
		}
	}

	if ( get_use_native_centroid_flag() ) {
		//rhiju assign centroids to be at actual centers of mass of
		//      side chains from input structure.
		centroid_place_useinputsidechain();
	}	else {
		//Default.
		centroid_place(); // centroid
	}

	for ( int  i = 1; i <= total_domains; ++i ) {
		if ( i == 1 ) {
			start = 1;
		} else {
			start = domain_end(i-1) + 1;
		}
		length = domain_end(i) - start + 1;
		if( length > 0 ) {
			torsion_from_position(length,Eposition(1,1,start),
														phi(start),psi(start),omega(start));
		}
	}

//jk read the header, if desired
	if ( preserve_header ) read_pdb_header( pdb_stream );

// jk temporarily turn off termini
// (because terminal variants aren't set up until the call to build_termini, below)
	termini_ns::use_N_terminus = false;
	termini_ns::use_C_terminus = false;

//car check for hetero atoms, see ligand.cc
	if( ligand::ligand_flag && ligand::ligand_mdlfile){
		stringafteroption("ligand_mdlfile", ( files_paths::start_path +
			files_paths::start_file + ".mdl" ),
					 files_paths::ligand_input_mdlfile);
		utility::io::izstream lig_iunit;
		lig_iunit.clear();
		lig_iunit.open(files_paths::start_path + files_paths::ligand_input_mdlfile);
	  delete ligand::ligand_one;
	  ligand::ligand_one = new Ligand();
		if (read_mdl( lig_iunit,(ligand::ligand_one))==-1){
			std::string ligand_mdlfile=(files_paths::start_path + files_paths::start_file + ".mol");
			lig_iunit.open(ligand_mdlfile);
			if( read_mdl( lig_iunit, (ligand::ligand_one)) == -1){
				std::cout << "Reading from mdlfile " << files_paths::ligand_input_mdlfile;
				std::cout << " failed." << std::endl;
				std::cout << "Looking in start pdb for ligand." << std::endl;
				read_pdb_hetero(pdb_stream, (*ligand::ligand_one));
			}
		}
		lig_iunit.close();
  } else if ( get_read_ligaa_flag() ) { //lin if read ligand as aa, setup ligand aa
		//using namespace enable_ligaa_ns;
    //lin read ligand from pdb and pass to the ligand_one
    enable_ligaa_ns::read_ligand( pdb_stream );
		if( total_residue <= 0 ) return;
  } else {
		read_pdb_hetero(pdb_stream, (*ligand::ligand_one));
	}

	// jk setup for symmetric packing
	if ( design_sym::num_clones > 0 )	setup_symmetric_packing(total_residue);

//lin check for water atoms, see water.cc:
	read_pdb_water(pdb_stream,res,full_coord,total_residue);

//lin building the inverse rotamer tree
	get_inverse_rotamers();

//car fill in missing sidechains, optimize hydrogens,aa_variants
	if ( fullatom ) {
		bool const clean_pdb_fail = clean_pdb(missing_input_atom);
		if ( clean_pdb_fail ) {
			fail = true;
			return;
		}
	}

// jk restore desired termini settings
	termini_ns::use_N_terminus = start_Nterm;
	termini_ns::use_C_terminus = start_Cterm;

//bq building termini
	build_termini( fullatom );

	return;

L50:
	std::cout << "can't find starting residue in pdb file" << std::endl;
	fail = true;
}


///////////////////////////////////////////////////////////////////////////////
//

bool
flip_symmetric_coords(
	int const aa,
	int const aav,
	FArray2Da_float coords
)
{
	using namespace param_aa;
	using aaproperties_pack::chi_atoms;
	using aaproperties_pack::chi_required;
	using aaproperties_pack::icoor;
	using numeric::conversions::radians;

	initialize_fullatom();

	coords.dimension(3,param::MAX_ATOM()());

	int chino(0);
	if ( aa == aa_asp || aa == aa_tyr || aa == aa_phe ) {
			chino = 2;
	} else if ( aa == aa_glu ) {
			chino = 3;
	} else if ( aa == aa_arg ) {
		// check for NH1 - NH2 flip by comparing distances to CD position
		int const nh1_atomno( LookupByName( aa, aav, " NH1") );
		int const nh2_atomno( LookupByName( aa, aav, " NH2") );
		int const cd_atomno ( LookupByName( aa, aav, " CD ") );

		numeric::xyzVector_float
			nh1(&coords(1,nh1_atomno)),
			nh2(&coords(1,nh2_atomno)),
			cd (&coords(1, cd_atomno));

		if ( distance(nh1,cd) > distance(nh2,cd) ) {
			std::cout << "flipping NH1 and NH2 on Arg" << std::endl;
			for ( int k=1; k<=3; ++k ) {
				coords(k,nh1_atomno) = nh2(k);
				coords(k,nh2_atomno) = nh1(k);
			}
			return true;
		}
		return false;
	} else {
		return false;
	}

	// save starting coords
	FArray2D_float save_coords(3,param::MAX_ATOM()() );
	for ( int j=1; j<= param::MAX_ATOM()(); ++j ) {
		for ( int k=1; k<= 3; ++k ) {
			save_coords(k,j) = coords(k,j);
		}
	}

	// calculate current chi:
	float const coord_chi
		( periodic_range( dihedral( coords(1,chi_atoms(1,chino,aa,aav)),
																coords(1,chi_atoms(2,chino,aa,aav)),
																coords(1,chi_atoms(3,chino,aa,aav)),
																coords(1,chi_atoms(4,chino,aa,aav))), 360.0));

	// get rosetta range for this chi angle and deviation
	float const rosetta_chi( set_chi_to_periodic_range( coord_chi, aa, chino ));
	float const angle_dev( subtract_degree_angles( rosetta_chi, coord_chi ) );

	if ( std::abs(angle_dev) < 0.1 ) return false; // no need to flip

	if ( std::abs( std::abs(angle_dev) - 180.0 ) > 0.1 ) {
		// this should not happen unless set_chi_to_periodic_range changes
		std::cout << "STOP: unexpected flip angle != 180.0!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// find corresponding atoms in icoor
	int const natoms( aaproperties_pack::natoms( aa, aav ) );
	FArray2D_float icoor_flipped(3,param::MAX_ATOM()());
	for ( int j=1; j<= natoms; ++j ) {
		for ( int k=1; k<= 3; ++k ) {
			icoor_flipped(k,j) = icoor(k,j,aa,aav);
		}
	}

	{ // flip icoor coordinates by 180.0 at this chi
		// stolen from ctsa -- minimize_chi
		float ichi, rot;
		FArray2D_float mat( 3, 3 );
		FArray1D_float vec( 3 );

		//// get initial chi angle
		ichi = dihedral(icoor_flipped(1,chi_atoms(1,chino,aa,aav)),
										icoor_flipped(1,chi_atoms(2,chino,aa,aav)),
										icoor_flipped(1,chi_atoms(3,chino,aa,aav)),
										icoor_flipped(1,chi_atoms(4,chino,aa,aav)));

		//// find rotation angle
		rot = radians( 180.0 );

		//// generate rotation vector and matrix
		Sgetrot(icoor_flipped(1,chi_atoms(2,chino,aa,aav)),
						icoor_flipped(1,chi_atoms(3,chino,aa,aav)),rot,mat,vec);

		//// move atoms
		for ( int i = 1, e = natoms; i <= e; ++i ) {
			if ( chi_required(chino,i,aa,aav) ) {
				Smove_in_place(icoor_flipped(1,i),mat,vec);
			}
		}
	}

	for ( int i=1; i<= natoms; ++i ) {
		if ( !chi_required(chino,i,aa,aav) ) continue;
		numeric::xyzVector_float ixyz( &icoor(1,i,aa,aav));
		int iflip(0);
		float min_d(1000.0);
		for ( int j=1; j<= natoms; ++j ) {
			if ( !chi_required(chino,j,aa,aav) ) continue;
			numeric::xyzVector_float jxyz( &icoor_flipped(1,j));
			float const d( distance(ixyz,jxyz) );
			if ( d<min_d ) {
				min_d = d;
				iflip = j;
			}
		}
		if ( iflip == i ) {
			//std::cout << "iflip=i: " << i << " min_d= " << min_d << std::endl;
		} else if ( min_d < 0.25 ) {
// 			std::cout << "Flipping atom: " << i << ' ' << iflip << ' ' <<
// 				aaproperties_pack::atom_name( i, aa, aav ) << ' ' << min_d <<
// 				std::endl;
			for ( int k=1; k<=3; ++k ) {
				coords(k,i) = save_coords(k,iflip);
			}
// 		} else {
// 			std::cout << "not flipping atom: " << i << ' ' << aa << ' ' <<
// 				aaproperties_pack::atom_name( i, aa, aav ) << ' ' << min_d <<
// 				std::endl;
		}
	}
	return true;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin clear_header
///
/// @brief
///   Clear the header info
///
/// @detailed
///
/// @param  iunit - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
clear_header()
{

	using namespace files_paths;
	header_info.clear();

	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin append_header_line
///
/// @brief
///   Take in a string, append this to the PDB header info
///
/// @detailed
///
/// @param  hline - [in] - string to append
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
append_header_line( std::string hline )
{

	using namespace files_paths;

	header_info.push_back( hline );

	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin read_pdb_header
///
/// @brief
///     read the header lines of a PDB
///
/// @detailed
///
/// @param  iunit - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified November 2005 John Karanicolas
/////////////////////////////////////////////////////////////////////////////////
void
read_pdb_header( utility::io::irstream & iunit )
{

	std::string line;
	std::string pdb_card;
	std::istringstream line_stream;
	bool done_reading = false;

	iunit.seek_beg(); // reposition to top of pdb file
	clear_header();

	while ( ! done_reading ) {

		iunit.getline( line );
		if ( iunit.eof() ) return;
		line_stream.clear();
		line_stream.str( line );
		line_stream.seekg( std::ios_base::beg );

		if ( line_stream.eof() ) {
			iunit.clear();
			done_reading=true;

		} else if ( iunit.fail() ) {
			iunit.clear();
			iunit >> skip;

		} else {
			line_stream >> bite( 6, pdb_card ) >> skip;
			if ( pdb_card == "ATOM  " ) {
				// jk stop upon finding the first atom
				done_reading=true;
			} else {
				// jk store the header
				append_header_line( line );
			}

		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin check_missing_residue
///
/// @brief
/// check if all backbone atoms in  residue are available
///
/// @detailed
/// returns is_missing true if any backbone atom in residue is missing
/// increments counters of missing residues and missing atoms
///
/// @param  missingatoms - [in/out]? -
/// @param  ires - [in/out]? -
/// @param  nmissing - [in/out]? -
/// @param  nmissing_res - [in/out]? -
/// @param  is_missing - [in/out]? - does residue have missing atoms
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
check_missing_residue(
	FArray1D_bool const & missing_atom,
	int const ires,
	int & nmissing,
	int & nmissing_res,
	bool & is_missing, // does residue have missing atoms
	int const res_num,
	char const chain_id,
	std::string const & res_name
)
{
	using namespace param;
	using namespace param_aa;
	using namespace runlevel_ns;

	assert( ( ires > 0 ) );

	is_missing = false;

//jjh escape nucleic acids
	int res_type;
	num_from_name( res_name, res_type );
	if (res_type > 0 && is_NA(res_type) ) return;

	for ( int j = 1; j <= 4; ++j ) {
		if ( missing_atom(j) ) {
			if ( runlevel > inform ) {
				// if you want more info here, you must pass into this routine
				// DO NOT USE VARIABLES IN MISC OR PDB NAMESPACES
				// there is no guarantee that these variables have been defined
				std::cout << "residue " << ires << " (PDB " << chain_id << ' ' <<
				 res_num << ' ' << res_name << ") is missing backbone atoms " <<
				 j << std::endl;
			}
			is_missing = true;
			++nmissing;
		}
	}
	if ( is_missing ) ++nmissing_res;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin read_pdb_phipsi
///
/// @brief
///car read phi,psi,omega, secstruct and name into low arrays
///
/// @detailed
///
/// @param  funit - [in/out]? -
/// @param  fail - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///cj  NOTE :: needs open file & total residue defined!
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
read_pdb_phipsi(
	utility::io::irstream & funit,
	bool & fail
)
{
	using namespace misc;
	using namespace runlevel_ns;

	int i = 0;
	std::string tag( "nothi" );
	std::string tmp;

	funit.clear();
	funit.seek_beg();

//cj    read into a tmp holder to save until get to right spot
L220:
	while ( ( tag != "compl" ) && ( funit ) ) {
		funit >> bite( tmp ) >> skip;
		if ( funit.eof() ) {
			goto L255;
		} else if ( funit.fail() ) {
			funit.clear();
			funit >> skip;
			goto L220;
		}
		if ( funit ) {
			tag = tmp.substr(0,5);
		}
	}

//cj    now get values
	for ( i = 1; i <= total_residue; ++i ) {
		float tempphi, temppsi, tempomega;
		char tempsec;
		std::string tempname;
		funit >> skip( 5 ) >> bite( tempsec ) >> skip( 1 ) >>
		 bite( 9, tempphi ) >> bite( 9, temppsi ) >> bite( 9, tempomega ) >>
		 skip( 1 ) >> bite( 4, tempname ) >> skip;
		if ( funit.eof() ) {
			goto L255;
		} else if ( funit.fail() ) {
			funit.clear();
			funit >> skip;
			goto L999;
		}
		secstruct(i) = tempsec;
		phi(i) = tempphi;
		psi(i) = temppsi;
		omega(i) = tempomega;
		name(i) = tempname;
		if ( name(i) == "    " ) name(i) = "NONE";
		if ( runlevel > inform ) std::cout << SS( i ) << '|' << secstruct(i) << '|' <<
		 SS( phi(i) ) << SS( psi(i) ) << SS( omega(i) ) << SS( residue3(i) ) <<
		 std::endl;
	}

//cj    checks for premature ending
L255:
	if ( tag != "compl" ) {
		std::cout << "WARNING:: end of pdb file reached: " <<
		 "angle, secstruct, & res info not found" << std::endl;
		fail = true;
	} else if ( i-1 != total_residue ) {
		std::cout << "WARNING :: protein length mismatch" << std::endl;
		std::cout << "total residue: " << total_residue <<
		 " read_pdb_phipsi nres: " << i-1 << std::endl;
		fail = true;
	} else {
		fail = false;
	}
	return;

L999:
	fail = true;
	std::cout << "WARNING:: unable to read torsion angles" << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_atom_num
///
/// @brief
///
/// @detailed
///
/// @param  name - [in/out]? - atom name
/// @param  number - [in/out]? - atom number
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_atom_num(
	std::string const & name, // atom name
	int & number // atom number
)
{

//** backbone atoms

	if ( name == " N  " ) {
		number = 1;
	} else if ( name == " CA " ) {
		number = 2;
	} else if ( name == " C  " ) {
		number = 4;
	} else if ( name == " O  " ) {
		number = 5;
	} else if ( name == " CB " ) {
		number = 3;
	} else {
		number = 0;
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin centroid_place_useinputsidechain
///
/// @brief Assigns centroid position based on actual center of mass of
///         side chain centroids rather than Rosetta's guesses (see centroid_place).
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors rhiju
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
centroid_place_useinputsidechain()
{
	using namespace aaproperties_pack;
	using namespace misc;
	using namespace param;
	using numeric::xyzVector_float;

	xyzVector_float coord_Ca, coord_heavy, coord_avg;
	int aa1, aav1, num_heavy, total_heavy_atoms;

	for ( int i = 1; i <= total_residue; ++i ) {
		aa1  = res(i);
		aav1 = res_variant(i);
		num_heavy = nheavyatoms(aa1,aav1);

		coord_Ca =  &Eposition(1, 2, i);

		total_heavy_atoms = num_heavy - 4;
		if (total_heavy_atoms > 0) { //Not glycine
			coord_avg = &Eposition(1, 3, i); //Cbeta
			for ( int atom1 = 6; atom1 <= num_heavy; ++atom1 ) {
				coord_heavy = &full_coord(1, atom1, i);
				coord_avg += coord_heavy;
			}

			coord_avg /= float(total_heavy_atoms);
		}
		else{ //glycine
			coord_avg = coord_Ca;
		}

		centroid(1,i) = coord_avg[0];
		centroid(2,i) = coord_avg[1];
		centroid(3,i) = coord_avg[2];
	}

}

//************
//------------------------------------------------------------------------------
//     centroid_place
//     compute_CB_position
//     dihedral
//     torsion_from_position

//////////////////////////////////////////////////////////////////////////////
/// @begin old_resr_initializer
///
/// @brief
///
/// @detailed
///
/// @param  old_resr - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
old_resr_initializer( FArray2D_float & old_resr )
{
	FArray1Da_float old_resr_1d( old_resr ); // 1D view
	int i = 0;
	old_resr_1d( ++i ) = 0.682; // n
	old_resr_1d( ++i ) = -0.382;
	old_resr_1d( ++i ) = -1.068;
	old_resr_1d( ++i ) = -0.370; // ca
	old_resr_1d( ++i ) = -0.760;
	old_resr_1d( ++i ) = -0.128;
	old_resr_1d( ++i ) = -1.060; // c
	old_resr_1d( ++i ) = 0.466;
	old_resr_1d( ++i ) = 0.445;
}


void
sum_stan_initializer( FArray1D_float & sum_stan )
{
	sum_stan( 1 ) = -0.2493333;
	sum_stan( 2 ) = -0.2253333;
	sum_stan( 3 ) = -0.2503333;
}


void
centroids_initializer( FArray2D_float & centroids )
{
	FArray1Da_float centroids_1d( centroids ); // 1D view
	int i = 0;
	centroids_1d( ++i ) = 0.197;
	centroids_1d( ++i ) = -1.614;
	centroids_1d( ++i ) = 0.980;
	centroids_1d( ++i ) = 0.422;
	centroids_1d( ++i ) = -2.503;
	centroids_1d( ++i ) = 1.191;
	centroids_1d( ++i ) = 0.305;
	centroids_1d( ++i ) = -2.596;
	centroids_1d( ++i ) = 1.271;
	centroids_1d( ++i ) = 0.487;
	centroids_1d( ++i ) = -3.148;
	centroids_1d( ++i ) = 1.697;
	centroids_1d( ++i ) = 0.326;
	centroids_1d( ++i ) = -3.156;
	centroids_1d( ++i ) = 1.382;
	centroids_1d( ++i ) = -0.370;
	centroids_1d( ++i ) = -0.760;
	centroids_1d( ++i ) = -0.128;
	centroids_1d( ++i ) = 0.345;
	centroids_1d( ++i ) = -3.001;
	centroids_1d( ++i ) = 1.323;
	centroids_1d( ++i ) = 0.442;
	centroids_1d( ++i ) = -2.625;
	centroids_1d( ++i ) = 1.346;
	centroids_1d( ++i ) = 0.453;
	centroids_1d( ++i ) = -3.529;
	centroids_1d( ++i ) = 1.875;
	centroids_1d( ++i ) = 0.160;
	centroids_1d( ++i ) = -3.133;
	centroids_1d( ++i ) = 1.293;
	centroids_1d( ++i ) = 0.239;
	centroids_1d( ++i ) = -3.251;
	centroids_1d( ++i ) = 1.631;
	centroids_1d( ++i ) = 0.293;
	centroids_1d( ++i ) = -2.651;
	centroids_1d( ++i ) = 1.200;
	centroids_1d( ++i ) = 1.818;
	centroids_1d( ++i ) = -1.172;
	centroids_1d( ++i ) = -0.008;
	centroids_1d( ++i ) = 0.402;
	centroids_1d( ++i ) = -3.237;
	centroids_1d( ++i ) = 1.608;
	centroids_1d( ++i ) = 0.688;
	centroids_1d( ++i ) = -3.738;
	centroids_1d( ++i ) = 2.239;
	centroids_1d( ++i ) = 0.567;
	centroids_1d( ++i ) = -1.794;
	centroids_1d( ++i ) = 1.351;
	centroids_1d( ++i ) = 0.459;
	centroids_1d( ++i ) = -2.088;
	centroids_1d( ++i ) = 1.230;
	centroids_1d( ++i ) = 0.197;
	centroids_1d( ++i ) = -2.250;
	centroids_1d( ++i ) = 1.291;
	centroids_1d( ++i ) = 0.112;
	centroids_1d( ++i ) = -3.212;
	centroids_1d( ++i ) = 1.718;
	centroids_1d( ++i ) = 0.339;
	centroids_1d( ++i ) = -3.294;
	centroids_1d( ++i ) = 1.489;
//KMa phospho_ser
	if ( add_pser () 	)
	{
	centroids_1d( ++i ) = 0.567;
	centroids_1d( ++i ) = -1.794;
	centroids_1d( ++i ) = 1.351;
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin centroid_place
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
centroid_place()
{
	using namespace centroid_private;
	using namespace misc;
	using namespace param;
	using namespace runlevel_ns;

//------------------------------------------------------------------------------

	static FArray2D_float const old_resr( 3, 3, old_resr_initializer );
	static FArray1D_float const sum_stan( 3, sum_stan_initializer );
//KMa phospho_ser
	static FArray2D_float const centroids( 3, MAX_AA_PLUS(), centroids_initializer );
	FArray1D_float sum( 3 );
	float centroid_rms;

//------------------------------------------------------------------------------
//  TRANSLATE RESIDUE TO THE ORIGIN
	int error = 0;
	for ( int resi = 1; resi <= total_residue; ++resi ) {
		int const type = res(resi);
//cmd do not calculate centroids for non-proteins
		//dr  don't think will effect nonnat aa
//KMa phospho_ser -> MAX_AA_PLUS()
		if ( res(resi) <= MAX_AA_PLUS()    ) {//MAX_AUTH_AA
			for ( int i = 1; i <= 3; ++i ) {
				float const sum_stan_i = sum_stan(i);
				for ( int j = 1; j <= 3; ++j ) {
					new_res(i,j) = old_resr(i,j) - sum_stan_i;
				}
				new_res(i,4) = centroids(i,type) - sum_stan_i;
			}

			for ( int i = 1; i <= 3; ++i ) {
				sum(i) = Eposition(i,1,resi);
				sum(i) += Eposition(i,2,resi);
				sum(i) += Eposition(i,4,resi);
				sum(i) /= 3;
				old_res(i,1) = Eposition(i,1,resi) - sum(i);
				old_res(i,2) = Eposition(i,2,resi) - sum(i);
				old_res(i,3) = Eposition(i,4,resi) - sum(i);
			}

			for ( int i = 1; i <= 3; ++i ) {
				for ( int j = 1; j <= 4; ++j ) {
					if ( Eposition(i,j,resi) != 0.0 ) goto L60;
				}
			}
//car no non-zero coords:
			for ( int i = 1; i <= 3; ++i ) {
				centroid(i,resi) = 0.0;
			}
			goto L70;

L60:

//  SETUP FOR ORIENTATION
			orient_centroid(centroid_rms);
			if ( centroid_rms > 0.2 ) {
				random_rot();
				orient_centroid(centroid_rms);
				if ( centroid_rms > 0.2 ) {
					++error;
					if ( runlevel >= inform ) {
						std::cout << "centroid_place error: " << resi << SS( centroid_rms )
						 << std::endl;
						std::cout << SS( old_res(1,1) ) << SS( old_res(2,1) ) <<
						 SS( old_res(3,1) ) << std::endl;
						std::cout << SS( old_res(1,2) ) << SS( old_res(2,2) ) <<
						 SS( old_res(3,2) ) << std::endl;
						std::cout << SS( old_res(1,3) ) << SS( old_res(2,3) ) <<
						 SS( old_res(3,3) ) << std::endl;
						std::cout << SS( new_res(1,1) ) << SS( new_res(2,1) ) <<
						 SS( new_res(3,1) ) << std::endl;
						std::cout << SS( new_res(1,2) ) << SS( new_res(2,2) ) <<
						 SS( new_res(3,2) ) << std::endl;
						std::cout << SS( new_res(1,3) ) << SS( new_res(2,3) ) <<
						 SS( new_res(3,3) ) << std::endl;
						std::cout << "placing centroid at (0,0,0)" << std::endl;
					}
					centroid(1,resi) = 0.0;
					centroid(2,resi) = 0.0;
					centroid(3,resi) = 0.0;
				}
			} else {
				for ( int i = 1; i <= 3; ++i ) {
					centroid(i,resi) = new_res(i,4) + sum(i);
				}
			}

L70:;
		} else {
			for ( int i = 1; i <= 3; ++i ) {
				centroid(i,resi) = full_coord(i,11,resi); //centroids for DNA
			}
		}
	}

	if ( error > 0 ) std::cout << "WARNING: " << error << " centroids were" <<
	 " placed at 0,0,0" << std::endl;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin orient_centroid
///
/// @brief
///
/// @detailed
///
/// @param  centroid_rms - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
orient_centroid( float & centroid_rms )
{
	using namespace centroid_private;

//car local
	float bb,cc;
	int ict,ix,iy,iz,iflag;
	FArray2D_double aa( 3, 3, 0.0 );
	FArray2D_double rot( 3, 3, 0.0 );
	FArray1D_double t( 3 );
	double sig,sg,gam;

	float const tol = 0.0001;
	for ( int k = 1; k <= 3; ++k ) {
		for ( int i = 1; i <= 3; ++i ) {
			for ( int j = 1; j <= 3; ++j ) {
				aa(i,j) += new_res(i,k)*old_res(j,k);
			}
		}
	}
	for ( int i = 1; i <= 3; ++i ) {
		rot(i,i) = 1.0;
	}
//
//       from here to 70, iterative rotation scheme
//
	ict = 0;
	iflag = 0;
	ix = 1;
	goto L52;
L50:
	++ix;
	if ( ix < 4 ) goto L52;
	if ( iflag == 0 ) goto L70;
	iflag = 0;
	ix = 1;
L52:
	++ict;
	if ( ict > 1000 ) goto L70;
	iy = ix+1;
	if ( iy == 4 ) iy = 1;
	iz = 6-ix-iy;
	sig = aa(iz,iy)-aa(iy,iz);
	gam = aa(iy,iy)+aa(iz,iz);
	sg = std::sqrt(sig*sig+gam*gam);
	if ( sg == 0. ) goto L50;
	sg = 1./sg;
	if ( std::abs(sig) <= tol*std::abs(gam) ) goto L50;
	for ( int k = 1; k <= 3; ++k ) {
		bb = gam*aa(iy,k)+sig*aa(iz,k);
		cc = gam*aa(iz,k)-sig*aa(iy,k);
		aa(iy,k) = bb*sg;
		aa(iz,k) = cc*sg;
		bb = gam*rot(iy,k)+sig*rot(iz,k);
		cc = gam*rot(iz,k)-sig*rot(iy,k);
		rot(iy,k) = bb*sg;
		rot(iz,k) = cc*sg;
	}

	iflag = 1;
	goto L50;
L70:
//
//       rotate new_res
//
	for ( int i = 1; i <= 4; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			t(j) = 0.;
			for ( int k = 1; k <= 3; ++k ) {
				t(j) += rot(j,k) * new_res(k,i);
			}
		}

		for ( int j = 1; j <= 3; ++j ) {
			new_res(j,i) = t(j);
		}
	}

//       rms computation
//
	centroid_rms = 0.0;
	for ( int i = 1; i <= 3; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			float const res_dif = old_res(j,i) - new_res(j,i);
			centroid_rms += res_dif * res_dif;
		}
	}
//
	centroid_rms = std::sqrt(centroid_rms/3.0f);
//
}

//////////////////////////////////////////////////////////////////////////////
/// @begin rot_initializer
///
/// @brief
///
/// @detailed
///
/// @param  rot - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
rot_initializer( FArray2D_double & rot )
{
	FArray1Da_double rot_1d( rot ); // 1D view
	int i = 0;
	rot_1d( ++i ) = -0.404876238953589;
	rot_1d( ++i ) = 0.561890823160855;
	rot_1d( ++i ) = 0.721355522055314;
	rot_1d( ++i ) = 0.400849677789197;
	rot_1d( ++i ) = 0.818131092288274;
	rot_1d( ++i ) = -0.412287490681113;
	rot_1d( ++i ) = -0.821824045219219;
	rot_1d( ++i ) = 0.122229740676607;
	rot_1d( ++i ) = -0.556475625871692;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin random_rot
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
random_rot()
{
	using namespace centroid_private;
//------------------------------------------------------------------------------
	static FArray2D_double const rot( 3, 3, rot_initializer );
	FArray1D_double t( 3 );
//------------------------------------------------------------------------------
	for ( int i = 1; i <= 4; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			t(j) = 0.0;
			for ( int k = 1; k <= 3; ++k ) {
				t(j) += rot(j,k) * new_res(k,i);
			}
		}
		for ( int j = 1; j <= 3; ++j ) {
			new_res(j,i) = t(j);
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin torsion_from_position
///
/// @brief
/// determines phi,psi angles from a protein structure
/// non-existant angles set to 0.0
///
/// @detailed
///
/// @param  nres - [in/out]? - # of residues in the protein, logical
/// @param  xyz - [in/out]? - (xyz,atom#,res#) atom order:N CA CB C O
/// @param  phi - [in/out]? -
/// @param  psi - [in/out]? -
/// @param  omega - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
torsion_from_position(
	int const nres, // # of residues in the protein, logical
	FArray3Da_float xyz, // (xyz,atom#,res#) atom order:N CA CB C O
	FArray1Da_float phi,
	FArray1Da_float psi,
	FArray1Da_float omega
)
{
	using namespace param;

	xyz.dimension( 3, MAX_POS, nres );
	phi.dimension( nres );
	psi.dimension( nres );
	omega.dimension( nres );

	phi = 0.;
	psi = 0.;
	omega = 0.;

	for ( int i = 1; i < nres; ++i ) {
			psi(i) = dihedral(xyz(1,1,i),xyz(1,2,i),xyz(1,4,i),xyz(1,1,i+1));
			omega(i) = dihedral(xyz(1,2,i),xyz(1,4,i),xyz(1,1,i+1),xyz(1,2,i+1));
			phi(i+1) = dihedral(xyz(1,4,i),xyz(1,1,i+1),xyz(1,2,i+1),xyz(1,4,i+1));
	}

	// jk Note: the following phi/psi/omega remain at zero
	//	phi(1) = 0.0;
	//	psi(nres) = 0.0;
	//	omega(nres) = 0.0;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin dihedral
///
/// @brief
///car returns dihedral angle from four points.
///
/// @detailed
///car  Define a coordinate system in which vector v23 lies on the Z axis
///car  vector v21 lies in the xy plane.
///
/// @param  a1 - [in/out]? -
/// @param  a2 - [in/out]? -
/// @param  a3 - [in/out]? -
/// @param  a4 - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
dihedral(
	FArray1Da_float a1,
	FArray1Da_float a2,
	FArray1Da_float a3,
	FArray1Da_float a4
)
{
	using numeric::conversions::degrees;
	using namespace runlevel_ns;
  using namespace::runlevel_ns;

//car functions from bystroff

	a1.dimension( 3 );
	a2.dimension( 3 );
	a3.dimension( 3 );
	a4.dimension( 3 );

	if ( (benchmark || runlevel > standard) &&
			 (a4(1)*a4(1) + a4(2)*a4(2) + a4(3)*a4(3) < 1e-6) ) { // jss debug warning
		std::cout << "Warning: dihedral of ("<<a1(1)<<","<<a1(2)<<","<<a1(3)<<") ("<<a2(1)<<","<<a2(2)<<","<<a2(3)<<")   ("<<
      a3(1)<<","<<a3(2)<<","<<a3(3)<<") ("<<a4(1)<<","<<a4(2)<<","<<a4(3)<<") \n  Since last is the origin, this is probably uninitialized."<<
      std::endl;
	}

	static FArray1D_float v23( 3 );
	static FArray1D_float v21( 3 );
	static FArray1D_float v34( 3 );
	static FArray1D_float vy( 3 );
	static FArray1D_float vx( 3 );
	static FArray1D_float uy( 3 );
	static FArray1D_float ux( 3 );

	subvec(a3,a2,v23);
	subvec(a1,a2,v21);
	cros(v23,v21,vy); // vy is vector along y axis
	unitvec(vy,uy); // uy is unit vector on y axis
	cros(uy,v23,vx); // vx is vector on x axis
	unitvec(vx,ux); // ux is unit vector on y axis
	subvec(a4,a3,v34);
	float const cx = dotprod(v34,ux); // cx: component of v34 on x axis
	float const cy = dotprod(v34,uy); // cy: component of v34 on y axis
	return ( cy == 0.0 && cx == 0.0 ) ? 0.0f : degrees( std::atan2( cy, cx ) );
}

//////////////////////////////////////////////////////////////////////////////
/// @begin CBpos_initializer
///
/// @brief
///
/// @detailed
///
/// @param  CBpos - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
CBpos_initializer( FArray1D_double & CBpos )
{
	CBpos( 1 ) = -0.532605943;
	CBpos( 2 ) = 0.775186438;
	CBpos( 3 ) = 1.19513831;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin compute_CB_position
///
/// @brief
///car The coordinates of three atoms are required to place the CB
///
/// @detailed
///car atom2 = atom to which CB is bound  (CA)
///car atom1 = N-terminally adjacent to atom2, (N)
///car atom3-C-terminally adjacent to atom2, (C)
///
/// @param  position - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
compute_CB_position( FArray2Da_float position )
{
	using namespace param;

	position.dimension( 3, MAX_POS );

//car internal
	FArray2D_float axes( 3, 3 );

//car paramters
	static FArray1D_double const CBpos( 3, CBpos_initializer );
	 // location of CB atom measured from CA with
	// x axis along N-CA bond,  yaxis in plane defined
	// by cA->C bond vector


	for ( int i = 1; i <= 3; ++i ) {
		for ( int j = 1; j <= 5; ++j ) {
			if ( j != 3 ) {
				if ( position(i,j) != 0.0 ) goto L100;
			}
		}
	}
	for ( int i = 1; i <= 3; ++i ) {
		position(i,3) = 0.0;
	}
	return;

L100:

	for ( int i = 1; i <= 3; ++i ) {
		axes(i,1) = position(i,4) - position(i,2);
		axes(i,2) = position(i,2) - position(i,1);
	}
	cros(axes(1,1),axes(1,2),axes(1,3)); // generate rotation matrix
	cros(axes(1,3),axes(1,1),axes(1,2)); // rosetta function cros
	unitvec(axes(1,1),axes(1,1));
	unitvec(axes(1,3),axes(1,3));
	unitvec(axes(1,2),axes(1,2));

	for ( int i = 1; i <= 3; ++i ) { // template coord x rotation matrix + base coord
		position(i,3) = position(i,2) +
		 CBpos(1) * axes(i,1) +
		 CBpos(2) * axes(i,2) +
		 CBpos(3) * axes(i,3);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin clean_pdb
///
/// @brief
///
/// @detailed
///
/// @param  missingatoms - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
clean_pdb( FArray2D_bool const & missingatoms )
{
	using namespace aaproperties_pack;
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace runlevel_ns;
	using namespace dna_variables;
	using namespace files_paths;

//car local
	FArray1D_bool missingres( MAX_RES()() );
	bool include_current, make_output_file;
	std::string pack_mode;

	bool fail( true );

	//yl, Create PackerTask and setup values before pass into pack_rotamers
	//PackerTask Task;
	//yl flags for pose_from_misc
	bool const fullatom( true );
	bool const ideal_pose( false ); // non-ideal backbone geometry
	bool const coords_init( true );

//bk set missingres to true if any heavy atoms were missing from a residue
	for ( int i = 1; i <= total_residue; ++i ) {
		missingres(i) = false;
		no_sidechain(i) = false;
		na_missing_phosphate(i) = false;
	}
	for ( int i = 1; i <= total_residue; ++i ) {
		int const aa ( res(i) );
		int const aav( res_variant(i) );

		for ( int j = 1, je = nheavyatoms( aa, aav ); j <= je; ++j ) {
			if ( missingatoms(j,i) ) {
				missingres(i) = true;
				no_sidechain(i) = true;
				if ( j == 1 && !is_protein(aa) ) {
					na_missing_phosphate(i) = true;
					std::cout << "Missing 5' phosphate for residue " <<
					 pdb::pdb_res_num(i) << std::endl;
				}
				if ( runlevel > inform )
					std::cout << " missing atom " << I( 3, j ) << " on aa "
								 << I( 3, aa ) << " at residue " << I( 3, i ) << std::endl;

				if ( is_NA( res(i) ) ) {
					missingres(i) = false;
					no_sidechain(i) = false;
					bool ta_ok( true );
					for ( int k=1; k<= 3; ++k ) {
						int const ta_atomno( ta(k,j,aa,aav));
						if ( ta_atomno < 1 || ta_atomno > MAX_ATOM()() ) {
							std::cout << "WARNING:: missing DNA atom with bad template: " <<
								i << ' ' << j << ' ' << ta_atomno << std::endl;
							ta_ok = false;
							break;
						}
					}
					if (!ta_ok ) {
						fail = true;
						return fail; // signal failure
					}
				}
			}
		}
	}

	//bk read in rotamer libraries ...
	initialize_fullatom();

	initialize_fullcoord_array( Eposition, full_coord, total_residue, res, res_variant );

	bool save_fullatom = fullatom_flag::full_atom;
	fullatom_flag::full_atom = true;

	if ( ! quiettruefalseoption( "no_clean" ) ) { //SGM Option leaves PDB atoms alone for testing
		std::cout << "Filling missing atoms" << std::endl;

		fill_missingatoms( missingatoms );

		include_current = false;
		pack_mode = "packrot";
		make_output_file = false;
		std::cout << "optimizing missing atoms" << std::endl;


		//yl create local pose and prepare the Epositions for misc
		pose_ns::Pose pose;

	pose_from_misc(pose,fullatom, ideal_pose, coords_init);
		PackerTask Task( pose );
		Task.set_task("packrot", false, missingres, include_current);
		Task.setup_residues_to_vary();

		pack_rotamers( pose, Task );
		pose.copy_to_misc();


		if (no_optH) {
			std::cout << "NOT optimizing hydrogen positions" << std::endl;
		} else if( get_cst_mode_flag() ) {
				enzyme::missingres=missingres;//save it for later
		} else {
			pack_mode = "optimizeH";
			std::cout << "optimizing hydrogen positions" << std::endl;
			pose_from_misc(pose,fullatom, ideal_pose, coords_init);

			PackerTask Task2( pose );
			Task2.set_task("optimizeH", false, missingres, include_current);
			Task2.setup_residues_to_vary();

			pack_rotamers( pose, Task2 );
			pose.copy_to_misc();
		}
	}
	if ( flip_symmetric_sidechains ) {
		for ( int i = 1; i <= total_residue; ++i ) {
			bool const flipped
				( flip_symmetric_coords(res(i),res_variant(i),full_coord(1,1,i) ) );
			if ( flipped  && runlevel_ns::runlevel > runlevel_ns::standard) {
				std::cout<< "Flipped " << i << ' ' << param_aa::aa_name3(res(i))
								 << pdb::pdb_res_num(i) << pdb::res_chain(i) << std::endl;
			}
		}
	}
	fullatom_flag::full_atom = save_fullatom;
	fail = false;
	return fail;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin count_neighbors_wrap
///
/// @brief
///
/// @detailed
///
/// @param  res - [in/out]? -
/// @param  total_residue - [in/out]? -
/// @param  full_coord - [in/out]? -
/// @param  nlist_out - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
count_neighbors_wrap(
	FArray1Da_int res,
	int & total_residue,
	FArray3Da_float full_coord,
	FArray2Da_bool nlist_out
)
{
	using namespace param;
	using namespace template_pack;

	res.dimension( MAX_RES() );
	full_coord.dimension( 3, MAX_ATOM(), MAX_RES() );
	nlist_out.dimension( MAX_RES(), MAX_RES() );

	make_neighbor_info(res,total_residue,full_coord,neighborlist,neighbors);
	for ( int i = 1; i <= total_residue; ++i ) {
		for ( int j = 1; j <= total_residue; ++j ) {
			nlist_out(i,j) = neighborlist(i,j);
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin fill_missingatoms
///
/// @brief
///bk fills in missing heavy atoms and hydrogens to a pdb structure
///
/// @detailed
///
/// @param  missingatoms - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
fill_missingatoms( FArray2D_bool const & missingatoms )
{
	using namespace aaproperties_pack;
	using namespace misc;
	using namespace param;
	using namespace param_aa;

//car local
	FArray2D_int rotarray( MAX_CHI, MAX_RES()() );
	FArray1D_float lphi( MAX_RES()() );
	FArray1D_float lpsi( MAX_RES()() );
	FArray2D_float chiarray( MAX_CHI, MAX_RES()() );
	FArray2D_float new_sccoord( 3, MAX_ATOM()() );
	FArray2D_bool missingchi( MAX_CHI, MAX_RES()() );

//bk determine which chi angles are not determined in the pdb structure
	get_missingchi(missingatoms,total_residue,missingchi,res,res_variant);

//bk build chi and rot arrays
	get_chi_and_rot_from_coords(total_residue,res,res_variant,full_coord,chiarray,rotarray);
	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		int aa = res(seqpos);
		int aav = res_variant(seqpos);
		for ( int ch = 1, ch_end = nchi(aa,aav); ch <= ch_end; ++ch ) {
			if ( missingchi(ch,seqpos) ) {
				chiarray(ch,seqpos) = 180.0;
			}
		}
		for ( int atm = 1, atme = natoms(aa,aav); atm <= atme; ++atm ) {
			if ( missingatoms( atm, seqpos ) ) {
//bk if heavy atom missing, rebuild entire sidechain
				if ( atm <= nheavyatoms(aa,aav) && ( is_protein(aa) || is_nonnatural(aa) ) ) {
					get_rot_coord( full_coord, aa, aav, chiarray(1,seqpos), new_sccoord, seqpos );
					for ( int i = 1; i <= 3; ++i ) {
						for ( int ii = 5; ii <= atme; ++ii ) {
							full_coord(i,ii,seqpos) = new_sccoord(i,ii);
						}
					}
					goto L50;

				} else {         // a hydrogen or nucleic acid
					if ( HNpos(aa,aav) != atm ) {   // already placed HN
						place_atom( aa, aav, atm, full_coord(1,1,seqpos) );
						if ( is_NA(res(seqpos)) && atom_name(atm,aa,aav)[1]!='H'){
							std::cout << "Missing " << atom_name(atm,aa,aav) <<
							 " atom was added to DNA " << pdb::pdb_res_num(seqpos) <<
							 std::endl;
						}
					}
				}   // heavy or hydrogen
			}    // missingatom
		}  // atm
L50:;
	}   // seqpos
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_missingchi
///
/// @brief
///bk what chi angles are not determined by the existing chi angles
///
/// @detailed
///
/// @param  missingatoms - [in/out]? -
/// @param  total_residue - [in/out]? -
/// @param  missingchi - [in/out]? -
/// @param  res - [in/out]? -
/// @param  res_variant - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_missingchi(
	FArray2D_bool const & missingatoms,
	int const total_residue,
	FArray2Da_bool missingchi,
	FArray1Da_int res,
	FArray1Da_int res_variant
)
{
	using namespace aaproperties_pack;
	using namespace param;

	missingchi.dimension( MAX_CHI, MAX_RES() );
	res.dimension( MAX_RES() );
	res_variant.dimension( MAX_RES() );

	missingchi = false;

	for ( int i = 1; i <= total_residue; ++i ) {
		int aa = res(i);
		int aav = res_variant(i);
		for ( int ch = 1, ch_end = nchi(aa,aav); ch <= ch_end; ++ch ) {
			for ( int j = 1; j <= 4; ++j ) {
				int atm = chi_atoms(j,ch,aa,aav);
				if ( missingatoms(atm,i) ) {
					missingchi(ch,i) = true;
				}
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin input_compressed
///
/// @brief
///
/// @detailed
///
/// @param  fail - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
input_compressed( bool & fail, bool & coord_fail )
{
	using namespace docking;
	using namespace files_paths;
	using namespace misc;
	using namespace param;

	if ( new_silent_reader ) {
		input_compressed_new( fail, coord_fail );
		return;
	}

	coord_fail = true;

//car local
	char temp_ss;
	std::string line;
	std::string sequence;

	std::string structure_label;
	std::string desired_label;
	std::string last_structure_label;

	int nres_start;
	int structure_index;
	int residue_index;
	int desired_index = 0;

	float CA_x, CA_y, CA_z;
	float temp_phi, temp_psi, temp_omega;

	fail = true;

	//BW Initialize the arrays
	phi = 0.0;
	psi = 0.0;
	omega = 0.0;
	secstruct = ' ';
	name = "    ";

	utility::io::izstream pdb_stream;

	std::string filename( open_compressed_start_pdb( pdb_stream ) );

	int nres = input_compressed_get_sequence( pdb_stream, sequence );

	if ( use_fasta && !map_start_sequence ) {
		if ( nres != total_residue ) {
			std::cout << "ERROR:: protein length mismatch" << std::endl;
			std::cout << "total_residue " << total_residue << " silent file nres "
			 << nres << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		//check for sequence correctness
		for ( int i = 1; i <= total_residue; ++i ) {
			if ( sequence[i] != residue1(i) ) {
				std::cout << "ERROR:: sequence mismatch, residue: " << i << std::endl;
				std::cout << "residue1 " << residue1(i) << "silent file "
				 << sequence[i] << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}
	} else {
		total_residue = nres;
		if ( total_residue <= 0 ) {
			std::cerr << "ERROR:: Unable to read sequence from compresssed input file "
			 << filename << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		// For symmetrical pose we need to add extra residues
		int const num_simulated_monomers = docking_pose_symm_subsystem ? docking_pose_symm_subsystem_size : pose_symm_n_monomers;
		int nres (  docking_pose_symmetry ? (monomer_input ? num_simulated_monomers*total_residue + num_simulated_monomers : total_residue + num_simulated_monomers ) : total_residue );
		MAX_RES_assign_res( nres ); // Set MAX_RES

		for ( int i = 1; i <= total_residue; ++i ) {
			residue1(i) = sequence[i];
			res_variant(i) = 1;
		}
		// PB 08/26/05 added these two calls:
		total_domains = 1;
		domain_end(1) = misc::total_residue;
		convert1_num();
		update_sequence();
	}

L11:
	while ( line.substr(0,6) != "SCORE:" ) {   // read until we find first score line
		pdb_stream >> bite( line ) >> skip;
		if ( pdb_stream.eof() ) {
			goto L40;
		} else if ( pdb_stream.fail() ) {
			pdb_stream.clear();
			pdb_stream >> skip;
			goto L11;
		}
	}

//car current line is probably SCORE labels, if so, read one more:
	if ( line.find( "score" ) != std::string::npos ) {
		pdb_stream >> bite( line ) >> skip;
		if ( line.substr(0,6) != "SCORE:" ) goto L70;
	}
	nres = 0;

	if ( by_index ) {
		structure_index = 1; // file at residue 0 (ie score line) of structure 1
		desired_index = 0;
//car convert start_file string to an integer index
		line = start_file;
		while ( line[0] != ' ' ) {
			desired_index = 10 * desired_index + ICHAR(line[0]) - ICHAR('0');
			line = line.substr(1);
		}

//BW Read  until you reach the desired structure
//BW Count structures by the "SCORE:" lines
//car check to ensure that all structures are the expected size
		while ( structure_index < desired_index ) {
L12:	pdb_stream >> bite( line ) >> skip;
			if ( pdb_stream.eof() ) {
				goto L40;
			} else if ( pdb_stream.fail() ) {
				pdb_stream.clear();
				pdb_stream >> skip;
				goto L12;
			}
			++nres;
			if ( line.substr(0,6) == "SCORE:" ) {
				if ( nres != total_residue + 1 ) goto L80;
				++structure_index;
				nres = 0;
			}
			if ( nres > total_residue ) goto L70;
		}
//car read first line of desired structure
		pdb_stream >> bite( 4, residue_index ) >> skip( 1 ) >>
		 bite( temp_ss ) >> skip( 1 ) >>
		 bite( 9, temp_phi ) >> bite( 9, temp_psi ) >> bite( 9, temp_omega ) >>
		 bite( 9, CA_x ) >> bite( 9, CA_y ) >> bite( 9, CA_z ) >> skip( 1 ) >>
		  structure_label >> skip;
		if ( !pdb_stream ) goto L40;
		desired_label = structure_label;
		start_file = desired_label;
	} else {

//car find correct structure by label
		last_structure_label = ' ';
		structure_index = 0; // label of structure 1 not yet identified
		desired_index = 0; // in case read fails and desired_index is not set to structure_index
		desired_label = start_file;
		while ( true ) {
L20:
			pdb_stream >> bite( 4, residue_index ) >> skip( 1 ) >>
			 bite( temp_ss ) >> skip( 1 ) >>
			 bite( 9, temp_phi ) >> bite( 9, temp_psi ) >> bite( 9, temp_omega ) >>
			 bite( 9, CA_x ) >> bite( 9, CA_y ) >> bite( 9, CA_z ) >> skip( 1 ) >>
			 structure_label >> skip;
			if ( pdb_stream.eof() ) {
				goto L50;
			} else if ( pdb_stream.fail() ) {
				pdb_stream.clear();
				pdb_stream >> skip;
				goto L20;
			}
			if ( structure_label != last_structure_label ) {
				++structure_index;
				last_structure_label = structure_label;
			}
			if ( structure_label == desired_label ) goto L30;
		}
L30:
		desired_index = structure_index;
	}

//car read in torsion angles, secstruct of desired structure
	std::cout << "start structure ::  label: " <<
	 desired_label << " index: " << desired_index << std::endl;
	nres_start = residue_index;
	nres = nres_start;
	while ( pdb_stream ) {
		if ( residue_index != nres ) {
			std::cout << "ERROR:: nonsequential numbering in input file!!" << std::endl;
			std::cout << "nres: " << nres << " residue_index: " << residue_index <<
			 std::endl;
			pdb_stream.close();
			pdb_stream.clear();
			return;
		}
		secstruct(residue_index) = temp_ss;
		phi(residue_index) = temp_phi;
		psi(residue_index) = temp_psi;
		omega(residue_index) = temp_omega;
		pdb_stream >> bite( 4, residue_index ) >> skip( 1 ) >>
		 bite( temp_ss ) >> skip( 1 ) >>
		 bite( 9, temp_phi ) >> bite( 9, temp_psi ) >> bite( 9, temp_omega ) >>
		 bite( 9, CA_x ) >> bite( 9, CA_y ) >> bite( 9, CA_z ) >> skip( 1 ) >>
		 structure_label >> skip;
		if ( pdb_stream ) {
			if ( desired_label != structure_label ) goto L60;
			res_variant(residue_index) = 1;
			++nres;
//       std::cout << I( 4, residue_index ) << ' ' << temp_ss << ' ' <<
//        F( 9, 3, temp_phi ) << F( 9, 3, temp_psi ) << F( 9, 3, temp_omega ) <<
//        F( 9, 3, CA_x ) << F( 9, 3, CA_y ) << F( 9, 3, CA_z ) << ' ' <<
//			  structure_label << std::endl;
		}
	}
L60:
	pdb_stream.close();
	pdb_stream.clear();

	if ( nres-nres_start+1 != total_residue ) {
		std::cout << "ERROR: incomplete starting structure" << std::endl;
		return;
	}

	fail = false;
	return;

L40:
	std::cout << "ERROR: Unable to find input structure by index: " <<
	 desired_index << std::endl;
	pdb_stream.close();
	pdb_stream.clear();
	return;
L50:
	std::cout << "ERROR: Unable to find input structure by label: " <<
	 desired_label << std::endl;
L70:
	std::cout << "ERROR:  silent_input file format error " << std::endl;
	std::cout << "        expected \"SCORE: \" line not found" << std::endl;
	std::cout << "file: " << filename << std::endl;
	std::cout << "line: " << line << std::endl;
	pdb_stream.close();
	pdb_stream.clear();
	return;
L80:
	std::cout << "ERROR:  silent_input file format error " << std::endl;
	std::cout << "        \"SCORE: \" line found before structure end" << std::endl;
	std::cout << "file: " << filename << std::endl;
	std::cout << "line: " << line << std::endl;
	pdb_stream.close();
	pdb_stream.clear();
	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin input_compressed_istream
///
/// @brief  Like normal input_compressed but reads from a
///         generic irstring
///
/// @detailed
///
/// @param  fail - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  Mike Tyka
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
input_compressed_istream(utility::io::irstream &pdb_stream, bool & fail )
{
	using namespace docking;
	using namespace files_paths;
	using namespace misc;
	using namespace param;

//car local
	char temp_ss;
	std::string line;
	std::string sequence;

	std::string structure_label;
	std::string desired_label;
	std::string last_structure_label;

	int nres_start;
	int structure_index;
	int residue_index;
	int desired_index = 0;

	float CA_x, CA_y, CA_z;
	float temp_phi, temp_psi, temp_omega;

	fail = true;

	//BW Initialize the arrays
	phi = 0.0;
	psi = 0.0;
	omega = 0.0;
	secstruct = ' ';
	name = "    ";

	int nres = input_compressed_get_sequence( pdb_stream, sequence );

	if ( use_fasta && !map_start_sequence ) {
		if ( nres != total_residue ) {
			std::cout << "ERROR:: protein length mismatch" << std::endl;
			std::cout << "total_residue " << total_residue << " silent file nres "
			 << nres << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		//check for sequence correctness
		for ( int i = 1; i <= total_residue; ++i ) {
			if ( sequence[i] != residue1(i) ) {
				std::cout << "ERROR:: sequence mismatch, residue: " << i << std::endl;
				std::cout << "residue1 " << residue1(i) << "silent file "
				 << sequence[i] << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}
	} else {
		total_residue = nres;
		if ( total_residue <= 0 ) {
			std::cerr << "ERROR:: Unable to read sequence from compresssed input file "  << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		// For symmetrical pose we need to add extra residues
		int const num_simulated_monomers = docking_pose_symm_subsystem ? docking_pose_symm_subsystem_size : pose_symm_n_monomers;
		int nres (  docking_pose_symmetry ? (monomer_input ? num_simulated_monomers*total_residue + num_simulated_monomers : total_residue + num_simulated_monomers ) : total_residue );
		MAX_RES_assign_res( nres ); // Set MAX_RES
		for ( int i = 1; i <= total_residue; ++i ) {
			residue1(i) = sequence[i];
			res_variant(i) = 1;
		}
		// PB 08/26/05 added these two calls:
		total_domains = 1;
		domain_end(1) = misc::total_residue;
		convert1_num();
		update_sequence();
	}

L11:
	while ( line.substr(0,6) != "SCORE:" ) {   // read until we find first score line
		pdb_stream >> bite( line ) >> skip;
		if ( pdb_stream.eof() ) {
			goto L40;
		} else if ( pdb_stream.fail() ) {
			pdb_stream.clear();
			pdb_stream >> skip;
			goto L11;
		}
	}

//car current line is probably SCORE labels, if so, read one more:
	if ( line.find( "score" ) != std::string::npos ) {
		pdb_stream >> bite( line ) >> skip;
		if ( line.substr(0,6) != "SCORE:" ) goto L70;
	}
	nres = 0;

	if ( by_index ) {
		structure_index = 1; // file at residue 0 (ie score line) of structure 1
		desired_index = 0;
//car convert start_file string to an integer index
		line = start_file;
		while ( line[0] != ' ' ) {
			desired_index = 10 * desired_index + ICHAR(line[0]) - ICHAR('0');
			line = line.substr(1);
		}

//BW Read  until you reach the desired structure
//BW Count structures by the "SCORE:" lines
//car check to ensure that all structures are the expected size
		while ( structure_index < desired_index ) {
L12:	pdb_stream >> bite( line ) >> skip;
			if ( pdb_stream.eof() ) {
				goto L40;
			} else if ( pdb_stream.fail() ) {
				pdb_stream.clear();
				pdb_stream >> skip;
				goto L12;
			}
			++nres;
			if ( line.substr(0,6) == "SCORE:" ) {
				if ( nres != total_residue + 1 ) goto L80;
				++structure_index;
				nres = 0;
			}
			if ( nres > total_residue ) goto L70;
		}
//car read first line of desired structure
		pdb_stream >> bite( 4, residue_index ) >> skip( 1 ) >>
		 bite( temp_ss ) >> skip( 1 ) >>
		 bite( 9, temp_phi ) >> bite( 9, temp_psi ) >> bite( 9, temp_omega ) >>
		 bite( 9, CA_x ) >> bite( 9, CA_y ) >> bite( 9, CA_z ) >> skip( 1 ) >>
		  structure_label >> skip;
		if ( !pdb_stream ) goto L40;
		desired_label = structure_label;
		start_file = desired_label;
	} else {

//car find correct structure by label
		last_structure_label = ' ';
		structure_index = 0; // label of structure 1 not yet identified
		desired_index = 0; // in case read fails and desired_index is not set to structure_index
	  desired_label = start_file;
		while ( true ) {
L20:
			pdb_stream >> bite( 4, residue_index ) >> skip( 1 ) >>
			 bite( temp_ss ) >> skip( 1 ) >>
			 bite( 9, temp_phi ) >> bite( 9, temp_psi ) >> bite( 9, temp_omega ) >>
			 bite( 9, CA_x ) >> bite( 9, CA_y ) >> bite( 9, CA_z ) >> skip( 1 ) >>
			 structure_label >> skip;
	    if(start_file == ""){   // if "" is given as parameter "accept" the first structure
			  desired_label = structure_label;
			}
			if ( pdb_stream.eof() ) {
				goto L50;
			} else if ( pdb_stream.fail() ) {
				pdb_stream.clear();
				pdb_stream >> skip;
				goto L20;
			}
			if ( structure_label != last_structure_label ) {
				++structure_index;
				last_structure_label = structure_label;
			}
			if ( structure_label == desired_label ) goto L30;
		}

L30:
		desired_index = structure_index;
	}

//car read in torsion angles, secstruct of desired structure
	std::cout << "start structure ::  label: " <<
	 desired_label << " index: " << desired_index << std::endl;
	nres_start = residue_index;
	nres = nres_start;
	while ( pdb_stream ) {
		if ( residue_index != nres ) {
			std::cout << "ERROR:: nonsequential numbering in input file!!" << std::endl;
			std::cout << "nres: " << nres << " residue_index: " << residue_index <<
			 std::endl;
			pdb_stream.clear();
			return;
		}
		secstruct(residue_index) = temp_ss;
		phi(residue_index) = temp_phi;
		psi(residue_index) = temp_psi;
		omega(residue_index) = temp_omega;
		pdb_stream >> bite( 4, residue_index ) >> skip( 1 ) >>
		 bite( temp_ss ) >> skip( 1 ) >>
		 bite( 9, temp_phi ) >> bite( 9, temp_psi ) >> bite( 9, temp_omega ) >>
		 bite( 9, CA_x ) >> bite( 9, CA_y ) >> bite( 9, CA_z ) >> skip( 1 ) >>
		 structure_label >> skip;
		if ( pdb_stream ) {
			if ( desired_label != structure_label ) goto L60;
			res_variant(residue_index) = 1;
			++nres;
//       std::cout << I( 4, residue_index ) << ' ' << temp_ss << ' ' <<
//        F( 9, 3, temp_phi ) << F( 9, 3, temp_psi ) << F( 9, 3, temp_omega ) <<
//        F( 9, 3, CA_x ) << F( 9, 3, CA_y ) << F( 9, 3, CA_z ) << ' ' <<
//			  structure_label << std::endl;
		}
	}
L60:
	pdb_stream.clear();

	if ( nres-nres_start+1 != total_residue ) {
		std::cout << "ERROR: incomplete starting structure"
		          << nres << "  "
							<< nres_start << "  "
							<< total_residue << std::endl;
		return;
	}

	fail = false;
	return;

L40:
	std::cout << "ERROR: Unable to find input structure by index: " <<
	 desired_index << std::endl;
	pdb_stream.clear();
	return;
L50:
	std::cout << "ERROR: Unable to find input structure by label: " <<
	 desired_label << std::endl;
L70:
	std::cout << "ERROR:  silent_input file format error " << std::endl;
	std::cout << "        expected \"SCORE: \" line not found" << std::endl;
	std::cout << "line: " << line << std::endl;
	pdb_stream.clear();
	return;
L80:
	std::cout << "ERROR:  silent_input file format error " << std::endl;
	std::cout << "        \"SCORE: \" line found before structure end" << std::endl;
	std::cout << "line: " << line << std::endl;
	pdb_stream.clear();
	return;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin input_compressed_new
///
/// @brief
///
/// @detailed
///
/// @param  fail - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
input_compressed_new( bool & fail, bool & coord_fail )
{

	using namespace docking;

	coord_fail = true;

	if ( files_paths::by_index ) {
		// by-index isn't that safe anyway
		std::cout << "SORRY: new reader doesnt support by-index reading." <<
			std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	std::string const filename( files_paths::start_path +
		files_paths::silent_input_file );

	bool const fa_input( files_paths::input_fa );

	// this object stores all the decoys
	// read the file once, on construction:
	static silent_io::Silent_file_data data( filename, fa_input );

	fail = true;

	//BW Initialize the arrays
	misc::phi = 0.0;
	misc::psi = 0.0;
	misc::omega = 0.0;
	misc::secstruct = ' ';
	misc::name = "    ";

	//////////////////////////////////////
	// update / check sequence information
	const std::string & sequence ( data.sequence() ); // 0-indexed !!!
	const int nres ( data.nres() );

	if ( files_paths::use_fasta && !files_paths::map_start_sequence ) {
		if ( nres != misc::total_residue ) {
			std::cout << "ERROR:: protein length/domain mismatch" << std::endl;
			std::cout << "total_residue " << misc::total_residue <<
				" silent file nres " << nres << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		for ( int i = 1; i <= misc::total_residue; ++i ) {
			if ( sequence[i-1] != misc::residue1(i) ){
//			||  misc::res_variant(i) != 1 ) { // no res_variants in silent-files
				std::cout << "ERROR:: sequence mismatch, residue: " << i << std::endl;
				std::cout << "residue1 " << misc::residue1(i) << "silent file " <<
					sequence[i-1] << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}
	} else {
		std::cout << "setting total_residue = " << nres << std::endl;
		misc::total_residue = nres;
		// For symmetrical pose we nned to add extra residues
		int const num_simulated_monomers = docking_pose_symm_subsystem ? docking_pose_symm_subsystem_size : pose_symm_n_monomers;
		int tot_res (  docking_pose_symmetry ? (monomer_input ? num_simulated_monomers*misc::total_residue + num_simulated_monomers : misc::total_residue + num_simulated_monomers ) : misc::total_residue );
		param::MAX_RES_assign_res( tot_res ); // Set MAX_RES
		for ( int i = 1; i <= misc::total_residue; ++i ) {
			misc::residue1(i) = sequence[i-1]; // sequence is 0-indexed std::string
			misc::res_variant(i) = 1;
		}
		//
		misc::total_domains = 1;
		misc::domain_end(1) = misc::total_residue;

		convert1_num();
		update_sequence();
	}

	////////////////////////////////////////
	// now look for our tag and get the data
	std::string const & tag ( files_paths::start_file );
	if ( ! data.has_key( tag ) ) {
		std::cout << "WARNING:: cant find tag in silent-file: " << tag <<
			std::endl;
		fail = true;
		return;
	}

	// get the data
	silent_io::Silent_structure const & decoy
		( data.get_structure ( tag ) );

	if ( fa_input ) {
		// fa_input --> need pose machinery for silent-file
		// reconstruction
		//		bool const save_pose_flag( pose_flag() );
		//		bool const save_fullatom_flag( get_fullatom_flag() );

		setup_termini();
		// turn on pose flag
		//		set_pose_flag( true );
		pose_ns::Pose pose;
		decoy.fill_pose( pose );
		//		std::cout << "pose_to_misc: "<< save_pose_flag << ' ' <<
		//			save_fullatom_flag << std::endl;
		pose_to_misc( pose ); // triggers a refold

		// restore flags
		//		set_fullatom_flag( save_fullatom_flag );
		//		set_pose_flag( save_pose_flag );

		//bq building termini
		build_termini( fa_input );
// fill in missing sidechains, optimize hydrogens,aa_variants
//		pdb::missing_input_atom = true;
//		if ( fa_input ) clean_pdb(pdb::missing_input_atom);
		monte_carlo_accept_best();
		initialize_maps();
		refold(1, misc::total_residue);

		// signal success
		fail = false;
		coord_fail = false;

	} else {
		// centroid only, can copy data from the silent file
		// fill in torsions:
		for ( int i=1; i<= misc::total_residue; ++i ) {
			misc::phi       ( i ) = decoy.phi       ( i );
			misc::psi       ( i ) = decoy.psi       ( i );
			misc::omega     ( i ) = decoy.omega     ( i );
			misc::secstruct ( i ) = decoy.secstruct ( i );
		}

		// success!
		fail = false;
	}


	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin get_size_from_first_pdb
///
/// @brief
///
///
/// @detailed
///
/// @param  require_start - [in/out]? -
/// @param  silent_input - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Carol Rohl
///
/// @last_modified 2005/03/25
/////////////////////////////////////////////////////////////////////////////////
void
get_size_from_first_pdb(
	bool const require_start,
	bool const silent_input
)
{
	using namespace docking;
	using namespace misc;
	using namespace param;

	if ( ! require_start ) {
		std::cerr << "ERROR:: Unable to obtain total_residue & sequence.\n";
		std::cerr << "        start pdb file must be provided.\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	utility::io::izstream pdb_stream;

	if ( silent_input ) {
		open_compressed_start_pdb( pdb_stream );
		std::string sequence;   //discarded here
		total_residue = input_compressed_get_sequence( pdb_stream, sequence );
	} else {
		open_start_pdb( pdb_stream );
		total_residue = input_pdb_get_size( pdb_stream );
	}
	pdb_stream.close();
	pdb_stream.clear();

	if ( total_residue <= 0 ) {
		std::cerr << "ERROR:: Unable to determine sequence length from starting structure coordinate file\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	// For symmetrical pose we need to add extra residues
	int nres (  docking_pose_symmetry ? total_residue + pose_symm_n_monomers : total_residue );
	MAX_RES_assign_res( nres ); // Set MAX_RES

}


////////////////////////////////////////////////////////////////////////////////
/// @begin open_start_pdb
///
/// @brief
///
///
/// @detailed
///
/// @param  pdb_stream - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Carol Rohl, Stuart Mentzer
///
/// @last_modified 2005/03/14
/////////////////////////////////////////////////////////////////////////////////
std::string
open_start_pdb( utility::io::izstream & pdb_stream )
{
	using namespace files_paths;

	std::string const pdb_file( start_path + start_file + ".pdb" );

	pdb_stream.clear();
	pdb_stream.open( pdb_file );
	if ( !pdb_stream ) {
		std::cout << "ERROR:: Can't find starting structure: " << pdb_stream.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	return pdb_file;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin open_start_pdb
///
/// @brief
///  	opens a pdb for readin
///
/// @detailed
///  	overloaded version of open_start_pdb, with an input argument to
///		define the input file instead of the default 'start_file'
///
/// @param  pdb_stream - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified March 31 2006
/////////////////////////////////////////////////////////////////////////////////
std::string
open_start_pdb(
				utility::io::izstream & pdb_stream,
				std::string filename
				)
{
	using namespace files_paths;

	std::string const pdb_file( start_path + filename + ".pdb" );

	pdb_stream.clear();
	pdb_stream.open( pdb_file );
	if ( !pdb_stream ) {
		std::cout << "ERROR:: Can't find starting structure: " << pdb_stream.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	return pdb_file;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin open_compressed_start_pdb
///
/// @brief
///
///
/// @detailed
///
/// @param  pdb_stream - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Carol Rohl, Stuart Mentzer
///
/// @last_modified 2005/03/14
/////////////////////////////////////////////////////////////////////////////////
std::string
open_compressed_start_pdb( utility::io::izstream & pdb_stream )
{
	using namespace files_paths;

	std::string const pdb_file( start_path + silent_input_file );

	std::cout << "Attempting to open silent_input file: " << pdb_file << std::endl;

	pdb_stream.clear();
	pdb_stream.open( pdb_file );
	if ( !pdb_stream ) {
		std::cout << "ERROR:: Unable to open silent_input file: " << pdb_stream.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	return pdb_file;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin input_compressed_get_sequence
///
/// @brief
/// Reads and returns the sequence string from the pdb file
/// The index 0 position of sequence is a dummy for 1-based indexing
/// Returns the sequence length or 0 if "SEQUENCE:" not present
///
/// @detailed
///
/// @param  pdb_stream - [in/out]? -
/// @param  sequence - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Carol Rohl
///
/// @last_modified 2005/03/14
/////////////////////////////////////////////////////////////////////////////////
int
input_compressed_get_sequence(
	utility::io::irstream & pdb_stream,
	std::string & sequence
)
{
	sequence = ' '; // Dummy sequence[0] for 1-based indexing on returned sequence
	std::string line;

	pdb_stream >> bite( line ) >> skip; // Read the whole line

	if ( line.substr(0,9) == "SEQUENCE:" ) {
		int const line_length = line.length();
		int i = 9; // Start after "SEQUENCE: " prefix
		while ( ( ++i < line_length ) && ( line[i] != ' ' ) ) {
			sequence.push_back( line[i] );
		}
	}
	return sequence.length() - 1;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin input_pdb_get_size
///
/// @brief
///
///
/// @detailed
///
/// @param  pdb_stream - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Carol Rohl
///
/// @last_modified 2005/03/14
/////////////////////////////////////////////////////////////////////////////////
int
input_pdb_get_size( utility::io::irstream & pdb_stream )
{
	using namespace files_paths;
	using namespace misc;
	using namespace param;
	using namespace param_aa;

//car local
	char ichain; // current chain being read
	char last_insert_let;
	int last_res_num; // detecting new residue

//car input fields from pdbfile
	std::string pdb_card;
	int atom_num; // atom number
	std::string atom_name;
	char res_conf;
	std::string pdb_resname, prev_pdb_resname( 3, ' ' );
	char chain_id, prev_chain_id = ' ';
	int res_num, prev_res_num = 0; // residue number
	char insert_let;
	float x,y,z; // coordinates
	float bfactor; // bfactor
	float occ; // occupancy


// initialize
	int ires = 0;
	ichain = protein_chain;
	if ( ichain == '_' ) {
		ichain = ' ';
	}

	last_res_num = -999;
	last_insert_let = 'Z';
	FArray1D_bool missing_atom( 4, true );
	bool is_missing = false;
	int nmissing_res = 0;
	int nmissing = 0;

//car-----------------------
//car find start of pdb file:
//car-----------------------

L40:
	pdb_stream >> bite( 6, pdb_card ) >> bite( 5, atom_num ) >> skip( 1 ) >>
	 bite( 4, atom_name ) >> bite( res_conf ) >> bite( 3, pdb_resname ) >>
	 skip( 1 ) >> bite( chain_id ) >> bite( 4, res_num ) >>
	 bite( insert_let ) >> skip( 3 ) >>
	 bite( 8, x ) >> bite( 8, y ) >> bite( 8, z ) >>
	 bite( 6, occ ) >> bite( 6, bfactor ) >> skip;
	if ( pdb_stream.eof() ) {
		pdb_stream.clear();
		if( get_read_ligaa_flag() ) {
			return 0;
		} else {
			goto L50;
		}
	} else if ( pdb_stream.fail() ) {
		pdb_stream.clear();
		pdb_stream >> skip;
		goto L40;
	}
	if ( pdb_card != "ATOM  " ) goto L40;

//car if the chain isn't defined yet, define as the first chain we find
	if ( ichain == '-' ) {
		ichain = chain_id;
		protein_chain = ichain;
		if ( protein_chain == ' ' ) protein_chain = '_';
	}
//car decide if its the correct chain:
	if ( !read_all_chains && ichain != chain_id ) goto L40;


//car-----------------------
//car read the coordinates:
//car-----------------------
	while ( true ) {

//car check termination conditions:
//car check for TER to signal new domain
		if ( pdb_card.substr(0,3) == "TER" ) {
			if ( !read_all_chains && !multi_chain ) goto L500;
		}
		//car different chain
		if ( !read_all_chains && chain_id != ichain ) goto L500;

//bk skip alternate configurations
		if ( res_conf == 'B' || res_conf == 'C' || res_conf == 'D' ) goto L300;

//car read ATOM records:
		if ( pdb_card == "ATOM  " ) {

//car detect new residue:
			if ( last_res_num != res_num || last_insert_let != insert_let ) {

				last_res_num = res_num;
				last_insert_let = insert_let;

// before starting new residue, check if previous residue is complete:
				if ( ires > 0 ) {
					check_missing_residue( missing_atom, ires, nmissing, nmissing_res,
					 is_missing, prev_res_num, prev_chain_id, prev_pdb_resname );
				}
				if ( !is_missing || !skip_missing ) {
					prev_res_num = res_num;
					prev_chain_id = chain_id;
					prev_pdb_resname = pdb_resname;
					++ires;
				}
				missing_atom = true;

			} // new residue

			// look for missing coordinates
			if ( atom_name == " N  " ) {
				if ( occ > 0.0 ) missing_atom(1) = false;
			} else if ( atom_name == " CA " ) {
				if ( occ > 0.0 ) missing_atom(2) = false;
			} else if ( atom_name == " C  " ) {
				if ( occ > 0.0 ) missing_atom(3) = false;
			} else if ( ( atom_name == " O  " ) || ( atom_name == " OT1" ) ) {
				if ( occ > 0.0 ) missing_atom(4) = false;
			}

		} // pdbcard == "ATOM  "

L300:;
		// Read the next line
		pdb_stream >> bite( 6, pdb_card ) >> bite( 5, atom_num ) >> skip( 1 ) >>
		 bite( 4, atom_name ) >> bite( res_conf ) >> bite( 3, pdb_resname ) >>
		 skip( 1 ) >> bite( chain_id ) >> bite( 4, res_num ) >>
		 bite( insert_let ) >> skip( 3 ) >>
		 bite( 8, x ) >> bite( 8, y ) >> bite( 8, z ) >>
		 bite( 6, occ ) >> bite( 6, bfactor ) >> skip;
		if ( pdb_stream.eof() ) {
			pdb_stream.clear();
			goto L500;
		} else if ( pdb_stream.fail() ) {
			pdb_stream.clear();
			pdb_stream >> skip;
			goto L300;
		}
	}

L500:; // exit do while loop

//car check last residue for missing atoms
	if ( ires > 0 ) {
		check_missing_residue( missing_atom, ires, nmissing, nmissing_res,
		 is_missing, prev_res_num, prev_chain_id, prev_pdb_resname );
		if ( is_missing && skip_missing ) --ires;
	}

	return ires;

L50:
	std::cout << "can't find starting residue in pdb file" << std::endl;
	return ires;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin save_missing_atoms
///
/// @brief
///
///
/// @detailed
///
/// @param  missing_atom - [in/out]? -
/// @param  ires - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Carol Rohl
///
/// @last_modified 2005/03/14
/////////////////////////////////////////////////////////////////////////////////
void
save_missing_atoms(
	FArray1D_bool const & missing_atom,
	int const ires
)
{
	using namespace param;
	using namespace pdb;

	assert( ( ires > 0 ) );

	for ( int i = 1, max_atom = MAX_ATOM(); i <= max_atom; ++i ) {
		missing_input_atom(i,ires) = missing_atom(i);
	}
}
