// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.12 $
//  $Date: 2005/08/25 21:39:29 $
//  $Author: johnk $

#ifndef INCLUDED_job_distributor
#define INCLUDED_job_distributor


// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>

// C++ Headers
#include <ctime>

// forward declaration of the base class
class job_distributor;
namespace main_job_distributor {
	extern job_distributor * jd;
}



// jk Class definition for job_distributor
// jk Note: the BOINC_job_distributor and MPI_generic_job_distributor (below) inherit from this
class job_distributor {

protected:

	std::string mode;

	int nstartnum; // number of input structures
	int number_of_output; // number of output structures per input

	FArray1D_string startnm; // starting pdb files
	FArray1D_string outnm; // output pdb files
	FArray1D_char startch; // starting chain id

	int curr_startnum; // current input structure to process
	int curr_outnum; // current output structure to process

  int num_decoys;
	int attempted_decoys;
	int starting_pdbs_skipped;

	// for Pose Monte_carlo checkpointing
	int mc_checkpoint_last_count;
	int mc_checkpoint_current_count;
	std::string mc_checkpoint_file;
	time_t mc_checkpoint_time;

	bool reinitialize;

	// Return the job_distributor type
	virtual std::string type();

	// Setup
	virtual void initialize_rosetta();

	// starting coors could not be obtained
	virtual void skip_starting_pdb();

	// If more jobs remain, set start_inx and output_inx to
	// values corresponding to the next job and return true.
	// If no jobs remain, return false.
	virtual bool get_next_job_num();

	// Essentially a stub, required by a derived class
	virtual void call_abrelax();
	virtual void call_abrelax_vf();

	// Set indices so that no more jobs will be run with this start_num
	virtual void prepare_for_next_startnum();

	// Called when an output decoy has been made
	virtual void made_output_decoy();

	// Called when an "attempt" at an output decoy has been completed
	// (ie. the output decoy filters have not yet been applied)
	virtual void done_attempt();

	// A stub, which exists to be used by derived classes
	virtual void report_progress();

	// Remove dynamically allocated memory in namespaces
	virtual void memory_cleanup();

	// Write final sets of files
	virtual void finish_writing_files();

	// Write the final summary to stdout/stderr
	virtual void finish_summary( std::ostream & iunit );

public:

	inline job_distributor() { return; };

	virtual ~job_distributor() { return; };

	// Setup
	virtual void initialize();

	// Used when mode == pose1
	virtual bool check_pose1();

	// Used when mode == pose1
	virtual void run_main_pose1();

	// Used for csa mode
	virtual bool check_csa();

	// Used for csa mode
	virtual void run_csa();

	// Setup for the next job
	virtual bool setup_job();

	// Run the next job
	virtual void run_job();

	// End Rosetta (cleanly)
	virtual void finish();

	// Dump contents of member data, eg. for debugging
	virtual void dump_status( std::ostream & iunit );

	// Loop over a function (other classes derive from this)
	virtual void loop_over_function(
		void(*function)(const void * data_class, int const i, int const j, int const k),
		void * data_class,
		int const imax=1,
		int const jmax=1,
		int const kmax=1
	);

	int get_curr_outnum() const;

	// for Pose Monte_carlo checkpointing
	int get_mc_checkpoint_last_count();

	void set_mc_checkpoint_last_count(int last_cnt);

	int get_mc_checkpoint_current_count();

	void iterate_mc_checkpoint_current_count();

	time_t get_mc_checkpoint_time() const;

	void update_mc_checkpoint_time();

	std::string get_mc_checkpoint_file() const;

	bool skip_to_mc_checkpoint();

	void reset_mc_checkpoint();

};




#ifdef BOINC

class BOINC_job_distributor : public job_distributor {

private:

	int farlx_stage;

	// Reset namespace vars as appropriate for BOINC
	void set_BOINC_defaults();

	void read_BOINC_params();

	void call_abrelax();

	void call_abrelax_vf();

	// Called when an output decoy has been made (does BOINC checkpointing)
	void made_output_decoy();

	void report_progress();

	bool get_next_job_num();

	bool checkNoProgressInitCount( int& attempts, int& farlxstage );

public:

	// Empty constructor
	inline BOINC_job_distributor() { return; };

	~BOINC_job_distributor() { return; };

	// Return the job_distributor type
	std::string type();

	void initialize();

	// Used when mode == pose1
	void run_main_pose1();

	// End Rosetta (cleanly)
	void finish();

	// Write the final summary to stdout/stderr
	void finish_summary( std::ostream & iunit );

};

#endif // BOINC





#ifdef USEMPI

class MPI_generic_job_distributor : public job_distributor {

protected:

	int cpu_num;
	bool is_server;
	int jd_i, jd_j, jd_k; // loop counters for job distribution
	int jd_imax, jd_jmax, jd_kmax; // termination condition for task distribution

	virtual int get_server_seed();

	// Ask the server for a new job (if jobs remain)
	virtual bool get_next_task();

	// Run the server, which distributes jobs
	virtual void run_server();

	virtual bool get_next_job_num();

public:

	// Empty constructor
	inline MPI_generic_job_distributor() { return; };

	~MPI_generic_job_distributor() { return; };

	// Return the job_distributor type
	virtual std::string type();

	// Setup, in a manner dependent on whether this is the "server" or a "client" node
	virtual void initialize();

	// Reset namespace vars as appropriate for MPI
	virtual void set_MPI_defaults();

	// Sync random seed across all nodes
	virtual void sync_random_seeds();

	// End Rosetta (cleanly)
	virtual void finish();

	// Loop over a function, using MPI
	virtual void loop_over_function(
		void(*function)(const void * data_class, int const i, int const j, int const k),
		void * data_class,
		int const imax=1,
		int const jmax=1,
		int const kmax=1
	);

};


// jk Note: MPI_full_job_distributor is a special case of MPI_generic_job_distributor,
// in which the distributed loop is the one over startnum + outnum. Other loops will
// not be distributed

class MPI_full_job_distributor : public MPI_generic_job_distributor {

private:

	// Ask the server for a new job (if jobs remain)
	bool get_next_task();

	// For the server, get the next start_inx and output_inx
	bool get_next_job_num();

	// Let the server know that no more jobs should be sent with this startnum
	void prepare_for_next_startnum();

public:

	// Empty constructor
	inline MPI_full_job_distributor() { return; };

	~MPI_full_job_distributor() { return; };

	// Return the job_distributor type
	std::string type();

	// Setup, in a manner dependent on whether this is the "server" or a "client" node
	void initialize();

	// Loop over a function, but don't use MPI
	void loop_over_function(
		void(*function)(const void * data_class, int const i, int const j, int const k),
		void * data_class,
		int const imax=1,
		int const jmax=1,
		int const kmax=1
	);

};

#endif // MPI


#endif

