// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 14080 $
//  $Date: 2007-04-09 23:55:07 +0300 (Mon, 09 Apr 2007) $
//  $Author: yab $


// Rosetta Headers
#include "jump_classes.h"
#include "angles.h"
#include "FArray_xyz_functions.h"
#include "jumping_util.h"
#include "param.h"
#include "pose.h"
#include "random_numbers.h"
#include "util_vector.h" // Ddotprod, etc

// ObjexxFCL Headers
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/FArray1D.io.hh>
#include <ObjexxFCL/FArray2D.io.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/conversions.hh>
#include <numeric/trig.functions.hh>
#include <numeric/xyzVector.hh>
#include <numeric/xyzMatrix.hh>
#include <numeric/xyz.functions.hh>
#include <numeric/xyzVector.io.hh>
#include <numeric/xyzMatrix.io.hh>

// C++ Headers
#include <cstdlib>
#include <iostream>
#include <string>


namespace pose_ns {

	///////////////////////////// JUMPS /////////////////////////////
	// constructor, takes in two single residue Epos arrays
	RT::RT( FArray2Da_float Epos1, FArray2Da_float Epos2):
		rotation(0.0),
		translation(0.0)
	{
		Epos1.dimension(3,param::MAX_POS);
		Epos2.dimension(3,param::MAX_POS);

		bool const local_debug ( false );

		numeric::xyzMatrix_double p1, p2, m1, m2;

		// get coordinate systems from both residues
		get_ncac(Epos1,p1);
		get_ncac(Epos2,p2);
		get_coordinate_system(p1,m1);
		get_coordinate_system(p2,m2);

		// consider this:       |xx xy xz|
		// coordinate frame M = |yx yy yz|
		//                      |zx zy zz|
		// each column is a unit vector written in genuine frame.
		//
		// vector A in frame M can be rewritten as B in genuine frame
		// by the formula B = M x A, thus A = M^T x B
		// a simple example of this would be: the unit vector (1,0,0) in frame M
		// is actually (xx,yx,zx) in genuine frame. mathematically,
		// |xx|   |xx xy xz|   |1|
		// |yx| = |yx yy yz| x |0| ==> B = M x A
		// |zx|   |zx zy zz|   |0|
		//
		// the above formula has another layer of meaning: rotation
		// keeping the genuine frame fixed, a vector can be rotated by applying
		// matrix M onto it, e.g., (1,0,0) rotated to (xx,yx,zx)

		numeric::xyzVector_double e1( &Epos1(1,2) );
		numeric::xyzVector_double e2( &Epos2(1,2) );

		// ( e2 - e1 ) is the vector in genuine frame,
		// translation is the vector in m1 frame. so m1^T is multiplied.
		translation = m1.transposed() * ( e2 - e1 );

		// let's look at the rotation matrix
		// A, B, C are three vectors in genuine frame and are related by rotation
		// B = M1 x A; C = M2 x A;
		// ==> A = M1^T x B = M2^T x C
		// ==> C = M2 x M1^T x B
		// but note that C and B are both in genuine frame and we want a rotation
		// matrix to be applied onto a vector in M1 frame, so comes another step of
		// conversion -- left-multiply M1^T on both sides:
		// M1^T x C = M1^T x M2 x M1^T x B
		// C' = M1^T x M2 x B', as C' and B' are written in M1 frame.
		// so we get the rotation matrix as M1^T x M2.
		// but wait a minute, what Phil orginally got below is M2^T x M1 and it is
		// impossible for that to be wrong, then what happens?

		// It turns out when this rotation matrix is further applied to a vector,
		// it uses Charlies' (col,row) convention (see Dvect_multiply()
		// in RT::make_jump) which means there is one more transpose to do.
		// Now an agreement is reached:
		//  (M2^T x M1)^T = M1^T x (M2^T)^T = M1^T x M2
		// since xyzMatrix uses the normal (row,col) convention, we will switch to
		// rotation = M1^T x M2

		rotation = m1.transposed() * m2;
		/************************Phil's legacy code *********************/
 		// rotation(j,*) is the j-th unit vector of 2nd coord sys written in 1st coord-sys
	// 	for ( int i=1; i<=3; ++i ) {
// 			for ( int j=1; j<=3; ++j ) {
// 				// DANGER: not sure about the order... ////////////////////////
// 				// should sync with make_jump
//  				rotation(j,i) = Ddotprod( m1(1,i), m2(1,j) ); // 2nd guess
//  				//rotation(j,i) = Ddotprod( m1(1,j), m2(1,i) ); // 1st guess
// 			}
// 		}
		/************************Phil's legacy code ********************/

		if ( local_debug ) {
			std::cout << " p1:\n" << p1 << std::endl;
			std::cout << " p2:\n" << p2 << std::endl;
			std::cout << " m1:\n" << m2 << std::endl;
			std::cout << " m2:\n" << m2 << std::endl;
			std::cout << " translation:\n" << translation << std::endl;
			std::cout << " rotation:\n" << rotation << std::endl;
			// debug building: ////////////////
			FArray2D_float tmp_pos1(3,param::MAX_POS), tmp_pos2(3,param::MAX_POS);

			for ( int i=1; i<= param::MAX_POS; ++i ) {
				for ( int j=1; j<= 3; ++j ) {
					tmp_pos1(j,i) = Epos1(j,i);
					tmp_pos2(j,i) = Epos2(j,i);
				}
			}

			double forward_dev(0), forward_dev2(0), backward_dev(0);

			// first: make the jump forward
			make_jump( tmp_pos1, tmp_pos2 );
			std::cout << "tmp_pos1:\n" << tmp_pos1 << std::endl;
			std::cout << "tmp_pos2:\n" << tmp_pos2 << std::endl;
			for ( int i=1; i<= param::MAX_POS; ++i ) {
				if ( i==1 || i==2 || i == 4 ) {
					for ( int j=1; j<= 3; ++j ) {
						forward_dev += std::abs( Epos2(j,i) - tmp_pos2(j,i) );
					}
				}
			}

			// now make backward:
			reverse();
			make_jump( tmp_pos2, tmp_pos1 );
			std::cout << "tmp_pos1:\n" << tmp_pos1 << std::endl;
			std::cout << "tmp_pos2:\n" << tmp_pos2 << std::endl;
			for ( int i=1; i<= param::MAX_POS; ++i ) {
				if ( i==1 || i==2 || i == 4 ) {
					for ( int j=1; j<= 3; ++j ) {
						backward_dev += std::abs( Epos1(j,i) - tmp_pos1(j,i) );
					}
				}
			}

			//restore original direction
			reverse();

			// final sanity check:
			make_jump( tmp_pos1, tmp_pos2 );
			std::cout << "tmp_pos1:\n" << tmp_pos1 << std::endl;
			std::cout << "tmp_pos2:\n" << tmp_pos2 << std::endl;
			for ( int i=1; i<= param::MAX_POS; ++i ) {
				if ( i==1 || i==2 || i == 4 ) {
					for ( int j=1; j<= 3; ++j ) {
						forward_dev2 += std::abs( Epos2(j,i) - tmp_pos2(j,i) );
					}
				}
			}

			std::cout << "RT::RT: debug make_jump:"
								<< " fdev= " << F(9,6,forward_dev)
								<< " bdev= " << F(9,6,backward_dev)
								<< " fdev2= " << F(9,6,forward_dev2) << std::endl;
		}
	}

	// write some simple conversion rules to follow
	// vector V written in a different frame:
	// XYZ -> M: M^T * V
	// M -> XYZ: M * V
	// M is also the rotation matrix which rotates XYZ to M in XYZ frame.
	// M1 -> M2: R^T * V, in which R = M1^T * M2
	// M2-> M1: R * V
	// R is also the rotation matrix which rotates M1 to M2 in M1 frame.
	///////////////////////////////////////////////////////////////

	/////////////////////////////////////////////////////////////////////////////
	RT::RT(
		kin::Stub const & stub1,
		kin::Stub const & stub2
	)
	{
		this->from_stubs( stub1, stub2 ); // silly, but explicit
	}

	void
	RT::from_stubs(
		kin::Stub const & stub1,
		kin::Stub const & stub2
	)
	{
		translation = stub1.M.transposed() * ( stub2.v - stub1.v );
		rotation = stub1.M.transposed() * stub2.M;
	}

	numeric::xyzMatrix_double const &
	RT::get_rotation() const
	{
		return rotation;
	}

	numeric::xyzVector_double const &
	RT::get_translation() const
	{
		return translation;
	}

	void RT::reset()
	{
		translation.zero();
		rotation.zero();
	}

	///////////////////////////////////////////////////////////////
	void RT::reverse() {
		// new rotation is the inversed(transposed) old rotation
		rotation.transpose();
		// new translation is the negated old tranlation in m1 frame rewritten
		// in m2 frame (left multiplied by old_rotation^T = new_rotation)
		translation = rotation * translation.negate();

// 		// temporaries, hold old settings:
// 		FArray1D_double const v( translation );
// 		FArray2D_double const R( rotation );
// 		//std::cout << "RT::reverse()\n";
// 		// new values: rotation = R^T; translation = -R*v ( or -R^T*v ?? )
// 		for ( int i=1; i<=3; ++i ) {
// 			translation(i) = - v(1) * R(i,1) - v(2) * R(i,2) - v(3) * R(i,3);
// //  			translation(i) = - v(1) * R(1,i) - v(2) * R(2,i) - v(3) * R(3,i); 1st try
// 			for ( int j=1; j<=3; ++j ) {
// 				rotation(j,i) = R(i,j);
// 			}
// 		}
	}


	///////////////////////////////////////////////////////////////
	// should move to using Vectors in RT class
	//
	void
	RT::set_translation(
		numeric::xyzVector_double const & t
	)
	{
		translation = t;
	}

	void
	RT::set_rotation( //chutmp
		numeric::xyzMatrix_double const & r
	)
	{
		rotation = r;
	}

	///////////////////////////////////////////////////////////////
	// rb_center is the center of rotation, written in the down-stream (jump_end)
	// coordinate system
	void
	RT::fold_in_rb_deltas(
		FArray1Da_double rb,
		numeric::xyzVector_double const & rb_center
	)
	{
		rb.dimension(6);

		// hold the old values
		//FArray2D_double const R(rotation);
		//FArray1D_double const t(translation);

		// simple: rotation gets multiplied by Rzyx,
		// translation (t) goes to center + Rzyx(t-center) + rb_trans

		// find position of rb-center in jump_start coord sys
		// m2 -> m1: R * V
		numeric::xyzVector_double new_center(translation + rotation * rb_center);
// 		FArray1D_double center(3);
// 		for ( int k=1; k<= 3; ++k ) {
// 			center(k) = t(k) +
// 				rb_center(1) * R(1,k) +
// 				rb_center(2) * R(2,k) +
// 				rb_center(3) * R(3,k);
// 			//std::cout << "center in jump_start coord-sys: " << k << ' ' <<
// 			//	center(k) << std::endl;
// 		}

		// create a transformation matrix from the 3 rb angles
		numeric::xyzMatrix_double Rzyx;
		Rzyx = Z_rot( rb(6) ) * ( Y_rot( rb(5) ) * X_rot( rb(4) ) );
// 		FArray2D_double Rzyx(3,3);
// 		jmp_rotation_matrix(Rzyx,rb(4),rb(5),rb(6)); // R = Rz * Ry * Rx (z is last
// 				)
		rotation = Rzyx * rotation;
		//		mat_multiply3( Rzyx, R, rotation ); // is this the correct order ???

		numeric::xyzVector_double rb_trans( &rb(1) ); // load the first 3
		translation = new_center + Rzyx * ( translation - new_center ) + rb_trans;
// 		FArray1D_double v1(3), v2(3);
// 		Dsubvec(t,center,v1);
// 		Dvect_multiply( Rzyx, v1, v2);

// 		for ( int k=1; k<= 3; ++k ) {
// 			translation(k) = center(k) + v2(k) + rb(k);
// 		}
	} // fold_in_rb_deltas


	/////////////////////////////////////////////////////////////////////////////
	// should be the inverse of "RT::from_stubs"
	void
	RT::make_stub_jump(
		kin::Stub const & stub1,
		kin::Stub & stub2
	) const
	{
		stub2.M = stub1.M * rotation;
		stub2.v = stub1.v + stub1.M * translation;
	}

	///////////////////////////////////////////////////////////////////////////
	// rotate and translate Epos_in to get Epos_out
	// note that bond geometry of Epos_out comes from Epos_in
	void RT::make_jump( FArray2Da_float Epos_in, FArray2Da_float Epos_out ) const {
		Epos_in.dimension(3,param::MAX_POS);
		Epos_out.dimension(3,param::MAX_POS);
		// static for speed?

		numeric::xyzMatrix_double p, m;
		get_ncac(Epos_in,p);
		get_coordinate_system(p,m);

		numeric::xyzVector_double old_ca( &Epos_in(1,2) ), new_ca;
		// m * translation is the translation vector written in xyz frame.
		new_ca = old_ca + m * translation;

// 		FArray1D_double new_ca(3);
// 		for ( int i=1; i<=3; ++i ) {
// 			new_ca(i) = Epos_in(i,2) +
// 				translation(1) * m(i,1) +
// 				translation(2) * m(i,2) +
// 				translation(3) * m(i,3);
// 		}

//		FArray1D_double v1(3),v2(3),v3(3);

		// rotation_in_xyz is the rotation expressed in xyz frame
		numeric::xyzMatrix_double rotation_in_xyz;
		rotation_in_xyz = m * ( rotation * m.transposed() );
		for ( int j = 1; j <= param::MAX_POS; ++j ) {
			numeric::xyzVector_double old_atom( &Epos_in(1,j) ), new_atom;
			new_atom = new_ca + rotation_in_xyz * ( old_atom - old_ca );
			// copy it to Epos_out
			FArray1Da_float atom_Epos( Epos_out(1,j), 3 );
			copy_to_FArray( new_atom, atom_Epos );
			//Epos_out(1,j) = new_atom.x();
			//Epos_out(2,j) = new_atom.y();
			//Epos_out(3,j) = new_atom.z();
		}
			// basically (in Charlie-speak):
			//  new_v = new_ca + M^T * R * M ( old_v - old_ca )
			// basically (in row,col indexing):
			//  new_v = new_ca + M * R * M^T ( old_v - old_ca )

			// we can think of multiplication by M (non-Charlie)
			// as expressing a vector given in the jump-coordinate system
			// (which is defined by N-CA-C of 1st jump point)
			// in the absolute coordinate system.

			// for ( int k=1; k<=3; ++k ) v1(k) = double( Epos_in(k,j) - Epos_in(k,2) );
// 			for ( int k=1; k<=3; ++k ) v2(k) = Ddotprod( v1, m(1,k) );
// 			// equivalent to: Dvect_multiply( m, v1, v2 )

// 			// either this or rotation transposed !!!!!!!!!!!!!!!!!!
// 			Dvect_multiply( rotation, v2, v3 );

// 			for ( int k=1; k<=3; ++k ) {
// 				Epos_out(k,j) = new_ca(k) +
// 					v3(1) * m(k,1) +
// 					v3(2) * m(k,2) +
// 					v3(3) * m(k,3);
// 			}
//		}
	} // make_jump

	/////////////////////////////////////////////////////////////////////////////
	void
	RT::identity_transform()
	{
		translation.zero();
		rotation.to_identity();
	}

	bool RT::ortho_check() const
	{
		// debug orthogonality
		double dev(0.0);
		for ( int i=1; i<=3; ++i ) {
			for ( int j=1; j<=3; ++j ) {
				double f =
					rotation(1,i) * rotation(1,j) +
					rotation(2,i) * rotation(2,j) +
					rotation(3,i) * rotation(3,j);
				if ( i==j ) dev += std::abs(1.0-f);
				else dev += std::abs(f);
			}
		}
		return ( dev < 0.01 );
	}


	/////////////////////////////////////////////////////////////////////////////
	// these two extractors must be inverses: ie if we write an RT
	// out with one and then read it back in using the other the
	// new RT should be the same
	// this is critical for pose silent input/output to work correctly.

	std::ostream & operator <<( std::ostream & os, const RT & rt )
	{
		assert( rt.ortho_check() );
		os << " RT ";
		for ( int i = 1; i <= 3; ++i ) {
			for ( int j = 1; j <= 3; ++j ) {
				//os << F(9,3,rt.rotation(i,j)); // output by row
				os << rt.rotation(i,j) << ' '; // chu -- more digits for precision
			}
		}
		for ( int i = 1; i <= 3; ++i ) {
			//os << F(9,3,rt.translation(i) );
			os << rt.translation(i) << ' '; // chu -- more digits for precision
		}
		os << ' ';
		return os;
	}
// 	std::ostream & operator <<( std::ostream & os, const RT & rt )
// 	{
// 		assert( rt.ortho_check() );
// 		os << " RT ";
// 		os << rt.rotation;
// 		os << rt.translation;
// 		os << ' ';
// 		return os;
// 	}

	std::istream & operator >>( std::istream & is, RT & rt )
	{
		std::string tag;
		is >> tag;
		if ( !is.fail() && tag == "RT" ) {
			for ( int i = 1; i <= 3; ++i ) {
				for ( int j = 1; j <= 3; ++j ) {
					is >> rt.rotation(i,j); // input by row
				}
			}
			for ( int i = 1; i <= 3; ++i ) {
				is >> rt.translation(i);
			}
			if ( !is.fail() ) {
				if ( !rt.ortho_check() ) {
					std::cout << "RT failed orthogonality check!" << std::endl;
					is.setstate( std::ios_base::failbit );
				}
			}
		}
		return is;
	}


	///////////////////////////////////////////////////////////////
	////////////////////////JUMP //////////////////////////////////
	///////////////////////////////////////////////////////////////

	void
	Jump::reset()
	{
		rt.reset();
		rb_delta = 0.0;
		rb_center = numeric::xyzVector_double(0.0);
	}

	numeric::xyzMatrix_double const &
	Jump::get_rotation() const
	{
		return rt.get_rotation();
	}

	numeric::xyzVector_double const &
	Jump::get_translation() const
	{
		return rt.get_translation();
	}


	bool Jump::ortho_check() const {
		return rt.ortho_check();
	}

	void Jump::fold_in_rb_deltas() {
		// n2c
		rt.fold_in_rb_deltas( rb_delta(1,1), rb_center(1) );
		rt.reverse();

		// c2n
		rt.fold_in_rb_deltas( rb_delta(1,2), rb_center(2) );
		rt.reverse();

		rb_delta = 0.0; // array assignment
	}

	// jump_Epos is the Eposition slice for the jump residue
	// It's the downstream residue for the given direction,
	// ie if dir == n2c then it's the higher-sequence-numbered residue
	// if dir == c2n it's the lower-sequence-numbered residue
	//
	// eg: the default center of rotation is the C-alpha of this
	// residue

	void Jump::set_rb_center(
		int const dir,
		FArray2Da_float jump_Epos,
		numeric::xyzVector_double const & center
	)
	{
		jump_Epos.dimension(3,param::MAX_POS);

		int const n2c(1), c2n(-1);
		assert( dir == n2c || dir == c2n );

		// clear existing deltas
		if ( nonzero_deltas() ) {
			std::cout << "WARNING:: Jump::set_rb_center: clearing the " <<
				"existing rb_deltas!" << std::endl;
		}
		fold_in_rb_deltas();

		// calculate coordinate system
		numeric::xyzMatrix_double p, m;
		get_ncac( jump_Epos, p );
		get_coordinate_system(p,m);

		// get vector from jump C-alpha to the desired rotation center
		// and fill in the new data
		int const index( dir == n2c ? 1 : 2 );
		// center-ca is the position coord in xyz frame
		// xyz -> M: M^T * V
		rb_center(index) =  m.transposed() * ( center - p.col(2)/*ca*/ );
	}
	///////////////////////////////////////////////////////////////
	// stub defines the reference frame for center
	//
	// the column vectors of stub are unit vectors defined in the
	// absolute reference frame( the frame in which center is defined)
	//
	// so left-multiplying by stub.M.transposed() puts an absolute ref
	// frame vector into the local reference frame
	//
	void
	Jump::set_rb_center(
		const int dir,
		kin::Stub const & stub,
		numeric::xyzVector_double const & center
		)
	{
		fold_in_rb_deltas();
		int const index( dir == 1 ? 1 : 2 );
		rb_center(index) = stub.M.transposed() * ( center - stub.v );
	}


	bool Jump::nonzero_deltas() const {
		const double tol(1.0e-3);
		double sum(0.0);
		for ( int i=1; i<=2; ++i ) {
			for ( int j=1; j<= 6; ++j ) {
				sum += std::abs( rb_delta(j,i) );
			}
		}
		return ( sum > tol );
	}

	///////////////////////////////////////////////////////////////
	// make a an additional random rotation of alpha degrees.
	// (in the n2c direction!) will be centered on rb_center

// 	void Jump::random_rot( const float alpha_in ) {
// 		// choose a random unit vector as the axis of rotation,
// 		// by selecting its spherical coordinates phi(0-180) and theta(0-360)
// 		using numeric::conversions::degrees;
// 		using numeric::sin_cos_range;

// 		const double theta( 360.0 * ran3());
// 		const double phi( degrees( std::acos(sin_cos_range(1.0-2.0*ran3()))));
// 		const double alpha( alpha_in );
// 		rotation_by_axis( phi, theta, alpha );
// 	}
	////////////////////////////////////////////////////////////////////////
	void Jump::random_trans( const float dist_in ) {
		// choose a random unit vector as the direction of translation,
		// by selecting its spherical coordinates phi(0-180) and theta(0-360)
		// then move dist_in along that direction
		using numeric::conversions::degrees;
		using numeric::sin_cos_range;

		const double theta( 360.0 * ran3());
		const double phi( degrees( std::acos(sin_cos_range(1.0-2.0*ran3()))));
		const double dist( dist_in );

		fold_in_rb_deltas();
		rt.set_translation( dist * (Y_rot(phi)*Z_rot(theta)).col_z() );
	}
	///////////////////////////////////////////////////////////////////////
	void
	Jump::gaussian_move(
		int const dir,
		float const trans_mag,
		float const rot_mag )
	{
		fold_in_rb_deltas(); // clear rb_delta
		for ( int i = 1; i <= 3; ++i ) {
			set_rb_delta( i, dir, double( trans_mag * gaussian() ) );
			set_rb_delta( i+3, dir, double( rot_mag * gaussian() ) );
		}
		fold_in_rb_deltas();
	}
	///////////////////////////////////////////////////////////////////////
	void
	Jump::rotation_by_matrix(
		FArray2Da_float Epos,
		numeric::xyzVector_double const & center, //in xyz frame
		numeric::xyzMatrix_double const & matrix //in xyz frame
	)
	{
		Epos.dimension(3,param::MAX_POS);

		fold_in_rb_deltas(); // clear rb_delta

		numeric::xyzMatrix_double p, m;

		// see RT::fold_in_rb_delta for similar logic
		// get jump coodinate system
		get_ncac(Epos, p);
		get_coordinate_system(p,m);
		// find the rotation center in jump coordinate system
		numeric::xyzVector_double new_center = m.transposed() * ( center - p.col_y() );
		// find the operator matrix when transofrmed to jump coord system
		numeric::xyzMatrix_double new_matrix = m.transposed() * ( matrix * m );
		// new translation vector after applying matrix in jump coord system
		rt.set_translation( new_center + new_matrix * ( rt.get_translation() - new_center ) );
		// new rotation after applying matrix in jump coord system
		rt.set_rotation( new_matrix * rt.get_rotation() );
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Jump::set_rotation(
		numeric::xyzMatrix_double const & R
	)
	{
		fold_in_rb_deltas(); // clear rb_delta
		rt.set_rotation( R );
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Jump::set_translation(
		numeric::xyzVector_double const & t
	)
	{
		fold_in_rb_deltas(); // clear rb_delta
		rt.set_translation( t );
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Jump::rotation_by_matrix(
		kin::Stub const & stub,
		numeric::xyzVector_double const & center, //in xyz frame
		numeric::xyzMatrix_double const & matrix //in xyz frame
	)
	{
		fold_in_rb_deltas(); // clear rb_delta

		numeric::xyzMatrix_double const & m( stub.M );

		// find the rotation center in jump coordinate system
		numeric::xyzVector_double new_center = m.transposed() * ( center - stub.v);

		// find the operator matrix when transofrmed to jump coord system
		numeric::xyzMatrix_double new_matrix = m.transposed() * ( matrix * m );

		// new translation vector after applying matrix in jump coord system
		rt.set_translation( new_center + new_matrix * ( rt.get_translation() -
																										new_center ) );

		// new rotation after applying matrix in jump coord system
		rt.set_rotation( new_matrix * rt.get_rotation() );
	}
	///////////////////////////////////////////////////////////////////////////////////////
	void
	Jump::rotation_by_axis(
		FArray2Da_float Epos,
		numeric::xyzVector_double const & axis, /* in xyz frame*/
		numeric::xyzVector_double const & center, /* in xyz frame*/
		float const alpha
	)
	{
		using numeric::conversions::radians;
		using numeric::conversions::degrees;
		using numeric::sin_cos_range;

		Epos.dimension(3,param::MAX_POS);

		fold_in_rb_deltas();

		numeric::xyzMatrix_double mat = numeric::rotation_matrix( axis, double(radians(alpha)) );

		rotation_by_matrix( Epos, center, mat );

	}
	//////////////////////////////////////////////////////////
	void
	Jump::translation_along_axis(
		FArray2Da_float Epos,
		numeric::xyzVector_double const & axis, //in xyz frame
		float const dist // in angstrom
	)
	{
		Epos.dimension(3,param::MAX_POS);
		fold_in_rb_deltas();

		numeric::xyzMatrix_double p, m;
		get_ncac(Epos,p);
		get_coordinate_system(p,m);

		numeric::xyzVector_double new_trans( double(dist) * axis.normalized() );
		rt.set_translation( rt.get_translation() + m.transposed() * new_trans );
	}

	//////////////////////////////////////////////////////////
	void
	Jump::translation_along_axis(
		kin::Stub const & stub,
		numeric::xyzVector_double const & axis, //in xyz frame
		float const dist // in angstrom
	)
	{
		fold_in_rb_deltas();
		numeric::xyzVector_double new_trans( double(dist) * axis.normalized() );
		rt.set_translation( rt.get_translation() + stub.M.transposed()*new_trans );
	}

	//////////////////////////////////////////////////////////
	void
	Jump::reverse()
	{
		fold_in_rb_deltas();
		rt.reverse();
	}

	//////////////////////////////////////////////////////////
	void
	Jump::identity_transform()
	{
		rb_delta = 0.0;
		rb_center = numeric::xyzVector_double(0.0);
		rt.identity_transform();
	}


	/////////////////////////////////////////////////////////////////////////////
	// make an additional rotation alpha about an axis determined by
	// phi and theta in sph coords
	//
	// void
// 	Jump::rotation_by_axis(
// 		double const phi,
// 		double const theta,
// 		double const alpha
// 	)
// 	{
// 		using numeric::conversions::degrees;
// 		using numeric::conversions::radians;

// 		const int n2c_index( 1 ); // n2c index into the rb_deltas
// 		const int zrot( 6 ); // the index in rb_delta for z-rotation
// 		const int yrot( 5 ); // the index in rb_delta for y-rotation

// 		// clear anything in the current deltas
// 		fold_in_rb_deltas();

// 		// compose our jump by a new rotation of the form:
// 		// Rz(theta) * Ry(phi) * Rz(alpha) * Ry(-phi) * Rz(-theta)
// 		rb_delta(zrot,n2c_index) = -theta;
// 		fold_in_rb_deltas();

// 		rb_delta(zrot,n2c_index) = alpha;
// 		rb_delta(yrot,n2c_index) = -phi;
// 		fold_in_rb_deltas();

// 		rb_delta(zrot,n2c_index) = theta;
// 		rb_delta(yrot,n2c_index) = phi;
// 		fold_in_rb_deltas();
// 	}


	/////////////////////////////////////////////////////////////////////////////
	// rotate and translate Epos_in to get Epos_out
	// note that bond geometry of Epos_out comes from Epos_in
	void Jump::make_jump( int const dir, FArray2Da_float Epos_in,
												FArray2Da_float Epos_out ) const {

		Epos_in.dimension(3,param::MAX_POS);
		Epos_out.dimension(3,param::MAX_POS);

		// debug:
		//std::cout << "Jump::make_jump\n" << *this;

		// make temporary local copy of our rotation-translation
		RT tmp_rt(rt);

		// n2c
		tmp_rt.fold_in_rb_deltas( rb_delta(1,1), rb_center(1) );
		tmp_rt.reverse();
		// c2n
		tmp_rt.fold_in_rb_deltas( rb_delta(1,2), rb_center(2) );
		// back to original direction
		if ( dir == pose_param::n2c ) tmp_rt.reverse();

		tmp_rt.make_jump( Epos_in, Epos_out);
	}

	/////////////////////////////////////////////////////////////////////////////
	// make jump with stubs instead of FArrays
	//
	void Jump::make_stub_jump(
		int const dir, // legacy
		kin::Stub const & stub1,
		kin::Stub & stub2
	) const
	{
		// make temporary local copy of our rotation-translation
		RT tmp_rt(rt);

		// n2c
		tmp_rt.fold_in_rb_deltas( rb_delta(1,1), rb_center(1) );
		tmp_rt.reverse();

		// c2n
		tmp_rt.fold_in_rb_deltas( rb_delta(1,2), rb_center(2) );

		// back to original direction
		if ( dir == pose_param::n2c ) tmp_rt.reverse();

		tmp_rt.make_stub_jump( stub1, stub2 );
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Jump::from_stubs(
		kin::Stub const & stub1,
		kin::Stub const & stub2
	)
	{
		rt.from_stubs( stub1, stub2 );
		rb_delta = 0.0;
		// NOTE: we dont reset rb_center!!!!!!!!!
	}

	/////////////////////////////////////////////////////////////////////////////
	// input
	std::istream & operator >>( std::istream & is, Jump & jump ) {
		is >> jump.rt;
		jump.rb_delta = 0.0;
		jump.rb_center = numeric::xyzVector_double( 0.0 );
		return is;
	}

	// stream output:
	std::ostream& operator <<( std::ostream& os, const Jump & jump ) {
		if ( jump.nonzero_deltas() ) {
			// old-style verbose output
			os << "Jump:: nonzero_deltas= " << jump.nonzero_deltas() << '\n';
			os << jump.rt;
			for ( int i=1; i<= 2; ++i ) {
				std::string tag ( (i==1) ? "n2c" : "c2n" );
				os << "rb_delta " << tag;
				for ( int j=1; j<= 6; ++j ) {
					//os << F(9,3,jump.rb_delta(j,i));
					os << jump.rb_delta(j,i) << ' '; //chu -- more digits for precision
				}
				os << '\n';
				os << "rb_center " << tag;
				for ( int j=1; j<= 3; ++j ) {
					os << jump.rb_center(i);
				}
				os << '\n';
			}
		} else {
			//
			os << jump.rt;
		}
		return os;
	}

	double
	distance( Jump const & a_in, Jump const & b_in )
	{

		Jump a( a_in ), b( b_in );

		a.fold_in_rb_deltas();
		b.fold_in_rb_deltas();

		return distance( a.rt, b.rt );
	}

	void
	jump_distance( Jump const & a_in, Jump const & b_in, double & dist, double & theta )
	{
		using numeric::xyzMatrix_double;
		using numeric::xyzVector_double;

		Jump a( a_in ), b( b_in );

		a.fold_in_rb_deltas();
		b.fold_in_rb_deltas();

		xyzMatrix_double A( a.get_rotation() );
		xyzVector_double v( a.get_translation() );

		xyzMatrix_double B( b.get_rotation() );
		xyzVector_double w( b.get_translation() );

		//double theta; // in radians
		rotation_axis( A * B.transposed(), theta ); // A * B^-1 = rotation to make A==B.

		dist = distance(v, w);

		//return distance( v, w ) + theta;
	}

}
