// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 23432 $
//  $Date: 2008-06-24 16:25:52 +0300 (Tue, 24 Jun 2008) $
//  $Author: yab $

// Rosetta Headers
#include "jumping_pairings.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "barcode_stats.h"
#include "crankshaft.h"
#include "fragments.h"
#include "fragment_class.h"
#include "files_paths.h"
#include "filters.h"
//#include "fold_abinitio.h"
#include "force_barcode.h"
#include "fullatom_setup.h"
//#include "fragment_class.h"
#include "fragments.h"
#include "fragments_ns.h"
#include "fullatom.h"
#include "idealize.h"
#include "initialize.h"
#include "jumping_diagnostics.h"
#include "jumping_ns.h"
#include "jumping_util.h"
#include "jumping_refold.h"
#include "make_pdb.h"
#include "map_sequence.h"
#include "maps.h"// for jump_relax
#include "maps_ns.h"
#include "misc.h"
#include "monte_carlo.h" // for jumprelax
#include "minimize.h"
#include "native.h"
#include "nblist.h"
#include "output_decoy.h" // for jumprelax
#include "pack_fwd.h"
#include "param.h"
#include "pose.h"
#include "pose_abinitio.h"
#include "pose_idealize.h" // for jumprelax
#include "pose_io.h"
#include "pose_relax.h"
#include "pose_rms.h"
#include "pose_util.h"
#include "pose_vdw.h" // for pose_update_cendist, to find disulfides.
#include "random_numbers.h"
#include "read_paths.h"
#include "read_aa_ss.h"
//#include "rebuild_loops.h"
#include "relax_structure.h" //For jump relax.
#include "relax.h" //For jump relax.
#include "runlevel.h"
#include "score.h"
#include "silent_input.h"
#include "termini.h"
#include "timer.h"

#include "options.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/string.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <algorithm>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <string>

// BOINC
#ifdef BOINC
#include "boinc_rosetta_util.h"
#include "counters.h"
#include "trajectory.h"
#endif

// Namespaces
//using namespace pose_ns;

namespace loop_sum_common {
	bool loop_sum_init = { false };
	FArray1D_float loop_sum;
}
namespace strand_sum_common {
	bool strand_sum_init = { false };
	FArray1D_float strand_sum;
}



class Pairing_template {
public:
	pose_ns::Jump jump;
	FArray1D_float phi;
	FArray1D_float psi;
	FArray1D_float omega;
	Pairing_template ( FArray2Da_float Epos1, FArray2Da_float Epos2 ):
		jump( Epos1, Epos2 ),
		phi  ( 2, 0.0 ),
		psi  ( 2, 0.0 ),
		omega( 2, 0.0 )
	{}
};

typedef std::vector< Pairing_template > Pairing_template_list;
typedef std::map< std::pair< int, int >, Pairing_template_list > Pairing_template_map;

namespace Pairing_template_map_ns {
	Pairing_template_map pairing_template_map;
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
// NOTES
//
////////////////////////
// PAIRINGS FILE FORMAT:
//
// one line per pairing, 4 numbers per line, white-space delimited
//
// format: "pos1 pos2 orientation pleating"
//
// pos1 and pos2 are the sequence numbers of the beta-paired positions
//
// orientation is the beta-strand orientation:
// "1" for antiparallel, "2" for parallel
// or just "A" or "P"
//
// pleating determines the pleat of the beta-carbons
// "1" if the N and O of pos1 are pointed away from pos2
// "2" if the N and O of pos1 are pointed toward pos2
//
// eg, in the antiparallel case, a pleat of 2 would mean that
// there are two backbone-backbone hydrogen bonds between pos1 and pos2
// In the parallel case a pleat of 2 means that pos1 is Hbonding with
// pos2-1 and pos2+1
//
// if you check out rosetta_benchmarks, in the directory 1d3z/
// is a pairing file "pairings.dat" with two pairings in it.
// These are native pairings, ie they match the pdb structure
// 1d3z.pdb in the same directory. In Fortran you had to tell
// Rosetta how many pairings to expect; that's why the first line
// has a "2" on it. This is no longer necessary.

//////////////////////
// COMMAND LINE SYNTAX
//
// for ab initio folding with pairings: the usual "xx 1xyz _ -silent"
// plus:
//
// -pairing_file <pairings-file>
//
// with no other arguments and it will try to build decoys with
// *all* the pairings in the file. You can also specify what
// kind of sheet topology Rosetta should try to construct:
//
// -sheet1 <N1> -sheet2 <N2> ... -sheetk <Nk>
//
// Here Nj is the number of strands in sheet number j. So the
// number of forced pairings will be (Nj-1). The sheet can
// get other strands during the folding simulation -- this is
// just specifying how many Rosetta should actual build from
// the start using the broken chain stuff.
//
// So the total number of forced pairings will be:
//  N1-1 + N2-1 + N3-1 + ... + Nk-1
//
// For example, to specify two strand pairings in two different
// sheets, use the args "-sheet1 2 -sheet2 2"
//
//
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////


///////////////////////////////////////////////////////////////////////////////
const pairing_ns::Pairing_list &
read_pairing_list()
{
	static bool init ( false );
	static pairing_ns::Pairing_list pairings;

	if (!init ) {
		// read the pairing file
		init = true;

		int a,b,c,d;
		std::string pairing_file;

		stringafteroption("pairing_file", "nonexistent_file", pairing_file);
		utility::io::izstream pairing_stream( pairing_file );
		if ( !pairing_stream ) {
			std::cout << "can't open pairings file!!!" << pairing_file << std::endl;
			pairing_stream.close();
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		std::string line;
		while ( getline( pairing_stream, line ) ) {
			std::istringstream line_stream( line );
			// a=i, b=j, c=orientation(1 or 2), d=pleating(1 or 2)
			std::string o;
			line_stream >> a >> b >> o >> d;
			if ( line_stream.fail() || o.size() != 1 ) {
				std::cout << "parse error: " << line << std::endl;
				continue;
			}

			if ( o == "A" || o == "1" ) {
				c = 1;
			} else if ( o == "P" || o == "2") {
				c = 2;
			} else {
				std::cout << "bad orientation: " << o << std::endl;
				continue;
			}

			if ( ( a < 1 || b < 1 ) || ( a == b ) || ( c != 1 && c != 2 ) ||
					 ( d != 1 && d != 2 ) ) {
				std::cout << "bad pairing:" <<
					SS( a ) << SS( b ) << SS( c ) << SS( d ) << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

			if ( a > b ) reverse_pairing(a,b,c,d);

			pairing_ns::Pairing p;
			p.pos1 = a;
			p.pos2 = b;
			p.orientation = c;
			p.pleating = d;

			pairings.push_back( p );
		}
		pairing_stream.close();
	} // if (!init )

	return pairings;
}


///////////////////////////////////////////////////////////////////////////////
void
read_jump_templates()
{
	const float MAX_NO_DIST ( 3.1 );

	static bool init ( false );
	if ( init ) return;
	init = true;

	using namespace Pairing_template_map_ns;

	utility::io::izstream & data( open_data_file( "jump_templates_v2.dat" ) );

	std::string line;
	std::string tag,filename;
	FArray2D_float Epos1(3,param::MAX_POS), Epos2(3,param::MAX_POS);
	int pos1,pos2;
	float o,p1,p2,mn_dist,mx_dist,
		phi1,psi1,omega1,phi2,psi2,omega2;

	while ( getline( data,line ) ) {
		std::istringstream is( line );
		is >> tag >> filename >> pos1 >> pos2 >> mn_dist >> mx_dist >>
			o >> p1 >> p2 >>
			Epos1(1,1) >> Epos1(2,1) >> Epos1(3,1) >>
			Epos1(1,2) >> Epos1(2,2) >> Epos1(3,2) >>
			Epos1(1,4) >> Epos1(2,4) >> Epos1(3,4) >>
			Epos2(1,1) >> Epos2(2,1) >> Epos2(3,1) >>
			Epos2(1,2) >> Epos2(2,2) >> Epos2(3,2) >>
			Epos2(1,4) >> Epos2(2,4) >> Epos2(3,4) >>
			phi1 >> psi1 >> omega1 >>
			phi2 >> psi2 >> omega2;
		if ( is.fail() || tag != "PAIR" ) continue;

		assert ( pos1 < pos2 && p1 * p2 > 0.0 &&
			std::abs(phi1) < 185 && std::abs(psi1) < 185 && std::abs(omega1) < 185 &&
			std::abs(phi2) < 185 && std::abs(psi2) < 185 && std::abs(omega2) < 185 );

		// filter
		// note that the filename contains info about the scop class so you could
		// in principle filter for b,c,or d class proteins individually
		if ( mx_dist > MAX_NO_DIST ||
				 phi1 >  0.0 || phi2 >  0.0 ||
				 psi1 < 50.0 || psi2 < 50.0 ||
				 std::abs( omega1 ) < 90 ||
				 std::abs( omega2 ) < 90 ) continue;

		// fill in a new beta-pairing template
		Pairing_template t( Epos1, Epos2);
		t.phi   (1) = phi1;
		t.phi   (2) = phi2;
		t.psi   (1) = psi1;
		t.psi   (2) = psi2;
		t.omega (1) = omega1;
		t.omega (2) = omega2;

		const int o_key( ( o  < 0.0 ) ? 1 : 2 ); // orientation
		const int p_key( ( p1 < 0.0 ) ? 1 : 2 ); // pleating

		pairing_template_map[ std::make_pair( o_key, p_key ) ].push_back( t );
		//std::cout << "new pairing: " << filename << ' ' << o_key << ' ' <<
		//	p_key << std::endl;
	}
}


///////////////////////////////////////////////////////////////////////////////
pose_ns::Jump
get_random_beta_sheet_jump(
	int const orientation,
	int const pleating
)
{
	read_jump_templates(); // self-initializing

	// key for looking up the template geometry:
	std::pair<int,int> key( orientation, pleating );

	// HACK to get it to compile -- fix this later
	assert ( Pairing_template_map_ns::pairing_template_map.count( key ) == 1 );

	const Pairing_template_list & templates
		( Pairing_template_map_ns::pairing_template_map.find( key )->second );

	const int ntemplates ( templates.size() );

	int const index( static_cast<int>( ran3() * ntemplates ) );
	const Pairing_template & t ( templates[ index ] );

	return t.jump;
}


///////////////////////////////////////////////////////////////////////////////
// decoy_pairings is cleared in this routine before new pairings are loaded

void
choose_random_pairings( pairing_ns::Pairing_list & decoy_pairings )
{
	// these guys return if already setup
	read_jump_templates();
	const pairing_ns::Pairing_list & original_protein_pairings( read_pairing_list());

	pairing_ns::Pairing_list protein_pairings(original_protein_pairings);

	//barcode stuff -- rhiju
	//	barcode_initialize_decoy(); // Chooses a barcode. // move this to main loop.
	int num_jumps_with_barcode = get_allowed_pairings_from_barcode(protein_pairings); // Filter. If no barcode, no change to protein_pairings.
	int num_protein_pairings( protein_pairings.size() );

	if ( num_protein_pairings == 0 ) {
		std::cout << "WARNING: choose_random_pairings:: num_protein_pairings == 0 !! Proceeding without jumping." <<
			std::endl;
		//		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		decoy_pairings.clear();
		return;
	}


	// currently there are two ways that we can construct decoys:
	// in the first: we specify on the command line the sizes
	// of the different sheets: " -sheet1 3 -sheet2 2 -sheet3 4 " etc
	// note that these sizes are the number of STRANDS per sheet,
	// not the number of PAIRINGS per sheet ( PAIRINGS = STRANDS - 1 )
	// pairings are chosen randomly from out list file to construct
	// the desired topology
	//
	// alternatively we just build decoys with the same set of pairings
	// each time, namely the full list of pairings in our pairing list file
	//
	// rhiju: added another method to phil's methods.
	// "-sheet_from_barcode" which is to be used in combination
	// with "-filter_jumps_with_barcode". In this mode, the number of
	// pairings is equal to the number of SSPAIR features that are given bonuses
	// in the barcode. The pairings aren't all required to be in the same sheet.
	//

	int num_sheets ( 0 );
	int num_decoy_pairings ( 0 );
	FArray1D_int sheet_size( jumping_pairings::max_sheets, 0 );
	bool check_compatible ( false );
	bool force_single_sheet ( true );

	if ( truefalseoption("sheet1") ) {
		// Method 1: here we fill the array sheet_size(num_sheets)
		// at the end we have set:: num_sheets, num_decoy_pairings, and sheet_size
		check_compatible = !truefalseoption("no_jump_compatibility_check");
		num_sheets = 0;
		num_decoy_pairings = 0;

		std::string flag;
		flag = "sheet"+string_of( num_sheets+1, 1 );

		while ( truefalseoption(flag) ) {
			++num_sheets;
			if ( num_sheets > 9 ) utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

			intafteroption(flag,0,sheet_size(num_sheets));
			std::cout << "sheet_size: " << flag << SS( sheet_size(num_sheets) ) <<
				std::endl;
			num_decoy_pairings += sheet_size(num_sheets)-1;
			flag = "sheet"+string_of( num_sheets+1,1 );
		}
		if ( num_decoy_pairings > num_protein_pairings ) {
			std::cout << "need more protein_pairings!" <<
				SS( num_decoy_pairings ) << SS( num_protein_pairings ) << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	} else if ( truefalseoption("sheet_from_barcode") ){
		// Method 2: Use all pairings consistent with barcode (as specified by SSPAIR features with negative weights)
		check_compatible = !truefalseoption("no_jump_compatibility_check");
		force_single_sheet = false; // Phil's convenient flag...
		num_sheets = 1;
		num_decoy_pairings =  num_jumps_with_barcode;
		std::cout << "Looking for " << num_decoy_pairings << " pairings based on current barcode." << std::endl;
		sheet_size(1) = num_decoy_pairings+1;
	} else {
		// Method 3: just use all the pairings in our list file
		check_compatible = false;
		num_sheets = 1;
		num_decoy_pairings = num_protein_pairings;
		sheet_size(1) = num_decoy_pairings+1;

		std::cout << "WARNING: no sheet info specified, using all the pairings" <<
			" from file: " << SS( num_decoy_pairings ) << SS( check_compatible ) <<
			std::endl;
	}

	//// fill the sheet_pairing array:
	FArray3D_int sheet_pairing( 4, jumping_pairings::max_sheet_size, jumping_pairings::max_sheets );


	int tries1 ( 0 );
	bool redo_same_strand ( true );
	while ( redo_same_strand ) {
		redo_same_strand = false;
		++tries1;
		if ( tries1 > 100 ) {
			std::cout << "problem in choose_random_pairings: tries1 > 100. Proceeding without jumping!!"
								<< std::endl;
			//utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			decoy_pairings.clear();
			return;
		}

		// makes stochastic decisions about strand boundaries:
		// some decisions may not be compatible with the desired pairings
		// and our logic for choosing sheets
		// so we want to have the option of redoing this, hence the
		// loop: while ( redo_same_strand )

		setup_same_strand( true ); // force reset

		int tries2 ( 0 );
		bool sheet_fail ( true );
		while ( sheet_fail ) {
			sheet_fail = false;
			tries2++;
			if ( tries2 > 100 ) {
				std::cout << "redo same_strand: too many tries with this one!" <<
					SS( tries1 ) << std::endl;
				redo_same_strand = true;
				break;
			}


			// here we fill the array "sheet_pairing" that has all the pairings
			// organized by sheets


			int tries3 ( 0 );
			for ( int sheet = 1; !sheet_fail && sheet <= num_sheets; ++sheet ) {
				for ( int pairing = 1, pairinge = sheet_size(sheet)-1;
							!sheet_fail && pairing <= pairinge;	++pairing ) {
					bool pairing_fail ( true );
					while ( pairing_fail && !sheet_fail ) {
						pairing_fail = false;
						++tries3;
						if ( tries3%10000 == 0 ) {
							//					std::cout << "tries:"	<< SS( tries1 ) << SS( tries2 ) << SS( tries3 ) << '\n';
						}

						if ( tries3 > 100000 ) {
							sheet_fail = true;
							break;
						}

						int const	p = static_cast< int >(ran3()*num_protein_pairings);
						//std::cout << "Picked pairing " << p << " out of " << num_protein_pairings << std::endl;

						// you should replace sheet_pairing with array of Pairings!!!!!
						// then just use operator=
						sheet_pairing(1,pairing,sheet) = protein_pairings[p].pos1;
						sheet_pairing(2,pairing,sheet) = protein_pairings[p].pos2;
						sheet_pairing(3,pairing,sheet) = protein_pairings[p].orientation;
						sheet_pairing(4,pairing,sheet) = protein_pairings[p].pleating;

						//////// evaluate this new pairing: ////////////
						if ( ! check_compatible ) {
							// at least make sure its not identical
							for ( int prev_sheet = 1; prev_sheet <= sheet; ++prev_sheet ) {
								for ( int prev_pairing = 1, pe = sheet_size(prev_sheet)-1;
											prev_pairing <= pe; ++prev_pairing ) {
									if ( prev_sheet != sheet || prev_pairing != pairing ) {
										bool same_pairing ( true );
										for ( int i = 1; i <= 4; ++i ) {
											if ( sheet_pairing(i,prev_pairing,prev_sheet) !=
													 sheet_pairing(i,pairing,sheet) )
												same_pairing = false;
										}
										if ( same_pairing ) {
											pairing_fail = true;
											break;
										}
									}
								}
							}
						} else {
							// full compatibility check
							for ( int prev_sheet = 1, pe = sheet-1; prev_sheet <= pe; ++prev_sheet ) {
								// dont want to intersect previous sheets
								for ( int prev_pairing = 1, pe = sheet_size(prev_sheet)-1;
											prev_pairing <= pe; ++prev_pairing ) {
									if ( pairing_intersect( sheet_pairing(1,prev_pairing,prev_sheet),
																					sheet_pairing(1,pairing,sheet) ) ) {
										pairing_fail = true;
										break;
									}
								}
							}
							//							const bool force_single_sheet ( true ); // old option
							if ( ! check_sheet_pairings( sheet_pairing(1,1,sheet),
																					 pairing,force_single_sheet) ) {
								pairing_fail = true;
								//break; // BUG: dont want to break out of the while loop
							}
						} // check_compatible?
					} // while ( pairing_fail && !sheet_fail )
				}
			}
		} // while ( sheet_fail )
	} // while ( redo_same_strand)

	// now load these pairings into the decoy_pairings list:
	decoy_pairings.clear();
	for ( int sheet = 1; sheet <= num_sheets; ++sheet ) {
		for ( int pairing = 1, pe = sheet_size(sheet)-1; pairing <= pe; ++pairing ) {

			pairing_ns::Pairing p;
			p.pos1        = sheet_pairing(1,pairing,sheet);
			p.pos2        = sheet_pairing(2,pairing,sheet);
			p.orientation = sheet_pairing(3,pairing,sheet);
			p.pleating    = sheet_pairing(4,pairing,sheet);

			decoy_pairings.push_back( p );

			std::cout << "sheet_pairing:" << SS(sheet) << SS(pairing) << ' ' <<
				p.pos1 << ' ' << p.pos2 << ' ' <<
				p.orientation << ' ' << p.pleating << std::endl;
		}
	}
}

///////////////////////////////////////////////////////////////////////////////
// assumes that pose has already been sized from fasta
//
// sets allow_bb_move false at the jump positions!!
//

void
pose_from_random_pairings( pose_ns::Pose & pose )
{
	pairing_ns::Pairing_list decoy_pairings;

	const int nres ( pose.total_residue() );

	// fill in num_jump jump_point
	int num_jump;
	FArray2D_int jump_point(2,nres); //plenty of space

	// local fold_tree:
	pose_ns::Fold_tree fold_tree;
	bool fold_tree_success ( false );
	while ( !fold_tree_success ) {
		// we keep choosing until we find a set that builds a tree
		// successfully. This should almost always happen on the first try
		choose_random_pairings( decoy_pairings );

		// first fill a jump_point array:
		num_jump = 0;
		for ( pairing_ns::Pairing_list::const_iterator it = decoy_pairings.begin(),
						it_end = decoy_pairings.end(); it != it_end; ++it ) {
			num_jump++;
			jump_point(1,num_jump) = it->pos1;
			jump_point(2,num_jump) = it->pos2;
		}

		// now try to construct a fold_tree from these jump_points and
		// the cut_bias array
		fold_tree_success =
			fold_tree.random_tree_from_jump_points( nres, num_jump, jump_point,
																							get_loop_fraction( nres) );

		if ( !fold_tree_success ) {
			std::cout << "failed to choose a fold_tree from these pairings !!\n";
		}
	} // while ( !fold_tree_success )

	// copy this tree into the pose
	pose.set_fold_tree( fold_tree );

	// now go back and fill in the pose variables
	int jump_number( 0 );
	for ( pairing_ns::Pairing_list::const_iterator it = decoy_pairings.begin(),
					it_end = decoy_pairings.end(); it != it_end; ++it ) {
		++jump_number;
		assert ( it->pos1 < it->pos2 &&
						 jump_point(1,jump_number) == it->pos1 &&
						 jump_point(2,jump_number) == it->pos2 );

		// key for looking up the template geometry:
		std::pair<int,int> key (it->orientation, it->pleating);

		// HACK to get it to compile -- fix this later
		assert ( Pairing_template_map_ns::pairing_template_map.count( key ) == 1 );

		const Pairing_template_list & templates
			( Pairing_template_map_ns::pairing_template_map.find( key )->second );

		const int ntemplates ( templates.size() );
		assert ( ntemplates > 0 );

		// choose one of these templates randomly:
		const int template_number ( static_cast< int > ( ran3() * ntemplates ) );
		assert ( 0 <= template_number && template_number < ntemplates );
		const Pairing_template & t ( templates[ template_number ] );

		for ( int j = 1; j <= 2; ++j ) {
			const int pos (jump_point(j,jump_number) );
			pose.set_allow_bb_move( pos, true );
			pose.set_phi  ( pos, t.phi(j) );
			pose.set_psi  ( pos, t.psi(j) );
			pose.set_omega( pos, t.omega(j) );
			pose.set_secstruct( pos, 'E' );
			pose.set_allow_bb_move( pos, false ); // DANGER!!!!!!!!!!!!!!!!!!!
		}
		pose.set_jump( jump_number, t.jump );
	}                     // i=1,num_jump

	// extends strings of E's so choose_fragment doesnt complain
	safe_secstruct( pose );
}


///////////////////////////////////////////////////////////////////////////////
bool
pairing_intersect(
	FArray1Da_int p1,
	FArray1Da_int p2
)
{
	setup_same_strand(); // returns if already done

	p1.dimension( 4 );
	p2.dimension( 4 );

	for ( int i = 1; i <= 2; ++i ) {
		for ( int j = 1; j <= 2; ++j ) {
			if ( same_strand_ns::same_strand(p1(i),p2(j)) ) return true;
		}
	}
	return false;
}


///////////////////////////////////////////////////////////////////////////////
bool
pairing_intersect_bothstrands(
	FArray1Da_int p1,
	FArray1Da_int p2
)
{
	setup_same_strand(); // returns if already done

	p1.dimension( 4 );
	p2.dimension( 4 );

	if ( same_strand_ns::same_strand(p1(1),p2(1)) & same_strand_ns::same_strand(p1(2),p2(2)) ) return true;
	if ( same_strand_ns::same_strand(p1(1),p2(2)) & same_strand_ns::same_strand(p1(2),p2(1)) ) return true;

	return false;
}


///////////////////////////////////////////////////////////////////////////////
bool
check_two_pairings(
	FArray1Da_int pairing1,
	FArray1Da_int pairing2,
	int & common_strands
)
{
	using namespace same_strand_ns;
	pairing1.dimension(4);
	pairing2.dimension(4);

	setup_same_strand();

	common_strands = 0;

	int i_start       = pairing1(1);
	int i_stop        = pairing1(2);
	int i_orientation = pairing1(3);
	int i_pleating    = pairing1(4);

	int j_start       = pairing2(1);
	int j_stop        = pairing2(2);
	int j_orientation = pairing2(3);
	int j_pleating    = pairing2(4);

	if ( same_strand(i_start,j_start) && same_strand(i_stop,j_stop) ) {
		common_strands = 2;
	} else if ( same_strand(i_start,j_stop) && same_strand(i_stop,j_start) ) {
		common_strands = 2;
		// could reverse either one:
		reverse_pairing( j_start, j_stop, j_orientation, j_pleating );
	} else if ( same_strand(i_start,j_start) ) {
		common_strands = 1;
	} else if ( same_strand(i_start,j_stop) ) {
		reverse_pairing( j_start, j_stop, j_orientation, j_pleating );
		common_strands = 1;
	} else if ( same_strand(i_stop,j_start) ) {
		reverse_pairing( i_start, i_stop, i_orientation, i_pleating );
		common_strands = 1;
	} else if ( same_strand(i_stop,j_stop) ) {
		reverse_pairing( j_start, j_stop, j_orientation, j_pleating );
		reverse_pairing( i_start, i_stop, i_orientation, i_pleating );
		common_strands = 1;
	}

	// now we have set things up so that i_start and j_start
	// are in the same strand.
	//
	// and maybe also i_stop and j_stop, if common_strands == 2
	assert ( common_strands == 0 || same_strand(i_start,j_start) );

	const int seqsep_mod2( mod(std::abs(i_start-j_start),2) );

	return ( ( common_strands == 0 ) ||
		( common_strands == 1 && seqsep_mod2 == 0 && i_pleating != j_pleating ) ||
		( common_strands == 1 && seqsep_mod2 == 1 && i_pleating == j_pleating ) ||
		( common_strands == 2 && seqsep_mod2 == 0 && i_pleating == j_pleating ) ||
		( common_strands == 2 && seqsep_mod2 == 1 && i_pleating != j_pleating ) );

}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
bool
check_sheet_pairings(
	FArray2Da_int pairing_list,
	const int last_pairing,
	const bool force_single_sheet
)
{
	if ( last_pairing < 2 ) return true; // first strand is OK by default

	pairing_list.dimension( 4, jumping_pairings::max_sheet_size );

	int total_common_strands( 0 );
	for ( int i = 1; i <= last_pairing-1; ++i ) {
		int common_strands(0);
		const bool ok( check_two_pairings( pairing_list(1,last_pairing),
																			 pairing_list(1,i), common_strands ) );
		if ( !ok || common_strands > 1 ) return false;
		total_common_strands += common_strands;
	}

	return ( ( total_common_strands >= 1 || !force_single_sheet ) &&
					 ( total_common_strands < 2 ) );
}

///////////////////////////////////////////////////////////////////////////////
// uses fragments_nres
//
void
setup_same_strand(
	const bool force_reset // = false
)
{
	using namespace same_strand_ns;
	//using namespace misc;
	using namespace param;

	static bool init( false );
	if ( init && !force_reset ) return;
	init = true;


	float strand1,strand2,strand_ceiling,k_strand;

	const int total_residue( fragments::fragments_nres );
	same_strand.dimension( total_residue, total_residue );

	FArray1D_float strand_sum( SRange( 0, total_residue ) );
	float loop, r;

	get_strand_predictions(strand_sum);

	for ( int pos1 = 1; pos1 <= total_residue; ++pos1 ) {
		for ( int pos2 = 1; pos2 <= total_residue; ++pos2 ) {
			strand1 = strand_sum(pos1)-strand_sum(pos1-1);
			strand2 = strand_sum(pos2)-strand_sum(pos2-1);
			strand_ceiling = max(0.2f, min(strand1,strand2));

			int const i = min(pos1,pos2);
			int const j = max(pos1,pos2);

			if ( j-i > 5 ) {
				same_strand(pos1,pos2) = false;
			} else if ( j-i < 2 ) {
				same_strand(pos1,pos2) = true;
			} else {
				same_strand(pos1,pos2) = true;
				for ( int k = i+1, ke = j-1; k <= ke; ++k ) {
					k_strand = ( strand_sum(k) - strand_sum(k-1) ) / strand_ceiling;
					loop = 1.0 - k_strand;
					if ( loop < 0.3 ) loop = 0.0;
					r = ran3();
					if ( r < loop ) {
						same_strand(pos1,pos2) = false;
//						std::cout << "loop cut:" << SS( i ) << SS( k ) << SS( j ) <<
//						 SS( loop ) << std::endl;
					}
				}            // k
			}               // sep>5
		}                  // pos2
	}                     // pos1
}

///////////////////////////////////////////////////////////////////////////////
void
reverse_pairing(
	int & pstart,
	int & pstop,
	int & orientation,
	int & pleating
)
{

	int tmp = pstop;
	pstop = pstart;
	pstart = tmp;

	if ( orientation == jumping_pairings::parallel_orientation ) { // flip pleating
		if ( pleating == 1 ) {
			pleating = 2;
		} else if ( pleating == 2 ) {
			pleating = 1;
		} else {
			std::cout << "unrecognized pleating:" << SS( pleating ) << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}
}
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
/// @begin get_strand_predictions
///
/// @brief
/// get fraction strand at central residues of 3-mers
/// set to 0 at 1 and total_residue
///
/// @detailed
///
/// @param[out]  output_strand_sum
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
get_strand_predictions( FArray1Da_float output_strand_sum /* output */ )
{
	using namespace fragments;
	//using namespace misc;
	using namespace param;
	using namespace strand_sum_common;

	int const total_residue( fragments::fragments_nres );

	output_strand_sum.dimension( SRange( 0, total_residue ) );

	if ( strand_sum_init ) {
		for ( int i = 0; i <= total_residue; ++i ) {
			output_strand_sum(i) = strand_sum(i);
		}
		return;
	}

	std::cout << "setting up strand_sum" << std::endl;
	strand_sum_init = true;
	strand_sum.dimension( SRange( 0, total_residue ) );

	// calculate fraction E in fragments
	FArray1D_float strand_fraction( calc_frag_E() );

	strand_sum(0) = 0.0;
	for ( int i = 1; i <= total_residue; ++i ) {
		strand_sum(i) = strand_sum(i-1) + strand_fraction(i);
	}

	for ( int i = 0; i <= total_residue; ++i ) {
		output_strand_sum(i) = strand_sum(i);
	}

}

///////////////////////////////////////////////////////////////////////////////
FArray1D_float
calc_frag_E()
{
	using namespace fragments;
	//using namespace misc;
	using namespace param;

	int const total_residue( fragments::fragments_nres );
	int const size( 3 );
	int const size_bin( get_index_by_frag_size(size) );

	std::cout << "calc_frag_E: " << total_residue << ' ' << size_bin <<
		std::endl;

	FArray1D_float strand_fraction( total_residue );
	strand_fraction(1) = 0.0;
	strand_fraction(total_residue) = 0.0;

	for ( int begin = 1, be = total_residue-size+1; begin <= be; ++begin ) {

		int const central_residue( 1 ); // numbering starts at 0 in fragments_ns.h

		int const total = align_depth(begin,size_bin);
		float strand_count( 0.0 );

		for ( int frag = 1; frag <= total; ++frag ) {
			if ( ss_type(begin,frag,central_residue,size_bin) == 'E') ++strand_count;
		}
		strand_fraction(begin+central_residue) = strand_count / total;
	}
	return strand_fraction;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_loop_predictions
///
/// @brief
/// get fraction loop at central residues of 3-mers
/// set to 0 at 1 and total_residue
///
/// @detailed
///
/// @param[out]  output_loop_sum -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
get_loop_predictions( FArray1Da_float output_loop_sum /* output */ )
{
	using namespace fragments;
	using namespace loop_sum_common;
	//using namespace misc;
	using namespace param;

	int const total_residue( fragments::fragments_nres );

	output_loop_sum.dimension( SRange( 0, total_residue ) );

	int size,size_bin,total,central_residue;
	float loop_count;
	FArray1D_float loop_fraction( total_residue );


	if ( loop_sum_init ) {
		for ( int i = 0; i <= total_residue; ++i ) {
			output_loop_sum(i) = loop_sum(i);
		}
		return;
	}

	std::cout << "setting up loop_sum" << std::endl;
	loop_sum_init = true;
	loop_sum.dimension( SRange( 0, total_residue ));

	size = 3;
	size_bin = get_index_by_frag_size(size);

	loop_fraction(1) = 0.0;
	loop_fraction(total_residue) = 0.0;

	for ( int begin = 1, be = total_residue-size+1; begin <= be; ++begin ) {

		central_residue = 1; // numbering starts at 0 in fragments_ns.h

		total = align_depth(begin,size_bin);
		loop_count = 0;

		for ( int frag = 1; frag <= total; ++frag ) {
			if ( ss_type(begin,frag,central_residue,size_bin) == 'L' ) ++loop_count;
		}

		loop_fraction(begin+central_residue) = loop_count / total;
	}


// loop_sum(i) = Sum (j=1 to i) loop_fraction(j)
	loop_sum(0) = 0.0;

	for ( int i = 1; i <= total_residue; ++i ) {
//		std::cout << "percent loop:  " << SS( i ) << ' '
//		 << F( 9, 3, 100*loop_fraction(i) ) << std::endl;
		loop_sum(i) = loop_sum(i-1) + loop_fraction(i);
	}

	for ( int i = 0; i <= total_residue; ++i ) {
		output_loop_sum(i) = loop_sum(i);
	}

}

// this is a little tricky
// the fragments have an implicit definition of total_residue
// here we assume that this agrees with passed-in nres
// but we flag changes in nres after initialization and die.

FArray1D_float const &
get_loop_fraction( int const nres )
{
	static FArray1D_float loop_fraction;
	static bool init(false);

	if ( init ) {
		if ( nres != int( loop_fraction.size1() ) ) {
			std::cout << "total_residue problem in get_loop_fraction!! "
								<< nres << ' ' << loop_fraction.size1() << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		return loop_fraction;
	}

	init = true;
	std::cout << "get_loop_fraction:: setting up loop_fraction" << std::endl;
	loop_fraction.dimension( nres );

	int const size(3);
	int const size_bin( get_index_by_frag_size(size) );

	for ( int pos=1; pos <= nres; ++pos ) {
		int frag_begin;
		if ( pos == 1 ) {
			frag_begin = pos;
		} else if ( pos == nres ) {
			frag_begin = pos-2;
		} else {
			frag_begin = pos-1;
		}

		int const frag_index ( pos - frag_begin );
		// debug:
		assert( 0<= frag_index && frag_index < size &&
						 1<= frag_begin && frag_begin <= nres - size + 1);

		int const total_frags ( fragments::align_depth(frag_begin,size_bin) );
		int loop_count (0);
		for ( int frag = 1; frag <= total_frags; ++frag ) {
			if ( fragments::ss_type(frag_begin,frag,frag_index,size_bin) == 'L' ) ++loop_count;
		}
		loop_fraction( pos ) = float( loop_count ) / total_frags;
	}
	return loop_fraction;
}

///////////////////////////////////////////////////////////////////////////////
// this subroutine is used to setup the new database of jump-transforms
// I used it with "-test" mode, hacking main.cc to call this routine
// after initialize_rosetta

void
get_beta_pairing_jumps()
{
	using namespace pose_ns;
	// read a bunch of pdbs, get jumps corresponding to pairings

	///////////////////////////////
	Pose pose;;

	///////////////////////////////
	std::string list_filename;
	assert( truefalseoption( "l" ) );
	stringafteroption( "l", "hey", list_filename );

	utility::io::izstream list_data( list_filename );

	std::string list_line;

	while ( std::getline( list_data(), list_line ) ) {
		std::cout << "read file: " << list_line << std::endl;
		bool ok ( pose_from_pdb( pose, list_line, false, false ) );
		if ( !ok ) {
			std::cout << "input failed: " << list_line << std::endl;
			continue;
		}
		int const nres ( pose.total_residue() );
		FArray3D_float const & Epos ( pose.Eposition() );

		float const max_ca_dis2 ( 30.25 ); // 5.5**2
		for ( int i=1; i<= nres; ++i ) {
			if ( pose.secstruct(i) != 'E' ) continue;
			int j_start=i+1;
			while ( j_start < nres && pose.secstruct( j_start ) == 'E' ) ++j_start;
			for ( int j = std::max( i+3, j_start ); j <= nres; ++j ) {
				if ( pose.secstruct(j) != 'E' ) continue;
				float ca_dis2( 0.0 ),tmp;
				for ( int k=1; k<= 3; ++k ) {
					tmp = ( Epos(k,2,i) - Epos(k,2,j) );
					ca_dis2 += tmp*tmp;
				}
				if ( ca_dis2 < max_ca_dis2 ) {
					float orientation, pleating1, pleating2;
					FArray2D_int pos(2,2);
					get_pairing_geometry( i, j, orientation, pleating1, pleating2 );
					if ( pleating1 * pleating2 > 0.0 ) {
						if ( orientation < 0 ) { // anti-parallel
							if ( pleating1 < 0 ) {
								pos(1,1) = j+1; // n
								pos(2,1) = i-1; // co
								pos(1,2) = i+1;
								pos(2,2) = j-1;
							} else {
								pos(1,1) = i; // n
								pos(2,1) = j; // co
								pos(1,2) = j;
								pos(2,2) = i;
							}
						} else { // parallel
							if ( pleating1 < 0 ) {
								pos(1,1) = j; // n
								pos(2,1) = i-1; // co
								pos(1,2) = i+1;
								pos(2,2) = j;
							} else {
								pos(1,1) = i; // n
								pos(2,1) = j-1; // co
								pos(1,2) = j+1;
								pos(2,2) = i;
							}
						}

						// calc hbond distances
						float mx_dist( 0.0 );
						float mn_dist( 100.0 );
						for ( int ii=1; ii<= 2; ++ii ) {
							float dist2(0.0);
							for ( int k=1; k<= 3; ++k ) {
								tmp = ( Epos(k,1,pos(1,ii)) - Epos(k,5,pos(2,ii) ) );
								dist2 += tmp*tmp;
							}
							mx_dist = std::max( mx_dist, std::sqrt(dist2) );
							mn_dist = std::min( mn_dist, std::sqrt(dist2) );
						}

						std::cout << "PAIR " << list_line << ' ' <<
							i << ' ' << j << ' ' <<
							mn_dist << ' ' << mx_dist << ' ' <<
							orientation << ' ' << pleating1 << ' ' << pleating2 << ' ' <<
							Epos(1,1,i) << ' ' << Epos(2,1,i) << ' ' << Epos(3,1,i) << ' ' <<
							Epos(1,2,i) << ' ' << Epos(2,2,i) << ' ' << Epos(3,2,i) << ' ' <<
							Epos(1,4,i) << ' ' << Epos(2,4,i) << ' ' << Epos(3,4,i) << ' ' <<
							Epos(1,1,j) << ' ' << Epos(2,1,j) << ' ' << Epos(3,1,j) << ' ' <<
							Epos(1,2,j) << ' ' << Epos(2,2,j) << ' ' << Epos(3,2,j) << ' ' <<
							Epos(1,4,j) << ' ' << Epos(2,4,j) << ' ' << Epos(3,4,j) << ' ' <<
							pose.phi  (i) << ' ' <<
							pose.psi  (i) << ' ' <<
							pose.omega(i) << ' ' <<
							pose.phi  (j) << ' ' <<
							pose.psi  (j) << ' ' <<
							pose.omega(j) << std::endl;
					}
				}
			}
		}
	}
	utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
}


///////////////////////////////////////////////////////////////////////////////
void
pose_set_termini( pose_ns::Pose & pose ){
	using namespace aaproperties_pack;
	using namespace termini_ns;
	using namespace param_aa;

	int respos, aan;
	respos = 1;
	aan = pose.res( respos );
	if ( use_N_terminus ) {
		int const n_var = nvar( aan );
		//		bool found_nterm = false;
		for ( int j = 1; j <= n_var; ++j ){
			if ( variant_type( aav_Nterm, aan, j ) ){
				pose.set_res_variant( respos, j);
				break;
			}
		}
	}

	respos = pose.total_residue();
	aan = pose.res( respos );
	if ( use_C_terminus ) {
		int const n_var = nvar( aan );
		//		bool found_nterm = false;
		for ( int j = 1; j <= n_var; ++j ){
			if ( variant_type( aav_Cterm, aan, j ) ){
				pose.set_res_variant( respos, j);
				break;
			}
		}
	}

}

///////////////////////////////////////////////////////////////////////////////
void
initialize_query_pose(
	pose_ns::Pose & pose
)
{

	// read sequence
	if ( !files_paths::query_defined ) {
		std::cout << "STOP:: need 3 args for initialize_query_pose" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	bool seq_undefined = true;
	read_aa( seq_undefined );
	if ( seq_undefined ) {
		std::cout << "STOP:: Need fasta for ai jumping" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// setup start pose
	pose.simple_fold_tree( misc::total_residue );
	for (int i=1; i<= misc::total_residue; ++i ) {
		pose.set_res( i, misc::res(i) );
	}

	//setup pdb info.
	Pdb_info pdb_info;
	pdb_info.pdb_info_from_global();
	pose.set_pdb_information( pdb_info );

	//termini.
	pose_set_termini( pose );
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
// mapping goes from the pose sequence to the native sequence

void
setup_homolog_native(
	pose_ns::Pose const & pose,
	bool & native_exists,
	pose_ns::Pose & native_pose,
	FArray1D_int & mapping
)
{

	native_exists = truefalseoption("n");
	if ( native_exists ) {
		int const nres( pose.total_residue() );
		mapping.dimension( nres );

		pose_from_pdb( native_pose, stringafteroption("n"), false, false );

		if ( native_pose.sequence() != pose.sequence() ) {
			std::cout << "native sequence mismatch: looking for align_file" <<
				std::endl;
			bool const fail( mapping_from_file( stringafteroption("align" ),
				pose.sequence(), native_pose.sequence(), mapping ));
			if ( fail ) {
				std::cout << "STOP: unable to map native sequence." << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		} else {
			for ( int i=1; i<= nres; ++i ) {
				mapping(i) = i;
			}
		}
	}
}


float
get_close_chainbreaks_cycle_ratio()
{
	static float close_chainbreaks_cycle_ratio = 1.0;
	static bool init = { false };

	if ( !init ) {
		realafteroption("close_chainbreaks_cycle_ratio",1.0,
				close_chainbreaks_cycle_ratio);
		init = true;
	}
	return close_chainbreaks_cycle_ratio;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
////////////////////////////
// returns false if we failed
//
// max_loop_length is the longest loop we're willing to search in order
// to close the break
//
// we assume that frags have been loaded, matching the pose sequence
//

bool
close_chainbreaks(
	pose_ns::Pose & pose,
	const pose_ns::Score_weight_map & eval_weight_map,
	const int max_loop_size,
	const int min_good_loops,
	const float vdw_delta // was hard-coded at 0.5 in rebuild_loops
)
{
	using namespace pose_ns;

	// params
	const int min_breakout_good_loops( std::max( min_good_loops, 5 ) );
	const int min_loop_size(6);

	const float cycle_ratio = get_close_chainbreaks_cycle_ratio(); // default is just 1

	// local pose copy:
	Pose loop_pose, best_pose;

	// close the breaks, one at a time
	// make a copy of the cutpoint array at the start:
	// used to be const ref but that's a bug! after the pose fold_tree changes!
	int num_fold_tree_cutpoint;
	FArray1D_int fold_tree_cutpoint
		( pose.fold_tree().get_fold_tree_cutpoint( num_fold_tree_cutpoint ) );
	const int nres( pose.total_residue() );

	bool fail(false);

	// loop through the existing cutpoints, closing one at a time:
	for ( int ncut=1; ncut <= num_fold_tree_cutpoint; ++ncut ) {
		const int cutpoint ( fold_tree_cutpoint( ncut ) );
		std::cout << "close chainbreak at position= " << cutpoint << std::endl;

		// for choosing the best loop, and jumping out once we've seen
		// enough good ones:
		pose.score( eval_weight_map );
		const float start_score ( pose.get_0D_score( SCORE ) );
		const float start_vdw_score ( pose.get_0D_score( VDW ) );
		const float really_bad_score ( 10000.0 );
		float best_score ( really_bad_score );
		int good_loop_count (0);

		//
		const Fold_tree & old_fold_tree( pose.fold_tree() );
		const FArray1D_bool & is_jump_point( old_fold_tree.get_is_jump_point() );

		if ( is_jump_point( cutpoint ) && is_jump_point( cutpoint+1 ) ) {
			std::cout << "cutpoint bounded by jump_points - cant close" << std::endl;
			fail = true;
			break;
		}
		// figure out where the loop can safely start and stop
		// these are loop residues, ie torsions at these positions will be moving
		// dont want either of these residues to be a jump_point
		int min_loop_begin( cutpoint+1 );
		int max_loop_end  ( cutpoint );
		while ( min_loop_begin > 2      && !is_jump_point( min_loop_begin-1 ) )
			--min_loop_begin;
		while ( max_loop_end   < nres-1 && !is_jump_point( max_loop_end+1 ) )
			++max_loop_end;
		const int actual_max_loop_size( max_loop_end - min_loop_begin + 1 );
		assert( min_loop_begin <= max_loop_end );


		// debug
		if ( !is_jump_point( min_loop_begin-1 ) ||
				 !is_jump_point( max_loop_end+1   ) ) {
			std::cout << "close_breaks:: funny tree!!\n" << old_fold_tree;
			fail = true;
			break;
		}

		bool loop_closed( false );

		// construct the new tree formed when we glue this cutpoint and
		// delete a single jump
		Fold_tree new_fold_tree;
		new_fold_tree = old_fold_tree;
		std::cout << "old_fold_tree: " << old_fold_tree;
		{ // scope
			const int jump_pos1( min_loop_begin-1 );
			const int jump_pos2( max_loop_end  +1 );
			const int num_jump( old_fold_tree.get_num_jump() );
			const FArray2D_int & jump_point( old_fold_tree.get_jump_point() );

			assert( is_jump_point( jump_pos1 ) && is_jump_point( jump_pos2 ) );
			if ( jump_pos1 < cutpoint )
				new_fold_tree.delete_unordered_edge( jump_pos1, cutpoint, -1);
			if ( cutpoint+1 < jump_pos2 )
				new_fold_tree.delete_unordered_edge( cutpoint+1, jump_pos2, -1);
			new_fold_tree.add_edge( jump_pos1, jump_pos2, -1 );

			// I think there may be more than one jump which
			// we could delete. This just chooses the first one
			for ( int i=1; i<= num_jump; ++i ) {
				// mark as "deleted" for the purposes of connectivity checking
				new_fold_tree.update_edge_label( jump_point(1,i), jump_point(2,i), i,
																				 0 ); // label with 0
				if ( new_fold_tree.connected() ) {
					// safe to cut this edge
					new_fold_tree.delete_unordered_edge( jump_point(1,i),
																							 jump_point(2,i), 0 );
				} else {
					// dont cut this edge after all
					new_fold_tree.update_edge_label( jump_point(1,i),
																					 jump_point(2,i), 0, i );
				}
			}

			if ( new_fold_tree.connected() ) {
				// we deleted a jump so the jump label numbers may be screwed up
				// note that this will screw up the jump_transform mapping
				new_fold_tree.renumber_jumps();
				// plus, there may be vertices in the tree which are not jumps and
				// not at a break
				new_fold_tree.delete_extra_vertices();
				new_fold_tree.reorder( 1 );
			}
		} // scope

		// this should not happen:
		if ( !( new_fold_tree.connected() && new_fold_tree.check_fold_tree() ) ) {
			std::cout << "close_breaks:: funny tree2\n" << new_fold_tree;
			fail = true;
			break;
		}

		///////////////////////////////////////////////////////////////////////////
		// try to close the chainbreak:
		//
		// start with a small loop, gradually increase loop size
		//
		// also, for each loop size, slide through all possible loop positions
		// that span the cutpoint and lie with min_loop_begin, max_loop_end
		// traverse the windows in order of decreasing loop content based on
		// the fragments
		//
		for ( int loop_size = min( actual_max_loop_size, min_loop_size );
					loop_size <= min( actual_max_loop_size,max_loop_size); ++loop_size ){

			// sort the window positions by loop content in frags
			// go from highest to lowest:

			const FArray1D_float & loop_fraction ( get_loop_fraction( nres ) );
			typedef std::list< std::pair< float, int > > Float_int_list;
			Float_int_list window_list;
			for ( int ii=0; ii<= loop_size; ++ii ) {
				const int loop_begin( cutpoint - loop_size + ii + 1);
				const int loop_end  ( cutpoint + ii );
				assert( loop_begin <= cutpoint+1 &&
								loop_end   >= cutpoint &&
								loop_end   == loop_begin + loop_size - 1 );
				if ( loop_begin < min_loop_begin || loop_end > max_loop_end ) continue;

				float f (0);
				for ( int i=loop_begin; i<= loop_end; ++i ) {
					f += loop_fraction(i);
				}
				window_list.push_back( std::make_pair( f, loop_begin ) );
			}
			window_list.sort();
			window_list.reverse();

			// now loop over different loop positions
			for ( Float_int_list::const_iterator w_it=window_list.begin(),
							w_it_end = window_list.end(); w_it != w_it_end; ++w_it ) {
				const int loop_begin( w_it->second );
				const int loop_end( loop_begin + loop_size - 1 );

				/////////////////////
				// setup the new pose
				loop_pose = pose;

				insert_init_frag( loop_pose, loop_begin, loop_size );
				pose.insert_ideal_bonds( loop_begin, loop_end );

				// collect a list of closed fragments:
				std::vector< Fragment > fragment_list;
				{
					// close with scored fragment insertions
					// setup the score function:
					Score_weight_map weight_map;
					weight_map.set_weight( VDW, 1.0 );
					weight_map.set_weight( ENV, 1.0 );
					weight_map.set_weight( PAIR, 1.0 );
					weight_map.set_weight( CHAINBREAK, 1.0 );
					weight_map.set_weight( CHAINBREAK_OVERLAP, 1.0 );

					const int frag_offset ( 0 ); // no sequence mapping
					bool const do_ccd_moves( loop_size >= 10 ? true : false);

					if (!runlevel_ns::benchmark) start_timer();
					scored_frag_close( weight_map, loop_pose, loop_begin, loop_end, 3,
						frag_offset, static_cast< int > (100*cycle_ratio),
						static_cast< int > (20*loop_size*cycle_ratio),
						do_ccd_moves, fragment_list );
					std::cout << "scored_frag_close:: found " << fragment_list.size() <<
						" frags";
					if (!runlevel_ns::benchmark) {
						stop_timer( std::cout );
					} else {
						std::cout << std::endl;
					}
					// found that ccdmoves for shorter loops didnt really increase the
					// closure rate, took about twice as long
				}


				if ( loop_size <= 10 ) {
					// try fragment closure w/o score, just deviations
					// here we assume that fragments have been read in for the
					// target sequence
					const int frag_offset ( 0 );
					const int cycles1(static_cast< int > (400*cycle_ratio)),
						cycles2(static_cast< int > (10*loop_size*cycle_ratio)),
						big_num_fragments(100), little_num_fragments(50),
						frag_close_ccd_cycles(50);

					frag_close( loop_pose, loop_begin, loop_size, cutpoint,
						frag_offset, cycles1, cycles2, big_num_fragments,
						little_num_fragments, frag_close_ccd_cycles, fragment_list );
					std::cout << "frag_close:: now found " << fragment_list.size() <<
						" frags" << std::endl;
				}

				if ( fragment_list.size() > 0 ) {
					// We found at least one fragment that perfectly closes the loop
					//
					// try inserting all fragments into the pose, evaluate score,
					// rmsd to starting pose
					//
					FArray1D_bool fixed_residues( nres, true );
					for ( int i=loop_begin; i<= loop_end; ++i ) fixed_residues(i) = false;

					loop_pose.set_fold_tree( new_fold_tree );

					// the numbering of jumps in new_fold_tree may not match current
					// we used to have to do this, now it's done automatically by
					// set_fold_tree
					//loop_pose.jumps_from_position();

					for ( std::vector< Fragment >::const_iterator
									it = fragment_list.begin(), it_end = fragment_list.end();
								it != it_end; ++it ) {
						it->insert( loop_pose, loop_begin );

						// check rmsd over un-moved region
						const float check_rmsd
							( CA_rmsd_by_subset(loop_pose, pose, fixed_residues ) );
						if ( check_rmsd > 0.1 ) {
							std::cout << "WARNING:: big loop_pose refold rmsd= " <<
								check_rmsd << std::endl;
						}
						//assert( check_rmsd < 0.1 );
						loop_pose.score( eval_weight_map );
#ifndef BOINC
						std::cout << "noloop refold: check_rmsd= " <<
							check_rmsd << ' ' << loop_pose.show_scores() << std::endl;
#endif
						const float score ( loop_pose.get_0D_score ( SCORE ) );
						const float vdw_score ( loop_pose.get_0D_score ( VDW ) );
						if ( vdw_score < start_vdw_score + vdw_delta ) {
							std::cout << "good loop " << vdw_score << ' ' <<
								start_vdw_score << std::endl;
							++good_loop_count;
						}

						if ( best_score > score ) {
							best_score = score;
							best_pose = loop_pose;
							std::cout << "best_accept: new-best= " << best_score <<
								" start_score= " << start_score << std::endl;
						}
					}
				}

				std::cout << "slide window: " << loop_begin << ' ' <<
					loop_size << ' ' << best_score << ' ' << good_loop_count <<
					std::endl;

				if ( good_loop_count >= min_breakout_good_loops ) {
					// skip out of the slide-window loop if we've seen enough
					// non-clashing loops; just an optimization
					std::cout << "seen " << good_loop_count << " good loops; done\n";
					break;
				}
			} // slide loop

			if ( good_loop_count >= min_good_loops &&
					 best_score < really_bad_score-1 ) {
				std::cout << "closed the loop!!! score,start_score,vdw,old_vdw " <<
					best_pose.get_0D_score ( SCORE ) << ' ' << start_score << ' ' <<
					best_pose.get_0D_score ( VDW ) << ' ' <<
					pose.get_0D_score ( VDW ) << std::endl;
				pose = best_pose;
				loop_closed = true;
				break;
			}
		} // loop size

		if ( !loop_closed ) {
			fail = true;
			break;
		}
	} // cutpoints

	if ( !fail ) {
		assert( pose.num_jump() == 0 );
		return true;
	} else {
		return false;
	}
}

///////////////////////////////////////////////////
// close chainbreaks on an input silent-file
//
// mode = "close_chainbreaks"
//
// called from pose_main

void
main_close_chainbreaks()
{
	using namespace pose_ns;
	using namespace silent_io;

	// these params control the loop_closing protocol
	// if vdw<start_vdw + vdw_delta, a loop is counted as "good"
	// we escape out of the loop-size loop at the end of a round
	// if we've seen min_good_loops good loops
	//
	// otherwise we keep increasing the loop-size until we close
	// or reach max_loop_size
	//
	const int max_loop_size( 12 );
	const int min_good_loops( 3 );
	const float vdw_delta( 0.5 );

	// allocate space
	Pose pose;

	// defines length, sequence of pose
	initialize_query_pose( pose );

	// read native
	bool native_exists;
	Pose native_pose;
	FArray1D_int native_mapping; // from pose to native
	setup_homolog_native( pose, native_exists,native_pose,native_mapping );

	// read fragments -- after reading native in case MAX_RES increases!
	read_fragments( pose.total_residue() );

	// scoring function: score4
	Score_weight_map weight_map( score4 );

	// read silent file
	Silent_file_data decoys( stringafteroption("s") );
	if ( !decoys.size() ) {
		std::cout << "couldnt open silent-file!!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( decoys.nres() != pose.total_residue() ||
			 decoys.sequence() != pose.sequence() ) {
		std::cout << "main_close_chainbreaks:: query/silent-file mismatch: " <<
			decoys.nres() << ' ' << pose.total_residue() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// open silent-out file
	Silent_out out( files_paths_pdb_out_prefix()+"close_chainbreaks.out" );

	for ( Silent_file_data::const_iterator it = decoys.begin(),
					it_end = decoys.end(); it != it_end; ++it) {
		std::string decoy_name( it->first + "_closed" );

		// already started?
		if ( !out.start_decoy( decoy_name ) ) {
			std::cout << "decoy exists: " << decoy_name << std::endl;
			continue;
		}

		// copy from silent-data
		it->second->fill_pose( pose );
		pose.score( weight_map );
		const float start_score( pose.get_0D_score( SCORE ) );

		start_timer("close_cb");
		const bool ok( close_chainbreaks( pose, weight_map,
			max_loop_size, min_good_loops, vdw_delta ) );

		if ( !ok ) {
			std::cout << "chainbreak closure failed!! " << decoy_name << std::endl;
			continue;
		}

		// output
		pose.score( weight_map );
		pose.set_extra_score("START_SCORE", start_score);
		pose.set_extra_score("TIME", get_timer("close_cb") );
		if ( native_exists ) {
			calc_homolog_rms( pose, native_pose, native_mapping );
		}

		out.write( decoy_name, pose );
	}

}

///////////////////////////////////////////////////////////////////////////////
////////////////////////////
// Which of the possible pairings is allowed by the barcode SSPAIR?
//

int
get_allowed_pairings_from_barcode(
	 pairing_ns::Pairing_list & protein_pairings
	 ){
	// Should probably make this information (whether allowed by barcode)
	// and this filtering method a part of the Pairing method, to be truly object oriented. Hmm.
	using namespace barcode_energy;

	static bool init = false;
	static bool filter_jumps_with_barcode;
	static bool filter_jumps_with_bonus  = false;

	int num_jumps_with_barcode = 0;

	if (!init){
		init = true;
		// This flag (filter_jumps_with_barcode) tells this routine to remove any pairings
		// that are given penalties in the barcode file (e.g., SSPAIR 4.0 A 5 20 61 80).
		filter_jumps_with_barcode =  truefalseoption("filter_jumps_with_barcode");
	// This flag (filter_jumps_with_bonus)  tells this routine to *stringently* filter the pairings --
	//  an actual bonus needs to be assigned in the barcode file (e.g., SSPAIR -4.0 A 5 20 45 60) for
	//  a pairing to be accepted. It will get triggered if there are any bonus features in the barcode.
		num_jumps_with_barcode = 1;
		filter_jumps_with_bonus   =  truefalseoption("filter_jumps_with_bonus");
		if (filter_jumps_with_bonus) filter_jumps_with_barcode = true;
	}

	if (filter_jumps_with_bonus){
		num_jumps_with_barcode = barcode_feature_set().num_bonus_sspair_constraints() + barcode_feature_set().num_bonus_beta_pair_constraints();
	}

	int num_protein_pairings = protein_pairings.size();

	bool const barcode_sspair_constraints_exist
		( barcode_exist() && barcode_feature_set().sspair_constraints_exist() );

	bool const barcode_beta_pair_constraints_exist
		( barcode_exist() && barcode_feature_set().beta_pair_constraints_exist() );

	if ((barcode_sspair_constraints_exist || barcode_beta_pair_constraints_exist) && filter_jumps_with_barcode) {
		pairing_ns::Pairing_list new_protein_pairings;

		for (int p = 0; p < num_protein_pairings; ++p) {

			float check_barcode_score = 0;
		  if(barcode_sspair_constraints_exist) {
				reset_barcode_sspair_limits();
				// To match definition used in barcode_sspair_constraint_score
				float const theta ( protein_pairings[p].orientation == 1 ? 175.0 : 5.0 );
				int const dimer2_offset( theta > 90.0 ? 1 : 0 );
				// Following from structure.cc SSpairscore calculation:
				check_barcode_score += barcode_sspair_constraint_score(
					 protein_pairings[p].pos1, protein_pairings[p].pos2 + dimer2_offset, theta, -1.0);
			}

			float beta_pair_score = 0.0;
			if(barcode_beta_pair_constraints_exist)
				check_barcode_score += barcode_beta_pair_energy(
						protein_pairings[p].pos1, protein_pairings[p].pos2, 
						protein_pairings[p].orientation, protein_pairings[p].pleating );

			if (check_barcode_score + beta_pair_score < 0.0){ // Barcode has a bonus for this pairing
				new_protein_pairings.push_back( protein_pairings[p] );
			}	else if (check_barcode_score + beta_pair_score <= 0.0 & !filter_jumps_with_bonus){ // Barcode has no penalty defined for this pairing
				new_protein_pairings.push_back( protein_pairings[p] );
			}
		}
		protein_pairings = new_protein_pairings;
	}

	int num_allowed_pairings = protein_pairings.size();

	std::cout << "Number of jumps that passed barcode filter: " << num_allowed_pairings <<
		" out of " << num_protein_pairings << std::endl;

	return num_jumps_with_barcode;

}



bool close_chainbreaks_after_jumping(	pose_ns::Pose & pose ){
	using namespace pose_ns;

	const int max_loop_size( 12 );
	const int min_good_loops( 3 );
	const float vdw_delta( 0.5 );

	Score_weight_map weight_map( score4 );
	pose.score( weight_map );
	bool chainbreaks_closed = close_chainbreaks( pose, weight_map,
																 max_loop_size, min_good_loops, vdw_delta );

	score_set_evaluate_all_terms(true);
	pose.score( weight_map ); // output-score
	score_set_evaluate_all_terms(false);

	return chainbreaks_closed;
}

bool apply_filters_after_jumping(	pose_ns::Pose & pose ){
	using namespace pose_ns;

	bool jump_accepted(true);
	pose_to_misc(pose);
	set_pose_flag( false );
	setup_filters();
	files_paths::idealized_structure = true;
	monte_carlo_reset();
	initialize_maps();
	apply_filters_wrapper( false /*fullatom*/, jump_accepted );
	set_pose_flag( true );

	return jump_accepted;
}

void relax_and_output_after_jumping( pose_ns::Pose & pose ){
	bool idealize_fail ( false );
//rhiju  Originally, I tried idealizing with full chain idealize
//rhiju  and results got *worse* on a test with 1bk2 (SH3 domain).
//rhiju  I don't know why. Anyway, turn off idealize for now.
// 	bool const save_fullatom_flag = get_fullatom_flag();
// 	set_pose_flag( true );
// 	pose_to_misc( pose );
// 	set_pose_flag( false );
// 	monte_carlo_reset();

// 	set_fullatom_flag(false);
// 	bool const save_no_faidl = get_no_faidl();
// 	set_no_faidl( true );
// 	idealize( idealize_fail );
// 	set_fullatom_flag( save_fullatom_flag );
// 	monte_carlo_reset();
// 	set_no_faidl( save_no_faidl );
// 	if( !files_paths::idealized_structure ) idealize_fail = true;

// The other option (pose idealize):
// pose.set_fullatom_flag( /* fullatom */ true, false); //Weird, I get errors if not fullatom.
// idealize_pose( pose, /*ignore_allow_move*/ true);
// files_paths::idealized_structure = true;

	pose_to_misc( pose );
	set_pose_flag( false );
	files_paths::idealized_structure = true;

	//OK, ready to relax
	if (!idealize_fail){
		monte_carlo_reset();
		initialize_maps();
		relax_structure();
		bool accepted = true;
		output_decoy( accepted);
	}

	set_pose_flag( true );
}

void
idealize_pose_before_relax( pose_ns::Pose & pose )
{
	pose.simple_fold_tree( pose.total_residue() );
	//		pose.set_fullatom_flag( true, false );
	pose.dump_pdb( "blah1.pdb" );
	std::cout << "IDEAL_BACKBONE" << pose.ideal_backbone() << std::endl;
	pose.update_backbone_bonds_and_torsions();
	std::cout << "IDEAL_BACKBONE" << pose.ideal_backbone() << std::endl;
	pose.dump_pdb( "blah2.pdb" );
	idealize_pose( pose, /*ignore_allow_move*/ true);
	std::cout << "IDEAL_BACKBONE" << pose.ideal_backbone() << std::endl;
	pose.dump_pdb( "blah3.pdb" );
	files_paths::idealized_structure = true;

	monte_carlo_reset();
}


void pose_relax_after_jumping( pose_ns::Pose & pose ){

	static const bool do_idealize = truefalseoption("idealize_before_relax");


	initialize_maps();
	pose.set_fullatom_flag( true );
	int lj_ramp_cycles = 8;
	int const cycles_per_residue = { 1 };
	int cycles = static_cast<int> ( cycles_per_residue * pose.total_residue() * get_farlx_cycle_ratio() );
	if (truefalseoption("benchmark")){
		lj_ramp_cycles = 1;
		cycles = 1;
	}
	static bool const vary_bond_angles = truefalseoption("vary_bond_angles");
	static float const filter1  = realafteroption("filter1" ,9999999999.);
	static float const filter1a = realafteroption("filter1a",9999999999.);
	static float const filter1b = realafteroption("filter1b",9999999999.);
	static float const filter2  = realafteroption("filter2" ,9999999999.);

	pose_ns::Score_weight_map weight_map( score12 );
	//Moved out from fast_relax_pose.
	weight_map.set_weight( pose_ns::CHAINBREAK,         1.0);
	weight_map.set_weight( pose_ns::CHAINBREAK_OVERLAP, 1.0);

	pose.set_allow_bb_move( true );

	if ( do_idealize ) idealize_pose_before_relax( pose );

	std::cout << "Starting full atom relax with the pose ... " << std::endl;
	fast_relax_pose( pose, weight_map, lj_ramp_cycles, cycles, "tmp", 1.0, vary_bond_angles,
									 filter1, filter1a, filter1b, filter2);
}


