// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 23432 $
//  $Date: 2008-06-24 16:25:52 +0300 (Tue, 24 Jun 2008) $
//  $Author: yab $


// Rosetta Headers
#include "jumping_util.h"
#include "aaproperties_pack.h"
#include "angles.h"
#include "current_pose.h"
#include "dna.h"
#include "dock_loop_ensemble_ns.h"
#include "fullatom.h"
#include "fragments.h"
#include "fragments_ns.h"
#include "jumping_ns.h"
#include "jumping_diagnostics.h"
#include "jumping_minimize.h"
#include "jumping_pairings.h"
#include "jumping_refold.h"
#include "maps.h"
#include "misc.h"
#include "namespace_best_position_init_common.h"
#include "param.h"
#include "param_aa.h"
#include "pose.h"
#include "random_numbers.h"
#include "refold.h"
#include "refold_ns.h"
#include "ssblocks.h"
#include "start.h"
#include "util_vector.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/formatted.o.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/conversions.hh>
#include <numeric/trig.functions.hh>
#include <numeric/xyzVector.hh>
#include <numeric/xyzMatrix.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>

// C++ Headers
#include <cstdlib>
#include <iostream>
#include <fstream>

// Namespaces

namespace use_rosetta_torsions_common {
	bool use_rosetta_torsions = { false };
}

// DANGER DANGER DANGER
using namespace pose_ns;

//------------------------------------------------------------------------------
// move the N position along the N-CA bond to match the current value
// for N-CA bond length

void
slide_N(
	int const seqpos,
	FArray1Da_float n_xyz,
	FArray1Da_float ca_xyz,
	FArray1Da_float n_xyz_out
)
{
	using namespace refold_ns;
	using namespace start;

	n_xyz.dimension( 3 );
	ca_xyz.dimension( 3 );
	n_xyz_out.dimension( 3 );

// local:
	FArray1D_float v( 3 );
	FArray1D_float u( 3 );

	subvec(n_xyz,ca_xyz,v);
	unitvec(v,u);

	float bond_length;
	assert( refold_check_current_pose() );
	if ( !refold_get_current_pose().ideal_backbone() ) {
		bond_length = refold_get_current_pose().bonds().D()(n,c2n,seqpos);
// 	} else if ( use_st_bond() ) {
// 		bond_length = st_D(n,c2n,seqpos);
	} else {
		bond_length = D(n,c2n);
	}

	for ( int k = 1; k <= 3; ++k ) {
		n_xyz_out(k) = ca_xyz(k) + u(k) * bond_length;
	}

}

//------------------------------------------------------------------------------
// uses:
// * the n_xyz position
// * the n->ca bond vector
// * the n,ca,c oriented plane
//
// DANGER:: pos is the position one residue after the C, ie the position to
// whom phi,n,ca,c belong!!!

void
build_C_coords(
	int const pos,
	float const phi,
	FArray1Da_float n_xyz,
	FArray1Da_float ca_xyz,
	FArray1Da_float c_xyz,
	FArray1Da_float c_xyz_out
)
{
	using namespace refold_ns;
	using namespace start;

	n_xyz.dimension( 3 );
	ca_xyz.dimension( 3 );
	c_xyz.dimension( 3 );
	c_xyz_out.dimension( 3 );

// local:
	FArray2D_float X( 3, 3 );
	FArray2D_float M( 3, 3 );

	subvec( n_xyz,ca_xyz,X(1,ph)); // order is tricky: we are folding c2n
	subvec(ca_xyz, c_xyz,X(1,ps)); // in the call to build_atom!

// the X-axis of this coord system (M(*,1)) is parallel to
// X(1,ph), ie along the n->ca bond vector

	refold_coord_sys(X(1,ps),X(1,ph),M(1,1),M(1,2),M(1,3));

	bool const pose_refold( refold_check_current_pose() );
	if ( !pose_refold && use_st_bond() ) {
 		build_atom(M,n_xyz, st_cT(c,c2n,pos-1),st_sT(c,c2n,pos-1),phi,
 		 st_D(c,c2n,pos-1),c_xyz_out,X(1,om));
	} else if ( pose_refold && !refold_get_current_pose().ideal_backbone() ) {
		const bonds_class::Bonds & pose_bonds
			( refold_get_current_pose().bonds() );
		build_atom( M, n_xyz,
								pose_bonds.cT()(c,c2n,pos-1),
								pose_bonds.sT()(c,c2n,pos-1), phi,
								pose_bonds.D ()(c,c2n,pos-1), c_xyz_out, X(1,om));
	} else {
		// use ideal bonds
		build_atom(M,n_xyz,cT(c,c2n),sT(c,c2n),phi, D(c,c2n),c_xyz_out,
		 X(1,om));
	}

}
//------------------------------------------------------------------------------
// uses:
// * the n_xyz position
// * the n->ca bond vector
// * the n,ca,c oriented plane

// same as previous routine but uses only ideal bond geometry, so no
// need to pass in pos --> no use of any stored info
//

void
build_ideal_C_coords(
	float const phi,
	FArray1Da_float n_xyz,
	FArray1Da_float ca_xyz,
	FArray1Da_float c_xyz,
	FArray1Da_float c_xyz_out
)
{
	using namespace refold_ns;

	n_xyz.dimension( 3 );
	ca_xyz.dimension( 3 );
	c_xyz.dimension( 3 );
	c_xyz_out.dimension( 3 );

// local:
	FArray2D_float X( 3, 3 );
	FArray2D_float M( 3, 3 );

	subvec( n_xyz,ca_xyz,X(1,ph)); // order is tricky: we are folding c2n
	subvec(ca_xyz, c_xyz,X(1,ps)); // in the call to build_atom!

// the X-axis of this coord system (M(*,1)) is parallel to
// X(1,ph), ie along the n->ca bond vector

	refold_coord_sys(X(1,ps),X(1,ph),M(1,1),M(1,2),M(1,3));

	build_atom(M,n_xyz,cT(c,c2n),sT(c,c2n),phi, D(c,c2n),c_xyz_out,X(1,om));
}

//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// same as above, but generates N,CA of next position forward in sequence

void
build_ideal_N_CA_coords(
	float const psi,
	float const omega,
	FArray1Da_float n_xyz,
	FArray1Da_float ca_xyz,
	FArray1Da_float c_xyz,
	FArray1Da_float n_xyz_out,
	FArray1Da_float ca_xyz_out
)
{
	using namespace refold_ns;

	n_xyz.dimension( 3 );
	ca_xyz.dimension( 3 );
	c_xyz.dimension( 3 );
	n_xyz_out.dimension( 3 );

// local:
	FArray2D_float X( 3, 3 );
	FArray2D_float M( 3, 3 );

	// N
	subvec(ca_xyz, n_xyz,X(1,ph)); // order is tricky: we are folding n2c
	subvec( c_xyz,ca_xyz,X(1,ps)); // in the call to build_atom!

	refold_coord_sys(X(1,ph),X(1,ps),M(1,1),M(1,2),M(1,3));
	build_atom( M, c_xyz, cT(n,n2c), sT(n,n2c), psi, D(n,n2c),
							n_xyz_out, X(1,om));
	// CA
	refold_coord_sys(X(1,ps),X(1,om),M(1,1),M(1,2),M(1,3));
	build_atom( M, n_xyz_out, cT(ca,n2c), sT(ca,n2c), omega, D(ca,n2c),
							ca_xyz_out, X(1,ph) );
}

//------------------------------------------------------------------------------
// uses non-ideal geometry if pose.use_pose_bonds
//
// DANGER: pos is the position to whom psi, omega, and n,ca,c_xyz belong
// ie one residue before the residue for whom we are actually building
// coordinates!!

void
build_N_CA_coords(
	int const pos,
	float const psi,
	float const omega,
	FArray1Da_float n_xyz,
	FArray1Da_float ca_xyz,
	FArray1Da_float c_xyz,
	FArray1Da_float n_xyz_out,
	FArray1Da_float ca_xyz_out
)
{
	//using namespace start;
	using namespace refold_ns;

	n_xyz.dimension( 3 );
	ca_xyz.dimension( 3 );
	c_xyz.dimension( 3 );
	n_xyz_out.dimension( 3 );

// local:
	FArray2D_float X( 3, 3 );
	FArray2D_float M( 3, 3 );

	subvec(ca_xyz, n_xyz,X(1,ph)); // order is tricky: we are folding n2c
	subvec( c_xyz,ca_xyz,X(1,ps)); // in the call to build_atom!

	bool const pose_refold( refold_check_current_pose() );
	assert( pose_refold || !use_st_bond() );

	if ( pose_refold && !refold_get_current_pose().ideal_backbone() ) {
		const bonds_class::Bonds & pose_bonds
			( refold_get_current_pose().bonds() );
		int const bonds_pos( pos+1 );
		// N
		refold_coord_sys(X(1,ph),X(1,ps),M(1,1),M(1,2),M(1,3));
		build_atom( M, c_xyz, pose_bonds.cT()(n,n2c,bonds_pos),
			pose_bonds.sT()(n,n2c,bonds_pos), psi, pose_bonds.D()(n,n2c,bonds_pos),
			n_xyz_out, X(1,om));
		// CA
		refold_coord_sys(X(1,ps),X(1,om),M(1,1),M(1,2),M(1,3));
		build_atom( M, n_xyz_out, pose_bonds.cT()(ca,n2c,bonds_pos),
			pose_bonds.sT()(ca,n2c,bonds_pos), omega, pose_bonds.D()(ca,n2c,bonds_pos),
			ca_xyz_out, X(1,ph) );
	} else {
		// N
		refold_coord_sys(X(1,ph),X(1,ps),M(1,1),M(1,2),M(1,3));
		build_atom( M, c_xyz, cT(n,n2c), sT(n,n2c), psi, D(n,n2c),
								n_xyz_out, X(1,om));
		// CA
		refold_coord_sys(X(1,ps),X(1,om),M(1,1),M(1,2),M(1,3));
		build_atom( M, n_xyz_out, cT(ca,n2c), sT(ca,n2c), omega, D(ca,n2c),
								ca_xyz_out, X(1,ph) );
	}
}

///////////////////////////////////////////////////////////////////////////////
void
get_ncac(
	FArray2Da_float pos,
	numeric::xyzMatrix_double & p
)
{
	pos.dimension( 3, 5 );
	using namespace numeric;

	xyzVector_double n( &pos(1,1) );
	xyzVector_double ca( &pos(1,2) );
	xyzVector_double c( &pos(1,4) );

	p = xyzMatrix_double::cols( n, ca, c );

}

///////////////////////////////////////////////////////////////////////////////
numeric::xyzMatrix_double
get_ncac
(
	FArray2Da_float pos
)
{
	pos.dimension( 3, 5 );
	using namespace numeric;

	xyzVector_double n( &pos(1,1) );
	xyzVector_double ca( &pos(1,2) );
	xyzVector_double c( &pos(1,4) );

	return xyzMatrix_double::cols( n, ca, c );
}

///////////////////////////////////////////////////////////////////////////////
void
get_ncac_full_coord(
	FArray2Da_float pos,
	numeric::xyzMatrix_double & p
)
{
	pos.dimension( 3, 3 );

	p = numeric::xyzMatrix_double::cols( &pos(1,1) );
// 	for ( int i = 1; i <= 3; ++i ) {
// 		p(i,1) = double(pos(i,1)); // N
// 		p(i,2) = double(pos(i,2)); // CA
// 		p(i,3) = double(pos(i,3)); // C
// 	}
}

//------------------------------------------------------------------------------
// the x-axis of this coordinate system is along the p(*,2) -> p(*,1) bond vector
// the y-axis is in the p(*,*) plane, with positive dot product to p(*,2) -> p(*,3) vector

// m(*,1) is the x-axis unit vector of this new coord system expressed in
// the absolute coordinates defining the positions p(*,j)
// this corresponds to a column in the matrix m (non-CHarlie convention)
//
// thus multiplication by m can be interpreted as taking a vector in the
// local coordinate system defined by m and expressing it in the absolute
// coordinate system in which the coords p are defined
//
// by multiplication I mean non-charlie multiplication (row,col) indexing
//
// likewise, multiplication by m^t = m inverse can be thought of
// as expressing a vector given in absolute coords in terms of the
// local coordinate system defined by m
//

void
get_coordinate_system(
	numeric::xyzMatrix_double const & p, //FArray2Da_double p, // input
	numeric::xyzMatrix_double & m //FArray2Da_double m // output
)
{
	using namespace numeric;

	xyzVector_double a1 = p.col_x() - p.col_y();
	xyzVector_double a2 = p.col_z() - p.col_y();
	a1.normalize();
	xyzVector_double a3 = cross( a1, a2 );
	a3.normalize();
	a2 = cross( a3, a1 );

	m = xyzMatrix_double::cols( a1, a2, a3 );
}


///////////////////////////////////////////////////////////////////////////////
// this could be faster if necessary, probably too much construction+copying
numeric::xyzMatrix_double
get_coordinate_system(
	FArray2Da_float Epos
)
{
	Epos.dimension( 3, param::MAX_POS );
	numeric::xyzMatrix_double p;
	numeric::xyzMatrix_double m;
	get_ncac( Epos, p );
	get_coordinate_system( p, m );
	return m;
}


///////////////////////////////////////////////////////////////////////////////

void
get_coordinate_system(
	FArray2Da_float Epos,
	numeric::xyzMatrix_double & m
)
{
	Epos.dimension(3, param::MAX_POS );

	numeric::xyzMatrix_double p;
	get_ncac( Epos, p );
	get_coordinate_system( p, m );
}

////////////////////////////////////////////////////////////////////////////////////////
// rebuilds the backbone for position pos with correct geometry
// this assumes that temporary N,CA,C coords already exist in Eposition
// the CA is kept fixed, N is slid along the N->CA bond to the correct distance
// C is moved in the N,CA,C plane

void
update_geometry( int const pos )
{
	FArray1D_float n_xyz(3), ca_xyz(3), c_xyz(3), tmp_n_xyz(3), tmp_c_xyz(3);

	for ( int k=1; k<= 3; ++k ) {
		tmp_n_xyz(k) = misc::Eposition(k,1,pos);
		ca_xyz   (k) = misc::Eposition(k,2,pos);
		tmp_c_xyz(k) = misc::Eposition(k,4,pos);
	}

	slide_N( pos, tmp_n_xyz, ca_xyz, n_xyz); // input,input,input,output

// now build the C for position pos-1 to get a stub
// this uses the n_xyz position, the n->ca bond vector, and the n,ca,c plane
//
	build_C_coords( pos, misc::phi(pos), n_xyz, ca_xyz, tmp_c_xyz, c_xyz );

	build_backbone(1, // 1 ==> n2c
								 c_xyz,n_xyz,ca_xyz, 1, // == number of residues to build
								 misc::phi(pos),misc::psi(pos),misc::omega(pos),
								 misc::res(pos), misc::Eposition(1,1,pos), misc::centroid(1,pos),
								 c_xyz, n_xyz, ca_xyz,  // outgoing stub
								 pos);
}

//------------------------------------------------------------------------------
// build coordinates for c,n,ca stub
// c of position stub_res-1
// n,ca of position stub_res
//
// if ( dir == n2c ) assumes coords for stub_res-1 exist in Eposition
// if ( dir == c2n ) assumes coords for stub_res   exist in Eposition

void
build_stub_old(
	int const stub_res,
	int const dir,
	FArray1Da_float c_xyz,
	FArray1Da_float n_xyz,
	FArray1Da_float ca_xyz
)
{
	c_xyz.dimension(3);
	n_xyz.dimension(3);
	ca_xyz.dimension(3);

	static int const n(1), ca(2), c(4), n2c(1);

	if ( dir == n2c ) {
		// n2c. can assume that coords for stub_res-1 have been built
		// have to build n,ca positions; c can be copied from Eposition
		// this is a pretty wasteful: re-build everything for residue stub_res-1
		// if speed is a problem, we could redo this using two calls to build_atom
		// but then we need access to all the bond geometry info
		// note that currently this routine is called by update_overlap_positions
		// as well as at least once when folding each segment
		FArray1D_float tmp_c_xyz(3);
		for ( int k=1; k<= 3; ++k ) {
			n_xyz     (k) = misc::Eposition(k, n , stub_res-1);
			ca_xyz    (k) = misc::Eposition(k, ca, stub_res-1);
			tmp_c_xyz (k) = misc::Eposition(k, c , stub_res-1);
		}
		// build C at position stub_res-2
		build_C_coords( stub_res-1, misc::phi(stub_res-1), n_xyz, ca_xyz, tmp_c_xyz, c_xyz); // in,in,in,in,out

		// now build the backbone of position stub_res-1
		FArray2D_float dummy_pos( 3, param::MAX_POS );
		FArray1D_float dummy_cen( 3 );
		build_backbone( n2c, c_xyz,n_xyz,ca_xyz, 1, // rsds to build
										misc::phi( stub_res-1), misc::psi( stub_res-1), misc::omega( stub_res-1),
										misc::res( stub_res-1), dummy_pos, dummy_cen,
										c_xyz,n_xyz,ca_xyz, // what we want: the outgoing stub
										stub_res-1 );
	} else {
		// c2n. assume that coords for stub_res already exist
		// just have to generate the position of C
		assert( dir == -1 );
		for ( int k=1; k<= 3; ++k ) {
			n_xyz (k) = misc::Eposition( k, n , stub_res );
			ca_xyz(k) = misc::Eposition( k, ca, stub_res );
		}
		build_C_coords( stub_res, misc::phi(stub_res), n_xyz, ca_xyz, misc::Eposition( 1, c, stub_res ), c_xyz );
	}
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
// build coordinates for c,n,ca stub
// c of position stub_res-1
// n,ca of position stub_res
//
// if ( dir == n2c ) assumes coords for stub_res-1 exist in Eposition
// if ( dir == c2n ) assumes coords for stub_res   exist in Eposition

void
build_stub(
	const int stub_res,
	const int dir,
	FArray1Da_float c_xyz,
	FArray1Da_float n_xyz,
	FArray1Da_float ca_xyz
)
{
	c_xyz.dimension(3);
	n_xyz.dimension(3);
	ca_xyz.dimension(3);

	static const int n(1), ca(2), c(4), n2c(1), c2n(-1);

	if ( dir == n2c ) {
		// n2c. can assume that coords for stub_res-1 have been built
		// have to build n,ca positions; c can be copied from Eposition
		for ( int k=1; k<= 3; ++k ) {
			c_xyz(k) = misc::Eposition(k, c, stub_res-1);
		}
		build_N_CA_coords( stub_res-1, misc::psi( stub_res-1 ),
			misc::omega( stub_res-1 ),
			misc::Eposition(1, n, stub_res-1), misc::Eposition(1, ca, stub_res-1),
			c_xyz, n_xyz, ca_xyz );
	} else {
		// c2n. assume that coords for stub_res already exist
		// just have to generate the position of C
		assert ( dir == c2n );
		for ( int k=1; k<= 3; ++k ) {
			n_xyz (k) = misc::Eposition( k, n , stub_res );
			ca_xyz(k) = misc::Eposition( k, ca, stub_res );
		}
		build_C_coords( stub_res, misc::phi(stub_res), n_xyz, ca_xyz,
										misc::Eposition( 1, c, stub_res ), c_xyz );
	}

	if ( false ) {
		// debug
		FArray1D_float tmp_c_xyz(3), tmp_n_xyz(3), tmp_ca_xyz(3);
		build_stub_old( stub_res, dir, tmp_c_xyz, tmp_n_xyz, tmp_ca_xyz );
		float dev(0.0);
		for ( int k=1; k<= 3; ++k ) {
			dev +=
				std::abs(  n_xyz(k) -  tmp_n_xyz(k) ) +
				std::abs( ca_xyz(k) - tmp_ca_xyz(k) ) +
				std::abs(  c_xyz(k) -  tmp_c_xyz(k) );
		}
		std::cout << "Debug the new build_stub: dev= " << dev << std::endl;
	}

}

//-----------------------------------------------------------------------------
// same as above
//
void
build_stub_pose(
	pose_ns::Pose const & pose,
	const int stub_res,
	const int dir,
	FArray1Da_float c_xyz,
	FArray1Da_float n_xyz,
	FArray1Da_float ca_xyz
)
{
	c_xyz.dimension(3);
	n_xyz.dimension(3);
	ca_xyz.dimension(3);

	static const int n(1), ca(2), c(4), n2c(1), c2n(-1);

	// we had better be in sync with misc
	// dont do .Eposition() b/c we dont want to trigger a refold...
	using misc::Eposition;

	if ( dir == n2c ) {
		// n2c. can assume that coords for stub_res-1 have been built
		// have to build n,ca positions; c can be copied from Eposition
		for ( int k=1; k<= 3; ++k ) {
			c_xyz(k) = Eposition(k, c, stub_res-1);
		}
		build_N_CA_coords( stub_res-1, pose.psi( stub_res-1 ),
			pose.omega( stub_res-1 ),
			Eposition(1, n, stub_res-1), Eposition(1, ca, stub_res-1),
			c_xyz, n_xyz, ca_xyz );
	} else {
		// c2n. assume that coords for stub_res already exist
		// just have to generate the position of C
		assert ( dir == c2n );
		for ( int k=1; k<= 3; ++k ) {
			n_xyz (k) = Eposition( k, n , stub_res );
			ca_xyz(k) = Eposition( k, ca, stub_res );
		}
		build_C_coords( stub_res, pose.phi(stub_res), n_xyz, ca_xyz,
										Eposition( 1, c, stub_res ), c_xyz );
	}

	if ( false ) {
		// debug
		FArray1D_float tmp_c_xyz(3), tmp_n_xyz(3), tmp_ca_xyz(3);
		build_stub_old( stub_res, dir, tmp_c_xyz, tmp_n_xyz, tmp_ca_xyz );
		float dev(0.0);
		for ( int k=1; k<= 3; ++k ) {
			dev +=
				std::abs(  n_xyz(k) -  tmp_n_xyz(k) ) +
				std::abs( ca_xyz(k) - tmp_ca_xyz(k) ) +
				std::abs(  c_xyz(k) -  tmp_c_xyz(k) );
		}
		std::cout << "Debug the new build_stub: dev= " << dev << std::endl;
	}

}

//-----------------------------------------------------------------------------
//

int
pick_loopy_cutpoint(
	int const nres,
  FArray1D_float const & cut_bias_sum )
{

	float r = ran3() * cut_bias_sum( nres );

	int cutpoint( 0 );

	for ( int i = 1; i <= nres; ++i ) {
		if ( r > cut_bias_sum(i-1) && r <= cut_bias_sum(i) ) {
			cutpoint = i;
		}
	}

	if ( cutpoint == 0 ) {
		std::cout << "pick loopy cutpoint = 0! setting = 1" << std::endl;
		cutpoint = 1;
	}

	return cutpoint;
}

////////////////////////////////////////////////////////////////////////////////
//pb  I'm taking this code from docking, but the conventions on the matrices
//    appear to be different

// angles are in degrees; everything is double
void
jmp_rotation_matrix(
	FArray2Da_double Rzyx,
	double const alpha_x,
	double const alpha_y,
	double const alpha_z
)
{
	Rzyx.dimension( 3, 3 );

	FArray2D_double Rx( 3, 3 );
	FArray2D_double Ry( 3, 3 );
	FArray2D_double Rz( 3, 3 );
	FArray2D_double Rzy( 3, 3 );

	jmp_zrotation(Rz,alpha_z);
	jmp_yrotation(Ry,alpha_y);
	jmp_xrotation(Rx,alpha_x);

	mat_multiply3(Rz ,Ry,Rzy);
	mat_multiply3(Rzy,Rx,Rzyx);
}
//------------------------------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
void
jmp_xrotation(
	FArray2Da_double R,
	double const alpha
)
{
	using numeric::conversions::radians;

	R.dimension( 3, 3 );

	double cosA, sinA;

	cosA = std::cos( radians( alpha ) );
	sinA = std::sin( radians( alpha ) );

	R(1,1) = 1.0;
	R(1,2) = 0.0;
	R(1,3) = 0.0;

	R(2,1) = 0.0;
	R(2,2) = cosA;
	R(2,3) = sinA;

	R(3,1) = 0.0;
	R(3,2) = -sinA;
	R(3,3) = cosA;
}


//------------------------------------------------------------------------------
void
jmp_yrotation(
	FArray2Da_double R,
	double const alpha
)
{
	using numeric::conversions::radians;

	R.dimension( 3, 3 );

	double cosA, sinA;

	cosA = std::cos( radians( alpha ) );
	sinA = std::sin( radians( alpha ) );

	R(1,1) = cosA;
	R(1,2) = 0.0;
	R(1,3) = -sinA;

	R(2,1) = 0.0;
	R(2,2) = 1.0;
	R(2,3) = 0.0;

	R(3,1) = sinA;
	R(3,2) = 0.0;
	R(3,3) = cosA;
}


//------------------------------------------------------------------------------
void
jmp_zrotation(
	FArray2Da_double R,
	double const alpha
)
{
	using numeric::conversions::radians;

	R.dimension( 3, 3 );

	double cosA, sinA;

	cosA = std::cos( radians( alpha ) );
	sinA = std::sin( radians( alpha ) );

	R(1,1) = cosA;
	R(1,2) = sinA;
	R(1,3) = 0.0;

	R(2,1) = -sinA;
	R(2,2) = cosA;
	R(2,3) = 0.0;

	R(3,1) = 0.0;
	R(3,2) = 0.0;
	R(3,3) = 1.0;
}


//-----------------------------------------------------------------------------
// uses (col,row) indexing of the matrix -- charlie-style

void
Dvect_multiply(
	FArray2Da_double R,
	FArray1Da_double v,
	FArray1Da_double Rv // output
)
{
	R.dimension( 3, 3 );
	v.dimension( 3 );
	Rv.dimension( 3 );

	for ( int j = 1; j <= 3; ++j ) {
		Rv(j) = v(1) * R(1,j) + v(2) * R(2,j) + v(3) * R(3,j);
	}
}

//------------------------------------------------------------------------------
void
get_rosetta_torsions(
	FArray1Da_float phi_out,
	FArray1Da_float psi_out,
	int const nres // input
)
{
	//using namespace misc;

	if ( ! get_use_rosetta_torsions() ) { // debug
		std::cout << "shouldnt be here!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( nres != misc::total_residue ) {
		std::cout << "get_rosetta_torsions:: dimension mismatch: " <<
			nres << ' ' << misc::total_residue << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	phi_out.dimension( nres );
	psi_out.dimension( nres );

	for ( int i = 1; i <= nres; ++i ) {
		phi_out(i) = misc::phi(i);
		psi_out(i) = misc::psi(i);
	}
}


//------------------------------------------------------------------------------
void
set_use_rosetta_torsions( bool const setting /* input */ )
{
	using namespace use_rosetta_torsions_common;

	std::cout << "change in use_rosetta_torsions: from,to" <<
	 SS( use_rosetta_torsions ) << SS( setting ) << std::endl;
	use_rosetta_torsions = setting;
}

//------------------------------------------------------------------------------
bool
get_use_rosetta_torsions()
{
	return use_rosetta_torsions_common::use_rosetta_torsions;
}


//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// this routine is called from get_rot_coord
// if we are building pose rotamers
//
// it looks inside the "current" pose for the value of phi at seqpos
// and to see if seqpos-1 is a cutpoint
//
void
put_nh_current_pose(
	int const seqpos,
	FArray3DB_float const & fcoord,
	FArray1D_float & nhcoord
)
{
	// full_coord indices:
	int const N ( 1 );
	int const CA( 2 );
	int const C ( 3 );

	Pose const * pose(0);
	if ( pack_check_current_pose() ) {
		// inside packing
		pose = &pack_get_current_pose();
	} else if ( score_check_current_pose() ) {
		// probably inside rotamer trials
		pose = &score_get_current_pose();
	}
	if ( !pose ) {
		std::cout << "put_nh_pose: no current_pose!" << std::endl;
		std::exit( EXIT_FAILURE );
	}

	FArray1D_float prev_c(3);
	if ( pose->fold_tree().get_is_cutpoint()(seqpos-1) ) {
		build_ideal_C_coords( pose->phi(seqpos), fcoord(1,N,seqpos),
													fcoord(1,CA,seqpos), fcoord(1,C,seqpos), prev_c );
	} else {
		for ( int k=1; k<= 3; ++k ) {
			prev_c(k) = fcoord(k,C,seqpos-1);
		}
	}
	get_HN_coords( prev_c, fcoord(1,N,seqpos), fcoord(1,CA,seqpos), nhcoord );

}

//-----------------------------------------------------------------------------
// full_coord numbering on the input array!!!!!!!!!!!!!!!!!
// build NH coord at position seqpos
// builds the C for the previous residue with ideal bond lengths, angles

void
build_nh_simple(
	int const aa,
	int const aav,
	float const phi,
	FArray2Da_float fcoord
)
{

	if ( aa == param_aa::aa_pro ) return;

	fcoord.dimension( 3, param::MAX_ATOM() );

	int const N(1), CA(2), C(3); // full_coord numbering!!!!!!!!!!!

	//car define initial bond vectors
	FArray2D_float X( 3, 3 );
	subvec( fcoord(1,CA),fcoord(1,C ), X(1,refold_ns::ps) );
	subvec( fcoord(1,N ),fcoord(1,CA), X(1,refold_ns::ph) );
	//car build c(i-1)
	FArray2D_float M( 3, 3 );
	refold_coord_sys( X(1,refold_ns::ps), X(1,refold_ns::ph),
										M(1,1), M(1,2), M(1,3) );
	FArray1D_float c_xyz( 3 );
	build_atom( M, fcoord(1,N),
							refold_ns::cT( refold_ns::c, refold_ns::c2n ),
							refold_ns::sT( refold_ns::c, refold_ns::c2n ),
							phi,
							refold_ns::D ( refold_ns::c, refold_ns::c2n ),
							c_xyz,
							X(1,refold_ns::om) );
	// build HN
	FArray1D_float nhcoord( 3 );
	get_HN_coords( c_xyz, fcoord(1,N), fcoord(1,CA), nhcoord);

	int const j ( aaproperties_pack::HNpos(aa,aav) );
	for ( int k = 1; k <= 3; ++k ) {
		fcoord(k,j) = nhcoord(k);
	}
}

void
put_one_full_coord_nh( int const seqpos /* input */ )
{
	using namespace aaproperties_pack;
	//using namespace jumping;
	//using namespace misc;

	FArray1D_float nhcoord( 3 ); // local
	FArray2D_float dummy_pos( 3, 5 );
	FArray1D_float dummy_cen( 3 );
	FArray1D_float prev_c( 3 );

	int const aa  ( misc::res         (seqpos) );
	int const aav ( misc::res_variant (seqpos) );

	if ( aa == param_aa::aa_pro ) return;

	// POSE //
	FArray1D_bool const & is_cutpoint
		( refold_get_current_pose().fold_tree().get_is_cutpoint() );

	// get C(seqpos-1)
	if ( ! is_cutpoint(seqpos-1) ) {
		for ( int k = 1; k <= 3; ++k ) {
			prev_c(k) = misc::Eposition(k,4,seqpos-1);
		}
	} else {                      // build C
		// used to be build_C_coords -- PB changed (05/27/05)
		build_ideal_C_coords( misc::phi(seqpos),
		 misc::Eposition(1,1,seqpos), /* N */
		 misc::Eposition(1,2,seqpos), /* CA */
		 misc::Eposition(1,4,seqpos), /* C */ prev_c);
	}

	// now call the regular function:
	get_HN_coords( prev_c, misc::full_coord(1,1,seqpos), misc::full_coord(1,2,seqpos),
								 nhcoord );

	int const hnpos ( HNpos(aa,aav) );
	for ( int i = 1; i <= 3; ++i ) {
		misc::full_coord(i,hnpos,seqpos) = nhcoord(i);
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin change_color
///
/// @brief
///
/// @detailed
///
/// @param[in,out]  color -
/// @param[in]  nres -
/// @param  old_color - [in/out]? -
/// @param  new_color - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
change_color(
	FArray1Da_int color, // input/output
	int const nres, // input
	int const old_color,
	int const new_color
)
{
	color.dimension( nres );

	int const init_color = { -1 };
	int const moving_color = { 0 };

	if ( new_color == init_color || old_color == init_color ||
	 new_color == moving_color || old_color == moving_color ) {
		std::cout << "change color problem:" <<
		 SS( new_color ) << SS( old_color ) << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	for ( int i = 1; i <= nres; ++i ) {
		if ( color(i) == old_color ) color(i) = new_color;
	}
}

///////////////////////////////////////
bool
jmp_end_bias_check(
	Fold_tree const & fold_tree,
	int const nres,
	int const begin,
	int const size
)
{
	int max_edge_count;
	int const residues_moved
		( nres - fold_tree.count_fixed_residues( begin, size, max_edge_count ) );

	return ( ran3() <=
	  std::exp( ( residues_moved - max_edge_count ) / jumping::jumping_end_bias ) );

}

//////////////////////////////////////////////////////////////////////////////
/// @begin build_default_stub
///
/// @brief
///
/// @detailed
/// C (seqpos-1) is along the -x axis
/// N (seqpos)   is at the origin
/// CA(seqpos)   is in the first quadrant
///
/// @param  seqpos - [in/out]? -
/// @param  c_xyz - [in/out]? -
/// @param  n_xyz - [in/out]? -
/// @param  ca_xyz - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
void
build_default_stub(
	int const seqpos,
	FArray1Da_float c_xyz,
	FArray1Da_float n_xyz,
	FArray1Da_float ca_xyz
)
{
	using namespace refold_ns;        // n2c etc

	c_xyz.dimension( 3 );
	n_xyz.dimension( 3 );
	ca_xyz.dimension( 3 );

	float D_n, D_ca, cT_ca, sT_ca;
	if ( refold_check_current_pose() &&
			 !refold_get_current_pose().ideal_backbone() ) {
		const bonds_class::Bonds & pose_bonds
			( refold_get_current_pose().bonds() );
		D_n   = pose_bonds.D() (n ,n2c,seqpos);
		D_ca  = pose_bonds.D() (ca,n2c,seqpos);
		cT_ca = pose_bonds.cT()(ca,n2c,seqpos);
		sT_ca = pose_bonds.sT()(ca,n2c,seqpos);
	} else {
		D_n   = D ( n , n2c );
		D_ca  = D ( ca, n2c );
		cT_ca = cT( ca, n2c );
		sT_ca = sT( ca, n2c );
	}

	c_xyz(1) = - D_n;
	c_xyz(2) = 0.0;
	c_xyz(3) = 0.0;

	n_xyz(1) = 0.0;
	n_xyz(2) = 0.0;
	n_xyz(3) = 0.0;

	ca_xyz(1) = cT_ca * D_ca;
	ca_xyz(2) = sT_ca * D_ca;
	ca_xyz(3) = 0.0;
}


///////////////////////////////////////////////////////////////////////////////
//
// generate a stub for n2c folding that will preserve the current location
// of seqpos. Assumes that coords for seqpos exist in misc:: AND that
// misc::phi(seqpos) really is the dihedral angle between C-N-CA-C
//
// specifically, try to preserve CA, N-CA vector, and N-CA-C plane.
//
void
rebuild_default_stub(
	int const seqpos,
	FArray1Da_float c_xyz,
	FArray1Da_float n_xyz,
	FArray1Da_float ca_xyz
)
{
	using namespace refold_ns;        // n2c etc
	using numeric::xyzVector_float;

	c_xyz.dimension( 3 );
	n_xyz.dimension( 3 );
	ca_xyz.dimension( 3 );

	// figure out the N-CA bond length:
	float D_n_ca;
	if ( refold_check_current_pose() &&
			 !refold_get_current_pose().ideal_backbone() ) {
		const bonds_class::Bonds & pose_bonds
			( refold_get_current_pose().bonds() );
		D_n_ca  = pose_bonds.D()(ca,n2c,seqpos);
	} else {
		D_n_ca  = D ( ca, n2c );
	}

	// rebuild the N position with the proper length:
	// this was a BUG caught by Chu Wang: used to be Eposition
	// rather than Ebest_position! Eposition is potentially
	// garbage before refold, only Ebest_position can be assumed
	// to be OK, at least in non-torsion-angle-moved regions
	xyzVector_float n_v  ( &misc::Ebest_position(1,1,seqpos) );
	xyzVector_float ca_v ( &misc::Ebest_position(1,2,seqpos) );

	//n -= ca;
	//n = D_n_ca * n.normalized();
	//n += ca;

	n_v = D_n_ca * ( n_v - ca_v ).normalized() + ca_v;

	for ( int k=1; k<= 3; ++k ) {
		n_xyz (k) =  n_v(k);
		ca_xyz(k) = ca_v(k);
	}

	build_C_coords( seqpos, misc::phi(seqpos), n_xyz, ca_xyz,
									misc::Ebest_position(1,4,seqpos), c_xyz );
}

//////////////////////////////////////////////////////////////////
int
aa2int( char const aa, bool & fail )
{
 	using namespace param_aa;

	fail = false;

	for ( int i = 1; i <= 20; ++i ) {
		if ( aa == aa_name1(i) ) {
			return i;
		}
	}

	//For RNA silent output/input also.
	if (dna_enabled()) {
		for ( int i = 26; i <= 29; ++i ) {
			if ( aa == aa_name1(i) ) {
				return i;
			}
		}
	}

	fail = true;

	return -1;
}

//////////////////////////////////////////////////////////////////
int
aa2int( char const aa )
{

	bool fail( true );

	int i = aa2int( aa, fail );

	if (fail) {
		std::cout << "bad aa: " << aa << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	return i;

}

///////////////////////////////////////////////////////////
void
choose_offset_frag(
	const int frag_size,
	pose_ns::Pose & pose,
	const int loop_begin,
	const int loop_end,
	const int frag_offset
)
{
	using namespace fragments;
	using namespace fragments::frag_pointer;

	const int loop_size ( loop_end - loop_begin + 1);
	const int actual_frag_size
		( loop_size < frag_size  ? loop_size : frag_size );

	// choose a window
	const int pos ( static_cast<int>( ran3() * ( loop_size - actual_frag_size + 1) ) );
	const int pose_pos ( loop_begin + pos );
	const int frag_pos ( loop_begin + pos + frag_offset );

	assert ( 1<= frag_pos && frag_pos <= fragments_nres );

	// choose a fragment
	const int size_bin( get_index_by_frag_size( frag_size ) );

	// this is tricky: what if we have a single residue we want to model with
	// fragments at the c-terminus? cant use frag_pos= fragments_nres

	const int last_frag_pos ( fragments_nres - frag_size + 1 );
	const int shift ( std::max( 0, frag_pos - last_frag_pos ) );
	// PB: 06/02/05 fix bug: was getting align_depth(frag_pos,size_bin)
	// could be nonsense if frag_pos > frag_nres-frag_size+1
//	const int nn_num ( static_cast<int>( ( ran3() *
//																				 align_depth(frag_pos-shift,size_bin))+1) );

	int nn_num;
	if ( get_ssblock_state() ) {
		int const jnn_num = static_cast< int >
			( ran3() * std::min(align_depth(frag_pos-shift,size_bin),block_depth(frag_pos-shift,size_bin)) ) + 1;
		nn_num = block_frag_pointer(jnn_num,frag_pos-shift,size_bin);
	} else if(antibody_ns::bias_frags) { // prefer frags from antibody
		int num_totfrags_this_pos( align_depth(frag_pos-shift,size_bin)) ;
		int num_abfrags_this_pos( num_totfrags_this_pos - 200 ) ;
		// bias_fraction default is 3.00
		float prob_pick_abfrags( antibody_ns::bias_fraction * num_abfrags_this_pos
														 / num_totfrags_this_pos );

		if( ran3() < prob_pick_abfrags ) {
			nn_num = static_cast< int >
				( ran3() * num_abfrags_this_pos ) + 201;
		}
		else {
			// frags after 200th position are antibody fragments
			// 200 is the number of rosetta fragments
			nn_num = static_cast< int >
				( ran3() * 200 ) + 1;
		}
	} else {
		nn_num = static_cast< int >
			( ran3() * align_depth(frag_pos-shift,size_bin)) + 1;
	}

	// insert the fragment
	for ( int k=shift; k<shift+actual_frag_size; ++k ) {
		assert ( k < frag_size );
		if ( pose.get_allow_bb_move( pose_pos + k - shift ) ) {
			pose.set_phi  ( pose_pos+k-shift, align_phi  (frag_pos-shift,nn_num,k,size_bin) );
			pose.set_psi  ( pose_pos+k-shift, align_psi  (frag_pos-shift,nn_num,k,size_bin) );
			pose.set_omega( pose_pos+k-shift, align_omega(frag_pos-shift,nn_num,k,size_bin) );

			pose.set_secstruct( pose_pos+k-shift, ss_type(frag_pos-shift,nn_num,k,size_bin) );
		}
	}
}


///////////////////////////////////////////////////////////////////////
void
insert_init_frag(
	pose_ns::Pose & pose,
	const int begin,
	const int size
)
{
	for ( int i=begin; i<begin+size; ++i ) {
		pose.set_phi       ( i, param::init_phi );
		pose.set_psi       ( i, param::init_psi );
		pose.set_omega     ( i, param::init_omega );
		pose.set_secstruct ( i, 'L' );
	}
}
///////////////////////////////////////////////////////////////////////
///////////////////////////////////////////
void
insert_random_frags(
	const int frag_size,
	pose_ns::Pose & pose,
	const int loop_begin,
	const int loop_end,
	const int frag_offset
)
{
	const int loop_size ( loop_end - loop_begin + 1 );
	if ( frag_size > loop_size ) { // bq bugfix!!!
		choose_offset_frag( frag_size, pose, loop_begin, loop_end, frag_offset );
	} else {
		for ( int i=0; i< loop_size-frag_size+1; ++i ) {
			const int begin( loop_begin+i );
			const int end( begin+frag_size-1);
			assert( begin >= loop_begin && end <= loop_end );
			choose_offset_frag( frag_size, pose, loop_begin+i,
				loop_begin+i+frag_size-1, frag_offset );
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
bool
score_filter(
	const float score,
	const std::string & tag,
	const float acceptance_rate
)
{
	const int rank_pad(2);

	typedef std::list< float > Float_list;
	static std::map< std::string, Float_list > list_map;

	if ( list_map.count( tag ) <= 0 ) {
		Float_list l;
		list_map.insert( std::make_pair( tag, l ) );
	}

	Float_list & scores ( list_map.find(tag)->second );
	scores.push_back( score );
	scores.sort();

	const int max_rank ( static_cast< int >( scores.size() * acceptance_rate ) );

	// rank starts at 1 and goes to nscores
	//

	int rank(1), index(1);
	float filter_score(0.0);
	for ( Float_list::const_iterator it = scores.begin(), it_end=scores.end();
				it != it_end; ++it, ++index ) {
		if ( *it < score ) ++rank;
		if ( index == max_rank ) filter_score = *it;
	}

	std::cout << "score_filter: tag= " << tag <<
		" score= " << score <<
		" rank= " << rank <<
		" max_rank= " << max_rank <<
		" nscores= " << scores.size() <<
		" filter_score= " << filter_score << std::endl;

	return ( rank <= max_rank + rank_pad );

}
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
numeric::xyzVector_double
random_unit_vector()
{
	using namespace numeric::constants::d;

	const double theta( pi_2 * ran3());
	const double z( numeric::sin_cos_range( 1.0 - 2.0 * ran3() ) );
	const double xy_norm( std::sqrt( 1-z*z ) );

	return numeric::xyzVector_double( xy_norm * std::cos( theta ),
																 xy_norm * std::sin( theta ),
																 z);
}

///////////////////////////////////////////////////////////////////////////////
numeric::xyzMatrix_double
random_axis_rotation_matrix(
	double const alpha // degrees!
)
{
	// this will generate a rotation around a randomly chosen axis by alpha degrees
	using numeric::conversions::degrees;

	const double theta( 360.0 * ran3() );
	const double phi( degrees( std::acos( numeric::sin_cos_range( 1.0 - 2.0 * ran3() ) ) ) );

	return
		Z_rot(  theta ) *
		Y_rot(    phi ) *
		Z_rot(  alpha ) *
		Y_rot(   -phi ) *
		Z_rot( -theta );
}
///////////////////////////////////////////////////////////////////////////////
numeric::xyzMatrix_double
random_reorientation_matrix()
{
	// a genunine rotation matrix which will randomly reorient the coord sys.
	// from Euler threom
	using numeric::conversions::degrees;

	const double phi( 360.0 * ran3() ); // [0,360.0]degrees!
	const double psi( 360.0 * ran3() ); // [0,360.0]degrees!
	const double theta(
		degrees( std::acos(numeric::sin_cos_range( 1.0 - 2.0 * ran3() ) ) )
	); // [0,180.0] degrees

	return
		Z_rot(  psi   ) *
		Y_rot(  theta ) *
		Z_rot(  phi   );
}
///////////////////////////////////////////////////////////////////////////////
// right-handed rotation about the y-axis

numeric::xyzMatrix_double
X_rot( const double alpha ) // degrees
{
	using numeric::conversions::radians;
	double const cosA = std::cos( radians( alpha ) );
	double const sinA = std::sin( radians( alpha ) );

	return numeric::xyzMatrix_double::rows(
		1.0,  0.0,   0.0,
		0.0, cosA, -sinA,
		0.0, sinA,   cosA );
}

///////////////////////////////////////////////////////////////////////////////
// right-handed rotation about the y-axis

numeric::xyzMatrix_double
Y_rot( const double alpha ) // degrees
{
	using numeric::conversions::radians;
	double const cosA = std::cos( radians( alpha ) );
	double const sinA = std::sin( radians( alpha ) );

	return numeric::xyzMatrix_double::rows(
		cosA , 0.0,  sinA,
		0.0  , 1.0,   0.0,
		-sinA, 0.0,  cosA );
}

///////////////////////////////////////////////////////////////////////////////
// right-handed rotation about the z-axis

numeric::xyzMatrix_double
Z_rot( const double alpha ) // degrees
{
	using numeric::conversions::radians;
	double const cosA = std::cos( radians( alpha ) );
	double const sinA = std::sin( radians( alpha ) );

	return numeric::xyzMatrix_double::rows(
		cosA, -sinA, 0.0,
		sinA,  cosA, 0.0,
		 0.0,   0.0, 1.0 );
}

///////////////////////////////////////////////////////////////////////////////
// right-handed rotation about the y-axis

numeric::xyzMatrix_double
X_rot_rad( const double alpha ) // radians
{
	double const cosA = std::cos( alpha );
	double const sinA = std::sin( alpha );

	return numeric::xyzMatrix_double::rows(
		1.0,  0.0,   0.0,
		0.0, cosA, -sinA,
		0.0, sinA,   cosA );
}

///////////////////////////////////////////////////////////////////////////////
// right-handed rotation about the y-axis

numeric::xyzMatrix_double
Y_rot_rad( const double alpha ) // radians
{
	double const cosA = std::cos( alpha );
	double const sinA = std::sin( alpha );

	return numeric::xyzMatrix_double::rows(
		cosA , 0.0,  sinA,
		0.0  , 1.0,   0.0,
		-sinA, 0.0,  cosA );
}

///////////////////////////////////////////////////////////////////////////////
// right-handed rotation about the z-axis

numeric::xyzMatrix_double
Z_rot_rad( const double alpha ) // radians
{
	double const cosA = std::cos( alpha );
	double const sinA = std::sin( alpha );

	return numeric::xyzMatrix_double::rows(
		cosA, -sinA, 0.0,
		sinA,  cosA, 0.0,
		 0.0,   0.0, 1.0 );
}

///////////////////////////////////////////////////////////////////////////////
// checks to see if N,CA,C are most-likely initialized
//
// The N-CA and C-CA vectors have reasonable lengths, and are
// not parallel to one another
//
// probably a pretty good test for initialized coordinates,
// assuming they would be zero'd out by default
//
bool
Epos_init(
	FArray2Da_float Epos
)
{
	using numeric::xyzVector_float;

	Epos.dimension(3,param::MAX_POS);

	xyzVector_float n  ( &Epos(1,1) ); // note: P   if DNA
	xyzVector_float ca ( &Epos(1,2) ); //       O1P
	xyzVector_float c  ( &Epos(1,4) ); //       O2P

	n -= ca;
	c -= ca;

	float const n_len( n.length_squared() );
	float const c_len( c.length_squared() );

	return ( n_len > 0.25 && n_len < 9 && // should be around 2.25
					 c_len > 0.25 && c_len < 9 &&
					 cross(n,c).length_squared() > 0.05 );
}

///////////////////////////////////////////////////////////////////////////////
// checks to see if N,CA,C are most-likely initialized,
// here initialized means not zero'd out
//
// The N-CA and C-CA vectors have reasonable lengths, and are
// not parallel to one another
//
// probably a pretty good test for initialized coordinates,
// assuming they would be zero'd out by default
//
bool
Epos_init_zero(
	FArray2Da_float Epos
)
{
	using numeric::xyzVector_float;

	Epos.dimension(3,param::MAX_POS);

	xyzVector_float n  ( &Epos(1,1) ); // note: P   if DNA
	xyzVector_float ca ( &Epos(1,2) ); //       O1P
	xyzVector_float c  ( &Epos(1,4) ); //       O2P

	n -= ca;
	c -= ca;

	float const n_len( n.length_squared() );
	float const c_len( c.length_squared() );

	return ( n_len > 0.25 && // should be around 2.25
					 c_len > 0.25 &&
					 cross(n,c).length_squared() > 0.05 );
}


///////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////
// two acceptable formats -- tags should be unique
//
// single-line:
// ALIGN <alignment> tag
//
// or aligned fasta:
// >tag
// <alignment>
//
// . or - are acceptable gap characters
//
// returns TRUE on failure
//

bool
mapping_from_file(
	std::string const & filename,
	std::string const & seq1,
	std::string const & seq2,
	FArray1D_int & mapping,
	std::string const target_tag1, // = "tag1"
	std::string const target_tag2  // = "tag2"
)
{
	bool fail( true );

	assert( mapping.size1() >= seq1.size() );

	// try to open file
	std::ifstream data( filename.c_str() );
	if (!data.good() ) {
		std::cout << "mapping_from_file: Unable to open file: " <<
			filename << std::endl;
		return fail;
	}

	std::string line;
	getline( data, line );
	data.seekg( std::ios_base::beg );

	// read file: generate a std::mapping from tags to alignment strings
	typedef std::map< std::string, std::string > Align_map;
	Align_map align_map;

	if ( line[0] == '>' ) { // assume aligned fasta file
		std::string tag;
		while ( getline( data,line ) ) {
			if ( line[0] == '>' ) {
				line.erase(0,1);
				tag = line;
				if ( align_map.count( tag ) ) {
					std::cout << "WARNING:: duplicate tags in the align-file; " <<
						" replacing old data!" << std::endl;
				}
			} else {
				std::replace( line.begin(), line.end(), '.', '-' );
				std::string align( align_map[tag] );
				align = align+line;
				align_map[tag] = align;
			}
		}
	} else { // ALIGN file format
		while ( getline(data,line) ) {
			std::istringstream line_stream( line );
			std::string line_tag,align,tag;
			line_stream >> line_tag >> align >> tag;
			if ( line_tag != "ALIGN" || line_stream.fail() ) {
				std::cout << "align_file parse error: " << line << std::endl;
				continue;
			}
			std::replace( align.begin(), align.end(), '.', '-' );
			if ( align_map.count( tag ) ) {
				std::cout << "WARNING:: duplicate tags in the align-file; replacing" <<
					" old data!" << std::endl;
			}
			align_map[tag] = align;
		}
	}

	/////////////////////
	// find the sequences
	std::string tag1, tag2;

	for ( Align_map::const_iterator it=align_map.begin(),
					it_end = align_map.end(); it != it_end; ++it ) {
		std::string const & tag( it->first );
		std::string const & align( it->second );
		int const L( align.size() );
		std::string seq;
		for ( int i=0; i<L; ++i ) {
			if ( align[i] != '-' ) seq += align[i];
		}
		if ( seq == seq1 && tag1 != target_tag1 &&
				 ( tag1.size() == 0 || tag == target_tag1 ) ) {
			// take the first sequence match, unless a later one has
			// the correct tag
			tag1 = tag;
		}
		if ( seq == seq2 && tag2 != target_tag2 &&
				 ( tag2.size() == 0 || tag == target_tag2 ) ) {
			// take the first sequence match, unless a later one has
			// the correct tag
			tag2 = tag;
		}
	}

	if ( tag1.size() < 1 || tag2.size() < 1 ) {
		std::cout << "WARNING:: mapping_from_file: cant sequences in mapping file"
							<< std::endl;
		return fail;
	}


	///////////////////////
	// generate the mapping
	std::string const & align1( align_map[ tag1 ] );
	std::string const & align2( align_map[ tag2 ] );
	int const L( align1.size() );

	if ( L != int( align2.size() ) ) {
		std::cout << "WARNING: align-length mismatch in mapping file: " << L <<
			' ' << align2.size() << std::endl;
		return fail;
	}

	{ // debugging
		std::string al1,al2;
		for ( int i=0; i<L; ++i ) {
			char s1( align1[i] );
			char s2( align2[i] );
			if ( s1 != '-' || s2 != '-' ) {
				al1 = al1 + s1;
				al2 = al2 + s2;
			}
		}
		std::cout << "align1: " << al1 << "\nalign2: " << al2 << std::endl;
	}


	for ( int i=0, pos1=0, pos2=0; i< L; ++i ) {
		if ( align2[i] != '-' ) {
			++pos2;
		}
		if ( align1[i] != '-' ) {
			++pos1;
			if ( align2[i] != '-' ) {
				mapping( pos1 ) = pos2;
			} else {
				mapping( pos1 ) = -1;
			}
		}
		if ( i == L-1 )
			assert( pos1 == int( seq1.size()) && pos2 == int( seq2.size()) );
	}
	fail = false;
	return fail;
}

///////////////////////////////////////////////////////////////////////////////
// returns TRUE if passed

bool
symm_check(
					 pose_ns::Pose const & pose
					 )
{
	using namespace pose_ns;
	bool ok( true );

	if ( !pose.symmetric() ) return true;

	Symmetry_info const & s( pose.symmetry_info() );

	int const nres( pose.total_residue_for_scoring() ); // exclude pseudo rsds
	int const num_jump( pose.num_jump() );

	// check backbone
	for ( int seqpos=1; seqpos<= nres; ++seqpos ) {
		if ( s.bb_independent( seqpos ) ) {

			std::vector< int > const & clones( s.bb_clones( seqpos ) );
			assert( int(clones.size()) == s.num_bb_clones() );

			for ( int i=0; i< int(clones.size()); ++i ) {
				int const clone_pos( clones[i] );
				float const dev
					( std::abs( pose.phi  (seqpos) - pose.phi  (clone_pos) ) +
						std::abs( pose.psi  (seqpos) - pose.psi  (clone_pos) ) +
						std::abs( pose.omega(seqpos) - pose.omega(clone_pos) ) );

				if ( dev > 0.1 ||
						 pose.secstruct  ( seqpos ) != pose.secstruct  ( clone_pos ) ||
						 pose.res        ( seqpos ) != pose.res        ( clone_pos ) ||
						 pose.res_variant( seqpos ) != pose.res_variant( clone_pos ) ) {
					std::cout << "symm_check: bbdev: " << seqpos << ' ' << clone_pos <<
						' ' << dev << ' ' <<
						pose.secstruct  ( seqpos    ) << ' ' <<
						pose.secstruct  ( clone_pos ) << ' ' <<
						pose.res        ( seqpos    ) << ' ' <<
						pose.res        ( clone_pos ) << ' ' <<
						pose.res_variant( seqpos    ) << ' ' <<
						pose.res_variant( clone_pos ) << std::endl;
					ok = false;
					break;
				}
			}
		}
	}

	// check chi angles
	if ( pose.fullatom() ) {
		for ( int seqpos=1; seqpos<= nres; ++seqpos ) {
			if ( s.chi_independent( seqpos ) ) {

				std::vector< int > const & clones( s.chi_clones( seqpos ) );
				//assert( int(clones.size()) == s.num_chi_clones() );

				int const nchi
					(aaproperties_pack::nchi(pose.res(seqpos),pose.res_variant(seqpos)));
				for ( int i=0; i< int(clones.size()); ++i ) {
					int const clone_pos( clones[i] );
					float dev(0.0);
					for ( int chino=1; chino<= nchi; ++chino ) {
						dev += std::abs( pose.chi( chino,    seqpos ) -
														 pose.chi( chino, clone_pos ) );
					}
					if ( dev > 0.1 ) {
						std::cout << "symm_check: chidev: " << seqpos << ' ' <<
							clone_pos << ' ' << dev << std::endl;
						ok = false;
					}
				}
			}
		}
	}


	// check jumps
	for ( int jump=1; jump<= num_jump; ++jump ){
		if ( s.jump_independent( jump ) ) {
			std::vector< int > const & clones( s.jump_clones( jump ) );

			for ( int i=0; i< int(clones.size()); ++i ) {
				float const dev( distance( pose.get_jump( jump ),
																	 pose.get_jump( clones[i]) ) );
				if ( dev > 0.1 ) {
					std::cout << "symm_check: chidev: " << jump << ' ' <<
						clones[i] << ' ' << dev << std::endl;
					ok = false;
					break;
				}
			}
		}
	}
	return ok;
}




///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
//---------------------THE END-------------------------------------------------
