// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.1.2.2 $
//  $Date: 2005/11/07 21:05:35 $
//  $Author: pbradley $

// Rosetta Headers
#include "kin_bonded_atom.h"
#include "jumping_util.h"
#include "kin_util.h"
#include "pose.h"
#include "aaproperties_pack.h"
#include "atom_tree_minimize.h" // update_nblist! should just make method?

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
//#include <ObjexxFCL/FArray4D.h>
//#include <ObjexxFCL/formatted.io.h>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/constants.hh>
#include <numeric/conversions.hh>
#include <numeric/xyz.functions.hh>
#include <numeric/xyzMatrix.hh>
#include <numeric/xyzVector.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/orstream.hh>

// C++ Headers
#include <cstdlib>
#include <cstdio>


namespace kin {
	/////////////////////////////////////////////////////////////////////////////
	void
	Bonded_atom::set_torsion(
		Kin_torsion_type const type,
		float const value
	)
	{
		torsion_moved = true;
		if ( type == PHI ) {
			assert( !phi_fixed ); // should replace with error message + failure
			phi = value;
		} else if ( type == THETA ) {
			assert( !theta_fixed );
			theta = value;
		} else if ( type == D ) {
			assert( !d_fixed );
			d = value;
		} else {
			std::cout << "bad torsion type for Atom: " << type << std::endl;
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	float
	Bonded_atom::get_torsion(
		Kin_torsion_type const type
		) const
	{
		if ( type == PHI ) {
			return phi;
		} else if ( type == THETA ) {
			return theta;
		} else if ( type == D ) {
			return d;
		} else {
			std::cout << "bad torsion type for Atom: " << type << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		return 0.0;
	}

	/////////////////////////////////////////////////////////////////////////////
	// pass out the stub we would pass out if we were actually updating
	// coords or torsions

	void
	Bonded_atom::update_stub(
		Stub & stub
	) const
	{
		stub.M *= X_rot_rad( phi ); // this gets passed out
	}

	/////////////////////////////////////////////////////////////////////////////

	void
	Bonded_atom::update_coords(
		Stub & stub,
		Coords & coords
	) const
	{
		using numeric::constants::d::pi;
// 		if ( true ) { // debug
// 			std::cout << "Bonded_atom::update_coords: input_stub_dev= " <<
// 				distance( get_input_stub( coords ), stub ) << ' ' << atom_id <<
// 				std::endl;
// 		}

		//std::cout << "update_coords: " << atom_id << std::endl;

		stub.M *= X_rot_rad( phi ); // this gets passed out

		Stub new_stub( stub.M * Z_rot_rad( theta ), stub.v );

		if ( std::abs( theta - pi ) < 1e-6 ) {
			// very special case
			if ( keep_torsion_fixed( THETA ) ) {
				new_stub.M *= X_rot_rad( pi );
			}
		}

		new_stub.v += d * new_stub.M.col_x();

		coords.set_xyz( atom_id, new_stub.v );

// 		if ( true ) { // debug
// 			std::cout << "Bonded_atom::update_coords: stub_dev= " <<
// 				distance( get_stub( coords ), new_stub ) << ' ' << atom_id <<
// 				std::endl;
// 		}

		for ( std::vector< Atom* >::const_iterator it= atoms.begin(),
						it_end = atoms.end(); it != it_end; ++it ) {
			(*it)->update_coords( new_stub, coords );
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	void
	Bonded_atom::update_torsions(
		Stub & stub,
		Coords const & coords,
		bool const recursive // = true
	)
	{
		using numeric::constants::d::pi;
		torsion_moved = true; // ensure complete scoring

// 		std::cout << "Bonded_atom::update_torsions: input_stub_dev= " <<
// 			distance( stub, get_input_stub( coords ) ) << ' ' << atom_id <<
// 			std::endl;

		numeric::xyzVector_double w;
		coords.get_xyz( atom_id, w );

		w-= stub.v;
		d = w.length();

		bool flip_stub( false );
		if ( d < 1e-2 ) {
			// phi, theta dont make much sense
			std::cout << "WARNING:: very small d= " << d << ' ' << atom_id <<
				std::endl;
			phi = 0.0;
			theta = 0.0;
		} else {
			if ( d < 1e-1 ) {
				std::cout << "WARNING:: small d but we are calculating phi,theta: " <<
					d << std::endl;
			}
			w.normalize();
			double const x( dot( w, stub.M.col_x() ) );
			double const y( dot( w, stub.M.col_y() ) );
			double const z( dot( w, stub.M.col_z() ) );

			double const tol( 1e-6 );
			if ( x < -1.0 + tol ) {
				// very special case:
				// confirm that we are the stub_atom2 of a jump:
				//assert( keep_torsion_fixed( THETA ) );
				theta = pi;
				phi = 0.0;
				if ( keep_torsion_fixed( THETA ) ) {
					flip_stub = true; // very special case
				}
			} else if ( x > 1.0 - tol ) {
				//std::cout << "WARNING:: update_torsions: exactly parallel? " <<
				//	atom_id << std::endl;
				theta = 0.0;
				phi = 0.0;
			} else {
				theta = std::acos( x ); // DANGER
				if ( theta < 1e-2 || pi - theta < 1e-2 ) {
					// less than 0.57 degrees
					std::cout << "WARNING:: small theta but we are calculating phi: " <<
						theta << std::endl;
				}
				phi  = std::atan2( z, y );
			} // small theta
		} // small d

		stub.M *= X_rot_rad( phi );



		if ( recursive ) {
			Stub new_stub( stub.M * Z_rot_rad( theta ),
										 coords.get_xyz( atom_id ) );
			if ( flip_stub ) {
				// special case if I'm stub_atom2 of my parent (whose a jump)
				new_stub.M *= X_rot_rad( pi );
			}
//  			if ( true ) { // debug
//  				std::cout << "Bonded_atom::update_torsions: stub_dev= " <<
// 					distance( get_stub( coords ), new_stub ) << ' ' << atom_id <<
// 					std::endl;
//  			}

			for ( std::vector< Atom* >::const_iterator it= atoms.begin(),
							it_end = atoms.end(); it != it_end; ++it ) {
				(*it)->update_torsions( new_stub, coords );
			}
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	Atom*
	Bonded_atom::copy( Atom* parent_in ) const
	{

		Bonded_atom* new_me = new Bonded_atom();

		new_me->atom_id = atom_id;
		new_me->parent = parent_in;
		new_me->torsion_moved = torsion_moved;

		// copy torsions
		new_me->phi = phi;
		new_me->theta = theta;
		new_me->d = d;

		// allow move:
		new_me->phi_fixed = phi_fixed;
		new_me->theta_fixed = theta_fixed;
		new_me->d_fixed = d_fixed;

		// copy atoms
		for ( std::vector< Atom* >::const_iterator a=atoms.begin(),
						a_end=atoms.end(); a != a_end; ++a ) {
			new_me->atoms.push_back( (*a)->copy( new_me /*the new parent*/) );
		}

		return new_me;
	}


	/////////////////////////////////////////////////////////////////////////////
	// this may modify last_torsion, if our bond torsion angle is changing
	//
	void
	Bonded_atom::setup_min_map(
		Torsion_id & last_torsion,
		Minimizer_map & min_map
	) const
	{

		if ( !phi_fixed ) {
			Torsion_id phi_torsion( atom_id, PHI );
			min_map.add_torsion( phi_torsion, last_torsion );
			last_torsion = phi_torsion;
		}

		// no more changes to last_torsion from now on //

		Torsion_id last_torsion_local( last_torsion );

		if ( !theta_fixed ) {
			Torsion_id theta_torsion( atom_id, THETA );
			min_map.add_torsion( theta_torsion, last_torsion_local );
			last_torsion_local = theta_torsion;
		}

		if ( !d_fixed ) {
			Torsion_id d_torsion( atom_id, D );
			min_map.add_torsion( d_torsion, last_torsion_local );
			last_torsion_local = d_torsion;
		}

		// add me to the min_map
		min_map.add_atom( atom_id, last_torsion_local );

		for ( std::vector< Atom* >::const_iterator it=atoms.begin(),
						it_end = atoms.end(); it != it_end; ++it ) {
			(*it)->setup_min_map( last_torsion_local, min_map );
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Bonded_atom::get_torsion_axis_and_end_pos(
		Coords const & coords,
		numeric::xyzVector_float & axis,
		numeric::xyzVector_float & end_pos,
		Kin_torsion_type const type
	) const
	{
		Stub const my_stub( this->get_stub( coords ) ),
			input_stub( this->get_input_stub( coords ));

		if ( type == PHI ) {
			end_pos = input_stub.v;
			axis = input_stub.M.col(1);
		} else if ( type == THETA ) {
			end_pos = input_stub.v;
			axis = my_stub.M.col(3);
		} else if ( type == D ) {
			axis = my_stub.M.col(1);
		} else {
			std::cout << "Bad torsion type for Atom" << type << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	bool
	Bonded_atom::get_allow_move(
		Kin_torsion_type const type
		) const
	{

		//std::cout << "set_allow_move: " << atom_id << ' ' << type << ' ' <<
		//	setting << std::endl;
		if ( type == PHI ) {
			return !phi_fixed;
		} else if ( type == THETA ) {
			return !theta_fixed;
		} else if ( type == D ) {
			return !d_fixed;
		} else if ( type == RB_TRANSLATION || type == RB_ROTATION ) {
			return false;
		} else if ( type == ALL ) {
			return !torsions_fixed();
		} else {
			std::cout << "Bonded_atom::get_allow_move: bad type: " << type <<
				std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		return false;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Bonded_atom::set_allow_move(
		Kin_torsion_type const type,
		bool const setting
		)
	{
// 		std::cout << "set_allow_move: " << atom_id << ' ' << type << ' ' <<
// 			setting << std::endl;
		if ( type == PHI && ( !setting || ! this->keep_torsion_fixed(type) ) ) {
			if ( ! ( setting && keep_torsion_fixed( PHI   ))) phi_fixed = !setting;
		} else if ( type == THETA ) {
			if ( ! ( setting && keep_torsion_fixed( THETA ))) theta_fixed = !setting;
		} else if ( type == D ) {
			if ( ! ( setting && keep_torsion_fixed( D     ))) d_fixed = !setting;
		} else if ( type == ALL ) {
			if ( ! ( setting && keep_torsion_fixed( PHI   ))) phi_fixed = !setting;
			if ( ! ( setting && keep_torsion_fixed( THETA ))) theta_fixed = !setting;
			if ( ! ( setting && keep_torsion_fixed( D     ))) d_fixed = !setting;
		}
		// If I'm stub_atom 2 or 3 for a jump atom, have to be careful:
		//
// 		if ( setting ) {
// 			if ( ( type == THETA || type == PHI || type == ALL ) &&
// 					 parent->is_jump() && atom_id == parent->stub_atom2() ) {
// 				std::cout << "WARNING:: this bond is fixed: (stub_atom2) " <<
// 					atom_id << ' ' << type << std::endl;
// 				phi_fixed = true;
// 				theta_fixed = true;
// 			} else if ( ( type == PHI || type == ALL ) &&
// 									( ( parent->is_jump() && atom_id == parent->stub_atom3() ) ||
// 										( parent->parent && parent->parent->is_jump() &&
// 											atom_id == parent->parent->stub_atom3() ) ) ) {
// 				// if I'm stub_atom3 for a jump also have to be careful:
// 				std::cout << "WARNING:: this bond is fixed: (stub_atom3) " <<
// 					atom_id << ' ' << type << std::endl;
// 				phi_fixed = true;
// 			}
// 		}
	}

	/////////////////////////////////////////////////////////////////////////////
	bool
	Bonded_atom::keep_torsion_fixed(
		Kin_torsion_type const type
	) const
	{
		if ( type == D ) {
			return false;
		} else if ( type == PHI ) {
			return
				( ( parent->is_jump() &&
						( atom_id == parent->stub_atom2() ||
							atom_id == parent->stub_atom3() ) ) ||
					( parent->parent && parent->parent->is_jump() &&
						atom_id == parent->parent->stub_atom3() ) );
		} else if ( type == THETA ) {
			return ( parent->is_jump() && atom_id == parent->stub_atom2() );
		} else {
			std::cout << "Bonded_atom::keep_torsion_fixed: BAD_TYPE: " <<type <<
				std::endl;
			assert( false );
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		return false;
	}


	/////////////////////////////////////////////////////////////////////////////
	// this asserts equal topology
	void
	Bonded_atom::copy_torsions(
		Atom const & src
	)
	{
		torsion_moved = src.get_torsion_moved();

		phi   = src.get_torsion( PHI );
		theta = src.get_torsion( THETA );
		d     = src.get_torsion( D );

		phi_fixed   = !src.get_allow_move( PHI );
		theta_fixed = !src.get_allow_move( THETA );
		d_fixed     = !src.get_allow_move( D );

		if ( atom_id != src.atom_id || int(atoms.size()) != src.nchildren() ) {
			std::cout << "Bonded_atom:: copy_torsions: topology_mismatch!" <<
				atom_id  << ' ' << src.atom_id << ' ' << atoms.size() << ' ' <<
				src.nchildren() << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		int i(0);
		for ( std::vector< Atom* >::iterator it = atoms.begin(),
						it_end = atoms.end(); it != it_end; ++it, ++i ) {
			(*it)->copy_torsions( *(src.child(i)) );
		}
 	}


	/////////////////////////////////////////////////////////////////////////////
	//
// 	void
// 	Bonded_atom::operator= (
// 		Atom const & src
// 	)
// 	{
// 		set_torsion( PHI  , src.get_torsion( PHI   ) );
// 		set_torsion( THETA, src.get_torsion( THETA ) );
// 		set_torsion( D    , src.get_torsion( D     ) );

// 		if ( atom_id != src.atom_id || atoms.size() != src.atoms.size() ) {
// 			std::cout << "Bonded_atom:: copy_torsions: topology_mismatch!" <<
// 				atom_id  << ' ' << src.atom_id << ' ' << atoms.size() << ' ' <<
// 				src.atoms.size() << std::endl;
// 			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
// 		}

// 		std::vector< Atom* >::const_iterator src_it = src.atoms.begin();
// 		for ( std::vector< Atom* >::iterator it = atoms.begin(),
// 						it_end = atoms.end(); it != it_end; ++it, ++src_it ) {
// 			(*it)->copy_torsions( **src_it );
// 		}
// 	}




} // namespace kin
