// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 23432 $
//  $Date: 2008-06-24 16:25:52 +0300 (Tue, 24 Jun 2008) $
//  $Author: yab $

#ifndef INCLUDED_ligand_ns
#define INCLUDED_ligand_ns

// Rosetta Headers
#include "LigandAtomPairEnergy.h"
#include "atom_chem.h"

// Numeric Headers
#include <numeric/xyzVector.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>

//utility Headers
#include <utility/PeriodicSplineReader.hh>

// C++ Headers
#include <vector>
#include <list>
#include <map>

//forward Declarations
class BondSmallMolecule;
class Atom;
namespace ligand{

	//kwk the following two typedefs may turn into class in time
	// so I am adding these ease the possible transition
	typedef std::vector< numeric::xyzVector < double > > LigandCartesianCoord;
	typedef std::vector< numeric::xyzVector < double > > LigandInternalCoord;
	typedef Atom * Ligand_AtomOP;
	typedef Atom * Ligand_AtomAP;
	typedef BondSmallMolecule * Ligand_BondAP;
}

class Ligand {

	public:
//==============================================================================
//
//     ligand_ns
//
//     Jens Meiler April 2003
//	   Kristian Kaufmann October 2005
//==============================================================================

//mj fields for hetero atoms, introduced for loop building and ligand docking
//mj and side chain packing for loop building only a simplified clash check is
//mj performed.  for this all atoms are mapped on atom types
//mj 1=N,2=CA,3=CB,4=C,5=O in hetero_atom_cent for docking/packing ROSETTA full
//mj atom types are necesarry => all atoms need to be mapped on the 25 ROSETTA
//mj full atom types in hetero_atom_type! this type is obtained from the PDB
//mj name of the atom.

//constructors and destructor
	Ligand();
	Ligand(Ligand & p);
	Ligand & operator=(Ligand & p);
	~Ligand();

	//MEMBER DATA
	bool attach_by_jump;
	bool attach_by_bond;
	bool use_only_defined_bonds;
	bool use_input_ligand_only;
	bool use_iterative_conformation_search;
	bool rotamerize_input_ligand;
	bool make_rotamer_nearest_start;
	bool read_ligand_conformations;
	std::string ligand_conformations_infile;
	bool use_input_ligand;
	bool fix_ligand_HO; // flag to disallow optimization of HO position
	// atom information: coordinates, type, centroid, name etc. see atom.h
	std::vector< ligand::Ligand_AtomOP > atom_vector;
	 // best_coordinates
	ligand::LigandInternalCoord best_bonds; // best internal coordinates
	ligand::LigandCartesianCoord best_coord;// best cartesian coordinates
	 // low_coordinates
	ligand::LigandInternalCoord start_bonds; // start internal coordinates
	ligand::LigandCartesianCoord start_coord;// start cartesian coordinates
	std::vector< BondSmallMolecule * > bond_vector; // owning pointers
	// of bonds between Atom objects in atom_vector
	std::map< std::pair< ligand::Ligand_AtomAP, ligand::Ligand_AtomAP >, std::vector< std::pair< int, int > > > dihedral_minima;
	// storage container for allowed dihedral torsions of bonds.
	// stores a pair for the Atom pointers on either side of the bond
	// and a vector of pairs of ints for the dihedral minima in degrees
	// and absolute deviation from the minima

	std::vector< ligand::Ligand_AtomAP > trial_roots; // non-owning
	// pointers to Atom objects at the root of bond trees.

	std::vector< std::vector< std::pair< size_t, numeric::xyzVector< float > > > > ligand_copy_trans_rot;
	//these are used to store the necessary rotations and translations required for
	//initializing multiple copies of the small molecule type in a pose

	std::vector< ligand::LigandInternalCoord > ligand_conformations_base;
	//container for complete ligand rotamer library
	std::vector< ligand::LigandInternalCoord > ligand_conformations;
	// internal coordinates of conformations for a flexible ligand
	std::string ligand_conformations_outfile;
	bool output_ligand_conformations;
	bool superimpose_output_rotamers;


	float rotamerize_ligand_lower_bound;
	//ligand_rotamer must be at least the lower bound RMSD
	//from the input conformation
	float rotamerize_ligand_upper_bound;
	//ligand_rotamer cannot be more than the upper bound RMSD
	//from the input ligand conformation

	int delta_angular_sampling;
	// The number of degrees a dihedral is to be rotated
	// during conformational sampling
	float minimal_rms_limit;
	// The minimal_rms between accepted conformations
	// during while generating conformations

	int maximum_number_of_conformations;



	numeric::xyzVector< double > anchor_first_start; // anchors used during flexible
	numeric::xyzVector< double > anchor_second_start;// ligand docking
	numeric::xyzVector< double > anchor_third_start;
	numeric::xyzVector< double > anchor_first_best; // anchors used during flexible
	numeric::xyzVector< double > anchor_second_best;// ligand docking
	numeric::xyzVector< double > anchor_third_best;
	numeric::xyzVector< double > anchor_first; // anchors used during flexible
	numeric::xyzVector< double > anchor_second;// ligand docking
	numeric::xyzVector< double > anchor_third;
	size_t anchor_state;


	std::list<LigandAtomPairEnergy> ligand_atom_pair_energy;

	FArray1D_float hetero_born_radius; // Born radii
	float hetero_atom_total_charge; // total charge of heteroatoms
	FArray1D_int virtual_atom_bnd;
	// index of hetero atom that is covalantly bond to a virtual atom
	FArray1D_int virtual_atom_map;
	// index of amino_acid and atom that is currently mapped on a virtual atom
	FArray1D_string hetero_atom_resid;
	// name of hetero residues
	int virtual_atom_count; // virtual atom count
	FArray1D_int hetero_atom_hbond_acc;
	// pairs of hetero atoms that may accept a h-bond (e.g. >C=O)
	std::map< size_t, size_t >  atom_base;
	std::map< size_t, size_t > abase2;
	FArray1D_bool donor_satisfied;
	FArray1D_bool acceptor_satisfied;
	FArray1D_int hetero_atom_hbond_don;
	FArray1D_int don_base_if_OH;
	// pairs of hetero atoms that may donate a h-bond (e.g. -O-H)
	int hetero_atom_hbond_acc_count;
	// count of hetero atoms that may accept a h-bond (e.g. >C=O)
	int hetero_atom_hbond_don_count;
	// count of hetero atoms that may donate a h-bond (e.g. -O-H)
	bool recompute_ligand_interface;
	// flag that reports if interface needs to be recomputed
	int lig_neighbors;
	float lig_repE;
	float lig_atrE;
	float lig_solE;
	float lig_couE;
	float lig_h2oE;
	float lig_hbE;
	float lig_virE;
	float lig_sumE;
	float lig_sasE;
	float lig_int_repE;
	float lig_int_atrE;
	float lig_int_solvE;
	float lig_int_dihedE;
	float lig_int_coulombE;
	float lig_int_hbE;

// functions
	void
	get_FArray2D_of_coordinates(
		std::vector < ligand::Ligand_AtomOP > & ligand_atom,
		FArray2D_float & het_coord
	);

	void
	get_FArray1D_of_charge(
		std::vector < ligand::Ligand_AtomOP > & ligand_atom,
		FArray1D_float & het_charge
	);

	void
	get_FArray1D_of_atom_type(
		std::vector < ligand::Ligand_AtomOP > & ligand_atom,
		FArray1D_int & het_atom_type
	);

	int
	setup_bond_graph(
		int const root_atom_index
	);

	int
	set_bondtypes_by_distance(
	);

	int
	get_minimal_spanning_bond_tree_from_coordinates(
		std::map < size_t, numeric::xyzVector < double > > & coordinates,
		std::list < numeric::xyzVector < size_t >  > & bond_tree,
		size_t start_vertex
	);

	size_t
	get_nearest_heavy_atom(
		ligand::Ligand_AtomAP const atom_i
	);

	int
	setup_bond_graph(
		std::list < numeric::xyzVector < size_t > > & bond_map,
		int const root_atom
	);

	int
	get_bond_map(
		std::list < numeric::xyzVector < size_t > > & bond_map
	);

	int
	set_ring_rigid(
		BondSmallMolecule & bond,
		ligand::Ligand_AtomOP const atom2,
		std::list< BondSmallMolecule * > & rigid_bonds
	);

	int
	setup_hetero_atom_hbond_arrays(
	);

	int
	set_ligand_atom_types(
	);

	void
	set_rosetta_atom_types(
	);

	size_t
	get_atom_closest_to_centroid(
	);

	int
	get_number_of_double_bonded_oxygens(
		ligand::Ligand_AtomOP const atom1
	);

	int
	get_hybridization(
		ligand::Ligand_AtomOP const atom1
	);

	int
	get_number_of_hydrogens(
		ligand::Ligand_AtomOP const atom1
	);

	int
	get_number_of_aromatic_bonds(
		ligand::Ligand_AtomOP const atom1
	);

	bool
	Are_Bonded(
		ligand::Ligand_AtomOP const atom1,
		ligand::Ligand_AtomOP const atom2
	);

	bool
	calculate_internal_coordinates(
		bool const initialize
	);

	int
	twist_dihedral_radians(
	double const dihedral,
	BondSmallMolecule & bond
	);

	int
	set_dihedral_degrees(
		ligand::Ligand_BondAP bond_ptr,
		double const degrees
	);

	int
	set_best_coordinates(
	);

	int
	recover_best_coordinates(
	);

	int
	set_start_coordinates(
	);

	int
	recover_start_coordinates(
	);

	inline
	int
	set_anchors(
		numeric::xyzVector< double > anchor_one,
		numeric::xyzVector< double > anchor_two,
		numeric::xyzVector< double > anchor_three
	){
		anchor_first=anchor_one;
		anchor_second=anchor_two;
		anchor_third=anchor_three;
		return 1;
	};

	float
	get_ligand_internal_energy(
	bool const initialize
	);

	int
	refold_ligand(
	);

	int
	refold_atom(
		FArray1Dp_bool up_to_date,
		ligand::Ligand_AtomOP atom_to_refold
	);

	int
	compute_ligand_conformations(
	);

	int
	load_crystal_structure_dihedral_minima(
		std::list< ligand::Ligand_BondAP > & rotable_bonds
	);

	float
	prune_ligand_conformations(
	std::multimap< float, ligand::LigandInternalCoord > & conformations,
	float const energy_delta_cutoff,
	int const num_max_conf
	);

	int
	diversify_ligand_conformations(
		std::map< ligand::Ligand_BondAP, int > dihedral_sampling,
		int angular_density,
		float rms_density,
		size_t maximum_num_conf
	);

	void
	make_conformer_nearest_start(
		ligand::LigandInternalCoord & output_bonds
	);

	float
	iterative_conformation_search(
		std::multimap< float, ligand::LigandInternalCoord > & conformations,
		size_t allowed_number_of_conformations
	);
	
	float
	monte_carlo_conformation_search(
		std::multimap < float, ligand::LigandInternalCoord > & conformations,
		size_t allowed_number_of_conformations
	);

	int
	change_to_ligand_conformation(
		size_t const x
	);


	void
	get_bonded_atoms(
		std::list< ligand::Ligand_AtomAP > & bonded_atoms,
		ligand::Ligand_AtomAP const first_atom
	);

	int
	rotamerize_ligand(
	);

	size_t
	find_nearest_conformation_by_rmsd(
		ligand::LigandCartesianCoord passed_conformation,
		float & rmsd
	);

	void
	optimize_H(
	);

	int
	setup_ligand_res(
		int const aa, // lig amino acid type
		int const aav, // lig amino acid variant set to 1
		bool const read_occ_weights,
		bool const read_bvalue_charge,
		FArray3D_float & coords // returns current coordinates
	);

	int
	fill_in_heavy_atoms(
		FArray1D< Atom * > & atom_array_for_pose_res,
		int const aa,
		int const aav
	);

	int
	fill_in_hydrogens_and_virtual_atoms(
		FArray1D< Atom * > & atom_array_for_pose_res,
		int const last_assigned_index,
		int const aa,
		int const aav
	);

	int
	set_up_chi_arrays(
		int const aa,
		int const aav
	);

	void
	get_base_atoms(
		ligand::Ligand_AtomAP & atom_1,
		ligand::Ligand_AtomAP & atom_2,
		ligand::Ligand_AtomAP & atom_3
	);

	void
	setup_bonded_neighbors_arrays(
		int const atom_count,
		int const aa,
		int const aav
	);

	int
	fill_acceptor_arrays(
		int const atom_count,
		int const aa,
		int const aav
	);


	int
	get_rotable_bonds(
		std::list< ligand::Ligand_BondAP > & rotable_bonds
	);

	int
	get_size_of_branch(
		const ligand::Ligand_BondAP
	);

	int
	fill_ligand_rotamer_array(
		FArray1D< ligand::Ligand_AtomAP > & atom_array_for_aaproperties,
		int const aa,
		int const aav
	);

	int output_ligand_conformations_sdfile(
		std::string const filename
	);



};
namespace ligand{
	extern std::vector<Ligand * > ligand_ptr_vector;
	extern Ligand * ligand_one;
	extern Ligand * ligand_two;
	extern bool ligand_flag; // flag for reading, handling and writing potential ligands in docking/packing
	extern bool ligand_flexible;
	extern bool ligand_mdlfile;
	typedef std::vector<ligand::Ligand_AtomOP > ::iterator Atom_Itr;
	typedef std::vector< BondSmallMolecule * > ::iterator Bond_Itr;
	extern std::list < std::pair <
					std::pair<atom_chem::LigandAtomType,atom_chem::LigandAtomType>,
					utility::PeriodicSplineReader > > dihedral_splines;
	extern std::vector<int> ligand_residue_numbers; //flo vector to keep track of ligand residue numbers
}

#endif
