// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13616 $
//  $Date: 2007-03-18 08:39:36 +0200 (Sun, 18 Mar 2007) $
//  $Author: stuartm $


// Rosetta Headers
#include "maps.h"
#include "after_opts.h"
#include "current_pose.h"
#include "files_paths.h"
#include "initialize.h"
#include "maps_ns.h"
#include "misc.h"
#include "monte_carlo.h"
#include "param.h"
#include "pose.h"
#include "refold.h"
#include "assemble_domains.h"
#include "namespace_assemble_options.h"
#include "score.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/formatted.io.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <algorithm>
#include <cassert>
#include <cmath>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <limits>

////////////////////////////////////////////////////////////////////////////////
/// @begin insert_map_get_max_contig_size
///
/// @brief
///   returns the length of the longest contiguous insertable region
///   and the total number of insertable residues
///
/// @detailed
///
/// @param[out]   max_contig_size - out -  length of longest insertable region
/// @param[out]   total_size - out - total number of insertable residues
///
/// @global_read
///  total_residue misc.h
///  allow_insert  maps_ns.h
///  total_size    maps_ns.h
///
/// @global_write
///
/// @remarks
///cems this is useful to know what the largest fragment you should try
///cems inserting is (ie dont try putting in 9's if there's no insertable
///cems  region of length 9 around
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
insert_map_get_max_contig_size(
	int & max_contig_size,
	int & total_size
)
{
	using namespace misc;
	using namespace protein_maps;

	int candidate_contig_size = 0;
	max_contig_size = 0;

	for ( int i = 1; i <= total_residue; ++i ) {
		if ( allow_insert(i) ) {
			++candidate_contig_size;
		} else {
			max_contig_size = std::max(max_contig_size,candidate_contig_size);
			candidate_contig_size = 0;
		}
	}
	max_contig_size = std::max(max_contig_size,candidate_contig_size); // check C-term
	total_size = total_insert;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin reset_insert_map
///
/// @brief
///    rebuilds the insert map  to be consistent with
///    the current allow_insert settings
///
/// @detailed
///
/// @param[out]   exist - out - do any insertable residue exist?
///
/// @global_read
/// total_residue  misc.h
/// allow_insert   maps_ns.h
///
/// @global_write
/// insert_map     maps_ns.h
/// total_insert   maps_ns.h
///
/// @remarks
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
reset_insert_map()
{
	using namespace misc;
	using namespace protein_maps;

	total_insert = 0;
	for ( int ires = 1; ires <= total_residue; ++ires ) {
		if ( allow_insert(ires) ) {
			++total_insert;
			insert_map(total_insert) = ires;
		}
	}
	if ( total_insert > 0 ) return true;
	return false;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_maps
///
/// @brief  initialize maps that define movable backbone and sidechain segments
///
/// @detailed
///
/// @global_read
/// total_residue misc.h
/// domain_end    misc.h
/// idealized_structure files_paths.h
///
/// @global_write
///  protein_maps block  maps_ns.h
///
/// @remarks
///car maps are required for efficient fragment insertion, folding, packing and
///car scoring
///car see maps_ns.h for explanations of the maps
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
initialize_maps()
{
	using namespace files_paths;
	using namespace misc;
	using namespace protein_maps;

	total_insert = 0;

	{ for ( int ires = 1; ires <= total_residue; ++ires ) {
		allow_rottrial(ires) = true;
		allow_repack(ires) = true;
		if ( idealized_structure ) {
			allow_insert(ires) = true;
			++total_insert;
			insert_map(total_insert) = ires;
		} else {
			allow_insert(ires) = false;
		}
	} } //Objexx:SGM Extra {} scope is a VC++ work-around

	//car initialize refold segments
	FArray1D_int seg_begin( total_domains );
	seg_begin( 1 ) = 1;
	for ( int iseg = 2; iseg <= total_domains; ++iseg ) {
		seg_begin( iseg ) = domain_end( iseg - 1 ) + 1;
	}
	refold_set_segments( 1, total_domains, seg_begin, domain_end,
    FArray1D_int( total_domains, 0 ) );  // automatic direction selection
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_allow_insert_by_region
///
/// @brief - set a single region of allow_insert
///
/// @detailed
///
/// @param[in]   first_res -in-
/// @param[in]   last_res- in -
/// @param[in]   yes_no - int -
/// @param[out]   exist - out -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  glb
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_allow_insert_by_region(
	int const first_res,
	int const last_res,
	bool const yes_no,
	bool & exist
)
{
	using namespace misc;
	using namespace protein_maps;

	for ( int i = 1; i <= total_residue; ++i ) {
		allow_insert(i) = !yes_no;
	}

	for ( int i = first_res; i <= last_res; ++i ) {
		allow_insert(i) = yes_no;
	}

	exist = reset_insert_map();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin retrieve_allow_insert
///
/// @brief retrieve current value of allow_insert global array
///
/// @detailed
///
/// @param[out]   allow - out - current setting of allow_insert map
/// @param[in]   total_residue - in - logical size of allow
///
/// @global_read
/// allow_insert  maps_ns.h
///
/// @global_write
///
/// @remarks
///car allow_insert indicates residues variable in torsion space (ie insertable)
///car not necessarily fixed in cartesian space
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
retrieve_allow_insert(
	FArray1DB_bool & allow,
	int total_residue
)
{
	using namespace protein_maps;

	if ( moves_check_current_pose() || score_check_current_pose() ) {
		// POSE
		pose_ns::Pose const * pose_ptr(0);
		if ( moves_check_current_pose() ) {
			pose_ptr = &(moves_get_current_pose() );
		} else if ( score_check_current_pose() ) {
			pose_ptr = &(score_get_current_pose() );
		} else {
			// not possible to get here
			std::cout << "STOP:: retrieve_allow_insert!" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		for ( int i=1; i<= total_residue; ++i ) {
			allow(i) = pose_ptr->get_allow_bb_move( i );
		}
	} else {
		// non-pose
		for ( int i = 1; i <= total_residue; ++i ) {
			allow(i) = allow_insert(i);
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin retrieve_insertmap
///
/// @brief retrieve list of insertable residues
///
/// @detailed
///
/// @param[out]   map - out - list of insertable residues
/// @param[out]   length - out - number of insertable residues
///
/// @global_read
///  insert_map   maps_ns.h
///  total_insert maps_ns.hc
///
/// @global_write
///
/// @remarks
///car insert_map is a list of residues that are variable in torsion space.
///car alway coincident with allow_insert array
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
retrieve_insertmap(
	FArray1DB_int & map, // should be declared of size MAX_RES
	int & length
)
{
	using namespace protein_maps;

	if ( moves_check_current_pose() || score_check_current_pose() ) {
		// POSE
		pose_ns::Pose const * pose_ptr(0);
		if ( moves_check_current_pose() ) {
			pose_ptr = &(moves_get_current_pose() );
		} else if ( score_check_current_pose() ) {
			pose_ptr = &(score_get_current_pose() );
		} else {
			std::cout << "STOP:: retrieve_insertmap!" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		FArray1D_int const & map_from_pose( pose_ptr->get_insert_map( length ) );
		assert( int(map.size1()) >= length );
		for ( int i=1; i<= length; ++i ) {
			map(i) = map_from_pose(i);
		}
	} else {
		// non-pose
		length = total_insert;
		for ( int i = 1; i <= length; ++i ) {
			map(i) = insert_map(i);
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin setup_variable_regions
///
/// @brief read variable regions from a file and set the allow_insert arrays appropriately
///
/// @detailed
///car implements -regions option which restricts insertions to a portion of a
///car structure as defined by a user-provided input file.  Chain continuity is
///car maintained, fixed regions are fixed in torsion space, but not cartesian
///
/// @global_read
///  start_path  files_paths.h
///  start_file  files_paths.h
///  total_residue misc.h
///  init_phi    param.h
///  init_psi    param.h
///  init_omega  param.h
///
/// @global_write
///  use_regions  static   option in-use flag
///  init         static   self-initialization flag
///  extension    static   region file extension
///  phi   misc.h
///  psi   misc.h
///  omega misc.h
///  allow_insert maps_ns.h
/// @remarks
///car link/region file format: 1 variable region per line
///car region_start region_end  zero_start zero_end
///car  zero_start,zero_end span the region for which torsion angles
///car  cannot be correctly determined from coordinates
///car  these regions will be initialized to extended chain
///
///car default file extension is 'regions', can be set with  -regionfile option
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
setup_variable_regions()
{

	using namespace files_paths;
	using namespace misc;
	using namespace param;
	using namespace protein_maps;

	std::string fullname;

	int begin,end,zero_begin,zero_end;
	bool exist;

	static bool init = { false };
	static bool use_regions;
	static std::string extension;

	if ( !init ) {
		use_regions = false;
		if ( truefalseoption("regions") ) {
			use_regions = true;
			stringafteroption( "regionfile", "regions", extension );
		}
		init = true;
	}
	if ( !use_regions ) return;

	fullname = start_path + start_file + '.' + extension;
	std::cout << "Looking for region file: " << fullname << std::endl;

	utility::io::izstream region_stream( fullname );

	if ( !region_stream ) {
		std::cout << "STOPPING! region file not found!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	for ( int i = 1; i <= total_residue; ++i ) {
		allow_insert(i) = false;
	}

	while ( true ) {
		region_stream >> bite( 5, begin ) >> skip( 1 ) >>
		 bite( 5, end ) >> skip( 1 ) >>
		 bite( 5, zero_begin ) >> skip( 1 ) >>
		 bite( 5, zero_end ) >> skip;
		if ( region_stream.eof() ) goto L100;
		begin = std::min(std::max(begin,1),total_residue); // dont write past end of arrays
		end = std::min(std::max(end,1),total_residue);
		zero_begin = std::min(std::max(zero_begin,1),total_residue);
		zero_end = std::min(std::max(zero_end,1),total_residue);
		std::cout << "begin,end,zero_begin,zero_end " <<
		 begin << ' ' << end << ' ' << zero_begin << ' ' << zero_end << std::endl;

		for ( int i = begin; i <= end; ++i ) {
			allow_insert(i) = true;
		}

		if (! get_assemble_flag() ) {
			 if ( zero_begin <= zero_end ) {
				 psi(zero_begin) = init_psi;
				 omega(zero_begin) = init_omega;
				 for ( int i = zero_begin + 1; i < zero_end; ++i ) {
					 phi(i) = init_phi;
					 psi(i) = init_psi;
					 omega(i) = init_omega;
				 }
				 phi(zero_end) = init_phi;
			 }
		}

	}
L100:

	exist = reset_insert_map();
	if ( !exist ) {
		std::cout << "ERROR:: no variable regions defined in region file!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	monte_carlo_accept_best(); // fill best_position array
	refold(1,total_residue);
	save_START();

	region_stream.close();
	region_stream.clear();
}


////////////////////////////////////////////////////////////////////////////////
/// @begin retrieve_pair_moved
///
/// @brief
///
/// @detailed
///
/// @param  begin - [in/out]? -
/// @param  size - [in/out]? -
/// @param  nres - [in/out]? -
/// @param  res_moved - [in/out]? -
/// @param  pair_moved - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///  could be a problem if allow_insert is changed--
///  need to control this (or skip the allow_insert checks)
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
FArray2D_bool const &
retrieve_pair_moved( bool const bb_only )
{
	using namespace move_map;
	using namespace misc;

	if ( score_check_current_pose() ) {
		return score_get_current_pose().get_pair_moved();
	}

	if ( ! movemap_current ) update_pair_moved();
	if ( bb_only ) return pair_moved_bbonly;
	return pair_moved;
}

FArray1D_bool const &
retrieve_res_moved( bool const bb_only )
{
	using namespace move_map;

	if ( score_check_current_pose() ) {
		return score_get_current_pose().get_res_moved();
	}

	if ( ! movemap_current ) update_pair_moved();
	if ( bb_only ) return res_moved_bbonly;
	return res_moved;
}

void
update_pair_moved()
{
	using namespace move_map;
	using namespace param;
	using namespace misc;

	assert( ( equal_dimensions( pair_moved, pair_moved_bbonly ) ) ); //Objexx: Assumed below
	assert( !score_check_current_pose() );

//car set pair_moved and res_moved to values determined by new_rotamer
//car and phipsi moved
	for ( int j = 1; j <= total_residue; ++j ) {
		res_moved(j) = ( new_rotamer(j) || new_phipsi(j) );
		res_moved_bbonly(j) = ( new_phipsi(j) );
  }

	for ( int j = 1; j <= total_residue; ++j ) {
		bool j_res_moved_bb = res_moved_bbonly(j);
		bool j_res_moved = res_moved(j);
		int j_newrb = new_rigidbody(j);
		for ( int i = 1, l = pair_moved.index(i,j); i <= total_residue; ++i, ++l ) {
			pair_moved[ l ] =  // pair_moved(i,j)
        (j_res_moved || res_moved(i) ||
         ( new_rigidbody(i) != j_newrb ) );
			pair_moved_bbonly[ l ] =  // pair_moved_bbonly(i,j)
        (j_res_moved_bb || res_moved_bbonly(i) ||
         (new_rigidbody(i) != j_newrb ) );

		}
		//		std::cout << SS(j) << SS(bbpair_moved_start(j)) << SS(pair_moved_end(j)) << std::endl;
	}
	movemap_current = true;


}
////////////////////////////////////////////////////////////////////////////////
/// @begin maps_set_new_phipsi
///
/// @brief marks residues as having new coordinates as a result of new torsion angles
///
/// @detailed
///
/// @param[in]   begin
/// @param[in]   end
///
/// @global_read
///
/// @global_write
///  new_phipsi
///
/// @remarks
///car  new_phipsi  should define the entire region in
///car  which the backbone torsion angles differ between the current and best
///car  pose.  This is required for accurate scoring and updating of coordinates.
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
maps_set_new_phipsi(int const begin, int const end, int const seg_begin, int const seg_end, int const dir )
{
	using namespace move_map;

	//car idiot hunt
	if ( begin > end ) return;

	int const n2c = 1;

	for ( int i = begin; i <= end; ++i ) {
		new_phipsi(i) = true;
	}

	//car in addition to changing torsions, refold made a rigid body perturbation
	// of a chain segment

	if ( dir == n2c ) {
		maps_set_new_rigidbody(end + 1, seg_end);
	} else {
		maps_set_new_rigidbody(seg_begin, begin-1);
	}
	movemap_current = false;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin maps_set_new_rotamer
///
/// @brief
///
/// @detailed
///
/// @param  residue - [in] -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
maps_set_new_rotamer(
  int const residue
)
{
	using namespace move_map;

	new_rotamer(residue) = true;
	movemap_current = false;
}
////////////////////////////////////////////////////////////////////////////////
/// @begin maps_set_new_rigid_body
///
/// @brief  -- mark residues begin to end as having undergone a rigid body
///            transformation
///
/// @detailed
///
/// @param  begin - [in] -
/// @param  end - [in] -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
maps_set_new_rigidbody( int const begin, int const end)
{
	using namespace misc;
	using namespace move_map;

	//car idiot check
	if ( begin > end ) return;
	//	std::cout << "NEWRB" << SS(begin) << SS(end) << SS(rbmoves.size()) << std::endl;

//car see if this is a move we've made before:
	{ for ( RbMoves::const_iterator iter = rbmoves.begin(), itere = rbmoves.end(); iter != itere; ++iter ) {
	 	if ( begin == (*iter).begin && end == (*iter).end ) return;
	} } //Objexx:SGM Extra {} scope is a VC++ work-around

//car  check for reset
	if ( rbmoves.size() == 0 ) nrb_max_value = 0;

//save this move
	rbmoves.push_back( RbMove( begin, end ) );


//car check for overflow or nrb_max_value
	if ( 2 * nrb_max_value + 1 > std::numeric_limits< int >::max() )  {
		//  if (false ) {
//car renumber map
		int new_value = 0;
		FArray1D_int nrb_copy( new_rigidbody );
		for ( int i = 1; i<= total_residue; ++i ) {
			int old_value = nrb_copy(begin);
			if ( old_value == 0 ) continue;  // leave zeros in new_rigidbody
			++new_value;
//car search for all occurences of old_value in the rest of the array
//car  and change them to new_value
			for ( int j = i; j <= total_residue; ++j ) {
				if ( nrb_copy(j) == old_value ) {
					new_rigidbody(j) = new_value;
					nrb_copy(j) = 0;  //mark as updated so skipped when i = this value
				}
			}  //j
		}  //i
		nrb_max_value = new_value;
		if ( 2 * nrb_max_value + 1 > std::numeric_limits< int >::max() )  {
			std::cerr << "Overflow of nrb_max_value in maps_set_new_rigid_body\n";
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}  //end overflow check

//car  mark the rigid body movement
//car everything in this region needs a new number, but if they have different
//car numbers already, they need to stay different
	int const increment = nrb_max_value + 1;
	for ( int i = begin; i <= end; ++i ) {
		new_rigidbody(i) += increment;
		//std::cout << "RBUP" << SS(i) << SS(new_rigidbody(i)) << SS(increment) << SS(largest_value) << std::endl;
	}
	nrb_max_value += increment;
	movemap_current = false;
	//std::cout << "NEWRB2" << SS(begin) << SS(end) << std::endl;

}
////////////////////////////////////////////////////////////////////////////////
/// @begin reset_move_map
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
reset_move_map()
{
	using namespace move_map;

	new_rotamer = false;
	new_phipsi = false;
  new_rigidbody = 0;
	nrb_max_value = 0;
  rbmoves.clear();
	movemap_current = false;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin maps_set_new_pose
///
/// @brief
///
/// @detailed
///car notify scorefxn that a new structure is in the arrays and
///car internal info shouldn't be used
///car this allows a new position array to be scored, but the internal
///car array data is not updated
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
maps_set_new_pose()
{

	using namespace move_map;

// car mark all residues as having moved
	new_rotamer = true;
	new_phipsi = true;

// car and since its trivial to calculate:
//  note its not sufficient to just set pair_moved and res_moved
//  because movemap_current may get set false by some other routine
//  so if pair_moved is recomputed, it must be computed properly.
	pair_moved = true;
	pair_moved_bbonly = true;
	res_moved = true;
	res_moved_bbonly = true;
	movemap_current = true;

}

///////////////////////////////////////////////////////////////////////////////
/// @begin debug_move_maps
///
/// @brief  saves move maps and then marks entire structure as changed;
///         saved maps are restored when debug is turned off
///
/// @detailed
/// @param  on [in]  turning on?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car
///
/// @last_modified 9/1/2004
////////////////////////////////////////////////////////////////////////////////
void
debug_move_maps( bool on )
{
	using namespace param;
	using namespace move_map;

	static FArray1D_bool save_new_phipsi( MAX_RES() );
	static FArray1D_bool save_new_rotamer( MAX_RES() );
	static FArray1D_int  save_new_rigidbody( MAX_RES() );
	static int save_nrb_max_value;
	static RbMoves save_rbmoves;

	static bool saved = { false };
	static int restore = { 0 };

	if ( on ) {
		saved = true;
		save_new_phipsi = new_phipsi;
		save_new_rotamer = new_rotamer;
		save_new_rigidbody = new_rigidbody;
		save_nrb_max_value = nrb_max_value;
		save_rbmoves = rbmoves;
		restore = 0;

		maps_set_new_pose();

	} else {
		std::cout << "WARNING:: RESTORING MOVE_MAPS!!! " <<
			" THIS ROUTINE SHOULD ONLY BE USED FOR DEBUGGING!!" << std::endl;
		if ( ! saved ) {
			std::cout << "WARNING:: move maps have not been saved, no restore possible!! " << std::endl;
			return;
		}
		if ( restore > 0 ) {
			std::cout << "WARNING:: this configuration of move maps has already been restored " << restore << " times" << std::endl;
		}
		++restore;

		new_phipsi = save_new_phipsi;
		new_rotamer = save_new_rotamer;
		new_rigidbody = save_new_rigidbody;
		nrb_max_value = save_nrb_max_value;
		rbmoves = save_rbmoves;
		movemap_current = false;
	}
}

///////////////////////////////////////////////////////////////////////////////
/// @begin maps_get_pair_moved_region
///
/// @brief  define region outside of which pair_moved is always false
///         considering only backbone movement
///
/// @detailed  when this function returns true, it is guaranteed that
///    pair_moved_bbonly(i,j) = false if i and j are both < begin or both > end
///
/// @param  begin [out] defines region for which pair_moved is false
/// @param  end [out] defines region for which pair_moved is false
///
/// @global_read
///
/// @global_write
///
/// @remarks
///     once we identify begin and end, we assume pair_moved looks like
///     the diagram below, allowing us to skip some regions in loops
///     this strategy is used in vdw_compute.

///    |<---- i axis ----->|
///    F . . . . . . . . . . __
///    F F . . . . . . . . .  ^
///    F F F . . . . . . . .  |
///    F F F F . . . . . . .  |
///    * * * * * . . . . . .  J   <----  Begin
///    * * * * * * . . . . .  a
///    * * * * * * * . . . .  x   <----  End
///    * * * * * * * F . . .  i
///    * * * * * * * F F . .  s
///    * * * * * * * F F F .  |
///    * * * * * * * F F F F  _
///    |<---- i axis ----->|
///
///   *: residue pairs that changed;
///   F: residue pairs that may have changed
///   .: mirror image
///
///
///  Here are comments from charlie strauss on the speedup enabled
///
/// worst_case scenario:
///  if we assume that total_residue >> (end_res-begin_res), then the area of the *
///  region is a little over 1/2 the whole triangle.  So we would have a factor
///  of two speed-up.
///
/// best_case scenario:
///   when the insert region is at the end or the beginning, then the area of
///   the * region is roughly  2*frag_size/total_residue  of the whole region
///   and there's an enormous speedup.
///
/// Amortized scenario:
///   Assuming begin_res is unifrmly distributed on average then the average
///   speed up is somewhat less than a factor of four.
///
/// @references
///
/// @authors car
///
/// @last_modified 10/19/2004
////////////////////////////////////////////////////////////////////////////////
void
maps_get_pair_moved_region(int & begin, int & end )
{

	using namespace move_map;

	int id;
	for ( begin = 1, id = new_rigidbody(begin);
				begin < misc::ints::total_residue && !new_phipsi(begin) &&
				new_rigidbody(begin) == id; ++begin );

	for ( end = misc::ints::total_residue, id = new_rigidbody(end);
				end > 1 &&  !new_phipsi(end) &&  new_rigidbody(end) == id; --end );

}
