// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 19150 $
//  $Date: 2007-12-15 09:18:46 +0200 (Sat, 15 Dec 2007) $
//  $Author: yab $


// Rosetta Headers
#include "minimize.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "all_atom_type.h"
#include "cenlist.h"
#include "constraints.h"
#include "count_pair.h"
#include "current_pose.h"
#include "disulfides.h"
#include "dna.h"
#include "docking.h"
#include "docking_minimize.h"
#include "dock_pivot.h"
#include "dock_structure.h"
#include "fullatom.h"
#include "fullatom_energy.h"
#include "hbonds.h"
#include "jumping_minimize.h"
#include "ligand.h"
#include "ligand_ns.h"
#include "loops.h"
#include "maps.h"
#include "maps_ns.h"
#include "minimize_chi.h"
#include "minimize_ns.h"
#include "misc.h"
#include "monte_carlo.h" // yab: misc removal
#include "nblist.h"
#include "nblist_ns.h"
#include "pack.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "param_torsion.h"
#include "pdbstatistics_pack.h"
#include "pose.h"
#include "pose_backrub.h"
#include "prof.h"
#include "ramachandran.h"
#include "read_aaproperties.h"
#include "refold.h"
#include "rotamer_trials.h"
#include "runlevel.h"
#include "score.h"
#include "score_ns.h"
#include "tether.h"
#include "tether_ns.h"
#include "util_basic.h"
#include "util_vector.h"
#include "vdw.h"
#include "wobble.h"
#include "water_ns.h" // PBHACK

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/formatted.o.hh>

// Numeric Headers
#include <numeric/conversions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>

// C++ Headers
#include <algorithm>
#include <cassert>
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <vector>


//cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
//------------------------------------------------------------------------------
// MINIMIZER INTERFACE
//    --all functions collected below here are for changing or querying the
//    minimizer's state variables ONLY.
//    -- minimization functions follow this block
//------------------------------------------------------------------------------
//cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_reset
///
/// @brief
///    This function completely resets the minimizer to its ground state.
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
minimize_reset()
{

	using namespace minimize_ns;

// default to using phi/psi only!!
	minimize_vary_phipsi = minimize_vary_phipsi_default;
	minimize_vary_omega = minimize_vary_omega_default;
	minimize_vary_chi = minimize_vary_chi_default;
	minimize_vary_rb_angle = minimize_vary_rb_angle_default;
	minimize_vary_rb_trans = minimize_vary_rb_trans_default;

	minimize_excludeH = minimize_excludeH_default;
	minimize_excludeE = minimize_excludeE_default;

	minimize_local_min = minimize_local_min_default;
	minimize_local_window = minimize_local_window_default;

//car set the dfpmin and frpmin tolerance for min moves
//car note that this is _not_ the tolerance for linmin
	minimize_tolerance = minimize_tolerance_default;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_set_tolerance
///
/// @brief
///car set the dfpmin and frpmin tolerance for min moves
///car note that this is _not_ the tolerance for linmin
///
/// @detailed
///
/// @param  setting - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
minimize_set_tolerance( float const setting )
{

	using namespace minimize_ns;

	minimize_tolerance = setting;
}

float
minimize_get_tolerance()
{

	using namespace minimize_ns;

	return minimize_tolerance;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_uninit_exclude_list
///
/// @brief
///car changes the state of the exclude list to uninitialized so that
///car when the next min move is started, the list will be reset according
///car to the current structure and settings
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
minimize_uninit_exclude_list()
{

	using namespace minimize_ns;

	minimize_exclude_list_init = false;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_exclude_sstype
///
/// @brief
///car mark helical and strand residues so that they are or are not minimized
///car by calls to minimize
///car the exclude list is used by pack_phispi
///
/// @detailed
///
/// @param  Hsetting - [in/out]? -
/// @param  Esetting - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
minimize_exclude_sstype(
	bool Hsetting,
	bool Esetting
)
{

	using namespace minimize_ns;

	minimize_excludeH = Hsetting;
	minimize_excludeE = Esetting;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_set_local_min
///
/// @brief
///
/// @detailed
///
/// @param  on_off - [in/out]? -
/// @param  window_size - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
minimize_set_local_min(
	bool on_off,
	int window_size
)
{
	using namespace minimize_ns;

	minimize_local_min = on_off;
	minimize_local_window = window_size;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_set_vary_phipsi
///
/// @brief
///
/// @detailed
///
/// @param  setting - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
minimize_set_vary_phipsi( bool const setting )
{
	using namespace minimize_ns;

	minimize_vary_phipsi = setting;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_set_vary_omega
///
/// @brief
///
/// @detailed
///
/// @param  setting - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
minimize_set_vary_omega( bool const setting )
{
	using namespace minimize_ns;

	minimize_vary_omega = setting;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_set_vary_chi
///
/// @brief
///
/// @detailed
///
/// @param  setting - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///ctsa this isn't yet ci for general use, because of:
///ctsa 1) conflicting rot trials and minimizer potentials
///ctsa     (P(aa|phi,psi))
///ctsa 2) no dunbrack torsion for chi 3,4 yet
///ctsa 3) sc-bb clashes need to be addressed
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
minimize_set_vary_chi( bool const setting )
{

	using namespace minimize_ns;

	minimize_vary_chi = setting;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_set_vary_rb_angle
///
/// @brief
///
/// @detailed
///
/// @param  setting - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
minimize_set_vary_rb_angle( bool const setting )
{

	using namespace minimize_ns;

	minimize_vary_rb_angle = setting;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_set_vary_rb_trans
///
/// @brief
///
/// @detailed
///
/// @param  setting - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
minimize_set_vary_rb_trans( bool const setting )
{

	using namespace minimize_ns;

	minimize_vary_rb_trans = setting;
}



//cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
//------------------------------------------------------------------------------
//   END MINIMIZER INTERFACE
//------------------------------------------------------------------------------
//cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc



////////////////////////////////////////////////////////////////////////////////
/// @begin minimize
///
/// @brief
///
/// @detailed
///
/// @param  min_type - [in/out]? -
/// @param  move_type - [in/out]? -
/// @param[in]  score_fxn - in - passed in to ensure correct setup of scorefxn
/// @param  gfrag - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
minimize(
	std::string const & min_type,
	std::string const & move_type,
	Scoring_Function score_fxn, // passed in to ensure correct setup of scorefxn
	int const fold_begin,
	int const fold_end,
	bool & gfrag
)
{
	using namespace minimize_ns;
	using namespace param;
	using namespace param_torsion;
	using namespace runlevel_ns;

	// temporary:
	if ( minimize_check_current_pose() ) {
		pose_minimize( minimize_get_current_pose(), min_type, move_type,
									 score_fxn, gfrag );
		return;
	}

//car parameters
//	float const ediff_cutoff = { 15.0 }; // energy cutoff for minimization
	float const ediff_cutoff = get_ediff_cutoff();
	float const warning_cutoff = { 1.0 }; // energy cutoff for warnings

//car local
	FArray1D_float phipsi( MAX_RES()() * total_torsion );
	int nangles;
	FArray1D_float dE_dphipsi( MAX_RES()() * total_torsion );
	float fret;
	int iter;
	float ediff;
	float start_score;

//car debug
	float start_func,end_func,end_score;
	float total_score;

	gfrag = true; // initialize

//car figure the starting score and check ediff cutoff
//car score with rotamer_trials to ensure that good sidechains are used
	update_nblist();

	start_score = score_fxn(); // score with rotamer trials if try_rotamer==true

	ediff = start_score - mc_global_track::mc_score::best_score;

	if ( move_type != "minimize" && ( ediff < 0.0 || ediff > ediff_cutoff ) ) {
		minimize_uninit_exclude_list(); // reset for next move
		return;
	}

//car put angles to minimize into phipsi array
	pack_phipsi(phipsi,nangles,fold_begin,fold_end);

//car have to update the nblist again if rotamer trials was called
	update_nblist();

	if ( nangles == 0 ) {
		minimize_uninit_exclude_list(); // reset for next move
		return;
	}

//car set appropriate flags
//car save best structure and replace with current pose so that new rotamers
//car are used
	hijack_best_pose();
	score_enable_rotamer_trials(false);
	score_set_use_subset_energy(true);
	minimize_set_func(2);

//car starting function value for debugging...
	start_func = func(gfrag,phipsi);


//car MINIMIZE
	if ( min_type == "linmin" ) {
		for ( int j = 1; j <= 1; ++j ) {
			dfunc(phipsi,dE_dphipsi,nangles);
			linmin(phipsi,dE_dphipsi,nangles,fret,gfrag);
			if ( !gfrag ) goto L300;
		}
	} else if ( min_type == "dfpmin" ) {
		dfpmin(phipsi,nangles,minimize_tolerance,iter,fret,gfrag);
	} else if ( min_type == "dfpmin_atol" ) { //absolute tolerance
		brent_set_abs_tolerance(minimize_tolerance);
		dfpmin_atol(phipsi,nangles,minimize_tolerance,iter,fret,gfrag);
	} else if ( min_type == "frpmin" ) {
		frprmn(phipsi,nangles,minimize_tolerance,iter,fret,gfrag);
	} else if ( min_type == "frpmin_atol" ) { //absolute tolerance
		brent_set_abs_tolerance(minimize_tolerance);
		frprmn_atol(phipsi,nangles,minimize_tolerance,iter,fret,gfrag);
	} else {
		std::cout << "min_type not defined in function minimize (debump.cc)" << std::endl;
		std::cout << "min_type: " << min_type << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
L300:

	if ( !gfrag ) {
		hijack_best_pose_release();
		score_enable_rotamer_trials(true);
		score_set_use_subset_energy(false);
		set_currently_minimizing(false); // was set to true in pack_phipsi
		minimize_uninit_exclude_list(); // reset for next move
		return;
	}

	end_func = func(gfrag,phipsi);

//car restore default conditions
	hijack_best_pose_release();
	score_set_use_subset_energy(false);
	set_currently_minimizing(false); // was set to true in pack_phipsi
	end_score = score_fxn();

 	if ( runlevel > quiet ) {
		if ( end_score - start_score > warning_cutoff ) {
			score_set_new_pose();
			total_score = score_fxn();
			std::cout << "WARNING!! " << move_type << " score increase:" <<
			 SS( start_score ) << SS( end_score ) << SS( start_func ) <<
				SS( end_func ) << SS( total_score ) << std::endl;
		}
	}

	score_enable_rotamer_trials(true);
	minimize_uninit_exclude_list(); // reset for next move

}

////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_reset_exclude_list
///
/// @brief
///
/// @detailed
///car resets the exclude list to exclude fixed residues and sstypes
///car based on the current structure and the current settings of
///car allow_insert and minimze_excludeH and minimize_excludeE.
///
///car this list should be reset at the beginning of each move and
///car is done so either by calling 'minimize_exclude_frag' or
///car 'minimize_exclude_residues'  or the list is reset at the beginning
///car of the minimization if it hasn't yet been reset for the current
///car move trial
///
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
minimize_reset_exclude_list()
{

	using namespace minimize_ns;
	using namespace misc;
	using namespace protein_maps;

	for ( int i = 1; i <= total_residue; ++i ) {
		minimize_allow_residue(i); // new logic
//		resinmap(i) = allow_insert(i);
//		if ( minimize_excludeH && secstruct(i) == 'H' ) resinmap(i) = false;
//		if ( minimize_excludeE && secstruct(i) == 'E' ) resinmap(i) = false;
	}
	minimize_exclude_list_init = true;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin minimize_allow_residue
///
/// @brief
///  determines which torsion angles and rigid body angles can be modified
///   by the minimizer for a specific residue
///
/// @detailed
///
/// @param i [in] residue number
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////

void
minimize_allow_residue( int const seqpos )
{
	using namespace minimize_ns;
	//using namespace misc;
	using namespace param;
	using namespace param_torsion;
	//using namespace protein_maps;
	using namespace disulfides::BOUNDARY;

	char ss( misc::secstruct( seqpos ) );
	bool allow_backbone_min( protein_maps::allow_insert( seqpos ) );
	bool allow_sidechain_min( protein_maps::allow_repack( seqpos ) );
	bool allow_jump_min( false );

	if ( minimize_check_current_pose() ) {
		pose_ns::Pose const & pose( minimize_get_current_pose() );
		ss = pose.secstruct( seqpos );
		allow_backbone_min = pose.get_allow_bb_move( seqpos );
		allow_sidechain_min = pose.fullatom() && pose.get_allow_chi_move( seqpos );
		allow_jump_min = ( pose.num_jump() >= seqpos &&
											 pose.get_allow_jump_move( seqpos ) );
	}

	if ( ( minimize_excludeH && ss == 'H' ) ||
			 ( minimize_excludeE && ss == 'E' ) ) {
		allow_backbone_min = false;
	}

	// initialize to TRUE
	for ( int tor = 1; tor <= jmp_total_torsion; ++tor ) {
		minimize_allow_torsion(tor,seqpos) = true;
	}

	if ( ! allow_backbone_min ) {
		minimize_allow_torsion( phi_torsion,   seqpos ) = false;
		minimize_allow_torsion( psi_torsion,   seqpos ) = false;
		minimize_allow_torsion( omega_torsion, seqpos  ) = false;
	}

	if ( ! allow_sidechain_min ) {
		for ( int k = 1; k <= MAX_CHI; ++k ) {
			minimize_allow_torsion( total_bb_torsion+k, seqpos ) = false;
		}
	}

	if ( ! minimize_vary_phipsi ) {
		minimize_allow_torsion( phi_torsion, seqpos ) = false;
		minimize_allow_torsion( psi_torsion, seqpos ) = false;
	}

	if ( ! minimize_vary_omega ) {
		minimize_allow_torsion( omega_torsion, seqpos ) = false;
	}

	if ( ! minimize_vary_chi ) {
		for ( int k = 1; k <= MAX_CHI; ++k ) {
			minimize_allow_torsion(total_bb_torsion+k, seqpos ) = false;
		}
	}

  //bills don't minimize disulf when "norepack_disulf" is true
  if ( get_disulf_flag() && get_norepack_disulf() && !get_minimize_disulf_chi() && cys_res_in_disulf(seqpos) ) {
    for ( int k = 1; k <= MAX_CHI; ++k ) {
      minimize_allow_torsion(total_bb_torsion+k, seqpos ) = false;
    }
  }

	if ( ! allow_jump_min || ! minimize_vary_rb_trans ) {
		for ( int k = 1; k <= 3; ++k ) {
			minimize_allow_torsion( first_rb_torsion-1+k, seqpos ) = false;
		}
	}

	if ( ! allow_jump_min || ! minimize_vary_rb_angle ) {
		for ( int k = 4; k <= 6; ++k ) {
			minimize_allow_torsion( first_rb_torsion-1+k, seqpos ) = false;
		}
	}

	// PB -- handling of individual rb deltas separately
	if ( pose_flag() && allow_jump_min ) {
		for ( int k=1; k<= 6; ++k ) {
			if ( !minimize_get_current_pose().get_allow_rb_move( k, seqpos ) )
				minimize_allow_torsion( first_rb_torsion-1+k, seqpos ) = false;
		}
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_disallow_residue
///
/// @brief
///
/// @detailed
///
/// @param  seqpos - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
minimize_disallow_residue( int const seqpos )
{
	using namespace minimize_ns;
	using namespace param_torsion;

	for ( int torsion = 1; torsion <= jmp_total_torsion; ++torsion ) {
		minimize_allow_torsion(torsion,seqpos) = false;
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_exclude_residues
///
/// @brief
///
/// @detailed
///car mark residues in the reslist so that they are not minimized by calls
///car to minimize
///car the exclude list is used by pack_phispi
///car in global_min mode, all other residues are minimized (unless fixed or
///car excluded by ss type
///car in local_min mode, only residues adjacent to residues in the reslist
///car (within local_window residues) are minimized
///
/// @param  reslist - [in/out]? -
/// @param  num - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
minimize_exclude_residues(
	FArray1DB_int & reslist,
	int & num
)
{

	using namespace minimize_ns;
	using namespace misc;
	using namespace protein_maps;

	if ( !minimize_exclude_list_init ) minimize_reset_exclude_list();

	if ( minimize_local_min ) {
		for ( int i = 1; i <= total_residue; ++i ) {
			minimize_disallow_residue(i);
//			resinmap(i) = false;
		}
		for ( int i = 1; i <= num; ++i ) {
			int const start = std::max(1,reslist(i)-minimize_local_window);
			int const end = std::min(total_residue,reslist(i)+minimize_local_window);
			for ( int k = start; k <= end; ++k ) {
				 minimize_allow_residue(k); // new logic
//				if ( allow_insert(k) ) resinmap(k) = true;
//				if ( minimize_excludeH && secstruct(k) == 'H' ) resinmap(i) = false;
//				if ( minimize_excludeE && secstruct(k) == 'E' ) resinmap(i) = false;
			}
		}
	}

//car turn off minimization for residues in the list
	for ( int i = 1; i <= num; ++i ) {
		minimize_disallow_residue(reslist(i));
//		resinmap(reslist(i)) = false;
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_exclude_frag
///
/// @brief
///
/// @detailed
///car mark residues in the inserted fragment so that they are not minimized
///car by calls to minimize in global_min mode.
///car in local_min mode, the fragment and residues within local_window residues
///car are the only residues minimized
///car the exclude list is used by pack_phispi
///
/// @param  begin - [in/out]? -
/// @param  size - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
minimize_exclude_frag(
	int begin,
	int size
)
{

	using namespace minimize_ns;
	using namespace misc;
	using namespace protein_maps;

	if ( !minimize_exclude_list_init ) minimize_reset_exclude_list();

	if ( minimize_local_min ) {
		for ( int i = 1; i <= total_residue; ++i ) {
			minimize_disallow_residue(i); // new logic
//			resinmap(i) = false;
		}

		int const start = std::max(1,begin-minimize_local_window);
		int const end = std::min(total_residue,begin+size+minimize_local_window-1);
		for ( int i = start; i <= end; ++i ) {
			minimize_allow_residue(i);
//			if ( allow_insert(i) ) resinmap(i) = true; // minimize frag +/- local window
//			if ( minimize_excludeH && secstruct(i) == 'H' ) resinmap(i) = false;
//			if ( minimize_excludeE && secstruct(i) == 'E' ) resinmap(i) = false;
		}
	} else {
		for ( int i = begin, ie = begin + size - 1; i <= ie; ++i ) {
			minimize_disallow_residue(i);
//			resinmap(i) = false;
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_res_is_excluded
///
/// @brief checks to see if a residue is currently being minimized
///
/// @detailed
///
/// @param[in]   residue - in - residue to check
///
/// @return   true if residue is NOT being minimized
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  car 10/7/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

bool
minimize_res_is_excluded( int residue )
{
	using namespace minimize_ns;
	using namespace param_torsion;

//	return !resinmap(residue);
	return !(
	 minimize_allow_torsion( phi_torsion,   residue ) ||
	 minimize_allow_torsion( psi_torsion,   residue ) ||
	 minimize_allow_torsion( omega_torsion, residue ) );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin pack_phipsi
///
/// @brief
///
/// @detailed
///
/// @param  phipsi - [in/out]? -
/// @param  nangles - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
pack_phipsi(
	FArray1DB_float & phipsi,
	int & nangles,
	int const fold_begin,
	int const fold_end
)
{
	using namespace aaproperties_pack;
	using namespace minimize_ns;
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace param_torsion;

//car local
	int this_aa,this_aav,tor;
	FArray2D_int rotarray( MAX_CHI, MAX_RES()() );
	FArray2D_float chiarray( MAX_CHI, MAX_RES()() );

	set_currently_minimizing(true);

	if ( minimize_check_current_pose() ) {
		std::cout << "STOP: pose shouldnt be here: pack_phipsi" <<
			std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		//jmp_pack_phipsi(phipsi,nangles); // should check exclude_list_init
		//return;
	}

//car derivative calculation relies on angles being in map in sequential order
//car from N-term to C-term
//car phi,psi,omega  phi,psi,omega   etc.
	if ( !minimize_exclude_list_init ) minimize_reset_exclude_list();

	bool fullatom = false;
	if ( minimize_vary_chi ) fullatom = get_fullatom_flag();

	if ( minimize_vary_chi && fullatom ) {
		////  if packing chi angles, then measure them from the current structure
		//// eventually these should be precalced and stored somewhere
		////
		get_chi_and_rot_from_coords(total_residue,res,res_variant,full_coord,
		 chiarray,rotarray);
	}

	map_length = 0;
	for ( int i = 1; i <= total_residue; ++i ) { // load phipsi array
//pb		if ( !resinmap(i) ) goto L100; // new logic

//phi
		if ( i != 1 && minimize_allow_torsion(phi_torsion,i) ) {
		 // phi of residue 1 not allowed
			++map_length;
			angle_map(map_length) = i;
			angle_type(map_length) = phi_torsion;
			phipsi(map_length) = phi(i);
		}

//ctsa - add chi packing
		if ( fullatom && minimize_vary_chi ) {
			////
			this_aa = res(i);
			this_aav = res_variant(i);
			if ( this_aa == aa_pro ) goto L345;
			for ( int j = 1; j <= total_chi_torsion; ++j ) {
				tor = total_bb_torsion + j;
				if ( minimize_allow_torsion(tor,i) ) {
					if ( nchi(this_aa,this_aav) < j ) goto L345;
					++map_length;
					angle_map(map_length) = i;
					angle_type(map_length) = total_bb_torsion+j;
					phipsi(map_length) = chiarray(j,i);
					//	std::cout << "chi in: " << i << ' ' << chiarray(j,i) <<
					//	" aa " << this_aa << std::endl;
				}
			}

		}

L345:

		if ( i == total_residue ) goto L100; // psi,omega of last res not allowed
//psi
		if ( minimize_allow_torsion(psi_torsion,i) ) {
			++map_length;
			angle_map(map_length) = i;
			angle_type(map_length) = psi_torsion;
			phipsi(map_length) = psi(i);
		}
//omega
		if ( minimize_allow_torsion(omega_torsion,i) ) {
			++map_length;
			phipsi(map_length) = omega(i);
			angle_map(map_length) = i;
			angle_type(map_length) = omega_torsion;
		}

L100:;
	}
	nangles = map_length;

//car save region that needs to be folded: may be larger than
//car region begin minimized

	minimize_fold_begin = fold_begin;
	minimize_fold_end = fold_end;
	for ( int i = 1; i <= map_length; ++i ) {
		if ( angle_type(i) <= total_bb_torsion ) {
			int res = angle_map(i);
			minimize_fold_begin = std::min(minimize_fold_begin,res);
			minimize_fold_end = std::max(minimize_fold_end,res);
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin unpack_phipsi
///
/// @brief
///
/// @detailed
///
/// @param  phipsi - [in/out]? -
/// @param  first - [in/out]? -
/// @param  last - [in/out]? -
/// @param  gfrag - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
unpack_phipsi(
	FArray1DB_float & phipsi,
	int & first,
	int & last,
	bool & gfrag
)
{
	using namespace minimize_ns;
	using namespace misc;
	using namespace param;
	using namespace param_torsion;
	using namespace minimize_ns::unpack_refoldchi_share;

//car local
	int chino,torsion;

	if ( minimize_check_current_pose() ) {
		std::cout << "STOP: pose shouldnt be here: unpack_phipsi" <<
			std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		//jmp_unpack_phipsi(phipsi,gfrag);
		//return;
	}

	if ( minimize_vary_chi ) {
		for ( int i = 1; i <= total_residue; ++i ) {
			for ( int j = 1; j <= MAX_CHI; ++j ) {
				update_chi(j,i) = false;
			}
		}
	}

	gfrag = true;
	for ( int i = 1; i <= map_length; ++i ) {
		torsion = angle_type(i);
		if ( phipsi(i) != phipsi(i) ) { // NaN detected (works for IEEE floating point)
			gfrag = false;
			std::cout << "MINIMIZER ERROR" << std::endl;
			std::cout << "angle_type: " << angle_type(i) << std::endl;
			std::cout << "residue:    " << angle_map(i) << std::endl;
			std::cout << "value:      " << phipsi(i) << std::endl;
			return;
		}
		if ( std::abs(phipsi(i)) > 360.0 ) angle_in_range(phipsi(i));

		if ( torsion == phi_torsion ) {
			phi(angle_map(i)) = phipsi(i);
		} else if ( torsion == psi_torsion ) {
			psi(angle_map(i)) = phipsi(i);
		} else if ( torsion == omega_torsion ) {
			omega(angle_map(i)) = phipsi(i);
		} else if ( torsion > total_bb_torsion && torsion <= total_torsion ) {
			chino = torsion-total_bb_torsion;
			update_chi(chino,angle_map(i)) = true;
			new_chi(chino,angle_map(i)) = periodic_range((phipsi(i)),360.);
		} else {
			std::cout << "unknown angle type in unpack_phipsi: " << i << ' ' <<
			 torsion << ' ' << angle_map(i) << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

	first = minimize_fold_begin;
	last = minimize_fold_end;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_update_fullcoord
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
minimize_update_fullcoord()
{
	using namespace misc;

	if ( get_fullatom_flag() ) return;
	for ( int i = 1; i <= total_residue; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			full_coord(j,1,i) = Eposition(j,1,i);
			full_coord(j,2,i) = Eposition(j,2,i);
			full_coord(j,3,i) = Eposition(j,4,i);
			full_coord(j,4,i) = Eposition(j,5,i);
			full_coord(j,5,i) = Eposition(j,3,i);
			full_coord(j,6,i) = centroid(j,i);
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin func_vdw
///
/// @brief
///car function for idealizing/minimizing: scorefxn,tether,angles
///
/// @detailed
///
/// @param  phipsi - [in/out]? -
/// @param  gfrag - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
func_vdw(
	FArray1DB_float & phipsi,
	bool & gfrag
)
{

	using namespace misc;
	using namespace minimize_ns;

//car local
	int first_bb,last_bb; // between which phi/psi change
	int first_sc,last_sc; // between which chi change

//------------------------------------------------------------------------------
	if ( minimize_check_current_pose() ) {
		return jmp_func_vdw( minimize_get_current_pose(), phipsi, gfrag );
	}

	unpack_phipsi(phipsi,first_bb,last_bb,gfrag);
	if ( !gfrag ) return 0.0f;

	if ( minimize_vary_phipsi || minimize_vary_omega ) {
		refold(first_bb,last_bb);
	}

	minimize_update_fullcoord();

	first_sc = total_residue+1;
	last_sc = 0;
	if ( minimize_vary_chi ) {
		//// update the minimized chi angles on full_coord after
		////   backbone refold is complete
		refold_new_chi(first_sc,last_sc);
	}

	return scorefxn();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin dfunc_vdw
///
/// @brief
///car recursive derivative calculation
///
/// @detailed
///car note that this calculates the deriv for all torsion angles even
///car those that are fixed. Because of the way that the derivative is
///car calculated, I don't believe this is a significant slow down (ie
///car have to run over all the atom pairs twice, regardless of the number
///car of torsion angles)
///
///car multiple neighborlists:
///car cendist       centroid distances in current structure, cutoff for vdw
///car dis2_tether   centroid distances in tether structure, cutoff for tether
///
///db computes the derivative of E  with respect to each
///db of the torsion angles.  Using the chain rule, we have
///db
///db dE/d phi = dE/dr *  dr/dphi
///db
///db dr/dphi = Eab x (V-Vb) . (V' - V)/|V-V'|
///db
///db (the first cross product is the displacement of V upon a rotation dphi
///db around the unit vector Eab, Vb is the coordinates of the second atom in
///db the bond)
///db
///car dE/dR = 2r  (for vdw at least)
///db since | V-V'| = r,
///db
///db dE/ dphi = 2 Eab x (V-Vb) . (V' - V)
///db
///db  note that Eab and Vb are different for each torsion angle, but V'
///db and V are the same.  rearranging:
///db
///db = -  2 Eab X Vb . (V' - V) - 2 Eab . (V' x V).
///db
///db now we need the averages over all Vi of the difference and the
///db crossproduct of V and V'.
///
///car below, Eab x Vb is 'vec'
///car        Eab      is 'unit'
///car        (V'-V)   is 'f2'
///car        'F2tot' = f2*dE_dR (cumulative)
///car        (V' X V) is 'f1' ('F1_xxxE' is cumulative for potential xxx)
///car        eval_dE_dR actually returns dE_dR/r
///
///car if two atoms are fixed relatively in cartesian space, then dr/dphi = 0
///car and there is no contribution to the derivative
///
/// @param  phipsi - [in/out]? -
/// @param  dE_dphipsi - [in/out]? -
/// @param  nangles - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///car     Braun and Go, JMB 1985 186:611-26
///car     Guntert,Braun & Wuthrich JMB 1991 217:517-53
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
dfunc_vdw(
	FArray1DB_float & phipsi,
	FArray1DB_float & dE_dphipsi,
	int nangles
)
{
	using namespace minimize_ns::deriv_precalc;
	using namespace minimize_ns;
	using namespace misc;
	using namespace param;
	using namespace param_pack;
	using namespace param_torsion;
	using namespace runlevel_ns;
	using namespace scorefxns;
	using namespace tether;
	using numeric::conversions::radians;

//car local
	int j;
	int imap; // current position in the angle map
	int i; // residue of current torsion angle
	int torsion; // angle type of current torsion angle
	int end_res,end_atom; // N-term atoms in torsion bond
	// body region
//      FArray1D_float ixyz(3),jxyz(3);  // coordinates of ires,iat and jres,jat
	float deriv,loop_deriv;
	float rama_deriv,fa_prob_deriv;
	float fa_pair_factor;
//      float dis2;
	FArray1D_float unit( 3 );
	FArray1D_float vect( 3 );
	FArray1D_float F1_allE( 3 );
	FArray1D_float F2_allE( 3 );
	FArray3D_float F1_atompairE( 3, total_torsion, MAX_RES()() );
	FArray3D_float F2_atompairE( 3, total_torsion, MAX_RES()() );
	FArray3D_float F1_respairE( 3, total_torsion, MAX_RES()() );
	FArray3D_float F2_respairE( 3, total_torsion, MAX_RES()() );

	FArray3D_float F1_hbondE( 3, total_torsion, MAX_RES()() );
	FArray3D_float F2_hbondE( 3, total_torsion, MAX_RES()() );

	bool fullatom;
	int first_res,last_res; // between which phi/psi/chi change //chu
	int first_res_bb, last_res_bb; // between which phi/psi change //chu
	int first_res_sc, last_res_sc; // between which chi change //chu

	FArray2D_float dunbrack_deriv( total_torsion, MAX_RES()() );
	FArray2D_float torsion_tether_deriv( total_bb_torsion, MAX_RES()() );
	float tmp1,tmp2;
	float sr_hbenergy,lr_hbenergy;

//ctsa local copy of rotarray and chiarray to evaluate dunbrack deriv
	FArray2D_int rotarray( MAX_CHI, MAX_RES()() );
	FArray2D_float chiarray( MAX_CHI, MAX_RES()() );

	float fa_dun_factor;

//car variables for numerical derivative check
	FArray1D_float phipsi_ori( 800 );
	float f11,f22;
	bool gfrag( true );
	float ratio;
	float increment;


	if ( minimize_check_current_pose() ) {
//pb PHIL: you need to keep the derivatives synced in jumping_minimize!!
//pb it would be a good idea to come up with a benchmark for this
		jmp_dfunc_vdw(minimize_get_current_pose(),phipsi,dE_dphipsi,nangles);
		return;
	}

//car precalculate constants (weights and constants in derivatives)
//car constants from form of derivatives
//car  except 0.8 for vdw which is weight applied inside vdw_compute
	tether_factor = 0.0;
	if ( tether_norm != 0.0 ) tether_factor = -4.0*tether_weight/tether_norm;
	vdw_factor = -4.*0.8*vdw_weight;

	fa_pair_factor = pack_wts.Wpair()*fa_pair_weight;
	fa_dun_factor = pack_wts.Wdun()*fa_dun_weight;

//car retrieve flag to pass to functions
	fullatom = get_fullatom_flag();
//car update coordinates

	unpack_phipsi(phipsi,first_res_bb,last_res_bb,gfrag);
	if ( !gfrag ) return; // set deriv to 0?

	if ( minimize_vary_phipsi || minimize_vary_omega ) {
		refold(first_res_bb,last_res_bb);
	}

//car if not fullatom mode, put position array+centroids in full_coord array
	minimize_update_fullcoord();

	first_res_sc = total_residue + 1;
	last_res_sc = 0; // chu initialize
	if ( minimize_vary_chi ) {
		//// update the minimized chi angles on full_coord after
		////   backbone refold is complete
		refold_new_chi(first_res_sc,last_res_sc);
	}

	first_res = std::min( first_res_bb, first_res_sc );
	last_res = std::max( last_res_bb, last_res_sc );
	//chu there must be either phi/psi or chi change
	assert( first_res <= last_res );
//car update nblist
	drv_update_nblist();

// Compute the H-bond energy and derivatives
	evaluate_hbenergy(hbonds::hbderiv_ABE_GO,sr_hbenergy,lr_hbenergy); // JSS to fill arrays; return values unused
	hbond_compute_deriv(first_res,last_res, F1_hbondE,F2_hbondE);

	//// ctsa update chi and rotamer maps for dunbrack derivative
	////   and pre-calculate dunbrack deriv in first-last res range
	////
	if ( fullatom && fa_dun_factor != 0. ) {
		get_chi_and_rot_from_coords(total_residue,res,res_variant,full_coord,
		 chiarray,rotarray);
		eval_total_dunbrack_deriv(first_res,last_res,chiarray,rotarray,
		 dunbrack_deriv);
	}

	//// evaluate bb torsion tether deriv
	////
	if ( phipsi_weight != 0.0 )
	 calculate_phipsi_torsion_tether_score(tmp1,torsion_tether_deriv);
	if ( omega_weight != 0.0 )
	 calculate_omega_torsion_tether_score(tmp2,torsion_tether_deriv);


	//// ctsa    calc pair energy derivative
	////
	if ( fullatom && fa_pair_factor != 0. ) {
		get_total_pair_deriv(first_res,last_res,F1_respairE,F2_respairE);
	}

	////  get derivative of all atom pair potentials
	////

	get_atompairE_deriv(first_res,last_res,F1_atompairE,F2_atompairE,fullatom);

//car for each torsion angle in the angle map:
//     1) combine all distance derivs and complete conversion to angular deriv
//     2) calc other anglular derivs and sum in
//
	for ( imap = map_length; imap >= 1; --imap ) { // iterate from C to N terminus
		i = angle_map(imap);
		torsion = angle_type(imap);
		deriv = 0.0;

//car these should never get into the map
		if ( i == 1             && torsion == phi_torsion ) goto L110;
		if ( i == total_residue && torsion == psi_torsion ) goto L110;
		if ( i == total_residue && torsion == omega_torsion ) goto L110;

		//// sum F1,F2 from different energy components
		////     atomic pair potentials and residue pair score
		////
		for ( j = 1; j <= 3; ++j ) {
			F1_allE(j) = F1_atompairE(j,torsion,i);
			F2_allE(j) = F2_atompairE(j,torsion,i);
		}

//car  pre-computed h-bond contribution
		for ( j = 1; j <= 3; ++j ) {
			F1_allE(j) += F1_hbondE(j,torsion,i);

			F2_allE(j) += F2_hbondE(j,torsion,i);
		}

//ctsa  add F1,F2 from the pair energy derivative to the total F1,F2
		if ( fullatom && fa_pair_factor > 0. ) {
			for ( j = 1; j <= 3; ++j ) {
				F1_allE(j) += fa_pair_factor*F1_respairE(j,torsion,i);
				F2_allE(j) += fa_pair_factor*F2_respairE(j,torsion,i);
			}
		}

		//// convert combined F1,F2 to an angular derivative
		////  using Abe Go trick
		////
		make_unit_vector(i,torsion,unit,end_atom,end_res);
		cros(unit,full_coord(1,end_atom,end_res),vect);

		deriv -= radians( dotprod(unit,F1_allE) + dotprod(vect,F2_allE) );

//car derivative for loop closure:
		eval_loop_deriv(i,torsion,unit,loop_deriv);
		deriv += loop_deriv;

		//// ctsa derivative of dunbrack score:
		////
		if ( fullatom && fa_dun_factor != 0. ) {
			deriv += (dunbrack_deriv(torsion,i)*fa_dun_factor);
		}

		//// ctsa - derivative of rama score:
		////   -- note that if necessary, this could be accelerated by storing the
		////   -- derivatives from the last call to func_vdw
		////
		if ( ramachandran_weight != 0. ) {
			rama_deriv = get_rama_score_residue_deriv(res(i),phi(i),psi(i),
			 secstruct(i),torsion);
			deriv += (rama_deriv*ramachandran_weight);
		}

		if ( fa_prob1b_weight != 0.0 ) {
			fa_prob_deriv = get_Paa_pp_deriv(res(i),phi(i),psi(i),torsion);
			deriv += fa_prob_deriv * fa_prob1b_weight;
		}

//car derivative for tethered bb torsion angles
		if ( phipsi_weight != 0.0 ) {
			if ( torsion == phi_torsion || torsion == psi_torsion ) {
				deriv += phipsi_weight*torsion_tether_deriv(torsion,i);
			}
		}
		if ( omega_weight != 0.0 ) {
			if ( torsion == omega_torsion ) {
				deriv += omega_weight*torsion_tether_deriv(torsion,i);
			}
		}

L110:; // skip torsion
		dE_dphipsi(imap) = deriv;
	}                     // loop over map

	if ( runlevel <= gush ) return;

//car numerical derivative check
	increment = 0.01;
	gfrag = true;
	for ( i = 1; i <= nangles; ++i ) {
		phipsi_ori(i) = phipsi(i);
	}

	for ( i = 1; i <= nangles; ++i ) {
		for ( j = 1; j <= 1; ++j ) {
			phipsi_ori(i) = phipsi(i)+float(j)*increment;
			angle_in_range(phipsi_ori(i));
			f11 = func(gfrag,phipsi_ori);
			phipsi_ori(i) = phipsi(i)-float(j)*increment;
			angle_in_range(phipsi_ori(i));
			f22 = func(gfrag,phipsi_ori);
//       write_hbonds(0)
			deriv = (f11-f22)/(float(j)*2*increment);
			phipsi_ori(i) = phipsi(i);
//       if ( std::abs(dE_dphipsi(i)) > 0.001 && std::abs(deriv) > 0.001 ) {
			if ( std::abs(dE_dphipsi(i)) > 0.001 ) {
				ratio = deriv/dE_dphipsi(i);
			} else {
				ratio = 0.0;
			}
			if ( std::abs(dE_dphipsi(i)) > 0.001 || std::abs(deriv) > 0.01 )
			 std::cout << "ratio" <<
			 I( 5, i ) << I( 5, j ) << I( 5, angle_map(i) ) << I( 5, angle_type(i) ) <<
			 F( 10, 4, deriv ) << F( 10, 4, dE_dphipsi(i) ) << F( 10, 4, ratio ) <<
			 F( 10, 4, f11 ) << F( 10, 4, f22 ) << F( 10, 4, phipsi(i) ) << std::endl;

		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_atompairE_deriv
///
/// @brief
///
/// @detailed
///
/// @param  first_res - [in/out]? -
/// @param  last_res - [in/out]? -
/// @param  F1_atompairE - [in/out]? -
/// @param  F2_atompairE - [in/out]? -
/// @param  fullatom - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_atompairE_deriv(
	int first_res,
	int last_res,
	FArray3DB_float & F1_atompairE,
	FArray3DB_float & F2_atompairE,
	bool fullatom
)
{
	using namespace aaproperties_pack;
	using namespace cenlist_ns;
	using namespace minimize_ns::deriv_precalc;
	using namespace minimize_ns;
	using namespace misc;
	using namespace nblist;
	using namespace param;
	using namespace param_pack;
	using namespace param_torsion;
	using namespace scorefxns;
	using namespace tether;

	assert( ( equal_dimensions( F1_atompairE, F2_atompairE ) ) ); //Objexx: Required below

// locals
	int aa,aav;
	int torsion;
	int ires,iat,jres,jat,jallat,jjend;
	int first_atom,last_atom; // first and last atom in list for rigid
	float dE_dR;
	FArray1D_float f1( 3 );
	FArray1D_float f2( 3 );
	bool constraints_exist;
	float pc_score; // local only
	int seg_begin, seg_end;
//$$$	FArray1D_float F1_disulfE( 3 ), F2_disulfE( 3 );

// tmp torsion order hack
	FArray1D_int torsion_order_map( total_torsion );

//------------------------------------------------------------------------------

	bool const use_nblist = get_use_nblist();

	//// get torsion order
	////
	torsion_order_map(1) = 1;
	for ( int k = 1; k <= total_chi_torsion; ++k ) {
		torsion_order_map(1+k) = 3+k;
	}
	torsion_order_map(2+total_chi_torsion) = 2;
	torsion_order_map(3+total_chi_torsion) = 3;

//car precalculate constants (weights and constants in derivatives)
//car constants from form of derivatives
//car  except 0.8 for vdw which is weight applied inside vdw_compute
	tether_factor = 0.0;
	if ( tether_norm != 0.0 ) tether_factor = -4.0*tether_weight/tether_norm;
	vdw_factor = -4.*0.8*vdw_weight;
	fa_rep_factor = pack_wts.Wrep()*fa_rep_weight;
	fa_atr_factor = pack_wts.Watr()*fa_atr_weight;
	fa_solv_factor = pack_wts.Wsol()*fa_solv_weight;

//car determine if constraints need to be minimized; can't skip some pairs
//car if constraints included
	classical_constraints::BOUNDARY::eval_pairConstraint(0,pc_score); // update distances,distances**2
	classical_constraints::BOUNDARY::precalc_cst_deriv(); // precalc factor for deriv
	constraints_exist = classical_constraints::BOUNDARY::get_constraints_exist();

//car obtain cendist  to determine which atoms can be skipped
	update_cendist(total_residue,centroid);

	//// car  initialize  joint atom-pair derivatives
	////
	F1_atompairE = 0.0;
	F2_atompairE = 0.0;

	first_atom = total_atoms+1; // zeroth rigid body begins after chain end
	int seg_num = identify_segment(last_res);
	retrieve_segment_ends(seg_num,seg_begin,seg_end);
	if ( seg_end < total_residue ) first_atom =
	 res_atm_num2global_atm_num_map(1,seg_end+1);

	//// get joint atom pair potential derivatives over all torsion angles
	////   between the first and last residues in the angle_map
	////
	for ( int seqpos = last_res; seqpos >= first_res; --seqpos ) {
		aa = res(seqpos);
		aav = res_variant(seqpos);
		int const total_bb_torsion_nchi = total_bb_torsion + nchi(aa,aav);
		for ( int tor = total_torsion; tor >= 1; --tor ) {
			torsion = torsion_order_map(tor);

//car these should never be calculated
			if ( seqpos == 1 && torsion == phi_torsion ) goto L110;
			if ( seqpos == seg_end && torsion == omega_torsion ) goto L110;

// special cases for chi angles
			if ( torsion > total_bb_torsion_nchi ) goto L110;
			if ( torsion > total_bb_torsion && ! minimize_vary_chi ) goto L110;

//car derivative for all atom pair terms
			get_rigid_body_region(seqpos,torsion,first_atom,last_atom);

			for ( int ii = first_atom, lFa = F1_atompairE.index(1,torsion,seqpos);
			 ii <= last_atom; ++ii ) {
			 // atoms between current and last torsion
				ires = global_atm_num2res_num_map(ii);
				iat = global_atm_num2res_atm_num_map(ii);
				int iseg = identify_segment(ires);
				int seg_begin,seg_end;
				retrieve_segment_ends(iseg,seg_begin,seg_end);
				int const res_ires = res(ires);
				int const res_variant_ires = res_variant(ires);
				if ( use_nblist ) {
					jjend = drv_nb_len(ii);
				} else {
					jjend = total_atoms;
				}
				for ( int jj = 1; jj <= jjend; ++jj ) {
					if ( use_nblist ) {
						jallat = nb(jj,ii);
					} else {
						jallat = jj;
					}

					if ( jallat >= first_atom && jallat <= last_atom ) goto L100;
					jat = global_atm_num2res_atm_num_map(jallat);
					jres = global_atm_num2res_num_map(jallat);

//car distance check (only has benefit if not using the nblist)
//car atoms that are far apart are going to have dE_dR = 0 unless they interact
//car by some energy term without a distance cutoff
					if ( ( ! use_nblist ) && ( ! constraints_exist )  &&
							( tether_weight == 0.0 || dis2_tether(ires,cutoff_atm,jres,cutoff_atm) <= cen_dist_cutoff2 )
            && ( cendist(ires,jres) > cen_dist_cutoff2 ) ) goto L100; //skip pair

					eval_dE_dR(ires,iat,jres,jat,res_ires,res(jres),res_variant_ires,
					 res_variant(jres),fullatom,dE_dR,f1,f2);
//$$$				if ( std::abs(dE_dR) > 0.00001 ) std::cout << "ires,iat,jres,jat " <<
//$$$				 ires << ' ' << iat << ' ' << jres << ' ' << jat <<
//$$$				 SS( dE_dR ) << ' ' << torsion << std::endl;

					//// sum F1,F2 over all atom pairs
					////
					if ( dE_dR == 0.0 ) goto L100;
					for ( int k = 1, l = lFa; k <= 3; ++k, ++l ) {
						F1_atompairE[ l ] += dE_dR * f1(k); // F1_atompairE(k,torsion,seqpos)
						F2_atompairE[ l ] += dE_dR * f2(k); // F2_atompairE(k,torsion,seqpos)
					}

//$$$ These calls for angular derivs of fullatom disulfides
//$$$					if ( get_disulf_flag() ) {
//$$$cbs  here call eval_disulf_deriv...this can call dist deriv and ang deriv and compile a complete F1_disulf,F2_disufl
//$$$cbs  the F's should be zero unless atom pairs are appropriate
//$$$cbs dist deriv for Cb-Cb,Cen-Cen,S-S
//$$$cbs ang deriv for S-S...what about ang deriv for Cb? I think I should put it there too!
//$$$
//$$$ 					call eval_disulf_ang_deriv(ires,iat,jres,jat,
//$$$ 					 res(ires),res(jres),
//$$$ 					 res_variant(ires),res_variant(jres),
//$$$ 					 fullatom,F1_disulfE,F2_disulfE);
//$$$ 					for ( k = 1; k <= 3; ++k ) {
//$$$ 						F1_atompairE(k,torsion,seqpos) += F1_disulfE(k);
//$$$ 						F2_atompairE(k,torsion,seqpos) += F2_disulfE(k);
//$$$ 					}
//$$$ 				}

L100:; // skip pair
				}            // jj list
			}            // ii list

			//// link F1,F2 to all immediately preceding torsions
			////
			link_torsion_vectors(seqpos,torsion,F1_atompairE,F2_atompairE);

L110:; // skip torsion
		}    // torsion list
	}    // residue list

}

////////////////////////////////////////////////////////////////////////////////
/// @begin link_torsion_vectors
///
/// @brief
///
/// @detailed
///
/// ctsa - for any f1,f2, find the one (or two) c/chi-terminal F1,F2
///   vectors and sum them in to solve for this torsion's F1,F2
///
///   This function assumes that, when called, Fx(x,torsion,seqpos)
///   actually contains the value of fx for the indicated torsion
///   angle. This value is replaced with the value for Fx via the sum:
///   Fx(current_torsion) = fx(current_torsion +
///      sum_over_upstream_torsion_angles( Fx(upstream_torsion_angle) )
///
///   Note that in the process the value of fx stored in
///   Fx(x,torsion,seqpos). is destroyed
///
///
/// @param  seqpos - [in/out]? -
/// @param  torsion - [in/out]? -
/// @param  F1 - [in/out]? -
/// @param  F2 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
link_torsion_vectors(
	int const seqpos,
	int const torsion,
	FArray3DB_float & F1,
	FArray3DB_float & F2
)
{
	using namespace misc;
	using namespace param_torsion;


	//  local
	int next_seqpos = 0, next_torsion = 0;
//------------------------------------------------------------------------------
	if ( torsion == phi_torsion ) {
		//  If current torsion is backbone phi, then add the
		//   sidechain torsion Fx....
		//
		if ( total_chi_torsion >= 1 ) {
			for ( int j = 1; j <= 3; ++j ) {
				F1(j,torsion,seqpos) += F1(j,chi1_torsion,seqpos);
				F2(j,torsion,seqpos) += F2(j,chi1_torsion,seqpos);
			}
		}
		//  ...and add the backbone phi torsion Fx
		//
		next_seqpos = seqpos;
		next_torsion = psi_torsion;
	} else if ( torsion == psi_torsion ) {
		//  If this is psi, then add the omega torsion Fx
		//
		next_seqpos = seqpos;
		next_torsion = omega_torsion;
	} else if ( torsion == omega_torsion ) {
		//  If this is omega, then add the phi torsion Fx from the
		//   next residue
		//
		next_seqpos = seqpos + 1;
		next_torsion = phi_torsion;
		if ( next_seqpos > total_residue ) return;

//car stop propagation across chain breaks
		int seg1 = identify_segment(seqpos);
		int seg2 = identify_segment(seqpos+1);
		if ( seg1 != seg2 ) {
			for ( int j = 1; j <= 3; ++j ) {
				F1(j,torsion,seqpos) = 0.0;
				F2(j,torsion,seqpos) = 0.0;
			}
			return;
		}
	} else if ( torsion > total_bb_torsion && torsion <= total_torsion ) {
		//  If this is a chi(n) torsion angle, then add the
		//   torsion Fx from chi(n+1)
		//
		next_seqpos = seqpos;
		next_torsion = torsion + 1;
		if ( next_torsion > total_torsion ) return;
	} else {
		std::cout << "ABORT: bad torsion in link_torsion_vectors" << std::endl;
		std::cout << "torsion: " << torsion << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	for ( int j = 1; j <= 3; ++j ) {
		F1(j,torsion,seqpos) += F1(j,next_torsion,next_seqpos);
		F2(j,torsion,seqpos) += F2(j,next_torsion,next_seqpos);
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin make_unit_vector
///
/// @brief
///car returns the unit vector along phi,psi,omega of residue i
///car also returns the atom and residue index  C-terminal atom in the bond
///
/// @detailed
///
/// @param  i - [in/out]? -
/// @param  torsion - [in/out]? -
/// @param  vec - [in/out]? -
/// @param  end_atom - [in/out]? -
/// @param  end_res - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
make_unit_vector(
	int const i,
	int const torsion,
	FArray1DB_float & vec,
	int & end_atom,
	int & end_res
)
{
	using namespace aaproperties_pack;
	using namespace misc;
	using namespace param_torsion;


	int start_atom = 0;

	if ( torsion == phi_torsion ) {
		start_atom = 1;
		end_atom = 2;
		end_res = i;
	} else if ( torsion == psi_torsion ) {
		start_atom = 2;
		end_atom = 3;
		end_res = i;
	} else if ( torsion == omega_torsion ) {
		start_atom = 3;
		end_atom = 1;
		end_res = i+1;
	} else if ( torsion > total_bb_torsion && torsion <= total_torsion ) {
		int chino = torsion - total_bb_torsion;
		start_atom = chi_atoms(2,chino,res(i),res_variant(i));
		end_atom = chi_atoms(3,chino,res(i),res_variant(i));
		end_res = i;
	} else {
		std::cout << "Invalid value of torsion: " << torsion << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	subvec(full_coord(1,end_atom,end_res),full_coord(1,start_atom,i),vec);

	vector_normalize(vec);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_rigid_body_region
///
/// @brief
///
/// @detailed
///    identify the rigid body on the c-term or higher chi side of the
///    given torsion angle and seqpos:
///
///car  set the first atom of the rigid body by looking it up from
///    torsion and seqpos
///
///car  set the last atom of the rigid body from the incoming value of
///    rigid_body_start_atm.
///
///    (ie rigid_body_start_atm is both an input and output argument;
///    before calling this function for the first time,
///    rigid_body_start_atom should be set to total_atoms+1)
///
///
/// @param  seqpos - [in/out]? - current res
/// @param  torsion_type - [in/out]? - current torsion
/// @param  rigid_body_start_atm - [in/out]? -
/// @param  rigid_body_end_atm - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_rigid_body_region(
	int const seqpos, // current res
	int const torsion_type, // current torsion
	int & rigid_body_start_atm,
	int & rigid_body_end_atm
)
{

//car copy allatom list and add protons on end? only do nangles times

	using namespace nblist;
	using namespace param_torsion;


	rigid_body_end_atm = std::min(rigid_body_start_atm-1,total_atoms);

	if ( torsion_type > total_torsion || torsion_type <= 0 ) {
		std::cout << "ABORT:: unidentified torsion type in " <<
		 "get_rigid_body_region" << std::endl;
		std::cout << "torsion: " << torsion_type << " residue: " << seqpos <<
		 std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	rigid_body_start_atm = torsion2rigid_body_start_map(torsion_type,seqpos);

	if ( rigid_body_start_atm == undefined_global_atm ) {
		std::cout << "ABORT:: unable to identify rigid body region: " <<
		 seqpos << ' ' << torsion_type << ' ' << total_atoms << ' ' <<
		 rigid_body_start_atm << ' ' << rigid_body_end_atm << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin eval_dE_dR
///
/// @brief
///
/// @detailed
///
/// @param  i - [in/out]? -
/// @param  ii - [in/out]? -
/// @param  j - [in/out]? -
/// @param  jj - [in/out]? -
/// @param  resi - [in/out]? -
/// @param  resj - [in/out]? -
/// @param  resvi - [in/out]? -
/// @param  resvj - [in/out]? -
/// @param  fullatom - [in/out]? -
/// @param  dE_dR - [in/out]? -
/// @param  f1 - [in/out]? - dimension( 3 )
/// @param  f2 - [in/out]? - dimension( 3 )
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
eval_dE_dR(
	int i,
	int ii,
	int j,
	int jj,
	int resi,
	int resj,
	int resvi,
	int resvj,
	bool fullatom,
	float & dE_dR,
	FArray1DB_float & f1, // dimension( 3 )
	FArray1DB_float & f2  // dimension( 3 )
)
{
	using namespace cenlist_ns;
	using namespace minimize_ns::deriv_precalc;
	using namespace pdbstatistics_pack;
	using namespace scorefxns;
	using namespace tether;
  using namespace aaproperties_pack;
  using namespace param_pack;

	static FArray1D_float ixyz( 3 );
	static FArray1D_float jxyz( 3 );

	int disbin1,atypei,atypej;
	// int disbin2;
	float radius2;
	float d2_bin,frac;
	float solv_deriv1,solv_deriv2;
	float rep_deriv1,rep_deriv2,atr_deriv1,atr_deriv2;
	float cp_weight;

//car fullatom lj or centroid bump_score
	dE_dR = 0.0;
	float tether_deriv = 0.0;
	float cst_deriv = 0.0;
	float vdw_deriv = 0.0;
	float solv_deriv = 0.0;
	float disulf_deriv = 0.0;
  float elec_deriv = 0.0;
  float d_elec1, d_elec2;
	float dis = -1.0; // flag dis so we know it's not defined yet
	float dis2 = -1.0; // flag dis2 so we know it's not defined yet
//standard deriv

	if ( ii < 1 || jj < 1 ) goto L100; // protons

//car this check required, might be on nblist for cst or tether
//car but this will require dis2?
	if ( ( !minimize_check_current_pose() || !fullatom ) &&
			 cendist(i,j) > cen_dist_cutoff2 ) {
		goto L200;
	}

//car this check only effective if nblist not in use
	if ( !count_pair(i,ii,resi,resvi,j,jj,resj,resvj,fullatom,cp_weight) )
	 goto L200;

	eval_dis2_f2(i,ii,j,jj,ixyz,jxyz,dis2,f2);

	if ( dis2 >= safe_max_dis2 ) goto L200; // cutoff for fullatom terms

	dis = std::sqrt( dis2 );
	if ( fullatom ) {
    int l1, l2;
		//// use the same interpolation scheme as in pairenergy
		d2_bin = dis2 * fa_bins_per_A2;
		disbin1 = static_cast< int >( d2_bin ) + 1;
		// disbin2 = disbin1+1;
		frac = d2_bin - ( disbin1 - 1 );

		atypej = all_atom_type(resj,resvj,jj,fullatom);
		atypei = all_atom_type(resi,resvi,ii,fullatom);

    int const typei = aaproperties_pack::fullatom_type(ii,resi,resvi);
    int const typej = aaproperties_pack::fullatom_type(jj,resj,resvj);
    bool aro_charged = aro_charged_pair( resi, typei, ii, resj,  typej, jj );
    if ( get_simple_elec() && aro_charged ) {
      float chrgi = atomic_charge(ii, resi, resvi);
      float chrgj = atomic_charge(jj, resj, resvj);

      elec_deriv = cp_weight * chrgi * chrgj * pack_wts.Wele() * scorefxns::fa_elec_weight;

      l1 = pCurrentEtable->d_elec_rep.index( disbin1, 1, 1 ); l2 = l1 + 1;
      if (elec_deriv > 0) { // repulsive
        d_elec1 = pCurrentEtable->d_elec_rep[ l1 ];
        d_elec2 = pCurrentEtable->d_elec_rep[ l2 ];
      } else { // attractive
        d_elec1 = pCurrentEtable->d_elec_atr[ l1 ];
        d_elec2 = pCurrentEtable->d_elec_atr[ l2 ];
      }
      elec_deriv *= d_elec1 + frac * ( d_elec2 - d_elec1);
    }

		//Objexx: Linear indexing assumes arrays have same dimensions
		assert( ( equal_dimensions( pCurrentEtable->dljrep, pCurrentEtable->dljatr ) ) );
		assert( ( equal_dimensions( pCurrentEtable->dljrep, pCurrentEtable->dsolv ) ) );
		l1 = pCurrentEtable->dljrep.index(disbin1,atypej,atypei); l2 = l1 + 1;
		rep_deriv1 = pCurrentEtable->dljrep[ l1 ]; // dljrep(disbin1,atypej,atypei)
		rep_deriv2 = pCurrentEtable->dljrep[ l2 ]; // dljrep(disbin2,atypej,atypei)
		atr_deriv1 = pCurrentEtable->dljatr[ l1 ]; // dljatr(disbin1,atypej,atypei)
		atr_deriv2 = pCurrentEtable->dljatr[ l2 ]; // dljatr(disbin2,atypej,atypei)
		if ( atypei == water::type_h2o || atypej == water::type_h2o ) {
			// WATER
			vdw_deriv = cp_weight * param_pack::pack_wts.Wh2o() * param_pack::pack_wts.Wh2o_lj() *
				scorefxns::fa_h2o_weight *
				( rep_deriv1 + frac * ( rep_deriv2 - rep_deriv1 ) );
			solv_deriv = 0.0;
		} else {
			vdw_deriv = cp_weight * (
				( rep_deriv1 + frac * ( rep_deriv2 - rep_deriv1 ) ) * fa_rep_factor +
				( atr_deriv1 + frac * ( atr_deriv2 - atr_deriv1 ) ) * fa_atr_factor );

			solv_deriv1 = pCurrentEtable->dsolv[ l1 ]; // dsolv(disbin1,atypej,atypei)
			solv_deriv2 = pCurrentEtable->dsolv[ l2 ]; // dsolv(disbin2,atypej,atypei)
			solv_deriv = ( solv_deriv1 + frac * ( solv_deriv2 - solv_deriv1 ) ) *
				fa_solv_factor * cp_weight;
		}
	} else {
		if ( std::abs(i-j) <= 1 ) {
			eval_local_vdw_deriv(i,ii,j,jj,resi,resj,resvi,resvj,dis,dis2,vdw_deriv);
		} else {
			radius2 = vdw::atom_vdw(all_atom_type(resi,resvi,ii,fullatom),
			 all_atom_type(resj,resvj,jj,fullatom));
			if ( dis2 < radius2 ) vdw_deriv =
			 vdw_factor * dis * ( radius2 - dis2 ) / radius2;
//$$$    if ( dis2 < radius2 ) std::cout << "dedr" <<
//$$$     I( 4, i ) << I( 4, ii ) << I( 4, j ) << I( 4, jj ) <<
//$$$     F( 6, 3, dis2 ) << F( 6, 3, radius2 ) << std::endl; // debug

    }
  }

L200:

//car tether to starting structure
  if ( tether_weight != 0.0 ) {
    if ( std::abs(i-j) > tether_window_size ) goto L100;
    if ( ii > max_tether_atoms || jj > max_tether_atoms ) goto L100;
    if ( dis2_tether(i,cutoff_atm,j,cutoff_atm) > cen_dist_cutoff2 ) goto L100;

    if ( dis2 < 0.0 ) eval_dis2_f2(i,ii,j,jj,ixyz,jxyz,dis2,f2);
    if ( dis < 0.0 ) dis = std::sqrt( dis2 ); // didn't take sqrt yet
    tether_deriv = tether_factor * tether_res_weight(i) * tether_res_weight(j) *
     dis * ( dis2_tether(i,ii,j,jj) - dis2 );
  }

L100:

	if ( dis2 < 0.0 ) eval_dis2_f2(i,ii,j,jj,ixyz,jxyz,dis2,f2);
	if ( dis < 0.0 ) dis = std::sqrt( dis2 ); // didn't take sqrt yet
	classical_constraints::BOUNDARY::eval_constraint_deriv(i,ii,j,jj,fullatom,dis,dis2,cst_deriv);

	disulfides::BOUNDARY::eval_disulf_dist_deriv(i,ii,j,jj,fullatom,disulf_deriv);

	if ( geometric_sol::geometric_sol_flag ) solv_deriv *= geometric_sol::lk_sol_weight; // a little bit of the usual solvation?

  dE_dR = vdw_deriv + solv_deriv + cst_deriv + tether_deriv + disulf_deriv + elec_deriv;

	if ( dE_dR == 0.0 ) return;
	if ( dis2 < 0.0 ) eval_dis2_f2(i,ii,j,jj,ixyz,jxyz,dis2,f2);
	if ( dis < 0.0 ) dis = std::sqrt( dis2 ); // dis not evaluated yet
	if ( dis > 0.0 ) dE_dR /= dis;
	cros(ixyz,jxyz,f1);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin eval_dis2_f2
///
/// @brief
///
/// @detailed
///
/// @param  i - [in/out]? -
/// @param  ii - [in/out]? -
/// @param  j - [in/out]? -
/// @param  jj - [in/out]? -
/// @param  ixyz - [in/out]? - dimension( 3 )
/// @param  jxyz - [in/out]? - dimension( 3 )
/// @param  dis2 - [in/out]? -
/// @param  f2 - [in/out]? - dimension( 3 )
///
/// @global_read
///
/// @global_write
///
/// @remarks
///car note: assume that if this function is being called for a proton,
///car it is because it is involved in a constraint, and therefore,
///car eval_pairConstraint has updated all necesssary coordinates
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
eval_dis2_f2(
	int i,
	int ii,
	int j,
	int jj,
	FArray1DB_float & ixyz, // dimension( 3 )
	FArray1DB_float & jxyz, // dimension( 3 )
	float & dis2,
	FArray1DB_float & f2 // dimension( 3 )
)
{
	using namespace misc;

	if ( ii > 0 ) {
		int l = full_coord.index(1,ii,i);
		ixyz(1) = full_coord[   l ]; // full_coord(1,ii,i)
		ixyz(2) = full_coord[ ++l ]; // full_coord(2,ii,i)
		ixyz(3) = full_coord[ ++l ]; // full_coord(3,ii,i)
	} else if ( ii == 0 ) {                     // protons
		ixyz(1) = classical_constraints::BOUNDARY::HN_position(1,i);
		ixyz(2) = classical_constraints::BOUNDARY::HN_position(2,i);
		ixyz(3) = classical_constraints::BOUNDARY::HN_position(3,i);
	} else if ( ii == -1 ) {
		ixyz(1) = classical_constraints::BOUNDARY::HA_position(1,i);
		ixyz(2) = classical_constraints::BOUNDARY::HA_position(2,i);
		ixyz(3) = classical_constraints::BOUNDARY::HA_position(3,i);
	} else {
		std::cout << "ERROR:: unknown atom type in eval_dis2_f2: "
		 << i << ' ' << ii << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( jj > 0 ) {
		int l = full_coord.index(1,jj,j);
		jxyz(1) = full_coord[   l ]; // full_coord(1,jj,j)
		jxyz(2) = full_coord[ ++l ]; // full_coord(2,jj,j)
		jxyz(3) = full_coord[ ++l ]; // full_coord(3,jj,j)
	} else if ( jj == 0 ) {                     // protons
		jxyz(1) = classical_constraints::BOUNDARY::HN_position(1,j);
		jxyz(2) = classical_constraints::BOUNDARY::HN_position(2,j);
		jxyz(3) = classical_constraints::BOUNDARY::HN_position(3,j);
	} else if ( jj == -1 ) {
		jxyz(1) = classical_constraints::BOUNDARY::HA_position(1,j);
		jxyz(2) = classical_constraints::BOUNDARY::HA_position(2,j);
		jxyz(3) = classical_constraints::BOUNDARY::HA_position(3,j);
	} else {
		std::cout << "ERROR:: unknown atom type in eval_dis2_f2: "
		 << j << ' ' << jj << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	f2(1) = ixyz(1) - jxyz(1);
	f2(2) = ixyz(2) - jxyz(2);
	f2(3) = ixyz(3) - jxyz(3);

	dis2 = ( f2(1) * f2(1) ) + ( f2(2) * f2(2) ) + ( f2(3) * f2(3) );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_atm_num_in_count_pair
///
/// @brief
///     given the full_coord atm num, decide the number for looking up the
///     cp_table. N=1,CA=2,C=3,O=4,CB=5,SC=6,HN=7,HA=8. This only applies in
///     fullatom mode  --Chu
///
/// @detailed
///
/// @param  aa - [in/out]? -
/// @param  aav - [in/out]? -
/// @param  atm_full - [in/out]? -
/// @param  atm_cp - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_atm_num_in_count_pair(
	int const aa,
	int const aav,
	int const atm_full,
	int & atm_cp
)
{

	using namespace aaproperties_pack;
	using namespace param_aa;

	if ( aa == aa_gly ) {
		if ( atm_full <= 4 ) { // backbone
			atm_cp = atm_full;
		} else if ( LookupByName( aa, aav, " OXT" ) == atm_full ) {
			atm_cp = 4;
		} else if ( LookupByName( aa, aav, "1H  " ) == atm_full ||
                LookupByName( aa, aav, "2H  " ) == atm_full ||
                LookupByName( aa, aav, "3H  " ) == atm_full){
			atm_cp = 7;
		} else if ( atm_full == 5 ) { //HN
			atm_cp = 7;
		} else { // HA
			atm_cp = 8;
		}
	} else {
		if ( atm_full <= 5 ) {
			atm_cp = atm_full;
		} else if ( LookupByName( aa, aav, " OXT" ) == atm_full ) {
			atm_cp = 4;
		} else if ( LookupByName( aa, aav, "1H  " ) == atm_full ||
								LookupByName( aa, aav, "2H  " ) == atm_full ||
								LookupByName( aa, aav, "3H  " ) == atm_full ) {
			atm_cp = 7;
		} else if ( atm_full == HNpos(aa,aav) ) {
			atm_cp = 7;
		} else if ( atm_full == HApos(aa,aav) ) {
			atm_cp = 8;
		} else {
			atm_cp = 6;
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin eval_total_dunbrack_deriv
///
/// @brief
///
/// @detailed
///
/// @param  first_res - [in/out]? -
/// @param  last_res - [in/out]? -
/// @param  chiarray - [in/out]? -
/// @param  rotarray - [in/out]? -
/// @param  dunbrack_deriv - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
eval_total_dunbrack_deriv(
	int const first_res,
	int const last_res,
	FArray2DB_float & chiarray,
	FArray2DB_int & rotarray,
	FArray2DB_float & dunbrack_deriv
)
{
	using namespace misc;
	using namespace param;
	using namespace param_torsion;

//	chiarray.dimension( MAX_CHI, MAX_RES() );
//	rotarray.dimension( MAX_CHI, MAX_RES() );
//	dunbrack_deriv.dimension( total_torsion, MAX_RES() );

// local
	FArray1D_float dE_dchi( MAX_CHI );
	float dE_dphi, dE_dpsi;
//------------------------------------------------------------------------------
	for ( int seqpos = first_res; seqpos <= last_res; ++seqpos ) {
		for ( int torsion = 1; torsion <= total_torsion; ++torsion ) {
			dunbrack_deriv(torsion,seqpos) = 0.;
		}
		eval_dunbrack_derivs(res(seqpos),res_variant(seqpos),phi(seqpos),
		 psi(seqpos),seqpos,total_residue,chiarray(1,seqpos),rotarray(1,seqpos),
		 dE_dphi,dE_dpsi,dE_dchi);
		dunbrack_deriv(phi_torsion,seqpos) = dE_dphi;
		dunbrack_deriv(psi_torsion,seqpos) = dE_dpsi;
		for ( int i = 1; i <= MAX_CHI; ++i ) {
			if ( total_chi_torsion >= i )
			 dunbrack_deriv(chi1_torsion+i-1,seqpos) = dE_dchi(i);
		}
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin get_total_pair_deriv
///
/// @brief
///
/// @detailed
///    Gets the F1 and F2 pair energy derivative vectors from the
///  Braun and Go recursive torsion derivative scheme for every
///  torsion angle on the protein backbone [in the range first_res-last_res]
///    These results can be
///  subsequently combined with any subset of torsion angles in imap
///  to get any set of dpairE_dtorsion.
///
///  ...lots of this is copied from code by DB and Bill Wedemeyer
///ctsa
///
/// @param  first_res - [in/out]? -
/// @param  last_res - [in/out]? -
/// @param  F1_pair - [in/out]? -
/// @param  F2_pair - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_total_pair_deriv(
	int const first_res,
	int const last_res,
	FArray3DB_float & F1_pair,
	FArray3DB_float & F2_pair
)
{
	using namespace minimize_ns;
	using namespace misc;
	using namespace param;
	using namespace param_torsion;

//	F1_pair.dimension( 3, total_torsion, total_residue );
//	F2_pair.dimension( 3, total_torsion, total_residue );

// locals
	FArray2D_float actcoord( 3, total_residue );
	FArray1D_int neighbors( total_residue );
	FArray2D_bool neighborlist( MAX_RES()(), MAX_RES()() );

	//// highest no. torsion angle that controls wcentroid
	FArray1D_int wcentroid_torsion( MAX_AA()() );

// tmp hack to get torsion order right
	FArray1D_int torsion_order_map( total_torsion );
//------------------------------------------------------------------------------
	//// define the torsion order
	////
	torsion_order_map(1) = 1;
	for ( int j = 1; j <= total_chi_torsion; ++j ) {
		torsion_order_map(1+j) = 3+j;
	}
	torsion_order_map(2+total_chi_torsion) = 2;
	torsion_order_map(3+total_chi_torsion) = 3;

	//// ctsa  get residue-residue neighborlist and density bins
	////
	make_neighbor_info(res,total_residue,full_coord,neighborlist,neighbors);
	//// ctsa    ...and determine chemical centers
	////
	for ( int seqpos1 = 1; seqpos1 <= total_residue; ++seqpos1 ) {
		int aa1 = res(seqpos1);
		put_wcentroid(full_coord(1,1,seqpos1),actcoord(1,seqpos1),aa1);
	}

	//// ctsa- get highest numbered torsion angles that control wcentroid for each aa
	////
	for ( int aa1 = 1, aae = MAX_AA(); aa1 <= aae; ++aa1 ) {
		get_wcentroid_closest_chi(wcentroid_torsion(aa1),aa1);
		if ( wcentroid_torsion(aa1) == 0 || !minimize_vary_chi ||
		 total_chi_torsion <= 0 ) {
			wcentroid_torsion(aa1) = phi_torsion;
		} else {
			wcentroid_torsion(aa1) += total_bb_torsion;
			wcentroid_torsion(aa1) = std::min(wcentroid_torsion(aa1),total_torsion);
		}
	}

	//// init F1,F2
	////
	F1_pair = 0.0f;
	F2_pair = 0.0f;
//	for ( int seqpos1 = 1; seqpos1 <= total_residue; ++seqpos1 ) {
//		for ( int torsion = 1; torsion <= total_torsion; ++torsion ) {
//			for ( int j = 1; j <= 3; ++j ) {
//				F1_pair(j,torsion,seqpos1) = 0.;
//				F2_pair(j,torsion,seqpos1) = 0.;
//			}
//		}
//	}

	//// ctsa - add in the C-terminal rigid body if one exists
	////
	int seg_begin, seg_end;
	int seg_num = identify_segment(last_res);
	retrieve_segment_ends(seg_num,seg_begin,seg_end);
	if ( last_res != seg_end ) {
		int torsion = omega_torsion;
		 // make the c-term effects start at the first active bb torsion
		for ( int seqpos1 = last_res+1; seqpos1 <= seg_end; ++seqpos1 ) {
			int aa1 = res(seqpos1);
			for ( int seqpos2 = 1; seqpos2 <= last_res; ++seqpos2 ) {
				if ( neighborlist(seqpos1,seqpos2) ) {
					int aa2 = res(seqpos2);
					calc_and_sum_pair_deriv(aa1,aa2,seqpos1,seqpos2,actcoord(1,seqpos1),
					 actcoord(1,seqpos2),F1_pair(1,torsion,last_res),
					 F2_pair(1,torsion,last_res));
				}
			}
		}
	}

	//// ctsa - now add in 1 res at a time through the flexible region
	////
	for ( int seqpos1 = last_res; seqpos1 >= first_res; --seqpos1 ) {
		int aa1 = res(seqpos1);

		//// over all torsion angles,
		for ( int tor = total_torsion; tor >= 1; --tor ) {
			int torsion = torsion_order_map(tor);

			if ( minimize_vary_chi ) {
				if ( torsion != wcentroid_torsion(aa1) ) goto L313;
			} else {
				//// use phi for a sc-sc potential when no chis are in
				////
				if ( torsion != phi_torsion ) goto L313;
			}

			//// ctsa- sum over all interactions between two sets of atoms:
			//// first is the set of atoms upstream of the current torsion
			//// affected by no other upstream torsion angles, the second
			//// is the complement of this set.
			////
			for ( int seqpos2 = 1; seqpos2 <= total_residue; ++seqpos2 ) {
				if ( seqpos1 == seqpos2 ) {
					if ( neighborlist(seqpos1,seqpos2) ) {
						int aa2 = res(seqpos2);
						calc_and_sum_pair_deriv(aa1,aa2,seqpos1,seqpos2,actcoord(1,seqpos1),
						 actcoord(1,seqpos2),F1_pair(1,torsion,seqpos1),
						 F2_pair(1,torsion,seqpos1));
					}
				}
			}

L313:
			//// ctsa - link this F1,F2 to the preceding C-terminal F1,F2
			////
			link_torsion_vectors(seqpos1,torsion,F1_pair,F2_pair);

		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin calc_and_sum_pair_deriv
///
/// @brief
///  get the pair deriv w.r.t distance, and transform and sum it into F1,F2
///ctsa
///
/// @detailed
///
/// @param  aa1 - [in/out]? -
/// @param  aa2 - [in/out]? -
/// @param  res1 - [in/out]? -
/// @param  res2 - [in/out]? -
/// @param  coord1 - [in/out]? -
/// @param  coord2 - [in/out]? -
/// @param  F1_pair - [in/out]? -
/// @param  F2_pair - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calc_and_sum_pair_deriv(
	int const aa1,
	int const aa2,
	int const res1,
	int const res2,
	FArray1Da_float coord1,
	FArray1Da_float coord2,
	FArray1Da_float F1_pair,
	FArray1Da_float F2_pair
)
{
	coord1.dimension( 3 );
	coord2.dimension( 3 );
	F1_pair.dimension( 3 );
	F2_pair.dimension( 3 );


// return
// parameters
// local
	FArray1D_float f1( 3 );
	FArray1D_float f2( 3 );
	float tmp1,dpairE_dr;
//------------------------------------------------------------------------------
	get_pairtermE(aa1,aa2,res1,res2,coord1,coord2,tmp1,dpairE_dr);

	if ( dpairE_dr == 0. ) return;

	cros(coord1,coord2,f1);
	subvec(coord1,coord2,f2);
	float dis =
	 std::sqrt( ( f2(1) * f2(1) ) + ( f2(2) * f2(2) ) + ( f2(3) * f2(3) ) );

	if ( dis == 0.0 ) {
		//// serious bug
		std::cout << "STOP :: Active site pair collision in " <<
		 "get_total_pair_deriv!" << std::endl;
		std::cout << "dis" << SS( dis ) << " f2" << SS( f2(1) ) << SS( f2(2) ) <<
		 SS( f2(3) ) << std::endl;
		std::cout << "xyz1" << SS( coord1(1) ) << SS( coord1(2) ) <<
		 SS( coord1(3) ) << std::endl;
		std::cout << "xyz2" << SS( coord2(1) ) << SS( coord2(2) ) <<
		 SS( coord2(3) ) << std::endl;
		std::cout << "  Aborting..." << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	dpairE_dr /= dis;

	for ( int i = 1; i <= 3; ++i ) {
		F1_pair(i) += dpairE_dr * f1(i);
		F2_pair(i) += dpairE_dr * f2(i);
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin eval_local_vdw_deriv
///
/// @brief
///
/// @detailed
///
/// @param  i - [in/out]? -
/// @param  ii - [in/out]? -
/// @param  j - [in/out]? -
/// @param  jj - [in/out]? -
/// @param  resi - [in/out]? -
/// @param  resj - [in/out]? -
/// @param  resvi - [in/out]? -
/// @param  resvj - [in/out]? -
/// @param  dis - [in/out]? -
/// @param  dis2 - [in/out]? -
/// @param  vdw_deriv - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
eval_local_vdw_deriv(
	int i,
	int ii,
	int j,
	int jj,
	int resi,
	int resj,
	int resvi,
	int resvj,
	float dis,
	float dis2,
	float & vdw_deriv
)
{
	using namespace minimize_ns::deriv_precalc;
	using namespace runlevel_ns;
	using namespace scorefxns;

//car for derivative!!!  1 = n,2 = ca,3 = c,4 = o,5 = cb, 6 = centroid
	vdw_deriv = 0.0;
	if ( vdw_weight == 0.0 ) return;
	float radius2 = vdw::atom_vdw(all_atom_type(resi,resvi,ii,false), /* not fullatom */
	 all_atom_type(resj,resvj,jj,false));

//car weights
//car 0.8 is in vdw_compute  (4*0.8 = 3.2   4 is from the derivative)
//car vdw_weight is applied to vdw_score in scorefxn (and thus in func_minimize)
//car local_vdw_weight is an additional weight just for local bumps
	if ( dis2 < radius2 ) vdw_deriv = vdw_factor*dis*(radius2-dis2)/radius2;

	if ( runlevel > gush && std::abs(vdw_deriv) > 0.00001f ) std::cout <<
	 "local_vdw_deriv " << SS( i ) << SS( j ) << SS( ii ) << SS( jj ) <<
	  SS( vdw_deriv ) << SS( radius2 ) << SS( dis2 ) << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin drv_update_nblist
///
/// @brief  add atom pairs to the nblist that are not neighbors but
///         that have a non-zero score
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///   constaints can have scores & derivs even at long distances
///      this function relies on constraints on nblist having been
///       marked as on the nblist when the nblist was updated
///   disulfides may also be long range
///   tether scoring functions
///
/// @references
///
/// @authors  car 9/22/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
drv_update_nblist()
{
	using namespace disulfides::BOUNDARY;
	using namespace classical_constraints::BOUNDARY;
	using namespace nblist;
	using namespace scorefxns;

//car local
	int res1,res2,atom1,atom2;
	int n_disulfides;

	if ( !get_use_nblist() ) return;
	int const stage = ( pc_mode >= 0 ? get_max_seqSep() : 9999 );
	if ( stage <= 0 ) return;

	bool const fullatom = get_fullatom_flag();
	for ( int i = 1; i <= total_atoms; ++i ) {
		drv_nb_len(i) = nb_len(i);
	}

//car add constraints
	if ( classical_constraints::BOUNDARY::get_constraints_exist() ) {
		int np_pairs = classical_constraints::BOUNDARY::np_pairs();
		for ( int i = 1; i <= np_pairs; ++i ) {
			if ( classical_constraints::nblist::cst_nb(i) ) goto L100; // already on list
			if ( pairSeqSep(i) > stage ) goto L100; // too long range
			if ( dist_constraint(i) < pairRadius(i) && dist_constraint(i) > pairMinRadius(i) ) goto L100;
			 // satisfied
//car add to list!
			res1 = constraintPair(i,1);
			res2 = constraintPair(i,2);
			atom1 = pair_atom_index(1,i);
			atom2 = pair_atom_index(2,i);
			if ( cst_pair_satisfies_conditions(i,fullatom,stage) ) {

				if ( atom1 == classical_constraints::constants::CENTROID_ATOMNUM ) atom1 = 6;
				if ( atom2 == classical_constraints::constants::CENTROID_ATOMNUM ) atom2 = 6; // convert cen numbering
				add_to_nblist(res1,atom1,res2,atom2,true);
				add_to_nblist(res2,atom2,res1,atom1,true);
			}
		L100:;
		}
	}

	if ( disulfides::options::find_disulf || disulfides::options::fix_disulf ) {

		if ( disulfides::options::fix_disulf ) {
			n_disulfides = get_n_known_disulfides();
		} else {
			n_disulfides = get_n_disulf_centroid();
		}

		for ( int i = 1; i <= n_disulfides; ++i ) {
			res1 = get_cys(get_disulf_partner_a(i));
			res2 = get_cys(get_disulf_partner_b(i));
//bs add cb-cb for cys-cys in disulf
			atom1 = 5;
			atom2 = 5;
			add_to_nblist(res1,atom1,res2,atom2,true);

//bs  add either cen-cen (! fullatom) or s-s (fullatom)
//bs in either case atom1 = atom2 = 6
			atom1 = 6;
			atom2 = 6;
			add_to_nblist(res1,atom1,res2,atom2,true);
		}
	}

//car add tether
//car assume if tethered, not using nb_list (idealizing cannot use the nblist)
//car otherwise tethered pairs must be added to the nblist if they arent
//car on it already
	if ( tether_weight != 0.0 ) {
		std::cout << "STOPPING:: nblist must be updated properly before" <<
		 " it can be used in combination with tether_score" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin hijack_best_pose_release
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
hijack_best_pose_release()
{
	using namespace aaproperties_pack;
	using namespace ligand;
	using namespace misc;
	using namespace minimize_ns::save_pose;

	if ( !hijack ) return;

	hijack = false;

	for ( int k = 1; k <= total_residue; ++k ) {
		best_phi(k) = save_phi(k);
		best_psi(k) = save_psi(k);
		best_omega(k) = save_omega(k);
		best_res(k) = save_res(k);
		best_res_variant(k) = save_res_variant(k);
		for ( int i = 1; i <= 3; ++i ) {
			for ( int j = 1; j <= 5; ++j ) {
				Ebest_position(i,j,k) = Esave_position(i,j,k);
			}
			best_centroid(i,k) = save_centroid(i,k);
		}
	}

	if ( get_fullatom_flag() ) {
		for ( int k = 1; k <= total_residue; ++k ) {
			int const atom_end = natoms(best_res(k),best_res_variant(k));
			for ( int j = 1; j <= atom_end; ++j ) {
				for ( int i = 1; i <= 3; ++i ) {
					best_full_coord(i,j,k) = save_full_coord(i,j,k);
				}
			}
		}
	}

 if ( get_ligand_flag() ) {

    for ( size_t j =0;j < ligand::ligand_one->best_coord.size(); ++j ) {
      if(ligand::ligand_one->best_coord.size()==save_hetero_atom_coord.size()){
        ligand::ligand_one->best_coord[j]=save_hetero_atom_coord[j];
      }else{
        std::cerr << "best_coord and save_hetero_atom_coord have different sizes " << save_hetero_atom_coord.size()<< " " << ligand::ligand_one->best_coord.size() << std::endl;
      }
    }
  }
}

////////////////////////////////////////////////////////////////////////////////
/// @begin hijack_best_pose
///
/// @brief
///
/// @detailed
///car fold relative to the faux best
///car but score relative to the true best
///car  thus we put the current in the best arrays, but DO NOT CHANGE
///car  the scoring functions (ie do not call monte_carlo_accept_best)
///car  note that we must still refold the entire region changed relative to
///car  TRUE best because this will determine what we score
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
hijack_best_pose()
{

	using namespace aaproperties_pack;
	using namespace ligand;
	using namespace misc;
	using namespace minimize_ns::save_pose;

//------------------------------------------------------------------------------

	hijack = true;

	for ( int i = 1; i <= total_residue; ++i ) {
		save_phi(i) = best_phi(i);
		save_psi(i) = best_psi(i);
		save_omega(i) = best_omega(i);
		save_res(i) = best_res(i);
		save_res_variant(i) = best_res_variant(i);
		for ( int k = 1; k <= 3; ++k ) {
			for ( int j = 1; j <= 5; ++j ) {
				Esave_position(k,j,i) = Ebest_position(k,j,i);
			}
			save_centroid(k,i) = best_centroid(k,i);
		}
	}

//car save rotamers
	if ( get_fullatom_flag() ) {
		for ( int i = 1; i <= total_residue; ++i ) {
			int const atom_end = natoms(best_res(i),best_res_variant(i));
			for ( int j = 1; j <= atom_end; ++j ) {
				for ( int k = 1; k <= 3; ++k ) {
					save_full_coord(k,j,i) = best_full_coord(k,j,i);
				}
			}
		}
	}

//mj save ligand
  if ( get_ligand_flag() ) {
    for ( size_t j = 0; j < ligand::ligand_one->best_coord.size(); ++j ) {
      if(j==0){
        save_hetero_atom_coord.clear();
        save_hetero_atom_coord.reserve(ligand::ligand_one->best_coord.size());
      }
      save_hetero_atom_coord.push_back(ligand::ligand_one->best_coord[j]);
    }
  }

//car put current in best
	for ( int i = 1; i <= total_residue; ++i ) {
		best_phi(i) = phi(i);
		best_psi(i) = psi(i);
		best_omega(i) = omega(i);
		best_res(i) = res(i);
		best_res_variant(i) = res_variant(i);
	}

	for ( int i = 1; i <= total_residue*5; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			best_position(j,i) = position(j,i);
		}
	}
	for ( int i = 1; i <= total_residue; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			best_centroid(j,i) = centroid(j,i);
		}
	}

	if ( get_fullatom_flag() ) {
		for ( int k = 1; k <= total_residue; ++k ) {
			int const atom_end = natoms(res(k),res_variant(k));
			for ( int j = 1; j <= atom_end; ++j ) {
				for ( int i = 1; i <= 3; ++i ) {
					best_full_coord(i,j,k) = full_coord(i,j,k);
				}
			}
		}
	}

  if ( get_ligand_flag() ) {
    ligand_one->set_best_coordinates();
  }
}

////////////////////////////////////////////////////////////////////////////////
/// @begin debug_minimize_allow_torsion
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
debug_minimize_allow_torsion()
{
	using namespace minimize_ns;
	using namespace misc;
	using namespace param;
	using namespace param_torsion;

	for ( int i = 1; i <= total_residue; ++i ) {
		for ( int torsion = 1; torsion <= jmp_total_torsion; ++torsion ) {
			if ( minimize_allow_torsion(torsion,i) ) {
				if ( ( ! minimize_vary_phipsi &&
				 ( torsion == phi_torsion || torsion == psi_torsion ) ) ||
				 ( ! minimize_vary_omega && torsion == omega_torsion ) ||
				 ( ! minimize_vary_chi && torsion > total_bb_torsion &&
				 torsion <= total_bb_torsion + MAX_CHI ) ||
				 ( ! minimize_vary_rb_trans && torsion >= first_rb_torsion &&
				 torsion < first_rb_torsion + 3 ) ||
				 ( ! minimize_vary_rb_angle && torsion >= first_rb_torsion+3 &&
				 torsion < first_rb_torsion + 6 ) ||
				 ( minimize_excludeH && secstruct(i) == 'H' &&
				 torsion_type_is_bb(torsion) ) ||
				 ( minimize_excludeE && secstruct(i) == 'E' &&
				 torsion_type_is_bb(torsion) ) ) {
					std::cout << "problemo in minimize_allow_torsion_torsion" <<
					 SS( i ) << SS( torsion ) << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_set_func
///
/// @brief
///
/// @detailed
///
/// @param  which_func - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
minimize_set_func( int which_func )
{
//car  what function to call:
//car  1 wobble          (msd of splint region)
//car  2 minimize trials (func=scoring_function; dfunc=vdw + constraints)
//car  3 idealizing      (vdw+dme)
//car  4 docking

	using namespace minimize_ns::func_switch;
	using namespace minimize_ns::linmin_param;

	func_evals = 0;
	switch_value = which_func; // JSS global value set and used later
	switch(switch_value) {
	case 1:          // wobble
		ax_init = 0.0;
		xx_init = 0.1;
		bx_init = 0.2;
		break;
	case 2:   // min moves
		if ( get_pose_docking_flag() &&  // pose_docking
			minimize_get_current_pose().get_allow_jump_move(pose_ns::pose_param::dock_jump)) {
			ax_init = 0.0;
			xx_init = 0.001;
			bx_init = 0.002;
		} else {
			ax_init = 0.0;
			xx_init = 0.1;
			bx_init = 0.2;
		}
		break;
	case 3:   // idealize
		ax_init = 0.0;
		xx_init = .001;
		bx_init = .002;
		break;
	case 4:   // docking
		ax_init = 0.0;
		xx_init = .001;
		bx_init = .002;
		break;
	case 5:   // sc min in rotamer trial
		ax_init = 0.0;
		xx_init = 1.0; // one degree from starting
		bx_init = 2.0;
		break;
	case 6:   //ora: currently not used; slot can be used for new minimization mode
		ax_init = 0.0;
		xx_init = .01;
		bx_init = .02;
		break;
	case 7:   // mdaily, dock_pivot
	  ax_init = 0.0;
	  xx_init = 0.3;
	  bx_init = 0.6;
		break;
	case 8:   // prot-dna
		ax_init = 0.0; // am these are some obscure linmin params, do what chu did for sc min in rotamer trial
		xx_init = 1.0;
		bx_init = 2.0;
		break;
	case 9:   // backrub
		ax_init = 0.0;
		xx_init = 1.0;
		bx_init = 2.0;
		break;
	default:
		ax_init = 0.0;
		xx_init = 1.0;
		bx_init = 2.0;
		break;
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin func
///
/// @brief
///
/// @detailed
///
/// @param  gfrag - [in/out]? - flag to indicate whether phi's and psi's have
///                  nonsense
/// @param  values - [in/out]? - (this can sometimes result from minimization)
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
func(
	bool & gfrag,
	FArray1DB_float & values
)
{
	using namespace minimize_ns::func_switch;

//car gfrag is a flag to indicate whether phi's and psi's have nonsense
//car values (this can sometimes result from minimization)

	++func_evals;
	switch (switch_value) {
	case 1:
		float msd_score, rama_score;
		return func_wobble(values,gfrag, msd_score, rama_score);
	case 2:
		return func_vdw(values,gfrag);
	case 3:
		return func_vdw(values,gfrag);
	case 4:
		return func_dvdw(values,gfrag);
	case 5:
		return func_scvdw(values,gfrag);
	case 6: //ora: currently not used - slot can be used for new minimization mode
		std::cout << "case 6 func currently undefined!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		return 0.0; // Keeps compiler happy
	case 7: // mdaily, dock_pivot
		return func_pivot(values);
	case 8: // prot-dna
		return func_dna(values,gfrag);
	case 9: // backrub
		return minimize_func_backrub_eval(values,gfrag);
	default:
		std::cout << "func undefined" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		return 0.0; // Keeps compiler happy
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin dfunc
///
/// @brief
///
/// @detailed
///
/// @param  values - [in/out]? -
/// @param  derivatives - [in/out]? -
/// @param  n - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
dfunc(
	FArray1DB_float & values,
	FArray1DB_float & derivatives,
	int n
)
{
	using namespace minimize_ns::func_switch;
	PROF_START( prof::DFUNC );
	switch ( switch_value) {
	case 1:
		dfunc_wobble(values,derivatives,n);
		break;
	case 2:
		dfunc_vdw(values,derivatives,n);
		break;
	case 3:
		dfunc_vdw(values,derivatives,n);
		break;
	case 4: // docking van der Waals
		dfunc_dvdw(values,derivatives,n);
		break;
	case 5: // side-chain minimization
		dfunc_scvdw(values,derivatives,n);
		break;
	case 9: // backrub
		minimize_dfunc_backrub_eval(values,derivatives,n);
		break;
	default:
		std::cout << "dfunc undefined" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	PROF_STOP( prof::DFUNC );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_func_evals
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
int
get_func_evals()
{
	using namespace minimize_ns::func_switch;

	return func_evals;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin powell
///
/// @brief
///
/// @detailed
///
/// @param  gfrag - [in/out]? -
/// @param  p - [in/out]? -
/// @param  xi - [in/out]? -
/// @param  n - [in/out]? -
/// @param  np - [in/out]? -
/// @param  ftol - [in/out]? -
/// @param  iter - [in/out]? -
/// @param  fret - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
powell(
	bool & gfrag,
	FArray1DB_float & p,
	int n,
	float ftol,
	int & iter,
	float & fret
)
{
//U    USES func,linmin

	int const ITMAX = { 20 };
	float del, fp, fptt, t;
	FArray1D_float pt( n );
	FArray1D_float ptt( n );
	FArray1D_float xit( n );
	FArray2D_float xi( n, n );

	xi.to_identity();
	fret = func(gfrag,p);
	if ( !gfrag ) return;
	for ( int j = 1; j <= n; ++j ) {
		pt(j) = p(j);
	}
	iter = 0;
L1:
	++iter;
	fp = fret;
	int ibig = 0;
	del = 0.;
	for ( int i = 1; i <= n; ++i ) {
		for ( int j = 1; j <= n; ++j ) {
			xit(j) = xi(j,i);
		}
		fptt = fret;
		linmin(p,xit,n,fret,gfrag);
		if ( !gfrag ) return;
		if ( std::abs(fptt-fret) > del ) {
			del = std::abs(fptt-fret);
			ibig = i;
		}
	}
	if ( 2.*std::abs(fp-fret) <= ftol*(std::abs(fp)+std::abs(fret)) ) return;
	if ( iter == ITMAX) {
#ifdef BOINC
		return;
#endif
#ifdef USEMPI
		return;
#endif
		error_stop( "powell exceeding maximum iterations" );
	}
	for ( int j = 1; j <= n; ++j ) {
		ptt(j) = 2.*p(j)-pt(j);
		xit(j) = p(j) - pt(j);
		pt(j) = p(j);
	}
	fptt = func(gfrag,ptt);
	if ( !gfrag ) return;
	if ( fptt >= fp ) goto L1;
	float const dif1 = fp - fret - del;
	float const dif2 = fp - fptt;
	t = 2.0*( fp - 2.0*fret + fptt )*( dif1 * dif1 ) - del*( dif2 * dif2 );
	if ( t >= 0.0 ) goto L1;
	linmin(p,xit,n,fret,gfrag);
	if ( !gfrag ) return;
	for ( int j = 1; j <= n; ++j ) {
		xi(j,ibig) = xi(j,n);
		xi(j,n) = xit(j);
	}
	goto L1;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin brent_set_abs_tolerance
///
/// @brief
///
/// @detailed
///
/// @param  new_tol - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
brent_set_abs_tolerance( float new_tol )
{
	using namespace minimize_ns::brent_common;

	brent_abs_tolerance = new_tol;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin BRENT
///
/// @brief
///
/// @detailed
///
/// @param  AX - [in/out]? -
/// @param  BX - [in/out]? -
/// @param  CX - [in/out]? -
/// @param  FA - [in/out]? -
/// @param  FB - [in/out]? -
/// @param  FC - [in/out]? -
/// @param  F - [in/out]? -
/// @param  TOL - [in/out]? -
/// @param  XMIN - [in/out]? -
/// @param  gfrag - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
BRENT(
	float const AX,
	float const BX,
	float const CX,
	float & FA,
	float & FB,
	float const FC,
	f_Function_bf F,
	float const TOL,
	float & XMIN,
	bool & gfrag
)
{
	using namespace minimize_ns::brent_common;

	float BRENT; // Return value

	float A, B, E,TOL1,TOL2;
	float V,W,X,FX,FV,FW,XM,R,Q,P,ETEMP,D,U,FU;

	int const ITMAX = { 100 };
	float const CGOLD = { 0.3819660 };
	float const ZEPS = { 1.0E-10 };
//$$$      int func_counter;       // diagnostic only

	A = std::min(AX,CX);
	B = std::max(AX,CX);
	V = BX;
	W = V;
	X = V;
	E = 0.0;
//     these two initializations added to remove warnings
	D = 0.0; // D will be set first time through loop, when if (E>tol) is false
	BRENT = -999.9; // needs to be set in case of early exit due to gfrag = false
//********************************************
	FX = FB;
	if ( A == AX ) {
		FB = FC;
	} else {
		FB = FA;
		FA = FC;
	}
//      FX = F(gfrag,X);
//********************************************
//$$$      func_counter = 1;
	if ( !gfrag ) return BRENT;
	FV = FX;
	FW = FX;
	for ( int iter = 1; iter <= ITMAX; ++iter ) {
		XM = 0.5*(A+B);
		TOL1 = TOL*std::abs(X)+ZEPS;
		TOL2 = 2.*TOL1;

//    std::cout << "A,X,XM,B = " << SS( A ) << SS( X ) << SS( XM ) << SS( B ) << std::endl;
//    std::cout << "FA,FX,FB = " << SS( FA ) << SS( FX ) << SS( FB ) << std::endl;

//********************************************
//     here, we exit BRENT if the function varies by less than a
//     tolerance over the interval
		if ( ( std::abs(FB-FX) <= brent_abs_tolerance ) &&
		 ( std::abs(FA-FX) <= brent_abs_tolerance ) ) goto L3;

//$$$        if ( ( std::abs(FB-FX) <= brent_abs_tolerance ) &&
//$$$         ( std::abs(FA-FX) <= brent_abs_tolerance ) ) {
//$$$           std::cout << "special escape" << std::endl;
//$$$           goto L3;
//$$$        }

//********************************************
//     here, we exit BRENT if we have narrowed the search down to a
//     small change in X
		if ( std::abs(X-XM) <= (TOL2-.5*(B-A)) ) goto L3;
		if ( std::abs(E) > TOL1 ) {
			R = (X-W)*(FX-FV);
			Q = (X-V)*(FX-FW);
			P = (X-V)*Q-(X-W)*R;
			Q = 2.*(Q-R);
			if ( Q > 0.0 ) P = -P;
			Q = std::abs(Q);
			ETEMP = E;
			E = D;
			if ( std::abs(P) >= std::abs(.5*Q*ETEMP) || P <= Q*(A-X) || P >= Q*(B-X) ) goto L1;
			D = P/Q;
			U = X+D;
			if ( U-A < TOL2 || B-U < TOL2 ) D = sign(TOL1,XM-X);
			goto L2;
		}
L1:
		if ( X >= XM ) {
			E = A-X;
		} else {
			E = B-X;
		}
		D = CGOLD*E;
L2:
		if ( std::abs(D) >= TOL1 ) {
			U = X+D;
		} else {
			U = X+sign(TOL1,D);
		}
		FU = F(gfrag,U);
//$$$        ++func_counter;
		if ( !gfrag ) return BRENT;
		if ( FU <= FX ) {
			if ( U >= X ) {
				A = X;
				FA = FX;
			} else {
				B = X;
				FB = FX;
			}
			V = W;
			FV = FW;
			W = X;
			FW = FX;
			X = U;
			FX = FU;
		} else {
			if ( U < X ) {
				A = U;
				FA = FU;
			} else {
				B = U;
				FB = FU;
			}
			if ( FU <= FW || W == X ) {
				V = W;
				FV = FW;
				W = U;
				FW = FU;
			} else if ( FU <= FV || V == X || V == W ) {
				V = U;
				FV = FU;
			}
		}
	}
#ifdef BOINC
	return BRENT;
#endif
	error_stop( "BRENT exceed maximum iterations." );
L3:
	XMIN = X;
	BRENT = FX;
//$$$      std::cout << "BRENT called func " << func_counter << " times" << std::endl;
	return BRENT;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin linmin
///
/// @brief
///
/// @detailed
///
/// @param  P - [in/out]? -
/// @param  XI - [in/out]? -
/// @param  N - [in/out]? -
/// @param  FRET - [in/out]? -
/// @param  gfrag - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
linmin(
	FArray1DB_float & P,
	FArray1DB_float & XI,
	int N,
	float & FRET,
	bool & gfrag
)
{
	using namespace minimize_ns::F1COM;
	using namespace minimize_ns::func_switch;
	using namespace minimize_ns::linmin_param;
	using namespace runlevel_ns;

//	P.dimension( N );
//	XI.dimension( N );

	float AX, XX, BX, FA, FX(0.), FB(0.), XMIN(0.);
	float const TOL = { .1 };

//car added namespace so bracketing parameters can
//car be adjusted to the function being minimized:
//chu added namespace so bracketing parameters can
//chu be adjusted to the functin being minimized inside linmin

//car check local parameters
	if ( N > int(PCOM.size1()) ) {
// 	if ( N > NMAX ) {
		std::cout << "increase NMAX in namespace F1COM" << std::endl;
		std::cout << "NMAX: " << NMAX << " N: " << N << std::endl;
		//utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		// PB: I don't think it's necessary to exit here...
		// uncomment the next two lines and comment out the previous one
		//
		PCOM.dimension( N );
		XICOM.dimension( N );
	}

	float derivmax = 0.0;
	for ( int i = 1; i <= N; ++i ) {
		if ( std::abs(XI(i)) > std::abs(derivmax) ) derivmax = XI(i);
	}
	if ( runlevel >= gush) std::cout << "derivmax," << SS( derivmax ) << std::endl;
	if ( std::abs(derivmax) > .0001 ) goto L10;
	FRET = func(gfrag,P); // deriv = 0, return value
	return;
L10:

	NCOM = N;
	for ( int j = 1; j <= N; ++j ) {
		PCOM(j) = P(j);
		XICOM(j) = XI(j);
	}

	if ( switch_value != 5 ) {
		AX = ax_init;
		XX = xx_init; // initial range for bracketing
		BX = bx_init; // now set in namespace in  minimize_set_func
		if ( runlevel >= gush ) std::cout << "mnbrak" << std::endl;
		MNBRAK(AX,XX,BX,FA,FX,FB,F1DIM,gfrag);
	} else {
//chu   ensure in sc_minimization the frist bracketing on CHI_1 is (XX-AX)
//     degree away from the starting position.
		AX = ax_init/std::max(std::abs(XICOM(1)),1.0E-8f);
		XX = xx_init/std::max(std::abs(XICOM(1)),1.0E-8f);
		BX = bx_init/std::max(std::abs(XICOM(1)),1.0E-8f);
		if ( runlevel >= gush ) std::cout << "mnbrak_small_step" << std::endl;
		MNBRAK_SMALL_STEP(AX,XX,BX,FA,FX,FB,F1DIM,gfrag);
	}
	if ( !gfrag ) std::cout << "linmin gfrag tripped" << std::endl;
	if ( !gfrag ) return;
	if ( runlevel >= gush ) std::cout << "brent" << std::endl;
	FRET = BRENT(AX,XX,BX,FA,FX,FB,F1DIM,TOL,XMIN,gfrag);
	if ( !gfrag ) return;
	for ( int j = 1; j <= N; ++j ) {
		XI(j) *= XMIN;
		P(j) += XI(j);
	}
}


float
F1DIM(
	bool & gfrag,
	float X
)
{
	using namespace minimize_ns::F1COM;
	using namespace runlevel_ns;

	FArray1D_float XT( NCOM );
//car pcom = x values
//car xicom = derivatives

	for ( int j = 1; j <= NCOM; ++j ) {
		XT(j) = PCOM(j) + X*XICOM(j);
	}
	float const F1DIM = func(gfrag,XT); // Return value
	if ( runlevel >= gush) std::cout << "f1dim" << SS( F1DIM ) << SS( X ) << std::endl;
	return F1DIM;
}



/////////////////////////////////////////////////////////////////////////////////
void
ARMIJO(
			 float & XX,
			 float & FP,
			 float & DRV,
			 f_Function_bf func,
			 bool & gfrag,
			 int & NF,
			 float & XMIN,
			 float & FRET
			 )
// INPUT PARAMETERS: XX,FP,DRV,func,gfrag,NF
// OUTPUT PARAMETERS: NF,XMIN,FRET
//
// Given a function FUNC, and initial stepsize XX
// such that 0 < XX and FUNC has negative derivative DRV at 0,
// this routine returns a stepsize XX at least as good as that
// given by Armijo rule.
// Reference:  D.P. Bertsekas, Nonlinear Programming, 2nd ed, 1999, page 29.
{
  float U, FU, BX;
  float const FACTOR =
	{
		0.5
	};
  float const SIGMA =
	{
		0.1
	};
  float const SIGMA2 =
	{
		0.8
	};

	//	FP = func(gfrag,0.0);
	//	if ( !gfrag ) return;
  FRET = func(gfrag,XX);
  NF += 1;
  if ( !gfrag ) return;
  XMIN = XX;

	//if large descent, test to increase it
	//      if (FRET < FP+XX*SIGMA*DRV && XX < 1.0) {
  if (FRET < FP+XX*SIGMA2*DRV )
	{
		U = XX/FACTOR;
		FU = func(gfrag,U);
		NF += 1;
//L1:
      if (FU < FRET)
			{
				XMIN = U;
				FRET = FU;
				//          U = U/FACTOR;
				//          FU = func(gfrag,U);
				//	    NF += 1;
				//          goto L1;
			}
		return;
	}
  BX=XX;
L2:
		if (FRET > FP+XX*SIGMA*DRV)
    {
			// Abort if function value is unlikely to improve.
			//        if ((FRET >= FP-0.01)&(FRET>=FP+0.5)) {
      if (XX <= BX*1e-5 || (XX < 1e-8 && FRET >= FP))
			{

				U=(FRET-FP)/XX;
				std::cout << "Inaccurate G! step= " << SS( XX ) << " Deriv= " << SS( DRV ) << " Finite Diff= " << SS( U ) << std::endl;

				XMIN = 0.0;
				FRET = FP;
				return;
			}
			//        }
      XX *= FACTOR*FACTOR;			// faster step decrease
																//        XX *= FACTOR;
      FRET = func(gfrag,XX);
      NF += 1;
      goto L2;
    }
  XMIN = XX;

  if (XX < 0.0)
	{
		// Parabola interpolate between 0 and XX for refinement
		U = -DRV*XX*XX/(2*(FRET-FP-DRV*XX));
		if (U > BX*1e-3 && U < BX)
		{
			FU = func(gfrag,U);
			NF += 1;
			if (FU < FRET)
			{
				XMIN = U;
				FRET = FU;
			}
		}
	}
  return;
}


/////////////////////////////////////////////////////////////////////////////////
void
linmin_armijo(
							FArray1DB_float & P,
							FArray1DB_float & XI,
							int & NF,
							float & DRV,
							float & FP,
							int & N,
							bool & gfrag,
							float & XMIN,
							float & FRET
							)
// INPUT PARAMETERS: DRV,FP,N,gfrag,P,XI,NF,XMIN
// OUTPUT PARAMETERS: P,XI,NF,XMIN,FRET
{
  using namespace minimize_ns::F1COM;
  using namespace minimize_ns::func_switch;
  using namespace minimize_ns::linmin_param;
  using namespace runlevel_ns;

	//	P.dimension( N );
	//	XI.dimension( N );

  //float AX, BX, FA, FX, FB;
	float XX;
	// , XMIN(0.);
  float const FACTOR = { 0.5 };

  //float const TOL = {	.1	};

	//car added namespace so bracketing parameters can
	//car be adjusted to the function being minimized:
	//chu added namespace so bracketing parameters can
	//chu be adjusted to the functin being minimized inside linmin

	//car check local parameters
  if ( N > int(PCOM.size1()) )
	{
		// 	if ( N > NMAX ) {
		std::cout << "increase NMAX in namespace F1COM" << std::endl;
		std::cout << "NMAX: " << NMAX << " N: " << N << std::endl;
		//utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		// PB: I don't think it's necessary to exit here...
		// uncomment the next two lines and comment out the previous one
		//
		PCOM.dimension( N );
		XICOM.dimension( N );
	}

  float derivmax = 0.0;
  for ( int i = 1; i <= N; ++i )
	{
		if ( std::abs(XI(i)) > std::abs(derivmax) ) derivmax = XI(i);
	}
  if ( runlevel >= gush) std::cout << "derivmax," << SS( derivmax ) << std::endl;
  if ( std::abs(derivmax) > .0001 ) goto L10;
  FRET = func(gfrag,P); // deriv = 0, return value
  return;
L10:

		NCOM = N;
  for ( int j = 1; j <= N; ++j )
	{
		PCOM(j) = P(j);
		XICOM(j) = XI(j);
	}

	//	if ( switch_value != 5 ) {
	//		AX = ax_init;
	//		XX = xx_init; // initial range for bracketing

  XX = XMIN/FACTOR;
  if ( XX > 1.0 ) XX = 1.0;   //initial trial stepsize

	//		BX = bx_init; // now set in namespace in  minimize_set_func
	//		if ( runlevel >= gush ) std::cout << "mnbrak" << std::endl;

  ARMIJO(XX,FP,DRV,F1DIM,gfrag,NF,XMIN,FRET);

	//	} else {
	//chu   ensure in sc_minimization the frist bracketing on CHI_1 is (XX-AX)
	//     degree away from the starting position.
	//		AX = ax_init/std::max(std::abs(XICOM(1)),1.0E-8f);
	//		XX = xx_init/std::max(std::abs(XICOM(1)),1.0E-8f);
	//		BX = bx_init/std::max(std::abs(XICOM(1)),1.0E-8f);
	//		if ( runlevel >= gush ) std::cout << "mnbrak_small_step" << std::endl;
	//		MNBRAK_SMALL_STEP(AX,XX,BX,FA,FX,FB,F1DIM,gfrag);
	//	}

  if ( !gfrag ) std::cout << "linmin gfrag tripped" << std::endl;
  if ( !gfrag ) return;

	//      if ( switch_value = 5 ) {
	//		if ( runlevel >= gush ) std::cout << "brent" << std::endl;
	//		FRET = BRENT(AX,XX,BX,FA,FX,FB,F1DIM,TOL,XMIN,gfrag);
	//		if ( !gfrag ) return;
	//	}

  for ( int j = 1; j <= N; ++j )
	{
		XI(j) *= XMIN;
		P(j) += XI(j);
	}
}




////////////////////////////////////////////////////////////////////////////////
/// @begin MNBRAK
///
/// @brief
///
/// @detailed
///
/// @param  AX - [in/out]? -
/// @param  BX - [in/out]? -
/// @param  CX - [in/out]? -
/// @param  FA - [in/out]? -
/// @param  FB - [in/out]? -
/// @param  FC - [in/out]? -
/// @param  func - [in/out]? -
/// @param  gfrag - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
MNBRAK(
	float & AX,
	float & BX,
	float & CX,
	float & FA,
	float & FB,
	float & FC,
	f_Function_bf func,
	bool & gfrag
)
{
	float DUM, R, Q, U, ULIM, FU;
	float const GOLD = { 1.618034 };
	float const GLIMIT = { 100.0 };
	float const TINY = { 1.E-20 };
//$$$      int func_counter;        // diagnostic

	FA = func(gfrag,AX);
	if ( !gfrag ) return;
	FB = func(gfrag,BX);
	if ( !gfrag ) return;
	if ( FB > FA ) {
		DUM = AX;
		AX = BX;
		BX = DUM;
		DUM = FB;
		FB = FA;
		FA = DUM;
	}
	CX = BX+GOLD*(BX-AX);
	FC = func(gfrag,CX);
//$$$      func_counter = 3;
//$$$      std::cout << "INITIAL VALUES:" << SS( AX ) << SS( BX ) <<
//$$$       SS( CX ) << SS( FA ) << SS( FB ) << SS( FC ) << std::endl;
	if ( !gfrag ) return;
L1:
	if ( FB >= FC ) {
		R = (BX-AX)*(FB-FC);
		Q = (BX-CX)*(FB-FA);
		U = BX-((BX-CX)*Q-(BX-AX)*R)/(2.*sign(std::max(std::abs(Q-R),TINY),Q-R));
		ULIM = BX+GLIMIT*(CX-BX);
		if ( (BX-U)*(U-CX) > 0.0 ) {
			FU = func(gfrag,U);
//$$$          ++func_counter;
//$$$          std::cout << "PARABOLIC U IS BETWEEN B AND C" << std::endl;
			if ( !gfrag ) return;
			if ( FU < FC ) {
				AX = BX;
				FA = FB;
				BX = U;
				FB = FU;
//$$$            std::cout << "GOT A MINIMUM BETWEEN B AND C" << std::endl;
				goto L1;
			} else if ( FU > FB ) {
				CX = U;
				FC = FU;
//$$$            std::cout << "GOT A MINIMUM BETWEEN A AND U" << std::endl;
				goto L1;
			}
			U = CX+GOLD*(CX-BX);
			FU = func(gfrag,U);
//$$$          ++func_counter;
//$$$          std::cout << "PARABOLIC FIT WAS NO USE. USE DEFAULT MAGNIFICATION" << std::endl;
			if ( !gfrag ) return;
		} else if ( (CX-U)*(U-ULIM) > 0. ) {
			FU = func(gfrag,U);
//$$$          ++func_counter;
//$$$          std::cout << "PARABOLIC FIT IS BETWEEN C AND ITS ALLOWED LIMIT" << std::endl;
			if ( !gfrag ) return;
			if ( FU < FC ) {
				BX = CX;
				CX = U;
				U = CX+GOLD*(CX-BX);
				FB = FC;
				FC = FU;
				FU = func(gfrag,U);
//$$$          ++func_counter;
				if ( !gfrag ) return;
			}
		} else if ( (U-ULIM)*(ULIM-CX) >= 0. ) {
			U = ULIM;
			FU = func(gfrag,U);
//$$$          ++func_counter;
//$$$          std::cout << "LIMIT PARABOLIC U TO MAXIMUM ALLOWD VALUE" << std::endl;
			if ( !gfrag ) return;
		} else {
			U = CX+GOLD*(CX-BX);
			FU = func(gfrag,U);
//$$$          ++func_counter;
//$$$          std::cout << "REJECT PARABOLIC U, USE DEFAULT MAGNIFICATION" << std::endl;
			if ( !gfrag ) return;
		}
		AX = BX;
		BX = CX;
		CX = U;
		FA = FB;
		FB = FC;
		FC = FU;
		goto L1;
	}
//$$$      std::cout << "MNBRAK called func " << func_counter << " times" << std::endl;
//$$$      std::cout << "VALUES RETURNED:" << SS( AX ) << SS( BX ) <<
//$$$       SS( CX ) << SS( FA ) << SS( FB ) << SS( FC ) << std::endl;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin MNBRAK_SMALL_STEP
///
/// @brief
///
/// @detailed
///chu   This is a modified version of MNBRAK used for rotamer_minimization.
///     During the process of bracketing the minimum, the size of each forcoming
///     step is further limited by a CONSTANT which is prefined by the initial
///     step * GLIMIT. This measure can void jumping too far by extrapolation since
///     in rotamer_minimization we really want to sample the local minimum around
///     each rotamer.
///
/// @param  AX - [in/out]? -
/// @param  BX - [in/out]? -
/// @param  CX - [in/out]? -
/// @param  FA - [in/out]? -
/// @param  FB - [in/out]? -
/// @param  FC - [in/out]? -
/// @param  func - [in/out]? -
/// @param  gfrag - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
MNBRAK_SMALL_STEP(
	float & AX,
	float & BX,
	float & CX,
	float & FA,
	float & FB,
	float & FC,
	f_Function_bf func,
	bool & gfrag
)
{
	float DUM, R, Q, U, ULIM, FU, STEP,STEP_LIM;
	float const GOLD = { 1.618034 };
	float const GLIMIT = { 10.0 };
	float const TINY = { 1.E-20 };
//$$$      int func_counter        // diagnostic

	FA = func(gfrag,AX);
	if ( !gfrag ) return;
	FB = func(gfrag,BX);
	if ( !gfrag ) return;
	if ( FB > FA ) {
		DUM = AX;
		AX = BX;
		BX = DUM;
		DUM = FB;
		FB = FA;
		FA = DUM;
	}
	STEP_LIM = GLIMIT*(BX-AX); // SET STEP_LIM BASED ON INITIAL AX AND BX
	STEP = GOLD*(BX-AX);
	if ( std::abs(STEP) <= std::abs(STEP_LIM) ) {
		CX = BX+STEP;
	} else {
		CX = BX+STEP_LIM;
	}
	FC = func(gfrag,CX);
//$$$      func_counter = 3;
//$$$      std::cout << "INITIAL VALUES:" << SS( AX ) << SS( BX ) << SS( CX ) <<
//$$$       SS( FA ) << SS( FB ) << SS( FC ) << std::endl;
	if ( !gfrag ) return;
L1:
	if ( FB >= FC ) {
		R = (BX-AX)*(FB-FC);
		Q = (BX-CX)*(FB-FA);
		U = BX-((BX-CX)*Q-(BX-AX)*R)/(2.*sign(std::max(std::abs(Q-R),TINY),Q-R));
		ULIM = CX+STEP_LIM;
		if ( (BX-U)*(U-CX) > 0. ) {
			FU = func(gfrag,U);
//$$$           ++func_counter;
//$$$          std::cout << "PARABOLIC U IS BETWEEN B AND C" << std::endl;
			if ( !gfrag ) return;
			if ( FU < FC ) {
				AX = BX;
				FA = FB;
				BX = U;
				FB = FU;
//$$$            std::cout << "GOT A MINIMUM BETWEEN B AND C" << std::endl;
				goto L1;
			} else if ( FU > FB ) {
				CX = U;
				FC = FU;
//$$$            std::cout << "GOT A MINIMUM BETWEEN A AND U" << std::endl;
				goto L1;
			}
			STEP = GOLD*(CX-BX);
			if ( std::abs(STEP) <= std::abs(STEP_LIM) ) {
				U = CX+STEP;
			} else {
				U = CX+STEP_LIM;
			}
			FU = func(gfrag,U);
//$$$           ++func_counter;
//$$$          std::cout << "PARABOLIC FIT WAS NO USE. USE DEFAULT MAGNIFICATION" << std::endl;
			if ( !gfrag ) return;
		} else if ( (CX-U)*(U-ULIM) > 0. ) {
			FU = func(gfrag,U);
//$$$           ++func_counter;
//$$$          std::cout << "PARABOLIC FIT IS BETWEEN C AND ITS ALLOWED LIMIT" << std::endl;
			if ( !gfrag ) return;
			if ( FU < FC ) {
				BX = CX;
				CX = U;
				STEP = GOLD*(CX-BX);
				if ( std::abs(STEP) < std::abs(STEP_LIM) ) {
					U = CX+STEP;
				} else {
					U = CX+STEP_LIM;
				}
				FB = FC;
				FC = FU;
				FU = func(gfrag,U);
//$$$           ++func_counter;
				if ( !gfrag ) return;
			}
		} else if ( (U-ULIM)*(ULIM-CX) >= 0. ) {
			U = ULIM;
			FU = func(gfrag,U);
//$$$           ++func_counter;
//$$$          std::cout << "LIMIT PARABOLIC U TO MAXIMUM ALLOWED VALUE" << std::endl;
			if ( !gfrag ) return;
		} else {
			STEP = GOLD*(CX-BX);
			if ( std::abs(STEP) <= std::abs(STEP_LIM) ) {
				U = CX+STEP;
			} else {
				U = CX+STEP_LIM;
			}
			FU = func(gfrag,U);
//$$$           ++func_counter;
//$$$           std::cout << "REJECT PARABOLIC U, USE DEFAULT MAGNIFICATION" << std::endl;
			if ( !gfrag ) return;
		}
		AX = BX;
		BX = CX;
		CX = U;
		FA = FB;
		FB = FC;
		FC = FU;
		goto L1;
	}
//$$$	std::cout << "MNBRAK called func " << func_counter << " times" << std::endl;
//$$$	std::cout << "VALUE RETURNED:" << SS( AX ) << SS( BX ) << SS( CX ) <<
//$$$	 SS( FA ) << SS( FB ) << SS( FC ) << std::endl;
}


void
dfpmin(
							FArray1DB_float & P,
							int N,
							float FTOL,
							int & ITER,
							float & FRET,
							bool & gfrag
							)
{
	static bool use_inexact_line_search = false;
	static bool checked_for_option = false;
	static bool use_non_monotone = false;

	if ( !checked_for_option ) {
		use_inexact_line_search = truefalseoption("use_inexact_line_search");
		use_non_monotone        = truefalseoption("use_non_monotone_line_search");
		checked_for_option = true;
		if( use_inexact_line_search && use_non_monotone ){
			std::cout << "ERROR: Cannot use -use_inexact_line_search and -use_non_monotone simultaneously." << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}
	if ( use_inexact_line_search ) {
		dfpmin_armijo( P, N, FTOL, ITER, FRET, gfrag );
	}else
	if(use_non_monotone){
		dfpmin_armijo_nonmonotone( P, N, FTOL, ITER, FRET, gfrag );
	} else {
		dfpmin_rosetta_r13068( P, N, FTOL, ITER, FRET, gfrag );
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin dfpmin
///
/// @brief
///
/// @detailed
///
/// @param  P - [in/out]? -
/// @param  N - [in/out]? -
/// @param  NP - [in/out]? -
/// @param  FTOL - [in/out]? -
/// @param  ITER - [in/out]? -
/// @param  FRET - [in/out]? -
/// @param  gfrag - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
dfpmin_armijo(
			 FArray1DB_float & P,
			 int N,
			 float FTOL,
			 int & ITER,
			 float & FRET,
			 bool & gfrag
			 )
{
  using namespace runlevel_ns;

	using namespace std;

  float FP, FAC, FAE, FAD, FAF, DRV, DRVNEW, GMAX, GNORM, XMIN, XINORM;
  int NF, HOPT;
  int const ITMAX =
	{
		200
	};
  float const EPS =
	{
		1.E-5
	};
  FArray2D_float HESSIN( N, N );
  FArray1D_float XI( N );
  FArray1D_float G( N );
  FArray1D_float DG( N );
  FArray1D_float HDG( N );
	//$$$   int linmin_count;     // diagnostic only
  PROF_START( prof::DFPMIN );
  if ( runlevel >= gush ) std::cout << "dfpmin" << std::endl;
	//$$$   linmin_count = 0;

	// When inexact line search is used, HESSIN need not remain positive definite, so
	// additional safeguard must be added to ensure XI is a desc. direction (or inexact
	// line search would fail).  Two options for safeguards are implemented below:
	//  	HOPT = 1  resets HESSIN to a multiple of identity when XI is not a desc. direction.
	//	HOPT = 2  leaves HESSIN unchanged if stepsize XMIN fails Wolfe's condition
	//		    for ensuring new HESSIN to be positive definite.
  HOPT = 2;

  FP = func(gfrag,P);
  NF = 1;  		// number of func evaluations
  if ( !gfrag ) return;
  dfunc(P,G,N);
  for ( int i = 1; i <= N; ++i )
	{
		for ( int j = 1; j <= N; ++j )
		{
			HESSIN(i,j) = 0.0;
		}
		HESSIN(i,i) = 1.0;
		XI(i) = -G(i);
	}
  XMIN = 1.0;   // initial trial stepsize
  //start_timer("dfpmin");
  for ( ITER = 1; ITER <= ITMAX; ++ITER )
	{
		//$$$   ++linmin_count;
		DRV=0.0;
		GMAX=0.0;
		GNORM=0.0;
		for ( int i = 1; i <= N; ++i )
		{
			DRV += XI(i)*G(i);
			GNORM += G(i)*G(i);
			if (std::abs(G(i))>GMAX)
			{
				GMAX=std::abs(G(i));
			}
		}
		GNORM = std::sqrt(GNORM);

		// P is returned as new pt, and XI is returned as the change.
		linmin_armijo(P,XI,NF,DRV,FP,N,gfrag,XMIN,FRET);

		// std::cout << "N= " << N << " ITER= " << ITER << " #F-eval= " << NF << " maxG= " << SS( GMAX ) << " Gnorm= " << SS( GNORM ) << " step= " << SS( XMIN ) << " func= " << SS( FRET ) << std::endl;

		if ( !gfrag ) return;
		if ( 2.0*std::abs(FRET-FP) <= FTOL*(std::abs(FRET)+std::abs(FP)+EPS) )
		{
			//$$$   std::cout << "dfpmin called linmin " << linmin_count << " times" << std::endl;
			if (GMAX<=1.0)
			{

				//std::cout << "N= " << N << " ITER= " << ITER << " #F-eval= " << NF << " maxG= " << SS( GMAX ) << " Gnorm= " << SS( GNORM ) << " step= " << SS( XMIN ) << " func= " << SS( FRET ) << " time= " << SS( get_timer("dfpmin") ) << std::endl;

				PROF_STOP( prof::DFPMIN );
				return;
			}
			else
			{
				if (std::abs(FRET-FP)<=EPS)
				{
					XINORM = 0.0;
					for ( int i = 1; i <= N; ++i )
					{
						XINORM += XI(i)*XI(i);
					}
					if ( DRV < -1e-3*GNORM*XINORM )
					{

						std::cout << "Failed line search while large DRV, quit! N= " << N << " ITER= " << ITER << " #F-eval= " << NF << " maxG= " << SS( GMAX ) << " Gnorm= " << SS( GNORM ) << " step= " << SS( XMIN ) << " func= " << SS( FRET ) /*<< " time= " << SS( get_timer("dfpmin") )*/ << std::endl;

						PROF_STOP( prof::DFPMIN );
						return;
					}
					// Not convergence yet. Reinitialize HESSIN to a diagonal matrix & update direction XI.
					// This requires G to be correctly the gradient of the function.

					std::cout << ":( reset HESSIN from failed line search" << std::endl;

					DRV=0.0;
					for ( int i = 1; i <= N; ++i )
					{
						for ( int j = 1; j < i; ++j )
						{
							HESSIN(i,j) = 0.0;
						}
						for ( int j = i+1; j <= N; ++j )
						{
							HESSIN(i,j) = 0.0;
						}
						if ( HESSIN(i,i) < 0.01 ) HESSIN(i,i) = 0.01;
						XI(i) = -HESSIN(i,i)*G(i);
						DRV += XI(i)*G(i);
					}
					linmin_armijo(P,XI,NF,DRV,FP,N,gfrag,XMIN,FRET);

					std::cout << "Failed line search again, quit! N= " << N << " ITER= " << ITER << " #F-eval= " << NF << " maxG= " << SS( GMAX ) << " Gnorm= " << SS( GNORM ) << " step= " << SS( XMIN ) << " func= " << SS( FRET ) /*<< " time= " << SS( get_timer("dfpmin") )*/ << std::endl;

					if (std::abs(FRET-FP)<=EPS)
					{
						PROF_STOP( prof::DFPMIN );
						return;
					}
				}
			}
		}
		FP = FRET;
		for ( int i = 1; i <= N; ++i )
		{
			DG(i) = G(i);
		}
		FRET = func(gfrag,P);
		if ( !gfrag ) return;
		dfunc(P,G,N);
		DRV=0.0;						//needed if HOPT = 2
		DRVNEW=0.0;						//needed if HOPT = 2
		for ( int i = 1; i <= N; ++i )
		{
			DRV += XI(i)*DG(i);			//needed if HOPT = 2
			DRVNEW += XI(i)*G(i);			//needed if HOPT = 2
			DG(i) = G(i)-DG(i);
		}

		//if ( XMIN = 0.0 ) std::cout << " XMIN = 0.0! " << std::endl;	//diagnostic

		if ( HOPT == 1 || DRVNEW > 0.95*DRV )
		{			//needed if HOPT = 2

			for ( int i = 1; i <= N; ++i )
			{
				HDG(i) = 0.0;
				for ( int j = 1; j <= N; ++j )
				{
					HDG(i) += HESSIN(i,j)*DG(j);
				}
			}
			FAC = 0.0;
			FAE = 0.0;
			FAF = 0.0;
			for ( int i = 1; i <= N; ++i )
			{
				FAC += DG(i)*XI(i);
				FAE += DG(i)*HDG(i);
				FAF += DG(i)*DG(i);
			}
			FAF = FAC/FAF;
			FAC = 1.0/FAC;
			FAD = 1.0/FAE;
			for ( int i = 1; i <= N; ++i )
			{
				DG(i) = FAC*XI(i) - FAD*HDG(i);
			}
			for ( int i = 1; i <= N; ++i )
			{
				for ( int j = 1; j <= N; ++j )
				{
					HESSIN(i,j) += FAC*XI(i)*XI(j) - FAD*HDG(i)*HDG(j) + FAE*DG(i)*DG(j);
				}
			}

		}									//needed if HOPT = 2

		for ( int i = 1; i <= N; ++i )
		{
			XI(i) = 0.0;
			for ( int j = 1; j <= N; ++j )
			{
				XI(i) -= HESSIN(i,j)*G(j);
			}
		}

		if ( HOPT == 1 )
		{

			DRVNEW=0.0;
			for ( int i = 1; i <= N; ++i )
			{
				DRVNEW += XI(i)*G(i);
			}
			// If direc. deriv >0, reset the Hessian inverse estimate
			if (DRVNEW>-EPS)
			{

				std::cout << "reset hessin; dirdg=" << SS( DRV ) << std::endl;

				if (FAF<0.01) FAF=0.01;
				for ( int i = 1; i <= N; ++i )
				{
					for ( int j = 1; j <= N; ++j )
					{
						HESSIN(i,j) = 0;
					}
					HESSIN(i,i) = FAF;
					XI(i) = -FAF*G(i);
				}
			}

		}

	}

	std::cout << "WARNING: DFPMIN MAX CYCLES " << ITMAX
		        << " EXCEEDED, BUT FUNC NOT CONVERGED!\n";
	return;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin dfpmin
///
/// @brief
///
/// @detailed
///
/// @param  P - [in/out]? -
/// @param  N - [in/out]? -
/// @param  NP - [in/out]? -
/// @param  FTOL - [in/out]? -
/// @param  ITER - [in/out]? -
/// @param  FRET - [in/out]? -
/// @param  gfrag - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
dfpmin_rosetta_r13068(
	FArray1DB_float & P,
	int N,
	float FTOL,
	int & ITER,
	float & FRET,
	bool & gfrag
)
{
	using namespace runlevel_ns;

	float FP, FAC, FAE, FAD;
	int const ITMAX = std::max( 200, N/10 );
	float const EPS = { 1.E-10 };
	FArray2D_float HESSIN( N, N );
	FArray1D_float XI( N );
	FArray1D_float G( N );
	FArray1D_float DG( N );
	FArray1D_float HDG( N );
//$$$   int linmin_count;     // diagnostic only
	PROF_START( prof::DFPMIN );
	if ( runlevel >= gush ) std::cout << "dfpmin  N " << N << "   FTOL " << FTOL << std::endl;
//$$$   linmin_count = 0;
	FP = func(gfrag,P);
	if ( !gfrag ) return;
	dfunc(P,G,N);
	for ( int i = 1; i <= N; ++i ) {
		for ( int j = 1; j <= N; ++j ) {
			HESSIN(i,j) = 0.0;
		}
		HESSIN(i,i) = 1.0;
		XI(i) = -G(i);
	}
	for ( ITER = 1; ITER <= ITMAX; ++ITER ) {
//$$$   ++linmin_count;
		linmin(P,XI,N,FRET,gfrag);
		if ( !gfrag ) return;
		if ( 2.0*std::abs(FRET-FP) <= FTOL*(std::abs(FRET)+std::abs(FP)+EPS) ) {
//$$$   std::cout << "dfpmin called linmin " << linmin_count << " times" << std::endl;
			PROF_STOP( prof::DFPMIN );
			return;
		}
		FP = FRET;
		for ( int i = 1; i <= N; ++i ) {
			DG(i) = G(i);
		}
		FRET = func(gfrag,P);
		if ( !gfrag ) return;
		dfunc(P,G,N);
		for ( int i = 1; i <= N; ++i ) {
			DG(i) = G(i)-DG(i);
		}
		for ( int i = 1; i <= N; ++i ) {
			HDG(i) = 0.0;
			for ( int j = 1; j <= N; ++j ) {
				HDG(i) += HESSIN(i,j)*DG(j);
			}
		}
		FAC = 0.;
		FAE = 0.;
		for ( int i = 1; i <= N; ++i ) {
			FAC += DG(i)*XI(i);
			FAE += DG(i)*HDG(i);
		}
		if ( FAC != 0.0 ) FAC = 1.0/FAC;
		if ( FAE != 0.0 ) {
			FAD = 1./FAE;
		} else {
			FAD = 0.;
		}
		for ( int i = 1; i <= N; ++i ) {
			DG(i) = FAC*XI(i) - FAD*HDG(i);
		}
		for ( int i = 1; i <= N; ++i ) {
			for ( int j = 1; j <= N; ++j ) {
				HESSIN(i,j) += FAC*XI(i)*XI(j) - FAD*HDG(i)*HDG(j) + FAE*DG(i)*DG(j);
			}
		}
		for ( int i = 1; i <= N; ++i ) {
			XI(i) = 0.;
			for ( int j = 1; j <= N; ++j ) {
				XI(i) -= HESSIN(i,j)*G(j);
			}
		}
	}
	std::cout << "WARNING: DFPMIN MAX CYCLES " << ITMAX
						<< " EXCEEDED, BUT FUNC NOT CONVERGED!\n";
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin dfpmin_atol
///
/// @brief
///
/// @detailed
///     JJG 2/18/02: this function is the same as dfpmin, except the tolerance
///     is *Absolute* instead of *relative* (fractional).  i've replaced
///     ftol with atol, an absolute tolerance on the function value
///
/// @param  P - [in/out]? -
/// @param  N - [in/out]? -
/// @param  NP - [in/out]? -
/// @param  ATOL - [in/out]? -
/// @param  ITER - [in/out]? -
/// @param  FRET - [in/out]? -
/// @param  gfrag - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
dfpmin_atol(
	FArray1DB_float & P,
	int N,
	float ATOL,
	int & ITER,
	float & FRET,
	bool & gfrag
)
{
	using namespace runlevel_ns;

	float FP, FAC, FAE, FAD;
	int const ITMAX = std::max( 200, N/10 );
	//float const EPS = { 1.E-10 };
	FArray2D_float HESSIN( N, N );
	FArray1D_float XI( N );
	FArray1D_float G( N );
	FArray1D_float DG( N );
	FArray1D_float HDG( N );
//$$$    int linmin_count;     // diagnostic only

	if ( runlevel >= gush ) std::cout << "dfpmin" << std::endl;
//$$$    linmin_count = 0;
	FP = func(gfrag,P);
	if ( !gfrag ) return;
	dfunc(P,G,N);
	for ( int i = 1; i <= N; ++i ) {
		for ( int j = 1; j <= N; ++j ) {
			HESSIN(i,j) = 0.;
		}
		HESSIN(i,i) = 1.;
		XI(i) = -G(i);
	}
	for ( ITER = 1; ITER <= ITMAX; ++ITER ) {
//$$$      ++linmin_count;
		linmin(P,XI,N,FRET,gfrag);
		if ( !gfrag ) return;
		if ( std::abs(FRET-FP) <= ATOL ) {
//$$$      std::cout << "dfpmin called linmin " << linmin_count << " times" << std::endl;
			return;
		}
		FP = FRET;
		for ( int i = 1; i <= N; ++i ) {
			DG(i) = G(i);
		}
		FRET = func(gfrag,P);
		if ( !gfrag ) return;
		dfunc(P,G,N);
		for ( int i = 1; i <= N; ++i ) {
			DG(i) = G(i)-DG(i);
		}
		for ( int i = 1; i <= N; ++i ) {
			HDG(i) = 0.;
			for ( int j = 1; j <= N; ++j ) {
				HDG(i) += HESSIN(i,j)*DG(j);
			}
		}
		FAC = 0.;
		FAE = 0.;
		for ( int i = 1; i <= N; ++i ) {
			FAC += DG(i)*XI(i);
			FAE += DG(i)*HDG(i);
		}
		FAC = 1./FAC;
		FAD = 1./FAE;
		for ( int i = 1; i <= N; ++i ) {
			DG(i) = FAC*XI(i)-FAD*HDG(i);
		}
		for ( int i = 1; i <= N; ++i ) {
			for ( int j = 1; j <= N; ++j ) {
				HESSIN(i,j) += FAC*XI(i)*XI(j)-FAD*HDG(i)*HDG(j)+FAE*DG(i)*DG(j);
			}
		}
		for ( int i = 1; i <= N; ++i ) {
			XI(i) = 0.;
			for ( int j = 1; j <= N; ++j ) {
				XI(i) -= HESSIN(i,j)*G(j);
			}
		}
	}
	std::cout << "WARNING: DFPMIN MAX CYCLES " << ITMAX
						<< " EXCEEDED, BUT FUNC NOT CONVERGED!\n";
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin frprmn
///
/// @brief
///
/// @detailed
///
/// @param  P - [in/out]? -
/// @param  N - [in/out]? -
/// @param  FTOL - [in/out]? -
/// @param  ITER - [in/out]? -
/// @param  FRET - [in/out]? -
/// @param  gfrag - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
frprmn(
	FArray1DB_float & P,
	int N,
	float FTOL,
	int & ITER,
	float & FRET,
	bool & gfrag
)
{
	float FP, GG, DGG, GAM;
	int const ITMAX = { 1000 };
	float const EPS = { 1.E-10 };
	FArray1D_float G( N );
	FArray1D_float H( N );
	FArray1D_float XI( N );

	FP = func(gfrag,P);
	if ( !gfrag ) return;
	dfunc(P,XI,N);
	for ( int j = 1; j <= N; ++j ) {
		G(j) = -XI(j);
		H(j) = G(j);
		XI(j) = H(j);
	}
	for ( int ITS = 1; ITS <= ITMAX; ++ITS ) {
		ITER = ITS;
		linmin(P,XI,N,FRET,gfrag);
		if ( !gfrag ) return;
		if ( 2.*std::abs(FRET-FP) <= FTOL*(std::abs(FRET)+std::abs(FP)+EPS) ) return;
		FP = func(gfrag,P);
		if ( !gfrag ) return;
		dfunc(P,XI,N);
		GG = 0.;
		DGG = 0.;
		for ( int j = 1; j <= N; ++j ) {
			float const G_j = G(j);
			GG += ( G_j * G_j );
//			DGG += ( XI(j) * XI(j) );
			DGG += (XI(j)+G_j)*XI(j);
		}
		if ( GG == 0. ) return;
		GAM = DGG/GG;
		for ( int j = 1; j <= N; ++j ) {
			G(j) = -XI(j);
			H(j) = G(j)+GAM*H(j);
			XI(j) = H(j);
		}
	}
#ifdef BOINC
	return;
#endif
	error_stop( "FRPR maximum iterations exceeded" );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin frprmn_atol
///
/// @brief
///  jjg 2/18/2: same as frprmn but convergence is measured with an *absolute*
///  tolerance, ATOL
///
/// @detailed
///
/// @param  P - [in/out]? -
/// @param  N - [in/out]? -
/// @param  aTOL - [in/out]? -
/// @param  ITER - [in/out]? -
/// @param  FRET - [in/out]? -
/// @param  gfrag - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
frprmn_atol(
	FArray1DB_float & P,
	int N,
	float ATOL,
	int & ITER,
	float & FRET,
	bool & gfrag
)
{
	float FP, GG, DGG, GAM;
	int const ITMAX = { 1000 };
	//float const EPS = { 1.E-10 };
	FArray1D_float G( N );
	FArray1D_float H( N );
	FArray1D_float XI( N );

	FP = func(gfrag,P);
	if ( !gfrag ) return;
	dfunc(P,XI,N);
	for ( int j = 1; j <= N; ++j ) {
		G(j) = -XI(j);
		H(j) = G(j);
		XI(j) = H(j);
	}
	for ( int ITS = 1; ITS <= ITMAX; ++ITS ) {
		ITER = ITS;
		linmin(P,XI,N,FRET,gfrag);
		if ( !gfrag ) return;
		if ( std::abs(FRET-FP) <= ATOL ) return;
		FP = func(gfrag,P);
		if ( !gfrag ) return;
		dfunc(P,XI,N);
		GG = 0.;
		DGG = 0.;
		for ( int j = 1; j <= N; ++j ) {
			float const G_j = G(j);
			GG += ( G_j * G_j );
//			DGG += ( XI(j) * XI(j) );
			DGG += (XI(j)+G_j)*XI(j);
		}
		if ( GG == 0. ) return;
		GAM = DGG/GG;
		for ( int j = 1; j <= N; ++j ) {
			G(j) = -XI(j);
			H(j) = G(j)+GAM*H(j);
			XI(j) = H(j);
		}
	}
#ifdef BOINC
	return;
#endif
	error_stop( "FRPR maximum iterations exceeded" );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_include_one_residue
///vds: pb put in to minimize only one residue
////////////////////////////////////////////////////////////////////////////////
void
minimize_include_one_residue( const int residue )
{

	using namespace minimize_ns;
	using namespace misc;
	using namespace protein_maps;

	if ( !minimize_exclude_list_init ) minimize_reset_exclude_list();

	for ( int i = 1; i <= total_residue; ++i ) {
		minimize_disallow_residue(i);
	}

	minimize_allow_residue( residue);
}
////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_only_these_residues
////////////////////////////////////////////////////////////////////////////////
void
minimize_only_these_residues(  std::vector <int> min_res )
{

	using namespace minimize_ns;
	using namespace misc;
	using namespace protein_maps;

	if ( !minimize_exclude_list_init ) minimize_reset_exclude_list();

	for ( int i = 1; i <= total_residue; ++i ) {
		minimize_disallow_residue(i);
	}

	int num_in = min_res.size();
  for ( int j = 0; j < num_in; ++j ) {
    int residue = min_res[j];
		minimize_allow_residue ( residue );
  }
}
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
float
get_ediff_cutoff()
{
	static bool init = {false};
	static float ediff_cutoff;

	if (!init) {
		//		realafteroption("ediff_cutoff", 15.0, ediff_cutoff ) ;
		// Increased default value (DB, RD 1-10-07)
		realafteroption("ediff_cutoff", 25.0, ediff_cutoff ) ;
		init = true;
	}

	return ediff_cutoff;
}
////////////////////////////////////////////////////////////////////////////////
bool
get_minimize_exclude_frag_in_wobble()
{
	static bool init = {false};
	static bool minimize_exclude_frag_in_wobble = {false};

	if (!init) {
		//		minimize_exclude_frag_in_wobble = !truefalseoption("no_minimize_exclude_frag_in_wobble");
		// Minimize_exclude_frag is no longer default option (DB,RD 1-10-07)
		minimize_exclude_frag_in_wobble = truefalseoption("minimize_exclude_frag_in_wobble");
		init = true;
	}

	return minimize_exclude_frag_in_wobble;
}
////////////////////////////////////////////////////////////////////////////////
bool
get_minimize_exclude_frag_in_crank()
{
	static bool init = {false};
	static bool minimize_exclude_frag_in_crank = {false};

	if (!init) {
		//		minimize_exclude_frag_in_crank = !truefalseoption("no_minimize_exclude_frag_in_crank");
		// Minimize_exclude_frag is no longer default option (DB,RD 1-10-07)
		minimize_exclude_frag_in_crank = truefalseoption("minimize_exclude_frag_in_crank");
		init = true;
	}

	return minimize_exclude_frag_in_crank;
}
////////////////////////////////////////////////////////////////////////////////
bool
get_minimize_set_local_min()
{
	static bool minimize_set_local_min( true );
	static bool init( false );

	if ( !init ) {
		minimize_set_local_min = !truefalseoption("no_minimize_set_local_min");
		init = true;
	}
	return minimize_set_local_min;
}










/////////////////////////////////////////////////////////////////////////////////
void
ARMIJO_nonmonotone(
	float & XX,
	float & FP,
	float & DRV,
	f_Function_bf func,    // <--- This is NOT func(gfrag, P) but is actually a pointer to F1DIM
	bool & gfrag,
	int & NF,
	float & XMIN,
	float & FRET
)
// INPUT PARAMETERS: XX,FP,DRV,func,gfrag,NF
// OUTPUT PARAMETERS: NF,XMIN,FRET
//
// Given a function FUNC, and initial stepsize XX
// such that 0 < XX and FUNC has negative derivative DRV at 0,
// this routine returns a stepsize XX at least as good as that
// given by Armijo rule.
// Reference:  D.P. Bertsekas, Nonlinear Programming, 2nd ed, 1999, page 29.
{
	float U, FU, BX;
	float const FACTOR = { 0.5 };
	float const SIGMA = { 0.1 };
	float const SIGMA2 = { 0.8 };

	FRET = func(gfrag,XX);
	NF += 1;
	if ( !gfrag ) return;
    XMIN = XX;

    if (FRET < FP+XX*SIGMA2*DRV ) {
      U = XX/FACTOR; //std::cout << "That one: " << U << "\n";
      FU = func(gfrag,U);
	  	NF += 1;
      if (FU < FRET) {
        XMIN = U;
        FRET = FU;
      }
      return;
    }
    BX=XX;
L2:
		if (FRET > FP+XX*SIGMA*DRV) {  // Abort if function value is unlikely to improve.
      if (XX <= BX*1e-5 || (XX < 1e-8 && FRET >= FP)) {
					U=(FRET-FP)/XX;
          XMIN = 0.0;
          FRET = FP;
          return;
      }
      XX *= FACTOR*FACTOR;			// faster step decrease
			//std::cout << "This one: " << XX << "\n";
      FRET = func(gfrag,XX); //std::cout << "This one \n";
	  	NF += 1;
      goto L2;
    }
    XMIN = XX;
		if (XX < 0.0) {  // Parabola interpolate between 0 and XX for refinement
      U = -DRV*XX*XX/(2*(FRET-FP-DRV*XX));
      if (U > BX*1e-3 && U < BX) {
				//std::cout << "The other one: " << U << "\n";
        FU = func(gfrag,U); //std::cout << "Other one \n";
	  		NF += 1;
        if (FU < FRET) {
          XMIN = U;
          FRET = FU;
        }
      }
		}
    return;
}


/////////////////////////////////////////////////////////////////////////////////
void
linmin_armijo_nonmonotone(
		FArray1DB_float & P,
		FArray1DB_float & XI,
		int & NF,
		float & DRV,
		float & FP,
		int & N,
		bool & gfrag,
		float & XMIN,
		float & FRET
		)
// INPUT PARAMETERS: DRV,FP,N,gfrag,P,XI,NF,XMIN
// OUTPUT PARAMETERS: P,XI,NF,XMIN,FRET
{
	using namespace minimize_ns::F1COM;
	using namespace minimize_ns::func_switch;
	using namespace minimize_ns::linmin_param;
	using namespace runlevel_ns;

	float XX;
	float const FACTOR = { 0.5 };
	float LIMITXX = 2.0;

	//car check local parameters
	if ( N > int(PCOM.size1()) ) {
		std::cout << "increase NMAX in namespace F1COM" << std::endl;
		std::cout << "NMAX: " << NMAX << " N: " << N << std::endl;
		PCOM.dimension( N );
		XICOM.dimension( N );
	}

	float derivmax = 0.0;
	for ( int i = 1; i <= N; ++i ) {
		if ( std::abs(XI(i)) > std::abs(derivmax) ) derivmax = XI(i);
	}
	if ( runlevel >= gush) std::cout << "derivmax," << SS( derivmax ) << std::endl;
	if ( std::abs(derivmax) > .0001 ) goto L10;
	FRET = func(gfrag,P); // deriv = 0, return value
	return;
L10:

	// set (global!) current position and search direction - the function F1DIM (!) uses these
	NCOM = N;
	for ( int j = 1; j <= N; ++j ) {
		PCOM(j) = P(j);
		XICOM(j) = XI(j);
	}

	//		AX = ax_init;
	//		XX = xx_init; // initial range for bracketing

	XX = XMIN/FACTOR;
	if ( XX > LIMITXX ) XX = LIMITXX;   //limit to 2.0 seems to work a tiny bit better than limiting it to 1.0

	ARMIJO_nonmonotone(XX,FP,DRV,F1DIM,gfrag,NF,XMIN,FRET);

	if ( !gfrag ) std::cout << "linmin gfrag tripped" << std::endl;
	if ( !gfrag ) return;

	for ( int j = 1; j <= N; ++j ) {
		XI(j) *= XMIN;
		P(j) += XI(j);
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin dfpmin_nonmonotone
///
/// @brief
///
/// @detailed
///
/// @param  P - [in/out]? -
/// @param  N - [in/out]? -
/// @param  NP - [in/out]? -
/// @param  FTOL - [in/out]? -
/// @param  ITER - [in/out]? -
/// @param  FRET - [in/out]? -
/// @param  gfrag - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
dfpmin_armijo_nonmonotone(
	FArray1DB_float & P,
	int N,
	float FTOL,
	int & ITER,
	float & FRET,
	bool & gfrag
)
{
	using namespace runlevel_ns;

	float FP, FAC, FAE, FAD, FAF, DRV, DRVNEW, GMAX, GNORM, XMIN, XINORM;
	int NF, HOPT, NFP;
	// NM is the memory size of non-monotone LS
	int const NM = { 3 };
	float FPMAX;
	FArray1D_float FPM( NM );
	int const ITMAX = { 200 };
	float const EPS = { 1.E-5 };
	FArray2D_float HESSIN( N, N );
	FArray1D_float XI( N );
	FArray1D_float G( N );
	FArray1D_float DG( N );
	FArray1D_float HDG( N );
	PROF_START( prof::DFPMIN );
	if ( runlevel >= gush ) std::cout << "dfpmin" << std::endl;

	// When inexact line search is used, HESSIN need not remain positive definite, so
	// additional safeguard must be added to ensure XI is a desc. direction (or inexact
	// line search would fail).  Two options for safeguards are implemented below:
	//  	HOPT = 1  resets HESSIN to a multiple of identity when XI is not a desc. direction.
	//	HOPT = 2  leaves HESSIN unchanged if stepsize XMIN fails Wolfe's condition
	//		    for ensuring new HESSIN to be positive definite.
	HOPT = 2;

	FP = func(gfrag,P);
	NF = 1;  		// number of func evaluations
	NFP = 1;
	FPM(NFP) = FP;
	if ( !gfrag ) return;
	dfunc(P,G,N);
	for ( int i = 1; i <= N; ++i ) {
		for ( int j = 1; j <= N; ++j ) {
			HESSIN(i,j) = 0.0;
		}
		HESSIN(i,i) = 1.0;
		XI(i) = -G(i);
	}
	//XMIN = 1.0;   // initial trial stepsize
	XMIN = 0.005;   // more appropriate for a perturbed structure (which clashes initially) mtyka

	//start_timer("dfpmin");
	for ( ITER = 1; ITER <= ITMAX; ++ITER ) {

		DRV=0.0;
		GMAX=0.0;
		GNORM=0.0;
		for ( int i = 1; i <= N; ++i ) {
			DRV += XI(i)*G(i);
			GNORM += G(i)*G(i);
			if (std::abs(G(i))>GMAX) {
				GMAX=std::abs(G(i));
			}
		}
		GNORM = std::sqrt(GNORM);

		// Compute maximum of function values in memory.
		FPMAX = FPM(1);
		for (int i = 2; i <= NFP; ++i ) {
			if ( FPMAX < FPM(i) ) {
				FPMAX = FPM(i);
			}
		}

		// P is returned as new pt, and XI is returned as the change.
		NF = 1;
		linmin_armijo_nonmonotone(P,XI,NF,DRV,FPMAX,N,gfrag,XMIN,FRET);

		if ( !gfrag ) return;
		if ( 2.0*std::abs(FRET-FP) <= FTOL*(std::abs(FRET)+std::abs(FP)+EPS) ) {
			//$$$   std::cout << "dfpmin called linmin " << linmin_count << " times" << std::endl;
			if (GMAX<=1.0) {


				PROF_STOP( prof::DFPMIN );
				return;
			}else {
				if (std::abs(FRET-FP)<=EPS) {
					XINORM = 0.0;
					for ( int i = 1; i <= N; ++i ) {
						XINORM += XI(i)*XI(i);
					}
					if ( DRV < -1e-3*GNORM*XINORM ) {


						PROF_STOP( prof::DFPMIN );
						return;
					}
					// No convergence yet. Reinitialize HESSIN to a diagonal matrix & update direction XI.
					// This requires G to be correctly the gradient of the function.

					DRV=0.0;
					for ( int i = 1; i <= N; ++i ) {
						for ( int j = 1; j < i; ++j ) {
							HESSIN(i,j) = 0.0;
						}
						for ( int j = i+1; j <= N; ++j ) {
							HESSIN(i,j) = 0.0;
						}
						if ( HESSIN(i,i) < 0.01 ) HESSIN(i,i) = 0.01;
						XI(i) = -HESSIN(i,i)*G(i);
						DRV += XI(i)*G(i);
					}
					linmin_armijo_nonmonotone(P,XI,NF,DRV,FPMAX,N,gfrag,XMIN,FRET);


					if (std::abs(FRET-FP)<=EPS) {
						PROF_STOP( prof::DFPMIN );
						return;
					}
				}
			}
		}
		FP = FRET;

		// Update function values in memory.
		if ( NFP < NM ) {
			NFP += 1;
		} else {
			for ( int i = 1; i < NFP; ++i ) {
				FPM(i) = FPM(i+1);
			}
		}
		FPM(NFP) = FP;

		for ( int i = 1; i <= N; ++i ) {
			DG(i) = G(i);
		}
		FRET = func(gfrag,P);
		if ( !gfrag ) return;
		dfunc(P,G,N);
		DRV=0.0;						//needed if HOPT = 2
		DRVNEW=0.0;						//needed if HOPT = 2
		for ( int i = 1; i <= N; ++i ) {
			DRV += XI(i)*DG(i);			//needed if HOPT = 2
			DRVNEW += XI(i)*G(i);			//needed if HOPT = 2
			DG(i) = G(i)-DG(i);
		}

		//if ( XMIN = 0.0 ) std::cout << " XMIN = 0.0! " << std::endl;	//diagnostic

		if ( HOPT == 1 || DRVNEW > 0.95*DRV ) {			//needed if HOPT = 2

			for ( int i = 1; i <= N; ++i ) {
				HDG(i) = 0.0;
				for ( int j = 1; j <= N; ++j ) {
					HDG(i) += HESSIN(i,j)*DG(j);
				}
			}
			FAC = 0.0;
			FAE = 0.0;
			FAF = 0.0;
			for ( int i = 1; i <= N; ++i ) {
				FAC += DG(i)*XI(i);
				FAE += DG(i)*HDG(i);
				FAF += DG(i)*DG(i);
			}
			FAF = FAC/FAF;
			FAC = 1.0/FAC;
			FAD = 1.0/FAE;
			for ( int i = 1; i <= N; ++i ) {
				DG(i) = FAC*XI(i) - FAD*HDG(i);
			}
			for ( int i = 1; i <= N; ++i ) {
				for ( int j = 1; j <= N; ++j ) {
					HESSIN(i,j) += FAC*XI(i)*XI(j) - FAD*HDG(i)*HDG(j) + FAE*DG(i)*DG(j);
				}
			}

		}									//needed if HOPT = 2

		for ( int i = 1; i <= N; ++i ) {
			XI(i) = 0.0;
			for ( int j = 1; j <= N; ++j ) {
				XI(i) -= HESSIN(i,j)*G(j);
			}
		}

		if ( HOPT == 1 ) {

			DRVNEW=0.0;
			for ( int i = 1; i <= N; ++i ) {
				DRVNEW += XI(i)*G(i);
			}
			// If direc. deriv >0, reset the Hessian inverse estimate
			if (DRVNEW>-EPS) {

				if (FAF<0.01) FAF=0.01;
				for ( int i = 1; i <= N; ++i ) {
					for ( int j = 1; j <= N; ++j ) {
						HESSIN(i,j) = 0;
					}
					HESSIN(i,i) = FAF;
					XI(i) = -FAF*G(i);
				}
			}

		}

	}
#ifdef BOINC
	return;
#endif
	std::cout << "Exit dfpmin!  #iter reaches limit of " << ITMAX << std::endl;
	return;
}













