// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 23432 $
//  $Date: 2008-06-24 16:25:52 +0300 (Tue, 24 Jun 2008) $
//  $Author: yab $

// Rosetta Headers
#include "output_decoy.h"
#include "after_opts.h"
#include "antibody_modeling.h"
#include "barcode_stats.h"
#include "decoystats.h"
#include "design.h"
#include "disulfides.h"
#include "docking.h"
#include "docking_ns.h"
#include "docking_scoring.h"
#include "files_paths.h"
#include "filters.h"
#include "force_barcode.h"
#include "fullatom.h"
#include "initialize.h"
#include "input_pdb.h"
#include "knots.h"
#include "loops.h"
#include "make_pdb.h"
#include "misc.h"
#include "monte_carlo.h" // yab: misc removal
#include "nblist.h"
#include "orient_rms.h"
#include "pack.h"
#include "param.h"
#include "param_aa.h"
#include "pdb.h"
#include "pKa_mode.h"
#include "pose.h"
#include "pose_io.h"
#include "random_numbers.h"
#include "recover.h"
#include "runlevel.h"
#include "saxs_model.h"
#include "score.h"
#include "score_ns.h"
#include "sheet_filter.h"
#include "silent_input.h"
#include "start.h"
#include "status.h"
#include "taboo_search.h"
#include "relax.h" //for relax options

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/string.functions.hh>
#include <ObjexxFCL/Time_Date.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>
#include <utility/io/ocstream.hh>
#include <utility/file/file_sys_util.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <fstream>

// BOINC
#ifdef BOINC
#ifdef _WIN32
#include "boinc_win.h"
#endif
#include "boinc_api.h"
#include "util.h"
#include "boinc_rosetta_util.h"
#endif

// Namespaces
namespace decoy_number_block {
	int global_number = { 0 };
}
namespace makepdb_decoy {
	int makepdb_number = { 0 };
	std::string decoy_cnt_filename = "rosetta_decoy_cnt.txt";
}
namespace makepdb_file {
	using namespace files_paths;
	std::string makepdb_name( "initialize.pdb" );
}
namespace makepdb_time {
	double makepdb_start_minutes = { -999.9 };
}
namespace scorefile_init {
	bool init = { false };
}


// Member functions for cat_pdb_writer

cat_pdb_writer::cat_pdb_writer() {
  using namespace files_paths;

  if ( concat_outpdbs ) {
	  curr_filename = pdb_out_path + cat_outpdb_filename;
		if ( files_paths::overwrite_pdbs ) {
			pdb_out_stream.open( curr_filename ); // open the outfile stream for writing
		} else {
			pdb_out_stream.open_append( curr_filename ); // open the outfile stream for writing
		}
  }
	else {
	  curr_filename = "";
    cat_list_filename = "";
	}
}

cat_pdb_writer::~cat_pdb_writer() {
	if ( pdb_out_stream ) { // flush the stream
		pdb_out_stream.close();
		pdb_out_stream.clear();
	}
	if( !files_paths::single_silent_writer )
	  remove( (files_paths::cat_list_filename).c_str() );
}

void cat_pdb_writer::set_filename( std::string const & fullname ) {
	if ( fullname != curr_filename ) { // opening a new filename
		if ( pdb_out_stream ) { // flush out the previous buffer
			pdb_out_stream.close();
			pdb_out_stream.clear();
		}
	  curr_filename = fullname; // open a new stream
		if ( files_paths::overwrite_pdbs ) {
			pdb_out_stream.open( curr_filename );
		} else {
			pdb_out_stream.open_append( curr_filename );
		}
	  if ( !pdb_out_stream ) {
		  std::cout << "Open failed for file: " << pdb_out_stream.filename() << std::endl;
		  utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	  }
	}
	std::cout << "writing pdb file: " << fullname << std::endl;
}

// returns whether a filename has been started in the concatenated list
bool cat_filename_exists( std::string const & fullname ) {

  utility::io::izstream list;
	list.open( files_paths::cat_list_filename );

	std::string line;
	while( getline( list, line ) ) {
	  if( line == fullname )
		  return true;
	}
	return false;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_decoy
///
/// @brief
///car handle all functions required to end a single decoy simulation
///car score final structure; output score file, pdbfiles; close status files
///car parallel to initialize_decoy
///
/// @detailed
///car option dependencies:
///car   output_coord    should pdb files be written?
///car   use_scorefile   should scorefile be written?
///car   use_filter      should a particular filter be applied?
///car   output_fa       is output fullatom or centroid?
///car   output_all      output structures that fail filters?
///
/// @param  accepted - [in/out]? - true if structure passes all cutoff criteria
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_decoy( bool & accepted /* true if structure passes all cutoff criteria */ )
{
	using namespace files_paths;
	using namespace param;
	using namespace runlevel_ns;

//car local
	int istruct;
	bool exist;
	static int i_alt = { 1 };
	std::string filename;
	std::string fullname;
	std::string tag("pass");

	accepted = true; // initialize

#ifdef BOINC
	boinc_begin_critical_section();
#endif

// activities that should happen to reset for next structure
	set_use_nblist(false); // make sure the nblist is off
	if ( get_loop_flag() ) {
		score_set_loop_weight(0.0);
		//chu ensure the decoy out of pose_loop modeling protocol will not be refolded
		set_allow_insert_by_looplength(MAX_RES(),!get_pose_loop_flag(),exist);
	}

//car first check if we need to evaluate final structure:
	if ( !output_coord && !use_scorefile ) {
		accepted = true;
		goto L99;
	}

//car turn on the flag if necessary to repack and score
	select_rotamer_set( "large" );
	set_fullatom_flag(output_fa);
	score_set_default_function(output_fa);
	score_set_evaluate_all_terms( true );
	score_enable_rotamer_trials( false );
	if( !files_paths::antibody_modeler)
		recover_LOW(scorefxn);

	if ( get_decoystats_flag() && output_fa) decoystats_store_decoy();

	calc_docking_interf_energy( get_flexbb_docking_flag() );

	if ( output_topology_info() ) update_topology_info();

//current structure
	calc_rms();

//car check appropriate cutoffs: rg,co,rms,knots,others...
//car requires that current structure by evaluated
//car     ie call scorefxn with eval_all_terms true
//car     ie call calc_rms
	if ( !disable_output_decoy_filters ) apply_filters( accepted, tag );
	if ( has_prefix( tag, "pass" ) ) tag = "output_decoy";

//car write out pdb/score info
//car note that rms_err which will
//car be overwritten in make_pdb and is therefore
//car recomputed in scorefile_output
//car scores are NOT recomputed in makepdb and are therefore current

//bqian: if want to accumulate taboo list for each decoy,
//bqian: update active_taboo_map here.
	if ( get_taboo_mode_exist() ) {
		if ( accepted || output_all ) output_taboo( accepted );
		conclude_taboo_decoy();
	}

	if ( runlevel == silent ) {
		if ( accepted || output_all ) output_compressed(accepted,output_fa);
		goto L99;
	}

	if ( accepted ) {
		if ( output_coord ) {
			retrieve_decoy_name( filename );
			fullname = pdb_out_path + filename;

			make_pdb( get_cat_pdb_stream( fullname ), output_fa);

			remove( (fullname+".in_progress").c_str() );

		} else {
			filename = "no_pdbfile";
		}
		scorefile_output( filename, tag, output_fa );
		reset_decoy_pdb();
	} else {
		if ( output_all ) {
      utility::io::izstream pdb_out_checkstream;
			bool looking = true;
			while ( looking ) {
				filename = code + protein_name + lead_zero_string_of( i_alt, NUMBER_MAX_LENGTH ) + "fail.pdb";
				if (output_pdb_gz) filename += ".gz";
				fullname = pdb_out_path + filename;
				pdb_out_checkstream.open( fullname, std::ios_base::in );
				if ( pdb_out_checkstream ) { // File exists: Increment name
					++i_alt;
				} else {
					looking = false;
				}
				pdb_out_checkstream.close();
				pdb_out_checkstream.clear();
			}

			make_pdb( get_cat_pdb_stream( fullname ),output_fa);

			remove( (fullname+".in_progress").c_str() );

		} else {
			filename = "no_pdbfile_fail";
		}

		scorefile_output( filename, tag, output_fa );
		retrieve_decoy_number(istruct);
		std::cout << "REPEAT STRUCTURE: " << istruct << std::endl;
	}

	//reset for next structure (decoy or new starting structure)
L99:

	output_saxs_profile( fullname );

	close_status_file();
	set_fullatom_flag( false );
	select_rotamer_set( "default" );
	score_enable_rotamer_trials( true );
	score_set_evaluate_all_terms( false );

#ifdef BOINC
	boinc_end_critical_section();
#endif
}





////////////////////////////////////////////////////////////////////////////////
/// @begin output_decoy
///
/// @brief
/// mtyka addition - analog of output_decoy but this one writes to a memory buffer
/// defined by the string stream output_strstream. THis is used in the distributed CSA
/// mode of rosetta
///
/// @detailed
///
/// @param  accepted - [in/out]? - true if structure passes all cutoff criteria
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
output_decoy_mem(std::stringstream &output_strstream,
                 bool & accepted /* true if structure passes all cutoff criteria */ )
{
	using namespace files_paths;
	using namespace param;
	using namespace runlevel_ns;
//  std::ostream *tmpstream = &output_strstream;

//car local
	bool exist;
	std::string filename;
	std::string fullname;
	std::string tag;
	std::string number;

  utility::io::ocstream tmpstream(output_strstream);

// activities that should happen to reset for next structure
	set_use_nblist(false); // make sure the nblist is off
	if ( get_loop_flag() ) {
		score_set_loop_weight(0.0);
		set_allow_insert_by_looplength(MAX_RES(),true,exist);
	}

//car first check if we need to evaluate final structure:
	if ( !output_coord && !use_scorefile ) {
		accepted = true;
		goto L99;
	}

//car turn on the flag if necessary to repack and score
	set_fullatom_flag(output_fa);
	score_set_default_function(output_fa);
	score_set_evaluate_all_terms( true );
	select_rotamer_set( "large" );
	score_enable_rotamer_trials( false );
	recover_LOW(scorefxn);

	if ( get_decoystats_flag() && output_fa) decoystats_store_decoy();

//current structure
	calc_rms();

//car check appropriate cutoffs: rg,co,rms,knots,others...
//car requires that current structure by evaluated
//car     ie call scorefxn with eval_all_terms true
//car     ie call calc_rms
	if ( !disable_output_decoy_filters ) apply_filters( accepted, tag );
	if ( has_prefix( tag, "pass" ) ) tag = "output_decoy";

  // output structure whether accepted or not - server will decide what to do
  make_pdb_orstream(tmpstream,output_fa);
  reset_decoy_pdb();

	//reset for next structure (decoy or new starting structure)
L99:

	close_status_file();
	set_fullatom_flag( false );
	select_rotamer_set( "default" );
	score_enable_rotamer_trials( true );
	score_set_evaluate_all_terms( false );

}








////////////////////////////////////////////////////////////////////////////////
/// @begin open_decoy_pdb
///
/// @brief
///car opens pdb file for final decoy output via output_decoy
///car DON'T USE THIS ROUTINE TO OPEN OTHER PDB FILES!  it will _always_ fail
///
/// @detailed
///
/// @param  ns - [in/out]? - number of the current starting structure
/// @param  no - [in/out]? - number of the current output structure for this
///        start structure
/// @param  fail - [in/out]? - true if pdb exists and no file opened
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
open_decoy_pdb(
	std::string const & mode,
	int const ns, // number of the current starting structure
	int const no, // number of the current output structure for this start structure
	bool & fail
)
{
	using namespace files_paths;
	using namespace makepdb_decoy;
	using namespace makepdb_file;
	using namespace runlevel_ns;

	std::string filename;
	std::string fullname;
	bool file_exists;

	fail = false;
	if ( !output_coord ) return;

	fail = true;
	check_decoy_exists( mode, ns, no, file_exists, filename );

	fullname = pdb_out_path + filename;

	utility::io::ozstream pdb_out_xstream;

	if ( overwrite_pdbs || touch_before_reading ) {
		pdb_out_xstream.open( fullname+".in_progress" );
	} else {
		if ( file_exists ) return;
		pdb_out_xstream.open( fullname+".in_progress" );
	}

	if ( !pdb_out_xstream ) {
		pdb_out_xstream.close();
		pdb_out_xstream.clear();
		return;
	}

	fail = false;
	pdb_out_xstream.close();
	pdb_out_xstream.clear();
	std::cout << "NEXT STRUCTURE: " << fullname << std::endl;
	makepdb_name = filename;
	makepdb_number = no;
	reset_timer();

//car increment counter for silent format
	if ( runlevel == silent ) {
		fullname = pdb_out_path + code + protein_name + ".last_pdb";
		pdb_out_xstream.open( fullname );
		if ( !pdb_out_xstream ) {
			pdb_out_xstream.close();
			pdb_out_xstream.clear();
			return;
		}
		pdb_out_xstream << SS( ns ) << '\n'; //std::endl;
		pdb_out_xstream << SS( no ) << '\n'; //std::endl;
		pdb_out_xstream.close();
		pdb_out_xstream.clear();
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin check_decoy_exists
///
/// @brief
///car opens pdb file for final decoy output via output_decoy
///car DON'T USE THIS ROUTINE TO OPEN OTHER PDB FILES!  it will _always_ fail
///
/// @detailed
///bs modified to allow (1) using list of start structures while running silent_mode.
///bs                   (2) multiple outputs for each start while running silent_mode.
///bs works in conjunction with function reset_last_pdb.
///
///bs start structures can be in a silent file themselves, or not.
///bs if start structures are in a silent_file, command-line should have:
///bs -refold -silent_input -s <silent_inputfile> -l <tagsfile>
///bs if start structures are in normal pdb files you only need:
///bs -s <pdbfile> or -l <pdbfilelist>
///
/// @param  ns - [in/out]? - number of the current starting structure
/// @param  no - [in/out]? -  number of the current output structure for this
///        start structure
/// @param  exists - [in/out]? -
/// @param  filename - [in/out]? - true if pdb exists and no file opened
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
check_decoy_exists(
	std::string const & mode,
	int const ns, // number of the current starting structure
	int const no, // number of the current output structure for this start structure
	bool & exists,
	std::string & filename
)
{
	using namespace design;
	using namespace files_paths;
	using namespace runlevel_ns;

//car local
	int last_start; // most recent start structure
	int last_pdb; // most recent pdb built for most recent start struct

	exists = false;

	set_decoy_number( no );

#ifdef BOINC
	if ( boinc_params::pct_complete >= 1 ) {
		exists = true;
		return;
	}
#endif

	// jk most design submodes don't put series_code in output pdbs
	std::string fname_code;
	if ((mode == "design") && ! (mvbb || design_loops)) {
		fname_code="";
	} else {
		fname_code=code;
	}

	if ( runlevel == silent ) {
		std::string const fullname( pdb_out_path + fname_code + protein_name + ".last_pdb" );
		utility::io::izstream pdb_stream( fullname );
		if ( !pdb_stream ) { // filename.last_pdb doesn't exist yet
			last_start = 0;
			last_pdb = 0;
		} else {
			pdb_stream >> last_start >> skip;
			pdb_stream >> last_pdb >> skip;
			int loop_counter(0);
			while ( pdb_stream.fail() ) {
				++loop_counter;
				if ( loop_counter > 120 ) {
					last_start = ns;
					last_pdb = no-1;
					break;
				}
				std::cout << "WARNING: check_decoy_exists: can not read from " <<
					fullname.c_str() << std::endl;
				pdb_stream.clear();
				pdb_stream.close();
				utility::sys_sleep(1);
				pdb_stream.open( fullname.c_str() );
				pdb_stream >> last_start >> skip;
				pdb_stream >> last_pdb >> skip;

				std::cout << "WARNING: ns no last_start last_pdb " << ns << ' ' <<
					no << ' ' << last_start << ' ' << last_pdb<< std::endl;
			}
#ifdef BOINC
			--last_pdb; // allows boinc jobs to start at the correct decoy if the run failed previously
                // otherwise the last pdb would get skipped.
#endif
		}
		pdb_stream.close();
		pdb_stream.clear();

		if ( ns < last_start ) {
			exists = true;
			return;
		} else if ( ns == last_start && no <= last_pdb ) {
			exists = true;
			return;
#ifdef BOINC
		} else if ( ns == last_start && no - last_pdb > 2 ) {
#else
		} else if ( ns == last_start && no - last_pdb > 1 ) {
#endif
			std::cout << "WARNING: check_decoy_exists: unexpected " <<
			 "decoy number: start#,decoy#,lastdecoy# " << ns << ' ' << last_start <<
			 ' ' << no << ' ' << last_pdb << std::endl;
		} else if ( ns - last_start > 1 ) {
			std::cout << "WARNING: check_decoy_exists: unexpected " <<
			 "start number: start#, laststart# " << ns << ' ' << last_start << std::endl;
		}
	}

	std::string const number( lead_zero_string_of( no, NUMBER_MAX_LENGTH ) );
	if ( require_start ) {
 		filename = output_file;

		if ( pKa_mode::get_pKa_flag() ) {
			pKa_mode::output_pdb_name(filename,no);
		}
 		filename += '_' + number + ".pdb";
		if ( output_pdb_gz ) filename += ".gz";
		if ( query_defined )
			filename.insert( ( filename.find_last_of( '/' ) + 1 ), fname_code );

	} else {
		filename = fname_code + protein_name + number + ".pdb";
		if ( pKa_mode::get_pKa_flag() ) {
			pKa_mode::output_pdb_name(filename,no);
		}
		if ( output_pdb_gz ) filename += ".gz";
	}

	std::string const fullname( pdb_out_path + filename );

#ifdef BOINC
	exists = ( utility::file::file_size( fullname ) > 0 );
	if ( ! exists ) {
		exists = ( utility::file::file_size( fullname+".in_progress" ) > 0 );
	}
#else
	utility::io::izstream pdb_stream( fullname );
  exists = pdb_stream.good() ||
	  ( files_paths::concat_outpdbs && cat_filename_exists( fullname ) );
	pdb_stream.close();
	pdb_stream.clear();
	if ( ! exists ) {
		utility::io::izstream tmp_stream( fullname+".in_progress" );
		exists = tmp_stream.good();
		exists = tmp_stream.good() ||
			( files_paths::concat_outpdbs && cat_filename_exists( fullname+".in_progress" ) );
		tmp_stream.close();
		tmp_stream.clear();
	}
#endif

}

////////////////////////////////////////////////////////////////////////////////
/// @begin retrieve_decoy_name
///
/// @brief
///car returns the filename associated with the current decoy being built
///
/// @detailed
///
/// @param  filename - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
retrieve_decoy_name( std::string & filename )
{ //car returns the filename associated with the current decoy being built
	using namespace makepdb_file;

	filename = makepdb_name;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin retrieve_decoy_number
///
/// @brief
///
/// @detailed
///
/// @param  number - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
retrieve_decoy_number( int & number )
{
	using namespace makepdb_decoy;

	number = makepdb_number;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin reset_timer
///
/// @brief  starts timing
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks  intended to time the operations per decoy;  DO NOT RESET THE
///        TIMER DURING A RUN. Should be reset only at initialization.
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
reset_timer()
{
	makepdb_time::makepdb_start_minutes = minutes();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin retrieve_timer_start
///
/// @brief
///
/// @detailed
///
/// @param[out]   time - out - time stamp at which timer was reset
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
double
retrieve_timer_start()
{
	return makepdb_time::makepdb_start_minutes;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin reset_decoy_pdb
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
reset_decoy_pdb()
{
	using namespace makepdb_decoy;
	using namespace makepdb_file;

	makepdb_number = 0;
	makepdb_name = "initialize.pdb";
}

////////////////////////////////////////////////////////////////////////////////
/// @begin scorefile_output
///
/// @brief
///
/// @detailed
///
/// @param  filename_in - [in/out]? -
/// @param  tag - [in/out]? -
/// @param  fullatom - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
scorefile_output(
	std::string const & filename_in,
	std::string const & tag,
	bool fullatom
)
{
	using namespace files_paths;

//car  option dependencies:
//car   use_scorefile     should scorefile be written?

	if ( !use_scorefile ) return;

	if ( !scorefile_init::init ) {
		score_file = '-';
		if ( query_defined ) score_file = code + protein_name;
		stringafteroption( "scorefile", score_file, score_file );
		if ( score_file == "-" ) {
			std::cout << "WARNING:: unable to make score file: " <<
			 " no filename defined" << std::endl;
			std::cout << "either specify series_code and protein_name" << std::endl;
			std::cout << "or use -scorefile option" << std::endl;
			use_scorefile = false;
			return;
		} else {
			std::cout << "score file: " << score_path << score_file << std::endl;
		}
		scorefile_init::init = true;
	}

	if ( !use_scorefile ) return;

//car evaluate rms, time, (and rg and co in case not in scorefxn)
//car native cannot evaluate rms; rg and co evaluated by scorefxn with
//car evaluate_all_terms true
	update_scorefile_info();

// determine filename and open file
	std::string outfile;
	if ( fullatom ) {
		outfile = score_path + score_file + ".fasc";
	} else {
		outfile = score_path + score_file + ".sc";
	}

	std::fstream score_teststream( outfile.c_str(), std::ios_base::in|std::ios_base::out );
	utility::io::izstream score_istream;
	utility::io::ozstream score_stream;

	int fml; // actual max-length of filename
	if ( scorefile_output_full_filename ) {
		//retrieve_filename_max_length(fml);
		//fml = std::max( 10, std::min( 99, fml ) );
		fml = 99;
	} else {
		fml = 21;
	}

	if ( !score_teststream ) { // file is new so write out headers
    score_teststream.close();
		score_teststream.clear();

		score_stream.open( outfile );
		if ( !score_stream ) {
			std::cout << "ABORT: cannot create or find scorefile" << std::endl;
			std::cout << "scorefile: " << outfile << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		score_stream << sized( "filename", fml );
		score_header_output( score_stream, fullatom ); score_stream << "\n";
	}

	std::string filename;
	strip_path( filename_in, filename );
	score_stream.close(); //Objexx: close/clear/open is safe but slow work-around
	score_stream.clear();
	score_istream.open( outfile );
	bool already_in_scorefile = false;

	if (!skip_scorefile_check) scorefile_scroll_and_detect( score_istream, filename, tag, already_in_scorefile );

	if ( !already_in_scorefile ) {
		//Objexx: Replaced by below: score_stream.seekp( std::ios_base::end ); // Set write position to end of file
		score_istream.close(); //Objexx: close/clear/open is safe but slow work-around
		score_istream.clear();
		score_stream.open_append( outfile );
		if( scorefile_output_full_filename ) // use the full path
			score_stream << sized( filename_in, fml );
		else
			score_stream << sized( filename, fml );
		score_output( score_stream, tag, fullatom );
		score_stream << "\n";
//bs Put all disulfide params & score info into the disulf log file.
//bs This includes info for each individual disulfide that is
//bs not included in the scorefile.
		disulfides::BOUNDARY::output_disulf_stats_to_log( filename, tag, fullatom );
	}

	score_stream.close();
	score_stream.clear();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin strip_path
///
/// @brief
/// strip the paths off the filename and force it to the desired length
///
/// @detailed
///
/// @param  filename_in - [in/out]? -
/// @param  filename_out - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
strip_path(
	std::string const & filename_in,
	std::string & filename_out
)
{
   filename_out = filename_in.substr( filename_in.find_last_of( '/' ) + 1 );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin scorefile_scroll_and_detect
///
/// @brief
///jg   scroll through the opened scorefile and detect whether the given
///jg   filename and tag are already there.  returns 'found_it'.
///
/// @detailed
///
/// @param  iunit - [in/out]? -
/// @param  filename - [in/out]? -
/// @param  tag - [in/out]? -
/// @param  found_it - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
scorefile_scroll_and_detect(
	utility::io::izstream & iunit,
	std::string const & filename,
	std::string const & tag,
	bool & found_it
)
{

//car local
	std::string line;
	std::string::size_type tag_location = 0;
	int bad_header_count = 0;
//------------------------------------------------------------------------------
	found_it = false;

//     get the description column from the header line
L110:
	iunit.seek_beg();
	iunit >> bite( line ) >> skip;
	if ( iunit.eof() ) {
		goto L105;
	} else if ( iunit.fail() ) {
		iunit.clear();
		iunit >> skip;
		goto L110;
	}

	tag_location = line.find( "description" );
L105:
	if ( tag_location == std::string::npos ) {
		//// ctsa - race condition has allowed this process to reach the file
		//// before the header was written, try a few more times
		////
		if ( bad_header_count > 100 ) {
			std::cout << "ABORT: problem reading scorefile" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		++bad_header_count;
		goto L110;
	}


//     line by line, look for the given filename and tag
	std::string::size_type tag_length = len_trim( tag );
	std::string::size_type filename_length = filename.length();
	while ( !found_it ) {   // scroll to end of score file
		iunit >> bite( line ) >> skip;
		if ( iunit.eof() ) {
			goto L111;
		} else if ( iunit.fail() ) {
			iunit.clear();
			iunit >> skip;
		} else {
			ObjexxFCL::strip_whitespace( line );
			if ( ( filename != "no_pdbfile" )
			 && ( filename == line.substr( 0, filename_length ) )
			 && ( tag == line.substr( tag_location, tag_length ) ) )
			 found_it = true;
		}
	}
L111:; // escape to here when the end-of-file is reached
	iunit.clear();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin scorefile_entry_present
///
/// @brief
///     a utility function to open the scorefile and check whether a particular
///     entry is already there.
///
/// @detailed
///
/// @param  filename_in - [in/out]? -
/// @param  tag - [in/out]? -
/// @param  fullatom - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
scorefile_entry_present(
	std::string const & filename_in,
	std::string const & tag,
	bool const fullatom
)
{
	using namespace files_paths;

	if ( !use_scorefile ) return false;
	if ( !scorefile_init::init ) return false;

	bool scorefile_entry_present = false; // Return value

//   determine filename and open file
	std::string outfile( score_path + score_file );
	if ( fullatom ) {
		outfile += ".fasc";
	} else {
		outfile += ".sc";
	}

	utility::io::izstream score_stream( outfile );

	if ( !score_stream ) return false; // scorefile does not exist

	std::string filename;
	strip_path( filename_in, filename );
	scorefile_scroll_and_detect( score_stream, filename, tag, scorefile_entry_present );

	score_stream.close();
	score_stream.clear();

	return scorefile_entry_present;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_compressed
///
/// @brief
///
/// @detailed
///car compressed file formats:
///car  if SEQUENCE: is to be included, it should be first line
///car  other tags should follow
///car  SCORE: labels should immediately precede decoys
///car         followed by entries for each decoy:
///car  SCORE:
///car  decoy angles/coords...
///
/// @param  accepted - [in/out]? -
/// @param  fullatom - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_compressed(
	bool const accepted,
	bool const fullatom
)
{
	using namespace files_paths;
	using namespace misc;
	using namespace param;

	if ( get_pose_silent_out_flag() ) {
		output_pose_compressed( accepted, fullatom );
		return;
	} else if ( get_docking_flag() ) {
		output_docking_compressed(accepted, fullatom);
		return;
	}
	/*
	//psh structures from nonideal silent output can be recovered by pose extract
	else if (relax_options::looprlx_nonideal_silent_output){
		output_pose_docking_compressed(accepted, fullatom);
	}
	*/

//car initializations required for every process:
//car initialize output_res array
//car NOTE: assumes only one query will be run

	//Objexx:SGM NOTE: This is not currently set up for use with multiple
	// proteins and as a temporary hack will trigger an exit if called with
	// a different total_residue than when it performed the one-time
	// initialization. This limitation will be removed in the future.

	static int fail = { 0 };
	static bool init = { false };
	static int init_residue = 0; //Objexx:SGM Hack to try and detect new protein
	static FArray1D_bool output_res( MAX_RES(), true ); // should residue be output for each decoy?

	if ( ! init ) {
		init = true;
		init_residue = total_residue; //Objexx:SGM Hack
		output_res = true;
		initialize_compressed_loops( total_residue, output_res );
	} else if ( total_residue != init_residue ) { //Objexx:SGM Hack
		std::cout << "ERROR: output_compressed called with multiple proteins" << std::endl;
		std::cout << "       Multiple protein support not yet available" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// prepare to output chi angles if desired:
	FArray2D_float decoy_chi( MAX_CHI, MAX_RES()() );
	if ( fullatom && output_chi_silent ) {
		FArray2D_int decoy_rot( MAX_CHI, MAX_RES()() );
		FArray1D_float decoy_phi( MAX_RES()() );
		FArray1D_float decoy_psi( MAX_RES()() );

		get_chi_and_rot_from_coords(total_residue,res,res_variant,full_coord,decoy_chi,decoy_rot);

	}

	update_scorefile_info();

//car figure out structure number
	int num;
	retrieve_decoy_number( num );
	std::string const label = lead_zero_string_of(
		static_cast< int >( std::pow( 10.0, NUMBER_MAX_LENGTH ) * ran3() ),
		NUMBER_MAX_LENGTH
	);
	std::string number;
	std::string tag;
	if ( accepted ) {
		number = lead_zero_string_of( num, NUMBER_MAX_LENGTH );
		tag = "S_" + number + '_' + label;
	} else {
		++fail;
		number = lead_zero_string_of( fail, NUMBER_MAX_LENGTH );
		tag = "F_" + number + '_' + label;
	}

//car check to see if output file has been initialized
	std::string const fullname( pdb_out_path + code + protein_name + ".out" );
	utility::io::ozstream pdb_out_checkstream( fullname, std::ios_base::in|std::ios_base::out );

	utility::io::ozstream pdb_out_stream;

	if ( !pdb_out_checkstream ) {
//car initializations require only once even if multiple processes occuring:
//car write header info:
    pdb_out_checkstream.close();
    pdb_out_checkstream.clear();

    pdb_out_stream.open( fullname );

		if ( !pdb_out_stream ) {
			std::cout << "Open failed for file: " << fullname << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		pdb_out_stream << "SEQUENCE: ";
		for ( int i = 1; i <= total_residue; ++i ) {
			pdb_out_stream << residue1(i);
		}
		pdb_out_stream << '\n';
//car write out any mode-specific lines to output here:
		output_compressed_loops_header(pdb_out_stream);
//car score headers should be the last in the header section:
		pdb_out_stream << "SCORE: ";
		score_header_output(pdb_out_stream,fullatom); pdb_out_stream << "\n";

	  pdb_out_stream.close();
	  pdb_out_stream.clear();
	}

//car open a temp file
	std::string filename;
	retrieve_decoy_name( filename );
	std::string const tempfile( pdb_out_path + filename + ".in_progress");
	pdb_out_stream.open( tempfile );
	if ( !pdb_out_stream ) {
		std::cout << "Open failed for file: " << tempfile << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
//car scores:
	pdb_out_stream << "SCORE: ";
	score_output( pdb_out_stream, tag, fullatom );
	pdb_out_stream << "\n";
	// topology info
	if ( output_topology_info() ) {
		pdb_out_stream << retrieve_topology_info() << ' ' << tag << '\n';
	}

//car structure
	for ( int i = 1; i <= total_residue; ++i ) {
		if ( output_res(i) ) {
			pdb_out_stream << I( 4, i ) << ' ' <<
				secstruct(i) << ' ' << F( 9, 3, phi(i) ) << F( 9, 3, psi(i) ) <<
				F( 9, 3, omega(i) ) << F( 9, 3, Eposition(1,2,i) ) <<
				F( 9, 3, Eposition(2,2,i) ) << F( 9, 3, Eposition(3,2,i) );
			if ( fullatom && output_chi_silent ) {
				for ( int chino=1; chino<= param::MAX_CHI; ++chino ) {
					pdb_out_stream << F( 9, 3, decoy_chi( chino, i ) );
				}
			}
			pdb_out_stream << ' ' << A( 11, tag ) << std::endl;
		}
	}
	pdb_out_stream.close();
	pdb_out_stream.clear();

#ifdef BOINC
	// cannot use system calls because of different platforms
	// so append tempfile to fullname using streams
	utility::io::ozstream z_ostream_app;
	z_ostream_app.open_append( fullname );
	if ( !z_ostream_app ) {
		std::cout << "Open failed for file: " << z_ostream_app.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	utility::io::izstream z_istream( tempfile );
	if ( !z_istream ) {
		std::cout << "Open failed for file: " << z_istream.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	z_ostream_app << z_istream.rdbuf();
	if ( z_ostream_app.fail() ) {
		std::cout << "Append failed for file: " << z_ostream_app.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	z_ostream_app.close();
	z_ostream_app.clear();
	z_istream.close();
	z_istream.clear();
	if ( accepted || output_all) utility::file::file_delete( tempfile );
#else
  std::ofstream out ( fullname.c_str(), std::ios_base::out | std::ios_base::app ) ;
	std::ifstream in( tempfile.c_str() );
	out << in.rdbuf() ;
	out.close(); in.close();
	if ( accepted || output_all) {
	  utility::file::file_delete( tempfile.c_str() );
	}
#endif

//bs Put all disulfide params & score info into the disulf log file.
//bs This includes info for each individual disulfide that is
//bs not included in the scorefile.
	disulfides::BOUNDARY::output_disulf_stats_to_log( filename, tag, fullatom );

}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_taboo
///
/// @brief
///
/// @detailed
///car compressed file formats:
///car  if SEQUENCE: is to be included, it should be first line
///car  other tags should follow
///car  SCORE: labels should immediately precede decoys
///car         followed by entries for each decoy:
///car  SCORE:
///car  decoy angles/coords...
///
/// @param  accepted - [in/out]? -
/// @param  fullatom - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_taboo( bool const accepted )
{
	using namespace files_paths;
	using namespace misc;
	using namespace param;

//car initializations required for every process:
	static int fail = { 0 };

//car figure out structure number
	int num;
	retrieve_decoy_number(num);
	std::string number;
	std::string tag;
	if ( accepted ) {
		number = lead_zero_string_of( num, NUMBER_MAX_LENGTH );
		tag = "S_" + number;
	} else {
		++fail;
		number = lead_zero_string_of( fail, NUMBER_MAX_LENGTH );
		tag = "F_" + number;
	}

//car check to see if output file has been initialized
	std::string const fullname( pdb_out_path + code + protein_name + ".taboo" );
	utility::io::ozstream taboo_out_stream( fullname, std::ios_base::in|std::ios_base::out );
	if ( !taboo_out_stream ) {
		taboo_out_stream.open( fullname );
		if ( !taboo_out_stream ) {
			std::cout << "Open failed for file: " << taboo_out_stream.filename() << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		taboo_out_stream << "TABOO LIST FILE: " << '\n';

	}
	taboo_out_stream.close();
	taboo_out_stream.clear();

//car open a temp file
	std::string filename;
	retrieve_decoy_name( filename );
	std::string const tempfile( pdb_out_path + filename + ".tabootemp" );
	taboo_out_stream.open( tempfile );
	if ( !taboo_out_stream ) {
		std::cout << "Open failed for file: " << taboo_out_stream.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
//car scores:

	std::string current_barcode = get_current_barcode();
//	taboo_out_stream << SS ( tag ) << SS( current_barcode ) << '\n';
	write_taboo_file( taboo_out_stream, tag );
	taboo_out_stream.close();
	taboo_out_stream.clear();

#ifdef BOINC
	// cannot use system calls because of different platforms
	// so append tempfile to fullname using streams
	utility::io::ozstream z_ostream_app;
	z_ostream_app.open_append( fullname );
	if ( !z_ostream_app ) {
		std::cout << "Open failed for file: " << z_ostream_app.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	utility::io::izstream z_istream( tempfile );
	if ( !z_istream ) {
		std::cout << "Open failed for file: " << z_istream.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	z_ostream_app << z_istream.rdbuf();
	if ( z_ostream_app.fail() ) {
		std::cout << "Append failed for file: " << z_ostream_app.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	z_ostream_app.close();
	z_ostream_app.clear();
	z_istream.close();
	z_istream.clear();
	if ( accepted ) utility::file::file_delete( tempfile );
#else
  std::ofstream out ( fullname.c_str(), std::ios_base::out | std::ios_base::app ) ;
	std::ifstream in( tempfile.c_str() );
	out << in.rdbuf() ;
	out.close(); in.close();
	if ( accepted || output_all) {
	  utility::file::file_delete( tempfile.c_str() );
	}
#endif

}


////////////////////////////////////////////////////////////////////////////////
/// @begin apply_filters
///
/// @brief  applies various filters to a decoy structure
///
/// @detailed
///
/// @param[out]   accepted - out -  did structure pass all filters?
/// @param[out]   tag - out -  filter type which failed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  car 10/7/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
apply_filters(
	bool & accepted,
	std::string & tag
)
{
//car option dependecies:
//car   use_filter: is filter in use?

	using namespace mc_global_track::diagnose; // yab: misc removal
	using namespace scores;

//car local
	int result;

	accepted = true;
	tag = "pass";

	sheet_filter::SheetFilter sf(&misc::position, &misc::secstruct, &misc::total_residue);

	if ( !rg_filter(rg) ) {
		accepted = false;
		std::cout << "structure exceeds rg cutoff: " << rg << std::endl;
		tag = "rg_fail";
	} else if ( !co_filter(co) ) {
		accepted = false;
		std::cout << "contact order too low: " << co << std::endl;
		tag = "co_fail";
	} else if ( files_paths::use_filter(files_paths::sheet_type) && !sf.compute_result(result) ) {
		accepted = false;
		std::cout << "bad sheets: " << result << std::endl;
		tag = std::string( "sheet_" ) + static_cast<char>( result + '0' ) + "_fail";
	} else if ( !rms_filter(rms_err) ) {
		accepted = false;
		std::cout << "rms too high" << std::endl;
		tag = "rms_fail";
	} else if ( !knot_filter() ) {
		accepted = false;
		std::cout << "knots in decoy" << std::endl;
		tag = "knot_fail";
	} else if ( !topology_filter() ) {
		accepted = false;
		std::cout << "topology-filter fail" << std::endl;
		tag = "topology_fail";
	} else if ( !antibody_modeling_h3_match() ) {
		accepted = false;
		std::cout << "H3-filter fail" << std::endl;
		tag = "H3 fail";
	}
	disulfides::BOUNDARY::disulf_apply_filter(accepted,tag);
	docking_apply_filters(accepted,tag);

	save_filter_tag( tag );
}

//////////////////////////////////////////////////////////////////////////////
void
apply_filters_wrapper(
	bool const fullatom,
	bool & accepted
)
{

	// setup -- I hope this is enough (PB)
	score_set_default_function( fullatom );
	score_set_evaluate_all_terms( true );
	// not disabling rotamer trials -- worried about downstream effects, but
	// this could be a problem...
	//score_enable_rotamer_trials( false );
	recover_LOW( scorefxn );
	score_set_evaluate_all_terms( false );
	calc_rms();

	//
	std::string tag;
	apply_filters( accepted, tag);
}


//////////////////////////////////////////////////////////////////////////////
/// @begin set_decoy_number
///
/// @brief
///
/// @detailed
///
/// @param  number - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
set_decoy_number( int const number )
{
	decoy_number_block::global_number = number;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_decoy_number
///
/// @brief
///
/// @detailed
///
/// @param  number - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
get_decoy_number( int & number )
{
	number = decoy_number_block::global_number;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_output_silent_gz_flag
///
/// @brief
///
/// @detailed
///
/// @param  number - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
bool
get_output_silent_gz_flag()
{
  using namespace files_paths;
  using namespace runlevel_ns;

	//  if ( runlevel == silent && output_silent_gz ) return true;
	if ( output_silent_gz ) return true;
  return false;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_output_scorefile_gz_flag
///
/// @brief
///
/// @detailed
///
/// @param  number - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
bool
get_output_scorefile_gz_flag()
{
  using namespace files_paths;
  using namespace runlevel_ns;

  if ( use_scorefile && scorefile_init::init && output_scorefile_gz ) return true;
  return false;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_scorefile_name
///
/// @brief
///
/// @detailed
///
/// @param  number - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
std::string
get_scorefile_name()
{
  using namespace files_paths;
  std::string scorefile_name;
  if ( get_fullatom_flag() ) {
    scorefile_name = score_path + score_file + ".fasc";
  } else {
    scorefile_name = score_path + score_file + ".sc";
  }
  return scorefile_name;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin saveDecoyInfo
///
/// @brief
///
/// @detailed
///
/// @param  number - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
bool
saveDecoyInfo(
	int attempts,
	int complete,
	int farlx_stage
)
{
	using namespace makepdb_decoy;

	bool saved = false;

	utility::io::ozstream decoy_stream( decoy_cnt_filename );

	if ( decoy_stream ) {
		decoy_stream << attempts << " " << complete << " " << farlx_stage;
		decoy_stream.close();
		decoy_stream.clear();
		saved = true;
	}

	return saved;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin restoreDecoyInfo
///
/// @brief
///
/// @detailed
///
/// @param  number - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks  Restores the decoy stat info.  attempts and complete are unchanged
///           if the function returns false.
///
/// @references
///
/// @authors
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
bool
restoreDecoyInfo(
	int& attempts,
	int& complete,
	int& farlx_stage
)
{
	using namespace makepdb_decoy;

	bool restored = false;

	utility::io::izstream decoy_stream( decoy_cnt_filename );

	if ( decoy_stream ) {
		int a, c, s;
		decoy_stream >> a >> c >> s;
		decoy_stream.close();
		decoy_stream.clear();

		attempts = a;
		complete = c;
		farlx_stage = s;

		restored = true;
	}

	return restored;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin checkpoint
///
/// @brief
///
/// @detailed
///
/// @param  number - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks  Restores the decoy stat info.  attempts and complete are unchanged
///           if the function returns false.
///
/// @references
///
/// @authors
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
void
checkpoint_decoys(
	int attempts,
	int complete,
	int farlx_stage
)
{
	if ( save_random_numbers() ) {
		if ( saveDecoyInfo( attempts, complete, farlx_stage ) ) {
			std::cout << "BOINC :: " << utility::timestamp() <<
				" :: checkpoint_decoys() :: saved decoy info :: attempted_decoys: " << attempts <<
				" :: num_decoys: " << complete << " :: farlx_stage: " << farlx_stage << std::endl;
		} else {
			std::cout << "checkpoint_decoys() :: WARNING :: failed to save decoy info!" << std::endl;
		}
	} else {
		std::cout << "checkpoint_decoys() :: WARNING :: failed to save random seed!" << std::endl;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_score_only
///
/// @brief output a sc score file in a middle of simulation
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_score_only()
{
	using namespace files_paths;
	using namespace param;

//local
	std::string filename;
	std::string tag;

//tell scorefxns there's a new structure in arrays...
	score_set_new_pose();
	score6();
//current structure
	calc_rms();

	tag = "output_decoy";

	retrieve_decoy_name( filename );
	scorefile_output( filename, tag, false );

}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_score_log
///
/// @brief this will take a file name as input and use it in the score file
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_score_log( std::string const & filename )
{
	using namespace files_paths;
	using namespace param;

	std::string tag;

// tell scorefxns there's a new structure in arrays...
	score_set_new_pose();
	score12();
//current structure
	calc_rms();

	tag = "output_decoy";

	scorefile_output( filename, tag, false );

}

//////////////////////////////////////////////////////////////////////////////
void
output_docking_compressed(
	bool const accepted,
	bool const fullatom
)
{

	using namespace files_paths;
	using namespace misc;
	using namespace param;
	using namespace docking;

	if ( ! get_docking_flag() ) return;
	//if ( get_pose_docking_flag() ) {
	//	output_pose_docking_compressed( accepted, fullatom );
	//	return;
	//}

//car initializations required for every process:
//car initialize output_res array
//car NOTE: assumes only one query will be run
	static int fail = { 0 };

	update_scorefile_info();

	docking_calculate_silent_matrix();

	FArray1D_bool ideal_sc( MAX_RES()(), true );
	FArray2D_int decoy_rot( MAX_CHI, MAX_RES()() );
	FArray2D_float decoy_chi( MAX_CHI, MAX_RES()() );
	if ( fullatom ) {
		get_real_chi_from_coords( total_residue, res, res_variant, full_coord,
			decoy_chi );
		docking_flag_ideal_sidechain( ideal_sc, decoy_chi );
	}
//car figure out structure number
	int num;
	retrieve_decoy_number( num );
	std::string const label = lead_zero_string_of(
		static_cast< int >( std::pow( 10.0, NUMBER_MAX_LENGTH ) * ran3() ),
		NUMBER_MAX_LENGTH );
	std::string number;
	std::string tag;
	if ( accepted ) {
		number = lead_zero_string_of( num, NUMBER_MAX_LENGTH );
		tag = "S_" + number + '_' + label;
	} else {
		++fail;
		number = lead_zero_string_of( fail, NUMBER_MAX_LENGTH );
		tag = "F_" + number + '_' + label;
	}

//car check to see if output file has been initialized
	std::string const fullname( pdb_out_path + code + protein_name + ".out" );;
	utility::io::ozstream pdb_out_checkstream( fullname, std::ios_base::in|std::ios_base::out );

	utility::io::ozstream pdb_out_stream;
	if ( !pdb_out_checkstream ) {
//car initializations require only once even if multiple processes occuring:
//car write header info:
		pdb_out_checkstream.close();
		pdb_out_checkstream.clear();

		pdb_out_stream.open( fullname );

		if ( !pdb_out_stream ) {
			std::cout << "Open failed for file: " << fullname << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		pdb_out_stream << "SEQUENCE: ";
		for ( int i = 1; i <= total_residue;++i ) {
			pdb_out_stream << residue1(i);
		}
		pdb_out_stream << "\n";

//car score headers should be the last in the header section:
		pdb_out_stream << "SCORE: ";
		score_header_output(pdb_out_stream,fullatom);
		pdb_out_stream << "\n";
		//pdb_out_stream<< " MATRIX: T1 T2 T3 R11 R21 R31 R12 R22 R32 R31 R32 R33 \n";
	}
	pdb_out_stream.close();
	pdb_out_stream.clear();

//car open a temp file
	std::string filename;
	retrieve_decoy_name( filename );
	std::string const tempfile( pdb_out_path + filename + ".in_progress" );
	pdb_out_stream.open( tempfile.c_str(), std::ios_base::in|std::ios_base::out );
	if ( !pdb_out_stream ) {
		std::cout << "Open failed for file: " << tempfile << " " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
//car scores:
	pdb_out_stream << "SCORE: ";
	score_output( pdb_out_stream, tag, fullatom );

	//chu tranformation matrix
	pdb_out_stream << "\nMATRIX:";
	docking_write_silent_matrix( pdb_out_stream );
	pdb_out_stream << " " << tag << "\n";

	//chu sidechain angles if fa_output

	if ( fullatom ) {
		for ( int i = 1; i <= total_residue; ++i ) {
			pdb_out_stream << I( 4, i ) << ' ';
			for ( int chino=1; chino<= param::MAX_CHI; ++chino ) {
				pdb_out_stream << F( 9, 3, decoy_chi( chino, i ) );
			}
			if ( ideal_sc(i) ) {
				pdb_out_stream << " I"; // ideal sc bonds
			} else {
				pdb_out_stream << " S"; // start sc bonds
			}
			pdb_out_stream << ' ' << A( 11, tag ) << std::endl;
		}
	}
	pdb_out_stream.close();
	pdb_out_stream.clear();

#ifdef BOINC
	// cannot use system calls because of different platforms
	// so append tempfile to fullname using streams
	utility::io::ozstream z_ostream_app;
	z_ostream_app.open_append( fullname );
	if ( !z_ostream_app ) {
		std::cout << "Open failed for file: " << z_ostream_app.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	utility::io::izstream z_istream( tempfile );
	if ( !z_istream ) {
		std::cout << "Open failed for file: " << z_istream.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	z_ostream_app << z_istream.rdbuf();
	if ( z_ostream_app.fail() ) {
		std::cout << "Append failed for file: " << z_ostream_app.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	z_ostream_app.close();
	z_ostream_app.clear();
	z_istream.close();
	z_istream.clear();
	if ( accepted || output_all) utility::file::file_delete( tempfile );
#else
  std::ofstream out ( fullname.c_str(), std::ios_base::out | std::ios_base::app ) ;
	std::ifstream in( tempfile.c_str() );
	out << in.rdbuf() ;
	out.close(); in.close();
	if ( accepted || output_all) {
	  utility::file::file_delete( tempfile.c_str() );
	}
#endif

//bs Put all disulfide params & score info into the disulf log file.
//bs This includes info for each individual disulfide that is
//bs not included in the scorefile.
	disulfides::BOUNDARY::output_disulf_stats_to_log( filename, tag, fullatom );

}
///////////////////////////////////////////////////////////////////////////////
void
output_pose_docking_compressed(
	bool const accepted,
	bool const fullatom
)
{

	using namespace files_paths;
	using namespace misc;
	using namespace param;
	using namespace docking;

// psh taken out to handle other non-ideal silent output cases
//	if ( ! get_docking_flag() ) return;

	static int fail = { 0 };
	update_scorefile_info();

//car check to see if output file has been initialized
	std::string fullname;
	fullname = pdb_out_path + code + protein_name + ".out";
	utility::io::ozstream pdb_out_checkstream( fullname, std::ios_base::in|std::ios_base::out );

	utility::io::ozstream pdb_out_stream;

	if ( !pdb_out_checkstream ) {
//car initializations require only once even if multiple processes occuring:
//car write header info:
		pdb_out_checkstream.close();
		pdb_out_checkstream.clear();

		pdb_out_stream.open( fullname );

		if ( !pdb_out_stream ) {
			std::cout << "Open failed for file: " << fullname << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		pdb_out_stream << "SEQUENCE: ";
		for ( int i = 1; i <= total_residue; ++i ) {
			pdb_out_stream << residue1(i);
		}
		pdb_out_stream << std::endl;

//car score headers should be the last in the header section:
		pdb_out_stream << "SCORE: ";
		score_header_output(pdb_out_stream,fullatom);
		pdb_out_stream << "\n";
	}
	pdb_out_stream.close();
	pdb_out_stream.clear();

	//check if a temp compressed pose file has been output
	std::string filename;
	retrieve_decoy_name( filename );
	std::string const tempfile = pdb_out_path + filename + ".in_progress";
	utility::io::izstream pdb_in_stream( tempfile, std::ios_base::in );
	if ( ! pdb_in_stream ) {
		std::cout << "Open failed for file: " << tempfile << std::endl;
		std::cout << "Skip silent output for " << filename << std::endl;
		return;
	}

//car figure out structure number
	int num;
	retrieve_decoy_number( num );
	std::string const label = lead_zero_string_of(
		static_cast< int >( std::pow( 10.0, NUMBER_MAX_LENGTH ) * ran3() ),
		NUMBER_MAX_LENGTH );

	std::string number;
	std::string tag;
	if ( accepted ) {
		number = lead_zero_string_of( num, NUMBER_MAX_LENGTH );
		tag = "S_" + number + '_' + label;
	} else {
		++fail;
		number = lead_zero_string_of( fail, NUMBER_MAX_LENGTH );
		tag = "F_" + number + '_' + label;
	}

	pdb_out_stream.open_append( fullname );
	if ( !pdb_out_stream ) {
		std::cout << "Open failed for file: " << pdb_out_stream.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	//car scores:
	pdb_out_stream << "SCORE: ";
	score_output( pdb_out_stream, tag, fullatom );
	pdb_out_stream << "\n";


	std::string line;
	while( getline( pdb_in_stream, line ) ) {
		pdb_out_stream << line << tag << "\n";
	}
	pdb_out_stream.close();
	pdb_out_stream.clear();
	pdb_in_stream.close();
	pdb_in_stream.clear();

#ifdef BOINC
	if ( accepted || output_all) utility::file::file_delete( tempfile );
#else
	if ( accepted || output_all) {
	  utility::file::file_delete( tempfile.c_str() );
	}
#endif

}
///////////////////////////////////////////////////////////////////////////////
namespace decoy_attempt_counter_ns {
	int attempts(0);
}

int
get_decoy_attempt_counter()
{
	return decoy_attempt_counter_ns::attempts;
}

///////////////////////////////////////////////////////////////////////////////
bool
no_prev_decoy_attempts()
{
	if ( decoy_attempt_counter_ns::attempts == 0 ) return true;
	return false;
}

///////////////////////////////////////////////////////////////////////////////
void
reset_decoy_attempt_counter()
{
	decoy_attempt_counter_ns::attempts = 0;
}


///////////////////////////////////////////////////////////////////////////////
void
increment_decoy_attempt_counter()
{
	++decoy_attempt_counter_ns::attempts;
}
///////////////////////////////////////////////////////////////////////////////
bool
get_do_checkpointing()
{
	static bool do_farlx_checkpointing( false );
	static bool init( false );

	if ( !init ) {
		do_farlx_checkpointing = truefalseoption("do_farlx_checkpointing");
		init = true;
	}

	return do_farlx_checkpointing;

}

bool
get_do_pose_checkpointing()
{
	static bool do_pose_checkpointing( false );
	static bool init( false );
	if ( !init ) {
#ifdef BOINC
		do_pose_checkpointing = !truefalseoption("disable_pose_checkpointing");
#else
		do_pose_checkpointing = truefalseoption("do_pose_checkpointing");
#endif
		init = true;
	}

	return do_pose_checkpointing;
}

int
get_checkpoint_interval()
{
	static int checkpointing_interval( 120 );
	static bool init( false );

	if ( !init ) {
		intafteroption( "checkpointing_interval", 120, checkpointing_interval );
		init = true;
	}

	return checkpointing_interval;
}
////////////////////////////////////////////////////////////////////////////
namespace pose_silent_out_ns {
	pose_ns::Pose silent_out_decoy_pose;
	pose_ns::Pose silent_out_start_pose;
}
///////////////////////////////////////////////////////////////////////////
bool
get_pose_silent_out_flag() {
	return files_paths::pose_silent_out;
}
///////////////////////////////////////////////////////////////////////////
void
silent_out_save_start_pose(
	bool const fullatom,
	bool const ideal_pose,
	bool const coords_init
)
{
	using namespace pose_silent_out_ns;

	if( !get_pose_silent_out_flag() ) return;

	pose_from_misc( silent_out_start_pose, fullatom, ideal_pose, coords_init );

	return;
}
//////////////////////////////////////////////////////////////////////////////
void
silent_out_save_decoy_pose(
	pose_ns::Pose & pose
)
{
	using namespace pose_silent_out_ns;

	if( !get_pose_silent_out_flag() ) return;

	// don't care as we don't use score data anyway
	silent_out_decoy_pose.copy_coords( pose, false );

	return;
}
///////////////////////////////////////////////////////////////////////////////
void
silent_out_write_nonideal_geometry(
	pose_ns::Pose & pose,
	silent_io::Silent_out & silent_out,
	std::string filename
)
{
	using namespace silent_io;

	// make .bonds file if necessary:
	if ( !pose.ideal_backbone() ) {
		std::string const bonds_filename( filename+".bonds" );
		utility::io::ozstream file_checkstream(
			bonds_filename, std::ios_base::in|std::ios_base::out );
		if ( ! file_checkstream ) {
			file_checkstream.close();
			file_checkstream.clear();
			utility::io::ozstream out;
			out.open( bonds_filename );
			pose.bonds().dump( out );
			out.close();
			out.clear();
		} else {
			file_checkstream.close();
			file_checkstream.clear();
		}
	}

	// make rotamers file if necessary:
	if ( ! silent_out.ideal_rotamers() ) {
		std::string const rot_filename( filename+".rot_templates" );
		utility::io::ozstream file_checkstream(
			rot_filename, std::ios_base::in|std::ios_base::out );
		if ( ! file_checkstream ) {
			file_checkstream.close();
			file_checkstream.clear();
			utility::io::ozstream out;
			out.open( rot_filename );
			std::map< int, std::vector<Rotamer_template> > const &
				rotamer_template_map ( silent_out.rotamer_template_map () );
			const int nres( pose.total_residue() );
			for( int i=1; i<= nres; ++i ) {
				if ( !rotamer_template_map.count(i) ) {
					std::cout << "i not in rotamer_template_map: " << i << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
				std::vector< Rotamer_template > const & rot_list
					( rotamer_template_map.find( i )->second );

				for( int j=0,j_end=rot_list.size(); j<j_end; ++j ) {
					out << i << ' ' << j << ' ' << rot_list[j] << '\n';
				}
			}
			out.close();
			out.clear();
		} else {
			file_checkstream.close();
			file_checkstream.clear();
		}
	}
	return;
}
///////////////////////////////////////////////////////////////////////////////
void
output_pose_compressed(
	bool const accepted,
	bool const fullatom
)
{
	using namespace files_paths;
	using namespace misc;
	using namespace param;
	using namespace pose_silent_out_ns;
	using namespace silent_io;


	if ( !get_pose_silent_out_flag() ) return;

	static int fail = { 0 };
	update_scorefile_info();

//car check to see if output file has been initialized
	std::string fullname;
	fullname = pdb_out_path + code + protein_name + ".out";
	if( files_paths::single_silent_writer ) fullname += ".gz";
	utility::io::ozstream pdb_out_checkstream(
		fullname, std::ios_base::in|std::ios_base::out );

  static cat_pdb_writer out_stream;

	if ( !pdb_out_checkstream ) {
//car initializations require only once even if multiple processes occuring:
//car write header info:
		pdb_out_checkstream.close();
		pdb_out_checkstream.clear();

// SJF
		out_stream.pdb_out_stream.open( fullname );
		if( files_paths::single_silent_writer )
      std::cout << "Opening "<<fullname<<" as a single silent file stream. WARNING: current setup requires that only one processor work on each such stream. MAKE SURE QUEUE=1!!!"<<std::endl;
		if ( !out_stream.pdb_out_stream ) {
			std::cout << "Open failed for file: " << fullname << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		out_stream.pdb_out_stream << "SEQUENCE: ";
		for ( int i = 1; i <= total_residue; ++i ) {
			out_stream.pdb_out_stream << residue1(i);
		}
		out_stream.pdb_out_stream << std::endl;

//car score headers should be the last in the header section:
		out_stream.pdb_out_stream << "SCORE: ";
		score_header_output(out_stream.pdb_out_stream,fullatom);
		out_stream.pdb_out_stream << "\n";

	}
	if( !files_paths::single_silent_writer ) { // SJF if single_silent_writer is on, maintain an open stream
	  out_stream.pdb_out_stream.close();
	  out_stream.pdb_out_stream.clear();
	}

//car figure out structure number
	int num;
	retrieve_decoy_number( num );
	std::string const label = lead_zero_string_of(
		static_cast< int >( std::pow( 10.0, NUMBER_MAX_LENGTH ) * ran3() ),
		NUMBER_MAX_LENGTH );

	std::string number;
	std::string tag;
	if ( accepted ) {
		number = lead_zero_string_of( num, NUMBER_MAX_LENGTH );
		tag = "S_" + number + '_' + label;
	} else {
		++fail;
		number = lead_zero_string_of( fail, NUMBER_MAX_LENGTH );
		tag = "F_" + number + '_' + label;
	}
	//chu output bonds file and rot_template for start_pose
	silent_io::Silent_out silent_out;
	if ( silent_out_start_pose.fullatom() )
		silent_out.store_rotamers( silent_out_start_pose );

	if( files_paths::single_silent_writer )
	  silent_out_write_nonideal_geometry( silent_out_start_pose,
		  silent_out, fullname.substr(0, fullname.length() - 3)); // remove the .gz suffix
	else
	  silent_out_write_nonideal_geometry( silent_out_start_pose,
		  silent_out, fullname);

  std::string tempfile;
	std::string filename;
	retrieve_decoy_name( filename );
	tempfile = pdb_out_path + filename + ".in_progress";

  if( !files_paths::single_silent_writer ) {
	  //open a temp file for current decoy output
//	  std::string filename;
//	  retrieve_decoy_name( filename );
//	  tempfile = pdb_out_path + filename + ".in_progress";
	  out_stream.pdb_out_stream.open( tempfile );
	  if ( !out_stream.pdb_out_stream ) {
		  std::cout << "Open failed for file: " << out_stream.pdb_out_stream.filename()
							<< std::endl;
		  utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	  }
  }
	//car scores:
	out_stream.pdb_out_stream << "SCORE: ";
	score_output( out_stream.pdb_out_stream, tag, fullatom );
	out_stream.pdb_out_stream << "\n";

	//chu current decoy in pose compressed format
	if ( ! silent_out_decoy_pose_exist() ) {
		// no previously saved pose info, copy from misc
		pose_from_misc( silent_out_decoy_pose, fullatom, false, true );
	}
	int const nres( silent_out_decoy_pose.total_residue() );
	FArray1D_bool output_rsd( nres, true );
	FArray1D_int rot_template( nres, 0 );
	if ( ! silent_out.ideal_rotamers() ) {
		silent_out.get_rot_templates( silent_out_decoy_pose, rot_template );
	}
	dump_compressed_pose( tag, silent_out_decoy_pose, output_rsd, rot_template,
		0/*symm*/, out_stream.pdb_out_stream );

	if( !files_paths::single_silent_writer ) {
	  out_stream.pdb_out_stream.close();
	  out_stream.pdb_out_stream.clear();
	}

  // append the file and then delete it.
#ifdef BOINC
	// cannot use system calls because of different platforms
	// so append tempfile to fullname using streams
	utility::io::ozstream z_ostream_app;
	z_ostream_app.open_append( fullname );
	if ( !z_ostream_app ) {
		std::cout << "Open failed for file: " << z_ostream_app.filename()
							<< std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	utility::io::izstream z_istream( tempfile );
	if ( !z_istream ) {
		std::cout << "Open failed for file: " << z_istream.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	z_ostream_app << z_istream.rdbuf();
	if ( z_ostream_app.fail() ) {
		std::cout << "Append failed for file: " << z_ostream_app.filename()
		  				<< std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	z_ostream_app.close();
	z_ostream_app.clear();
	z_istream.close();
	z_istream.clear();
	if ( accepted || output_all) utility::file::file_delete( tempfile );
#else
  if( !files_paths::single_silent_writer ) {
    std::ofstream out ( fullname.c_str(), std::ios_base::out | std::ios_base::app ) ;
	  std::ifstream in( tempfile.c_str() );
	  out << in.rdbuf() ;
	  out.close(); in.close();
	}
	if ( accepted || output_all) {
	  utility::file::file_delete( tempfile.c_str() );
 	}
#endif
	return;
}
///////////////////////////////////////////////////////////////////////////////
bool
silent_out_decoy_pose_exist() {

	using namespace pose_silent_out_ns;

	static bool exist( false );;
	static bool init( false );

	if ( !init ) {
		if ( silent_out_decoy_pose.total_residue() == 0 ){
			exist = false;
		} else {
			exist = true;
		}
		init = true;
	}
	return exist;
 }

///////////////////////////////////////////////////////////////////////////////

utility::io::ozstream & get_cat_pdb_stream( std::string const & fullname ) {

	using namespace files_paths;

	static cat_pdb_writer out_stream;

	if ( concat_outpdbs ) {

		// add fullname to the list of completed PDBs

	  utility::io::ozstream list;
		list.open_append( files_paths::cat_list_filename );

		list << fullname << std::endl;
		list.close(); list.clear();

		// return the (static) shared ozstream
		out_stream.set_filename( out_stream.curr_filename );

		// write current PDB name (ie. fullname)
		out_stream.pdb_out_stream << "HEADER  CURRENT FILENAME: " << fullname << std::endl;

		remove( (fullname).c_str() ); // remove the placeholder pdb file
		return out_stream.pdb_out_stream;

	}

	// create and return a new (static) ozstream
	out_stream.set_filename(fullname);
	return out_stream.pdb_out_stream;

 }


