// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 23432 $
//  $Date: 2008-06-24 16:25:52 +0300 (Tue, 24 Jun 2008) $
//  $Author: yab $


// Rosetta Headers
#include "pose.h"
#include "aaproperties_pack.h" // nchi
#include "after_opts.h" // stringafteroption
#include "angles.h"
#include "constraints.h" // set_use_pose_constraints
#include "current_pose.h"
#include "decoystats.h" // allatom_rms
#include "files_paths.h" // for resfile and use_status for MC
#include "fullatom.h" // for get_fullatom_flag()
#include "gl_graphics.h"
#include "input_pdb.h"  // dihedral
#include "job_distributor.h" // for pose monte carlo checkpointing
#include "jumping_diagnostics.h" // REMOVE ME LATER!!!!!!!!!!!!!
#include "jumping_minimize.h"
#include "jumping_refold.h"
#include "jumping_util.h"
#include "kin_util.h"
#include "kin_atom.h"
#include "make_pdb.h"
#include "minimize.h"
#include "misc.h" // for set_current_pose(...)
#include "misc_removal.h" // for misc setup ... yab/rhiju need to clean this!
#include "monte_carlo.h"
#include "namespace_low_pose.h" // for BOINC stuff (pose_to_low), may be useful otherwise.
#include "native.h"
#include "orient_rms.h"
#include "output_decoy.h" // for checkpointing interval
#include "pack.h" // pack_rotamers
//#include "packing_measures.h"
#include "PackerTask.h"
#include "param.h" // MAX_POS, MAX_CHI, MAX_ATOM
#include "param_aa.h" // aa_XXX
#include "pose_io.h"
#include "pose_constraints.h"
#include "dna.h" // is_DNA_backbone_atom -- move?
#include "pose_rotamer_trials.h"
#include "prof.h"
#include "random_numbers.h" // ran3 in Monte_carlo
#include "read_aaproperties.h" // LookupByName
#include "refold.h" // get_GL_matrix
#include "RotamerSet.h"
#include "runlevel.h"
#include "score.h"
#include "silent_input.h"
#include "symmetric_design.h"
#include "symmetry_info.h"
#include "trajectory.h"
#include "util_basic.h" // subtract_degree_angles, periodic_range
#include "util_vector.h" // Ddotprod, etc

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh> //Pretty output.

// C++ Headers
#include <algorithm>
#include <cstddef>
#include <iostream>
#include <fstream>

//Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>
#include <utility/file/file_sys_util.hh>

#include <numeric/xyz.functions.hh>

//BOINC
#ifdef BOINC
#ifdef _WIN32
#include "boinc_win.h"
#endif
#include "trajectory.h" // for BOINC stuff
#include "pose_rms.h" // for BOINC stuff (calc_rms)
#include "counters.h" // for BOINC stuff (ntrials)
#include "boinc_api.h"
#endif


namespace pose_ns {


	/////////////////////////////////////////////////////////
	///////////////////// POSE STUFF ////////////////////////
	/////////////////////////////////////////////////////////

	/////////////////////////////////////////////////////////////////////////////
	//
	// WARNING: this subroutine is not currently as fast as it could be.
	// if it is getting called in a performance critical loop it could be
	// further optimized: it recalculates all the backbone bonds and torsions
	// rather than just the relevant stretch. Because the internal pose
	// representation will be changing in the near future I decided to
	// wait on this.
	//
	// seg_coord contains full_coord info for the segment in question
	//
	// the 1-index position should correspond to seg_begin, ie
	// if you have a full_coord array for the whole protein you would
	// want to pass in full_coord(1,1,seg_begin) to get the correct
	// slice.

	void
	Pose::set_segment_full_coord(
		int const seg_size,
		int const seg_begin,
		FArray3Da_float seg_coord
	)
	{
		seg_coord.dimension( 3, param::MAX_ATOM()(), seg_size );

		if ( new_torsions ) refold(); // ensure coords up to date

		assert( fullatom_ ); // temporary
		assert( seg_begin >= 1 && seg_begin+seg_size-1 <= total_residue_ );

		// update coords + set the coord-moved arrays for correct scoring
		for ( int i=1; i<=seg_size; ++i ) {
			int const pos( seg_begin+i-1 );
			int const natoms
				( aaproperties_pack::natoms( res_(pos), res_variant_(pos) ) );
			for ( int k=1; k<=3; ++k ) {
				for ( int j=1; j<= natoms; ++j ) {
					full_coord_(k,j,pos) = seg_coord(k,j,i);
				}
				Eposition_(k,1,pos) = full_coord_(k,1,pos); // N
				Eposition_(k,2,pos) = full_coord_(k,2,pos); // CA
				Eposition_(k,3,pos) = full_coord_(k,5,pos); // CB
				Eposition_(k,4,pos) = full_coord_(k,3,pos); // C
				Eposition_(k,5,pos) = full_coord_(k,4,pos); // O
			}

			// ensure correct scoring:
			// use "new_chi_score" rather than "new_bb_score" since the change
			// does not propagate downstream outside of the segment region
			// as it might if we had changed backbone torsion angles.
			//
			// this will guarantee that all interactions with this residue are
			// updated in the next score calculation.
			new_chi_score(pos) = true;
		}

		// update centroids -- all of them!
		rebuild_centroids();

		// update torsions -- all of them!
		if (atom_tree_) {
			atom_tree_->update_torsions(full_coord_, total_residue_, res_, res_variant_);
		} else {
			// update bonds -- all of them!
			calc_bonds();

			backbone_torsions_from_position();
			update_chi();
		}

		// ensure that change to new_chi_score is recognized
		domain_map_updated = false;
		score_data.after_refolding();
	}

	void
	Pose::set_full_coord(
			FArray3Da_float fcoord
			){
		full_coord_ = fcoord;
		update_chi();
	}

	// set sidechain coordinates from rotamer set
	void
	Pose::set_rotcoord(
										 int const & ires,
										 RotamerSet const & rotamer_set,
										 int const & rot_index
		 )
	{
		int je=aaproperties_pack::natoms( res(ires), res_variant(ires) );
		for(int j=5; j<=je; ++j){
			for(int k=1; k<=3; ++k){
				full_coord_(k,j,ires) = rotamer_set.get_rotcoord(k,j,rot_index);
			}
		}
		update_chi_residue(ires);
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::setup_atom_tree()
	{
		// things are very different after this call!
		assert( fullatom_ && check_xyz( full_coord_ ) );
		FArray1D_bool bb_move,chi_move,jump_move;
		bool restore_allow_move( false );
		if ( atom_tree_ ) {
			atom_tree_->clear();
		} else {
			retrieve_allow_move( bb_move, chi_move, jump_move );
			restore_allow_move = true;
			atom_tree_ = new kin::Atom_tree;
		}
		atom_tree_->setup( fold_tree_, full_coord_, res_,res_variant_);
		if ( pose_bonds_ ) {
			delete pose_bonds_;
			pose_bonds_ = 0;
		}
		if ( restore_allow_move ) {
			set_allow_bb_move  (   bb_move );
			set_allow_chi_move (  chi_move );
			set_allow_jump_move( jump_move );
		}
		this->new_score_pose();
	}


	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::setup_atom_tree_preserving_allow_move()
	{
		// things are very different after this call!
		assert( fullatom_ && check_xyz( full_coord_ ) );
		FArray1D_bool bb_move,chi_move,jump_move;
		bool restore_allow_move( true );
		retrieve_allow_move( bb_move, chi_move, jump_move );
		if ( atom_tree_ ) {
			atom_tree_->clear();
		} else {
			retrieve_allow_move( bb_move, chi_move, jump_move );
			restore_allow_move = true;
			atom_tree_ = new kin::Atom_tree;
		}
		atom_tree_->setup( fold_tree_, full_coord_, res_,res_variant_);
		if ( pose_bonds_ ) {
			delete pose_bonds_;
			pose_bonds_ = 0;
		}
		if ( restore_allow_move ) {
			set_allow_bb_move  (   bb_move );
			set_allow_chi_move (  chi_move );
			set_allow_jump_move( jump_move );
		}
		this->new_score_pose();
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::clear_atom_tree()
	{
		assert( fullatom_ && check_xyz( full_coord_ ) );

		if ( atom_tree_ == 0 ) return;

		//Convert back to old-style jumps.
		for (int n = 1; n <= num_jump_; n++){
			int const u = fold_tree_.upstream_jump_residue( n );
			int const d = fold_tree_.downstream_jump_residue( n );
			Jump new_jump( Eposition_(1,1,u), Eposition_(1,1,d));
			set_jump( n, new_jump);
		}

		delete atom_tree_;
		atom_tree_ = 0;
		this->new_score_pose();

	}


	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::retrieve_allow_move(
		FArray1D_bool & bb_move,
		FArray1D_bool & chi_move,
		FArray1D_bool & jump_move
	) const
	{
		bb_move  .dimension( total_residue_ );
		chi_move .dimension( total_residue_ );
		jump_move.dimension( num_jump_ );
		bb_move   = false;
		chi_move  = false;
		jump_move = false;
		for ( int i=1; i<= total_residue_; ++i ) {
			if ( get_allow_bb_move  (i) )   bb_move(i) = true;
			if ( get_allow_chi_move (i) )  chi_move(i) = true;
		}
		for ( int i=1; i<= num_jump_; ++i ) {
			if ( get_allow_jump_move(i) ) jump_move(i) = true;
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::set_allow_move(
											 kin::Kin_torsion_type const type,
											 bool const setting
											 )
	{
		assert( atom_tree_ );
		for ( int i=1; i<= total_residue_; ++i ) {
			int const natoms( aaproperties_pack::natoms( res_(i), res_variant_(i) ));
			for ( int j=1; j<= natoms; ++j ) {
				atom_tree_->atom(j,i)->set_allow_move( type, setting );
			}
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::show_allow_move(){
		std::cout<<"Show_allow_move "<<std::endl;
		assert( atom_tree_ );
		atom_tree_->root()->show_allow_move();
	}

	/////////////////////////////////////////////////////////////////////////////
	bool
	Pose::set_phi(
		int const pos,
		float const value
	)
	{
		if ( atom_tree_ ) {
			new_torsions = true;
 			atom_tree_->set_rosetta_torsion( param_torsion::phi_torsion, pos, value);
 			return true;
 		}

		new_bb_refold(pos) = true;
		new_torsions = true;
		phi_(pos) = value;

		// handle symmetry:
		if ( symm_info ) {
			for ( std::vector< int >::const_iterator
							clone     = symm_info->bb_clones( pos ).begin(),
							clone_end = symm_info->bb_clones( pos ).end();
						clone != clone_end; ++clone ) {
				new_bb_refold( *clone ) = true;
				phi_( *clone ) = value;
			}
		}
		return true;
	}

	/////////////////////////////////////////////////////////////////////////////
	bool
	Pose::set_psi(
		int const pos,
		float const value
	)
	{
		if ( atom_tree_ ) {
			new_torsions = true;
 			atom_tree_->set_rosetta_torsion( param_torsion::psi_torsion, pos,
																			 value);
 			return true;
 		}

		new_bb_refold(pos) = true;
		new_torsions = true;
		psi_(pos) = value;

		// handle symmetry:
		if ( symm_info ) {
			for ( std::vector< int >::const_iterator
							clone     = symm_info->bb_clones( pos ).begin(),
							clone_end = symm_info->bb_clones( pos ).end();
						clone != clone_end; ++clone ) {
				new_bb_refold( *clone ) = true;
				psi_( *clone ) = value;
			}
		}
		return true;
	}

	/////////////////////////////////////////////////////////////////////////////
	bool
	Pose::set_omega(
		int const pos,
		float const value
	)
	{
		if ( atom_tree_ ) {
			if ( fold_tree_.get_is_cutpoint()(pos) ) {
				// no omega at cutpoints
				return true;
			}
			new_torsions = true;
 			atom_tree_->set_rosetta_torsion( param_torsion::omega_torsion,pos,
																			 value);
 			return true;
 		}

		new_bb_refold(pos) = true;
		new_torsions = true;
		omega_(pos) = value;

		// handle symmetry:
		if ( symm_info ) {
			for ( std::vector< int >::const_iterator
							clone     = symm_info->bb_clones( pos ).begin(),
							clone_end = symm_info->bb_clones( pos ).end();
						clone != clone_end; ++clone ) {
				new_bb_refold( *clone ) = true;
				omega_( *clone ) = value;
			}
		}
		return true;
	}

	/////////////////////////////////////////////////////////////////////////////
	bool
	Pose::set_secstruct(
		int const pos,
		char const value
	)
	{

		new_bb_refold(pos) = true;
		new_torsions = true; // do we really need this??
		secstruct_(pos) = value;

		// handle symmetry:
		if ( symm_info ) {
			for ( std::vector< int >::const_iterator
							clone     = symm_info->bb_clones( pos ).begin(),
							clone_end = symm_info->bb_clones( pos ).end();
						clone != clone_end; ++clone ) {
				new_bb_refold( *clone ) = true;
				secstruct_( *clone ) = value;
			}
		}
		return true;
	}

	/////////////////////////////////////////////////////////////////////////////
	bool
	Pose::set_name(
		int const pos,
		std::string const value
	)
	{

		new_bb_refold(pos) = true;
		new_torsions = true;
		name_(pos) = value;

		// handle symmetry:
		if ( symm_info ) {
			for ( std::vector< int >::const_iterator
							clone     = symm_info->bb_clones( pos ).begin(),
							clone_end = symm_info->bb_clones( pos ).end();
						clone != clone_end; ++clone ) {
				new_bb_refold( *clone ) = true;
				name_( *clone ) = value;
			}
		}

		return true;
	}

	/////////////////////////////////////////////////////////////////////////////
	// no freakin allow-move checking
	bool
	Pose::set_res(
		const FArray1D_int & value
	)
	{
		res_ = value;
		new_bb_refold = true;
		new_torsions = true;
		return true;
	}


	/////////////////////////////////////////////////////////////////////////////
	bool
	Pose::set_res_variant(
		const FArray1D_int & value
	)
	{
		res_variant_ = value;
		new_bb_refold = true;
		new_torsions = true;
		return true;
	}


	/////////////////////////////////////////////////////////////////////////////
	bool
	Pose::set_res(
		int const pos,
		int const value
	)
	{

		assert( value > 0 );
		if ( value != res(pos ) ) {
			new_bb_refold(pos) = true;
			new_torsions = true;
			res_(pos) = value;

			// handle symmetry:
			if ( symm_info ) {
				for ( std::vector< int >::const_iterator
 								clone     = symm_info->bb_clones( pos ).begin(),
								clone_end = symm_info->bb_clones( pos ).end();
							clone != clone_end; ++clone ) {
					new_bb_refold( *clone ) = true;
					res_( *clone ) = value;
				}
			}
		}

		return true;
	}

	/////////////////////////////////////////////////////////////////////////////
	bool
	Pose::set_res_variant(
		int const pos,
		int const value
	)
	{

		new_bb_refold(pos) = true;
		new_torsions = true;
		res_variant_(pos) = value;

		// handle symmetry:
		if ( symm_info ) {
			for ( std::vector< int >::const_iterator
							clone     = symm_info->bb_clones( pos ).begin(),
							clone_end = symm_info->bb_clones( pos ).end();
						clone != clone_end; ++clone ) {
				new_bb_refold( *clone ) = true;
				res_variant_( *clone ) = value;
			}
		}

		return true;
	}

	/////////////////////////////////////////////////////////////////////////////
	bool
	Pose::set_res_all(
		const int value
	)
	{
		res_ = value;
		new_bb_refold = true;
		new_torsions = true;
		return true;
	}


	/////////////////////////////////////////////////////////////////////////////
	bool
	Pose::set_res_variant_all(
		const int value
	)
	{
		res_variant_ = value;
		new_bb_refold = true;
		new_torsions = true;
		return true;
	}


	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::refold_sidechains_from_chi()
	{
		for ( int i=1; i<= total_residue_; ++i ) {
			int const aa( res(i) );
			if ( !param_aa::is_protein( aa ) ) continue; // DNA etc
			int const aav( res_variant(i) );
			int const nchi( aaproperties_pack::nchi(aa,aav) );
			for ( int chino=1; chino<= nchi; ++chino ) {
				// the value returned by get_chi is already standardized
				// by get_chi_and_rot_from_coords!
				// ie, it might not actually agree with the calculated torsion
				// angle
				set_chi( chino, i, chi_( chino, i ) );
			}
		}
		refold();
	}


	/////////////////////////////////////////////////////////////////////////////
	// args in same order as chi(chino,pos)
	bool
	Pose::set_chi(
		int const chino,
		int const pos,
		float const value
	)
	{
		if ( atom_tree_ ) {
			new_torsions = true;
			int const torsion_number( 3 + chino ); // offset by phi,psi,omega
			atom_tree_->set_rosetta_torsion( torsion_number, pos,
																			 value );
			return true;
		}

		assert( pos<= total_residue_ && chino <= 4 );

		new_chi_refold(pos) = true;
		new_torsions = true;
		chi_(chino,pos) = value;

		// handle symmetry:
		if ( symm_info ) {
			for ( std::vector< int >::const_iterator
							clone     = symm_info->chi_clones( pos ).begin(),
							clone_end = symm_info->chi_clones( pos ).end();
						clone != clone_end; ++clone ) {
				new_chi_refold( *clone ) = true;
				chi_( chino, *clone ) = value;
			}
		}

		return true;
	}

	/////////////////////////////////////////////////////////////////////////////
	bool
	Pose::set_jump(
		int const jump_number,
		const Jump & new_jump
	)
	{
		if ( atom_tree_ ) {
			int const seqpos( fold_tree_.downstream_jump_residue( jump_number ) );
			kin::Atom* atom( atom_tree_->get_jump_atom( seqpos ) );
			if ( atom ) {
				new_torsions = true;
				atom->jump() = new_jump;
			} else {
				std::cout << "WARNING: cant find jump atom: " << jump_number <<
					std::endl;
			}
			return true;
		}

		new_jump_refold( jump_number ) = true;
		new_torsions = true;
		jump_transform( jump_number ) = new_jump;

		// handle symmetry:
		if ( symm_info ) {
			for ( std::vector< int >::const_iterator
							clone     = symm_info->jump_clones( jump_number ).begin(),
							clone_end = symm_info->jump_clones( jump_number ).end();
						clone != clone_end; ++clone ) {
				new_jump_refold( *clone ) = true;
				jump_transform( *clone ) = new_jump;
			}
		}

		return true;
	}

	/////////////////////////////////////////////////////////////////////////////
	bool
	Pose::set_jump_rb_delta (
		int const jump_number,
		int const rb_no,
		int const dir,
		double const value
		)
	{
		if ( atom_tree_ ) {
			int const seqpos( fold_tree_.downstream_jump_residue( jump_number ) );
			kin::Atom* atom( atom_tree_->get_jump_atom( seqpos ) );
			if ( atom ) {
				new_torsions = true;
				atom->jump().set_rb_delta( rb_no, dir, value );
			} else {
				std::cout << "WARNING: cant find jump atom: " << jump_number <<
					std::endl;
			}
			return true;
		}

		new_jump_refold( jump_number ) = true;
		new_torsions = true;
		jump_transform(jump_number).set_rb_delta( rb_no, dir, value );

		// handle symmetry:
		if ( symm_info ) {
			for ( std::vector< int >::const_iterator
							clone     = symm_info->jump_clones( jump_number ).begin(),
							clone_end = symm_info->jump_clones( jump_number ).end();
						clone != clone_end; ++clone ) {
				new_jump_refold( *clone ) = true;
				jump_transform( *clone ).set_rb_delta( rb_no, dir, value );
			}
		}

		return true;
	}

	/////////////////////////////////////////////////////////////////////////////
	bool
	Pose::set_torsion_by_number(
		int const pos,
		int const torsion,
		float const value
	)
	{
		if ( atom_tree() ) {
			new_torsions = true;
			atom_tree_->set_rosetta_torsion( torsion, pos, value );
		} else if ( torsion == param_torsion::phi_torsion ) {
			return set_phi( pos, value );
		} else if ( torsion == param_torsion::psi_torsion ) {
			return set_psi( pos, value );
		} else if ( torsion == param_torsion::omega_torsion ) {
			return set_omega( pos, value );
		} else if ( torsion >= param_torsion::chi1_torsion &&
								torsion <= param_torsion::chi4_torsion ) {
			return set_chi( torsion - param_torsion::chi1_torsion + 1, pos, value );
		} else {
			std::cout << "Pose::set_torsion_by_number: cant handle rb-torsions: "
								<< torsion << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		return false; // cant get here
	}

	void
	Pose::set_pdb_information(
		Pdb_info const & p
		)
	{
			pdb_info_ = p;
	}

	/////////////////////////////////////////////////////////////////////////////
	// DANGER: this doesnt  save the current jump_transforms,
	// rather it recalculates them from Eposition in jumps_from_position
	//
	void
	Pose::set_fold_tree(
		Fold_tree const & f
	)
	{
		if ( total_residue_ > 0 ) {
			// already had a tree set, just replacing with a new one.
			//
			// refold here in case we need coords first.
			// eg, suppose we set a simple tree, fill all the torsion angles,
			// and then want to set a different tree and calculate jumps
			// from Eposition??
			//
			// refold just returns if the torsions are not init...
			//
			refold();
		}

		if ( atom_tree_ ) {
			if ( f.get_nres() != total_residue_ ) {
				std::cout << "cant change total_residue after setting up an "
									<< "atom_tree. deleting current atom_tree" << std::endl;
				//utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				delete atom_tree_;
				atom_tree_ = 0;
			} else {
				atom_tree_->clear();
			}
		}

		new_bb_refold = true;
		new_torsions = true;

		fold_tree_ = f;

		if ( !fold_tree_.check_fold_tree() ) {
			std::cout << "WARNING: pose::set_fold_tree reordering tree at 1" <<
				std::endl;
			fold_tree_.reorder(1);
			if ( !fold_tree_.check_fold_tree() ) {
				std::cout << "STOP:: bad fold_tree!" << fold_tree_ << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}

		update_dimensions();

		if ( atom_tree_ ) {
			// this will fail if total_residue has changed, b/c update_dimensions
			// will have cleared the full_coord array
			setup_atom_tree();
		} else {
			jumps_from_position();
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::reset_extra_scores()
	{
		extra_scores.clear();
	}

	void
	Pose::unset_extra_score( std::string const & name)
	{
		extra_scores.erase( name );
	}

	void
	Pose::reset_score_data()
	{
		score_data.clear();
	}

	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	//// get/set allow move maps //
	// bb
	void
	Pose::set_allow_bb_move(
		int const pos,
		bool const setting
	)
	{
		insert_maps_updated = false;
		if ( atom_tree_ ) {
			int const aa( res_(pos ) );
			int const max_bb_tor( param_aa::is_protein( aa ) ? 3 :
														( param_aa::is_DNA( aa ) || param_aa::is_RNA( aa ) ? 6 : 0 ));
			for ( int tor=1; tor<= max_bb_tor; ++tor ) {
				kin::Atom* atom( atom_tree_->get_rosetta_torsion_atom( tor, pos ) );
				if ( atom ) {
					atom->set_allow_move( kin::PHI, setting );
				}
			}
		} else {
			allow_bb_move(pos) = setting;
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	bool
	Pose::get_allow_bb_move(
		int const pos
	) const
	{
		if ( atom_tree_ ) {
			int const aa( res_(pos ) );
			//int const aav( res_variant_(pos ) );
			int const max_bb_tor( param_aa::is_protein( aa ) ? 3 :
														( param_aa::is_DNA( aa ) || param_aa::is_RNA( aa ) ? 6 : 0 ));

			bool allow( true );
			for ( int tor=1; tor<= max_bb_tor; ++tor ) {
				kin::Atom* atom( atom_tree_->get_rosetta_torsion_atom( tor, pos ) );
				if ( atom && !atom->get_allow_move( kin::PHI ) ) {
					allow = false;
					break;
				}
			}
			return allow;
		} else {
			return allow_bb_move(pos);
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	// jump
	bool
	Pose::get_allow_jump_move(
		int const jump_number
	) const
	{
		if ( atom_tree_ ) {
			int const seqpos( fold_tree_.downstream_jump_residue( jump_number ) );
			kin::Atom* atom( atom_tree_->get_jump_atom( seqpos ) );
			return ( atom && atom->get_allow_move( kin::RB_TRANSLATION ) &&
							 atom->get_allow_move( kin::RB_TRANSLATION ) );
		} else {
			return allow_jump_move( jump_number );
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	// jump
	bool
	Pose::get_allow_rb_move(
		int const rb_no,
		int const jump_number
	) const
	{
		// need to add atom_tree stuff here
		return allow_rb_move( rb_no, jump_number );
	}

	/////////////////////////////////////////////////////////////////////////////
	// chi
	bool
	Pose::get_allow_chi_move( int const pos ) const
	{
		if ( atom_tree_ ) {
			using aaproperties_pack::nchi;
			int const aa( res_(pos ) );
			int const aav( res_variant_(pos ) );
			int const max_bb_tor( param_aa::is_protein( aa ) ? 3 :
														( param_aa::is_DNA( aa ) ||
															param_aa::is_RNA( aa) ? 6 : 0 ));
			int const max_chi( param_aa::is_protein( aa ) ? aaproperties_pack::nchi(aa,aav):
											 ( param_aa::is_DNA( aa ) ? 1 :
												 ( param_aa::is_RNA( aa ) ? 4 : 0 )));
			bool allow( true );
			for ( int chino=1; chino<= max_chi; ++chino ) {
				kin::Atom* atom( atom_tree_->get_rosetta_torsion_atom(max_bb_tor+chino,
																															pos ));
				if ( atom && !atom->get_allow_move( kin::PHI ) ) {
					allow = false;
					break;
				}
			}
			return allow;
		} else {
			return allow_chi_move( pos );
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::set_allow_bb_move(
		bool const setting
	)
	{
		insert_maps_updated = false;
		for( int i=1; i<= total_residue_; ++i ) {
			set_allow_bb_move(i,setting);
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::set_allow_bb_move(
		const FArray1D_bool & setting
	)
	{
		if ( int(setting.size1()) < total_residue_ ) {
			std::cout << "Pose::set_allow_bb_move: setting array is too small" <<
				std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		for( int i=1; i<= total_residue_; ++i ) {
			set_allow_bb_move(i,setting(i));
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	// jump
	void
	Pose::set_allow_jump_move(
		bool const setting
	)
	{
		insert_maps_updated = false;
		for( int i=1; i<= num_jump_; ++i ) {
			set_allow_jump_move(i,setting);
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::set_allow_rb_move(
		int const rb_no,
		int const jump_number,
		bool const setting
	)
	{
		insert_maps_updated = false;
		allow_rb_move( rb_no, jump_number ) = setting;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::set_allow_jump_move(
		int const jump_number,
		bool const setting
	)
	{
		insert_maps_updated = false;
		if ( atom_tree_ ) {
			int const seqpos( fold_tree_.downstream_jump_residue( jump_number ) );
			kin::Atom* atom( atom_tree_->get_jump_atom( seqpos ) );
			if ( atom ) {
				atom->set_allow_move( kin::ALL, setting );
			} else {
				std::cout << "WARNING: cant find jump atom: " << jump_number <<
					std::endl;
			}
		}

		allow_jump_move( jump_number ) = setting;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::set_allow_jump_move(
		const FArray1D_bool & setting
	)
	{
		if ( int(setting.size1()) < num_jump_ ) {
			std::cout << "Pose::set_allow_jump_move: setting array is too small" <<
				std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		for( int i=1; i<= num_jump_; ++i ) {
			set_allow_jump_move(i,setting(i));
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	// chi
	void
	Pose::set_allow_chi_move(
		bool const setting
	)
	{
		insert_maps_updated = false;
		for( int i=1; i<= total_residue_; ++i ) {
			set_allow_chi_move(i,setting);
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::set_allow_chi_move(
		int const pos,
		bool const setting
	)
	{
		//assert( fullatom_ );
		insert_maps_updated = false;

		if ( atom_tree_ ) {
			using aaproperties_pack::nchi;
			int const aa( res_(pos ) );
			int const aav( res_variant_(pos ) );
			int const max_bb_tor( param_aa::is_protein( aa ) ? 3 :
														( param_aa::is_DNA( aa ) ||
															param_aa::is_RNA( aa) ? 6 : 0 ));
			int const max_chi( param_aa::is_protein( aa ) ? aaproperties_pack::nchi(aa,aav):
											 ( param_aa::is_DNA( aa ) ? 1 :
												 ( param_aa::is_RNA( aa ) ? 4 : 0 )));

			for ( int chino=1; chino<= max_chi; ++chino ) {
				kin::Atom* atom( atom_tree_->get_rosetta_torsion_atom(max_bb_tor+chino,
																															pos ));
				if ( atom ) {
					atom->set_allow_move( kin::PHI, setting );
				} else {
					std::cout << "couldnt find this chi torsion angle: " << pos << ' ' <<
						aa << ' ' << chino << std::endl;
				}
			}
		} else {
			allow_chi_move(pos) = setting;
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::set_allow_chi_move(
		const FArray1D_bool & setting
	)
	{
		if ( int(setting.size1()) < total_residue_ ) {
			std::cout << "Pose::set_allow_chi_move: setting array is too small" <<
				std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		for( int i=1; i<= total_residue_; ++i ) {
			set_allow_chi_move(i,setting(i));
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::set_allow_bbchi_move(
		const FArray1D_bool & bb_move,
		const FArray1D_bool & chi_move
	)
	{
		if ( int( bb_move.size1()) < total_residue_ ||
				 int(chi_move.size1()) < total_residue_ ) {
			std::cout << "Pose::set_allow_chi_move: setting array is too small" <<
				std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		for( int i=1; i<= total_residue_; ++i ) {
			set_allow_bb_move (i, bb_move(i));
			set_allow_chi_move(i,chi_move(i));
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////
	// score -- should remove this one later
	float
 	Pose::score(
 		pose_param::Scoring_Function score_fxn
 	)
 	{
 		before_scoring();
 		// call scorefxn
 		float const score_val = score_fxn();
 		after_scoring();
		return score_val;
 	}

	/////////////////////////////////////////////////////////////////////////////
	// probably could do this with some kind of bind1st function adaptor:
	float
	Pose::score(
		const Score_weight_map & wt_map
	)
	{
		// call scorefxn
		before_scoring();

		float const score_val = score_by_weight_map ( wt_map );

		after_scoring();
		return score_val;
	}



	/////////////////////////////////////////////////////////////////////////////
	FArray1D_int const &
	Pose::get_domain_map() const
	{
		update_domain_map();
		return domain_map;
	}

	/////////////////////////////////////////////////////////////////////////////
	FArray1D_bool const &
	Pose::get_res_moved() const
	{
		update_pair_moved(); // same as below:
		return res_moved;
	}

	/////////////////////////////////////////////////////////////////////////////
	FArray2D_bool const &
	Pose::get_pair_moved() const
	{
		update_pair_moved();
		return pair_moved;
	}


	/////////////////////////////////////////////////////////////////////////////
	FArray4D_float const &
	Pose::get_overlap_Eposition(
		int const overlap
	) const
	{
		refold(); // depends on Eposition

		// assumes that we have already refolded
		// and that jump_transform is properly dimensioned
		// and that we dont have more cutpoints than jumps
		// (ie fold_tree is connected )
		if ( overlap <= 0 ) return overlap_Eposition_;
		bool recalculate( false );
		std::size_t const max_jump( jump_transform.size1() );
		if ( max_jump <= 0 ) return overlap_Eposition_; // no cutpoints
		if ( overlap_Eposition_.size4() < max_jump ||
				 overlap_Eposition_.size3() < static_cast< std::size_t >( 2 * overlap ) ) {
			recalculate = true;
			overlap_Eposition_.dimension
				( 3, param::MAX_POS, DRange( 1 - overlap, overlap ), max_jump );
		} else if ( !check_xyz( overlap_Eposition_ ) ) {
			recalculate = true;
		}
		if ( recalculate ) {
			jmp_update_overlap_positions( *this, overlap, overlap_Eposition_ );
		}

		return overlap_Eposition_;
	}

	/////////////////////////////////////////////////////////////////////////////
	bool
	Pose::native_pose_exists() const
	{
		return ( native_pose_ != 0 );
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::set_native_pose(
		Pose const & pose
	)
	{

		if ( native_pose_ != 0 ) {
			--(native_pose_->ptr_references_);
			if (native_pose_->ptr_references_ == 0) {
				delete native_pose_;
			}
		}

		native_pose_ = &pose;
		(native_pose_->ptr_references_)++;

	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::set_native_pose()
	{
		Pose & native_pose( *(new Pose ));
		native_pose.ptr_references_ = 0; // native_pose is "anonymous" - prevent memory leak!
		set_native_pose( native_pose );

		int const nres = total_residue();
		native_pose.simple_fold_tree( nres );
		bool const ideal_pose( false ); //
		//bool const coords_init( true ); // new_*_refold ==> false

		//pose_from_misc( native_pose, fullatom(), ideal_pose, coords_init );

	// fill in backbone torsions, sequence
		for ( int i=1; i<= nres; ++i ) {
			native_pose.set_res         ( i, res(i) );
			native_pose.set_res_variant ( i, res_variant(i) );
			//native_pose.set_phi         ( i, native::native_phi(i) );
			//native_pose.set_psi         ( i, native::native_psi(i) );
			//native_pose.set_omega       ( i, native::native_omega(i) );
			native_pose.set_secstruct   ( i, native::nat_secstruct(i) );
			// now pose.new_torsions() will be true...
		}

		// this will recalculate the torsion angles:
		native_pose.set_coords( ideal_pose, native::native_Eposition, native::native_coord );
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::reset_native_pose()
	{

		if ( native_pose_ != 0 ) {
			--(native_pose_->ptr_references_);
			if (native_pose_->ptr_references_ == 0) {
				delete native_pose_;
			}
		}

		native_pose_ = 0;
	}

	/////////////////////////////////////////////////////////////////////////////
	Pose const &
	Pose::native_pose() const
	{
		assert( native_pose_ != 0 );
		return *native_pose_;
	}

	/////////////////////////////////////////////////////////////////////////////
	bool
	Pose::start_pose_exists() const
	{
		return ( start_pose_ != 0 );
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::set_start_pose(
		Pose const & pose
	)
	{

		if ( start_pose_ != 0 ) {
			--(start_pose_->ptr_references_);
			if (start_pose_->ptr_references_ == 0) {
				delete start_pose_;
			}
		}

		start_pose_ = &pose;
		(start_pose_->ptr_references_)++;

	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::reset_start_pose()
	{

		if ( start_pose_ != 0 ) {
			--(start_pose_->ptr_references_);
			if (start_pose_->ptr_references_ == 0) {
				delete start_pose_;
			}
		}

		start_pose_ = 0;
	}

	/////////////////////////////////////////////////////////////////////////////
	Pose const &
	Pose::start_pose() const
	{
		assert( start_pose_ != 0 );
		return *start_pose_;
	}


	/////////////////////////////////////////////////////////////////////////////
	FArray1D_bool
	Pose::flag_nonideal()
	{
		if ( pose_bonds_ == 0 ) {
			FArray1D_bool all_false( total_residue_, false );
			return all_false;
		} else {
			return pose_bonds_->flag_nonideal();
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	// set bond geometry
	void
	Pose::set_bonds(
		bonds_class::Bonds const & bonds
	)
	{
		if ( bonds.nres() != total_residue_ ) {
			std::cout << "pose::set_bonds: nres mismatch: " <<
				bonds.nres() << ' ' << total_residue_ << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		if ( pose_bonds_ == 0 ) {
			pose_bonds_ = new bonds_class::Bonds;
		}
		*pose_bonds_ = bonds;
		new_bb_refold = true;
		new_torsions = true;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::reset_bonds()
	{
		if ( pose_bonds_ != 0 ) delete pose_bonds_;
		pose_bonds_ = 0;
		new_bb_refold = true;
		new_torsions = true;
	}

	/////////////////////////////////////////////////////////////////////////////
	// access constraints
	bool
	Pose::constraints_exist() const
	{
		return ( pose_constraints_ != 0 );
	}

	/////////////////////////////////////////////////////////////////////////////
	cst_set_ns::Cst_set const &
	Pose::constraints() const
	{
		return *pose_constraints_;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::set_constraints(
		cst_set_ns::Cst_set const & constraints_in
	)
	{
		pose_constraints_ = &constraints_in;
		pose_constraints_->add_ptr_reference();
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::reset_constraints()
	{
		if ( pose_constraints_ != 0 ) pose_constraints_->remove_ptr_reference();
		pose_constraints_ = 0;
	}

	/////////////////////////////////////////////////////////////////////////////
	//lin compare the pose aa aav to those in Fullatom_id
	bool Pose::match_aa_aav(  kin::Fullatom_id const & atom ) const {
		return ( atom.aa() == res( atom.rsd() )
						 && atom.aav() == res_variant( atom.rsd() ) );
	}

	/////////////////////////////////////////////////////////////////////////////
	//lin get Fullatom_id which get the aa aav from pose
	kin::Fullatom_id Pose::get_fullatom_id(  kin::Atom_id const & atom ) const {
		kin::Fullatom_id full_atom
			( atom, res_( atom.rsd() ), res_variant_( atom.rsd() ) );
		return full_atom;
	}

	/////////////////////////////////////////////////////////////////////////////
	/////////////
	// operator=
	//
	// doesnt copy:
	// 1. native_pose, start_pose
	// 2. insert_maps
	// 3. pair_moved,jmp_domain_map
	// 4. refolding pose
	//
	Pose &
	Pose::operator =(
		Pose const & src
	)
	{
		// this handles fold_tree, torsions, bonds, coords, allow_move,new_*_refold
		copy_coords( src, true /* we will copy score_data */ );

		// pdb info
		pdb_info_ = src.pdb_info_;

		// DNA basepair partner information
		basepair_partner_ = src.basepair_partner_;

		// score_data:
		PROF_START( prof::POSE_COPY_SCORE_DATA );
		score_data = src.score_data;
		PROF_STOP ( prof::POSE_COPY_SCORE_DATA );

		extra_scores = src.extra_scores;

		// constraints: just copy pointers! we don't own this memory
		pose_constraints_ = src.pose_constraints_;
		if ( pose_constraints_ != 0 ) pose_constraints_->add_ptr_reference();

		// reset book-keeping since we're not copying pair_moved, jmp_domain_map,
		// insert_map, insert_size
		insert_maps_updated = false;
		domain_map_updated = false;

		// pose pointers:
		ptr_references_ = 1;
		native_pose_ = src.native_pose_;
		start_pose_ = src.start_pose_;
		if ( native_pose_ != 0 ) {
			(native_pose_->ptr_references_)++;
		}
		if ( start_pose_ != 0 ) {
			(start_pose_->ptr_references_)++;
		}

		// copy the symmtry information
		if ( src.symm_info ) {
			if ( !symm_info ) symm_info = new Symmetry_info();
			*symm_info = *src.symm_info;
		} else if ( symm_info ) {
			delete symm_info;
			symm_info = 0;
		}


		// things not relevant to atom_tree_ pose:
		if ( !atom_tree_ ) {
			// new_*_score
			new_bb_score = src.new_bb_score;
			new_jump_score = src.new_jump_score;
			if ( fullatom_ ) new_chi_score = src.new_chi_score;

			// we dont copy the refolding pose:
			refold_from_current = true;
			if ( !src.refold_from_current ) {
				std::cout << "WARNING:: Pose::operator= not copying the refolding pose"
									<< std::endl;
			}
		}

		return *this;
	}

	/////////////////////////////////////////////////////////////////////////////
// 	void
// 	Pose::set_symm(
// 		int const N
// 	)
// 	{
// 		symm_ = N;
// 	}


	/////////////////////////////////////////////////////////////////////////////
	int
	Pose::get_total_insert() const
	{
		update_insert_maps();
		return total_insert;
	}

	/////////////////////////////////////////////////////////////////////////////
	FArray1D_int const &
	Pose::get_insert_map(
		int & total_insert_out
	) const
	{
		update_insert_maps();
		total_insert_out = total_insert;
		return insert_map;
	}

	/////////////////////////////////////////////////////////////////////////////
	FArray1D_int const &
	Pose::get_insert_size() const
	{
		update_insert_maps();
		return insert_size;
	}


	/////////////////////////////////////////////////////////////////////////////
	// puts ideal bond-lenths into segment from start to stop
	void
	Pose::insert_ideal_bonds(
		int const start,
		int const stop
	)
	{

		assert( !atom_tree_ );
		if ( start > stop || start < 1 || stop > total_residue_ ) {
			std::cout << "bad args to Pose::insert_ideal_bonds: " << start << ' '
								<< stop << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		if ( pose_bonds_ == 0 ) {
			std::cout << "entire pose already ideal!!!" << std::endl;
			return;
		}
// 		for ( int i=start; i<= stop; ++i ) {
// 			if ( !allow_bb_move(i) ) {
// 				std::cout << "Pose::insert_ideal_bonds: no bb moves allowed here: "
// 									<< i << std::endl;
// 				assert( false ); // DIE
// 				return;
// 			}
// 		}
		pose_bonds_->idealize( start, stop );

		// add extra residue on either side because of potential
		// problems in jumping_refold: we need to be able to superimpose
		// current onto best, only works at positions with same geometry
		// see comments to this effect in jumping_refold.cc
		//
		// PB 10-04-05 I don't think this is still necessary, as of
		// the bug fix that Chu discovered in jumping_refold
		// now there are explicit tests, and it should only be using stub
		// atoms from residues with new_bb_refold = FALSE
		// BUT I guess it can't hurt...
		for ( int i = std::max( 1, start-1 ),
						iend = std::min( total_residue_, stop+1 ); i <= iend; ++i ) {
			if ( i>= start && i<= stop ){
				assert( param_aa::is_protein( res_(i) ) );
			}
			new_bb_refold(i) = true;
		}
		new_torsions = true;
	}

	/////////////////////////////////////////////////////////////////////////////
	// puts extended phi/psi/omega into segment from start to stop
	void
	Pose::set_segment_extended(
		int const start,
		int const stop
	)
	{
		if ( start > stop || start < 1 || stop > total_residue_ ) {
			std::cout << "bad args to set_segment_extended: " << start << ' '
								<< stop << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		if ( total_residue_ == 0 ) {
			std::cout << "pose not initialized yet!!!" << std::endl;
			return;
		}
// 		for ( int i=start; i<= stop; ++i ) {
// 			if ( !allow_bb_move(i) ) {
// 				std::cout << "set_segment_extended:: no bb moves allowed here: "
// 									<< i << std::endl;
// 				return;
// 			}
// 		}
		for ( int i=start; i<=stop; ++i ) {
			set_phi(i, param::init_phi);
			set_psi(i, param::init_psi );
			set_omega(i, param::init_omega );
			set_secstruct(i, 'L' );
			set_name(i,"EXTD");
		}
	}
	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::set_fullatom_flag (
		bool const setting,
		bool const repack_rotamers //= true
	)
	{
		if ( setting == fullatom_ ) return;
		fullatom_ = setting;
		if ( fullatom_ ) {
			if (runlevel_ns::runlevel >= runlevel_ns::yap) std::cout << "setting pose fullatom flag to TRUE" << std::endl;
			if ( total_residue_ > 0 ) {
				redimension_fullatom_arrays();
			}
			// set rosetta fullatom flag:
			if ( repack_rotamers ) {
				bool const rotamers_exist( false );
				full_repack( rotamers_exist );
			}
		} else {
			if (runlevel_ns::runlevel >= runlevel_ns::yap) std::cout << "setting pose fullatom flag to FALSE" << std::endl;
			// set rosetta fullatom flag:
		}
	}
	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::recover_sidechain( pose_ns::Pose & src_pose )
	{
		assert( fullatom_ && src_pose.fullatom() );
		assert( total_residue_ == src_pose.total_residue() );

		FArray3D_float src_full_coord( src_pose.full_coord() );
		assert( check_xyz( src_full_coord ) );

		for ( int i = 1; i <= total_residue_; ++i ) {
			assert( res_(i) == src_pose.res(i) );
			assert( res_variant_(i) == src_pose.res_variant(i) );
			copy_sidechain( i, res_(i), res_variant_(i), src_full_coord(1,1,i) );
			update_chi_residue(i);
		}
	}
 /////////////////////////////////////////////////////////////////////////////
	// full repack:
	void
	Pose::full_repack(
		bool const rotamers_exist
	)
	{
		if ( !fullatom_ ) {
			std::cout << "Pose::full_repack called but fullatom= FALSE" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		FArray1D_bool allow_repack( total_residue_ );
		if ( ! rotamers_exist ) {
			// should be handled by refold now
			// setup_full_coord();
			// this will trigger a warning inside repack()
			// if allow_chi_move is F anywhere
			allow_repack = true;
		} else {
			for ( int i=1; i<= total_residue_; ++i ) {
				allow_repack(i) = allow_chi_move(i);
			}
		}
		// call the following function
		repack( allow_repack, rotamers_exist );
	}

 	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::repack(
		FArray1D_bool const & allow_repack,
		bool const include_current
	)
	{
		PROF_START( prof::MAIN_REPACK );

		if ( !fullatom_ ) {
			std::cout << "Pose::repack called but fullatom= FALSE" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		if ( new_torsions ) refold();

		// make a local copy so we can pass to pack_rotamers
		pose_update_MAX_RES( *this );
		static FArray1D_bool allow_repack_local( param::MAX_RES() );

		for ( int i=1; i<= total_residue_; ++i ) {
			if ( allow_repack(i) ) {
				allow_repack_local(i) = true;
				if ( !allow_chi_move(i) ) {
					allow_repack_local(i) = false;
					std::cout << "Hey: you requested a repack for a disallowed " <<
						"sidechain: " << i << std::endl;
					std::cout << "I'm giving it to you!" << std::endl;
				}
			} else {
				allow_repack_local(i) = false;
			}
		}

		///////////////////////////////////////////////////////////////////////////
		// call pack_rotamers:
		std::string const packmode( "packrot" );
		bool make_output_file( false );
		copy_to_misc(); // updates full_coord_ if necessary
		pack_set_current_pose( *this );

		if ( symmetric() ) {
			misc::total_residue = total_residue_for_scoring();
			// fills *clone* info in design_sym namespace
			setup_symmetric_packing( total_residue_for_scoring(), *symm_info );
			symm_info->set_exclude_pseudo_from_total_residue( true );
		}

		// hijack get_res_res_cstE:
// 		if ( this->constraints_exist() ) {
// 			set_pose_constraints( *pose_constraints_ );
// 		}

		// this is a pain: pack_rotamers expects MAX_RES() sized arrays
		// so we have to use the misc:: versions
		/*		Task.set_mode(packmode);
			Task.set_make_output_file(make_output_file);
			Task.set_allow_repack(allow_repack_local);
			Task.set_include_current(include_current);
			Task.set_include_extra(include_extra);
			Task.set_extra_rot(extra_rot);
			Task.set_extra_chi(extra_chi);*/

		//yl, Create PackerTask and setup values before pass into pack_rotamers
		PackerTask Task( *this );

		Task.set_task(packmode, make_output_file, allow_repack_local, include_current);
		//bk set variables that specify which residues to vary
		Task.setup_residues_to_vary();

		pack_rotamers( *this, Task );
		copy_to_misc();


		if ( symmetric() ) {
			misc::total_residue = total_residue_;
			reset_symmetric_packing(); // num_clones-->0
			symm_info->set_exclude_pseudo_from_total_residue( false );
		}

		// un-set current pose
		pack_reset_current_pose();

		for ( int i=1; i<= total_residue_for_scoring(); ++i ) {
			if ( misc::res(i) != res_(i) ) {
				std::cout << "WHOAH! pack changed the res array! " << i << ' ' <<
					misc::res(i) << ' ' << res_(i) << std::endl;
				std::cerr << "WHOAH! pack changed the res array! " << i << ' ' <<
					misc::res(i) << ' ' << res_(i) << std::endl;
				std::exit( EXIT_FAILURE );
			}
			copy_sidechain( i, res_(i), misc::res_variant(i),
											misc::full_coord(1,1,i));
		}

// 		// copy from misc back to full_coord_
// 		{
// 			assert( full_coord_.index(1,1,1) == misc::full_coord.index(1,1,1) &&
// 							full_coord_.size1()      == misc::full_coord.size1() &&
// 							full_coord_.size2()      == misc::full_coord.size2() );
// 			for ( int l = full_coord_.index(1,1,1), l_end = full_coord_.index(1,1,1)+
// 							total_residue_for_scoring() * full_coord_.size1() *
// 							full_coord_.size2();
// 						l < l_end; ++l ) {
// 				full_coord_[l] = misc::full_coord[l];
// 			}
// 		}


// 		// the domain map needs to be recalculated since some torsions
// 		// have changed
// 		domain_map_updated = false;

// 		// we've just updated the structure, effectively refolded, so
// 		// declare scored data no longer up-to-date
// 		score_data.after_refolding();

// 		if ( atom_tree_ ) {
// 			// will trigger a full rescore:
// 			atom_tree_->update_torsions( full_coord_, total_residue_, res_,
// 																	 res_variant_ );

// 			// this it too tricky for the time being:
// 			// and compared to the immense cost of a repack, who cares?
// 			//
// 			// of course if we were designing we'd have to be more careful...

// // 			for ( int i=1; i<= total_residue_; ++i ) {
// // 				if ( allow_repack(i) ) {
// // 					atom_tree_->new_sidechain_coords( i, res_(i), res_variant(i),
// // 																						full_coord_ );
// // 				}
// // 			}
// 		} else {
// 			update_chi(); // update our chi array

// 			if ( symmetric() ) symm_check( *this );

// 			// we just "refolded" a bunch of chi torsions, do things we do after
// 			// refolding:
// 			for ( int i=1; i<= total_residue_; ++i ) {
// 				new_chi_score(i) = ( new_chi_score(i) || allow_repack_local(i) );
// 			}
// 			if ( symmetric() ) new_chi_score = true; // could do better

// 		}

		PROF_STOP( prof::MAIN_REPACK );
	}


	///jjh Ooooo it's a nasty copy of pose::repack() to add more general access
	///jjh to pack_rotamers().  I totally promise to merge/refactor with
	///jjh pose::repack() when I get it to final form.
	///jjh Notes:
	///jjh	- This behaves like fixed_backbone(), but doesn't do clusters
 	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::pack_hack(
		bool const include_current,
		std::string const & packmode
	)
	{
		assert( pose_flag() );
		if ( !fullatom_ ) {
			std::cout << "Pose::repack called but fullatom= FALSE" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		if ( new_torsions ) refold();

		/// SET UP INFO FOR PACKER:
		///   1. Pose member data accessors
		///   2. Packer Task

		// make a local copy so we can pass to pack_rotamers
		pose_update_MAX_RES( *this );

		copy_to_misc(); // updates full_coord_ if necessary
		pack_set_current_pose( *this );

		//yl, Create PackerTask and setup values before pass into pack_rotamers
		PackerTask Task( *this );

		static FArray1D_bool which_residue( param::MAX_RES() );
		which_residue = true;

		Task.set_task(	packmode,
										false, 						// don't make an output file
										which_residue,		// Default to true, modified by setup_residues_to_vary
										include_current
										);
		stringafteroption( "resfile", "none", files_paths::resfile );
		design::active_rotamer_options.use_input_sc = truefalseoption("use_input_sc");

		Task.setup_residues_to_vary();
		/// CALL PACK ROTAMERS
		pack_rotamers( *this, Task );

		copy_to_misc();
		/// DONE PACK ROTAMERS

		// un-set current pose
		pack_reset_current_pose();

		///jjh copy over the coordinate set.  This is probably
		///jjh not as efficient as it could be.
		for ( int i=1; i<= total_residue_; ++i ) {
			copy_sidechain( i, misc::res(i), misc::res_variant(i), misc::full_coord(1,1,i), true );
		}

		// the domain map needs to be recalculated since some torsions
		// have changed
		domain_map_updated = false;

		// we've just updated the structure, effectively refolded, so
		// declare scored data no longer up-to-date
		score_data.after_refolding();

/// START OF ATOMTREE STUFF
		if ( atom_tree_ ) {
			// will trigger a full rescore:
			atom_tree_->update_torsions( full_coord_, total_residue_, res_,
																	 res_variant_ );
		} else {
			update_chi(); // update our chi array

			// we just "refolded" a bunch of chi torsions, do things we do after
			// refolding:
			for ( int i=1; i<= total_residue_; ++i ) {
				new_chi_score(i) = ( new_chi_score(i) || which_residue(i) );
			}

		}
/// END OF ATOMTREE STUFF

	}


 	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::pack(
		PackerTask const & task
	)
	{
		if ( !fullatom_ ) {
			std::cout << "Pose::repack called but fullatom= FALSE" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		// ensure most recent torsion angle changesa are reflected in the coords
		refold();

		// make a local copy so we can pass to pack_rotamers
		copy_to_misc(); // updates full_coord_ if necessary
		pack_set_current_pose( *this );

		design::active_rotamer_options.use_input_sc = truefalseoption("use_input_sc");

	  pack_rotamers( *this, task);

	  copy_to_misc();
		// un-set current pose
		pack_reset_current_pose();

		///jjh copy over the coordinate set.  This is probably
		///jjh not as efficient as it could be.
		for ( int i=1; i<= total_residue_; ++i ) {
			copy_sidechain( i, misc::res(i), misc::res_variant(i),
											misc::full_coord(1,1,i), true );
		}

		// the domain map needs to be recalculated since some torsions
		// have changed
		domain_map_updated = false;

		// we've just updated the structure, effectively refolded, so
		// declare scored data no longer up-to-date
		score_data.after_refolding();

		if ( atom_tree_ ) {
			// will trigger a full rescore:
			atom_tree_->update_torsions( full_coord_, total_residue_, res_,
																	 res_variant_ );
		} else {
			update_chi(); // update our chi array

			// we just "refolded" a bunch of chi torsions, do things we do after
			// refolding:
			for ( int i=1; i<= total_residue_; ++i ) {
				new_chi_score(i) = true;
			}

		}
	}


	/////////////////////////////////////////////////////////////////////////////
	// this routine calculates torsion angles from the coordinates
	//
	// and fills in jump_transform based on Eposition and fold_tree (by
	// calling jumps_from_position )
	//
	void
	Pose::set_coords(
		bool const ideal_pos,
		FArray3DB_float const & Epos,
		FArray3DB_float const & fcoord,
		bool const check_missing // = true,
		)
	{
		update_dimensions();
		if ( total_residue_ <= 0 ) {
			std::cout << "STOP: You need to fill the fold_tree before calling " <<
				"Pose::set_coords" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		// book-keeping
		new_bb_score = true;
		domain_map_updated = false;
		new_bb_refold = false;
		new_jump_refold = false;
		new_chi_refold = false;
		new_torsions = false;

		if ( fullatom_ ) {
			// fill full_coord_
			assert( fcoord.size2() == full_coord_.size2() );
			const int off1( full_coord_.index(1,1,1) );
			const int off2( fcoord.index(1,1,1) );
			const int size( 3 * full_coord_.size2() * total_residue_ );
			for ( int l=0; l< size; ++l ) full_coord_[ l+off1 ] = fcoord[ l+off2 ];
		}


		// check for non-protein, trigger atom-tree setup
		for ( int i=1; i<= total_residue_; ++i ) {
			bool all_protein( true );
			if ( !param_aa::is_protein( res(i) ) ) {
				all_protein = false;
				if ( !param_aa::is_NA( res(i) ) && !param_aa::is_ligand( res(i) ) ) {
					std::cout << "res array needs to be initialized before " <<
						" calling pose.set_coords(..)" << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
			}
			if ( !all_protein ) {
				std::cout << "Non protein atoms detected, using Atom_tree object" <<
					std::endl;
				setup_atom_tree();
				new_torsions = true;
				atom_tree_refold(); // builds eposition, centroid
				return; // NOTE EARLY RETURN!!
			}
		}



		// flag positions that appear to be missing:
		FArray1D_bool bb_missing( total_residue_, false );

		// debug
		assert( Epos.size1() == 3 && Epos.size2() == Eposition_.size2() );

		// fill Eposition_
		const int off1( Eposition_.index(1,1,1) );
		const int off2( Epos.index(1,1,1) );
		const int size( 3 * param::MAX_POS * total_residue_ );
		for ( int l=0; l< size; ++l ) Eposition_[ l+off1 ] = Epos[ l+off2 ];

		if ( check_missing ) {
			for ( int i=1; i<= total_residue_; ++i ) {
				if ( is_NA( i ) ) continue; // skip for NA

				if ( !backbone_coords_init( i ) ) {
					std::cout << "WARNING: Pose::set_coords: backbone missing for " <<
						"residue " << i << std::endl;
					if ( fold_tree_.get_is_jump_point()( i ) ) {
						std::cout << "DANGER!!! Pose::set_coords: backbone missing for " <<
							"a JUMP residue " << i << std::endl;
					}
					bb_missing(i) = true;
					// put in bogus coords for calculating bonds, torsions
					for ( int j=1; j<= param::MAX_POS; ++j ) {
						for ( int k=1; k<= 3; ++k ) {
							Eposition_(k,j,i) = 200+ran3()*799;
						}
					}
				}
			}
		}

		// calculate backbone torsion angles
		backbone_torsions_from_position();


		// calculate the jump_transforms
		jumps_from_position();

		// rebuild centroids
		rebuild_centroids();

		if ( !ideal_pos ) {
			// calculate bonds: note that these values will be strange around
			// chainbreaks. You can call idealize_chainbreaks to fix things
			// later.
			calc_bonds();
		}

		if ( fullatom_ && check_missing ) {
			for ( int i=1; i<= total_residue_; ++i ) {
				if ( bb_missing(i) ) {
					for ( int j=1; j<= param::MAX_ATOM()(); ++j ) {
						for ( int k=1; k<= 3; ++k ) {
							full_coord_(k,j,i) = 200+ran3()*799;
						}
					}
				} else {
					for ( int k=1; k<= 3; ++k ) {
						float const coord_dev =
							std::abs( full_coord_(k,1,i) - Eposition_(k,1,i) ) + // N
							std::abs( full_coord_(k,2,i) - Eposition_(k,2,i) ) + // CA
							std::abs( full_coord_(k,3,i) - Eposition_(k,4,i) );  // C
							//bqian - termini OXT Eposition will be different from fullcord
							// so I have to relax the O position check
//							std::abs( full_coord_(k,4,i) - Eposition_(k,5,i) );  // O

// debugging output:
// 						std::cout << i << ' ' << k << ' ' << coord_dev << ' ' <<
// 							std::abs( full_coord_(k,1,i) - Eposition_(k,1,i) ) << ' ' <<
// 							std::abs( full_coord_(k,2,i) - Eposition_(k,2,i) ) << ' ' <<
// 							std::abs( full_coord_(k,3,i) - Eposition_(k,4,i) ) <<std::endl;
						if ( coord_dev > 0.03 /*chuwang*/ ) {
							std::cout << "STOP:: Pose::set_coords(): mismatch between " <<
								" N,CA,C,O coords between full-coord and Eposition. dev= " <<
								coord_dev << std::endl;
							std::cout << k << " " << i << " "
												<<  full_coord_(k,1,i) << " " << Eposition_(k,1,i) << std::endl;

							//utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
						}
					}
				}
			}
		}

		if ( fullatom_ ) {
			// calculate chi angles, will set new_chi_refold = FALSE
			update_chi();
		}

		new_Eposition();
	} // set_coords


	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::update_backbone_bonds_and_torsions()
	{
		backbone_torsions_from_position();
		calc_bonds();
	}


	/////////////////////////////////////////////////////////////////////////////
	// this changes torsion angles around the fold_tree cutpoints

 	void
 	Pose::insert_ideal_bonds_at_fold_tree_cutpoints()
 	{
		if ( pose_bonds_ == 0 ) {
			std::cout << "Entire backbone is already ideal!!" << std::endl;
			return;
		}

 		if ( num_jump_ <= 0 ) return;

 		std::vector< int > cuts( fold_tree_.cutpoints() );
 		for( int i=0; i< int(cuts.size()) ; ++i ) {
			insert_ideal_bonds_at_cutpoint( cuts[i] );
 		}
 	}


	/////////////////////////////////////////////////////////////////////////////
	// this changes torsion angles around the fold_tree cutpoints

 	void
 	Pose::insert_ideal_bonds_at_backbone_chainbreaks(
		float const tolerance // = 1.0
	)
 	{
		std::cout << "Pose::insert_ideal_bonds_at_backbone_chainbreaks: " <<
			"I may be refolding now." << std::endl;
		refold();

		if ( pose_bonds_ == 0 ) {
			std::cout << "Entire backbone is already ideal!!" << std::endl;
			return;
		}

		for ( int i=1; i<total_residue_; ++i ) {
			FArray1D_float n_xyz(3), ca_xyz(3), c_xyz(3);
			build_ideal_C_coords( phi_(i+1), Eposition_(1,1,i+1),
														Eposition_(1,2,i+1), Eposition_(1,4,i+1), c_xyz);

			build_ideal_N_CA_coords( psi_(i), omega_(i), Eposition_(1,1,i),
															 Eposition_(1,2,i), Eposition_(1,4,i),
															 n_xyz, ca_xyz);
			float dev(0);
			for ( int k=1; k<= 3;++k ){
				float const tmp1( Eposition_(k,1,i+1) - n_xyz(k) );
				float const tmp2( Eposition_(k,4,i  ) - c_xyz(k) );
				dev += tmp1*tmp1 + tmp2*tmp2;
			}
			dev = std::sqrt( dev/2 );
			std::cout << "chainbreak dev: " << i << ' ' << dev << std::endl;
			if ( dev > tolerance ) {
				std::cout << "Chainbreak detected: " << i << ' ' << dev << std::endl;
				insert_ideal_bonds_at_cutpoint( i );
			}
		} // i
	}

	/////////////////////////////////////////////////////////////////////////////
	// this changes the torsion angles omega,phi around the cutpoint residue
  // on the assumption that they were bogus anyway. psi is recalculated

	void
	Pose::insert_ideal_bonds_at_cutpoint(
		int const cutpoint
	)
	{
		if ( atom_tree_ ) {
			using numeric::constants::d::pi;
			if ( fold_tree_.get_is_cutpoint()(cutpoint) ) {
				std::cout << "WARNING:: No need to idealize an existing fold_tree " <<
					"cutpoint.\n There is no internal coordinate information that " <<
					"spans the break if you are using an Atom_tree." << std::endl;
				return;
			}

			/////////////////////////////////////////////////////////////////////////
			// this is a bit of a pain!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
			//
			// but actually this is a tricky operation....
			// we want to put ideal bonds at a peptide connection;
			// this will pull one side or the other
			// we want to get phi and psi from the coords of single residues
			// rather than measuring across the chain break!

			FArray3D_float fcoord( full_coord_ ); // local copy

			int const H
				( chainbreak_phi_atomno(res_(cutpoint+1),res_variant_(cutpoint+1)));

			// calculate psi,phi from coords, without spanning chainbreak
			float const coord_psi
				( periodic_range( dihedral( full_coord_(1,1,cutpoint ), // N
																		full_coord_(1,2,cutpoint ), // CA
																		full_coord_(1,3,cutpoint ), // C
																		full_coord_(1,4,cutpoint )) // O
													+ 180.0, 360.0 ) );
			float const coord_phi
				( periodic_range( dihedral( full_coord_(1,H,cutpoint+1 ), // H
																		full_coord_(1,1,cutpoint+1 ), // N
																		full_coord_(1,2,cutpoint+1 ), // CA
																		full_coord_(1,3,cutpoint+1 )) // C
													+ 180.0, 360.0 ) );
			float const coord_omega( 180.0 );


//			std::cout << "psi,omega,phi calc-pop: " << psi(cutpoint) << ' ' <<
//				omega(cutpoint) << ' ' << phi(cutpoint+1) << ' ' <<
//				coord_psi << ' ' << coord_omega << ' ' << coord_phi << std::endl;

			FArray2D_float overlap_xyz(3,6);
			build_overlap_coords( cutpoint, coord_psi, coord_omega, coord_phi,
														full_coord_, overlap_xyz );

			// Atom_tree Atoms
			kin::Atom
				*N_atom( atom_tree_->atom( 1, cutpoint+1) ),
				*C_atom( atom_tree_->atom( 3, cutpoint  ) );

			//////////////////////////////////////////////////////////////
			// determine which direction we are folding through the break:
			if ( C_atom->parent == N_atom ) {
//				std::cout << "insert_ideal:: c2n!!" << std::endl;
				// we are folding c2n through the cutpoint
				assert( C_atom->input_stub_atom0() == kin::Atom_id( 1, cutpoint+1 ) &&
								C_atom->input_stub_atom1() == kin::Atom_id( 1, cutpoint+1 ) &&
								C_atom->input_stub_atom2() == kin::Atom_id( 2, cutpoint+1 ) &&
								C_atom->input_stub_atom3() == kin::Atom_id( 3, cutpoint+1 ) );

				// reorient full_coord_ onto overlap_xyz
				FArray2D_float Mgl( 4, 4 );
				get_GL_matrix( full_coord_(1,1,cutpoint),
											 full_coord_(1,2,cutpoint),
											 full_coord_(1,3,cutpoint),
											 overlap_xyz(1,1),
											 overlap_xyz(1,2),
											 overlap_xyz(1,3),
											 Mgl );

				// update coords
				// C, O, Ca, N
				GL_rot_in_place( Mgl, full_coord_(1,1,cutpoint) );
				GL_rot_in_place( Mgl, full_coord_(1,2,cutpoint) );
				GL_rot_in_place( Mgl, full_coord_(1,3,cutpoint) );
				GL_rot_in_place( Mgl, full_coord_(1,4,cutpoint) );
				// now update internal coords in Atom_tree
				kin::Coords_FArray fcoord( full_coord_ );
				atom_tree_->atom( 1, cutpoint )->update_torsions(fcoord,false);
				atom_tree_->atom( 2, cutpoint )->update_torsions(fcoord,false);
				atom_tree_->atom( 3, cutpoint )->update_torsions(fcoord,false);
				atom_tree_->atom( 4, cutpoint )->update_torsions(fcoord,false);
				new_torsions = true; // ensure a refold
				{ // debug:
					double const O_psi_offset
						( atom_tree_->atom( 4, cutpoint )->get_torsion( kin::PHI ) );
//					std::cout << "O_psi_offset " << O_psi_offset << std::endl;
					assert( std::abs( std::abs( O_psi_offset ) - pi ) < 1e-3 );
				}
			} else if ( N_atom->parent == C_atom ) {
				assert( N_atom->input_stub_atom0() == kin::Atom_id( 3, cutpoint ) &&
								N_atom->input_stub_atom1() == kin::Atom_id( 3, cutpoint ) &&
								N_atom->input_stub_atom2() == kin::Atom_id( 2, cutpoint ) &&
								N_atom->input_stub_atom3() == kin::Atom_id( 1, cutpoint ) );

//				std::cout << "insert_ideal:: n2c!!" << std::endl;
				N_atom->show();
				// we are folding n2c through the cutpoint

				// reorient full_coord_ onto overlap_xyz
				FArray2D_float Mgl( 4, 4 );
				get_GL_matrix( full_coord_(1,1,cutpoint+1),
											 full_coord_(1,2,cutpoint+1),
											 full_coord_(1,3,cutpoint+1),
											 overlap_xyz(1,4),
											 overlap_xyz(1,5),
											 overlap_xyz(1,6),
											 Mgl );

				// update coords
				// N, CA, C, H
				GL_rot_in_place( Mgl, full_coord_(1,1,cutpoint+1) );
				GL_rot_in_place( Mgl, full_coord_(1,2,cutpoint+1) );
				GL_rot_in_place( Mgl, full_coord_(1,3,cutpoint+1) );
				GL_rot_in_place( Mgl, full_coord_(1,H,cutpoint+1) );
				// now update internal coords in Atom_tree -- order matters!!!!!
				kin::Coords_FArray fcoord( full_coord_ );
				atom_tree_->atom( 1,cutpoint+1)->update_torsions(fcoord,false);
				atom_tree_->atom( 2,cutpoint+1)->update_torsions(fcoord,false);
				atom_tree_->atom( 3,cutpoint+1)->update_torsions(fcoord,false);
				atom_tree_->atom( H,cutpoint+1)->update_torsions(fcoord,false);
				new_torsions = true; // ensure a refold
				{ // debug:
					double const H_phi_offset
						( atom_tree_->atom( H, cutpoint+1 )->get_torsion( kin::PHI ) );
//					std::cout << "H_phi_offset" << H_phi_offset << std::endl;
					assert( std::abs( std::abs( H_phi_offset ) - pi ) < 1e-3 );
				}
			} else {
				std::cout << "can't update cutpoint bond info!!! " << cutpoint <<
					' ' << fold_tree_ << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			return;
		}



		if ( pose_bonds_ == 0 ) {
			std::cout << "entire pose already ideal!" << std::endl;
			return;
		}

		if ( cutpoint < 1 || cutpoint >= total_residue_ ) {
			std::cout << "bad cutpoint residue: " << cutpoint << std::endl;
			return;
		}

		// this changes ox_psi_offset to ideal value(180)
		// so we have to update psi as well, or the oxygen will move
		pose_bonds_->idealize_chainbreak( cutpoint );

		// need to update psi, omega of cutpoint and phi of cutpoint + 1
		// temporary: assume we dont have an NH position to determine
		// phi
//		std::cout << "setting phi = -90 at position: " << cutpoint+1 <<
//			std::endl;
//		std::cout << "setting omega = 180 at position: " << cutpoint <<
//			std::endl;
		set_phi( cutpoint+1, -90.0 ); // use pack.cc default value
		set_omega( cutpoint, 180.0 );

		// calculate psi from the position of the O??
		// but what if no coordinates exist for that position?
		// if no coords, keep current psi value, may have been
		// put there by set_psi(...)
		if ( backbone_coords_init( cutpoint ) ) {
			//std::cout << "recalculate psi from O position: " << cutpoint <<
			//	std::endl;
			float const new_psi
				( periodic_range( dihedral( Eposition_(1,1,cutpoint ), // N
																		Eposition_(1,2,cutpoint ), // CA
																		Eposition_(1,4,cutpoint ), // C
																		Eposition_(1,5,cutpoint ) )// O
													+ 180.0, 360.0 ) );
			set_psi( cutpoint, new_psi );
		}
	}
	/////////////////////////////////////////////////////////////////////////
	// flag the backbone chainbreaks
	void
 	Pose::identify_backbone_chainbreaks(
		FArray1D_bool & chainbreaks,
		float const tolerance // = 1.0
	)
 	{
		if ( runlevel_ns::runlevel >= runlevel_ns::inform ){
			std::cout << "Pose::identify_backbone_chainbreaks: " <<
				"I may be refolding now." << std::endl;
		}
		refold();

		if ( pose_bonds_ == 0 ) {
			std::cout << "Entire backbone is already ideal!!" << std::endl;
			return;
		}

		for ( int i=1; i<total_residue_; ++i ) {
			FArray1D_float n_xyz(3), ca_xyz(3), c_xyz(3);
			build_ideal_C_coords( phi_(i+1), Eposition_(1,1,i+1),
														Eposition_(1,2,i+1), Eposition_(1,4,i+1), c_xyz);

			build_ideal_N_CA_coords( psi_(i), omega_(i), Eposition_(1,1,i),
															 Eposition_(1,2,i), Eposition_(1,4,i),
															 n_xyz, ca_xyz);
			float dev(0);
			for ( int k=1; k<= 3;++k ){
				float const tmp1( Eposition_(k,1,i+1) - n_xyz(k) );
				float const tmp2( Eposition_(k,4,i  ) - c_xyz(k) );
				dev += tmp1*tmp1 + tmp2*tmp2;
			}
			dev = std::sqrt( dev/2 );
			if ( runlevel_ns::runlevel >= runlevel_ns::inform )
				std::cout << "chainbreak dev: " << i << ' ' << dev << std::endl;
			if ( dev > tolerance ) {
				std::cout << "Chainbreak detected: " << i << ' ' << dev << std::endl;
				chainbreaks(i) = true;
			}
		} // i
	}

	/////////////////////////////////////////////////////////////////////////////
	// rottrial, based on allow_chi
	void
	Pose::rottrial(
		const Score_weight_map & weight_map,
		const int cycles
	)
	{
		score_set_try_rotamers( true );
		set_rt_pos_list_by_allow( allow_chi_move, total_residue_, cycles );
		score( weight_map );
		score_set_try_rotamers( false );
	}

	/////////////////////////////////////////////////////////////////////////////
	// called by rotamer_trials_pose
	// which is called in the middle of scoring, after most/all fullatom
	// score calculations have been done
	void
	Pose::after_rotamer_trials(
		int const nres,
		FArray1DB_int const & resv,
		FArray3DB_float const & xyz
	)
	{
		for ( int i=1; i<= nres; ++i ) {
			copy_sidechain( i, res(i), resv(i), xyz(1,1,i) );
		}

// 		// copy the new coords into the full_coord array
// 		assert( xyz.size1() == full_coord_.size1() &&
// 						 int(xyz.size1()) == 3 &&
// 						 int(xyz.size2()) == param::MAX_ATOM() &&
// 						 xyz.size2() == full_coord_.size2() );
// 		int const xyz_offset ( xyz.index(1,1,1) );
// 		int const fc_offset ( full_coord_.index(1,1,1) );
// 		for ( int l=0, l_end = 3 * param::MAX_ATOM()() * total_residue_; l < l_end; ++l ) {
// 			full_coord_[ l + fc_offset ] = xyz[ l + xyz_offset ];
// 		}

// 		// copy new coords to misc:
// 		// update_misc_full_coord( this );

// 		if ( atom_tree_ ) {
// 			atom_tree_->update_torsions( full_coord_, total_residue_,
// 																	 res_, res_variant_ );
// 		} else {
// 			// update chi-torsions
// 			update_chi();
// 		}

		//this->dump_pdb("after_rotamer_trials.pdb");

		// declare that some rotamers have changed
		// not sure we have to do this, since rotamer_trials updates
		// score arrays like atr_pair and intraresenergy ...
		// OK - not doing this right now, since we dont know which
		// residues changed anyhow...


		// undo hijack of get_res_res_cstE:
// 		if ( this->constraints_exist() ) {
// 			unset_pose_constraints();
// 		}

	}

	/////////////////////////////////////////////////////////////////////////////
	// takes sidechain atoms, transformed so that backbones overlay,
	// and copies into full_coord array
	//
	// currently, this leaves the chi array out of sync, which is bad
	//
	// it should be the case that the backbone geometry is the same
	// it appears that the C-alphas will be superimposed by the GL matrix
	// which at least should prevent bad things...

	void
	Pose::copy_sidechain(
		const int seqpos,
		int const aa, // could be the same as the old aa
		int const aav,
		FArray2Da_float fcoord,
		bool keep_bb,  ///jjh defaults to true, set to false to copy entire res.
		bool align_bb // = true
	)
	{
		fcoord.dimension(3,param::MAX_ATOM()());
		using aaproperties_pack::HNpos;

		int const old_aa ( res_        (seqpos) );
		int const old_aav( res_variant_(seqpos) );
		bool const bb_move( get_allow_bb_move( seqpos ) );
		bool const chi_move( get_allow_chi_move( seqpos ) );
		float const current_phi( phi( seqpos ) );

		refold(); // ensure that full_coord_ is synced with Eposition

		bool const sequence_change( old_aa != aa || old_aav != aav );
		bool const rebuild_HN( aa != param_aa::aa_pro && param_aa::is_protein(aa)&&
													 HNpos( old_aa, old_aav ) != HNpos( aa, aav ) );
		if ( sequence_change ) {
// 			std::cout << "[DEBUG] sequence change: " << seqpos << " from " <<
// 				param_aa::aa_name1( old_aa ) << " " << old_aav << " to " <<
// 				param_aa::aa_name1( aa ) << " " << aav << std::endl;

			// update res,res_variant
			res_        ( seqpos ) = aa;
			res_variant_( seqpos ) = aav;
		}

		int const natoms( aaproperties_pack::natoms(aa,aav));

		if ( param_aa::is_ligand( aa ) ) {
			// work-around
			assert( !sequence_change );
			for ( int j=1; j<= natoms; ++j ) {
				for ( int k=1; k<= 3; ++k ) {
					full_coord_( k, j, seqpos ) = fcoord( k, j );
				}
			}


			// this is short-term hack, will be slow
			// add ligand-safe routines to atom_tree in future
			atom_tree_->update_torsions( full_coord_, total_residue_,
																	 res_, res_variant_ );

			// notify that domain_map / current scores are out of date
			domain_map_updated = false;
			score_data.after_refolding();

			return;
		}



		if ( keep_bb ) {
			FArray2D_float Mgl( 4, 4 );
			if ( param_aa::is_protein( aa ) && param_aa::is_protein( old_aa ) ) {

				get_GL_matrix( fcoord( 1, 1 ), // N
											 fcoord( 1, 2 ), // CA
											 fcoord( 1, 3 ), // C
											 full_coord_( 1, 1, seqpos ), // N
											 full_coord_( 1, 2, seqpos ), // CA
											 full_coord_( 1, 3, seqpos ), // C
											 Mgl);

				const int HN_pos ( HNpos (aa,aav));
				for ( int j=1; j<= natoms; ++j ) {
					if ( j >= 5 && j != HN_pos ) {
						if (align_bb) {
							GL_rot( Mgl, fcoord( 1, j ), full_coord_( 1, j, seqpos ) );
						} else {
							// just copy from fcoord, ignoring GL transformation
							for ( int k=1; k<= 3; ++k ) {
								full_coord_(k,j,seqpos) = fcoord(k,j);
							}
						}
					}
				}
			} else if ( param_aa::is_NA( aa ) && param_aa::is_NA( old_aa ) ) {
				// this will preserve the second point and the vector
				// from the second point to the third point:
				using aaproperties_pack::na_anchor;
				using aaproperties_pack::atom_name;
				int const
					old_na_anchor1( na_anchor( 1, old_aa, old_aav ) ),
					old_na_anchor2( na_anchor( 2, old_aa, old_aav ) ),
					old_na_anchor3( na_anchor( 3, old_aa, old_aav ) );
				get_GL_matrix( fcoord( 1, na_anchor( 3, aa, aav ) ),
											 fcoord( 1, na_anchor( 1, aa, aav ) ),
											 fcoord( 1, na_anchor( 2, aa, aav ) ),
											 full_coord_( 1, old_na_anchor3, seqpos ),
											 full_coord_( 1, old_na_anchor1, seqpos ),
											 full_coord_( 1, old_na_anchor2, seqpos ),
											 Mgl);
				// save old coords since some atoms we are copying have different
				// atom numbers
				int const natoms_old( aaproperties_pack::natoms( old_aa, old_aav ) );
				FArray2D_float old_coord(3,param::MAX_ATOM()());
				for ( int j=1; j<= natoms_old; ++j ) {
					for ( int k=1; k<= 3; ++k ) {
						old_coord(k,j) = full_coord_(k,j,seqpos);
					}
				}

				// now fill in the full_coord_ array for seqpos
				// either by transforming from the input coords
				// or by copying from the saved coords
				for ( int j=1; j<= natoms; ++j ) {
					if ( is_NA_backbone_atom( j, aa, aav ) ) {
						// copy from old_coord
						int const atomno
							( LookupByName( old_aa, old_aav, atom_name( j,aa,aav) ) );
						for ( int k=1; k<= 3; ++k ) {
							full_coord_(k,j,seqpos) = old_coord(k,atomno);
						}
					} else {
						if (align_bb) {
							// transform from fcoord (input coords)
							GL_rot( Mgl, fcoord( 1, j ), full_coord_( 1, j, seqpos ) );
						} else {
							// just copy from old_coord, ignoring GL transformation
							for ( int k=1; k<= 3; ++k ) {
								full_coord_(k,j,seqpos) = old_coord(k,j);
							}
						}
					}
				}
			} else {
				std::cout << "dont know how to handle this aa change! " <<
					old_aa << ' ' << aa << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		} else {
			// keep_bb == false ==> moving everything!
			for ( int j=1; j<= natoms; ++j ) {
				for ( int k=1; k<= 3; ++k ) {
					full_coord_( k, j, seqpos ) = fcoord( k, j );
				}
			}
		}

		if ( sequence_change || !keep_bb ) {
			// have to update Eposition!
			for ( int k=1; k<= 3; ++k ) {
				Eposition_(k,1,seqpos) = full_coord_(k,1,seqpos);
				Eposition_(k,2,seqpos) = full_coord_(k,2,seqpos);
				Eposition_(k,4,seqpos) = full_coord_(k,3,seqpos);
				Eposition_(k,5,seqpos) = full_coord_(k,4,seqpos);
				Eposition_(k,3,seqpos) = full_coord_(k,5,seqpos);
			}
		}

		if ( sequence_change ) {
			// rebuild the centroid -- uses Eposition if protein, else full_coord_
			rebuild_centroid( seqpos );
		}

		if ( rebuild_HN ) {
			// rebuild HN
			build_nh_simple( aa, aav, current_phi, full_coord_(1,1,seqpos) );
		}

		if ( atom_tree_ ) {
			/////////////////////////////////////////////////////////////////////////
			// update Atom_tree
			if ( sequence_change || !keep_bb ) {
				atom_tree_->new_aa_and_coords( seqpos, old_aa, old_aav, aa, aav,
																			 full_coord_ );
			} else {
				atom_tree_->new_sidechain_coords( seqpos, aa, aav, full_coord_ );
			}

			// restore allow-move settings -- these may not have been preserved
			// the atom_tree operations... alas
			set_allow_bb_move ( seqpos,  bb_move );
			set_allow_chi_move( seqpos, chi_move );

		} else {
			/////////////////////////////////////////////////////////////////////////
			// set the new chi angles
			assert( keep_bb );
			update_chi_residue( seqpos );
			new_chi_score( seqpos ) = true;
		}


		// notify that domain_map / current scores are out of date
		domain_map_updated = false;
		score_data.after_refolding();

	}


	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	// this could be faster
	//
	// tricky if we have things like constraints!!
	//
	void
	Pose::delete_segment(
		int const seg_begin, // inclusive
		int const seg_end
	)
	{
		// debugging
		int cutpoint(-1);
		for ( int i=seg_begin; i<= seg_end; ++i ) {
			if ( fold_tree_.get_is_jump_point()(i) ) {
				std::cout << "STOP:: Pose::delete_segment( " << seg_begin << " , " <<
					seg_end << "): cant delete segments that contain jump points.\n" <<
					"Residue " << i << " is a jump_point." <<	std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			if ( fold_tree_.get_is_cutpoint()(i) ) cutpoint = i;
		}


		// create temporary pose
		Pose tmp_pose;

		Fold_tree f( fold_tree_ );
		if ( cutpoint >= seg_begin && cutpoint < seg_end ) {
			f.delete_segment( cutpoint+1, seg_end );
			f.delete_segment( seg_begin, cutpoint );
		} else {
			f.delete_segment( seg_begin, seg_end );
		}

		tmp_pose.set_fold_tree( f );
		tmp_pose.set_fullatom_flag( fullatom_, false );

		int size( seg_begin -1 );
		if ( size > 0 ) tmp_pose.copy_segment( size, *this, 1, 1 );

		size = total_residue_ - seg_end;
		if ( size > 0 ) tmp_pose.copy_segment( size, *this, seg_begin,
																					 seg_end + 1 );

		set_fold_tree( f );
		assert( total_residue_ == tmp_pose.total_residue() );
		copy_segment( total_residue_, tmp_pose, 1, 1 );

		///////////////////////////////
		// have to renumber stuff here:
		assert( pose_constraints_ == 0 );

	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::copy_segment(
		int const size,
		Pose const & src,
		int const begin,
		int const src_begin,
		bool const update_jumps // = true
	)
	{
		assert( !atom_tree_ ); //ja need to make this accept an atom tree..
		assert( !symmetric() ); // pb -- short term

		//update_dimensions();
		int const src_total_residue( src.total_residue() );
		bool const src_fullatom ( src.fullatom() );
		assert( size+begin-1 <= total_residue_ &&
						 size+src_begin-1 <= src_total_residue );
/*		if ( fullatom_ != src_fullatom ) {							//commented out by Sid for ensemble moves
			std::cout << "WARNING Pose::copy_segment: fullatom == " << fullatom_ <<
				" and src.fullatom() == " << src_fullatom << std::endl;
		}*/
		// fill in PDB information
		pdb_info_.copy_segment(size, src.pdb_info(), begin, src_begin);

		// bb-torsions, ss, res
		for ( int i=0; i< size; ++i ) {
			int const pos ( begin+i );
			int const src_pos ( src_begin+i );

			set_phi         ( pos, src.phi         ( src_pos ) );
			set_psi         ( pos, src.psi         ( src_pos ) );
			set_omega       ( pos, src.omega       ( src_pos ) );
			set_secstruct   ( pos, src.secstruct   ( src_pos ) );
			set_name        ( pos, src.name        ( src_pos ) );
			set_res         ( pos, src.res         ( src_pos ) );
			set_res_variant ( pos, src.res_variant ( src_pos ) );
			new_bb_score    ( pos ) = true;
			new_bb_refold   ( pos ) = true;
			if ( fullatom_ && src_fullatom ) {
				for ( int k=1; k<= param::MAX_CHI; ++k ) {
					set_chi( k, pos, src.chi( k, src_pos ) );
				}
			}
		}
		new_torsions = true;

		if ( !src.ideal_backbone() ) {
			if ( !pose_bonds_ ) {
				// allocate bonds object, fill with ideal values
				pose_bonds_ = new bonds_class::Bonds( total_residue_ );
				new_bb_refold = true; // slight change to structure on refolding
			}
			pose_bonds_->copy_segment( size, src.bonds(), begin, src_begin );
		} else if ( pose_bonds_ ) {
			pose_bonds_->idealize( begin, begin+size-1 );
		}


		///////////
		// copy coords:
		// Note that these coordinate access calls may trigger a refold
		// of the src pose.
		//
		int offset,src_offset,l_end;
		// Eposition
		FArray3D_float const & src_Epos ( src.Eposition() );
		offset = Eposition_.index(1,1,begin);
		src_offset = src_Epos.index(1,1,src_begin);
		l_end = 3 * param::MAX_POS * size;
		for ( int l=0; l<l_end; ++l ) {
			Eposition_[l+offset] = src_Epos[l+src_offset];
		}
		new_Eposition();

		// centroid
		FArray2D_float const & src_cen ( src.centroid() );
		offset = centroid_.index(1,begin);
		src_offset = src_cen.index(1,src_begin);
		l_end = 3 * size;
		for ( int l=0; l<l_end; ++l ) {
			centroid_[l+offset] = src_cen[l+src_offset];
		}
		// fullcoord
		if ( fullatom_ && src_fullatom ) {
			FArray3D_float const & src_fcoord ( src.full_coord() );
			offset = full_coord_.index(1,1,begin);
			src_offset = src_fcoord.index(1,1,src_begin);
			l_end = 3 * param::MAX_ATOM()() * size;
			for ( int l=0; l<l_end; ++l ) {
				full_coord_[l+offset] = src_fcoord[l+src_offset];
			}
		}
		////////////////////////////////////////////////////////////
		// update any jumps that are affected by this copy operation
		if ( update_jumps ) {
			FArray2D_int const & jump_point( fold_tree_.get_jump_point() );
			for ( int i=1; i<= num_jump_; ++i ) {
				int const pos1( jump_point(1,i));
				int const pos2( jump_point(2,i));
				if ( pos1 >= begin && pos1 < begin + size ||
						 pos2 >= begin && pos2 < begin + size ) {
					if ( backbone_coords_init( pos1 ) &&
							 backbone_coords_init( pos2 ) ) {
						update_jump_from_coords( i );
					} else {
						//std::cout << "cant update jump_transform for jump " <<
						//	"between " << pos1 << " and " << pos2 << std::endl;
					}
				}
			}
		}


		// hack for DNA copy:
		if ( size == total_residue_ ) {

			// check for non-protein, trigger atom-tree setup
			bool all_protein( true );
			for ( int i=1; i<= total_residue_; ++i ) {
				if ( !param_aa::is_protein( res(i) ) ) {
					all_protein = false;
					break;
				}
			}
			if ( !all_protein ) {
				std::cout << "Non protein atoms detected, using Atom_tree object" <<
					std::endl;
				setup_atom_tree();
				return;
			}
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::simple_fold_tree(
		int const nres
	)
	{
		Fold_tree f;
		f.simple_tree( nres );

		set_fold_tree( f );
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::fold_in_rb_deltas(
	)
	{
		if ( atom_tree_ ) {
			for ( int i=1; i<= num_jump_; ++i ) {
				atom_tree_->jump( fold_tree_.downstream_jump_residue(i) ).\
					fold_in_rb_deltas();
			}
			return;
		}
		for ( int i=1; i<= num_jump_; ++i ) {
			// dont check allow-move
			jump_transform(i).fold_in_rb_deltas();
		}

	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::one_jump_tree(
		int const nres,
		int const jump_pos1,
		int const jump_pos2,
		int const cutpoint,
		int const root
	)
	{
		assert( jump_pos1 >= 1 &&
						jump_pos1 <= jump_pos2 &&
						jump_pos1 <= cutpoint &&
						jump_pos2 >= cutpoint+1 &&
						jump_pos2 <= nres );

		Fold_tree f;

		f.clear();
		f.add_edge( jump_pos1, jump_pos2, 1 );
		f.add_edge(1,jump_pos1, pose_param::PEPTIDE );
		f.add_edge(jump_pos1,cutpoint, pose_param::PEPTIDE );
		f.add_edge(cutpoint+1,jump_pos2, pose_param::PEPTIDE);
		f.add_edge(jump_pos2,nres, pose_param::PEPTIDE);
		f.reorder(root);

		set_fold_tree(f);
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::main_minimize(
		const Score_weight_map & wt_map,
		std::string const & min_type
	)
	{
		//refold(); // why do we need this?
		// setup the scoring function weights
		// since we are just passing in the generic routine "scorefxn" to the minimizer
		score_set_weight_map_weights( wt_map );

		//score( wt_map );
		//float const start_score ( get_0D_score( SCORE ) );
		int const dummy1(0), dummy2(0);
		bool gfrag( true );
		minimize_set_current_pose( *this );
		minimize( min_type, "minimize", scorefxn, dummy1, dummy2, gfrag );
		if ( !gfrag ) {
			std::cout << "gfrag failed during minimization!!" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		minimize_reset_current_pose();
		assert( misc_in_sync( *this ) );
	}


	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::dump_pdb(
		std::string const & filename
	) const
	{
		std::cout << "Pose::dump_pdb: " << filename << "\n";
		utility::io::ozstream out( filename.c_str() );
		if (!out.good() ) {
			std::cout << "STOP: cant open file: " << filename << "\n";
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		if ( fullatom_ ) {
			dump_full_coord( *this, out );
		} else {
			dump_Eposition( *this, out );
		}

		out << show_scores() << '\n';

		output_torsions( *this, out );
		out.close();
	}

	/////////////////////////////////////////////////////////////////////////////
	/// @begin Pose::pose_to_design_output
	///
	/// @brief
	/// Calls conventional design_output() from Pose
	///
	/// @authors
	/// ashworth
	///
	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::dump_scored_pdb(
		std::string const & filename,
		Score_weight_map const & w
	)
	{

		utility::io::ozstream out;
		open_scored_pdb_outfile( filename, out );
		dump_scored_pdb(out, w);
		out.close();

	}
/*
	void
	Pose::dump_scored_bsasa_pdb(
		std::string const & filename,
		Score_weight_map const & w
	)
	{
		using namespace packing_ns;
		copy_to_misc();
		ProteinSasa ps;
		ps.compute_atom_bsasa_score();
		float bsasaWLN = ps.bsasa_score_weighted_log();

		utility::io::ozstream out;
		open_scored_pdb_outfile( filename, out );
		dump_scored_pdb(out, w);
		out << "bsasaWLN: " << bsasaWLN << std::endl;
		out.close();

	}
*/
	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::open_scored_pdb_outfile(
    std::string const & filename,
		utility::io::ozstream & out )
	{

		std::cout << "Pose::dump_scored_pdb: " << filename << "\n";
		out.clear();
		out.open( filename.c_str() );
		if (!out.good() ) {
			std::cout << "STOP: cant open file: " << filename << "\n";
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::dump_scored_pdb( utility::io::ozstream & out, Score_weight_map const & w )
	{
		if ( fullatom_ ) {
			//assert( w.get_weight( FA_SCOREFXN ) );
			score_set_evaluate_all_terms( true );
			bool const enabled( score_get_try_rotamers() );
			score_enable_rotamer_trials(false);
			this->score( w );
			score_enable_rotamer_trials(enabled);
			score_set_evaluate_all_terms( false );

			dump_full_coord( *this, out );

			score_set_current_pose( *this );
			output_pdb_stats( out, true );
			score_reset_current_pose();

		} else {
			assert( false );
			//dump_Eposition( *this, out );
		}

		out << show_scores() << '\n';

		output_torsions( *this, out );
	}


	void
	Pose::pose_to_design_output( int const run )
	{
		design_output( total_residue_, res_, res_variant_, full_coord_, run );
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::set_extra_score(
		std::string const & name,
		float const value
	)
	{
		extra_scores[name] = value;
	}

	/////////////////////////////////////////////////////////////////////////////
	float
	Pose::get_extra_score(
		std::string const & name
	)
	{
		float extra_score( 0.0 );
		if ( has_extra_score( name ))
			extra_score =  extra_scores[name];
		return extra_score;
	}

	/////////////////////////////////////////////////////////////////////////////
	bool
	Pose::has_extra_score(
		std::string const & name
	)
	{
		if (extra_scores.count( name ) > 0)
			return true;
		return false;
	}

	/////////////////////////////////////////////////////////////////////////////
	std::string
	Pose::show_scores() const
	{
		std::ostringstream os;
		show_scores( os );
		return os.str();
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::show_scores(
		std::ostream & os
	) const
	{
		score_data.show_scores(os,0);
//	score_data.show_scores(os,3); //ja easier to read
		for ( std::map< std::string, float >::const_iterator it=extra_scores.begin(),
						it_end=extra_scores.end(); it != it_end; ++it ) {
			os << it->first << ' ' << it->second << ' ';
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::show_score_values(
		std::ostream & os
	) const
	{
		score_data.show_scores(os,1);
		for ( std::map< std::string, float >::const_iterator it=extra_scores.begin(),
						it_end=extra_scores.end(); it != it_end; ++it ) {
			os << F( 9, 4, it->second ) << ' ';
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::show_score_tags(
		std::ostream & os
	) const
	{
		score_data.show_scores(os,2);
		for ( std::map< std::string, float >::const_iterator it=extra_scores.begin(),
						it_end=extra_scores.end(); it != it_end; ++it ) {
			os << A( 9, it->first ) << ' ';
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	std::string
	Pose::sequence() const
	{
		std::string seq;

		for ( int i=1; i<= total_residue_; ++i ) {
			int const aa( res_(i) );
			if ( aa >= 1 && aa <= param::MAX_AA() ) {
				seq += param_aa::aa_name1( aa );
			} else {
				seq += 'X';
			}
		}
		return seq;
	}


	/////////////////////////////////////////////////////////////////////////////
	// NOTE: still assumes that one side-effect of refold is copying into
	// the misc arrays
	//
	void
	Pose::copy_to_misc() const
	{
		if ( new_torsions ) {
			refold();

			//////////////////////////////////////////
			//yab/rhiju This is ugly...
			misc_removal::init_modules_TEMPORARY();
			/////////////////////////////////////////

		} else {
			pose_to_misc( *this );
		}
		assert( misc_in_sync( *this ) );
	}

	/////////////////////////////////////////////////////////////////////////////
	// subtle point: this recenters the jump to rotate around
	// this new position in both directions: ie if the fold-tree
	// changes so that the jump is being folded in the opposite
	// direction eg from c2n rather than n2c, then the center will
	// be the same. By default, the center of rotation is the
	// C-alpha of the downstream jump residue, which changes
	// depending on the direction the jump is folded!

	void
	Pose::set_jump_rb_center(
		int const jump_number,
		numeric::xyzVector_double & center
	)
	{
	//	assert( !atom_tree_ );

		refold(); // need Eposition up to date

// 		if ( !allow_jump_move( jump_number ) ) {
// 			std::cout << "set_jump_rb_center called for non-moving jump" <<
// 				std::endl;
// 			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
// 		}

		//debug:
		assert( jump_number <= num_jump_ );

		// get jump residues:
		FArray2D_int const & jump_point( fold_tree_.get_jump_point() );
		int const pos1( jump_point(1,jump_number) );
		int const pos2( jump_point(2,jump_number) );

		// sanity check:
		assert( pos1 < pos2 );

		// get handle on jump:
		Jump & jump( jump_transform( jump_number ) );

		// note the residues: pos2 is the downstream residue for
		// n2c folding, while pos1 is the downstream residue for
		// c2n folding.
		int const n2c(1), c2n(-1);
		jump.set_rb_center( n2c, Eposition_(1,1,pos2), center);
		jump.set_rb_center( c2n, Eposition_(1,1,pos1), center);
	}


	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::fold_from_here()
	{
		assert( !atom_tree_ );

		// for minimization, to prevent coords from getting slowly
		// warped by multiple copy operations
		// refolding_pose is a copy of the pose when we called fold_from_here
		// with the torsions continually updated but no calls to refold
		if ( refolding_pose == 0 ) {
			refolding_pose = new Pose;
		}
		// dont copy score data, constraints
		// and especially: want refolding_pose.fold_from_current == true !
		refolding_pose->copy_coords( *this, false );
		refold_from_current = false;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::fold_from_current()
	{
		// this is the default behavior
		refold_from_current = true;
		if (!new_torsions ) {
			new_bb_refold = false;
			new_jump_refold = false;
			if ( fullatom_ ) new_chi_refold = false;
		}
	}



	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::reattach_atom(
		kin::Atom_id const & root_id,
		kin::Atom_id const & anchor_id
	)
	{
		using namespace kin;
		assert( atom_tree_ );
		if ( new_torsions ) refold();

		Atom* root  ( atom_tree_->atom(   root_id.atomno(),   root_id.rsd()));
		Atom* anchor( atom_tree_->atom( anchor_id.atomno(), anchor_id.rsd()));

		root->parent->delete_atom( root );
		root->parent = anchor;
		anchor->add_atom( root );
		atom_tree_->update_torsions( full_coord_, total_residue_, res_,
																 res_variant_ );
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::setup_symm_info(
		int const nres_monomer,
		int const njump_monomer,
		int const N,
		std::string const & type
	)
	{
		this->new_score_pose();
		if ( symm_info ) {
			delete symm_info;
		}
		symm_info = new Symmetry_info( nres_monomer, njump_monomer, N, type );
	}


  /////////////////////////////////////////////////////////////////////////////
	void
  Pose::setup_symm_info(
      int const nres_monomer,
      int const njump_monomer,
      int const N,
      int score_monomer,
      std::string const & type
    )
		{
			this->new_score_pose();
			if ( symm_info ) {
				delete symm_info;
			}
    symm_info = new Symmetry_info( nres_monomer, njump_monomer, N, score_monomer, type );

	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::setup_symm_info(
		Symmetry_info const & s
	)
	{
		this->new_score_pose();
		if ( symm_info ) {
			delete symm_info;
		}
		symm_info = new Symmetry_info( s );
	}


	/////////////////////////////////////////////////////////////////////////////
	// attaches the ligand at the end, ie the sequence number for the ligand
	// will be nres+1 where nres is the current number of residues
	//
	void
	Pose::attach_ligand(
		int const lig_aa,
		int const lig_aav,
		int const anchor_atomno,
		int const anchor_rsd,
		int const lig_atomno,
		FArray2Da_float lig_coord,
		bool const attach_by_jump
	)
	{
		assert( fullatom_ );
		Pose tmp_pose;
		tmp_pose = *this;
		//std::cout<<"DEBUG:attach_ligand "<<lig_aa<<" "<<lig_aav<<" "<<anchor_atomno<<" "<<anchor_rsd<<" "<<lig_atomno<<" "<<attach_by_jump<<std::endl;

		// a little tricky -- have to set up a new tree
		// either we are attaching the ligand by a jump or by a bond
		lig_coord.dimension( 3, param::MAX_ATOM() );

		int const nres( total_residue_ + 1 );

		{ // setup new fold_tree
			Fold_tree f( fold_tree_ );
			if( attach_by_jump ) {
				f.add_edge( nres-1, nres, pose_param::PEPTIDE );//tell fold tree nres is increased
				f.new_jump( anchor_rsd, nres, nres-1 );//it will erase the the above edge
				//f.add_edge( anchor_rsd, nres, num_jump_+1 );
			} else {
				f.add_edge( nres-1, nres, pose_param::PEPTIDE );
				f.delete_extra_vertices(); // remove non-jump vertices, prob not neces
			}
			std::cout << "new fold_tree: " << f << std::endl;
			this->set_fold_tree( f );
		}

		// copy the old data up to nres-1
		copy_segment( nres -1, tmp_pose, 1, 1 );

		this->set_res( nres, lig_aa );
		this->set_res_variant( nres, lig_aav );
		//lin update the pdb_info for ligand
		this->pdb_info_.set_pdb_res( nres, nres );
		this->pdb_info_.set_pdb_chain( nres, ' ' );
		this->pdb_info_.set_pdb_insert_let( nres, ' ' );

		for ( int i=1; i<= aaproperties_pack::natoms(lig_aa,lig_aav); ++i ) {
			for ( int k=1; k<= 3; ++k ) {
				full_coord_(k,i,nres) = lig_coord(k,i);
			}
		}

		// this determines where the ligand gets attached during atom_tree
		// building inside kin_util
		set_default_root_atomno( lig_aa, lig_atomno );

		// now build the atom_tree
		setup_atom_tree();

		// update coords --  will copy to Eposition, build centroid, etc
		atom_tree_refold();

		// rewire ligand placement:

		// delete current attachment
		kin::Atom* const ligand_root( atom_tree_->atom( lig_atomno, nres ) );
		ligand_root->parent->delete_atom( ligand_root );

		// new attachment
		kin::Atom * const new_anchor
			( atom_tree_->atom(anchor_atomno,anchor_rsd));
		new_anchor->add_atom( ligand_root );
		ligand_root->parent = new_anchor;

		//chu 2008/01/16 save fold_tree jump_atoms info
		if ( attach_by_jump) {
			assert( ( anchor_rsd == fold_tree_.upstream_jump_residue(num_jump_) )
				&& (nres == fold_tree_.downstream_jump_residue(num_jump_) ) );
			fold_tree_.set_jump_atoms(num_jump_, anchor_atomno, lig_atomno);
		}
		// debugging:
		//ligand_root->show();

		// recalculate torsions
		std::cout << "new update torsions:" << std::endl;

		atom_tree_->update_torsions( full_coord_, total_residue_, res_,
																 res_variant_ );

		// ensure full score calc'n
		this->new_score_pose();
	}


	/////////////////////////////////////////////////////////////////////////////
	// forget all of the internal score info
	void
	Pose::new_score_pose()
	{
		domain_map_updated = false;
		if ( atom_tree_ ) {
			atom_tree_->new_score_pose();
		}
		new_bb_score = true;
	}

	/////////////////////////////////////////////////////////////////////////////
	// recenter coords with seqpos in standard position
	//
	void
	Pose::center(
							 int const seqpos
							 )
	{
		refold();

		FArray2D_float xyz(3,3,0.0); // CA at (0,0,0)
		xyz(1,1) = 1.0; // N at (1,0,0)
		xyz(2,3) = 1.0; // C at (0,1,0)

		FArray2D_float Mgl( 4, 4 );
		// aligns CA and CA->N vector
		get_GL_matrix( Eposition_(1,4,seqpos), // C
									 Eposition_(1,2,seqpos), // CA
									 Eposition_(1,1,seqpos), // N
 									 xyz(1,3),
 									 xyz(1,2),
 									 xyz(1,1),
									 Mgl );

		transform_GL( Mgl );
		assert( std::abs( Eposition_(1,2,seqpos)) < 1e-2 &&
						std::abs( Eposition_(2,2,seqpos)) < 1e-2 &&
						std::abs( Eposition_(3,2,seqpos)) < 1e-2 &&
						std::abs( Eposition_(2,1,seqpos)) < 1e-2 &&
						std::abs( Eposition_(3,1,seqpos)) < 1e-2 &&
						std::abs( Eposition_(3,4,seqpos)) < 1e-2 );
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::transform_GL(
		FArray2D_float const & Mgl
	)
	{
		for ( int i = 1; i<= total_residue_; ++i ) {
			int const aa ( res_(i));
			int const aav( res_variant_(i));
			for ( int j=1; j<= param::MAX_POS; ++j ) {
				GL_rot_in_place( Mgl, Eposition_(1,j,i) );
			}
			GL_rot_in_place( Mgl, centroid_(1,i) );
			if ( fullatom_ ) {
				for ( int j=1; j<= aaproperties_pack::natoms(aa,aav); ++j ) {
					GL_rot_in_place( Mgl, full_coord_(1,j,i));
				}
			}
		}

		new_Eposition();
	}


	/////////////////////////////////////////////////////////////////////////////
	void
	Pose::transform_Ax_plus_b(
														numeric::xyzMatrix_float const & A,
														numeric::xyzVector_float const & b
	)
	{
		typedef numeric::xyzVector_float Vec;
		for ( int i = 1; i<= total_residue_; ++i ) {
			int const aa ( res_(i));
			int const aav( res_variant_(i));
			for ( int j=1; j<= param::MAX_POS; ++j ) {
				Vec v( &Eposition_(1,j,i) );
				v = A * v + b;
				for ( int k=1; k<= 3; ++k ) Eposition_(k,j,i) = v(k);
			}
			Vec v( &centroid_(1,i) );
			v = A * v + b;
			for ( int k=1; k<= 3; ++k ) centroid_(k,i) = v(k);

			if ( fullatom_ ) {
				for ( int j=1; j<= aaproperties_pack::natoms(aa,aav); ++j ) {
					Vec v( &full_coord_(1,j,i) );
					v = A * v + b;
					for ( int k=1; k<= 3; ++k ) full_coord_(k,j,i) = v(k);
				}
			}
		}

		new_Eposition();
	}


	/////////////////////////////////////////////////////////////////////////////
	// just sets res, not resv
	void
	Pose::set_sequence(
		std::string const & seq
	)
	{
		int const nres_seq( seq.size() );
		if ( total_residue_ != nres_seq ) {
			std::cout << "need to setup fold_tree before calling set_sequence !\n" <<
				"Total_residue does not match the length of the sequence: " <<
				total_residue_ << ' ' << seq.size() << std::endl;
			std::exit( EXIT_FAILURE );
		}

		for ( int i=0; i< nres_seq; ++i ) {
			this->set_res( i+1, aa2int( seq[i] ) );
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	// debugging
	/////////////////////////////////////////////////////////////////////////////
	bool
	Pose::debug_torsions() const
	{
		std::cout << "Phil -- put something here: debug_torsions" << std::endl;
		return true;
	}

	//////////////////////////////////////////////////////
	float
	Pose::get_distance(
	 kin::Atom_id const & a1,
	 kin::Atom_id const & a2 ) const {

		typedef numeric::xyzVector_float Vec;

		kin::Coords_FArray_const coords( full_coord() );
		numeric::xyzVector_float p1 ( coords.get_xyz(a1) );
		numeric::xyzVector_float p2 ( coords.get_xyz(a2) );
		Vec v1( p1-p2 );

		return v1.length() ;
	}

	//////////////////////////////////////////////////////
	float
	Pose::get_angle(
	 kin::Atom_id const & a1,
	 kin::Atom_id const & a2,
	 kin::Atom_id const & a3 ) const {

		kin::Coords_FArray_const coords( full_coord() );
		numeric::xyzVector_float p1 ( coords.get_xyz(a1) );
		numeric::xyzVector_float p2 ( coords.get_xyz(a2) );
		numeric::xyzVector_float p3 ( coords.get_xyz(a3) );
		numeric::xyzVector_float u1 ( p1 - p2 );
		numeric::xyzVector_float u2 ( p3 - p2 );
		float const n1( u1.length() );
		float const n2( u2.length() );
		if ( n1 > 1e-12 && n2 > 1e-12 ) {
			return std::acos( numeric::sin_cos_range( dot( u1,u2 ) / ( n1 * n2 ) ) );
		}
		return 0.0;
	}

	//////////////////////////////////////////////////////
	float
	Pose::get_torsion_angle(
	 kin::Atom_id const & a1,
	 kin::Atom_id const & a2,
	 kin::Atom_id const & a3,
	 kin::Atom_id const & a4 ) const {

		float const deg2rad( numeric::conversions::radians(1.0) );

		kin::Coords_FArray_const coords( full_coord() );
		numeric::xyzVector_float p1 ( coords.get_xyz(a1) );
		numeric::xyzVector_float p2 ( coords.get_xyz(a2) );
		numeric::xyzVector_float p3 ( coords.get_xyz(a3) );
		numeric::xyzVector_float p4 ( coords.get_xyz(a4) );

		return deg2rad * dihedral( p1, p2, p3, p4 );
	}

	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////

	//////////////////////////////////////
	// Monte_carlo
	//////////////////////////////////////

	/////////////////////////////////////////////////////////////////////////////
	// constructor for monte_carlo object
	Monte_carlo::Monte_carlo(
		Pose const & init_pose,
		Score_weight_map const & weight_map,
		float const temperature
	)
	{
		weight_map_ = weight_map;

		temperature_ = temperature;

		autotemp_ = false;
		save_mc_accepted_ = true;

		reset( init_pose );

		// for checkpointing
		do_checkpointing = get_do_pose_checkpointing();
		if (do_checkpointing) {
			checkpointfile = main_job_distributor::jd->get_mc_checkpoint_file();
			currposeoutfile = checkpointfile + "_current.out";
			lowposeoutfile = checkpointfile + "_low.out";
			bestposeoutfile = checkpointfile + "_best.out";
			mc_checkpoint_isfullatom = 0;
			mc_checkpoint_ntrials = 0;
			restore_checkpoint_counts();
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Monte_carlo::save_mc_accepted()
	{
		save_mc_accepted_ = true;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Monte_carlo::set_temperature(
		float const temp
	)
	{
		temperature_ = temp;
	}

	/////////////////////////////////////////////////////////////////////////////
	Score_weight_map const &
	Monte_carlo::weight_map() const
	{
		return weight_map_;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Monte_carlo::reset(
		Pose const & pose
	)
	{
		best_pose_ = pose;
		best_pose_.score( weight_map_ );
		low_pose_ = best_pose_;
#ifdef GL_GRAPHICS
		update_monte_carlo_globals(3);
		gl_graphics_mc_reset( low_score(), 0.0 );
		//gl_graphics_reset_trajectory( low_score() );
#endif
#ifdef BOINC_GRAPHICS
		update_monte_carlo_globals(3);
#endif

	}

	/////////////////////////////////////////////////////////////////////////////
	Pose const &
	Monte_carlo::best_pose(
  ) const
	{
		return best_pose_;
	}

	/////////////////////////////////////////////////////////////////////////////
	Pose const &
	Monte_carlo::low_pose(
  ) const
	{
		return low_pose_;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Monte_carlo::set_weight_map(
		Score_weight_map const & weight_map
	)
	{
		weight_map_ = weight_map;

		bool const save_enable( score_get_try_rotamers() );
		score_enable_rotamer_trials( false );
		best_pose_.score( weight_map_ );
#ifdef GL_GRAPHICS
		// assumes we just scored the best_pose
		gl_graphics_save_best_scores();
#endif

		low_pose_.score( weight_map_ );
		score_enable_rotamer_trials( save_enable );

#ifdef GL_GRAPHICS
		// assumes we just scored the low_pose
		gl_graphics_mc_new_scorefxn( best_score(), low_score(), calc_rms( best_pose_ ) );
#endif

	}

	/////////////////////////////////////////////////////////////////////////////
	float Monte_carlo::last_score() const
	{
		return last_score_;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Monte_carlo::recover_low(
		Pose & pose
	)
	{
		pose = low_pose_;
		best_pose_ = low_pose_;

#ifdef GL_GRAPHICS
		update_monte_carlo_globals( 2 /* best accept */ );
		gl_graphics_mc_recover_low( low_score(), calc_rms(low_pose_) );
#endif
	}


	/////////////////////////////////////////////////////////////////////////////
	void Monte_carlo::show_scores() const
	{
		std::cout << "Monte_carlo:: best_score,low_score: " <<
			best_score() << ' ' << low_score() << std::endl;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Monte_carlo::reset_counters()
	{
		trial_counter.clear();
		accept_counter.clear();
		energy_drop_counter.clear();
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Monte_carlo::store_counters_in_pose( Pose & pose )
	{
		for ( std::map< std::string, int >::const_iterator
						it=trial_counter.begin(); it != trial_counter.end(); ++it ) {
			std::string const & tag( it->first );
			int const ntrials( it->second );

			float accept_rate( -1.0 );

			if  (accept_counter.count( tag )){
				int const accepts( accept_counter.find( tag )->second );
				//				float const energy_drop( energy_drop_counter.find( tag )->second );
				accept_rate = float( accepts )/ntrials;
			}

			pose.set_extra_score( tag, accept_rate );
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Monte_carlo::show_counters() const
	{
		for ( std::map< std::string, int >::const_iterator
						it=trial_counter.begin(); it != trial_counter.end(); ++it ) {
			std::string const & tag( it->first );
			int const ntrials( it->second );
			if  (accept_counter.count( tag )){
				int const accepts( accept_counter.find( tag )->second );
				float const energy_drop( energy_drop_counter.find( tag )->second );
				std::cout << A( 16, tag ) <<
					" trials= " << I( 6, ntrials ) <<
					" accepts= " << F( 6, 4, float( accepts )/ntrials ) <<
					" e/trial= " << F( 9, 5, energy_drop / ntrials )<<
				std::endl;
			} else {
				std::cout << A( 16, tag ) <<
					" trials= " << I( 6, ntrials ) <<
					" NO ACCEPTS." << std::endl;
			}
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	void
	Monte_carlo::update_monte_carlo_globals(
																					int const accept_type
																					)
	{
		// copy coords to the global arrays!
		int const nres( best_pose_.total_residue() );
		bool const fullatom( best_pose_.fullatom() );
		FArray3D_float dummy_fcoord;
		{
			FArray3D_float const & fcoord
				( fullatom ? best_pose_.full_coord() : dummy_fcoord );
			copy_coordinates_pose( fullatom, nres, best_pose_.Eposition(),
														 best_pose_.centroid(), fcoord,
														 misc::Ebest_position,
														 misc::best_centroid,
														 misc::best_full_coord );
			for ( int i=1; i<= nres; ++i ) {
				misc::best_res        (i) = best_pose_.res        (i);
				misc::best_res_variant(i) = best_pose_.res_variant(i);
			}
		}
		{
			FArray3D_float const & fcoord
				( fullatom ? low_pose_.full_coord() : dummy_fcoord );
			copy_coordinates_pose( fullatom, nres, low_pose_.Eposition(),
														 low_pose_.centroid(), fcoord,
														 low_pose::Elow_position,
														 low_pose::low_centroid,
														 low_pose::low_full_coord );


			for ( int i=1; i<= nres; ++i ) {
				low_pose::low_res        (i) = low_pose_.res        (i);
				low_pose::low_res_variant(i) = low_pose_.res_variant(i);
			}
		}

		monte_carlo_set_accept(accept_type);

// #ifdef GL_GRAPHICS
// 		if ( accept_type >= 1 ) {
// 			gl_graphics_new_best_pose( best_pose_.total_residue() );
// 		}

// 		if ( accept_type == 3 ) {
// 			gl_graphics_new_low_pose( low_pose_.total_residue() );
// 		}
// #endif

		mc_global_track::mc_score::score      = best_score();
		mc_global_track::mc_score::best_score = best_score();
		mc_global_track::mc_score::low_score  = low_score();

	}

//////////////////////////////////////////////////
// CHECKPOINTING

//////////////////////////////////////////////////////////////////////////////
/// @begin Monte_carlo::restore_checkpoint_counts
///
/// @brief
/// retores checkpoint counts
///
/// @detailed
/// This restores the checkpoint counts in the Monte_carlo object by reading
/// them from the checkpoint file.
///
/// @authors: dekim
///
/// @last_modified: 04/11/07
//////////////////////////////////////////////////////////////////////////////
	void Monte_carlo::restore_checkpoint_counts(){
		utility::io::izstream in_stream( checkpointfile );
		if (in_stream) {
			int cnt, isfullatom, ntrials;
			in_stream >> cnt >> isfullatom >> ntrials;
			in_stream.close();
			in_stream.clear();

			assert(	cnt >= 0 && isfullatom >= 0 );

			mc_checkpoint_isfullatom = isfullatom;
			mc_checkpoint_ntrials = ntrials;

			// must be saved outside of Monte_carlo class
			main_job_distributor::jd->set_mc_checkpoint_last_count( cnt );

			std::cout<<"Pose mc checkpoint!! restored counts for mc object last_count:"<<
					cnt << " isfullatom:" << isfullatom << std::endl;
		} else {
			std::cout<<"WARNING!! cant restore counts from checkpoint file:" << checkpointfile << std::endl;
		}
	}

//////////////////////////////////////////////////////////////////////////////
/// @begin Monte_carlo::time_to_checkpoint
///
/// @brief
/// is it time to checkpoint?
///
/// @detailed
///
/// @authors: dekim
///
/// @last_modified: 04/11/07
//////////////////////////////////////////////////////////////////////////////
	bool Monte_carlo::time_to_checkpoint() {
		if (!do_checkpointing) return false;
#ifdef BOINC
		// honor user's disk write interval preference
		if (!boinc_time_to_checkpoint() && !boinc_is_standalone()) return false;
#endif
		bool retval = false;
		double time_diff(0.0);
		time_t curr_time;
		time(&curr_time);
		time_diff = difftime( curr_time,
			main_job_distributor::jd->get_mc_checkpoint_time() );
		if ( time_diff > get_checkpoint_interval() ) {
			main_job_distributor::jd->update_mc_checkpoint_time();
			retval = true;
		}
#ifdef BOINC
		// boinc_time_to_checkpoint sets critical section to true
		// so make sure it is false and deal with the critical sections
		// in checkpoint() below
		boinc_end_critical_section();
#endif
		return retval;
	}

//////////////////////////////////////////////////////////////////////////////
/// @begin Monte_carlo::get_pose_from_checkpoint
///
/// @brief
/// gets checkpointed pose
///
/// @detailed
///
/// @authors: dekim
///
/// @last_modified: 04/11/07
//////////////////////////////////////////////////////////////////////////////
void Monte_carlo::restore_pose_from_checkpoint(
		Pose & pose,
		std::string const & outfile,
		bool const & isfullatom
) {
	using namespace silent_io;
	using namespace native;
	Silent_file_data decoys( outfile, isfullatom );
	if ( !decoys.size() ) {
		std::cout << "STOP:: couldnt open silent-file " << outfile << "!!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( ! decoys.has_key( "S_0" ) ) {
		std::cout << "STOP:: couldnt find tag in silent-file " << outfile << "!!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	Silent_structure const & decoy( decoys.get_structure( "S_0" ) );
	Pose restoredpose;
	restoredpose = pose; //copies in symmetry info, etc.
	decoy.fill_pose( restoredpose, true );

	// native? This isn't taken care of by the "restoredpose = pose"  above.
	if (pose.native_pose_exists()) restoredpose.set_native_pose( pose.native_pose() );

	pose = restoredpose;
	pose.score( weight_map_ );
}

//////////////////////////////////////////////////////////////////////////////
/// @begin Monte_carlo::checkpoint
///
/// @brief
/// checkpoint or skip to checkpoint and restore
///
/// @detailed
/// This is the main call for checkpointing and should be called before a
/// monte carlo move. If it returns true, the move should be skipped to
/// get back to the last move that was checkpointed. When the last checkpoint
/// step is reached, this will restore the current, best, and low pose and
/// then continue on and checkpoint when necessary according to the time
/// interval.
///
/// example in a loop:
///
/// for ( int j = 1; j <= nloop; ++j ) {
/// 	if (mc.checkpoint(pose)) continue;
/// 	choose_fragment_pose( pose, size, choose_frag_ss_check, true );
/// 	mc.boltzmann( pose );
/// }
///
/// This should not be called in a loop that has a break because it keeps
/// track of the last checkpoint by iterating a step count. If there is a
/// break and a checkpoint is made before generating the final structure,
/// the step count will not reflect the actual place to restore the
/// checkpoint. The current, best, and low pose get saved at checkpoint
/// time intervals or restored at the last checkpointed step. Checkpoint
/// counts should be reset after each model using:
///
///   main_job_distributor::jd->reset_mc_checkpoint()
///
/// If there are moves that should ideally be skipped when trying to restore
/// the last checkpoint, like closing chain breaks, use
///
///  main_job_distributor::jd->::skip_to_mc_checkpoint()
///
/// If it returns true, skip the moves.
///
/// @authors: dekim
///
/// @last_modified: 04/11/07
//////////////////////////////////////////////////////////////////////////////
	bool Monte_carlo::checkpoint(
		Pose & pose
	)
	{
		using namespace main_job_distributor;

		if (!do_checkpointing) return false;

		jd->iterate_mc_checkpoint_current_count();

		if ( jd->skip_to_mc_checkpoint() ) {
			// skip step!!
			return true;
		} else if ( jd->get_mc_checkpoint_current_count() ==
								jd->get_mc_checkpoint_last_count() ) {
			// restore checkpoint!!

			std::cout<<"Pose mc checkpoint!! restoring curr, best, and low at last_count:"
					<< jd->get_mc_checkpoint_last_count() << " isfullatom:"
					<< mc_checkpoint_isfullatom << std::endl;

			// read in currentpose, bestpose, lowpose
			bool isfullatom = false;
			if (mc_checkpoint_isfullatom) isfullatom = true;

			// current pose
			restore_pose_from_checkpoint( pose, currposeoutfile, isfullatom );

			// low pose
			restore_pose_from_checkpoint( low_pose_, lowposeoutfile, isfullatom );
			pose_to_low(low_pose_);

			// best pose
			restore_pose_from_checkpoint( best_pose_, bestposeoutfile, isfullatom );
			pose_to_best(best_pose_);

#ifdef BOINC
			// need this information in misc and trajectory for BOINC graphics.
			float const score ( pose.get_0D_score ( SCORE ) );
			float rms_err  = calc_rms( pose );
			float low_rms  = calc_rms( low_pose_ );
			float best_rms = calc_rms( best_pose_ );
			fill_misc_mc_score( score, best_score(), low_score(), rms_err, best_rms,
					low_rms);
			// restore counter for graphics
			counters::monte_carlo_ints::ntrials = mc_checkpoint_ntrials;
#endif

		} else if ( time_to_checkpoint() ) {
			// checkpoint!!

			int ntrials = 0;
#ifdef BOINC
			boinc_begin_critical_section();
			ntrials = counters::monte_carlo_ints::ntrials;
#endif

			int currcnt = jd->get_mc_checkpoint_current_count();

			std::cout<<"Pose mc checkpoint!! saving curr, best, and low pose current_count:"
					<< currcnt << " isfullatom:" << pose.fullatom() << std::endl;

			// prevent appending to existing checkpoints
			utility::file::file_delete( currposeoutfile );
			utility::file::file_delete( bestposeoutfile );
			utility::file::file_delete( lowposeoutfile );

			// should we make sure the files are deleted here?
			// save currentpose, bestpose, lowpose
			silent_io::Silent_out currout( currposeoutfile );
			currout.write( "S_0", pose );
			silent_io::Silent_out bestout( bestposeoutfile );
			bestout.write( "S_0", best_pose_ );
			silent_io::Silent_out lowout( lowposeoutfile );
			lowout.write( "S_0", low_pose_ );
			if ( pose.fullatom() ) {
				currout.store_rotamers( pose );
				bestout.store_rotamers( best_pose_ );
				lowout.store_rotamers( low_pose_ );
			}
			silent_out_write_nonideal_geometry( pose,
				currout, currposeoutfile );
			silent_out_write_nonideal_geometry( best_pose_,
				bestout, bestposeoutfile );
			silent_out_write_nonideal_geometry( low_pose_,
				lowout, lowposeoutfile );

			// update checkpoint file
			utility::io::ozstream out_stream( checkpointfile );
			if (!out_stream.good() ) {
				std::cout << "STOP: cant open file: " << checkpointfile << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			out_stream << currcnt << " " << pose.fullatom() << " " << ntrials;
			out_stream.close();
			out_stream.clear();
#ifdef BOINC
			boinc_checkpoint_completed();
#endif
		}
		return false;
	}



	/////////////////////////////////////////////////////////////////////////////
	//////////////////////////////
	// behavior depends on setting of temperature
	// return values:
	// 	mc_accepted
	// 		3 = accepted:score beat low score and best score
	// 		2 = accepted:score beat best score
	// 		1 = thermally accepted: score worse than best score
	// 		0 = not accepted

	bool Monte_carlo::boltzmann(
		Pose & pose,
		std::string const & move_type, // = unk
		bool const & movie // = true
	)
	{
		bool const debug( false );
		using files_paths::use_status;
		pose.score( weight_map_ );
		float const score ( pose.get_0D_score ( SCORE ) );
		last_score_ = score;

		mc_accepted_ = 2;

		if ( debug ) { // mcdebug
			pose.new_score_pose();
			pose.score( weight_map_ );
			std::cout << "MCDEBUG: score-del1: " << pose.get_0D_score( SCORE ) -
				score << std::endl;
		}

		++trial_counter[ move_type ];
#ifdef BOINC
// diagnostics for BOINC graphics.
		counters::monte_carlo_ints::ntrials++;
#endif

		if ( score > best_score() ) {
			float const boltz_factor = ( best_score() - score ) / temperature_;
			float const probability = std::exp( std::min (40.0f,
				std::max(-40.0f,boltz_factor)) );

			if ( ran3() > probability ) {
				if ( score - best_score() > 50 &&
						 move_type.find("repack") != std::string::npos ) {
					std::cout << "WARNING:: bad_repack: " << score << ' ' <<
						best_score() << std::endl;
					//pose.dump_scored_pdb( "bad_repack.pdb", weight_map_ );
					//exit(0);
				}

				mc_accepted_ = 0; // rejected
				autotemp_reject();
				pose = best_pose_;
				pose_to_best(best_pose_);
				if ( save_mc_accepted_ ) {
					pose.set_extra_score ( "MC_ACCEPTED", mc_accepted_ );
				}
				if ( debug ) {
					pose.new_score_pose();
					pose.score( weight_map_ );
					std::cout << "MCDEBUG: score-del2: " << pose.get_0D_score( SCORE ) -
						best_score() << std::endl;
				}
				if ( use_status ) {
					std::cout << "STATUS reject " << I(4,trial_counter[ move_type ] ) <<
						F(9,3,score) << ' ' << F(9,3,best_score()) << ' ' <<
						F(9,3,low_score() ) << ' ' << move_type << std::endl;
				}
#ifdef GL_GRAPHICS
				gl_graphics_mc_trial( score, best_score(), low_score(),
															calc_rms(pose), move_type, mc_accepted_ );
#endif
				return false;
			}
			mc_accepted_ = 1; // accepted by chance
		}

		++accept_counter[ move_type ];
		energy_drop_counter[ move_type ] += score - best_score();
		best_pose_ = pose;
		if ( debug ) { // mc debug
			best_pose_.new_score_pose();
			best_pose_.score( weight_map_ );
			std::cout << "MCDEBUG: score-del3: " << best_pose_.get_0D_score( SCORE) -
				score << std::endl;
		}
		autotemp_accept();

		if ( score < low_score() ) {
			low_pose_ = pose;
			if ( debug ) { // mc debug
				low_pose_.new_score_pose();
				low_pose_.score( weight_map_ );
				std::cout << "MCDEBUG: score-del4: " << low_pose_.get_0D_score(SCORE) -
					score << std::endl;
			}
			pose_to_low(low_pose_);
			mc_accepted_ = 3;
			if ( use_status ) {
				std::cout << "STATUS low-accept " << I(4,trial_counter[ move_type ] )<<
					F(9,3,score) << ' ' << F(9,3,best_score()) << ' ' <<
					F(9,3,low_score() ) << ' ' << move_type << std::endl;
			}
		} else {
			if ( use_status ) {
				std::cout << "STATUS accept " << I(4,trial_counter[ move_type ] ) <<
					F(9,3,score) << ' ' << F(9,3,best_score()) << ' ' <<
					F(9,3,low_score() ) << ' ' << move_type << std::endl;
			}
		}

		if ( save_mc_accepted_ ) {
			pose.set_extra_score ( "MC_ACCEPTED", mc_accepted_ );
		}

		if ( movie ) {
			make_trajectory( move_type, mc_accepted_, trial_counter[ move_type ],
											 accept_counter[ move_type]);
		}

#ifdef BOINC
// need this information in misc and trajectory for BOINC graphics.
		float rms_err  = calc_rms( pose );
		float low_rms  = calc_rms( low_pose_ );
		float best_rms = calc_rms( best_pose_ );
		fill_misc_mc_score( score, best_score(), low_score(), rms_err, best_rms,
												low_rms);
		append_trajectory(score, rms_err);
#endif

#ifdef GL_GRAPHICS
		update_monte_carlo_globals( mc_accepted_ );
		gl_graphics_mc_trial( score, best_score(), low_score(),
													calc_rms( pose ), move_type, mc_accepted_ );
		while ( wait_on_graphics() ) {
		}
#endif

		return true;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Monte_carlo::set_autotemp(
		bool const setting,
		float const quench_temp
	)
	{
		autotemp_ = setting;
		quench_temp_ = quench_temp;
		last_accept_ = 0;
	}


	/////////////////////////////////////////////////////////////////////////////
	// replicate logic from monte_carlo.cc
	//
	// should probably make these parameters ( 150, 1.0 )
	void
	Monte_carlo::autotemp_reject()
	{
		int const heat_after_cycles( 150 );
		float const heat_delta( quench_temp_ * 0.5 );
		float const max_temperature( quench_temp_ * 10.0 );

		if ( !autotemp_ ) return;
		if ( last_accept_ >= heat_after_cycles ) {
			if ( temperature_ > max_temperature * 0.25 )
				std::cout << "heat: " << last_accept_<< ' ' << temperature_ <<
					std::endl;
			last_accept_ = -1;
			temperature_ = std::min( temperature_ + heat_delta, max_temperature );
		}
		++last_accept_;
	}

	/////////////////////////////////////////////////////////////////////////////
	// replicate logic from monte_carlo.cc
	void
	Monte_carlo::autotemp_accept()
	{
		if ( !autotemp_ ) return;
		temperature_ = quench_temp_;
		last_accept_ = 0;
	}

	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
} // Pose_ns

////////////////////////////////////////////////////
void
copy_secstruct_pose( FArray1D_char const & secstruct_src, FArray1D_char & secstruct_dest, int const nres ){
	for (int i = 1; i <= nres; i++){
		secstruct_dest(i) = secstruct_src(i);
	}
}

////////////////////////////////////////////////////
void
pose_to_native(
	pose_ns::Pose const & native_pose
){
	using namespace native;
	FArray3D_float dummy_full_coord;

	native_exists = true;

	copy_coordinates_pose( true /*native_pose.fullatom()*/, native_pose.total_residue(),
												 native_pose.Eposition(), native_pose.centroid(),
												 //												 native_pose.fullatom() ? native_pose.full_coord() : dummy_full_coord,
												 native_pose.full_coord(),
												 native_Eposition, native_centroid, native_coord );
	//	pose_to_misc( native_pose );
	//	store_native( native_pose.fullatom, native_pose.fullatom );

	//Needed for graphics... copy other stuff too?
	copy_secstruct_pose( native_pose.secstruct(), native::nat_secstruct, native_pose.total_residue() );

	// Necessary for maxsub calculation:
	for ( int i = 1; i <= native_pose.total_residue(); ++i ) {
		for ( int k = 1; k <= 3; ++k ) {
			native_ca(k,i) = native_Eposition(k,2,i);
		}
	}

	return;
}
////////////////////////////////////////////////////

void
pose_to_low(
	pose_ns::Pose const & low_pose
){
	using namespace low_pose;
	FArray3D_float dummy_full_coord;

	copy_coordinates_pose( low_pose.fullatom(), low_pose.total_residue(),
		low_pose.Eposition(), low_pose.centroid(),
		( low_pose.fullatom() ? low_pose.full_coord() : dummy_full_coord ),
		low_pose::Elow_position, low_pose::low_centroid, low_pose::low_full_coord );

	//Needed for graphics... copy other stuff too?
	copy_secstruct_pose( low_pose.secstruct(), low_pose::low_secstruct, low_pose.total_residue() );

	for ( int i=1; i<= low_pose.total_residue(); ++i ) {
		low_pose::low_res        (i) = low_pose.res        (i);
		low_pose::low_res_variant(i) = low_pose.res_variant(i);
	}

	return;
}

////////////////////////////////////////////////////

void
pose_to_best(
	pose_ns::Pose const & best_pose
){
	using namespace misc;
	FArray3D_float dummy_full_coord;

	copy_coordinates_pose( best_pose.fullatom(), best_pose.total_residue(),
		best_pose.Eposition(), best_pose.centroid(),
		( best_pose.fullatom() ? best_pose.full_coord() : dummy_full_coord ),
		best_pose::Ebest_position, best_pose::best_centroid, best_pose::best_full_coord );

	copy_secstruct_pose( best_pose.secstruct(), best_pose::best_secstruct, best_pose.total_residue() );

	for ( int i=1; i<= best_pose.total_residue(); ++i ) {
		misc::best_res        (i) = best_pose.res        (i);
		misc::best_res_variant(i) = best_pose.res_variant(i);
	}

	return;
}

//////////////////////////////////////////////////////
void
fill_misc_mc_score( float const & score, float const & best_score, float const & low_score, float const & rms_err, float const & best_rms, float const & low_rms)
{
	mc_global_track::mc_score::score      = score;
	mc_global_track::mc_score::best_score = best_score;
	mc_global_track::mc_score::low_score  = low_score;
	mc_global_track::diagnose::rms_err  = rms_err;
	mc_global_track::diagnose::best_rms = best_rms;
	mc_global_track::diagnose::low_rms  = low_rms;
}

