// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 23432 $
//  $Date: 2008-06-24 16:25:52 +0300 (Tue, 24 Jun 2008) $
//  $Author: yab $


// Rosetta Headers
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "bond_angle.h"
#include "design.h"
#include "disulfides.h"
#include "files_paths.h"
#include "hbonds.h"
#include "histogram.h"
#include "initialize.h"
#include "minimize_ns.h"
#include "minimize.h"
#include "PackerTask.h"
#include "pack_geom_inline.h"
#include "param_aa.h"
#include "param_pack.h"
#include "param.h"
#include "pose_backrub_controller.h"
#include "pose_backrub.h"
#include "pose_io.h"
#include "pose_movie.h"
#include "pose_rms.h"
#include "pose_rotamer_controller.h"
#include "pose.h"
#include "ramachandran.h"
#include "random_numbers.h"
#include "read_aaproperties.h"
#include "runlevel.h"
#include "score_data.h"
#include "score_name.h"
#include "score.h"
#include "util_vector.h"
#include <IntervalSet.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/formatted.io.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/constants.hh>
#include <numeric/conversions.hh>
#include <numeric/numeric.functions.hh>
#include <numeric/NumericTraits.hh>
#include <numeric/trig.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/vector1.hh>

// C++ Headers
#include <cstdlib>
#include <iostream>
#include <iomanip>
#include <cassert>
#include <fstream>
#include <set>
#include <sstream>


/// @brief generate a random number drawn from a sin distribution
float
random_isotropic_zenith_angle(
	float const current_angle,
	float const min_angle = 0,
	float const max_angle = numeric::constants::f::pi
)
{
	assert(current_angle >= 0 && current_angle <= numeric::constants::f::pi);
	assert(min_angle <= max_angle && min_angle >= 0 && max_angle <= numeric::constants::f::pi);

	float const threshold(ran3());
	float const sin_current_angle(std::sin(current_angle));
	float const angle_range(max_angle - min_angle);
	float angle;

	for (int i = 0; i < 10000000; i++) {
		angle = min_angle + ran3()*angle_range;
		if (std::sin(angle)/sin_current_angle >= threshold) break;
	}

	return angle;
}

/// @brief test difference between constraining CB & HA positions and using quadratic fits
void
bond_angle_free_energy_test()
{
	pose_ns::Pose pose;

	int ntrials = 1000;
	intafteroption("ntrials", ntrials, ntrials);
	float mc_temp = 0.6;
	realafteroption("mc_temp", mc_temp, mc_temp);
	int const nres(2);
	bool const constrain(truefalseoption("constrain"));
	float const delta(numeric::conversions::radians(5.));


	std::cout << "Running Bond Angle Free Energy" << std::endl;

	bond_angle::initialize_bond_angle_params();

	pose.simple_fold_tree(nres);

	for (int i = 1; i <= nres; ++i) {
		pose.set_phi(i, 180);
		pose.set_psi(i, 180);
		pose.set_omega(i, 180);
		pose.set_res(i, param_aa::aa_ala);
		pose.set_res_variant(i, 1);
	}

	pose.set_fullatom_flag(true, false);
	replace_cbha_pose(pose);
	pose.setup_atom_tree();

	int const cbp = bond_angle::CBpos[pose.res(1)];
	int const hap = aaproperties_pack::HApos(pose.res(1), pose.res_variant(1));
	pose.set_allow_move(3, 1, kin::THETA, true);
	pose.set_allow_move(cbp, 1, kin::PHI, true);
	pose.set_allow_move(cbp, 1, kin::THETA, true);
	pose.set_allow_move(hap, 1, kin::PHI, true);
	pose.set_allow_move(hap, 1, kin::THETA, true);

	histogram<float> n_ca_c_hist;
	histogram<float> cb_phi_hist;
	histogram<float> cb_theta_hist;
	histogram<float> ha_phi_hist;
	histogram<float> ha_theta_hist;

	n_ca_c_hist.init(0, numeric::constants::f::pi, 360);
	cb_phi_hist.init(-numeric::constants::f::pi, numeric::constants::f::pi, 720);
	cb_theta_hist.init(0, numeric::constants::f::pi, 360);
	ha_phi_hist.init(-numeric::constants::f::pi, numeric::constants::f::pi, 720);
	ha_theta_hist.init(0, numeric::constants::f::pi, 360);

	pose_ns::Pose last_pose;
	last_pose = pose;
	float last_energy(get_bond_angleE(pose.res(1), pose.res_variant(1), pose.full_coord()(1,1,1)));

	for (int i = 0; i < ntrials; ++i) {
		//angle = random_isotropic_zenith_angle(angle, 0, numeric::constants::f::pi);
		//n_ca_c_hist.record(angle);

		float angle;

		angle = pose.get_atom_tree_torsion(3, 1, kin::THETA);
		angle = random_isotropic_zenith_angle(angle, std::max(angle - delta, (float)0.0),
		                                      std::min(angle + delta, numeric::constants::f::pi));
		//std::cout << numeric::conversions::degrees(angle);
		pose.set_atom_tree_torsion(3, 1,  kin::THETA, angle);

		if (constrain) {
			float thetaCB, phiCB, thetaHA, phiHA;
			get_cbha_angles(pose.res(1), pose.res_variant(1), numeric::constants::f::pi - angle, thetaCB, phiCB, thetaHA, phiHA);
			//std::cout << thetaCB << "\t" << phiCB << "\t" << thetaHA << "\t" << phiHA << std::endl;
			thetaHA -= thetaCB;
			center_angle_radians(thetaHA);
			/*
			std::cout << -thetaCB << "\t"
			          << numeric::constants::f::pi - phiCB << "\t"
								<< -thetaHA << "\t"
								<< numeric::constants::f::pi - phiHA << std::endl;
			std::cout << pose.get_atom_tree_torsion(cbp, 1, kin::PHI) << "\t"
								<< pose.get_atom_tree_torsion(cbp, 1, kin::THETA) << "\t"
								<< pose.get_atom_tree_torsion(hap, 1, kin::PHI) << "\t"
								<< pose.get_atom_tree_torsion(hap, 1, kin::THETA) << std::endl;
			*/
			pose.set_atom_tree_torsion(cbp, 1,  kin::PHI, -thetaCB);
			pose.set_atom_tree_torsion(cbp, 1,  kin::THETA, numeric::constants::f::pi - phiCB);
			pose.set_atom_tree_torsion(hap, 1,  kin::PHI, -thetaHA);
			pose.set_atom_tree_torsion(hap, 1,  kin::THETA, numeric::constants::f::pi - phiHA);
		} else {
			angle = pose.get_atom_tree_torsion(cbp, 1, kin::PHI);
			angle += 2*delta*ran3() - delta;
			//std::cout << "\t" << numeric::conversions::degrees(angle);
			pose.set_atom_tree_torsion(cbp, 1,  kin::PHI, angle);

			angle = pose.get_atom_tree_torsion(cbp, 1, kin::THETA);
			angle = random_isotropic_zenith_angle(angle, angle - delta, angle + delta);
			//std::cout << "\t" << numeric::conversions::degrees(angle);
			pose.set_atom_tree_torsion(cbp, 1,  kin::THETA, angle);

			angle = pose.get_atom_tree_torsion(hap, 1, kin::PHI);
			angle += 2*delta*ran3() - delta;
			//std::cout << "\t" << numeric::conversions::degrees(angle);
			pose.set_atom_tree_torsion(hap, 1,  kin::PHI, angle);

			angle = pose.get_atom_tree_torsion(hap, 1, kin::THETA);
			angle = random_isotropic_zenith_angle(angle, angle - delta, angle + delta);
			//std::cout << "\t" << numeric::conversions::degrees(angle);
			pose.set_atom_tree_torsion(hap, 1,  kin::THETA, angle);
		}

		float new_energy(get_bond_angleE(pose.res(1), pose.res_variant(1), pose.full_coord()(1,1,1)));
		//std::cout << "\t" << new_energy << "\t" << last_energy;

		float const probability(std::exp((last_energy - new_energy) / mc_temp));

		if (probability < ran3()) {
			// Rejected
			pose = last_pose;
			//std::cout << "\t" << "R";
		} else {
			// Accepted
			last_energy = new_energy;
			last_pose = pose;
			//std::cout << "\t" << "A";
		}
		//std::cout << std::endl;

		n_ca_c_hist.record(numeric::constants::f::pi - pose.get_atom_tree_torsion(3, 1, kin::THETA));
		cb_phi_hist.record(pose.get_atom_tree_torsion(cbp, 1, kin::PHI));
		cb_theta_hist.record(numeric::constants::f::pi - pose.get_atom_tree_torsion(cbp, 1, kin::THETA));
		ha_phi_hist.record(pose.get_atom_tree_torsion(hap, 1, kin::PHI));
		ha_theta_hist.record(numeric::constants::f::pi - pose.get_atom_tree_torsion(hap, 1, kin::THETA));
	}

	std::ofstream outfile;
	outfile.open("n_ca_c_hist.txt");

	n_ca_c_hist.print_counts(outfile);
	cb_phi_hist.print_counts(outfile);
	cb_theta_hist.print_counts(outfile);
	ha_phi_hist.print_counts(outfile);
	ha_theta_hist.print_counts(outfile);

	outfile.close();
}

void
get_pdb_bond_angles()
{
	int nstartnum; // number of input structures
	FArray1D_string startnm; // starting pdb files
	FArray1D_string outnm; // output pdb files
	FArray1D_char startch; // starting chain id

	startnm.dimension( param::MAX_START );
	outnm.dimension( param::MAX_START );
	startch.dimension( param::MAX_START);
	startch = ' ';

	setup_start_list( nstartnum, startnm, outnm, startch );

	param::MAX_RES_assign_res(1000);
	std::cout << "Max Res: " << param::MAX_RES()() << std::endl;

	for (int curr_startnum = 1; curr_startnum <= nstartnum; ++curr_startnum) {

	files_paths::start_file = startnm(curr_startnum);
	files_paths::output_file = outnm(curr_startnum);
	files_paths::protein_chain = startch(curr_startnum);

	std::string startfile = files_paths::start_path + files_paths::start_file + ".pdb";
	pose_ns::Pose pose;
	pose_from_pdb(pose, startfile, true, false,
	              files_paths::io_options_bool::read_all_chains,
	              files_paths::protein_chain,
								files_paths::skip_missing,
								files_paths::allow_missing);

	pose.setup_atom_tree();

	std::ofstream angles;

	bond_angle::initialize_bond_angle_params();

	angles.open((files_paths::start_path + files_paths::start_file + "_angles").c_str());

	for (int i = 1; i <= pose.total_residue(); ++i) {
		int const aatype = pose.res(i);
		int const aavariant = pose.res_variant(i);\
		int const Cpos = 3;
		int const CBpos = bond_angle::CBpos[aatype];
		int const HApos = aaproperties_pack::HApos(aatype, aavariant);
		float alpha = numeric::constants::f::pi - pose.get_atom_tree_torsion(Cpos, i, kin::THETA);
		float thetaCB = pose.get_atom_tree_torsion(CBpos, i, kin::PHI);
		float phiCB = numeric::constants::f::pi - pose.get_atom_tree_torsion(CBpos, i, kin::THETA);
		float thetaHA = pose.get_atom_tree_torsion(HApos, i, kin::PHI) + thetaCB;
		center_angle_radians(thetaHA);
		float phiHA = numeric::constants::f::pi - pose.get_atom_tree_torsion(HApos, i, kin::THETA);
		angles << param_aa::aa_name3(aatype)  << "\t" << alpha << "\t" << thetaCB << "\t" << phiCB << "\t" << thetaHA << "\t" << phiHA << std::endl;
	}

	angles.close();

	} // end starting loop
}


////////////////////////////////////////////////////////////////////////////////
/// @begin detailed_balance_test
///
/// @brief
/// test for isotropic internal coordinate distributions with the Backrub move
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 16, 2007
////////////////////////////////////////////////////////////////////////////////
void
detailed_balance_test()
{
	using numeric::conversions::radians;

	pose_ns::Pose pose;

	int const nres = 8;
	int ntrials = 1000;
	float backrub_prob = 0;
	float alpha_min = 10;
	float alpha_max = 170;
	float angle_max = 180;
	bool const jacobian = truefalseoption("jacobian");
	int res_dependent = truefalseoption("res_dependent");
	intafteroption("ntrials", ntrials, ntrials);
	realafteroption("backrub_prob", backrub_prob, backrub_prob);
	realafteroption("alpha_min", alpha_min, alpha_min);
	realafteroption("alpha_max", alpha_max, alpha_max);
	realafteroption("angle_max", angle_max, angle_max);
	utility::vector0<utility::vector0<int> > allowed_moves;
	utility::vector0<histogram<float> > hists(nres*3);
	utility::vector0<histogram<float> > testhists(3);
	utility::vector0<float> testangles(3);
	std::ofstream histfile;
	double res_opt = 0;

	numeric::conversions::to_radians(alpha_min);
	numeric::conversions::to_radians(alpha_max);
	numeric::conversions::to_radians(angle_max);

	/*

	std::cout << "Running Histogram Test" << std::endl;

	for (int i = 0; i < (signed)testhists.size(); i++) {
		testhists[i].init(0, numeric::constants::f::pi, 36);
		testangles[i] = alpha_min + ran3()*(alpha_max-alpha_min);
		testangles[i] = alpha_min + ran3()*(alpha_max-alpha_min);
	}

	histfile.open("anglerecord.txt");

	float num = ran3();

	for (int i = 0; i < ntrials; i++) {

		float randnum = ran3();
		float newnum = ran3();
		while (newnum/num <= randnum) {
					newnum = ran3();
		}
		num = newnum;
		histfile << num << std::endl;
	}

	histfile.close();

	int max_iter = 0;

	for (int i = 0; i < ntrials; i++) {
		testangles[0] = alpha_min + ran3()*(alpha_max-alpha_min);

		float const J1 = testangles[1];
		float const R1 = ran3();
		for (int j = 0; j < 1000000000; j++) {
			testangles[1] = alpha_min + ran3()*(alpha_max-alpha_min);
			if (j > max_iter) max_iter = j;
			if (testangles[1]/J1 >= R1) break;
		}

		float const J2 = std::sin(testangles[2]);
		float const R2 = ran3();
		for (int j = 0; j < 10000000; j++) {
			testangles[2] = alpha_min + ran3()*(alpha_max-alpha_min);
			if (j > max_iter) max_iter = j;
			if (std::sin(testangles[2])/J2 >= R2) break;
		}

		for (int j = 0; j < (signed)testhists.size(); j++) {
			testhists[j].record(testangles[j]);
		}
	}

	std::cout << max_iter << " Iterations" << std::endl;

	histfile.open("testhist.txt");

	for (int i = 0; i < (signed)testhists.size(); i++) {

		testhists[i].print_frequencies(histfile);
	}

	histfile.close();

	return;

	*/

	std::cout << "Running Detailed Balance Test" << std::endl;

	bond_angle::initialize_bond_angle_params();

	pose.simple_fold_tree(nres);

	for (int i = 1; i <= nres; i++) {
		pose.set_phi(i, 180);
		pose.set_psi(i, 180);
		pose.set_omega(i, 180);
		pose.set_res(i, param_aa::aa_ala);
		pose.set_res_variant(i, 1);
	}

	pose.set_fullatom_flag(true, false);
	replace_cbha_pose(pose);

	allowed_moves = pose_backrub_allowed_moves(pose, nres);

	for (int i = 0; i < (signed)hists.size(); i += 3) {
		hists[i].init(-numeric::constants::d::pi, numeric::constants::d::pi, 36);
		hists[i+1].init(-numeric::constants::d::pi, numeric::constants::d::pi, 36);
		hists[i+2].init(0, numeric::constants::d::pi, 36);
	}

	for (int i = 1; i <= nres; i++) {
		pose.set_phi(i, (ran3() - 0.5)*360);
		pose.set_psi(i, (ran3() - 0.5)*360);

		if (res_dependent) {
			// Bond angle potential ideal angle
			res_opt = bond_angle::Theta0_bb[pose.res(i)];
		}
	}

	/* Test uniform sampling of a single backrub angle
	histogram<float> anglehist;
	anglehist.init(-numeric::constants::d::pi, numeric::constants::d::pi, 180);
	float theangle = 0;
	for (int i = 0; i < ntrials; i++) {
		float const angle = backrub_select_angle(pose, 2, 7, alpha_min, alpha_max,
			                                       res_dependent, angle_max, jacobian);
		backrub_rot_pose(pose, 2, 7, angle);
		theangle += angle;
		center_angle_radians(theangle);
		anglehist.record(theangle);
	}
	histfile.open("anglehist.txt");
	anglehist.print_counts(histfile);
	histfile.close();

	return;
	*/

	for (int i = 0; i < ntrials; i++) {

		if (ran3() < backrub_prob) {

			int size = random_range(2, nres);
			int start = allowed_moves[size-2][random_range(0, allowed_moves[size-2].size() - 1)];
			int end = start + size - 1;

			float const angle = backrub_select_angle(pose, start, end, alpha_min, alpha_max,
			                                         res_dependent, angle_max, jacobian);

			backrub_rot_pose(pose, start, end, angle);

		} else {

			int angle_type = random_range(1, 3);
			int res = random_range(1, nres);

			if (angle_type == 1) {
			  float const angle = (ran3() - 0.5)*360;
				pose.set_phi(res, angle);
			} else if (angle_type == 2) {
			  float const angle = (ran3() - 0.5)*360;
				pose.set_psi(res, angle);
			} else {
				if (res_dependent) {
					// Bond angle potential ideal angle
					res_opt = bond_angle::Theta0_bb[pose.res(res)];
				}

				float const J = std::sin(planar_angle(pose.full_coord(1,res), pose.full_coord(2,res),
				                                      pose.full_coord(3,res)));

			  float const threshold = ran3();
			  float angle;

			  for (int j = 0; j < 10000000; j++) {
			  	angle = res_opt + alpha_min + ran3()*(alpha_max-alpha_min);
			  	if (std::sin(angle)/J >= threshold) break;
			  }
			  //angle = alpha_min + ran3()*(alpha_max-alpha_min);
				set_ca_angle_pose(pose, res, angle);
			}
		}

		for (int j = 0; j < nres; j++) {

			hists[3*j].record(radians(pose.phi(j+1)));
			hists[3*j+1].record(radians(pose.psi(j+1)));
			hists[3*j+2].record(planar_angle(pose.full_coord(1,j+1), pose.full_coord(2,j+1),
			                                 pose.full_coord(3,j+1)));
		}

	}

	histfile.open("hist.txt");

	for (int i = 0; i < (signed)hists.size(); i++) {

		hists[i].print_counts(histfile);
	}

	histfile.close();

	pose.dump_pdb("dbtest.pdb");
}

////////////////////////////////////////////////////////////////////////////////
/// @begin pose_backrub_test
///
/// @brief
/// function for placing Backrub test code
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 16, 2007
////////////////////////////////////////////////////////////////////////////////
void
pose_backrub_test()
{
	using namespace param_aa;
	using namespace pose_ns;
	using numeric::conversions::radians;
	using numeric::conversions::degrees;

	//bond_angle_free_energy_test();
	//get_pdb_bond_angles();

	{ // scope tau_intervals test

	pose_ns::Pose pose;
	std::string startfile = files_paths::start_path + files_paths::start_file + ".pdb";
	bond_angle::initialize_bond_angle_params();
	pose_from_pdb(pose, startfile, true, false,
	              files_paths::io_options_bool::read_all_chains,
	              files_paths::protein_chain);

	for (int i = 2; i <= 12; i++) {
		std::cout << ")," <<  std::endl << "\"" << i << "\" = c(";
		for (int j = 1; j + i - 1 <= pose.total_residue(); j++) {
			if (pose.res(j) == param_aa::aa_pro) continue;
			if (pose.res(j + i - 1) == param_aa::aa_pro) continue;
			backrub_select_angle(pose, j, j + i - 1, numeric::conversions::radians(-10.),
			                     numeric::conversions::radians(10.), true,
													 numeric::conversions::radians(90.), false);
		}
	}

	}

	return;

	{ // scope

	pose_ns::Score_weight_map weight_map, cst_weight_map;
	std::string startfile = files_paths::start_path + files_paths::start_file + ".pdb";
	pose_ns::Pose pose, pose_ref;

	pose_from_pdb(pose_ref, "1N7Tali.pdb", true, false,
	              files_paths::io_options_bool::read_all_chains,
	              files_paths::protein_chain);
	pose_from_pdb(pose, startfile, true, false,
	              files_paths::io_options_bool::read_all_chains,
	              files_paths::protein_chain);

	setup_score_weight_map(weight_map, score12);

	weight_map.set_weight(pose_ns::COORD_CST, 1.);
	cst_weight_map.set_weight(pose_ns::COORD_CST, 1.);

	pose_ns::Fold_tree ft;

	ft.clear();
	ft.add_edge(16, 102, 1); // Gly 16 to terminal V jump
	ft.add_edge(1, 16, pose_param::PEPTIDE); // PDZ domain
	ft.add_edge(16, 95, pose_param::PEPTIDE); // PDZ domain
	ft.add_edge(96, 102, pose_param::PEPTIDE); // peptide
	ft.reorder(1); // root at the first residue

	pose.set_fold_tree(ft);

	pose.setup_atom_tree();

	pose.score(weight_map);
	std::cout << "Before Constraints" << std::endl;
	std::cout << pose.show_scores() << std::endl;

	cst_set_ns::Cst_set cset;

	/*
	int CAatomno(LookupByName(pose.res(20), pose.res_variant(20), " CA "));
	int CZ3atomno(LookupByName(pose.res(98), pose.res_variant(98), " CZ3"));
	int CBatomno(LookupByName(pose.res(98), pose.res_variant(98), " CB "));

	kin::Atom_id CZ3atomid(pose.atom_tree()->atom(CZ3atomno, 98)->atom_id);
	kin::Atom_id CBatomid(pose.atom_tree()->atom(CBatomno, 98)->atom_id);
	kin::Atom_id CAatomid;

	CAatomid = pose.atom_tree()->atom(CAatomno, 20)->atom_id;
	cset.add_atompair_constraint(CZ3atomid, CAatomid, 5.752601, 1, true);
	CAatomid = pose.atom_tree()->atom(CAatomno, 29)->atom_id;
	cset.add_atompair_constraint(CZ3atomid, CAatomid, 5.604912, 1, true);
	CAatomid = pose.atom_tree()->atom(CAatomno, 40)->atom_id;
	cset.add_atompair_constraint(CZ3atomid, CAatomid, 4.871484, 1, true);

	CAatomid = pose.atom_tree()->atom(CAatomno, 20)->atom_id;
	cset.add_atompair_constraint(CBatomid, CAatomid, 5.639821, 1, true);
	CAatomid = pose.atom_tree()->atom(CAatomno, 29)->atom_id;
	cset.add_atompair_constraint(CBatomid, CAatomid, 7.802580, 1, true);
	CAatomid = pose.atom_tree()->atom(CAatomno, 40)->atom_id;
	cset.add_atompair_constraint(CBatomid, CAatomid, 8.899708, 1, true);

	pose.set_constraints(cset);

	pose.score(weight_map);
	std::cout << "After Constraints" << std::endl;
	std::cout << pose.show_scores() << std::endl;
	*/

	pose.set_allow_bb_move(false);
	pose.set_allow_chi_move(false);
	pose.set_allow_jump_move(false);

	for (int i = 96; i <= 100; i++) {
		pose.set_allow_bb_move(i, true);
		pose.set_allow_chi_move(i, true);
		pose.set_phi(i, pose_ref.phi(i+8));
		pose.set_phi(i, pose_ref.psi(i+8));
		int const nchi(aaproperties_pack::nchi(pose.res(i), pose.res_variant(i)));
		for (int j = 1; j <= nchi; j++) {
			pose.set_chi(j, i, pose_ref.chi(j, i+8));
		}

		coord_cst_residue(cset, pose, i, pose_ref, i+8, 4);
	}

	pose.set_constraints(cset);

	pose.score(weight_map);
	std::cout << "After Constraints" << std::endl;
	std::cout << pose.show_scores() << std::endl;

	//pose.main_minimize(weight_map, "dfpmin");

	//pose.set_chi(1, 98, 42.3919500);
	//pose.set_chi(2, 98, -86.6037619);

	//pose.set_allow_chi_move(98, false);

	pose.main_minimize(cst_weight_map, "dfpmin");
	pose.main_minimize(weight_map, "dfpmin");

	pose.score(weight_map);
	std::cout << "After Minimization (Full constraints)" << std::endl;
	std::cout << pose.show_scores() << std::endl;

	for (int i = 1; i <= 10; i++) {
		weight_map.set_weight(pose_ns::COORD_CST, weight_map.get_weight(pose_ns::COORD_CST)/2);
		pose.main_minimize(weight_map, "dfpmin");

		pose.score(weight_map);
		std::cout << "After Minimization " << i << std::endl;
		std::cout << pose.show_scores() << std::endl;
	}
	weight_map.set_weight(pose_ns::COORD_CST, 0.);
	pose.main_minimize(weight_map, "dfpmin");

	pose.score(weight_map);
	std::cout << "After Minimization (No constraints)" << std::endl;
	std::cout << pose.show_scores() << std::endl;

	pose.dump_pdb("minimized.pdb");

	return;

	} // scope

	{ // scope

	pose_ns::Pose pose;
	pose_ns::Score_weight_map weight_map;
	float bond_angle_weight = 1;

	bond_angle::initialize_bond_angle_params();

	setup_score_weight_map(weight_map, score12);
	realafteroption("bond_angle_weight", bond_angle_weight, bond_angle_weight);
	weight_map.set_weight(pose_ns::BOND_ANGLE, bond_angle_weight);

	/*
	pose.simple_fold_tree(3);

	for (int i = 1; i <= pose.total_residue(); i++) {
		pose.set_phi(i, 180);
		pose.set_psi(i, 180);
		pose.set_omega(i, 180);
		pose.set_res(i, 1);
		pose.set_res_variant(i, 1);
	}
	pose.set_fullatom_flag(true, true);
	pose.setup_atom_tree();

	for (int i = 1; i <= pose.total_residue(); i++) {
		for (int j = 1; j <= aaproperties_pack::nchi(pose.res(i), pose.res_variant(i)); j++) {
			pose.set_chi(j, i, 180);
		}
		pose.set_allow_bb_move(i, false);
		pose.set_allow_chi_move(i, false);
	}

	int const cbp = bond_angle::CBpos[pose.res(2)];
	int const hap = aaproperties_pack::HApos(pose.res(2), pose.res_variant(2));

	pose.dump_pdb("ba_start.pdb");

	pose.set_allow_move(3, 2, kin::THETA, true);
	pose.set_atom_tree_torsion(3, 2, kin::THETA, radians(48.8));

	pose.dump_pdb("ba_bent.pdb");
	*/

	std::string startfile = files_paths::start_path + files_paths::start_file + ".pdb";
	pose_from_pdb(pose, startfile, true, false,
	              files_paths::io_options_bool::read_all_chains,
	              files_paths::protein_chain);

	pose.setup_atom_tree();

	pose.dump_pdb("ba_start.pdb");

	for (int i = 1; i <= pose.total_residue(); i++) {
		if (pose.res(i) == param_aa::aa_pro) {
			pose.set_allow_chi_move(i, false); // freeze proline side chain
			pose.set_allow_move(2, i, kin::PHI, false); // freeze proline phi
			continue;
		}
		int const cbp = bond_angle::CBpos[pose.res(i)];
		int const hap = aaproperties_pack::HApos(pose.res(i), pose.res_variant(i));
		pose.set_allow_move(3, i, kin::THETA, true);
		pose.set_allow_move(cbp, i, kin::PHI, true);
		pose.set_allow_move(cbp, i, kin::THETA, true);
		pose.set_allow_move(hap, i, kin::PHI, true);
		pose.set_allow_move(hap, i, kin::THETA, true);
	}

	pose.main_minimize(weight_map, "dfpmin");

	pose.dump_pdb("ba_minimized.pdb");

	return;

	}

	print_idealzed_ca_angles(std::cout);

	return;

	detailed_balance_test();

	return;

	/*

	utility::vector0<double> r1;
	utility::vector0<double> r2;
	utility::vector0<double> r;

	r1.push_back(0);
	r1.push_back(1);
	r1.push_back(2);
	r1.push_back(4);
	r1.push_back(5);
	r1.push_back(6);
	r2.push_back(0.5);
	r2.push_back(2.5);
	r2.push_back(3.5);
	r2.push_back(5.5);
	range_and(r1, r2, r);


	for (int i = 0; i < r1.size(); i += 2) std::cout << r1[i] << ":" << r1[i+1] << " ";
	std::cout << std::endl;
	for (int i = 0; i < r2.size(); i += 2) std::cout << r2[i] << ":" << r2[i+1] << " ";
	std::cout << std::endl;
	for (int i = 0; i < r.size(); i += 2) std::cout << r[i] << ":" << r[i+1] << " ";
	std::cout << std::endl;

	std::cout << range_length(r) << std::endl;

	histogram<float> hist;
	hist.init(r[0], r[r.size()-1], (r[r.size()-1] - r[0]) * 2);

	for (int i = 0; i < 100000000; i++) {
		hist.record(range_random(r));
	}

	hist.print_frequencies(std::cout);

	return;
	*/

	Pose pose, start_pose;
	Score_weight_map weight_map;
	setup_score_weight_map(weight_map, score12);
	float score;
	utility::vector0<int> res;
	FArray1D_float angles(2), lowangles(2), den(2);
	//float en, en_fc, lowen;
	//bool gfrag = false;
	std::ofstream func, func_fc, dfunc1, dfunc2, trace;
	//int const ang_max = 180;
	std::cout << "init startfile" << std::endl;
	std::string startfile = files_paths::start_path + files_paths::start_file +
	                        ".pdb";
	std::cout << "init movie" << std::endl;
	pose_ns::Movie movie;

	std::cout << "Running Backrub Test" << std::endl;

	pose_from_pdb(pose, startfile, true, false,
	              files_paths::io_options_bool::read_all_chains,
	              files_paths::protein_chain);

	std::cout << "init rotcontrol" << std::endl;
	pose_ns::Rotamer_controller rotcontrol;
	/*
	rotcontrol.init(&pose);
	param::MAXROTPERAA() = 5000;
	rotcontrol.set_include_current(false);
	rotcontrol.init_rotamers();
	std::cout << "Including Current: " << rotcontrol.include_current() << std::endl;
	std::cout << "Total Rotamers: " << rotcontrol.nrotamers() << std::endl;
	utility::vector1<int> closerot = rotcontrol.closest_rotamer(pose);
	for (unsigned int i = 1; i < closerot.size(); i++) {
		if (rotcontrol.nrotpos(i)) {
			rotcontrol.set_rotamer(i, closerot[i]);
		}
		//std::cout << closerot[i] << "\t" << rotcontrol.nrotpos(i) << std::endl;
	}
	pose.dump_pdb("rotamerized.pdb");

	return;
	*/

	weight_map.set_weight(BOND_ANGLE, 1.0);
	std::cout << "BOND_ANGLE weight: " << weight_map.get_weight(BOND_ANGLE) << std::endl;

	score = pose.score(weight_map);
	std::cout << pose.show_scores() << std::endl;
	//for (int i = 1; i < pose.total_residue(); i++) {
	//	std::cout << i << "\t" << pose.get_1D_score(BONDANGLEENERGY)(i) << std::endl;
	//}
	replace_cbha_pose(pose);
	score = pose.score(weight_map);
	std::cout << pose.show_scores() << std::endl;
	//for (int i = 1; i < pose.total_residue(); i++) {
	//	std::cout << i << "\t" << pose.get_1D_score(BONDANGLEENERGY)(i) << std::endl;
	//}
	//dump_scorefxn_weights(std::cout);

	//backrub_dD_debug(pose, 166, 176, radians((float)52), std::cout);
	//backrub_dD_debug(pose, 166, 176, radians((float)53), std::cout);
	//backrub_dD_debug(pose, 166, 176, radians((float)54), std::cout);
	//backrub_dD_debug(pose, 166, 176, 0.9250245, std::cout);
	//backrub_dD_debug(pose, 166, 176, 1.9024089, std::cout);



	for (int i = 1; i < pose.total_residue(); i++) {
		for (int j = i+1; j < pose.total_residue(); j++) {
	//for (int i = 2; i < 3; i++) {
		//for (int j = 5; j < 6; j++) {
			std::stringstream fnamestream;
			fnamestream << "dD2/" << i << "-" << j << ".tsv";
			std::string fname(fnamestream.str());
			trace.open(fname.c_str());
			backrub_dD_write(pose, i, j, trace);
			trace.close();
		}
	}

	return;
	float rama1, rama2, drama_dphi, drama_dpsi;
	eval_rama_score_residue(pose.res(166), pose.phi(166), pose.psi(166),
	                        pose.secstruct(166), rama1, drama_dphi, drama_dpsi);
	eval_rama_score_residue(pose.res(176), pose.phi(176), pose.psi(176),
	                        pose.secstruct(176), rama2, drama_dphi, drama_dpsi);
	std::cout << param_aa::aa_name3(pose.res(166)) << "\t"
	          << param_aa::aa_name3(pose.res(176)) << std::endl;
	score = pose.score(weight_map);
	movie.write_frame(pose);
	std::cout << 0 << "\t" << pose.get_1D_score(BONDANGLEENERGY)(166) << "\t" << rama1
	          << "\t" << pose.get_1D_score(BONDANGLEENERGY)(176) << "\t" << rama2
	          << std::endl;
	for (int i = 0; i < 50; i++) {
		eval_rama_score_residue(pose.res(166), pose.phi(166), pose.psi(166),
		                        pose.secstruct(166), rama1, drama_dphi, drama_dpsi);
		eval_rama_score_residue(pose.res(176), pose.phi(176), pose.psi(176),
		                        pose.secstruct(176), rama2, drama_dphi, drama_dpsi);
		backrub_rot_pose(pose, 166, 176, radians(1.0));
		score = pose.score(weight_map);
		movie.increment_frame();
		movie.write_frame(pose);
		std::cout << i+1 << "\t" << pose.get_1D_score(BONDANGLEENERGY)(166) << "\t" << rama1
		          << "\t" << pose.get_1D_score(BONDANGLEENERGY)(176) << "\t" << rama2
		          << std::endl;
	}

	return;

	pose.dump_pdb("old.pdb");

	utility::vector1<bool> chain_breaks = pose_chain_breaks(pose);
	std::cout << "Chain Breaks Detected:" << std::endl;
	for (unsigned int i = 1; i <= chain_breaks.size(); i++)
		std::cout << i << "\t" << chain_breaks[i] << std::endl;

	return;

	score = pose.score(score12);
	std::cout << pose.show_scores() << std::endl;

	replace_cbha_pose(pose);
	std::string opt_filename = startfile;
	opt_filename.replace(opt_filename.size()-4, 8, "_opt.pdb");
	pose.dump_pdb(opt_filename);
	return;
	FArray1D_float cbold(3);
	score = pose.score(score12);
	std::cout << pose.show_scores() << std::endl;

	start_pose = pose;
	pose.set_start_pose(start_pose);

	pose.dump_pdb("prerotamer.pdb");
	rotcontrol.init(&pose);
	std::cout << "Total Rotamers: " << rotcontrol.nrotamers() << std::endl;
	//utility::vector1<int> closerot = rotcontrol.closest_rotamer(pose);
	//for (unsigned int i = 1; i < closerot.size(); i++) {
	//	std::cout << closerot[i] << std::endl;
	//}

	rotcontrol.print(15);
	for (int i = 1; i <= rotcontrol.nrotpos(15); i++) {
		std::cout << "Setting to rotamer " << i << std::endl;
		rotcontrol.set_rotamer(pose, 15, i);
		std::cout << param_aa::aa_name3(pose.res(15)) << std::setw(4) << 15 << ":";
		int const nchi = aaproperties_pack::nchi(pose.res(15), pose.res_variant(15));
		for (int j = 1; j <= nchi; j++) {
			std::cout << std::fixed << std::setprecision(1) << std::setw(7)
			          << pose.chi(j, 15) << std::setprecision(2);
		}
		std::cout << std::endl;
	}
	score = pose.score(score12);

	std::cout << pose.start_pose_exists() << std::endl;

	std::cout << backrub_get_angle(pose.start_pose(), pose, 4, 6) << std::endl;
	backrub_rot_pose(pose, 4, 6, -.35);
	std::cout << backrub_get_angle(pose.start_pose(), pose, 4, 6) << std::endl;

	/*
	std::cout << "Residue 75: " << rotcontrol.nrotpos(75) << std::endl;
	rotcontrol.set_rotamer(75, 1);
	std::cout << "Residue 76: " << rotcontrol.nrotpos(76) << std::endl;
	rotcontrol.set_rotamer(76, 1);
	std::cout << "Residue 77: " << rotcontrol.nrotpos(77) << std::endl;
	rotcontrol.set_rotamer(pose, 77, 4);
	pose.dump_pdb("rotamer.pdb");



	pose_read_resfile(pose, files_paths::resfile);
	for (int i = 1; i <= pose.total_residue(); i++) {
		std::cout << i << "\t" << pose.res_variant(i) << "\t"
		          << pose.get_allow_chi_move(i) << "\t"
		          << pose.get_allow_bb_move(i) << std::endl;
	}

	backrub_rot_pose(pose, 4, 6, -.35);

	res.push_back(4);
	res.push_back(5);
	res.push_back(5);
	res.push_back(6);
	backrub_ns::Backrub_eval energy(&pose, res);

	trace.open("mintrace.txt");
	energy.set_func_trace(trace);

	std::cout << "Minimizing..." << std::endl;
	angles(1) = radians(-50);
	angles(2) = radians(150);
	energy.minimize(angles, "dfpmin");
	std::cout << "Angle 1: " << degrees(angles(1)) << "   Angle 2: "
	          << degrees(angles(2)) << std::endl;
	std::cout << "Function Evaluations: " << get_func_evals() << std::endl;

	trace.close();

	rotcontrol.print();
	*/
	/*
	for (int i = 15; i <= 15; i++) {
		std::cout << param_aa::aa_name3(pose.res(i)) << std::setw(4) << i << ":";
		int const nchi = aaproperties_pack::nchi(pose.res(i), pose.res_variant(i));
		for (int j = 1; j <= nchi; j++) {
			std::cout << std::fixed << std::setprecision(1) << std::setw(7)
			          << pose.chi(j, i) << std::setprecision(2);
		}
		std::cout << std::endl;
	}
	*/
	/*
	lowen = 100;
	std::cout << "Starting search" << std::endl;
	func.open("func.txt");
	func_fc.open("func_fc.txt");
	dfunc1.open("dfunc1.txt");
	dfunc2.open("dfunc2.txt");
	for (int i = -ang_max; i < ang_max; i++) {
		angles(2) = radians((float)i);
		for (int j = -ang_max; j < ang_max; j++) {
			angles(1) = radians((float)j);
			energy.set_use_full_coord(false);
			en = energy.func(angles, gfrag);
			if (en < lowen) {
				lowen = en;
				lowangles = angles;
			}
			func << en << std::endl;
			energy.set_use_full_coord(true);
			en_fc = energy.func(angles, gfrag);
			func_fc << en_fc << std::endl;
			energy.dfunc(angles, den, 2);
			dfunc1 << den(1) << std::endl;
			dfunc2 << den(2) << std::endl;
		}
	}
	func.close();
	func_fc.close();
	dfunc1.close();
	dfunc2.close();
	std::cout << "Lowest Energy: " << lowen << "\t" << degrees(lowangles(1))
	          << "\t" << degrees(lowangles(2)) << std::endl;

	pose.dump_pdb("new1.pdb");
	backrub_rot_pose(pose, 4, 5, lowangles(1));
	backrub_rot_pose(pose, 5, 6, lowangles(2));

	score = pose.score(score12);
	std::cout << pose.show_scores() << std::endl;

	pose.dump_pdb("new2.pdb");
	*/
}

////////////////////////////////////////////////////////////////////////////////
/// @begin pose_backrub_mc
///
/// @brief
/// perform Monte Carlo using backrub moves
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 25, 2007
////////////////////////////////////////////////////////////////////////////////
void
pose_backrub_mc(
	int const n_struct,
	int const nstart
)
{
	pose_ns::Pose pose;
	pose_ns::Pose start_pose;
	pose_ns::Pose low_pose;
	pose_ns::Score_weight_map weight_map;
	std::string startfile = files_paths::start_path + files_paths::start_file + ".pdb";

	int ntrials = 1000;
	float mc_temp = 0.6;
	float mc_temp_initial = 0.6;
	float bond_angle_weight = 1;
	float score;

	if (nstart) {} // avoid unused parameter warning

	std::cout << "Running Backrub Monte Carlo" << std::endl;

	intafteroption("ntrials", ntrials, ntrials);
	realafteroption("mc_temp", mc_temp, mc_temp);
	realafteroption("mc_temp_initial", mc_temp, mc_temp_initial);
	float const gamma = std::pow( (mc_temp/mc_temp_initial), (1.0f/(ntrials-1)) );
	bool initial_pack = truefalseoption("initial_pack");
	bool iterative_design = truefalseoption("iterative_design");

	setup_score_weight_map(weight_map, score12);
	realafteroption("bond_angle_weight", bond_angle_weight, bond_angle_weight);
	weight_map.set_weight(pose_ns::BOND_ANGLE, bond_angle_weight);

	pose_from_pdb(pose, startfile, true, false,
	              files_paths::io_options_bool::read_all_chains,
	              files_paths::protein_chain);

	Pdb_info pdb_info;
	pdb_info.pdb_info_from_global();
	pose.set_pdb_information(pdb_info);

	//pose.setup_atom_tree();

	pose_ns::Backrub_controller backcontrol;
	backcontrol.set_pose(&pose);
	backcontrol.init_with_args();

	score = pose.score(weight_map);
	std::cout << std::endl << "Initial Scores:" << std::endl;
	std::cout << pose.show_scores() << std::endl;

	replace_cbha_pose(pose);

	score = pose.score(weight_map);
	std::cout << std::endl << "Scores After CB/HA Optimization:" << std::endl;
	std::cout << pose.show_scores() << std::endl;

	start_pose = pose;
	pose.set_start_pose(start_pose);
	if (files_paths::query_defined) {
		dump_full_coord(pose, "_initial", true);
	} else {
		pose.dump_scored_pdb("backrub_initial.pdb", weight_map);
	}

	DesignMap & dm = backcontrol.rotcontrol().packer_task().get_designmap();
	for (int i = 1; i <= pose.total_residue(); i++) {
		if (dm.repack_residue(i) && !dm.get(i, pose.res(i))) {
			initial_pack = true;
			std::cout << "Amino acid at position " << i << " ("
			          << param_aa::aa_name1(pose.res(i)) << ") not allowed in resfile: ";
			for (int j = 1; j <= param::MAX_AA()(); ++j) {
				if (dm.get(i, j)) std::cout << param_aa::aa_name1(j);
			}
			std::cout << std::endl;
		}
	}

	backcontrol.init_histograms(10);

	for (int i = 1; i <= n_struct; i++) {

		if (initial_pack && (i == 1 || !iterative_design)) {

			std::cout << std::endl << "Running initial pack/redesign" << std::endl;

			backcontrol.rotcontrol().pack(pose);
			if (iterative_design) {
				std::cout << "Initial designed sequence: ";
				backcontrol.rotcontrol().print_designable_sequence(pose, std::cout);
				std::cout << std::endl;
			}

			score = pose.score(weight_map);
			std::cout << std::endl << "Scores After Initial Pack/Redesign:" << std::endl;
			std::cout << pose.show_scores() << std::endl;
		}

		std::cout << std::endl << "Running " << ntrials << " Monte Carlo trials"
		          << std::endl;
		pose_ns::Monte_carlo mc(pose, weight_map, mc_temp_initial);
		int low_trial(0);

		for (int i = 1; i <= ntrials; i++) {
			backcontrol.trial(mc);
			if (pose.get_extra_score("MC_ACCEPTED") == 3) low_trial = i;
			mc.set_temperature(mc.temperature()*gamma);
		}

		std::cout << std::endl;
		mc.show_counters();
		std::cout << std::endl;
		backcontrol.print_histograms(std::cout);
		std::cout << std::endl;
		backcontrol.print_bond_angle_stat(pose, mc, std::cout);

		std::cout << std::endl;
		if (iterative_design) {
			std::cout << "Running full pack/redesign on last pose" << std::endl;
			backcontrol.rotcontrol().pack(pose);
			std::cout << "Last designed sequence: ";
			backcontrol.rotcontrol().print_designable_sequence(pose, std::cout);
			std::cout << std::endl;
		}
		score = pose.score(weight_map);
		std::cout << std::endl << "Last Scores:" << std::endl << pose.show_scores() << std::endl;

		utility::io::ozstream out;

		if (files_paths::query_defined) {
			open_full_coord_outfile(out, "last");
		} else {
			pose.open_scored_pdb_outfile("backrub_last.pdb", out);
		}
		out << "REMARK  99 SAMPLE  " << ntrials << std::endl;
		out << "REMARK  99 SCORE   " << score << std::endl;
		out << "REMARK  99 CA_RMSD " << CA_rmsd(pose, start_pose) << std::endl;
		if (files_paths::query_defined) {
			dump_full_coord(pose, out);
			out << pose.show_scores() << std::endl;
		} else {
			pose.dump_scored_pdb(out, weight_map);
		}
		out.close();
		out.clear();

		low_pose = mc.low_pose();

		std::cout << std::endl;
		if (iterative_design) {
			std::cout << "Running full pack/redesign on low pose" << std::endl;
			backcontrol.rotcontrol().pack(low_pose);
			std::cout << "Low designed sequence: ";
			backcontrol.rotcontrol().print_designable_sequence(low_pose, std::cout);
			std::cout << std::endl;
		}
		score = low_pose.score(weight_map);
		std::cout << std::endl << "Low Scores:" << std::endl << low_pose.show_scores() << std::endl;

		if (files_paths::query_defined) {
			open_full_coord_outfile(out, "low");
		} else {
			low_pose.open_scored_pdb_outfile("backrub_low.pdb", out);
		}
		out << "REMARK  99 SAMPLE  " << low_trial << std::endl;
		out << "REMARK  99 SCORE   " << score << std::endl;
		out << "REMARK  99 CA_RMSD " << CA_rmsd(low_pose, start_pose) << std::endl;
		if (files_paths::query_defined) {
			dump_full_coord(low_pose, out);
			out << low_pose.show_scores() << std::endl;
		} else {
			low_pose.dump_scored_pdb(out, weight_map);
		}
		out.close();
		out.clear();

		if (!files_paths::query_defined) break;

		if (iterative_design) {

			std::cout << std::endl;
			if (low_pose.get_0D_score(pose_ns::SCORE) < pose.get_0D_score(pose_ns::SCORE)) {
				std::cout << "Low score better than last, retaining low for next iteration" << std::endl;
				pose = low_pose;
			} else {
				std::cout << "Last score better than low, retaining last for next iteration" << std::endl;
			}

		} else {

			// reset the pose back to its starting state
			pose = start_pose;
			pose.set_start_pose(start_pose);
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin pose_read_resfile
///
/// @brief
/// apply resfile rules to the Pose object backbone/chi move allowed fields
///
/// @detailed
/// allow_chi_move is set to false if the id (columns 14-18) is "NATRO".
/// Otherwise it is set to true.
///
/// allow_bb_move is set to true if the 19th column of the file is set to "B".
/// Otherwise, it is set to false. This is a modification of the file format
/// that does not interfere with the existing uses.
///
/// @param[in,out] pose - Pose object
/// @param[in] resfile - path to the resfile
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified May 5, 2006
////////////////////////////////////////////////////////////////////////////////
void
pose_read_resfile(
	pose_ns::Pose & pose,
	std::string resfile
)
{
	int seqpos;
	std::string charid;
	std::string bbflag;
	utility::io::izstream data_x;

	if (resfile == "none") {
		return;
	}	else if (resfile == "auto") {
		data_x.open(files_paths::start_path + files_paths::start_file + ".resfile");
	} else {
		data_x.open(resfile);
	}

	if (!data_x) {
		std::cout << "Open failed for file: " << data_x.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// read through header information
	for (int i = 1; i <= 26; i++) {
		data_x >> skip;
	}

	for (int i = 1; i <= pose.total_residue(); i++) {
		data_x >> skip(3) >> bite(4, seqpos) >> skip(6)
		       >> bite(5, charid) >> bite(1, bbflag) >> skip;
		if (seqpos != i) {
			std::cout << "problem reading resfile" << std::endl;
			std::cout << "seqpos is " << seqpos << " and i is " << i << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		// set allow_chi_move
		if (charid == "NATRO") {
			pose.set_allow_chi_move(i, false);
		} else {
			pose.set_allow_chi_move(i, true);
		}

		// set_allow_bb_move
		if (bbflag == "B") {
			pose.set_allow_bb_move(i, true);
		} else {
			pose.set_allow_bb_move(i, false);
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin pose_pick_free_sc
///
/// @brief
/// pick a random number of free side chains from the pose object
///
/// @detailed
///
/// @param[in] pose - Pose object
/// @param[in] start_res - starting residue
/// @param[in] end_res - ending residue
/// @param[in] max_res - maximum number of side chains to return
/// @param[in] prob_zero - probability of returning zero side chains
///
/// @global_read
///
/// @global_write
///
/// @return
/// vector with the side chain numbers
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified July 14, 2006
////////////////////////////////////////////////////////////////////////////////
utility::vector0<int>
pose_pick_free_sc(
	pose_ns::Pose const & pose,
	utility::vector1<int> const & nrotpos,
	int const start_res,
	int const end_res,
	int max_res,
	float const prob_zero
)
{
	utility::vector0<int> side_chains;
	utility::vector0<int> free_sc;

	if (ran3() < prob_zero || max_res == 0) return side_chains;

	for (int i = start_res; i <= end_res; i++) {
		if (pose.get_allow_chi_move(i) && nrotpos[i] > 1) {
			free_sc.push_back(i);
		}
	}
	if (max_res > (signed)free_sc.size()) max_res = free_sc.size();

	int num_res = 1;
	if (max_res > 1 && ran3() < 0.25) {
		num_res = 2;
	}

	if (num_res >= (signed)free_sc.size()) return free_sc;

	std::set<int> side_chains_set;

	while((signed)side_chains_set.size() < num_res) {
		side_chains_set.insert(free_sc[random_range(0, free_sc.size()-1)]);
	}

	side_chains.insert(side_chains.end(), side_chains_set.begin(),
	                   side_chains_set.end());

	return side_chains;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin pose_chain_breaks
///
/// @brief
/// compute a list of chain breaks
///
/// @detailed
///
/// @param[in] pose - Pose object
/// @param[in] chain_break_cutoff - squared distance cutoff
///
/// @global_read
///
/// @global_write
///
/// @return
/// a boolean vector indicating a chain break after the given residue position
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified July 14, 2006
////////////////////////////////////////////////////////////////////////////////
utility::vector1<bool>
pose_chain_breaks(
	pose_ns::Pose const & pose,
	float const chain_break_cutoff // = 4.0
)
{
	utility::vector1<bool> chain_breaks(pose.total_residue()-1, false);
	FArray3D_float const Eposition = pose.Eposition();

	for (int i = 1; i < pose.total_residue(); i++) {

		float const Edif1 = Eposition(1,4,i) - Eposition(1,1,i+1);
		float const Edif2 = Eposition(2,4,i) - Eposition(2,1,i+1);
		float const Edif3 = Eposition(3,4,i) - Eposition(3,1,i+1);
		float const dist2 = ( Edif1 * Edif1 ) + ( Edif2 * Edif2 ) +
		                    ( Edif3 * Edif3 );
		if (dist2 > chain_break_cutoff) {
			chain_breaks[i] = true;
		} else if ( dist2 < 0.1 ) {
			std::cout << "WARNING: zero length peptide bond: " << i << '-'
			          << i+1 << std::endl;
			std::cout << SS(Eposition(1,4,i)) << SS(Eposition(1,1,i+1))
			          << std::endl;
			chain_breaks[i] = true;
		}
	}

	return chain_breaks;
}

/// @brief create coordinate constraints for a residue using a reference pose
/// @param[in] cstset - constraint set to store the additional constraints
/// @param[in] src - the Pose to be constrained
/// @param[in] src_pos - the Pose sequence position to be constrained
/// @param[in] dest - the Pose with the coordinates for the constraint
/// @param[in] dest_pos - the Pose sequence position to be constrained to
/// @param[in] level - level of constraint (5: all, 4: heavy, 3: N CA C O CB, 2: N CA C, 1: C)
void
coord_cst_residue(
	cst_set_ns::Cst_set & cstset,
	pose_ns::Pose const & src,
	int const src_pos,
	pose_ns::Pose const & dest,
	int const dest_pos,
	int const level
)
{
	assert(src.res(src_pos) == dest.res(dest_pos) &&
	       src.res_variant(src_pos) == dest.res_variant(dest_pos));

	int const aa(src.res(src_pos));
	int const aav(src.res_variant(src_pos));
	int const natoms(aaproperties_pack::natoms(aa, aav));

	for (int i = 1; i <= natoms; i++) {

		if (level >= 5) { // all atoms
			// no action required
		} else if (level == 4) { // heavy atoms only
			if (aaproperties_pack::atom_name(i, aa, aav)[1] == 'H') continue;
		} else if (level == 3) { // N CA C O CB atoms only
			if (i > 5) continue;
		} else if (level == 2) { // N CA C atoms only
			if (i > 3) continue;
		} else if (level == 1) { // CA atoms only
			if (aaproperties_pack::atom_name(i, aa, aav) != " CA ") continue;
		}

		std::cout << "COORD CST: " << src_pos << "\t" << i << "\t" << aaproperties_pack::atom_name(i, aa, aav) << std::endl;
		kin::Atom_id atomid(src.atom_tree()->atom(i, src_pos)->atom_id);
		cstset.add_coordinate_constraint(atomid, dest.full_coord(i, dest_pos));
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin pose_backrub_allowed_moves
///
/// @brief
/// compute alist of allowed starting residue positions for backrub moves
///
/// @detailed
///
/// @param[in] pose - Pose object
/// @param[in] max_res - maximum size of backrub move
///
/// @global_read
///
/// @global_write
///
/// @return
/// a vector of vectors of backrub starting positions (The first vector contains
/// starting positions for moves of size 2, and so on.)
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified May 5, 2006
////////////////////////////////////////////////////////////////////////////////
utility::vector0<utility::vector0<int> >
pose_backrub_allowed_moves(
	pose_ns::Pose & pose,
	const int max_res // = 3
)
{
	utility::vector0<utility::vector0<int> > allowed_moves(max_res-1);
	utility::vector1<bool> chain_breaks = pose_chain_breaks(pose);
	bool allowed;

	for (int nres = 2; nres <= max_res; nres++) {
		for (int i = 1; i <= pose.total_residue()-nres+1; i++) {
			std::cout << "loop i" << i << std::endl;
			allowed = true;
			for (int j = i; j <= i + nres - 1; j++) {
				std::cout << "loop j" << j << std::endl;
				if (!pose.get_allow_bb_move(j)) {
					allowed = false;
					break;
				}
				if (j < i + nres - 1 && chain_breaks[j]) {
					allowed = false;
					break;
				}
			}
			if (pose.res(i) == param_aa::aa_pro) allowed = false;
			if (pose.res(i + nres - 1) == param_aa::aa_pro) allowed = false;
			if (allowed) allowed_moves[nres-2].push_back(i);
		}
	}

	return allowed_moves;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin backrub_rot_pose
///
/// @brief
/// do backrub motion on a Pose object
///
/// @detailed
/// angle is clockwise looking from CA[start] to CA[end]
///
/// @param[in,out] pose - Pose object
/// @param[in] start - starting residue position
/// @param[in] end - ending residue position
/// @param[in] angle - angle (radians)
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 19, 2006
////////////////////////////////////////////////////////////////////////////////
void backrub_rot_pose(
	pose_ns::Pose & pose,
	int const start,
	int const end,
	float const angle
)
{
	using namespace pose_ns;

	int const seg_size = end - start + 1;
	FArray3D_float fcoord = pose.full_coord();

	backrub_rot_full_coord(fcoord, pose.res(), pose.res_variant(), start, end,
	                       angle);

	pose.set_segment_full_coord(seg_size, start, fcoord(1,1,start));
}

////////////////////////////////////////////////////////////////////////////////
/// @begin backrub_rot_Eposition
///
/// @brief
/// do backrub motion on an Eposition array
///
/// @detailed
/// angle is clockwise looking from CA[start] to CA[end]
///
/// @param[in,out] Eposition - input array
/// @param[in] aa - array of amino acid numbers
/// @param[in] aav - array of amino acid variant numbers
/// @param[in] start - starting residue position
/// @param[in] end - ending residue position
/// @param[in] angle - angle (radians)
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 19, 2006
////////////////////////////////////////////////////////////////////////////////
void backrub_rot_Eposition(
	FArray3D_float & Eposition,
	int const start,
	int const end,
	float const angle
)
{
	FArray1D_float vec(3);
	FArray2D_float mat(3,3);
	int i, j;

	// Get rotation matrix/vector
	getrot_bk(Eposition(1,2,start), Eposition(1,2,end), angle, mat, vec);

	// Move atoms in start and end residues
	move_bk(Eposition(1,4,start), mat, vec); // Start C
	move_bk(Eposition(1,5,start), mat, vec); // Start O
	move_bk(Eposition(1,1,end), mat, vec); // End N

	// Move all atoms in between
	for (i = start + 1; i < end; i++) {
		for (j = 1; j <= (signed)Eposition.size2(); j++) {
			move_bk(Eposition(1,j,i), mat, vec);
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin backrub_rot_full_coord
///
/// @brief
/// do backrub motion on a full_coord array
///
/// @detailed
/// angle is clockwise looking from CA[start] to CA[end]
///
/// @param[in,out] full_coord - input array
/// @param[in] aa - array of amino acid numbers
/// @param[in] aav - array of amino acid variant numbers
/// @param[in] start - starting residue position
/// @param[in] end - ending residue position
/// @param[in] angle - angle (radians)
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 19, 2006
////////////////////////////////////////////////////////////////////////////////
void backrub_rot_full_coord(
	FArray3D_float & full_coord,
	FArray1D_int const & aa,
	FArray1D_int const & aav,
	int const start,
	int const end,
	float const angle
)
{
	FArray1D_float vec(3);
	FArray2D_float mat(3,3);
	int i, j, num_atoms;

	// Get rotation matrix/vector
	getrot_bk(full_coord(1,2,start), full_coord(1,2,end), angle, mat, vec);

	// Move atoms in start and end residues
	move_bk(full_coord(1,3,start), mat, vec); // Start C
	move_bk(full_coord(1,4,start), mat, vec); // Start 0
	move_bk(full_coord(1,1,end), mat, vec); // End N
	int const HNpos(aaproperties_pack::HNpos(aa(end),aav(end)));
	if (HNpos) move_bk(full_coord(1,HNpos,end), mat, vec); // End HN

	// Optimize CB/HA at start and end residues
	replace_cb(aa(start), aav(start), full_coord(1,1,start));
	replace_ha(aa(start), aav(start), full_coord(1,1,start));
	replace_cb(aa(end), aav(end), full_coord(1,1,end));
	replace_ha(aa(end), aav(end), full_coord(1,1,end));

	// Move all atoms in between
	for (i = start + 1; i < end; i++) {
		num_atoms = aaproperties_pack::natoms(aa(i), aav(i));
		for (j = 1; j <= num_atoms; j++) {
			move_bk(full_coord(1,j,i), mat, vec);
		}
	}

	// Check bond lengths
	for (i = start; i <= end; i++) {
		assert_bond_length_res(aa(i), aav(i), full_coord(1,1,i));
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_ca_angle_pose
///
/// @brief
/// set CA angle on a Pose object
///
/// @detailed
///
/// @param[in,out] pose - Pose object
/// @param[in] res - residue position
/// @param[in] angle - angle (radians)
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified January 16, 2006
////////////////////////////////////////////////////////////////////////////////
void set_ca_angle_pose(
	pose_ns::Pose & pose,
	int const res,
	float const angle
)
{
	using namespace pose_ns;

	int const seg_size = pose.total_residue() - res + 1;
	FArray3D_float fcoord = pose.full_coord();

	set_ca_angle_full_coord(fcoord, pose.res(), pose.res_variant(), res, pose.total_residue(),
	                        angle);

	pose.set_segment_full_coord(seg_size, res, fcoord(1,1,res));
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_ca_angle_full_coord
///
/// @brief
/// set CA angle on a full_coord array
///
/// @detailed
///
/// @param[in,out] full_coord - input array
/// @param[in] aa - array of amino acid numbers
/// @param[in] aav - array of amino acid variant numbers
/// @param[in] res - starting residue position
/// @param[in] totres - ending residue position
/// @param[in] angle - angle (radians)
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified January 16, 2007
////////////////////////////////////////////////////////////////////////////////
void set_ca_angle_full_coord(
	FArray3D_float & full_coord,
	FArray1D_int const & aa,
	FArray1D_int const & aav,
	int const res,
	int const totres,
	float const angle
)
{
	using namespace aaproperties_pack;
	using numeric::conversions::radians;

	FArray1D_float vec(3);
	FArray2D_float mat(3,3);
	FArray1D_float can(3);
	FArray1D_float cac(3);
	FArray1D_float norm(3);

	for (int i = 1; i <= 3; i++) {
		can(i) = full_coord(i,1,res) - full_coord(i,2,res);
		cac(i) = full_coord(i,3,res) - full_coord(i,2,res);
	}

	// generate the normal line of the N-CA-C plane
	cross_bk(can,cac,norm);
	for (int i = 1; i <= 3; i++) {
		norm(i) += full_coord(i,2,res);
	}

	// Previous N-CA-C angle
	float pangle = radians(vec_angle(full_coord(1,1,res), full_coord(1,2,res), full_coord(1,3,res)));

	// Get rotation matrix/vector
	getrot_bk(full_coord(1,2,res), norm, angle - pangle, mat, vec);

	// Move atoms in pivoting residue
	move_bk(full_coord(1,3,res), mat, vec); // Pivoting C
	move_bk(full_coord(1,4,res), mat, vec); // Pivoting 0

	// Optimize CB/HA at pivoting residue
	replace_cb(aa(res), aav(res), full_coord(1,1,res));
	replace_ha(aa(res), aav(res), full_coord(1,1,res));

	// Move all atoms beyond pivoting residue
	for (int i = res + 1; i <= totres; i++) {
		int const num_atoms = natoms(aa(i), aav(i));
		for (int j = 1; j <= num_atoms; j++) {
			move_bk(full_coord(1,j,i), mat, vec);
		}
	}

	// Check bond lengths
	for (int i = res; i <= totres; i++) {
		assert_bond_length_res(aa(i), aav(i), full_coord(1,1,i));
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin copy_full_coord_Eposition
///
/// @brief
/// copy atoms from full_coord array to Eposition array
///
/// @detailed
///
/// @param[in] full_coord - input array
/// @param[in,out] Eposition - output array
/// @param[in] start - starting residue position
/// @param[in] end - ending residue position
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 19, 2006
////////////////////////////////////////////////////////////////////////////////
void copy_full_coord_Eposition(
	FArray3D_float const & full_coord,
	FArray3D_float & Eposition,
	int const start,
	int const end
)
{
	int i, j;

	for (i = start; i <= end; i++) {
		for (j = 1; j <= 3; j++) {
			Eposition(j,1,i) = full_coord(j,1,i); // N
			Eposition(j,2,i) = full_coord(j,2,i); // CA
			Eposition(j,3,i) = full_coord(j,5,i); // CB
			Eposition(j,4,i) = full_coord(j,3,i); // C
			Eposition(j,5,i) = full_coord(j,4,i); // O
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin replace_cbha_pose
///
/// @brief
/// replace CB, sidechain, and HA atoms in a Pose object with optimal positions
///
/// @detailed
/// only residues with chi moves allowed will be altered
///
/// @param[in,out] pose - Pose object
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 19, 2006
////////////////////////////////////////////////////////////////////////////////
void replace_cbha_pose(
	pose_ns::Pose & pose
)
{
	FArray3D_float pose_full_coord = pose.full_coord();
	int const num_res = pose.total_residue();

	for (int i = 1; i <= num_res; i++) {
		if ((pose.get_allow_chi_move(i) || pose.get_allow_bb_move(i))
		    && pose.res(i) != param_aa::aa_pro) {
			replace_cb(pose.res(i), pose.res_variant(i), pose_full_coord(1,1,i));
			replace_ha(pose.res(i), pose.res_variant(i), pose_full_coord(1,1,i));
			//pose.copy_sidechain(i, pose.res(i), pose.res_variant(i),
			//                    pose_full_coord(1,1,i));
			pose.set_segment_full_coord(1, i, pose_full_coord(1,1,i));
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin replace_cb
///
/// @brief
/// reposition CB (and sidechain) to minimize bond angle energies
///
/// @detailed
///
/// @param[in] aa - amino acid type
/// @param[in] aav - amino acid variant
/// @param[in,out] res_coor - residue atom coordinates
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 13, 2006
////////////////////////////////////////////////////////////////////////////////
void replace_cb(
	int const aa,
	int const aav,
	FArray2Da_float res_coor
)
{
	using namespace aaproperties_pack;
	using namespace param;
	using namespace param_aa;
	using namespace numeric::conversions;

	bond_angle::initialize_bond_angle_params();

	FArray1D_float cb(3);
	FArray1D_float vec(3);
	FArray2D_float mat(3,3);
	float thetaCB, phiCB, thetaHA, phiHA, lenCB, chi1 = 0;
	const int CBpos ( bond_angle::CBpos[aa] );
	const int HNpos ( aaproperties_pack::HNpos (aa,aav));
	const int HApos ( aaproperties_pack::HApos (aa,aav));
	const int natoms( aaproperties_pack::natoms(aa,aav));

	if ( !is_protein(aa) ) {
		std::cout << "WARNING: Attempt to replace CB position for non-amino acid"
		          << std::endl;
		return;
	}

	res_coor.dimension(3, MAX_ATOM());

	// save original chi1
	if (aaproperties_pack::nchi(aa, aav) >= 1) {
		chi1 = get_chi_angle(aa, aav, res_coor, 1);
	}

	// place CB using polar coordinates
	get_cbha_angles(aa, aav, res_coor, thetaCB, phiCB, thetaHA, phiHA);
	distance_bk(res_coor(1,2), res_coor(1,CBpos), lenCB);
	place_atom_polar(res_coor(1,1), res_coor(1,2), res_coor(1,3), cb, lenCB,
	                 thetaCB, phiCB);

	// check: get CB displacement
	//distance_bk(cb, res_coor(1,CBpos), lenCB);
	//std::cout << "CB displacement: " << lenCB << std::endl;

	// rotate sidechain atoms to follow CB
	lineup_bk(res_coor(1,2), res_coor(1,CBpos), res_coor(1,2), cb, mat, vec);
	for (int i = 5; i <= natoms; i++) {
		if (i != HNpos && i != HApos) {
			move_bk(res_coor(1,i), mat, vec);
		}
	}

	// restore original chi1
	if (aaproperties_pack::nchi(aa, aav) >= 1) {

		FArray2D_float mat( 3, 3 );
		FArray1D_float vec( 3 );

		//bk retrieve atoms that determine chi angle
		int const c1 = chi_atoms(1,1,aa,aav);
		int const c2 = chi_atoms(2,1,aa,aav);
		int const c3 = chi_atoms(3,1,aa,aav);
		int const c4 = chi_atoms(4,1,aa,aav);

		//bk calculate initial chi angle
		float schi;
		dihedral_bk( res_coor(1,c1), res_coor(1,c2), res_coor(1,c3), res_coor(1,c4), schi );

		//bk rotate angle by new-initial degrees
		float const rot = radians( chi1 - schi );

		//bk generate rotation vector and matrix
		getrot_bk( res_coor(1,c2), res_coor(1,c3), rot, mat, vec );

		//bk move atoms
		for ( int i = 1; i <= natoms; ++i ) {
			if ( chi_required( 1, i, aa, aav ) ) {
				move_bk( res_coor(1,i), mat, vec );
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin replace_ha
///
/// @brief
/// reposition HA to minimize bond angle energies
///
/// @detailed
///
/// @param[in] aa - amino acid type
/// @param[in] aav - amino acid variant
/// @param[in,out] res_coor - residue atom coordinates
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 13, 2006
////////////////////////////////////////////////////////////////////////////////
void replace_ha(
	int const aa,
	int const aav,
	FArray2Da_float res_coor
)
{
	using namespace aaproperties_pack;
	using namespace param;
	using namespace param_aa;
	using namespace numeric::conversions;

	FArray1D_float ha(3);
	float thetaCB, phiCB, thetaHA, phiHA, lenHA;

	if ( !is_protein(aa) ) {
		std::cout << "WARNING: Attempt to replace HA position for non-amino acid"
		          << std::endl;
		return;
	}

	res_coor.dimension(3, MAX_ATOM());

	// place HA using polar coordinates
	get_cbha_angles(aa, aav, res_coor, thetaCB, phiCB, thetaHA, phiHA);
	distance_bk(res_coor(1,2), res_coor(1,HApos(aa,aav)), lenHA);
	place_atom_polar(res_coor(1,1), res_coor(1,2), res_coor(1,3), ha(1), lenHA,
	                 thetaHA, phiHA);

	// check: get HA displacement
	//distance_bk(ha, res_coor(1,HApos(aa,aav)), lenHA);
	//std::cout << "HA displacement: " << lenHA << std::endl;

	// store new HA coordinates
	for (int i = 1; i <= 3; i++) {
		res_coor(i,HApos(aa,aav)) = ha(i);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin place_atom_polar
///
/// @brief
/// place an atom using polar coordinates, using three other atoms as references
///
/// @detailed
///
/// @param[in] a1 - atom along positive z-axis (z > 0)
/// @param[in] a2 - atom at origin
/// @param[in] a3 - atom in the x-z plane (x > 0)
/// @param[out] a4 - position of placed atom
/// @param[in] len - distance from origin
/// @param[in] theta - azimuthal angle (radians)
/// @param[in] phi - polar angle (radians)
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
/// http://mathworld.wolfram.com/SphericalCoordinates.html
///
/// @authors Colin A. Smith
///
/// @last_modified April 13, 2006
////////////////////////////////////////////////////////////////////////////////
void place_atom_polar(
	FArray1Da_float a1,
	FArray1Da_float a2,
	FArray1Da_float a3,
	FArray1Da_float a4,
	float len,
	float theta,
	float phi
)
{
	FArray1D_float vec(3);
	FArray2D_float mat(3,3);
	float len12;

	a1.dimension(3);
	a2.dimension(3);
	a3.dimension(3);
	a4.dimension(3);

	// place a4 along a2 -> a1 vector
	distance_bk(a1, a2, len12);
	for (int i = 1; i <= 3; i++) {
		a4(i) = (a1(i)-a2(i))/len12*len + a2(i);
	}

	// rotate in a1-a2-a3 plane (phi)
	getrot_parallel_plane(a2, a1, a2, a3, phi, mat, vec);
	move_bk(a4, mat, vec);

	// rotate about a2 -> a1 vector (theta)
	getrot_bk(a2, a1, theta, mat, vec);
	move_bk(a4, mat, vec);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin getrot_parallel_plane
///
/// @brief
/// get mat/vec for rotating around a line perpendicular to a plane
///
/// @detailed
///
/// @param[in] a1 - atom in line perpendicular to the plane
/// @param[in] b1 - atom 1 defining the plane
/// @param[in] b2 - atom 2 defining the plane
/// @param[in] b3 - atom 3 defining the plane
/// @param[in] rot_ang - rotation angle (radians)
/// @param[out] mat - rotation matrix
/// @param[out] vec - translation vector
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
/// based on rotate_in_plane()
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 13, 2006
////////////////////////////////////////////////////////////////////////////////
void
getrot_parallel_plane(
	FArray1Da_float a1,
	FArray1Da_float b1,
	FArray1Da_float b2,
	FArray1Da_float b3,
	float & rot_ang, // rotate angle
	FArray2Da_float mat,
	FArray1Da_float vec
)
{
	a1.dimension( 3 );
	b1.dimension( 3 );
	b2.dimension( 3 );
	b3.dimension( 3 );
	mat.dimension( 3, 3 );
	vec.dimension( 3 );

	FArray1D_float b12( 3 );
	FArray1D_float b13( 3 );
	FArray1D_float norm1( 3 );

	// generate the normal line of the plane
	for ( int i = 1; i <= 3; i++ ) {
		b12(i) = b2(i)-b1(i);
		b13(i) = b3(i)-b1(i);
	}
	cross_bk(b13,b12,norm1);
	norm1 += a1;

	// generate rotation vector and matrix
	getrot_bk(a1, norm1, rot_ang, mat, vec);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_chi_angle
///
/// @brief
/// get chi angle with a numerical index
///
/// @detailed
///
/// @param[in] aa - amino acid type
/// @param[in] aav - amino acid variant
/// @param[in] coor - residue atom coordinates
/// @param[in] ichi - chi angle index
///
/// @global_read
///
/// @global_write
///
/// @return
/// the chi angle (degrees)
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 13, 2006
////////////////////////////////////////////////////////////////////////////////
float get_chi_angle(
	int const aa,
	int const aav,
	FArray2Da_float coor,
	int ichi
)
{
	using namespace aaproperties_pack;
	using numeric::conversions::radians;

	coor.dimension(3, param::MAX_ATOM());

	int const c1 = chi_atoms(1,ichi,aa,aav);
	int const c2 = chi_atoms(2,ichi,aa,aav);
	int const c3 = chi_atoms(3,ichi,aa,aav);
	int const c4 = chi_atoms(4,ichi,aa,aav);
	float chi;

	dihedral_bk( coor(1,c1), coor(1,c2), coor(1,c3), coor(1,c4), chi );

	return chi;
}

float backrub_get_angle(
	pose_ns::Pose const & pose1,
	pose_ns::Pose const & pose2,
	int const start,
	int const end
)
{
	FArray3D_float const Eposition1 = pose1.Eposition();
	FArray3D_float const Eposition2 = pose2.Eposition();
	float angle = 0, dihedral1, dihedral2;
	int nangles = 0;

	// get the angle around the start position
	if (start > 1) {
		dihedral_bk(Eposition1(1,2,start-1), Eposition1(1,2,start),
		            Eposition1(1,2,end), Eposition1(1,2,start+1), dihedral1);
		dihedral_bk(Eposition2(1,2,start-1), Eposition2(1,2,start),
		            Eposition2(1,2,end), Eposition2(1,2,start+1), dihedral2);
		angle += dihedral2-dihedral1;
		nangles += 1;
	}

	// get the angle around the end position
	if (end < pose1.total_residue()) {
		dihedral_bk(Eposition1(1,2,end+1), Eposition1(1,2,start),
		            Eposition1(1,2,end), Eposition1(1,2,end-1), dihedral1);
		dihedral_bk(Eposition2(1,2,end+1), Eposition2(1,2,start),
		            Eposition2(1,2,end), Eposition2(1,2,end-1), dihedral2);
		angle += dihedral2-dihedral1;
		nangles += 1;
	}

	if (nangles == 0) {
		std::cout << "Cannot get backrub angle for the entire chain" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	return numeric::conversions::radians(angle/nangles);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin assert_bond_length_res
///
/// @brief
///	check that residue bond lengths are close to optimal values
///
/// @detailed
///
/// @param[in] aa - amino acid type
/// @param[in] aav - amino acid variant
/// @param[in] coor - residue atom coordinates
///
/// @global_read
///
/// @global_write
///
/// @return
/// true if all bond lengths within 15% of the template, false otherwise
/// (currently it only returns true, and prints out all bad lengths)
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 20, 2005
////////////////////////////////////////////////////////////////////////////////
bool assert_bond_length_res(
	int const aa,
	int const aav,
	FArray2Da_float coor
)
{
	using namespace aaproperties_pack;
	using namespace param_aa;

	int const natoms = aaproperties_pack::natoms(aa, aav);
	int nb;
	float dist, temp_dist;

	coor.dimension(3, param::MAX_ATOM());

	for (int i = 1; i <= natoms; i++) {
		for (int j = 1; j <= nbonded_neighbors(i, aa, aav); j++) {
			nb = bonded_neighbor(j, i, aa, aav);
			if (nb < i) continue; // don't check bond lengths twice
			distance_bk(coor(1,i), coor(1,nb), dist);
			distance_bk(icoor(1,i,aa,aav), icoor(1,nb,aa,aav),
			            temp_dist);
			if (std::abs(temp_dist - dist)/temp_dist > 0.15) {
				std::cout << "Unusual " << aa_name3(aa) << " "
				          << atom_name(i, aa, aav)
				          << "-" << atom_name(nb, aa, aav)
				          << " bond length: " << dist << " (template "
				          << temp_dist << ")" << std::endl;
			//	return false;
			}
		}
	}

	return true;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin backrub_dE_dtheta
///
/// @brief
/// calculate bond angle energy derivative for a backrub motion
///
/// @detailed
/// The four atom arguments should be given in the order in which they appear
/// on the backbone, either in forward or reverse order.
///
/// @param[in] ca0 - reference CA atom defining rotation axis/hinge point
/// @param[in] nc1 - N (C if reversed) atom for the N-CA-C bond angle energy
/// @param[in] ca1 - CA atom between the N and C atoms
/// @param[in] cn1 - C (N if reversed) atom for the N-CA-C bond angle energy
/// @param[in] Ktheta - bond angle energy constant
/// @param[in] Theta0 - ideal bond angle
///
/// @global_read
///
/// @global_write
///
/// @return
/// The derivative dE_dtheta. If the atoms are given in reverse order, the sign
/// will be the opposite of the true derivative.
///
/// @remarks
/// The derivative is calculated in polar coordinates with ca1 defining the
/// origin and ca0 defining the direction of the positive z-axis.
/// The algorithmic derivative was calculated in R with the following call:
///
/// deriv(expression(Ktheta*(acos(sin(phiNC)*sin(phiCN)*cos(theta) +
///                  cos(phiNC)*cos(phiCN))-Theta0)^2), "theta")
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 26, 2006
////////////////////////////////////////////////////////////////////////////////
float
backrub_dE_dtheta(
	FArray1Da_float const ca0,
	FArray1Da_float const nc1,
	FArray1Da_float const ca1,
	FArray1Da_float const cn1,
	float const Ktheta,
	float const Theta0
)
{
	using numeric::conversions::radians;

	float phiNC, phiCN, theta, expr1, expr2, expr3, dE_dtheta;

	phiNC = vec_angle(ca0, ca1, nc1);
	phiCN = vec_angle(ca0, ca1, cn1);
	dihedral_bk(nc1, ca0, ca1, cn1, theta);

	phiNC = radians(phiNC);
	phiCN = radians(phiCN);
	theta = radians(theta);

	expr1 = sin(phiNC) * sin(phiCN);
	expr2 = expr1 * cos(theta) + cos(phiNC) * cos(phiCN);
	expr3 = acos(expr2) - radians(Theta0);
	dE_dtheta = Ktheta * 2 * expr1 * expr3 * sin(theta) / sqrt(1-expr2*expr2);

	return dE_dtheta;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin backrub_dE_dtheta_Eposition
///
/// @brief
/// calculate the derivative of a backrub motion using the Eposition array
///
/// @detailed
///
/// @param[in] Eposition - Eposition array
/// @param[in] aa - array of amino acid types
/// @param[in] aav - array of amino acid variants
/// @param[in] start - starting residue number
/// @param[in] end - ending residue number
///
/// @global_read
///
/// @global_write
///
/// @return
/// the derivative dE_dtheta
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 26, 2006
////////////////////////////////////////////////////////////////////////////////
float
backrub_dE_dtheta_Eposition(
	FArray3D_float const & Eposition,
	FArray1D_int const & aa,
	FArray1D_int const & aav,
	int const start,
	int const end
)
{
	float dE_dtheta = 0;

	if (aav.size()) {} // avoid unused parameter warning

	bond_angle::initialize_bond_angle_params();

	dE_dtheta -= backrub_dE_dtheta(Eposition(1,2,end), Eposition(1,1,start),
	                               Eposition(1,2,start), Eposition(1,4,start),
	                               bond_angle::Ktheta_bb[aa(start)],
	                               bond_angle::Theta0_bb[aa(start)]);

	dE_dtheta += backrub_dE_dtheta(Eposition(1,2,start), Eposition(1,4,end),
	                               Eposition(1,2,end), Eposition(1,1,end),
	                               bond_angle::Ktheta_bb[aa(end)],
	                               bond_angle::Theta0_bb[aa(end)]);

	return dE_dtheta;
}

namespace numeric {

template< typename T >
inline
void
planar_angle(
	xyzVector< T > const & p1,
	xyzVector< T > const & p2,
	xyzVector< T > const & p3,
	T & angle
)
{
	using numeric::conversions::degrees;

	// Two normalized directional vectors hold the relative directions of the points
	xyzVector< T > const a( ( p2 - p1 ).normalize_or_zero() );
	xyzVector< T > const b( ( p2 - p3 ).normalize_or_zero() );

	angle = std::acos(numeric::sin_cos_range(dot(a, b)));
}

template< typename T >
inline
T
planar_angle(
	xyzVector< T > const & p1,
	xyzVector< T > const & p2,
	xyzVector< T > const & p3
)
{
	T angle( 0 );
	planar_angle( p1, p2, p3, angle );
	return angle;
}

}

template< typename T >
void
center_angle_radians( T & ang )
{
	typedef numeric::NumericTraits< T >  Traits;

	if (ang > 0) {
		ang -= static_cast< long int >( ( ang + Traits::pi() ) / Traits::pi_2() ) *
		       Traits::pi_2();
	} else if (ang < 0) {
		ang -= static_cast< long int >( ( ang - Traits::pi() ) / Traits::pi_2() ) *
		       Traits::pi_2();
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin backrub_tau_intervals
///
/// @brief
/// calculate the intervals of tau permissible for a given range of alpha angles
///
/// @detailed
///
/// @param[in] pose - Pose object
/// @param[in] res - residue with alpha constraint
/// @param[in] res_ref - residue with other hinge axis
/// @param[in] alpha_min - minimum allowed alpha angle
/// @param[in] alpha_max - maximum allowed alpha angle
/// @param[out] tau_range - range vector with allowed tau angles
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 16, 2007
////////////////////////////////////////////////////////////////////////////////
void
backrub_tau_intervals(
	pose_ns::Pose const & pose,
	int const res,
	int const res_ref,
	double const alpha_min,
	double const alpha_max,
	IntervalSet<double> & tau_intervals
)
{
	using namespace numeric;
	using numeric::conversions::radians;
	using numeric::sin_cos_range;
	using numeric::in_sin_cos_range;

	tau_intervals.clear();

	xyzVector_double const N(pose.full_coord(1,res));
	xyzVector_double const CA(pose.full_coord(2,res));
	xyzVector_double const C(pose.full_coord(3,res));
	xyzVector_double const CAref(pose.full_coord(2,res_ref));

	xyzVector_double const V( (CA - N).normalize() );
	xyzVector_double const U( (C - CA).normalize() );
	xyzVector_double const R( (CAref - CA).normalize() );

	double const sigma_v = std::acos(sin_cos_range(-dot(V, R))); // N->CA
	double const sigma_u = std::acos(sin_cos_range(-dot(U, R))); // CA->C

	double const alpha = std::acos(sin_cos_range(-dot(V, U))); // N-CA-C

	double const sin_sigma_v = std::sin(sigma_v);
	double const cos_sigma_v = std::cos(sigma_v);
	double const sin_sigma_u = std::sin(sigma_u);
	double const cos_sigma_u = std::cos(sigma_u);
	double const cos_alpha = std::cos(alpha);

	double const tau_u = ((dot(U, cross(V, R)) < 0) ? -1 : 1) *
	                     std::acos(sin_cos_range((cos_alpha + cos_sigma_v * cos_sigma_u) /
	                                             (sin_sigma_v * sin_sigma_u)));

	// Calculate ranges of allowed tau values
	if (alpha_min >= 0 && alpha_max > alpha_min) {

		IntervalSet<double> min_intervals;
		IntervalSet<double> max_intervals;

		double const min_value = (cos_sigma_v * cos_sigma_u + std::cos(alpha_min)) /
		                         (sin_sigma_v * sin_sigma_u);

		if (in_sin_cos_range(min_value, 0.00000001)) {

			double const acos_min = std::acos(sin_cos_range(min_value));
			double tau1 = -acos_min - tau_u;
			double tau2 = acos_min - tau_u;

			center_angle_radians(tau1);
			center_angle_radians(tau2);

			if (tau1 > tau2) {
				double const temp = tau1;
				tau1 = tau2;
				tau2 = temp;
			}

			double const alpha_mid = std::acos(sin_cos_range(sin_sigma_v * sin_sigma_u *
			                                                 std::cos(tau_u + (tau1 + tau2)/2) -
			                                                 cos_sigma_v * cos_sigma_u));

			if (alpha_mid > alpha_min) {
				min_intervals.set(tau1, tau2);
			} else {
				min_intervals.set(-numeric::constants::d::pi, tau1, tau2, numeric::constants::d::pi);
			}
		} else if (alpha > alpha_min) {
			min_intervals.set(-numeric::constants::d::pi, numeric::constants::d::pi);
		} else {
			//std::cout << "Error: No tau angles meet minimum alpha bond angle" << std::endl;
		}

		/*
		if (min_intervals.endpoints().size() && !min_intervals.is_inside(0)) {
			std::cout << "Min not inside " << tau_u << " "
			          << (in_sin_cos_range(min_value) ? std::acos(sin_cos_range(min_value)) : -1) << " "
			          << min_value << std::endl;
			std::cout << min_intervals << std::endl;
		}
		*/

		double const max_value = (cos_sigma_v * cos_sigma_u + std::cos(alpha_max)) /
		                         (sin_sigma_v * sin_sigma_u);

		if (in_sin_cos_range(max_value, 0.00000001)) {

			double const acos_max = std::acos(sin_cos_range(max_value));
			double tau1 = -acos_max - tau_u;
			double tau2 = acos_max - tau_u;

			center_angle_radians(tau1);
			center_angle_radians(tau2);

			if (tau1 > tau2) {
				double const temp = tau1;
				tau1 = tau2;
				tau2 = temp;
			}

			double const alpha_mid = std::acos(sin_cos_range(sin_sigma_v * sin_sigma_u *
			                                                 std::cos(tau_u + (tau1 + tau2)/2) -
			                                                 cos_sigma_v * cos_sigma_u));

			if (alpha_mid < alpha_max) {
				max_intervals.set(tau1, tau2);
			} else {
				max_intervals.set(-numeric::constants::d::pi, tau1, tau2, numeric::constants::d::pi);
			}
		} else if (alpha < alpha_max) {
			max_intervals.set(-numeric::constants::d::pi, numeric::constants::d::pi);
		} else {
			//std::cout << "Error: No tau angles meet maximum alpha bond angle" << std::endl;
		}

		/*
		if (max_intervals.endpoints().size() && !max_intervals.is_inside(0)) {
			std::cout << "Max not inside " << tau_u << " "
			          << (in_sin_cos_range(max_value) ? std::acos(sin_cos_range(max_value)) : -1) << " "
			          << max_value << std::endl;
			std::cout << max_intervals << std::endl;
		}
		*/

		tau_intervals = min_intervals & max_intervals;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin backrub_dangle_dtau_constants
///
/// @brief
/// calculate constants necessary for calculating internal angles/derivatives
///
/// @detailed
/// The first 9 constants returned represent A1-A3, B1-B3, & C1-C3 as described
/// in Betancourt 2005. The last 6 constants allow calculation of the signs of
/// phi and psi. They could be called B4-B6 & C4-C6. For a given tau angle, phi
/// is negative if the following is true:
///
/// B4 < B5 * cos(B6 + tau)
///
/// Similarly, psi is negative if the following is true:
///
/// C4 < C5 * cos(C6 + tau)
///
/// @param[in] pose - Pose object
/// @param[in] res - residue with alpha constraint
/// @param[in] res_ref - residue with other hinge axis
/// @param[out] constants - vector with 15 constants
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 16, 2007
////////////////////////////////////////////////////////////////////////////////
void
backrub_dangle_dtau_constants(
	pose_ns::Pose const & pose,
	int const res,
	int const res_ref,
	utility::vector0<double> & constants
)
{
	using namespace numeric;
	using numeric::conversions::radians;
	using numeric::sin_cos_range;

	constants.resize(15);
	for (int i = 0; i < 15; i++) {
		constants[i] = 0;
	}

	xyzVector_double const N(pose.full_coord(1,res));
	xyzVector_double const CA(pose.full_coord(2,res));
	xyzVector_double const C(pose.full_coord(3,res));
	xyzVector_double const CAref(pose.full_coord(2,res_ref));

	xyzVector_double const V( (CA - N).normalize() );
	xyzVector_double const U( (C - CA).normalize() );
	xyzVector_double const R( (CAref - CA).normalize() );

	double const sigma_v = std::acos(sin_cos_range(-dot(V, R))); // N->CA
	double const sigma_u = std::acos(sin_cos_range(-dot(U, R))); // CA->C

	double const alpha = std::acos(sin_cos_range(-dot(V, U))); // N-CA-C

	double const sin_sigma_v = std::sin(sigma_v);
	double const cos_sigma_v = std::cos(sigma_v);
	double const sin_sigma_u = std::sin(sigma_u);
	double const cos_sigma_u = std::cos(sigma_u);
	double const cos_alpha = std::cos(alpha);

	double const tau_u = ((dot(U, cross(V, R)) < 0) ? -1 : 1) *
	                     std::acos(sin_cos_range((cos_alpha + cos_sigma_v * cos_sigma_u) /
	                                             (sin_sigma_v * sin_sigma_u)));

	double const a1 = constants[0] = sin_sigma_v * sin_sigma_u * std::cos(tau_u);
	double const b1 = constants[1] = -sin_sigma_v * sin_sigma_u * std::sin(tau_u);
	double const c1 = constants[2] = -cos_sigma_v * cos_sigma_u;

	if (res > 1) {

		xyzVector_double const Cm1(pose.full_coord(3,res-1));

		xyzVector_double const W( (N - Cm1).normalize() );

		double const sigma_w = std::acos(sin_cos_range(-dot(W, R))); // C(-1)->N

		double const gamma = std::acos(sin_cos_range(-dot(W, V))); // C(-1)-N-CA

		double const sin_sigma_w = std::sin(sigma_w);
		double const cos_sigma_w = std::cos(sigma_w);
		double const sin_gamma = std::sin(gamma);
		double const cos_gamma = std::cos(gamma);

		double const tau_w = ((dot(U, cross(W, R)) < 0) ? -1 : 1) *
	                       std::acos(sin_cos_range((-dot(W, U) + cos_sigma_w * cos_sigma_u) /
	                                               (sin_sigma_w * sin_sigma_u)));

		constants[3] = (sin_sigma_w * sin_sigma_u * std::cos(tau_w) + a1 * cos_gamma) / sin_gamma;
		constants[4] = (-sin_sigma_w * sin_sigma_u * std::sin(tau_w) + b1 * cos_gamma) / sin_gamma;
		constants[5] = (-cos_sigma_u * cos_sigma_w + c1 * cos_gamma) / sin_gamma;

		xyzVector_double const Wn( (cross(W, V)).normalize() ); // V-W plane normal

		double const sigma_wn = std::acos(sin_cos_range(-dot(Wn, R)));

		double const sin_sigma_wn = std::sin(sigma_wn);
		double const cos_sigma_wn = std::cos(sigma_wn);

		double const tau_wn = ((dot(U, cross(Wn, R)) < 0) ? -1 : 1) *
	                        std::acos(sin_cos_range((-dot(Wn, U) + cos_sigma_wn * cos_sigma_u) /
		                                              (sin_sigma_wn * sin_sigma_u)));

		constants[9] = cos_sigma_wn * cos_sigma_u;
		constants[10] = sin_sigma_wn * sin_sigma_u;
		constants[11] = tau_wn;
	}

	if (res < pose.total_residue()) {

		xyzVector_double const Np1(pose.full_coord(1,res+1));

		xyzVector_double const W1( (Np1 - C).normalize() );

		double const sigma_w1 = std::acos(sin_cos_range(-dot(W1, R))); // C->N(+1)

		double const beta = std::acos(sin_cos_range(-dot(U, W1))); // CA-C-N(+1)

		double const sin_sigma_w1 = std::sin(sigma_w1);
		double const cos_sigma_w1 = std::cos(sigma_w1);
		double const sin_beta = std::sin(beta);
		double const cos_beta = std::cos(beta);

		double const tau_w1 = ((dot(W1, cross(V, R)) < 0) ? -1 : 1) *
	                        std::acos(sin_cos_range((-dot(V, W1) + cos_sigma_v * cos_sigma_w1) /
	                                                (sin_sigma_v * sin_sigma_w1)));

		constants[6] = (sin_sigma_v * sin_sigma_w1 * std::cos(tau_w1) + a1 * cos_beta) / sin_beta;
		constants[7] = (-sin_sigma_v * sin_sigma_w1 * std::sin(tau_w1) + b1 * cos_beta) / sin_beta;
		constants[8] = (-cos_sigma_v * cos_sigma_w1 + c1 * cos_beta) / sin_beta;

		xyzVector_double const W1n( (cross(U, W1)).normalize() ); // U-W1 plane normal

		double const sigma_w1n = std::acos(sin_cos_range(-dot(W1n, R)));

		double const sin_sigma_w1n = std::sin(sigma_w1n);
		double const cos_sigma_w1n = std::cos(sigma_w1n);

		double const tau_w1n = ((dot(W1n, cross(V, R)) < 0) ? -1 : 1) *
		                       std::acos(sin_cos_range((-dot(V, W1n) + cos_sigma_v * cos_sigma_w1n) /
		                                               (sin_sigma_v * sin_sigma_w1n)));

		constants[12] = cos_sigma_w1n * cos_sigma_v;
		constants[13] = sin_sigma_w1n * sin_sigma_v;
		constants[14] = tau_w1n;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin backrub_dangle_dtau
///
/// @brief
/// calculate internal angles derivatives and values for any tau value
///
/// @detailed
///
/// @param[in] constants - vector with 15 constants
/// @param[in] tau - Backrub rotation angle
/// @param[out] dalpha_dtau - dalpha/dtau derivative
/// @param[out] dphi_dtau - dphi/dtau derivative
/// @param[out] dpsi_dtau - dpsi/dtau derivative
/// @param[out] alpha - alpha angle value
/// @param[out] phi - phi angle value
/// @param[out] psi - psi angle value
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 16, 2007
////////////////////////////////////////////////////////////////////////////////
void
backrub_dangle_dtau(
	utility::vector0<double> const & constants,
	double const tau,
	double & dalpha_dtau,
	double & dphi_dtau,
	double & dpsi_dtau,
	double & alpha,
	double & phi,
	double & psi
)
{
	using numeric::sin_cos_range;
	using numeric::constants::d::pi;

	double const sin_tau = std::sin(tau);
	double const cos_tau = std::cos(tau);

	alpha = acos(sin_cos_range(constants[0]*cos_tau + constants[1]*sin_tau + constants[2]));

	double const sin_alpha = std::sin(alpha);
	double const cos_alpha = std::cos(alpha);

	phi = acos(sin_cos_range((constants[3]*cos_tau + constants[4]*sin_tau + constants[5]) /
	                         sin_alpha));
	if (constants[9] < constants[10]*std::cos(constants[11] + tau)) phi = -phi;

	psi = acos(sin_cos_range((constants[6]*cos_tau + constants[7]*sin_tau + constants[8]) /
	                         sin_alpha));
	if (constants[12] < constants[13]*std::cos(constants[14] + tau)) psi = -psi;

	dalpha_dtau = (constants[0]*sin_tau - constants[1]*cos_tau) / sin_alpha;

	double const min_torsion = 0.00000001;//0.01745329252; // 1 degree

	if (std::abs(phi) >= min_torsion && std::abs(phi) <= pi - min_torsion) {
		dphi_dtau = (constants[3]*sin_tau - constants[4]*cos_tau + dalpha_dtau*cos_alpha*std::cos(phi))
		            / (sin_alpha*std::sin(phi));
	} else {

		// Linear interpolate discontinuity
		std::cout << "Interpolating dphi/dtau (tau = " << tau << ", phi = " << phi << ")" << std::endl;
		double dtau = min_torsion*2;
		dphi_dtau = 0;

		for (double tau1 = tau-dtau; tau1 <= tau+2*dtau; tau1 += 2*dtau) {
			double const sin_tau1 = std::sin(tau1);
			double const cos_tau1 = std::cos(tau1);

			double const alpha1 = acos(sin_cos_range(constants[0]*cos_tau1 + constants[1]*sin_tau1 +
			                                         constants[2]));

			double const sin_alpha1 = std::sin(alpha1);
			double const cos_alpha1 = std::cos(alpha1);

			double phi1 = acos(sin_cos_range((constants[3]*cos_tau1 + constants[4]*sin_tau1 +
			                                  constants[5]) / sin_alpha1));
			if (constants[9] < constants[10]*std::cos(constants[11] + tau1)) phi1 = -phi1;

			if (!(std::abs(phi1) >= min_torsion && std::abs(phi1) <= pi - min_torsion)) {
				dphi_dtau = 0;
				if (dtau > 10*min_torsion) break;
				std::cout << "Incrementing dtau" << std::endl;
				dtau *= 2;
				tau1 = tau-3*dtau;
				continue;
			}

			double const dalpha_dtau1 = (constants[0]*sin_tau1 - constants[1]*cos_tau1) / sin_alpha1;

			dphi_dtau += (constants[3]*sin_tau1 - constants[4]*cos_tau1 +
			             dalpha_dtau1*cos_alpha1*std::cos(phi1)) / (sin_alpha1*std::sin(phi1));
		}

		dphi_dtau /= 2;
	}

	if (std::abs(psi) >= min_torsion && std::abs(psi) <= pi - min_torsion) {
		dpsi_dtau = (constants[6]*sin_tau - constants[7]*cos_tau + dalpha_dtau*cos_alpha*std::cos(psi))
		            / (sin_alpha*std::sin(psi));
	} else {

		// Linear interpolate discontinuity
		std::cout << "Interpolating dpsi/dtau (tau = " << tau << ", psi = " << psi << ")" << std::endl;
		double dtau = min_torsion*2;
		dpsi_dtau = 0;

		for (double tau1 = tau-dtau; tau1 <= tau+2*dtau; tau1 += 2*dtau) {
			double const sin_tau1 = std::sin(tau1);
			double const cos_tau1 = std::cos(tau1);

			double const alpha1 = acos(sin_cos_range(constants[0]*cos_tau1 + constants[1]*sin_tau1 +
			                                         constants[2]));

			double const sin_alpha1 = std::sin(alpha1);
			double const cos_alpha1 = std::cos(alpha1);

			double psi1 = acos(sin_cos_range((constants[6]*cos_tau1 + constants[7]*sin_tau1 +
			                                  constants[8]) / sin_alpha1));
			if (constants[12] < constants[13]*std::cos(constants[14] + tau1)) psi1 = -psi1;

			if (!(std::abs(psi1) >= min_torsion && std::abs(psi1) <= pi - min_torsion)) {
				dpsi_dtau = 0;
				if (dtau > 10*min_torsion) break;
				std::cout << "Incrementing dtau" << std::endl;
				dtau *= 2;
				tau1 = tau-3*dtau;
				continue;
			}

			double const dalpha_dtau1 = (constants[0]*sin_tau1 - constants[1]*cos_tau1) / sin_alpha1;

			dpsi_dtau += (constants[6]*sin_tau1 - constants[7]*cos_tau1 +
			             dalpha_dtau1*cos_alpha1*std::cos(psi1)) / (sin_alpha1*std::sin(psi1));
		}

		dpsi_dtau /= 2;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin backrub_dangle_dtau_test
///
/// @brief
/// test angular derivatives with a finite difference approximation
///
/// @detailed
/// This function writes two lines for each of alpha, phi, & psi:
///
/// line 1: calculated value, calculated derivative
///
/// line 2: actual value, finite difference derivative
///
/// @param[in] pose - Pose object
/// @param[in] res - residue to test angles
/// @param[in] res_ref - residue with other hinge axis
/// @param[in] constants - vector with 15 constants
/// @param[in] tau - backrub angle
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 16, 2007
////////////////////////////////////////////////////////////////////////////////
void
backrub_dangle_dtau_test(
	pose_ns::Pose const & pose,
	int const res,
	int const res_ref,
	utility::vector0<double> const & constants,
	double const tau
)
{
	using numeric::conversions::radians;

	pose_ns::Pose tmp_pose, dtmp_pose;
	double dalpha_dtau, dphi_dtau, dpsi_dtau, alpha, phi, psi;

	bool const phi_exists = res > 1; // Can we calculate leading phi?
	bool const psi_exists = res < pose.total_residue(); // Can we calculate lagging psi?

	backrub_dangle_dtau(constants, tau, dalpha_dtau, dphi_dtau, dpsi_dtau, alpha, phi, psi);

	tmp_pose = pose;
	if (res < res_ref && tau) {
		backrub_rot_pose(tmp_pose, res, res_ref, tau);
	} else if (tau) {
		backrub_rot_pose(tmp_pose, res_ref, res, tau);
	}

	double const dtau = 0.0004;

	double const alpha1 = radians(vec_angle(tmp_pose.full_coord()(1,1,res),
	                                       tmp_pose.full_coord()(1,2,res),
	                                       tmp_pose.full_coord()(1,3,res))); // N-CA-C
	double const phi1 = radians(tmp_pose.phi(res));
	double const psi1 = radians(tmp_pose.psi(res));

	dtmp_pose = tmp_pose;
	if (res < res_ref) {
		backrub_rot_pose(dtmp_pose, res, res_ref, -dtau/2);
	} else {
		backrub_rot_pose(dtmp_pose, res_ref, res, -dtau/2);
	}

	double const alpha2 = radians(vec_angle(dtmp_pose.full_coord()(1,1,res),
	                                       dtmp_pose.full_coord()(1,2,res),
	                                       dtmp_pose.full_coord()(1,3,res))); // N-CA-C
	double const phi2 = radians(dtmp_pose.phi(res));
	double const psi2 = radians(dtmp_pose.psi(res));

	dtmp_pose = tmp_pose;
	if (res < res_ref) {
		backrub_rot_pose(dtmp_pose, res, res_ref, dtau/2);
	} else {
		backrub_rot_pose(dtmp_pose, res_ref, res, dtau/2);
	}

	double const alpha3 = radians(vec_angle(dtmp_pose.full_coord()(1,1,res),
	                                       dtmp_pose.full_coord()(1,2,res),
	                                       dtmp_pose.full_coord()(1,3,res))); // N-CA-C
	double const phi3 = radians(dtmp_pose.phi(res));
	double const psi3 = radians(dtmp_pose.psi(res));

	std::cout << std::fixed << std::setprecision(6);
	std::cout << "alpha:" << std::setw(12) << alpha << std::setw(12) << dalpha_dtau << std::endl;
	std::cout << "      " << std::setw(12) << alpha1 << std::setw(12) << (alpha3-alpha2)/dtau << std::endl;

	if (phi_exists) {
		std::cout << "phi:  " << std::setw(12) << phi << std::setw(12) << dphi_dtau << std::endl;
		std::cout << "      " << std::setw(12) << phi1 << std::setw(12) << (phi3-phi2)/dtau << std::endl;
	}
	if (psi_exists) {
		std::cout << "psi:  " << std::setw(12) << psi << std::setw(12) << dpsi_dtau << std::endl;
		std::cout << "      " << std::setw(12) << psi1 << std::setw(12) << (psi3-psi2)/dtau << std::endl;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin backrub_select_angle
///
/// @brief
/// select a random Backrub angle satisfying various constraints
///
/// @detailed
///
/// @param[in] pose - Pose object
/// @param[in] res1 - starting residue
/// @param[in] res2 - ending residue
/// @param[in] alpha_min - minimum C-alpha bond angle
/// @param[in] alpha_max - maximum C-alpha bond angle
/// @param[in] angle_disp - maximum angular displacement
/// @param[in] res_dependent - are min/max bond angles relative to optimal
/// @param[in] jacobian - should Jacobian weighting be applied to angle selction
///
/// @global_read
///
/// @global_write
///
/// @return
/// a random backrub angle or 0 if any constraint could not be satisfied
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 16, 2007
////////////////////////////////////////////////////////////////////////////////
double
backrub_select_angle(
	pose_ns::Pose const & pose,
	int const res1,
	int const res2,
	double const alpha_min,
	double const alpha_max,
	bool const res_dependent,
	double const angle_disp,
	bool const jacobian
)
{
	using numeric::constants::d::pi;
	using numeric::conversions::radians;

	double angle = 0;
	utility::vector0<double> constants1;
	utility::vector0<double> constants2;
	IntervalSet<double> tau_intervals1;
	IntervalSet<double> tau_intervals2;
	IntervalSet<double> tau_intervals;
	IntervalSet<double> tau_intervals_maskp;
	double dalpha_dtau1, dphi_dtau1, dpsi_dtau1, dalpha_dtau2, dphi_dtau2, dpsi_dtau2;
	double alpha1, phi1, psi1, alpha2, phi2, psi2;
	double res1_opt = 0;
	double res2_opt = 0;

	if (res_dependent) {
		// Bond angle potential ideal angle
		res1_opt = bond_angle::Theta0_bb[pose.res(res1)];
		res2_opt = bond_angle::Theta0_bb[pose.res(res2)];
	}

	backrub_tau_intervals(pose, res1, res2, alpha_min+res1_opt, alpha_max+res1_opt, tau_intervals1);
	backrub_tau_intervals(pose, res2, res1, alpha_min+res2_opt, alpha_max+res2_opt, tau_intervals2);
	tau_intervals = tau_intervals1 & tau_intervals2;

	if (tau_intervals.length() <= 0) {
		/*
		// Begin Debug Code
		std::cout << "Error " << res1 << ":" << res2 << std::endl;
		backrub_dangle_dtau_constants(pose, res1, res2, constants1);
		backrub_dangle_dtau_constants(pose, res2, res1, constants2);
		int size;
		size = tau_intervals1.endpoints().size();
		for (int i = 0; i < size; i += 2) {
			backrub_dangle_dtau(constants1, tau_intervals1.endpoints()[i], dalpha_dtau1, dphi_dtau1,
			                    dpsi_dtau1, alpha1, phi1, psi1);
			std::cout << tau_intervals1.endpoints()[i] << "(" << alpha1 << "):";
			backrub_dangle_dtau(constants1, tau_intervals1.endpoints()[i+1], dalpha_dtau1, dphi_dtau1,
			                    dpsi_dtau1, alpha1, phi1, psi1);
			std::cout << tau_intervals1.endpoints()[i+1] << "(" << alpha1 << ") ";
		}
		std::cout << std::endl;
		size = tau_intervals2.endpoints().size();
		for (int i = 0; i < size; i += 2) {
			backrub_dangle_dtau(constants2, tau_intervals2.endpoints()[i], dalpha_dtau2, dphi_dtau2,
			                    dpsi_dtau2, alpha2, phi2, psi2);
			std::cout << tau_intervals2.endpoints()[i] << "(" << alpha2 << "):";
			backrub_dangle_dtau(constants2, tau_intervals2.endpoints()[i+1], dalpha_dtau2, dphi_dtau2,
			                    dpsi_dtau2, alpha2, phi2, psi2);
			std::cout << tau_intervals2.endpoints()[i+1] << "(" << alpha2 << ") ";
		}
		std::cout << std::endl;
		size = tau_intervals.endpoints().size();
		for (int i = 0; i < size; i += 2) {
			backrub_dangle_dtau(constants1, tau_intervals.endpoints()[i], dalpha_dtau1, dphi_dtau1,
			                    dpsi_dtau1, alpha1, phi1, psi1);
			backrub_dangle_dtau(constants2, tau_intervals.endpoints()[i], dalpha_dtau2, dphi_dtau2,
			                    dpsi_dtau2, alpha2, phi2, psi2);
			std::cout << tau_intervals.endpoints()[i] << "(" << alpha1 << "," << alpha2 << "):";
			backrub_dangle_dtau(constants1, tau_intervals.endpoints()[i+1], dalpha_dtau1, dphi_dtau1,
			                    dpsi_dtau1, alpha1, phi1, psi1);
			backrub_dangle_dtau(constants2, tau_intervals.endpoints()[i+1], dalpha_dtau2, dphi_dtau2,
			                    dpsi_dtau2, alpha2, phi2, psi2);
			std::cout << tau_intervals.endpoints()[i+1] << "(" << alpha1 << "," << alpha2 << ") ";
		}
		std::cout << std::endl;
		// End Debug Code
		*/
		return 0;
	}

	IntervalSet<double> tau_intervals_mask(-angle_disp, angle_disp);
	IntervalSet<double> tau_intervals_sub(tau_intervals & tau_intervals_mask);

	double const intervallen = tau_intervals_sub.length();
	//std::cout << ", " << intervallen;
	if (intervallen <= 0) {
		return 0;
	}

	double const threshold = ran3();
	int i;
	for (i = 0; i < 100000; i++) {

		angle = tau_intervals_sub.random_point();

		double min_anglep = angle - angle_disp;
		double max_anglep = angle + angle_disp;
		center_angle_radians(min_anglep);
		center_angle_radians(max_anglep);
		if (min_anglep < max_anglep) {
			tau_intervals_maskp.set(min_anglep, max_anglep);
		} else {
			tau_intervals_maskp.set(-pi, max_anglep, min_anglep, pi);
		}

		double intervallenp = (tau_intervals & tau_intervals_maskp).length();

		if (intervallenp == 0 || intervallen/intervallenp >= threshold) break;
	}

	if (jacobian) {

		bool const phi_exists = res1 > 1; // Can we calculate leading phi?
		bool const psi_exists = res2 < pose.total_residue(); // Can we calculate lagging psi?

		backrub_dangle_dtau_constants(pose, res1, res2, constants1);
		backrub_dangle_dtau_constants(pose, res2, res1, constants2);

		backrub_dangle_dtau(constants1, 0, dalpha_dtau1, dphi_dtau1, dpsi_dtau1, alpha1, phi1, psi1);
		if (!phi_exists) dphi_dtau1 = 0;
		backrub_dangle_dtau(constants2, 0, dalpha_dtau2, dphi_dtau2, dpsi_dtau2, alpha2, phi2, psi2);
		if (!psi_exists) dpsi_dtau2 = 0;

		double const dD = std::sqrt(dalpha_dtau1*dalpha_dtau1 + dphi_dtau1*dphi_dtau1 +
		                            dpsi_dtau1*dpsi_dtau1 + dalpha_dtau2*dalpha_dtau2 +
		                            dphi_dtau2*dphi_dtau2 + dpsi_dtau2*dpsi_dtau2);

		double const threshold = ran3();
		for (int i = 0; i < 100000; i++) {

			angle = tau_intervals.random_point();

			backrub_dangle_dtau(constants1, angle, dalpha_dtau1, dphi_dtau1, dpsi_dtau1, alpha1, phi1, psi1);
			if (!phi_exists) dphi_dtau1 = 0;
			backrub_dangle_dtau(constants2, angle, dalpha_dtau2, dphi_dtau2, dpsi_dtau2, alpha2, phi2, psi2);
			if (!psi_exists) dpsi_dtau2 = 0;

			double const dDp = std::sqrt(dalpha_dtau1*dalpha_dtau1 + dphi_dtau1*dphi_dtau1 +
			                             dpsi_dtau1*dpsi_dtau1 + dalpha_dtau2*dalpha_dtau2 +
			                             dphi_dtau2*dphi_dtau2 + dpsi_dtau2*dpsi_dtau2);

			if (dDp/dD >= threshold) break;
		}
	}

	//std::cout << i << "\t" << angle << std::endl;

	return angle;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin backrub_dD_write
///
/// @brief
/// write internal coordinate derivatives and values for 360 degrees of a
/// backrub angle
///
/// @detailed
/// This function writes tab delimeted output for every degree from -180 to 180:
///
/// tau, dD, dalpha1/dtau, dphi1/dtau, dpsi1/dtau, dalpha2/dtau, dphi2/dtau,
/// dpsi2/dtau, alpha1, phi1, psi1, alpha2, phi2, psi2
///
/// @param[in] pose - Pose object
/// @param[in] res1 - starting residue
/// @param[in] res2 - ending residue
/// @param[in] stream - output stream to write
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 16, 2007
////////////////////////////////////////////////////////////////////////////////
void
backrub_dD_write(
	pose_ns::Pose const & pose,
	int res1,
	int res2,
	std::ostream & stream
)
{
	using numeric::conversions::radians;

	utility::vector0<double> constants1;
	utility::vector0<double> constants2;
	IntervalSet<double> tau_intervals1;
	IntervalSet<double> tau_intervals2;
	IntervalSet<double> tau_intervals;
	double dalpha_dtau1, dphi_dtau1, dpsi_dtau1, dalpha_dtau2, dphi_dtau2, dpsi_dtau2;
	double alpha1, phi1, psi1, alpha2, phi2, psi2;

	bool const phi_exists = res1 > 1; // Can we calculate leading phi?
	bool const psi_exists = res2 < pose.total_residue(); // Can we calculate lagging psi?

	backrub_dangle_dtau_constants(pose, res1, res2, constants1);
	backrub_tau_intervals(pose, res1, res2, radians(96.7951), radians(116.7951), tau_intervals1);
	backrub_dangle_dtau_constants(pose, res2, res1, constants2);
	backrub_tau_intervals(pose, res2, res1, radians(96.7951), radians(116.7951), tau_intervals2);

	tau_intervals = tau_intervals1 & tau_intervals2;

	for (int i = 0; i < (signed)tau_intervals1.endpoints().size(); i += 1) {
		stream << tau_intervals1.endpoints()[i] << "\t";
	}
	stream << std::endl;
	for (int i = 0; i < (signed)tau_intervals2.endpoints().size(); i += 1) {
		stream << tau_intervals2.endpoints()[i] << "\t";
	}
	stream << std::endl;
	for (int i = 0; i < (signed)tau_intervals.endpoints().size(); i += 1) {
		stream << tau_intervals.endpoints()[i] << "\t";
	}
	stream << std::endl;

	for (int i = -180; i <= 180; i++) {

		double angle = numeric::conversions::radians((double)i);

		backrub_dangle_dtau(constants1, angle, dalpha_dtau1, dphi_dtau1, dpsi_dtau1,
		                    alpha1, phi1, psi1);
		if (!phi_exists) dphi_dtau1 = 0;
		backrub_dangle_dtau(constants2, angle, dalpha_dtau2, dphi_dtau2, dpsi_dtau2,
		                    alpha2, phi2, psi2);
		if (!psi_exists) dpsi_dtau2 = 0;

		double const dD = std::sqrt(dalpha_dtau1*dalpha_dtau1 + dphi_dtau1*dphi_dtau1 +
		                           dpsi_dtau1*dpsi_dtau1 + dalpha_dtau2*dalpha_dtau2 +
		                           dphi_dtau2*dphi_dtau2 + dpsi_dtau2*dpsi_dtau2);

		stream << angle << "\t" << dD << "\t" << dalpha_dtau1 << "\t" << dphi_dtau1 << "\t"
		       << dpsi_dtau1 << "\t" << dalpha_dtau2 << "\t" << dphi_dtau2 << "\t" << dpsi_dtau2 << "\t"
		       << alpha1 << "\t" << phi1 << "\t" << psi1 << "\t" << alpha2 << "\t" << phi2 << "\t"
		       << psi2 << "\t" << std::endl;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin backrub_dD_debug
///
/// @brief
/// debug calculation of dD
///
/// @detailed
/// Lots of output. See source for full expanation.
///
/// @param[in] pose - Pose object
/// @param[in] res1 - starting residue
/// @param[in] res2 - ending residue
/// @param[in] tau - backrub angle
/// @param[in] stream - output stream to write
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 16, 2007
////////////////////////////////////////////////////////////////////////////////
void
backrub_dD_debug(
	pose_ns::Pose const & pose,
	int const res1,
	int const res2,
	double const tau,
	std::ostream & stream
)
{
	utility::vector0<double> constants1;
	utility::vector0<double> constants2;
	double dalpha_dtau1, dphi_dtau1, dpsi_dtau1, dalpha_dtau2, dphi_dtau2, dpsi_dtau2;
	double alpha, phi, psi;

	bool const phi_exists = res1 > 1; // Can we calculate leading phi?
	bool const psi_exists = res2 < pose.total_residue(); // Can we calculate lagging psi?

	backrub_dangle_dtau_constants(pose, res1, res2, constants1);
	backrub_dangle_dtau_constants(pose, res2, res1, constants2);

	for (int i = 0; i < (signed)constants1.size(); i++) {
		stream << constants1[i] << "\t";
	}
	stream << std::endl;
	for (int i = 0; i < (signed)constants1.size(); i++) {
		stream << constants2[i] << "\t";
	}
	stream << std::endl;

	backrub_dangle_dtau(constants1, tau, dalpha_dtau1, dphi_dtau1, dpsi_dtau1, alpha, phi, psi);
	if (!phi_exists) dphi_dtau1 = 0;
	stream << dalpha_dtau1 << "\t" << dphi_dtau1 << "\t" << dpsi_dtau1 << "\t"
	       << alpha << "\t" << phi << "\t" << psi << std::endl;
	backrub_dangle_dtau(constants2, tau, dalpha_dtau2, dphi_dtau2, dpsi_dtau2, alpha, phi, psi);
	if (!psi_exists) dpsi_dtau2 = 0;
	stream << dalpha_dtau2 << "\t" << dphi_dtau2 << "\t" << dpsi_dtau2 << "\t"
	       << alpha << "\t" << phi << "\t" << psi << std::endl;

	double const dD = std::sqrt(dalpha_dtau1*dalpha_dtau1 + dphi_dtau1*dphi_dtau1 +
	                           dpsi_dtau1*dpsi_dtau1 + dalpha_dtau2*dalpha_dtau2 +
	                           dphi_dtau2*dphi_dtau2 + dpsi_dtau2*dpsi_dtau2);

	stream << tau << "\t" << dD << std::endl;

	stream << constants1[9] << "\t" << constants1[10] << "\t" << constants1[11] << "\t"
	       << constants1[12] << "\t" << constants1[13] << "\t" << constants1[14] << std::endl;
	stream << constants2[9] << "\t" << constants2[10] << "\t" << constants2[11] << "\t"
	       << constants2[12] << "\t" << constants2[13] << "\t" << constants2[14] << std::endl;

	backrub_dangle_dtau_test(pose, res1, res2, constants1, tau);
	backrub_dangle_dtau_test(pose, res2, res1, constants2, tau);
}

namespace backrub_ns {

Backrub_eval * minimize_backrub_eval;

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_eval::Backrub_eval
///
/// @brief
/// constructor for Backrub_eval objects
///
/// @detailed
///
/// @param[in] pose - Pose object with default (0) backrub angles
/// @param[in] residue_pairs - list (vector0) of residue pairs to rotate (length
///                            2N)
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 26, 2006
////////////////////////////////////////////////////////////////////////////////
Backrub_eval::Backrub_eval(
	pose_ns::Pose * pose
)
{
	assert(pose);

	pose_ = pose;
	use_full_coord_ = false;
	Eposition_ = pose_->Eposition();
	full_coord_ = pose_->full_coord();
	func_trace_ = 0;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_eval::Backrub_eval
///
/// @brief
/// constructor for Backrub_eval objects
///
/// @detailed
///
/// @param[in] pose - Pose object with default (0) backrub angles
/// @param[in] residue_pairs - list (vector0) of residue pairs to rotate (length
///                            2N)
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 26, 2006
////////////////////////////////////////////////////////////////////////////////
Backrub_eval::Backrub_eval(
	pose_ns::Pose * pose,
	utility::vector0<int> residue_pairs
)
{
	assert(pose);
	assert(residue_pairs.size() >= 2);

	pose_ = pose;
	use_full_coord_ = false;
	Eposition_ = pose_->Eposition();
	full_coord_ = pose_->full_coord();
	func_trace_ = 0;
	residue_pairs_ = residue_pairs;

	update_residues();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_eval::update_residues
///
/// @brief
/// update vector of resides that must be scored
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified March 2, 2006
////////////////////////////////////////////////////////////////////////////////
void
Backrub_eval::update_residues()
{
	assert(residue_pairs_.size() >= 2);

	utility::vector0<bool> res_eval(pose_->total_residue());
	int num_res = 0;

	for (int i = 0; i < (signed)residue_pairs_.size(); i += 2) {
		int const end_res = residue_pairs_[i+1];
		for (int j = residue_pairs_[i]; j <= end_res; j++) {
			if (!res_eval[j-1]) {
				res_eval[j-1] = true;
				num_res++;
			}
		}
	}
	residues_.resize(num_res, false);
	int j = 0;
	for (int i = 0; i < (signed)res_eval.size(); i++) {
		if (res_eval[i]) residues_[j++] = i+1;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_eval::push_back
///
/// @brief
/// add a residue pair to the stack
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified March 2, 2006
////////////////////////////////////////////////////////////////////////////////
void
Backrub_eval::push_back(
	int start,
	int end
)
{
	assert(start >= 1 && start < end && end <= pose_->total_residue());

	size_t const size = residue_pairs_.size();

	residue_pairs_.resize(size+2);

	residue_pairs_[size] = start;
	residue_pairs_[size+1] = end;

	// Invalidate the residues_ vector
	residues_.clear();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_eval::residue_pairs
///
/// @brief
/// get list of residue pairs
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
/// vector0 list of residue pairs
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 26, 2006
////////////////////////////////////////////////////////////////////////////////
utility::vector0<int>
Backrub_eval::residue_pairs()
{
	return residue_pairs_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_eval::residues
///
/// @brief
/// get list of residues that have a pivot
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
/// vector0 list of residue pairs
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified March 2, 2007
////////////////////////////////////////////////////////////////////////////////
utility::vector0<int>
Backrub_eval::residues()
{
	return residues_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_eval::use_full_coord
///
/// @brief
/// get full coordinate flag
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
/// boolean indicating whether full coordinates are used for function
/// evaluations
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 26, 2006
////////////////////////////////////////////////////////////////////////////////
bool
Backrub_eval::use_full_coord()
{
	return use_full_coord_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_eval::set_use_full_coord
///
/// @brief
/// set full coordinate flag
///
/// @detailed
/// If this flag is set, full coordinates are used to calculate the bond angle
/// energy. This is slower as it involves moving side chains around.
/// Derivatives are never calculated with full coordinates for simplicity.
///
/// @param[in] use_full_coord - boolean for new setting of full coordinate flag
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 26, 2006
////////////////////////////////////////////////////////////////////////////////
void
Backrub_eval::set_use_full_coord(
	bool use_full_coord
)
{
	use_full_coord_ = use_full_coord;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_eval::set_func_trace
///
/// @brief
/// set output stream for recording function evaluations
///
/// @detailed
///
/// @param[in] func_trace - ostream object to write function calls
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 26, 2006
////////////////////////////////////////////////////////////////////////////////
void
Backrub_eval::set_func_trace(
	std::ostream & func_trace
)
{
	func_trace_ = &func_trace;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_eval::reset_func_trace
///
/// @brief
/// turn off recording of function evaluations
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 26, 2006
////////////////////////////////////////////////////////////////////////////////
void
Backrub_eval::reset_func_trace()
{
	func_trace_ = 0;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_eval::update_Eposition
///
/// @brief
/// update scratch Eposition array with backrub rotation angles
///
/// @detailed
///
/// @param[in] values - list of angular values (radians)
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 26, 2006
////////////////////////////////////////////////////////////////////////////////
void
Backrub_eval::update_Eposition(
	FArray1DB_float & values
)
{
	unsigned int const npairs = residue_pairs_.size()/2;

	assert(values.size() == npairs);

	Eposition_ = pose_->Eposition();
	for (unsigned int i = 0; i < npairs; i++) {
		backrub_rot_Eposition(Eposition_, residue_pairs_[i*2],
		                      residue_pairs_[i*2+1], values[i]);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_eval::update_full_coord
///
/// @brief
/// update scratch full_coord array with backrub rotation angles
///
/// @detailed
///
/// @param[in] values - list of angular values (radians)
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 26, 2006
////////////////////////////////////////////////////////////////////////////////
void
Backrub_eval::update_full_coord(
	FArray1DB_float & values
)
{
	unsigned int const npairs = residue_pairs_.size()/2;

	assert(values.size() == npairs);

	full_coord_ = pose_->full_coord();
	for (unsigned int i = 0; i < npairs; i++) {
		backrub_rot_full_coord(full_coord_, pose_->res(), pose_->res_variant(),
		                       residue_pairs_[i*2], residue_pairs_[i*2+1],
		                       values[i]);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_eval::func
///
/// @brief
/// evaluate bond angle energy
///
/// @detailed
///
/// @param[in] values - angular values for Backrub motions (radians)
/// @param[out] gfrag - flag to indicate if angles are valid numbers (unused)
///
/// @global_read
///
/// @global_write
///
/// @return
/// bond angle energy (kcal/mole)
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 26, 2006
////////////////////////////////////////////////////////////////////////////////
float
Backrub_eval::func(
	FArray1DB_float & values,
	bool & gfrag
 )
{
	float energy = 0;
	unsigned int i;

	if (gfrag) {} // avoid unused parameter warning

	if (residues_.size() == 0) update_residues(); // update the residue array

	if (use_full_coord_) {
		update_full_coord(values);
		for (i = 0; i < residues_.size(); i++) {
			energy += get_bond_angleE(pose_->res(residues_[i]),
			                          pose_->res_variant(residues_[i]),
			                          full_coord_(1,1,residues_[i]));
		}
	} else {
		update_Eposition(values);
		for (i = 0; i < residues_.size(); i++) {
			energy += get_bond_angleE_Ep(pose_->res(residues_[i]),
			                             pose_->res_variant(residues_[i]),
			                             Eposition_(1,1,residues_[i]));
		}
	}

	if (func_trace_ && func_trace_->good()) {
		(*func_trace_) << numeric::conversions::degrees(values(1));
		for (unsigned int i = 2; i <= values.size(); i++) {
			(*func_trace_) << "\t" << numeric::conversions::degrees(values(i));
		}
		(*func_trace_) << std::endl;
	}

	return energy;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_eval::dfunc
///
/// @brief
/// calculate backrub angle derivatives for bond angle energy
///
/// @detailed
///
/// @param[in] values - angular values for Backrub motions (radians)
/// @param[out] derivatives - derivative for each Backrub motion
/// @param[in] n - number of backrub motions
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 26, 2006
////////////////////////////////////////////////////////////////////////////////
void
Backrub_eval::dfunc(
	FArray1DB_float & values,
	FArray1DB_float & derivatives,
	int n
)
{
	const int npairs = residue_pairs_.size()/2;

	if (n) {} // avoid unused parameter warning

	/*
	std::cout << "dfunc:\t";
	for (unsigned int i = 1; i <= values.size(); i++)
		std::cout << numeric::conversions::degrees(values(i)) << "\t";
	std::cout << std::endl;
	*/

	update_Eposition(values);
	for (int i = 0; i < npairs; i++) {
		derivatives(i+1) = backrub_dE_dtheta_Eposition(Eposition_, pose_->res(),
		                                               pose_->res_variant(),
		                                               residue_pairs_[2*i],
		                                               residue_pairs_[2*i+1]);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Backrub_eval::minimize
///
/// @brief
/// minimize bond angle energy by changing angles defined by residue_pairs
///
/// @detailed
///
/// @param[out] values - angular values for Backrub motions (radians)
/// @param[in] min_type - type of minimization
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
Backrub_eval::minimize(
	FArray1DB_float & values,
	std::string const & min_type
)
{
	using minimize_ns::minimize_state::minimize_tolerance;

	int nangles = (signed)residue_pairs_.size()/2;
	FArray1D_float dE_dangles(nangles);
	float fret;
	int iter;
	bool gfrag = true;

	//minimize_reset();
	minimize_set_backrub_eval(*this);
	minimize_set_func(9);

	if ( min_type == "linmin" ) {
		for ( int j = 1; j <= 1; ++j ) {
			dfunc(values,dE_dangles,nangles);
			linmin(values,dE_dangles,nangles,fret,gfrag);
			//if ( !gfrag ) goto L300;
		}
	} else if ( min_type == "dfpmin" ) {
		dfpmin(values,nangles,minimize_tolerance,iter,fret,
		       gfrag);
	} else if ( min_type == "dfpmin_atol" ) { /* absolute tolerence */
		dfpmin_atol(values,nangles,minimize_tolerance,iter,fret,gfrag);
	} else if ( min_type == "frpmin" ) {
		frprmn(values,nangles,minimize_tolerance,iter,fret,gfrag);
	} else if ( min_type == "frpmin_atol" ) { /* absolute tolerance */
		frprmn_atol(values,nangles,minimize_tolerance,iter,fret,gfrag);
	} else {
		std::cout << "min_type not defined in function minimize (debump.cc)"
		          << std::endl << "min_type: " << min_type << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	for (size_t i = 1; i <= values.size(); i++) {
		center_angle_radians(values(i));
	}

	minimize_reset_backrub_eval();
}

} // end backrub_ns namespace

////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_set_backrub_eval
///
/// @brief
/// set global Backrub_eval object for minimization
///
/// @detailed
///
/// @param[in] backrub_eval - Backrub_eval object for minimization
///
/// @global_read
///
/// @global_write
/// backrub_ns::minimize_backrub_eval
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 26, 2006
////////////////////////////////////////////////////////////////////////////////
void
minimize_set_backrub_eval(
	backrub_ns::Backrub_eval & backrub_eval
)
{
	assert(backrub_ns::minimize_backrub_eval == 0);
	backrub_ns::minimize_backrub_eval = &backrub_eval;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_get_backrub_eval
///
/// @brief
/// get global Backrub_eval object for minimization
///
/// @detailed
///
/// @param
///
/// @global_read
/// backrub_ns::minimize_backrub_eval
///
/// @global_write
///
/// @return
/// Backrub_eval object
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 26, 2006
////////////////////////////////////////////////////////////////////////////////
backrub_ns::Backrub_eval &
minimize_get_backrub_eval()
{
	if (backrub_ns::minimize_backrub_eval == 0) {
		std::cout << "minimize_get_backrub_eval called with uninit backrub_eval"
		          << std::endl;
		assert(false);
		utility::exit(EXIT_FAILURE, __FILE__, __LINE__);
	}
	return *(backrub_ns::minimize_backrub_eval);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_reset_backrub_eval
///
/// @brief
/// clear reference to global Backrub_eval object
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
/// This object is not freed. That is the responsibility of the caller.
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 26, 2006
////////////////////////////////////////////////////////////////////////////////
void
minimize_reset_backrub_eval()
{
	backrub_ns::minimize_backrub_eval = 0;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_func_backrub_eval
///
/// @brief
/// evalutate bond angle energies using global Backrub_eval object
///
/// @detailed
///
/// @param[in] values - angular values for Backrub motions (radians)
/// @param[out] gfrag - flag to indicate if angles are valid numbers (unused)
///
/// @global_read
///
/// @global_write
///
/// @return
/// bond angle energy (kcal/mole)
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 26, 2006
////////////////////////////////////////////////////////////////////////////////
float
minimize_func_backrub_eval(
	FArray1DB_float & values,
	bool & gfrag
)
{
	assert(backrub_ns::minimize_backrub_eval != 0);
	return backrub_ns::minimize_backrub_eval->func(values, gfrag);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin minimize_dfunc_backrub_eval
///
/// @brief
/// evalutate bond angle energy derivatives using global Backrub_eval object
///
/// @detailed
///
/// @param[in] values - angular values for Backrub motions (radians)
/// @param[out] derivatives - derivative for each Backrub motion
/// @param[in] n - number of backrub motions
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified April 26, 2006
////////////////////////////////////////////////////////////////////////////////
void
minimize_dfunc_backrub_eval(
	FArray1DB_float & values,
	FArray1DB_float & derivatives,
	int n
)
{
	assert(backrub_ns::minimize_backrub_eval != 0);
	backrub_ns::minimize_backrub_eval->dfunc(values, derivatives, n);
}
