// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13616 $
//  $Date: 2007-03-18 08:39:36 +0200 (Sun, 18 Mar 2007) $
//  $Author: stuartm $


// Rosetta Headers
#include "pose_idealize.h"
#include "after_opts.h"
#include "aaproperties_pack.h"
#include "atom_tree_routines.h"
//#include "capri6.h"
#include "cst_set.h"
#include "disulfides.h"
#include "files_paths.h"
//  #include "filters.h"
//  #include "fragment_class.h"
//  #include "fragments.h"
//  #include "fragments_pose.h"
//  #include "fragments_ns.h"
#include "fullatom.h"
//  #include "fullatom_setup.h"
#include "initialize.h"
//  #include "jumping_util.h"
//  #include "jumping_loops.h"
//  #include "jumping_pairings.h" // get_loop_fraction
#include "jumping_diagnostics.h"
#include "loop_relax.h"
//  #include "make_pdb.h"
//  #include "misc.h" // damn
#include "minimize.h"
//  #include "monte_carlo.h"
#include "nblist.h"
#include "output_decoy.h"
#include "pack_fwd.h"
#include "param.h"
#include "param_aa.h"
//  #include "param_pack.h"
//  #include "design.h"
#include "pose.h"
#include "pose_constraints.h"
#include "pose_io.h"
#include "pose_rms.h"
#include "relax.h"
//  #include "random_numbers.h"
#include "rotamer_trials.h" // random_order
#include "score.h"
//  #include "silent_input.h"
//  #include "smallmove.h"
//  #include "ssblocks.h" // charlie's heap stuff
//#include "tether.h"
#include "timer.h"
//  #include "t267.h"
//  #include "util_vector.h"
#include "util_basic.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/Time_Date.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <sstream>
#include <fstream>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>
#include <utility/io/ocstream.hh>
#include <utility/file/file_sys_util.hh>

// called from main inside decoy loop
//
void
main_pose_idealize()
{
	pose_ns::Pose pose;
	std::string const filename( stringafteroption("s") );
	pose_from_pdb( pose, filename, true, false ); // fa,ideal
	float const start_score( pose.score(score12));
	pose.refold_sidechains_from_chi();
	std::cout << "Score delta from sidechain flipping: " <<
		pose.score( score12 ) - start_score << std::endl;
	idealize_pose( pose );
	pose.dump_pdb( filename+".idl_pdb" );
}


//////////////////////////////////////////////////////////////////
// this does call refold on the pose at the start
// so it should already have bonds setup or at least no
// new_*_refold!

void
idealize_pose(
	pose_ns::Pose & pose,
	bool const ignore_allow_move // = true

)
{
	// 2005-08-11: refactoring pose stuff
	// now all the calc_bonds stuff should already have been done
	// also assuming that sidechains have already been flipped

	// 09-16: allow idealizing subset of rsds

	using namespace pose_ns;
	using namespace cst_set_ns;
	using namespace relax_options;

	if ( pose.ideal_backbone() ) {
		std::cout << "idealize_pose:: backbone is already ideal!\n";
	}
	const int nres( pose.total_residue() );
	const int num_jump( pose.num_jump() );
	const bool fullatom( get_fullatom_flag() );

	/////////
	// params
	int const window_size_default( 1 ); // rsd on either side
	int const window_size_pro    ( 2 ); // rsd on either side

	if ( fullatom )	pose.set_fullatom_flag( true, true );

	std::cout << "In pose_idealize fullatom " << fullatom << std::endl;

	float const atr_weight_low( 0.05 );
	float const rep_weight_low( 0.05 );
	float const atr_weight( 1.0 );
	float const rep_weight( 1.0 );
	float const dun_weight( 0.1 );
	float const chi_weight( 1.0 );

	float const vdw_weight( 1.0 );
	float const vdw_weight_low( 0.05 );
	float const rama_weight( 0.1 );

	// tether weights:
	float const atompair_weight( 500.0 );
	float const coord_weight( 1000.0 );
	float const phipsi_weight( 0.2 );
	float const omega_weight( 1.0 );



	// which residues to idealize?
	FArray1D_bool idealize_bb( nres, true ); //svn version set to true
	FArray1D_bool idealize_sc( nres, fullatom ); //svn version set to true
	FArray1D_bool idealize_jump( num_jump, true );

	std::vector < int > segment_begin;
	std::vector < int > segment_end;

	if( !ignore_allow_move ){
		if ( get_pose_idlz_flag() ){
			//idealize based on the input segment file
			assert( num_jump == 0 );
			read_segment_file( segment_begin, segment_end );
			for (int i = 1; i <= pose.total_residue(); ++i) {
				idealize_bb( i ) = false;
				if ( fullatom )	idealize_sc( i ) = false;
			}
			for ( int j = 0; j < int(segment_begin.size()); ++j) {
				for ( int k = segment_begin[j]; k <= segment_end[j]; ++k){
					std::cout << "idealize pos k " << k << std::endl;
					idealize_bb( k ) = true;
					if ( fullatom )	idealize_sc( k ) = true;
				}
			}
		} else {
			//idealize based on the allow_move_array
			for ( int i=1; i<= nres; ++i ) {
				idealize_bb(i) = pose.get_allow_bb_move(i);
				if ( fullatom ) idealize_sc(i) = pose.get_allow_chi_move(i);
			}
			for( int i = 1; i <= num_jump; ++i ){
				idealize_jump( i ) = pose.get_allow_jump_move( i );
				if( idealize_jump ( i ) ){
					std::cout << "will idealize at jump: " << i << std::endl;
				}
			}
		}
	}

	// status output:
	for ( int i = 1; i <= nres; ++i ){
		if ( idealize_bb( i ) ) std::cout << "will idealize bb at pos " << i <<
															std::endl;
		if ( fullatom ) {
			if( idealize_sc( i ) ) std::cout << "will idealize sc at pos " << i <<
															 std::endl;
		}
	}
	for ( int i = 1; i <= num_jump; ++i ){
		if ( idealize_jump ( i ) ){
			std::cout << "will idealize at jump: " << i << std::endl;
		}
	}


	///////////////
	// pose objects
	Pose start_pose;
	Cst_set cst_set;
	bool const just_use_bb_heavy_atoms ( !fullatom );
	// scorefxn, more terms later
	Score_weight_map weight_map;
	fill_cst_set( cst_set, pose, just_use_bb_heavy_atoms );
	pose.set_constraints( cst_set );

	//psh include disulfide constraints
	bool const disulf_flag = disulfides::BOUNDARY::get_disulf_flag();
	if (disulf_flag){
	  add_disulf_pose_constraints( pose);
	}

	weight_map.set_weight( RAMACHANDRAN, rama_weight );
	weight_map.set_weight( ATOMPAIR_CST,
		atompair_weight / cst_set.count_atompair_constraints() );
	weight_map.set_weight( PHIPSI_CST, phipsi_weight / (2*nres) );
	weight_map.set_weight( OMEGA_CST, omega_weight / nres );
	if ( fullatom ) {
		weight_map.set_weight( COORD_CST,
			coord_weight / cst_set.count_coordinate_constraints() );
		weight_map.set_weight( CHI_CST,
			chi_weight / ( cst_set.count_chi_constraints() ) );
		weight_map.set_weight( FA_SCOREFXN, 1.0 );
		weight_map.set_weight( FA_ATR, atr_weight );
		weight_map.set_weight( FA_REP, rep_weight );
		weight_map.set_weight( FA_DUN, dun_weight );
	} else {
		weight_map.set_weight( VDW, vdw_weight );
	}

	pose.score( weight_map );
	start_pose = pose;

	pose.set_start_pose( start_pose );

	// setup constraint set, includes atompairs and torsion tethers
	/*
	Cst_set cst_set;
	bool const just_use_bb_heavy_atoms ( true );
	fill_cst_set( cst_set, pose, just_use_bb_heavy_atoms );
	weight_map.set_weight( ATOMPAIR_CST, atompair_weight );
												 //		atompair_weight / cst_set.count_atompair_constraints() );
	weight_map.set_weight( COORD_CST, coord_weight );
												 //												 coord_weight / cst_set.count_coordinate_constraints() );
	std::cout << "coord_weight " << coord_weight << " cst_set.count_coordinate_constraints() " << cst_set.count_coordinate_constraints() << " ratio " << coord_weight / cst_set.count_coordinate_constraints() << std::endl;

	*/
	//	if ( fullatom ) 	weight_map.set_weight( CHI_CST,chi_weight / ( cst_set.count_chi_constraints() ) );


	///////////////////////////////
	// start minimizing
	// params:
	minimize_set_vary_phipsi( true );
	minimize_set_vary_omega( true ); // only this is changed later
	minimize_set_vary_rb_angle( true ); // controlled by allow_jump_move
	minimize_set_vary_rb_trans( true );
	minimize_set_tolerance( 0.001 ); // small
	//	minimize_set_xx_init( 0.1 ); // same setting as during smallmin
//  	minimize_set_xx_init( 0.001 ); // same setting as during idealization
	minimize_set_local_min( false, 0 );
	score_set_try_rotamers(false);
	pose_set_use_nblist(true);
	minimize_exclude_sstype(false,false);
	if ( fullatom ) minimize_set_vary_chi( true );



	/////////////////////////////////////////////////////////////////////////////
	// idealize sidechains:
	if( fullatom )	{
		pose.set_allow_bb_move( false );
		pose.set_allow_jump_move( false );
		weight_map.set_weight( FA_ATR, atr_weight_low);
		weight_map.set_weight( FA_REP, rep_weight_low);

		// use a ran3-based shuffle from rosetta
		FArray1D_int pos_list(nres);
		for ( int i=1; i<= nres; ++i ) {
			pos_list(i) = i;
		}
		random_order(pos_list,nres);
		for ( int ipos = 1; ipos <= nres; ++ipos ) {
			int const pos( pos_list(ipos) );
			if ( !idealize_sc(pos) ) continue;
			int const aa( pose.res(pos ));
			int const aav( pose.res_variant(pos ));
			if ( aa == param_aa::aa_ala || aa == param_aa::aa_gly ) continue;

			// allow movement here:
			pose.set_allow_chi_move( false );
			pose.set_allow_chi_move( pos, true );

			// store the starting chi angles
			int const nchi( aaproperties_pack::nchi(aa,aav));
			FArray1D_float chi(param::MAX_CHI);
			for( int chino=1; chino<= nchi; ++chino ) {
				chi(chino) = pose.chi( chino, pos );
			}

			// idealize sidechain at this position
			// bit of a hack: call repack with include_current = false
			// then reset to original chi torsions
			FArray1D_bool allow_repack( nres, false );
			allow_repack( pos ) = true;
			bool const include_current( false );
			pose.repack( allow_repack, include_current );
			for( int chino=1; chino<= nchi; ++chino ) {
				pose.set_chi( chino, pos, chi(chino) );
			}

			// minimize w/ moderate atr,rep

			pose.main_minimize( weight_map, "dfpmin" );
			idl_log( pose, start_pose, "sc_idl "+string_of( pos ), fullatom );
		}
	}

	////////////////////
	// idealize backbone
	// dont use coord constraints anymore
	cst_set.clear_coordinate_constraints();

	// for minimizing the jumps, setup a map from residues to jumps
	std::map< int, std::vector< int > > jump_map;
	const FArray2D_int & jump_point( pose.fold_tree().get_jump_point() );
	for( int i=1; i<= num_jump; ++i ) {
		if ( idealize_jump(i) ) {
			jump_map[ jump_point(1,i) ].push_back(i);
			jump_map[ jump_point(2,i) ].push_back(i);
		}
	}


	{ // loop over residues to idealize
		// use a ran3-based shuffle from rosetta
		FArray1D_int pos_list(nres);
		for ( int i=1; i<= nres; ++i ) {
			pos_list(i) = i;
		}
		random_order(pos_list,nres);
		for ( int ipos=1; ipos <= nres; ++ipos ) {
			int const pos( pos_list(ipos) );
			if ( !idealize_bb(pos) ) continue;
			const int aa( pose.res(pos));

			// allow movement in a window around the residue
			pose.set_allow_bb_move( false );
			pose.set_allow_chi_move( false );
			pose.set_allow_jump_move( false );
			const int window_size
				( aa == param_aa::aa_pro ? window_size_pro : window_size_default );
			for ( int i=pos-window_size; i<= pos+window_size; ++i ) {
				if ( i>=1 && i<= nres ) {
					if ( pose.res(i) != param_aa::aa_pro && idealize_bb(i) ) {
						pose.set_allow_bb_move(i,true);
					}
				}
			}

			if ( jump_map.count(pos) ) {
				// idealize jumps
				for ( std::vector< int >::const_iterator it=jump_map[pos].begin(),
								it_end = jump_map[pos].end(); it != it_end; ++it ) {
					const int jump_number( *it );
					assert ( idealize_jump(jump_number));
					std::cout << "idealizing jump: " << pos << ' ' << jump_number <<
						std::endl;
					pose.set_allow_jump_move( jump_number, true );
				}
			}

			// idealize at this position
			pose.set_allow_bb_move( pos, true ); // in case PRO
			pose.insert_ideal_bonds( pos, pos );
			if ( pose.res(pos) == param_aa::aa_pro ) {
				// dont minimize pro torsions!
				pose.set_allow_bb_move( pos, false );
			}

			// minimize w/o atr,rep, omega
			minimize_set_vary_omega( false );
			if( fullatom ){
				weight_map.set_weight( FA_ATR, 0.0 );
				weight_map.set_weight( FA_REP, 0.0 );
			} else {
				weight_map.set_weight( VDW, 0.0 );
			}
			pose.main_minimize( weight_map, "dfpmin" );
			idl_log( pose, start_pose, "bb_min1 "+string_of( pos ), fullatom );

			// minimize w/ moderate atr,rep, omega varying
			minimize_set_vary_omega( true );
			if ( fullatom ){
				weight_map.set_weight( FA_ATR, atr_weight_low );
				weight_map.set_weight( FA_REP, rep_weight_low );
			} else {
				weight_map.set_weight( VDW, vdw_weight_low );
			}
			pose.main_minimize( weight_map, "dfpmin" );
			idl_log( pose, start_pose, "bb_min2 "+string_of( pos ), fullatom );
		}
	}


	if ( ignore_allow_move == true ) {
		float const score_before( pose.score( weight_map ) );
		pose.reset_bonds();
		pose.score( weight_map );
		std::cout << "score-delta-from-ideal-bonds: " <<
			pose.get_0D_score( SCORE ) - score_before << std::endl;
	}

	////////////////////////////////////////////////
	// round1 of full minimization, at all positions
	// finally use full atr weight, bb-only, no omega
	if ( fullatom ) {
		weight_map.set_weight( FA_ATR, atr_weight );
		weight_map.set_weight( FA_REP, rep_weight );
		pose.set_allow_chi_move( false );
	} else {
		weight_map.set_weight( VDW, vdw_weight );
	}
	minimize_set_vary_omega( false );
	pose.set_allow_bb_move( idealize_bb );
	pose.set_allow_jump_move( false );
	pose.main_minimize( weight_map, "dfpmin" );
	idl_log( pose, start_pose, "main_min1", fullatom );

	///////////////////////////////////////////
	// round2 of minimization, at all positions
	// finally use full atr weight, vary omega, chi
	minimize_set_vary_omega( true );
	pose.set_allow_bb_move( idealize_bb );
	if ( fullatom ) pose.set_allow_chi_move( idealize_sc );
	pose.set_allow_jump_move( idealize_jump );
	pose.main_minimize( weight_map, "dfpmin" );
	idl_log( pose, start_pose, "main_min1", fullatom );

	// clear constraints pointer
	pose.reset_constraints();
	pose.reset_start_pose();

}

/////////////////////////////////////////////////////////////////
// writes info to stdout, does not add a newline
//
void
idl_log(
	pose_ns::Pose & pose,
	pose_ns::Pose const & start_pose,
	std::string const & tag,
	bool const full_atom
)
{
	int const nres( pose.total_residue() );

	float max_phipsi_dev( 0.0 );
	float max_omega_dev( 0.0 );
	float max_chi_dev( 0.0 );
	for ( int i=1; i<= nres; ++i ) {
		float phi_dev
			( std::abs( subtract_degree_angles( pose.phi(i),
																					start_pose.phi(i))));
		float psi_dev
			( std::abs( subtract_degree_angles( pose.psi(i),
																					start_pose.psi(i) )));
		float omega_dev
			( std::abs( subtract_degree_angles( pose.omega(i),
																					start_pose.omega(i))));

		max_phipsi_dev = std::max( max_phipsi_dev, std::max( phi_dev, psi_dev ) );
		max_omega_dev = std::max( max_omega_dev, omega_dev );

		int const aa( pose.res(i));
		int const aav( pose.res_variant(i));
		if( full_atom ) {
			int const nchi( aaproperties_pack::nchi(aa,aav));
			for ( int chino=1; chino<= nchi; ++chino ) {
				float chi_dev
					( std::abs( subtract_degree_angles( pose.chi(chino,i),
																							start_pose.chi(chino,i))));
				max_chi_dev = std::max( max_chi_dev, chi_dev );
			}
		}
	}

	calc_rms( pose );

	// will fail if atompair not in the weight_map
	std::cout <<
		" s_rms: "      << F(6,3,pose.get_0D_score( pose_ns::START_RMSD )) <<
		" ap_cst: "   << F(9,2,pose.get_0D_score( pose_ns::ATOMPAIR_CST) ) <<
		" mx_pp_dev: " << F(6,1,max_phipsi_dev) <<
		" mx_om_dev: "  << F(6,1,max_omega_dev) <<
		" mx_ch_dev: "    << F(6,1,max_chi_dev) <<
		' ' << tag << std::endl;

	//pose.set_extra_score( "MAX_PHIPSI_DEV", max_phipsi_dev );
	//pose.set_extra_score( "MAX_OMEGA_DEV", max_omega_dev );
	//pose.set_extra_score( "MAX_CHI_DEV", max_chi_dev );
}


////////////////////////////////////////////////////////////////////////////////
/// @begin read_segment_file
///
/// @brief read in the segment file defined as -loop_file in command line
///
/// @detailed
///
/// @param  tolerance - [in/out]? -
/// @param  size - [in/out]? -
/// @param  frag_begin - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors sraman
///
/// @last_modified Mar 24th 2006
/////////////////////////////////////////////////////////////////////////////////
void
read_segment_file(
	std::vector<int> & loop_begin,
	std::vector<int> & loop_end
)
{
	std::string filename;
	stringafteroption( "loop_file", "", filename);

	std::cout << "Reading segments file for idealization...." << std::endl;
	if ( filename == "" ){
		std::cout << "No loop_file specified! Try to find loopfile in start_path!"
		<< std::endl;
		filename = files_paths::start_path + files_paths::start_file + ".loopfile";
	}
	utility::io::izstream data( filename.c_str() );
	if ( !data ) {
		std::cout << "Couldn't open loop file " << filename << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	std::string line;
	while ( getline( data, line ) ) {
		std::istringstream line_stream( line );
		int begin,end;
		line_stream >> begin >> end;
		if ( !line_stream.fail() ) {
			loop_begin.push_back ( begin );
			loop_end.push_back ( end );
			std::cout << "segments file for idealization: " << ' ' << begin << ' ' << end << std::endl;
		}
	}
	data.close();
}
