// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 7212 $
//  $Date: 2006-01-04 07:49:01 -0800 (Wed, 04 Jan 2006) $
//  $Author: pbradley $


// Rosetta Headers
#include "pose_relax.h"
#include "after_opts.h"
#include "aaproperties_pack.h"
#include "atom_tree_routines.h"
#include "disulfides.h"
#include "fullatom.h"
#include "job_distributor.h" // for pose Monte_carlo checkpointing
#include "jumping_util.h"
#include "minimize.h"
#include "misc.h" // pose_flag()
#include "monte_carlo.h"
#include "nblist.h"
#include "output_decoy.h"
#include "pose.h"
#include "pose_benchmark.h"
#include "pose_correlated_angles.h"
#include "pose_io.h"
#include "pose_design.h"
#include "pose_rotamer_trials.h"
#include "prof.h"
#include "relax_structure.h"
#include "read_aaproperties.h"
#include "refold.h" //for setting up segment map.
#include "score.h"
#include "score_ns.h" //for tether score setup
#include "smallmove.h"
#include "tether.h" // pairwise coordinate tethers (classical)
#include "tether_ns.h" // omega_tether
#include "torsion_bbmove_trials.h"
#include "util_vector.h"

//rhiju -- useful in setting up a test routine.
#include "initialize.h"
#include "jumping_pairings.h"
#include "maps.h"
#include "pose_rms.h"
#include "files_paths.h"
#include "fragments.h"
#include "silent_input.h"

// Numeric Headers
#include <numeric/xyz.functions.hh> // for dihedral()

// ObjexxFCL Headers
#include <ObjexxFCL/string.functions.hh>
#include <ObjexxFCL/formatted.o.hh>

// C++ Headers
#include <cstdlib>
#include <iostream>
#include <fstream>


///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////
// initial repack with lowered repulsive
//
// couple of cycles of:
// repack,minimize,small trials w rotamer-trials on
//
// score_filter_tag lets us distinguish between different sequences
// for the purpose of score-filtering
//
// Added filter1 and filter2 for user-defined score-filters.
// Set score_filter_acceptance_rate to 1.0 if you want to use those.
///////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void
fast_relax_pose(
	pose_ns::Pose & pose,
	pose_ns::Score_weight_map & weight_map,
	int const lj_ramp_cycles,
	int const cycles,
	std::string const & score_filter_tag,
	float const score_filter_acceptance_rate,
	bool vary_sidechain_bond_angles,
	float const filter1, //default is a really big number
	float const filter1a, //default is a really big number
	float const filter1b, //default is a really big number
	float const filter2 //default is a really big number
)
{
	using namespace pose_ns;

	// params
	float const temperature ( 0.8 );
	float const energycut(0.01); // for rotamer_trials
	bool const use_nblist( true );
	//	bool const geometric_ramp( false ); // ramping the repulsive
	static bool const geometric_ramp = truefalseoption( "geometric_ramp");

	float const start_rep_weight( 0.02 );
	float const end_rep_weight( 1.0 );
	int const lj_ramp_inner_cycles( 10 );
	// Halved number of final cycles (stage 3), in favor of
	// more stringent minimization in earlier, more important stage (stage 2).
	// (DB,RD 1-10-07)
	int final_cycles = cycles/2;
	get_final_cycles( final_cycles ); //default: final_cycles = cycles/2.

	bool const deriv_checking( truefalseoption("pose_relax_deriv_check") );
	bool const do_fragment_moves = truefalseoption( "pose_relax_fragment_moves" );
	if (do_fragment_moves) initialize_segment_map_from_pose( pose );

	bool symm_jump_moves_during_relax;
	float trans_mag, rot_mag;
	get_symm_jump_moves_during_relax( symm_jump_moves_during_relax, trans_mag, rot_mag );

	// these are the stringent relax settings, I think
	float round1_tolerance(0.00025);
	float round2_tolerance(0.00025); //can get changed below.
	float round3_tolerance(0.000025);
	float final_min_tolerance(0.000005);
	std::string min_type = "dfpmin";
	// Stringent relax before stage2 on by default (DB,RD 1-10-07)
	if (get_stringentrelax_before_stage2()) round2_tolerance = 0.000025;		// uses relax_structure.cc
	if (use_abs_tolerance()) { //chu use absolute tolerance, uses relax_structure.cc
		round1_tolerance = 0.001;
		round2_tolerance = 0.001;
		round3_tolerance = 0.0001;
		final_min_tolerance = 0.00005;
		if(get_stringentrelax_before_stage2()) round2_tolerance = 0.0001;
		min_type = "dfpmin_atol";
	}
	//Wobble and crank settings, use relax_structure.cc
	int const nwobble1_crank = get_nwobble1_crank(); //default 3
	int const nwobble2_crank = get_nwobble2_crank(); //default 2
	int const nwobble_wobble = get_nwobble_wobble(); //default 1
	int const crank_size  = get_crank_size(); //default 3
	int const wobble_size = get_wobble_size();//default 3
	int const stage2_repack_period = get_stage2_repack_period(); //default 25

	// setup
	score_set_try_rotamers(false);
	pose_set_use_nblist( use_nblist );

	// Local min stuff currently ignored by pose torsion bbmoves?
	int local_min_window = get_local_min_window(); //default 5, uses relax_structure.cc
	if (get_minimize_set_local_min()) minimize_set_local_min(true,local_min_window);
	else minimize_set_local_min(false,local_min_window);

	minimize_exclude_sstype(false,false);
	minimize_set_vary_phipsi(true);
	minimize_set_vary_chi(false); // turns on in round3
	set_smallmove_size(2.0,2.0,3.0); // helix,strand,other

	// vary omega:
	//score_set_vary_omega( false ); // PBHACK
	if( score_get_vary_omega() ){		// uses relax_structure.cc
		pose.copy_to_misc(); // uses misc::omega
		setup_omega_tether( pose );
		minimize_set_vary_omega(true);
	} else {
		minimize_set_vary_omega(false);
	}


	static bool const tether_during_stage1 = truefalseoption( "tether_during_stage1" );
	if (tether_during_stage1) {
		setup_backbone_pairwise_tethers( pose, weight_map );
		pose_set_use_nblist( false );
	}

	// Sort of arbitrary chainbreak parameters... might be worth ramping up.
	// Wait, not all pose_relaxers want chainbreaks closed! E.g. symmetric docking.
	//	weight_map.set_weight( CHAINBREAK,         1.0);
	//	weight_map.set_weight( CHAINBREAK_OVERLAP, 1.0);

	// monte-carlo object
	Monte_carlo mc( pose, weight_map, temperature );

	// turn on profiling
	//	prof::reset( true );

	// Initialize new score columns for constraints related to varying bond geometry
	bool vary_sidechain_bond_angles_before_stage2;
	bool vary_all_bond_distances, vary_all_bond_distances_before_stage2;
	bool vary_all_bond_angles, vary_all_bond_angles_before_stage2;
	initialize_vary_bond_geometry_flags( vary_sidechain_bond_angles, vary_sidechain_bond_angles_before_stage2,
																			 vary_all_bond_distances, vary_all_bond_distances_before_stage2,
																			 vary_all_bond_angles, vary_all_bond_angles_before_stage2, weight_map, mc);

	// for tracking the simulation: how many filters did we pass
	int final_round(1);

	///////////////////////////////////////////////////////
	// packer params:
	//initialize_fullatom(); //ensure that this has been done
	set_rot_limit( 45, 24 ); //like in relax_structure.
	set_perc_limit( 0.9999, 0.9999 );
	get_packing_options(); // does it hurt to set this up again?

	minimize_set_tolerance( round1_tolerance );
	if (get_vary_chi_before_stage1()) minimize_set_vary_chi(true);

	if ( lj_ramp_cycles ) {
		///////////////////////////////////////////////////////////////////////////
		///////////////////////////////////////////////////////////////////////////
		///////////////////////////////////////////////////////////////////////////
		// PART I
		// lj-ramp cycles
		///////////////////////////////////////////////////////////////////////////
		///////////////////////////////////////////////////////////////////////////
		///////////////////////////////////////////////////////////////////////////

		if ( deriv_checking ) pose_deriv_check( pose, weight_map, "round1" );

		//////////////////////////////////////
		// setup ramping of the repulsive term
		// geometric
		// rep_delta^(lj_ramp_cycles-1) = end_rep_weight / start_rep_weight
		float rep_delta, rep_weight( 1.0 );
		if ( geometric_ramp ) {
			rep_delta =
				( std::exp( std::log( end_rep_weight / start_rep_weight ) /
										(lj_ramp_cycles) ) );
			rep_weight = start_rep_weight / rep_delta;
		} else {
			rep_delta = ( end_rep_weight - start_rep_weight ) / lj_ramp_cycles;
		}

		// lj_ramp_cycles
		for ( int n=0; n<= lj_ramp_cycles; ++n ) {

			// ONLY CHECKPOINT HERE IN THIS LOOP
			if (mc.checkpoint( pose )) continue;

			if ( geometric_ramp ) {
				rep_weight *= rep_delta;
			} else {
				rep_weight = start_rep_weight + n * rep_delta;
			}

			assert( n < lj_ramp_cycles ||
							std::abs( rep_weight - end_rep_weight ) < 1e-2 );

			//weight_map.set_weight( FA_REP, rep_weight );
			score_set_lj_rep_weight( rep_weight );
			mc.set_weight_map( weight_map );

			// recover low
			mc.recover_low( pose );

			///////////////////////
			// start with a repack:
			if ( !pose.fullatom() ) {
				pose.set_fullatom_flag( true, true /* repack rotamers */);
				mc.reset( pose );
			} else {
				pose.full_repack( true );
				mc.boltzmann( pose, "ljramp_repack" );
			}

			///////////////
			// now minimize
			bool const use_nblist_local = get_use_nblist();
			pose_set_use_nblist( false );
			pose.main_minimize( weight_map, min_type );
			pose_set_use_nblist( use_nblist_local );
			mc.boltzmann( pose, "ljramp_minimize" );

			//////////////
			// small moves
			for ( int m=1; m<= lj_ramp_inner_cycles; ++m ) {

				pose_small_min_trial( pose, mc, weight_map, "linmin",5,true,energycut,
															"round1" );
				pose_shear_min_trial( pose, mc, weight_map, "linmin",10,true,energycut,
															"round1" );
			}
		} //ramp_outer_cycle

		// recover low
		mc.recover_low( pose );
	} else {
		if ( !pose.fullatom() ) {
			std::cout << "WARNING:: pose_relax called on centroid pose w/ " <<
				"lj_ramp_cycles = 0!" << std::endl;
			pose.set_fullatom_flag( true, true /* repack rotamers */);
			mc.reset( pose );
		}
	} // ramp_rep or not?

	// repack
	if (!mc.checkpoint( pose )) pose.full_repack( true );

	static bool const relax_rtmin = truefalseoption("relax_rtmin");
	if (!mc.checkpoint( pose )) {
		if ( relax_rtmin ) {
			pose_rottrial_minimization(pose, weight_map );
			mc.boltzmann( pose, "repack_rtmin");
		} else {
			mc.boltzmann( pose, "repack" );
		}

		// minimize
		minimize_set_tolerance( round2_tolerance );
		// 	pose.main_minimize( weight_map, min_type );
		// 	mc.boltzmann( pose, "minimize" );
	}

	if (tether_during_stage1) {
		turn_off_backbone_pairwise_tethers( weight_map );
		pose_set_use_nblist( use_nblist );
	}

	mc.recover_low( pose );

	float current_score( 0.0 );
	bool pass_score_filter ( true );

	pose_save_relax_score_filters( pose, current_score, mc, "RLXSCOR1",
																 filter1, pass_score_filter );

	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	// PART II
	// small/shear moves, no vary chi
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	// Vary chi angles  before stage 2 on by default (DB,RD 1-10-07)
	if (get_vary_chi_before_stage2()) minimize_set_vary_chi(true);

	// check score
	if ( main_job_distributor::jd->skip_to_mc_checkpoint() ||
					( score_filter( current_score, score_filter_tag+"round1",
								score_filter_acceptance_rate ) && pass_score_filter)) {
		// record that we got to 2nd round
		final_round = 2;

		initialize_vary_bond_geometry_constraints( vary_sidechain_bond_angles_before_stage2, vary_all_bond_distances_before_stage2,
																							 vary_all_bond_angles_before_stage2, pose, weight_map, mc);

		// params
		minimize_set_tolerance( round2_tolerance );

		// deriv-check
		if ( deriv_checking ) pose_deriv_check( pose, weight_map, "round2" );

		// minimize -- no, sync w/ fullatom_relax
// 		pose.main_minimize( weight_map, min_type );
// 		mc.boltzmann( pose, "minimize" );

		static bool const replace_crank_with_wobble =  truefalseoption("replace_crank_with_wobble");


		// cycles:
		for ( int n=1; n<= cycles; ++n ) {

			// ONLY CHECKPOINT HERE IN THIS LOOP
			if (mc.checkpoint( pose )) continue;

			if (!do_fragment_moves) {
				if ( symm_jump_moves_during_relax ) {
					pose_symm_rb_min_trial( pose, mc, weight_map, min_type, trans_mag, rot_mag, true,energycut,"round2");
				}
				pose_small_min_trial( pose, mc, weight_map, min_type, 5 ,true,energycut,
															"round2");
				pose_shear_min_trial( pose, mc, weight_map, min_type, 10,true,energycut,
															"round2");

				//rhiju -- make comparison to wobble fair -- equal number of moves!
				pose_small_min_trial( pose, mc, weight_map, min_type, 5 ,true,energycut,
															"round2");
				pose_shear_min_trial( pose, mc, weight_map, min_type, 10,true,energycut,
															"round2");

			} else { // match fullatom_relax
 				if ( symm_jump_moves_during_relax ) {
 					pose_symm_rb_min_trial( pose, mc, weight_map, min_type, trans_mag, rot_mag, true,energycut,"round2");
 				}
 				pose_small_wobble_min_trial( pose, mc, weight_map, min_type,
 																		 1, 7, 0, true, energycut, "round2");
 				pose_small_min_trial( pose, mc, weight_map, min_type, 5 ,true, energycut,
 															"round2");

				if ( replace_crank_with_wobble ){
					pose_wobble_min_trial( pose, mc, weight_map, min_type,
																 wobble_size, 60.0, nwobble_wobble, 0, true, energycut,"round2");
				}	else {
					pose_crank_min_trial( pose, mc, weight_map, min_type,
																crank_size, 60.0, nwobble1_crank, nwobble2_crank,
																true, energycut,"round2");
				}

				pose_wobble_min_trial( pose, mc, weight_map, min_type,
															 wobble_size, 60.0, nwobble_wobble, 0, true, energycut,"round2");

			}
			if ( mod(n,stage2_repack_period) == 0 ) {
				// repack
				pose.full_repack( true );
				mc.boltzmann( pose, "repack" );
			}

			if ( n == cycles/4 )	{
				pose_save_relax_score_filters( pose, current_score, mc, "RLXSCOR1A",
																			 filter1a, pass_score_filter );
				if (!pass_score_filter) break;
			}
			if ( n == cycles/2 )	{
				pose_save_relax_score_filters( pose, current_score, mc, "RLXSCOR1B",
																			 filter1b, pass_score_filter );
				if (!pass_score_filter) break;
			}
		}

		mc.recover_low( pose );

		// repack
		pose.full_repack( true );
		if ( relax_rtmin ) {
			pose_rottrial_minimization(pose, weight_map );
			mc.boltzmann( pose, "repack_rtmin");
		} else {
			mc.boltzmann( pose, "repack" );
		}

		// minimize
// 		pose.main_minimize( weight_map, min_type );
// 		mc.boltzmann( pose, "minimize" );

		mc.recover_low( pose );


	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	// PART III
	// small/shear moves, vary chi
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
		if (pass_score_filter) pose_save_relax_score_filters( pose, current_score, mc, "RLXSCOR2",
																													filter2, pass_score_filter);
		// check score
		if ( main_job_distributor::jd->skip_to_mc_checkpoint() ||
						( score_filter( current_score, score_filter_tag+"round2",
										score_filter_acceptance_rate ) && pass_score_filter)) {
			// record that we got to the 3rd round
			final_round = 3;

			// params:
			minimize_set_vary_chi(true); // chi min!
			minimize_set_tolerance(round3_tolerance);

			initialize_vary_bond_geometry_constraints( vary_sidechain_bond_angles, vary_all_bond_distances,
																								 vary_all_bond_angles, pose, weight_map, mc);

			if ( truefalseoption("force_atom_tree") ) {
				//PBHACK
				if ( !pose.atom_tree() ) {
					pose.setup_atom_tree();
				}
			}

			// deriv-check
			if ( deriv_checking ) pose_deriv_check( pose, weight_map, "round3" );

			// minimize
			if (!mc.checkpoint( pose )) {
				pose.main_minimize( weight_map, min_type );
				mc.reset( pose );
			}

			// cycles
			for ( int n=1; n<= final_cycles; ++n ) {

				// ONLY CHECKPOINT HERE IN THIS LOOP
				if (mc.checkpoint( pose )) continue;

				pose_small_min_trial( pose, mc, weight_map,min_type,5 ,true,energycut,
															"round3" );
				pose_shear_min_trial( pose, mc, weight_map,min_type,10,true,energycut,
															"round3" );
				if ( mod(n,25) == 0 ) {
					// repack
					pose.full_repack( true );
					mc.boltzmann( pose, "repack" );
				}
			}

			mc.recover_low( pose );

			// repack
			pose.full_repack( true );
			if ( relax_rtmin ) {
				pose_rottrial_minimization(pose, weight_map );
				mc.boltzmann( pose, "repack_rtmin");
			} else {
				mc.boltzmann( pose, "repack" );
			}

			// minimize
			minimize_set_tolerance( final_min_tolerance );
			pose.main_minimize( weight_map, min_type );
			mc.boltzmann( pose, "minimize" );

		} // score-filter
	} // score-filter


	// status info
	prof::show();
	mc.show_counters();

	/////////////////////////////////////////////////////////////////////////////
	// now score final structure
	mc.recover_low( pose );
	score_set_evaluate_all_terms(true);
	pose.score(weight_map);
	score_set_evaluate_all_terms(false);
	pose.set_extra_score("ROUND",final_round);
}

//////////////////////////////////////////////////////////////////////////////
void
pose_save_relax_score_filters(
  pose_ns::Pose & pose,
	float & current_score,
	pose_ns::Monte_carlo & mc,
	std::string const extrascorename,
	float const filter,
	bool & pass_score_filter
)
{
	using namespace pose_ns;

	if (main_job_distributor::jd->skip_to_mc_checkpoint()) {
		pass_score_filter = true;
		return;
	}

	pose = mc.low_pose(); // a little redundant; this was probably called already.

	current_score = pose.get_0D_score( SCORE );

	//Yes there are shorter and more elegant ways to code this:
	if (extrascorename == "RLXSCOR1" )
		pose.set_extra_score("RLXSCOR1", current_score );

	if (extrascorename == "RLXSCOR1" || extrascorename == "RLXSCOR1A" )
		pose.set_extra_score("RLXSCOR1A", current_score );

	if (extrascorename == "RLXSCOR1" || extrascorename == "RLXSCOR1A" ||
			extrascorename == "RLXSCOR1B")
		pose.set_extra_score("RLXSCOR1B", current_score );

	if (extrascorename == "RLXSCOR1" || extrascorename == "RLXSCOR1A" ||
			extrascorename == "RLXSCOR1B" || extrascorename == "RLXSCOR2" )
		pose.set_extra_score("RLXSCOR2", current_score );

	mc.reset( pose ); //put extra scores in low and best.
	pass_score_filter = ( current_score < filter );

	if (!pass_score_filter) std::cout << "Didn't pass scorefilter: " << extrascorename << "." << std::endl;
}

/////////////////////////////////////////////////////////////////////////////////////
void
initialize_vary_bond_geometry_flags( bool & vary_sidechain_bond_angles, bool & vary_sidechain_bond_angles_before_stage2,
																		 bool & vary_all_bond_distances, bool & vary_all_bond_distances_before_stage2,
																		 bool & vary_all_bond_angles, bool & vary_all_bond_angles_before_stage2,
																		 pose_ns::Score_weight_map & weight_map,
																		 pose_ns::Monte_carlo & mc)
{
	using namespace pose_ns;

	vary_sidechain_bond_angles_before_stage2 = truefalseoption( "vary_sidechain_bond_angles_before_stage2");
	vary_all_bond_distances = truefalseoption( "vary_all_bond_distances");
	vary_all_bond_angles = truefalseoption( "vary_all_bond_angles");
	vary_all_bond_distances_before_stage2 =	truefalseoption( "vary_all_bond_distances_before_stage2");
 	vary_all_bond_angles_before_stage2 =	truefalseoption( "vary_all_bond_angles_before_stage2");

	//Ensure that bond constraints aren't set up twice .. before stage2, after stage2.
	if (vary_sidechain_bond_angles_before_stage2)        vary_sidechain_bond_angles = false;
	if (vary_all_bond_angles_before_stage2)    vary_all_bond_angles = false;
	if (vary_all_bond_distances_before_stage2) vary_all_bond_distances = false;

	if ( vary_sidechain_bond_angles || vary_sidechain_bond_angles_before_stage2 ){
		weight_map.set_weight( SIDECHAIN_BOND_ANGLE,
													 realafteroption("bond_angle_weight",50.0));
			mc.set_weight_map( weight_map );
	}

	if ( vary_all_bond_distances || vary_all_bond_distances_before_stage2 ){
		weight_map.set_weight( KIN_1D_CST, 1.0 )  ; // the torsion/bond/angle tether
		weight_map.set_weight( ATOMPAIR_CST, 1.0 ); // the ring constraint tether
		mc.set_weight_map( weight_map );
	}


	if ( vary_all_bond_angles || vary_all_bond_angles_before_stage2 ){
		weight_map.set_weight( KIN_1D_CST, 1.0 )  ; // the torsion/bond/angle tether
		weight_map.set_weight( ATOMPAIR_CST, 1.0 ); // the ring constraint tether
		mc.set_weight_map( weight_map );
	}

}

/////////////////////////////////////////////////////////////////////////////////////
void
initialize_vary_bond_geometry_constraints(
   bool const vary_sidechain_bond_angles,
	 bool const vary_all_bond_distances,
	 bool const vary_all_bond_angles,
	 pose_ns::Pose & pose,
	 pose_ns::Score_weight_map weight_map,
	 pose_ns::Monte_carlo & mc
	 )
{

	using namespace pose_ns;

	const std::string min_type = use_abs_tolerance() ? "dfpmin_atol" : "dfpmin";
	if ( vary_sidechain_bond_angles ){
		std::cout << "Varying bond angles in pose_relax!" << std::endl;
		if ( !pose.atom_tree() ) {
			pose.setup_atom_tree();
		}
		set_sidechain_bond_angle_allow_move( pose );
		pose.main_minimize( weight_map, min_type );
		mc.reset( pose );
	}

	if ((vary_all_bond_distances || vary_all_bond_angles) && !pose.atom_tree() ) {
		pose.setup_atom_tree();
	}

	if ( vary_all_bond_distances ) {
		std::cout << "Varying all bond distances in pose_relax!" << std::endl;
		set_all_bond_distance_allow_move( pose );
		pose.main_minimize( weight_map, min_type );
		mc.reset( pose );
	}

	if ( vary_all_bond_angles ) {
		std::cout << "Varying all bond angles in pose_relax!" << std::endl;
		set_all_bond_angle_allow_move( pose );
		pose.main_minimize( weight_map, min_type );
		mc.reset( pose );
	}

}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
typedef numeric::xyzVector_float Vec;
typedef numeric::xyzVector_double DVec;

///////////////////////////////////////////////////////////////////////////////
void
pose_relax_test()
{
	using namespace pose_ns;
	using namespace kin;
	using namespace param_aa;

	Pose pose;
	pose_from_pdb( pose, stringafteroption("s"), true, false );


	pose.setup_atom_tree();

	// first just try minimizing angles

	pose.set_allow_move( ALL, false );
	pose.set_allow_bb_move( false );
	pose.set_allow_chi_move( false );

	// now allow the angles to wiggle
	int const nres( pose.total_residue() );
	for ( int i=1; i<= nres; ++i ) {
		if ( pose.res(i) == aa_gly ) continue;

		// c-beta atom
		Atom_id cbeta( 5, i );

		pose.set_allow_move( cbeta, PHI, true );
		pose.set_allow_move( cbeta, THETA, true );

		// cgamma
		int const cgamma
			( LookupByName(pose.res(i),pose.res_variant(i)," CG " ) );
		if ( cgamma >= 1) {
			pose.set_allow_move( Atom_id( cgamma, i ), THETA, true );


		}
	}

 	Score_weight_map w( score12 );
	w.set_weight( SIDECHAIN_BOND_ANGLE, 50.0 );

	if (truefalseoption("crazy_fold_tree")){
		Fold_tree f = pose.fold_tree();
		f.new_jump( 3, 17, 6);
		pose.set_fold_tree( f );
	}

	if (truefalseoption("vary_all_bond_angles")){
		w.set_weight( KIN_1D_CST, 1.0 )  ; // the torsion/bond/angle tether
		set_all_bond_angle_allow_move( pose );
	}

	if (truefalseoption("vary_all_bond_distances")){
		w.set_weight( KIN_1D_CST, 1.0 )  ; // the torsion/bond/angle tether
		w.set_weight( ATOMPAIR_CST, 1.0 ); // the ring constraint tether
		set_all_bond_distance_allow_move( pose );
	}


	if (truefalseoption("colin")){
		Pose pose_with_desired_geometry;
		create_totally_new_ideal_pose(pose_with_desired_geometry, pose);
		w.set_weight( KIN_1D_CST, 1.0 )  ; // the torsion/bond/angle tether
		w.set_weight( KIN_3D_CST, 1.0 )  ; // the torsion/bond/angle tether
		set_all_bond_angle_allow_move_CHARMM( pose, pose_with_desired_geometry );
	}


	pose.score(w);
	pose.dump_pdb("before.pdb");
	std::cout << 	pose.show_scores() << std::endl;
	pose_setup_minimizer( 0.001 );
	pose_set_use_nblist( false );

	pose.main_minimize( w, "dfpmin" );

	pose.dump_pdb("after.pdb");
	std::cout << 	pose.show_scores() << std::endl;
	exit(0);
}

///////////////////////////////////////////////////////////////////////////////
void
pose_relax_wrapper(
	int const lj_ramp_cycles,
	int const cycles,
	bool const vary_sidechain_bond_angles
)
{
	using namespace pose_ns;

	bool const start_pose_flag( pose_flag() );
	set_pose_flag( true );

	Pose pose;
	pose_from_misc( pose, get_fullatom_flag(), false/*idl strx*/, true );

	pose.set_allow_bb_move( true );
	int const nres( pose.total_residue() );
	FArray1D_bool chainbreaks( nres, false );
	pose.identify_backbone_chainbreaks( chainbreaks );
	for ( int i = 1; i < nres; ++i ) {
		if ( chainbreaks(i) ) {
			pose.set_allow_bb_move(i,false);
			pose.set_allow_bb_move(i+1,false);
		}
	}
	pose.set_allow_chi_move( true );

	Score_weight_map weight_map( score12 );
	static float const filter1  = realafteroption("filter1" ,9999999999.);
	static float const filter1a = realafteroption("filter1a",9999999999.);
	static float const filter1b = realafteroption("filter1b",9999999999.);
	static float const filter2  = realafteroption("filter2" ,9999999999.);

	//psh include disulfide constraints
	bool const disulf_flag = disulfides::BOUNDARY::get_disulf_flag();
	if (disulf_flag){
		add_disulf_pose_constraints( pose );
		//turn on the weight_map terms that correspond to disulf_constraints
		weight_map.set_weight( KIN_3D_CST, 1.0 )  ; // the torsion/bond/angle tether
	  weight_map.set_weight( ATOMPAIR_CST, 1.0 ); // the ring constraint tether
	}
	//psh score weight_map to get the terms calculated
	pose.score(weight_map);

	fast_relax_pose( pose, weight_map, lj_ramp_cycles, cycles, "tmp", 1.0, vary_sidechain_bond_angles,
									 filter1, filter1a, filter1b, filter2);

	// save this pose for silent_output if necessary
	silent_out_save_decoy_pose( pose );

	pose.copy_to_misc();

	set_pose_flag( start_pose_flag );

	if ( !start_pose_flag ) monte_carlo_reset();
}


///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void
read_decoys_silent_or_not( silent_io::Silent_file_data * & decoys_p, std::vector< std::string> & files, bool & silent_input )
{
  using namespace silent_io;

	static bool const silent_input_local = truefalseoption( "silent" );
	static bool const fullatom = truefalseoption( "fa_input" );
	silent_input = silent_input_local;

  if (silent_input){
  // read silent file
    decoys_p = new Silent_file_data( stringafteroption("s"), fullatom );
    files = decoys_p->tags();
  } else {
		if (truefalseoption("s")) {
			files.push_back( stringafteroption("s") );
		}

		if (truefalseoption("l")) {
			std::ifstream data( stringafteroption("l").c_str() );
			std::string line;
			while ( getline( data,line ) ) {
				files.push_back(line);
			}
			data.close();
		}

		decoys_p = NULL; //silent file data not used.
	}

}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void
fill_decoy_silent_or_not( pose_ns::Pose & pose, silent_io::Silent_file_data * decoys_p,
													std::string const start_file, bool const silent_input) {

  using namespace silent_io;

	static bool const fullatom = truefalseoption( "fa_input" );

	if ( silent_input ){
		Silent_structure const & decoy( decoys_p->get_structure( start_file ) );
		decoy.fill_pose( pose, true );
	} else {
		pose_from_pdb( pose, start_file, fullatom, false /*ideal_pose*/, true /*read_all_chains*/ );
	}

}



//////////////////////////////////////////////////////////////////
//
// provides alternative route to pose_relax via -pose1 and enables
// proper output of structures even if bond angles/length get modified.
//
// invoke this with -pose1 -pose_relax
void
pose1_relax_main(
)
{

	std::cout << "=== POSE1 RELAX MAIN =============================== \n";

	using namespace silent_io;
	using namespace pose_ns;
	using namespace kin;
	using namespace param_aa;

  const bool fullatom( truefalseoption("fa_input") );

	Pose start_pose;

	initialize_query(); // Reads in bbdep, and all the stuff we need for full atom relax.
	initialize_maps();
	score_set_vary_omega( true );

	// defines length, sequence of start_pose
	initialize_query_pose( start_pose );

	// read native
	bool native_exists;
	Pose native_pose;
	FArray1D_int native_mapping; // from pose to native
	setup_homolog_native( start_pose, native_exists,native_pose,native_mapping );
	if (native_exists){
		pose_to_native(native_pose);
		start_pose.set_native_pose();
	}

	// read fragments -- after reading native in case MAX_RES increases!
	read_fragments( start_pose.total_residue() );
	//fragment_diversity(); // needed for crankshaft?

	// open silent-output object
	Silent_out out( files_paths_pdb_out_prefix_nochain()+"_relax.out" ); //The most common name of an outfile....

  Silent_file_data * decoys_p;
	std::vector< std::string > tag_list;
  bool silent_input;

	read_decoys_silent_or_not( decoys_p, tag_list, silent_input );

	files_paths::use_status = truefalseoption("status");

	int istruct,nstruct = intafteroption("nstruct");
  for ( std::vector< std::string >::const_iterator it=tag_list.begin(),
					it_end = tag_list.end(); it != it_end; ++it ) {
		for ( istruct=0; istruct < nstruct; istruct++ ){
			std::string const & tag( *it );
			std::string decoy_name( tag + "_" + right_string_of(istruct,5,'0') );
			// already started?
			if ( !out.start_decoy( decoy_name ) ) {
				std::cout << "Decoy already processed: " << decoy_name << std::endl;
				continue;
			}

			std::cout << "\n\n==========================================================\n";
			std::cout << "Starting work on structure:  " << decoy_name << std::endl;

			Pose pose;

			fill_decoy_silent_or_not( pose, decoys_p, tag, silent_input );
			pose.copy_to_misc(); // need to set total_residue correctly, etc.

			pose.set_fullatom_flag( fullatom );

			int lj_ramp_cycles = intafteroption("lj_ramp_cycles",8) ;
			int const cycles_per_residue = { 1 };
			int cycles = static_cast<int> ( cycles_per_residue * pose.total_residue() * get_farlx_cycle_ratio() );
			bool const vary_sidechain_bond_angles = truefalseoption("vary_sidechain_bond_angles");

			pose.set_allow_bb_move( true );
			int const nres( pose.total_residue() );
			FArray1D_bool chainbreaks( nres, false );
			pose.identify_backbone_chainbreaks( chainbreaks );
			for ( int i = 1; i < nres; ++i ) {
				if ( chainbreaks(i) ) {
					pose.set_allow_bb_move(i,false);
					pose.set_allow_bb_move(i+1,false);
				}
			}
			pose.set_allow_chi_move( true );

 			Score_weight_map weight_map( score12 );

			int starttime = time(NULL);
			fast_relax_pose( pose, weight_map, lj_ramp_cycles, cycles, "tmp", 1.0, vary_sidechain_bond_angles);
			int endtime = time(NULL);
 			std::cout << "Pose Relax Time:  " <<  endtime - starttime << " s\n";

 			// output
			if ( native_exists ) {
				calc_homolog_rms( pose, native_pose, native_mapping );
			}

			score_set_evaluate_all_terms(true);
 			pose.score( score12 ); // output-score
 			out.write( decoy_name, pose );
 		}
	}



}












/////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
void
pose_rhiju(){
	using namespace silent_io;
	using namespace pose_ns;
	using namespace kin;
	using namespace param_aa;

	Pose start_pose;

	initialize_query(); // Reads in bbdep, and all the stuff we need for full atom relax.
	initialize_maps();

	// defines length, sequence of start_pose
	initialize_query_pose( start_pose );

	// read native
	bool native_exists;
	Pose native_pose;
	FArray1D_int native_mapping; // from pose to native
	setup_homolog_native( start_pose, native_exists,native_pose,native_mapping );
	if (native_exists){
		pose_to_native(native_pose);
		start_pose.set_native_pose();
	}

	// read fragments -- after reading native in case MAX_RES increases!
	read_fragments( start_pose.total_residue() );
	//fragment_diversity(); // needed for crankshaft?

	Silent_file_data decoys( stringafteroption("s") );

	// open silent-output object
	Silent_out out( files_paths_pdb_out_prefix_nochain()+"_relax.out" ); //The most common name of an outfile....

	for ( Silent_file_data::const_iterator it = decoys.begin(),
					it_end = decoys.end(); it != it_end; ++it) {
		std::string decoy_name( it->first + "_relax" );

		std::cout << "POSE_RHIJU getting started on: " << decoy_name << std::endl;

		// already started?
		if ( !out.start_decoy( decoy_name ) ) {
			std::cout << "decoy exists: " << decoy_name << std::endl;
			continue;
		}

		// copy from silent-data
		Pose pose;
		it->second->fill_pose( pose );
		pose.set_fullatom_flag( true );

	//Rhiju Hack.
	// Put jumps into protein L by hand.
		if (truefalseoption("1hz6_pose_relax")){
			Fold_tree f = pose.fold_tree();
			f.new_jump(46, 57, 51);
			f.new_jump( 7, 15, 11);
			f.new_jump( 8, 58, 42);
			pose.set_fold_tree( f );
			std::cout << "New FOLD TREEL: " << pose.fold_tree() << std::endl;
		}

		int lj_ramp_cycles = 8;
		int const cycles_per_residue = { 1 };
		int cycles = static_cast<int> ( cycles_per_residue * pose.total_residue() * get_farlx_cycle_ratio() );
		if (truefalseoption("fast")){
			lj_ramp_cycles = 1;
			cycles = 1;
		}
		bool const vary_sidechain_bond_angles = truefalseoption("vary_sidechain_bond_angles");
		Score_weight_map weight_map( score12 );
		fast_relax_pose( pose, weight_map, lj_ramp_cycles, cycles, "tmp", 1.0, vary_sidechain_bond_angles);

		// output
		if ( native_exists ) {
			calc_homolog_rms( pose, native_pose, native_mapping );
		}

		score_set_evaluate_all_terms(true);
		pose.score( score12 ); // output-score

		out.write( decoy_name, pose );
		pose.dump_pdb( decoy_name+".pdb");
	}
}
///////////////////////////////////////////////////////////////////////////////
void
setup_omega_tether( pose_ns::Pose & pose ){
	using namespace tether;
	using namespace misc;

	setup_omega_tether();

	for ( int i = 1; i <= total_residue; ++i ) {
		if ( pose.is_cutpoint( i ) ){
			tether_angle_res_weight(i) = 0.0;
		}
	}

	for ( int i = 1; i < total_residue; ++i ) {
		if ( pose.pseudo( i+1 ) ){
			tether_angle_res_weight(i) = 0.0;
		}
	}

}


///////////////////////////////////////////////////////////////////////////////
void
setup_backbone_pairwise_tethers( pose_ns::Pose & pose, pose_ns::Score_weight_map & weight_map  ){
	using namespace tether;
	using namespace misc;

	pose.copy_to_misc();

	std::cout << "Setting up pairwise tethers! " << std::endl;

	tether_setup();

	static int const pose_relax_tether_window_size = intafteroption( "pose_relax_tether_window_size", 5 );
	tether_set_window_size( pose_relax_tether_window_size );

	float const pose_relax_tether_weight = realafteroption( "pose_relax_tether_weight", 1.0 );
	scorefxn_set_tether_weight( pose_relax_tether_weight );
	weight_map.set_weight( pose_ns::TETHER, pose_relax_tether_weight );

	for ( int i = 1; i <= total_residue; ++i ) {
		tether_res_weight(i) = 1.0;
	}

}

///////////////////////////////////////////////////////////////////////////////
void
turn_off_backbone_pairwise_tethers( pose_ns::Score_weight_map & weight_map ){
	using namespace tether;

	std::cout << "Turning off pairwise tethers! " << std::endl;

	scorefxn_set_tether_weight( 0.0 );
	weight_map.set_weight( pose_ns::TETHER, 0.0 );
	tether_res_weight = 0.0;

}

///////////////////////////////////////////////////////////////////////////////
void
pose_relax_fullatom(
	pose_ns::Pose & pose
)
{
	using namespace pose_ns;

	bool const start_pose_flag( pose_flag() );
	set_pose_flag( true );

	int const lj_ramp_cycles(7);
	int const cycles(pose.total_residue());
	bool const vary_sidechain_bond_angles(false);

	pose.set_allow_bb_move( true );
	int const nres( pose.total_residue() );
	FArray1D_bool chainbreaks( nres, false );
	pose.identify_backbone_chainbreaks( chainbreaks );
	for ( int i = 1; i < nres; ++i ) {
		if ( chainbreaks(i) ) {
			pose.set_allow_bb_move(i,false);
			pose.set_allow_bb_move(i+1,false);
		}
	}
	pose.set_allow_chi_move( true );

	pose.set_fullatom_flag( true, false ); //repack

	Score_weight_map weight_map( score12 );
	static float const filter1  = realafteroption("filter1" ,9999999999.);
	static float const filter1a = realafteroption("filter1a",9999999999.);
	static float const filter1b = realafteroption("filter1b",9999999999.);
	static float const filter2  = realafteroption("filter2" ,9999999999.);

	//psh include disulfide constraints
	bool const disulf_flag = disulfides::BOUNDARY::get_disulf_flag();
	if (disulf_flag){
		add_disulf_pose_constraints( pose );
		//turn on the weight_map terms that correspond to disulf_constraints
		weight_map.set_weight( KIN_3D_CST, 1.0 )  ; // the torsion/bond/angle tether
	  weight_map.set_weight( ATOMPAIR_CST, 1.0 ); // the ring constraint tether
	}
	//psh score weight_map to get the terms calculated
	pose.score(weight_map);

	fast_relax_pose( pose, weight_map, lj_ramp_cycles, cycles, "tmp", 1.0, vary_sidechain_bond_angles,
									 filter1, filter1a, filter1b, filter2);

	// save this pose for silent_output if necessary
	silent_out_save_decoy_pose( pose );

	pose.copy_to_misc();

	set_pose_flag( start_pose_flag );

	if ( !start_pose_flag ) monte_carlo_reset();
}

//////////////////////////////////////////////////////////////////////////////

void
get_symm_jump_moves_during_relax(
	bool & symm_jump_moves_during_relax,
	float & transmag,
	float & rotmag
)
{
	static bool symm_jump_moves_during_relax_local( false );
	static float transmag_local ( 1.0 );
	static float rotmag_local ( 1.0 );
	static bool init( false );


	if ( !init ) {
		symm_jump_moves_during_relax_local = truefalseoption ( "pose_symm_jump_moves_during_relax");
		if( symm_jump_moves_during_relax_local ){
			realafteroption( "transmag", 1.0, transmag_local );
			realafteroption( "rotmag", 1.0, rotmag_local );
		}
		init = true;
	}

	symm_jump_moves_during_relax = symm_jump_moves_during_relax_local;
	transmag = transmag_local;
	rotmag = rotmag_local ;
}

////////////////////////////////////////////////////////////////////////////////////
// To run wobble and crank, the "segment_map" needs to be filled so
// that refold can work. Its a little silly to be using misc refold
// when we're trying to relax a pose with a crazy fold tree, but
// no one has taken the time to rewrite wobble and crank to
// work on poses...
//
void
initialize_segment_map_from_pose( pose_ns::Pose & pose )
{

	//Figure out what the segment boundaries are.
	int const nres = pose.total_residue();
	std::vector<int> segment_begin_list, segment_end_list;

	segment_begin_list.push_back( 1 );
	bool prev_allow_move = pose.get_allow_bb_move( 1 );

	for (int i = 1; i <= nres; i++ ){

		if (pose.get_allow_bb_move(i) != prev_allow_move) {
			segment_end_list.push_back( i-1 );
			segment_begin_list.push_back( i );
		}

		prev_allow_move = pose.get_allow_bb_move( i );

	}

	segment_end_list.push_back( nres );

	assert( segment_end_list.size() == segment_begin_list.size() ) ;

	//Fill FArrays with segment information
	int const total_domains = segment_end_list.size();
	FArray1D_int seg_begin( total_domains );
	FArray1D_int seg_end  ( total_domains );
	for (int i = 1; i <= total_domains; i++ ) {
		seg_begin( i ) = segment_begin_list[ i-1 ];
		seg_end  ( i ) = segment_end_list  [ i-1 ];
		//		std::cout << "SEGMENT MAP " << i << "  :  " << seg_begin(i) << " to " << seg_end( i )  << std::endl;
	}


	//Then put that stuff in "segment_map", which is used during wobble/crank in pose_relax.
	refold_set_segments( 1, total_domains, seg_begin, seg_end,
											 FArray1D_int( total_domains, 0 ) );  // automatic direction selection

}



