// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.1.2.1 $
//  $Date: 2005/11/07 21:05:35 $
//  $Author: pbradley $


// Rosetta Headers
#include "pose_rna_fragments_classes.h"
#include "pose.h"
#include "pose_rna_ns.h"
#include "random_numbers.h"
#include "read_paths.h" //For rosetta_database file read in.

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/StaticIndexRange.hh>
#include <ObjexxFCL/formatted.i.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/constants.hh>
#include <numeric/xyz.functions.hh>
#include <numeric/xyzVector.hh>

// C++ Headers
#include <cstdlib>
#include <iostream>
#include <sstream>
#include <fstream>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/file/file_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>

namespace rna_fragments_class{

	int const NUM_RNA_TORSIONS = 9;
	int const CHI_TORSION = 7;
	int const MAX_VALL_SIZE = 3000;

	/////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////
	TorsionSet::TorsionSet( int const size ){
		torsions.dimension( NUM_RNA_TORSIONS, SRange(0, size) );
		torsion_source_name.dimension( SRange(0, size), std::string( 4, ' ' )  );
		size_ = size;
	}

//////////////////////////////////////////////////////////////////////
	TorsionSet &
	TorsionSet::operator =(
			TorsionSet const & src
	){
		size_ = src.size_;

		for (int offset = 0; offset < size_; offset++){
			for (int j = 1; j <= NUM_RNA_TORSIONS; j++ ){
				torsions( j, offset) = src.torsions( j, offset);
			}
			torsion_source_name( offset ) = src.torsion_source_name( offset );
		}

		return *this;
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////
	// I really wish that this function wasn't necessary.
	// For historical reasons, the default atom tree torsion that controls the glycosydic
	// angle is not equal to chi. What's worse, though, is that its 180 degrees different for
	// purines than for pyrimidines. Its because the default atom tree trajectory depends
	// on the numbering schemes for the bases, which are opposite for purine vs. pyrimidine.
	void
	TorsionSet::flip_chi_purines( std::string const sequence ){

		assert( static_cast<int> ( sequence.length() )  == size_ );

		for (int i = 0; i < size_; i++){

			if 	( compare_RNA_char( sequence[i], 'r' ) ) {

				if (torsions( CHI_TORSION, i) < 0.0) torsions( CHI_TORSION, i ) += 360.0;
				//This is the flip:
				torsions( CHI_TORSION, i) -= 180.0;

			}
		}

	}


	/////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////
	float FragmentLibrary::get_fragment_torsion( int const num_torsion,  int const which_frag, int const offset ){
		return align_torsions_[ which_frag - 1 ].torsions( num_torsion, offset) ;
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////
	TorsionSet const FragmentLibrary::get_fragment_torsion_set( int const which_frag ){
		return align_torsions_[ which_frag - 1 ];
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////
	void  FragmentLibrary::add_torsion( TorsionSet const torsion_set ){
		align_torsions_.push_back( torsion_set );
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////
	void  FragmentLibrary::add_torsion(
																		 FArray2D_float & source_array,
																		 int const position,
																		 int const size,
																		 FArray1D_string & source_name ){
		TorsionSet torsion_set( size );

		for (int offset = 0; offset < size; offset++){
			for (int j = 1; j <= NUM_RNA_TORSIONS; j++ ){
				torsion_set.torsions( j, offset) = source_array( j, position+offset);
			}
			torsion_set.torsion_source_name( offset ) = source_name( position+offset );
		}

		align_torsions_.push_back( torsion_set );
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////
	void  FragmentLibrary::add_torsion( FArray2D_float & source_array, int const position, int const size,
																			bool flip_the_chi_purines, std::string const RNA_sequence,
																			FArray1D_string & source_name ){
		TorsionSet torsion_set( size );

		for (int offset = 0; offset < size; offset++){
			for (int j = 1; j <= NUM_RNA_TORSIONS; j++ ){
				torsion_set.torsions( j, offset) = source_array( j, position+offset);
			}
			torsion_set.torsion_source_name( offset ) = source_name( position+offset );
		}

		if (flip_the_chi_purines) torsion_set.flip_chi_purines( RNA_sequence );

		align_torsions_.push_back( torsion_set );
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////
	int FragmentLibrary::get_align_depth() {
		return align_torsions_.size();
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////
	//Constructor -- needs vall_torsions_file to get started.
	//	RNA_Fragments::RNA_Fragments( std::string const filename ){
	//		//Set up all the vall stuff.
	//		read_vall_torsions( filename );
	//	}
	//////////////////////////////////////////////////////////////////////////////
	void
	RNA_Fragments::pick_random_fragment(
			 TorsionSet & torsion_set,
			 std::string const RNA_string,
			 int const type /* = MATCH_YR */){

		std::string const RNA_string_local = convert_based_on_match_type( RNA_string, type );

		if (! fragment_library_pointer_map.count( RNA_string_local ) ){
			pick_fragment_library( RNA_string_local );
		}

		FragmentLibrary * fragment_library_pointer = fragment_library_pointer_map[ RNA_string_local ];

		int const num_frags = fragment_library_pointer->get_align_depth();

		if (num_frags == 0) { //trouble.
			std::cout << "Fragment Library: zero fragments found for " << RNA_string_local << std::endl;
			std::cerr << "Fragment Library: zero fragments found for " << RNA_string_local << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		int const which_frag = static_cast <int> ( ran3() * num_frags) + 1;

		torsion_set = fragment_library_pointer->get_fragment_torsion_set( which_frag );

		//Oh, I *hate* doing this...
		torsion_set.flip_chi_purines( RNA_string );

	}

	/////////////////////////////////////////////////////////////////////////////////////////////////
	void
	RNA_Fragments::pick_random_fragment(
			 TorsionSet & torsion_set,
			 pose_ns::Pose & pose,
			 int const position,
			 int const size,
			 int const type /* = MATCH_YR */){

		std::string const RNA_sequence( pose.sequence() );
		std::string const RNA_string = RNA_sequence.substr( position - 1, size );

		pick_random_fragment( torsion_set, RNA_string, type );

	}
	/////////////////////////////////////////////////////////////////////////////////////////////////
	void
	RNA_Fragments::pick_fragment_library( std::string const RNA_string ){

		//Don't worry, I coded a destructor in later.
		FragmentLibrary * fragment_library_p;
		fragment_library_p = new FragmentLibrary;

		int const size = RNA_string.length();

		std::string vall_current_sequence = RNA_string;

		for (int i = 1; i <= vall_size_ - size + 1; i++ ){

			bool match( true );

			/////////////////////////////////////
			//replace with sequence matcher!
			/////////////////////////////////////

			for (int offset = 0; offset < size; offset++ ){
				vall_current_sequence[offset] = vall_sequence_( i + offset );
				if ( vall_is_chainbreak_( i + offset ) ||
						 !compare_RNA_char( vall_current_sequence[offset], RNA_string[ offset ] ) )
					{
						match = false;
						break;
					}
			}

			if (match) {
				fragment_library_p->add_torsion( vall_torsions_, i, size,
																				 true /*flip_chi_purines*/,  vall_current_sequence,
																				 vall_name_ );
			}

		}


		std::cout << "Picked Fragment Library for sequence " << RNA_string << " ... found " <<
			fragment_library_p->get_align_depth() << " potential fragments" << std::endl;


		fragment_library_pointer_map[ RNA_string ] = fragment_library_p;

	}


	/////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////
	//random helper functions. Not very elegant, actually lots of copied code.
	/////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////
	void
	RNA_Fragments::get_fragment(
			 TorsionSet & torsion_set,
			 std::string const RNA_string,
			 int const type,
			 int const which_frag ){

		std::string const RNA_string_local = convert_based_on_match_type( RNA_string, type );

		if (! fragment_library_pointer_map.count( RNA_string_local ) ){
			pick_fragment_library( RNA_string_local );
		}

		FragmentLibrary * fragment_library_pointer = fragment_library_pointer_map[ RNA_string_local ];

		int const num_frags = fragment_library_pointer->get_align_depth();

		if (num_frags == 0) { //trouble.
			std::cout << "Fragment Library: zero fragments found for " << RNA_string_local << std::endl;
			std::cerr << "Fragment Library: zero fragments found for " << RNA_string_local << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		torsion_set = fragment_library_pointer->get_fragment_torsion_set( which_frag );

		//Oh, I *hate* doing this...
		torsion_set.flip_chi_purines( RNA_string );

	}
	/////////////////////////////////////////////////////////////////////////////////////////////////
	void
	RNA_Fragments::get_fragment(
			 TorsionSet & torsion_set,
			 pose_ns::Pose & pose,
			 int const position,
			 int const size,
			 int const type,
       int const which_frag ){

		std::string const RNA_sequence( pose.sequence() );
		std::string const RNA_string = RNA_sequence.substr( position - 1, size );

		get_fragment( torsion_set, RNA_string, type, which_frag );

	}

	//////////////////////////////////////////////////////////////////////////////
	int
	RNA_Fragments::get_align_depth(
			 std::string const RNA_string,
			 int const type /* = MATCH_YR */){

		std::string const RNA_string_local = convert_based_on_match_type( RNA_string, type );

		if (! fragment_library_pointer_map.count( RNA_string_local ) ){
			pick_fragment_library( RNA_string_local );
		}

		FragmentLibrary * fragment_library_pointer = fragment_library_pointer_map[ RNA_string_local ];

		int const num_frags = fragment_library_pointer->get_align_depth();

		return num_frags;

	}

	//////////////////////////////////////////////////////////////////////////////
	int
	RNA_Fragments::get_align_depth(
			 pose_ns::Pose & pose,
			 int const position,
			 int const size,
			 int const type /* = MATCH_YR */){

		std::string const RNA_sequence( pose.sequence() );
		std::string const RNA_string = RNA_sequence.substr( position - 1, size );

		return get_align_depth( RNA_string, type );
	}


	/////////////////////////////////////////////////////////////////////////////////////////////////
	void
	RNA_Fragments::read_vall_torsions( std::string const filename ){

		//Just read in this file once.
		static bool init ( false );
		if (init) return;
		init = true;

		vall_torsions_.dimension ( SRange(0, NUM_RNA_TORSIONS), MAX_VALL_SIZE );
		vall_sequence_.dimension ( MAX_VALL_SIZE );
		vall_is_chainbreak_.dimension ( MAX_VALL_SIZE );
		vall_edge_is_base_pairing_.dimension( MAX_VALL_SIZE, rna_scoring::NUM_EDGES );
		vall_name_.dimension( MAX_VALL_SIZE );

		std::cout << "Reading in vall_torsions file: " <<  filename << std::endl;

		//This will check in rosetta_database first.
		utility::io::izstream & vall_in( open_data_file( filename ) );
		if ( vall_in.fail() ){
			std::cout << "Bad vall torsions file? " << filename << std::endl;
			std::cerr << "Bad vall torsions file? " << filename << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}


		std::string line, tag;

		int count( 0 );
		while (  getline( vall_in, line) ){

			std::istringstream line_stream( line );

			count++;

			line_stream >> vall_sequence_( count );

			for (int i = 1; i <= NUM_RNA_TORSIONS; i++ ) {
				line_stream >> vall_torsions_( i, count );
			}

			line_stream >> vall_is_chainbreak_( count );

			for (int i = 1; i <= rna_scoring::NUM_EDGES; i++ ) {
				line_stream >> vall_edge_is_base_pairing_( count, i );
			}

			//In principle could look for one more string in the vall
			// torsions file as a "name", but for now just keep track
			// of line number.
			vall_name_( count ) = I( 4, count );

		} // line_stream

		vall_size_ = count;

		vall_in.close();

		std::cout << "Lines read from vall_torsions file: " << vall_size_ << std::endl;

	}


	/////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////
	std::string const convert_based_on_match_type( std::string const RNA_string, int const type ){

		std::string RNA_string_local = RNA_string;

		int const size = RNA_string.length();

		//Obey orders to match exactly, match pyrimidine/purine, or match all.
		if (type == MATCH_ALL){

			for (int i = 0; i < size; i++) 	RNA_string_local[ i ] = 'n';

		} else if ( type == MATCH_YR ) {

			for (int i = 0; i < size; i++) {
				if (RNA_string[ i ] == 'g' || RNA_string[ i ] == 'a' ){
					RNA_string_local[ i ] = 'r';
				} else {
					assert( RNA_string[ i ] == 'u' || RNA_string[ i ] == 'c' );
					RNA_string_local[ i ] = 'y';
				}
			}

		}

		return RNA_string_local;
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////
	bool
	compare_RNA_char( char const char1, char const char2 ) {
		//Man this is silly, there must be a more elegant way to do this.
		if (char1 == char2) return true;
		if (char1 == 'n' || char2 == 'n') return true;
		if (char1 == 'r' && (char2 == 'a' || char2 == 'g')) return true;
		if (char1 == 'y' && (char2 == 'c' || char2 == 'u')) return true;
		if (char2 == 'r' && (char1 == 'a' || char1 == 'g')) return true;
		if (char2 == 'y' && (char1 == 'c' || char1 == 'u')) return true;
		return false;
	}


}

