// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.1.2.1 $
//  $Date: 2005/11/07 21:05:35 $
//  $Author: pbradley $


// Rosetta Headers
#include "pose_rna_fullatom.h"
#include "pose_rna.h"
#include "pose_dna.h"
#include "pose_rna_ns.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "atom_tree_routines.h"
#include "current_pose.h"
#include "dna.h"
#include "dna_am_min.h"
#include "dna_ns.h"
#include "files_paths.h"
#include "fragments_pose.h"
#include "force_barcode.h"
#include "fullatom_energies.h"
#include "fullatom_energy.h"
#include "fullatom_sasa.h"
#include "hbonds.h"
#include "kin_stub.h"
#include "minimize.h"
#include "misc.h"
#include "nblist.h"
#include "param.h"
#include "param_aa.h"
#include "pose.h"
#include "pose_io.h"
#include "pose_rna_fragments.h"
#include "pose_rna_pdbstats.h"
#include "read_aaproperties.h"
#include "random_numbers.h"
#include "score.h"
#include "score_ns.h"
#include "structure.h"
#include "silent_input.h"

#include "prof.h"

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/StaticIndexRange.hh>
#include <ObjexxFCL/string.functions.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/conversions.hh>
#include <numeric/trig.functions.hh>
#include <numeric/xyz.functions.hh>
#include <numeric/xyz.io.hh>
#include <numeric/xyzVector.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <sstream>
#include <fstream>


///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
pose_ns::Score_weight_map setup_rna_fullatom_weight_map(){
	using namespace pose_ns;

	static Score_weight_map w;

	static bool init = false;

	if (init) return w;

	w.set_weight( FA_ATR, realafteroption("fa_atr_weight",1.0));
	w.set_weight( FA_REP, realafteroption("fa_rep_weight",1.0));
	w.set_weight( FA_SOL, realafteroption("fa_sol_weight",1.0));

	w.set_weight( FA_ELEC, realafteroption("fa_elec_weight",1.0));

	w.set_weight( KIN_1D_CST, realafteroption("kin_1d_weight", 1.0) )  ; // the torsion/bond/angle tether

	// the ring constraint tethers
	w.set_weight( KIN_3D_CST, 1.0 );
	w.set_weight( ATOMPAIR_CST, 1.0 );

	set_use_W_hb_env_dep_tk( false );
	set_use_W_hb_env_dep_lin( false );

	w.set_weight( HB_SC,   realafteroption("hb_sc_weight",1.0) );
	w.set_weight( HB_SRBB, realafteroption("hb_srbb_weight",1.0));
	w.set_weight( HB_LRBB, realafteroption("hb_lrbb_weight",1.0));

	minimize_set_tolerance( realafteroption("tolerance", 0.0000025) );
	//	minimize_set_tolerance( 0.00025 );

	pose_set_use_nblist( truefalseoption("use_nblist") );

//Tell count_pair.h and hbonds.cc to go ahead and count RNA-RNA interactions.
	rna_scoring::rna_fullatom = true;
	rna_scoring::count_pair_ignore_neighbor_backbone = truefalseoption( "count_pair_ignore_neighbor_backbone" );

	init = true;

	return w;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
//
// This potential keeps the various RNA torsions reasonable.
// Several notes:
//  1. This probably shouldn't be set up as a Kin-1D torsion constraint -- its
//      getting added to ring constraints, and that's confusing! But this was
//      very quick to implement. The Kin-1D constraints are *harmonic*; I am
//      assuming that we are minimizing to optmize base pairs and not to
//      switch sugar puckers or glycosidic torsion angles or anything crazy.
//  2. Big issue -- the torsions alpha, beta, gamma, etc. are offset from
//      conventional values. WATCH OUT!
//  3. Fits were carried out to 1ffk torsions in MATLAB. In some cases I fit a single "peak"
//      to the the sum of two Gaussians, and the sharpest Gaussian was assumed
//      to be due to regular A-form helical structure; I am conservatively using
//      the broader width below.
//  4. I made NO ATTEMPT to parametrize the well-depths of each harmonic well for
//      e.g., the trimodal alpha, according to, e.g.,
//      quantum mechanical calculations or observed distributions.
//  5. This function might seem overcomplex because of an attempt to reproduce correlations
//      in delta/chi/epsilon torsions and in zeta(i)/alpha(i+1) torsions.
//      Note again that I'm not trying to get well-depths right -- only the positions
//      of well minima. [The principle here is that geometry is always important, while
//      energy depths are maybe less important due to cooperativity.]
//  6. The constraints on nu2 and nu3 aren't really done in a totally kosher manner,
//      because the covalent ring closure of the sugar enforce correlations of
//      nu2, nu3, and delta... I don't think this constraint will do much anyway, though.
//
//
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
class Gaussian_parameter {
public:
	float amplitude, center, width;

	Gaussian_parameter ( float const amplitude_in, float const center_in, float const width_in ):
		amplitude( amplitude_in ),
		center   ( center_in ),
		width    ( width_in )
	{}
};

typedef std::vector< Gaussian_parameter > Gaussian_parameter_set;

namespace rna_torsion_gaussian_parameters{
	Gaussian_parameter_set gaussian_parameter_set_alpha;
	Gaussian_parameter_set gaussian_parameter_set_beta;
	Gaussian_parameter_set gaussian_parameter_set_gamma;
	Gaussian_parameter_set gaussian_parameter_set_delta_north;
	Gaussian_parameter_set gaussian_parameter_set_delta_south;
	Gaussian_parameter_set gaussian_parameter_set_epsilon_north;
	Gaussian_parameter_set gaussian_parameter_set_epsilon_south;
	Gaussian_parameter_set gaussian_parameter_set_zeta_alpha_sc_minus;
	Gaussian_parameter_set gaussian_parameter_set_zeta_alpha_sc_plus;
	Gaussian_parameter_set gaussian_parameter_set_zeta_alpha_ap;
	Gaussian_parameter_set gaussian_parameter_set_chi_north;
	Gaussian_parameter_set gaussian_parameter_set_chi_south;
	Gaussian_parameter_set gaussian_parameter_set_nu2_north;
	Gaussian_parameter_set gaussian_parameter_set_nu2_south;
	Gaussian_parameter_set gaussian_parameter_set_nu3_north;
	Gaussian_parameter_set gaussian_parameter_set_nu3_south;
}

///////////////////////////////////////////////////////////////////////////////
void
init_rna_torsion_gaussian_parameters()
{
	using namespace rna_torsion_gaussian_parameters;

	static bool init ( false );

	// These numbers refer to amplitude, mean, and width of fitted Gaussians.

	// "Tight torsion" means use the width of the major peak in the torsion
	// histogram to determine how strong the harmonic constraint will be. Otherwise, be less restrictive,
	// and use width of the minor fatter peak that lies under the sharp peak.
	static const bool rna_tight_torsions = truefalseoption( "rna_tight_torsions" );

	if ( rna_tight_torsions ) {
		gaussian_parameter_set_alpha.push_back( Gaussian_parameter(222.03, -64.11,  9.64) );
	} else {
		gaussian_parameter_set_alpha.push_back( Gaussian_parameter(13.12, -73.74,  35.97) );
	}
	gaussian_parameter_set_alpha.push_back( Gaussian_parameter(10.51, 66.01,  18.09) );
	gaussian_parameter_set_alpha.push_back( Gaussian_parameter(18.40, 161.80,  18.12) );

	if ( rna_tight_torsions ) {
		gaussian_parameter_set_beta.push_back( Gaussian_parameter(181.33, 176.33,  11.54) );
	} else {
		gaussian_parameter_set_beta.push_back( Gaussian_parameter(32.30, 174.52,  43.56) );
	}

	if ( rna_tight_torsions ) {
		gaussian_parameter_set_gamma.push_back( Gaussian_parameter(366.90, 53.08,  6.64) );
	} else {
		gaussian_parameter_set_gamma.push_back( Gaussian_parameter(18.26, 56.59,  20.57) );
	}
	gaussian_parameter_set_gamma.push_back( Gaussian_parameter(21.61, 178.19,  13.61) );
	gaussian_parameter_set_gamma.push_back( Gaussian_parameter(3.98, -64.02,  17.76) );

	gaussian_parameter_set_delta_north.push_back( Gaussian_parameter(687.92, 82.90,  3.99) );
	gaussian_parameter_set_delta_south.push_back( Gaussian_parameter(53.18, 145.25,  6.35) );

	gaussian_parameter_set_epsilon_north.push_back( Gaussian_parameter(178.08, -150.17,  14.64) );
	gaussian_parameter_set_epsilon_north.push_back( Gaussian_parameter(2.52, 68.28,  32.29) );

	gaussian_parameter_set_epsilon_south.push_back( Gaussian_parameter(11.95, -98.45, 26.80) );
	gaussian_parameter_set_epsilon_south.push_back( Gaussian_parameter( 0.58, 159.70, 103.86) );

	if ( rna_tight_torsions ) {
		gaussian_parameter_set_zeta_alpha_sc_minus.push_back( Gaussian_parameter( 143.97, -71.45, 7.91) );
	} else {
		gaussian_parameter_set_zeta_alpha_sc_minus.push_back( Gaussian_parameter( 78.74, -68.60, 16.19) );
	}
	gaussian_parameter_set_zeta_alpha_sc_minus.push_back( Gaussian_parameter(  2.43, 178.84, 114.82) );

	gaussian_parameter_set_zeta_alpha_sc_plus.push_back( Gaussian_parameter(2.08, -137.28,  63.12) );
	gaussian_parameter_set_zeta_alpha_sc_plus.push_back( Gaussian_parameter(3.37, 87.07,  32.69) );

	gaussian_parameter_set_zeta_alpha_ap.push_back( Gaussian_parameter(13.65, -69.74,  15.28) );
	gaussian_parameter_set_zeta_alpha_ap.push_back( Gaussian_parameter(2.69, 63.03,  33.61) );


	gaussian_parameter_set_chi_north.push_back( Gaussian_parameter(228.92, -100.57, 11.43) );
	gaussian_parameter_set_chi_north.push_back( Gaussian_parameter( 1.07, 129.24, 26.11) );

	gaussian_parameter_set_chi_south.push_back( Gaussian_parameter(12.53, -63.40, 25.23) );
	gaussian_parameter_set_chi_south.push_back( Gaussian_parameter( 0.64, 130.09, 28.10) );

	gaussian_parameter_set_nu2_north.push_back( Gaussian_parameter(1074.06, 35.80,  -2.60) );
	gaussian_parameter_set_nu2_south.push_back( Gaussian_parameter(134.05, -33.99,  2.55) );

	gaussian_parameter_set_nu3_north.push_back( Gaussian_parameter(631.56, 95.34,  4.20) );
	gaussian_parameter_set_nu3_south.push_back( Gaussian_parameter(57.04, 155.51,  6.00) );

	init = true;

}

bool check_RNA_torsion_chainbreak( pose_ns::Pose & pose, int const position_offset, int const which_torsion) {

	//obviously some torsions don't make sense for first and last residues.
	// There must be a more clever way to deal with this.
 	if (which_torsion == 1 && check_chainbreak( position_offset-1, pose )) return true;
	if (which_torsion == 5 && check_chainbreak( position_offset, pose )) return true;
	if (which_torsion == 6 && check_chainbreak( position_offset, pose )) return true;

	return false;

}


///////////////////////////////////////////////////////////////////////////////
void
add_RNA_torsion_constraint(
     pose_ns::Pose & pose,
		 int const i,
		 cst_set_ns::Cst_set & cst_set,
		 int const torsion_number,
		 Gaussian_parameter_set const gaussian_parameter_set
)
{
	using namespace kin;

	if (!check_RNA_torsion_insertable( pose, i, torsion_number )) return;
	if (check_RNA_torsion_chainbreak( pose, i, torsion_number )) return;

	// Get current value.
	float const current_torsion = pose.get_torsion_by_number( i, torsion_number );

	// If there are multiple harmonic tethers available choose the "closest" one.

	assert( gaussian_parameter_set.size() > 0 );
	float best_center( 0.0 ), best_weight( 0.0 ), best_sigma( 10000000.0 );

	for (unsigned int n = 0; n < gaussian_parameter_set.size(); n++ ){

		float center = gaussian_parameter_set[n].center;

		//Absolutely stupid. Definition of chi changes for purines due
		// to atom_tree silliness.
		if ( torsion_number == 7 /*CHI*/ &&
				 (param_aa::aa_name1(pose.res(i)) == 'a' ||
					param_aa::aa_name1(pose.res(i)) == 'g' ) ) {
			center += 180.0;
		}

		float const weight = 1.0/(gaussian_parameter_set[n].width * gaussian_parameter_set[n].width);
		float deviation = (current_torsion - center );

		//Copied from elsewhere -- take into account periodicity.
		float const x = 360.0;
		float const halfx = 180.0;
		deviation = ( ( deviation >= halfx || deviation < -halfx ) ?
						 mod( mod( deviation, x ) + ( x + halfx ), x ) - halfx :
						 deviation );

		float const sigma = deviation * deviation * weight;

		if (sigma < best_sigma ){
			best_center = center;
			best_weight = weight;
			best_sigma = sigma;
		}

	}

	//	std::cout << "TETHER_ENERGY: " << i << " " << torsion_number << " " << current_torsion <<
	//		" " << best_center  << " " << std::sqrt( best_sigma ) << std::endl;

	assert( best_weight > 0.0 );

	// Now add in the constraint.
	Atom_tree const * atom_tree = pose.atom_tree();

	const Atom* atom( atom_tree->get_rosetta_torsion_atom( torsion_number, i ) );

	Torsion_id this_bond_torsion( atom->atom_id, PHI );

	static float const deg2rad( numeric::conversions::radians( 1.0 ) );

	//Here's the deal. The "PHI" torsions stored inside the atom tree are not
	// the dihedrals but are off by some offset (confusing). This is an ugly hack
	// to figure out what that internal offset is...
	float const tricky_offset = atom_tree->torsion( this_bond_torsion ) /* radians */
		- deg2rad * pose.get_torsion_by_number( i, torsion_number ) /*degrees --> convert to radians*/;

	//	std::cout << "Tethering torsion number " << torsion_number << " of residue " << i << " to value: " << best_center << "  with weight " << best_weight << ".   Present value? " <<  pose.get_torsion_by_number( i, torsion_number ) << " " << atom_tree->torsion( this_bond_torsion) << std::endl;

	float const torsion_center = deg2rad * best_center + tricky_offset;

	static const float scale_rna_torsion_tether = realafteroption("scale_rna_torsion_tether", 0.05 );
	float const torsion_weight = scale_rna_torsion_tether * best_weight/ (deg2rad*deg2rad);

	cst_set.add_kin_torsion_constraint( this_bond_torsion, torsion_center, torsion_weight );

}

///////////////////////////////////////////////////////////////////////////////
void
add_o2star_torsion_constraint(
															pose_ns::Pose & pose,
															int const i,
															cst_set_ns::Cst_set & cst_set )
{

	using namespace kin;
	using namespace rna_variables;
	using namespace aaproperties_pack;

	// Just need to figure out atom numbers to define the torsion.
	// Might as well use C3'-C2'-O2'-HO2'
	// Make it a periodic constraint too.

	Atom_id const atom1( c3star, i );
	Atom_id const atom2( c2star, i );
	Atom_id const atom3( o2star, i );

	int ho2star;
	atom_num_from_atom_name( "2HO*", pose.res(i), pose.res_variant(i), ho2star );
	Atom_id const atom4( ho2star, i );


	float const theta0    =  numeric::conversions::radians( rna_scoring::o2star_torsion_offset );
	float const dummy  =  numeric::conversions::radians( 20.0 ); // not actually used!!
	float periodic_value = numeric::conversions::radians( 120.0 );

	//wild guess. In principle should be much larger, but
	// minimizer tends to over minimize dof's that are directly controlled by the atom tree!
	float weight = 2.7; //This determines how strong the periodic potential actually is.

	cst_set_ns::Torsion_cst torsion_cst( atom1, atom2, atom3, atom4,
													 theta0, dummy, periodic_value, weight,
													 "PERIODIC" );

	//Just to break degeneracy, add a very weak potential to favor one rotamer
	// a la Auffinger/Westhof. This is a very weak potential indeed.
	periodic_value = numeric::conversions::radians(360.0);
	weight = 0.1; //super weak.
	torsion_cst = cst_set_ns::Torsion_cst( atom1, atom2, atom3, atom4, theta0, dummy, periodic_value, weight, "PERIODIC");
	cst_set.add_atom_torsion_constraint( torsion_cst ); //wait does this overwrite the previous potential?

}

///////////////////////////////////////////////////////////////////////////////
void
add_RNA_torsion_constraints( pose_ns::Pose & pose ){

	using namespace cst_set_ns;
	using namespace kin;
	using namespace rna_torsion_gaussian_parameters;

  // Dynamically allocate a cst_set, which will be pointed to by a pose.
	// The cst_set will delete itself when the pose no longer points to it, but this
	// requires that we call "remove_ptr_reference()" once the pose pointer is setup (below)
	Cst_set * cst_set_p;
	cst_set_p = new Cst_set();

	int const total_residue = pose.total_residue();

	init_rna_torsion_gaussian_parameters();

	enum{ WHATEVER, ALPHA, BETA, GAMMA, DELTA, EPSILON, ZETA, CHI, NU2, NU3};

	float const DELTA_CUTOFF = 115.0;

	for (int i = 1; i <= total_residue; i++ ){

		// These are in order of "rosetta torsion numbers", as is traced
		// out by the atom tree.

		/////////////////////////////////////
		// alpha --  3 harmonic potentials
		/////////////////////////////////////

		if ( check_RNA_torsion_insertable_strict( pose, i, ALPHA ) ) {
			add_RNA_torsion_constraint( pose, i, *cst_set_p, ALPHA, gaussian_parameter_set_alpha);
		}

		/////////////////////////////////////
		// beta -- 1 harmonic potential
		/////////////////////////////////////
		add_RNA_torsion_constraint( pose, i, *cst_set_p, BETA, gaussian_parameter_set_beta);

		/////////////////////////////////////
		// gamma
		/////////////////////////////////////
		add_RNA_torsion_constraint( pose, i, *cst_set_p, GAMMA, gaussian_parameter_set_gamma);

		/////////////////////////////////////
		// delta
		/////////////////////////////////////
		float const delta = pose.get_torsion_by_number( i, DELTA );

		if (delta <= DELTA_CUTOFF) { // North, or 3'-endo sugar pucker, favored by RNA.
			add_RNA_torsion_constraint( pose, i, *cst_set_p, DELTA, gaussian_parameter_set_delta_north);
		} else { // South, or 2'-endo sugar pucker.
			add_RNA_torsion_constraint( pose, i, *cst_set_p, DELTA, gaussian_parameter_set_delta_south);
		}

		/////////////////////////////////////
		// epsilon
		/////////////////////////////////////

		if ( check_RNA_torsion_insertable_strict( pose, i, EPSILON ) ) {

			if (delta <= DELTA_CUTOFF) { // North, or 3'-endo sugar pucker, favored by RNA.
				add_RNA_torsion_constraint( pose, i, *cst_set_p, EPSILON, gaussian_parameter_set_epsilon_north);
			} else { // South, or 2'-endo sugar pucker.
				add_RNA_torsion_constraint( pose, i, *cst_set_p, EPSILON, gaussian_parameter_set_epsilon_south);
			}

		}

		/////////////////////////////////////
		// zeta
		/////////////////////////////////////

		if ( check_RNA_torsion_insertable_strict( pose, i, ZETA ) ) {

			float const next_alpha = pose.get_torsion_by_number( i+1, ALPHA );

			if ( next_alpha > -120.0 && next_alpha <= 0.0 ) { //default A-form, alpha sc-
				add_RNA_torsion_constraint( pose, i, *cst_set_p, ZETA, gaussian_parameter_set_zeta_alpha_sc_minus );
			} else if ( next_alpha > 0.0 && next_alpha < 100.0 ) { // alpha sc+
				add_RNA_torsion_constraint( pose, i, *cst_set_p, ZETA, gaussian_parameter_set_zeta_alpha_sc_plus );
			} else { // alpha ap
				add_RNA_torsion_constraint( pose, i, *cst_set_p, ZETA, gaussian_parameter_set_zeta_alpha_ap );
			}

		}


		/////////////////////////////////////
		// chi
		/////////////////////////////////////
		if (delta <= DELTA_CUTOFF) { // North, or 3'-endo sugar pucker, favored by RNA.
			add_RNA_torsion_constraint( pose, i, *cst_set_p, CHI, gaussian_parameter_set_chi_north);
		} else { // South, or 2'-endo sugar pucker.
			add_RNA_torsion_constraint( pose, i, *cst_set_p, CHI, gaussian_parameter_set_chi_south);
		}

		/////////////////////////////////////
		// nu2
		/////////////////////////////////////
		if (delta <= DELTA_CUTOFF) { // North, or 3'-endo sugar pucker, favored by RNA.
			add_RNA_torsion_constraint( pose, i, *cst_set_p, NU2, gaussian_parameter_set_nu2_north);
		} else { // South, or 2'-endo sugar pucker.
			add_RNA_torsion_constraint( pose, i, *cst_set_p, NU2, gaussian_parameter_set_nu2_south);
		}

		/////////////////////////////////////
		// nu3
		/////////////////////////////////////
		if (delta <= DELTA_CUTOFF) { // North, or 3'-endo sugar pucker, favored by RNA.
			add_RNA_torsion_constraint( pose, i, *cst_set_p, NU3, gaussian_parameter_set_nu3_north);
		} else { // South, or 2'-endo sugar pucker.
			add_RNA_torsion_constraint( pose, i, *cst_set_p, NU3, gaussian_parameter_set_nu3_south);
		}

		/////////////////////////////////////
		// 2'-OH torsional constraints...
		/////////////////////////////////////
		add_o2star_torsion_constraint( pose, i, *cst_set_p );

	}

	if (pose.constraints_exist()){
		cst_set_p -> collect( pose.constraints() );
	}

	pose.set_constraints( *cst_set_p );

	// this was allocated by "new" (above), and will cause a memory leak otherwise
	cst_set_p->remove_ptr_reference();

}

///////////////////////////////////////////////////////////////////////////////
void
setup_RNA_geometry_constraints( pose_ns::Pose & pose ){

	using namespace pose_ns;

	// Allow distances to move?
	static bool const vary_bond_geometry_RNA = truefalseoption("vary_bond_geometry_RNA" );

	//	pose.set_vary_bond_geometry_flag( true );

	if (vary_bond_geometry_RNA){

		//		Pose pose_with_ideal_geometry;
		//		initialize_query_pose_rna( pose_with_ideal_geometry, false /*use_fasta*/ );

		Pose & pose_with_ideal_geometry = pose;

		set_all_bond_distance_allow_move( pose, pose_with_ideal_geometry ); // allow bonds lengths to move.
		set_all_bond_angle_allow_move( pose, pose_with_ideal_geometry ); // allow bonds lengths to move.

	}

	//any ring constraints
	add_all_bond_constraints( pose );

	// alpha, beta, gamma, delta, epsilon, zeta, chi, nu3, nu2:
	add_RNA_torsion_constraints( pose );

	add_heavyatom_coordinate_constraints( pose ); // These will only be activated if COORD_CST weight > 0.0.

	//		score_the_pose( pose, rna_weight_map, start_file, out );
}


///////////////////////////////////////////////////////////////////////////////
// Following was just to make sure 2'-OH torsion was getting minimized.
void crazy_test( pose_ns::Pose & pose ){
	using namespace kin;

	Atom_tree const * atom_tree = pose.atom_tree(); //Need the atom tree to see where jumps are.

	int const nres( pose.total_residue() );
	for (int i = 1; i <= nres; ++i) {
		int const aa  = pose.res(i);
		int const aav = pose.res_variant(i);
		int const natoms( aaproperties_pack::natoms( aa, aav ) );

		for ( int j = 1; j <= natoms; ++j ) {

			Atom_id current_atom_id( j, i );

			bool const this_is_a_jump = atom_tree->atom( current_atom_id )->is_jump();

			if (!this_is_a_jump) {

				Torsion_id this_bond_torsion( current_atom_id, PHI );

				pose.set_allow_move( this_bond_torsion, false );

				if ( i==1 && j==30 ) pose.set_allow_move( this_bond_torsion, true );


			}
		}
	}

}

///////////////////////////////////////////////////////////////////////////////
// Move this to pose_rna_jumping or something?
// Decided that this was a bad idea -- it doesn't really make sense
// to have angle constraints when the chainbreak ends are really
// far apart.... better to take a pseudoatom approach (i.e.
// do it in minirosetta) or to try to do everything by
// distance constraints -- see jmp_chainbreak_score in jumping_loops.cc.
void
setup_rna_chainbreak_constraints( pose_ns::Pose & pose )
{
	using namespace cst_set_ns;
	using namespace kin;
	using namespace rna_variables;

  // Dynamically allocate a cst_set, which will be pointed to by a pose.
	// The cst_set will delete itself when the pose no longer points to it, but this
	// requires that we call "remove_ptr_reference()" once the pose pointer is setup (below)
	Cst_set * cst_set_p;
	cst_set_p = new Cst_set();

	int const total_residue = pose.total_residue();

	//Find chainbreaks -- if any!
	for ( int i = 1; i < total_residue; i++ ){

		if  ( !pose.is_cutpoint( i ) ) continue;

		std::cout << "Setting up chainbreak constraints at position " << i << std::endl;

		//Relevant atoms?
		Atom_id c4star_id     ( c4star, i );
		Atom_id c3star_id     ( c3star, i );
		Atom_id o3star_id     ( o3star, i );
		Atom_id next_p_id     ( p,      i+1);
		Atom_id next_o5star_id( o5star, i+1);
		Atom_id next_c5star_id( c5star, i+1);

		//One distance constraint
		float const ideal_distance = 1.61;
		float const distance_tether = 10.0;
		cst_set_p->add_atompair_constraint( o3star_id, next_p_id, ideal_distance,
																		 distance_tether );

		// Two angle constraints
		float const angle_tether = 10.0;

		float const c3star_o3star_p = numeric::conversions::radians( 119.8 );
		cst_set_p->add_atom_angle_constraint( c3star_id, o3star_id, next_p_id,
																					c3star_o3star_p, angle_tether );

		float const o3star_p_o5star = numeric::conversions::radians( 109.0 );
		cst_set_p->add_atom_angle_constraint( o3star_id, next_p_id, next_o5star_id,
																					o3star_p_o5star, angle_tether );

		// Three dihedral constraints... epsilon, zeta, and next residue's alpha.
		float const dummy  =  numeric::conversions::radians( 20.0 ); // not actually used!!

		// epsilon
		float theta0 = numeric::conversions::radians( 170.0 );
		float periodic_value = numeric::conversions::radians( 360.0 );
		cst_set_ns::Torsion_cst torsion_cst( c4star_id, c3star_id, o3star_id, next_p_id,
																				 theta0, dummy, periodic_value, angle_tether,
																				 "PERIODIC" );
		cst_set_p->add_atom_torsion_constraint( torsion_cst );

		// zeta
		theta0 = numeric::conversions::radians( -67.0 );
		periodic_value = numeric::conversions::radians( 120.0 );
		torsion_cst = cst_set_ns::Torsion_cst ( c3star_id, o3star_id, next_p_id, next_o5star_id,
																				 theta0, dummy, periodic_value, angle_tether,
																				 "PERIODIC" );
		cst_set_p->add_atom_torsion_constraint( torsion_cst );

		// alpha
		theta0 = numeric::conversions::radians( -67.0 );
		periodic_value = numeric::conversions::radians( 120.0 );
		torsion_cst = cst_set_ns::Torsion_cst ( o3star_id, next_p_id, next_o5star_id, next_c5star_id,
																				 theta0, dummy, periodic_value, angle_tether,
																				 "PERIODIC" );
		cst_set_p->add_atom_torsion_constraint( torsion_cst );


	}

	if (pose.constraints_exist()){
		cst_set_p -> collect( pose.constraints() );
	}

	pose.set_constraints( *cst_set_p );

	// this was allocated by "new" (above), and will cause a memory leak otherwise
	cst_set_p->remove_ptr_reference();

}

///////////////////////////////////////////////////////////////////////////////
void
fill_bval_with_atom_energy( pose_ns::Pose & pose, bool const subtract_reference_state = false )
{

	using namespace aaproperties_pack;

	Pdb_info & pdb_info = pose.pdb_info(); //not const, let's change this sucker.

	//local array to keep track of atom energy.
	int const total_residue = pose.total_residue();
	FArray2D_float atom_energy( param::MAX_ATOM(), total_residue, 0.0 );

	evaluate_atom_energy( atom_energy );

	if ( subtract_reference_state ) {
		FArray2D_float reference_atom_energy( param::MAX_ATOM(), total_residue, 0.0 );
		pose_ns::Pose reference_pose;
		initialize_query_pose_rna( reference_pose, false /*use_fasta*/ );
		evaluate_atom_energy( reference_atom_energy );
		atom_energy = atom_energy - reference_atom_energy;
	}



	float atom_energy_total( 0.0f ); //Just a diagnostic.

	for ( int i = 1; i <= total_residue; i++ ){
		int const aa = pose.res( i );
		int const aav = pose.res_variant( i );
		for ( int j = 1; j <= natoms( aa, aav ); j++ ) {
			pdb_info.set_pdb_bvalue( j, i, atom_energy(j , i) );
			atom_energy_total += atom_energy( j, i );
		}
	}

	std::cout << "DISPLAY_ENERGY  " <<  atom_energy_total << std::endl;

}

///////////////////////////////////////////////////////////////////////////////
void
dump_fullatom_pdb_with_color(  std::string const start_file, pose_ns::Pose & pose )
{
	pose_io::bval_src = pose_io::PDB_BVAL;
	fill_bval_with_atom_energy( pose );
	pose.dump_pdb( start_file );
}


///////////////////////////////////////////////////////////////////////////////
// Poor man's rotamer trials for 2'-OH only. Note this might also be
// doable through optE or something, but I like having full control
// over these very important hydrogens!
void
rna_o2star_crude_rotamer_trials( pose_ns::Pose & pose, pose_ns::Score_weight_map & rna_weight_map, int const seqpos )
{
	using namespace pose_ns;

	static int const o2star_torsion_num = 10;

	// Currently, hard to find experimental info on RNA 2'-OH positions -- no neutron diffraction data!
	static float const torsion_offset = rna_scoring::o2star_torsion_offset;
	static float const torsion_period = 120.0; // Again, not absolutely necessary to test three periodic angles.

	static float const torsion_standard_deviation = 15.0; // typical width seen in MD?

	pose.score( rna_weight_map );
	float const start_score = pose.get_0D_score( SCORE );
	float best_score( 0.0 ), best_chi( 0.0 );

	best_score = start_score;
	best_chi   = pose.get_torsion_by_number( seqpos, o2star_torsion_num );

	//	bool init( false );
	for (int k = 1; k <= 3; k++ ){

		for (int m = -1; m <= 1; m++ ){
			float const chi = torsion_offset + (k-1) * torsion_period + m * torsion_standard_deviation;

			pose.set_torsion_by_number( seqpos, o2star_torsion_num, chi );

			pose.score( rna_weight_map );
			float const current_score = pose.get_0D_score( SCORE );

			if (current_score < best_score){
				best_score = current_score;
				best_chi = chi;
				//				init = true;
			}

		}

	}

	pose.set_torsion_by_number( seqpos, o2star_torsion_num, best_chi );
	pose.score( rna_weight_map ); //just in case...

}

///////////////////////////////////////////////////////////////////////////////
void
rna_o2star_crude_rotamer_trials( pose_ns::Pose & pose, pose_ns::Score_weight_map & rna_weight_map )
{

	std::cout << "Rotamer trials for 2'-OH torsions... " << std::endl;

	int const nres = pose.total_residue();

	for (int i = 1; i <= nres; i++ ){
		rna_o2star_crude_rotamer_trials( pose, rna_weight_map, i );
	}

}

///////////////////////////////////////////////////////////////////////////////
void
rna_o2star_repack_test()
{
  using namespace pose_ns;
  using namespace param_aa;
  using namespace silent_io;

	Pose pose;
	bool success = pose_from_pdb( pose, stringafteroption("s","tltr_RNA.pdb"),
																true, false, true );
	if (!success){
		std::cout << "Had trouble with input pdb" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	Score_weight_map rna_weight_map = setup_rna_fullatom_weight_map();

	setup_RNA_geometry_constraints( pose );

	//Try different rotamers for my favorite O2', and dump.

	//	int const	o2star_torsion_num = 10;

	// 	pose.set_torsion_by_number( 16, o2star_torsion_num, 60.0 );
	// 	pose.score( rna_weight_map );
	// 	std::cout << "ANGLE 60.0 " << pose.show_scores() << std::endl;
	// 	pose.dump_pdb( "blah_60.pdb");

	// 	pose.set_torsion_by_number( 16, o2star_torsion_num, 180.0 );
	// 	pose.score( rna_weight_map );
	// 	std::cout << "ANGLE 180.0 " << pose.show_scores() << std::endl;
	// 	pose.dump_pdb( "blah_180.pdb");

	// 	pose.set_torsion_by_number( 16, o2star_torsion_num, 240.0 );
	// 	pose.score( rna_weight_map );
	// 	std::cout << "ANGLE 305.0 " << pose.show_scores() << std::endl;
	// 	pose.dump_pdb( "blah_300.pdb");

	pose.dump_pdb( "START.pdb" );
	pose.score( rna_weight_map );
	std::cout << "START " << pose.show_scores() << std::endl;

	rna_o2star_crude_rotamer_trials( pose, rna_weight_map );

	pose.dump_pdb( "FINAL.pdb" );
	pose.score( rna_weight_map );
	std::cout << "FINAL " << pose.show_scores() << std::endl;

}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void
rna_fullatom_minimize_test()
{
  using namespace pose_ns;
  using namespace param_aa;
  using namespace silent_io;

	std::vector< std::string > files;

	//Hmm, shouldn't be reproducing code between here and rna_score_test()
	// Oh well.

	Pose native_pose;
	bool native_pose_exists( false );
	if (truefalseoption("n")) {
		native_pose_exists = true;
		std::string native_file_name = stringafteroption("n","blah.pdb");
		bool success = pose_from_pdb( native_pose, native_file_name,
									 true, false, true );
		if (!success){
			std::cout << "Had trouble with native pdb " << native_file_name << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

  bool const silent_input = truefalseoption( "silent" );

  Silent_file_data * decoys_p;

  if (silent_input){
  // read silent file
    decoys_p = new Silent_file_data( stringafteroption("s"), true /*fullatom*/ );
    files = decoys_p->tags();
  } else {
		if (truefalseoption("s")) {
			files.push_back( stringafteroption("s") );
		}

		if (truefalseoption("l")) {
			std::ifstream data( stringafteroption("l").c_str() );
			std::string line;
			while ( getline( data,line ) ) {
				files.push_back(line);
			}
			data.close();
		}

		decoys_p = NULL; //silent file data not used.
	}

	std::string const scorefilename = stringafteroption("o","minimize.out");
	Silent_out out( scorefilename );

	Score_weight_map rna_weight_map = setup_rna_fullatom_weight_map();

	for ( std::vector< std::string >::const_iterator file=files.begin();
				file != files.end(); ++file ) {

		// read pose
		Pose pose;

		std::string start_file( *file);

		if (!out.start_decoy( start_file ) ) continue;

		std::cout << "Reading in ... " << start_file << std::endl;

    if ( silent_input ){
      Silent_structure const & decoy( decoys_p->get_structure( start_file ) );
      decoy.fill_pose( pose, true );
    } else {
      pose_from_pdb( pose, start_file, true /*fullatom*/, false /*ideal_pose*/, true /*read_all_chains*/ );
    }

		pose.copy_to_misc(); // need to set total_residue correctly, etc.

		//Diagnostics.
		if (native_pose_exists){
			pose.set_native_pose( native_pose );
		}

		barcode_initialize_start( pose.total_residue() );
		barcode_initialize_decoy();

		std::string const tag  = "minimize_" + start_file;

		//following includes minimize...
		pose.score( rna_weight_map );
		put_the_final_touch_on_rna( pose, out, tag, true /*minimize*/);

		//		score_the_pose( pose, rna_weight_map );
		//		out.write( start_file, pose );

		if ( start_file.rfind(".pdb") == std::string::npos) start_file += ".pdb";
		static bool const dump_minimized_pdb = truefalseoption( "dump_minimized_pdb" );
		if (dump_minimized_pdb) dump_fullatom_pdb_with_color( start_file, pose );

	}


}


///////////////////////////////////////////////////////////////////////////////
void
compute_low_res_rna_score_terms( pose_ns::Pose & pose, pose_ns::Score_weight_map & minimize_weight_map )
{
	using namespace pose_ns;

	float const original_rna_bp_w_weight = minimize_weight_map.get_weight( RNA_BP_W );


	if (  original_rna_bp_w_weight == 0.0 ){
		Score_weight_map hires_weight_map_with_a_little_lores( minimize_weight_map );
		hires_weight_map_with_a_little_lores.set_weight( RG, 1e-5 );
		hires_weight_map_with_a_little_lores.set_weight( VDW, 1e-5 );
		hires_weight_map_with_a_little_lores.set_weight( RNA_BS, 1e-5 );

		// This was last scores with a hi res energy function -- let's calculate the low res terms too!
		pose.new_score_pose();
		pose.score( hires_weight_map_with_a_little_lores );

		//To tally up the low res score, its best to make a copy of
		// the pose, since otherwise some of the hires scores
		// will be zeroed out.
		Score_weight_map lores_weight_map = setup_rna_weight_map();
		Pose pose_copy;
		pose_copy = pose;
		pose_copy.score( lores_weight_map );
		float const lores_score = pose_copy.get_0D_score( SCORE );

		pose.set_extra_score( "LOW_RES", lores_score );
	}

}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void
minimize_rna( pose_ns::Pose & pose ) {
	using namespace pose_ns;

	std::string mode_title_save = files_paths::mode_title;

	std::cout << "Beginning minimize... " << std::endl;

	atom_tree_set_allow_move( pose, true /* move bb*/, true /*move sc*/, true /*move jump*/ );

	Score_weight_map rna_weight_map = setup_rna_fullatom_weight_map();

	setup_RNA_geometry_constraints( pose );

	int const rounds = intafteroption("minimize_rounds", 2);

	static float const coord_cst_weight = realafteroption("coord_cst_weight", 0.01);

	static bool const skip_o2star_rotamer_trials = truefalseoption( "skip_o2star_rotamer_trials" );

	if (!skip_o2star_rotamer_trials) rna_o2star_crude_rotamer_trials( pose, rna_weight_map );

	for (int r = 1; r <= rounds; r++ ){
		std::cout << "Minimize round " << r << " of " << rounds << std::endl;
		files_paths::mode_title = "RNA min round " + string_of( r );
		Score_weight_map minimize_weight_map( rna_weight_map );

		minimize_weight_map.set_weight( FA_REP, static_cast<float>( r )/ rounds );

		//Discrete optimization of hydrogen positions...
		if (!skip_o2star_rotamer_trials) rna_o2star_crude_rotamer_trials( pose, minimize_weight_map );

		//Prevent explosions on first minimize.
		if (r==1) minimize_weight_map.set_weight( COORD_CST, coord_cst_weight );

		pose.new_score_pose(); // doesn't hurt

		pose.main_minimize( minimize_weight_map, "dfpmin" );
	}

	// A little rigamarole to also get the "low res" potential computed. Why not?
	compute_low_res_rna_score_terms( pose, rna_weight_map );

	files_paths::mode_title = mode_title_save;


}

