// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 23432 $
//  $Date: 2008-06-24 16:25:52 +0300 (Tue, 24 Jun 2008) $
//  $Author: yab $


// Rosetta Headers
#include "aaproperties_pack.h"
#include "current_pose.h"
#include "files_paths.h"
#include "pack.h"
#include "pack_geom_inline.h"
#include "param.h"
#include "param_aa.h"
#include "pose_backrub.h"
#include "pose_rotamer_controller.h"
#include "random_numbers.h"
#include "RotamerSet.h"

// C++ Headers
#include <iomanip>

namespace pose_ns {

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::Rotamer_controller
///
/// @brief
/// initialize the new Rotamer_controller object empty
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
Rotamer_controller::Rotamer_controller():
	pose_(0),
	packer_task_(0)
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::Rotamer_controller
///
/// @brief
/// initialize the new Rotamer_controller object using a Pose object
///
/// @detailed
///
/// @param[in] pose - Pose object
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
Rotamer_controller::Rotamer_controller(
	pose_ns::Pose * pose
):
	packer_task_(0)
{
	init(pose);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::~Rotamer_controller
///
/// @brief
/// deallocate the new Rotamer_controller sub-objects
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
Rotamer_controller::~Rotamer_controller()
{
	if (packer_task_) {
		delete packer_task_;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::set_include_current
///
/// @brief
/// set the include_current flag in the packer task
///
/// @detailed
///
/// @param[in] include_current - flag value
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
Rotamer_controller::set_include_current(
	bool include_current
)
{
	if (packer_task_->get_include_current() != include_current) {
		// invalidate current rotamer library if changing flag
		rotamer_set_.reset();
		packer_task_->set_include_current(include_current);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::include_current
///
/// @brief
/// get the include_current flag in the packer task
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
Rotamer_controller::include_current()
{
	return packer_task_->get_include_current();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::init
///
/// @brief
/// initialize the rotamer library using a Pose object
///
/// @detailed
///
/// @param[in] pose - Pose object
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
Rotamer_controller::init(
	pose_ns::Pose * pose
)
{
	pose_ = pose;

	init_packer_task();
	init_rotamers();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::init_packer_task
///
/// @brief
/// initialize the PackerTask object
///
/// @detailed
/// use global resfile and Pose allow_chi_move to setup residue freedom
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
Rotamer_controller::init_packer_task()
{
	assert(pose_ != 0);

	FArray1D_bool allow_repack_local(param::MAX_RES()());

	packer_task_ = new PackerTask(*pose_);

	packer_task_->set_mode((files_paths::resfile == "none") ? "packrot" : "design");

	// set variables that specify which residues to vary
	for (int i = 1; i <= pose_->total_residue(); i++) {
		allow_repack_local(i) = pose_->get_allow_chi_move(i);
	}
	packer_task_->set_allow_repack(allow_repack_local);

	packer_task_->setup_residues_to_vary();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::init_rotamers
///
/// @brief
/// initialize the RotamerSet object
///
/// @detailed
/// use current conformation of the Pose object to generate rotamers
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
Rotamer_controller::init_rotamers()
{
	assert(pose_ != 0);
	pose_->copy_to_misc();

	FArray1D_int current_rot_index(param::MAX_RES()());

	// Tell get_rotamers_seqpos_aa_aav not to screen for backbone clashes
	bool const old_rotamerize = design::design_commands::rotamerize;
	design::design_commands::rotamerize = true;

	pack_set_current_pose(*pose_);

	//select_rotamer_set("large");
	packer_task_->set_include_extra(false);
	rotamer_set_.get_rotamers(pose_->total_residue(), pose_->res(), pose_->res_variant(),
	                          pose_->full_coord(), current_rot_index, *packer_task_);

	index_rotamers();

	pack_reset_current_pose();
	design::design_commands::rotamerize = old_rotamerize;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::index_rotamers
///
/// @brief
/// create an index of the data in the RotamerSet
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
Rotamer_controller::index_rotamers()
{
	int size = pose_->total_residue();

	// create per-residue indecies
	indexpos_.resize(size, 0);
	nrotpos_.resize(size, 0);
	indexposaa_.resize(size);
	nrotposaa_.resize(size);
	for (int i = 1; i <= size; i++) {
		indexposaa_[i].resize((int)param::MAX_AA(), 0);
		nrotposaa_[i].resize((int)param::MAX_AA(), 0);
	}
	for (int i = 1; i <= rotamer_set_.nrotamers(); i++) {
		int const respos = rotamer_set_.report_seqpos(i);
		int const resaa = rotamer_set_.report_aa(i);
		if (indexpos_[respos] == 0) {
			indexpos_[respos] = i;
		}
		if (indexposaa_[respos][resaa] == 0) {
			indexposaa_[respos][resaa] = i - indexpos_[respos] + 1;
		}
		nrotpos_[respos]++;
		nrotposaa_[respos][resaa]++;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::print
///
/// @brief
/// print entire rotamer library
///
/// @detailed
///
/// @param[in] chi - boolean, print chi angles for each rotamer
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
Rotamer_controller::print(
	bool chi // = true
)
{
	std::cout << "Rotamer Library (" << rotamer_set_.nrotamers()
	          << " rotamers)" << std::endl;

	for (unsigned int pos = 1; pos <= nrotpos_.size(); pos++) {
		if (nrotpos_[pos] == 0) continue;
		print(pos, chi);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::print
///
/// @brief
/// print rotamer library for a single position
///
/// @detailed
///
/// @param[in] pos - amino acid position
/// @param[in] chi - boolean, print chi angles for each rotamer
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
Rotamer_controller::print(
	int pos,
	bool chi // = true
)
{
	std::cout << " Position " << pos << ": "
	          << param_aa::aa_name3(pose_->res(pos)) << " (" << nrotpos_[pos]
	          << " rotamers)" << std::endl;

	for (unsigned int aa = 1; aa <= nrotposaa_[pos].size(); aa++) {
		if (nrotposaa_[pos][aa] == 0) continue;
		print(pos, aa, chi);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::print
///
/// @brief
/// print rotamer library for a single position and amino acid type
///
/// @detailed
///
/// @param[in] pos - amino acid position
/// @param[in] aa - amino acid type
/// @param[in] chi - boolean, print chi angles for each rotamer
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
Rotamer_controller::print(
	int pos,
	int aa,
	bool chi // = true
)
{
	int const aav = rotamer_set_.report_aav(indexposaa_[pos][aa]);

	std::cout << "  Amino Acid " << aa << ": " << param_aa::aa_name3(aa)
	          << " (" << nrotposaa_[pos][aa] << " rotamers)" << std::endl;

	if (!chi) return;
	for (int i = 1; i <= nrotposaa_[pos][aa]; i++) {
		std::cout << "   " << std::setw(2) << i << ":";
		for (int j = 1; j <= aaproperties_pack::nchi(aa, aav); j++) {
			std::cout << std::fixed << std::setprecision(2) << std::setw(8)
			          << rotamer_set_.get_rchi(j,indexposaa_[pos][aa]+i-1);
		}
		std::cout << std::endl;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::closest_rotamer
///
/// @brief
/// for each amino acid, find the closest rotamer in the RotamerSet
///
/// @detailed
///
/// @param[in] pose - Pose object
///
/// @global_read
///
/// @global_write
///
/// @return
/// a vector with the per-residue rotamer numbers
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
utility::vector1<int>
Rotamer_controller::closest_rotamer(
	pose_ns::Pose & pose
)
{
	int const size = pose_->total_residue();
	utility::vector1<int> closestrot(size);
	FArray1D_int bestrotamer_at_seqpos(param::MAX_RES());

	pose.copy_to_misc();

	rotamer_set_.select_closest_rotamer(pose, bestrotamer_at_seqpos);
	for (int i = 1; i <= size; i++) {
		closestrot[i] = bestrotamer_at_seqpos(i) - indexpos_[i] + 1;
	}

	return closestrot;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::nrotamers
///
/// @brief
/// get the total number of rotamers in the RotamerSet
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
/// the total number of rotamers in the RotamerSet
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int
Rotamer_controller::nrotamers()
{
	return rotamer_set_.nrotamers();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::nrotpos
///
/// @brief
/// get the number of rotamers at each amino acid position
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
/// a vector with the number of rotamers at each position
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
utility::vector1<int>
Rotamer_controller::nrotpos()
{
	return nrotpos_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::nrotpos
///
/// @brief
/// get the number of rotamers at an amino acid position
///
/// @detailed
///
/// @param[in] pos - amino acid position
///
/// @global_read
///
/// @global_write
///
/// @return
/// the number of rotamers at the specified position
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int
Rotamer_controller::nrotpos(
	int const pos
)
{
	return nrotpos_[pos];
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::res
///
/// @brief
/// get the amino acid type for a rotamer
///
/// @detailed
///
/// @param[in] pos - amino acid position
/// @param[in] rotnum - rotamer number
///
/// @global_read
///
/// @global_write
///
/// @return
/// the amino acid type for the given position/rotamer number
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int
Rotamer_controller::res(
	int const pos,
	int const rotnum
)
{
	return rotamer_set_.report_aa(indexpos_[pos]+rotnum-1);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::set_rotamer
///
/// @brief
/// set the rotamer for a given amino acid position
///
/// @detailed
///
/// @param[in] pos - amino acid position
/// @param[in] rotnum - rotamer number
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
/// this method uses the internally referenced Pose object
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
Rotamer_controller::set_rotamer(
	int const pos,
	int const rotnum
)
{
	set_rotamer(*pose_, pos, rotnum);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::set_rotamer
///
/// @brief
/// set the rotamer for a given amino acid position
///
/// @detailed
///
/// @param[in] pose - Pose object
/// @param[in] pos - amino acid position
/// @param[in] rotnum - rotamer number
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
Rotamer_controller::set_rotamer(
	pose_ns::Pose & pose,
	int const pos,
	int const rotnum
)
{
	assert(pose.total_residue() == pose_->total_residue());

	int const rotpos = indexpos_[pos] + rotnum - 1;
	
	if (rotamer_set_.report_aa(rotpos) != param_aa::aa_pro) {
		FArray3D_float full_coord = pose.full_coord();
	
		set_rotamer(full_coord, pose.res(pos), pose.res_variant(pos), pos, rotnum);

		pose.copy_sidechain(pos, rotamer_set_.report_aa(rotpos), rotamer_set_.report_aav(rotpos),
		                    full_coord(1,1,pos), true, false);
	} else {
		pose.copy_sidechain(pos, rotamer_set_.report_aa(rotpos), rotamer_set_.report_aav(rotpos),
		                    rotamer_set_.get_rotcoord(rotpos)(1,1), true);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::set_rotamer
///
/// @brief
/// set the rotamer for a given amino acid position
///
/// @detailed
///
/// @param[in] full_coord - full_coord FArray
/// @param[in] pos - amino acid position
/// @param[in] rotnum - rotamer number
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
copy_sidechain_cb(
	int const aa,
	int const aav,
	FArray2Da_float res_coor_source,
	int const aa_old,
	int const aav_old,
	FArray2Da_float res_coor_dest
)
{
	res_coor_source.dimension(3, param::MAX_ATOM());
	res_coor_dest.dimension(3, param::MAX_ATOM());

	int const CBpos = (aa == param_aa::aa_gly) ? 6 : 5; // 2HA is gly pseudo CB
	int const CBpos_old = (aa_old == param_aa::aa_gly) ? 6 : 5; // 2HA is gly pseudo CB
	int const HNpos_old = aaproperties_pack::HNpos(aa_old,aav_old);
	int const HApos_old = aaproperties_pack::HApos(aa_old,aav_old);
	int const HNpos = aaproperties_pack::HNpos(aa,aav);
	int const HApos = aaproperties_pack::HApos(aa,aav);
	int const natoms = aaproperties_pack::natoms(aa,aav);
	FArray1D_float static tmppos(3);
	FArray1D_float static vec(3);
	FArray2D_float static mat(3,3);

	// get transformation to lineup CA->CB vectors
	lineup_bk(res_coor_source(1,2), res_coor_source(1,CBpos),
	          res_coor_dest(1,2), res_coor_dest(1,CBpos_old), mat, vec);

	tmppos(1) = res_coor_dest(1,HApos_old);
	tmppos(2) = res_coor_dest(2,HApos_old);
	tmppos(3) = res_coor_dest(3,HApos_old);
	if (HNpos > 0 && HNpos_old > 0) {
		res_coor_dest(1,HNpos) = res_coor_dest(1,HNpos_old);
		res_coor_dest(2,HNpos) = res_coor_dest(2,HNpos_old);
		res_coor_dest(3,HNpos) = res_coor_dest(3,HNpos_old);
	} else if (HNpos > 0) {
		// The HN position should be reset by build_nh_simple() in Pose::copy_sidechain
		res_coor_dest(1,HNpos) = 0;
		res_coor_dest(2,HNpos) = 0;
		res_coor_dest(3,HNpos) = 0;
	}
	res_coor_dest(1,HApos) = tmppos(1);
	res_coor_dest(2,HApos) = tmppos(2);
	res_coor_dest(3,HApos) = tmppos(3);

	// copy rotamer atom positions and lineup with current CA->CB vector
	for (int i = 5; i <= natoms; i++) {
		if (i != HNpos && i != HApos) {
			res_coor_dest(1,i) = res_coor_source(1,i);
			res_coor_dest(2,i) = res_coor_source(2,i);
			res_coor_dest(3,i) = res_coor_source(3,i);
			move_bk(res_coor_dest(1,i), mat, vec);
		}
	}

	// reset chi1 to rotamer definition
	if (aaproperties_pack::nchi(aa, aav) >= 1) {

		//bk retrieve atoms that determine chi angle
		int const c1 = aaproperties_pack::chi_atoms(1,1,aa,aav);
		int const c2 = aaproperties_pack::chi_atoms(2,1,aa,aav);
		int const c3 = aaproperties_pack::chi_atoms(3,1,aa,aav);
		int const c4 = aaproperties_pack::chi_atoms(4,1,aa,aav);

		//bk calculate initial chi angle
		float schi;
		dihedral_bk( res_coor_dest(1,c1), res_coor_dest(1,c2), res_coor_dest(1,c3),
		             res_coor_dest(1,c4), schi );

		float echi;
		dihedral_bk( res_coor_source(1,c1), res_coor_source(1,c2), res_coor_source(1,c3),
		             res_coor_source(1,c4), echi );

		//bk rotate angle by new-initial degrees
		float const rot = numeric::conversions::radians( echi - schi );

		//bk generate rotation vector and matrix
		getrot_bk( res_coor_dest(1,c2), res_coor_dest(1,c3), rot, mat, vec );

		//bk move atoms
		for ( int i = 1; i <= natoms; ++i ) {
			if ( aaproperties_pack::chi_required( 1, i, aa, aav ) ) {
				move_bk( res_coor_dest(1,i), mat, vec );
			}
		}
	}
	
	if (aa_old == param_aa::aa_pro && aa != param_aa::aa_pro) {
		replace_cb(aa, aav, res_coor_dest);
		replace_ha(aa, aav, res_coor_dest);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::set_rotamer
///
/// @brief
/// set the rotamer for a given amino acid position
///
/// @detailed
///
/// @param[in] full_coord - full_coord FArray
/// @param[in] pos - amino acid position
/// @param[in] rotnum - rotamer number
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
Rotamer_controller::set_rotamer(
	FArray3D_float & full_coord,
	int const aa_old,
	int const aav_old,
	int const pos,
	int const rotnum
)
{
	assert(pos <= pose_->total_residue());
	assert(rotnum <= nrotpos_[pos]);

	int const rotpos = indexpos_[pos] + rotnum - 1;
	int const aa = rotamer_set_.report_aa(rotpos);
	int const aav = rotamer_set_.report_aav(rotpos);
	
	assert(rotpos <= rotamer_set_.nrotamers());

	copy_sidechain_cb(aa, aav, rotamer_set_.get_rotcoord(rotpos)(1,1),
	                  aa_old, aav_old, full_coord(1,1,pos));
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::random_rotnum
///
/// @brief
/// get a random rotamer number for the given amino acid position
///
/// @detailed
///
/// @param[in] pos - amino acid position
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
/// this method give equal probability to all amino acid types
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int
Rotamer_controller::random_rotnum(
	int const pos
)
{
	// simple: all rotamers are the same amino acid type
	if (rotamer_set_.report_aa(indexpos_[pos]) ==
	    rotamer_set_.report_aa(indexpos_[pos]+nrotpos_[pos]-1)) {
		return random_range(1, nrotpos_[pos]);
	}

	int numaa = 0;
	int randaa;

	// more complex: first pick random amino acid, then pick random rotamer
	for (int i = 1; i <= param::MAX_AA(); i++) {
		if (nrotposaa_[pos][i] > 0) numaa++;
	}
	randaa = random_range(1, numaa);
	numaa = 0;
	for (int i = 1; i <= param::MAX_AA(); i++) {
		if (nrotposaa_[pos][i] > 0) numaa++;
		if (numaa == randaa) {
			randaa = i;
			break;
		}
	}

	return indexposaa_[pos][randaa] + random_range(1, nrotposaa_[pos][randaa])-1;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::pack
///
/// @brief
/// call the packer to re(pack/design) all moving residues
///
/// @detailed
///
/// @param[in] pose - Pose object
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
Rotamer_controller::pack(
	pose_ns::Pose & pose
)
{
	assert(packer_task_);
	assert(pose.total_residue() == pose_->total_residue());

	pose_ns::Pose pose_pack;
	pose_pack = pose;

	bool old_use_input_cb = design::active_rotamer_options.use_input_cb;
	design::active_rotamer_options.use_input_cb = true;
	
	pose_pack.pack(*packer_task_);

	design::active_rotamer_options.use_input_cb = old_use_input_cb;
	
	FArray3D_float pose_full_coord = pose.full_coord();
	FArray3D_float const & pose_pack_full_coord = pose_pack.full_coord();
	int const num_res = pose.total_residue();

	for (int i = 1; i <= num_res; i++) {
		if (packer_task_->get_designmap().repack_residue(i)) {
			if (pose_pack.res(i) != param_aa::aa_pro) {
				copy_sidechain_cb(pose_pack.res(i), pose_pack.res_variant(i), pose_pack_full_coord(1,1,i),
				                  pose.res(i), pose.res_variant(i), pose_full_coord(1,1,i));
				pose.copy_sidechain(i, pose_pack.res(i), pose_pack.res_variant(i),
				                    pose_full_coord(1,1,i), true, false);
			} else {
				pose.copy_sidechain(i, pose_pack.res(i), pose_pack.res_variant(i),
				                    pose_pack_full_coord(1,1,i), true);
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer_controller::print_designable_sequence
///
/// @brief
/// print the current amino acids of the designable positions
///
/// @detailed
///
/// @param[in] out - ostream to write to
///
/// @authors Colin A. Smith
////////////////////////////////////////////////////////////////////////////////
void
Rotamer_controller::print_designable_sequence(
	pose_ns::Pose & pose,
	std::ostream & out
)
{
	assert(packer_task_);

	DesignMap & dm = packer_task_->get_designmap();

	bool subsequent_line(false);

	for (int i = 1; i <= pose.total_residue(); i++) {
		if (dm.repack_residue(i) && dm.num_allowed_aa(i) > 1) {
			if (subsequent_line) out << " ";
			out << param_aa::aa_name1(pose.res(i)) << i;
			subsequent_line = true;
		}
	}
}

} // end pose_ns namespace
