// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13616 $
//  $Date: 2007-03-18 08:39:36 +0200 (Sun, 18 Mar 2007) $
//  $Author: stuartm $


// Rosetta Headers
#include "after_opts.h"
#include "ramachandran.h"
#include "current_pose.h"
#include "filters.h"
#include "param.h"
#include "param_aa.h"
#include "param_torsion.h"
#include "pose.h"
#include "read_paths.h"
#include "runlevel.h"
#include "score_ns.h"
#include "util_interpolate.h"
#include "wobble.h"
//KMa add_phospho_ser 2006-01
#include "options.h"
#include "add_pser.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/formatted.io.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdio>
#include <cstdlib>
#include <fstream>
#include <iostream>


// Namespaces
//KMa phospho_ser 2006-01
namespace ramachandran {
	FArray4D_float ram_probabil( 36, 36, 3, param::MAX_AA_PLUS() );
	FArray4D_int ram_counts( 36, 36, 3, param::MAX_AA_PLUS() );
	FArray4D_float ram_energ( 36, 36, 3, param::MAX_AA_PLUS() );
	int n_phi;
	int n_psi;
	FArray2D_float ram_entropy( 3, param::MAX_AA_PLUS() );
}


//     =========================================================================
//cj    take from ~cems/cems00/fullatom/wobble_rama2/
//cj    kindly developed by Charlie Strauss and co-opted by jotter
//cj    22Mar00
//     =========================================================================

//     -------------------------------------------------------------------------
//     Reads in an amino acid specific ramachandran table generated from
//     aproximately 950 pdb x-ray structures of < 2.0A resolution
//     - dimensions: 20X36x36 total bins
//     - in 10 degree bins  0...180,190...350s in both phi and psi
//
//     Amino Acid numbering scheme
//     A=1, C=2, D=3, E=4, F=5, G=6, H=7, I=8, K=9, L=10
//     M=11, N=12, P=13, Q=14, R=15, S=16, T=17, V=18, W=19, Y=20
////////////////////////////////////////////////////////////////////////////////
/// @begin read_rama
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
read_rama()
{
	using namespace ramachandran;

	int aa_num,phi_bin,psi_bin,ss_type;
	float check,min_prob,max_prob;
	double entropy;
	n_phi = 36;
	n_psi = 36;
	char line[60];
	int scan_count;

	utility::io::izstream & iunit( open_data_file( "Rama_smooth_dyn.dat_ss_6.4" ) );
//cj      std::cout << "index" << "aa" << "ramachandran entropy" << std::endl;
//KMa add_phospho_ser 2006-01
	for ( int i = 1; i <= param::MAX_AUTH_AA; ++i ) {
		for ( int ii = 1; ii <= 3; ++ii ) {
			entropy = 0.0;
			check = 0.0;
			min_prob = 1e36;
			max_prob = -min_prob;
			for ( int j = 1; j <= 36; ++j ) {
				for ( int k = 1; k <= 36; ++k ) {
					iunit.getline( line, 60 );
					if ( iunit.eof() ) {
						goto L100;
					} else if ( iunit.fail() ) { // Clear and continue: NO ERROR DETECTION
						iunit.clear();
					}
					std::sscanf( line, "%5d", &aa_num );
					std::sscanf( line+6, "%5d", &ss_type );
					std::sscanf( line+12, "%5d", &phi_bin );
					std::sscanf( line+18, "%5d", &psi_bin );
					std::sscanf( line+24, "%5d", &ram_counts(j,k,ii,i) );
					std::sscanf( line+30, "%12f", &ram_probabil(j,k,ii,i) );
					scan_count = std::sscanf( line+43, "%12f", &ram_energ(j,k,ii,i) );

//KMa phospho_ser 2006-01
					if ( add_pser () )	if (i==param_aa::aa_ser)
						{
						ram_probabil(j,k,ii,param_aa::aa_sep) =ram_probabil(j,k,ii,i) ;
					    ram_probabil(j,k,ii,param_aa::aa_sep) =ram_probabil(j,k,ii,i) ;
						}

					if ( scan_count == EOF ) continue; // Read problem: NO ERROR DETECTION

// This is the Slick & Slow (S&S) stream-based method that is too slow for large
// files like this one, at least under the GCC 3.3.1 stream implementation.
// It should be retried on future releases and target compilers because there is
// no reason it cannot be competitive with good optimization and inlining.
// If this is used the <cstdio> can be removed.
//
//					iunit >> bite( 5, aa_num ) >> skip( 1 ) >>
//					 bite( 5, ss_type ) >> skip( 1 ) >>
//					 bite( 5, phi_bin ) >> skip( 1 ) >>
//					 bite( 5, psi_bin ) >> skip( 1 ) >>
//					 bite( 5, ram_counts(j,k,ii,i) ) >> skip( 1 ) >>
//					 bite( 12, ram_probabil(j,k,ii,i) ) >> skip( 1 ) >>
//					 bite( 12, ram_energ(j,k,ii,i) ) >> skip;
//					if ( iunit.eof() ) {
//						goto L100;
//					} else if ( iunit.fail() ) { // Clear and continue: NO ERROR DETECTION
//						iunit.clear();
//						iunit >> skip;
//					}

					check += ram_probabil(j,k,ii,i);
					entropy += ram_probabil(j,k,ii,i) *
					 std::log( static_cast< double >( ram_probabil(j,k,ii,i) ) );
					min_prob = std::min(ram_probabil(j,k,ii,i),min_prob);
					max_prob = std::max(ram_probabil(j,k,ii,i),max_prob);
				}
			}
			ram_entropy(ii,i) = entropy;
		}
//cj		std::cout << SS( check ) << SS( std::log(min_prob) ) <<
//cj		 SS( std::log(max_prob) ) << SS( entropy ) << std::endl;
	}
L100:
	iunit.close();
	iunit.clear();

//cj      std::cout << "========================================" << std::endl;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin eval_rama_score_all
///
/// @brief
///car compute ramachandran score over all residues in the protein
///
/// @detailed
///
/// @param  phi - [in/out]? -
/// @param  psi - [in/out]? -
/// @param  res - [in/out]? -
/// @param  ss - [in/out]? -
/// @param  total_residue - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///db  added by db for use in minimizer
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
eval_rama_score_all(
	FArray1Da_float phi,
	FArray1Da_float psi,
	FArray1Da_int res,
	FArray1Da_char ss,
	int & total_residue
)
{
	phi.dimension( total_residue );
	psi.dimension( total_residue );
	res.dimension( total_residue );
	ss.dimension( total_residue );


	float rama,tmp1,tmp2;
	double rama_sum = 0.0;
	// in pose mode, we use fold_tree.cutpoint info to exclude terminus
	// residues from rama calculation. A cutpoint could be either an artificial
	// cutpoint such as loop cutpoint or a real physical chain break such as
	// multiple-chain complex. For the artificial cutpoint, we may need to
	// calculate rama scores for cutpoint residues, but for the real chain break
	// cutpoint, we don't want to do that. So here we first loop over all the
	// residue in the protein and exclude those ones which are the cutpoints.
	// Then we loop over the cutpoint residues and add rama score for residues
	// at artificial cutpoints, i.e., cut_weight != 0.0, which means that
	// jmp_chainbreak_score is also calculated for this cutpoint. Note that the
	// default value for cut_weight here is dependent on whether
	// jmp_chainbreak_weight is set. This is to ensure that rama score for
	// termini residues are not calculated when jmp_chainbreak_weight is 0.0,
	// e.g normal pose docking.
	if ( score_check_current_pose() ) {
		assert( total_residue == score_get_current_pose().total_residue() ||
			total_residue == score_get_current_pose().total_residue_for_scoring() );

		// retrieve cutpoint info
		int n_cut;
		FArray1D_int const & fold_tree_cutpoint(
			score_get_current_pose().fold_tree().get_fold_tree_cutpoint( n_cut ) );
		FArray1D_bool const & is_cutpoint(
			score_get_current_pose().fold_tree().get_is_cutpoint() );
		FArray1D_float cut_weight( n_cut,
			scorefxns::jmp_chainbreak_weight == 0.0 ? 0.0 : 1.0 );
		if( cut_weight.size1() == scorefxns::cut_weight.size1() )
			cut_weight = scorefxns::cut_weight;
		// exclude all the cutpoint residues first
		for ( int i = 1; i <= total_residue; ++i ) {
			if ( ! is_cutpoint(i) && ! is_cutpoint(i-1) ) {
				eval_rama_score_residue(res(i),phi(i),psi(i),ss(i),rama,tmp1,tmp2);
				rama_sum += rama;
			}
		}
		// add back those for which cut_weight is non-zero.
		for ( int j = 1; j <= n_cut; ++j ) {
			if ( cut_weight(j) != 0.0 ) {
				int i = fold_tree_cutpoint(j);
				eval_rama_score_residue(res(i),phi(i),psi(i),ss(i),rama,tmp1,tmp2);
				rama_sum += rama;
				i++;
				eval_rama_score_residue(res(i),phi(i),psi(i),ss(i),rama,tmp1,tmp2);
				rama_sum += rama;
			}
		}
	} else {
		for ( int i = 1; i <= total_residue; ++i ) {
			eval_rama_score_residue(res(i),phi(i),psi(i),ss(i),rama,tmp1,tmp2);
			rama_sum += rama;
		}
	}
	return rama_sum;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin write_rama_score_all
///
/// @brief
///car writes the rama score of each residue to the logfile
///
/// @detailed
///
/// @param  phi - [in/out]? -
/// @param  psi - [in/out]? -
/// @param  res - [in/out]? -
/// @param  ss - [in/out]? -
/// @param  total_residue - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///db  diagnostic
///db  added by db for use in minimizer
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
write_rama_score_all(
	FArray1Da_float phi,
	FArray1Da_float psi,
	FArray1Da_int res,
	FArray1Da_char ss,
	int & total_residue
)
{
	using namespace param_aa;

	phi.dimension( total_residue );
	psi.dimension( total_residue );
	res.dimension( total_residue );
	ss.dimension( total_residue );

//     local
	float rama,tmp1,tmp2;

	for ( int i = 1; i <= total_residue; ++i ) {
		if ( is_protein(res(i)) ) {
			eval_rama_score_residue(res(i),phi(i),psi(i),ss(i),rama,tmp1,tmp2);
			std::cout << I( 5, i ) << F( 9, 3, phi(i) ) << F( 9, 3, psi(i) ) <<
			 I( 4, res(i) ) << ' ' << A( 1, ss(i) ) << F( 8, 3, rama ) << std::endl;
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin eval_rama_score_residue
///
/// @brief
///
/// @detailed
///
/// @param  res - [in/out]? -
/// @param  phi - [in/out]? -
/// @param  psi - [in/out]? -
/// @param  ss - [in/out]? -
/// @param  rama - [in/out]? -
/// @param  drama_dphi - [in/out]? -
/// @param  drama_dpsi - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///ctsa all finite phi,psi will produce valid results, so exception
///ctsa handling has been removed from this function
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
eval_rama_score_residue(
	int const res,
	float const phi,
	float const psi,
	char const ss,
	float & rama,
	float & drama_dphi,
	float & drama_dpsi
)
{

	using namespace param_aa;
	using namespace ramachandran;
	using namespace runlevel_ns;

	if ( !is_protein(res) ) {
		rama = 0.0;
		drama_dphi = 0.0;
		drama_dpsi = 0.0;
		return;
	}
	if ( phi == 0.0 || psi == 0.0 ) { // begin or end of chain
		rama = 0.0;
		drama_dphi = 0.0;
		drama_dpsi = 0.0;
		return;
	}

//db
//db secondary structure dependent tables favor helix slightly.
//db only use if have predicted all alpha protein
//db
// rhiju and db: no longer use alpha-specific rama, after
//  tests on 1yrf and other all alpha proteins. 2-8-07

//	std::string protein_sstype = get_protein_sstype();
	int ss_type;
	if ( use_alpha_rama_flag() && get_protein_sstype() == "a" ) {
		ss_type = ( ( ss == 'H' ) ? 1 : ( ( ss == 'E' ) ? 2 : 3 ) );
	} else {
		ss_type = 3;
	}

//     do I (?cems/cj/cdb?) want to interpolate probabilities or log probs???
//     currently am interpolating  probs then logging.

	float interp_p,dp_dphi,dp_dpsi;
	interpolate_2d_func_of_angles(phi,psi,ram_probabil(1,1,ss_type,res),interp_p,
	 dp_dphi,dp_dpsi);

	if ( interp_p > 0.0 ) {
		rama = ram_entropy(ss_type,res) - std::log( static_cast< double >( interp_p ) );
		double const interp_p_inv_neg = -1.0 / interp_p;
		drama_dphi = interp_p_inv_neg * dp_dphi;
		drama_dpsi = interp_p_inv_neg * dp_dpsi;
	} else {
		if ( runlevel > silent ) {
			std::cout << "rama prob = 0. in eval_rama_score_residue!" << std::endl;
			std::cout << "phi" << SS( phi ) << " psi" << SS( psi ) <<
			 " ss " << SS( ss ) << std::endl;
		}
		drama_dphi = 0.0;
		drama_dpsi = 0.0;
		rama = 20.0;
	}

	if ( rama > 1.0 ) {
		float const rama_squared = rama * rama;
		if ( rama_squared > 20.0 ) {
			////  limit the score, but give the true derivative
			////   as guidance out of the flat section of map
			drama_dphi = 0.0;
			drama_dpsi = 0.0;
			rama = 20.0;
		} else {
			drama_dphi = 2 * rama * drama_dphi;
			drama_dpsi = 2 * rama * drama_dpsi;
			rama = rama_squared;
		}
	}

}

///////////////////////////////////////////////////////////////////////////////
bool
use_alpha_rama_flag()
{
	static bool use_alpha_rama( false );
	static bool init( false );

	if ( !init ) {
		use_alpha_rama = truefalseoption("use_alpha_rama");
		init = true;
	}
	return use_alpha_rama;
}
////////////////////////////////////////////////////////////////////////////////
/// @begin get_rama_score_residue_deriv
///
/// @brief
///ctsa  cheap interface to rama derivs
///
/// @detailed
///
/// @param  aa - [in/out]? -
/// @param  phi - [in/out]? -
/// @param  psi - [in/out]? -
/// @param  ss - [in/out]? -
/// @param  torsion - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
get_rama_score_residue_deriv(
	int const aa,
	float const phi,
	float const psi,
	char const ss,
	int const torsion
)
{
	using namespace param_torsion;

	float get_rama_score_residue_deriv; // Return value


	float const rama_dont_care = { 0.0 };

	float tmp1,tmp2;

	if ( torsion == phi_torsion ) {
		eval_rama_score_residue(aa,phi,psi,ss,tmp1,get_rama_score_residue_deriv,
		 tmp2);
	} else if ( torsion == psi_torsion ) {
		eval_rama_score_residue(aa,phi,psi,ss,tmp1,tmp2,
		 get_rama_score_residue_deriv);
	} else if ( torsion <= total_torsion ) {
		//// rama don't give a hoot' bout megas and chis
		get_rama_score_residue_deriv = rama_dont_care;
	} else {
		std::cout << "invalid torsion code in get_rama_score_residue_deriv:" <<
		 torsion << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	return get_rama_score_residue_deriv;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin read_rama
///
/// @brief
///
/// @param  map - [in/out]? -
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///car Map obtained from PROCHECK v3.5.4
///car                   file ps.cc function RAMREG
///car see also
///car  Morris et al, Proteins (1992) 12:354
///car "Laskowski R A, MacArthur M W, Moss D S & Thornton J M (1993). PROCHECK:
///car   a program to check the stereochemical quality of protein
///car   structures. J. Appl.  Cryst., 26, 283-291."
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
map_initializer( FArray1D_string & map )
{

	//car                    -180         psi                 180

	map( 1) = "bBBBBBBBBBBbbbxxooooowwwwwoooooxxxxb"; // 180
	map( 2) = "bBBBBBBBBBBBbbbxxooooooooooooooxxxbb";
	map( 3) = "bBBBBBBBBBBBbbbxxxxooooooooooooxxbbb";
	map( 4) = "bbBBBBBBBBBBBbbbbxxooooooooooooxxxbb";
	map( 5) = "bbBBBBBBBBBBBBbbxxxooooooooooooxxxxb";
	map( 6) = "bbBBBBBBBBBBBbbbxxxoooooooooooooxxxb";
	map( 7) = "bbbBBBBBBBBBbbbbxxxooozzzzzzoooooxxb";
	map( 8) = "bbbbBBBBBBBbbbbbbxxzzzzzzzzzoooooxxb";
	map( 9) = "bbbbbBbbBbbbbbbbbxxzzzzzllzzoooooxxb";
	map(10) = "bbbbbbbbbbbbbbxxxxxzzllllzzzoooooxxx";
	map(11) = "bbbbbbbbbbbbxxxxxxxzzllllzzzoooooxxx";
	map(12) = "xbbbbbbbbbbbbxxoooozzzlllzzzzooooxxx";
	map(13) = "xxbbbbbbbbbbbxxoooozzllllllzzooooxxx";
	map(14) = "yyaaaaaaaaaayyyoooozzllLllzzzooooxxx";
	map(15) = "yaaaaaaaaaaayyyoooozzzlLLlzzzooooxxx";
	map(16) = "yaaaaaaAaaaaayyyooozzzlllllzzooooxxx";
	map(17) = "yaaaaaAAAAaaayyyyooozzzlllzzzooooxxx";
	map(18) = "yaaaaaAAAAAaaayyyooozzzzlllzzooooxxx"; // phi
	map(19) = "yaaaaAAAAAAAaaayyyoozzzlzllzzooooxxx";
	map(20) = "yaaaaaAAAAAAAaayyyyozzzzzzzzzooooxxx";
	map(21) = "yyaaaaAAAAAAAAaayyyozzzzzzzzzooooxxx";
	map(22) = "yyaaaaaAAAAAAAaaayyyoooooooooooooxxx";
	map(23) = "yyyaaaaaAAAAAAAaayyyoooooooooooooxxx";
	map(24) = "yaaaaaaaaAAAAAAaaayyyooooooooooooxxx";
	map(25) = "aayaaaaaaaaAAAAaaayyyooooooooooooxxx";
	map(26) = "yyyaaaaaaaaaaaaaaaayyooooooooooooxxx";
	map(27) = "yyyyyaaaaaaaaaaaaayyyooooooooooooxxx";
	map(28) = "oyyyyaaaaaaaaaayyyyyyooooooooooooooo";
	map(29) = "oooyyyyyyyyayyyyyyyyoooooooooooooooo";
	map(30) = "oooxxxxbbxxxxxxxxooooooooooooooooooo";
	map(31) = "xxxxxbbxxxxxxxooooooowwwwwoooooooooo";
	map(32) = "xxxxxxbbbbxxxxooooooowwwwwoooooooooo";
	map(33) = "xbbbxbbbbbxxxxxoooooowwewwwooooooooo";
	map(34) = "xbbbbbbbbbbxxxxoooooowwewwwooooooxxx";
	map(35) = "xbbbbbbbbbbbbxxoooooowweewwooooooxxx";
	map(36) = "bbbbbbbbbbbbbxxoooooowwwwwwooooooxxb"; // -180

	//car o disallowed
	//car B favorable
	//car b allowed
	//car A favorable
	//car a allowed
	//car L favorable
	//car l allowed

	//car p
	//car e allowed
	//car x generous
	//car y generous
	//car z generous
	//car w generous
}

////////////////////////////////////////////////////////////////////////////////
/// @begin eval_procheck_rama
///
/// @brief
///
/// @detailed
///
/// @param  phi - [in/out]? -
/// @param  psi - [in/out]? -
/// @param  res - [in/out]? -
/// @param  total_residue - [in/out]? -
/// @param  favorable - [in/out]? -
/// @param  allowed - [in/out]? -
/// @param  generous - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
eval_procheck_rama(
	FArray1Da_float phi,
	FArray1Da_float psi,
	FArray1Da_int res,
	int & total_residue,
	float & favorable,
	float & allowed,
	float & generous
)
{
	using namespace param_aa;

	phi.dimension( total_residue );
	psi.dimension( total_residue );
	res.dimension( total_residue );

	int phibin,psibin;
	float phix,psix;
	char region;

	static FArray1D_string const map( 36, map_initializer );

	favorable = 0.;
	allowed = 0.;
	generous = 0.;

	int counter = 0;

	for ( int i = 2; i < total_residue; ++i ) {
		if ( res(i) == aa_gly || res(i) == aa_pro ) goto L100; // exclude G P
		++counter;
		phix = phi(i);
		psix = psi(i);
		angle_in_range(phix);
		angle_in_range(psix);
		phibin = static_cast< int >((phix+180.00)/10.0)+1;
		psibin = static_cast< int >((psix+180.00)/10.0)+1;
		psibin = 37-psibin; // cause map order is reversed
		if ( phibin < 1 || phibin > 36 || psibin < 1 || psibin > 36 ) std::cout <<
		 "binning error" << I( 4, i ) << F( 7, 1, phi(i) ) << F( 7, 1, psi(i) ) <<
		 F( 7, 1, phix ) << F( 7, 1, psix ) << F( 7, 1, (phix+180.00)/10.0 ) <<
		 F( 7, 1, (psix+180.00)/10.0 ) << I( 4, phibin ) << I( 4, psibin ) << std::endl;
		region = map(psibin)[phibin];
		if ( region == 'B' || region == 'A' || region == 'L' ) {
			favorable += 1.;
		} else if ( region == 'b' || region == 'a' || region == 'l' || region == 'e' ) {
			allowed += 1.;
		} else if ( region == 'x' || region == 'y' || region == 'z' || region == 'w' ) {
			generous += 1.;
		}
L100:;
	}

	if ( counter == 0 ) return;
	favorable /= counter;
	allowed /= counter;
	generous /= counter;

}
