// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 23432 $
//  $Date: 2008-06-24 16:25:52 +0300 (Tue, 24 Jun 2008) $
//  $Author: yab $


// Rosetta Headers
#include "read_aaproperties.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "design.h"
#include "dna.h"
#include "enzyme.h"
#include "hbonds.h" // jss use get_fine_hb_categories
#include "hbonds_ns.h" // for hybridization types
#include "files_paths.h"
#include "fullatom_setup.h"
#include "minimize.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "pH_main.h"
#include "pH_ns.h"
#include "read_paths.h"
#include "runlevel.h"
#include "termini.h"
#include "water.h"
#include "add_pser.h" //KMa phospho_ser 2006-01

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/formatted.i.hh>
#include <ObjexxFCL/string.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <cstdlib>
#include <string>
#include <fstream>
#include <iostream>

// Namespaces
namespace atom_mode_common {
	std::string atom_mode( "all" ); // default setting for atom mode
}

// index for fullatom_type
// fullatom type number -> human readable abbreviation
//  1   // CNH2   CD
//  1   // CNH2   CG
//  2   // COO    CD
//  2   // COO    CG
//  3   // CH1    CB
//  3   // CH1    CG
//  4   // CH2    CB
//  4   // CH2    CD
//  4   // CH2    CE
//  4   // CH2    CG
//  4   // CH2    CG1
//  5   // CH3    CB
//  5   // CH3    CD1
//  5   // CH3    CD2
//  5   // CH3    CE
//  5   // CH3    CG1
//  5   // CH3    CG2
//  6   // aroC   CD1
//  6   // aroC   CD2
//  6   // aroC   CE1
//  6   // aroC   CE2
//  6   // aroC   CE3
//  6   // aroC   CG
//  6   // aroC   CH2
//  6   // aroC   CZ
//  6   // aroC   CZ2
//  6   // aroC   CZ3
//  7   // Ntrp   NE1
//  8   // Nhis   ND1
//  8   // Nhis   NE2
//  9   // NH2O   ND2
//  9   // NH2O   NE2
// 10   // Nlys   NZ
// 11   // Narg   NE
// 11   // Narg   NH1
// 11   // Narg   NH2
// 12   // Npro   N
// 13   // OH     OG
// 13   // OH     OG1
// 13   // OH     OH
// 14   // ONH2   OD1
// 14   // ONH2   OE1
// 15   // OOC    OD1
// 15   // OOC    OD2
// 15   // OOC    OE1
// 15   // OOC    OE2
// 16   // S      SD
// 16   // S      SG
// 17   // Nbb    N
// 18   // CAbb   CA
// 19   // CObb   C
// 20   // OCbb   O
// 21   // Phos   P
// 22   // Hpol  1HD2
// 22   // Hpol  1HE2
// 22   // Hpol  1HH1
// 22   // Hpol  1HH2
// 22   // Hpol  1HZ
// 22   // Hpol  2HD2
// 22   // Hpol  2HE2
// 22   // Hpol  2HH1
// 22   // Hpol  2HH2
// 22   // Hpol  2HZ
// 22   // Hpol  3HZ
// 22   // Hpol   HE
// 22   // Hpol   HE1
// 22   // Hpol   HG
// 22   // Hpol   HG1
// 22   // Hpol   HH
// 23   // Hapo  1HB
// 23   // Hapo  1HD1
// 23   // Hapo  1HD2
// 23   // Hapo  1HE
// 23   // Hapo  1HG1
// 23   // Hapo  1HG2
// 23   // Hapo  2HA
// 23   // Hapo  2HB
// 23   // Hapo  2HD
// 23   // Hapo  2HD1
// 23   // Hapo  2HD2
// 23   // Hapo  2HE
// 23   // Hapo  2HG
// 23   // Hapo  2HG1
// 23   // Hapo  2HG2
// 23   // Hapo  3HA
// 23   // Hapo  3HB
// 23   // Hapo  3HD
// 23   // Hapo  3HD1
// 23   // Hapo  3HD2
// 23   // Hapo  3HE
// 23   // Hapo  3HG
// 23   // Hapo  3HG1
// 23   // Hapo  3HG2
// 23   // Hapo   HA
// 23   // Hapo   HB
// 23   // Hapo   HD2
// 23   // Hapo   HE2
// 23   // Hapo   HG
// 24   // Haro   HD1
// 24   // Haro   HD2
// 24   // Haro   HE1
// 24   // Haro   HE2
// 24   // Haro   HE3
// 24   // Haro   HH2
// 24   // Haro   HZ
// 24   // Haro   HZ2
// 24   // Haro   HZ3
// 25   // HNbb   H
// 26   // H2O


// forward declarations for local functions that need not be in .h
void zero_aaproperties();
void copy_aaproperties(int aa, int aav1, int aav2);
void create_alternate_his_tautomer();


////////////////////////////////////////////////////////////////////////////////
/// @begin set_atom_mode
///
/// @brief Initializes atom arrays based on mode passed by variable amode
/// amode can equal: initialize; all  (all is default)
///
/// @detailed
///
/// @param  amode - [in/out]? - flag passed in to specify run mode.  amode unchanged.
///
/// @global_read
/// MAX_CHI
/// chi_required
///
/// @global_write Variables changed depending on amode :
/// nside
/// abase2
/// atom_type_char
/// chiu_required
/// chi_rigid_atom_list
/// length_chi_rigid_atom_list
///
/// @remarks
/// There used to be other modes (noH; polH; polH&aroH),
/// but they weren't consistently implemented or used.
/// There also used to be a bunch of _complete arrays that maintained,
/// copies of the following, but since neither changed in mode="all",
/// we don't need the _complete arrays.
/// nchi_complete
/// nheavyatoms
/// nH_polar_complete
/// nhydrogens_on_atm_complete
/// nH_aromatic_complete
/// nH_apolar_complete
/// nh2o_complete
/// h2opos_complete
/// Hpos_aromatic_complete
/// nh2o_on_atm_complete
/// icoor_complete
/// atom_name_complete
/// fullatom_type_complete
/// chi_required_complete
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified JSS Dec 2006
/////////////////////////////////////////////////////////////////////////////////
void
set_atom_mode( std::string const & amode )
{
//bk set number of atoms to include for each amino acid, depends on amode
//bk which indicates which hydrogens to use

	using namespace aaproperties_pack;
	using namespace atom_mode_common;
	using namespace param;
	using namespace param_aa;

	if ( amode != "initialize" ) atom_mode = amode;
	std::cout << "Atom_mode set to " << atom_mode << std::endl;

  if ( atom_mode != "all" ) { // JSS only atom mode remaining Dec 2006
		std::cout << "STOPPING:: atom_mode: " << atom_mode <<
			" not recognized" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	for ( int aa = 1, aae = MAX_AA(); aa <= aae; ++aa ) {
		for ( int aav = 1, aave = nvar(aa); aav <= aave; ++aav ) {
			nside(aa,aav) = natoms(aa,aav) - first_scatom(aa,aav) + 1;
			for ( int atm = 1, atme = natoms( aa, aav ); atm <= atme; ++atm ) {
 				atom_type_char(atm,aa,aav) = atom_type_char_from_id(atm,aa,aav);
				get_atm_num_in_count_pair(aa,aav,atm,cp_atom_num(atm,aa,aav));
        if (abase2(atm,aa,aav) == 0) { //JSS if abase2 not set specially, make it atom_base^2
          if (atom_base(atm,aa,aav) > 0)
            abase2(atm,aa,aav) = atom_base(atom_base(atm,aa,aav),aa,aav);
          else {
            std::cout<< "WARNING: in set_atom_mode, can't set abase2("<<atm<<","<<aa<<","<<aav<<") since atom_base = "<<atom_base(atm,aa,aav)<< std::endl;
          }
        }
			}

//chu  chi_rigid_atom_list contains all the atoms whose positions depend only on
//    one certain chi angle.
			for ( int ch = 1; ch <= MAX_CHI; ++ch ) {
				if ( ch == MAX_CHI ) { // chi4
					int i = 0;
					for ( int atm = 1, atme = natoms( aa, aav ); atm <= atme; ++atm ) {
						if ( chi_required( ch, atm, aa, aav ) ) {
							++i;
							chi_rigid_atom_list(i,ch,aa,aav) = atm;
						}
					}
					length_chi_rigid_atom_list(ch,aa,aav) = i;
				} else {             // chi1, chi2, chi3
					int i = 0;
					for ( int atm = 1, atme = natoms( aa, aav ); atm <= atme; ++atm ) {
						if ( chi_required(ch,atm,aa,aav) && !chi_required(ch+1,atm,aa,aav) ) {
							++i;
							chi_rigid_atom_list(i,ch,aa,aav) = atm;
						}
					}
					length_chi_rigid_atom_list(ch,aa,aav) = i;
				}
			} // chi
		} // aav
	} // aa

	setup_intra_cp_table();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin atom_num_from_atom_name
///
/// @brief function to assign variable 'num' the index number for the
///       corresponding atom name 'name' for amino acid 'aa'
///
/// @detailed
///
/// @param[in]   name - in - variable passed in as basis for nam eassignment
/// @param[in]   aa - in - amino acid index
/// @param[in]   aav - in - amino acid variant index
/// @param[out]   num - out - variable assigned based on atom 'name'
///
/// @global_read - only variables listed above
///
/// @global_write - none
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
atom_num_from_atom_name(
	std::string const & name,
	int aa,
	int aav,
	int & num
)
{
	using namespace aaproperties_pack;
	using namespace runlevel_ns;

	num = 0;

	for ( int atm = 1, atme = natoms( aa, aav ); atm <= atme; ++atm ) {
		if ( name == atom_name( atm, aa, aav ) ) {
			num = atm;
			return;
		}
	}

	if ( runlevel >= inform ) std::cout << "Unknown atom name " << name << ' ' <<
	 aa << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin atom_name_from_atom_num
///
/// @brief assign atom name 'name' based on atom number 'num' from amino acid 'aa'
///
/// @detailed
///
/// @param[in]   num - in - index for atom number
/// @param[in]   aa - in - amino acid index
/// @param[in]   aav - in - amino acid variant index
/// @param[out]   name - out - variable being assigned
///
/// @global_read - only variables listed above
///
/// @global_write - none
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
atom_name_from_atom_num(
	int num,
	int aa,
	int aav,
	std::string & name
)
{
	using namespace aaproperties_pack;

	name = atom_name( num, aa, aav );
}



////////////////////////////////////////////////////////////////////////////////
/// @begin atom_type_char_from_id
///
/// @detailed
///
///  given an atom's *topological* id number in a given residue type
///  (aa,aav), return the 4-letter name associated with its *chemical*
///  id number
///
/// @param[in]   atmno - in - atom number in residue
/// @param[in]   aa - in - residue aa type
/// @param[in]   aav - in - " variant type
///
/// @return string name of chemical atom type
///
/// @remarks this function replaces needless array of same name
///  jss changed name to better suit function when I got rid of all other _complete arrays
/// @references
///
/// @authors ctsa 9-2003
/// jss Dec 2006
/////////////////////////////////////////////////////////////////////////////////
std::string
atom_type_char_from_id(
	int const atmno,
	int const aa,
	int const aav
)
{
	using namespace aaproperties_pack;
	using namespace param;

	std::string atom_type_char; // Return value

//------------------------------------------------------------------------------
	int fatype = fullatom_type( atmno, aa, aav );
	if ( fatype > 0 && fatype <= MAX_ATOMTYPES() ) {
		atom_type_char = atom_type_name( fatype );
	} else {
		atom_type_char = "    ";
	}

	return atom_type_char;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin LookupByName
///
/// @brief For a given aa and aav, takes an atomname (string) and returns the
///					number of the atom in the residue
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Jim Havranek  07/29/04
///
/// @last_modified 2006/10/25
/////////////////////////////////////////////////////////////////////////////////
int
LookupByName( int const aa, int const aav, std::string const & name )
{
	using namespace aaproperties_pack;

	std::string const stripped_name( stripped_whitespace( name ) );
	for ( int i = 1, e = natoms( aa, aav ); i <= e; ++i ) {
		if ( stripped_name == stripped_whitespace( atom_name( i, aa, aav ) ) ) return i;
	}

	return -1;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_ala
///
/// @brief arrays filled to define the proerties of the amino acid Alanine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// aa_ala - index number for ALA (1)
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to atom_base(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_ala()
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

//bk   Parameters for ALA -----------------------------
	nvar( aa_ala ) = 1; // starting starting number of variants for ALA
	variant_type( aav_base, aa_ala, 1 ) = true; // type of aa variant

//bk   variant1 ---------- ALA

	natoms( aa_ala,1) = 10; // number of total atoms
	nheavyatoms( aa_ala,1) = 5; // number of heavy atoms
	nchi( aa_ala,1) = 0; // number of chi angles

//bk   atom names ALA
	atom_name( 1, aa_ala,1) = " N  ";
	atom_name( 2, aa_ala,1) = " CA ";
	atom_name( 3, aa_ala,1) = " C  ";
	atom_name( 4, aa_ala,1) = " O  ";
	atom_name( 5, aa_ala,1) = " CB ";
	atom_name( 6, aa_ala,1) = " H  ";
	atom_name( 7, aa_ala,1) = " HA ";
	atom_name( 8, aa_ala,1) = "1HB ";
	atom_name( 9, aa_ala,1) = "2HB ";
	atom_name(10, aa_ala,1) = "3HB ";

//jjh intra residue bonding
	nbonded_neighbors( 1, aa_ala, 1) = 2; // N
		bonded_neighbor(1, 1, aa_ala, 1) = 2; // N--CA
		bonded_neighbor(2, 1, aa_ala, 1) = 6; // N--H
	nbonded_neighbors( 2, aa_ala, 1) = 4; // CA
		bonded_neighbor(1, 2, aa_ala, 1) = 1; // CA--N
		bonded_neighbor(2, 2, aa_ala, 1) = 3; // CA--C
		bonded_neighbor(3, 2, aa_ala, 1) = 5; // CA--CB
		bonded_neighbor(4, 2, aa_ala, 1) = 7; // CA--HA
	nbonded_neighbors( 3, aa_ala, 1) = 2; // C
		bonded_neighbor(1, 3, aa_ala, 1) = 2; // C--CA
		bonded_neighbor(2, 3, aa_ala, 1) = 4; // C--O
	nbonded_neighbors( 4, aa_ala, 1) = 1; // O
		bonded_neighbor(1, 4, aa_ala, 1) = 3; // O--C
	nbonded_neighbors( 5, aa_ala, 1) = 4; // CB
		bonded_neighbor(1, 5, aa_ala, 1) = 2; // CB--CA
		bonded_neighbor(2, 5, aa_ala, 1) = 8; // CB--1HB
		bonded_neighbor(3, 5, aa_ala, 1) = 9; // CB--2HB
		bonded_neighbor(4, 5, aa_ala, 1) =10; // CB--3HB
	nbonded_neighbors( 6, aa_ala, 1) = 1; // H
		bonded_neighbor(1, 6, aa_ala, 1) = 1; // H--N
	nbonded_neighbors( 7, aa_ala, 1) = 1; // HA
		bonded_neighbor(1, 7, aa_ala, 1) = 2; // HA--CA
	nbonded_neighbors( 8, aa_ala, 1) = 1; // 1HB
		bonded_neighbor(1, 8, aa_ala, 1) = 5; // 1HB--CB
	nbonded_neighbors( 9, aa_ala, 1) = 1; // 2HB
		bonded_neighbor(1, 9, aa_ala, 1) = 5; // 2HB--CB
	nbonded_neighbors( 10, aa_ala, 1) = 1; // 3HB
		bonded_neighbor(1,10, aa_ala, 1) = 5; // 3HB--CB

//jjh get the angles automatically
	get_angles_from_bonds(aa_ala, 1);

//bk   atom type numbers ALA
	fullatom_type( 1, aa_ala,1) = 17; // Nbb    N
	fullatom_type( 2, aa_ala,1) = 18; // CAbb   CA
	fullatom_type( 3, aa_ala,1) = 19; // CObb   C
	fullatom_type( 4, aa_ala,1) = 20; // OCbb   O
	fullatom_type( 5, aa_ala,1) =  5; // CH3    CB
	fullatom_type( 6, aa_ala,1) = 25; // HNbb   H
	fullatom_type( 7, aa_ala,1) = 23; // Hapo   HA
	fullatom_type( 8, aa_ala,1) = 23; // Hapo  1HB
	fullatom_type( 9, aa_ala,1) = 23; // Hapo  2HB
	fullatom_type(10, aa_ala,1) = 23; // Hapo  3HB

//cmd  atomic charges ALA
	atomic_charge( 1, aa_ala,1) = -0.47; //    N
	atomic_charge( 2, aa_ala,1) =  0.07; //    CA
	atomic_charge( 3, aa_ala,1) =  0.51; //    C
	atomic_charge( 4, aa_ala,1) = -0.51; //    O
	atomic_charge( 5, aa_ala,1) = -0.27; //    CB
	atomic_charge( 6, aa_ala,1) =  0.31; //    H
	atomic_charge( 7, aa_ala,1) =  0.09; //    HA
	atomic_charge( 8, aa_ala,1) =  0.09; //   1HB
	atomic_charge( 9, aa_ala,1) =  0.09; //   2HB
	atomic_charge(10, aa_ala,1) =  0.09; //   3HB

	assert_total_charge( aa_ala, 1, 0.0);

//cmd  first sidechain atom
	first_scatom( aa_ala,1) = 5; //CB

//bk   template atoms used for placing atoms AA

//bk   template for building  HA
	ta(1, 7, aa_ala,1) =  2; //   CA
	ta(2, 7, aa_ala,1) =  1; //   N
	ta(3, 7, aa_ala,1) =  3; //   C

//bk   template for building 1HB
	ta(1, 8, aa_ala,1) =  5; //   CB
	ta(2, 8, aa_ala,1) =  2; //   CA
	ta(3, 8, aa_ala,1) =  1; //   N

//bk   template for building 2HB
	ta(1, 9, aa_ala,1) =  5; //   CB
	ta(2, 9, aa_ala,1) =  2; //   CA
	ta(3, 9, aa_ala,1) =  1; //   N

//bk   template for building 3HB
	ta(1,10, aa_ala,1) =  5; //   CB
	ta(2,10, aa_ala,1) =  2; //   CA
	ta(3,10, aa_ala,1) =  1; //   N

//bk   identify atoms that require chi angles to be built, for ALA this is none

//bk   hbonding information
	HNpos( aa_ala,1) =  6; // position of backbone HN
	HApos( aa_ala,1) =  7; // position of backbone HA
	nH_polar( aa_ala,1) =  1; // number of polar hydrogens
	nH_aromatic( aa_ala,1) =  0; // number of aromatic hydrogens
	nH_apolar( aa_ala,1) =  4; // number of apolar hydrogens
	nacceptors( aa_ala,1) =  1; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1, aa_ala,1) =  6; //  H

//bk   atom numbers for aromatic hydrogens

//bk   atom numbers for apolar hydrogens
	Hpos_apolar( 1, aa_ala,1) =  7; //  HA
	Hpos_apolar( 2, aa_ala,1) =  8; // 1HB
	Hpos_apolar( 3, aa_ala,1) =  9; // 2HB
	Hpos_apolar( 4, aa_ala,1) = 10; // 3HB

//chu   base atom numbers ALA
	atom_base( 1, aa_ala,1) = 2; //  N  -> CA
	atom_base( 2, aa_ala,1) = 1; //  CA -> N
	atom_base( 3, aa_ala,1) = 2; //  C  -> CA
	atom_base( 4, aa_ala,1) = 3; //  0  -> C
	atom_base( 5, aa_ala,1) = 2; //  CB -> CA
	atom_base( 6, aa_ala,1) = 1; //  H  -> N
	atom_base( 7, aa_ala,1) = 2; //  HA -> CA
	atom_base( 8, aa_ala,1) = 5; // 1HB -> CB
	atom_base( 9, aa_ala,1) = 5; // 2HB -> CB
	atom_base(10, aa_ala,1) = 5; // 3HB -> CB

//bk   acceptor positions
	accpt_pos( 1, aa_ala, 1) =  4; //  O

//bk   hydrogens on each atom

	nhydrogens_on_atm( 1, aa_ala, 1) =  1; //  N
	hydrogens_on_atm( 1, 1, aa_ala, 1) =  6; //  H

	nhydrogens_on_atm( 2, aa_ala, 1) =  1; //  CA
	hydrogens_on_atm( 1, 2, aa_ala, 1) =  7; //  HA

	nhydrogens_on_atm( 5, aa_ala, 1) =  3; //  CB
	hydrogens_on_atm( 1, 5, aa_ala, 1) =  8; // 1HB
	hydrogens_on_atm( 2, 5, aa_ala, 1) =  9; // 2HB
	hydrogens_on_atm( 3, 5, aa_ala, 1) = 10; // 3HB

//bk   template coordinates for the amino acid
	icoor( 1, 1, aa_ala, 1) =    0.0000; //   N
	icoor( 2, 1, aa_ala, 1) =    1.5340; //   N
	icoor( 3, 1, aa_ala, 1) =    3.3340; //   N
	icoor( 1, 2, aa_ala, 1) =    0.0000; //   CA
	icoor( 2, 2, aa_ala, 1) =    2.8360; //   CA
	icoor( 3, 2, aa_ala, 1) =    3.9900; //   CA
	icoor( 1, 3, aa_ala, 1) =    0.0000; //   C
	icoor( 2, 3, aa_ala, 1) =    2.6880; //   C
	icoor( 3, 3, aa_ala, 1) =    5.5060; //   C
	icoor( 1, 4, aa_ala, 1) =    0.0020; //   O
	icoor( 2, 4, aa_ala, 1) =    1.5660; //   O
	icoor( 3, 4, aa_ala, 1) =    6.0130; //   O
	icoor( 1, 5, aa_ala, 1) =   -1.1960; //   CB
	icoor( 2, 5, aa_ala, 1) =    3.6590; //   CB
	icoor( 3, 5, aa_ala, 1) =    3.5340; //   CB
	icoor( 1, 6, aa_ala, 1) =   -0.0470; //   H
	icoor( 2, 6, aa_ala, 1) =    0.7070; //   H
	icoor( 3, 6, aa_ala, 1) =    3.9120; //   H
	icoor( 1, 7, aa_ala, 1) =    0.9120; //   HA
	icoor( 2, 7, aa_ala, 1) =    3.3660; //   HA
	icoor( 3, 7, aa_ala, 1) =    3.7150; //   HA
	icoor( 1, 8, aa_ala, 1) =   -1.1810; //  1HB
	icoor( 2, 8, aa_ala, 1) =    4.6280; //  1HB
	icoor( 3, 8, aa_ala, 1) =    4.0330; //  1HB
	icoor( 1, 9, aa_ala, 1) =   -1.1470; //  2HB
	icoor( 2, 9, aa_ala, 1) =    3.8060; //  2HB
	icoor( 3, 9, aa_ala, 1) =    2.4550; //  2HB
	icoor( 1,10, aa_ala, 1) =   -2.1160; //  3HB
	icoor( 2,10, aa_ala, 1) =    3.1350; //  3HB
	icoor( 3,10, aa_ala, 1) =    3.7880; //  3HB

	// H positions from REDUCE
	if( param_pack::packer_logical::reduce_templates ) {
		icoor( 1 , 6 , aa_ala , 1) = -0.148 ; // H
		icoor( 2 , 6 , aa_ala , 1) = 0.817 ; // H
		icoor( 3 , 6 , aa_ala , 1) = 4.015 ; // H
		icoor( 1 , 7 , aa_ala , 1) = 0.922 ; // HA
		icoor( 2 , 7 , aa_ala , 1) = 3.363 ; // HA
		icoor( 3 , 7 , aa_ala , 1) = 3.702 ; // HA
		icoor( 1 , 8 , aa_ala , 1) = -1.182 ; // 1HB
		icoor( 2 , 8 , aa_ala , 1) = 4.637 ; // 1HB
		icoor( 3 , 8 , aa_ala , 1) = 4.037 ; // 1HB
		icoor( 1 , 9 , aa_ala , 1) = -1.146 ; // 2HB
		icoor( 2 , 9 , aa_ala , 1) = 3.807 ; // 2HB
		icoor( 3 , 9 , aa_ala , 1) = 2.445 ; // 2HB
		icoor( 1 , 10 ,aa_ala , 1) = -2.125 ; // 3HB
		icoor( 2 , 10 ,aa_ala , 1) = 3.129 ; // 3HB
		icoor( 3 , 10 ,aa_ala , 1) = 3.790 ; // 3HB
	}


//bq create N and Cterm variants
	create_termini_variants( aa_ala );

//bk   create variants that have h2o connected to the backbone nh and carbonyl
	if ( explicit_h2o ) {
		create_backbone_h2o_variants(aa_ala);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_cys
///
/// @brief arrays filled to define the proerties of the amino acid Cysteine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// aa_cys - index number for CYS (2)
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/// 10 Oct 2007 Jack Snoeyink (JSS) Hg made apolar so it doesn't Hbond.
/////////////////////////////////////////////////////////////////////////////////
void
init_cys()
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

//bk   Parameters for CYS -----------------------------
	nvar( aa_cys ) = 1; // starting number of variants for CYS
	variant_type( aav_base, aa_cys, 1 ) = true; // type of aa variant

//bk   variant1 ---------- CYS

	natoms( aa_cys,1) = 11; // number of total atoms
	nheavyatoms( aa_cys,1) =  6; // number of heavy atoms
	nchi( aa_cys,1) =  1; // number of chi angles

//bk   atom names CYS
	atom_name( 1, aa_cys,1) = " N  ";
	atom_name( 2, aa_cys,1) = " CA ";
	atom_name( 3, aa_cys,1) = " C  ";
	atom_name( 4, aa_cys,1) = " O  ";
	atom_name( 5, aa_cys,1) = " CB ";
	atom_name( 6, aa_cys,1) = " SG ";
	atom_name( 7, aa_cys,1) = " H  ";
	atom_name( 8, aa_cys,1) = " HA ";
	atom_name( 9, aa_cys,1) = "2HB ";
	atom_name(10, aa_cys,1) = "3HB ";
	atom_name(11, aa_cys,1) = " HG ";

//jjh intra residue bonding
	nbonded_neighbors( 1, aa_cys, 1) = 2; // N
		bonded_neighbor(1, 1, aa_cys, 1) = 2; // N--CA
		bonded_neighbor(2, 1, aa_cys, 1) = 7; // N--H
	nbonded_neighbors( 2, aa_cys, 1) = 4; // CA
		bonded_neighbor(1, 2, aa_cys, 1) = 1; // CA--N
		bonded_neighbor(2, 2, aa_cys, 1) = 3; // CA--C
		bonded_neighbor(3, 2, aa_cys, 1) = 5; // CA--CB
		bonded_neighbor(4, 2, aa_cys, 1) = 8; // CA--HA
	nbonded_neighbors( 3, aa_cys, 1) = 2; // C
		bonded_neighbor(1, 3, aa_cys, 1) = 2; // C--CA
		bonded_neighbor(2, 3, aa_cys, 1) = 4; // C--O
	nbonded_neighbors( 4, aa_cys, 1) = 1; // O
		bonded_neighbor(1, 4, aa_cys, 1) = 3; // O--C
	nbonded_neighbors( 5, aa_cys, 1) = 4; // CB
		bonded_neighbor(1, 5, aa_cys, 1) = 2; // CB--CA
		bonded_neighbor(2, 5, aa_cys, 1) = 6; // CB--SG
		bonded_neighbor(3, 5, aa_cys, 1) = 9; // CB--2HB
		bonded_neighbor(4, 5, aa_cys, 1) =10; // CB--3HB
	nbonded_neighbors( 6, aa_cys, 1) = 2; // SG
		bonded_neighbor(1, 6, aa_cys, 1) = 5; // SG--CB
		bonded_neighbor(2, 6, aa_cys, 1) =11; // SG--HG1
	nbonded_neighbors( 7, aa_cys, 1) = 1; // H
		bonded_neighbor(1, 7, aa_cys, 1) = 1; // H--N
	nbonded_neighbors( 8, aa_cys, 1) = 1; // HA
		bonded_neighbor(1, 8, aa_cys, 1) = 2; // HA--CA
	nbonded_neighbors( 9, aa_cys, 1) = 1; //2HB
		bonded_neighbor(1, 9, aa_cys, 1) = 5; //2HB--CB
	nbonded_neighbors( 10, aa_cys, 1) = 1; //3HB
		bonded_neighbor(1,10, aa_cys, 1) = 5; //3HB--CB
	nbonded_neighbors( 11, aa_cys, 1) = 1; // HG
		bonded_neighbor(1,11, aa_cys, 1) = 6; // HG--SG

//jjh get the angles automatically
	get_angles_from_bonds(aa_cys, 1);

//chu   base atom numbers CYS
	atom_base( 1, aa_cys,1) = 2; //  N  -> CA
	atom_base( 2, aa_cys,1) = 1; //  CA -> N
	atom_base( 3, aa_cys,1) = 2; //  C  -> CA
	atom_base( 4, aa_cys,1) = 3; //  O  -> C
	atom_base( 5, aa_cys,1) = 2; //  CB -> CA
	atom_base( 6, aa_cys,1) = 5; //  SG -> CB
	atom_base( 7, aa_cys,1) = 1; //  H  -> N
	atom_base( 8, aa_cys,1) = 2; //  HA -> CA
	atom_base( 9, aa_cys,1) = 5; // 2HB -> CB
	atom_base(10, aa_cys,1) = 5; // 3HB -> CB
	atom_base(11, aa_cys,1) = 6; //  HG -> SG

//bk   atom type numbers CYS
	fullatom_type( 1, aa_cys,1) = 17; // Nbb    N
	fullatom_type( 2, aa_cys,1) = 18; // CAbb   CA
	fullatom_type( 3, aa_cys,1) = 19; // CObb   C
	fullatom_type( 4, aa_cys,1) = 20; // OCbb   O
	fullatom_type( 5, aa_cys,1) =  4; // CH2    CB
	fullatom_type( 6, aa_cys,1) = 16; // S      SG
	fullatom_type( 7, aa_cys,1) = 25; // HNbb   H
	fullatom_type( 8, aa_cys,1) = 23; // Hapo   HA
	fullatom_type( 9, aa_cys,1) = 23; // Hapo  2HB
	fullatom_type(10, aa_cys,1) = 23; // Hapo  3HB
	fullatom_type(11, aa_cys,1) = 23; // Hapo   HG

//cmd  atomic charge CYS
	atomic_charge( 1, aa_cys,1) = -0.47; //    N
	atomic_charge( 2, aa_cys,1) =  0.07; //    CA
	atomic_charge( 3, aa_cys,1) =  0.51; //    C
	atomic_charge( 4, aa_cys,1) = -0.51; //    O
	atomic_charge( 5, aa_cys,1) = -0.11; //    CB
	atomic_charge( 6, aa_cys,1) = -0.23; //    SG
	atomic_charge( 7, aa_cys,1) =  0.31; //    H
	atomic_charge( 8, aa_cys,1) =  0.09; //    HA
	atomic_charge( 9, aa_cys,1) =  0.09; //   2HB
	atomic_charge(10, aa_cys,1) =  0.09; //   3HB
	atomic_charge(11, aa_cys,1) =  0.16; //    HG

	assert_total_charge( aa_cys, 1, 0.0);

//cmd  first sidechain atom
	first_scatom( aa_cys,1) = 5; //CB

//bk   template atoms used for placing atoms CYS

//bk   template for building  HA
	ta(1, 8, aa_cys,1) =  2; //   CA
	ta(2, 8, aa_cys,1) =  1; //   N
	ta(3, 8, aa_cys,1) =  3; //   C

//bk   template for building 2HB
	ta(1, 9, aa_cys,1) =  5; //   CB
	ta(2, 9, aa_cys,1) =  2; //   CA
	ta(3, 9, aa_cys,1) =  6; //   SG

//bk   template for building 3HB
	ta(1,10, aa_cys,1) =  5; //   CB
	ta(2,10, aa_cys,1) =  2; //   CA
	ta(3,10, aa_cys,1) =  6; //   SG

//bk   template for building  HG
	ta(1,11, aa_cys,1) =  6; //   SG
	ta(2,11, aa_cys,1) =  5; //   CB
	ta(3,11, aa_cys,1) =  2; //   CA

//bk   chi angles required to build atoms CYS
//bk   chi angles needed for building  SG
	chi_required(1, 6, aa_cys,1) = true;

//bk   chi angles needed for building 2HB
	chi_required(1, 9, aa_cys,1) = true;

//bk   chi angles needed for building 3HB
	chi_required(1,10, aa_cys,1) = true;

//bk   chi angles needed for building  HG
	chi_required(1,11, aa_cys,1) = true;

//bk   four atoms that define chi angle  1
	chi_atoms( 1, 1, aa_cys, 1) =  1; //   N
	chi_atoms( 2, 1, aa_cys, 1) =  2; //   CA
	chi_atoms( 3, 1, aa_cys, 1) =  5; //   CB
	chi_atoms( 4, 1, aa_cys, 1) =  6; //   SG

	HNpos( aa_cys,1) =  7; // position of backbone HN
	HApos( aa_cys,1) =  8; // position of backbone HA
	nH_polar( aa_cys,1) =  1; // number of polar hydrogens
	nH_aromatic( aa_cys,1) =  0; // number of aromatic hydrogens
	nH_apolar( aa_cys,1) =  4; // number of apolar hydrogens
	nacceptors( aa_cys,1) =  1; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1, aa_cys,1) =  7; //  H
//JSS moved HG to apolar list below for consistency with etables & observations in natives.
//bk   atom numbers foraromatic hydrogens

//bk   atom numbers forapolar hydrogens
	Hpos_apolar( 1, aa_cys,1) =  8; //  HA
	Hpos_apolar( 2, aa_cys,1) =  9; // 2HB
	Hpos_apolar( 3, aa_cys,1) = 10; // 3HB
	Hpos_apolar( 4, aa_cys,1) = 11; //  HG

//bk   acceptor positions
	accpt_pos( 1, aa_cys, 1) =  4; //  O

//bk   hydrogens on each atom

	nhydrogens_on_atm( 1, aa_cys, 1) =  1; //  N
	hydrogens_on_atm( 1, 1, aa_cys, 1) =  7; //  H

	nhydrogens_on_atm( 2, aa_cys, 1) =  1; //  CA
	hydrogens_on_atm( 1, 2, aa_cys, 1) =  8; //  HA

	nhydrogens_on_atm( 5, aa_cys, 1) =  2; //  CB
	hydrogens_on_atm( 1, 5, aa_cys, 1) =  9; // 2HB
	hydrogens_on_atm( 2, 5, aa_cys, 1) = 10; // 3HB

	nhydrogens_on_atm( 6, aa_cys, 1) =  1; //  SG
	hydrogens_on_atm( 1, 6, aa_cys, 1) = 11; //  HG

//bk   template coordinates for the amino acid
	icoor( 1, 1, aa_cys, 1) =    0.0000; //   N
	icoor( 2, 1, aa_cys, 1) =    3.8070; //   N
	icoor( 3, 1, aa_cys, 1) =    6.2240; //   N
	icoor( 1, 2, aa_cys, 1) =    0.0000; //   CA
	icoor( 2, 2, aa_cys, 1) =    3.7830; //   CA
	icoor( 3, 2, aa_cys, 1) =    7.6880; //   CA
	icoor( 1, 3, aa_cys, 1) =    0.0000; //   C
	icoor( 2, 3, aa_cys, 1) =    5.1680; //   C
	icoor( 3, 3, aa_cys, 1) =    8.3290; //   C
	icoor( 1, 4, aa_cys, 1) =    0.0050; //   O
	icoor( 2, 4, aa_cys, 1) =    6.1860; //   O
	icoor( 3, 4, aa_cys, 1) =    7.6360; //   O
	icoor( 1, 5, aa_cys, 1) =    1.3120; //   CB
	icoor( 2, 5, aa_cys, 1) =    3.0690; //   CB
	icoor( 3, 5, aa_cys, 1) =    8.0140; //   CB
	icoor( 1, 6, aa_cys, 1) =    1.6350; //   SG
	icoor( 2, 6, aa_cys, 1) =    2.8650; //   SG
	icoor( 3, 6, aa_cys, 1) =    9.7820; //   SG
	icoor( 1, 7, aa_cys, 1) =   -0.0250; //   H
	icoor( 2, 7, aa_cys, 1) =    4.7040; //   H
	icoor( 3, 7, aa_cys, 1) =    5.7600; //   H
	icoor( 1, 8, aa_cys, 1) =   -0.8240; //   HA
	icoor( 2, 8, aa_cys, 1) =    3.2090; //   HA
	icoor( 3, 8, aa_cys, 1) =    8.1120; //   HA
	icoor( 1, 9, aa_cys, 1) =    1.3110; //  2HB
	icoor( 2, 9, aa_cys, 1) =    2.0650; //  2HB
	icoor( 3, 9, aa_cys, 1) =    7.5890; //  2HB
	icoor( 1,10, aa_cys, 1) =    2.1560; //  3HB
	icoor( 2,10, aa_cys, 1) =    3.6320; //  3HB
	icoor( 3,10, aa_cys, 1) =    7.6160; //  3HB
	icoor( 1,11, aa_cys, 1) =    2.7990; //   HG
	icoor( 2,11, aa_cys, 1) =    2.2390; //   HG
	icoor( 3,11, aa_cys, 1) =    9.6390; //   HG

	// H positions from REDUCE
	if( param_pack::packer_logical::reduce_templates ) {
		icoor( 1 , 7 , aa_cys , 1) = -0.033 ; // H
		icoor( 2 , 7 , aa_cys , 1) = 4.754 ; // H
		icoor( 3 , 7 , aa_cys , 1) = 5.906 ; // H
		icoor( 1 , 8 , aa_cys , 1) = -0.909 ; // HA
		icoor( 2 , 8 , aa_cys , 1) = 3.299 ; // HA
		icoor( 3 , 8 , aa_cys , 1) = 8.076 ; // HA
		icoor( 1 , 9 , aa_cys , 1) = 1.303 ; // 1HB
		icoor( 2 , 9 , aa_cys , 1) = 2.077 ; // 1HB
		icoor( 3 , 9 , aa_cys , 1) = 7.539 ; // 1HB
		icoor( 1 , 10 , aa_cys , 1) = 2.143 ; // 2HB
		icoor( 2 , 10 , aa_cys , 1) = 3.633 ; // 2HB
		icoor( 3 , 10 , aa_cys , 1) = 7.565 ; // 2HB
		icoor( 1 , 11 , aa_cys , 1) = 2.767 ; // HG
		icoor( 2 , 11 , aa_cys , 1) = 2.248 ; // HG
		icoor( 3 , 11 , aa_cys , 1) = 9.948 ; // HG
	}

//bq create N and Cterm variants
	create_termini_variants( aa_cys );

//bk   create variants that have h2o connected to the backbone nh and carbonyl
	if ( explicit_h2o ) {
		create_backbone_h2o_variants(aa_cys);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_asp
///
/// @brief arrays filled to define the proerties of the amino acid Aspartate
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// aa_asp - index number for ASP (3)
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_asp()
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

//bk   Parameters for ASP -----------------------------
	nvar( aa_asp ) = 1; // starting number of variants for ASP
	variant_type( aav_base, aa_asp, 1 ) = true; // type of aa variant

//bk   variant1 ---------- ASP

	natoms( aa_asp,1) = 12; // number of total atoms
	nheavyatoms( aa_asp,1) =  8; // number of heavy atoms
	nchi( aa_asp,1) =  2; // number of chi angles

//bk   atom names ASP
	atom_name( 1, aa_asp,1) = " N  ";
	atom_name( 2, aa_asp,1) = " CA ";
	atom_name( 3, aa_asp,1) = " C  ";
	atom_name( 4, aa_asp,1) = " O  ";
	atom_name( 5, aa_asp,1) = " CB ";
	atom_name( 6, aa_asp,1) = " CG ";
	atom_name( 7, aa_asp,1) = " OD1";
	atom_name( 8, aa_asp,1) = " OD2";
	atom_name( 9, aa_asp,1) = " H  ";
	atom_name(10, aa_asp,1) = " HA ";
	atom_name(11, aa_asp,1) = "2HB ";
	atom_name(12, aa_asp,1) = "3HB ";

//jjh intra residue bonding
	nbonded_neighbors( 1, aa_asp, 1) = 2; // N
		bonded_neighbor(1, 1, aa_asp, 1) = 2; // N--CA
		bonded_neighbor(2, 1, aa_asp, 1) = 9; // N--H
	nbonded_neighbors( 2, aa_asp, 1) = 4; // CA
		bonded_neighbor(1, 2, aa_asp, 1) = 1; // CA--N
		bonded_neighbor(2, 2, aa_asp, 1) = 3; // CA--C
		bonded_neighbor(3, 2, aa_asp, 1) = 5; // CA--CB
		bonded_neighbor(4, 2, aa_asp, 1) =10; // CA--HA
	nbonded_neighbors( 3, aa_asp, 1) = 2; // C
		bonded_neighbor(1, 3, aa_asp, 1) = 2; // C--CA
		bonded_neighbor(2, 3, aa_asp, 1) = 4; // C--O
	nbonded_neighbors( 4, aa_asp, 1) = 1; // O
		bonded_neighbor(1, 4, aa_asp, 1) = 3; // O--C
	nbonded_neighbors( 5, aa_asp, 1) = 4; // CB
		bonded_neighbor(1, 5, aa_asp, 1) = 2; // CB--CA
		bonded_neighbor(2, 5, aa_asp, 1) = 6; // CB--CG
		bonded_neighbor(3, 5, aa_asp, 1) =11; // CB--2HB
		bonded_neighbor(4, 5, aa_asp, 1) =12; // CB--3HB
	nbonded_neighbors( 6, aa_asp, 1) = 3; // CG
		bonded_neighbor(1, 6, aa_asp, 1) = 5; // CG--CB
		bonded_neighbor(2, 6, aa_asp, 1) = 7; // CG--OD1
		bonded_neighbor(3, 6, aa_asp, 1) = 8; // CG--OD2
	nbonded_neighbors( 7, aa_asp, 1) = 1; // OD1
		bonded_neighbor(1, 7, aa_asp, 1) = 6; // OD1--CG
	nbonded_neighbors( 8, aa_asp, 1) = 1; // OD2
		bonded_neighbor(1, 8, aa_asp, 1) = 6; // OD2--CG
	nbonded_neighbors( 9, aa_asp, 1) = 1; // H
		bonded_neighbor(1, 9, aa_asp, 1) = 1; // H--N
	nbonded_neighbors( 10, aa_asp, 1) = 1; // HA
		bonded_neighbor(1,10, aa_asp, 1) = 2; // HA--CA
	nbonded_neighbors( 11, aa_asp, 1) = 1; //2HB
		bonded_neighbor(1,11, aa_asp, 1) = 5; //2HB--CB
	nbonded_neighbors( 12, aa_asp, 1) = 1; //3HB
		bonded_neighbor(1,12, aa_asp, 1) = 5; //3HB--CB

//jjh get the angles automatically
	get_angles_from_bonds(aa_asp, 1);

//chu   base atom numbers ASP
	atom_base( 1, aa_asp,1) = 2; //  N  -> CA
	atom_base( 2, aa_asp,1) = 1; //  CA -> N
	atom_base( 3, aa_asp,1) = 2; //  C  -> CA
	atom_base( 4, aa_asp,1) = 3; //  O  -> C
	atom_base( 5, aa_asp,1) = 2; //  CB -> CA
	atom_base( 6, aa_asp,1) = 5; //  CG -> CB
	atom_base( 7, aa_asp,1) = 6; //  OD1-> CG
	atom_base( 8, aa_asp,1) = 6; //  OD2-> CG
	atom_base( 9, aa_asp,1) = 1; //  H  -> N
	atom_base(10, aa_asp,1) = 2; //  HA -> CA
	atom_base(11, aa_asp,1) = 5; // 2HB -> CB
	atom_base(12, aa_asp,1) = 5; // 3HB -> CB

//bk   atom type numbers ASP
	fullatom_type( 1, aa_asp,1) = 17; // Nbb    N
	fullatom_type( 2, aa_asp,1) = 18; // CAbb   CA
	fullatom_type( 3, aa_asp,1) = 19; // CObb   C
	fullatom_type( 4, aa_asp,1) = 20; // OCbb   O
	fullatom_type( 5, aa_asp,1) =  4; // CH2    CB
	fullatom_type( 6, aa_asp,1) =  2; // COO    CG
	fullatom_type( 7, aa_asp,1) = 15; // OOC    OD1
	fullatom_type( 8, aa_asp,1) = 15; // OOC    OD2
	fullatom_type( 9, aa_asp,1) = 25; // HNbb   H
	fullatom_type(10, aa_asp,1) = 23; // Hapo   HA
	fullatom_type(11, aa_asp,1) = 23; // Hapo  2HB
	fullatom_type(12, aa_asp,1) = 23; // Hapo  3HB

//cmd  atomic charge ASP
	atomic_charge( 1, aa_asp,1) = -0.47; //    N
	atomic_charge( 2, aa_asp,1) =  0.07; //    CA
	atomic_charge( 3, aa_asp,1) =  0.51; //    C
	atomic_charge( 4, aa_asp,1) = -0.51; //    O
	atomic_charge( 5, aa_asp,1) = -0.28; //    CB
	atomic_charge( 6, aa_asp,1) =  0.62; //    CG
	atomic_charge( 7, aa_asp,1) = -0.76; //    OD1
	atomic_charge( 8, aa_asp,1) = -0.76; //    OD2
	atomic_charge( 9, aa_asp,1) =  0.31; //    H
	atomic_charge(10, aa_asp,1) =  0.09; //    HA
	atomic_charge(11, aa_asp,1) =  0.09; //   2HB
	atomic_charge(12, aa_asp,1) =  0.09; //   3HB

	assert_total_charge( aa_asp, 1, -1.0);

//cmd  first sidechain atom
	first_scatom( aa_asp,1) = 5; //CB

//bk   template atoms used for placing atoms ASP

//bk   template for building  HA
	ta(1,10, aa_asp,1) =  2; //   CA
	ta(2,10, aa_asp,1) =  1; //   N
	ta(3,10, aa_asp,1) =  3; //   C

//bk   template for building 2HB
	ta(1,11, aa_asp,1) =  5; //   CB
	ta(2,11, aa_asp,1) =  2; //   CA
	ta(3,11, aa_asp,1) =  6; //   CG

//bk   template for building 3HB
	ta(1,12, aa_asp,1) =  5; //   CB
	ta(2,12, aa_asp,1) =  2; //   CA
	ta(3,12, aa_asp,1) =  6; //   CG

//bk   chi angles required to build atoms ASP
//bk   chi angles needed for building  CG
	chi_required(1, 6, aa_asp,1) =  true;

//bk   chi angles needed for building  OD1
	chi_required(1, 7, aa_asp,1) =  true;
	chi_required(2, 7, aa_asp,1) =  true;

//bk   chi angles needed for building  OD2
	chi_required(1, 8, aa_asp,1) =  true;
	chi_required(2, 8, aa_asp,1) =  true;

//bk   chi angles needed for building 2HB
	chi_required(1,11, aa_asp,1) =  true;

//bk   chi angles needed for building 3HB
	chi_required(1,12, aa_asp,1) =  true;

//bk   four atoms that define chi angle  1
	chi_atoms( 1, 1, aa_asp, 1) =  1; //   N
	chi_atoms( 2, 1, aa_asp, 1) =  2; //   CA
	chi_atoms( 3, 1, aa_asp, 1) =  5; //   CB
	chi_atoms( 4, 1, aa_asp, 1) =  6; //   CG
//bk   four atoms that define chi angle  2
	chi_atoms( 1, 2, aa_asp, 1) =  2; //   CA
	chi_atoms( 2, 2, aa_asp, 1) =  5; //   CB
	chi_atoms( 3, 2, aa_asp, 1) =  6; //   CG
	chi_atoms( 4, 2, aa_asp, 1) =  7; //   OD1

	HNpos( aa_asp,1) =  9; // atom numbers forbackbone HN
	HApos( aa_asp,1) = 10; // atom numbers forbackbone HA
	nH_polar( aa_asp,1) =  1; // number of polar hydrogens
	nH_aromatic( aa_asp,1) =  0; // number of aromatic hydrogens
	nH_apolar( aa_asp,1) =  3; // number of apolar hydrogens
	nacceptors( aa_asp,1) =  3; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1, aa_asp,1) =  9; //  H

//bk   atom numbers for aromatic hydrogens

//bk   atom numbers forapolar hydrogens
	Hpos_apolar( 1, aa_asp,1) = 10; //  HA
	Hpos_apolar( 2, aa_asp,1) = 11; // 2HB
	Hpos_apolar( 3, aa_asp,1) = 12; // 3HB

//bk   acceptor positions
	accpt_pos( 1, aa_asp, 1) =  4; //  O
	accpt_pos( 2, aa_asp, 1) =  7; //  OD1
	accpt_pos( 3, aa_asp, 1) =  8; //  OD2

//bk   hydrogens on each atom

	nhydrogens_on_atm( 1, aa_asp, 1) =  1; //  N
	hydrogens_on_atm( 1, 1, aa_asp, 1) =  9; //  H

	nhydrogens_on_atm( 2, aa_asp, 1) =  1; //  CA
	hydrogens_on_atm( 1, 2, aa_asp, 1) = 10; //  HA

	nhydrogens_on_atm( 5, aa_asp, 1) =  2; //  CB
	hydrogens_on_atm( 1, 5, aa_asp, 1) = 11; // 2HB
	hydrogens_on_atm( 2, 5, aa_asp, 1) = 12; // 3HB

//bk   template coordinates for the amino acid
	icoor( 1, 1, aa_asp, 1) =    0.0000; //   N
	icoor( 2, 1, aa_asp, 1) =    5.2000; //   N
	icoor( 3, 1, aa_asp, 1) =    9.6580; //   N
	icoor( 1, 2, aa_asp, 1) =    0.0000; //   CA
	icoor( 2, 2, aa_asp, 1) =    6.4570; //   CA
	icoor( 3, 2, aa_asp, 1) =   10.3940; //   CA
	icoor( 1, 3, aa_asp, 1) =    0.0000; //   C
	icoor( 2, 3, aa_asp, 1) =    6.2160; //   C
	icoor( 3, 3, aa_asp, 1) =   11.8980; //   C
	icoor( 1, 4, aa_asp, 1) =    0.0010; //   O
	icoor( 2, 4, aa_asp, 1) =    5.0720; //   O
	icoor( 3, 4, aa_asp, 1) =   12.3530; //   O
	icoor( 1, 5, aa_asp, 1) =   -1.2080; //   CB
	icoor( 2, 5, aa_asp, 1) =    7.3100; //   CB
	icoor( 3, 5, aa_asp, 1) =    9.9990; //   CB
	icoor( 1, 6, aa_asp, 1) =   -1.2950; //   CG
	icoor( 2, 6, aa_asp, 1) =    8.6490; //   CG
	icoor( 3, 6, aa_asp, 1) =   10.7190; //   CG
	icoor( 1, 7, aa_asp, 1) =   -2.2210; //   OD1
	icoor( 2, 7, aa_asp, 1) =    9.3810; //   OD1
	icoor( 3, 7, aa_asp, 1) =   10.4610; //   OD1
	icoor( 1, 8, aa_asp, 1) =   -0.5300; //   OD2
	icoor( 2, 8, aa_asp, 1) =    8.8620; //   OD2
	icoor( 3, 8, aa_asp, 1) =   11.6290; //   OD2
	icoor( 1, 9, aa_asp, 1) =   -0.0400; //   H
	icoor( 2, 9, aa_asp, 1) =    4.3370; //   H
	icoor( 3, 9, aa_asp, 1) =   10.1830; //   H
	icoor( 1,10, aa_asp, 1) =    0.9090; //   HA
	icoor( 2,10, aa_asp, 1) =    7.0160; //   HA
	icoor( 3,10, aa_asp, 1) =   10.1700; //   HA
	icoor( 1,11, aa_asp, 1) =   -1.2940; //  2HB
	icoor( 2,11, aa_asp, 1) =    7.4700; //  2HB
	icoor( 3,11, aa_asp, 1) =    8.9240; //  2HB
	icoor( 1,12, aa_asp, 1) =   -2.0180; //  3HB
	icoor( 2,12, aa_asp, 1) =    6.6630; //  3HB
	icoor( 3,12, aa_asp, 1) =   10.3370; //  3HB

	// H positions from REDUCE
	if( param_pack::packer_logical::reduce_templates ) {
		icoor( 1 , 9 , aa_asp , 1) = -0.000 ; // H
		icoor( 2 , 9 , aa_asp , 1) = 4.436 ; // H
		icoor( 3 , 9 , aa_asp , 1) = 10.303 ; // H
		icoor( 1 , 10 , aa_asp , 1) = 0.921 ; // HA
		icoor( 2 , 10 , aa_asp , 1) = 6.998 ; // HA
		icoor( 3 , 10 , aa_asp , 1) = 10.132 ; // HA
		icoor( 1 , 11 , aa_asp , 1) = -1.174 ; // 1HB
		icoor( 2 , 11 , aa_asp , 1) = 7.492 ; // 1HB
		icoor( 3 , 11 , aa_asp , 1) = 8.915 ; // 1HB
		icoor( 1 , 12 , aa_asp , 1) = -2.126 ; // 2HB
		icoor( 2 , 12 , aa_asp , 1) = 6.739 ; // 2HB
		icoor( 3 , 12 , aa_asp , 1) = 10.201 ; // 2HB
	}

	//rh create protonation state variants
	create_pH_variants( aa_asp);

	//bq create N and Cterm variants
	create_termini_variants( aa_asp );

//bk   create variants that have h2o connected to the backbone nh and carbonyl
	if ( explicit_h2o ) {
		create_backbone_h2o_variants(aa_asp);
		create_sidechain_h2o_variants(aa_asp);
	}


}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_glu
///
/// @brief arrays filled to define the proerties of the amino acid Glutamate
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// aa_glu - index number for GLU (4)
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_glu()
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

//bk   Parameters for GLU -----------------------------
	nvar( aa_glu ) = 1; // starting number of variants for GLU
	variant_type( aav_base, aa_glu, 1 ) = true; // type of aa variant

//bk   variant1 ---------- GLU

	natoms( aa_glu,1) = 15; // number of total atoms
	nheavyatoms( aa_glu,1) =  9; // number of heavy atoms
	nchi( aa_glu,1) =  3; // number of chi angles

//bk   atom names GLU
	atom_name( 1, aa_glu,1) = " N  ";
	atom_name( 2, aa_glu,1) = " CA ";
	atom_name( 3, aa_glu,1) = " C  ";
	atom_name( 4, aa_glu,1) = " O  ";
	atom_name( 5, aa_glu,1) = " CB ";
	atom_name( 6, aa_glu,1) = " CG ";
	atom_name( 7, aa_glu,1) = " CD ";
	atom_name( 8, aa_glu,1) = " OE1";
	atom_name( 9, aa_glu,1) = " OE2";
	atom_name(10, aa_glu,1) = " H  ";
	atom_name(11, aa_glu,1) = " HA ";
	atom_name(12, aa_glu,1) = "2HB ";
	atom_name(13, aa_glu,1) = "3HB ";
	atom_name(14, aa_glu,1) = "2HG ";
	atom_name(15, aa_glu,1) = "3HG ";

//jjh intra residue bonding
	nbonded_neighbors( 1, aa_glu, 1) = 2; // N
		bonded_neighbor(1, 1, aa_glu, 1) = 2; // N--CA
		bonded_neighbor(2, 1, aa_glu, 1) =10; // N--H
	nbonded_neighbors( 2, aa_glu, 1) = 4; // CA
		bonded_neighbor(1, 2, aa_glu, 1) = 1; // CA--N
		bonded_neighbor(2, 2, aa_glu, 1) = 3; // CA--C
		bonded_neighbor(3, 2, aa_glu, 1) = 5; // CA--CB
		bonded_neighbor(4, 2, aa_glu, 1) =11; // CA--HA
	nbonded_neighbors( 3, aa_glu, 1) = 2; // C
		bonded_neighbor(1, 3, aa_glu, 1) = 2; // C--CA
		bonded_neighbor(2, 3, aa_glu, 1) = 4; // C--O
	nbonded_neighbors( 4, aa_glu, 1) = 1; // O
		bonded_neighbor(1, 4, aa_glu, 1) = 3; // O--C
	nbonded_neighbors( 5, aa_glu, 1) = 4; // CB
		bonded_neighbor(1, 5, aa_glu, 1) = 2; // CB--CA
		bonded_neighbor(2, 5, aa_glu, 1) = 6; // CB--CG
		bonded_neighbor(3, 5, aa_glu, 1) =12; // CB--2HB
		bonded_neighbor(4, 5, aa_glu, 1) =13; // CB--3HB
	nbonded_neighbors( 6, aa_glu, 1) = 4; // CG
		bonded_neighbor(1, 6, aa_glu, 1) = 5; // CG--CB
		bonded_neighbor(2, 6, aa_glu, 1) = 7; // CG--CD
		bonded_neighbor(3, 6, aa_glu, 1) =14; // CG--2HG
		bonded_neighbor(4, 6, aa_glu, 1) =15; // CG--3HG
	nbonded_neighbors( 7, aa_glu, 1) = 3; // CD
		bonded_neighbor(1, 7, aa_glu, 1) = 6; // CD--CG
		bonded_neighbor(2, 7, aa_glu, 1) = 8; // CD--OE1
		bonded_neighbor(3, 7, aa_glu, 1) = 9; // CD--OE2
	nbonded_neighbors( 8, aa_glu, 1) = 1; // OE1
		bonded_neighbor(1, 8, aa_glu, 1) = 7; // OE1--CD
	nbonded_neighbors( 9, aa_glu, 1) = 1; // OE2
		bonded_neighbor(1, 9, aa_glu, 1) = 7; // OE2--CD
	nbonded_neighbors( 10, aa_glu, 1) = 1; // H
		bonded_neighbor(1,10, aa_glu, 1) = 1; // H--N
	nbonded_neighbors( 11, aa_glu, 1) = 1; // HA
		bonded_neighbor(1,11, aa_glu, 1) = 2; // HA--CA
	nbonded_neighbors( 12, aa_glu, 1) = 1; //2HB
		bonded_neighbor(1,12, aa_glu, 1) = 5; //2HB--CB
	nbonded_neighbors( 13, aa_glu, 1) = 1; //3HB
		bonded_neighbor(1,13, aa_glu, 1) = 5; //3HB--CB
	nbonded_neighbors( 14, aa_glu, 1) = 1; //2HG
		bonded_neighbor(1,14, aa_glu, 1) = 6; //2HG--CG
	nbonded_neighbors( 15, aa_glu, 1) = 1; //3HG
		bonded_neighbor(1,15, aa_glu, 1) = 6; //3HG--CG

//jjh get the angles automatically
	get_angles_from_bonds(aa_glu, 1);

//chu   base atom numbers GLU
	atom_base( 1, aa_glu,1) = 2; //  N  -> CA
	atom_base( 2, aa_glu,1) = 1; //  CA -> N
	atom_base( 3, aa_glu,1) = 2; //  C  -> CA
	atom_base( 4, aa_glu,1) = 3; //  O  -> C
	atom_base( 5, aa_glu,1) = 2; //  CB -> CA
	atom_base( 6, aa_glu,1) = 5; //  CG -> CB
	atom_base( 7, aa_glu,1) = 6; //  CD -> CG
	atom_base( 8, aa_glu,1) = 7; //  OE1-> CD
	atom_base( 9, aa_glu,1) = 7; //  OE2-> CD
	atom_base(10, aa_glu,1) = 1; //  H  -> N
	atom_base(11, aa_glu,1) = 2; //  HA -> CA
	atom_base(12, aa_glu,1) = 5; // 2HB -> CB
	atom_base(13, aa_glu,1) = 5; // 3HB -> CB
	atom_base(14, aa_glu,1) = 6; // 2HG -> CG
	atom_base(15, aa_glu,1) = 6; // 3HG -> CG

//bk   atom type numbers GLU
	fullatom_type( 1, aa_glu,1) = 17; // Nbb    N
	fullatom_type( 2, aa_glu,1) = 18; // CAbb   CA
	fullatom_type( 3, aa_glu,1) = 19; // CObb   C
	fullatom_type( 4, aa_glu,1) = 20; // OCbb   O
	fullatom_type( 5, aa_glu,1) =  4; // CH2    CB
	fullatom_type( 6, aa_glu,1) =  4; // CH2    CG
	fullatom_type( 7, aa_glu,1) =  2; // COO    CD
	fullatom_type( 8, aa_glu,1) = 15; // OOC    OE1
	fullatom_type( 9, aa_glu,1) = 15; // OOC    OE2
	fullatom_type(10, aa_glu,1) = 25; // HNbb   H
	fullatom_type(11, aa_glu,1) = 23; // Hapo   HA
	fullatom_type(12, aa_glu,1) = 23; // Hapo  2HB
	fullatom_type(13, aa_glu,1) = 23; // Hapo  3HB
	fullatom_type(14, aa_glu,1) = 23; // Hapo  2HG
	fullatom_type(15, aa_glu,1) = 23; // Hapo  3HG

//cmd  atomic charge GLU
	atomic_charge( 1, aa_glu,1) = -0.47; //    N
	atomic_charge( 2, aa_glu,1) =  0.07; //    CA
	atomic_charge( 3, aa_glu,1) =  0.51; //    C
	atomic_charge( 4, aa_glu,1) = -0.51; //    O
	atomic_charge( 5, aa_glu,1) = -0.18; //    CB
	atomic_charge( 6, aa_glu,1) = -0.28; //    CG
	atomic_charge( 7, aa_glu,1) =  0.62; //    CD
	atomic_charge( 8, aa_glu,1) = -0.76; //    OE1
	atomic_charge( 9, aa_glu,1) = -0.76; //    OE2
	atomic_charge(10, aa_glu,1) =  0.31; //    H
	atomic_charge(11, aa_glu,1) =  0.09; //    HA
	atomic_charge(12, aa_glu,1) =  0.09; //   2HB
	atomic_charge(13, aa_glu,1) =  0.09; //   3HB
	atomic_charge(14, aa_glu,1) =  0.09; //   2HG
	atomic_charge(15, aa_glu,1) =  0.09; //   3HG

	assert_total_charge( aa_glu, 1, -1.0);

//cmd  first sidechain atom
	first_scatom( aa_glu,1) = 5; //CB

//bk   template atoms used for placing atoms GLU

//bk   template for building  HA
	ta(1,11, aa_glu,1) =  2; //   CA
	ta(2,11, aa_glu,1) =  1; //   N
	ta(3,11, aa_glu,1) =  3; //   C

//bk   template for building 2HB
	ta(1,12, aa_glu,1) =  5; //   CB
	ta(2,12, aa_glu,1) =  2; //   CA
	ta(3,12, aa_glu,1) =  6; //   CG

//bk   template for building 3HB
	ta(1,13, aa_glu,1) =  5; //   CB
	ta(2,13, aa_glu,1) =  2; //   CA
	ta(3,13, aa_glu,1) =  6; //   CG

//bk   template for building 2HG
	ta(1,14, aa_glu,1) =  6; //   CG
	ta(2,14, aa_glu,1) =  5; //   CB
	ta(3,14, aa_glu,1) =  7; //   CD

//bk   template for building 3HG
	ta(1,15, aa_glu,1) =  6; //   CG
	ta(2,15, aa_glu,1) =  5; //   CB
	ta(3,15, aa_glu,1) =  7; //   CD

//bk   chi angles required to build atoms GLU
//bk   chi angles needed for building  CG
	chi_required(1, 6, aa_glu,1) =  true;

//bk   chi angles needed for building  CD
	chi_required(1, 7, aa_glu,1) =  true;
	chi_required(2, 7, aa_glu,1) =  true;

//bk   chi angles needed for building  OE1
	chi_required(1, 8, aa_glu,1) =  true;
	chi_required(2, 8, aa_glu,1) =  true;
	chi_required(3, 8, aa_glu,1) =  true;

//bk   chi angles needed for building  OE2
	chi_required(1, 9, aa_glu,1) =  true;
	chi_required(2, 9, aa_glu,1) =  true;
	chi_required(3, 9, aa_glu,1) =  true;

//bk   chi angles needed for building 2HB
	chi_required(1,12, aa_glu,1) =  true;

//bk   chi angles needed for building 3HB
	chi_required(1,13, aa_glu,1) =  true;

//bk   chi angles needed for building 2HG
	chi_required(1,14, aa_glu,1) =  true;
	chi_required(2,14, aa_glu,1) =  true;

//bk   chi angles needed for building 3HG
	chi_required(1,15, aa_glu,1) =  true;
	chi_required(2,15, aa_glu,1) =  true;

//bk   four atoms that define chi angle  1
	chi_atoms( 1, 1, aa_glu, 1) =  1; //   N
	chi_atoms( 2, 1, aa_glu, 1) =  2; //   CA
	chi_atoms( 3, 1, aa_glu, 1) =  5; //   CB
	chi_atoms( 4, 1, aa_glu, 1) =  6; //   CG
//bk   four atoms that define chi angle  2
	chi_atoms( 1, 2, aa_glu, 1) =  2; //   CA
	chi_atoms( 2, 2, aa_glu, 1) =  5; //   CB
	chi_atoms( 3, 2, aa_glu, 1) =  6; //   CG
	chi_atoms( 4, 2, aa_glu, 1) =  7; //   CD
//bk   four atoms that define chi angle  3
	chi_atoms( 1, 3, aa_glu, 1) =  5; //   CB
	chi_atoms( 2, 3, aa_glu, 1) =  6; //   CG
	chi_atoms( 3, 3, aa_glu, 1) =  7; //   CD
	chi_atoms( 4, 3, aa_glu, 1) =  8; //   OE1

	HNpos( aa_glu,1) = 10; // atom numbers forbackbone HN
	HApos( aa_glu,1) = 11; // atom numbers forbackbone HA
	nH_polar( aa_glu,1) =  1; // number of polar hydrogens
	nH_aromatic( aa_glu,1) =  0; // number of aromatic hydrogens
	nH_apolar( aa_glu,1) =  5; // number of apolar hydrogens
	nacceptors( aa_glu,1) =  3; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1, aa_glu,1) = 10; //  H

//bk   atom number for aromatic hydrogens

//bk   atom number for apolar hydrogens
	Hpos_apolar( 1, aa_glu,1) = 11; //  HA
	Hpos_apolar( 2, aa_glu,1) = 12; // 2HB
	Hpos_apolar( 3, aa_glu,1) = 13; // 3HB
	Hpos_apolar( 4, aa_glu,1) = 14; // 2HG
	Hpos_apolar( 5, aa_glu,1) = 15; // 3HG

//bk   acceptor positions
	accpt_pos( 1, aa_glu, 1) =  4; //  O
	accpt_pos( 2, aa_glu, 1) =  8; //  OE1
	accpt_pos( 3, aa_glu, 1) =  9; //  OE2

//bk   hydrogens on each atom

	nhydrogens_on_atm( 1, aa_glu, 1) =  1; //  N
	hydrogens_on_atm( 1, 1, aa_glu, 1) = 10; //  H

	nhydrogens_on_atm( 2, aa_glu, 1) =  1; //  CA
	hydrogens_on_atm( 1, 2, aa_glu, 1) = 11; //  HA

	nhydrogens_on_atm( 5, aa_glu, 1) =  2; //  CB
	hydrogens_on_atm( 1, 5, aa_glu, 1) = 12; // 2HB
	hydrogens_on_atm( 2, 5, aa_glu, 1) = 13; // 3HB

	nhydrogens_on_atm( 6, aa_glu, 1) =  2; //  CG
	hydrogens_on_atm( 1, 6, aa_glu, 1) = 14; // 2HG
	hydrogens_on_atm( 2, 6, aa_glu, 1) = 15; // 3HG

//bk   template coordinates for the amino acid
	icoor( 1, 1, aa_glu, 1) =    0.0000; //   N
	icoor( 2, 1, aa_glu, 1) =    7.3000; //   N
	icoor( 3, 1, aa_glu, 1) =   12.6660; //   N
	icoor( 1, 2, aa_glu, 1) =    0.0000; //   CA
	icoor( 2, 2, aa_glu, 1) =    7.2080; //   CA
	icoor( 3, 2, aa_glu, 1) =   14.1210; //   CA
	icoor( 1, 3, aa_glu, 1) =    0.0000; //   C
	icoor( 2, 3, aa_glu, 1) =    8.5910; //   C
	icoor( 3, 3, aa_glu, 1) =   14.7610; //   C
	icoor( 1, 4, aa_glu, 1) =   -0.0030; //   O
	icoor( 2, 4, aa_glu, 1) =    9.6070; //   O
	icoor( 3, 4, aa_glu, 1) =   14.0660; //   O
	icoor( 1, 5, aa_glu, 1) =    1.2080; //   CB
	icoor( 2, 5, aa_glu, 1) =    6.4040; //   CB
	icoor( 3, 5, aa_glu, 1) =   14.6070; //   CB
	icoor( 1, 6, aa_glu, 1) =    1.2950; //   CG
	icoor( 2, 6, aa_glu, 1) =    6.2520; //   CG
	icoor( 3, 6, aa_glu, 1) =   16.1190; //   CG
	icoor( 1, 7, aa_glu, 1) =    2.5040; //   CD
	icoor( 2, 7, aa_glu, 1) =    5.4530; //   CD
	icoor( 3, 7, aa_glu, 1) =   16.5190; //   CD
	icoor( 1, 8, aa_glu, 1) =    2.6990; //   OE1
	icoor( 2, 8, aa_glu, 1) =    5.2540; //   OE1
	icoor( 3, 8, aa_glu, 1) =   17.6940; //   OE1
	icoor( 1, 9, aa_glu, 1) =    3.1710; //   OE2
	icoor( 2, 9, aa_glu, 1) =    4.9460; //   OE2
	icoor( 3, 9, aa_glu, 1) =   15.6480; //   OE2
	icoor( 1,10, aa_glu, 1) =    0.0400; //   H
	icoor( 2,10, aa_glu, 1) =    8.2170; //   H
	icoor( 3,10, aa_glu, 1) =   12.2440; //   H
	icoor( 1,11, aa_glu, 1) =   -0.9090; //   HA
	icoor( 2,11, aa_glu, 1) =    6.7100; //   HA
	icoor( 3,11, aa_glu, 1) =   14.4600; //   HA
	icoor( 1,12, aa_glu, 1) =    1.1400; //  2HB
	icoor( 2,12, aa_glu, 1) =    5.4180; //  2HB
	icoor( 3,12, aa_glu, 1) =   14.1470; //  2HB
	icoor( 1,13, aa_glu, 1) =    2.0990; //  3HB
	icoor( 2,13, aa_glu, 1) =    6.9150; //  3HB
	icoor( 3,13, aa_glu, 1) =   14.2420; //  3HB
	icoor( 1,14, aa_glu, 1) =    1.2970; //  2HG
	icoor( 2,14, aa_glu, 1) =    7.2030; //  2HG
	icoor( 3,14, aa_glu, 1) =   16.6510; //  2HG
	icoor( 1,15, aa_glu, 1) =    0.3930; //  3HG
	icoor( 2,15, aa_glu, 1) =    5.6970; //  3HG
	icoor( 3,15, aa_glu, 1) =   16.3730; //  3HG


	// H positions from REDUCE
	if( param_pack::packer_logical::reduce_templates ) {
		icoor( 1 , 10 , aa_glu , 1) = 0.033 ; // H
		icoor( 2 , 10 , aa_glu , 1) = 8.261 ; // H
		icoor( 3 , 10 , aa_glu , 1) = 12.392 ; // H
		icoor( 1 , 11 , aa_glu , 1) = -0.921 ; // HA
		icoor( 2 , 11 , aa_glu , 1) = 6.689 ; // HA
		icoor( 3 , 11 , aa_glu , 1) = 14.425 ; // HA
		icoor( 1 , 12 , aa_glu , 1) = 1.174 ; // 1HB
		icoor( 2 , 12 , aa_glu , 1) = 5.403 ; // 1HB
		icoor( 3 , 12 , aa_glu , 1) = 14.152 ; // 1HB
		icoor( 1 , 13 , aa_glu , 1) = 2.126 ; // 2HB
		icoor( 2 , 13 , aa_glu , 1) = 6.891 ; // 2HB
		icoor( 3 , 13 , aa_glu , 1) = 14.246 ; // 2HB
		icoor( 1 , 14 , aa_glu , 1) = 1.338 ; // 1HG
		icoor( 2 , 14 , aa_glu , 1) = 7.246 ; // 1HG
		icoor( 3 , 14 , aa_glu , 1) = 16.588 ; // 1HG
		icoor( 1 , 15 , aa_glu , 1) = 0.386 ; // 2HG
		icoor( 2 , 15 , aa_glu , 1) = 5.758 ; // 2HG
		icoor( 3 , 15 , aa_glu , 1) = 16.493 ; // 2HG
	}

	//rh create protonation state variants
	create_pH_variants( aa_glu);

//bq create N and Cterm variants
	create_termini_variants( aa_glu );

//bk   create variants that have h2o connected to the backbone nh and carbonyl
	if ( explicit_h2o ) {
		create_backbone_h2o_variants(aa_glu);
		create_sidechain_h2o_variants(aa_glu);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_phe
///
/// @brief arrays filled to define the proerties of the amino acid Phenylalanine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// aa_phe - index number for PHE (5)
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_phe()
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

//bk   Parameters for PHE -----------------------------
	nvar( aa_phe ) = 1; // starting number of variants for PHE
	variant_type( aav_base, aa_phe, 1 ) = true; // type of aa variant

//bk   variant1 ---------- PHE

	natoms( aa_phe, 1) = 20; // number of total atoms
	nheavyatoms( aa_phe, 1) = 11; // number of heavy atoms
	nchi( aa_phe, 1) =  2; // number of chi angles

//bk   atom names PHE
	atom_name( 1, aa_phe, 1) = " N  ";
	atom_name( 2, aa_phe, 1) = " CA ";
	atom_name( 3, aa_phe, 1) = " C  ";
	atom_name( 4, aa_phe, 1) = " O  ";
	atom_name( 5, aa_phe, 1) = " CB ";
	atom_name( 6, aa_phe, 1) = " CG ";
	atom_name( 7, aa_phe, 1) = " CD1";
	atom_name( 8, aa_phe, 1) = " CD2";
	atom_name( 9, aa_phe, 1) = " CE1";
	atom_name(10, aa_phe, 1) = " CE2";
	atom_name(11, aa_phe, 1) = " CZ ";
	atom_name(12, aa_phe, 1) = " H  ";
	atom_name(13, aa_phe, 1) = " HD1";
	atom_name(14, aa_phe, 1) = " HE1";
	atom_name(15, aa_phe, 1) = " HZ ";
	atom_name(16, aa_phe, 1) = " HE2";
	atom_name(17, aa_phe, 1) = " HD2";
	atom_name(18, aa_phe, 1) = " HA ";
	atom_name(19, aa_phe, 1) = "2HB ";
	atom_name(20, aa_phe, 1) = "3HB ";

//jjh intra residue bonding
	nbonded_neighbors( 1, aa_phe, 1) = 2; // N
		bonded_neighbor(1, 1, aa_phe, 1) = 2; // N--CA
		bonded_neighbor(2, 1, aa_phe, 1) =12; // N--H
	nbonded_neighbors( 2, aa_phe, 1) = 4; // CA
		bonded_neighbor(1, 2, aa_phe, 1) = 1; // CA--N
		bonded_neighbor(2, 2, aa_phe, 1) = 3; // CA--C
		bonded_neighbor(3, 2, aa_phe, 1) = 5; // CA--CB
		bonded_neighbor(4, 2, aa_phe, 1) =18; // CA--HA
	nbonded_neighbors( 3, aa_phe, 1) = 2; // C
		bonded_neighbor(1, 3, aa_phe, 1) = 2; // C--CA
		bonded_neighbor(2, 3, aa_phe, 1) = 4; // C--O
	nbonded_neighbors( 4, aa_phe, 1) = 1; // O
		bonded_neighbor(1, 4, aa_phe, 1) = 3; // O--C
	nbonded_neighbors( 5, aa_phe, 1) = 4; // CB
		bonded_neighbor(1, 5, aa_phe, 1) = 2; // CB--CA
		bonded_neighbor(2, 5, aa_phe, 1) = 6; // CB--CG
		bonded_neighbor(3, 5, aa_phe, 1) =19; // CB--2HB
		bonded_neighbor(4, 5, aa_phe, 1) =20; // CB--3HB
	nbonded_neighbors( 6, aa_phe, 1) = 3; // CG
		bonded_neighbor(1, 6, aa_phe, 1) = 5; // CG--CB
		bonded_neighbor(2, 6, aa_phe, 1) = 7; // CG--CD1
		bonded_neighbor(3, 6, aa_phe, 1) = 8; // CG--CD2
	nbonded_neighbors( 7, aa_phe, 1) = 3; // CD1
		bonded_neighbor(1, 7, aa_phe, 1) = 6; // CD1--CG
		bonded_neighbor(2, 7, aa_phe, 1) = 9; // CD1--CE1
		bonded_neighbor(3, 7, aa_phe, 1) =13; // CD1--HD1
	nbonded_neighbors( 8, aa_phe, 1) = 3; // CD2
		bonded_neighbor(1, 8, aa_phe, 1) = 6; // CD2--CG
		bonded_neighbor(2, 8, aa_phe, 1) =10; // CD2--CE2
		bonded_neighbor(3, 8, aa_phe, 1) =17; // CD2--HD2
	nbonded_neighbors( 9, aa_phe, 1) = 3; // CE1
		bonded_neighbor(1, 9, aa_phe, 1) = 7; // CE1--CD1
		bonded_neighbor(2, 9, aa_phe, 1) =11; // CE1--CZ
		bonded_neighbor(3, 9, aa_phe, 1) =14; // CE1--HE1
	nbonded_neighbors( 10, aa_phe, 1) = 3; // CE2
		bonded_neighbor(1,10, aa_phe, 1) = 8; // CE2--CD2
		bonded_neighbor(2,10, aa_phe, 1) =11; // CE2--CZ
		bonded_neighbor(3,10, aa_phe, 1) =16; // CE2--HE2
	nbonded_neighbors( 11, aa_phe, 1) = 3; // CZ
		bonded_neighbor(1,11, aa_phe, 1) = 9; // CZ--CE1
		bonded_neighbor(2,11, aa_phe, 1) =10; // CZ--CE2
		bonded_neighbor(3,11, aa_phe, 1) =15; // CZ--HZ
	nbonded_neighbors( 12, aa_phe, 1) = 1; // H
		bonded_neighbor(1,12, aa_phe, 1) = 1; // H--N
	nbonded_neighbors( 13, aa_phe, 1) = 1; // HD1
		bonded_neighbor(1,13, aa_phe, 1) = 7; // HD1--CD1
	nbonded_neighbors( 14, aa_phe, 1) = 1; // HE1
		bonded_neighbor(1,14, aa_phe, 1) = 9; // HE1--CE1
	nbonded_neighbors( 15, aa_phe, 1) = 1; // HZ
		bonded_neighbor(1,15, aa_phe, 1) =11; // HZ--CZ
	nbonded_neighbors( 16, aa_phe, 1) = 1; // HE2
		bonded_neighbor(1,16, aa_phe, 1) =10; // HE2--CE2
	nbonded_neighbors( 17, aa_phe, 1) = 1; // HD2
		bonded_neighbor(1,17, aa_phe, 1) = 8; // HD2--CD2
	nbonded_neighbors( 18, aa_phe, 1) = 1; // HA
		bonded_neighbor(1,18, aa_phe, 1) = 2; // HA--CA
	nbonded_neighbors( 19, aa_phe, 1) = 1; //2HB
		bonded_neighbor(1,19, aa_phe, 1) = 5; //2HB--CB
	nbonded_neighbors( 20, aa_phe, 1) = 1; //3HB
		bonded_neighbor(1,20, aa_phe, 1) = 5; //3HB--CB

//jjh get the angles automatically
	get_angles_from_bonds(aa_phe, 1);

//chu   base atom numbers PHE
	atom_base( 1, aa_phe, 1) = 2; //  N  -> CA
	atom_base( 2, aa_phe, 1) = 1; //  CA -> N
	atom_base( 3, aa_phe, 1) = 2; //  C  -> CA
	atom_base( 4, aa_phe, 1) = 3; //  O  -> C
	atom_base( 5, aa_phe, 1) = 2; //  CB -> CA
	atom_base( 6, aa_phe, 1) = 5; //  CG -> CB
	atom_base( 7, aa_phe, 1) = 6; //  CD1-> CG
	atom_base( 8, aa_phe, 1) = 6; //  CD2-> CG
	atom_base( 9, aa_phe, 1) = 7; //  CE1-> CD1
	atom_base(10, aa_phe, 1) = 8; //  CE2-> CD2
	atom_base(11, aa_phe, 1) = 9; //  CZ -> CE1 (not consider CZ->CE2)
  abase2(11, aa_phe, 1) = 10; // for completeness
	atom_base(12, aa_phe, 1) = 1; //  H   -> N
	atom_base(13, aa_phe, 1) = 7; //  HD1 -> CD1
	atom_base(14, aa_phe, 1) = 9; //  HE1 -> CE1
	atom_base(15, aa_phe, 1) =11; //  HZ  -> CZ
	atom_base(16, aa_phe, 1) =10; //  HE2 -> CE2
	atom_base(17, aa_phe, 1) = 8; //  HD2 -> CD2
	atom_base(18, aa_phe, 1) = 2; //  HA  -> CA
	atom_base(19, aa_phe, 1) = 5; // 2HB  -> CB
	atom_base(20, aa_phe, 1) = 5; // 3HB  -> CB

//bk   atom type numbers PHE
	fullatom_type( 1, aa_phe, 1) = 17; // Nbb    N
	fullatom_type( 2, aa_phe, 1) = 18; // CAbb   CA
	fullatom_type( 3, aa_phe, 1) = 19; // CObb   C
	fullatom_type( 4, aa_phe, 1) = 20; // OCbb   O
	fullatom_type( 5, aa_phe, 1) =  4; // CH2    CB
	fullatom_type( 6, aa_phe, 1) =  6; // aroC   CG
	fullatom_type( 7, aa_phe, 1) =  6; // aroC   CD1
	fullatom_type( 8, aa_phe, 1) =  6; // aroC   CD2
	fullatom_type( 9, aa_phe, 1) =  6; // aroC   CE1
	fullatom_type(10, aa_phe, 1) =  6; // aroC   CE2
	fullatom_type(11, aa_phe, 1) =  6; // aroC   CZ
	fullatom_type(12, aa_phe, 1) = 25; // HNbb   H
	fullatom_type(13, aa_phe, 1) = 24; // Haro   HD1
	fullatom_type(14, aa_phe, 1) = 24; // Haro   HE1
	fullatom_type(15, aa_phe, 1) = 24; // Haro   HZ
	fullatom_type(16, aa_phe, 1) = 24; // Haro   HE2
	fullatom_type(17, aa_phe, 1) = 24; // Haro   HD2
	fullatom_type(18, aa_phe, 1) = 23; // Hapo   HA
	fullatom_type(19, aa_phe, 1) = 23; // Hapo  2HB
	fullatom_type(20, aa_phe, 1) = 23; // Hapo  3HB

//cmd  atomic charges PHE
	atomic_charge( 1, aa_phe, 1) = -0.47; //    N
	atomic_charge( 2, aa_phe, 1) =  0.07; //    CA
	atomic_charge( 3, aa_phe, 1) =  0.51; //    C
	atomic_charge( 4, aa_phe, 1) = -0.51; //    O
	atomic_charge( 5, aa_phe, 1) = -0.18; //    CB
	atomic_charge( 6, aa_phe, 1) =  0.00; //    CG
	atomic_charge( 7, aa_phe, 1) = -0.115; //    CD1
	atomic_charge( 8, aa_phe, 1) = -0.115; //    CD2
	atomic_charge( 9, aa_phe, 1) = -0.115; //    CE1
	atomic_charge(10, aa_phe, 1) = -0.115; //    CE2
	atomic_charge(11, aa_phe, 1) = -0.115; //    CZ
	atomic_charge(12, aa_phe, 1) =  0.31; //    H
	atomic_charge(13, aa_phe, 1) =  0.115; //    HD1
	atomic_charge(14, aa_phe, 1) =  0.115; //    HE1
	atomic_charge(15, aa_phe, 1) =  0.115; //    HZ
	atomic_charge(16, aa_phe, 1) =  0.115; //    HE2
	atomic_charge(17, aa_phe, 1) =  0.115; //    HD2
	atomic_charge(18, aa_phe, 1) =  0.09; //    HA
	atomic_charge(19, aa_phe, 1) =  0.09; //   2HB
	atomic_charge(20, aa_phe, 1) =  0.09; //   3HB

	assert_total_charge( aa_phe, 1, 0.0);

//cmd  first sidechain atom
	first_scatom( aa_phe,1) = 5; //CB

//bk   template atoms used for placing atoms PHE

//bk   template for building  HD1
	ta(1,13, aa_phe, 1) =  7; //   CD1
	ta(2,13, aa_phe, 1) =  6; //   CG
	ta(3,13, aa_phe, 1) =  9; //   CE1

//bk   template for building  HE1
	ta(1,14, aa_phe, 1) =  9; //   CE1
	ta(2,14, aa_phe, 1) =  7; //   CD1
	ta(3,14, aa_phe, 1) = 11; //   CZ

//bk   template for building  HZ
	ta(1,15, aa_phe, 1) = 11; //   CZ
	ta(2,15, aa_phe, 1) =  9; //   CE1
	ta(3,15, aa_phe, 1) = 10; //   CE2

//bk   template for building  HE2
	ta(1,16, aa_phe, 1) = 10; //   CE2
	ta(2,16, aa_phe, 1) =  8; //   CD2
	ta(3,16, aa_phe, 1) = 11; //   CZ

//bk   template for building  HD2
	ta(1,17, aa_phe, 1) =  8; //   CD2
	ta(2,17, aa_phe, 1) =  6; //   CG
	ta(3,17, aa_phe, 1) = 10; //   CE2

//bk   template for building  HA
	ta(1,18, aa_phe, 1) =  2; //   CA
	ta(2,18, aa_phe, 1) =  1; //   N
	ta(3,18, aa_phe, 1) =  3; //   C

//bk   template for building 2HB
	ta(1,19, aa_phe, 1) =  5; //   CB
	ta(2,19, aa_phe, 1) =  2; //   CA
	ta(3,19, aa_phe, 1) =  6; //   CG

//bk   template for building 3HB
	ta(1,20, aa_phe, 1) =  5; //   CB
	ta(2,20, aa_phe, 1) =  2; //   CA
	ta(3,20, aa_phe, 1) =  6; //   CG

//bk   chi angles required to build atoms PHE
//bk   chi angles needed for building  CG
	chi_required(1, 6, aa_phe, 1) =  true;

//bk   chi angles needed for building  CD1
	chi_required(1, 7, aa_phe, 1) =  true;
	chi_required(2, 7, aa_phe, 1) =  true;

//bk   chi angles needed for building  CD2
	chi_required(1, 8, aa_phe, 1) =  true;
	chi_required(2, 8, aa_phe, 1) =  true;

//bk   chi angles needed for building  CE1
	chi_required(1, 9, aa_phe, 1) =  true;
	chi_required(2, 9, aa_phe, 1) =  true;

//bk   chi angles needed for building  CE2
	chi_required(1,10, aa_phe, 1) =  true;
	chi_required(2,10, aa_phe, 1) =  true;

//bk   chi angles needed for building  CZ
	chi_required(1,11, aa_phe, 1) =  true;
	chi_required(2,11, aa_phe, 1) =  true;
	chi_required(4,12, aa_phe, 1) = false;

//bk   chi angles needed for building  HD1
	chi_required(1,13, aa_phe, 1) =  true;
	chi_required(2,13, aa_phe, 1) =  true;

//bk   chi angles needed for building  HE1
	chi_required(1,14, aa_phe, 1) =  true;
	chi_required(2,14, aa_phe, 1) =  true;
	chi_required(3,14, aa_phe, 1) = false;
	chi_required(4,14, aa_phe, 1) = false;

//bk   chi angles needed for building  HZ
	chi_required(1,15, aa_phe, 1) =  true;
	chi_required(2,15, aa_phe, 1) =  true;

//bk   chi angles needed for building  HE2
	chi_required(1,16, aa_phe, 1) =  true;
	chi_required(2,16, aa_phe, 1) =  true;

//bk   chi angles needed for building  HD2
	chi_required(1,17, aa_phe, 1) =  true;
	chi_required(2,17, aa_phe, 1) =  true;

//bk   chi angles needed for building  HA

//bk   chi angles needed for building 2HB
	chi_required(1,19, aa_phe, 1) =  true;

//bk   chi angles needed for building 3HB
	chi_required(1,20, aa_phe, 1) =  true;

//bk   four atoms that define chi angle  1
	chi_atoms( 1, 1, aa_phe, 1) =  1; //   N
	chi_atoms( 2, 1, aa_phe, 1) =  2; //   CA
	chi_atoms( 3, 1, aa_phe, 1) =  5; //   CB
	chi_atoms( 4, 1, aa_phe, 1) =  6; //   CG
//bk   four atoms that define chi angle  2
	chi_atoms( 1, 2, aa_phe, 1) =  2; //   CA
	chi_atoms( 2, 2, aa_phe, 1) =  5; //   CB
	chi_atoms( 3, 2, aa_phe, 1) =  6; //   CG
	chi_atoms( 4, 2, aa_phe, 1) =  7; //   CD1

	HNpos( aa_phe, 1) = 12; // atom number for backbone HN
	HApos( aa_phe, 1) = 18; // atom number for backbone HA
	nH_polar( aa_phe, 1) =  1; // number of polar hydrogens
	nH_aromatic( aa_phe, 1) =  5; // number of aromatic hydrogens
	nH_apolar( aa_phe, 1) =  3; // number of apolar hydrogens
	nacceptors( aa_phe, 1) =  1; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1, aa_phe, 1) = 12; //  H

//bk   atom number for aromatic hydrogens
	Hpos_aromatic( 1, aa_phe, 1) = 13; //  HD1
	Hpos_aromatic( 2, aa_phe, 1) = 14; //  HE1
	Hpos_aromatic( 3, aa_phe, 1) = 15; //  HZ
	Hpos_aromatic( 4, aa_phe, 1) = 16; //  HE2
	Hpos_aromatic( 5, aa_phe, 1) = 17; //  HD2

//bk   atom number for apolar hydrogens
	Hpos_apolar( 1, aa_phe, 1) = 18; //  HA
	Hpos_apolar( 2, aa_phe, 1) = 19; // 2HB
	Hpos_apolar( 3, aa_phe, 1) = 20; // 3HB

//bk   acceptor positions
	accpt_pos( 1, aa_phe, 1) =  4; //  O

//bk   hydrogens on each atom

	nhydrogens_on_atm( 1, aa_phe, 1) =  1; //  N
	hydrogens_on_atm( 1, 1, aa_phe, 1) = 12; //  H

	nhydrogens_on_atm( 2, aa_phe, 1) =  1; //  CA
	hydrogens_on_atm( 1, 2, aa_phe, 1) = 18; //  HA

	nhydrogens_on_atm( 5, aa_phe, 1) =  2; //  CB
	hydrogens_on_atm( 1, 5, aa_phe, 1) = 19; // 2HB
	hydrogens_on_atm( 2, 5, aa_phe, 1) = 20; // 3HB

	nhydrogens_on_atm( 7, aa_phe, 1) =  1; //  CD1
	hydrogens_on_atm( 1, 7, aa_phe, 1) = 13; //  HD1

	nhydrogens_on_atm( 8, aa_phe, 1) =  1; //  CD2
	hydrogens_on_atm( 1, 8, aa_phe, 1) = 17; //  HD2

	nhydrogens_on_atm( 9, aa_phe, 1) =  1; //  CE1
	hydrogens_on_atm( 1, 9, aa_phe, 1) = 14; //  HE1

	nhydrogens_on_atm(10, aa_phe, 1) =  1; //  CE2
	hydrogens_on_atm( 1,10, aa_phe, 1) = 16; //  HE2

	nhydrogens_on_atm(11, aa_phe, 1) =  1; //  CZ
	hydrogens_on_atm( 1,11, aa_phe, 1) = 15; //  HZ

//bk   template coordinates for the amino acid
	icoor( 1, 1, aa_phe, 1) =    0.0000; //   N
	icoor( 2, 1, aa_phe, 1) =    8.6220; //   N
	icoor( 3, 1, aa_phe, 1) =   16.0890; //   N
	icoor( 1, 2, aa_phe, 1) =    0.0000; //   CA
	icoor( 2, 2, aa_phe, 1) =    9.8810; //   CA
	icoor( 3, 2, aa_phe, 1) =   16.8260; //   CA
	icoor( 1, 3, aa_phe, 1) =    0.0000; //   C
	icoor( 2, 3, aa_phe, 1) =    9.6390; //   C
	icoor( 3, 3, aa_phe, 1) =   18.3300; //   C
	icoor( 1, 4, aa_phe, 1) =    0.0020; //   O
	icoor( 2, 4, aa_phe, 1) =    8.4940; //   O
	icoor( 3, 4, aa_phe, 1) =   18.7830; //   O
	icoor( 1, 5, aa_phe, 1) =   -1.2080; //   CB
	icoor( 2, 5, aa_phe, 1) =   10.7320; //   CB
	icoor( 3, 5, aa_phe, 1) =   16.4300; //   CB
	icoor( 1, 6, aa_phe, 1) =   -1.2820; //   CG
	icoor( 2, 6, aa_phe, 1) =   12.0500; //   CG
	icoor( 3, 6, aa_phe, 1) =   17.1470; //   CG
	icoor( 1, 7, aa_phe, 1) =   -2.3200; //   CD1
	icoor( 2, 7, aa_phe, 1) =   12.9350; //   CD1
	icoor( 3, 7, aa_phe, 1) =   16.8960; //   CD1
	icoor( 1, 8, aa_phe, 1) =   -0.3140; //   CD2
	icoor( 2, 8, aa_phe, 1) =   12.4080; //   CD2
	icoor( 3, 8, aa_phe, 1) =   18.0730; //   CD2
	icoor( 1, 9, aa_phe, 1) =   -2.3890; //   CE1
	icoor( 2, 9, aa_phe, 1) =   14.1480; //   CE1
	icoor( 3, 9, aa_phe, 1) =   17.5550; //   CE1
	icoor( 1,10, aa_phe, 1) =   -0.3810; //   CE2
	icoor( 2,10, aa_phe, 1) =   13.6190; //   CE2
	icoor( 3,10, aa_phe, 1) =   18.7340; //   CE2
	icoor( 1,11, aa_phe, 1) =   -1.4200; //   CZ
	icoor( 2,11, aa_phe, 1) =   14.4900; //   CZ
	icoor( 3,11, aa_phe, 1) =   18.4740; //   CZ
	icoor( 1,12, aa_phe, 1) =   -0.0400; //   H
	icoor( 2,12, aa_phe, 1) =    7.7600; //   H
	icoor( 3,12, aa_phe, 1) =   16.6140; //   H
	icoor( 1,13, aa_phe, 1) =   -3.0870; //   HD1
	icoor( 2,13, aa_phe, 1) =   12.6640; //   HD1
	icoor( 3,13, aa_phe, 1) =   16.1700; //   HD1
	icoor( 1,14, aa_phe, 1) =   -3.2100; //   HE1
	icoor( 2,14, aa_phe, 1) =   14.8340; //   HE1
	icoor( 3,14, aa_phe, 1) =   17.3480; //   HE1
	icoor( 1,15, aa_phe, 1) =   -1.4730; //   HZ
	icoor( 2,15, aa_phe, 1) =   15.4460; //   HZ
	icoor( 3,15, aa_phe, 1) =   18.9930; //   HZ
	icoor( 1,16, aa_phe, 1) =    0.3860; //   HE2
	icoor( 2,16, aa_phe, 1) =   13.8880; //   HE2
	icoor( 3,16, aa_phe, 1) =   19.4600; //   HE2
	icoor( 1,17, aa_phe, 1) =    0.5070; //   HD2
	icoor( 2,17, aa_phe, 1) =   11.7200; //   HD2
	icoor( 3,17, aa_phe, 1) =   18.2780; //   HD2
	icoor( 1,18, aa_phe, 1) =    0.9090; //   HA
	icoor( 2,18, aa_phe, 1) =   10.4410; //   HA
	icoor( 3,18, aa_phe, 1) =   16.6020; //   HA
	icoor( 1,19, aa_phe, 1) =   -1.1750; //  2HB
	icoor( 2,19, aa_phe, 1) =   10.9580; //  2HB
	icoor( 3,19, aa_phe, 1) =   15.3650; //  2HB
	icoor( 1,20, aa_phe, 1) =   -2.1310; //  3HB
	icoor( 2,20, aa_phe, 1) =   10.2010; //  3HB
	icoor( 3,20, aa_phe, 1) =   16.6590; //  3HB

	// H positions from REDUCE
	if( param_pack::packer_logical::reduce_templates ) {
		icoor( 1 , 12 , aa_phe , 1) = -0.000 ; // H
		icoor( 2 , 12 , aa_phe , 1) = 7.858 ; // H
		icoor( 3 , 12 , aa_phe , 1) = 16.734 ; // H
		icoor( 1 , 18 , aa_phe , 1) = 0.921 ; // HA
		icoor( 2 , 18 , aa_phe , 1) = 10.424 ; // HA
		icoor( 3 , 18 , aa_phe , 1) = 16.566 ; // HA
		icoor( 1 , 19 , aa_phe , 1) = -1.175 ; // 1HB
		icoor( 2 , 19 , aa_phe , 1) = 10.916 ; // 1HB
		icoor( 3 , 19 , aa_phe , 1) = 15.346 ; // 1HB
		icoor( 1 , 20 , aa_phe , 1) = -2.128 ; // 2HB
		icoor( 2 , 20 , aa_phe , 1) = 10.163 ; // 2HB
		icoor( 3 , 20 , aa_phe , 1) = 16.632 ; // 2HB
		icoor( 1 , 13 , aa_phe , 1) = -3.097 ; // HD1
		icoor( 2 , 13 , aa_phe , 1) = 12.669 ; // HD1
		icoor( 3 , 13 , aa_phe , 1) = 16.164 ; // HD1
		icoor( 1 , 17 , aa_phe , 1) = 0.517 ; // HD2
		icoor( 2 , 17 , aa_phe , 1) = 11.719 ; // HD2
		icoor( 3 , 17 , aa_phe , 1) = 18.284 ; // HD2
		icoor( 1 , 14 , aa_phe , 1) = -3.217 ; // HE1
		icoor( 2 , 14 , aa_phe , 1) = 14.841 ; // HE1
		icoor( 3 , 14 , aa_phe , 1) = 17.346 ; // HE1
		icoor( 1 , 16 , aa_phe , 1) = 0.393 ; // HE2
		icoor( 2 , 16 , aa_phe , 1) = 13.889 ; // HE2
		icoor( 3 , 16 , aa_phe , 1) = 19.467 ; // HE2
		icoor( 1 , 15 , aa_phe , 1) = -1.475 ; // HZ
		icoor( 2 , 15 , aa_phe , 1) = 15.455 ; // HZ
		icoor( 3 , 15 , aa_phe , 1) = 18.999 ; // HZ
	}

//bq create N and Cterm variants
	create_termini_variants( aa_phe );


//bk   create variants that have h2o connected to the backbone nh and carbonyl
	if ( explicit_h2o ) {
		create_backbone_h2o_variants(aa_phe);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_gly
///
/// @brief arrays filled to define the proerties of the amino acid Glycine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// aa_gly - index number for GLY (6)
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_gly()
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

//bk   Parameters for GLY -----------------------------
	nvar( aa_gly ) = 1; // starting number of variants for GLY
	variant_type( aav_base, aa_gly, 1 ) = true; // type of aa variant

//bk   variant1 ---------- GLY

	natoms( aa_gly,1) =  7; // number of total atoms
	nheavyatoms( aa_gly,1) =  4; // number of heavy atoms
	nchi( aa_gly,1) =  0; // number of chi angles

//bk   atom names GLY
	atom_name( 1, aa_gly,1) = " N  ";
	atom_name( 2, aa_gly,1) = " CA ";
	atom_name( 3, aa_gly,1) = " C  ";
	atom_name( 4, aa_gly,1) = " O  ";
	atom_name( 5, aa_gly,1) = " H  ";
	atom_name( 6, aa_gly,1) = "2HA ";
	atom_name( 7, aa_gly,1) = "3HA ";

//jjh intra residue bonding
	nbonded_neighbors( 1, aa_gly, 1) = 2; // N
		bonded_neighbor(1, 1, aa_gly, 1) = 2; // N--CA
		bonded_neighbor(2, 1, aa_gly, 1) = 5; // N--H
	nbonded_neighbors( 2, aa_gly, 1) = 4; // CA
		bonded_neighbor(1, 2, aa_gly, 1) = 1; // CA--N
		bonded_neighbor(2, 2, aa_gly, 1) = 3; // CA--C
		bonded_neighbor(3, 2, aa_gly, 1) = 6; // CA--2HA
		bonded_neighbor(4, 2, aa_gly, 1) = 7; // CA--3HA
	nbonded_neighbors( 3, aa_gly, 1) = 2; // C
		bonded_neighbor(1, 3, aa_gly, 1) = 2; // C--CA
		bonded_neighbor(2, 3, aa_gly, 1) = 4; // C--O
	nbonded_neighbors( 4, aa_gly, 1) = 1; // O
		bonded_neighbor(1, 4, aa_gly, 1) = 3; // O--C
	nbonded_neighbors( 5, aa_gly, 1) = 1; // H
		bonded_neighbor(1, 5, aa_gly, 1) = 1; // H--N
	nbonded_neighbors( 6, aa_gly, 1) = 1; //2HA
		bonded_neighbor(1, 6, aa_gly, 1) = 2; //2HA--CA
	nbonded_neighbors( 7, aa_gly, 1) = 1; //3HA
		bonded_neighbor(1, 7, aa_gly, 1) = 2; //3HA--CA

//jjh get the angles automatically
	get_angles_from_bonds(aa_gly, 1);

//chu   atom base GLY
	atom_base( 1, aa_gly,1) = 2; //  N  -> CA
	atom_base( 2, aa_gly,1) = 1; //  CA -> N
	atom_base( 3, aa_gly,1) = 2; //  C  -> CA
	atom_base( 4, aa_gly,1) = 3; //  O  -> C
	atom_base( 5, aa_gly,1) = 1; //  H  -> N
	atom_base( 6, aa_gly,1) = 2; // 2HA -> CA
	atom_base( 7, aa_gly,1) = 2; // 3HA -> CA

//bk   atom type numbers GLY
	fullatom_type( 1, aa_gly,1) = 17; // Nbb    N
	fullatom_type( 2, aa_gly,1) = 18; // CAbb   CA
	fullatom_type( 3, aa_gly,1) = 19; // CObb   C
	fullatom_type( 4, aa_gly,1) = 20; // OCbb   O
	fullatom_type( 5, aa_gly,1) = 25; // HNbb   H
	fullatom_type( 6, aa_gly,1) = 23; // Hapo  2HA
	fullatom_type( 7, aa_gly,1) = 23; // Hapo  3HA

//cmd  atomic charge GLY
	atomic_charge( 1, aa_gly,1) = -0.47; //    N
	atomic_charge( 2, aa_gly,1) = -0.02; //    CA
	atomic_charge( 3, aa_gly,1) =  0.51; //    C
	atomic_charge( 4, aa_gly,1) = -0.51; //    O
	atomic_charge( 5, aa_gly,1) =  0.31; //    H
	atomic_charge( 6, aa_gly,1) =  0.09; //   2HA
	atomic_charge( 7, aa_gly,1) =  0.09; //   3HA

	assert_total_charge( aa_gly, 1, 0.0);

//cmd  first sidechain atom
	first_scatom( aa_gly,1) = 5; //CB

//bk   template atoms used for placing atoms GLY

//bk   template for building 2HA
	ta(1, 6, aa_gly,1) =  2; //   CA
	ta(2, 6, aa_gly,1) =  1; //   N
	ta(3, 6, aa_gly,1) =  3; //   C

//bk   template for building 3HA
	ta(1, 7, aa_gly,1) =  2; //   CA
	ta(2, 7, aa_gly,1) =  1; //   N
	ta(3, 7, aa_gly,1) =  3; //   C

//bk   chi angles required to build atoms GLY


	HNpos( aa_gly,1) =  5; // atom number for backbone HN
	HApos( aa_gly,1) =  7; // atom number for backbone HA
	nH_polar( aa_gly,1) =  1; // number of polar hydrogens
	nH_aromatic( aa_gly,1) =  0; // number of aromatic hydrogens
	nH_apolar( aa_gly,1) =  2; // number of apolar hydrogens
	nacceptors( aa_gly,1) =  1; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1, aa_gly,1) =  5; //  H

//bk   atom number for aromatic hydrogens

//bk   atom number for apolar hydrogens
	Hpos_apolar( 1, aa_gly,1) =  6; // 2HA
	Hpos_apolar( 2, aa_gly,1) =  7; // 3HA

//bk   acceptor positions
	accpt_pos( 1, aa_gly, 1) =  4; //  O

//bk   hydrogens on each atom

	nhydrogens_on_atm( 1, aa_gly, 1) =  1; //  N
	hydrogens_on_atm( 1, 1, aa_gly, 1) =  5; //  H

	nhydrogens_on_atm( 2, aa_gly, 1) =  2; //  CA
	hydrogens_on_atm( 1, 2, aa_gly, 1) =  6; // 2HA
	hydrogens_on_atm( 2, 2, aa_gly, 1) =  7; // 3HA

//bk   template coordinates for the amino acid
	icoor( 1, 1, aa_gly, 1) =    0.0000; //   N
	icoor( 2, 1, aa_gly, 1) =   10.7220; //   N
	icoor( 3, 1, aa_gly, 1) =   19.0990; //   N
	icoor( 1, 2, aa_gly, 1) =    0.0000; //   CA
	icoor( 2, 2, aa_gly, 1) =   10.6280; //   CA
	icoor( 3, 2, aa_gly, 1) =   20.5540; //   CA
	icoor( 1, 3, aa_gly, 1) =    0.0000; //   C
	icoor( 2, 3, aa_gly, 1) =   12.0090; //   C
	icoor( 3, 3, aa_gly, 1) =   21.1960; //   C
	icoor( 1, 4, aa_gly, 1) =   -0.0030; //   O
	icoor( 2, 4, aa_gly, 1) =   13.0390; //   O
	icoor( 3, 4, aa_gly, 1) =   20.5210; //   O
	icoor( 1, 5, aa_gly, 1) =    0.0000; //   H
	icoor( 2, 5, aa_gly, 1) =   11.6400; //   H
	icoor( 3, 5, aa_gly, 1) =   18.6790; //   H
	icoor( 1, 6, aa_gly, 1) =   -0.8900; //  2HA
	icoor( 2, 6, aa_gly, 1) =   10.0870; //  2HA
	icoor( 3, 6, aa_gly, 1) =   20.8760; //  2HA
	icoor( 1, 7, aa_gly, 1) =    0.8890; //  3HA
	icoor( 2, 7, aa_gly, 1) =   10.0870; //  3HA
	icoor( 3, 7, aa_gly, 1) =   20.8760; //  3HA

	// H positions from REDUCE
	if( param_pack::packer_logical::reduce_templates ) {
		icoor( 1 , 5 , aa_gly , 1) = 0.033 ; // H
		icoor( 2 , 5 , aa_gly , 1) = 11.684 ; // H
		icoor( 3 , 5 , aa_gly , 1) = 18.827 ; // H
		icoor( 1 , 6 , aa_gly , 1) = -0.884 ; // 1HA
		icoor( 2 , 6 , aa_gly , 1) = 10.065 ; // 1HA
		icoor( 3 , 6 , aa_gly , 1) = 20.888 ; // 1HA
		icoor( 1 , 7 , aa_gly , 1) = 0.884 ; // 2HA
		icoor( 2 , 7 , aa_gly , 1) = 10.065 ; // 2HA
		icoor( 3 , 7 , aa_gly , 1) = 20.888 ; // 2HA
	}

	//bq create N and Cterm variants
	create_termini_variants( aa_gly );


//bk   create variants that have h2o connected to the backbone nh and carbonyl
	if ( explicit_h2o ) {
		create_backbone_h2o_variants(aa_gly);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_his
///
/// @brief arrays filled to define the proerties of the amino acid Histidine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// aa_his - index number for HIS (7)
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_his()
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

//bk   Parameters for HIS -----------------------------
	nvar( aa_his ) = 1; // starting number of variants for HIS
	variant_type( aav_base, aa_his, 1 ) = true; // type of aa variant

//bk   variant1 ---------- HIS

	natoms( aa_his,1) = 17; // number of total atoms
	nheavyatoms( aa_his,1) = 10; // number of heavy atoms
	nchi( aa_his,1) =  2; // number of chi angles

//bk   atom names HIS
	atom_name( 1, aa_his,1) = " N  ";
	atom_name( 2, aa_his,1) = " CA ";
	atom_name( 3, aa_his,1) = " C  ";
	atom_name( 4, aa_his,1) = " O  ";
	atom_name( 5, aa_his,1) = " CB ";
	atom_name( 6, aa_his,1) = " CG ";
	atom_name( 7, aa_his,1) = " ND1";
	atom_name( 8, aa_his,1) = " CD2";
	atom_name( 9, aa_his,1) = " CE1";
	atom_name(10, aa_his,1) = " NE2";
	atom_name(11, aa_his,1) = " H  ";
	atom_name(12, aa_his,1) = " HE2";
	atom_name(13, aa_his,1) = " HA ";
	atom_name(14, aa_his,1) = "2HB ";
	atom_name(15, aa_his,1) = "3HB ";
	atom_name(16, aa_his,1) = " HE1";
	atom_name(17, aa_his,1) = " HD2";

//jjh intra residue bonding
	nbonded_neighbors( 1, aa_his, 1) = 2; // N
		bonded_neighbor(1, 1, aa_his, 1) = 2; // N--CA
		bonded_neighbor(2, 1, aa_his, 1) =11; // N--H
	nbonded_neighbors( 2, aa_his, 1) = 4; // CA
		bonded_neighbor(1, 2, aa_his, 1) = 1; // CA--N
		bonded_neighbor(2, 2, aa_his, 1) = 3; // CA--C
		bonded_neighbor(3, 2, aa_his, 1) = 5; // CA--CB
		bonded_neighbor(4, 2, aa_his, 1) =13; // CA--HA
	nbonded_neighbors( 3, aa_his, 1) = 2; // C
		bonded_neighbor(1, 3, aa_his, 1) = 2; // C--CA
		bonded_neighbor(2, 3, aa_his, 1) = 4; // C--O
	nbonded_neighbors( 4, aa_his, 1) = 1; // O
		bonded_neighbor(1, 4, aa_his, 1) = 3; // O--C
	nbonded_neighbors( 5, aa_his, 1) = 4; // CB
		bonded_neighbor(1, 5, aa_his, 1) = 2; // CB--CA
		bonded_neighbor(2, 5, aa_his, 1) = 6; // CB--CG
		bonded_neighbor(3, 5, aa_his, 1) =14; // CB--2HB
		bonded_neighbor(4, 5, aa_his, 1) =15; // CB--3HB
	nbonded_neighbors( 6, aa_his, 1) = 3; // CG
		bonded_neighbor(1, 6, aa_his, 1) = 5; // CG--CB
		bonded_neighbor(2, 6, aa_his, 1) = 7; // CG--ND1
		bonded_neighbor(3, 6, aa_his, 1) = 8; // CG--CD2
	nbonded_neighbors( 7, aa_his, 1) = 2; // ND1
		bonded_neighbor(1, 7, aa_his, 1) = 6; // ND1--CG
		bonded_neighbor(2, 7, aa_his, 1) = 9; // ND1--CE1
	nbonded_neighbors( 8, aa_his, 1) = 3; // CD2
		bonded_neighbor(1, 8, aa_his, 1) = 6; // CD2--CG
		bonded_neighbor(2, 8, aa_his, 1) =10; // CD2--NE2
		bonded_neighbor(3, 8, aa_his, 1) =17; // CD2--HD2
	nbonded_neighbors( 9, aa_his, 1) = 3; // CE1
		bonded_neighbor(1, 9, aa_his, 1) = 7; // CE1--ND1
		bonded_neighbor(2, 9, aa_his, 1) =10; // CE1--NE2
		bonded_neighbor(3, 9, aa_his, 1) =16; // CE1--HE1
	nbonded_neighbors( 10, aa_his, 1) = 3; // NE2
		bonded_neighbor(1,10, aa_his, 1) = 8; // NE2--CD2
		bonded_neighbor(2,10, aa_his, 1) = 9; // NE2--CE1
		bonded_neighbor(3,10, aa_his, 1) =12; // NE2--HE2
	nbonded_neighbors( 11, aa_his, 1) = 1; // H
		bonded_neighbor(1,11, aa_his, 1) = 1; // H--N
	nbonded_neighbors( 12, aa_his, 1) = 1; // HE2
		bonded_neighbor(1,12, aa_his, 1) =10; // HE2--NE2
	nbonded_neighbors( 13, aa_his, 1) = 1; // HA
		bonded_neighbor(1,13, aa_his, 1) = 2; // HA--CA
	nbonded_neighbors( 14, aa_his, 1) = 1; //2HB
		bonded_neighbor(1,14, aa_his, 1) = 5; //2HB--CB
	nbonded_neighbors( 15, aa_his, 1) = 1; //3HB
		bonded_neighbor(1,15, aa_his, 1) = 5; //3HB--CB
	nbonded_neighbors( 16, aa_his, 1) = 1; // HE1
		bonded_neighbor(1,16, aa_his, 1) = 9; // HE1--CE1
	nbonded_neighbors( 17, aa_his, 1) = 1; // HD2
		bonded_neighbor(1,17, aa_his, 1) = 8; // HD2--CD2

//jjh get the angles automatically
	get_angles_from_bonds(aa_his, 1);

//chu   base atom numbers HIS
	atom_base( 1, aa_his,1) = 2; //  N  -> CA
	atom_base( 2, aa_his,1) = 1; //  CA -> N
	atom_base( 3, aa_his,1) = 2; //  C  -> CA
	atom_base( 4, aa_his,1) = 3; //  O  -> C
	atom_base( 5, aa_his,1) = 2; //  CB -> CA
	atom_base( 6, aa_his,1) = 5; //  CG -> CB
	atom_base( 7, aa_his,1) = 6; //  ND1-> CG
	abase2(7, aa_his, 1) =  9; //  CE1 base2 for ND1
	atom_base( 8, aa_his,1) = 6; //  CD2-> CG
	atom_base( 9, aa_his,1) = 7; //  CE1-> ND1
	atom_base(10, aa_his,1) = 8; //  NE2-> CD2 (not consider CE1<->NE2)
  abase2(10, aa_his,1) = 9; // for completeness
	atom_base(11, aa_his,1) = 1; //  H   -> N
	atom_base(12, aa_his,1) =10; //  HE2 -> NE2
	atom_base(13, aa_his,1) = 2; //  HA  -> CA
	atom_base(14, aa_his,1) = 5; // 2HB  -> CB
	atom_base(15, aa_his,1) = 5; // 3HB  -> CB
	atom_base(16, aa_his,1) = 9; //  HE1 -> CE1
	atom_base(17, aa_his,1) = 8; //  HD2 -> CD2

//bk   atom type numbers HIS
	fullatom_type( 1, aa_his,1) = 17; // Nbb    N
	fullatom_type( 2, aa_his,1) = 18; // CAbb   CA
	fullatom_type( 3, aa_his,1) = 19; // CObb   C
	fullatom_type( 4, aa_his,1) = 20; // OCbb   O
	fullatom_type( 5, aa_his,1) =  4; // CH2    CB
	fullatom_type( 6, aa_his,1) =  6; // aroC   CG
	fullatom_type( 7, aa_his,1) =  8; // Nhis   ND1
	fullatom_type( 8, aa_his,1) =  6; // aroC   CD2
	fullatom_type( 9, aa_his,1) =  6; // aroC   CE1
	fullatom_type(10, aa_his,1) =  7; // Ntrp   NE2
	fullatom_type(11, aa_his,1) = 25; // HNbb   H
	fullatom_type(12, aa_his,1) = 22; // Hpol   HE2
	fullatom_type(13, aa_his,1) = 23; // Hapo   HA
	fullatom_type(14, aa_his,1) = 23; // Hapo  2HB
	fullatom_type(15, aa_his,1) = 23; // Hapo  3HB
	fullatom_type(16, aa_his,1) = 23; // Hapo   HE1
	fullatom_type(17, aa_his,1) = 23; // Hapo   HD2

//cmd  atomic charge HIS
	atomic_charge( 1, aa_his,1) = -0.47; //    N
	atomic_charge( 2, aa_his,1) =  0.07; //    CA
	atomic_charge( 3, aa_his,1) =  0.51; //    C
	atomic_charge( 4, aa_his,1) = -0.51; //    O
	atomic_charge( 5, aa_his,1) = -0.08; //    CB
	atomic_charge( 6, aa_his,1) =  0.22; //    CG
	atomic_charge( 7, aa_his,1) = -0.70; //    ND1
	atomic_charge( 8, aa_his,1) = -0.05; //    CD2
	atomic_charge( 9, aa_his,1) =  0.25; //    CE1
	atomic_charge(10, aa_his,1) = -0.36; //    NE2
	atomic_charge(11, aa_his,1) =  0.31; //    H
	atomic_charge(12, aa_his,1) =  0.32; //    HE2
	atomic_charge(13, aa_his,1) =  0.09; //    HA
	atomic_charge(14, aa_his,1) =  0.09; //   2HB
	atomic_charge(15, aa_his,1) =  0.09; //   3HB
	atomic_charge(16, aa_his,1) =  0.13; //    HE1
	atomic_charge(17, aa_his,1) =  0.09; //    HD2

	assert_total_charge( aa_his, 1, 0.0);

//cmd  first sidechain atom
	first_scatom( aa_his,1) = 5; //CB

//bk   template atoms used for placing atoms HIS

//bk   template for building  HE2
	ta(1,12, aa_his,1) = 10; //   NE2
	ta(2,12, aa_his,1) =  8; //   CD2
	ta(3,12, aa_his,1) =  9; //   CE1

//bk   template for building  HA
	ta(1,13, aa_his,1) =  2; //   CA
	ta(2,13, aa_his,1) =  1; //   N
	ta(3,13, aa_his,1) =  3; //   C

//bk   template for building 2HB
	ta(1,14, aa_his,1) =  5; //   CB
	ta(2,14, aa_his,1) =  2; //   CA
	ta(3,14, aa_his,1) =  6; //   CG

//bk   template for building 3HB
	ta(1,15, aa_his,1) =  5; //   CB
	ta(2,15, aa_his,1) =  2; //   CA
	ta(3,15, aa_his,1) =  6; //   CG

//bk   template for building  HE1
	ta(1,16, aa_his,1) =  9; //   CE1
	ta(2,16, aa_his,1) =  7; //   ND1
	ta(3,16, aa_his,1) = 10; //   NE2

//bk   template for building  HD2
	ta(1,17, aa_his,1) =  8; //   CD2
	ta(2,17, aa_his,1) =  6; //   CG
	ta(3,17, aa_his,1) = 10; //   NE2

//bk   chi angles required to build atoms HIS
//bk   chi angles needed for building  CG
	chi_required(1, 6, aa_his,1) =  true;

//bk   chi angles needed for building  ND1
	chi_required(1, 7, aa_his,1) =  true;
	chi_required(2, 7, aa_his,1) =  true;

//bk   chi angles needed for building  CD2
	chi_required(1, 8, aa_his,1) =  true;
	chi_required(2, 8, aa_his,1) =  true;

//bk   chi angles needed for building  CE1
	chi_required(1, 9, aa_his,1) =  true;
	chi_required(2, 9, aa_his,1) =  true;

//bk   chi angles needed for building  NE2
	chi_required(1,10, aa_his,1) =  true;
	chi_required(2,10, aa_his,1) =  true;

//bk   chi angles needed for building  HE2
	chi_required(1,12, aa_his,1) =  true;
	chi_required(2,12, aa_his,1) =  true;

//bk   chi angles needed for building 2HB
	chi_required(1,14, aa_his,1) =  true;

//bk   chi angles needed for building 3HB
	chi_required(1,15, aa_his,1) =  true;

//bk   chi angles needed for building  HE1
	chi_required(1,16, aa_his,1) =  true;
	chi_required(2,16, aa_his,1) =  true;

//bk   chi angles needed for building  HD2
	chi_required(1,17, aa_his,1) =  true;
	chi_required(2,17, aa_his,1) =  true;

//bk   four atoms that define chi angle  1
	chi_atoms( 1, 1, aa_his, 1) =  1; //   N
	chi_atoms( 2, 1, aa_his, 1) =  2; //   CA
	chi_atoms( 3, 1, aa_his, 1) =  5; //   CB
	chi_atoms( 4, 1, aa_his, 1) =  6; //   CG
//bk   four atoms that define chi angle  2
	chi_atoms( 1, 2, aa_his, 1) =  2; //   CA
	chi_atoms( 2, 2, aa_his, 1) =  5; //   CB
	chi_atoms( 3, 2, aa_his, 1) =  6; //   CG
	chi_atoms( 4, 2, aa_his, 1) =  7; //   ND1

	HNpos( aa_his,1) = 11; // atom number for backbone HN
	HApos( aa_his,1) = 13; // atom number for backbone HA
	nH_polar( aa_his,1) =  2; // number of polar hydrogens
	nH_aromatic( aa_his,1) =  0; // number of aromatic hydrogens
	nH_apolar( aa_his,1) =  5; // number of apolar hydrogens
	nacceptors( aa_his,1) =  2; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1, aa_his,1) = 11; //  H
	Hpos_polar( 2, aa_his,1) = 12; //  HE2

//bk   atom number for aromatic hydrogens

//bk   atom number for apolar hydrogens
	Hpos_apolar( 1, aa_his,1) = 13; //  HA
	Hpos_apolar( 2, aa_his,1) = 14; // 2HB
	Hpos_apolar( 3, aa_his,1) = 15; // 3HB
	Hpos_apolar( 4, aa_his,1) = 16; //  HE1
	Hpos_apolar( 5, aa_his,1) = 17; //  HD2

//bk   acceptor positions
	accpt_pos( 1, aa_his, 1) =  4; //  O
	accpt_pos( 2, aa_his, 1) =  7; //  ND1

//bk   hydrogens on each atom

	nhydrogens_on_atm( 1, aa_his, 1) =  1; //  N
	hydrogens_on_atm( 1, 1, aa_his, 1) = 11; //  H

	nhydrogens_on_atm( 2, aa_his, 1) =  1; //  CA
	hydrogens_on_atm( 1, 2, aa_his, 1) = 13; //  HA

	nhydrogens_on_atm( 5, aa_his, 1) =  2; //  CB
	hydrogens_on_atm( 1, 5, aa_his, 1) = 14; // 2HB
	hydrogens_on_atm( 2, 5, aa_his, 1) = 15; // 3HB

	nhydrogens_on_atm( 8, aa_his, 1) =  1; //  CD2
	hydrogens_on_atm( 1, 8, aa_his, 1) = 17; //  HD2

	nhydrogens_on_atm( 9, aa_his, 1) =  1; //  CE1
	hydrogens_on_atm( 1, 9, aa_his, 1) = 16; //  HE1

	nhydrogens_on_atm(10, aa_his, 1) =  1; //  NE2
	hydrogens_on_atm( 1,10, aa_his, 1) = 12; //  HE2

//bk   template coordinates for the amino acid
	icoor( 1, 1, aa_his, 1) =    0.0000; //   N
	icoor( 2, 1, aa_his, 1) =   12.0140; //   N
	icoor( 3, 1, aa_his, 1) =   22.5220; //   N
	icoor( 1, 2, aa_his, 1) =    0.0000; //   CA
	icoor( 2, 2, aa_his, 1) =   13.2300; //   CA
	icoor( 3, 2, aa_his, 1) =   23.3160; //   CA
	icoor( 1, 3, aa_his, 1) =    0.0000; //   C
	icoor( 2, 3, aa_his, 1) =   12.6360; //   C
	icoor( 3, 3, aa_his, 1) =   24.7310; //   C
	icoor( 1, 4, aa_his, 1) =    0.1540; //   O
	icoor( 2, 4, aa_his, 1) =   11.4260; //   O
	icoor( 3, 4, aa_his, 1) =   24.9130; //   O
	icoor( 1, 5, aa_his, 1) =   -1.2080; //   CB
	icoor( 2, 5, aa_his, 1) =   14.1050; //   CB
	icoor( 3, 5, aa_his, 1) =   22.9660; //   CB
	icoor( 1, 6, aa_his, 1) =   -1.2740; //   CG
	icoor( 2, 6, aa_his, 1) =   15.3800; //   CG
	icoor( 3, 6, aa_his, 1) =   23.7480; //   CG
	icoor( 1, 7, aa_his, 1) =   -2.2840; //   ND1
	icoor( 2, 7, aa_his, 1) =   16.3040; //   ND1
	icoor( 3, 7, aa_his, 1) =   23.5800; //   ND1
	icoor( 1, 8, aa_his, 1) =   -0.4560; //   CD2
	icoor( 2, 8, aa_his, 1) =   15.8850; //   CD2
	icoor( 3, 8, aa_his, 1) =   24.7010; //   CD2
	icoor( 1, 9, aa_his, 1) =   -2.0840; //   CE1
	icoor( 2, 9, aa_his, 1) =   17.3230; //   CE1
	icoor( 3, 9, aa_his, 1) =   24.3980; //   CE1
	icoor( 1,10, aa_his, 1) =   -0.9820; //   NE2
	icoor( 2,10, aa_his, 1) =   17.0930; //   NE2
	icoor( 3,10, aa_his, 1) =   25.0880; //   NE2
	icoor( 1,11, aa_his, 1) =   -0.0410; //   H
	icoor( 2,11, aa_his, 1) =   11.1270; //   H
	icoor( 3,11, aa_his, 1) =   23.0030; //   H
	icoor( 1,12, aa_his, 1) =   -0.5830; //   HE2
	icoor( 2,12, aa_his, 1) =   17.7010; //   HE2
	icoor( 3,12, aa_his, 1) =   25.7890; //   HE2
	icoor( 1,13, aa_his, 1) =    0.9100; //   HA
	icoor( 2,13, aa_his, 1) =   13.7960; //   HA
	icoor( 3,13, aa_his, 1) =   23.1190; //   HA
	icoor( 1,14, aa_his, 1) =   -1.1760; //  2HB
	icoor( 2,14, aa_his, 1) =   14.3870; //  2HB
	icoor( 3,14, aa_his, 1) =   21.9130; //  2HB
	icoor( 1,15, aa_his, 1) =   -2.1330; //  3HB
	icoor( 2,15, aa_his, 1) =   13.5660; //  3HB
	icoor( 3,15, aa_his, 1) =   23.1690; //  3HB
	icoor( 1,16, aa_his, 1) =   -2.7780; //   HE1
	icoor( 2,16, aa_his, 1) =   18.1630; //   HE1
	icoor( 3,16, aa_his, 1) =   24.4180; //   HE1
	icoor( 1,17, aa_his, 1) =    0.4620; //   HD2
	icoor( 2,17, aa_his, 1) =   15.5170; //   HD2
	icoor( 3,17, aa_his, 1) =   25.1600; //   HD2


	// H positions from REDUCE
	if( param_pack::packer_logical::reduce_templates ) {
		icoor( 1 , 11 , aa_his , 1) = 0.196 ; // H
		icoor( 2 , 11 , aa_his , 1) = 11.230 ; // H
		icoor( 3 , 11 , aa_his , 1) = 23.112 ; // H
		icoor( 1 , 13 , aa_his , 1) = 0.838 ; // HA
		icoor( 2 , 13 , aa_his , 1) = 13.925 ; // HA
		icoor( 3 , 13 , aa_his , 1) = 23.160 ; // HA
		icoor( 1 , 14 , aa_his , 1) = -1.177 ; // 1HB
		icoor( 2 , 14 , aa_his , 1) = 14.345 ; // 1HB
		icoor( 3 , 14 , aa_his , 1) = 21.893 ; // 1HB
		icoor( 1 , 15 , aa_his , 1) = -2.129 ; // 2HB
		icoor( 2 , 15 , aa_his , 1) = 13.530 ; // 2HB
		icoor( 3 , 15 , aa_his , 1) = 23.141 ; // 2HB
		icoor( 1 , 17 , aa_his , 1) = 0.459 ; // HD2
		icoor( 2 , 17 , aa_his , 1) = 15.416 ; // HD2
		icoor( 3 , 17 , aa_his , 1) = 25.092 ; // HD2
		icoor( 1 , 16 , aa_his , 1) = -2.726 ; // HE1
		icoor( 2 , 16 , aa_his , 1) = 18.211 ; // HE1
		icoor( 3 , 16 , aa_his , 1) = 24.488 ; // HE1
		icoor( 1 , 12 , aa_his , 1) = -0.593 ; // HE2
		icoor( 2 , 12 , aa_his , 1) = 17.700 ; // HE2
		icoor( 3 , 12 , aa_his , 1) = 25.781 ; // HE2
	}

//jjh make alternate tautomer for histidine as variant 2
	if ( try_both_his_tautomers || get_pH_packing_flag()) {
		create_alternate_his_tautomer();
		nvar(aa_his) = 2;
	}

	//rh create protonation state variants
	create_pH_variants( aa_his);

//bq create N and Cterm variants
	create_termini_variants( aa_his );

//bk   create variants that have h2o connected to the backbone nh and carbonyl
	if ( explicit_h2o ) {
		create_backbone_h2o_variants(aa_his);
		create_sidechain_h2o_variants(aa_his);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_ile
///
/// @brief arrays filled to define the proerties of the amino acid Isoleucine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// aa_ile - index number for ILE (8)
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_ile()
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

//bk   Parameters for ILE -----------------------------
	nvar( aa_ile ) = 1; // starting number of variants for ILE
	variant_type( aav_base, aa_ile, 1 ) = true; // type of aa variant

//bk   variant1 ---------- ILE

	natoms( aa_ile,1) = 19; // number of total atoms
	nheavyatoms( aa_ile,1) =  8; // number of heavy atoms
	nchi( aa_ile,1) =  2; // number of chi angles

//bk   atom names ILE
	atom_name( 1, aa_ile,1) = " N  ";
	atom_name( 2, aa_ile,1) = " CA ";
	atom_name( 3, aa_ile,1) = " C  ";
	atom_name( 4, aa_ile,1) = " O  ";
	atom_name( 5, aa_ile,1) = " CB ";
	atom_name( 6, aa_ile,1) = " CG1";
	atom_name( 7, aa_ile,1) = " CG2";
	atom_name( 8, aa_ile,1) = " CD1";
	atom_name( 9, aa_ile,1) = " H  ";
	atom_name(10, aa_ile,1) = " HA ";
	atom_name(11, aa_ile,1) = " HB ";
	atom_name(12, aa_ile,1) = "1HG2";
	atom_name(13, aa_ile,1) = "2HG2";
	atom_name(14, aa_ile,1) = "3HG2";
	atom_name(15, aa_ile,1) = "2HG1";
	atom_name(16, aa_ile,1) = "3HG1";
	atom_name(17, aa_ile,1) = "1HD1";
	atom_name(18, aa_ile,1) = "2HD1";
	atom_name(19, aa_ile,1) = "3HD1";

//jjh intra residue bonding
	nbonded_neighbors( 1, aa_ile, 1) = 2; // N
		bonded_neighbor(1, 1, aa_ile, 1) = 2; // N--CA
		bonded_neighbor(2, 1, aa_ile, 1) = 9; // N--H
	nbonded_neighbors( 2, aa_ile, 1) = 4; // CA
		bonded_neighbor(1, 2, aa_ile, 1) = 1; // CA--N
		bonded_neighbor(2, 2, aa_ile, 1) = 3; // CA--C
		bonded_neighbor(3, 2, aa_ile, 1) = 5; // CA--CB
		bonded_neighbor(4, 2, aa_ile, 1) =10; // CA--HA
	nbonded_neighbors( 3, aa_ile, 1) = 2; // C
		bonded_neighbor(1, 3, aa_ile, 1) = 2; // C--CA
		bonded_neighbor(2, 3, aa_ile, 1) = 4; // C--O
	nbonded_neighbors( 4, aa_ile, 1) = 1; // O
		bonded_neighbor(1, 4, aa_ile, 1) = 3; // O--C
	nbonded_neighbors( 5, aa_ile, 1) = 4; // CB
		bonded_neighbor(1, 5, aa_ile, 1) = 2; // CB--CA
		bonded_neighbor(2, 5, aa_ile, 1) = 6; // CB--CG1
		bonded_neighbor(3, 5, aa_ile, 1) = 7; // CB--CG2
		bonded_neighbor(4, 5, aa_ile, 1) =11; // CB--HB
	nbonded_neighbors( 6, aa_ile, 1) = 4; // CG1
		bonded_neighbor(1, 6, aa_ile, 1) = 5; // CG1--CB
		bonded_neighbor(2, 6, aa_ile, 1) = 8; // CG1--CD1
		bonded_neighbor(3, 6, aa_ile, 1) =15; // CG1--2HG1
		bonded_neighbor(4, 6, aa_ile, 1) =16; // CG1--3HG1
	nbonded_neighbors( 7, aa_ile, 1) = 4; // CG2
		bonded_neighbor(1, 7, aa_ile, 1) = 5; // CG2--CB
		bonded_neighbor(2, 7, aa_ile, 1) =12; // CG2--1HG2
		bonded_neighbor(3, 7, aa_ile, 1) =13; // CG2--2HG2
		bonded_neighbor(4, 7, aa_ile, 1) =14; // CG2--3HG2
	nbonded_neighbors( 8, aa_ile, 1) = 4; // CD1
		bonded_neighbor(1, 8, aa_ile, 1) = 6; // CD1--CG1
		bonded_neighbor(2, 8, aa_ile, 1) =17; // CD1--1HD1
		bonded_neighbor(3, 8, aa_ile, 1) =18; // CD1--2HD1
		bonded_neighbor(4, 8, aa_ile, 1) =19; // CD1--3HD1
	nbonded_neighbors( 9, aa_ile, 1) = 1; // H
		bonded_neighbor(1, 9, aa_ile, 1) = 1; // H--N
	nbonded_neighbors( 10, aa_ile, 1) = 1; // HA
		bonded_neighbor(1,10, aa_ile, 1) = 2; // HA--CA
	nbonded_neighbors( 11, aa_ile, 1) = 1; // HB
		bonded_neighbor(1,11, aa_ile, 1) = 5; // HB--CB
	nbonded_neighbors( 12, aa_ile, 1) = 1; //1HG2
		bonded_neighbor(1,12, aa_ile, 1) = 7; //1HG2--CG2
	nbonded_neighbors( 13, aa_ile, 1) = 1; //2HG2
		bonded_neighbor(1,13, aa_ile, 1) = 7; //2HG2--CG2
	nbonded_neighbors( 14, aa_ile, 1) = 1; //3HG2
		bonded_neighbor(1,14, aa_ile, 1) = 7; //3HG2--CG2
	nbonded_neighbors( 15, aa_ile, 1) = 1; //2HG1
		bonded_neighbor(1,15, aa_ile, 1) = 6; //2HG1--CG1
	nbonded_neighbors( 16, aa_ile, 1) = 1; //3HG1
		bonded_neighbor(1,16, aa_ile, 1) = 6; //3HG1--CG1
	nbonded_neighbors( 17, aa_ile, 1) = 1; //1HD1
		bonded_neighbor(1,17, aa_ile, 1) = 8; //1HD1--CD1
	nbonded_neighbors( 18, aa_ile, 1) = 1; //2HD1
		bonded_neighbor(1,18, aa_ile, 1) = 8; //2HD1--CD1
	nbonded_neighbors( 19, aa_ile, 1) = 1; //3HD1
		bonded_neighbor(1,19, aa_ile, 1) = 8; //3HD1--CD1

//jjh get the angles automatically
	get_angles_from_bonds(aa_ile, 1);

//chu   base atom numbers ILE
	atom_base( 1, aa_ile,1) = 2; //  N  -> CA
	atom_base( 2, aa_ile,1) = 1; //  CA -> N
	atom_base( 3, aa_ile,1) = 2; //  C  -> CA
	atom_base( 4, aa_ile,1) = 3; //  O  -> C
	atom_base( 5, aa_ile,1) = 2; //  CB -> CA
	atom_base( 6, aa_ile,1) = 5; //  CG1 -> CB
	atom_base( 7, aa_ile,1) = 5; //  CG2 -> CB
	atom_base( 8, aa_ile,1) = 6; //  CD1 -> CG1
	atom_base( 9, aa_ile,1) = 1; //  H   -> N
	atom_base(10, aa_ile,1) = 2; //  HA  -> CA
	atom_base(11, aa_ile,1) = 5; //  HB  -> CB
	atom_base(12, aa_ile,1) = 7; // 1HG2 -> CG2
	atom_base(13, aa_ile,1) = 7; // 2HG2 -> CG2
	atom_base(14, aa_ile,1) = 7; // 3HG2 -> CG2
	atom_base(15, aa_ile,1) = 6; // 2HG1 -> CG1
	atom_base(16, aa_ile,1) = 6; // 3HG1 -> CG1
	atom_base(17, aa_ile,1) = 8; // 1HD1 -> CD1
	atom_base(18, aa_ile,1) = 8; // 2HD1 -> CD1
	atom_base(19, aa_ile,1) = 8; // 3HD1 -> CD1

//bk   atom type numbers ILE
	fullatom_type( 1, aa_ile,1) = 17; // Nbb    N
	fullatom_type( 2, aa_ile,1) = 18; // CAbb   CA
	fullatom_type( 3, aa_ile,1) = 19; // CObb   C
	fullatom_type( 4, aa_ile,1) = 20; // OCbb   O
	fullatom_type( 5, aa_ile,1) =  3; // CH1    CB
	fullatom_type( 6, aa_ile,1) =  4; // CH2    CG1
	fullatom_type( 7, aa_ile,1) =  5; // CH3    CG2
	fullatom_type( 8, aa_ile,1) =  5; // CH3    CD1
	fullatom_type( 9, aa_ile,1) = 25; // HNbb   H
	fullatom_type(10, aa_ile,1) = 23; // Hapo   HA
	fullatom_type(11, aa_ile,1) = 23; // Hapo   HB
	fullatom_type(12, aa_ile,1) = 23; // Hapo  1HG2
	fullatom_type(13, aa_ile,1) = 23; // Hapo  2HG2
	fullatom_type(14, aa_ile,1) = 23; // Hapo  3HG2
	fullatom_type(15, aa_ile,1) = 23; // Hapo  2HG1
	fullatom_type(16, aa_ile,1) = 23; // Hapo  3HG1
	fullatom_type(17, aa_ile,1) = 23; // Hapo  1HD1
	fullatom_type(18, aa_ile,1) = 23; // Hapo  2HD1
	fullatom_type(19, aa_ile,1) = 23; // Hapo  3HD1

//cmd  atomic charge ILE
	atomic_charge( 1, aa_ile,1) = -0.47; //    N
	atomic_charge( 2, aa_ile,1) =  0.07; //    CA
	atomic_charge( 3, aa_ile,1) =  0.51; //    C
	atomic_charge( 4, aa_ile,1) = -0.51; //    O
	atomic_charge( 5, aa_ile,1) = -0.09; //    CB
	atomic_charge( 6, aa_ile,1) = -0.18; //    CG1
	atomic_charge( 7, aa_ile,1) = -0.27; //    CG2
	atomic_charge( 8, aa_ile,1) = -0.27; //    CD1
	atomic_charge( 9, aa_ile,1) =  0.31; //    H
	atomic_charge(10, aa_ile,1) =  0.09; //    HA
	atomic_charge(11, aa_ile,1) =  0.09; //    HB
	atomic_charge(12, aa_ile,1) =  0.09; //   1HG2
	atomic_charge(13, aa_ile,1) =  0.09; //   2HG2
	atomic_charge(14, aa_ile,1) =  0.09; //   3HG2
	atomic_charge(15, aa_ile,1) =  0.09; //   2HG1
	atomic_charge(16, aa_ile,1) =  0.09; //   3HG1
	atomic_charge(17, aa_ile,1) =  0.09; //   1HD1
	atomic_charge(18, aa_ile,1) =  0.09; //   2HD1
	atomic_charge(19, aa_ile,1) =  0.09; //   3HD1

	assert_total_charge( aa_ile, 1, 0.0);

//cmd  first sidechain atom
	first_scatom( aa_ile,1) = 5; //CB

//bk   template atoms used for placing atoms ILE

//bk   template for building  HA
	ta(1,10, aa_ile,1) =  2; //   CA
	ta(2,10, aa_ile,1) =  1; //   N
	ta(3,10, aa_ile,1) =  3; //   C

//bk   template for building  HB
	ta(1,11, aa_ile,1) =  5; //   CB
	ta(2,11, aa_ile,1) =  2; //   CA
	ta(3,11, aa_ile,1) =  6; //   CG1

//bk   template for building 1HG2
	ta(1,12, aa_ile,1) =  7; //   CG2
	ta(2,12, aa_ile,1) =  5; //   CB
	ta(3,12, aa_ile,1) =  2; //   CA

//bk   template for building 2HG2
	ta(1,13, aa_ile,1) =  7; //   CG2
	ta(2,13, aa_ile,1) =  5; //   CB
	ta(3,13, aa_ile,1) =  2; //   CA

//bk   template for building 3HG2
	ta(1,14, aa_ile,1) =  7; //   CG2
	ta(2,14, aa_ile,1) =  5; //   CB
	ta(3,14, aa_ile,1) =  2; //   CA

//bk   template for building 2HG1
	ta(1,15, aa_ile,1) =  6; //   CG1
	ta(2,15, aa_ile,1) =  5; //   CB
	ta(3,15, aa_ile,1) =  8; //   CD1

//bk   template for building 3HG1
	ta(1,16, aa_ile,1) =  6; //   CG1
	ta(2,16, aa_ile,1) =  5; //   CB
	ta(3,16, aa_ile,1) =  8; //   CD1

//bk   template for building 1HD1
	ta(1,17, aa_ile,1) =  8; //   CD1
	ta(2,17, aa_ile,1) =  6; //   CG1
	ta(3,17, aa_ile,1) =  5; //   CB

//bk   template for building 2HD1
	ta(1,18, aa_ile,1) =  8; //   CD1
	ta(2,18, aa_ile,1) =  6; //   CG1
	ta(3,18, aa_ile,1) =  5; //   CB

//bk   template for building 3HD1
	ta(1,19, aa_ile,1) =  8; //   CD1
	ta(2,19, aa_ile,1) =  6; //   CG1
	ta(3,19, aa_ile,1) =  5; //   CB

//bk   chi angles required to build atoms ILE

//bk   chi angles needed for building  CG1
	chi_required(1, 6, aa_ile,1) =  true;

//bk   chi angles needed for building  CG2
	chi_required(1, 7, aa_ile,1) =  true;

//bk   chi angles needed for building  CD1
	chi_required(1, 8, aa_ile,1) =  true;
	chi_required(2, 8, aa_ile,1) =  true;

//bk   chi angles needed for building  HB
	chi_required(1,11, aa_ile,1) =  true;

//bk   chi angles needed for building 1HG2
	chi_required(1,12, aa_ile,1) =  true;

//bk   chi angles needed for building 2HG2
	chi_required(1,13, aa_ile,1) =  true;

//bk   chi angles needed for building 3HG2
	chi_required(1,14, aa_ile,1) =  true;

//bk   chi angles needed for building 2HG1
	chi_required(1,15, aa_ile,1) =  true;
	chi_required(2,15, aa_ile,1) =  true;

//bk   chi angles needed for building 3HG1
	chi_required(1,16, aa_ile,1) =  true;
	chi_required(2,16, aa_ile,1) =  true;

//bk   chi angles needed for building 1HD1
	chi_required(1,17, aa_ile,1) =  true;
	chi_required(2,17, aa_ile,1) =  true;

//bk   chi angles needed for building 2HD1
	chi_required(1,18, aa_ile,1) =  true;
	chi_required(2,18, aa_ile,1) =  true;

//bk   chi angles needed for building 3HD1
	chi_required(1,19, aa_ile,1) =  true;
	chi_required(2,19, aa_ile,1) =  true;

//bk   four atoms that define chi angle  1
	chi_atoms( 1, 1, aa_ile, 1) =  1; //   N
	chi_atoms( 2, 1, aa_ile, 1) =  2; //   CA
	chi_atoms( 3, 1, aa_ile, 1) =  5; //   CB
	chi_atoms( 4, 1, aa_ile, 1) =  6; //   CG1
//bk   four atoms that define chi angle  2
	chi_atoms( 1, 2, aa_ile, 1) =  2; //   CA
	chi_atoms( 2, 2, aa_ile, 1) =  5; //   CB
	chi_atoms( 3, 2, aa_ile, 1) =  6; //   CG1
	chi_atoms( 4, 2, aa_ile, 1) =  8; //   CD1

	HNpos( aa_ile,1) =  9; // atom number for backbone HN
	HApos( aa_ile,1) = 10; // atom number for backbone HA
	nH_polar( aa_ile,1) =  1; // number of polar hydrogens
	nH_aromatic( aa_ile,1) =  0; // number of aromatic hydrogens
	nH_apolar( aa_ile,1) = 10; // number of apolar hydrogens
	nacceptors( aa_ile,1) =  1; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1, aa_ile,1) =  9; //  H

//bk   atom number for aromatic hydrogens

//bk   atom number for apolar hydrogens
	Hpos_apolar( 1, aa_ile,1) = 10; //  HA
	Hpos_apolar( 2, aa_ile,1) = 11; //  HB
	Hpos_apolar( 3, aa_ile,1) = 12; // 1HG2
	Hpos_apolar( 4, aa_ile,1) = 13; // 2HG2
	Hpos_apolar( 5, aa_ile,1) = 14; // 3HG2
	Hpos_apolar( 6, aa_ile,1) = 15; // 2HG1
	Hpos_apolar( 7, aa_ile,1) = 16; // 3HG1
	Hpos_apolar( 8, aa_ile,1) = 17; // 1HD1
	Hpos_apolar( 9, aa_ile,1) = 18; // 2HD1
	Hpos_apolar(10, aa_ile,1) = 19; // 3HD1

//bk   acceptor positions
	accpt_pos( 1, aa_ile, 1) =  4; //  O

//bk   hydrogens on each atom

	nhydrogens_on_atm( 1, aa_ile, 1) =  1; //  N
	hydrogens_on_atm( 1, 1, aa_ile, 1) =  9; //  H

	nhydrogens_on_atm( 2, aa_ile, 1) =  1; //  CA
	hydrogens_on_atm( 1, 2, aa_ile, 1) = 10; //  HA

	nhydrogens_on_atm( 5, aa_ile, 1) =  1; //  CB
	hydrogens_on_atm( 1, 5, aa_ile, 1) = 11; //  HB

	nhydrogens_on_atm( 6, aa_ile, 1) =  2; //  CG1
	hydrogens_on_atm( 1, 6, aa_ile, 1) = 15; // 2HG1
	hydrogens_on_atm( 2, 6, aa_ile, 1) = 16; // 3HG1

	nhydrogens_on_atm( 7, aa_ile, 1) =  3; //  CG2
	hydrogens_on_atm( 1, 7, aa_ile, 1) = 12; // 1HG2
	hydrogens_on_atm( 2, 7, aa_ile, 1) = 13; // 2HG2
	hydrogens_on_atm( 3, 7, aa_ile, 1) = 14; // 3HG2

	nhydrogens_on_atm( aa_ile, aa_ile, 1) =  3; //  CD1
	hydrogens_on_atm( 1, aa_ile, aa_ile, 1) = 17; // 1HD1
	hydrogens_on_atm( 2, aa_ile, aa_ile, 1) = 18; // 2HD1
	hydrogens_on_atm( 3, aa_ile, aa_ile, 1) = 19; // 3HD1

//bk   template coordinates for the amino acid
	icoor( 1, 1, aa_ile, 1) =    0.0000; //   N
	icoor( 2, 1, aa_ile, 1) =   13.5100; //   N
	icoor( 3, 1, aa_ile, 1) =   25.7330; //   N
	icoor( 1, 2, aa_ile, 1) =    0.0000; //   CA
	icoor( 2, 2, aa_ile, 1) =   13.0790; //   CA
	icoor( 3, 2, aa_ile, 1) =   27.1250; //   CA
	icoor( 1, 3, aa_ile, 1) =    0.0000; //   C
	icoor( 2, 3, aa_ile, 1) =   14.2670; //   C
	icoor( 3, 3, aa_ile, 1) =   28.0870; //   C
	icoor( 1, 4, aa_ile, 1) =    0.0680; //   O
	icoor( 2, 4, aa_ile, 1) =   15.4230; //   O
	icoor( 3, 4, aa_ile, 1) =   27.6610; //   O
	icoor( 1, 5, aa_ile, 1) =    1.2140; //   CB
	icoor( 2, 5, aa_ile, 1) =   12.1860; //   CB
	icoor( 3, 5, aa_ile, 1) =   27.4400; //   CB
	icoor( 1, 6, aa_ile, 1) =    1.1890; //   CG1
	icoor( 2, 6, aa_ile, 1) =   11.7490; //   CG1
	icoor( 3, 6, aa_ile, 1) =   28.9070; //   CG1
	icoor( 1, 7, aa_ile, 1) =    2.5090; //   CG2
	icoor( 2, 7, aa_ile, 1) =   12.9170; //   CG2
	icoor( 3, 7, aa_ile, 1) =   27.1210; //   CG2
	icoor( 1, 8, aa_ile, 1) =    2.3510; //   CD1
	icoor( 2, 8, aa_ile, 1) =   10.8660; //   CD1
	icoor( 3, 8, aa_ile, 1) =   29.3010; //   CD1
	icoor( 1, 9, aa_ile, 1) =    0.0310; //   H
	icoor( 2, 9, aa_ile, 1) =   14.5010; //   H
	icoor( 3, 9, aa_ile, 1) =   25.5390; //   H
	icoor( 1,10, aa_ile, 1) =   -0.9170; //   HA
	icoor( 2,10, aa_ile, 1) =   12.5400; //   HA
	icoor( 3,10, aa_ile, 1) =   27.3600; //   HA
	icoor( 1,11, aa_ile, 1) =    1.1520; //   HB
	icoor( 2,11, aa_ile, 1) =   11.2780; //   HB
	icoor( 3,11, aa_ile, 1) =   26.8410; //   HB
	icoor( 1,12, aa_ile, 1) =    3.3570; //  1HG2
	icoor( 2,12, aa_ile, 1) =   12.2720; //  1HG2
	icoor( 3,12, aa_ile, 1) =   27.3500; //  1HG2
	icoor( 1,13, aa_ile, 1) =    2.5280; //  2HG2
	icoor( 2,13, aa_ile, 1) =   13.1790; //  2HG2
	icoor( 3,13, aa_ile, 1) =   26.0640; //  2HG2
	icoor( 1,14, aa_ile, 1) =    2.5720; //  3HG2
	icoor( 2,14, aa_ile, 1) =   13.8250; //  3HG2
	icoor( 3,14, aa_ile, 1) =   27.7210; //  3HG2
	icoor( 1,15, aa_ile, 1) =    1.1970; //  2HG1
	icoor( 2,15, aa_ile, 1) =   12.6530; //  2HG1
	icoor( 3,15, aa_ile, 1) =   29.5150; //  2HG1
	icoor( 1,16, aa_ile, 1) =    0.2540; //  3HG1
	icoor( 2,16, aa_ile, 1) =   11.2120; //  3HG1
	icoor( 3,16, aa_ile, 1) =   29.0690; //  3HG1
	icoor( 1,17, aa_ile, 1) =    2.2640; //  1HD1
	icoor( 2,17, aa_ile, 1) =   10.5970; //  1HD1
	icoor( 3,17, aa_ile, 1) =   30.3540; //  1HD1
	icoor( 1,18, aa_ile, 1) =    2.3430; //  2HD1
	icoor( 2,18, aa_ile, 1) =    9.9600; //  2HD1
	icoor( 3,18, aa_ile, 1) =   28.6940; //  2HD1
	icoor( 1,19, aa_ile, 1) =    3.2860; //  3HD1
	icoor( 2,19, aa_ile, 1) =   11.4010; //  3HD1
	icoor( 3,19, aa_ile, 1) =   29.1410; //  3HD1


	// H positions from REDUCE
	if( param_pack::packer_logical::reduce_templates ) {
		icoor( 1 , 9 , aa_ile , 1) = -0.933 ; // H
		icoor( 2 , 9 , aa_ile , 1) = 13.483 ; // H
		icoor( 3 , 9 , aa_ile , 1) = 25.375 ; // H
		icoor( 1 , 10 , aa_ile , 1) = -0.925 ; // HA
		icoor( 2 , 10 , aa_ile , 1) = 12.501 ; // HA
		icoor( 3 , 10 , aa_ile , 1) = 27.267 ; // HA
		icoor( 1 , 11 , aa_ile , 1) = 1.160 ; // HB
		icoor( 2 , 11 , aa_ile , 1) = 11.287 ; // HB
		icoor( 3 , 11 , aa_ile , 1) = 26.809 ; // HB
		icoor( 1 , 15 , aa_ile , 1) = 1.188 ; // 1HG1
		icoor( 2 , 15 , aa_ile , 1) = 12.644 ; // 1HG1
		icoor( 3 , 15 , aa_ile , 1) = 29.546 ; // 1HG1
		icoor( 1 , 16 , aa_ile , 1) = 0.249 ; // 2HG1
		icoor( 2 , 16 , aa_ile , 1) = 11.212 ; // 2HG1
		icoor( 3 , 16 , aa_ile , 1) = 29.104 ; // 2HG1
		icoor( 1 , 12 , aa_ile , 1) = 3.365 ; // 1HG2
		icoor( 2 , 12 , aa_ile , 1) = 12.266 ; // 1HG2
		icoor( 3 , 12 , aa_ile , 1) = 27.352 ; // 1HG2
		icoor( 1 , 13 , aa_ile , 1) = 2.528 ; // 2HG2
		icoor( 2 , 13 , aa_ile , 1) = 13.182 ; // 2HG2
		icoor( 3 , 13 , aa_ile , 1) = 26.054 ; // 2HG2
		icoor( 1 , 14 , aa_ile , 1) = 2.572 ; // 3HG2
		icoor( 2 , 14 , aa_ile , 1) = 13.833 ; // 3HG2
		icoor( 3 , 14 , aa_ile , 1) = 27.727 ; // 3HG2
		icoor( 1 , 17 , aa_ile , 1) = 2.263 ; // 1HD1
		icoor( 2 , 17 , aa_ile , 1) = 10.595 ; // 1HD1
		icoor( 3 , 17 , aa_ile , 1) = 30.364 ; // 1HD1
		icoor( 1 , 18 , aa_ile , 1) = 2.342 ; // 2HD1
		icoor( 2 , 18 , aa_ile , 1) = 9.952 ; // 2HD1
		icoor( 3 , 18 , aa_ile , 1) = 28.688 ; // 2HD1
		icoor( 1 , 19 , aa_ile , 1) = 3.295 ; // 3HD1
		icoor( 2 , 19 , aa_ile , 1) = 11.407 ; // 3HD1
		icoor( 3 , 19 , aa_ile , 1) = 29.138 ; // 3HD1
	}

//bq create N and Cterm variants
	create_termini_variants( aa_ile );


//bk   create variants that have h2o connected to the backbone nh and carbonyl
	if ( explicit_h2o ) {
		create_backbone_h2o_variants(aa_ile);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_lys
///
/// @brief arrays filled to define the proerties of the amino acid Lysine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// aa_lys - index number for LYS (9)
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_lys()
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

//bk   Parameters for LYS -----------------------------
	nvar( aa_lys ) = 1; // starting number of variants for LYS
	variant_type( aav_base, aa_lys, 1 ) = true; // type of aa variant

//bk   variant1 ---------- LYS

	natoms( aa_lys,1) = 22; // number of total atoms
	nheavyatoms( aa_lys,1) =  9; // number of heavy atoms
	nchi( aa_lys,1) =  4; // number of chi angles

//bk   atom names LYS
	atom_name( 1, aa_lys,1) = " N  ";
	atom_name( 2, aa_lys,1) = " CA ";
	atom_name( 3, aa_lys,1) = " C  ";
	atom_name( 4, aa_lys,1) = " O  ";
	atom_name( 5, aa_lys,1) = " CB ";
	atom_name( 6, aa_lys,1) = " CG ";
	atom_name( 7, aa_lys,1) = " CD ";
	atom_name( 8, aa_lys,1) = " CE ";
	atom_name( 9, aa_lys,1) = " NZ ";
	atom_name(10, aa_lys,1) = " H  ";
	atom_name(11, aa_lys,1) = "1HZ ";
	atom_name(12, aa_lys,1) = "2HZ ";
	atom_name(13, aa_lys,1) = "3HZ ";
	atom_name(14, aa_lys,1) = " HA ";
	atom_name(15, aa_lys,1) = "2HB ";
	atom_name(16, aa_lys,1) = "3HB ";
	atom_name(17, aa_lys,1) = "2HG ";
	atom_name(18, aa_lys,1) = "3HG ";
	atom_name(19, aa_lys,1) = "2HD ";
	atom_name(20, aa_lys,1) = "3HD ";
	atom_name(21, aa_lys,1) = "2HE ";
	atom_name(22, aa_lys,1) = "3HE ";

//jjh intra residue bonding
	nbonded_neighbors( 1, aa_lys, 1) = 2; // N
		bonded_neighbor(1, 1, aa_lys, 1) = 2; // N--CA
		bonded_neighbor(2, 1, aa_lys, 1) =10; // N--H
	nbonded_neighbors( 2, aa_lys, 1) = 4; // CA
		bonded_neighbor(1, 2, aa_lys, 1) = 1; // CA--N
		bonded_neighbor(2, 2, aa_lys, 1) = 3; // CA--C
		bonded_neighbor(3, 2, aa_lys, 1) = 5; // CA--CB
		bonded_neighbor(4, 2, aa_lys, 1) =14; // CA--HA
	nbonded_neighbors( 3, aa_lys, 1) = 2; // C
		bonded_neighbor(1, 3, aa_lys, 1) = 2; // C--CA
		bonded_neighbor(2, 3, aa_lys, 1) = 4; // C--O
	nbonded_neighbors( 4, aa_lys, 1) = 1; // O
		bonded_neighbor(1, 4, aa_lys, 1) = 3; // O--C
	nbonded_neighbors( 5, aa_lys, 1) = 4; // CB
		bonded_neighbor(1, 5, aa_lys, 1) = 2; // CB--CA
		bonded_neighbor(2, 5, aa_lys, 1) = 6; // CB--CG
		bonded_neighbor(3, 5, aa_lys, 1) =15; // CB--2HB
		bonded_neighbor(4, 5, aa_lys, 1) =16; // CB--3HB
	nbonded_neighbors( 6, aa_lys, 1) = 4; // CG
		bonded_neighbor(1, 6, aa_lys, 1) = 5; // CG--CB
		bonded_neighbor(2, 6, aa_lys, 1) = 7; // CG--CD
		bonded_neighbor(3, 6, aa_lys, 1) =17; // CG--2HG
		bonded_neighbor(4, 6, aa_lys, 1) =18; // CG--3HG
	nbonded_neighbors( 7, aa_lys, 1) = 4; // CD
		bonded_neighbor(1, 7, aa_lys, 1) = 6; // CD--CG
		bonded_neighbor(2, 7, aa_lys, 1) = 8; // CD--CE
		bonded_neighbor(3, 7, aa_lys, 1) =19; // CD--2HD
		bonded_neighbor(4, 7, aa_lys, 1) =20; // CD--3HD
	nbonded_neighbors( 8, aa_lys, 1) = 4; // CE
		bonded_neighbor(1, 8, aa_lys, 1) = 7; // CE--CD
		bonded_neighbor(2, 8, aa_lys, 1) = 9; // CE--NZ
		bonded_neighbor(3, 8, aa_lys, 1) =21; // CE--2HE
		bonded_neighbor(4, 8, aa_lys, 1) =22; // CE--3HE
	nbonded_neighbors( 9, aa_lys, 1) = 4; // NZ
		bonded_neighbor(1, 9, aa_lys, 1) = 8; // NZ--CE
		bonded_neighbor(2, 9, aa_lys, 1) =11; // NZ--1HZ
		bonded_neighbor(3, 9, aa_lys, 1) =12; // NZ--2HZ
		bonded_neighbor(4, 9, aa_lys, 1) =13; // NZ--3HZ
	nbonded_neighbors( 10, aa_lys, 1) = 1; // H
		bonded_neighbor(1,10, aa_lys, 1) = 1; // H--N
	nbonded_neighbors( 11, aa_lys, 1) = 1; //1HZ
		bonded_neighbor(1,11, aa_lys, 1) = 9; //1HZ--NZ
	nbonded_neighbors( 12, aa_lys, 1) = 1; //2HZ
		bonded_neighbor(1,12, aa_lys, 1) = 9; //2HZ--NZ
	nbonded_neighbors( 13, aa_lys, 1) = 1; //3HZ
		bonded_neighbor(1,13, aa_lys, 1) = 9; //3HZ--NZ
	nbonded_neighbors( 14, aa_lys, 1) = 1; // HA
		bonded_neighbor(1,14, aa_lys, 1) = 2; // HA--CA
	nbonded_neighbors( 15, aa_lys, 1) = 1; //2HB
		bonded_neighbor(1,15, aa_lys, 1) = 5; //2HB--CB
	nbonded_neighbors( 16, aa_lys, 1) = 1; //3HB
		bonded_neighbor(1,16, aa_lys, 1) = 5; //3HB--CB
	nbonded_neighbors( 17, aa_lys, 1) = 1; //2HG
		bonded_neighbor(1,17, aa_lys, 1) = 6; //2HG--CG
	nbonded_neighbors( 18, aa_lys, 1) = 1; //3HG
		bonded_neighbor(1,18, aa_lys, 1) = 6; //3HG--CG
	nbonded_neighbors( 19, aa_lys, 1) = 1; //2HD
		bonded_neighbor(1,19, aa_lys, 1) = 7; //2HD--CD
	nbonded_neighbors( 20, aa_lys, 1) = 1; //3HD
		bonded_neighbor(1,20, aa_lys, 1) = 7; //3HD--CD
	nbonded_neighbors( 21, aa_lys, 1) = 1; //2HE
		bonded_neighbor(1,21, aa_lys, 1) = 8; //2HE--CE
	nbonded_neighbors( 22, aa_lys, 1) = 1; //3HE
		bonded_neighbor(1,22, aa_lys, 1) = 8; //3HE--CE

//jjh get the angles automatically
	get_angles_from_bonds(aa_lys, 1);

//chu   base atom numbers LYS
	atom_base( 1, aa_lys,1) =  2; //  N  -> CA
	atom_base( 2, aa_lys,1) =  1; //  CA -> N
	atom_base( 3, aa_lys,1) =  2; //  C  -> CA
	atom_base( 4, aa_lys,1) =  3; //  O  -> C
	atom_base( 5, aa_lys,1) =  2; //  CB -> CA
	atom_base( 6, aa_lys,1) =  5; //  CG -> CB
	atom_base( 7, aa_lys,1) =  6; //  CD -> CG
	atom_base( 8, aa_lys,1) =  7; //  CE -> CD
	atom_base( 9, aa_lys,1) =  8; //  NZ -> CE
	atom_base(10, aa_lys,1) =  1; //  H   on  N
	atom_base(11, aa_lys,1) =  9; // 1HZ  on  NZ
	atom_base(12, aa_lys,1) =  9; // 2HZ  on  NZ
	atom_base(13, aa_lys,1) =  9; // 3HZ  on  NZ
	atom_base(14, aa_lys,1) =  2; //  HA  on  CA
	atom_base(15, aa_lys,1) =  5; // 2HB  on  CB
	atom_base(16, aa_lys,1) =  5; // 3HB  on  CB
	atom_base(17, aa_lys,1) =  6; // 2HG  on  CG
	atom_base(18, aa_lys,1) =  6; // 3HG  on  CG
	atom_base(19, aa_lys,1) =  7; // 2HD  on  CD
	atom_base(20, aa_lys,1) =  7; // 3HD  on  CD
	atom_base(21, aa_lys,1) =  8; // 2HE  on  CE
	atom_base(22, aa_lys,1) =  8; // 3HE  on  CE

//bk   atom type numbers LYS
	fullatom_type( 1, aa_lys,1) = 17; // Nbb    N
	fullatom_type( 2, aa_lys,1) = 18; // CAbb   CA
	fullatom_type( 3, aa_lys,1) = 19; // CObb   C
	fullatom_type( 4, aa_lys,1) = 20; // OCbb   O
	fullatom_type( 5, aa_lys,1) =  4; // CH2    CB
	fullatom_type( 6, aa_lys,1) =  4; // CH2    CG
	fullatom_type( 7, aa_lys,1) =  4; // CH2    CD
	fullatom_type( 8, aa_lys,1) =  4; // CH2    CE
	fullatom_type( 9, aa_lys,1) = 10; // Nlys   NZ
	fullatom_type(10, aa_lys,1) = 25; // HNbb   H
	fullatom_type(11, aa_lys,1) = 22; // Hpol  1HZ
	fullatom_type(12, aa_lys,1) = 22; // Hpol  2HZ
	fullatom_type(13, aa_lys,1) = 22; // Hpol  3HZ
	fullatom_type(14, aa_lys,1) = 23; // Hapo   HA
	fullatom_type(15, aa_lys,1) = 23; // Hapo  2HB
	fullatom_type(16, aa_lys,1) = 23; // Hapo  3HB
	fullatom_type(17, aa_lys,1) = 23; // Hapo  2HG
	fullatom_type(18, aa_lys,1) = 23; // Hapo  3HG
	fullatom_type(19, aa_lys,1) = 23; // Hapo  2HD
	fullatom_type(20, aa_lys,1) = 23; // Hapo  3HD
	fullatom_type(21, aa_lys,1) = 23; // Hapo  2HE
	fullatom_type(22, aa_lys,1) = 23; // Hapo  3HE

//cmd  atomic charge LYS
	atomic_charge( 1, aa_lys,1) = -0.47; //    N
	atomic_charge( 2, aa_lys,1) =  0.07; //    CA
	atomic_charge( 3, aa_lys,1) =  0.51; //    C
	atomic_charge( 4, aa_lys,1) = -0.51; //    O
	atomic_charge( 5, aa_lys,1) = -0.18; //    CB
	atomic_charge( 6, aa_lys,1) = -0.18; //    CG
	atomic_charge( 7, aa_lys,1) = -0.18; //    CD
	atomic_charge( 8, aa_lys,1) =  0.21; //    CE
	atomic_charge( 9, aa_lys,1) = -0.30; //    NZ
	atomic_charge(10, aa_lys,1) =  0.31; //    H
	atomic_charge(11, aa_lys,1) =  0.33; //   1HZ
	atomic_charge(12, aa_lys,1) =  0.33; //   2HZ
	atomic_charge(13, aa_lys,1) =  0.33; //   3HZ
	atomic_charge(14, aa_lys,1) =  0.09; //    HA
	atomic_charge(15, aa_lys,1) =  0.09; //   2HB
	atomic_charge(16, aa_lys,1) =  0.09; //   3HB
	atomic_charge(17, aa_lys,1) =  0.09; //   2HG
	atomic_charge(18, aa_lys,1) =  0.09; //   3HG
	atomic_charge(19, aa_lys,1) =  0.09; //   2HD
	atomic_charge(20, aa_lys,1) =  0.09; //   3HD
	atomic_charge(21, aa_lys,1) =  0.05; //   2HE
	atomic_charge(22, aa_lys,1) =  0.05; //   3HE

	assert_total_charge( aa_lys, 1, 1.0);

//cmd  first sidechain atom
	first_scatom( aa_lys,1) = 5; //CB

//bk   template atoms used for placing atoms LYS

//bk   template for building 1HZ
	ta(1,11, aa_lys,1) =  9; //   NZ
	ta(2,11, aa_lys,1) =  8; //   CE
	ta(3,11, aa_lys,1) =  7; //   CD

//bk   template for building 2HZ
	ta(1,12, aa_lys,1) =  9; //   NZ
	ta(2,12, aa_lys,1) =  8; //   CE
	ta(3,12, aa_lys,1) =  7; //   CD

//bk   template for building 3HZ
	ta(1,13, aa_lys,1) =  9; //   NZ
	ta(2,13, aa_lys,1) =  8; //   CE
	ta(3,13, aa_lys,1) =  7; //   CD

//bk   template for building  HA
	ta(1,14, aa_lys,1) =  2; //   CA
	ta(2,14, aa_lys,1) =  1; //   N
	ta(3,14, aa_lys,1) =  3; //   C

//bk   template for building 2HB
	ta(1,15, aa_lys,1) =  5; //   CB
	ta(2,15, aa_lys,1) =  2; //   CA
	ta(3,15, aa_lys,1) =  6; //   CG

//bk   template for building 3HB
	ta(1,16, aa_lys,1) =  5; //   CB
	ta(2,16, aa_lys,1) =  2; //   CA
	ta(3,16, aa_lys,1) =  6; //   CG

//bk   template for building 2HG
	ta(1,17, aa_lys,1) =  6; //   CG
	ta(2,17, aa_lys,1) =  5; //   CB
	ta(3,17, aa_lys,1) =  7; //   CD

//bk   template for building 3HG
	ta(1,18, aa_lys,1) =  6; //   CG
	ta(2,18, aa_lys,1) =  5; //   CB
	ta(3,18, aa_lys,1) =  7; //   CD

//bk   template for building 2HD
	ta(1,19, aa_lys,1) =  7; //   CD
	ta(2,19, aa_lys,1) =  6; //   CG
	ta(3,19, aa_lys,1) =  8; //   CE

//bk   template for building 3HD
	ta(1,20, aa_lys,1) =  7; //   CD
	ta(2,20, aa_lys,1) =  6; //   CG
	ta(3,20, aa_lys,1) =  8; //   CE

//bk   template for building 2HE
	ta(1,21, aa_lys,1) =  8; //   CE
	ta(2,21, aa_lys,1) =  7; //   CD
	ta(3,21, aa_lys,1) =  9; //   NZ

//bk   template for building 3HE
	ta(1,22, aa_lys,1) =  8; //   CE
	ta(2,22, aa_lys,1) =  7; //   CD
	ta(3,22, aa_lys,1) =  9; //   NZ

//bk   chi angles required to build atoms LYS

//bk   chi angles needed for building  N
	chi_required(1, 1, aa_lys,1) = false;
	chi_required(2, 1, aa_lys,1) = false;
	chi_required(3, 1, aa_lys,1) = false;
	chi_required(4, 1, aa_lys,1) = false;

//bk   chi angles needed for building  CA
	chi_required(1, 2, aa_lys,1) = false;
	chi_required(2, 2, aa_lys,1) = false;
	chi_required(3, 2, aa_lys,1) = false;
	chi_required(4, 2, aa_lys,1) = false;

//bk   chi angles needed for building  C
	chi_required(1, 3, aa_lys,1) = false;
	chi_required(2, 3, aa_lys,1) = false;
	chi_required(3, 3, aa_lys,1) = false;
	chi_required(4, 3, aa_lys,1) = false;

//bk   chi angles needed for building  O
	chi_required(1, 4, aa_lys,1) = false;
	chi_required(2, 4, aa_lys,1) = false;
	chi_required(3, 4, aa_lys,1) = false;
	chi_required(4, 4, aa_lys,1) = false;

//bk   chi angles needed for building  CB
	chi_required(1, 5, aa_lys,1) = false;
	chi_required(2, 5, aa_lys,1) = false;
	chi_required(3, 5, aa_lys,1) = false;
	chi_required(4, 5, aa_lys,1) = false;

//bk   chi angles needed for building  CG
	chi_required(1, 6, aa_lys,1) =  true;
	chi_required(2, 6, aa_lys,1) = false;
	chi_required(3, 6, aa_lys,1) = false;
	chi_required(4, 6, aa_lys,1) = false;

//bk   chi angles needed for building  CD
	chi_required(1, 7, aa_lys,1) =  true;
	chi_required(2, 7, aa_lys,1) =  true;
	chi_required(3, 7, aa_lys,1) = false;
	chi_required(4, 7, aa_lys,1) = false;

//bk   chi angles needed for building  CE
	chi_required(1, 8, aa_lys,1) =  true;
	chi_required(2, 8, aa_lys,1) =  true;
	chi_required(3, 8, aa_lys,1) =  true;
	chi_required(4, 8, aa_lys,1) = false;

//bk   chi angles needed for building  NZ
	chi_required(1, 9, aa_lys,1) =  true;
	chi_required(2, 9, aa_lys,1) =  true;
	chi_required(3, 9, aa_lys,1) =  true;
	chi_required(4, 9, aa_lys,1) =  true;

//bk   chi angles needed for building  H
	chi_required(1,10, aa_lys,1) = false;
	chi_required(2,10, aa_lys,1) = false;
	chi_required(3,10, aa_lys,1) = false;
	chi_required(4,10, aa_lys,1) = false;

//bk   chi angles needed for building 1HZ
	chi_required(1,11, aa_lys,1) =  true;
	chi_required(2,11, aa_lys,1) =  true;
	chi_required(3,11, aa_lys,1) =  true;
	chi_required(4,11, aa_lys,1) =  true;

//bk   chi angles needed for building 2HZ
	chi_required(1,12, aa_lys,1) =  true;
	chi_required(2,12, aa_lys,1) =  true;
	chi_required(3,12, aa_lys,1) =  true;
	chi_required(4,12, aa_lys,1) =  true;

//bk   chi angles needed for building 3HZ
	chi_required(1,13, aa_lys,1) =  true;
	chi_required(2,13, aa_lys,1) =  true;
	chi_required(3,13, aa_lys,1) =  true;
	chi_required(4,13, aa_lys,1) =  true;

//bk   chi angles needed for building  HA
	chi_required(1,14, aa_lys,1) = false;
	chi_required(2,14, aa_lys,1) = false;
	chi_required(3,14, aa_lys,1) = false;
	chi_required(4,14, aa_lys,1) = false;

//bk   chi angles needed for building 2HB
	chi_required(1,15, aa_lys,1) =  true;
	chi_required(2,15, aa_lys,1) = false;
	chi_required(3,15, aa_lys,1) = false;
	chi_required(4,15, aa_lys,1) = false;

//bk   chi angles needed for building 3HB
	chi_required(1,16, aa_lys,1) =  true;
	chi_required(2,16, aa_lys,1) = false;
	chi_required(3,16, aa_lys,1) = false;
	chi_required(4,16, aa_lys,1) = false;

//bk   chi angles needed for building 2HG
	chi_required(1,17, aa_lys,1) =  true;
	chi_required(2,17, aa_lys,1) =  true;
	chi_required(3,17, aa_lys,1) = false;
	chi_required(4,17, aa_lys,1) = false;

//bk   chi angles needed for building 3HG
	chi_required(1,18, aa_lys,1) =  true;
	chi_required(2,18, aa_lys,1) =  true;
	chi_required(3,18, aa_lys,1) = false;
	chi_required(4,18, aa_lys,1) = false;

//bk   chi angles needed for building 2HD
	chi_required(1,19, aa_lys,1) =  true;
	chi_required(2,19, aa_lys,1) =  true;
	chi_required(3,19, aa_lys,1) =  true;
	chi_required(4,19, aa_lys,1) = false;

//bk   chi angles needed for building 3HD
	chi_required(1,20, aa_lys,1) =  true;
	chi_required(2,20, aa_lys,1) =  true;
	chi_required(3,20, aa_lys,1) =  true;
	chi_required(4,20, aa_lys,1) = false;

//bk   chi angles needed for building 2HE
	chi_required(1,21, aa_lys,1) =  true;
	chi_required(2,21, aa_lys,1) =  true;
	chi_required(3,21, aa_lys,1) =  true;
	chi_required(4,21, aa_lys,1) =  true;

//bk   chi angles needed for building 3HE
	chi_required(1,22, aa_lys,1) =  true;
	chi_required(2,22, aa_lys,1) =  true;
	chi_required(3,22, aa_lys,1) =  true;
	chi_required(4,22, aa_lys,1) =  true;

//bk   four atoms that define chi angle  1
	chi_atoms( 1, 1, aa_lys, 1) =  1; //   N
	chi_atoms( 2, 1, aa_lys, 1) =  2; //   CA
	chi_atoms( 3, 1, aa_lys, 1) =  5; //   CB
	chi_atoms( 4, 1, aa_lys, 1) =  6; //   CG
//bk   four atoms that define chi angle  2
	chi_atoms( 1, 2, aa_lys, 1) =  2; //   CA
	chi_atoms( 2, 2, aa_lys, 1) =  5; //   CB
	chi_atoms( 3, 2, aa_lys, 1) =  6; //   CG
	chi_atoms( 4, 2, aa_lys, 1) =  7; //   CD
//bk   four atoms that define chi angle  3
	chi_atoms( 1, 3, aa_lys, 1) =  5; //   CB
	chi_atoms( 2, 3, aa_lys, 1) =  6; //   CG
	chi_atoms( 3, 3, aa_lys, 1) =  7; //   CD
	chi_atoms( 4, 3, aa_lys, 1) =  8; //   CE
//bk   four atoms that define chi angle  4
	chi_atoms( 1, 4, aa_lys, 1) =  6; //   CG
	chi_atoms( 2, 4, aa_lys, 1) =  7; //   CD
	chi_atoms( 3, 4, aa_lys, 1) =  8; //   CE
	chi_atoms( 4, 4, aa_lys, 1) =  9; //   NZ

	HNpos( aa_lys,1) = 10; // atom number for backbone HN
	HApos( aa_lys,1) = 14; // atom number for backbone HA
	nH_polar( aa_lys,1) =  4; // number of polar hydrogens
	nH_aromatic( aa_lys,1) =  0; // number of aromatic hydrogens
	nH_apolar( aa_lys,1) =  9; // number of apolar hydrogens
	nacceptors( aa_lys,1) =  1; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1, aa_lys,1) = 10; //  H
	Hpos_polar( 2, aa_lys,1) = 11; // 1HZ
	Hpos_polar( 3, aa_lys,1) = 12; // 2HZ
	Hpos_polar( 4, aa_lys,1) = 13; // 3HZ

//bk   atom number for aromatic hydrogens

//bk   atom number for apolar hydrogens
	Hpos_apolar( 1, aa_lys,1) = 14; //  HA
	Hpos_apolar( 2, aa_lys,1) = 15; // 2HB
	Hpos_apolar( 3, aa_lys,1) = 16; // 3HB
	Hpos_apolar( 4, aa_lys,1) = 17; // 2HG
	Hpos_apolar( 5, aa_lys,1) = 18; // 3HG
	Hpos_apolar( 6, aa_lys,1) = 19; // 2HD
	Hpos_apolar( 7, aa_lys,1) = 20; // 3HD
	Hpos_apolar( 8, aa_lys,1) = 21; // 2HE
	Hpos_apolar( 9, aa_lys,1) = 22; // 3HE

//bk   acceptor positions
	accpt_pos( 1, aa_lys, 1) =  4; //  O

//bk   hydrogens on each atom

	nhydrogens_on_atm( 1, aa_lys, 1) =  1; //  N
	hydrogens_on_atm( 1, 1, aa_lys, 1) = 10; //  H

	nhydrogens_on_atm( 2, aa_lys, 1) =  1; //  CA
	hydrogens_on_atm( 1, 2, aa_lys, 1) = 14; //  HA

	nhydrogens_on_atm( 5, aa_lys, 1) =  2; //  CB
	hydrogens_on_atm( 1, 5, aa_lys, 1) = 15; // 2HB
	hydrogens_on_atm( 2, 5, aa_lys, 1) = 16; // 3HB

	nhydrogens_on_atm( 6, aa_lys, 1) =  2; //  CG
	hydrogens_on_atm( 1, 6, aa_lys, 1) = 17; // 2HG
	hydrogens_on_atm( 2, 6, aa_lys, 1) = 18; // 3HG

	nhydrogens_on_atm( 7, aa_lys, 1) =  2; //  CD
	hydrogens_on_atm( 1, 7, aa_lys, 1) = 19; // 2HD
	hydrogens_on_atm( 2, 7, aa_lys, 1) = 20; // 3HD

	nhydrogens_on_atm( 8, aa_lys, 1) =  2; //  CE
	hydrogens_on_atm( 1, 8, aa_lys, 1) = 21; // 2HE
	hydrogens_on_atm( 2, 8, aa_lys, 1) = 22; // 3HE

	nhydrogens_on_atm( 9, aa_lys, 1) =  3; //  NZ
	hydrogens_on_atm( 1, 9, aa_lys, 1) = 11; // 1HZ
	hydrogens_on_atm( 2, 9, aa_lys, 1) = 12; // 2HZ
	hydrogens_on_atm( 3, 9, aa_lys, 1) = 13; // 3HZ

//bk   template coordinates for the amino acid
	icoor( 1, 1, aa_lys, 1) =    0.0000; //   N
	icoor( 2, 1, aa_lys, 1) =   13.9730; //   N
	icoor( 3, 1, aa_lys, 1) =   29.3820; //   N
	icoor( 1, 2, aa_lys, 1) =    0.0000; //   CA
	icoor( 2, 2, aa_lys, 1) =   15.0130; //   CA
	icoor( 3, 2, aa_lys, 1) =   30.4040; //   CA
	icoor( 1, 3, aa_lys, 1) =    0.0000; //   C
	icoor( 2, 3, aa_lys, 1) =   14.4110; //   C
	icoor( 3, 3, aa_lys, 1) =   31.8030; //   C
	icoor( 1, 4, aa_lys, 1) =    0.0030; //   O
	icoor( 2, 4, aa_lys, 1) =   13.1890; //   O
	icoor( 3, 4, aa_lys, 1) =   31.9540; //   O
	icoor( 1, 5, aa_lys, 1) =   -1.2070; //   CB
	icoor( 2, 5, aa_lys, 1) =   15.9360; //   CB
	icoor( 3, 5, aa_lys, 1) =   30.2290; //   CB
	icoor( 1, 6, aa_lys, 1) =   -1.2940; //   CG
	icoor( 2, 6, aa_lys, 1) =   17.0590; //   CG
	icoor( 3, 6, aa_lys, 1) =   31.2540; //   CG
	icoor( 1, 7, aa_lys, 1) =   -2.5170; //   CD
	icoor( 2, 7, aa_lys, 1) =   17.9320; //   CD
	icoor( 3, 7, aa_lys, 1) =   31.0160; //   CD
	icoor( 1, 8, aa_lys, 1) =   -2.6050; //   CE
	icoor( 2, 8, aa_lys, 1) =   19.0540; //   CE
	icoor( 3, 8, aa_lys, 1) =   32.0400; //   CE
	icoor( 1, 9, aa_lys, 1) =   -3.7970; //   NZ
	icoor( 2, 9, aa_lys, 1) =   19.9170; //   NZ
	icoor( 3, 9, aa_lys, 1) =   31.8190; //   NZ
	icoor( 1,10, aa_lys, 1) =   -0.0400; //   H
	icoor( 2,10, aa_lys, 1) =   13.0090; //   H
	icoor( 3,10, aa_lys, 1) =   29.6800; //   H
	icoor( 1,11, aa_lys, 1) =   -3.8180; //  1HZ
	icoor( 2,11, aa_lys, 1) =   20.6470; //  1HZ
	icoor( 3,11, aa_lys, 1) =   32.5170; //  1HZ
	icoor( 1,12, aa_lys, 1) =   -3.7470; //  2HZ
	icoor( 2,12, aa_lys, 1) =   20.3320; //  2HZ
	icoor( 3,12, aa_lys, 1) =   30.8990; //  2HZ
	icoor( 1,13, aa_lys, 1) =   -4.6350; //  3HZ
	icoor( 2,13, aa_lys, 1) =   19.3580; //  3HZ
	icoor( 3,13, aa_lys, 1) =   31.8910; //  3HZ
	icoor( 1,14, aa_lys, 1) =    0.9090; //   HA
	icoor( 2,14, aa_lys, 1) =   15.6100; //   HA
	icoor( 3,14, aa_lys, 1) =   30.3230; //   HA
	icoor( 1,15, aa_lys, 1) =   -1.1400; //  2HB
	icoor( 2,15, aa_lys, 1) =   16.3630; //  2HB
	icoor( 3,15, aa_lys, 1) =   29.2280; //  2HB
	icoor( 1,16, aa_lys, 1) =   -2.0990; //  3HB
	icoor( 2,16, aa_lys, 1) =   15.3130; //  3HB
	icoor( 3,16, aa_lys, 1) =   30.2980; //  3HB
	icoor( 1,17, aa_lys, 1) =   -1.3510; //  2HG
	icoor( 2,17, aa_lys, 1) =   16.6160; //  2HG
	icoor( 3,17, aa_lys, 1) =   32.2490; //  2HG
	icoor( 1,18, aa_lys, 1) =   -0.3930; //  3HG
	icoor( 2,18, aa_lys, 1) =   17.6670; //  3HG
	icoor( 3,18, aa_lys, 1) =   31.1790; //  3HG
	icoor( 1,19, aa_lys, 1) =   -2.4500; //  2HD
	icoor( 2,19, aa_lys, 1) =   18.3580; //  2HD
	icoor( 3,19, aa_lys, 1) =   30.0140; //  2HD
	icoor( 1,20, aa_lys, 1) =   -3.4080; //  3HD
	icoor( 2,20, aa_lys, 1) =   17.3080; //  3HD
	icoor( 3,20, aa_lys, 1) =   31.0850; //  3HD
	icoor( 1,21, aa_lys, 1) =   -2.6590; //  2HE
	icoor( 2,21, aa_lys, 1) =   18.6080; //  2HE
	icoor( 3,21, aa_lys, 1) =   33.0320; //  2HE
	icoor( 1,22, aa_lys, 1) =   -1.7010; //  3HE
	icoor( 2,22, aa_lys, 1) =   19.6580; //  3HE
	icoor( 3,22, aa_lys, 1) =   31.9630; //  3HE

	// H positions from REDUCE
	if( param_pack::packer_logical::reduce_templates ) {
		icoor( 1 , 10 , aa_lys , 1) = -0.033 ; // H
		icoor( 2 , 10 , aa_lys , 1) = 13.075 ; // H
		icoor( 3 , 10 , aa_lys , 1) = 29.820 ; // H
		icoor( 1 , 14 , aa_lys , 1) = 0.921 ; // HA
		icoor( 2 , 14 , aa_lys , 1) = 15.602 ; // HA
		icoor( 3 , 14 , aa_lys , 1) = 30.284 ; // HA
		icoor( 1 , 15 , aa_lys , 1) = -1.173 ; // 1HB
		icoor( 2 , 15 , aa_lys , 1) = 16.377 ; // 1HB
		icoor( 3 , 15 , aa_lys , 1) = 29.222 ; // 1HB
		icoor( 1 , 16 , aa_lys , 1) = -2.125 ; // 2HB
		icoor( 2 , 16 , aa_lys , 1) = 15.333 ; // 2HB
		icoor( 3 , 16 , aa_lys , 1) = 30.285 ; // 2HB
		icoor( 1 , 17 , aa_lys , 1) = -1.337 ; // 1HG
		icoor( 2 , 17 , aa_lys , 1) = 16.632 ; // 1HG
		icoor( 3 , 17 , aa_lys , 1) = 32.267 ; // 1HG
		icoor( 1 , 18 , aa_lys , 1) = -0.385 ; // 2HG
		icoor( 2 , 18 , aa_lys , 1) = 17.676 ; // 2HG
		icoor( 3 , 18 , aa_lys , 1) = 31.204 ; // 2HG
		icoor( 1 , 19 , aa_lys , 1) = -2.474 ; // 1HD
		icoor( 2 , 19 , aa_lys , 1) = 18.359 ; // 1HD
		icoor( 3 , 19 , aa_lys , 1) = 30.003 ; // 1HD
		icoor( 1 , 20 , aa_lys , 1) = -3.426 ; // 2HD
		icoor( 2 , 20 , aa_lys , 1) = 17.315 ; // 2HD
		icoor( 3 , 20 , aa_lys , 1) = 31.066 ; // 2HD
		icoor( 1 , 21 , aa_lys , 1) = -2.647 ; // 1HE
		icoor( 2 , 21 , aa_lys , 1) = 18.624 ; // 1HE
		icoor( 3 , 21 , aa_lys , 1) = 33.052 ; // 1HE
		icoor( 1 , 22 , aa_lys , 1) = -1.694 ; // 2HE
		icoor( 2 , 22 , aa_lys , 1) = 19.668 ; // 2HE
		icoor( 3 , 22 , aa_lys , 1) = 31.989 ; // 2HE
		icoor( 1 , 11 , aa_lys , 1) = -3.819 ; // 1HZ
		icoor( 2 , 11 , aa_lys , 1) = 20.640 ; // 1HZ
		icoor( 3 , 11 , aa_lys , 1) = 32.510 ; // 1HZ
		icoor( 1 , 12 , aa_lys , 1) = -3.747 ; // 2HZ
		icoor( 2 , 12 , aa_lys , 1) = 20.328 ; // 2HZ
		icoor( 3 , 12 , aa_lys , 1) = 30.909 ; // 2HZ
		icoor( 1 , 13 , aa_lys , 1) = -4.627 ; // 3HZ
		icoor( 2 , 13 , aa_lys , 1) = 19.364 ; // 3HZ
		icoor( 3 , 13 , aa_lys , 1) = 31.890 ; // 3HZ
	}

	//rh create protonation state variants
	create_pH_variants( aa_lys);

//bq create N and Cterm variants
	create_termini_variants( aa_lys );

//bk   create variants that have h2o connected to the backbone nh and carbonyl
	if ( explicit_h2o ) {
		create_backbone_h2o_variants(aa_lys);
		create_sidechain_h2o_variants(aa_lys);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_leu
///
/// @brief arrays filled to define the proerties of the amino acid Leucine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// aa_leu - index number for LEU (10)
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_leu()
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

//bk   Parameters for LEU -----------------------------
	nvar(aa_leu) = 1; // starting number of variants for LEU
	variant_type( aav_base, aa_leu, 1 ) = true; // type of aa variant

//bk   variant1 ---------- LEU

	natoms(aa_leu,1) = 19; // number of total atoms
	nheavyatoms(aa_leu,1) =  8; // number of heavy atoms
	nchi(aa_leu,1) =  2; // number of chi angles

//bk   atom names LEU
	atom_name( 1,aa_leu,1) = " N  ";
	atom_name( 2,aa_leu,1) = " CA ";
	atom_name( 3,aa_leu,1) = " C  ";
	atom_name( 4,aa_leu,1) = " O  ";
	atom_name( 5,aa_leu,1) = " CB ";
	atom_name( 6,aa_leu,1) = " CG ";
	atom_name( 7,aa_leu,1) = " CD1";
	atom_name( 8,aa_leu,1) = " CD2";
	atom_name( 9,aa_leu,1) = " H  ";
	atom_name(10,aa_leu,1) = " HA ";
	atom_name(11,aa_leu,1) = "2HB ";
	atom_name(12,aa_leu,1) = "3HB ";
	atom_name(13,aa_leu,1) = " HG ";
	atom_name(14,aa_leu,1) = "1HD1";
	atom_name(15,aa_leu,1) = "2HD1";
	atom_name(16,aa_leu,1) = "3HD1";
	atom_name(17,aa_leu,1) = "1HD2";
	atom_name(18,aa_leu,1) = "2HD2";
	atom_name(19,aa_leu,1) = "3HD2";

//jjh intra residue bonding
	nbonded_neighbors( 1, aa_leu, 1) = 2; // N
		bonded_neighbor(1, 1, aa_leu, 1) = 2; // N--CA
		bonded_neighbor(2, 1, aa_leu, 1) = 9; // N--H
	nbonded_neighbors( 2, aa_leu, 1) = 4; // CA
		bonded_neighbor(1, 2, aa_leu, 1) = 1; // CA--N
		bonded_neighbor(2, 2, aa_leu, 1) = 3; // CA--C
		bonded_neighbor(3, 2, aa_leu, 1) = 5; // CA--CB
		bonded_neighbor(4, 2, aa_leu, 1) =10; // CA--HA
	nbonded_neighbors( 3, aa_leu, 1) = 2; // C
		bonded_neighbor(1, 3, aa_leu, 1) = 2; // C--CA
		bonded_neighbor(2, 3, aa_leu, 1) = 4; // C--O
	nbonded_neighbors( 4, aa_leu, 1) = 1; // O
		bonded_neighbor(1, 4, aa_leu, 1) = 3; // O--C
	nbonded_neighbors( 5, aa_leu, 1) = 4; // CB
		bonded_neighbor(1, 5, aa_leu, 1) = 2; // CB--CA
		bonded_neighbor(2, 5, aa_leu, 1) = 6; // CB--CG
		bonded_neighbor(3, 5, aa_leu, 1) =11; // CB--2HB
		bonded_neighbor(4, 5, aa_leu, 1) =12; // CB--3HB
	nbonded_neighbors( 6, aa_leu, 1) = 4; // CG
		bonded_neighbor(1, 6, aa_leu, 1) = 5; // CG--CB
		bonded_neighbor(2, 6, aa_leu, 1) = 7; // CG--CD1
		bonded_neighbor(3, 6, aa_leu, 1) = 8; // CG--CD2
		bonded_neighbor(4, 6, aa_leu, 1) =13; // CG--HG
	nbonded_neighbors( 7, aa_leu, 1) = 4; // CD1
		bonded_neighbor(1, 7, aa_leu, 1) = 6; // CD1--CG
		bonded_neighbor(2, 7, aa_leu, 1) =14; // CD1--1HD1
		bonded_neighbor(3, 7, aa_leu, 1) =15; // CD1--2HD1
		bonded_neighbor(4, 7, aa_leu, 1) =16; // CD1--3HD1
	nbonded_neighbors( 8, aa_leu, 1) = 4; // CD2
		bonded_neighbor(1, 8, aa_leu, 1) = 6; // CD2--CG
		bonded_neighbor(2, 8, aa_leu, 1) =17; // CD2--1HD2
		bonded_neighbor(3, 8, aa_leu, 1) =18; // CD2--2HD2
		bonded_neighbor(4, 8, aa_leu, 1) =19; // CD2--3HD2
	nbonded_neighbors( 9, aa_leu, 1) = 1; // H
		bonded_neighbor(1, 9, aa_leu, 1) = 1; // H--N
	nbonded_neighbors( 10, aa_leu, 1) = 1; // HA
		bonded_neighbor(1,10, aa_leu, 1) = 2; // HA--CA
	nbonded_neighbors( 11, aa_leu, 1) = 1; //2HB
		bonded_neighbor(1,11, aa_leu, 1) = 5; //2HB--CB
	nbonded_neighbors( 12, aa_leu, 1) = 1; //3HB
		bonded_neighbor(1,12, aa_leu, 1) = 5; //3HB--CB
	nbonded_neighbors( 13, aa_leu, 1) = 1; // HG
		bonded_neighbor(1,13, aa_leu, 1) = 6; // HG--CG
	nbonded_neighbors( 14, aa_leu, 1) = 1; //1HD1
		bonded_neighbor(1,14, aa_leu, 1) = 7; //1HD1--CD1
	nbonded_neighbors( 15, aa_leu, 1) = 1; //2HD1
		bonded_neighbor(1,15, aa_leu, 1) = 7; //2HD1--CD1
	nbonded_neighbors( 16, aa_leu, 1) = 1; //3HD1
		bonded_neighbor(1,16, aa_leu, 1) = 7; //3HD1--CD1
	nbonded_neighbors( 17, aa_leu, 1) = 1; //1HD2
		bonded_neighbor(1,17, aa_leu, 1) = 8; //1HD2--CD2
	nbonded_neighbors( 18, aa_leu, 1) = 1; //2HD2
		bonded_neighbor(1,18, aa_leu, 1) = 8; //2HD2--CD2
	nbonded_neighbors( 19, aa_leu, 1) = 1; //3HD2
		bonded_neighbor(1,19, aa_leu, 1) = 8; //3HD2--CD2

//jjh get the angles automatically
	get_angles_from_bonds(aa_leu, 1);

//chu   base atom numbers LEU
	atom_base( 1,aa_leu,1) =  2; //  N  -> CA
	atom_base( 2,aa_leu,1) =  1; //  CA -> N
	atom_base( 3,aa_leu,1) =  2; //  C  -> CA
	atom_base( 4,aa_leu,1) =  3; //  O  -> C
	atom_base( 5,aa_leu,1) =  2; //  CB -> CA
	atom_base( 6,aa_leu,1) =  5; //  CG -> CB
	atom_base( 7,aa_leu,1) =  6; //  CD1 on  CG
	atom_base( 8,aa_leu,1) =  6; //  CD2 on  CG
	atom_base( 9,aa_leu,1) =  1; //  H   on  N
	atom_base(10,aa_leu,1) =  2; //  HA  on  CA
	atom_base(11,aa_leu,1) =  5; // 2HB  on  CB
	atom_base(12,aa_leu,1) =  5; // 3HB  on  CB
	atom_base(13,aa_leu,1) =  6; //  HG  on  CG
	atom_base(14,aa_leu,1) =  7; // 1HD1 on  CD1
	atom_base(15,aa_leu,1) =  7; // 2HD1 on  CD1
	atom_base(16,aa_leu,1) =  7; // 3HD1 on  CD1
	atom_base(17,aa_leu,1) =  8; // 1HD2 on  CD2
	atom_base(18,aa_leu,1) =  8; // 2HD2 on  CD2
	atom_base(19,aa_leu,1) =  8; // 3HD2 on  CD2
//bk   atom type numbers LEU
	fullatom_type( 1,aa_leu,1) = 17; // Nbb    N
	fullatom_type( 2,aa_leu,1) = 18; // CAbb   CA
	fullatom_type( 3,aa_leu,1) = 19; // CObb   C
	fullatom_type( 4,aa_leu,1) = 20; // OCbb   O
	fullatom_type( 5,aa_leu,1) =  4; // CH2    CB
	fullatom_type( 6,aa_leu,1) =  3; // CH1    CG
	fullatom_type( 7,aa_leu,1) =  5; // CH3    CD1
	fullatom_type( 8,aa_leu,1) =  5; // CH3    CD2
	fullatom_type( 9,aa_leu,1) = 25; // HNbb   H
	fullatom_type(10,aa_leu,1) = 23; // Hapo   HA
	fullatom_type(11,aa_leu,1) = 23; // Hapo  2HB
	fullatom_type(12,aa_leu,1) = 23; // Hapo  3HB
	fullatom_type(13,aa_leu,1) = 23; // Hapo   HG
	fullatom_type(14,aa_leu,1) = 23; // Hapo  1HD1
	fullatom_type(15,aa_leu,1) = 23; // Hapo  2HD1
	fullatom_type(16,aa_leu,1) = 23; // Hapo  3HD1
	fullatom_type(17,aa_leu,1) = 23; // Hapo  1HD2
	fullatom_type(18,aa_leu,1) = 23; // Hapo  2HD2
	fullatom_type(19,aa_leu,1) = 23; // Hapo  3HD2

//cmd  atomic charge LEU
	atomic_charge( 1,aa_leu,1) = -0.47; //    N
	atomic_charge( 2,aa_leu,1) =  0.07; //    CA
	atomic_charge( 3,aa_leu,1) =  0.51; //    C
	atomic_charge( 4,aa_leu,1) = -0.51; //    O
	atomic_charge( 5,aa_leu,1) = -0.18; //    CB
	atomic_charge( 6,aa_leu,1) = -0.09; //    CG
	atomic_charge( 7,aa_leu,1) = -0.27; //    CD1
	atomic_charge( 8,aa_leu,1) = -0.27; //    CD2
	atomic_charge( 9,aa_leu,1) =  0.31; //    H
	atomic_charge(10,aa_leu,1) =  0.09; //    HA
	atomic_charge(11,aa_leu,1) =  0.09; //   2HB
	atomic_charge(12,aa_leu,1) =  0.09; //   3HB
	atomic_charge(13,aa_leu,1) =  0.09; //    HG
	atomic_charge(14,aa_leu,1) =  0.09; //   1HD1
	atomic_charge(15,aa_leu,1) =  0.09; //   2HD1
	atomic_charge(16,aa_leu,1) =  0.09; //   3HD1
	atomic_charge(17,aa_leu,1) =  0.09; //   1HD2
	atomic_charge(18,aa_leu,1) =  0.09; //   2HD2
	atomic_charge(19,aa_leu,1) =  0.09; //   3HD2

	assert_total_charge( aa_leu, 1, 0.0);

//cmd  first sidechain atom
	first_scatom( aa_leu,1) = 5; //CB

//bk   template atoms used for placing atoms LEU

//bk   template for building  HA
	ta(1,10,aa_leu,1) =  2; //   CA
	ta(2,10,aa_leu,1) =  1; //   N
	ta(3,10,aa_leu,1) =  3; //   C

//bk   template for building 2HB
	ta(1,11,aa_leu,1) =  5; //   CB
	ta(2,11,aa_leu,1) =  2; //   CA
	ta(3,11,aa_leu,1) =  6; //   CG

//bk   template for building 3HB
	ta(1,12,aa_leu,1) =  5; //   CB
	ta(2,12,aa_leu,1) =  2; //   CA
	ta(3,12,aa_leu,1) =  6; //   CG

//bk   template for building  HG
	ta(1,13,aa_leu,1) =  6; //   CG
	ta(2,13,aa_leu,1) =  5; //   CB
	ta(3,13,aa_leu,1) =  7; //   CD1

//bk   template for building 1HD1
	ta(1,14,aa_leu,1) =  7; //   CD1
	ta(2,14,aa_leu,1) =  6; //   CG
	ta(3,14,aa_leu,1) =  5; //   CB

//bk   template for building 2HD1
	ta(1,15,aa_leu,1) =  7; //   CD1
	ta(2,15,aa_leu,1) =  6; //   CG
	ta(3,15,aa_leu,1) =  5; //   CB

//bk   template for building 3HD1
	ta(1,16,aa_leu,1) =  7; //   CD1
	ta(2,16,aa_leu,1) =  6; //   CG
	ta(3,16,aa_leu,1) =  5; //   CB

//bk   template for building 1HD2
	ta(1,17,aa_leu,1) =  8; //   CD2
	ta(2,17,aa_leu,1) =  6; //   CG
	ta(3,17,aa_leu,1) =  5; //   CB

//bk   template for building 2HD2
	ta(1,18,aa_leu,1) =  8; //   CD2
	ta(2,18,aa_leu,1) =  6; //   CG
	ta(3,18,aa_leu,1) =  5; //   CB

//bk   template for building 3HD2
	ta(1,19,aa_leu,1) =  8; //   CD2
	ta(2,19,aa_leu,1) =  6; //   CG
	ta(3,19,aa_leu,1) =  5; //   CB

//bk   chi angles required to build atoms LEU

//bk   chi angles needed for building  N
	chi_required(1, 1,aa_leu,1) = false;
	chi_required(2, 1,aa_leu,1) = false;
	chi_required(3, 1,aa_leu,1) = false;
	chi_required(4, 1,aa_leu,1) = false;

//bk   chi angles needed for building  CA
	chi_required(1, 2,aa_leu,1) = false;
	chi_required(2, 2,aa_leu,1) = false;
	chi_required(3, 2,aa_leu,1) = false;
	chi_required(4, 2,aa_leu,1) = false;

//bk   chi angles needed for building  C
	chi_required(1, 3,aa_leu,1) = false;
	chi_required(2, 3,aa_leu,1) = false;
	chi_required(3, 3,aa_leu,1) = false;
	chi_required(4, 3,aa_leu,1) = false;

//bk   chi angles needed for building  O
	chi_required(1, 4,aa_leu,1) = false;
	chi_required(2, 4,aa_leu,1) = false;
	chi_required(3, 4,aa_leu,1) = false;
	chi_required(4, 4,aa_leu,1) = false;

//bk   chi angles needed for building  CB
	chi_required(1, 5,aa_leu,1) = false;
	chi_required(2, 5,aa_leu,1) = false;
	chi_required(3, 5,aa_leu,1) = false;
	chi_required(4, 5,aa_leu,1) = false;

//bk   chi angles needed for building  CG
	chi_required(1, 6,aa_leu,1) =  true;
	chi_required(2, 6,aa_leu,1) = false;
	chi_required(3, 6,aa_leu,1) = false;
	chi_required(4, 6,aa_leu,1) = false;

//bk   chi angles needed for building  CD1
	chi_required(1, 7,aa_leu,1) =  true;
	chi_required(2, 7,aa_leu,1) =  true;
	chi_required(3, 7,aa_leu,1) = false;
	chi_required(4, 7,aa_leu,1) = false;

//bk   chi angles needed for building  CD2
	chi_required(1, 8,aa_leu,1) =  true;
	chi_required(2, 8,aa_leu,1) =  true;
	chi_required(3, 8,aa_leu,1) = false;
	chi_required(4, 8,aa_leu,1) = false;

//bk   chi angles needed for building  H
	chi_required(1, 9,aa_leu,1) = false;
	chi_required(2, 9,aa_leu,1) = false;
	chi_required(3, 9,aa_leu,1) = false;
	chi_required(4, 9,aa_leu,1) = false;

//bk   chi angles needed for building  HA
	chi_required(1,10,aa_leu,1) = false;
	chi_required(2,10,aa_leu,1) = false;
	chi_required(3,10,aa_leu,1) = false;
	chi_required(4,10,aa_leu,1) = false;

//bk   chi angles needed for building 2HB
	chi_required(1,11,aa_leu,1) =  true;
	chi_required(2,11,aa_leu,1) = false;
	chi_required(3,11,aa_leu,1) = false;
	chi_required(4,11,aa_leu,1) = false;

//bk   chi angles needed for building 3HB
	chi_required(1,12,aa_leu,1) =  true;
	chi_required(2,12,aa_leu,1) = false;
	chi_required(3,12,aa_leu,1) = false;
	chi_required(4,12,aa_leu,1) = false;

//bk   chi angles needed for building  HG
	chi_required(1,13,aa_leu,1) =  true;
	chi_required(2,13,aa_leu,1) =  true;
	chi_required(3,13,aa_leu,1) = false;
	chi_required(4,13,aa_leu,1) = false;

//bk   chi angles needed for building 1HD1
	chi_required(1,14,aa_leu,1) =  true;
	chi_required(2,14,aa_leu,1) =  true;
	chi_required(3,14,aa_leu,1) = false;
	chi_required(4,14,aa_leu,1) = false;

//bk   chi angles needed for building 2HD1
	chi_required(1,15,aa_leu,1) =  true;
	chi_required(2,15,aa_leu,1) =  true;
	chi_required(3,15,aa_leu,1) = false;
	chi_required(4,15,aa_leu,1) = false;

//bk   chi angles needed for building 3HD1
	chi_required(1,16,aa_leu,1) =  true;
	chi_required(2,16,aa_leu,1) =  true;
	chi_required(3,16,aa_leu,1) = false;
	chi_required(4,16,aa_leu,1) = false;

//bk   chi angles needed for building 1HD2
	chi_required(1,17,aa_leu,1) =  true;
	chi_required(2,17,aa_leu,1) =  true;
	chi_required(3,17,aa_leu,1) = false;
	chi_required(4,17,aa_leu,1) = false;

//bk   chi angles needed for building 2HD2
	chi_required(1,18,aa_leu,1) =  true;
	chi_required(2,18,aa_leu,1) =  true;
	chi_required(3,18,aa_leu,1) = false;
	chi_required(4,18,aa_leu,1) = false;

//bk   chi angles needed for building 3HD2
	chi_required(1,19,aa_leu,1) =  true;
	chi_required(2,19,aa_leu,1) =  true;
	chi_required(3,19,aa_leu,1) = false;
	chi_required(4,19,aa_leu,1) = false;

//bk   four atoms that define chi angle  1
	chi_atoms( 1, 1,aa_leu, 1) =  1; //   N
	chi_atoms( 2, 1,aa_leu, 1) =  2; //   CA
	chi_atoms( 3, 1,aa_leu, 1) =  5; //   CB
	chi_atoms( 4, 1,aa_leu, 1) =  6; //   CG
//bk   four atoms that define chi angle  2
	chi_atoms( 1, 2,aa_leu, 1) =  2; //   CA
	chi_atoms( 2, 2,aa_leu, 1) =  5; //   CB
	chi_atoms( 3, 2,aa_leu, 1) =  6; //   CG
	chi_atoms( 4, 2,aa_leu, 1) =  7; //   CD1

	HNpos(aa_leu,1) =  9; // atom number for backbone HN
	HApos(aa_leu,1) = 10; // atom number for backbone HA
	nH_polar(aa_leu,1) =  1; // number of polar hydrogens
	nH_aromatic(aa_leu,1) =  0; // number of aromatic hydrogens
	nH_apolar(aa_leu,1) = 10; // number of apolar hydrogens
	nacceptors(aa_leu,1) =  1; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1,aa_leu,1) =  9; //  H

//bk   atom number for aromatic hydrogens

//bk   atom number for apolar hydrogens
	Hpos_apolar( 1,aa_leu,1) = 10; //  HA
	Hpos_apolar( 2,aa_leu,1) = 11; // 2HB
	Hpos_apolar( 3,aa_leu,1) = 12; // 3HB
	Hpos_apolar( 4,aa_leu,1) = 13; //  HG
	Hpos_apolar( 5,aa_leu,1) = 14; // 1HD1
	Hpos_apolar( 6,aa_leu,1) = 15; // 2HD1
	Hpos_apolar( 7,aa_leu,1) = 16; // 3HD1
	Hpos_apolar( 8,aa_leu,1) = 17; // 1HD2
	Hpos_apolar( 9,aa_leu,1) = 18; // 2HD2
	Hpos_apolar(10,aa_leu,1) = 19; // 3HD2

//bk   acceptor positions
	accpt_pos( 1,aa_leu, 1) =  4; //  O

//bk   hydrogens on each atom

	nhydrogens_on_atm( 1,aa_leu, 1) =  1; //  N
	hydrogens_on_atm( 1, 1,aa_leu, 1) =  9; //  H

	nhydrogens_on_atm( 2,aa_leu, 1) =  1; //  CA
	hydrogens_on_atm( 1, 2,aa_leu, 1) = 10; //  HA

	nhydrogens_on_atm( 5,aa_leu, 1) =  2; //  CB
	hydrogens_on_atm( 1, 5,aa_leu, 1) = 11; // 2HB
	hydrogens_on_atm( 2, 5,aa_leu, 1) = 12; // 3HB

	nhydrogens_on_atm( 6,aa_leu, 1) =  1; //  CG
	hydrogens_on_atm( 1, 6,aa_leu, 1) = 13; //  HG

	nhydrogens_on_atm( 7,aa_leu, 1) =  3; //  CD1
	hydrogens_on_atm( 1, 7,aa_leu, 1) = 14; // 1HD1
	hydrogens_on_atm( 2, 7,aa_leu, 1) = 15; // 2HD1
	hydrogens_on_atm( 3, 7,aa_leu, 1) = 16; // 3HD1

	nhydrogens_on_atm( 8,aa_leu, 1) =  3; //  CD2
	hydrogens_on_atm( 1, 8,aa_leu, 1) = 17; // 1HD2
	hydrogens_on_atm( 2, 8,aa_leu, 1) = 18; // 2HD2
	hydrogens_on_atm( 3, 8,aa_leu, 1) = 19; // 3HD2

//bk   template coordinates for the amino acid
//jjh new template from Kevin Karplus
	icoor( 1, 1,aa_leu, 1) =   1.460; //   N
	icoor( 2, 1,aa_leu, 1) =   0.000; //   N
	icoor( 3, 1,aa_leu, 1) =   0.000; //   N
	icoor( 1, 2,aa_leu, 1) =   0.000; //   CA
	icoor( 2, 2,aa_leu, 1) =   0.000; //   CA
	icoor( 3, 2,aa_leu, 1) =   0.000; //   CA
	icoor( 1, 3,aa_leu, 1) =   -0.559; //   C
	icoor( 2, 3,aa_leu, 1) =   1.422; //   C
	icoor( 3, 3,aa_leu, 1) =   -0.000; //   C
	icoor( 1, 4,aa_leu, 1) =   -1.545; //   O
	icoor( 2, 4,aa_leu, 1) =   1.703; //   O
	icoor( 3, 4,aa_leu, 1) =   -0.678; //   O
	icoor( 1, 5,aa_leu, 1) =   -0.530; //   CB
	icoor( 2, 5,aa_leu, 1) =   -0.778; //   CB
	icoor( 3, 5,aa_leu, 1) =   1.211; //   CB
	icoor( 1, 6,aa_leu, 1) =   -0.286; //   CG
	icoor( 2, 6,aa_leu, 1) =   -2.292; //   CG
	icoor( 3, 6,aa_leu, 1) =   1.172; //   CG
	icoor( 1, 7,aa_leu, 1) =   -0.695; //   CD1
	icoor( 2, 7,aa_leu, 1) =   -2.921; //   CD1
	icoor( 3, 7,aa_leu, 1) =   2.497; //   CD1
	icoor( 1, 8,aa_leu, 1) =   -1.069; //   CD2
	icoor( 2, 8,aa_leu, 1) =   -2.904; //   CD2
	icoor( 3, 8,aa_leu, 1) =    0.020; //   CD2
	icoor( 1, 9,aa_leu, 1) =   2.424; //   H
	icoor( 2, 9,aa_leu, 1) =   -0.301; //   H
	icoor( 3, 9,aa_leu, 1) =   0.000; //   H
	icoor( 1,10,aa_leu, 1) =   -0.365; //   HA
	icoor( 2,10,aa_leu, 1) =   -0.473; //   HA
	icoor( 3,10,aa_leu, 1) =   -0.911; //   HA
	icoor( 1,11,aa_leu, 1) =   0.075; //  1HB
	icoor( 2,11,aa_leu, 1) =   -0.322; //  1HB
	icoor( 3,11,aa_leu, 1) =   1.993; //  1HB
	icoor( 1,12,aa_leu, 1) =   -1.583; //  2HB
	icoor( 2,12,aa_leu, 1) =   -0.567; //  2HB
	icoor( 3,12,aa_leu, 1) =   1.397; //  2HB
	icoor( 1,13,aa_leu, 1) =   0.775; //   HG
	icoor( 2,13,aa_leu, 1) =   -2.444; //   HG
	icoor( 3,13,aa_leu, 1) =   0.972; //   HG
	icoor( 1,14,aa_leu, 1) =   -0.518; //  1HD1
	icoor( 2,14,aa_leu, 1) =   -3.996; //  1HD1
	icoor( 3,14,aa_leu, 1) =   2.460; //  1HD1
	icoor( 1,15,aa_leu, 1) =   -0.106; //  2HD1
	icoor( 2,15,aa_leu, 1) =   -2.485; //  2HD1
	icoor( 3,15,aa_leu, 1) =   3.304; //  2HD1
	icoor( 1,16,aa_leu, 1) =   -1.753; //  3HD1
	icoor( 2,16,aa_leu, 1) =   -2.734; //  3HD1
	icoor( 3,16,aa_leu, 1) =   2.677; //  3HD1
	icoor( 1,17,aa_leu, 1) =   -0.894; //  1HD2
	icoor( 2,17,aa_leu, 1) =   -3.980; //  1HD2
	icoor( 3,17,aa_leu, 1) =   -0.007; //  1HD2
	icoor( 1,18,aa_leu, 1) =   -2.133; //  2HD2
	icoor( 2,18,aa_leu, 1) =   -2.712; //  2HD2
	icoor( 3,18,aa_leu, 1) =   0.160; //  2HD2
	icoor( 1,19,aa_leu, 1) =   -0.741; //  3HD2
	icoor( 2,19,aa_leu, 1) =   -2.460; //  3HD2
	icoor( 3,19,aa_leu, 1) =   -0.920; //  3HD2

	// H positions from REDUCE
	if( param_pack::packer_logical::reduce_templates ) {
		icoor( 1 , 9  , aa_leu , 1) = 1.794 ; // H
		icoor( 2 , 9  , aa_leu , 1) = 0.567 ; // H
		icoor( 3 , 9  , aa_leu , 1) = -0.753 ; // H
		icoor( 1 , 10 , aa_leu , 1) = -0.337 ; // HA
		icoor( 2 , 10 , aa_leu , 1) = -0.495 ; // HA
		icoor( 3 , 10 , aa_leu , 1) = -0.923 ; // HA
		icoor( 1 , 11 , aa_leu , 1) = -0.064 ; // 1HB
		icoor( 2 , 11 , aa_leu , 1) = -0.371 ; // 1HB
		icoor( 3 , 11 , aa_leu , 1) = 2.121 ; // 1HB
		icoor( 1 , 12 , aa_leu , 1) = -1.612 ; // 2HB
		icoor( 2 , 12 , aa_leu , 1) = -0.599 ; // 2HB
		icoor( 3 , 12 , aa_leu , 1) = 1.296 ; // 2HB
		icoor( 1 , 13 , aa_leu , 1) = 0.785 ; // HG
		icoor( 2 , 13 , aa_leu , 1) = -2.487 ; // HG
		icoor( 3 , 13 , aa_leu , 1) = 1.015 ; // HG
		icoor( 1 , 14 , aa_leu , 1) = -0.516 ; // 1HD1
		icoor( 2 , 14 , aa_leu , 1) = -4.006 ; // 1HD1
		icoor( 3 , 14 , aa_leu , 1) = 2.459 ; // 1HD1
		icoor( 1 , 15 , aa_leu , 1) = -0.101 ; // 2HD1
		icoor( 2 , 15 , aa_leu , 1) = -2.480 ; // 2HD1
		icoor( 3 , 15 , aa_leu , 1) = 3.311 ; // 2HD1
		icoor( 1 , 16 , aa_leu , 1) = -1.763 ; // 3HD1
		icoor( 2 , 16 , aa_leu , 1) = -2.732 ; // 3HD1
		icoor( 3 , 16 , aa_leu , 1) = 2.679 ; // 3HD1
		icoor( 1 , 17 , aa_leu , 1) = -0.894 ; // 1HD2
		icoor( 2 , 17 , aa_leu , 1) = -3.990 ; // 1HD2
		icoor( 3 , 17 , aa_leu , 1) = -0.007 ; // 1HD2
		icoor( 1 , 18 , aa_leu , 1) = -2.142 ; // 2HD2
		icoor( 2 , 18 , aa_leu , 1) = -2.710 ; // 2HD2
		icoor( 3 , 18 , aa_leu , 1) = 0.162 ; // 2HD2
		icoor( 1 , 19 , aa_leu , 1) = -0.738 ; // 3HD2
		icoor( 2 , 19 , aa_leu , 1) = -2.456 ; // 3HD2
		icoor( 3 , 19 , aa_leu , 1) = -0.928 ; // 3HD2
	}

//bq create N and Cterm variants
	create_termini_variants( aa_leu );


//bk   create variants that have h2o connected to the backbone nh and carbonyl
	if ( explicit_h2o ) {
		create_backbone_h2o_variants(aa_leu);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_met
///
/// @brief arrays filled to define the proerties of the amino acid Methionine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// aa_met - index number for MET (11)
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_met()
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

//bk   Parameters for MET -----------------------------
	nvar(aa_met) = 1; // starting number of variants for MET
	variant_type( aav_base, aa_met, 1 ) = true; // type of aa variant

//bk   variant1 ---------- MET

	natoms(aa_met,1) = 17; // number of total atoms
	nheavyatoms(aa_met,1) =  8; // number of heavy atoms
	nchi(aa_met,1) =  3; // number of chi angles

//bk   atom names MET
	atom_name( 1,aa_met,1) = " N  ";
	atom_name( 2,aa_met,1) = " CA ";
	atom_name( 3,aa_met,1) = " C  ";
	atom_name( 4,aa_met,1) = " O  ";
	atom_name( 5,aa_met,1) = " CB ";
	atom_name( 6,aa_met,1) = " CG ";
	atom_name( 7,aa_met,1) = " SD ";
	atom_name( 8,aa_met,1) = " CE ";
	atom_name( 9,aa_met,1) = " H  ";
	atom_name(10,aa_met,1) = " HA ";
	atom_name(11,aa_met,1) = "2HB ";
	atom_name(12,aa_met,1) = "3HB ";
	atom_name(13,aa_met,1) = "2HG ";
	atom_name(14,aa_met,1) = "3HG ";
	atom_name(15,aa_met,1) = "1HE ";
	atom_name(16,aa_met,1) = "2HE ";
	atom_name(17,aa_met,1) = "3HE ";

//jjh intra residue bonding
	nbonded_neighbors( 1, aa_met, 1) = 2; // N
		bonded_neighbor(1, 1, aa_met, 1) = 2; // N--CA
		bonded_neighbor(2, 1, aa_met, 1) = 9; // N--H
	nbonded_neighbors( 2, aa_met, 1) = 4; // CA
		bonded_neighbor(1, 2, aa_met, 1) = 1; // CA--N
		bonded_neighbor(2, 2, aa_met, 1) = 3; // CA--C
		bonded_neighbor(3, 2, aa_met, 1) = 5; // CA--CB
		bonded_neighbor(4, 2, aa_met, 1) =10; // CA--HA
	nbonded_neighbors( 3, aa_met, 1) = 2; // C
		bonded_neighbor(1, 3, aa_met, 1) = 2; // C--CA
		bonded_neighbor(2, 3, aa_met, 1) = 4; // C--O
	nbonded_neighbors( 4, aa_met, 1) = 1; // O
		bonded_neighbor(1, 4, aa_met, 1) = 3; // O--C
	nbonded_neighbors( 5, aa_met, 1) = 4; // CB
		bonded_neighbor(1, 5, aa_met, 1) = 2; // CB--CA
		bonded_neighbor(2, 5, aa_met, 1) = 6; // CB--CG
		bonded_neighbor(3, 5, aa_met, 1) =11; // CB--2HB
		bonded_neighbor(4, 5, aa_met, 1) =12; // CB--3HB
	nbonded_neighbors( 6, aa_met, 1) = 4; // CG
		bonded_neighbor(1, 6, aa_met, 1) = 5; // CG--CB
		bonded_neighbor(2, 6, aa_met, 1) = 7; // CG--SD
		bonded_neighbor(3, 6, aa_met, 1) =13; // CG--2HG
		bonded_neighbor(4, 6, aa_met, 1) =14; // CG--3HG
	nbonded_neighbors( 7, aa_met, 1) = 2; // SD
		bonded_neighbor(1, 7, aa_met, 1) = 6; // SD--CG
		bonded_neighbor(2, 7, aa_met, 1) = 8; // SD--CE
	nbonded_neighbors( 8, aa_met, 1) = 4; // CE
		bonded_neighbor(1, 8, aa_met, 1) = 7; // CE--SD
		bonded_neighbor(2, 8, aa_met, 1) =15; // CE--1HE
		bonded_neighbor(3, 8, aa_met, 1) =16; // CE--2HE
		bonded_neighbor(4, 8, aa_met, 1) =17; // CE--3HE
	nbonded_neighbors( 9, aa_met, 1) = 1; // H
		bonded_neighbor(1, 9, aa_met, 1) = 1; // H--N
	nbonded_neighbors( 10, aa_met, 1) = 1; // HA
		bonded_neighbor(1,10, aa_met, 1) = 2; // HA--CA
	nbonded_neighbors( 11, aa_met, 1) = 1; //2HB
		bonded_neighbor(1,11, aa_met, 1) = 5; //2HB--CB
	nbonded_neighbors( 12, aa_met, 1) = 1; //3HB
		bonded_neighbor(1,12, aa_met, 1) = 5; //3HB--CB
	nbonded_neighbors( 13, aa_met, 1) = 1; //2HG
		bonded_neighbor(1,13, aa_met, 1) = 6; //2HG--CG
	nbonded_neighbors( 14, aa_met, 1) = 1; //3HG
		bonded_neighbor(1,14, aa_met, 1) = 6; //3HG--CG
	nbonded_neighbors( 15, aa_met, 1) = 1; //1HE
		bonded_neighbor(1,15, aa_met, 1) = 8; //1HE--CE
	nbonded_neighbors( 16, aa_met, 1) = 1; //2HE
		bonded_neighbor(1,16, aa_met, 1) = 8; //2HE--CE
	nbonded_neighbors( 17, aa_met, 1) = 1; //3HE
		bonded_neighbor(1,17, aa_met, 1) = 8; //3HE--CE

//jjh get the angles automatically
	get_angles_from_bonds(aa_met, 1);

//chu   base atom numbers MET
	atom_base( 1,aa_met,1) =  2; //  N  -> CA
	atom_base( 2,aa_met,1) =  1; //  CA -> N
	atom_base( 3,aa_met,1) =  2; //  C  -> CA
	atom_base( 4,aa_met,1) =  3; //  O  -> C
	atom_base( 5,aa_met,1) =  2; //  CB -> CA
	atom_base( 6,aa_met,1) =  5; //  CG -> CB
	atom_base( 7,aa_met,1) =  6; //  SD -> CG
	atom_base( 8,aa_met,1) =  7; //  CE -> SD
	atom_base( 9,aa_met,1) =  1; //  H   on  N
	atom_base(10,aa_met,1) =  2; //  HA  on  CA
	atom_base(11,aa_met,1) =  5; // 2HB  on  CB
	atom_base(12,aa_met,1) =  5; // 3HB  on  CB
	atom_base(13,aa_met,1) =  6; // 2HG  on  CG
	atom_base(14,aa_met,1) =  6; // 3HG  on  CG
	atom_base(15,aa_met,1) =  8; // 1HE  on  CE
	atom_base(16,aa_met,1) =  8; // 2HE  on  CE
	atom_base(17,aa_met,1) =  8; // 3HE  on  CE

//bk   atom type numbers MET
	fullatom_type( 1,aa_met,1) = 17; // Nbb    N
	fullatom_type( 2,aa_met,1) = 18; // CAbb   CA
	fullatom_type( 3,aa_met,1) = 19; // CObb   C
	fullatom_type( 4,aa_met,1) = 20; // OCbb   O
	fullatom_type( 5,aa_met,1) =  4; // CH2    CB
	fullatom_type( 6,aa_met,1) =  4; // CH2    CG
	fullatom_type( 7,aa_met,1) = 16; // S      SD
	fullatom_type( 8,aa_met,1) =  5; // CH3    CE
	fullatom_type( 9,aa_met,1) = 25; // HNbb   H
	fullatom_type(10,aa_met,1) = 23; // Hapo   HA
	fullatom_type(11,aa_met,1) = 23; // Hapo  2HB
	fullatom_type(12,aa_met,1) = 23; // Hapo  3HB
	fullatom_type(13,aa_met,1) = 23; // Hapo  2HG
	fullatom_type(14,aa_met,1) = 23; // Hapo  3HG
	fullatom_type(15,aa_met,1) = 23; // Hapo  1HE
	fullatom_type(16,aa_met,1) = 23; // Hapo  2HE
	fullatom_type(17,aa_met,1) = 23; // Hapo  3HE

//cmd  atomic charge MET
	atomic_charge( 1,aa_met,1) = -0.47; //    N
	atomic_charge( 2,aa_met,1) =  0.07; //    CA
	atomic_charge( 3,aa_met,1) =  0.51; //    C
	atomic_charge( 4,aa_met,1) = -0.51; //    O
	atomic_charge( 5,aa_met,1) = -0.18; //    CB
	atomic_charge( 6,aa_met,1) = -0.14; //    CG
	atomic_charge( 7,aa_met,1) = -0.09; //    SD
	atomic_charge( 8,aa_met,1) = -0.22; //    CE
	atomic_charge( 9,aa_met,1) =  0.31; //    H
	atomic_charge(10,aa_met,1) =  0.09; //    HA
	atomic_charge(11,aa_met,1) =  0.09; //   2HB
	atomic_charge(12,aa_met,1) =  0.09; //   3HB
	atomic_charge(13,aa_met,1) =  0.09; //   2HG
	atomic_charge(14,aa_met,1) =  0.09; //   3HG
	atomic_charge(15,aa_met,1) =  0.09; //   1HE
	atomic_charge(16,aa_met,1) =  0.09; //   2HE
	atomic_charge(17,aa_met,1) =  0.09; //   3HE

	assert_total_charge( aa_met, 1, 0.0);

//cmd  first sidechain atom
	first_scatom( aa_met,1) = 5; //CB

//bk   template atoms used for placing atoms MET

//bk   template for building  HA
	ta(1,10,aa_met,1) =  2; //   CA
	ta(2,10,aa_met,1) =  1; //   N
	ta(3,10,aa_met,1) =  3; //   C

//bk   template for building 2HB
	ta(1,11,aa_met,1) =  5; //   CB
	ta(2,11,aa_met,1) =  2; //   CA
	ta(3,11,aa_met,1) =  6; //   CG

//bk   template for building 3HB
	ta(1,12,aa_met,1) =  5; //   CB
	ta(2,12,aa_met,1) =  2; //   CA
	ta(3,12,aa_met,1) =  6; //   CG

//bk   template for building 2HG
	ta(1,13,aa_met,1) =  6; //   CG
	ta(2,13,aa_met,1) =  5; //   CB
	ta(3,13,aa_met,1) =  7; //   SD

//bk   template for building 3HG
	ta(1,14,aa_met,1) =  6; //   CG
	ta(2,14,aa_met,1) =  5; //   CB
	ta(3,14,aa_met,1) =  7; //   SD

//bk   template for building 1HE
	ta(1,15,aa_met,1) =  8; //   CE
	ta(2,15,aa_met,1) =  7; //   SD
	ta(3,15,aa_met,1) =  6; //   CG

//bk   template for building 2HE
	ta(1,16,aa_met,1) =  8; //   CE
	ta(2,16,aa_met,1) =  7; //   SD
	ta(3,16,aa_met,1) =  6; //   CG

//bk   template for building 3HE
	ta(1,17,aa_met,1) =  8; //   CE
	ta(2,17,aa_met,1) =  7; //   SD
	ta(3,17,aa_met,1) =  6; //   CG

//bk   chi angles required to build atoms MET

//bk   chi angles needed for building  N
	chi_required(1, 1,aa_met,1) = false;
	chi_required(2, 1,aa_met,1) = false;
	chi_required(3, 1,aa_met,1) = false;
	chi_required(4, 1,aa_met,1) = false;

//bk   chi angles needed for building  CA
	chi_required(1, 2,aa_met,1) = false;
	chi_required(2, 2,aa_met,1) = false;
	chi_required(3, 2,aa_met,1) = false;
	chi_required(4, 2,aa_met,1) = false;

//bk   chi angles needed for building  C
	chi_required(1, 3,aa_met,1) = false;
	chi_required(2, 3,aa_met,1) = false;
	chi_required(3, 3,aa_met,1) = false;
	chi_required(4, 3,aa_met,1) = false;

//bk   chi angles needed for building  O
	chi_required(1, 4,aa_met,1) = false;
	chi_required(2, 4,aa_met,1) = false;
	chi_required(3, 4,aa_met,1) = false;
	chi_required(4, 4,aa_met,1) = false;

//bk   chi angles needed for building  CB
	chi_required(1, 5,aa_met,1) = false;
	chi_required(2, 5,aa_met,1) = false;
	chi_required(3, 5,aa_met,1) = false;
	chi_required(4, 5,aa_met,1) = false;

//bk   chi angles needed for building  CG
	chi_required(1, 6,aa_met,1) =  true;
	chi_required(2, 6,aa_met,1) = false;
	chi_required(3, 6,aa_met,1) = false;
	chi_required(4, 6,aa_met,1) = false;

//bk   chi angles needed for building  SD
	chi_required(1, 7,aa_met,1) =  true;
	chi_required(2, 7,aa_met,1) =  true;
	chi_required(3, 7,aa_met,1) = false;
	chi_required(4, 7,aa_met,1) = false;

//bk   chi angles needed for building  CE
	chi_required(1, 8,aa_met,1) =  true;
	chi_required(2, 8,aa_met,1) =  true;
	chi_required(3, 8,aa_met,1) =  true;
	chi_required(4, 8,aa_met,1) = false;

//bk   chi angles needed for building  H
	chi_required(1, 9,aa_met,1) = false;
	chi_required(2, 9,aa_met,1) = false;
	chi_required(3, 9,aa_met,1) = false;
	chi_required(4, 9,aa_met,1) = false;

//bk   chi angles needed for building  HA
	chi_required(1,10,aa_met,1) = false;
	chi_required(2,10,aa_met,1) = false;
	chi_required(3,10,aa_met,1) = false;
	chi_required(4,10,aa_met,1) = false;

//bk   chi angles needed for building 2HB
	chi_required(1,11,aa_met,1) =  true;
	chi_required(2,11,aa_met,1) = false;
	chi_required(3,11,aa_met,1) = false;
	chi_required(4,11,aa_met,1) = false;

//bk   chi angles needed for building 3HB
	chi_required(1,12,aa_met,1) =  true;
	chi_required(2,12,aa_met,1) = false;
	chi_required(3,12,aa_met,1) = false;
	chi_required(4,12,aa_met,1) = false;

//bk   chi angles needed for building 2HG
	chi_required(1,13,aa_met,1) =  true;
	chi_required(2,13,aa_met,1) =  true;
	chi_required(3,13,aa_met,1) = false;
	chi_required(4,13,aa_met,1) = false;

//bk   chi angles needed for building 3HG
	chi_required(1,14,aa_met,1) =  true;
	chi_required(2,14,aa_met,1) =  true;
	chi_required(3,14,aa_met,1) = false;
	chi_required(4,14,aa_met,1) = false;

//bk   chi angles needed for building 1HE
	chi_required(1,15,aa_met,1) =  true;
	chi_required(2,15,aa_met,1) =  true;
	chi_required(3,15,aa_met,1) =  true;
	chi_required(4,15,aa_met,1) = false;

//bk   chi angles needed for building 2HE
	chi_required(1,16,aa_met,1) =  true;
	chi_required(2,16,aa_met,1) =  true;
	chi_required(3,16,aa_met,1) =  true;
	chi_required(4,16,aa_met,1) = false;

//bk   chi angles needed for building 3HE
	chi_required(1,17,aa_met,1) =  true;
	chi_required(2,17,aa_met,1) =  true;
	chi_required(3,17,aa_met,1) =  true;
	chi_required(4,17,aa_met,1) = false;

//bk   four atoms that define chi angle  1
	chi_atoms( 1, 1,aa_met, 1) =  1; //   N
	chi_atoms( 2, 1,aa_met, 1) =  2; //   CA
	chi_atoms( 3, 1,aa_met, 1) =  5; //   CB
	chi_atoms( 4, 1,aa_met, 1) =  6; //   CG
//bk   four atoms that define chi angle  2
	chi_atoms( 1, 2,aa_met, 1) =  2; //   CA
	chi_atoms( 2, 2,aa_met, 1) =  5; //   CB
	chi_atoms( 3, 2,aa_met, 1) =  6; //   CG
	chi_atoms( 4, 2,aa_met, 1) =  7; //   SD
//bk   four atoms that define chi angle  3
	chi_atoms( 1, 3,aa_met, 1) =  5; //   CB
	chi_atoms( 2, 3,aa_met, 1) =  6; //   CG
	chi_atoms( 3, 3,aa_met, 1) =  7; //   SD
	chi_atoms( 4, 3,aa_met, 1) =  8; //   CE

	HNpos(aa_met,1) =  9; // atom number for backbone HN
	HApos(aa_met,1) = 10; // atom number for backbone HA
	nH_polar(aa_met,1) =  1; // number of polar hydrogens
	nH_aromatic(aa_met,1) =  0; // number of aromatic hydrogens
	nH_apolar(aa_met,1) =  8; // number of apolar hydrogens
	nacceptors(aa_met,1) =  1; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1,aa_met,1) =  9; //  H

//bk   atom number for aromatic hydrogens

//bk   atom number for apolar hydrogens
	Hpos_apolar( 1,aa_met,1) = 10; //  HA
	Hpos_apolar( 2,aa_met,1) = 11; // 2HB
	Hpos_apolar( 3,aa_met,1) = 12; // 3HB
	Hpos_apolar( 4,aa_met,1) = 13; // 2HG
	Hpos_apolar( 5,aa_met,1) = 14; // 3HG
	Hpos_apolar( 6,aa_met,1) = 15; // 1HE
	Hpos_apolar( 7,aa_met,1) = 16; // 2HE
	Hpos_apolar( 8,aa_met,1) = 17; // 3HE

//bk   acceptor positions
	accpt_pos( 1,aa_met, 1) =  4; //  O

//bk   hydrogens on each atom

	nhydrogens_on_atm( 1,aa_met, 1) =  1; //  N
	hydrogens_on_atm( 1, 1,aa_met, 1) =  9; //  H

	nhydrogens_on_atm( 2,aa_met, 1) =  1; //  CA
	hydrogens_on_atm( 1, 2,aa_met, 1) = 10; //  HA

	nhydrogens_on_atm( 5,aa_met, 1) =  2; //  CB
	hydrogens_on_atm( 1, 5,aa_met, 1) = 11; // 2HB
	hydrogens_on_atm( 2, 5,aa_met, 1) = 12; // 3HB

	nhydrogens_on_atm( 6,aa_met, 1) =  2; //  CG
	hydrogens_on_atm( 1, 6,aa_met, 1) = 13; // 2HG
	hydrogens_on_atm( 2, 6,aa_met, 1) = 14; // 3HG

	nhydrogens_on_atm( 8,aa_met, 1) =  3; //  CE
	hydrogens_on_atm( 1, 8,aa_met, 1) = 15; // 1HE
	hydrogens_on_atm( 2, 8,aa_met, 1) = 16; // 2HE
	hydrogens_on_atm( 3, 8,aa_met, 1) = 17; // 3HE

//bk   template coordinates for the amino acid
	icoor( 1, 1,aa_met, 1) =    0.0000; //   N
	icoor( 2, 1,aa_met, 1) =   15.5330; //   N
	icoor( 3, 1,aa_met, 1) =   36.4900; //   N
	icoor( 1, 2,aa_met, 1) =    0.0000; //   CA
	icoor( 2, 2,aa_met, 1) =   16.4650; //   CA
	icoor( 3, 2,aa_met, 1) =   37.6220; //   CA
	icoor( 1, 3,aa_met, 1) =    0.0000; //   C
	icoor( 2, 3,aa_met, 1) =   15.5850; //   C
	icoor( 3, 3,aa_met, 1) =   38.8610; //   C
	icoor( 1, 4,aa_met, 1) =    0.0630; //   O
	icoor( 2, 4,aa_met, 1) =   14.3610; //   O
	icoor( 3, 4,aa_met, 1) =   38.7570; //   O
	icoor( 1, 5,aa_met, 1) =   -1.2080; //   CB
	icoor( 2, 5,aa_met, 1) =   17.3960; //   CB
	icoor( 3, 5,aa_met, 1) =   37.5390; //   CB
	icoor( 1, 6,aa_met, 1) =   -1.3030; //   CG
	icoor( 2, 6,aa_met, 1) =   18.4070; //   CG
	icoor( 3, 6,aa_met, 1) =   38.6730; //   CG
	icoor( 1, 7,aa_met, 1) =   -2.7450; //   SD
	icoor( 2, 7,aa_met, 1) =   19.4810; //   SD
	icoor( 3, 7,aa_met, 1) =   38.5280; //   SD
	icoor( 1, 8,aa_met, 1) =   -2.5470; //   CE
	icoor( 2, 8,aa_met, 1) =   20.5110; //   CE
	icoor( 3, 8,aa_met, 1) =   39.9790; //   CE
	icoor( 1, 9,aa_met, 1) =   -0.0400; //   H
	icoor( 2, 9,aa_met, 1) =   14.5430; //   H
	icoor( 3, 9,aa_met, 1) =   36.6850; //   H
	icoor( 1,10,aa_met, 1) =    0.9070; //   HA
	icoor( 2,10,aa_met, 1) =   17.0690; //   HA
	icoor( 3,10,aa_met, 1) =   37.6060; //   HA
	icoor( 1,11,aa_met, 1) =   -1.1380; //  2HB
	icoor( 2,11,aa_met, 1) =   17.9220; //  2HB
	icoor( 3,11,aa_met, 1) =   36.5880; //  2HB
	icoor( 1,12,aa_met, 1) =   -2.0970; //  3HB
	icoor( 2,12,aa_met, 1) =   16.7640; //  3HB
	icoor( 3,12,aa_met, 1) =   37.5410; //  3HB
	icoor( 1,13,aa_met, 1) =   -1.3580; //  2HG
	icoor( 2,13,aa_met, 1) =   17.8590; //  2HG
	icoor( 3,13,aa_met, 1) =   39.6130; //  2HG
	icoor( 1,14,aa_met, 1) =   -0.4000; //  3HG
	icoor( 2,14,aa_met, 1) =   19.0170; //  3HG
	icoor( 3,14,aa_met, 1) =   38.6600; //  3HG
	icoor( 1,15,aa_met, 1) =   -3.3650; //  1HE
	icoor( 2,15,aa_met, 1) =   21.2300; //  1HE
	icoor( 3,15,aa_met, 1) =   40.0300; //  1HE
	icoor( 1,16,aa_met, 1) =   -2.5560; //  2HE
	icoor( 2,16,aa_met, 1) =   19.8860; //  2HE
	icoor( 3,16,aa_met, 1) =   40.8730; //  2HE
	icoor( 1,17,aa_met, 1) =   -1.5980; //  3HE
	icoor( 2,17,aa_met, 1) =   21.0450; //  3HE
	icoor( 3,17,aa_met, 1) =   39.9190; //  3HE

	// H positions from REDUCE
	if( param_pack::packer_logical::reduce_templates ) {
		icoor( 1 , 9 , aa_met , 1) = -0.049 ; // H
		icoor( 2 , 9 , aa_met , 1) = 14.594 ; // H
		icoor( 3 , 9 , aa_met , 1) = 36.831 ; // H
		icoor( 1 , 10 , aa_met , 1) = 0.874 ; // HA
		icoor( 2 , 10 , aa_met , 1) = 17.133 ; // HA
		icoor( 3 , 10 , aa_met , 1) = 37.634 ; // HA
		icoor( 1 , 11 , aa_met , 1) = -1.172 ; // 1HB
		icoor( 2 , 11 , aa_met , 1) = 17.939 ; // 1HB
		icoor( 3 , 11 , aa_met , 1) = 36.583 ; // 1HB
		icoor( 1 , 12 , aa_met , 1) = -2.124 ; // 2HB
		icoor( 2 , 12 , aa_met , 1) = 16.787 ; // 2HB
		icoor( 3 , 12 , aa_met , 1) = 37.529 ; // 2HB
		icoor( 1 , 13 , aa_met , 1) = -1.344 ; // 1HG
		icoor( 2 , 13 , aa_met , 1) = 17.872 ; // 1HG
		icoor( 3 , 13 , aa_met , 1) = 39.633 ; // 1HG
		icoor( 1 , 14 , aa_met , 1) = -0.392 ; // 2HG
		icoor( 2 , 14 , aa_met , 1) = 19.023 ; // 2HG
		icoor( 3 , 14 , aa_met , 1) = 38.687 ; // 2HG
		icoor( 1 , 15 , aa_met , 1) = -3.373 ; // 1HE
		icoor( 2 , 15 , aa_met , 1) = 21.236 ; // 1HE
		icoor( 3 , 15 , aa_met , 1) = 40.030 ; // 1HE
		icoor( 1 , 16 , aa_met , 1) = -2.557 ; // 2HE
		icoor( 2 , 16 , aa_met , 1) = 19.881 ; // 2HE
		icoor( 3 , 16 , aa_met , 1) = 40.881 ; // 2HE
		icoor( 1 , 17 , aa_met , 1) = -1.590 ; // 3HE
		icoor( 2 , 17 , aa_met , 1) = 21.050 ; // 3HE
		icoor( 3 , 17 , aa_met , 1) = 39.919 ; // 3HE
	}

//bq create N and Cterm variants
	create_termini_variants( aa_met );


//bk   create variants that have h2o connected to the backbone nh and carbonyl
	if ( explicit_h2o ) {
		create_backbone_h2o_variants(aa_met);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_asn
///
/// @brief arrays filled to define the proerties of the amino acid Asparagine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// aa_asn - index number for ASN (12)
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
///------------------------------------------------------------------------------
void
init_asn()
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

//bk   Parameters for ASN -----------------------------
	nvar(aa_asn) = 1; // starting number of variants for ASN
	variant_type( aav_base, aa_asn, 1 ) = true; // type of aa variant

//bk   variant1 ---------- ASN

	natoms(aa_asn,1) = 14; // number of total atoms
	nheavyatoms(aa_asn,1) =  8; // number of heavy atoms
	nchi(aa_asn,1) =  2; // number of chi angles

//bk   atom names ASN
	atom_name( 1,aa_asn,1) = " N  ";
	atom_name( 2,aa_asn,1) = " CA ";
	atom_name( 3,aa_asn,1) = " C  ";
	atom_name( 4,aa_asn,1) = " O  ";
	atom_name( 5,aa_asn,1) = " CB ";
	atom_name( 6,aa_asn,1) = " CG ";
	atom_name( 7,aa_asn,1) = " OD1";
	atom_name( 8,aa_asn,1) = " ND2";
	atom_name( 9,aa_asn,1) = " H  ";
	atom_name(10,aa_asn,1) = "1HD2";
	atom_name(11,aa_asn,1) = "2HD2";
	atom_name(12,aa_asn,1) = " HA ";
	atom_name(13,aa_asn,1) = "2HB ";
	atom_name(14,aa_asn,1) = "3HB ";

//jjh intra residue bonding
	nbonded_neighbors( 1, aa_asn, 1) = 2; // N
		bonded_neighbor(1, 1, aa_asn, 1) = 2; // N--CA
		bonded_neighbor(2, 1, aa_asn, 1) = 9; // N--H
	nbonded_neighbors( 2, aa_asn, 1) = 4; // CA
		bonded_neighbor(1, 2, aa_asn, 1) = 1; // CA--N
		bonded_neighbor(2, 2, aa_asn, 1) = 3; // CA--C
		bonded_neighbor(3, 2, aa_asn, 1) = 5; // CA--CB
		bonded_neighbor(4, 2, aa_asn, 1) =12; // CA--HA
	nbonded_neighbors( 3, aa_asn, 1) = 2; // C
		bonded_neighbor(1, 3, aa_asn, 1) = 2; // C--CA
		bonded_neighbor(2, 3, aa_asn, 1) = 4; // C--O
	nbonded_neighbors( 4, aa_asn, 1) = 1; // O
		bonded_neighbor(1, 4, aa_asn, 1) = 3; // O--C
	nbonded_neighbors( 5, aa_asn, 1) = 4; // CB
		bonded_neighbor(1, 5, aa_asn, 1) = 2; // CB--CA
		bonded_neighbor(2, 5, aa_asn, 1) = 6; // CB--CG
		bonded_neighbor(3, 5, aa_asn, 1) =13; // CB--2HB
		bonded_neighbor(4, 5, aa_asn, 1) =14; // CB--3HB
	nbonded_neighbors( 6, aa_asn, 1) = 3; // CG
		bonded_neighbor(1, 6, aa_asn, 1) = 5; // CG--CB
		bonded_neighbor(2, 6, aa_asn, 1) = 7; // CG--OD1
		bonded_neighbor(3, 6, aa_asn, 1) = 8; // CG--ND2
	nbonded_neighbors( 7, aa_asn, 1) = 1; // OD1
		bonded_neighbor(1, 7, aa_asn, 1) = 6; // OD1--CG
	nbonded_neighbors( 8, aa_asn, 1) = 3; // ND2
		bonded_neighbor(1, 8, aa_asn, 1) = 6; // ND2--CG
		bonded_neighbor(2, 8, aa_asn, 1) =10; // ND2--1HD2
		bonded_neighbor(3, 8, aa_asn, 1) =11; // ND2--2HD2
	nbonded_neighbors( 9, aa_asn, 1) = 1; // H
		bonded_neighbor(1, 9, aa_asn, 1) = 1; // H--N
	nbonded_neighbors( 10, aa_asn, 1) = 1; //1HD2
		bonded_neighbor(1,10, aa_asn, 1) = 8; //1HD2--ND2
	nbonded_neighbors( 11, aa_asn, 1) = 1; //2HD2
		bonded_neighbor(1,11, aa_asn, 1) = 8; //2HD2--ND2
	nbonded_neighbors( 12, aa_asn, 1) = 1; // HA
		bonded_neighbor(1,12, aa_asn, 1) = 2; // HA--CA
	nbonded_neighbors( 13, aa_asn, 1) = 1; //2HB
		bonded_neighbor(1,13, aa_asn, 1) = 5; //2HB--CB
	nbonded_neighbors( 14, aa_asn, 1) = 1; //3HB
		bonded_neighbor(1,14, aa_asn, 1) = 5; //3HB--CB

//jjh get the angles automatically
	get_angles_from_bonds(aa_asn, 1);

//chu   base atom numbers ASN
	atom_base( 1,aa_asn,1) =  2; //  N  -> CA
	atom_base( 2,aa_asn,1) =  1; //  CA -> N
	atom_base( 3,aa_asn,1) =  2; //  C  -> CA
	atom_base( 4,aa_asn,1) =  3; //  O  -> C
	atom_base( 5,aa_asn,1) =  2; //  CB -> CA
	atom_base( 6,aa_asn,1) =  5; //  CG -> CB
	atom_base( 7,aa_asn,1) =  6; //  OD1-> CG
	atom_base( 8,aa_asn,1) =  6; //  ND2 ->  CG
	atom_base( 9,aa_asn,1) =  1; //  H   ->  N
	atom_base(10,aa_asn,1) =  8; // 1HD2 ->  ND2
	atom_base(11,aa_asn,1) =  8; // 2HD2 ->  ND2
	atom_base(12,aa_asn,1) =  2; //  HA  ->  CA
	atom_base(13,aa_asn,1) =  5; // 2HB  ->  CB
	atom_base(14,aa_asn,1) =  5; // 3HB  ->  CB

//bk   atom type numbers ASN
	fullatom_type( 1,aa_asn,1) = 17; // Nbb    N
	fullatom_type( 2,aa_asn,1) = 18; // CAbb   CA
	fullatom_type( 3,aa_asn,1) = 19; // CObb   C
	fullatom_type( 4,aa_asn,1) = 20; // OCbb   O
	fullatom_type( 5,aa_asn,1) =  4; // CH2    CB
	fullatom_type( 6,aa_asn,1) =  1; // CNH2   CG
	fullatom_type( 7,aa_asn,1) = 14; // ONH2   OD1
	fullatom_type( 8,aa_asn,1) =  9; // NH2O   ND2
	fullatom_type( 9,aa_asn,1) = 25; // HNbb   H
	fullatom_type(10,aa_asn,1) = 22; // Hpol  1HD2
	fullatom_type(11,aa_asn,1) = 22; // Hpol  2HD2
	fullatom_type(12,aa_asn,1) = 23; // Hapo   HA
	fullatom_type(13,aa_asn,1) = 23; // Hapo  2HB
	fullatom_type(14,aa_asn,1) = 23; // Hapo  3HB

//cmd  atomic charge ASN
	atomic_charge( 1,aa_asn,1) = -0.47; //    N
	atomic_charge( 2,aa_asn,1) =  0.07; //    CA
	atomic_charge( 3,aa_asn,1) =  0.51; //    C
	atomic_charge( 4,aa_asn,1) = -0.51; //    O
	atomic_charge( 5,aa_asn,1) = -0.18; //    CB
	atomic_charge( 6,aa_asn,1) =  0.55; //    CG
	atomic_charge( 7,aa_asn,1) = -0.55; //    OD1
	atomic_charge( 8,aa_asn,1) = -0.62; //    ND2
	atomic_charge( 9,aa_asn,1) =  0.31; //    H
	atomic_charge(10,aa_asn,1) =  0.32; //   1HD2
	atomic_charge(11,aa_asn,1) =  0.30; //   2HD2
	atomic_charge(12,aa_asn,1) =  0.09; //    HA
	atomic_charge(13,aa_asn,1) =  0.09; //   2HB
	atomic_charge(14,aa_asn,1) =  0.09; //   3HB

	assert_total_charge( aa_asn, 1, 0.0);

//cmd  first sidechain atom
	first_scatom( aa_asn,1) = 5; //CB

//bk   template atoms used for placing atoms ASN

//bk   template for building 1HD2
	ta(1,10,aa_asn,1) =  8; //   ND2
	ta(2,10,aa_asn,1) =  6; //   CG
	ta(3,10,aa_asn,1) =  5; //   CB

//bk   template for building 2HD2
	ta(1,11,aa_asn,1) =  8; //   ND2
	ta(2,11,aa_asn,1) =  6; //   CG
	ta(3,11,aa_asn,1) =  5; //   CB

//bk   template for building  HA
	ta(1,12,aa_asn,1) =  2; //   CA
	ta(2,12,aa_asn,1) =  1; //   N
	ta(3,12,aa_asn,1) =  3; //   C

//bk   template for building 2HB
	ta(1,13,aa_asn,1) =  5; //   CB
	ta(2,13,aa_asn,1) =  2; //   CA
	ta(3,13,aa_asn,1) =  6; //   CG

//bk   template for building 3HB
	ta(1,14,aa_asn,1) =  5; //   CB
	ta(2,14,aa_asn,1) =  2; //   CA
	ta(3,14,aa_asn,1) =  6; //   CG

//bk   chi angles required to build atoms ASN

//bk   chi angles needed for building  N
	chi_required(1, 1,aa_asn,1) = false;
	chi_required(2, 1,aa_asn,1) = false;
	chi_required(3, 1,aa_asn,1) = false;
	chi_required(4, 1,aa_asn,1) = false;

//bk   chi angles needed for building  CA
	chi_required(1, 2,aa_asn,1) = false;
	chi_required(2, 2,aa_asn,1) = false;
	chi_required(3, 2,aa_asn,1) = false;
	chi_required(4, 2,aa_asn,1) = false;

//bk   chi angles needed for building  C
	chi_required(1, 3,aa_asn,1) = false;
	chi_required(2, 3,aa_asn,1) = false;
	chi_required(3, 3,aa_asn,1) = false;
	chi_required(4, 3,aa_asn,1) = false;

//bk   chi angles needed for building  O
	chi_required(1, 4,aa_asn,1) = false;
	chi_required(2, 4,aa_asn,1) = false;
	chi_required(3, 4,aa_asn,1) = false;
	chi_required(4, 4,aa_asn,1) = false;

//bk   chi angles needed for building  CB
	chi_required(1, 5,aa_asn,1) = false;
	chi_required(2, 5,aa_asn,1) = false;
	chi_required(3, 5,aa_asn,1) = false;
	chi_required(4, 5,aa_asn,1) = false;

//bk   chi angles needed for building  CG
	chi_required(1, 6,aa_asn,1) =  true;
	chi_required(2, 6,aa_asn,1) = false;
	chi_required(3, 6,aa_asn,1) = false;
	chi_required(4, 6,aa_asn,1) = false;

//bk   chi angles needed for building  OD1
	chi_required(1, 7,aa_asn,1) =  true;
	chi_required(2, 7,aa_asn,1) =  true;
	chi_required(3, 7,aa_asn,1) = false;
	chi_required(4, 7,aa_asn,1) = false;

//bk   chi angles needed for building  ND2
	chi_required(1, 8,aa_asn,1) =  true;
	chi_required(2, 8,aa_asn,1) =  true;
	chi_required(3, 8,aa_asn,1) = false;
	chi_required(4, 8,aa_asn,1) = false;

//bk   chi angles needed for building  H
	chi_required(1, 9,aa_asn,1) = false;
	chi_required(2, 9,aa_asn,1) = false;
	chi_required(3, 9,aa_asn,1) = false;
	chi_required(4, 9,aa_asn,1) = false;

//bk   chi angles needed for building 1HD2
	chi_required(1,10,aa_asn,1) =  true;
	chi_required(2,10,aa_asn,1) =  true;
	chi_required(3,10,aa_asn,1) = false;
	chi_required(4,10,aa_asn,1) = false;

//bk   chi angles needed for building 2HD2
	chi_required(1,11,aa_asn,1) =  true;
	chi_required(2,11,aa_asn,1) =  true;
	chi_required(3,11,aa_asn,1) = false;
	chi_required(4,11,aa_asn,1) = false;

//bk   chi angles needed for building  HA
	chi_required(1,12,aa_asn,1) = false;
	chi_required(2,12,aa_asn,1) = false;
	chi_required(3,12,aa_asn,1) = false;
	chi_required(4,12,aa_asn,1) = false;

//bk   chi angles needed for building 2HB
	chi_required(1,13,aa_asn,1) =  true;
	chi_required(2,13,aa_asn,1) = false;
	chi_required(3,13,aa_asn,1) = false;
	chi_required(4,13,aa_asn,1) = false;

//bk   chi angles needed for building 3HB
	chi_required(1,14,aa_asn,1) =  true;
	chi_required(2,14,aa_asn,1) = false;
	chi_required(3,14,aa_asn,1) = false;
	chi_required(4,14,aa_asn,1) = false;

//bk   four atoms that define chi angle  1
	chi_atoms( 1, 1,aa_asn, 1) =  1; //   N
	chi_atoms( 2, 1,aa_asn, 1) =  2; //   CA
	chi_atoms( 3, 1,aa_asn, 1) =  5; //   CB
	chi_atoms( 4, 1,aa_asn, 1) =  6; //   CG
//bk   four atoms that define chi angle  2
	chi_atoms( 1, 2,aa_asn, 1) =  2; //   CA
	chi_atoms( 2, 2,aa_asn, 1) =  5; //   CB
	chi_atoms( 3, 2,aa_asn, 1) =  6; //   CG
	chi_atoms( 4, 2,aa_asn, 1) =  7; //   OD1

	HNpos(aa_asn,1) =  9; // atom number for backbone HN
	HApos(aa_asn,1) = 12; // atom number for backbone HA
	nH_polar(aa_asn,1) =  3; // number of polar hydrogens
	nH_aromatic(aa_asn,1) =  0; // number of aromatic hydrogens
	nH_apolar(aa_asn,1) =  3; // number of apolar hydrogens
	nacceptors(aa_asn,1) =  2; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1,aa_asn,1) =  9; //  H
	Hpos_polar( 2,aa_asn,1) = 10; // 1HD2
	Hpos_polar( 3,aa_asn,1) = 11; // 2HD2

//bk   atom number for aromatic hydrogens

//bk   atom number for apolar hydrogens
	Hpos_apolar( 1,aa_asn,1) = 12; //  HA
	Hpos_apolar( 2,aa_asn,1) = 13; // 2HB
	Hpos_apolar( 3,aa_asn,1) = 14; // 3HB

//bk   acceptor positions
	accpt_pos( 1,aa_asn, 1) =  4; //  O
	accpt_pos( 2,aa_asn, 1) =  7; //  OD1

//bk   hydrogens on each atom

	nhydrogens_on_atm( 1,aa_asn, 1) =  1; //  N
	hydrogens_on_atm( 1, 1,aa_asn, 1) =  9; //  H

	nhydrogens_on_atm( 2,aa_asn, 1) =  1; //  CA
	hydrogens_on_atm( 1, 2,aa_asn, 1) = 12; //  HA

	nhydrogens_on_atm( 5,aa_asn, 1) =  2; //  CB
	hydrogens_on_atm( 1, 5,aa_asn, 1) = 13; // 2HB
	hydrogens_on_atm( 2, 5,aa_asn, 1) = 14; // 3HB

	nhydrogens_on_atm( 8,aa_asn, 1) =  2; //  ND2
	hydrogens_on_atm( 1, 8,aa_asn, 1) = 10; // 1HD2
	hydrogens_on_atm( 2, 8,aa_asn, 1) = 11; // 2HD2

//bk   template coordinates for the amino acid
//jjh new template from Kevin Karplus
	icoor( 1, 1,aa_asn, 1) =   1.463; //   N
	icoor( 2, 1,aa_asn, 1) =   0.000; //   N
	icoor( 3, 1,aa_asn, 1) =   0.000; //   N
	icoor( 1, 2,aa_asn, 1) =   0.000; //   CA
	icoor( 2, 2,aa_asn, 1) =   0.000; //   CA
	icoor( 3, 2,aa_asn, 1) =   0.000; //   CA
	icoor( 1, 3,aa_asn, 1) =   -0.565; //   C
	icoor( 2, 3,aa_asn, 1) =   1.411; //   C
	icoor( 3, 3,aa_asn, 1) =   0.000; //   C
	icoor( 1, 4,aa_asn, 1) =   -1.619; //   O
	icoor( 2, 4,aa_asn, 1) =   1.666; //   O
	icoor( 3, 4,aa_asn, 1) =   -0.603; //   O
	icoor( 1, 5,aa_asn, 1) =   -0.555; //   CB
	icoor( 2, 5,aa_asn, 1) =   -0.778; //   CB
	icoor( 3, 5,aa_asn, 1) =   1.179; //   CB
	icoor( 1, 6,aa_asn, 1) =   -0.282; //   CG
	icoor( 2, 6,aa_asn, 1) =   -2.255; //   CG
	icoor( 3, 6,aa_asn, 1) =   1.112; //   CG
	icoor( 1, 7,aa_asn, 1) =   0.050; //   OD1
	icoor( 2, 7,aa_asn, 1) =   -2.798; //   OD1
	icoor( 3, 7,aa_asn, 1) =   0.052; //   OD1
	icoor( 1, 8,aa_asn, 1) =   -0.499; //   ND2
	icoor( 2, 8,aa_asn, 1) =   -2.920; //   ND2
	icoor( 3, 8,aa_asn, 1) =   2.218; //   ND2
	icoor( 1, 9,aa_asn, 1) =   2.425; //   H
	icoor( 2, 9,aa_asn, 1) =   -0.306; //   H
	icoor( 3, 9,aa_asn, 1) =   0.000; //   H
	icoor( 1,10,aa_asn, 1) =   -0.338; //  1HD2
	icoor( 2,10,aa_asn, 1) =   -3.907; //  1HD2
	icoor( 3,10,aa_asn, 1) =   2.247; //  1HD2
	icoor( 1,11,aa_asn, 1) =   -0.825; //  2HD2
	icoor( 2,11,aa_asn, 1) =   -2.442; //  2HD2
	icoor( 3,11,aa_asn, 1) =    3.033; //  2HD2
	icoor( 1,12,aa_asn, 1) =   -0.370; //   HA
	icoor( 2,12,aa_asn, 1) =   -0.475; //   HA
	icoor( 3,12,aa_asn, 1) =   -0.910; //   HA
	icoor( 1,13,aa_asn, 1) =   -0.397; //  2HB
	icoor( 2,13,aa_asn, 1) =   -0.431; //  2HB
	icoor( 3,13,aa_asn, 1) =   2.201; //  2HB
	icoor( 1,14,aa_asn, 1) =   -1.591; //  3HB
	icoor( 2,14,aa_asn, 1) =   -0.599; //  3HB
	icoor( 3,14,aa_asn, 1) =   0.891; //  3HB

	// H positions from REDUCE
	if( param_pack::packer_logical::reduce_templates ) {
		icoor( 1 , 9 , aa_asn , 1) = 1.797 ; // H
		icoor( 2 , 9 , aa_asn , 1) = 0.471 ; // H
		icoor( 3 , 9 , aa_asn , 1) = -0.816 ; // H
		icoor( 1 , 12 , aa_asn , 1) = -0.314 ; // HA
		icoor( 2 , 12 , aa_asn , 1) = -0.497 ; // HA
		icoor( 3 , 12 , aa_asn , 1) = -0.929 ; // HA
		icoor( 1 , 13 , aa_asn , 1) = -0.123 ; // 1HB
		icoor( 2 , 13 , aa_asn , 1) = -0.376 ; // 1HB
		icoor( 3 , 13 , aa_asn , 1) = 2.108 ; // 1HB
		icoor( 1 , 14 , aa_asn , 1) = -1.642 ; // 2HB
		icoor( 2 , 14 , aa_asn , 1) = -0.617 ; // 2HB
		icoor( 3 , 14 , aa_asn , 1) = 1.234 ; // 2HB
		icoor( 1 , 10 , aa_asn , 1) = -0.398 ; // 1HD2
		icoor( 2 , 10 , aa_asn , 1) = -3.915 ; // 1HD2
		icoor( 3 , 10 , aa_asn , 1) = 2.231 ; // 1HD2
		icoor( 1 , 11 , aa_asn , 1) = -0.766 ; // 2HD2
		icoor( 2 , 11 , aa_asn , 1) = -2.433 ; // 2HD2
		icoor( 3 , 11 , aa_asn , 1) = 3.050 ; // 2HD2
	}

//bq create N and Cterm variants
	create_termini_variants( aa_asn );



//bk   create variants that have h2o connected to the backbone nh and carbonyl
	if ( explicit_h2o ) {
		create_backbone_h2o_variants(aa_asn);
		create_sidechain_h2o_variants(aa_asn);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_pro
///
/// @brief arrays filled to define the proerties of the amino acid Proline
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// aa_pro - index number for PRO (13)
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
init_pro()
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

//bk   Parameters for PRO -----------------------------
	nvar(aa_pro) = 1; // starting number of variants for PRO
	variant_type( aav_base, aa_pro, 1 ) = true; // type of aa variant

//bk   variant1 ---------- PRO

	natoms(aa_pro,1) = 14; // number of total atoms
	nheavyatoms(aa_pro,1) =  7; // number of heavy atoms
	nchi(aa_pro,1) =  1; // number of chi angles

//bk   atom names PRO
	atom_name( 1,aa_pro,1) = " N  ";
	atom_name( 2,aa_pro,1) = " CA ";
	atom_name( 3,aa_pro,1) = " C  ";
	atom_name( 4,aa_pro,1) = " O  ";
	atom_name( 5,aa_pro,1) = " CB ";
	atom_name( 6,aa_pro,1) = " CG ";
	atom_name( 7,aa_pro,1) = " CD ";
	atom_name( 8,aa_pro,1) = "2HD ";
	atom_name( 9,aa_pro,1) = "3HD ";
	atom_name(10,aa_pro,1) = "2HG ";
	atom_name(11,aa_pro,1) = "3HG ";
	atom_name(12,aa_pro,1) = "2HB ";
	atom_name(13,aa_pro,1) = "3HB ";
	atom_name(14,aa_pro,1) = " HA ";

//jjh intra residue bonding
	nbonded_neighbors( 1, aa_pro, 1) = 2; // N
		bonded_neighbor(1, 1, aa_pro, 1) = 2; // N--CA
		bonded_neighbor(2, 1, aa_pro, 1) = 7; // N--CD
	nbonded_neighbors( 2, aa_pro, 1) = 4; // CA
		bonded_neighbor(1, 2, aa_pro, 1) = 1; // CA--N
		bonded_neighbor(2, 2, aa_pro, 1) = 3; // CA--C
		bonded_neighbor(3, 2, aa_pro, 1) = 5; // CA--CB
		bonded_neighbor(4, 2, aa_pro, 1) =14; // CA--HA
	nbonded_neighbors( 3, aa_pro, 1) = 2; // C
		bonded_neighbor(1, 3, aa_pro, 1) = 2; // C--CA
		bonded_neighbor(2, 3, aa_pro, 1) = 4; // C--O
	nbonded_neighbors( 4, aa_pro, 1) = 1; // O
		bonded_neighbor(1, 4, aa_pro, 1) = 3; // O--C
	nbonded_neighbors( 5, aa_pro, 1) = 4; // CB
		bonded_neighbor(1, 5, aa_pro, 1) = 2; // CB--CA
		bonded_neighbor(2, 5, aa_pro, 1) = 6; // CB--CG
		bonded_neighbor(3, 5, aa_pro, 1) =12; // CB--2HB
		bonded_neighbor(4, 5, aa_pro, 1) =13; // CB--3HB
	nbonded_neighbors( 6, aa_pro, 1) = 4; // CG
		bonded_neighbor(1, 6, aa_pro, 1) = 5; // CG--CB
		bonded_neighbor(2, 6, aa_pro, 1) = 7; // CG--CD
		bonded_neighbor(3, 6, aa_pro, 1) =10; // CG--2HG
		bonded_neighbor(4, 6, aa_pro, 1) =11; // CG--3HG
	nbonded_neighbors( 7, aa_pro, 1) = 4; // CD
		bonded_neighbor(1, 7, aa_pro, 1) = 1; // CD--N
		bonded_neighbor(2, 7, aa_pro, 1) = 6; // CD--CG
		bonded_neighbor(3, 7, aa_pro, 1) = 8; // CD--2HD
		bonded_neighbor(4, 7, aa_pro, 1) = 9; // CD--3HD
	nbonded_neighbors( 8, aa_pro, 1) = 1; //2HD
		bonded_neighbor(1, 8, aa_pro, 1) = 7; //2HD--CD
	nbonded_neighbors( 9, aa_pro, 1) = 1; //3HD
		bonded_neighbor(1, 9, aa_pro, 1) = 7; //3HD--CD
	nbonded_neighbors( 10, aa_pro, 1) = 1; //2HG
		bonded_neighbor(1,10, aa_pro, 1) = 6; //2HG--CG
	nbonded_neighbors( 11, aa_pro, 1) = 1; //3HG
		bonded_neighbor(1,11, aa_pro, 1) = 6; //3HG--CG
	nbonded_neighbors( 12, aa_pro, 1) = 1; //2HB
		bonded_neighbor(1,12, aa_pro, 1) = 5; //2HB--CB
	nbonded_neighbors( 13, aa_pro, 1) = 1; //3HB
		bonded_neighbor(1,13, aa_pro, 1) = 5; //3HB--CB
	nbonded_neighbors( 14, aa_pro, 1) = 1; // HA
		bonded_neighbor(1,14, aa_pro, 1) = 2; // HA--CA

//jjh get the angles automatically
	get_angles_from_bonds(aa_pro, 1);

//chu   base atom numbers PRO
	atom_base( 1,aa_pro,1) =  2; //  N  -> CA
	atom_base( 2,aa_pro,1) =  1; //  CA -> N
	atom_base( 3,aa_pro,1) =  2; //  C  -> CA
	atom_base( 4,aa_pro,1) =  3; //  O  -> C
	atom_base( 5,aa_pro,1) =  2; //  CB -> CA
	atom_base( 6,aa_pro,1) =  5; //  CG -> CB
	atom_base( 7,aa_pro,1) =  6; //  CD -> CG (not consider CD <-> N)
//matan_kob, correct definition of ring: CG now connected to N (instead of CA),2008-02
  abase2(7,aa_pro,1) = 1; // for completeness
	atom_base( 8,aa_pro,1) =  7; // 2HD  on  CD
	atom_base( 9,aa_pro,1) =  7; // 3HD  on  CD
	atom_base(10,aa_pro,1) =  6; // 2HG  on  CG
	atom_base(11,aa_pro,1) =  6; // 3HG  on  CG
	atom_base(12,aa_pro,1) =  5; // 2HB  on  CB
	atom_base(13,aa_pro,1) =  5; // 3HB  on  CB
	atom_base(14,aa_pro,1) =  2; //  HA  on  CA
//bk   atom type numbers PRO
	fullatom_type( 1,aa_pro,1) = 12; // Npro   N
	fullatom_type( 2,aa_pro,1) = 18; // CAbb   CA
	fullatom_type( 3,aa_pro,1) = 19; // CObb   C
	fullatom_type( 4,aa_pro,1) = 20; // OCbb   O
	fullatom_type( 5,aa_pro,1) =  4; // CH2    CB
	fullatom_type( 6,aa_pro,1) =  4; // CH2    CG
	fullatom_type( 7,aa_pro,1) =  4; // CH2    CD
	fullatom_type( 8,aa_pro,1) = 23; // Hapo  2HD
	fullatom_type( 9,aa_pro,1) = 23; // Hapo  3HD
	fullatom_type(10,aa_pro,1) = 23; // Hapo  2HG
	fullatom_type(11,aa_pro,1) = 23; // Hapo  3HG
	fullatom_type(12,aa_pro,1) = 23; // Hapo  2HB
	fullatom_type(13,aa_pro,1) = 23; // Hapo  3HB
	fullatom_type(14,aa_pro,1) = 23; // Hapo   HA

//cmd  atomic charge PRO
	atomic_charge( 1,aa_pro,1) = -0.29; //    N
	atomic_charge( 2,aa_pro,1) =  0.02; //    CA
	atomic_charge( 3,aa_pro,1) =  0.51; //    C
	atomic_charge( 4,aa_pro,1) = -0.51; //    O
	atomic_charge( 5,aa_pro,1) = -0.18; //    CB
	atomic_charge( 6,aa_pro,1) = -0.18; //    CG
	atomic_charge( 7,aa_pro,1) =  0.00; //    CD
	atomic_charge( 8,aa_pro,1) =  0.09; //   2HD
	atomic_charge( 9,aa_pro,1) =  0.09; //   3HD
	atomic_charge(10,aa_pro,1) =  0.09; //   2HG
	atomic_charge(11,aa_pro,1) =  0.09; //   3HG
	atomic_charge(12,aa_pro,1) =  0.09; //   2HB
	atomic_charge(13,aa_pro,1) =  0.09; //   3HB
	atomic_charge(14,aa_pro,1) =  0.09; //    HA

	assert_total_charge( aa_pro, 1, 0.0);

//cmd  first sidechain atom
	first_scatom( aa_pro,1) = 5; //CB

//bk   template atoms used for placing atoms PRO

//bk   template for building 2HD
	ta(1, 8,aa_pro,1) =  7; //   CD
	ta(2, 8,aa_pro,1) =  1; //   N
	ta(3, 8,aa_pro,1) =  6; //   CG

//bk   template for building 3HD
	ta(1, 9,aa_pro,1) =  7; //   CD
	ta(2, 9,aa_pro,1) =  1; //   N
	ta(3, 9,aa_pro,1) =  6; //   CG

//bk   template for building 2HG
	ta(1,10,aa_pro,1) =  6; //   CG
	ta(2,10,aa_pro,1) =  5; //   CB
	ta(3,10,aa_pro,1) =  7; //   CD

//bk   template for building 3HG
	ta(1,11,aa_pro,1) =  6; //   CG
	ta(2,11,aa_pro,1) =  5; //   CB
	ta(3,11,aa_pro,1) =  7; //   CD

//bk   template for building 2HB
	ta(1,12,aa_pro,1) =  5; //   CB
	ta(2,12,aa_pro,1) =  2; //   CA
	ta(3,12,aa_pro,1) =  6; //   CG

//bk   template for building 3HB
	ta(1,13,aa_pro,1) =  5; //   CB
	ta(2,13,aa_pro,1) =  2; //   CA
	ta(3,13,aa_pro,1) =  6; //   CG

//bk   template for building  HA
	ta(1,14,aa_pro,1) =  2; //   CA
	ta(2,14,aa_pro,1) =  1; //   N
	ta(3,14,aa_pro,1) =  3; //   C

//bk   chi angles required to build atoms PRO

//bk   chi angles needed for building  N
	chi_required(1, 1,aa_pro,1) = false;
	chi_required(2, 1,aa_pro,1) = false;
	chi_required(3, 1,aa_pro,1) = false;
	chi_required(4, 1,aa_pro,1) = false;

//bk   chi angles needed for building  CA
	chi_required(1, 2,aa_pro,1) = false;
	chi_required(2, 2,aa_pro,1) = false;
	chi_required(3, 2,aa_pro,1) = false;
	chi_required(4, 2,aa_pro,1) = false;

//bk   chi angles needed for building  C
	chi_required(1, 3,aa_pro,1) = false;
	chi_required(2, 3,aa_pro,1) = false;
	chi_required(3, 3,aa_pro,1) = false;
	chi_required(4, 3,aa_pro,1) = false;

//bk   chi angles needed for building  O
	chi_required(1, 4,aa_pro,1) = false;
	chi_required(2, 4,aa_pro,1) = false;
	chi_required(3, 4,aa_pro,1) = false;
	chi_required(4, 4,aa_pro,1) = false;

//bk   chi angles needed for building  CB
	chi_required(1, 5,aa_pro,1) = false;
	chi_required(2, 5,aa_pro,1) = false;
	chi_required(3, 5,aa_pro,1) = false;
	chi_required(4, 5,aa_pro,1) = false;

//bk   chi angles needed for building  CG
	chi_required(1, 6,aa_pro,1) =  true;
	chi_required(2, 6,aa_pro,1) = false;
	chi_required(3, 6,aa_pro,1) = false;
	chi_required(4, 6,aa_pro,1) = false;

//bk   chi angles needed for building  CD
	chi_required(1, 7,aa_pro,1) = false;
	chi_required(2, 7,aa_pro,1) = false;
	chi_required(3, 7,aa_pro,1) = false;
	chi_required(4, 7,aa_pro,1) = false;

//bk   chi angles needed for building 2HD
	chi_required(1, 8,aa_pro,1) = false;
	chi_required(2, 8,aa_pro,1) = false;
	chi_required(3, 8,aa_pro,1) = false;
	chi_required(4, 8,aa_pro,1) = false;

//bk   chi angles needed for building 3HD
	chi_required(1, 9,aa_pro,1) = false;
	chi_required(2, 9,aa_pro,1) = false;
	chi_required(3, 9,aa_pro,1) = false;
	chi_required(4, 9,aa_pro,1) = false;

//bk   chi angles needed for building 2HG
	chi_required(1,10,aa_pro,1) =  true;
	chi_required(2,10,aa_pro,1) = false;
	chi_required(3,10,aa_pro,1) = false;
	chi_required(4,10,aa_pro,1) = false;

//bk   chi angles needed for building 3HG
	chi_required(1,11,aa_pro,1) =  true;
	chi_required(2,11,aa_pro,1) = false;
	chi_required(3,11,aa_pro,1) = false;
	chi_required(4,11,aa_pro,1) = false;

//bk   chi angles needed for building 2HB
	chi_required(1,12,aa_pro,1) =  true;
	chi_required(2,12,aa_pro,1) = false;
	chi_required(3,12,aa_pro,1) = false;
	chi_required(4,12,aa_pro,1) = false;

//bk   chi angles needed for building 3HB
	chi_required(1,13,aa_pro,1) =  true;
	chi_required(2,13,aa_pro,1) = false;
	chi_required(3,13,aa_pro,1) = false;
	chi_required(4,13,aa_pro,1) = false;

//bk   chi angles needed for building  HA
	chi_required(1,14,aa_pro,1) = false;
	chi_required(2,14,aa_pro,1) = false;
	chi_required(3,14,aa_pro,1) = false;
	chi_required(4,14,aa_pro,1) = false;

//bk   four atoms that define chi angle  1
	chi_atoms( 1, 1,aa_pro, 1) =  1; //   N
	chi_atoms( 2, 1,aa_pro, 1) =  2; //   CA
	chi_atoms( 3, 1,aa_pro, 1) =  5; //   CB
	chi_atoms( 4, 1,aa_pro, 1) =  6; //   CG

	HNpos(aa_pro,1) =  0; // atom number for backbone HN
	HApos(aa_pro,1) = 14; // atom number for backbone HA
	nH_polar(aa_pro,1) =  0; // number of polar hydrogens
	nH_aromatic(aa_pro,1) =  0; // number of aromatic hydrogens
	nH_apolar(aa_pro,1) =  7; // number of apolar hydrogens
	nacceptors(aa_pro,1) =  1; // number of acceptors

//bk   atom numbers for polar H

//bk   atom number for aromatic hydrogens

//bk   atom number for apolar hydrogens
	Hpos_apolar( 1,aa_pro,1) =  8; // 2HD
	Hpos_apolar( 2,aa_pro,1) =  9; // 3HD
	Hpos_apolar( 3,aa_pro,1) = 10; // 2HG
	Hpos_apolar( 4,aa_pro,1) = 11; // 3HG
	Hpos_apolar( 5,aa_pro,1) = 12; // 2HB
	Hpos_apolar( 6,aa_pro,1) = 13; // 3HB
	Hpos_apolar( 7,aa_pro,1) = 14; //  HA

//bk   acceptor positions
	accpt_pos( 1,aa_pro, 1) =  4; //  O

//bk   hydrogens on each atom

	nhydrogens_on_atm( 2,aa_pro, 1) =  1; //  CA
	hydrogens_on_atm( 1, 2,aa_pro, 1) = 14; //  HA

	nhydrogens_on_atm( 5,aa_pro, 1) =  2; //  CB
	hydrogens_on_atm( 1, 5,aa_pro, 1) = 12; // 2HB
	hydrogens_on_atm( 2, 5,aa_pro, 1) = 13; // 3HB

	nhydrogens_on_atm( 6,aa_pro, 1) =  2; //  CG
	hydrogens_on_atm( 1, 6,aa_pro, 1) = 10; // 2HG
	hydrogens_on_atm( 2, 6,aa_pro, 1) = 11; // 3HG

	nhydrogens_on_atm( 7,aa_pro, 1) =  2; //  CD
	hydrogens_on_atm( 1, 7,aa_pro, 1) =  8; // 2HD
	hydrogens_on_atm( 2, 7,aa_pro, 1) =  9; // 3HD

//bk   template coordinates for the amino acid
	icoor( 1, 1,aa_pro, 1) =   13.895; //   N
	icoor( 2, 1,aa_pro, 1) =    1.827; //   N
	icoor( 3, 1,aa_pro, 1) =    9.451; //   N
	icoor( 1, 2,aa_pro, 1) =   15.259; //   CA
	icoor( 2, 2,aa_pro, 1) =    2.242; //   CA
	icoor( 3, 2,aa_pro, 1) =    9.192; //   CA
	icoor( 1, 3,aa_pro, 1) =   15.959; //   C
	icoor( 2, 3,aa_pro, 1) =    1.321; //   C
	icoor( 3, 3,aa_pro, 1) =    8.222; //   C
	icoor( 1, 4,aa_pro, 1) =   15.702; //   O
	icoor( 2, 4,aa_pro, 1) =    0.132; //   O
	icoor( 3, 4,aa_pro, 1) =    8.143; //   O
	icoor( 1, 5,aa_pro, 1) =   15.927; //   CB
	icoor( 2, 5,aa_pro, 1) =    2.223; //   CB
	icoor( 3, 5,aa_pro, 1) =   10.577; //   CB
	icoor( 1, 6,aa_pro, 1) =   15.069; //   CG
	icoor( 2, 6,aa_pro, 1) =    1.310; //   CG
	icoor( 3, 6,aa_pro, 1) =   11.409; //   CG
	icoor( 1, 7,aa_pro, 1) =   13.672; //   CD
	icoor( 2, 7,aa_pro, 1) =    1.471; //   CD
	icoor( 3, 7,aa_pro, 1) =   10.862; //   CD
	icoor( 1, 8,aa_pro, 1) =   13.189; //  2HD
	icoor( 2, 8,aa_pro, 1) =    0.650; //  2HD
	icoor( 3, 8,aa_pro, 1) =   10.821; //  2HD
	icoor( 1, 9,aa_pro, 1) =   13.219; //  3HD
	icoor( 2, 9,aa_pro, 1) =    2.235; //  3HD
	icoor( 3, 9,aa_pro, 1) =   11.206; //  3HD
	icoor( 1,10,aa_pro, 1) =   15.229; //  2HG
	icoor( 2,10,aa_pro, 1) =    0.510; //  2HG
	icoor( 3,10,aa_pro, 1) =   11.182; //  2HG
	icoor( 1,11,aa_pro, 1) =   14.957; //  3HG
	icoor( 2,11,aa_pro, 1) =    1.715; //  3HG
	icoor( 3,11,aa_pro, 1) =   12.247; //  3HG
	icoor( 1,12,aa_pro, 1) =   16.695; //  2HB
	icoor( 2,12,aa_pro, 1) =    1.763; //  2HB
	icoor( 3,12,aa_pro, 1) =   10.389; //  2HB
	icoor( 1,13,aa_pro, 1) =   15.782; //  3HB
	icoor( 2,13,aa_pro, 1) =    3.034; //  3HB
	icoor( 3,13,aa_pro, 1) =   10.843; //  3HB
	icoor( 1,14,aa_pro, 1) =   15.270; //   HA
	icoor( 2,14,aa_pro, 1) =    3.160; //   HA
	icoor( 3,14,aa_pro, 1) =    8.805; //   HA

	// H positions from REDUCE
	if( param_pack::packer_logical::reduce_templates ) {
		icoor( 1 , 14 , aa_pro , 1) = 0.889 ; // HA
		icoor( 2 , 14 , aa_pro , 1) = 18.090 ; // HA
		icoor( 3 , 14 , aa_pro , 1) = 52.283 ; // HA
		icoor( 1 , 12 , aa_pro , 1) = -1.704 ; // 1HB
		icoor( 2 , 12 , aa_pro , 1) = 18.480 ; // 1HB
		icoor( 3 , 12 , aa_pro , 1) = 53.100 ; // 1HB
		icoor( 1 , 13 , aa_pro , 1) = -1.183 ; // 2HB
		icoor( 2 , 13 , aa_pro , 1) = 19.165 ; // 2HB
		icoor( 3 , 13 , aa_pro , 1) = 51.554 ; // 2HB
		icoor( 1 , 10 , aa_pro , 1) = -2.729 ; // 1HG
		icoor( 2 , 10 , aa_pro , 1) = 16.608 ; // 1HG
		icoor( 3 , 10 , aa_pro , 1) = 52.085 ; // 1HG
		icoor( 1 , 11 , aa_pro , 1) = -3.031 ; // 2HG
		icoor( 2 , 11 , aa_pro , 1) = 17.856 ; // 2HG
		icoor( 3 , 11 , aa_pro , 1) = 50.870 ; // 2HG
		icoor( 1 , 8 , aa_pro , 1) = -1.714 ; // 1HD
		icoor( 2 , 8 , aa_pro , 1) = 15.500 ; // 1HD
		icoor( 3 , 8 , aa_pro , 1) = 50.294 ; // 1HD
		icoor( 1 , 9 , aa_pro , 1) = -1.427 ; // 2HD
		icoor( 2 , 9 , aa_pro , 1) = 16.997 ; // 2HD
		icoor( 3 , 9 , aa_pro , 1) = 49.397 ; // 2HD
	}

//bq create N and Cterm variants
	create_termini_variants( aa_pro );


//bk   create variants that have h2o connected to the backbone carbonyl
	if ( explicit_h2o ) {
		create_backbone_h2o_variants(aa_pro);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_gln
///
/// @brief arrays filled to define the proerties of the amino acid Glutamine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// aa_gln - index number for GLN (14)
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_gln()
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

//bk   Parameters for GLN -----------------------------
	nvar(aa_gln) = 1; // starting number of variants for GLN
	variant_type( aav_base, aa_gln, 1 ) = true; // type of aa variant

//bk   variant1 ---------- GLN

	natoms(aa_gln,1) = 17; // number of total atoms
	nheavyatoms(aa_gln,1) =  9; // number of heavy atoms
	nchi(aa_gln,1) =  3; // number of chi angles

//bk   atom names GLN
	atom_name( 1,aa_gln,1) = " N  ";
	atom_name( 2,aa_gln,1) = " CA ";
	atom_name( 3,aa_gln,1) = " C  ";
	atom_name( 4,aa_gln,1) = " O  ";
	atom_name( 5,aa_gln,1) = " CB ";
	atom_name( 6,aa_gln,1) = " CG ";
	atom_name( 7,aa_gln,1) = " CD ";
	atom_name( 8,aa_gln,1) = " OE1";
	atom_name( 9,aa_gln,1) = " NE2";
	atom_name(10,aa_gln,1) = " H  ";
	atom_name(11,aa_gln,1) = "1HE2";
	atom_name(12,aa_gln,1) = "2HE2";
	atom_name(13,aa_gln,1) = " HA ";
	atom_name(14,aa_gln,1) = "2HB ";
	atom_name(15,aa_gln,1) = "3HB ";
	atom_name(16,aa_gln,1) = "2HG ";
	atom_name(17,aa_gln,1) = "3HG ";

//jjh intra residue bonding
	nbonded_neighbors( 1, aa_gln, 1) = 2; // N
		bonded_neighbor(1, 1, aa_gln, 1) = 2; // N--CA
		bonded_neighbor(2, 1, aa_gln, 1) =10; // N--H
	nbonded_neighbors( 2, aa_gln, 1) = 4; // CA
		bonded_neighbor(1, 2, aa_gln, 1) = 1; // CA--N
		bonded_neighbor(2, 2, aa_gln, 1) = 3; // CA--C
		bonded_neighbor(3, 2, aa_gln, 1) = 5; // CA--CB
		bonded_neighbor(4, 2, aa_gln, 1) =13; // CA--HA
	nbonded_neighbors( 3, aa_gln, 1) = 2; // C
		bonded_neighbor(1, 3, aa_gln, 1) = 2; // C--CA
		bonded_neighbor(2, 3, aa_gln, 1) = 4; // C--O
	nbonded_neighbors( 4, aa_gln, 1) = 1; // O
		bonded_neighbor(1, 4, aa_gln, 1) = 3; // O--C
	nbonded_neighbors( 5, aa_gln, 1) = 4; // CB
		bonded_neighbor(1, 5, aa_gln, 1) = 2; // CB--CA
		bonded_neighbor(2, 5, aa_gln, 1) = 6; // CB--CG
		bonded_neighbor(3, 5, aa_gln, 1) =14; // CB--2HB
		bonded_neighbor(4, 5, aa_gln, 1) =15; // CB--3HB
	nbonded_neighbors( 6, aa_gln, 1) = 4; // CG
		bonded_neighbor(1, 6, aa_gln, 1) = 5; // CG--CB
		bonded_neighbor(2, 6, aa_gln, 1) = 7; // CG--CD
		bonded_neighbor(3, 6, aa_gln, 1) =16; // CG--2HG
		bonded_neighbor(4, 6, aa_gln, 1) =17; // CG--3HG
	nbonded_neighbors( 7, aa_gln, 1) = 3; // CD
		bonded_neighbor(1, 7, aa_gln, 1) = 6; // CD--CG
		bonded_neighbor(2, 7, aa_gln, 1) = 8; // CD--OE1
		bonded_neighbor(3, 7, aa_gln, 1) = 9; // CD--NE2
	nbonded_neighbors( 8, aa_gln, 1) = 1; // OE1
		bonded_neighbor(1, 8, aa_gln, 1) = 7; // OE1--CD
	nbonded_neighbors( 9, aa_gln, 1) = 3; // NE2
		bonded_neighbor(1, 9, aa_gln, 1) = 7; // NE2--CD
		bonded_neighbor(2, 9, aa_gln, 1) =11; // NE2--1HE2
		bonded_neighbor(3, 9, aa_gln, 1) =12; // NE2--2HE2
	nbonded_neighbors( 10, aa_gln, 1) = 1; // H
		bonded_neighbor(1,10, aa_gln, 1) = 1; // H--N
	nbonded_neighbors( 11, aa_gln, 1) = 1; //1HE2
		bonded_neighbor(1,11, aa_gln, 1) = 9; //1HE2--NE2
	nbonded_neighbors( 12, aa_gln, 1) = 1; //2HE2
		bonded_neighbor(1,12, aa_gln, 1) = 9; //2HE2--NE2
	nbonded_neighbors( 13, aa_gln, 1) = 1; // HA
		bonded_neighbor(1,13, aa_gln, 1) = 2; // HA--CA
	nbonded_neighbors( 14, aa_gln, 1) = 1; //2HB
		bonded_neighbor(1,14, aa_gln, 1) = 5; //2HB--CB
	nbonded_neighbors( 15, aa_gln, 1) = 1; //3HB
		bonded_neighbor(1,15, aa_gln, 1) = 5; //3HB--CB
	nbonded_neighbors( 16, aa_gln, 1) = 1; //2HG
		bonded_neighbor(1,16, aa_gln, 1) = 6; //2HG--CG
	nbonded_neighbors( 17, aa_gln, 1) = 1; //3HG
		bonded_neighbor(1,17, aa_gln, 1) = 6; //3HG--CG

//jjh get the angles automatically
	get_angles_from_bonds(aa_gln, 1);

//chu   base atom numbers GLN
	atom_base( 1,aa_gln,1) =  2; //  N  -> CA
	atom_base( 2,aa_gln,1) =  1; //  CA -> N
	atom_base( 3,aa_gln,1) =  2; //  C  -> CA
	atom_base( 4,aa_gln,1) =  3; //  O  -> C
	atom_base( 5,aa_gln,1) =  2; //  CB -> CA
	atom_base( 6,aa_gln,1) =  5; //  CG -> CB
	atom_base( 7,aa_gln,1) =  6; //  CD -> CG
	atom_base( 8,aa_gln,1) =  7; //  OE1 on  CD
	atom_base( 9,aa_gln,1) =  7; //  NE2 on  CD
	atom_base(10,aa_gln,1) =  1; //  H   on  N
	atom_base(11,aa_gln,1) =  9; // 1HE2 on  NE2
	atom_base(12,aa_gln,1) =  9; // 2HE2 on  NE2
	atom_base(13,aa_gln,1) =  2; //  HA  on  CA
	atom_base(14,aa_gln,1) =  5; // 2HB  on  CB
	atom_base(15,aa_gln,1) =  5; // 3HB  on  CB
	atom_base(16,aa_gln,1) =  6; // 2HG  on  CG
	atom_base(17,aa_gln,1) =  6; // 3HG  on  CG
//bk   atom type numbers GLN
	fullatom_type( 1,aa_gln,1) = 17; // Nbb    N
	fullatom_type( 2,aa_gln,1) = 18; // CAbb   CA
	fullatom_type( 3,aa_gln,1) = 19; // CObb   C
	fullatom_type( 4,aa_gln,1) = 20; // OCbb   O
	fullatom_type( 5,aa_gln,1) =  4; // CH2    CB
	fullatom_type( 6,aa_gln,1) =  4; // CH2    CG
	fullatom_type( 7,aa_gln,1) =  1; // CNH2   CD
	fullatom_type( 8,aa_gln,1) = 14; // ONH2   OE1
	fullatom_type( 9,aa_gln,1) =  9; // NH2O   NE2
	fullatom_type(10,aa_gln,1) = 25; // HNbb   H
	fullatom_type(11,aa_gln,1) = 22; // Hpol  1HE2
	fullatom_type(12,aa_gln,1) = 22; // Hpol  2HE2
	fullatom_type(13,aa_gln,1) = 23; // Hapo   HA
	fullatom_type(14,aa_gln,1) = 23; // Hapo  2HB
	fullatom_type(15,aa_gln,1) = 23; // Hapo  3HB
	fullatom_type(16,aa_gln,1) = 23; // Hapo  2HG
	fullatom_type(17,aa_gln,1) = 23; // Hapo  3HG

//cmd  atomic charge GLN
	atomic_charge( 1,aa_gln,1) = -0.47; //    N
	atomic_charge( 2,aa_gln,1) =  0.07; //    CA
	atomic_charge( 3,aa_gln,1) =  0.51; //    C
	atomic_charge( 4,aa_gln,1) = -0.51; //    O
	atomic_charge( 5,aa_gln,1) = -0.18; //    CB
	atomic_charge( 6,aa_gln,1) = -0.18; //    CG
	atomic_charge( 7,aa_gln,1) =  0.55; //    CD
	atomic_charge( 8,aa_gln,1) = -0.55; //    OE1
	atomic_charge( 9,aa_gln,1) = -0.62; //    NE2
	atomic_charge(10,aa_gln,1) =  0.31; //    H
	atomic_charge(11,aa_gln,1) =  0.32; //   1HE2
	atomic_charge(12,aa_gln,1) =  0.30; //   2HE2
	atomic_charge(13,aa_gln,1) =  0.09; //    HA
	atomic_charge(14,aa_gln,1) =  0.09; //   2HB
	atomic_charge(15,aa_gln,1) =  0.09; //   3HB
	atomic_charge(16,aa_gln,1) =  0.09; //   2HG
	atomic_charge(17,aa_gln,1) =  0.09; //   3HG

	assert_total_charge( aa_gln, 1, 0.0);

//cmd  first sidechain atom
	first_scatom( aa_gln,1) = 5; //CB

//bk   template atoms used for placing atoms GLN

//bk   template for building 1HE2
	ta(1,11,aa_gln,1) =  9; //   NE2
	ta(2,11,aa_gln,1) =  7; //   CD
	ta(3,11,aa_gln,1) =  6; //   CG

//bk   template for building 2HE2
	ta(1,12,aa_gln,1) =  9; //   NE2
	ta(2,12,aa_gln,1) =  7; //   CD
	ta(3,12,aa_gln,1) =  6; //   CG

//bk   template for building  HA
	ta(1,13,aa_gln,1) =  2; //   CA
	ta(2,13,aa_gln,1) =  1; //   N
	ta(3,13,aa_gln,1) =  3; //   C

//bk   template for building 2HB
	ta(1,14,aa_gln,1) =  5; //   CB
	ta(2,14,aa_gln,1) =  2; //   CA
	ta(3,14,aa_gln,1) =  6; //   CG

//bk   template for building 3HB
	ta(1,15,aa_gln,1) =  5; //   CB
	ta(2,15,aa_gln,1) =  2; //   CA
	ta(3,15,aa_gln,1) =  6; //   CG

//bk   template for building 2HG
	ta(1,16,aa_gln,1) =  6; //   CG
	ta(2,16,aa_gln,1) =  5; //   CB
	ta(3,16,aa_gln,1) =  7; //   CD

//bk   template for building 3HG
	ta(1,17,aa_gln,1) =  6; //   CG
	ta(2,17,aa_gln,1) =  5; //   CB
	ta(3,17,aa_gln,1) =  7; //   CD

//bk   chi angles required to build atoms GLN

//bk   chi angles needed for building  N
	chi_required(1, 1,aa_gln,1) = false;
	chi_required(2, 1,aa_gln,1) = false;
	chi_required(3, 1,aa_gln,1) = false;
	chi_required(4, 1,aa_gln,1) = false;

//bk   chi angles needed for building  CA
	chi_required(1, 2,aa_gln,1) = false;
	chi_required(2, 2,aa_gln,1) = false;
	chi_required(3, 2,aa_gln,1) = false;
	chi_required(4, 2,aa_gln,1) = false;

//bk   chi angles needed for building  C
	chi_required(1, 3,aa_gln,1) = false;
	chi_required(2, 3,aa_gln,1) = false;
	chi_required(3, 3,aa_gln,1) = false;
	chi_required(4, 3,aa_gln,1) = false;

//bk   chi angles needed for building  O
	chi_required(1, 4,aa_gln,1) = false;
	chi_required(2, 4,aa_gln,1) = false;
	chi_required(3, 4,aa_gln,1) = false;
	chi_required(4, 4,aa_gln,1) = false;

//bk   chi angles needed for building  CB
	chi_required(1, 5,aa_gln,1) = false;
	chi_required(2, 5,aa_gln,1) = false;
	chi_required(3, 5,aa_gln,1) = false;
	chi_required(4, 5,aa_gln,1) = false;

//bk   chi angles needed for building  CG
	chi_required(1, 6,aa_gln,1) =  true;
	chi_required(2, 6,aa_gln,1) = false;
	chi_required(3, 6,aa_gln,1) = false;
	chi_required(4, 6,aa_gln,1) = false;

//bk   chi angles needed for building  CD
	chi_required(1, 7,aa_gln,1) =  true;
	chi_required(2, 7,aa_gln,1) =  true;
	chi_required(3, 7,aa_gln,1) = false;
	chi_required(4, 7,aa_gln,1) = false;

//bk   chi angles needed for building  OE1
	chi_required(1, 8,aa_gln,1) =  true;
	chi_required(2, 8,aa_gln,1) =  true;
	chi_required(3, 8,aa_gln,1) =  true;
	chi_required(4, 8,aa_gln,1) = false;

//bk   chi angles needed for building  NE2
	chi_required(1, 9,aa_gln,1) =  true;
	chi_required(2, 9,aa_gln,1) =  true;
	chi_required(3, 9,aa_gln,1) =  true;
	chi_required(4, 9,aa_gln,1) = false;

//bk   chi angles needed for building  H
	chi_required(1,10,aa_gln,1) = false;
	chi_required(2,10,aa_gln,1) = false;
	chi_required(3,10,aa_gln,1) = false;
	chi_required(4,10,aa_gln,1) = false;

//bk   chi angles needed for building 1HE2
	chi_required(1,11,aa_gln,1) =  true;
	chi_required(2,11,aa_gln,1) =  true;
	chi_required(3,11,aa_gln,1) =  true;
	chi_required(4,11,aa_gln,1) = false;

//bk   chi angles needed for building 2HE2
	chi_required(1,12,aa_gln,1) =  true;
	chi_required(2,12,aa_gln,1) =  true;
	chi_required(3,12,aa_gln,1) =  true;
	chi_required(4,12,aa_gln,1) = false;

//bk   chi angles needed for building  HA
	chi_required(1,13,aa_gln,1) = false;
	chi_required(2,13,aa_gln,1) = false;
	chi_required(3,13,aa_gln,1) = false;
	chi_required(4,13,aa_gln,1) = false;

//bk   chi angles needed for building 2HB
	chi_required(1,14,aa_gln,1) =  true;
	chi_required(2,14,aa_gln,1) = false;
	chi_required(3,14,aa_gln,1) = false;
	chi_required(4,14,aa_gln,1) = false;

//bk   chi angles needed for building 3HB
	chi_required(1,15,aa_gln,1) =  true;
	chi_required(2,15,aa_gln,1) = false;
	chi_required(3,15,aa_gln,1) = false;
	chi_required(4,15,aa_gln,1) = false;

//bk   chi angles needed for building 2HG
	chi_required(1,16,aa_gln,1) =  true;
	chi_required(2,16,aa_gln,1) =  true;
	chi_required(3,16,aa_gln,1) = false;
	chi_required(4,16,aa_gln,1) = false;

//bk   chi angles needed for building 3HG
	chi_required(1,17,aa_gln,1) =  true;
	chi_required(2,17,aa_gln,1) =  true;
	chi_required(3,17,aa_gln,1) = false;
	chi_required(4,17,aa_gln,1) = false;

//bk   four atoms that define chi angle  1
	chi_atoms( 1, 1,aa_gln, 1) =  1; //   N
	chi_atoms( 2, 1,aa_gln, 1) =  2; //   CA
	chi_atoms( 3, 1,aa_gln, 1) =  5; //   CB
	chi_atoms( 4, 1,aa_gln, 1) =  6; //   CG
//bk   four atoms that define chi angle  2
	chi_atoms( 1, 2,aa_gln, 1) =  2; //   CA
	chi_atoms( 2, 2,aa_gln, 1) =  5; //   CB
	chi_atoms( 3, 2,aa_gln, 1) =  6; //   CG
	chi_atoms( 4, 2,aa_gln, 1) =  7; //   CD
//bk   four atoms that define chi angle  3
	chi_atoms( 1, 3,aa_gln, 1) =  5; //   CB
	chi_atoms( 2, 3,aa_gln, 1) =  6; //   CG
	chi_atoms( 3, 3,aa_gln, 1) =  7; //   CD
	chi_atoms( 4, 3,aa_gln, 1) =  8; //   OE1

	HNpos(aa_gln,1) = 10; // atom number for backbone HN
	HApos(aa_gln,1) = 13; // atom number for backbone HA
	nH_polar(aa_gln,1) =  3; // number of polar hydrogens
	nH_aromatic(aa_gln,1) =  0; // number of aromatic hydrogens
	nH_apolar(aa_gln,1) =  5; // number of apolar hydrogens
	nacceptors(aa_gln,1) =  2; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1,aa_gln,1) = 10; //  H
	Hpos_polar( 2,aa_gln,1) = 11; // 1HE2
	Hpos_polar( 3,aa_gln,1) = 12; // 2HE2

//bk   atom number for aromatic hydrogens

//bk   atom number for apolar hydrogens
	Hpos_apolar( 1,aa_gln,1) = 13; //  HA
	Hpos_apolar( 2,aa_gln,1) = 14; // 2HB
	Hpos_apolar( 3,aa_gln,1) = 15; // 3HB
	Hpos_apolar( 4,aa_gln,1) = 16; // 2HG
	Hpos_apolar( 5,aa_gln,1) = 17; // 3HG

//bk   acceptor positions
	accpt_pos( 1,aa_gln, 1) =  4; //  O
	accpt_pos( 2,aa_gln, 1) =  8; //  OE1

//bk   hydrogens on each atom

	nhydrogens_on_atm( 1,aa_gln, 1) =  1; //  N
	hydrogens_on_atm( 1, 1,aa_gln, 1) = 10; //  H

	nhydrogens_on_atm( 2,aa_gln, 1) =  1; //  CA
	hydrogens_on_atm( 1, 2,aa_gln, 1) = 13; //  HA

	nhydrogens_on_atm( 5,aa_gln, 1) =  2; //  CB
	hydrogens_on_atm( 1, 5,aa_gln, 1) = 14; // 2HB
	hydrogens_on_atm( 2, 5,aa_gln, 1) = 15; // 3HB

	nhydrogens_on_atm( 6,aa_gln, 1) =  2; //  CG
	hydrogens_on_atm( 1, 6,aa_gln, 1) = 16; // 2HG
	hydrogens_on_atm( 2, 6,aa_gln, 1) = 17; // 3HG

	nhydrogens_on_atm( 9,aa_gln, 1) =  2; //  NE2
	hydrogens_on_atm( 1, 9,aa_gln, 1) = 11; // 1HE2
	hydrogens_on_atm( 2, 9,aa_gln, 1) = 12; // 2HE2

//bk   template coordinates for the amino acid
//jjh new template from Kevin Karplus
	icoor( 1, 1,aa_gln, 1) =   1.458; //   N
	icoor( 2, 1,aa_gln, 1) =   0.000; //   N
	icoor( 3, 1,aa_gln, 1) =   0.000; //   N
	icoor( 1, 2,aa_gln, 1) =   0.000; //   CA
	icoor( 2, 2,aa_gln, 1) =   0.000; //   CA
	icoor( 3, 2,aa_gln, 1) =   0.000; //   CA
	icoor( 1, 3,aa_gln, 1) =   -0.553; //   C
	icoor( 2, 3,aa_gln, 1) =   1.420; //   C
	icoor( 3, 3,aa_gln, 1) =   0.000; //   C
	icoor( 1, 4,aa_gln, 1) =   -1.607; //   O
	icoor( 2, 4,aa_gln, 1) =   1.682; //   O
	icoor( 3, 4,aa_gln, 1) =   -0.580; //   O
	icoor( 1, 5,aa_gln, 1) =   -0.536; //   CB
	icoor( 2, 5,aa_gln, 1) =   -0.762; //   CB
	icoor( 3, 5,aa_gln, 1) =   1.215; //   CB
	icoor( 1, 6,aa_gln, 1) =   -0.317; //   CG
	icoor( 2, 6,aa_gln, 1) =   -2.264; //   CG
	icoor( 3, 6,aa_gln, 1) =   1.154; //   CG
	icoor( 1, 7,aa_gln, 1) =   -0.856; //   CD
	icoor( 2, 7,aa_gln, 1) =   -2.876; //   CD
	icoor( 3, 7,aa_gln, 1) =   -0.125; //   CD
	icoor( 1, 8,aa_gln, 1) =   -2.024; //   OE1
	icoor( 2, 8,aa_gln, 1) =   -2.687; //   OE1
	icoor( 3, 8,aa_gln, 1) =   -0.476; //   OE1
	icoor( 1, 9,aa_gln, 1) =   -0.008; //   NE2
	icoor( 2, 9,aa_gln, 1) =   -3.618; //   NE2
	icoor( 3, 9,aa_gln, 1) =   -0.828; //   NE2
	icoor( 1,10,aa_gln, 1) =   2.424; //   H
	icoor( 2,10,aa_gln, 1) =   -0.294; //   H
	icoor( 3,10,aa_gln, 1) =   0.000; //   H
	icoor( 1,11,aa_gln, 1) =    -0.307; //  1HE2
	icoor( 2,11,aa_gln, 1) =   -4.048; //  1HE2
	icoor( 3,11,aa_gln, 1) =   -1.681; //  1HE2
	icoor( 1,12,aa_gln, 1) =   0.930; //  2HE2
	icoor( 2,12,aa_gln, 1) =   -3.747; //  2HE2
	icoor( 3,12,aa_gln, 1) =   -0.506; //  2HE2
	icoor( 1,13,aa_gln, 1) =    -0.365; //   HA
	icoor( 2,13,aa_gln, 1) =   -0.474; //   HA
	icoor( 3,13,aa_gln, 1) =   -0.911; //   HA
	icoor( 1,14,aa_gln, 1) =   -0.034; //  2HB
	icoor( 2,14,aa_gln, 1) =   -0.351; //  2HB
	icoor( 3,14,aa_gln, 1) =   2.091; //  2HB
	icoor( 1,15,aa_gln, 1) =   -1.603; //  3HB
	icoor( 2,15,aa_gln, 1) =   -0.548; //  3HB
	icoor( 3,15,aa_gln, 1) =   1.275; //  3HB
	icoor( 1,16,aa_gln, 1) =   0.645; //  2HG
	icoor( 2,16,aa_gln, 1) =   -2.729; //  2HG
	icoor( 3,16,aa_gln, 1) =   1.368; //  2HG
	icoor( 1,17,aa_gln, 1) =   -1.014; //  3HG
	icoor( 2,17,aa_gln, 1) =   -2.486; //  3HG
	icoor( 3,17,aa_gln, 1) =    1.963; //  3HG

	// H positions from REDUCE
	if( param_pack::packer_logical::reduce_templates ) {
		icoor( 1 , 10 , aa_gln , 1) = 1.792 ; // H
		icoor( 2 , 10 , aa_gln , 1) = 0.471 ; // H
		icoor( 3 , 10 , aa_gln , 1) = -0.816 ; // H
		icoor( 1 , 13 , aa_gln , 1) = -0.333 ; // HA
		icoor( 2 , 13 , aa_gln , 1) = -0.503 ; // HA
		icoor( 3 , 13 , aa_gln , 1) = -0.920 ; // HA
		icoor( 1 , 14 , aa_gln , 1) = -0.054 ; // 1HB
		icoor( 2 , 14 , aa_gln , 1) = -0.369 ; // 1HB
		icoor( 3 , 14 , aa_gln , 1) = 2.122 ; // 1HB
		icoor( 1 , 15 , aa_gln , 1) = -1.614 ; // 2HB
		icoor( 2 , 15 , aa_gln , 1) = -0.564 ; // 2HB
		icoor( 3 , 15 , aa_gln , 1) = 1.312 ; // 2HB
		icoor( 1 , 16 , aa_gln , 1) = 0.759 ; // 1HG
		icoor( 2 , 16 , aa_gln , 1) = -2.478 ; // 1HG
		icoor( 3 , 16 , aa_gln , 1) = 1.235 ; // 1HG
		icoor( 1 , 17 , aa_gln , 1) = -0.805 ; // 2HG
		icoor( 2 , 17 , aa_gln , 1) = -2.738 ; // 2HG
		icoor( 3 , 17 , aa_gln , 1) = 2.018 ; // 2HG
		icoor( 1 , 11 , aa_gln , 1) = -0.308 ; // 1HE2
		icoor( 2 , 11 , aa_gln , 1) = -4.050 ; // 1HE2
		icoor( 3 , 11 , aa_gln , 1) = -1.679 ; // 1HE2
		icoor( 1 , 12 , aa_gln , 1) = 0.930 ; // 2HE2
		icoor( 2 , 12 , aa_gln , 1) = -3.745 ; // 2HE2
		icoor( 3 , 12 , aa_gln , 1) = -0.507 ; // 2HE2
	}

//bq create N and Cterm variants
	create_termini_variants( aa_gln );


//bk   create variants that have h2o connected to the backbone nh and carbonyl
	if ( explicit_h2o ) {
		create_backbone_h2o_variants(aa_gln);
		create_sidechain_h2o_variants(aa_gln);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_arg
///
/// @brief arrays filled to define the proerties of the amino acid Arginine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// aa_arg - index number for ARG (15)
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_arg()
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

//bk   Parameters for ARG -----------------------------
		nvar(aa_arg) = 1; // starting number of variants for ARG
		variant_type( aav_base, aa_arg, 1 ) = true; // type of aa variant

//bk   variant1 ---------- ARG

	natoms(aa_arg,1) = 24; // number of total atoms
	nheavyatoms(aa_arg,1) = 11; // number of heavy atoms
	nchi(aa_arg,1) =  4; // number of chi angles

//bk   atom names ARG
	atom_name( 1,aa_arg,1) = " N  ";
	atom_name( 2,aa_arg,1) = " CA ";
	atom_name( 3,aa_arg,1) = " C  ";
	atom_name( 4,aa_arg,1) = " O  ";
	atom_name( 5,aa_arg,1) = " CB ";
	atom_name( 6,aa_arg,1) = " CG ";
	atom_name( 7,aa_arg,1) = " CD ";
	atom_name( 8,aa_arg,1) = " NE ";
	atom_name( 9,aa_arg,1) = " CZ ";
	atom_name(10,aa_arg,1) = " NH1";
	atom_name(11,aa_arg,1) = " NH2";
	atom_name(12,aa_arg,1) = " H  ";
	atom_name(13,aa_arg,1) = "1HH1";
	atom_name(14,aa_arg,1) = "2HH1";
	atom_name(15,aa_arg,1) = "1HH2";
	atom_name(16,aa_arg,1) = "2HH2";
	atom_name(17,aa_arg,1) = " HE ";
	atom_name(18,aa_arg,1) = " HA ";
	atom_name(19,aa_arg,1) = "2HB ";
	atom_name(20,aa_arg,1) = "3HB ";
	atom_name(21,aa_arg,1) = "2HG ";
	atom_name(22,aa_arg,1) = "3HG ";
	atom_name(23,aa_arg,1) = "2HD ";
	atom_name(24,aa_arg,1) = "3HD ";

//jjh intra residue bonding
	nbonded_neighbors( 1, aa_arg, 1) = 2; // N
		bonded_neighbor(1, 1, aa_arg, 1) = 2; // N--CA
		bonded_neighbor(2, 1, aa_arg, 1) =12; // N--H
	nbonded_neighbors( 2, aa_arg, 1) = 4; // CA
		bonded_neighbor(1, 2, aa_arg, 1) = 1; // CA--N
		bonded_neighbor(2, 2, aa_arg, 1) = 3; // CA--C
		bonded_neighbor(3, 2, aa_arg, 1) = 5; // CA--CB
		bonded_neighbor(4, 2, aa_arg, 1) =18; // CA--HA
	nbonded_neighbors( 3, aa_arg, 1) = 2; // C
		bonded_neighbor(1, 3, aa_arg, 1) = 2; // C--CA
		bonded_neighbor(2, 3, aa_arg, 1) = 4; // C--O
	nbonded_neighbors( 4, aa_arg, 1) = 1; // O
		bonded_neighbor(1, 4, aa_arg, 1) = 3; // O--C
	nbonded_neighbors( 5, aa_arg, 1) = 4; // CB
		bonded_neighbor(1, 5, aa_arg, 1) = 2; // CB--CA
		bonded_neighbor(2, 5, aa_arg, 1) = 6; // CB--CG
		bonded_neighbor(3, 5, aa_arg, 1) =19; // CB--2HB
		bonded_neighbor(4, 5, aa_arg, 1) =20; // CB--3HB
	nbonded_neighbors( 6, aa_arg, 1) = 4; // CG
		bonded_neighbor(1, 6, aa_arg, 1) = 5; // CG--CB
		bonded_neighbor(2, 6, aa_arg, 1) = 7; // CG--CD
		bonded_neighbor(3, 6, aa_arg, 1) =21; // CG--2HG
		bonded_neighbor(4, 6, aa_arg, 1) =22; // CG--3HG
	nbonded_neighbors( 7, aa_arg, 1) = 4; // CD
		bonded_neighbor(1, 7, aa_arg, 1) = 6; // CD--CG
		bonded_neighbor(2, 7, aa_arg, 1) = 8; // CD--NE
		bonded_neighbor(3, 7, aa_arg, 1) =23; // CD--2HD
		bonded_neighbor(4, 7, aa_arg, 1) =24; // CD--3HD
	nbonded_neighbors( 8, aa_arg, 1) = 3; // NE
		bonded_neighbor(1, 8, aa_arg, 1) = 7; // NE--CD
		bonded_neighbor(2, 8, aa_arg, 1) = 9; // NE--CZ
		bonded_neighbor(3, 8, aa_arg, 1) =17; // NE--HE
	nbonded_neighbors( 9, aa_arg, 1) = 3; // CZ
		bonded_neighbor(1, 9, aa_arg, 1) = 8; // CZ--NE
		bonded_neighbor(2, 9, aa_arg, 1) =10; // CZ--NH1
		bonded_neighbor(3, 9, aa_arg, 1) =11; // CZ--NH2
	nbonded_neighbors( 10, aa_arg, 1) = 3; // NH1
		bonded_neighbor(1,10, aa_arg, 1) = 9; // NH1--CZ
		bonded_neighbor(2,10, aa_arg, 1) =13; // NH1--1HH1
		bonded_neighbor(3,10, aa_arg, 1) =14; // NH1--2HH1
	nbonded_neighbors( 11, aa_arg, 1) = 3; // NH2
		bonded_neighbor(1,11, aa_arg, 1) = 9; // NH2--CZ
		bonded_neighbor(2,11, aa_arg, 1) =15; // NH2--1HH2
		bonded_neighbor(3,11, aa_arg, 1) =16; // NH2--2HH2
	nbonded_neighbors( 12, aa_arg, 1) = 1; // H
		bonded_neighbor(1,12, aa_arg, 1) = 1; // H--N
	nbonded_neighbors( 13, aa_arg, 1) = 1; //1HH1
		bonded_neighbor(1,13, aa_arg, 1) =10; //1HH1--NH1
	nbonded_neighbors( 14, aa_arg, 1) = 1; //2HH1
		bonded_neighbor(1,14, aa_arg, 1) =10; //2HH1--NH1
	nbonded_neighbors( 15, aa_arg, 1) = 1; //1HH2
		bonded_neighbor(1,15, aa_arg, 1) =11; //1HH2--NH2
	nbonded_neighbors( 16, aa_arg, 1) = 1; //2HH2
		bonded_neighbor(1,16, aa_arg, 1) =11; //2HH2--NH2
	nbonded_neighbors( 17, aa_arg, 1) = 1; // HE
		bonded_neighbor(1,17, aa_arg, 1) = 8; // HE--NE
	nbonded_neighbors( 18, aa_arg, 1) = 1; // HA
		bonded_neighbor(1,18, aa_arg, 1) = 2; // HA--CA
	nbonded_neighbors( 19, aa_arg, 1) = 1; //2HB
		bonded_neighbor(1,19, aa_arg, 1) = 5; //2HB--CB
	nbonded_neighbors( 20, aa_arg, 1) = 1; //3HB
		bonded_neighbor(1,20, aa_arg, 1) = 5; //3HB--CB
	nbonded_neighbors( 21, aa_arg, 1) = 1; //2HG
		bonded_neighbor(1,21, aa_arg, 1) = 6; //2HG--CG
	nbonded_neighbors( 22, aa_arg, 1) = 1; //3HG
		bonded_neighbor(1,22, aa_arg, 1) = 6; //3HG--CG
	nbonded_neighbors( 23, aa_arg, 1) = 1; //2HD
		bonded_neighbor(1,23, aa_arg, 1) = 7; //2HD--CD
	nbonded_neighbors( 24, aa_arg, 1) = 1; //3HD
		bonded_neighbor(1,24, aa_arg, 1) = 7; //3HD--CD

//jjh get the angles automatically
	get_angles_from_bonds(aa_arg, 1);

//chu   base atom numbers ARG
	atom_base( 1,aa_arg,1) =  2; //  N  -> CA
	atom_base( 2,aa_arg,1) =  1; //  CA -> N
	atom_base( 3,aa_arg,1) =  2; //  C  -> CA
	atom_base( 4,aa_arg,1) =  3; //  O  -> C
	atom_base( 5,aa_arg,1) =  2; //  CB -> CA
	atom_base( 6,aa_arg,1) =  5; //  CG -> CB
	atom_base( 7,aa_arg,1) =  6; //  CD -> CG
	atom_base( 8,aa_arg,1) =  7; //  NE -> CD
	atom_base( 9,aa_arg,1) =  8; //  CZ  on  NE
	atom_base(10,aa_arg,1) =  9; //  NH1 on  CZ
	atom_base(11,aa_arg,1) =  9; //  NH2 on  CZ
	atom_base(12,aa_arg,1) =  1; //  H   on  N
	atom_base(13,aa_arg,1) = 10; // 1HH1 on  NH1
	atom_base(14,aa_arg,1) = 10; // 2HH1 on  NH1
	atom_base(15,aa_arg,1) = 11; // 1HH2 on  NH2
	atom_base(16,aa_arg,1) = 11; // 2HH2 on  NH2
	atom_base(17,aa_arg,1) =  8; //  HE  on  NE
	atom_base(18,aa_arg,1) =  2; //  HA  on  CA
	atom_base(19,aa_arg,1) =  5; // 2HB  on  CB
	atom_base(20,aa_arg,1) =  5; // 3HB  on  CB
	atom_base(21,aa_arg,1) =  6; // 2HG  on  CG
	atom_base(22,aa_arg,1) =  6; // 3HG  on  CG
	atom_base(23,aa_arg,1) =  7; // 2HD  on  CD
	atom_base(24,aa_arg,1) =  7; // 3HD  on  CD
//bk   atom type numbers ARG
	fullatom_type( 1,aa_arg,1) = 17; // Nbb    N
	fullatom_type( 2,aa_arg,1) = 18; // CAbb   CA
	fullatom_type( 3,aa_arg,1) = 19; // CObb   C
	fullatom_type( 4,aa_arg,1) = 20; // OCbb   O
	fullatom_type( 5,aa_arg,1) =  4; // CH2    CB
	fullatom_type( 6,aa_arg,1) =  4; // CH2    CG
	fullatom_type( 7,aa_arg,1) =  4; // CH2    CD
	fullatom_type( 8,aa_arg,1) = 11; // Narg   NE
	fullatom_type( 9,aa_arg,1) =  6; // aroC   CZ
	fullatom_type(10,aa_arg,1) = 11; // Narg   NH1
	fullatom_type(11,aa_arg,1) = 11; // Narg   NH2
	fullatom_type(12,aa_arg,1) = 25; // HNbb   H
	fullatom_type(13,aa_arg,1) = 22; // Hpol  1HH1
	fullatom_type(14,aa_arg,1) = 22; // Hpol  2HH1
	fullatom_type(15,aa_arg,1) = 22; // Hpol  1HH2
	fullatom_type(16,aa_arg,1) = 22; // Hpol  2HH2
	fullatom_type(17,aa_arg,1) = 22; // Hpol   HE
	fullatom_type(18,aa_arg,1) = 23; // Hapo   HA
	fullatom_type(19,aa_arg,1) = 23; // Hapo  2HB
	fullatom_type(20,aa_arg,1) = 23; // Hapo  3HB
	fullatom_type(21,aa_arg,1) = 23; // Hapo  2HG
	fullatom_type(22,aa_arg,1) = 23; // Hapo  3HG
	fullatom_type(23,aa_arg,1) = 23; // Hapo  2HD
	fullatom_type(24,aa_arg,1) = 23; // Hapo  3HD

//cmd  atomic charge ARG
	atomic_charge( 1,aa_arg,1) = -0.47; //    N
	atomic_charge( 2,aa_arg,1) =  0.07; //    CA
	atomic_charge( 3,aa_arg,1) =  0.51; //    C
	atomic_charge( 4,aa_arg,1) = -0.51; //    O
	atomic_charge( 5,aa_arg,1) = -0.18; //    CB
	atomic_charge( 6,aa_arg,1) = -0.18; //    CG
	atomic_charge( 7,aa_arg,1) =  0.20; //    CD
	atomic_charge( 8,aa_arg,1) = -0.70; //    NE
	atomic_charge( 9,aa_arg,1) =  0.64; //    CZ
	atomic_charge(10,aa_arg,1) = -0.80; //    NH1
	atomic_charge(11,aa_arg,1) = -0.80; //    NH2
	atomic_charge(12,aa_arg,1) =  0.31; //    H
	atomic_charge(13,aa_arg,1) =  0.46; //   1HH1
	atomic_charge(14,aa_arg,1) =  0.46; //   2HH1
	atomic_charge(15,aa_arg,1) =  0.46; //   1HH2
	atomic_charge(16,aa_arg,1) =  0.46; //   2HH2
	atomic_charge(17,aa_arg,1) =  0.44; //    HE
	atomic_charge(18,aa_arg,1) =  0.09; //    HA
	atomic_charge(19,aa_arg,1) =  0.09; //   2HB
	atomic_charge(20,aa_arg,1) =  0.09; //   3HB
	atomic_charge(21,aa_arg,1) =  0.09; //   2HG
	atomic_charge(22,aa_arg,1) =  0.09; //   3HG
	atomic_charge(23,aa_arg,1) =  0.09; //   2HD
	atomic_charge(24,aa_arg,1) =  0.09; //   3HD

	assert_total_charge( aa_arg, 1, 1.0);

//cmd  first sidechain atom
	first_scatom( aa_arg,1) = 5; //CB

//bk   template atoms used for placing atoms ARG

//bk   template for building 1HH1
	ta(1,13,aa_arg,1) = 10; //   NH1
	ta(2,13,aa_arg,1) =  9; //   CZ
	ta(3,13,aa_arg,1) =  8; //   NE

//bk   template for building 2HH1
	ta(1,14,aa_arg,1) = 10; //   NH1
	ta(2,14,aa_arg,1) =  9; //   CZ
	ta(3,14,aa_arg,1) =  8; //   NE

//bk   template for building 1HH2
	ta(1,15,aa_arg,1) = 11; //   NH2
	ta(2,15,aa_arg,1) =  9; //   CZ
	ta(3,15,aa_arg,1) =  8; //   NE

//bk   template for building 2HH2
	ta(1,16,aa_arg,1) = 11; //   NH2
	ta(2,16,aa_arg,1) =  9; //   CZ
	ta(3,16,aa_arg,1) =  8; //   NE

//bk   template for building  HE
	ta(1,17,aa_arg,1) =  8; //   NE
	ta(2,17,aa_arg,1) =  7; //   CD
	ta(3,17,aa_arg,1) =  9; //   CZ

//bk   template for building  HA
	ta(1,18,aa_arg,1) =  2; //   CA
	ta(2,18,aa_arg,1) =  1; //   N
	ta(3,18,aa_arg,1) =  3; //   C

//bk   template for building 2HB
	ta(1,19,aa_arg,1) =  5; //   CB
	ta(2,19,aa_arg,1) =  2; //   CA
	ta(3,19,aa_arg,1) =  6; //   CG

//bk   template for building 3HB
	ta(1,20,aa_arg,1) =  5; //   CB
	ta(2,20,aa_arg,1) =  2; //   CA
	ta(3,20,aa_arg,1) =  6; //   CG

//bk   template for building 2HG
	ta(1,21,aa_arg,1) =  6; //   CG
	ta(2,21,aa_arg,1) =  5; //   CB
	ta(3,21,aa_arg,1) =  7; //   CD

//bk   template for building 3HG
	ta(1,22,aa_arg,1) =  6; //   CG
	ta(2,22,aa_arg,1) =  5; //   CB
	ta(3,22,aa_arg,1) =  7; //   CD

//bk   template for building 2HD
	ta(1,23,aa_arg,1) =  7; //   CD
	ta(2,23,aa_arg,1) =  6; //   CG
	ta(3,23,aa_arg,1) =  8; //   NE

//bk   template for building 3HD
	ta(1,24,aa_arg,1) =  7; //   CD
	ta(2,24,aa_arg,1) =  6; //   CG
	ta(3,24,aa_arg,1) =  8; //   NE

//bk   chi angles required to build atoms ARG

//bk   chi angles needed for building  N
	chi_required(1, 1,aa_arg,1) = false;
	chi_required(2, 1,aa_arg,1) = false;
	chi_required(3, 1,aa_arg,1) = false;
	chi_required(4, 1,aa_arg,1) = false;

//bk   chi angles needed for building  CA
	chi_required(1, 2,aa_arg,1) = false;
	chi_required(2, 2,aa_arg,1) = false;
	chi_required(3, 2,aa_arg,1) = false;
	chi_required(4, 2,aa_arg,1) = false;

//bk   chi angles needed for building  C
	chi_required(1, 3,aa_arg,1) = false;
	chi_required(2, 3,aa_arg,1) = false;
	chi_required(3, 3,aa_arg,1) = false;
	chi_required(4, 3,aa_arg,1) = false;

//bk   chi angles needed for building  O
	chi_required(1, 4,aa_arg,1) = false;
	chi_required(2, 4,aa_arg,1) = false;
	chi_required(3, 4,aa_arg,1) = false;
	chi_required(4, 4,aa_arg,1) = false;

//bk   chi angles needed for building  CB
	chi_required(1, 5,aa_arg,1) = false;
	chi_required(2, 5,aa_arg,1) = false;
	chi_required(3, 5,aa_arg,1) = false;
	chi_required(4, 5,aa_arg,1) = false;

//bk   chi angles needed for building  CG
	chi_required(1, 6,aa_arg,1) =  true;
	chi_required(2, 6,aa_arg,1) = false;
	chi_required(3, 6,aa_arg,1) = false;
	chi_required(4, 6,aa_arg,1) = false;

//bk   chi angles needed for building  CD
	chi_required(1, 7,aa_arg,1) =  true;
	chi_required(2, 7,aa_arg,1) =  true;
	chi_required(3, 7,aa_arg,1) = false;
	chi_required(4, 7,aa_arg,1) = false;

//bk   chi angles needed for building  NE
	chi_required(1, 8,aa_arg,1) =  true;
	chi_required(2, 8,aa_arg,1) =  true;
	chi_required(3, 8,aa_arg,1) =  true;
	chi_required(4, 8,aa_arg,1) = false;

//bk   chi angles needed for building  CZ
	chi_required(1, 9,aa_arg,1) =  true;
	chi_required(2, 9,aa_arg,1) =  true;
	chi_required(3, 9,aa_arg,1) =  true;
	chi_required(4, 9,aa_arg,1) =  true;

//bk   chi angles needed for building  NH1
	chi_required(1,10,aa_arg,1) =  true;
	chi_required(2,10,aa_arg,1) =  true;
	chi_required(3,10,aa_arg,1) =  true;
	chi_required(4,10,aa_arg,1) =  true;

//bk   chi angles needed for building  NH2
	chi_required(1,11,aa_arg,1) =  true;
	chi_required(2,11,aa_arg,1) =  true;
	chi_required(3,11,aa_arg,1) =  true;
	chi_required(4,11,aa_arg,1) =  true;

//bk   chi angles needed for building  H
	chi_required(1,12,aa_arg,1) = false;
	chi_required(2,12,aa_arg,1) = false;
	chi_required(3,12,aa_arg,1) = false;
	chi_required(4,12,aa_arg,1) = false;

//bk   chi angles needed for building 1HH1
	chi_required(1,13,aa_arg,1) =  true;
	chi_required(2,13,aa_arg,1) =  true;
	chi_required(3,13,aa_arg,1) =  true;
	chi_required(4,13,aa_arg,1) =  true;

//bk   chi angles needed for building 2HH1
	chi_required(1,14,aa_arg,1) =  true;
	chi_required(2,14,aa_arg,1) =  true;
	chi_required(3,14,aa_arg,1) =  true;
	chi_required(4,14,aa_arg,1) =  true;

//bk   chi angles needed for building 1HH2
	chi_required(1,15,aa_arg,1) =  true;
	chi_required(2,15,aa_arg,1) =  true;
	chi_required(3,15,aa_arg,1) =  true;
	chi_required(4,15,aa_arg,1) =  true;

//bk   chi angles needed for building 2HH2
	chi_required(1,16,aa_arg,1) =  true;
	chi_required(2,16,aa_arg,1) =  true;
	chi_required(3,16,aa_arg,1) =  true;
	chi_required(4,16,aa_arg,1) =  true;

//bk   chi angles needed for building  HE
	chi_required(1,17,aa_arg,1) =  true;
	chi_required(2,17,aa_arg,1) =  true;
	chi_required(3,17,aa_arg,1) =  true;
	chi_required(4,17,aa_arg,1) =  true;

//bk   chi angles needed for building  HA
	chi_required(1,18,aa_arg,1) = false;
	chi_required(2,18,aa_arg,1) = false;
	chi_required(3,18,aa_arg,1) = false;
	chi_required(4,18,aa_arg,1) = false;

//bk   chi angles needed for building 2HB
	chi_required(1,19,aa_arg,1) =  true;
	chi_required(2,19,aa_arg,1) = false;
	chi_required(3,19,aa_arg,1) = false;
	chi_required(4,19,aa_arg,1) = false;

//bk   chi angles needed for building 3HB
	chi_required(1,20,aa_arg,1) =  true;
	chi_required(2,20,aa_arg,1) = false;
	chi_required(3,20,aa_arg,1) = false;
	chi_required(4,20,aa_arg,1) = false;

//bk   chi angles needed for building 2HG
	chi_required(1,21,aa_arg,1) =  true;
	chi_required(2,21,aa_arg,1) =  true;
	chi_required(3,21,aa_arg,1) = false;
	chi_required(4,21,aa_arg,1) = false;

//bk   chi angles needed for building 3HG
	chi_required(1,22,aa_arg,1) =  true;
	chi_required(2,22,aa_arg,1) =  true;
	chi_required(3,22,aa_arg,1) = false;
	chi_required(4,22,aa_arg,1) = false;

//bk   chi angles needed for building 2HD
	chi_required(1,23,aa_arg,1) =  true;
	chi_required(2,23,aa_arg,1) =  true;
	chi_required(3,23,aa_arg,1) =  true;
	chi_required(4,23,aa_arg,1) = false;

//bk   chi angles needed for building 3HD
	chi_required(1,24,aa_arg,1) =  true;
	chi_required(2,24,aa_arg,1) =  true;
	chi_required(3,24,aa_arg,1) =  true;
	chi_required(4,24,aa_arg,1) = false;

//bk   four atoms that define chi angle  1
	chi_atoms( 1, 1,aa_arg, 1) =  1; //   N
	chi_atoms( 2, 1,aa_arg, 1) =  2; //   CA
	chi_atoms( 3, 1,aa_arg, 1) =  5; //   CB
	chi_atoms( 4, 1,aa_arg, 1) =  6; //   CG
//bk   four atoms that define chi angle  2
	chi_atoms( 1, 2,aa_arg, 1) =  2; //   CA
	chi_atoms( 2, 2,aa_arg, 1) =  5; //   CB
	chi_atoms( 3, 2,aa_arg, 1) =  6; //   CG
	chi_atoms( 4, 2,aa_arg, 1) =  7; //   CD
//bk   four atoms that define chi angle  3
	chi_atoms( 1, 3,aa_arg, 1) =  5; //   CB
	chi_atoms( 2, 3,aa_arg, 1) =  6; //   CG
	chi_atoms( 3, 3,aa_arg, 1) =  7; //   CD
	chi_atoms( 4, 3,aa_arg, 1) =  8; //   NE
//bk   four atoms that define chi angle  4
	chi_atoms( 1, 4,aa_arg, 1) =  6; //   CG
	chi_atoms( 2, 4,aa_arg, 1) =  7; //   CD
	chi_atoms( 3, 4,aa_arg, 1) =  8; //   NE
	chi_atoms( 4, 4,aa_arg, 1) =  9; //   CZ

	HNpos(aa_arg,1) = 12; // atom number for backbone HN
	HApos(aa_arg,1) = 18; // atom number for backbone HA
	nH_polar(aa_arg,1) =  6; // number of polar hydrogens
	nH_aromatic(aa_arg,1) =  0; // number of aromatic hydrogens
	nH_apolar(aa_arg,1) =  7; // number of apolar hydrogens
	nacceptors(aa_arg,1) =  1; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1,aa_arg,1) = 12; //  H
	Hpos_polar( 2,aa_arg,1) = 13; // 1HH1
	Hpos_polar( 3,aa_arg,1) = 14; // 2HH1
	Hpos_polar( 4,aa_arg,1) = 15; // 1HH2
	Hpos_polar( 5,aa_arg,1) = 16; // 2HH2
	Hpos_polar( 6,aa_arg,1) = 17; //  HE

//bk   atom number for aromatic hydrogens

//bk   atom number for apolar hydrogens
	Hpos_apolar( 1,aa_arg,1) = 18; //  HA
	Hpos_apolar( 2,aa_arg,1) = 19; // 2HB
	Hpos_apolar( 3,aa_arg,1) = 20; // 3HB
	Hpos_apolar( 4,aa_arg,1) = 21; // 2HG
	Hpos_apolar( 5,aa_arg,1) = 22; // 3HG
	Hpos_apolar( 6,aa_arg,1) = 23; // 2HD
	Hpos_apolar( 7,aa_arg,1) = 24; // 3HD

//bk   acceptor positions
	accpt_pos( 1,aa_arg, 1) =  4; //  O

//bk   hydrogens on each atom

	nhydrogens_on_atm( 1,aa_arg, 1) =  1; //  N
	hydrogens_on_atm( 1, 1,aa_arg, 1) = 12; //  H

	nhydrogens_on_atm( 2,aa_arg, 1) =  1; //  CA
	hydrogens_on_atm( 1, 2,aa_arg, 1) = 18; //  HA

	nhydrogens_on_atm( 5,aa_arg, 1) =  2; //  CB
	hydrogens_on_atm( 1, 5,aa_arg, 1) = 19; // 2HB
	hydrogens_on_atm( 2, 5,aa_arg, 1) = 20; // 3HB

	nhydrogens_on_atm( 6,aa_arg, 1) =  2; //  CG
	hydrogens_on_atm( 1, 6,aa_arg, 1) = 21; // 2HG
	hydrogens_on_atm( 2, 6,aa_arg, 1) = 22; // 3HG

	nhydrogens_on_atm( 7,aa_arg, 1) =  2; //  CD
	hydrogens_on_atm( 1, 7,aa_arg, 1) = 23; // 2HD
	hydrogens_on_atm( 2, 7,aa_arg, 1) = 24; // 3HD

	nhydrogens_on_atm( 8,aa_arg, 1) =  1; //  NE
	hydrogens_on_atm( 1, 8,aa_arg, 1) = 17; //  HE

	nhydrogens_on_atm(10,aa_arg, 1) =  2; //  NH1
	hydrogens_on_atm( 1,10,aa_arg, 1) = 13; // 1HH1
	hydrogens_on_atm( 2,10,aa_arg, 1) = 14; // 2HH1

	nhydrogens_on_atm(11,aa_arg, 1) =  2; //  NH2
	hydrogens_on_atm( 1,11,aa_arg, 1) = 15; // 1HH2
	hydrogens_on_atm( 2,11,aa_arg, 1) = 16; // 2HH2

//bk   template coordinates for the amino acid
//jjh new template from Kevin Karplus
	icoor( 1, 1,aa_arg, 1) =   1.455; //   N
	icoor( 2, 1,aa_arg, 1) =   0.000; //   N
	icoor( 3, 1,aa_arg, 1) =   0.000; //   N
	icoor( 1, 2,aa_arg, 1) =   0.000; //   CA
	icoor( 2, 2,aa_arg, 1) =   0.000; //   CA
	icoor( 3, 2,aa_arg, 1) =   0.000; //   CA
	icoor( 1, 3,aa_arg, 1) =   -0.569; //   C
	icoor( 2, 3,aa_arg, 1) =   1.399; //   C
	icoor( 3, 3,aa_arg, 1) =   -0.000; //   C
	icoor( 1, 4,aa_arg, 1) =   -1.568; //   O
	icoor( 2, 4,aa_arg, 1) =   1.674; //   O
	icoor( 3, 4,aa_arg, 1) =   -0.695; //   O
	icoor( 1, 5,aa_arg, 1) =   -0.574; //   CB
	icoor( 2, 5,aa_arg, 1) =   -0.820; //   CB
	icoor( 3, 5,aa_arg, 1) =   1.146; //   CB
	icoor( 1, 6,aa_arg, 1) =   -0.335; //   CG
	icoor( 2, 6,aa_arg, 1) =   -2.318; //   CG
	icoor( 3, 6,aa_arg, 1) =   1.043; //   CG
	icoor( 1, 7,aa_arg, 1) =   -1.198; //   CD
	icoor( 2, 7,aa_arg, 1) =   -3.009; //   CD
	icoor( 3, 7,aa_arg, 1) =   0.051; //   CD
	icoor( 1, 8,aa_arg, 1) =   -2.624; //   NE
	icoor( 2, 8,aa_arg, 1) =   -2.901; //   NE
	icoor( 3, 8,aa_arg, 1) =   0.314; //   NE
	icoor( 1, 9,aa_arg, 1) =   -3.279; //   CZ
	icoor( 2, 9,aa_arg, 1) =   -3.584; //   CZ
	icoor( 3, 9,aa_arg, 1) =   1.273; //   CZ
	icoor( 1,10,aa_arg, 1) =   -2.651; //   NH1
	icoor( 2,10,aa_arg, 1) =   -4.451; //   NH1
	icoor( 3,10,aa_arg, 1) =   2.036; //   NH1
	icoor( 1,11,aa_arg, 1) =   -4.577; //   NH2
	icoor( 2,11,aa_arg, 1) =   -3.377; //   NH2
	icoor( 3,11,aa_arg, 1) =   1.411; //   NH2
	icoor( 1,12,aa_arg, 1) =   2.418; //   H
	icoor( 2,12,aa_arg, 1) =   -0.306; //   H
	icoor( 3,12,aa_arg, 1) =   0.000; //   H
	icoor( 1,13,aa_arg, 1) =   -1.662; //  1HH1
	icoor( 2,13,aa_arg, 1) =   -4.611; //  1HH1
	icoor( 3,13,aa_arg, 1) =   1.906; //  1HH1
	icoor( 1,14,aa_arg, 1) =   -3.160; //  2HH1
	icoor( 2,14,aa_arg, 1) =   -4.953; //  2HH1
	icoor( 3,14,aa_arg, 1) =   2.749; //  2HH1
	icoor( 1,15,aa_arg, 1) =   -5.048; //  1HH2
	icoor( 2,15,aa_arg, 1) =    -2.720; //  1HH2
	icoor( 3,15,aa_arg, 1) =   0.804; //  1HH2
	icoor( 1,16,aa_arg, 1) =   -5.092; //  2HH2
	icoor( 2,16,aa_arg, 1) =   -3.876; //  2HH2
	icoor( 3,16,aa_arg, 1) =   2.121; //  2HH2
	icoor( 1,17,aa_arg, 1) =   -3.323; //   HE
	icoor( 2,17,aa_arg, 1) =   -2.333; //   HE
	icoor( 3,17,aa_arg, 1) =   -0.146; //   HE
	icoor( 1,18,aa_arg, 1) =   -0.369; //   HA
	icoor( 2,18,aa_arg, 1) =   -0.476; //   HA
	icoor( 3,18,aa_arg, 1) =   -0.909; //   HA
	icoor( 1,19,aa_arg, 1) =   -0.121; //  2HB
	icoor( 2,19,aa_arg, 1) =   -0.446; //  2HB
	icoor( 3,19,aa_arg, 1) =   2.063; //  2HB
	icoor( 1,20,aa_arg, 1) =   -1.647; //  3HB
	icoor( 2,20,aa_arg, 1) =   -0.629; //  3HB
	icoor( 3,20,aa_arg, 1) =   1.167; //  3HB
	icoor( 1,21,aa_arg, 1) =   0.704; //  2HG
	icoor( 2,21,aa_arg, 1) =   -2.485; //  2HG
	icoor( 3,21,aa_arg, 1) =   0.758; //  2HG
	icoor( 1,22,aa_arg, 1) =   -0.520; //  3HG
	icoor( 2,22,aa_arg, 1) =   -2.766; //  3HG
	icoor( 3,22,aa_arg, 1) =    2.020; //  3HG
	icoor( 1,23,aa_arg, 1) =   -1.015; //  2HD
	icoor( 2,23,aa_arg, 1) =   -2.583; //  2HD
	icoor( 3,23,aa_arg, 1) =   -0.935; //  2HD
	icoor( 1,24,aa_arg, 1) =   -0.947; //  3HD
	icoor( 2,24,aa_arg, 1) =   -4.069; //  3HD
	icoor( 3,24,aa_arg, 1) =   0.041; //  3HD

	// H positions from REDUCE
	if( param_pack::packer_logical::reduce_templates ) {
		icoor( 1 , 12 , aa_arg , 1) = 1.789 ; // H
		icoor( 2 , 12 , aa_arg , 1) = 0.667 ; // H
		icoor( 3 , 12 , aa_arg , 1) = -0.666 ; // H
		icoor( 1 , 18 , aa_arg , 1) = -0.307 ; // HA
		icoor( 2 , 18 , aa_arg , 1) = -0.483 ; // HA
		icoor( 3 , 18 , aa_arg , 1) = -0.939 ; // HA
		icoor( 1 , 19 , aa_arg , 1) = -0.140 ; // 1HB
		icoor( 2 , 19 , aa_arg , 1) = -0.458 ; // 1HB
		icoor( 3 , 19 , aa_arg , 1) = 2.090 ; // 1HB
		icoor( 1 , 20 , aa_arg , 1) = -1.658 ; // 2HB
		icoor( 2 , 20 , aa_arg , 1) = -0.639 ; // 2HB
		icoor( 3 , 20 , aa_arg , 1) = 1.200 ; // 2HB
		icoor( 1 , 21, aa_arg , 1) = 0.719 ; // 1HG
		icoor( 2 , 21, aa_arg , 1) = -2.492 ; // 1HG
		icoor( 3 , 21, aa_arg , 1) = 0.779 ; // 1HG
		icoor( 1 , 22 , aa_arg , 1) = -0.498 ; // 2HG
		icoor( 2 , 22 , aa_arg , 1) = -2.772 ; // 2HG
		icoor( 3 , 22 , aa_arg , 1) = 2.032 ; // 2HG
		icoor( 1 , 23, aa_arg , 1) = -0.990 ; // 1HD
		icoor( 2 , 23, aa_arg , 1) = -2.597 ; // 1HD
		icoor( 3 , 23, aa_arg , 1) = -0.947 ; // 1HD
		icoor( 1 , 24 , aa_arg , 1) = -0.923 ; // 2HD
		icoor( 2 , 24 , aa_arg , 1) = -4.074 ; // 2HD
		icoor( 3 , 24 , aa_arg , 1) = 0.023 ; // 2HD
		icoor( 1 , 17 , aa_arg , 1) = -3.156 ; // HE
		icoor( 2 , 17 , aa_arg , 1) = -2.276 ; // HE
		icoor( 3 , 17 , aa_arg , 1) = -0.257 ; // HE
		icoor( 1 , 13 , aa_arg , 1) = -1.672 ; // 1HH1
		icoor( 2 , 13 , aa_arg , 1) = -4.612 ; // 1HH1
		icoor( 3 , 13 , aa_arg , 1) = 1.909 ; // 1HH1
		icoor( 1 , 14 , aa_arg , 1) = -3.152 ; // 2HH1
		icoor( 2 , 14 , aa_arg , 1) = -4.950 ; // 2HH1
		icoor( 3 , 14 , aa_arg , 1) = 2.743 ; // 2HH1
		icoor( 1 , 15 , aa_arg , 1) = -5.046 ; // 1HH2
		icoor( 2 , 15 , aa_arg , 1) = -2.728 ; // 1HH2
		icoor( 3 , 15 , aa_arg , 1) = 0.812 ; // 1HH2
		icoor( 1 , 16 , aa_arg , 1) = -5.090 ; // 2HH2
		icoor( 2 , 16 , aa_arg , 1) = -3.870 ; // 2HH2
		icoor( 3 , 16 , aa_arg , 1) = 2.114 ; // 2HH2
	}

	//rh create protonation state variants
	create_pH_variants( aa_arg);

//bq create N and Cterm variants
	create_termini_variants( aa_arg );

//bk   create variants that have h2o connected to the backbone nh and carbonyl
	if ( explicit_h2o ) {
		create_backbone_h2o_variants(aa_arg);
		create_sidechain_h2o_variants(aa_arg);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_ser
///
/// @brief arrays filled to define the proerties of the amino acid Serine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// aa_ser - index number for SER (16)
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_ser()
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

//bk   Parameters for SER -----------------------------
	nvar(aa_ser) = 1; // starting number of variants for SER
	variant_type( aav_base, aa_ser, 1 ) = true; // type of aa variant

//bk   variant1 ---------- SER

	natoms(aa_ser,1) = 11; // number of total atoms
	nheavyatoms(aa_ser,1) =  6; // number of heavy atoms
	nchi(aa_ser,1) =  2; // number of chi angles

//bk   atom names SER
	atom_name( 1,aa_ser,1) = " N  ";
	atom_name( 2,aa_ser,1) = " CA ";
	atom_name( 3,aa_ser,1) = " C  ";
	atom_name( 4,aa_ser,1) = " O  ";
	atom_name( 5,aa_ser,1) = " CB ";
	atom_name( 6,aa_ser,1) = " OG ";
	atom_name( 7,aa_ser,1) = " H  ";
	atom_name( 8,aa_ser,1) = " HG ";
	atom_name( 9,aa_ser,1) = " HA ";
	atom_name(10,aa_ser,1) = "2HB ";
	atom_name(11,aa_ser,1) = "3HB ";

//jjh intra residue bonding
	nbonded_neighbors( 1, aa_ser, 1) = 2; // N
		bonded_neighbor(1, 1, aa_ser, 1) = 2; // N--CA
		bonded_neighbor(2, 1, aa_ser, 1) = 7; // N--H
	nbonded_neighbors( 2, aa_ser, 1) = 4; // CA
		bonded_neighbor(1, 2, aa_ser, 1) = 1; // CA--N
		bonded_neighbor(2, 2, aa_ser, 1) = 3; // CA--C
		bonded_neighbor(3, 2, aa_ser, 1) = 5; // CA--CB
		bonded_neighbor(4, 2, aa_ser, 1) = 9; // CA--HA
	nbonded_neighbors( 3, aa_ser, 1) = 2; // C
		bonded_neighbor(1, 3, aa_ser, 1) = 2; // C--CA
		bonded_neighbor(2, 3, aa_ser, 1) = 4; // C--O
	nbonded_neighbors( 4, aa_ser, 1) = 1; // O
		bonded_neighbor(1, 4, aa_ser, 1) = 3; // O--C
	nbonded_neighbors( 5, aa_ser, 1) = 4; // CB
		bonded_neighbor(1, 5, aa_ser, 1) = 2; // CB--CA
		bonded_neighbor(2, 5, aa_ser, 1) = 6; // CB--OG
		bonded_neighbor(3, 5, aa_ser, 1) =10; // CB--2HB
		bonded_neighbor(4, 5, aa_ser, 1) =11; // CB--3HB
	nbonded_neighbors( 6, aa_ser, 1) = 2; // OG
		bonded_neighbor(1, 6, aa_ser, 1) = 5; // OG--CB
		bonded_neighbor(2, 6, aa_ser, 1) = 8; // OG--HG
	nbonded_neighbors( 7, aa_ser, 1) = 1; // H
		bonded_neighbor(1, 7, aa_ser, 1) = 1; // H--N
	nbonded_neighbors( 8, aa_ser, 1) = 1; // HG
		bonded_neighbor(1, 8, aa_ser, 1) = 6; // HG--OG
	nbonded_neighbors( 9, aa_ser, 1) = 1; // HA
		bonded_neighbor(1, 9, aa_ser, 1) = 2; // HA--CA
	nbonded_neighbors( 10, aa_ser, 1) = 1; //2HB
		bonded_neighbor(1,10, aa_ser, 1) = 5; //2HB--CB
	nbonded_neighbors( 11, aa_ser, 1) = 1; //3HB
		bonded_neighbor(1,11, aa_ser, 1) = 5; //3HB--CB

//jjh get the angles automatically
	get_angles_from_bonds(aa_ser, 1);

//chu   base atom numbers SER
	atom_base( 1,aa_ser,1) =  2; //  N  -> CA
	atom_base( 2,aa_ser,1) =  1; //  CA -> N
	atom_base( 3,aa_ser,1) =  2; //  C  -> CA
	atom_base( 4,aa_ser,1) =  3; //  O  -> C
	atom_base( 5,aa_ser,1) =  2; //  CB -> CA
	atom_base( 6,aa_ser,1) =  5; //  OG  on  CB
	atom_base( 7,aa_ser,1) =  1; //  H   on  N
	atom_base( 8,aa_ser,1) =  6; //  HG  on  OG
	atom_base( 9,aa_ser,1) =  2; //  HA  on  CA
	atom_base(10,aa_ser,1) =  5; // 2HB  on  CB
	atom_base(11,aa_ser,1) =  5; // 3HB  on  CB
//bk   atom type numbers SER
	fullatom_type( 1,aa_ser,1) = 17; // Nbb    N
	fullatom_type( 2,aa_ser,1) = 18; // CAbb   CA
	fullatom_type( 3,aa_ser,1) = 19; // CObb   C
	fullatom_type( 4,aa_ser,1) = 20; // OCbb   O
	fullatom_type( 5,aa_ser,1) =  4; // CH2    CB
	fullatom_type( 6,aa_ser,1) = 13; // OH     OG
	fullatom_type( 7,aa_ser,1) = 25; // HNbb   H
	fullatom_type( 8,aa_ser,1) = 22; // Hpol   HG
	fullatom_type( 9,aa_ser,1) = 23; // Hapo   HA
	fullatom_type(10,aa_ser,1) = 23; // Hapo  2HB
	fullatom_type(11,aa_ser,1) = 23; // Hapo  3HB

//cmd  atomic charge SER
	atomic_charge( 1,aa_ser,1) = -0.47; //    N
	atomic_charge( 2,aa_ser,1) =  0.07; //    CA
	atomic_charge( 3,aa_ser,1) =  0.51; //    C
	atomic_charge( 4,aa_ser,1) = -0.51; //    O
	atomic_charge( 5,aa_ser,1) =  0.05; //    CB
	atomic_charge( 6,aa_ser,1) = -0.66; //    OG
	atomic_charge( 7,aa_ser,1) =  0.31; //    H
	atomic_charge( 8,aa_ser,1) =  0.43; //    HG
	atomic_charge( 9,aa_ser,1) =  0.09; //    HA
	atomic_charge(10,aa_ser,1) =  0.09; //   2HB
	atomic_charge(11,aa_ser,1) =  0.09; //   3HB

	assert_total_charge( aa_ser, 1, 0.0);

//cmd  first sidechain atom
	first_scatom( aa_ser,1) = 5; //CB

//bk   template atoms used for placing atoms SER

//bk   template for building  HG
	ta(1, 8,aa_ser,1) =  6; //   OG
	ta(2, 8,aa_ser,1) =  5; //   CB
	ta(3, 8,aa_ser,1) =  2; //   CA

//bk   template for building  HA
	ta(1, 9,aa_ser,1) =  2; //   CA
	ta(2, 9,aa_ser,1) =  1; //   N
	ta(3, 9,aa_ser,1) =  3; //   C

//bk   template for building 2HB
	ta(1,10,aa_ser,1) =  5; //   CB
	ta(2,10,aa_ser,1) =  2; //   CA
	ta(3,10,aa_ser,1) =  6; //   OG

//bk   template for building 3HB
	ta(1,11,aa_ser,1) =  5; //   CB
	ta(2,11,aa_ser,1) =  2; //   CA
	ta(3,11,aa_ser,1) =  6; //   OG

//bk   chi angles required to build atoms SER

//bk   chi angles needed for building  OG
	chi_required(1, 6,aa_ser,1) =  true;

//bk   chi angles needed for building  HG
	chi_required(1, 8,aa_ser,1) =  true;
	chi_required(2, 8,aa_ser,1) =  true;

//bk   chi angles needed for building 2HB
	chi_required(1,10,aa_ser,1) =  true;

//bk   chi angles needed for building 3HB
	chi_required(1,11,aa_ser,1) =  true;

//bk   four atoms that define chi angle  1
	chi_atoms( 1, 1,aa_ser, 1) =  1; //   N
	chi_atoms( 2, 1,aa_ser, 1) =  2; //   CA
	chi_atoms( 3, 1,aa_ser, 1) =  5; //   CB
	chi_atoms( 4, 1,aa_ser, 1) =  6; //   OG
//bk   four atoms that define chi angle  2
	chi_atoms( 1, 2,aa_ser, 1) =  2; //   CA
	chi_atoms( 2, 2,aa_ser, 1) =  5; //   CB
	chi_atoms( 3, 2,aa_ser, 1) =  6; //   OG
	chi_atoms( 4, 2,aa_ser, 1) =  8; //   HG

	HNpos(aa_ser,1) =  7; // atom number for backbone HN
	HApos(aa_ser,1) =  9; // atom number for backbone HA
	nH_polar(aa_ser,1) =  2; // number of polar hydrogens
	nH_aromatic(aa_ser,1) =  0; // number of aromatic hydrogens
	nH_apolar(aa_ser,1) =  3; // number of apolar hydrogens
	nacceptors(aa_ser,1) =  2; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1,aa_ser,1) =  7; //  H
	Hpos_polar( 2,aa_ser,1) =  8; //  HG

//bk   atom number for aromatic hydrogens

//bk   atom number for apolar hydrogens
	Hpos_apolar( 1,aa_ser,1) =  9; //  HA
	Hpos_apolar( 2,aa_ser,1) = 10; // 2HB
	Hpos_apolar( 3,aa_ser,1) = 11; // 3HB

//bk   acceptor positions
	accpt_pos( 1,aa_ser, 1) =  4; //  O
	accpt_pos( 2,aa_ser, 1) =  6; //  OG

//bk   acceptor base positions
	abase2( 6,aa_ser, 1) =  8; //  HG  base2 for  OG

//bk   hydrogens on each atom

	nhydrogens_on_atm( 1,aa_ser, 1) =  1; //  N
	hydrogens_on_atm( 1, 1,aa_ser, 1) =  7; //  H

	nhydrogens_on_atm( 2,aa_ser, 1) =  1; //  CA
	hydrogens_on_atm( 1, 2,aa_ser, 1) =  9; //  HA

	nhydrogens_on_atm( 5,aa_ser, 1) =  2; //  CB
	hydrogens_on_atm( 1, 5,aa_ser, 1) = 10; // 2HB
	hydrogens_on_atm( 2, 5,aa_ser, 1) = 11; // 3HB

	nhydrogens_on_atm( 6,aa_ser, 1) =  1; //  OG
	hydrogens_on_atm( 1, 6,aa_ser, 1) =  8; //  HG

//bk   template coordinates for the amino acid
//jjh new template from Kevin Karplus
	icoor( 1, 1,aa_ser, 1) =   1.470; //   N
	icoor( 2, 1,aa_ser, 1) =   0.000; //   N
	icoor( 3, 1,aa_ser, 1) =   0.000; //   N
	icoor( 1, 2,aa_ser, 1) =   0.000; //   CA
	icoor( 2, 2,aa_ser, 1) =   0.000; //   CA
	icoor( 3, 2,aa_ser, 1) =   0.000; //   CA
	icoor( 1, 3,aa_ser, 1) =   -0.572; //   C
	icoor( 2, 3,aa_ser, 1) =   1.422; //   C
	icoor( 3, 3,aa_ser, 1) =   0.000; //   C
	icoor( 1, 4,aa_ser, 1) =   -1.553; //   O
	icoor( 2, 4,aa_ser, 1) =   1.698; //   O
	icoor( 3, 4,aa_ser, 1) =   -0.675; //   O
	icoor( 1, 5,aa_ser, 1) =   -0.522; //   CB
	icoor( 2, 5,aa_ser, 1) =   -0.769; //   CB
	icoor( 3, 5,aa_ser, 1) =   1.198; //   CB
	icoor( 1, 6,aa_ser, 1) =   -0.231; //   OG
	icoor( 2, 6,aa_ser, 1) =   -2.137; //   OG
	icoor( 3, 6,aa_ser, 1) =   1.113; //   OG
	icoor( 1, 7,aa_ser, 1) =   2.435; //   H
	icoor( 2, 7,aa_ser, 1) =   -0.299; //   H
	icoor( 3, 7,aa_ser, 1) =   0.000; //   H
	icoor( 1, 8,aa_ser, 1) =   -0.577; //   HG
	icoor( 2, 8,aa_ser, 1) =   -2.586; //   HG
	icoor( 3, 8,aa_ser, 1) =   1.888; //   HG
	icoor( 1, 9,aa_ser, 1) =   -0.433; //   HA
	icoor( 2, 9,aa_ser, 1) =   -0.566; //   HA
	icoor( 3, 9,aa_ser, 1) =   -0.826; //   HA
	icoor( 1,10,aa_ser, 1) =   -0.062; //  2HB
	icoor( 2,10,aa_ser, 1) =   -0.364; //  2HB
	icoor( 3,10,aa_ser, 1) =   2.099; //  2HB
	icoor( 1,11,aa_ser, 1) =   -1.602; //  3HB
	icoor( 2,11,aa_ser, 1) =   -0.639; //  3HB
	icoor( 3,11,aa_ser, 1) =   1.253; //  3HB

	// H positions from REDUCE
	if( param_pack::packer_logical::reduce_templates ) {
		icoor( 1 , 7 , aa_ser , 1) = 1.804 ; // H
		icoor( 2 , 7 , aa_ser , 1) = 0.471 ; // H
		icoor( 3 , 7 , aa_ser , 1) = -0.816 ; // H
		icoor( 1 , 9 , aa_ser , 1) = -0.331 ; // HA
		icoor( 2 , 9 , aa_ser , 1) = -0.493 ; // HA
		icoor( 3 , 9 , aa_ser , 1) = -0.926 ; // HA
		icoor( 1 , 10 , aa_ser , 1) = -1.611 ; // 1HB
		icoor( 2 , 10 , aa_ser , 1) = -0.631 ; // 1HB
		icoor( 3 , 10 , aa_ser , 1) = 1.275 ; // 1HB
		icoor( 1 , 11 , aa_ser , 1) = -0.080 ; // 2HB
		icoor( 2 , 11 , aa_ser , 1) = -0.358 ; // 2HB
		icoor( 3 , 11 , aa_ser , 1) = 2.117 ; // 2HB
		icoor( 1 , 8 , aa_ser , 1) = -0.590 ; // HG
		icoor( 2 , 8 , aa_ser , 1) = -2.605 ; // HG
		icoor( 3 , 8 , aa_ser , 1) = 1.920 ; // HG
	}

//bq create N and Cterm variants
	create_termini_variants( aa_ser );


//bk   create variants that have h2o connected to the backbone nh and carbonyl
	if ( explicit_h2o ) {
		create_backbone_h2o_variants(aa_ser);
		create_sidechain_h2o_variants(aa_ser);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_thr
///
/// @brief arrays filled to define the proerties of the amino acid Threonine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// aa_thr - index number for THR (17)
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_thr()
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

//bk   Parameters for THR -----------------------------
	nvar(aa_thr) = 1; // starting number of variants for THR
	variant_type( aav_base, aa_thr, 1 ) = true; // type of aa variant

//bk   variant1 ---------- THR

	natoms(aa_thr,1) = 14; // number of total atoms
	nheavyatoms(aa_thr,1) =  7; // number of heavy atoms
	nchi(aa_thr,1) =  2; // number of chi angles

//bk   atom names THR
	atom_name( 1,aa_thr,1) = " N  ";
	atom_name( 2,aa_thr,1) = " CA ";
	atom_name( 3,aa_thr,1) = " C  ";
	atom_name( 4,aa_thr,1) = " O  ";
	atom_name( 5,aa_thr,1) = " CB ";
	atom_name( 6,aa_thr,1) = " OG1";
	atom_name( 7,aa_thr,1) = " CG2";
	atom_name( 8,aa_thr,1) = " H  ";
	atom_name( 9,aa_thr,1) = " HG1";
	atom_name(10,aa_thr,1) = " HA ";
	atom_name(11,aa_thr,1) = " HB ";
	atom_name(12,aa_thr,1) = "1HG2";
	atom_name(13,aa_thr,1) = "2HG2";
	atom_name(14,aa_thr,1) = "3HG2";

//jjh intra residue bonding
	nbonded_neighbors( 1, aa_thr, 1) = 2; // N
		bonded_neighbor(1, 1, aa_thr, 1) = 2; // N--CA
		bonded_neighbor(2, 1, aa_thr, 1) = 8; // N--H
	nbonded_neighbors( 2, aa_thr, 1) = 4; // CA
		bonded_neighbor(1, 2, aa_thr, 1) = 1; // CA--N
		bonded_neighbor(2, 2, aa_thr, 1) = 3; // CA--C
		bonded_neighbor(3, 2, aa_thr, 1) = 5; // CA--CB
		bonded_neighbor(4, 2, aa_thr, 1) =10; // CA--HA
	nbonded_neighbors( 3, aa_thr, 1) = 2; // C
		bonded_neighbor(1, 3, aa_thr, 1) = 2; // C--CA
		bonded_neighbor(2, 3, aa_thr, 1) = 4; // C--O
	nbonded_neighbors( 4, aa_thr, 1) = 1; // O
		bonded_neighbor(1, 4, aa_thr, 1) = 3; // O--C
	nbonded_neighbors( 5, aa_thr, 1) = 4; // CB
		bonded_neighbor(1, 5, aa_thr, 1) = 2; // CB--CA
		bonded_neighbor(2, 5, aa_thr, 1) = 6; // CB--OG1
		bonded_neighbor(3, 5, aa_thr, 1) = 7; // CB--CG2
		bonded_neighbor(4, 5, aa_thr, 1) =11; // CB--HB
	nbonded_neighbors( 6, aa_thr, 1) = 2; // OG1
		bonded_neighbor(1, 6, aa_thr, 1) = 5; // OG1--CB
		bonded_neighbor(2, 6, aa_thr, 1) = 9; // OG1--HG1
	nbonded_neighbors( 7, aa_thr, 1) = 4; // CG2
		bonded_neighbor(1, 7, aa_thr, 1) = 5; // CG2--CB
		bonded_neighbor(2, 7, aa_thr, 1) =12; // CG2--1HG2
		bonded_neighbor(3, 7, aa_thr, 1) =13; // CG2--2HG2
		bonded_neighbor(4, 7, aa_thr, 1) =14; // CG2--3HG2
	nbonded_neighbors( 8, aa_thr, 1) = 1; // H
		bonded_neighbor(1, 8, aa_thr, 1) = 1; // H--N
	nbonded_neighbors( 9, aa_thr, 1) = 1; // HG1
		bonded_neighbor(1, 9, aa_thr, 1) = 6; // HG1--OG1
	nbonded_neighbors( 10, aa_thr, 1) = 1; // HA
		bonded_neighbor(1,10, aa_thr, 1) = 2; // HA--CA
	nbonded_neighbors( 11, aa_thr, 1) = 1; // HB
		bonded_neighbor(1,11, aa_thr, 1) = 5; // HB--CB
	nbonded_neighbors( 12, aa_thr, 1) = 1; //1HG2
		bonded_neighbor(1,12, aa_thr, 1) = 7; //1HG2--CG2
	nbonded_neighbors( 13, aa_thr, 1) = 1; //2HG2
		bonded_neighbor(1,13, aa_thr, 1) = 7; //2HG2--CG2
	nbonded_neighbors( 14, aa_thr, 1) = 1; //3HG2
		bonded_neighbor(1,14, aa_thr, 1) = 7; //3HG2--CG2

//jjh get the angles automatically
	get_angles_from_bonds(aa_thr, 1);

//chu   base atom numbers THR
	atom_base( 1,aa_thr,1) =  2; //  N  -> CA
	atom_base( 2,aa_thr,1) =  1; //  CA -> N
	atom_base( 3,aa_thr,1) =  2; //  C  -> CA
	atom_base( 4,aa_thr,1) =  3; //  O  -> C
	atom_base( 5,aa_thr,1) =  2; //  CB -> CA
	atom_base( 6,aa_thr,1) =  5; //  OG1 on  CB
	atom_base( 7,aa_thr,1) =  5; //  CG2 on  CB
	atom_base( 8,aa_thr,1) =  1; //  H   on  N
	atom_base( 9,aa_thr,1) =  6; //  HG1 on  OG1
	atom_base(10,aa_thr,1) =  2; //  HA  on  CA
	atom_base(11,aa_thr,1) =  5; //  HB  on  CB
	atom_base(12,aa_thr,1) =  7; // 1HG2 on  CG2
	atom_base(13,aa_thr,1) =  7; // 2HG2 on  CG2
	atom_base(14,aa_thr,1) =  7; // 3HG2 on  CG2
//bk   atom type numbers THR
	fullatom_type( 1,aa_thr,1) = 17; // Nbb    N
	fullatom_type( 2,aa_thr,1) = 18; // CAbb   CA
	fullatom_type( 3,aa_thr,1) = 19; // CObb   C
	fullatom_type( 4,aa_thr,1) = 20; // OCbb   O
	fullatom_type( 5,aa_thr,1) =  3; // CH1    CB
	fullatom_type( 6,aa_thr,1) = 13; // OH     OG1
	fullatom_type( 7,aa_thr,1) =  5; // CH3    CG2
	fullatom_type( 8,aa_thr,1) = 25; // HNbb   H
	fullatom_type( 9,aa_thr,1) = 22; // Hpol   HG1
	fullatom_type(10,aa_thr,1) = 23; // Hapo   HA
	fullatom_type(11,aa_thr,1) = 23; // Hapo   HB
	fullatom_type(12,aa_thr,1) = 23; // Hapo  1HG2
	fullatom_type(13,aa_thr,1) = 23; // Hapo  2HG2
	fullatom_type(14,aa_thr,1) = 23; // Hapo  3HG2

//cmd  atomic charge THR
	atomic_charge( 1,aa_thr,1) = -0.47; //    N
	atomic_charge( 2,aa_thr,1) =  0.07; //    CA
	atomic_charge( 3,aa_thr,1) =  0.51; //    C
	atomic_charge( 4,aa_thr,1) = -0.51; //    O
	atomic_charge( 5,aa_thr,1) =  0.14; //    CB
	atomic_charge( 6,aa_thr,1) = -0.66; //    OG1
	atomic_charge( 7,aa_thr,1) = -0.27; //    CG2
	atomic_charge( 8,aa_thr,1) =  0.31; //    H
	atomic_charge( 9,aa_thr,1) =  0.43; //    HG1
	atomic_charge(10,aa_thr,1) =  0.09; //    HA
	atomic_charge(11,aa_thr,1) =  0.09; //    HB
	atomic_charge(12,aa_thr,1) =  0.09; //   1HG2
	atomic_charge(13,aa_thr,1) =  0.09; //   2HG2
	atomic_charge(14,aa_thr,1) =  0.09; //   3HG2

	assert_total_charge( aa_thr, 1, 0.0);

//cmd  first sidechain atom
	first_scatom( aa_thr,1) = 5; //CB

//bk   template atoms used for placing atoms THR

//bk   template for building  HG1
	ta(1, 9,aa_thr,1) =  6; //   OG1
	ta(2, 9,aa_thr,1) =  5; //   CB
	ta(3, 9,aa_thr,1) =  2; //   CA

//bk   template for building  HA
	ta(1,10,aa_thr,1) =  2; //   CA
	ta(2,10,aa_thr,1) =  1; //   N
	ta(3,10,aa_thr,1) =  3; //   C

//bk   template for building  HB
	ta(1,11,aa_thr,1) =  5; //   CB
	ta(2,11,aa_thr,1) =  2; //   CA
	ta(3,11,aa_thr,1) =  6; //   OG1

//bk   template for building 1HG2
	ta(1,12,aa_thr,1) =  7; //   CG2
	ta(2,12,aa_thr,1) =  5; //   CB
	ta(3,12,aa_thr,1) =  2; //   CA

//bk   template for building 2HG2
	ta(1,13,aa_thr,1) =  7; //   CG2
	ta(2,13,aa_thr,1) =  5; //   CB
	ta(3,13,aa_thr,1) =  2; //   CA

//bk   template for building 3HG2
	ta(1,14,aa_thr,1) =  7; //   CG2
	ta(2,14,aa_thr,1) =  5; //   CB
	ta(3,14,aa_thr,1) =  2; //   CA

//bk   chi angles required to build atoms THR

//bk   chi angles needed for building  N
	chi_required(1, 1,aa_thr,1) = false;
	chi_required(2, 1,aa_thr,1) = false;
	chi_required(3, 1,aa_thr,1) = false;
	chi_required(4, 1,aa_thr,1) = false;

//bk   chi angles needed for building  CA
	chi_required(1, 2,aa_thr,1) = false;
	chi_required(2, 2,aa_thr,1) = false;
	chi_required(3, 2,aa_thr,1) = false;
	chi_required(4, 2,aa_thr,1) = false;

//bk   chi angles needed for building  C
	chi_required(1, 3,aa_thr,1) = false;
	chi_required(2, 3,aa_thr,1) = false;
	chi_required(3, 3,aa_thr,1) = false;
	chi_required(4, 3,aa_thr,1) = false;

//bk   chi angles needed for building  O
	chi_required(1, 4,aa_thr,1) = false;
	chi_required(2, 4,aa_thr,1) = false;
	chi_required(3, 4,aa_thr,1) = false;
	chi_required(4, 4,aa_thr,1) = false;

//bk   chi angles needed for building  CB
	chi_required(1, 5,aa_thr,1) = false;
	chi_required(2, 5,aa_thr,1) = false;
	chi_required(3, 5,aa_thr,1) = false;
	chi_required(4, 5,aa_thr,1) = false;

//bk   chi angles needed for building  OG1
	chi_required(1, 6,aa_thr,1) =  true;
	chi_required(2, 6,aa_thr,1) = false;
	chi_required(3, 6,aa_thr,1) = false;
	chi_required(4, 6,aa_thr,1) = false;

//bk   chi angles needed for building  CG2
	chi_required(1, 7,aa_thr,1) =  true;
	chi_required(2, 7,aa_thr,1) = false;
	chi_required(3, 7,aa_thr,1) = false;
	chi_required(4, 7,aa_thr,1) = false;

//bk   chi angles needed for building  H
	chi_required(1, 8,aa_thr,1) = false;
	chi_required(2, 8,aa_thr,1) = false;
	chi_required(3, 8,aa_thr,1) = false;
	chi_required(4, 8,aa_thr,1) = false;

//bk   chi angles needed for building  HG1
	chi_required(1, 9,aa_thr,1) =  true;
	chi_required(2, 9,aa_thr,1) =  true;
	chi_required(3, 9,aa_thr,1) = false;
	chi_required(4, 9,aa_thr,1) = false;

//bk   chi angles needed for building  HA
	chi_required(1,10,aa_thr,1) = false;
	chi_required(2,10,aa_thr,1) = false;
	chi_required(3,10,aa_thr,1) = false;
	chi_required(4,10,aa_thr,1) = false;

//bk   chi angles needed for building  HB
	chi_required(1,11,aa_thr,1) =  true;
	chi_required(2,11,aa_thr,1) = false;
	chi_required(3,11,aa_thr,1) = false;
	chi_required(4,11,aa_thr,1) = false;

//bk   chi angles needed for building 1HG2
	chi_required(1,12,aa_thr,1) =  true;
	chi_required(2,12,aa_thr,1) = false;
	chi_required(3,12,aa_thr,1) = false;
	chi_required(4,12,aa_thr,1) = false;

//bk   chi angles needed for building 2HG2
	chi_required(1,13,aa_thr,1) =  true;
	chi_required(2,13,aa_thr,1) = false;
	chi_required(3,13,aa_thr,1) = false;
	chi_required(4,13,aa_thr,1) = false;

//bk   chi angles needed for building 3HG2
	chi_required(1,14,aa_thr,1) =  true;
	chi_required(2,14,aa_thr,1) = false;
	chi_required(3,14,aa_thr,1) = false;
	chi_required(4,14,aa_thr,1) = false;

//bk   four atoms that define chi angle  1
	chi_atoms( 1, 1,aa_thr, 1) =  1; //   N
	chi_atoms( 2, 1,aa_thr, 1) =  2; //   CA
	chi_atoms( 3, 1,aa_thr, 1) =  5; //   CB
	chi_atoms( 4, 1,aa_thr, 1) =  6; //   OG1
//bk   four atoms that define chi angle  2
	chi_atoms( 1, 2,aa_thr, 1) =  2; //   CA
	chi_atoms( 2, 2,aa_thr, 1) =  5; //   CB
	chi_atoms( 3, 2,aa_thr, 1) =  6; //   OG1
	chi_atoms( 4, 2,aa_thr, 1) =  9; //   HG1

	HNpos(aa_thr,1) =  8; // atom number for backbone HN
	HApos(aa_thr,1) = 10; // atom number for backbone HA
	nH_polar(aa_thr,1) =  2; // number of polar hydrogens
	nH_aromatic(aa_thr,1) =  0; // number of aromatic hydrogens
	nH_apolar(aa_thr,1) =  5; // number of apolar hydrogens
	nacceptors(aa_thr,1) =  2; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1,aa_thr,1) =  8; //  H
	Hpos_polar( 2,aa_thr,1) =  9; //  HG1

//bk   atom number for aromatic hydrogens

//bk   atom number for apolar hydrogens
	Hpos_apolar( 1,aa_thr,1) = 10; //  HA
	Hpos_apolar( 2,aa_thr,1) = 11; //  HB
	Hpos_apolar( 3,aa_thr,1) = 12; // 1HG2
	Hpos_apolar( 4,aa_thr,1) = 13; // 2HG2
	Hpos_apolar( 5,aa_thr,1) = 14; // 3HG2

//bk   acceptor positions
	accpt_pos( 1,aa_thr, 1) =  4; //  O
	accpt_pos( 2,aa_thr, 1) =  6; //  OG1

//bk   acceptor base positions
	abase2( 6,aa_thr, 1) =  9; //  HG1 base for  OG1

//bk   hydrogens on each atom

	nhydrogens_on_atm( 1,aa_thr, 1) =  1; //  N
	hydrogens_on_atm( 1, 1,aa_thr, 1) =  8; //  H

	nhydrogens_on_atm( 2,aa_thr, 1) =  1; //  CA
	hydrogens_on_atm( 1, 2,aa_thr, 1) = 10; //  HA

	nhydrogens_on_atm( 5,aa_thr, 1) =  1; //  CB
	hydrogens_on_atm( 1, 5,aa_thr, 1) = 11; //  HB

	nhydrogens_on_atm( 6,aa_thr, 1) =  1; //  OG1
	hydrogens_on_atm( 1, 6,aa_thr, 1) =  9; //  HG1

	nhydrogens_on_atm( 7,aa_thr, 1) =  3; //  CG2
	hydrogens_on_atm( 1, 7,aa_thr, 1) = 12; // 1HG2
	hydrogens_on_atm( 2, 7,aa_thr, 1) = 13; // 2HG2
	hydrogens_on_atm( 3, 7,aa_thr, 1) = 14; // 3HG2

//bk   template coordinates for the amino acid
	icoor( 1, 1,aa_thr, 1) =    0.0000; //   N
	icoor( 2, 1,aa_thr, 1) =   16.9170; //   N
	icoor( 3, 1,aa_thr, 1) =   58.2740; //   N
	icoor( 1, 2,aa_thr, 1) =    0.0000; //   CA
	icoor( 2, 2,aa_thr, 1) =   17.7300; //   CA
	icoor( 3, 2,aa_thr, 1) =   59.4840; //   CA
	icoor( 1, 3,aa_thr, 1) =    0.0000; //   C
	icoor( 2, 3,aa_thr, 1) =   16.8580; //   C
	icoor( 3, 3,aa_thr, 1) =   60.7340; //   C
	icoor( 1, 4,aa_thr, 1) =    0.0030; //   O
	icoor( 2, 4,aa_thr, 1) =   15.6300; //   O
	icoor( 3, 4,aa_thr, 1) =   60.6450; //   O
	icoor( 1, 5,aa_thr, 1) =   -1.2150; //   CB
	icoor( 2, 5,aa_thr, 1) =   18.6750; //   CB
	icoor( 3, 5,aa_thr, 1) =   59.5300; //   CB
	icoor( 1, 6,aa_thr, 1) =   -1.1740; //   OG1
	icoor( 2, 6,aa_thr, 1) =   19.4520; //   OG1
	icoor( 3, 6,aa_thr, 1) =   60.7340; //   OG1
	icoor( 1, 7,aa_thr, 1) =   -2.5110; //   CG2
	icoor( 2, 7,aa_thr, 1) =   17.8800; //   CG2
	icoor( 3, 7,aa_thr, 1) =   59.4880; //   CG2
	icoor( 1, 8,aa_thr, 1) =   -0.0310; //   H
	icoor( 2, 8,aa_thr, 1) =   15.9120; //   H
	icoor( 3, 8,aa_thr, 1) =   58.3720; //   H
	icoor( 1, 9,aa_thr, 1) =   -1.9320; //   HG1
	icoor( 2, 9,aa_thr, 1) =   20.0410; //   HG1
	icoor( 3, 9,aa_thr, 1) =   60.7600; //   HG1
	icoor( 1,10,aa_thr, 1) =    0.9100; //   HA
	icoor( 2,10,aa_thr, 1) =   18.3290; //   HA
	icoor( 3,10,aa_thr, 1) =   59.5260; //   HA
	icoor( 1,11,aa_thr, 1) =   -1.1740; //   HB
	icoor( 2,11,aa_thr, 1) =   19.3470; //   HB
	icoor( 3,11,aa_thr, 1) =   58.6730; //   HB
	icoor( 1,12,aa_thr, 1) =   -3.3580; //  1HG2
	icoor( 2,12,aa_thr, 1) =   18.5650; //  1HG2
	icoor( 3,12,aa_thr, 1) =   59.5210; //  1HG2
	icoor( 1,13,aa_thr, 1) =   -2.5500; //  2HG2
	icoor( 2,13,aa_thr, 1) =   17.2970; //  2HG2
	icoor( 3,13,aa_thr, 1) =   58.5680; //  2HG2
	icoor( 1,14,aa_thr, 1) =   -2.5530; //  3HG2
	icoor( 2,14,aa_thr, 1) =   17.2090; //  3HG2
	icoor( 3,14,aa_thr, 1) =   60.3450; //  3HG2


	// H positions from REDUCE
	if( param_pack::packer_logical::reduce_templates ) {
		icoor( 1 , 8 , aa_thr , 1) = 0.922 ; // H
		icoor( 2 , 8 , aa_thr , 1) = 16.893 ; // H
		icoor( 3 , 8 , aa_thr , 1) = 57.888 ; // H
		icoor( 1 , 10 , aa_thr , 1) = 0.920 ; // HA
		icoor( 2 , 10 , aa_thr , 1) = 18.332 ; // HA
		icoor( 3 , 10 , aa_thr , 1) = 59.461 ; // HA
		icoor( 1 , 11 , aa_thr , 1) = -1.176 ; // HB
		icoor( 2 , 11 , aa_thr , 1) = 19.340 ; // HB
		icoor( 3 , 11 , aa_thr , 1) = 58.655 ; // HB
		icoor( 1 , 9 , aa_thr , 1) = -1.962 ; // HG1
		icoor( 2 , 9 , aa_thr , 1) = 20.066 ; // HG1
		icoor( 3 , 9 , aa_thr , 1) = 60.762 ; // HG1
		icoor( 1 , 12 , aa_thr , 1) = -3.367 ; // 1HG2
		icoor( 2 , 12 , aa_thr , 1) = 18.571 ; // 1HG2
		icoor( 3 , 12 , aa_thr , 1) = 59.522 ; // 1HG2
		icoor( 1 , 13 , aa_thr , 1) = -2.551 ; // 2HG2
		icoor( 2 , 13 , aa_thr , 1) = 17.292 ; // 2HG2
		icoor( 3 , 13 , aa_thr , 1) = 58.559 ; // 2HG2
		icoor( 1 , 14 , aa_thr , 1) = -2.554 ; // 3HG2
		icoor( 2 , 14 , aa_thr , 1) = 17.202 ; // 3HG2
		icoor( 3 , 14 , aa_thr , 1) = 60.353 ; // 3HG2
	}

//bq create N and Cterm variants
	create_termini_variants( aa_thr );


//bk   create variants that have h2o connected to the backbone nh and carbonyl

	if ( explicit_h2o ) {
		create_backbone_h2o_variants(aa_thr);
		create_sidechain_h2o_variants(aa_thr);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_val
///
/// @brief arrays filled to define the proerties of the amino acid Valine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// aa_val - index number for VAL (18)
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_val()
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

//bk   Parameters for VAL -----------------------------
	nvar(aa_val) = 1; // starting number of variants for VAL
	variant_type( aav_base, aa_val, 1 ) = true; // type of aa variant

//bk   variant1 ---------- VAL

	natoms(aa_val,1) = 16; // number of total atoms
	nheavyatoms(aa_val,1) =  7; // number of heavy atoms
	nchi(aa_val,1) =  1; // number of chi angles

//bk   atom names VAL
	atom_name( 1,aa_val,1) = " N  ";
	atom_name( 2,aa_val,1) = " CA ";
	atom_name( 3,aa_val,1) = " C  ";
	atom_name( 4,aa_val,1) = " O  ";
	atom_name( 5,aa_val,1) = " CB ";
	atom_name( 6,aa_val,1) = " CG1";
	atom_name( 7,aa_val,1) = " CG2";
	atom_name( 8,aa_val,1) = " H  ";
	atom_name( 9,aa_val,1) = " HA ";
	atom_name(10,aa_val,1) = " HB ";
	atom_name(11,aa_val,1) = "1HG1";
	atom_name(12,aa_val,1) = "2HG1";
	atom_name(13,aa_val,1) = "3HG1";
	atom_name(14,aa_val,1) = "1HG2";
	atom_name(15,aa_val,1) = "2HG2";
	atom_name(16,aa_val,1) = "3HG2";

//jjh intra residue bonding
	nbonded_neighbors( 1, aa_val, 1) = 2; // N
		bonded_neighbor(1, 1, aa_val, 1) = 2; // N--CA
		bonded_neighbor(2, 1, aa_val, 1) = 8; // N--H
	nbonded_neighbors( 2, aa_val, 1) = 4; // CA
		bonded_neighbor(1, 2, aa_val, 1) = 1; // CA--N
		bonded_neighbor(2, 2, aa_val, 1) = 3; // CA--C
		bonded_neighbor(3, 2, aa_val, 1) = 5; // CA--CB
		bonded_neighbor(4, 2, aa_val, 1) = 9; // CA--HA
	nbonded_neighbors( 3, aa_val, 1) = 2; // C
		bonded_neighbor(1, 3, aa_val, 1) = 2; // C--CA
		bonded_neighbor(2, 3, aa_val, 1) = 4; // C--O
	nbonded_neighbors( 4, aa_val, 1) = 1; // O
		bonded_neighbor(1, 4, aa_val, 1) = 3; // O--C
	nbonded_neighbors( 5, aa_val, 1) = 4; // CB
		bonded_neighbor(1, 5, aa_val, 1) = 2; // CB--CA
		bonded_neighbor(2, 5, aa_val, 1) = 6; // CB--CG1
		bonded_neighbor(3, 5, aa_val, 1) = 7; // CB--CG2
		bonded_neighbor(4, 5, aa_val, 1) =10; // CB--HB
	nbonded_neighbors( 6, aa_val, 1) = 4; // CG1
		bonded_neighbor(1, 6, aa_val, 1) = 5; // CG1--CB
		bonded_neighbor(2, 6, aa_val, 1) =11; // CG1--1HG1
		bonded_neighbor(3, 6, aa_val, 1) =12; // CG1--2HG1
		bonded_neighbor(4, 6, aa_val, 1) =13; // CG1--3HG1
	nbonded_neighbors( 7, aa_val, 1) = 4; // CG2
		bonded_neighbor(1, 7, aa_val, 1) = 5; // CG2--CB
		bonded_neighbor(2, 7, aa_val, 1) =14; // CG2--1HG2
		bonded_neighbor(3, 7, aa_val, 1) =15; // CG2--2HG2
		bonded_neighbor(4, 7, aa_val, 1) =16; // CG2--3HG2
	nbonded_neighbors( 8, aa_val, 1) = 1; // H
		bonded_neighbor(1, 8, aa_val, 1) = 1; // H--N
	nbonded_neighbors( 9, aa_val, 1) = 1; // HA
		bonded_neighbor(1, 9, aa_val, 1) = 2; // HA--CA
	nbonded_neighbors( 10, aa_val, 1) = 1; // HB
		bonded_neighbor(1,10, aa_val, 1) = 5; // HB--CB
	nbonded_neighbors( 11, aa_val, 1) = 1; //1HG1
		bonded_neighbor(1,11, aa_val, 1) = 6; //1HG1--CG1
	nbonded_neighbors( 12, aa_val, 1) = 1; //2HG1
		bonded_neighbor(1,12, aa_val, 1) = 6; //2HG1--CG1
	nbonded_neighbors( 13, aa_val, 1) = 1; //3HG1
		bonded_neighbor(1,13, aa_val, 1) = 6; //3HG1--CG1
	nbonded_neighbors( 14, aa_val, 1) = 1; //1HG2
		bonded_neighbor(1,14, aa_val, 1) = 7; //1HG2--CG2
	nbonded_neighbors( 15, aa_val, 1) = 1; //2HG2
		bonded_neighbor(1,15, aa_val, 1) = 7; //2HG2--CG2
	nbonded_neighbors( 16, aa_val, 1) = 1; //3HG2
		bonded_neighbor(1,16, aa_val, 1) = 7; //3HG2--CG2

//jjh get the angles automatically
	get_angles_from_bonds(aa_val, 1);

//chu   base atom numbers VAL
	atom_base( 1,aa_val,1) =  2; //  N  -> CA
	atom_base( 2,aa_val,1) =  1; //  CA -> N
	atom_base( 3,aa_val,1) =  2; //  C  -> CA
	atom_base( 4,aa_val,1) =  3; //  O  -> C
	atom_base( 5,aa_val,1) =  2; //  CB -> CA
	atom_base( 6,aa_val,1) =  5; //  CG1 on  CB
	atom_base( 7,aa_val,1) =  5; //  CG2 on  CB
	atom_base( 8,aa_val,1) =  1; //  H   on  N
	atom_base( 9,aa_val,1) =  2; //  HA  on  CA
	atom_base(10,aa_val,1) =  5; //  HB  on  CB
	atom_base(11,aa_val,1) =  6; // 1HG1 on  CG1
	atom_base(12,aa_val,1) =  6; // 2HG1 on  CG1
	atom_base(13,aa_val,1) =  6; // 3HG1 on  CG1
	atom_base(14,aa_val,1) =  7; // 1HG2 on  CG2
	atom_base(15,aa_val,1) =  7; // 2HG2 on  CG2
	atom_base(16,aa_val,1) =  7; // 3HG2 on  CG2
//bk   atom type numbers VAL
	fullatom_type( 1,aa_val,1) = 17; // Nbb    N
	fullatom_type( 2,aa_val,1) = 18; // CAbb   CA
	fullatom_type( 3,aa_val,1) = 19; // CObb   C
	fullatom_type( 4,aa_val,1) = 20; // OCbb   O
	fullatom_type( 5,aa_val,1) =  3; // CH1    CB
	fullatom_type( 6,aa_val,1) =  5; // CH3    CG1
	fullatom_type( 7,aa_val,1) =  5; // CH3    CG2
	fullatom_type( 8,aa_val,1) = 25; // HNbb   H
	fullatom_type( 9,aa_val,1) = 23; // Hapo   HA
	fullatom_type(10,aa_val,1) = 23; // Hapo   HB
	fullatom_type(11,aa_val,1) = 23; // Hapo  1HG1
	fullatom_type(12,aa_val,1) = 23; // Hapo  2HG1
	fullatom_type(13,aa_val,1) = 23; // Hapo  3HG1
	fullatom_type(14,aa_val,1) = 23; // Hapo  1HG2
	fullatom_type(15,aa_val,1) = 23; // Hapo  2HG2
	fullatom_type(16,aa_val,1) = 23; // Hapo  3HG2

//cmd  atomic charge VAL
	atomic_charge( 1,aa_val,1) = -0.47; //    N
	atomic_charge( 2,aa_val,1) =  0.07; //    CA
	atomic_charge( 3,aa_val,1) =  0.51; //    C
	atomic_charge( 4,aa_val,1) = -0.51; //    O
	atomic_charge( 5,aa_val,1) = -0.09; //    CB
	atomic_charge( 6,aa_val,1) = -0.27; //    CG1
	atomic_charge( 7,aa_val,1) = -0.27; //    CG2
	atomic_charge( 8,aa_val,1) =  0.31; //    H
	atomic_charge( 9,aa_val,1) =  0.09; //    HA
	atomic_charge(10,aa_val,1) =  0.09; //    HB
	atomic_charge(11,aa_val,1) =  0.09; //   1HG1
	atomic_charge(12,aa_val,1) =  0.09; //   2HG1
	atomic_charge(13,aa_val,1) =  0.09; //   3HG1
	atomic_charge(14,aa_val,1) =  0.09; //   1HG2
	atomic_charge(15,aa_val,1) =  0.09; //   2HG2
	atomic_charge(16,aa_val,1) =  0.09; //   3HG2

	assert_total_charge( aa_val, 1, 0.0);

//cmd  first sidechain atom
	first_scatom( aa_val,1) = 5; //CB

//bk   template atoms used for placing atoms VAL

//bk   template for building  HA
	ta(1, 9,aa_val,1) =  2; //   CA
	ta(2, 9,aa_val,1) =  1; //   N
	ta(3, 9,aa_val,1) =  3; //   C

//bk   template for building  HB
	ta(1,10,aa_val,1) =  5; //   CB
	ta(2,10,aa_val,1) =  2; //   CA
	ta(3,10,aa_val,1) =  6; //   CG1

//bk   template for building 1HG1
	ta(1,11,aa_val,1) =  6; //   CG1
	ta(2,11,aa_val,1) =  5; //   CB
	ta(3,11,aa_val,1) =  2; //   CA

//bk   template for building 2HG1
	ta(1,12,aa_val,1) =  6; //   CG1
	ta(2,12,aa_val,1) =  5; //   CB
	ta(3,12,aa_val,1) =  2; //   CA

//bk   template for building 3HG1
	ta(1,13,aa_val,1) =  6; //   CG1
	ta(2,13,aa_val,1) =  5; //   CB
	ta(3,13,aa_val,1) =  2; //   CA

//bk   template for building 1HG2
	ta(1,14,aa_val,1) =  7; //   CG2
	ta(2,14,aa_val,1) =  5; //   CB
	ta(3,14,aa_val,1) =  2; //   CA

//bk   template for building 2HG2
	ta(1,15,aa_val,1) =  7; //   CG2
	ta(2,15,aa_val,1) =  5; //   CB
	ta(3,15,aa_val,1) =  2; //   CA

//bk   template for building 3HG2
	ta(1,16,aa_val,1) =  7; //   CG2
	ta(2,16,aa_val,1) =  5; //   CB
	ta(3,16,aa_val,1) =  2; //   CA

//bk   chi angles required to build atoms VAL

//bk   chi angles needed for building  N
	chi_required(1, 1,aa_val,1) = false;
	chi_required(2, 1,aa_val,1) = false;
	chi_required(3, 1,aa_val,1) = false;
	chi_required(4, 1,aa_val,1) = false;

//bk   chi angles needed for building  CA
	chi_required(1, 2,aa_val,1) = false;
	chi_required(2, 2,aa_val,1) = false;
	chi_required(3, 2,aa_val,1) = false;
	chi_required(4, 2,aa_val,1) = false;

//bk   chi angles needed for building  C
	chi_required(1, 3,aa_val,1) = false;
	chi_required(2, 3,aa_val,1) = false;
	chi_required(3, 3,aa_val,1) = false;
	chi_required(4, 3,aa_val,1) = false;

//bk   chi angles needed for building  O
	chi_required(1, 4,aa_val,1) = false;
	chi_required(2, 4,aa_val,1) = false;
	chi_required(3, 4,aa_val,1) = false;
	chi_required(4, 4,aa_val,1) = false;

//bk   chi angles needed for building  CB
	chi_required(1, 5,aa_val,1) = false;
	chi_required(2, 5,aa_val,1) = false;
	chi_required(3, 5,aa_val,1) = false;
	chi_required(4, 5,aa_val,1) = false;

//bk   chi angles needed for building  CG1
	chi_required(1, 6,aa_val,1) =  true;
	chi_required(2, 6,aa_val,1) = false;
	chi_required(3, 6,aa_val,1) = false;
	chi_required(4, 6,aa_val,1) = false;

//bk   chi angles needed for building  CG2
	chi_required(1, 7,aa_val,1) =  true;
	chi_required(2, 7,aa_val,1) = false;
	chi_required(3, 7,aa_val,1) = false;
	chi_required(4, 7,aa_val,1) = false;

//bk   chi angles needed for building  H
	chi_required(1, 8,aa_val,1) = false;
	chi_required(2, 8,aa_val,1) = false;
	chi_required(3, 8,aa_val,1) = false;
	chi_required(4, 8,aa_val,1) = false;

//bk   chi angles needed for building  HA
	chi_required(1, 9,aa_val,1) = false;
	chi_required(2, 9,aa_val,1) = false;
	chi_required(3, 9,aa_val,1) = false;
	chi_required(4, 9,aa_val,1) = false;

//bk   chi angles needed for building  HB
	chi_required(1,10,aa_val,1) =  true;
	chi_required(2,10,aa_val,1) = false;
	chi_required(3,10,aa_val,1) = false;
	chi_required(4,10,aa_val,1) = false;

//bk   chi angles needed for building 1HG1
	chi_required(1,11,aa_val,1) =  true;
	chi_required(2,11,aa_val,1) = false;
	chi_required(3,11,aa_val,1) = false;
	chi_required(4,11,aa_val,1) = false;

//bk   chi angles needed for building 2HG1
	chi_required(1,12,aa_val,1) =  true;
	chi_required(2,12,aa_val,1) = false;
	chi_required(3,12,aa_val,1) = false;
	chi_required(4,12,aa_val,1) = false;

//bk   chi angles needed for building 3HG1
	chi_required(1,13,aa_val,1) =  true;
	chi_required(2,13,aa_val,1) = false;
	chi_required(3,13,aa_val,1) = false;
	chi_required(4,13,aa_val,1) = false;

//bk   chi angles needed for building 1HG2
	chi_required(1,14,aa_val,1) =  true;
	chi_required(2,14,aa_val,1) = false;
	chi_required(3,14,aa_val,1) = false;
	chi_required(4,14,aa_val,1) = false;

//bk   chi angles needed for building 2HG2
	chi_required(1,15,aa_val,1) =  true;
	chi_required(2,15,aa_val,1) = false;
	chi_required(3,15,aa_val,1) = false;
	chi_required(4,15,aa_val,1) = false;

//bk   chi angles needed for building 3HG2
	chi_required(1,16,aa_val,1) =  true;
	chi_required(2,16,aa_val,1) = false;
	chi_required(3,16,aa_val,1) = false;
	chi_required(4,16,aa_val,1) = false;

//bk   four atoms that define chi angle  1
	chi_atoms( 1, 1,aa_val, 1) =  1; //   N
	chi_atoms( 2, 1,aa_val, 1) =  2; //   CA
	chi_atoms( 3, 1,aa_val, 1) =  5; //   CB
	chi_atoms( 4, 1,aa_val, 1) =  6; //   CG1

	HNpos(aa_val,1) =  8; // atom number for backbone HN
	HApos(aa_val,1) =  9; // atom number for backbone HA
	nH_polar(aa_val,1) =  1; // number of polar hydrogens
	nH_aromatic(aa_val,1) =  0; // number of aromatic hydrogens
	nH_apolar(aa_val,1) =  8; // number of apolar hydrogens
	nacceptors(aa_val,1) =  1; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1,aa_val,1) =  8; //  H

//bk   atom number for aromatic hydrogens

//bk   atom number for apolar hydrogens
	Hpos_apolar( 1,aa_val,1) =  9; //  HA
	Hpos_apolar( 2,aa_val,1) = 10; //  HB
	Hpos_apolar( 3,aa_val,1) = 11; // 1HG1
	Hpos_apolar( 4,aa_val,1) = 12; // 2HG1
	Hpos_apolar( 5,aa_val,1) = 13; // 3HG1
	Hpos_apolar( 6,aa_val,1) = 14; // 1HG2
	Hpos_apolar( 7,aa_val,1) = 15; // 2HG2
	Hpos_apolar( 8,aa_val,1) = 16; // 3HG2

//bk   acceptor positions
	accpt_pos( 1,aa_val, 1) =  4; //  O

//bk   hydrogens on each atom

	nhydrogens_on_atm( 1,aa_val, 1) =  1; //  N
	hydrogens_on_atm( 1, 1,aa_val, 1) =  8; //  H

	nhydrogens_on_atm( 2,aa_val, 1) =  1; //  CA
	hydrogens_on_atm( 1, 2,aa_val, 1) =  9; //  HA

	nhydrogens_on_atm( 5,aa_val, 1) =  1; //  CB
	hydrogens_on_atm( 1, 5,aa_val, 1) = 10; //  HB

	nhydrogens_on_atm( 6,aa_val, 1) =  3; //  CG1
	hydrogens_on_atm( 1, 6,aa_val, 1) = 11; // 1HG1
	hydrogens_on_atm( 2, 6,aa_val, 1) = 12; // 2HG1
	hydrogens_on_atm( 3, 6,aa_val, 1) = 13; // 3HG1

	nhydrogens_on_atm( 7,aa_val, 1) =  3; //  CG2
	hydrogens_on_atm( 1, 7,aa_val, 1) = 14; // 1HG2
	hydrogens_on_atm( 2, 7,aa_val, 1) = 15; // 2HG2
	hydrogens_on_atm( 3, 7,aa_val, 1) = 16; // 3HG2

//bk   template coordinates for the amino acid
	icoor( 1, 1,aa_val, 1) =    0.0000; //   N
	icoor( 2, 1,aa_val, 1) =   17.5010; //   N
	icoor( 3, 1,aa_val, 1) =   61.8970; //   N
	icoor( 1, 2,aa_val, 1) =    0.0000; //   CA
	icoor( 2, 2,aa_val, 1) =   16.7860; //   CA
	icoor( 3, 2,aa_val, 1) =   63.1670; //   CA
	icoor( 1, 3,aa_val, 1) =    0.0000; //   C
	icoor( 2, 3,aa_val, 1) =   17.7530; //   C
	icoor( 3, 3,aa_val, 1) =   64.3440; //   C
	icoor( 1, 4,aa_val, 1) =   -0.0020; //   O
	icoor( 2, 4,aa_val, 1) =   18.9700; //   O
	icoor( 3, 4,aa_val, 1) =   64.1600; //   O
	icoor( 1, 5,aa_val, 1) =    1.2150; //   CB
	icoor( 2, 5,aa_val, 1) =   15.8470; //   CB
	icoor( 3, 5,aa_val, 1) =   63.2870; //   CB
	icoor( 1, 6,aa_val, 1) =    1.1920; //   CG1
	icoor( 2, 6,aa_val, 1) =   15.1140; //   CG1
	icoor( 3, 6,aa_val, 1) =   64.6200; //   CG1
	icoor( 1, 7,aa_val, 1) =    1.2370; //   CG2
	icoor( 2, 7,aa_val, 1) =   14.8540; //   CG2
	icoor( 3, 7,aa_val, 1) =   62.1350; //   CG2
	icoor( 1, 8,aa_val, 1) =    0.0310; //   H
	icoor( 2, 8,aa_val, 1) =   18.5100; //   H
	icoor( 3, 8,aa_val, 1) =   61.9150; //   H
	icoor( 1, 9,aa_val, 1) =   -0.9090; //   HA
	icoor( 2, 9,aa_val, 1) =   16.1960; //   HA
	icoor( 3, 9,aa_val, 1) =   63.2900; //   HA
	icoor( 1,10,aa_val, 1) =    2.1280; //   HB
	icoor( 2,10,aa_val, 1) =   16.4380; //   HB
	icoor( 3,10,aa_val, 1) =   63.2130; //   HB
	icoor( 1,11,aa_val, 1) =    2.0580; //  1HG1
	icoor( 2,11,aa_val, 1) =   14.4550; //  1HG1
	icoor( 3,11,aa_val, 1) =   64.6880; //  1HG1
	icoor( 1,12,aa_val, 1) =    1.2230; //  2HG1
	icoor( 2,12,aa_val, 1) =   15.8380; //  2HG1
	icoor( 3,12,aa_val, 1) =   65.4340; //  2HG1
	icoor( 1,13,aa_val, 1) =    0.2800; //  3HG1
	icoor( 2,13,aa_val, 1) =   14.5220; //  3HG1
	icoor( 3,13,aa_val, 1) =   64.6940; //  3HG1
	icoor( 1,14,aa_val, 1) =    2.1020; //  1HG2
	icoor( 2,14,aa_val, 1) =   14.1990; //  1HG2
	icoor( 3,14,aa_val, 1) =   62.2350; //  1HG2
	icoor( 1,15,aa_val, 1) =    0.3250; //  2HG2
	icoor( 2,15,aa_val, 1) =   14.2570; //  2HG2
	icoor( 3,15,aa_val, 1) =   62.1530; //  2HG2
	icoor( 1,16,aa_val, 1) =    1.2990; //  3HG2
	icoor( 2,16,aa_val, 1) =   15.3940; //  3HG2
	icoor( 3,16,aa_val, 1) =   61.1900; //  3HG2

	// H positions from REDUCE
	if( param_pack::packer_logical::reduce_templates ) {
		icoor( 1 , 8 , aa_val , 1) = -0.275 ; // H
		icoor( 2 , 8 , aa_val , 1) = 18.450 ; // H
		icoor( 3 , 8 , aa_val , 1) = 62.048 ; // H
		icoor( 1 , 9 , aa_val , 1) = -0.920 ; // HA
		icoor( 2 , 9 , aa_val , 1) = 16.184 ; // HA
		icoor( 3 , 9 , aa_val , 1) = 63.191 ; // HA
		icoor( 1 , 10 , aa_val , 1) = 2.130 ; // HB
		icoor( 2 , 10 , aa_val , 1) = 16.456 ; // HB
		icoor( 3 , 10 , aa_val , 1) = 63.240 ; // HB
		icoor( 1 , 11 , aa_val , 1) = 2.065 ; // 1HG1
		icoor( 2 , 11 , aa_val , 1) = 14.449 ; // 1HG1
		icoor( 3 , 11 , aa_val , 1) = 64.688 ; // 1HG1
		icoor( 1 , 12 , aa_val , 1) = 1.223 ; // 2HG1
		icoor( 2 , 12 , aa_val , 1) = 15.845 ; // 2HG1
		icoor( 3 , 12 , aa_val , 1) = 65.442 ; // 2HG1
		icoor( 1 , 13 , aa_val , 1) = 0.271 ; // 3HG1
		icoor( 2 , 13 , aa_val , 1) = 14.517 ; // 3HG1
		icoor( 3 , 13 , aa_val , 1) = 64.695 ; // 3HG1
		icoor( 1 , 14 , aa_val , 1) = 2.110 ; // 1HG2
		icoor( 2 , 14 , aa_val , 1) = 14.193 ; // 1HG2
		icoor( 3 , 14 , aa_val , 1) = 62.237 ; // 1HG2
		icoor( 1 , 15 , aa_val , 1) = 0.317 ; // 2HG2
		icoor( 2 , 15 , aa_val , 1) = 14.251 ; // 2HG2
		icoor( 3 , 15 , aa_val , 1) = 62.152 ; // 2HG2
		icoor( 1 , 16 , aa_val , 1) = 1.300 ; // 3HG2
		icoor( 2 , 16 , aa_val , 1) = 15.399 ; // 3HG2
		icoor( 3 , 16 , aa_val , 1) = 61.182 ; // 3HG2
	}

//bq create N and Cterm variants
	create_termini_variants( aa_val );


//bk   create variants that have h2o connected to the backbone nh and carbonyl
	if ( explicit_h2o ) {
		create_backbone_h2o_variants(aa_val);
	}
}
////////////////////////////////////////////////////////////////////////////////
/// @begin init_trp
///
/// @brief arrays filled to define the proerties of the amino acid Tryptophan
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// aa_trp - index number for TRP (19)
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_trp()
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

//bk   Parameters for TRP -----------------------------
	nvar(aa_trp) = 1; // starting number of variants for TRP
	variant_type( aav_base, aa_trp, 1 ) = true; // type of aa variant

//bk   variant1 ---------- TRP

	natoms(aa_trp,1) = 24; // number of total atoms
	nheavyatoms(aa_trp,1) = 14; // number of heavy atoms
	nchi(aa_trp,1) =  2; // number of chi angles

//bk   atom names TRP
	atom_name( 1,aa_trp,1) = " N  ";
	atom_name( 2,aa_trp,1) = " CA ";
	atom_name( 3,aa_trp,1) = " C  ";
	atom_name( 4,aa_trp,1) = " O  ";
	atom_name( 5,aa_trp,1) = " CB ";
	atom_name( 6,aa_trp,1) = " CG ";
	atom_name( 7,aa_trp,1) = " CD1";
	atom_name( 8,aa_trp,1) = " CD2";
	atom_name( 9,aa_trp,1) = " NE1";
	atom_name(10,aa_trp,1) = " CE2";
	atom_name(11,aa_trp,1) = " CE3";
	atom_name(12,aa_trp,1) = " CZ2";
	atom_name(13,aa_trp,1) = " CZ3";
	atom_name(14,aa_trp,1) = " CH2";
	atom_name(15,aa_trp,1) = " H  ";
	atom_name(16,aa_trp,1) = " HE1";
	atom_name(17,aa_trp,1) = " HD1";
	atom_name(18,aa_trp,1) = " HZ2";
	atom_name(19,aa_trp,1) = " HH2";
	atom_name(20,aa_trp,1) = " HZ3";
	atom_name(21,aa_trp,1) = " HE3";
	atom_name(22,aa_trp,1) = " HA ";
	atom_name(23,aa_trp,1) = "2HB ";
	atom_name(24,aa_trp,1) = "3HB ";

//jjh intra residue bonding
	nbonded_neighbors( 1, aa_trp, 1) = 2; // N
		bonded_neighbor(1, 1, aa_trp, 1) = 2; // N--CA
		bonded_neighbor(2, 1, aa_trp, 1) =15; // N--H
	nbonded_neighbors( 2, aa_trp, 1) = 4; // CA
		bonded_neighbor(1, 2, aa_trp, 1) = 1; // CA--N
		bonded_neighbor(2, 2, aa_trp, 1) = 3; // CA--C
		bonded_neighbor(3, 2, aa_trp, 1) = 5; // CA--CB
		bonded_neighbor(4, 2, aa_trp, 1) =22; // CA--HA
	nbonded_neighbors( 3, aa_trp, 1) = 2; // C
		bonded_neighbor(1, 3, aa_trp, 1) = 2; // C--CA
		bonded_neighbor(2, 3, aa_trp, 1) = 4; // C--O
	nbonded_neighbors( 4, aa_trp, 1) = 1; // O
		bonded_neighbor(1, 4, aa_trp, 1) = 3; // O--C
	nbonded_neighbors( 5, aa_trp, 1) = 4; // CB
		bonded_neighbor(1, 5, aa_trp, 1) = 2; // CB--CA
		bonded_neighbor(2, 5, aa_trp, 1) = 6; // CB--CG
		bonded_neighbor(3, 5, aa_trp, 1) =23; // CB--2HB
		bonded_neighbor(4, 5, aa_trp, 1) =24; // CB--3HB
	nbonded_neighbors( 6, aa_trp, 1) = 3; // CG
		bonded_neighbor(1, 6, aa_trp, 1) = 5; // CG--CB
		bonded_neighbor(2, 6, aa_trp, 1) = 7; // CG--CD1
		bonded_neighbor(3, 6, aa_trp, 1) = 8; // CG--CD2
	nbonded_neighbors( 7, aa_trp, 1) = 3; // CD1
		bonded_neighbor(1, 7, aa_trp, 1) = 6; // CD1--CG
		bonded_neighbor(2, 7, aa_trp, 1) = 9; // CD1--NE1
		bonded_neighbor(3, 7, aa_trp, 1) =17; // CD1--HD1
	nbonded_neighbors( 8, aa_trp, 1) = 3; // CD2
		bonded_neighbor(1, 8, aa_trp, 1) = 6; // CD2--CG
		bonded_neighbor(2, 8, aa_trp, 1) =10; // CD2--CE2
		bonded_neighbor(3, 8, aa_trp, 1) =11; // CD2--CE3
	nbonded_neighbors( 9, aa_trp, 1) = 3; // NE1
		bonded_neighbor(1, 9, aa_trp, 1) = 7; // NE1--CD1
		bonded_neighbor(2, 9, aa_trp, 1) =10; // NE1--CE2
		bonded_neighbor(3, 9, aa_trp, 1) =16; // NE1--HE1
	nbonded_neighbors( 10, aa_trp, 1) = 3; // CE2
		bonded_neighbor(1,10, aa_trp, 1) = 8; // CE2--CD2
		bonded_neighbor(2,10, aa_trp, 1) = 9; // CE2--NE1
		bonded_neighbor(3,10, aa_trp, 1) =12; // CE2--CZ2
	nbonded_neighbors( 11, aa_trp, 1) = 3; // CE3
		bonded_neighbor(1,11, aa_trp, 1) = 8; // CE3--CD2
		bonded_neighbor(2,11, aa_trp, 1) =13; // CE3--CZ3
		bonded_neighbor(3,11, aa_trp, 1) =21; // CE3--HE3
	nbonded_neighbors( 12, aa_trp, 1) = 3; // CZ2
		bonded_neighbor(1,12, aa_trp, 1) =10; // CZ2--CE2
		bonded_neighbor(2,12, aa_trp, 1) =14; // CZ2--CH2
		bonded_neighbor(3,12, aa_trp, 1) =18; // CZ2--HZ2
	nbonded_neighbors( 13, aa_trp, 1) = 3; // CZ3
		bonded_neighbor(1,13, aa_trp, 1) =11; // CZ3--CE3
		bonded_neighbor(2,13, aa_trp, 1) =14; // CZ3--CH2
		bonded_neighbor(3,13, aa_trp, 1) =20; // CZ3--HZ3
	nbonded_neighbors( 14, aa_trp, 1) = 3; // CH2
		bonded_neighbor(1,14, aa_trp, 1) =12; // CH2--CZ2
		bonded_neighbor(2,14, aa_trp, 1) =13; // CH2--CZ3
		bonded_neighbor(3,14, aa_trp, 1) =19; // CH2--HH2
	nbonded_neighbors( 15, aa_trp, 1) = 1; // H
		bonded_neighbor(1,15, aa_trp, 1) = 1; // H--N
	nbonded_neighbors( 16, aa_trp, 1) = 1; // HE1
		bonded_neighbor(1,16, aa_trp, 1) = 9; // HE1--NE1
	nbonded_neighbors( 17, aa_trp, 1) = 1; // HD1
		bonded_neighbor(1,17, aa_trp, 1) = 7; // HD1--CD1
	nbonded_neighbors( 18, aa_trp, 1) = 1; // HZ2
		bonded_neighbor(1,18, aa_trp, 1) =12; // HZ2--CZ2
	nbonded_neighbors( 19, aa_trp, 1) = 1; // HH2
		bonded_neighbor(1,19, aa_trp, 1) =14; // HH2--CH2
	nbonded_neighbors( 20, aa_trp, 1) = 1; // HZ3
		bonded_neighbor(1,20, aa_trp, 1) =13; // HZ3--CZ3
	nbonded_neighbors( 21, aa_trp, 1) = 1; // HE3
		bonded_neighbor(1,21, aa_trp, 1) =11; // HE3--CE3
	nbonded_neighbors( 22, aa_trp, 1) = 1; // HA
		bonded_neighbor(1,22, aa_trp, 1) = 2; // HA--CA
	nbonded_neighbors( 23, aa_trp, 1) = 1; //2HB
		bonded_neighbor(1,23, aa_trp, 1) = 5; //2HB--CB
	nbonded_neighbors( 24, aa_trp, 1) = 1; //3HB
		bonded_neighbor(1,24, aa_trp, 1) = 5; //3HB--CB

//jjh get the angles automatically
	get_angles_from_bonds(aa_trp, 1);

//chu   base atom numbers TRP
	atom_base( 1,aa_trp,1) =  2; //  N  -> CA
	atom_base( 2,aa_trp,1) =  1; //  CA -> N
	atom_base( 3,aa_trp,1) =  2; //  C  -> CA
	atom_base( 4,aa_trp,1) =  3; //  O  -> C
	atom_base( 5,aa_trp,1) =  2; //  CB -> CA
	atom_base( 6,aa_trp,1) =  5; //  CG -> CB
	atom_base( 7,aa_trp,1) =  6; //  CD1-> CG
	atom_base( 8,aa_trp,1) =  6; //  CD2-> CG
	atom_base( 9,aa_trp,1) =  7; //  NE1-> CD1
	atom_base(10,aa_trp,1) =  8; //  CE2-> CD2 (not consider NE1<->CE2)
  abase2(10,aa_trp,1) = 9; // for completeness
	atom_base(11,aa_trp,1) =  8; //  CE3-> CD2
	atom_base(12,aa_trp,1) = 10; //  CZ2-> CE2
	atom_base(13,aa_trp,1) = 11; //  CZ3-> CE3
	atom_base(14,aa_trp,1) = 12; //  CH2-> CZ2 (not consider CH2->CZ3)
  abase2(14,aa_trp,1) = 13; // for completeness
	atom_base(15,aa_trp,1) =  1; //  H   on  N
	atom_base(16,aa_trp,1) =  9; //  HE1 on  NE1
	atom_base(17,aa_trp,1) =  7; //  HD1 on  CD1
	atom_base(18,aa_trp,1) = 12; //  HZ2 on  CZ2
	atom_base(19,aa_trp,1) = 14; //  HH2 on  CH2
	atom_base(20,aa_trp,1) = 13; //  HZ3 on  CZ3
	atom_base(21,aa_trp,1) = 11; //  HE3 on  CE3
	atom_base(22,aa_trp,1) =  2; //  HA  on  CA
	atom_base(23,aa_trp,1) =  5; // 2HB  on  CB
	atom_base(24,aa_trp,1) =  5; // 3HB  on  CB

//bk   atom type numbers TRP
	fullatom_type( 1,aa_trp,1) = 17; // Nbb    N
	fullatom_type( 2,aa_trp,1) = 18; // CAbb   CA
	fullatom_type( 3,aa_trp,1) = 19; // CObb   C
	fullatom_type( 4,aa_trp,1) = 20; // OCbb   O
	fullatom_type( 5,aa_trp,1) =  4; // CH2    CB
	fullatom_type( 6,aa_trp,1) =  6; // aroC   CG
	fullatom_type( 7,aa_trp,1) =  6; // aroC   CD1
	fullatom_type( 8,aa_trp,1) =  6; // aroC   CD2
	fullatom_type( 9,aa_trp,1) =  7; // Ntrp   NE1
	fullatom_type(10,aa_trp,1) =  6; // aroC   CE2
	fullatom_type(11,aa_trp,1) =  6; // aroC   CE3
	fullatom_type(12,aa_trp,1) =  6; // aroC   CZ2
	fullatom_type(13,aa_trp,1) =  6; // aroC   CZ3
	fullatom_type(14,aa_trp,1) =  6; // aroC   CH2
	fullatom_type(15,aa_trp,1) = 25; // HNbb   H
	fullatom_type(16,aa_trp,1) = 22; // Hpol   HE1
	fullatom_type(17,aa_trp,1) = 24; // Haro   HD1
	fullatom_type(18,aa_trp,1) = 24; // Haro   HZ2
	fullatom_type(19,aa_trp,1) = 24; // Haro   HH2
	fullatom_type(20,aa_trp,1) = 24; // Haro   HZ3
	fullatom_type(21,aa_trp,1) = 24; // Haro   HE3
	fullatom_type(22,aa_trp,1) = 23; // Hapo   HA
	fullatom_type(23,aa_trp,1) = 23; // Hapo  2HB
	fullatom_type(24,aa_trp,1) = 23; // Hapo  3HB

//cmd  atomic charge TRP
	atomic_charge( 1,aa_trp,1) = -0.47; //    N
	atomic_charge( 2,aa_trp,1) =  0.07; //    CA
	atomic_charge( 3,aa_trp,1) =  0.51; //    C
	atomic_charge( 4,aa_trp,1) = -0.51; //    O
	atomic_charge( 5,aa_trp,1) = -0.18; //    CB
	atomic_charge( 6,aa_trp,1) = -0.03; //    CG
	atomic_charge( 7,aa_trp,1) =  0.035; //    CD1
	atomic_charge( 8,aa_trp,1) = -0.02; //    CD2
	atomic_charge( 9,aa_trp,1) = -0.61; //    NE1
	atomic_charge(10,aa_trp,1) =  0.13; //    CE2
	atomic_charge(11,aa_trp,1) = -0.115; //    CE3
	atomic_charge(12,aa_trp,1) = -0.115; //    CZ2
	atomic_charge(13,aa_trp,1) = -0.115; //    CZ3
	atomic_charge(14,aa_trp,1) = -0.115; //    CH2
	atomic_charge(15,aa_trp,1) =  0.31; //    H
	atomic_charge(16,aa_trp,1) =  0.38; //    HE1
	atomic_charge(17,aa_trp,1) =  0.115; //    HD1
	atomic_charge(18,aa_trp,1) =  0.115; //    HZ2
	atomic_charge(19,aa_trp,1) =  0.115; //    HH2
	atomic_charge(20,aa_trp,1) =  0.115; //    HZ3
	atomic_charge(21,aa_trp,1) =  0.115; //    HE3
	atomic_charge(22,aa_trp,1) =  0.09; //    HA
	atomic_charge(23,aa_trp,1) =  0.09; //   2HB
	atomic_charge(24,aa_trp,1) =  0.09; //   3HB

	assert_total_charge( aa_trp, 1, 0.0);

//cmd  first sidechain atom
	first_scatom( aa_trp,1) = 5; //CB

//bk   template atoms used for placing atoms TRP

//bk   template for building  HE1
	ta(1,16,aa_trp,1) =  9; //   NE1
	ta(2,16,aa_trp,1) =  7; //   CD1
	ta(3,16,aa_trp,1) = 10; //   CE2

//bk   template for building  HD1
	ta(1,17,aa_trp,1) =  7; //   CD1
	ta(2,17,aa_trp,1) =  6; //   CG
	ta(3,17,aa_trp,1) =  9; //   NE1

//bk   template for building  HZ2
	ta(1,18,aa_trp,1) = 12; //   CZ2
	ta(2,18,aa_trp,1) = 10; //   CE2
	ta(3,18,aa_trp,1) = 14; //   CH2

//bk   template for building  HH2
	ta(1,19,aa_trp,1) = 14; //   CH2
	ta(2,19,aa_trp,1) = 12; //   CZ2
	ta(3,19,aa_trp,1) = 13; //   CZ3

//bk   template for building  HZ3
	ta(1,20,aa_trp,1) = 13; //   CZ3
	ta(2,20,aa_trp,1) = 11; //   CE3
	ta(3,20,aa_trp,1) = 14; //   CH2

//bk   template for building  HE3
	ta(1,21,aa_trp,1) = 11; //   CE3
	ta(2,21,aa_trp,1) =  8; //   CD2
	ta(3,21,aa_trp,1) = 13; //   CZ3

//bk   template for building  HA
	ta(1,22,aa_trp,1) =  2; //   CA
	ta(2,22,aa_trp,1) =  1; //   N
	ta(3,22,aa_trp,1) =  3; //   C

//bk   template for building 2HB
	ta(1,23,aa_trp,1) =  5; //   CB
	ta(2,23,aa_trp,1) =  2; //   CA
	ta(3,23,aa_trp,1) =  6; //   CG

//bk   template for building 3HB
	ta(1,24,aa_trp,1) =  5; //   CB
	ta(2,24,aa_trp,1) =  2; //   CA
	ta(3,24,aa_trp,1) =  6; //   CG

//bk   chi angles required to build atoms TRP

//bk   chi angles needed for building  N
	chi_required(1, 1,aa_trp,1) = false;
	chi_required(2, 1,aa_trp,1) = false;
	chi_required(3, 1,aa_trp,1) = false;
	chi_required(4, 1,aa_trp,1) = false;

//bk   chi angles needed for building  CA
	chi_required(1, 2,aa_trp,1) = false;
	chi_required(2, 2,aa_trp,1) = false;
	chi_required(3, 2,aa_trp,1) = false;
	chi_required(4, 2,aa_trp,1) = false;

//bk   chi angles needed for building  C
	chi_required(1, 3,aa_trp,1) = false;
	chi_required(2, 3,aa_trp,1) = false;
	chi_required(3, 3,aa_trp,1) = false;
	chi_required(4, 3,aa_trp,1) = false;

//bk   chi angles needed for building  O
	chi_required(1, 4,aa_trp,1) = false;
	chi_required(2, 4,aa_trp,1) = false;
	chi_required(3, 4,aa_trp,1) = false;
	chi_required(4, 4,aa_trp,1) = false;

//bk   chi angles needed for building  CB
	chi_required(1, 5,aa_trp,1) = false;
	chi_required(2, 5,aa_trp,1) = false;
	chi_required(3, 5,aa_trp,1) = false;
	chi_required(4, 5,aa_trp,1) = false;

//bk   chi angles needed for building  CG
	chi_required(1, 6,aa_trp,1) =  true;
	chi_required(2, 6,aa_trp,1) = false;
	chi_required(3, 6,aa_trp,1) = false;
	chi_required(4, 6,aa_trp,1) = false;

//bk   chi angles needed for building  CD1
	chi_required(1, 7,aa_trp,1) =  true;
	chi_required(2, 7,aa_trp,1) =  true;
	chi_required(3, 7,aa_trp,1) = false;
	chi_required(4, 7,aa_trp,1) = false;

//bk   chi angles needed for building  CD2
	chi_required(1, 8,aa_trp,1) =  true;
	chi_required(2, 8,aa_trp,1) =  true;
	chi_required(3, 8,aa_trp,1) = false;
	chi_required(4, 8,aa_trp,1) = false;

//bk   chi angles needed for building  NE1
	chi_required(1, 9,aa_trp,1) =  true;
	chi_required(2, 9,aa_trp,1) =  true;
	chi_required(3, 9,aa_trp,1) = false;
	chi_required(4, 9,aa_trp,1) = false;

//bk   chi angles needed for building  CE2
	chi_required(1,10,aa_trp,1) =  true;
	chi_required(2,10,aa_trp,1) =  true;
	chi_required(3,10,aa_trp,1) = false;
	chi_required(4,10,aa_trp,1) = false;

//bk   chi angles needed for building  CE3
	chi_required(1,11,aa_trp,1) =  true;
	chi_required(2,11,aa_trp,1) =  true;
	chi_required(3,11,aa_trp,1) = false;
	chi_required(4,11,aa_trp,1) = false;

//bk   chi angles needed for building  CZ2
	chi_required(1,12,aa_trp,1) =  true;
	chi_required(2,12,aa_trp,1) =  true;
	chi_required(3,12,aa_trp,1) = false;
	chi_required(4,12,aa_trp,1) = false;

//bk   chi angles needed for building  CZ3
	chi_required(1,13,aa_trp,1) =  true;
	chi_required(2,13,aa_trp,1) =  true;
	chi_required(3,13,aa_trp,1) = false;
	chi_required(4,13,aa_trp,1) = false;

//bk   chi angles needed for building  CH2
	chi_required(1,14,aa_trp,1) =  true;
	chi_required(2,14,aa_trp,1) =  true;
	chi_required(3,14,aa_trp,1) = false;
	chi_required(4,14,aa_trp,1) = false;

//bk   chi angles needed for building  H
	chi_required(1,15,aa_trp,1) = false;
	chi_required(2,15,aa_trp,1) = false;
	chi_required(3,15,aa_trp,1) = false;
	chi_required(4,15,aa_trp,1) = false;

//bk   chi angles needed for building  HE1
	chi_required(1,16,aa_trp,1) =  true;
	chi_required(2,16,aa_trp,1) =  true;
	chi_required(3,16,aa_trp,1) = false;
	chi_required(4,16,aa_trp,1) = false;

//bk   chi angles needed for building  HD1
	chi_required(1,17,aa_trp,1) =  true;
	chi_required(2,17,aa_trp,1) =  true;
	chi_required(3,17,aa_trp,1) = false;
	chi_required(4,17,aa_trp,1) = false;

//bk   chi angles needed for building  HZ2
	chi_required(1,18,aa_trp,1) =  true;
	chi_required(2,18,aa_trp,1) =  true;
	chi_required(3,18,aa_trp,1) = false;
	chi_required(4,18,aa_trp,1) = false;

//bk   chi angles needed for building  HH2
	chi_required(1,19,aa_trp,1) =  true;
	chi_required(2,19,aa_trp,1) =  true;
	chi_required(3,19,aa_trp,1) = false;
	chi_required(4,19,aa_trp,1) = false;

//bk   chi angles needed for building  HZ3
	chi_required(1,20,aa_trp,1) =  true;
	chi_required(2,20,aa_trp,1) =  true;
	chi_required(3,20,aa_trp,1) = false;
	chi_required(4,20,aa_trp,1) = false;

//bk   chi angles needed for building  HE3
	chi_required(1,21,aa_trp,1) =  true;
	chi_required(2,21,aa_trp,1) =  true;
	chi_required(3,21,aa_trp,1) = false;
	chi_required(4,21,aa_trp,1) = false;

//bk   chi angles needed for building  HA
	chi_required(1,22,aa_trp,1) = false;
	chi_required(2,22,aa_trp,1) = false;
	chi_required(3,22,aa_trp,1) = false;
	chi_required(4,22,aa_trp,1) = false;

//bk   chi angles needed for building 2HB
	chi_required(1,23,aa_trp,1) =  true;
	chi_required(2,23,aa_trp,1) = false;
	chi_required(3,23,aa_trp,1) = false;
	chi_required(4,23,aa_trp,1) = false;

//bk   chi angles needed for building 3HB
	chi_required(1,24,aa_trp,1) =  true;
	chi_required(2,24,aa_trp,1) = false;
	chi_required(3,24,aa_trp,1) = false;
	chi_required(4,24,aa_trp,1) = false;

//bk   four atoms that define chi angle  1
	chi_atoms( 1, 1,aa_trp, 1) =  1; //   N
	chi_atoms( 2, 1,aa_trp, 1) =  2; //   CA
	chi_atoms( 3, 1,aa_trp, 1) =  5; //   CB
	chi_atoms( 4, 1,aa_trp, 1) =  6; //   CG
//bk   four atoms that define chi angle  2
	chi_atoms( 1, 2,aa_trp, 1) =  2; //   CA
	chi_atoms( 2, 2,aa_trp, 1) =  5; //   CB
	chi_atoms( 3, 2,aa_trp, 1) =  6; //   CG
	chi_atoms( 4, 2,aa_trp, 1) =  7; //   CD1

	HNpos(aa_trp,1) = 15; // atom number for backbone HN
	HApos(aa_trp,1) = 22; // atom number for backbone HA
	nH_polar(aa_trp,1) =  2; // number of polar hydrogens
	nH_aromatic(aa_trp,1) =  5; // number of aromatic hydrogens
	nH_apolar(aa_trp,1) =  3; // number of apolar hydrogens
	nacceptors(aa_trp,1) =  1; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1,aa_trp,1) = 15; //  H
	Hpos_polar( 2,aa_trp,1) = 16; //  HE1

//bk   atom number for aromatic hydrogens
	Hpos_aromatic( 1,aa_trp,1) = 17; //  HD1
	Hpos_aromatic( 2,aa_trp,1) = 18; //  HZ2
	Hpos_aromatic( 3,aa_trp,1) = 19; //  HH2
	Hpos_aromatic( 4,aa_trp,1) = 20; //  HZ3
	Hpos_aromatic( 5,aa_trp,1) = 21; //  HE3

//bk   atom number for apolar hydrogens
	Hpos_apolar( 1,aa_trp,1) = 22; //  HA
	Hpos_apolar( 2,aa_trp,1) = 23; // 2HB
	Hpos_apolar( 3,aa_trp,1) = 24; // 3HB

//bk   acceptor positions
	accpt_pos( 1,aa_trp, 1) =  4; //  O

//bk   hydrogens on each atom

	nhydrogens_on_atm( 1,aa_trp, 1) =  1; //  N
	hydrogens_on_atm( 1, 1,aa_trp, 1) = 15; //  H

	nhydrogens_on_atm( 2,aa_trp, 1) =  1; //  CA
	hydrogens_on_atm( 1, 2,aa_trp, 1) = 22; //  HA

	nhydrogens_on_atm( 5,aa_trp, 1) =  2; //  CB
	hydrogens_on_atm( 1, 5,aa_trp, 1) = 23; // 2HB
	hydrogens_on_atm( 2, 5,aa_trp, 1) = 24; // 3HB

	nhydrogens_on_atm( 7,aa_trp, 1) =  1; //  CD1
	hydrogens_on_atm( 1, 7,aa_trp, 1) = 17; //  HD1

	nhydrogens_on_atm( 9,aa_trp, 1) =  1; //  NE1
	hydrogens_on_atm( 1, 9,aa_trp, 1) = 16; //  HE1

	nhydrogens_on_atm(11,aa_trp, 1) =  1; //  CE3
	hydrogens_on_atm( 1,11,aa_trp, 1) = 21; //  HE3

	nhydrogens_on_atm(12,aa_trp, 1) =  1; //  CZ2
	hydrogens_on_atm( 1,12,aa_trp, 1) = 18; //  HZ2

	nhydrogens_on_atm(13,aa_trp, 1) =  1; //  CZ3
	hydrogens_on_atm( 1,13,aa_trp, 1) = 20; //  HZ3

	nhydrogens_on_atm(14,aa_trp, 1) =  1; //  CH2
	hydrogens_on_atm( 1,14,aa_trp, 1) = 19; //  HH2

//bk   template coordinates for the amino acid
	icoor( 1, 1,aa_trp, 1) =    0.0000; //   N
	icoor( 2, 1,aa_trp, 1) =   17.2040; //   N
	icoor( 3, 1,aa_trp, 1) =   65.5540; //   N
	icoor( 1, 2,aa_trp, 1) =    0.0000; //   CA
	icoor( 2, 2,aa_trp, 1) =   18.0180; //   CA
	icoor( 3, 2,aa_trp, 1) =   66.7640; //   CA
	icoor( 1, 3,aa_trp, 1) =    0.0000; //   C
	icoor( 2, 3,aa_trp, 1) =   17.1480; //   C
	icoor( 3, 3,aa_trp, 1) =   68.0140; //   C
	icoor( 1, 4,aa_trp, 1) =    0.0020; //   O
	icoor( 2, 4,aa_trp, 1) =   15.9210; //   O
	icoor( 3, 4,aa_trp, 1) =   67.9270; //   O
	icoor( 1, 5,aa_trp, 1) =   -1.2100; //   CB
	icoor( 2, 5,aa_trp, 1) =   18.9540; //   CB
	icoor( 3, 5,aa_trp, 1) =   66.7770; //   CB
	icoor( 1, 6,aa_trp, 1) =   -1.2790; //   CG
	icoor( 2, 6,aa_trp, 1) =   19.8260; //   CG
	icoor( 3, 6,aa_trp, 1) =   67.9940; //   CG
	icoor( 1, 7,aa_trp, 1) =   -2.2370; //   CD1
	icoor( 2, 7,aa_trp, 1) =   20.7520; //   CD1
	icoor( 3, 7,aa_trp, 1) =   68.2800; //   CD1
	icoor( 1, 8,aa_trp, 1) =   -0.3540; //   CD2
	icoor( 2, 8,aa_trp, 1) =   19.8530; //   CD2
	icoor( 3, 8,aa_trp, 1) =   69.0880; //   CD2
	icoor( 1, 9,aa_trp, 1) =   -1.9670; //   NE1
	icoor( 2, 9,aa_trp, 1) =   21.3560; //   NE1
	icoor( 3, 9,aa_trp, 1) =   69.4830; //   NE1
	icoor( 1,10,aa_trp, 1) =   -0.8150; //   CE2
	icoor( 2,10,aa_trp, 1) =   20.8190; //   CE2
	icoor( 3,10,aa_trp, 1) =   70.0000; //   CE2
	icoor( 1,11,aa_trp, 1) =    0.8210; //   CE3
	icoor( 2,11,aa_trp, 1) =   19.1510; //   CE3
	icoor( 3,11,aa_trp, 1) =   69.3840; //   CE3
	icoor( 1,12,aa_trp, 1) =   -0.1480; //   CZ2
	icoor( 2,12,aa_trp, 1) =   21.1040; //   CZ2
	icoor( 3,12,aa_trp, 1) =   71.1810; //   CZ2
	icoor( 1,13,aa_trp, 1) =    1.4890; //   CZ3
	icoor( 2,13,aa_trp, 1) =   19.4360; //   CZ3
	icoor( 3,13,aa_trp, 1) =   70.5690; //   CZ3
	icoor( 1,14,aa_trp, 1) =    1.0180; //   CH2
	icoor( 2,14,aa_trp, 1) =   20.3840; //   CH2
	icoor( 3,14,aa_trp, 1) =   71.4420; //   CH2
	icoor( 1,15,aa_trp, 1) =   -0.0380; //   H
	icoor( 2,15,aa_trp, 1) =   16.1990; //   H
	icoor( 3,15,aa_trp, 1) =   65.6530; //   H
	icoor( 1,16,aa_trp, 1) =   -2.5240; //   HE1
	icoor( 2,16,aa_trp, 1) =   22.0780; //   HE1
	icoor( 3,16,aa_trp, 1) =   69.9170; //   HE1
	icoor( 1,17,aa_trp, 1) =   -3.0370; //   HD1
	icoor( 2,17,aa_trp, 1) =   20.8740; //   HD1
	icoor( 3,17,aa_trp, 1) =   67.5520; //   HD1
	icoor( 1,18,aa_trp, 1) =   -0.5570; //   HZ2
	icoor( 2,18,aa_trp, 1) =   21.8650; //   HZ2
	icoor( 3,18,aa_trp, 1) =   71.8460; //   HZ2
	icoor( 1,19,aa_trp, 1) =    1.5720; //   HH2
	icoor( 2,19,aa_trp, 1) =   20.5770; //   HH2
	icoor( 3,19,aa_trp, 1) =   72.3610; //   HH2
	icoor( 1,20,aa_trp, 1) =    2.4030; //   HZ3
	icoor( 2,20,aa_trp, 1) =   18.8830; //   HZ3
	icoor( 3,20,aa_trp, 1) =   70.7870; //   HZ3
	icoor( 1,21,aa_trp, 1) =    1.2370; //   HE3
	icoor( 2,21,aa_trp, 1) =   18.3870; //   HE3
	icoor( 3,21,aa_trp, 1) =   68.7280; //   HE3
	icoor( 1,22,aa_trp, 1) =    0.9070; //   HA
	icoor( 2,22,aa_trp, 1) =   18.6210; //   HA
	icoor( 3,22,aa_trp, 1) =   66.8040; //   HA
	icoor( 1,23,aa_trp, 1) =   -1.1800; //  2HB
	icoor( 2,23,aa_trp, 1) =   19.6220; //  2HB
	icoor( 3,23,aa_trp, 1) =   65.9160; //  2HB
	icoor( 1,24,aa_trp, 1) =   -2.1330; //  3HB
	icoor( 2,24,aa_trp, 1) =   18.3750; //  3HB
	icoor( 3,24,aa_trp, 1) =   66.7550; //  3HB

	// H positions from REDUCE
	if( param_pack::packer_logical::reduce_templates ) {
		icoor( 1 , 15 , aa_trp , 1) = -0.000 ; // H
		icoor( 2 , 15 , aa_trp , 1) = 16.236 ; // H
		icoor( 3 , 15 , aa_trp , 1) = 65.803 ; // H
		icoor( 1 , 22 , aa_trp , 1) = 0.920 ; // HA
		icoor( 2 , 22 , aa_trp , 1) = 18.620 ; // HA
		icoor( 3 , 22 , aa_trp , 1) = 66.765 ; // HA
		icoor( 1 , 23 , aa_trp , 1) = -1.180 ; // 1HB
		icoor( 2 , 23 , aa_trp , 1) = 19.591 ; // 1HB
		icoor( 3 , 23 , aa_trp , 1) = 65.881 ; // 1HB
		icoor( 1 , 24 , aa_trp , 1) = -2.129 ; // 2HB
		icoor( 2 , 24 , aa_trp , 1) = 18.353 ; // 2HB
		icoor( 3 , 24 , aa_trp , 1) = 66.714 ; // 2HB
		icoor( 1 , 17 , aa_trp , 1) = -3.101 ; // HD1
		icoor( 2 , 17 , aa_trp , 1) = 20.982 ; // HD1
		icoor( 3 , 17 , aa_trp , 1) = 67.640 ; // HD1
		icoor( 1 , 16 , aa_trp , 1) = -2.520 ; // HE1
		icoor( 2 , 16 , aa_trp , 1) = 22.070 ; // HE1
		icoor( 3 , 16 , aa_trp , 1) = 69.912 ; // HE1
		icoor( 1 , 21 , aa_trp , 1) = 1.208 ; // HE3
		icoor( 2 , 21 , aa_trp , 1) = 18.389 ; // HE3
		icoor( 3 , 21 , aa_trp , 1) = 68.692 ; // HE3
		icoor( 1 , 18 , aa_trp , 1) = -0.522 ; // HZ2
		icoor( 2 , 18 , aa_trp , 1) = 21.865 ; // HZ2
		icoor( 3 , 18 , aa_trp , 1) = 71.882 ; // HZ2
		icoor( 1 , 20 , aa_trp , 1) = 2.413 ; // HZ3
		icoor( 2 , 20 , aa_trp , 1) = 18.891 ; // HZ3
		icoor( 3 , 20 , aa_trp , 1) = 70.812 ; // HZ3
		icoor( 1 , 19 , aa_trp , 1) = 1.574 ; // HH2
		icoor( 2 , 19 , aa_trp , 1) = 20.581 ; // HH2
		icoor( 3 , 19 , aa_trp , 1) = 72.371 ; // HH2
	}

//bq create N and Cterm variants
	create_termini_variants( aa_trp );


//bk   create variants that have h2o connected to the backbone nh and carbonyl
	if ( explicit_h2o ) {
		create_backbone_h2o_variants(aa_trp);
		create_sidechain_h2o_variants(aa_trp);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_tyr
///
/// @brief arrays filled to define the proerties of the amino acid Tyrosine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// aa_tyr - index number for TYR (20)
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_tyr()
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

//bk   Parameters for TYR -----------------------------
	nvar(aa_tyr) = 1; // starting number of variants for TYR
	variant_type( aav_base, aa_tyr, 1 ) = true; // type of aa variant

//bk   variant1 ---------- TYR

	natoms(aa_tyr,1) = 21; // number of total atoms
	nheavyatoms(aa_tyr,1) = 12; // number of heavy atoms
	nchi(aa_tyr,1) =  3; // number of chi angles

//bk   atom names TYR
	atom_name( 1,aa_tyr,1) = " N  ";
	atom_name( 2,aa_tyr,1) = " CA ";
	atom_name( 3,aa_tyr,1) = " C  ";
	atom_name( 4,aa_tyr,1) = " O  ";
	atom_name( 5,aa_tyr,1) = " CB ";
	atom_name( 6,aa_tyr,1) = " CG ";
	atom_name( 7,aa_tyr,1) = " CD1";
	atom_name( 8,aa_tyr,1) = " CD2";
	atom_name( 9,aa_tyr,1) = " CE1";
	atom_name(10,aa_tyr,1) = " CE2";
	atom_name(11,aa_tyr,1) = " CZ ";
	atom_name(12,aa_tyr,1) = " OH ";
	atom_name(13,aa_tyr,1) = " H  ";
	atom_name(14,aa_tyr,1) = " HH ";
	atom_name(15,aa_tyr,1) = " HD1";
	atom_name(16,aa_tyr,1) = " HE1";
	atom_name(17,aa_tyr,1) = " HE2";
	atom_name(18,aa_tyr,1) = " HD2";
	atom_name(19,aa_tyr,1) = " HA ";
	atom_name(20,aa_tyr,1) = "2HB ";
	atom_name(21,aa_tyr,1) = "3HB ";

//jjh intra residue bonding
	nbonded_neighbors( 1, aa_tyr, 1) = 2; // N
		bonded_neighbor(1, 1, aa_tyr, 1) = 2; // N--CA
		bonded_neighbor(2, 1, aa_tyr, 1) =13; // N--H
	nbonded_neighbors( 2, aa_tyr, 1) = 4; // CA
		bonded_neighbor(1, 2, aa_tyr, 1) = 1; // CA--N
		bonded_neighbor(2, 2, aa_tyr, 1) = 3; // CA--C
		bonded_neighbor(3, 2, aa_tyr, 1) = 5; // CA--CB
		bonded_neighbor(4, 2, aa_tyr, 1) =19; // CA--HA
	nbonded_neighbors( 3, aa_tyr, 1) = 2; // C
		bonded_neighbor(1, 3, aa_tyr, 1) = 2; // C--CA
		bonded_neighbor(2, 3, aa_tyr, 1) = 4; // C--O
	nbonded_neighbors( 4, aa_tyr, 1) = 1; // O
		bonded_neighbor(1, 4, aa_tyr, 1) = 3; // O--C
	nbonded_neighbors( 5, aa_tyr, 1) = 4; // CB
		bonded_neighbor(1, 5, aa_tyr, 1) = 2; // CB--CA
		bonded_neighbor(2, 5, aa_tyr, 1) = 6; // CB--CG
		bonded_neighbor(3, 5, aa_tyr, 1) =20; // CB--2HB
		bonded_neighbor(4, 5, aa_tyr, 1) =21; // CB--3HB
	nbonded_neighbors( 6, aa_tyr, 1) = 3; // CG
		bonded_neighbor(1, 6, aa_tyr, 1) = 5; // CG--CB
		bonded_neighbor(2, 6, aa_tyr, 1) = 7; // CG--CD1
		bonded_neighbor(3, 6, aa_tyr, 1) = 8; // CG--CD2
	nbonded_neighbors( 7, aa_tyr, 1) = 3; // CD1
		bonded_neighbor(1, 7, aa_tyr, 1) = 6; // CD1--CG
		bonded_neighbor(2, 7, aa_tyr, 1) = 9; // CD1--CE1
		bonded_neighbor(3, 7, aa_tyr, 1) =15; // CD1--HD1
	nbonded_neighbors( 8, aa_tyr, 1) = 3; // CD2
		bonded_neighbor(1, 8, aa_tyr, 1) = 6; // CD2--CG
		bonded_neighbor(2, 8, aa_tyr, 1) =10; // CD2--CE2
		bonded_neighbor(3, 8, aa_tyr, 1) =18; // CD2--HD2
	nbonded_neighbors( 9, aa_tyr, 1) = 3; // CE1
		bonded_neighbor(1, 9, aa_tyr, 1) = 7; // CE1--CD1
		bonded_neighbor(2, 9, aa_tyr, 1) =11; // CE1--CZ
		bonded_neighbor(3, 9, aa_tyr, 1) =16; // CE1--HE1
	nbonded_neighbors( 10, aa_tyr, 1) = 3; // CE2
		bonded_neighbor(1,10, aa_tyr, 1) = 8; // CE2--CD2
		bonded_neighbor(2,10, aa_tyr, 1) =11; // CE2--CZ
		bonded_neighbor(3,10, aa_tyr, 1) =17; // CE2--HE2
	nbonded_neighbors( 11, aa_tyr, 1) = 3; // CZ
		bonded_neighbor(1,11, aa_tyr, 1) = 9; // CZ--CE1
		bonded_neighbor(2,11, aa_tyr, 1) =10; // CZ--CE2
		bonded_neighbor(3,11, aa_tyr, 1) =12; // CZ--OH
	nbonded_neighbors( 12, aa_tyr, 1) = 2; // OH
		bonded_neighbor(1,12, aa_tyr, 1) =11; // OH--CZ
		bonded_neighbor(2,12, aa_tyr, 1) =14; // OH--HH
	nbonded_neighbors( 13, aa_tyr, 1) = 1; // H
		bonded_neighbor(1,13, aa_tyr, 1) = 1; // H--N
	nbonded_neighbors( 14, aa_tyr, 1) = 1; // HH
		bonded_neighbor(1,14, aa_tyr, 1) =12; // HH--OH
	nbonded_neighbors( 15, aa_tyr, 1) = 1; // HD1
		bonded_neighbor(1,15, aa_tyr, 1) = 7; // HD1--CD1
	nbonded_neighbors( 16, aa_tyr, 1) = 1; // HE1
		bonded_neighbor(1,16, aa_tyr, 1) = 9; // HE1--CE1
	nbonded_neighbors( 17, aa_tyr, 1) = 1; // HE2
		bonded_neighbor(1,17, aa_tyr, 1) =10; // HE2--CE2
	nbonded_neighbors( 18, aa_tyr, 1) = 1; // HD2
		bonded_neighbor(1,18, aa_tyr, 1) = 8; // HD2--CD2
	nbonded_neighbors( 19, aa_tyr, 1) = 1; // HA
		bonded_neighbor(1,19, aa_tyr, 1) = 2; // HA--CA
	nbonded_neighbors( 20, aa_tyr, 1) = 1; //2HB
		bonded_neighbor(1,20, aa_tyr, 1) = 5; //2HB--CB
	nbonded_neighbors( 21, aa_tyr, 1) = 1; //3HB
		bonded_neighbor(1,21, aa_tyr, 1) = 5; //3HB--CB

//jjh get the angles automatically
	get_angles_from_bonds(aa_tyr, 1);

//chu   base atom numbers TYR
	atom_base( 1, aa_tyr, 1) = 2; //  N  -> CA
	atom_base( 2, aa_tyr, 1) = 1; //  CA -> N
	atom_base( 3, aa_tyr, 1) = 2; //  C  -> CA
	atom_base( 4, aa_tyr, 1) = 3; //  O  -> C
	atom_base( 5, aa_tyr, 1) = 2; //  CB -> CA
	atom_base( 6, aa_tyr, 1) = 5; //  CG -> CB
	atom_base( 7, aa_tyr, 1) = 6; //  CD1-> CG
	atom_base( 8, aa_tyr, 1) = 6; //  CD2-> CG
	atom_base( 9, aa_tyr, 1) = 7; //  CE1-> CD1
	atom_base(10, aa_tyr, 1) = 8; //  CE2-> CD2
	atom_base(11, aa_tyr, 1) = 9; //  CZ -> CE1 (not consider CZ->CE2)
  abase2(11,aa_tyr,1) = 10; // for completeness
	atom_base(12,aa_tyr,1) = 11; //  OH  on CZ
  abase2(12,aa_tyr, 1) = 14; //  HH  base2 for  OH (SP3 hybrid)
	atom_base(13,aa_tyr,1) =  1; //  H   on  N
	atom_base(14,aa_tyr,1) = 12; //  HH  on  OH
	atom_base(15,aa_tyr,1) =  7; //  HD1 on  CD1
	atom_base(16,aa_tyr,1) =  9; //  HE1 on  CE1
	atom_base(17,aa_tyr,1) = 10; //  HE2 on  CE2
	atom_base(18,aa_tyr,1) =  8; //  HD2 on  CD2
	atom_base(19,aa_tyr,1) =  2; //  HA  on  CA
	atom_base(20,aa_tyr,1) =  5; // 2HB  on  CB
	atom_base(21,aa_tyr,1) =  5; // 3HB  on  CB
//bk   atom type numbers TYR
	fullatom_type( 1,aa_tyr,1) = 17; // Nbb    N
	fullatom_type( 2,aa_tyr,1) = 18; // CAbb   CA
	fullatom_type( 3,aa_tyr,1) = 19; // CObb   C
	fullatom_type( 4,aa_tyr,1) = 20; // OCbb   O
	fullatom_type( 5,aa_tyr,1) =  4; // CH2    CB
	fullatom_type( 6,aa_tyr,1) =  6; // aroC   CG
	fullatom_type( 7,aa_tyr,1) =  6; // aroC   CD1
	fullatom_type( 8,aa_tyr,1) =  6; // aroC   CD2
	fullatom_type( 9,aa_tyr,1) =  6; // aroC   CE1
	fullatom_type(10,aa_tyr,1) =  6; // aroC   CE2
	fullatom_type(11,aa_tyr,1) =  6; // aroC   CZ
	fullatom_type(12,aa_tyr,1) = 13; // OH     OH
	fullatom_type(13,aa_tyr,1) = 25; // HNbb   H
	fullatom_type(14,aa_tyr,1) = 22; // Hpol   HH
	fullatom_type(15,aa_tyr,1) = 24; // Haro   HD1
	fullatom_type(16,aa_tyr,1) = 24; // Haro   HE1
	fullatom_type(17,aa_tyr,1) = 24; // Haro   HE2
	fullatom_type(18,aa_tyr,1) = 24; // Haro   HD2
	fullatom_type(19,aa_tyr,1) = 23; // Hapo   HA
	fullatom_type(20,aa_tyr,1) = 23; // Hapo  2HB
	fullatom_type(21,aa_tyr,1) = 23; // Hapo  3HB

//cmd  atomic charge TYR
	atomic_charge( 1,aa_tyr,1) = -0.47; //    N
	atomic_charge( 2,aa_tyr,1) =  0.07; //    CA
	atomic_charge( 3,aa_tyr,1) =  0.51; //    C
	atomic_charge( 4,aa_tyr,1) = -0.51; //    O
	atomic_charge( 5,aa_tyr,1) = -0.18; //    CB
	atomic_charge( 6,aa_tyr,1) =  0.00; //    CG
	atomic_charge( 7,aa_tyr,1) = -0.115; //    CD1
	atomic_charge( 8,aa_tyr,1) = -0.115; //    CD2
	atomic_charge( 9,aa_tyr,1) = -0.115; //    CE1
	atomic_charge(10,aa_tyr,1) = -0.115; //    CE2
	atomic_charge(11,aa_tyr,1) =  0.11; //    CZ
	atomic_charge(12,aa_tyr,1) = -0.54; //    OH
	atomic_charge(13,aa_tyr,1) =  0.31; //    H
	atomic_charge(14,aa_tyr,1) =  0.43; //    HH
	atomic_charge(15,aa_tyr,1) =  0.115; //    HD1
	atomic_charge(16,aa_tyr,1) =  0.115; //    HE1
	atomic_charge(17,aa_tyr,1) =  0.115; //    HE2
	atomic_charge(18,aa_tyr,1) =  0.115; //    HD2
	atomic_charge(19,aa_tyr,1) =  0.09; //    HA
	atomic_charge(20,aa_tyr,1) =  0.09; //   2HB
	atomic_charge(21,aa_tyr,1) =  0.09; //   3HB

	assert_total_charge( aa_tyr, 1, 0.0);

//cmd  first sidechain atom
	first_scatom( aa_tyr,1) = 5; //CB

//bk   template atoms used for placing atoms TYR

//bk   template for building  HH
	ta(1,14,aa_tyr,1) = 12; //   OH
	ta(2,14,aa_tyr,1) = 11; //   CZ
	ta(3,14,aa_tyr,1) =  9; //   CE1

//bk   template for building  HD1
	ta(1,15,aa_tyr,1) =  7; //   CD1
	ta(2,15,aa_tyr,1) =  6; //   CG
	ta(3,15,aa_tyr,1) =  9; //   CE1

//bk   template for building  HE1
	ta(1,16,aa_tyr,1) =  9; //   CE1
	ta(2,16,aa_tyr,1) =  7; //   CD1
	ta(3,16,aa_tyr,1) = 11; //   CZ

//bk   template for building  HE2
	ta(1,17,aa_tyr,1) = 10; //   CE2
	ta(2,17,aa_tyr,1) =  8; //   CD2
	ta(3,17,aa_tyr,1) = 11; //   CZ

//bk   template for building  HD2
	ta(1,18,aa_tyr,1) =  8; //   CD2
	ta(2,18,aa_tyr,1) =  6; //   CG
	ta(3,18,aa_tyr,1) = 10; //   CE2

//bk   template for building  HA
	ta(1,19,aa_tyr,1) =  2; //   CA
	ta(2,19,aa_tyr,1) =  1; //   N
	ta(3,19,aa_tyr,1) =  3; //   C

//bk   template for building 2HB
	ta(1,20,aa_tyr,1) =  5; //   CB
	ta(2,20,aa_tyr,1) =  2; //   CA
	ta(3,20,aa_tyr,1) =  6; //   CG

//bk   template for building 3HB
	ta(1,21,aa_tyr,1) =  5; //   CB
	ta(2,21,aa_tyr,1) =  2; //   CA
	ta(3,21,aa_tyr,1) =  6; //   CG

//bk   chi angles required to build atoms TYR

//bk   chi angles needed for building  N
	chi_required(1, 1,aa_tyr,1) = false;
	chi_required(2, 1,aa_tyr,1) = false;
	chi_required(3, 1,aa_tyr,1) = false;
	chi_required(4, 1,aa_tyr,1) = false;

//bk   chi angles needed for building  CA
	chi_required(1, 2,aa_tyr,1) = false;
	chi_required(2, 2,aa_tyr,1) = false;
	chi_required(3, 2,aa_tyr,1) = false;
	chi_required(4, 2,aa_tyr,1) = false;

//bk   chi angles needed for building  C
	chi_required(1, 3,aa_tyr,1) = false;
	chi_required(2, 3,aa_tyr,1) = false;
	chi_required(3, 3,aa_tyr,1) = false;
	chi_required(4, 3,aa_tyr,1) = false;

//bk   chi angles needed for building  O
	chi_required(1, 4,aa_tyr,1) = false;
	chi_required(2, 4,aa_tyr,1) = false;
	chi_required(3, 4,aa_tyr,1) = false;
	chi_required(4, 4,aa_tyr,1) = false;

//bk   chi angles needed for building  CB
	chi_required(1, 5,aa_tyr,1) = false;
	chi_required(2, 5,aa_tyr,1) = false;
	chi_required(3, 5,aa_tyr,1) = false;
	chi_required(4, 5,aa_tyr,1) = false;

//bk   chi angles needed for building  CG
	chi_required(1, 6,aa_tyr,1) =  true;
	chi_required(2, 6,aa_tyr,1) = false;
	chi_required(3, 6,aa_tyr,1) = false;
	chi_required(4, 6,aa_tyr,1) = false;

//bk   chi angles needed for building  CD1
	chi_required(1, 7,aa_tyr,1) =  true;
	chi_required(2, 7,aa_tyr,1) =  true;
	chi_required(3, 7,aa_tyr,1) = false;
	chi_required(4, 7,aa_tyr,1) = false;

//bk   chi angles needed for building  CD2
	chi_required(1, 8,aa_tyr,1) =  true;
	chi_required(2, 8,aa_tyr,1) =  true;
	chi_required(3, 8,aa_tyr,1) = false;
	chi_required(4, 8,aa_tyr,1) = false;

//bk   chi angles needed for building  CE1
	chi_required(1, 9,aa_tyr,1) =  true;
	chi_required(2, 9,aa_tyr,1) =  true;
	chi_required(3, 9,aa_tyr,1) = false;
	chi_required(4, 9,aa_tyr,1) = false;

//bk   chi angles needed for building  CE2
	chi_required(1,10,aa_tyr,1) =  true;
	chi_required(2,10,aa_tyr,1) =  true;
	chi_required(3,10,aa_tyr,1) = false;
	chi_required(4,10,aa_tyr,1) = false;

//bk   chi angles needed for building  CZ
	chi_required(1,11,aa_tyr,1) =  true;
	chi_required(2,11,aa_tyr,1) =  true;
	chi_required(3,11,aa_tyr,1) = false;
	chi_required(4,11,aa_tyr,1) = false;

//bk   chi angles needed for building  OH
	chi_required(1,12,aa_tyr,1) =  true;
	chi_required(2,12,aa_tyr,1) =  true;
	chi_required(3,12,aa_tyr,1) = false;
	chi_required(4,12,aa_tyr,1) = false;

//bk   chi angles needed for building  H
	chi_required(1,13,aa_tyr,1) = false;
	chi_required(2,13,aa_tyr,1) = false;
	chi_required(3,13,aa_tyr,1) = false;
	chi_required(4,13,aa_tyr,1) = false;

//bk   chi angles needed for building  HH
	chi_required(1,14,aa_tyr,1) =  true;
	chi_required(2,14,aa_tyr,1) =  true;
	chi_required(3,14,aa_tyr,1) =  true;
	chi_required(4,14,aa_tyr,1) = false;

//bk   chi angles needed for building  HD1
	chi_required(1,15,aa_tyr,1) =  true;
	chi_required(2,15,aa_tyr,1) =  true;
	chi_required(3,15,aa_tyr,1) = false;
	chi_required(4,15,aa_tyr,1) = false;

//bk   chi angles needed for building  HE1
	chi_required(1,16,aa_tyr,1) =  true;
	chi_required(2,16,aa_tyr,1) =  true;
	chi_required(3,16,aa_tyr,1) = false;
	chi_required(4,16,aa_tyr,1) = false;

//bk   chi angles needed for building  HE2
	chi_required(1,17,aa_tyr,1) =  true;
	chi_required(2,17,aa_tyr,1) =  true;
	chi_required(3,17,aa_tyr,1) = false;
	chi_required(4,17,aa_tyr,1) = false;

//bk   chi angles needed for building  HD2
	chi_required(1,18,aa_tyr,1) =  true;
	chi_required(2,18,aa_tyr,1) =  true;
	chi_required(3,18,aa_tyr,1) = false;
	chi_required(4,18,aa_tyr,1) = false;

//bk   chi angles needed for building  HA
	chi_required(1,19,aa_tyr,1) = false;
	chi_required(2,19,aa_tyr,1) = false;
	chi_required(3,19,aa_tyr,1) = false;
	chi_required(4,19,aa_tyr,1) = false;

//bk   chi angles needed for building 2HB
	chi_required(1,20,aa_tyr,1) =  true;
	chi_required(2,20,aa_tyr,1) = false;
	chi_required(3,20,aa_tyr,1) = false;
	chi_required(4,20,aa_tyr,1) = false;

//bk   chi angles needed for building 3HB
	chi_required(1,21,aa_tyr,1) =  true;
	chi_required(2,21,aa_tyr,1) = false;
	chi_required(3,21,aa_tyr,1) = false;
	chi_required(4,21,aa_tyr,1) = false;

//bk   four atoms that define chi angle  1
	chi_atoms( 1, 1,aa_tyr, 1) =  1; //   N
	chi_atoms( 2, 1,aa_tyr, 1) =  2; //   CA
	chi_atoms( 3, 1,aa_tyr, 1) =  5; //   CB
	chi_atoms( 4, 1,aa_tyr, 1) =  6; //   CG
//bk   four atoms that define chi angle  2
	chi_atoms( 1, 2,aa_tyr, 1) =  2; //   CA
	chi_atoms( 2, 2,aa_tyr, 1) =  5; //   CB
	chi_atoms( 3, 2,aa_tyr, 1) =  6; //   CG
	chi_atoms( 4, 2,aa_tyr, 1) =  7; //   CD1
//bk   four atoms that define chi angle  3
	chi_atoms( 1, 3,aa_tyr, 1) = 10; //   CE2
	chi_atoms( 2, 3,aa_tyr, 1) = 11; //   CZ
	chi_atoms( 3, 3,aa_tyr, 1) = 12; //   OH
	chi_atoms( 4, 3,aa_tyr, 1) = 14; //   HH

	HNpos(aa_tyr,1) = 13; // atom number for backbone HN
	HApos(aa_tyr,1) = 19; // atom number for backbone HA
	nH_polar(aa_tyr,1) =  2; // number of polar hydrogens
	nH_aromatic(aa_tyr,1) =  4; // number of aromatic hydrogens
	nH_apolar(aa_tyr,1) =  3; // number of apolar hydrogens
	nacceptors(aa_tyr,1) =  2; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1,aa_tyr,1) = 13; //  H
	Hpos_polar( 2,aa_tyr,1) = 14; //  HH

//bk   atom number for aromatic hydrogens
	Hpos_aromatic( 1,aa_tyr,1) = 15; //  HD1
	Hpos_aromatic( 2,aa_tyr,1) = 16; //  HE1
	Hpos_aromatic( 3,aa_tyr,1) = 17; //  HE2
	Hpos_aromatic( 4,aa_tyr,1) = 18; //  HD2

//bk   atom number for apolar hydrogens
	Hpos_apolar( 1,aa_tyr,1) = 19; //  HA
	Hpos_apolar( 2,aa_tyr,1) = 20; // 2HB
	Hpos_apolar( 3,aa_tyr,1) = 21; // 3HB

//bk   acceptor positions
	accpt_pos( 1,aa_tyr, 1) =  4; //  O
	accpt_pos( 2,aa_tyr, 1) = 12; //  OH

//bk   hydrogens on each atom

	nhydrogens_on_atm( 1,aa_tyr, 1) =  1; //  N
	hydrogens_on_atm( 1, 1,aa_tyr, 1) = 13; //  H

	nhydrogens_on_atm( 2,aa_tyr, 1) =  1; //  CA
	hydrogens_on_atm( 1, 2,aa_tyr, 1) = 19; //  HA

	nhydrogens_on_atm( 5,aa_tyr, 1) =  2; //  CB
	hydrogens_on_atm( 1, 5,aa_tyr, 1) = 20; // 2HB
	hydrogens_on_atm( 2, 5,aa_tyr, 1) = 21; // 3HB

	nhydrogens_on_atm( 7,aa_tyr, 1) =  1; //  CD1
	hydrogens_on_atm( 1, 7,aa_tyr, 1) = 15; //  HD1

	nhydrogens_on_atm( 8,aa_tyr, 1) =  1; //  CD2
	hydrogens_on_atm( 1, 8,aa_tyr, 1) = 18; //  HD2

	nhydrogens_on_atm( 9,aa_tyr, 1) =  1; //  CE1
	hydrogens_on_atm( 1, 9,aa_tyr, 1) = 16; //  HE1

	nhydrogens_on_atm(10,aa_tyr, 1) =  1; //  CE2
	hydrogens_on_atm( 1,10,aa_tyr, 1) = 17; //  HE2

	nhydrogens_on_atm(12,aa_tyr, 1) =  1; //  OH
	hydrogens_on_atm( 1,12,aa_tyr, 1) = 14; //  HH

//bk   template coordinates for the amino acid
	icoor( 1, 1,aa_tyr, 1) =    0.0000; //   N
	icoor( 2, 1,aa_tyr, 1) =   17.7910; //   N
	icoor( 3, 1,aa_tyr, 1) =   69.1770; //   N
	icoor( 1, 2,aa_tyr, 1) =    0.0000; //   CA
	icoor( 2, 2,aa_tyr, 1) =   17.0770; //   CA
	icoor( 3, 2,aa_tyr, 1) =   70.4470; //   CA
	icoor( 1, 3,aa_tyr, 1) =    0.0000; //   C
	icoor( 2, 3,aa_tyr, 1) =   18.0440; //   C
	icoor( 3, 3,aa_tyr, 1) =   71.6230; //   C
	icoor( 1, 4,aa_tyr, 1) =   -0.0010; //   O
	icoor( 2, 4,aa_tyr, 1) =   19.2610; //   O
	icoor( 3, 4,aa_tyr, 1) =   71.4380; //   O
	icoor( 1, 5,aa_tyr, 1) =    1.2090; //   CB
	icoor( 2, 5,aa_tyr, 1) =   16.1430; //   CB
	icoor( 3, 5,aa_tyr, 1) =   70.5360; //   CB
	icoor( 1, 6,aa_tyr, 1) =    1.2850; //   CG
	icoor( 2, 6,aa_tyr, 1) =   15.3600; //   CG
	icoor( 3, 6,aa_tyr, 1) =   71.8280; //   CG
	icoor( 1, 7,aa_tyr, 1) =    2.3230; //   CD1
	icoor( 2, 7,aa_tyr, 1) =   14.4700; //   CD1
	icoor( 3, 7,aa_tyr, 1) =   72.0620; //   CD1
	icoor( 1, 8,aa_tyr, 1) =    0.3190; //   CD2
	icoor( 2, 8,aa_tyr, 1) =   15.5150; //   CD2
	icoor( 3, 8,aa_tyr, 1) =   72.8110; //   CD2
	icoor( 1, 9,aa_tyr, 1) =    2.3970; //   CE1
	icoor( 2, 9,aa_tyr, 1) =   13.7520; //   CE1
	icoor( 3, 9,aa_tyr, 1) =   73.2400; //   CE1
	icoor( 1,10,aa_tyr, 1) =    0.3830; //   CE2
	icoor( 2,10,aa_tyr, 1) =   14.8030; //   CE2
	icoor( 3,10,aa_tyr, 1) =   73.9930; //   CE2
	icoor( 1,11,aa_tyr, 1) =    1.4240; //   CZ
	icoor( 2,11,aa_tyr, 1) =   13.9220; //   CZ
	icoor( 3,11,aa_tyr, 1) =   74.2040; //   CZ
	icoor( 1,12,aa_tyr, 1) =    1.4930; //   OH
	icoor( 2,12,aa_tyr, 1) =   13.2110; //   OH
	icoor( 3,12,aa_tyr, 1) =   75.3800; //   OH
	icoor( 1,13,aa_tyr, 1) =    0.0390; //   H
	icoor( 2,13,aa_tyr, 1) =   18.8010; //   H
	icoor( 3,13,aa_tyr, 1) =   69.1950; //   H
	icoor( 1,14,aa_tyr, 1) =    0.7680; //   HH
	icoor( 2,14,aa_tyr, 1) =   13.4050; //   HH
	icoor( 3,14,aa_tyr, 1) =   75.9790; //   HH
	icoor( 1,15,aa_tyr, 1) =    3.0880; //   HD1
	icoor( 2,15,aa_tyr, 1) =   14.3410; //   HD1
	icoor( 3,15,aa_tyr, 1) =   71.2960; //   HD1
	icoor( 1,16,aa_tyr, 1) =    3.2180; //   HE1
	icoor( 2,16,aa_tyr, 1) =   13.0560; //   HE1
	icoor( 3,16,aa_tyr, 1) =   73.4100; //   HE1
	icoor( 1,17,aa_tyr, 1) =   -0.3870; //   HE2
	icoor( 2,17,aa_tyr, 1) =   14.9380; //   HE2
	icoor( 3,17,aa_tyr, 1) =   74.7530; //   HE2
	icoor( 1,18,aa_tyr, 1) =   -0.5010; //   HD2
	icoor( 2,18,aa_tyr, 1) =   16.2120; //   HD2
	icoor( 3,18,aa_tyr, 1) =   72.6380; //   HD2
	icoor( 1,19,aa_tyr, 1) =   -0.9080; //   HA
	icoor( 2,19,aa_tyr, 1) =   16.4800; //   HA
	icoor( 3,19,aa_tyr, 1) =   70.5360; //   HA
	icoor( 1,20,aa_tyr, 1) =    1.1450; //  2HB
	icoor( 2,20,aa_tyr, 1) =   15.4510; //  2HB
	icoor( 3,20,aa_tyr, 1) =   69.6950; //  2HB
	icoor( 1,21,aa_tyr, 1) =    2.1010; //  3HB
	icoor( 2,21,aa_tyr, 1) =   16.7600; //  3HB
	icoor( 3,21,aa_tyr, 1) =   70.4320; //  3HB

	// H positions from REDUCE
	if( param_pack::packer_logical::reduce_templates ) {
		icoor( 1 , 13 , aa_tyr , 1) = -0.016 ; // H
		icoor( 2 , 13 , aa_tyr , 1) = 18.776 ; // H
		icoor( 3 , 13 , aa_tyr , 1) = 69.348 ; // H
		icoor( 1 , 19 , aa_tyr , 1) = -0.921 ; // HA
		icoor( 2 , 19 , aa_tyr , 1) = 16.477 ; // HA
		icoor( 3 , 19 , aa_tyr , 1) = 70.495 ; // HA
		icoor( 1 , 20 , aa_tyr , 1) = 1.178 ; // 1HB
		icoor( 2 , 20 , aa_tyr , 1) = 15.438 ; // 1HB
		icoor( 3 , 20 , aa_tyr , 1) = 69.693 ; // 1HB
		icoor( 1 , 21 , aa_tyr , 1) = 2.128 ; // 2HB
		icoor( 2 , 21 , aa_tyr , 1) = 16.737 ; // 2HB
		icoor( 3 , 21 , aa_tyr , 1) = 70.424 ; // 2HB
		icoor( 1 , 15 , aa_tyr , 1) = 3.101 ; // HD1
		icoor( 2 , 15 , aa_tyr , 1) = 14.334 ; // HD1
		icoor( 3 , 15 , aa_tyr , 1) = 71.296 ; // HD1
		icoor( 1 , 18 , aa_tyr , 1) = -0.512 ; // HD2
		icoor( 2 , 18 , aa_tyr , 1) = 16.217 ; // HD2
		icoor( 3 , 18 , aa_tyr , 1) = 72.647 ; // HD2
		icoor( 1 , 16 , aa_tyr , 1) = 3.226 ; // HE1
		icoor( 2 , 16 , aa_tyr , 1) = 13.049 ; // HE1
		icoor( 3 , 16 , aa_tyr , 1) = 73.409 ; // HE1
		icoor( 1 , 17 , aa_tyr , 1) = -0.392 ; // HE2
		icoor( 2 , 17 , aa_tyr , 1) = 14.938 ; // HE2
		icoor( 3 , 17 , aa_tyr , 1) = 74.762 ; // HE2
		icoor( 1 , 14 , aa_tyr , 1) = 0.717 ; // HH
		icoor( 2 , 14 , aa_tyr , 1) = 13.453 ; // HH
		icoor( 3 , 14 , aa_tyr , 1) = 75.962 ; // HH
	}
	//rh create protonation state variants
	create_pH_variants( aa_tyr);

//bq create N and Cterm variants
	create_termini_variants( aa_tyr );

//bk   create variants that have h2o connected to the backbone nh and carbonyl
	if ( explicit_h2o ) {
		create_backbone_h2o_variants(aa_tyr);
		create_sidechain_h2o_variants(aa_tyr);
	}

}

//------------------------------------------------------------------------------
// start dna definitions - JJH 5/23/03
////////////////////////////////////////////////////////////////////////////////
/// @begin init_dna_gua
///
/// @brief arrays filled to define the proerties of the nucleic acid dGuanine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// na_gua - index number for GUA (22)
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_dna_gua()
{
	using namespace aaproperties_pack;
	using namespace param_aa;

//bk   Parameters for GUA -----------------------------
	nvar(na_gua) = 1; // starting starting number of variants for GUA
	variant_type( aav_base, na_gua, 1 ) = true; // type of na variant

//bk   variant1 ---------- GUA

	natoms(na_gua,1) = 33; // number of total atoms
	nheavyatoms(na_gua,1) = 22; // number of heavy atoms
	nchi(na_gua,1) = 0; // number of chi angles

//bk   atom names GUA
	atom_name( 1, na_gua,1) = " P  ";
	atom_name( 2, na_gua,1) = " O1P";
	atom_name( 3, na_gua,1) = " O2P";
	atom_name( 4, na_gua,1) = " O5*";
	atom_name( 5, na_gua,1) = " C5*";
	atom_name( 6, na_gua,1) = " C4*";
	atom_name( 7, na_gua,1) = " O4*";
	atom_name( 8, na_gua,1) = " C3*";
	atom_name( 9, na_gua,1) = " O3*";
	atom_name(10, na_gua,1) = " C2*";
	atom_name(11, na_gua,1) = " C1*";
	atom_name(12, na_gua,1) = " N1 ";
	atom_name(13, na_gua,1) = " C2 ";
	atom_name(14, na_gua,1) = " N2 ";
	atom_name(15, na_gua,1) = " N3 ";
	atom_name(16, na_gua,1) = " C4 ";
	atom_name(17, na_gua,1) = " C5 ";
	atom_name(18, na_gua,1) = " C6 ";
	atom_name(19, na_gua,1) = " O6 ";
	atom_name(20, na_gua,1) = " N7 ";
	atom_name(21, na_gua,1) = " C8 ";
	atom_name(22, na_gua,1) = " N9 ";
	atom_name(23, na_gua,1) = "1H5*";
	atom_name(24, na_gua,1) = "2H5*";
	atom_name(25, na_gua,1) = " H4*";
	atom_name(26, na_gua,1) = " H3*";
	atom_name(27, na_gua,1) = "1H2*";
	atom_name(28, na_gua,1) = "2H2*";
	atom_name(29, na_gua,1) = " H1*";
	atom_name(30, na_gua,1) = " H1 ";
	atom_name(31, na_gua,1) = "1H2 ";
	atom_name(32, na_gua,1) = "2H2 ";
	atom_name(33, na_gua,1) = " H8 ";

//jjh intra residue bonding
	nbonded_neighbors( 1, na_gua, 1) = 3; // P
		bonded_neighbor(1, 1, na_gua, 1) = 2; // P--O1P
		bonded_neighbor(2, 1, na_gua, 1) = 3; // P--O2P
		bonded_neighbor(3, 1, na_gua, 1) = 4; // P--O5*
	nbonded_neighbors( 2, na_gua, 1) = 1; // O1P
		bonded_neighbor(1, 2, na_gua, 1) = 1; // O1P--P
	nbonded_neighbors( 3, na_gua, 1) = 1; // O2P
		bonded_neighbor(1, 3, na_gua, 1) = 1; // O2P--P
	nbonded_neighbors( 4, na_gua, 1) = 2; // O5*
		bonded_neighbor(1, 4, na_gua, 1) = 1; // O5*--P
		bonded_neighbor(2, 4, na_gua, 1) = 5; // O5*--C5*
	nbonded_neighbors( 5, na_gua, 1) = 4; // C5*
		bonded_neighbor(1, 5, na_gua, 1) = 4; // C5*--O5*
		bonded_neighbor(2, 5, na_gua, 1) = 6; // C5*--C4*
		bonded_neighbor(3, 5, na_gua, 1) =23; // C5*--1H5*
		bonded_neighbor(4, 5, na_gua, 1) =24; // C5*--2H5*
	nbonded_neighbors( 6, na_gua, 1) = 4; // C4*
		bonded_neighbor(1, 6, na_gua, 1) = 5; // C4*--C5*
		bonded_neighbor(2, 6, na_gua, 1) = 7; // C4*--O4*
		bonded_neighbor(3, 6, na_gua, 1) = 8; // C4*--C3*
		bonded_neighbor(4, 6, na_gua, 1) =25; // C4*--H4*
	nbonded_neighbors( 7, na_gua, 1) = 2; // O4*
		bonded_neighbor(1, 7, na_gua, 1) = 6; // O4*--C4*
		bonded_neighbor(2, 7, na_gua, 1) =11; // O4*--C1*
	nbonded_neighbors( 8, na_gua, 1) = 4; // C3*
		bonded_neighbor(1, 8, na_gua, 1) = 6; // C3*--C4*
		bonded_neighbor(2, 8, na_gua, 1) = 9; // C3*--O3*
		bonded_neighbor(3, 8, na_gua, 1) =10; // C3*--C2*
		bonded_neighbor(4, 8, na_gua, 1) =26; // C3*--H3*
	nbonded_neighbors( 9, na_gua, 1) = 1; // O3*
		bonded_neighbor(1, 9, na_gua, 1) = 8; // O3*--C3*
	nbonded_neighbors( 10, na_gua, 1) = 4; // C2*
		bonded_neighbor(1,10, na_gua, 1) = 8; // C2*--C3*
		bonded_neighbor(2,10, na_gua, 1) =11; // C2*--C1*
		bonded_neighbor(3,10, na_gua, 1) =27; // C2*--1H2*
		bonded_neighbor(4,10, na_gua, 1) =28; // C2*--2H2*
	nbonded_neighbors( 11, na_gua, 1) = 4; // C1*
		bonded_neighbor(1,11, na_gua, 1) = 7; // C1*--O4*
		bonded_neighbor(2,11, na_gua, 1) =10; // C1*--C2*
		bonded_neighbor(3,11, na_gua, 1) =22; // C1*--N9
		bonded_neighbor(4,11, na_gua, 1) =29; // C1*--H1*
	nbonded_neighbors( 12, na_gua, 1) = 3; // N1
		bonded_neighbor(1,12, na_gua, 1) =13; // N1--C2
		bonded_neighbor(2,12, na_gua, 1) =18; // N1--C6
		bonded_neighbor(3,12, na_gua, 1) =30; // N1--H1
	nbonded_neighbors( 13, na_gua, 1) = 3; // C2
		bonded_neighbor(1,13, na_gua, 1) =12; // C2--N1
		bonded_neighbor(2,13, na_gua, 1) =14; // C2--N2
		bonded_neighbor(3,13, na_gua, 1) =15; // C2--N3
	nbonded_neighbors( 14, na_gua, 1) = 3; // N2
		bonded_neighbor(1,14, na_gua, 1) =13; // N2--C2
		bonded_neighbor(2,14, na_gua, 1) =31; // N2--1H2
		bonded_neighbor(3,14, na_gua, 1) =32; // N2--2H2
	nbonded_neighbors( 15, na_gua, 1) = 2; // N3
		bonded_neighbor(1,15, na_gua, 1) =13; // N3--C2
		bonded_neighbor(2,15, na_gua, 1) =16; // N3--C4
	nbonded_neighbors( 16, na_gua, 1) = 3; // C4
		bonded_neighbor(1,16, na_gua, 1) =15; // C4--N3
		bonded_neighbor(2,16, na_gua, 1) =17; // C4--C5
		bonded_neighbor(3,16, na_gua, 1) =22; // C4--N9
	nbonded_neighbors( 17, na_gua, 1) = 3; // C5
		bonded_neighbor(1,17, na_gua, 1) =16; // C5--C4
		bonded_neighbor(2,17, na_gua, 1) =18; // C5--C6
		bonded_neighbor(3,17, na_gua, 1) =20; // C5--N7
	nbonded_neighbors( 18, na_gua, 1) = 3; // C6
		bonded_neighbor(1,18, na_gua, 1) =12; // C6--N1
		bonded_neighbor(2,18, na_gua, 1) =17; // C6--C5
		bonded_neighbor(3,18, na_gua, 1) =19; // C6--O6
	nbonded_neighbors( 19, na_gua, 1) = 1; // O6
		bonded_neighbor(1,19, na_gua, 1) =18; // O6--C6
	nbonded_neighbors( 20, na_gua, 1) = 2; // N7
		bonded_neighbor(1,20, na_gua, 1) =17; // N7--C5
		bonded_neighbor(2,20, na_gua, 1) =21; // N7--C8
	nbonded_neighbors( 21, na_gua, 1) = 3; // C8
		bonded_neighbor(1,21, na_gua, 1) =20; // C8--N7
		bonded_neighbor(2,21, na_gua, 1) =22; // C8--N9
		bonded_neighbor(3,21, na_gua, 1) =33; // C8--H8
	nbonded_neighbors( 22, na_gua, 1) = 3; // N9
		bonded_neighbor(1,22, na_gua, 1) =11; // N9--C1*
		bonded_neighbor(2,22, na_gua, 1) =16; // N9--C4
		bonded_neighbor(3,22, na_gua, 1) =21; // N9--C8
	nbonded_neighbors( 23, na_gua, 1) = 1; //1H5*
		bonded_neighbor(1,23, na_gua, 1) = 5; //1H5*--C5*
	nbonded_neighbors( 24, na_gua, 1) = 1; //2H5*
		bonded_neighbor(1,24, na_gua, 1) = 5; //2H5*--C5*
	nbonded_neighbors( 25, na_gua, 1) = 1; // H4*
		bonded_neighbor(1,25, na_gua, 1) = 6; // H4*--C4*
	nbonded_neighbors( 26, na_gua, 1) = 1; // H3*
		bonded_neighbor(1,26, na_gua, 1) = 8; // H3*--C3*
	nbonded_neighbors( 27, na_gua, 1) = 1; //1H2*
		bonded_neighbor(1,27, na_gua, 1) =10; //1H2*--C2*
	nbonded_neighbors( 28, na_gua, 1) = 1; //2H2*
		bonded_neighbor(1,28, na_gua, 1) =10; //2H2*--C2*
	nbonded_neighbors( 29, na_gua, 1) = 1; // H1*
		bonded_neighbor(1,29, na_gua, 1) =11; // H1*--C1*
	nbonded_neighbors( 30, na_gua, 1) = 1; // H1
		bonded_neighbor(1,30, na_gua, 1) =12; // H1--N1
	nbonded_neighbors( 31, na_gua, 1) = 1; //1H2
		bonded_neighbor(1,31, na_gua, 1) =14; //1H2--N2
	nbonded_neighbors( 32, na_gua, 1) = 1; //2H2
		bonded_neighbor(1,32, na_gua, 1) =14; //2H2--N2
	nbonded_neighbors( 33, na_gua, 1) = 1; // H8
		bonded_neighbor(1,33, na_gua, 1) =21; // H8--C8

//jjh get the angles automatically
	get_angles_from_bonds(na_gua, 1);

	atom_base( 1,na_gua,1) =  2; // P    --> O1P
	atom_base( 2,na_gua,1) =  1; // O1P  --> P
	atom_base( 3,na_gua,1) =  1; // O2P  --> P
	atom_base( 4,na_gua,1) =  1; // O5*  --> P
	abase2( 4, na_gua, 1) =   5; //  C5* base2 for O5*
	atom_base( 5,na_gua,1) =  4; // C5*  --> O5*
	atom_base( 6,na_gua,1) =  5; // C4*  --> C5*
	atom_base( 7,na_gua,1) =  6; // O4*  --> C1*
	abase2( 7, na_gua, 1) =  11; //  C4* base2 for O4*
	atom_base( 8,na_gua,1) =  6; // C3*  --> C4*
	atom_base( 9,na_gua,1) =  8; // O3*  --> C3*
	atom_base(10,na_gua,1) =  8; // C2*  --> C3*
	atom_base(11,na_gua,1) = 10; // C1*  --> C2*
	atom_base(12,na_gua,1) = 13; // N1   --> C2
	atom_base(13,na_gua,1) = 15; // C2   --> N3
	atom_base(14,na_gua,1) = 13; // N2   --> C2
	atom_base(15,na_gua,1) = 16; // N3   --> C4
  abase2(15, na_gua, 1) =  13; //  C2  second base for  N3
	atom_base(16,na_gua,1) = 22; // C4   --> N9
	atom_base(17,na_gua,1) = 20; // C5   --> N7
	atom_base(18,na_gua,1) = 17; // C6   --> C5
	atom_base(19,na_gua,1) = 18; // O6   --> C6
	atom_base(20,na_gua,1) = 21; // N7   --> C8
	atom_base(21,na_gua,1) = 22; // C8   --> N9
	abase2(20, na_gua, 1) =  17; //  C5  second base for  N7
	atom_base(22,na_gua,1) = 11; // N9   --> C1*
	atom_base(23,na_gua,1) =  5; // 1H5* --> C5*
	atom_base(24,na_gua,1) =  5; // 2H5* --> C5*
	atom_base(25,na_gua,1) =  6; // H4*  --> C4*
	atom_base(26,na_gua,1) =  8; // H3*  --> C3*
	atom_base(27,na_gua,1) = 10; // 1H2* --> C2*
	atom_base(28,na_gua,1) = 10; // 2H2* --> C2*
	atom_base(29,na_gua,1) = 11; // H1*  --> C1*
	atom_base(30,na_gua,1) = 12; // H1   --> N1
	atom_base(31,na_gua,1) = 14; // 1H2  --> N2
	atom_base(32,na_gua,1) = 14; // 2H2  --> N2
	atom_base(33,na_gua,1) = 21; // H8   --> C8

//bk   atom type numbers GUA
	fullatom_type( 1,na_gua,1) = 21; // Phos   P
	fullatom_type( 2,na_gua,1) = 15; // OOC    O1P
	fullatom_type( 3,na_gua,1) = 15; // OOC    O2P
	fullatom_type( 4,na_gua,1) = 14; // ONH2   O5*
	fullatom_type( 5,na_gua,1) =  4; // CH2    C5*
	fullatom_type( 6,na_gua,1) =  3; // CH1    C4*
	fullatom_type( 7,na_gua,1) = 13; // OH     O4* // need Oether type
	fullatom_type( 8,na_gua,1) =  3; // CH1    C3*
	fullatom_type( 9,na_gua,1) = 14; // ONH2   O3*
	fullatom_type(10,na_gua,1) =  4; // CH2    C2*
	fullatom_type(11,na_gua,1) =  3; // CH1    C1*
	fullatom_type(12,na_gua,1) =  7; // Ntrp   N1
	fullatom_type(13,na_gua,1) =  6; // aroC   C2
	fullatom_type(14,na_gua,1) =  9; // NH2O   N2
	fullatom_type(15,na_gua,1) =  8; // Nhis   N3
	fullatom_type(16,na_gua,1) =  6; // aroC   C4
	fullatom_type(17,na_gua,1) =  6; // aroC   C5
	fullatom_type(18,na_gua,1) = 19; // CObb   C6
	fullatom_type(19,na_gua,1) = 20; // OCbb   O6
	fullatom_type(20,na_gua,1) =  8; // Nhis   N7
	fullatom_type(21,na_gua,1) =  6; // aroC   C8
	fullatom_type(22,na_gua,1) =  7; // Ntrp   N9
	fullatom_type(23,na_gua,1) = 23; // Hapo  1H5*
	fullatom_type(24,na_gua,1) = 23; // Hapo  2H5*
	fullatom_type(25,na_gua,1) = 23; // Hapo   H4*
	fullatom_type(26,na_gua,1) = 23; // Hapo   H3*
	fullatom_type(27,na_gua,1) = 23; // Hapo  1H2*
	fullatom_type(28,na_gua,1) = 23; // Hapo  2H2*
	fullatom_type(29,na_gua,1) = 23; // Hapo   H1*
	fullatom_type(30,na_gua,1) = 22; // Hpol   H1
	fullatom_type(31,na_gua,1) = 22; // Hpol  1H2
	fullatom_type(32,na_gua,1) = 22; // Hpol  2H2
	fullatom_type(33,na_gua,1) = 24; // Haro   H8

//bk   atom partial charges GUA
	atomic_charge( 1, na_gua,1) = 1.500; // P
	atomic_charge( 2, na_gua,1) = -.780; // O1P
	atomic_charge( 3, na_gua,1) = -.780; // O2P
	atomic_charge( 4, na_gua,1) = -.570; // O5*
	atomic_charge( 5, na_gua,1) = -.080; // C5*
	atomic_charge( 6, na_gua,1) =  .160; // C4*
	atomic_charge( 7, na_gua,1) = -.500; // O4*
	atomic_charge( 8, na_gua,1) =  .010; // C3*
	atomic_charge( 9, na_gua,1) = -.570; // O3*
	atomic_charge(10, na_gua,1) = -.180; // C2* !!!
	atomic_charge(11, na_gua,1) =  .160; // C1*
	atomic_charge(12, na_gua,1) = -.340; // N1
	atomic_charge(13, na_gua,1) =  .750; // C2
	atomic_charge(14, na_gua,1) = -.680; // N2
	atomic_charge(15, na_gua,1) = -.740; // N3
	atomic_charge(16, na_gua,1) =  .260; // C4
	atomic_charge(17, na_gua,1) =  .000; // C5
	atomic_charge(18, na_gua,1) =  .540; // C6
	atomic_charge(19, na_gua,1) = -.510; // O6
	atomic_charge(20, na_gua,1) = -.600; // N7
	atomic_charge(21, na_gua,1) =  .250; // C8
	atomic_charge(22, na_gua,1) = -.020; // N9
	atomic_charge(23, na_gua,1) =  .090; // 1H5*
	atomic_charge(24, na_gua,1) =  .090; // 2H5*
	atomic_charge(25, na_gua,1) =  .090; // H4*
	atomic_charge(26, na_gua,1) =  .090; // H3*
	atomic_charge(27, na_gua,1) =  .090; // 1H2*
	atomic_charge(28, na_gua,1) =  .090; // 2H2*
	atomic_charge(29, na_gua,1) =  .090; // H1*
	atomic_charge(30, na_gua,1) =  .260; // H1
	atomic_charge(31, na_gua,1) =  .320; // 1H2
	atomic_charge(32, na_gua,1) =  .350; // 2H2
	atomic_charge(33, na_gua,1) =  .160; // H8

	assert_total_charge( na_gua, 1, -1.0);

//cmd  first sidechain atom
	first_scatom( na_gua,1) = 12; //N1

//jjh anchor atoms for base mutations
	na_anchor(1, na_gua, 1) = 11;
	na_anchor(2, na_gua, 1) = 22;
	na_anchor(3, na_gua, 1) = 16;

//bk   template atoms used for placing hydrogens

//bk   template for building  P
	ta(1, 1, na_gua,1) =  4; //   O5*
	ta(2, 1, na_gua,1) =  5; //   C5*
	ta(3, 1, na_gua,1) =  6; //   C4*

//bk   template for building  O1P
	ta(1, 2, na_gua,1) =  1; //   P
	ta(2, 2, na_gua,1) =  5; //   C5*
	ta(3, 2, na_gua,1) =  4; //   O5*

//bk   template for building  O2P
	ta(1, 3, na_gua,1) =  1; //   P
	ta(2, 3, na_gua,1) =  2; //   O1P
	ta(3, 3, na_gua,1) =  4; //   O5*

//bk   template for building 1H5*
	ta(1,23, na_gua,1) =  5; //   C5*
	ta(2,23, na_gua,1) =  4; //   O5*
	ta(3,23, na_gua,1) =  6; //   C4*

//bk   template for building 2H5*
	ta(1,24, na_gua,1) =  5; //   C5*
	ta(2,24, na_gua,1) =  4; //   O5*
	ta(3,24, na_gua,1) =  6; //   C4*

//bk   template for building  H4*
	ta(1,25, na_gua,1) =  6; //   C4*
	ta(2,25, na_gua,1) =  7; //   O4*
	ta(3,25, na_gua,1) =  5; //   C5*

//bk   template for building  H3*
	ta(1,26, na_gua,1) =  8; //   C3*
	ta(2,26, na_gua,1) = 10; //   C2*
	ta(3,26, na_gua,1) =  6; //   C4*

//bk   template for building 1H2*
	ta(1,27, na_gua,1) = 10; //   C2*
	ta(2,27, na_gua,1) =  8; //   C3*
	ta(3,27, na_gua,1) = 11; //   C1*

//bk   template for building 2H2*
	ta(1,28, na_gua,1) = 10; //   C2*
	ta(2,28, na_gua,1) =  8; //   C3*
	ta(3,28, na_gua,1) = 11; //   C1*

//bk   template for building  H1*
	ta(1,29, na_gua,1) = 11; //   C1*
	ta(2,29, na_gua,1) = 10; //   C2*
	ta(3,29, na_gua,1) =  7; //   O4*

//bk   template for building  H1
	ta(1,30, na_gua,1) = 12; //   N1
	ta(2,30, na_gua,1) = 13; //   C2
	ta(3,30, na_gua,1) = 18; //   C6

//bk   template for building 1H2
	ta(1,31, na_gua,1) = 14; //   N2
	ta(2,31, na_gua,1) = 13; //   C2
	ta(3,31, na_gua,1) = 15; //   N3

//bk   template for building 2H2
	ta(1,32, na_gua,1) = 14; //   N2
	ta(2,32, na_gua,1) = 13; //   C2
	ta(3,32, na_gua,1) = 15; //   N3

//bk   template for building  H8
	ta(1,33, na_gua,1) = 21; //   C8
	ta(2,33, na_gua,1) = 20; //   N7
	ta(3,33, na_gua,1) = 22; //   N9

//bk   identify atoms that require chi angles to be built
//jjh  this is none for nucleic acids (for now)

//bk   hbonding information
//jjh set the next two to nonsense values
	HNpos( na_gua,1) = -1; // position of backbone HN
	HApos( na_gua,1) = -1; // position of backbone HA

	nH_polar( na_gua,1) =  3; // number of polar hydrogens
	nH_aromatic( na_gua,1) =  1; // number of aromatic hydrogens
	nH_apolar( na_gua,1) =  7; // number of apolar hydrogens
	nacceptors( na_gua,1) =  8; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1, na_gua,1) = 30; //  H1
	Hpos_polar( 2, na_gua,1) = 31; // 1H2
	Hpos_polar( 3, na_gua,1) = 32; // 2H2

//bk   atom numbers for aromatic hydrogens
	Hpos_aromatic( 1, na_gua,1) = 33; //  H8

//bk   atom numbers for apolar hydrogens
	Hpos_apolar( 1, na_gua,1) = 23; // 1H5*
	Hpos_apolar( 2, na_gua,1) = 24; // 2H5*
	Hpos_apolar( 3, na_gua,1) = 25; // H4*
	Hpos_apolar( 4, na_gua,1) = 26; // H3*
	Hpos_apolar( 5, na_gua,1) = 27; // 1H2*
	Hpos_apolar( 6, na_gua,1) = 28; // 2H2*
	Hpos_apolar( 7, na_gua,1) = 29; // H1*

//bk   acceptor positions
	accpt_pos( 1, na_gua, 1) =  2; //  O1P
	accpt_pos( 2, na_gua, 1) =  3; //  O2P
	accpt_pos( 3, na_gua, 1) =  4; //  O5*
	accpt_pos( 4, na_gua, 1) =  9; //  O3*
	accpt_pos( 5, na_gua, 1) = 15; //  N3
	accpt_pos( 6, na_gua, 1) = 19; //  O6
	accpt_pos( 7, na_gua, 1) = 20; //  N7
	accpt_pos( 8, na_gua, 1) =  7; //  O4*

//ja major groove specificity hbonders
//	// map of vectors: first initialize vector at this key
	major_groove_hb_atom[ na_gua ].assign( natoms(na_gua,1), false );
	major_groove_hb_atom[ na_gua ][ 19 ] = true;
	major_groove_hb_atom[ na_gua ][ 20 ] = true;

//bk   hydrogens on each atom

	nhydrogens_on_atm( 5, na_gua, 1) =  2; //  C5*
	hydrogens_on_atm( 1, 5, na_gua, 1) = 23; // 1H5*
	hydrogens_on_atm( 2, 5, na_gua, 1) = 24; // 2H5*

	nhydrogens_on_atm( 6, na_gua, 1) =  1; //  C4*
	hydrogens_on_atm( 1, 6, na_gua, 1) =  25; //  H4*

	nhydrogens_on_atm( 8, na_gua, 1) =  1; //  C3*
	hydrogens_on_atm( 1, 8, na_gua, 1) =  26; //  H3*

	nhydrogens_on_atm(10, na_gua, 1) =  2; //  C2*
	hydrogens_on_atm( 1,10, na_gua, 1) =  27; // 1H2*
	hydrogens_on_atm( 2,10, na_gua, 1) =  28; // 2H2*

	nhydrogens_on_atm(11, na_gua, 1) =  1; //  C1*
	hydrogens_on_atm( 1,11, na_gua, 1) =  29; //  H1*

	nhydrogens_on_atm(12, na_gua, 1) =  1; //  N1
	hydrogens_on_atm( 1,12, na_gua, 1) =  30; //  H1

	nhydrogens_on_atm(14, na_gua, 1) =  2; //  N2
	hydrogens_on_atm( 1,14, na_gua, 1) =  31; // 1H2
	hydrogens_on_atm( 2,14, na_gua, 1) =  32; // 2H2

	nhydrogens_on_atm(21, na_gua, 1) =  1; //  C8
	hydrogens_on_atm( 1,21, na_gua, 1) = 33; //  H8

//bk   template coordinates for the nucleic acid

///jjh All nucleic acid template coordinates are from:
///jjh
///jjh Geometric Parameters in Nucleic Acids: Sugar and Phosphate Constituents.
///jjh Anke Gelbin, Bohdan Schneider, Lester Clowney, Shu-Hsin Hsieh,
///jjh Wilma K. Olson, and Helen M. Berman. (1996), J. Am. Chem. Soc., 118, 519-529.
///jjh
///jjh Geometric Parameters in Nucleic Acids: Nitrogenous Bases.
///jjh Lester Clowney, Shri C. Jain, A. R. Srinivasan, John Westbrook,
///jjh Wilma K. Olson, and Helen M. Berman. (1996), J. Am. Chem. Soc., 118, 509-518.
///jjh
///jjh with hydrogens placed by REDUCE:
///jjh
///jjh Word, et. al. (1999) J. Mol. Biol. 285, 1735-1747.

	icoor( 1, 1, na_gua, 1) =    0.224; //   P
	icoor( 2, 1, na_gua, 1) =   -4.365; //   P
	icoor( 3, 1, na_gua, 1) =    2.383; //   P
	icoor( 1, 2, na_gua, 1) =    1.336; //   O1P
	icoor( 2, 2, na_gua, 1) =   -3.982; //   O1P
	icoor( 3, 2, na_gua, 1) =    3.290; //   O1P
	icoor( 1, 3, na_gua, 1) =    0.278; //   O2P
	icoor( 2, 3, na_gua, 1) =   -5.664; //   O2P
	icoor( 3, 3, na_gua, 1) =    1.666; //   O2P
	icoor( 1, 4, na_gua, 1) =    0.042; //   O5*
	icoor( 2, 4, na_gua, 1) =   -3.205; //   O5*
	icoor( 3, 4, na_gua, 1) =    1.307; //   O5*
	icoor( 1, 5, na_gua, 1) =   -1.014; //   C5*
	icoor( 2, 5, na_gua, 1) =   -3.256; //   C5*
	icoor( 3, 5, na_gua, 1) =    0.347; //   C5*
	icoor( 1, 6, na_gua, 1) =   -0.913; //   C4*
	icoor( 2, 6, na_gua, 1) =   -2.083; //   C4*
	icoor( 3, 6, na_gua, 1) =   -0.600; //   C4*
	icoor( 1, 7, na_gua, 1) =   -1.127; //   O4*
	icoor( 2, 7, na_gua, 1) =   -0.853; //   O4*
	icoor( 3, 7, na_gua, 1) =    0.133; //   O4*
	icoor( 1, 8, na_gua, 1) =    0.445; //   C3*
	icoor( 2, 8, na_gua, 1) =   -1.932; //   C3*
	icoor( 3, 8, na_gua, 1) =   -1.287; //   C3*
	icoor( 1, 9, na_gua, 1) =    0.272; //   O3*
	icoor( 2, 9, na_gua, 1) =   -1.450; //   O3*
	icoor( 3, 9, na_gua, 1) =   -2.624; //   O3*
	icoor( 1,10, na_gua, 1) =    1.149; //   C2*
	icoor( 2,10, na_gua, 1) =   -0.891; //   C2*
	icoor( 3,10, na_gua, 1) =   -0.438; //   C2*
	icoor( 1,11, na_gua, 1) =    0.000; //   C1*
	icoor( 2,11, na_gua, 1) =    0.000; //   C1*
	icoor( 3,11, na_gua, 1) =    0.000; //   C1*
	icoor( 1,12, na_gua, 1) =    0.444; //   N1
	icoor( 2,12, na_gua, 1) =    4.437; //   N1
	icoor( 3,12, na_gua, 1) =    2.427; //   N1
	icoor( 1,13, na_gua, 1) =    0.142; //   C2
	icoor( 2,13, na_gua, 1) =    4.190; //   C2
	icoor( 3,13, na_gua, 1) =    1.110; //   C2
	icoor( 1,14, na_gua, 1) =   -0.047; //   N2
	icoor( 2,14, na_gua, 1) =    5.269; //   N2
	icoor( 3,14, na_gua, 1) =    0.336; //   N2
	icoor( 1,15, na_gua, 1) =    0.034; //   N3
	icoor( 2,15, na_gua, 1) =    2.979; //   N3
	icoor( 3,15, na_gua, 1) =    0.591; //   N3
	icoor( 1,16, na_gua, 1) =    0.254; //   C4
	icoor( 2,16, na_gua, 1) =    2.014; //   C4
	icoor( 3,16, na_gua, 1) =    1.509; //   C4
	icoor( 1,17, na_gua, 1) =    0.562; //   C5
	icoor( 2,17, na_gua, 1) =    2.154; //   C5
	icoor( 3,17, na_gua, 1) =    2.846; //   C5
	icoor( 1,18, na_gua, 1) =    0.676; //   C6
	icoor( 2,18, na_gua, 1) =    3.459; //   C6
	icoor( 3,18, na_gua, 1) =    3.389; //   C6
	icoor( 1,19, na_gua, 1) =    0.941; //   O6
	icoor( 2,19, na_gua, 1) =    3.789; //   O6
	icoor( 3,19, na_gua, 1) =    4.552; //   O6
	icoor( 1,20, na_gua, 1) =    0.712; //   N7
	icoor( 2,20, na_gua, 1) =    0.912; //   N7
	icoor( 3,20, na_gua, 1) =    3.448; //   N7
	icoor( 1,21, na_gua, 1) =    0.498; //   C8
	icoor( 2,21, na_gua, 1) =    0.057; //   C8
	icoor( 3,21, na_gua, 1) =    2.485; //   C8
	icoor( 1,22, na_gua, 1) =    0.214; //   N9
	icoor( 2,22, na_gua, 1) =    0.659; //   N9
	icoor( 3,22, na_gua, 1) =    1.283; //   N9
	icoor( 1,23, na_gua, 1) =   -0.962; //  1H5*
	icoor( 2,23, na_gua, 1) =   -4.199; //  1H5*
	icoor( 3,23, na_gua, 1) =   -0.217; //  1H5*
	icoor( 1,24, na_gua, 1) =   -1.987; //  2H5*
	icoor( 2,24, na_gua, 1) =   -3.241; //  2H5*
	icoor( 3,24, na_gua, 1) =    0.859; //  2H5*
	icoor( 1,25, na_gua, 1) =   -1.673; //   H4*
	icoor( 2,25, na_gua, 1) =   -2.277; //   H4*
	icoor( 3,25, na_gua, 1) =   -1.372; //   H4*
	icoor( 1,26, na_gua, 1) =    1.002; //   H3*
	icoor( 2,26, na_gua, 1) =   -2.877; //   H3*
	icoor( 3,26, na_gua, 1) =   -1.364; //   H3*
	icoor( 1,27, na_gua, 1) =    1.904; //  1H2*
	icoor( 2,27, na_gua, 1) =   -0.335; //  1H2*
	icoor( 3,27, na_gua, 1) =   -1.013; //  1H2*
	icoor( 1,28, na_gua, 1) =    1.668; //  2H2*
	icoor( 2,28, na_gua, 1) =   -1.344; //  2H2*
	icoor( 3,28, na_gua, 1) =    0.420; //  2H2*
	icoor( 1,29, na_gua, 1) =   -0.119; //   H1*
	icoor( 2,29, na_gua, 1) =    0.800; //   H1*
	icoor( 3,29, na_gua, 1) =   -0.745; //   H1*
	icoor( 1,30, na_gua, 1) =    0.501; //   H1
	icoor( 2,30, na_gua, 1) =    5.392; //   H1
	icoor( 3,30, na_gua, 1) =    2.716; //   H1
	icoor( 1,31, na_gua, 1) =   -0.270; //  1H2
	icoor( 2,31, na_gua, 1) =    5.157; //  1H2
	icoor( 3,31, na_gua, 1) =   -0.632; //  1H2
	icoor( 1,32, na_gua, 1) =    0.035; //  2H2
	icoor( 2,32, na_gua, 1) =    6.186; //  2H2
	icoor( 3,32, na_gua, 1) =    0.727; //  2H2
	icoor( 1,33, na_gua, 1) =    0.541; //   H8
	icoor( 2,33, na_gua, 1) =   -1.034; //   H8
	icoor( 3,33, na_gua, 1) =    2.622; //   H8

	if ( design::hydrate_dna ) {
		create_dna_h2o_variants( na_gua );
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_dna_ade
///
/// @brief arrays filled to define the proerties of the amino acid dAdenine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// na_ade - index number for ADE
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_dna_ade()
{
	using namespace aaproperties_pack;
	using namespace param_aa;

//bk   Parameters for ADE -----------------------------
	nvar(na_ade) = 1; // starting starting number of variants for ADE
	variant_type( aav_base, na_ade, 1 ) = true; // type of na variant

//bk   variant1 ---------- ADE

	natoms(na_ade,1) = 32; // number of total atoms
	nheavyatoms(na_ade,1) = 21; // number of heavy atoms
	nchi(na_ade,1) = 0; // number of chi angles

//bk   atom names ADE
	atom_name( 1, na_ade,1) = " P  ";
	atom_name( 2, na_ade,1) = " O1P";
	atom_name( 3, na_ade,1) = " O2P";
	atom_name( 4, na_ade,1) = " O5*";
	atom_name( 5, na_ade,1) = " C5*";
	atom_name( 6, na_ade,1) = " C4*";
	atom_name( 7, na_ade,1) = " O4*";
	atom_name( 8, na_ade,1) = " C3*";
	atom_name( 9, na_ade,1) = " O3*";
	atom_name(10, na_ade,1) = " C2*";
	atom_name(11, na_ade,1) = " C1*";
	atom_name(12, na_ade,1) = " N1 ";
	atom_name(13, na_ade,1) = " C2 ";
	atom_name(14, na_ade,1) = " N3 ";
	atom_name(15, na_ade,1) = " C4 ";
	atom_name(16, na_ade,1) = " C5 ";
	atom_name(17, na_ade,1) = " C6 ";
	atom_name(18, na_ade,1) = " N6 ";
	atom_name(19, na_ade,1) = " N7 ";
	atom_name(20, na_ade,1) = " C8 ";
	atom_name(21, na_ade,1) = " N9 ";
	atom_name(22, na_ade,1) = "1H5*";
	atom_name(23, na_ade,1) = "2H5*";
	atom_name(24, na_ade,1) = " H4*";
	atom_name(25, na_ade,1) = " H3*";
	atom_name(26, na_ade,1) = "1H2*";
	atom_name(27, na_ade,1) = "2H2*";
	atom_name(28, na_ade,1) = " H1*";
	atom_name(29, na_ade,1) = " H2 ";
	atom_name(30, na_ade,1) = "1H6 ";
	atom_name(31, na_ade,1) = "2H6 ";
	atom_name(32, na_ade,1) = " H8 ";

//jjh intra residue bonding
	nbonded_neighbors( 1, na_ade, 1) = 3; // P
		bonded_neighbor(1, 1, na_ade, 1) = 2; // P--O1P
		bonded_neighbor(2, 1, na_ade, 1) = 3; // P--O2P
		bonded_neighbor(3, 1, na_ade, 1) = 4; // P--O5*
	nbonded_neighbors( 2, na_ade, 1) = 1; // O1P
		bonded_neighbor(1, 2, na_ade, 1) = 1; // O1P--P
	nbonded_neighbors( 3, na_ade, 1) = 1; // O2P
		bonded_neighbor(1, 3, na_ade, 1) = 1; // O2P--P
	nbonded_neighbors( 4, na_ade, 1) = 2; // O5*
		bonded_neighbor(1, 4, na_ade, 1) = 1; // O5*--P
		bonded_neighbor(2, 4, na_ade, 1) = 5; // O5*--C5*
	nbonded_neighbors( 5, na_ade, 1) = 4; // C5*
		bonded_neighbor(1, 5, na_ade, 1) = 4; // C5*--O5*
		bonded_neighbor(2, 5, na_ade, 1) = 6; // C5*--C4*
		bonded_neighbor(3, 5, na_ade, 1) =22; // C5*--1H5*
		bonded_neighbor(4, 5, na_ade, 1) =23; // C5*--2H5*
	nbonded_neighbors( 6, na_ade, 1) = 4; // C4*
		bonded_neighbor(1, 6, na_ade, 1) = 5; // C4*--C5*
		bonded_neighbor(2, 6, na_ade, 1) = 7; // C4*--O4*
		bonded_neighbor(3, 6, na_ade, 1) = 8; // C4*--C3*
		bonded_neighbor(4, 6, na_ade, 1) =24; // C4*--H4*
	nbonded_neighbors( 7, na_ade, 1) = 2; // O4*
		bonded_neighbor(1, 7, na_ade, 1) = 6; // O4*--C4*
		bonded_neighbor(2, 7, na_ade, 1) =11; // O4*--C1*
	nbonded_neighbors( 8, na_ade, 1) = 4; // C3*
		bonded_neighbor(1, 8, na_ade, 1) = 6; // C3*--C4*
		bonded_neighbor(2, 8, na_ade, 1) = 9; // C3*--O3*
		bonded_neighbor(3, 8, na_ade, 1) =10; // C3*--C2*
		bonded_neighbor(4, 8, na_ade, 1) =25; // C3*--H3*
	nbonded_neighbors( 9, na_ade, 1) = 1; // O3*
		bonded_neighbor(1, 9, na_ade, 1) = 8; // O3*--C3*
	nbonded_neighbors( 10, na_ade, 1) = 4; // C2*
		bonded_neighbor(1,10, na_ade, 1) = 8; // C2*--C3*
		bonded_neighbor(2,10, na_ade, 1) =11; // C2*--C1*
		bonded_neighbor(3,10, na_ade, 1) =26; // C2*--1H2*
		bonded_neighbor(4,10, na_ade, 1) =27; // C2*--2H2*
	nbonded_neighbors( 11, na_ade, 1) = 4; // C1*
		bonded_neighbor(1,11, na_ade, 1) = 7; // C1*--O4*
		bonded_neighbor(2,11, na_ade, 1) =10; // C1*--C2*
		bonded_neighbor(3,11, na_ade, 1) =21; // C1*--N9
		bonded_neighbor(4,11, na_ade, 1) =28; // C1*--H1*
	nbonded_neighbors( 12, na_ade, 1) = 2; // N1
		bonded_neighbor(1,12, na_ade, 1) =13; // N1--C2
		bonded_neighbor(2,12, na_ade, 1) =17; // N1--C6
	nbonded_neighbors( 13, na_ade, 1) = 3; // C2
		bonded_neighbor(1,13, na_ade, 1) =12; // C2--N1
		bonded_neighbor(2,13, na_ade, 1) =14; // C2--N3
		bonded_neighbor(3,13, na_ade, 1) =29; // C2--H2
	nbonded_neighbors( 14, na_ade, 1) = 2; // N3
		bonded_neighbor(1,14, na_ade, 1) =13; // N3--C2
		bonded_neighbor(2,14, na_ade, 1) =15; // N3--C4
	nbonded_neighbors( 15, na_ade, 1) = 3; // C4
		bonded_neighbor(1,15, na_ade, 1) =14; // C4--N3
		bonded_neighbor(2,15, na_ade, 1) =16; // C4--C5
		bonded_neighbor(3,15, na_ade, 1) =21; // C4--N9
	nbonded_neighbors( 16, na_ade, 1) = 3; // C5
		bonded_neighbor(1,16, na_ade, 1) =15; // C5--C4
		bonded_neighbor(2,16, na_ade, 1) =17; // C5--C6
		bonded_neighbor(3,16, na_ade, 1) =19; // C5--N7
	nbonded_neighbors( 17, na_ade, 1) = 3; // C6
		bonded_neighbor(1,17, na_ade, 1) =12; // C6--N1
		bonded_neighbor(2,17, na_ade, 1) =16; // C6--C5
		bonded_neighbor(3,17, na_ade, 1) =18; // C6--N6
	nbonded_neighbors( 18, na_ade, 1) = 3; // N6
		bonded_neighbor(1,18, na_ade, 1) =17; // N6--C6
		bonded_neighbor(2,18, na_ade, 1) =30; // N6--1H6
		bonded_neighbor(3,18, na_ade, 1) =31; // N6--2H6
	nbonded_neighbors( 19, na_ade, 1) = 2; // N7
		bonded_neighbor(1,19, na_ade, 1) =16; // N7--C5
		bonded_neighbor(2,19, na_ade, 1) =20; // N7--C8
	nbonded_neighbors( 20, na_ade, 1) = 3; // C8
		bonded_neighbor(1,20, na_ade, 1) =19; // C8--N7
		bonded_neighbor(2,20, na_ade, 1) =21; // C8--N9
		bonded_neighbor(3,20, na_ade, 1) =32; // C8--H8
	nbonded_neighbors( 21, na_ade, 1) = 3; // N9
		bonded_neighbor(1,21, na_ade, 1) =11; // N9--C1*
		bonded_neighbor(2,21, na_ade, 1) =15; // N9--C4
		bonded_neighbor(3,21, na_ade, 1) =20; // N9--C8
	nbonded_neighbors( 22, na_ade, 1) = 1; //1H5*
		bonded_neighbor(1,22, na_ade, 1) = 5; //1H5*--C5*
	nbonded_neighbors( 23, na_ade, 1) = 1; //2H5*
		bonded_neighbor(1,23, na_ade, 1) = 5; //2H5*--C5*
	nbonded_neighbors( 24, na_ade, 1) = 1; // H4*
		bonded_neighbor(1,24, na_ade, 1) = 6; // H4*--C4*
	nbonded_neighbors( 25, na_ade, 1) = 1; // H3*
		bonded_neighbor(1,25, na_ade, 1) = 8; // H3*--C3*
	nbonded_neighbors( 26, na_ade, 1) = 1; //1H2*
		bonded_neighbor(1,26, na_ade, 1) =10; //1H2*--C2*
	nbonded_neighbors( 27, na_ade, 1) = 1; //2H2*
		bonded_neighbor(1,27, na_ade, 1) =10; //2H2*--C2*
	nbonded_neighbors( 28, na_ade, 1) = 1; // H1*
		bonded_neighbor(1,28, na_ade, 1) =11; // H1*--C1*
	nbonded_neighbors( 29, na_ade, 1) = 1; // H2
		bonded_neighbor(1,29, na_ade, 1) =13; // H2--C2
	nbonded_neighbors( 30, na_ade, 1) = 1; //1H6
		bonded_neighbor(1,30, na_ade, 1) =18; //1H6--N6
	nbonded_neighbors( 31, na_ade, 1) = 1; //2H6
		bonded_neighbor(1,31, na_ade, 1) =18; //2H6--N6
	nbonded_neighbors( 32, na_ade, 1) = 1; // H8
		bonded_neighbor(1,32, na_ade, 1) =20; // H8--C8

//jjh get the angles automatically
	get_angles_from_bonds(na_ade, 1);

	atom_base( 1,na_ade,1) =  2; //   P   --> O1P
	atom_base( 2,na_ade,1) =  1; //   O1P --> P
	atom_base( 3,na_ade,1) =  1; //   O2P --> P
	atom_base( 4,na_ade,1) =  1; //   O5* --> P
	abase2( 4, na_ade, 1)  =  5; //  C5* base for  O5*
	atom_base( 5,na_ade,1) =  4; //   C5* --> O5*
	atom_base( 6,na_ade,1) =  5; //   C4* --> C5*
	atom_base( 7,na_ade,1) = 11; // O4*  --> C1*
	abase2( 7, na_ade, 1) =   6; //  C4* base2 for  O4*
	atom_base( 8,na_ade,1) =  6; //   C3* --> C4*
	atom_base( 9,na_ade,1) =  8; //   O3* --> C3*
	atom_base(10,na_ade,1) =  8; //   C2* --> C3*
	atom_base(11,na_ade,1) = 10; //   C1* --> C2*
	atom_base(12,na_ade,1) = 17; //   N1  --> C6
	abase2(12, na_ade, 1)  = 13; //  C2  base2 for  N1
	atom_base(13,na_ade,1) = 14; //   C2  --> N3
	atom_base(14,na_ade,1) = 13; //   N3  --> C2
	abase2(14, na_ade, 1)  = 15; //  C4  base2 for  N3
	atom_base(15,na_ade,1) = 21; //   C4  --> N9
	atom_base(16,na_ade,1) = 19; //   C5  --> N7
	atom_base(17,na_ade,1) = 16; //   C6  --> C5
	atom_base(18,na_ade,1) = 17; //   N6  --> C6
	atom_base(19,na_ade,1) = 16; //   N7  --> C5
	abase2(19, na_ade, 1)  = 20; //  C8  base2 for  N7
	atom_base(20,na_ade,1) = 21; //   C8  --> N9
	atom_base(21,na_ade,1) = 11; //   N9  --> C1*
	atom_base(22,na_ade,1) =  5; //  1H5* --> C5*
	atom_base(23,na_ade,1) =  5; //  2H5* --> C5*`
	atom_base(24,na_ade,1) =  6; //   H4* --> C4*
	atom_base(25,na_ade,1) =  8; //   H3* --> C3*
	atom_base(26,na_ade,1) = 10; //  1H2* --> C2*
	atom_base(27,na_ade,1) = 10; //  2H2* --> C2*
	atom_base(28,na_ade,1) = 11; //   H1* --> C1*
	atom_base(29,na_ade,1) = 13; //   H2  --> C2
	atom_base(30,na_ade,1) = 18; //  1H6  --> N6
	atom_base(31,na_ade,1) = 18; //  2H6  --> N6
	atom_base(32,na_ade,1) = 20; //   H8  --> C8

//bk   atom type numbers ADE
	fullatom_type( 1,na_ade,1) = 21; // Phos   P
	fullatom_type( 2,na_ade,1) = 15; // OOC    O1P
	fullatom_type( 3,na_ade,1) = 15; // OOC    O2P
	fullatom_type( 4,na_ade,1) = 14; // ONH2   O5*
	fullatom_type( 5,na_ade,1) =  4; // CH2    C5*
	fullatom_type( 6,na_ade,1) =  3; // CH1    C4*
	fullatom_type( 7,na_ade,1) = 13; // OH     O4* // need Oether type
	fullatom_type( 8,na_ade,1) =  3; // CH1    C3*
	fullatom_type( 9,na_ade,1) = 14; // ONH2   O3*
	fullatom_type(10,na_ade,1) =  4; // CH2    C2*
	fullatom_type(11,na_ade,1) =  3; // CH1    C1*
	fullatom_type(12,na_ade,1) =  8; // Nhis   N1
	fullatom_type(13,na_ade,1) =  6; // aroC   C2
	fullatom_type(14,na_ade,1) =  8; // Nhis   N3
	fullatom_type(15,na_ade,1) =  6; // aroC   C4
	fullatom_type(16,na_ade,1) =  6; // aroC   C5
	fullatom_type(17,na_ade,1) =  6; // aroC   C6
	fullatom_type(18,na_ade,1) =  9; // NH2O   N6
	fullatom_type(19,na_ade,1) =  8; // Nhis   N7
	fullatom_type(20,na_ade,1) =  6; // aroC   C8
	fullatom_type(21,na_ade,1) =  7; // Ntrp   N9
	fullatom_type(22,na_ade,1) = 23; // Hapo  1H5*
	fullatom_type(23,na_ade,1) = 23; // Hapo  2H5*
	fullatom_type(24,na_ade,1) = 23; // Hapo   H4*
	fullatom_type(25,na_ade,1) = 23; // Hapo   H3*
	fullatom_type(26,na_ade,1) = 23; // Hapo  1H2*
	fullatom_type(27,na_ade,1) = 23; // Hapo  2H2*
	fullatom_type(28,na_ade,1) = 23; // Hapo   H1*
	fullatom_type(29,na_ade,1) = 24; // Haro   H2
	fullatom_type(30,na_ade,1) = 22; // Hpol  1H6
	fullatom_type(31,na_ade,1) = 22; // Hpol  2H6
	fullatom_type(32,na_ade,1) = 24; // Haro   H8

//bk   atom partial charges ADE
	atomic_charge( 1, na_ade,1) = 1.500; // P
	atomic_charge( 2, na_ade,1) = -.780; // O1P
	atomic_charge( 3, na_ade,1) = -.780; // O2P
	atomic_charge( 4, na_ade,1) = -.570; // O5*
	atomic_charge( 5, na_ade,1) = -.080; // C5*
	atomic_charge( 6, na_ade,1) =  .160; // C4*
	atomic_charge( 7, na_ade,1) = -.500; // O4*
	atomic_charge( 8, na_ade,1) =  .010; // C3*
	atomic_charge( 9, na_ade,1) = -.570; // O3*
	atomic_charge(10, na_ade,1) = -.180; // C2*
	atomic_charge(11, na_ade,1) =  .160; // C1*
	atomic_charge(12, na_ade,1) = -.740; // N1
	atomic_charge(13, na_ade,1) =  .500; // C2
	atomic_charge(14, na_ade,1) = -.750; // N3
	atomic_charge(15, na_ade,1) =  .430; // C4
	atomic_charge(16, na_ade,1) =  .280; // C5
	atomic_charge(17, na_ade,1) =  .460; // C6
	atomic_charge(18, na_ade,1) = -.770; // N6
	atomic_charge(19, na_ade,1) = -.710; // N7
	atomic_charge(20, na_ade,1) =  .340; // C8
	atomic_charge(21, na_ade,1) = -.050; // N9
	atomic_charge(22, na_ade,1) =  .090; // 1H5*
	atomic_charge(23, na_ade,1) =  .090; // 2H5*
	atomic_charge(24, na_ade,1) =  .090; // H4*
	atomic_charge(25, na_ade,1) =  .090; // H3*
	atomic_charge(26, na_ade,1) =  .090; // 1H2*
	atomic_charge(27, na_ade,1) =  .090; // 2H2*
	atomic_charge(28, na_ade,1) =  .090; // H1*
	atomic_charge(29, na_ade,1) =  .130; // H2
	atomic_charge(30, na_ade,1) =  .380; // 1H6
	atomic_charge(31, na_ade,1) =  .380; // 2H6
	atomic_charge(32, na_ade,1) =  .120; // H8

	assert_total_charge( na_ade, 1, -1.0);

//cmd  first sidechain atom
	first_scatom( na_ade,1) = 12; //N1

//jjh anchor atoms for base mutations
	na_anchor(1, na_ade, 1) = 11;
	na_anchor(2, na_ade, 1) = 21;
	na_anchor(3, na_ade, 1) = 15;

//bk   template atoms used for placing hydrogens

//bk   template for building  P
	ta(1, 1, na_ade,1) =  4; //   O5*
	ta(2, 1, na_ade,1) =  5; //   C5*
	ta(3, 1, na_ade,1) =  6; //   C4*

//bk   template for building  O1P
	ta(1, 2, na_ade,1) =  1; //   P
	ta(2, 2, na_ade,1) =  5; //   C5*
	ta(3, 2, na_ade,1) =  4; //   O5*

//bk   template for building  O2P
	ta(1, 3, na_ade,1) =  1; //   P
	ta(2, 3, na_ade,1) =  2; //   O1P
	ta(3, 3, na_ade,1) =  4; //   O5*

//bk   template for building 1H5*
	ta(1,22, na_ade,1) =  5; //   C5*
	ta(2,22, na_ade,1) =  4; //   O5*
	ta(3,22, na_ade,1) =  6; //   C4*

//bk   template for building 2H5*
	ta(1,23, na_ade,1) =  5; //   C5*
	ta(2,23, na_ade,1) =  4; //   O5*
	ta(3,23, na_ade,1) =  6; //   C4*

//bk   template for building  H4*
	ta(1,24, na_ade,1) =  6; //   C4*
	ta(2,24, na_ade,1) =  7; //   O4*
	ta(3,24, na_ade,1) =  5; //   C5*

//bk   template for building  H3*
	ta(1,25, na_ade,1) =  8; //   C3*
	ta(2,25, na_ade,1) = 10; //   C2*
	ta(3,25, na_ade,1) =  6; //   C4*

//bk   template for building 1H2*
	ta(1,26, na_ade,1) = 10; //   C2*
	ta(2,26, na_ade,1) =  8; //   C3*
	ta(3,26, na_ade,1) = 11; //   C1*

//bk   template for building  2H2*
	ta(1,27, na_ade,1) = 10; //   C2*
	ta(2,27, na_ade,1) =  8; //   C3*
	ta(3,27, na_ade,1) = 11; //   C1*

//bk   template for building  H1*
	ta(1,28, na_ade,1) = 11; //   C1*
	ta(2,28, na_ade,1) = 10; //   C2*
	ta(3,28, na_ade,1) =  7; //   O4*

//bk   template for building  H2
	ta(1,29, na_ade,1) = 13; //   C2
	ta(2,29, na_ade,1) = 12; //   N1
	ta(3,29, na_ade,1) = 14; //   N3

//bk   template for building 1H6
	ta(1,30, na_ade,1) = 18; //   N6
	ta(2,30, na_ade,1) = 17; //   C6
	ta(3,30, na_ade,1) = 12; //   N1

//bk   template for building 2H6
	ta(1,31, na_ade,1) = 18; //   N6
	ta(2,31, na_ade,1) = 17; //   C6
	ta(3,31, na_ade,1) = 12; //   N1

//bk   template for building  H8
	ta(1,32, na_ade,1) = 20; //   C8
	ta(2,32, na_ade,1) = 19; //   N7
	ta(3,32, na_ade,1) = 21; //   N9

//bk   identify atoms that require chi angles to be built
//jjh  this is none for nucleic acids (for now)

//bk   hbonding information
//jjh set the next two to nonsense values
	HNpos( na_ade,1) = -1; // position of backbone HN
	HApos( na_ade,1) = -1; // position of backbone HA

	nH_polar( na_ade,1) =  2; // number of polar hydrogens
	nH_aromatic( na_ade,1) =  2; // number of aromatic hydrogens
	nH_apolar( na_ade,1) =  7; // number of apolar hydrogens
	nacceptors( na_ade,1) =  8; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1, na_ade,1) = 30; // 1H6
	Hpos_polar( 2, na_ade,1) = 31; // 2H6

//bk   atom numbers for aromatic hydrogens
	Hpos_aromatic( 1, na_ade,1) = 29; //  H2
	Hpos_aromatic( 2, na_ade,1) = 32; //  H8

//bk   atom numbers for apolar hydrogens
	Hpos_apolar( 1, na_ade,1) = 22; //1H5*
	Hpos_apolar( 2, na_ade,1) = 23; //2H5*
	Hpos_apolar( 3, na_ade,1) = 24; // H4*
	Hpos_apolar( 4, na_ade,1) = 25; // H3*
	Hpos_apolar( 5, na_ade,1) = 26; //1H2*
	Hpos_apolar( 6, na_ade,1) = 27; //2H2*
	Hpos_apolar( 7, na_ade,1) = 28; // H1*

//bk   acceptor positions
	accpt_pos( 1, na_ade, 1) =  2; //  O1P
	accpt_pos( 2, na_ade, 1) =  3; //  O2P
	accpt_pos( 3, na_ade, 1) =  4; //  O5*
	accpt_pos( 4, na_ade, 1) =  9; //  O3*
	accpt_pos( 5, na_ade, 1) = 14; //  N3
	accpt_pos( 6, na_ade, 1) = 19; //  N7
	accpt_pos( 7, na_ade, 1) =  7; //  O4*
	accpt_pos( 8, na_ade, 1) = 12; //  N1

//ja major groove specificity hbonders
	major_groove_hb_atom[ na_ade ].assign( natoms(na_ade,1), false );
	major_groove_hb_atom[ na_ade ][ 19 ] = true;
	major_groove_hb_atom[ na_ade ][ 31 ] = true;

//bk   hydrogens on each atom

	nhydrogens_on_atm( 5, na_ade, 1) =  2; //  C5*
	hydrogens_on_atm( 1, 5, na_ade, 1) = 22; // 1H5*
	hydrogens_on_atm( 2, 5, na_ade, 1) = 23; // 2H5*

	nhydrogens_on_atm( 6, na_ade, 1) =  1; //  C4*
	hydrogens_on_atm( 1, 6, na_ade, 1) =  24; //  H4*

	nhydrogens_on_atm( 8, na_ade, 1) =  1; //  C3*
	hydrogens_on_atm( 1, 8, na_ade, 1) =  25; //  H3*

	nhydrogens_on_atm(10, na_ade, 1) =  2; //  C2*
	hydrogens_on_atm( 1,10, na_ade, 1) =  26; //1H2*
	hydrogens_on_atm( 2,10, na_ade, 1) =  27; // 2H2*

	nhydrogens_on_atm(11, na_ade, 1) =  1; //  C1*
	hydrogens_on_atm( 1,11, na_ade, 1) =  28; //  H1*

	nhydrogens_on_atm(13, na_ade, 1) =  1; //  C2
	hydrogens_on_atm( 1,13, na_ade, 1) =  29; //  H2

	nhydrogens_on_atm(18, na_ade, 1) =  2; //  N6
	hydrogens_on_atm( 1,18, na_ade, 1) =  30; // 1H6
	hydrogens_on_atm( 2,18, na_ade, 1) =  31; // 2H6

	nhydrogens_on_atm(20, na_ade, 1) =  1; //  C8
	hydrogens_on_atm( 1,20, na_ade, 1) = 32; //  H8

//bk   template coordinates for the nucleic acid

///jjh See init_dna_gua() for references

	icoor( 1, 1, na_ade, 1) =    0.224; //   P
	icoor( 2, 1, na_ade, 1) =   -4.365; //   P
	icoor( 3, 1, na_ade, 1) =    2.383; //   P
	icoor( 1, 2, na_ade, 1) =    1.336; //   O1P
	icoor( 2, 2, na_ade, 1) =   -3.982; //   O1P
	icoor( 3, 2, na_ade, 1) =    3.290; //   O1P
	icoor( 1, 3, na_ade, 1) =    0.278; //   O2P
	icoor( 2, 3, na_ade, 1) =   -5.664; //   O2P
	icoor( 3, 3, na_ade, 1) =    1.666; //   O2P
	icoor( 1, 4, na_ade, 1) =    0.042; //   O5*
	icoor( 2, 4, na_ade, 1) =   -3.205; //   O5*
	icoor( 3, 4, na_ade, 1) =    1.307; //   O5*
	icoor( 1, 5, na_ade, 1) =   -1.014; //   C5*
	icoor( 2, 5, na_ade, 1) =   -3.256; //   C5*
	icoor( 3, 5, na_ade, 1) =    0.347; //   C5*
	icoor( 1, 6, na_ade, 1) =   -0.913; //   C4*
	icoor( 2, 6, na_ade, 1) =   -2.083; //   C4*
	icoor( 3, 6, na_ade, 1) =   -0.600; //   C4*
	icoor( 1, 7, na_ade, 1) =   -1.127; //   O4*
	icoor( 2, 7, na_ade, 1) =   -0.853; //   O4*
	icoor( 3, 7, na_ade, 1) =    0.133; //   O4*
	icoor( 1, 8, na_ade, 1) =    0.445; //   C3*
	icoor( 2, 8, na_ade, 1) =   -1.932; //   C3*
	icoor( 3, 8, na_ade, 1) =   -1.287; //   C3*
	icoor( 1, 9, na_ade, 1) =    0.272; //   O3*
	icoor( 2, 9, na_ade, 1) =   -1.450; //   O3*
	icoor( 3, 9, na_ade, 1) =   -2.624; //   O3*
	icoor( 1,10, na_ade, 1) =    1.149; //   C2*
	icoor( 2,10, na_ade, 1) =   -0.891; //   C2*
	icoor( 3,10, na_ade, 1) =   -0.438; //   C2*
	icoor( 1,11, na_ade, 1) =    0.000; //   C1*
	icoor( 2,11, na_ade, 1) =    0.000; //   C1*
	icoor( 3,11, na_ade, 1) =    0.000; //   C1*
	icoor( 1,12, na_ade, 1) =    0.451; //   N1
	icoor( 2,12, na_ade, 1) =    4.493; //   N1
	icoor( 3,12, na_ade, 1) =    2.459; //   N1
	icoor( 1,13, na_ade, 1) =    0.142; //   C2
	icoor( 2,13, na_ade, 1) =    4.189; //   C2
	icoor( 3,13, na_ade, 1) =    1.194; //   C2
	icoor( 1,14, na_ade, 1) =    0.016; //   N3
	icoor( 2,14, na_ade, 1) =    2.995; //   N3
	icoor( 3,14, na_ade, 1) =    0.619; //   N3
	icoor( 1,15, na_ade, 1) =    0.250; //   C4
	icoor( 2,15, na_ade, 1) =    2.016; //   C4
	icoor( 3,15, na_ade, 1) =    1.509; //   C4
	icoor( 1,16, na_ade, 1) =    0.579; //   C5
	icoor( 2,16, na_ade, 1) =    2.170; //   C5
	icoor( 3,16, na_ade, 1) =    2.844; //   C5
	icoor( 1,17, na_ade, 1) =    0.681; //   C6
	icoor( 2,17, na_ade, 1) =    3.485; //   C6
	icoor( 3,17, na_ade, 1) =    3.329; //   C6
	icoor( 1,18, na_ade, 1) =    0.990; //   N6
	icoor( 2,18, na_ade, 1) =    3.787; //   N6
	icoor( 3,18, na_ade, 1) =    4.592; //   N6
	icoor( 1,19, na_ade, 1) =    0.747; //   N7
	icoor( 2,19, na_ade, 1) =    0.934; //   N7
	icoor( 3,19, na_ade, 1) =    3.454; //   N7
	icoor( 1,20, na_ade, 1) =    0.520; //   C8
	icoor( 2,20, na_ade, 1) =    0.074; //   C8
	icoor( 3,20, na_ade, 1) =    2.491; //   C8
	icoor( 1,21, na_ade, 1) =    0.213; //   N9
	icoor( 2,21, na_ade, 1) =    0.660; //   N9
	icoor( 3,21, na_ade, 1) =    1.287; //   N9
	icoor( 1,22, na_ade, 1) =   -0.962; //  1H5*
	icoor( 2,22, na_ade, 1) =   -4.199; //  1H5*
	icoor( 3,22, na_ade, 1) =   -0.217; //  1H5*
	icoor( 1,23, na_ade, 1) =   -1.987; //  2H5*
	icoor( 2,23, na_ade, 1) =   -3.241; //  2H5*
	icoor( 3,23, na_ade, 1) =    0.859; //  2H5*
	icoor( 1,24, na_ade, 1) =   -1.673; //   H4*
	icoor( 2,24, na_ade, 1) =   -2.277; //   H4*
	icoor( 3,24, na_ade, 1) =   -1.372; //   H4*
	icoor( 1,25, na_ade, 1) =    1.002; //   H3*
	icoor( 2,25, na_ade, 1) =   -2.877; //   H3*
	icoor( 3,25, na_ade, 1) =   -1.364; //   H3*
	icoor( 1,26, na_ade, 1) =    1.904; //  1H2*
	icoor( 2,26, na_ade, 1) =   -0.335; //  1H2*
	icoor( 3,26, na_ade, 1) =   -1.013; //  1H2*
	icoor( 1,27, na_ade, 1) =    1.668; //  2H2*
	icoor( 2,27, na_ade, 1) =   -1.344; //  2H2*
	icoor( 3,27, na_ade, 1) =    0.420; //  2H2*
	icoor( 1,28, na_ade, 1) =   -0.119; //   H1*
	icoor( 2,28, na_ade, 1) =    0.800; //   H1*
	icoor( 3,28, na_ade, 1) =   -0.745; //   H1*
	icoor( 1,29, na_ade, 1) =   -0.033; //   H2
	icoor( 2,29, na_ade, 1) =    5.050; //   H2
	icoor( 3,29, na_ade, 1) =    0.533; //   H2
	icoor( 1,30, na_ade, 1) =    1.048; //  1H6
	icoor( 2,30, na_ade, 1) =    4.744; //  1H6
	icoor( 3,30, na_ade, 1) =    4.878; //  1H6
	icoor( 1,31, na_ade, 1) =    1.164; //  2H6
	icoor( 2,31, na_ade, 1) =    3.057; //  2H6
	icoor( 3,31, na_ade, 1) =    5.252; //  2H6
	icoor( 1,32, na_ade, 1) =    0.571; //   H8
	icoor( 2,32, na_ade, 1) =   -1.016; //   H8
	icoor( 3,32, na_ade, 1) =    2.634; //   H8

	if ( design::hydrate_dna ) {
		create_dna_h2o_variants( na_ade );
	}


}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_dna_cyt
///
/// @brief arrays filled to define the proerties of the nucleic acid dCytosine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// na_cyt - index number for CYT
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_dna_cyt()
{
	using namespace aaproperties_pack;
	using namespace param_aa;

//bk   Parameters for CYT -----------------------------
	nvar(na_cyt) = 1; // starting starting number of variants for CYT
	variant_type( aav_base, na_cyt, 1 ) = true; // type of na variant

//bk   variant1 ---------- CYT

	natoms(na_cyt,1) = 30; // number of total atoms
	nheavyatoms(na_cyt,1) = 19; // number of heavy atoms
	nchi(na_cyt,1) = 0; // number of chi angles

//bk   atom names CYT
	atom_name( 1, na_cyt,1) = " P  ";
	atom_name( 2, na_cyt,1) = " O1P";
	atom_name( 3, na_cyt,1) = " O2P";
	atom_name( 4, na_cyt,1) = " O5*";
	atom_name( 5, na_cyt,1) = " C5*";
	atom_name( 6, na_cyt,1) = " C4*";
	atom_name( 7, na_cyt,1) = " O4*";
	atom_name( 8, na_cyt,1) = " C3*";
	atom_name( 9, na_cyt,1) = " O3*";
	atom_name(10, na_cyt,1) = " C2*";
	atom_name(11, na_cyt,1) = " C1*";
	atom_name(12, na_cyt,1) = " N1 ";
	atom_name(13, na_cyt,1) = " C2 ";
	atom_name(14, na_cyt,1) = " O2 ";
	atom_name(15, na_cyt,1) = " N3 ";
	atom_name(16, na_cyt,1) = " C4 ";
	atom_name(17, na_cyt,1) = " N4 ";
	atom_name(18, na_cyt,1) = " C5 ";
	atom_name(19, na_cyt,1) = " C6 ";
	atom_name(20, na_cyt,1) = "1H5*";
	atom_name(21, na_cyt,1) = "2H5*";
	atom_name(22, na_cyt,1) = " H4*";
	atom_name(23, na_cyt,1) = " H3*";
	atom_name(24, na_cyt,1) = "1H2*";
	atom_name(25, na_cyt,1) = "2H2*";
	atom_name(26, na_cyt,1) = " H1*";
	atom_name(27, na_cyt,1) = "1H4 ";
	atom_name(28, na_cyt,1) = "2H4 ";
	atom_name(29, na_cyt,1) = " H5 ";
	atom_name(30, na_cyt,1) = " H6 ";

//jjh intra residue bonding
	nbonded_neighbors( 1, na_cyt, 1) = 3; // P
		bonded_neighbor(1, 1, na_cyt, 1) = 2; // P--O1P
		bonded_neighbor(2, 1, na_cyt, 1) = 3; // P--O2P
		bonded_neighbor(3, 1, na_cyt, 1) = 4; // P--O5*
	nbonded_neighbors( 2, na_cyt, 1) = 1; // O1P
		bonded_neighbor(1, 2, na_cyt, 1) = 1; // O1P--P
	nbonded_neighbors( 3, na_cyt, 1) = 1; // O2P
		bonded_neighbor(1, 3, na_cyt, 1) = 1; // O2P--P
	nbonded_neighbors( 4, na_cyt, 1) = 2; // O5*
		bonded_neighbor(1, 4, na_cyt, 1) = 1; // O5*--P
		bonded_neighbor(2, 4, na_cyt, 1) = 5; // O5*--C5*
	nbonded_neighbors( 5, na_cyt, 1) = 4; // C5*
		bonded_neighbor(1, 5, na_cyt, 1) = 4; // C5*--O5*
		bonded_neighbor(2, 5, na_cyt, 1) = 6; // C5*--C4*
		bonded_neighbor(3, 5, na_cyt, 1) =20; // C5*--1H5*
		bonded_neighbor(4, 5, na_cyt, 1) =21; // C5*--2H5*
	nbonded_neighbors( 6, na_cyt, 1) = 4; // C4*
		bonded_neighbor(1, 6, na_cyt, 1) = 5; // C4*--C5*
		bonded_neighbor(2, 6, na_cyt, 1) = 7; // C4*--O4*
		bonded_neighbor(3, 6, na_cyt, 1) = 8; // C4*--C3*
		bonded_neighbor(4, 6, na_cyt, 1) =22; // C4*--H4*
	nbonded_neighbors( 7, na_cyt, 1) = 2; // O4*
		bonded_neighbor(1, 7, na_cyt, 1) = 6; // O4*--C4*
		bonded_neighbor(2, 7, na_cyt, 1) =11; // O4*--C1*
	nbonded_neighbors( 8, na_cyt, 1) = 4; // C3*
		bonded_neighbor(1, 8, na_cyt, 1) = 6; // C3*--C4*
		bonded_neighbor(2, 8, na_cyt, 1) = 9; // C3*--O3*
		bonded_neighbor(3, 8, na_cyt, 1) =10; // C3*--C2*
		bonded_neighbor(4, 8, na_cyt, 1) =23; // C3*--H3*
	nbonded_neighbors( 9, na_cyt, 1) = 1; // O3*
		bonded_neighbor(1, 9, na_cyt, 1) = 8; // O3*--C3*
	nbonded_neighbors( 10, na_cyt, 1) = 4; // C2*
		bonded_neighbor(1,10, na_cyt, 1) = 8; // C2*--C3*
		bonded_neighbor(2,10, na_cyt, 1) =11; // C2*--C1*
		bonded_neighbor(3,10, na_cyt, 1) =24; // C2*--1H2*
		bonded_neighbor(4,10, na_cyt, 1) =25; // C2*--2H2*
	nbonded_neighbors( 11, na_cyt, 1) = 4; // C1*
		bonded_neighbor(1,11, na_cyt, 1) = 7; // C1*--O4*
		bonded_neighbor(2,11, na_cyt, 1) =10; // C1*--C2*
		bonded_neighbor(3,11, na_cyt, 1) =12; // C1*--N1
		bonded_neighbor(4,11, na_cyt, 1) =26; // C1*--H1*
	nbonded_neighbors( 12, na_cyt, 1) = 3; // N1
		bonded_neighbor(1,12, na_cyt, 1) =11; // N1--C1*
		bonded_neighbor(2,12, na_cyt, 1) =13; // N1--C2
		bonded_neighbor(3,12, na_cyt, 1) =19; // N1--C6
	nbonded_neighbors( 13, na_cyt, 1) = 3; // C2
		bonded_neighbor(1,13, na_cyt, 1) =12; // C2--N1
		bonded_neighbor(2,13, na_cyt, 1) =14; // C2--O2
		bonded_neighbor(3,13, na_cyt, 1) =15; // C2--N3
	nbonded_neighbors( 14, na_cyt, 1) = 1; // O2
		bonded_neighbor(1,14, na_cyt, 1) =13; // O2--C2
	nbonded_neighbors( 15, na_cyt, 1) = 2; // N3
		bonded_neighbor(1,15, na_cyt, 1) =13; // N3--C2
		bonded_neighbor(2,15, na_cyt, 1) =16; // N3--C4
	nbonded_neighbors( 16, na_cyt, 1) = 3; // C4
		bonded_neighbor(1,16, na_cyt, 1) =15; // C4--N3
		bonded_neighbor(2,16, na_cyt, 1) =17; // C4--N4
		bonded_neighbor(3,16, na_cyt, 1) =18; // C4--C5
	nbonded_neighbors( 17, na_cyt, 1) = 3; // N4
		bonded_neighbor(1,17, na_cyt, 1) =16; // N4--C4
		bonded_neighbor(2,17, na_cyt, 1) =27; // N4--1H4
		bonded_neighbor(3,17, na_cyt, 1) =28; // N4--2H4
	nbonded_neighbors( 18, na_cyt, 1) = 3; // C5
		bonded_neighbor(1,18, na_cyt, 1) =16; // C5--C4
		bonded_neighbor(2,18, na_cyt, 1) =19; // C5--C6
		bonded_neighbor(3,18, na_cyt, 1) =29; // C5--H5
	nbonded_neighbors( 19, na_cyt, 1) = 3; // C6
		bonded_neighbor(1,19, na_cyt, 1) =12; // C6--N1
		bonded_neighbor(2,19, na_cyt, 1) =18; // C6--C5
		bonded_neighbor(3,19, na_cyt, 1) =30; // C6--H6
	nbonded_neighbors( 20, na_cyt, 1) = 1; //1H5*
		bonded_neighbor(1,20, na_cyt, 1) = 5; //1H5*--C5*
	nbonded_neighbors( 21, na_cyt, 1) = 1; //2H5*
		bonded_neighbor(1,21, na_cyt, 1) = 5; //2H5*--C5*
	nbonded_neighbors( 22, na_cyt, 1) = 1; // H4*
		bonded_neighbor(1,22, na_cyt, 1) = 6; // H4*--C4*
	nbonded_neighbors( 23, na_cyt, 1) = 1; // H3*
		bonded_neighbor(1,23, na_cyt, 1) = 8; // H3*--C3*
	nbonded_neighbors( 24, na_cyt, 1) = 1; //1H2*
		bonded_neighbor(1,24, na_cyt, 1) =10; // 1H2*--C2*
	nbonded_neighbors( 25, na_cyt, 1) = 1; //2H2*
		bonded_neighbor(1,25, na_cyt, 1) =10; // 2H2*--C2*
	nbonded_neighbors( 26, na_cyt, 1) = 1; // H1*
		bonded_neighbor(1,26, na_cyt, 1) =11; // H1*--C1*
	nbonded_neighbors( 27, na_cyt, 1) = 1; //1H4
		bonded_neighbor(1,27, na_cyt, 1) =17; //1H4--N4
	nbonded_neighbors( 28, na_cyt, 1) = 1; //2H4
		bonded_neighbor(1,28, na_cyt, 1) =17; //2H4--N4
	nbonded_neighbors( 29, na_cyt, 1) = 1; // H5
		bonded_neighbor(1,29, na_cyt, 1) =18; // H5--C5
	nbonded_neighbors( 30, na_cyt, 1) = 1; // H6
		bonded_neighbor(1,30, na_cyt, 1) =19; // H6--C6

//jjh get the angles automatically
	get_angles_from_bonds(na_cyt, 1);

	atom_base( 1,na_cyt,1) =  2; //   P   --> O1P
	atom_base( 2,na_cyt,1) =  1; //   O1P --> P
	atom_base( 3,na_cyt,1) =  1; //   O2P --> P
	atom_base( 4,na_cyt,1) =  1; //   O5* --> P
	abase2( 4, na_cyt, 1)  =  5; //  C5* base2 for  O5*
	atom_base( 5,na_cyt,1) =  4; //   C5* --> O5*
	atom_base( 6,na_cyt,1) =  5; //   C4* --> C5*
	atom_base( 7,na_cyt,1) = 11; // O4*  --> C1* //abase=C4*
	abase2( 7, na_cyt, 1)  =  6; //  C4* base2 for  O4*
	atom_base( 8,na_cyt,1) =  6; //   C3* --> C4*
	atom_base( 9,na_cyt,1) =  8; //   O3* --> C3*
	atom_base(10,na_cyt,1) =  8; //   C2* --> C3*
	atom_base(11,na_cyt,1) = 10; //   C1* --> C2*
	atom_base(12,na_cyt,1) = 11; //   N1  --> C1*
	atom_base(13,na_cyt,1) = 12; //   C2  --> N1
	atom_base(14,na_cyt,1) = 13; //   O2  --> C2
	atom_base(15,na_cyt,1) = 13; //   N3  --> C2
	abase2(15, na_cyt, 1)  = 16; //  C4  base2 for  N3
	atom_base(16,na_cyt,1) = 18; //   C4  --> C5
	atom_base(17,na_cyt,1) = 16; //   N4  --> C4
	atom_base(18,na_cyt,1) = 19; //   C5  --> C6
	atom_base(19,na_cyt,1) = 12; //   C6  --> N1
	atom_base(20,na_cyt,1) =  5; //  1H5* --> C5*
	atom_base(21,na_cyt,1) =  5; //  2H5* --> C5*
	atom_base(22,na_cyt,1) =  6; //   H4* --> C4*
	atom_base(23,na_cyt,1) =  8; //   H3* --> C3*
	atom_base(24,na_cyt,1) = 10; //  1H2* --> C2*
	atom_base(25,na_cyt,1) = 10; //  2H2* --> C2*
	atom_base(26,na_cyt,1) = 11; //   H1* --> C1*
	atom_base(27,na_cyt,1) = 17; //  1H4  --> N4
	atom_base(28,na_cyt,1) = 17; //  2H4  --> N4
	atom_base(29,na_cyt,1) = 18; //   H5  --> C5
	atom_base(30,na_cyt,1) = 19; //   H6  --> C6

//bk   atom type numbers CYT
	fullatom_type( 1,na_cyt,1) = 21; // Phos   P
	fullatom_type( 2,na_cyt,1) = 15; // OOC    O1P
	fullatom_type( 3,na_cyt,1) = 15; // OOC    O2P
	fullatom_type( 4,na_cyt,1) = 14; // ONH2   O5*
	fullatom_type( 5,na_cyt,1) =  4; // CH2    C5*
	fullatom_type( 6,na_cyt,1) =  3; // CH1    C4*
	fullatom_type( 7,na_cyt,1) = 13; // OH     O4* // Oether type
	fullatom_type( 8,na_cyt,1) =  3; // CH1    C3*
	fullatom_type( 9,na_cyt,1) = 14; // ONH2   O3*
	fullatom_type(10,na_cyt,1) =  4; // CH2    C2*
	fullatom_type(11,na_cyt,1) =  3; // CH1    C1*
	fullatom_type(12,na_cyt,1) =  7; // Ntrp   N1
	fullatom_type(13,na_cyt,1) = 19; // CObb   C2
	fullatom_type(14,na_cyt,1) = 20; // OCbb   O2
	fullatom_type(15,na_cyt,1) =  8; // Nhis   N3
	fullatom_type(16,na_cyt,1) =  6; // aroC   C4
	fullatom_type(17,na_cyt,1) =  9; // NH2O   N4
	fullatom_type(18,na_cyt,1) =  6; // aroC   C5
	fullatom_type(19,na_cyt,1) =  6; // aroC   C6
	fullatom_type(20,na_cyt,1) = 23; // Hapo  1H5*
	fullatom_type(21,na_cyt,1) = 23; // Hapo  2H5*
	fullatom_type(22,na_cyt,1) = 23; // Hapo   H4*
	fullatom_type(23,na_cyt,1) = 23; // Hapo   H3*
	fullatom_type(24,na_cyt,1) = 23; // Hapo  1H2*
	fullatom_type(25,na_cyt,1) = 23; // Hapo  2H2*
	fullatom_type(26,na_cyt,1) = 23; // Hapo   H1*
	fullatom_type(27,na_cyt,1) = 22; // Hpol  1H4
	fullatom_type(28,na_cyt,1) = 22; // Hpol  2H4
	fullatom_type(29,na_cyt,1) = 24; // Haro   H5
	fullatom_type(30,na_cyt,1) = 24; // Haro   H6

//bk   atom partial charges CYT
	atomic_charge( 1, na_cyt,1) = 1.500; // P
	atomic_charge( 2, na_cyt,1) = -.780; // O1P
	atomic_charge( 3, na_cyt,1) = -.780; // O2P
	atomic_charge( 4, na_cyt,1) = -.570; // O5*
	atomic_charge( 5, na_cyt,1) = -.080; // C5*
	atomic_charge( 6, na_cyt,1) =  .160; // C4*
	atomic_charge( 7, na_cyt,1) = -.500; // O4*
	atomic_charge( 8, na_cyt,1) =  .010; // C3*
	atomic_charge( 9, na_cyt,1) = -.570; // O3*
	atomic_charge(10, na_cyt,1) = -.180; // C2*
	atomic_charge(11, na_cyt,1) =  .160; // C1*
	atomic_charge(12, na_cyt,1) = -.130; // N1
	atomic_charge(13, na_cyt,1) =  .520; // C2
	atomic_charge(14, na_cyt,1) = -.490; // O2
	atomic_charge(15, na_cyt,1) = -.660; // N3
	atomic_charge(16, na_cyt,1) =  .650; // C4
	atomic_charge(17, na_cyt,1) = -.750; // N4
	atomic_charge(18, na_cyt,1) = -.130; // C5
	atomic_charge(19, na_cyt,1) =  .050; // C6
	atomic_charge(20, na_cyt,1) =  .090; // 1H5*
	atomic_charge(21, na_cyt,1) =  .090; // 2H5*
	atomic_charge(22, na_cyt,1) =  .090; // H4*
	atomic_charge(23, na_cyt,1) =  .090; // H3*
	atomic_charge(24, na_cyt,1) =  .090; // 1H2*
	atomic_charge(25, na_cyt,1) =  .090; // 2H2*
	atomic_charge(26, na_cyt,1) =  .090; // H1*
	atomic_charge(27, na_cyt,1) =  .370; // 1H4
	atomic_charge(28, na_cyt,1) =  .330; // 2H4
	atomic_charge(29, na_cyt,1) =  .070; // H5
	atomic_charge(30, na_cyt,1) =  .170; // H6

	assert_total_charge( na_cyt, 1, -1.0);

//cmd  first sidechain atom
	first_scatom( na_cyt,1) = 12; //N1

//jjh anchor atoms for base mutations
	na_anchor(1, na_cyt, 1) = 11;
	na_anchor(2, na_cyt, 1) = 12;
	na_anchor(3, na_cyt, 1) = 13;

//bk   template atoms used for placing hydrogens

//bk   template for building  P
	ta(1, 1, na_cyt,1) =  4; //   O5*
	ta(2, 1, na_cyt,1) =  5; //   C5*
	ta(3, 1, na_cyt,1) =  6; //   C4*

//bk   template for building  O1P
	ta(1, 2, na_cyt,1) =  1; //   P
	ta(2, 2, na_cyt,1) =  5; //   C5*
	ta(3, 2, na_cyt,1) =  4; //   O5*

//bk   template for building  O2P
	ta(1, 3, na_cyt,1) =  1; //   P
	ta(2, 3, na_cyt,1) =  2; //   O1P
	ta(3, 3, na_cyt,1) =  4; //   O5*

//bk   template for building 1H5*
	ta(1,20, na_cyt,1) =  5; //   C5*
	ta(2,20, na_cyt,1) =  4; //   O5*
	ta(3,20, na_cyt,1) =  6; //   C4*

//bk   template for building 2H5*
	ta(1,21, na_cyt,1) =  5; //   C5*
	ta(2,21, na_cyt,1) =  4; //   O5*
	ta(3,21, na_cyt,1) =  6; //   C4*

//bk   template for building  H4*
	ta(1,22, na_cyt,1) =  6; //   C4*
	ta(2,22, na_cyt,1) =  7; //   O4*
	ta(3,22, na_cyt,1) =  5; //   C5*

//bk   template for building  H3*
	ta(1,23, na_cyt,1) =  8; //   C3*
	ta(2,23, na_cyt,1) = 10; //   C2*
	ta(3,23, na_cyt,1) =  6; //   C4*

//bk   template for building 1H2*
	ta(1,24, na_cyt,1) = 10; //   C2*
	ta(2,24, na_cyt,1) =  8; //   C3*
	ta(3,24, na_cyt,1) = 11; //   C1*

//bk   template for building 2H2*
	ta(1,25, na_cyt,1) = 10; //   C2*
	ta(2,25, na_cyt,1) =  8; //   C3*
	ta(3,25, na_cyt,1) = 11; //   C1*

//bk   template for building  H1*
	ta(1,26, na_cyt,1) = 11; //   C1*
	ta(2,26, na_cyt,1) = 10; //   C2*
	ta(3,26, na_cyt,1) =  7; //   O4*

//bk   template for building 1H4
	ta(1,27, na_cyt,1) = 17; //   N4
	ta(2,27, na_cyt,1) = 16; //   C4
	ta(3,27, na_cyt,1) = 15; //   N3

//bk   template for building 2H4
	ta(1,28, na_cyt,1) = 17; //   N4
	ta(2,28, na_cyt,1) = 16; //   C4
	ta(3,28, na_cyt,1) = 15; //   N3

//bk   template for building  H5
	ta(1,29, na_cyt,1) = 18; //   C5
	ta(2,29, na_cyt,1) = 19; //   C6
	ta(3,29, na_cyt,1) = 12; //   N1

//bk   template for building  H6
	ta(1,30, na_cyt,1) = 19; //   C6
	ta(2,30, na_cyt,1) = 18; //   C5
	ta(3,30, na_cyt,1) = 16; //   C4

//bk   identify atoms that require chi angles to be built
//jjh  this is none for nucleic acids (for now)

//bk   hbonding information
//jjh set the next two to nonsense values
	HNpos( na_cyt,1) = -1; // position of backbone HN
	HApos( na_cyt,1) = -1; // position of backbone HA

	nH_polar( na_cyt,1) =  2; // number of polar hydrogens
	nH_aromatic( na_cyt,1) =  2; // number of aromatic hydrogens
	nH_apolar( na_cyt,1) =  7; // number of apolar hydrogens
	nacceptors( na_cyt,1) =  7; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1, na_cyt,1) = 27; // 1H4
	Hpos_polar( 2, na_cyt,1) = 28; // 2H4

//bk   atom numbers for aromatic hydrogens
	Hpos_aromatic( 1, na_cyt,1) = 29; //  H5
	Hpos_aromatic( 2, na_cyt,1) = 30; //  H6

//bk   atom numbers for apolar hydrogens
	Hpos_apolar( 1, na_cyt,1) = 20; // 1H5*
	Hpos_apolar( 2, na_cyt,1) = 21; // 2H5*
	Hpos_apolar( 3, na_cyt,1) = 22; //  H4*
	Hpos_apolar( 4, na_cyt,1) = 23; //  H3*
	Hpos_apolar( 5, na_cyt,1) = 24; // 1H2*
	Hpos_apolar( 6, na_cyt,1) = 25; // 2H2*
	Hpos_apolar( 7, na_cyt,1) = 26; //  H1*

//bk   acceptor positions
	accpt_pos( 1, na_cyt, 1) =  2; //  O1P
	accpt_pos( 2, na_cyt, 1) =  3; //  O2P
	accpt_pos( 3, na_cyt, 1) =  4; //  O5*
	accpt_pos( 4, na_cyt, 1) =  9; //  O3*
	accpt_pos( 5, na_cyt, 1) = 14; //  O2
	accpt_pos( 6, na_cyt, 1) = 15; //  N3
	accpt_pos( 7, na_cyt, 1) =  7; //  O4*

//ja major groove specificity hbonders
	major_groove_hb_atom[ na_cyt ].assign( natoms(na_cyt,1), false );
	major_groove_hb_atom[ na_cyt ][ 27 ] = true;

//bk   hydrogens on each atom

	nhydrogens_on_atm( 5, na_cyt, 1) =  2; //  C5*
	hydrogens_on_atm( 1, 5, na_cyt, 1) = 20; // 1H5*
	hydrogens_on_atm( 2, 5, na_cyt, 1) = 21; // 2H5*

	nhydrogens_on_atm( 6, na_cyt, 1) =  1; //  C4*
	hydrogens_on_atm( 1, 6, na_cyt, 1) =  22; //  H4*

	nhydrogens_on_atm( 8, na_cyt, 1) =  1; //  C3*
	hydrogens_on_atm( 1, 8, na_cyt, 1) =  23; //  H3*

	nhydrogens_on_atm(10, na_cyt, 1) =  2; //  C2*
	hydrogens_on_atm( 1,10, na_cyt, 1) =  24; // 1H2*
	hydrogens_on_atm( 2,10, na_cyt, 1) =  25; // 2H2*

	nhydrogens_on_atm(11, na_cyt, 1) =  1; //  C1*
	hydrogens_on_atm( 1,11, na_cyt, 1) =  26; //  H1*

	nhydrogens_on_atm(17, na_cyt, 1) =  2; //  N4
	hydrogens_on_atm( 1,17, na_cyt, 1) =  27; // 1H4
	hydrogens_on_atm( 2,17, na_cyt, 1) =  28; // 2H4

	nhydrogens_on_atm(18, na_cyt, 1) =  1; //  C5
	hydrogens_on_atm( 1,18, na_cyt, 1) =  29; //  H5

	nhydrogens_on_atm(19, na_cyt, 1) =  1; //  C6
	hydrogens_on_atm( 1,19, na_cyt, 1) = 30; //  H6

//bk   template coordinates for the nucleic acid

///jjh See init_dna_gua() for references

	icoor( 1, 1, na_cyt, 1) =    0.224; //   P
	icoor( 2, 1, na_cyt, 1) =   -4.365; //   P
	icoor( 3, 1, na_cyt, 1) =    2.383; //   P
	icoor( 1, 2, na_cyt, 1) =    1.336; //   O1P
	icoor( 2, 2, na_cyt, 1) =   -3.982; //   O1P
	icoor( 3, 2, na_cyt, 1) =    3.290; //   O1P
	icoor( 1, 3, na_cyt, 1) =    0.278; //   O2P
	icoor( 2, 3, na_cyt, 1) =   -5.664; //   O2P
	icoor( 3, 3, na_cyt, 1) =    1.666; //   O2P
	icoor( 1, 4, na_cyt, 1) =    0.042; //   O5*
	icoor( 2, 4, na_cyt, 1) =   -3.205; //   O5*
	icoor( 3, 4, na_cyt, 1) =    1.307; //   O5*
	icoor( 1, 5, na_cyt, 1) =   -1.014; //   C5*
	icoor( 2, 5, na_cyt, 1) =   -3.256; //   C5*
	icoor( 3, 5, na_cyt, 1) =    0.347; //   C5*
	icoor( 1, 6, na_cyt, 1) =   -0.913; //   C4*
	icoor( 2, 6, na_cyt, 1) =   -2.083; //   C4*
	icoor( 3, 6, na_cyt, 1) =   -0.600; //   C4*
	icoor( 1, 7, na_cyt, 1) =   -1.127; //   O4*
	icoor( 2, 7, na_cyt, 1) =   -0.853; //   O4*
	icoor( 3, 7, na_cyt, 1) =    0.133; //   O4*
	icoor( 1, 8, na_cyt, 1) =    0.445; //   C3*
	icoor( 2, 8, na_cyt, 1) =   -1.932; //   C3*
	icoor( 3, 8, na_cyt, 1) =   -1.287; //   C3*
	icoor( 1, 9, na_cyt, 1) =    0.272; //   O3*
	icoor( 2, 9, na_cyt, 1) =   -1.450; //   O3*
	icoor( 3, 9, na_cyt, 1) =   -2.624; //   O3*
	icoor( 1,10, na_cyt, 1) =    1.149; //   C2*
	icoor( 2,10, na_cyt, 1) =   -0.891; //   C2*
	icoor( 3,10, na_cyt, 1) =   -0.438; //   C2*
	icoor( 1,11, na_cyt, 1) =    0.000; //   C1*
	icoor( 2,11, na_cyt, 1) =    0.000; //   C1*
	icoor( 3,11, na_cyt, 1) =    0.000; //   C1*
	icoor( 1,12, na_cyt, 1) =    0.212; //   N1
	icoor( 2,12, na_cyt, 1) =    0.668; //   N1
	icoor( 3,12, na_cyt, 1) =    1.294; //   N1
	icoor( 1,13, na_cyt, 1) =    0.374; //   C2
	icoor( 2,13, na_cyt, 1) =    2.055; //   C2
	icoor( 3,13, na_cyt, 1) =    1.315; //   C2
	icoor( 1,14, na_cyt, 1) =    0.388; //   O2
	icoor( 2,14, na_cyt, 1) =    2.673; //   O2
	icoor( 3,14, na_cyt, 1) =    0.240; //   O2
	icoor( 1,15, na_cyt, 1) =    0.511; //   N3
	icoor( 2,15, na_cyt, 1) =    2.687; //   N3
	icoor( 3,15, na_cyt, 1) =    2.504; //   N3
	icoor( 1,16, na_cyt, 1) =    0.491; //   C4
	icoor( 2,16, na_cyt, 1) =    1.984; //   C4
	icoor( 3,16, na_cyt, 1) =    3.638; //   C4
	icoor( 1,17, na_cyt, 1) =    0.631; //   N4
	icoor( 2,17, na_cyt, 1) =    2.649; //   N4
	icoor( 3,17, na_cyt, 1) =    4.788; //   N4
	icoor( 1,18, na_cyt, 1) =    0.328; //   C5
	icoor( 2,18, na_cyt, 1) =    0.569; //   C5
	icoor( 3,18, na_cyt, 1) =    3.645; //   C5
	icoor( 1,19, na_cyt, 1) =    0.193; //   C6
	icoor( 2,19, na_cyt, 1) =   -0.043; //   C6
	icoor( 3,19, na_cyt, 1) =    2.462; //   C6
	icoor( 1,20, na_cyt, 1) =   -0.962; //  1H5*
	icoor( 2,20, na_cyt, 1) =   -4.199; //  1H5*
	icoor( 3,20, na_cyt, 1) =   -0.217; //  1H5*
	icoor( 1,21, na_cyt, 1) =   -1.987; //  2H5*
	icoor( 2,21, na_cyt, 1) =   -3.241; //  2H5*
	icoor( 3,21, na_cyt, 1) =    0.859; //  2H5*
	icoor( 1,22, na_cyt, 1) =   -1.673; //   H4*
	icoor( 2,22, na_cyt, 1) =   -2.277; //   H4*
	icoor( 3,22, na_cyt, 1) =   -1.372; //   H4*
	icoor( 1,23, na_cyt, 1) =    1.002; //   H3*
	icoor( 2,23, na_cyt, 1) =   -2.877; //   H3*
	icoor( 3,23, na_cyt, 1) =   -1.364; //   H3*
	icoor( 1,24, na_cyt, 1) =    1.904; //  1H2*
	icoor( 2,24, na_cyt, 1) =   -0.335; //  1H2*
	icoor( 3,24, na_cyt, 1) =   -1.013; //  1H2*
	icoor( 1,25, na_cyt, 1) =    1.668; //  2H2*
	icoor( 2,25, na_cyt, 1) =   -1.344; //  2H2*
	icoor( 3,25, na_cyt, 1) =    0.420; //  2H2*
	icoor( 1,26, na_cyt, 1) =   -0.119; //   H1*
	icoor( 2,26, na_cyt, 1) =    0.800; //   H1*
	icoor( 3,26, na_cyt, 1) =   -0.745; //   H1*
	icoor( 1,27, na_cyt, 1) =    0.621; //  1H4
	icoor( 2,27, na_cyt, 1) =    2.154; //  1H4
	icoor( 3,27, na_cyt, 1) =    5.657; //  1H4
	icoor( 1,28, na_cyt, 1) =    0.746; //  2H4
	icoor( 2,28, na_cyt, 1) =    3.642; //  2H4
	icoor( 3,28, na_cyt, 1) =    4.780; //  2H4
	icoor( 1,29, na_cyt, 1) =    0.314; //   H5
	icoor( 2,29, na_cyt, 1) =    0.001; //   H5
	icoor( 3,29, na_cyt, 1) =    4.587; //   H5
	icoor( 1,30, na_cyt, 1) =    0.065; //   H6
	icoor( 2,30, na_cyt, 1) =   -1.135; //   H6
	icoor( 3,30, na_cyt, 1) =    2.429; //   H6

	if ( design::hydrate_dna ) {
		create_dna_h2o_variants( na_cyt );
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_dna_thy
///
/// @brief arrays filled to define the proerties of the nucleic acid dThymine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// na_thy - index number for THY
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_dna_thy()
{
	using namespace aaproperties_pack;
	using namespace param_aa;

//bk   Parameters for THY -----------------------------
	nvar(na_thy) = 1; // starting starting number of variants for THY
	variant_type( aav_base, na_thy, 1 ) = true; // type of na variant

//bk   variant1 ---------- THY

	natoms(na_thy,1) = 32; // number of total atoms
	nheavyatoms(na_thy,1) = 20; // number of heavy atoms
	nchi(na_thy,1) = 0; // number of chi angles

//bk   atom names THY
	atom_name( 1, na_thy,1) = " P  ";
	atom_name( 2, na_thy,1) = " O1P";
	atom_name( 3, na_thy,1) = " O2P";
	atom_name( 4, na_thy,1) = " O5*";
	atom_name( 5, na_thy,1) = " C5*";
	atom_name( 6, na_thy,1) = " C4*";
	atom_name( 7, na_thy,1) = " O4*";
	atom_name( 8, na_thy,1) = " C3*";
	atom_name( 9, na_thy,1) = " O3*";
	atom_name(10, na_thy,1) = " C2*";
	atom_name(11, na_thy,1) = " C1*";
	atom_name(12, na_thy,1) = " N1 ";
	atom_name(13, na_thy,1) = " C2 ";
	atom_name(14, na_thy,1) = " O2 ";
	atom_name(15, na_thy,1) = " N3 ";
	atom_name(16, na_thy,1) = " C4 ";
	atom_name(17, na_thy,1) = " O4 ";
	atom_name(18, na_thy,1) = " C5 ";
	atom_name(19, na_thy,1) = " C5M";
	atom_name(20, na_thy,1) = " C6 ";
	atom_name(21, na_thy,1) = "1H5*";
	atom_name(22, na_thy,1) = "2H5*";
	atom_name(23, na_thy,1) = " H4*";
	atom_name(24, na_thy,1) = " H3*";
	atom_name(25, na_thy,1) = "1H2*";
	atom_name(26, na_thy,1) = "2H2*";
	atom_name(27, na_thy,1) = " H1*";
	atom_name(28, na_thy,1) = " H3 ";
	atom_name(29, na_thy,1) = "1H5M";
	atom_name(30, na_thy,1) = "2H5M";
	atom_name(31, na_thy,1) = "3H5M";
	atom_name(32, na_thy,1) = " H6 ";

//jjh intra residue bonding
	nbonded_neighbors( 1, na_thy, 1) = 3; // P
		bonded_neighbor(1, 1, na_thy, 1) = 2; // P--O1P
		bonded_neighbor(2, 1, na_thy, 1) = 3; // P--O2P
		bonded_neighbor(3, 1, na_thy, 1) = 4; // P--O5*
	nbonded_neighbors( 2, na_thy, 1) = 1; // O1P
		bonded_neighbor(1, 2, na_thy, 1) = 1; // O1P--P
	nbonded_neighbors( 3, na_thy, 1) = 1; // O2P
		bonded_neighbor(1, 3, na_thy, 1) = 1; // O2P--P
	nbonded_neighbors( 4, na_thy, 1) = 2; // O5*
		bonded_neighbor(1, 4, na_thy, 1) = 1; // O5*--P
		bonded_neighbor(2, 4, na_thy, 1) = 5; // O5*--C5*
	nbonded_neighbors( 5, na_thy, 1) = 4; // C5*
		bonded_neighbor(1, 5, na_thy, 1) = 4; // C5*--O5*
		bonded_neighbor(2, 5, na_thy, 1) = 6; // C5*--C4*
		bonded_neighbor(3, 5, na_thy, 1) =21; // C5*--1H5*
		bonded_neighbor(4, 5, na_thy, 1) =22; // C5*--2H5*
	nbonded_neighbors( 6, na_thy, 1) = 4; // C4*
		bonded_neighbor(1, 6, na_thy, 1) = 5; // C4*--C5*
		bonded_neighbor(2, 6, na_thy, 1) = 7; // C4*--O4*
		bonded_neighbor(3, 6, na_thy, 1) = 8; // C4*--C3*
		bonded_neighbor(4, 6, na_thy, 1) =23; // C4*--H4*
	nbonded_neighbors( 7, na_thy, 1) = 2; // O4*
		bonded_neighbor(1, 7, na_thy, 1) = 6; // O4*--C4*
		bonded_neighbor(2, 7, na_thy, 1) =11; // O4*--C1*
	nbonded_neighbors( 8, na_thy, 1) = 4; // C3*
		bonded_neighbor(1, 8, na_thy, 1) = 6; // C3*--C4*
		bonded_neighbor(2, 8, na_thy, 1) = 9; // C3*--O3*
		bonded_neighbor(3, 8, na_thy, 1) =10; // C3*--C2*
		bonded_neighbor(4, 8, na_thy, 1) =24; // C3*--H3*
	nbonded_neighbors( 9, na_thy, 1) = 1; // O3*
		bonded_neighbor(1, 9, na_thy, 1) = 8; // O3*--C3*
	nbonded_neighbors( 10, na_thy, 1) = 4; // C2*
		bonded_neighbor(1,10, na_thy, 1) = 8; // C2*--C3*
		bonded_neighbor(2,10, na_thy, 1) =11; // C2*--C1*
		bonded_neighbor(3,10, na_thy, 1) =25; // C2*--1H2*
		bonded_neighbor(4,10, na_thy, 1) =26; // C2*--2H2*
	nbonded_neighbors( 11, na_thy, 1) = 4; // C1*
		bonded_neighbor(1,11, na_thy, 1) = 7; // C1*--O4*
		bonded_neighbor(2,11, na_thy, 1) =10; // C1*--C2*
		bonded_neighbor(3,11, na_thy, 1) =12; // C1*--N1
		bonded_neighbor(4,11, na_thy, 1) =27; // C1*--H1*
	nbonded_neighbors( 12, na_thy, 1) = 3; // N1
		bonded_neighbor(1,12, na_thy, 1) =11; // N1--C1*
		bonded_neighbor(2,12, na_thy, 1) =13; // N1--C2
		bonded_neighbor(3,12, na_thy, 1) =20; // N1--C6
	nbonded_neighbors( 13, na_thy, 1) = 3; // C2
		bonded_neighbor(1,13, na_thy, 1) =12; // C2--N1
		bonded_neighbor(2,13, na_thy, 1) =14; // C2--O2
		bonded_neighbor(3,13, na_thy, 1) =15; // C2--N3
	nbonded_neighbors( 14, na_thy, 1) = 1; // O2
		bonded_neighbor(1,14, na_thy, 1) =13; // O2--C2
	nbonded_neighbors( 15, na_thy, 1) = 3; // N3
		bonded_neighbor(1,15, na_thy, 1) =13; // N3--C2
		bonded_neighbor(2,15, na_thy, 1) =16; // N3--C4
		bonded_neighbor(3,15, na_thy, 1) =28; // N3--H3
	nbonded_neighbors( 16, na_thy, 1) = 3; // C4
		bonded_neighbor(1,16, na_thy, 1) =15; // C4--N3
		bonded_neighbor(2,16, na_thy, 1) =17; // C4--O4
		bonded_neighbor(3,16, na_thy, 1) =18; // C4--C5
	nbonded_neighbors( 17, na_thy, 1) = 1; // O4
		bonded_neighbor(1,17, na_thy, 1) =16; // O4--C4
	nbonded_neighbors( 18, na_thy, 1) = 3; // C5
		bonded_neighbor(1,18, na_thy, 1) =16; // C5--C4
		bonded_neighbor(2,18, na_thy, 1) =19; // C5--C5M
		bonded_neighbor(3,18, na_thy, 1) =20; // C5--C6
	nbonded_neighbors( 19, na_thy, 1) = 4; // C5M
		bonded_neighbor(1,19, na_thy, 1) =18; // C5M--C5
		bonded_neighbor(2,19, na_thy, 1) =29; // C5M--1H5M
		bonded_neighbor(3,19, na_thy, 1) =30; // C5M--2H5M
		bonded_neighbor(4,19, na_thy, 1) =31; // C5M--3H5M
	nbonded_neighbors( 20, na_thy, 1) = 3; // C6
		bonded_neighbor(1,20, na_thy, 1) =12; // C6--N1
		bonded_neighbor(2,20, na_thy, 1) =18; // C6--C5
		bonded_neighbor(3,20, na_thy, 1) =32; // C6--H6
	nbonded_neighbors( 21, na_thy, 1) = 1; //1H5*
		bonded_neighbor(1,21, na_thy, 1) = 5; //1H5*--C5*
	nbonded_neighbors( 22, na_thy, 1) = 1; //2H5*
		bonded_neighbor(1,22, na_thy, 1) = 5; //2H5*--C5*
	nbonded_neighbors( 23, na_thy, 1) = 1; // H4*
		bonded_neighbor(1,23, na_thy, 1) = 6; // H4*--C4*
	nbonded_neighbors( 24, na_thy, 1) = 1; // H3*
		bonded_neighbor(1,24, na_thy, 1) = 8; // H3*--C3*
	nbonded_neighbors( 25, na_thy, 1) = 1; //1H2*
		bonded_neighbor(1,25, na_thy, 1) =10; //1H2*--C2*
	nbonded_neighbors( 26, na_thy, 1) = 1; //2H2*
		bonded_neighbor(1,26, na_thy, 1) =10; //2H2*--C2*
	nbonded_neighbors( 27, na_thy, 1) = 1; // H1*
		bonded_neighbor(1,27, na_thy, 1) =11; // H1*--C1*
	nbonded_neighbors( 28, na_thy, 1) = 1; // H3
		bonded_neighbor(1,28, na_thy, 1) =15; // H3--N3
	nbonded_neighbors( 29, na_thy, 1) = 1; //1H5M
		bonded_neighbor(1,29, na_thy, 1) =19; //1H5M--C5M
	nbonded_neighbors( 30, na_thy, 1) = 1; //2H5M
		bonded_neighbor(1,30, na_thy, 1) =19; //2H5M--C5M
	nbonded_neighbors( 31, na_thy, 1) = 1; //3H5M
		bonded_neighbor(1,31, na_thy, 1) =19; //3H5M--C5M
	nbonded_neighbors( 32, na_thy, 1) = 1; // H6
		bonded_neighbor(1,32, na_thy, 1) =20; // H6--C6

//jjh get the angles automatically
	get_angles_from_bonds(na_thy, 1);

	atom_base( 1,na_thy,1) =  2; //   P   --> O1P
	atom_base( 2,na_thy,1) =  1; //   O1P --> P
	atom_base( 3,na_thy,1) =  1; //   O2P --> P
	atom_base( 4,na_thy,1) =  1; //   O5* --> P
	abase2( 4, na_thy, 1) =  5; //  C5* base2 for  O5*
	atom_base( 5,na_thy,1) =  4; //   C5* --> O5*
	atom_base( 6,na_thy,1) =  5; //   C4* --> C5*
	atom_base( 7,na_thy,1) = 11; // O4*  --> C1* //abase2=C4*
	abase2( 7, na_thy, 1) =  6; //  C4* base2 for  O4*
	atom_base( 8,na_thy,1) =  6; //   C3* --> C4*
	atom_base( 9,na_thy,1) =  8; //   O3* --> C3*
	atom_base(10,na_thy,1) =  8; //   C2* --> C3*
	atom_base(11,na_thy,1) = 10; //   C1* --> C2*
	atom_base(12,na_thy,1) = 11; //   N1  --> C1*
	atom_base(13,na_thy,1) = 12; //   C2  --> N1
	atom_base(14,na_thy,1) = 13; //   O2  --> C2
	atom_base(15,na_thy,1) = 13; //   N3  --> C2
	atom_base(16,na_thy,1) = 18; //   C4  --> C5
	atom_base(17,na_thy,1) = 16; //   O4  --> C4
	atom_base(18,na_thy,1) = 20; //   C5  --> C6
	atom_base(19,na_thy,1) = 18; //   C5M --> C5
	atom_base(20,na_thy,1) = 12; //   C6  --> N1
	atom_base(21,na_thy,1) =  5; //  1H5* --> C5*
	atom_base(22,na_thy,1) =  5; //  2H5* --> C5*
	atom_base(23,na_thy,1) =  6; //   H4* --> C4*
	atom_base(24,na_thy,1) =  8; //   H3* --> C3*
	atom_base(25,na_thy,1) = 10; //  1H2* --> C2*
	atom_base(26,na_thy,1) = 10; //  2H2* --> C2*
	atom_base(27,na_thy,1) = 11; //   H1* --> C1*
	atom_base(28,na_thy,1) = 15; //   H3  --> N3
	atom_base(29,na_thy,1) = 19; //  1H5M --> C5M
	atom_base(30,na_thy,1) = 19; //  2H5M --> C5M
	atom_base(31,na_thy,1) = 19; //  3H5M --> C5M
	atom_base(32,na_thy,1) = 20; //   H6  --> C6

//bk   atom type numbers THY
	fullatom_type( 1,na_thy,1) = 21; // Phos   P
	fullatom_type( 2,na_thy,1) = 15; // OOC    O1P
	fullatom_type( 3,na_thy,1) = 15; // OOC    O2P
	fullatom_type( 4,na_thy,1) = 14; // ONH2   O5*
	fullatom_type( 5,na_thy,1) =  4; // CH2    C5*
	fullatom_type( 6,na_thy,1) =  3; // CH1    C4*
	fullatom_type( 7,na_thy,1) = 13; // OH     O4* // Oether type
	fullatom_type( 8,na_thy,1) =  3; // CH1    C3*
	fullatom_type( 9,na_thy,1) = 14; // ONH2   O3*
	fullatom_type(10,na_thy,1) =  4; // CH2    C2*
	fullatom_type(11,na_thy,1) =  3; // CH1    C1*
	fullatom_type(12,na_thy,1) =  7; // Ntrp   N1
	fullatom_type(13,na_thy,1) = 19; // CObb   C2
	fullatom_type(14,na_thy,1) = 20; // OCbb   O2
	fullatom_type(15,na_thy,1) =  7; // Ntrp   N3
	fullatom_type(16,na_thy,1) = 19; // CObb   C4
	fullatom_type(17,na_thy,1) = 20; // OCbb   O4
	fullatom_type(18,na_thy,1) =  6; // aroC   C5
	fullatom_type(19,na_thy,1) =  5; // CH3    C5M
	fullatom_type(20,na_thy,1) =  6; // aroC   C6
	fullatom_type(21,na_thy,1) = 23; // Hapo  1H5*
	fullatom_type(22,na_thy,1) = 23; // Hapo  2H5*
	fullatom_type(23,na_thy,1) = 23; // Hapo   H4*
	fullatom_type(24,na_thy,1) = 23; // Hapo   H3*
	fullatom_type(25,na_thy,1) = 23; // Hapo  1H2*
	fullatom_type(26,na_thy,1) = 23; // Hapo  2H2*
	fullatom_type(27,na_thy,1) = 23; // Hapo   H1*
	fullatom_type(28,na_thy,1) = 22; // Hpol   H3
	fullatom_type(29,na_thy,1) = 23; // Hapo  1H5M
	fullatom_type(30,na_thy,1) = 23; // Hapo  2H5M
	fullatom_type(31,na_thy,1) = 23; // Hapo  3H5M
	fullatom_type(32,na_thy,1) = 24; // Haro   H6

//bk   atom partial charges THY
	atomic_charge( 1, na_thy,1) = 1.500; // P
	atomic_charge( 2, na_thy,1) = -.780; // O1P
	atomic_charge( 3, na_thy,1) = -.780; // O2P
	atomic_charge( 4, na_thy,1) = -.570; // O5*
	atomic_charge( 5, na_thy,1) = -.080; // C5*
	atomic_charge( 6, na_thy,1) =  .160; // C4*
	atomic_charge( 7, na_thy,1) = -.500; // O4*
	atomic_charge( 8, na_thy,1) =  .010; // C3*
	atomic_charge( 9, na_thy,1) = -.570; // O3*
	atomic_charge(10, na_thy,1) = -.180; // C2*
	atomic_charge(11, na_thy,1) =  .160; // C1*
	atomic_charge(12, na_thy,1) = -.340; // N1
	atomic_charge(13, na_thy,1) =  .510; // C2
	atomic_charge(14, na_thy,1) = -.410; // O2
	atomic_charge(15, na_thy,1) = -.460; // N3
	atomic_charge(16, na_thy,1) =  .500; // C4
	atomic_charge(17, na_thy,1) = -.450; // O4
	atomic_charge(18, na_thy,1) = -.150; // C5
	atomic_charge(19, na_thy,1) = -.110; // C5M
	atomic_charge(20, na_thy,1) =  .170; // C6
	atomic_charge(21, na_thy,1) =  .090; // 1H5*
	atomic_charge(22, na_thy,1) =  .090; // 2H5*
	atomic_charge(23, na_thy,1) =  .090; // H4*
	atomic_charge(24, na_thy,1) =  .090; // H3*
	atomic_charge(25, na_thy,1) =  .090; // 1H2*
	atomic_charge(26, na_thy,1) =  .090; // 2H2*
	atomic_charge(27, na_thy,1) =  .090; // H1*
	atomic_charge(28, na_thy,1) =  .360; // H3
	atomic_charge(29, na_thy,1) =  .070; // 1H5M
	atomic_charge(30, na_thy,1) =  .070; // 2H5M
	atomic_charge(31, na_thy,1) =  .070; // 3H5M
	atomic_charge(32, na_thy,1) =  .170; // H6

	assert_total_charge( na_thy, 1, -1.0);

//cmd  first sidechain atom
	first_scatom( na_thy,1) = 12; //N1

//jjh anchor atoms for base mutations
	na_anchor(1, na_thy, 1) = 11;
	na_anchor(2, na_thy, 1) = 12;
	na_anchor(3, na_thy, 1) = 13;

//bk   template atoms used for placing hydrogens

//bk   template for building  P
	ta(1, 1, na_thy,1) =  4; //   O5*
	ta(2, 1, na_thy,1) =  5; //   C5*
	ta(3, 1, na_thy,1) =  6; //   C4*

//bk   template for building  O1P
	ta(1, 2, na_thy,1) =  1; //   P
	ta(2, 2, na_thy,1) =  5; //   C5*
	ta(3, 2, na_thy,1) =  4; //   O5*

//bk   template for building  O2P
	ta(1, 3, na_thy,1) =  1; //   P
	ta(2, 3, na_thy,1) =  2; //   O1P
	ta(3, 3, na_thy,1) =  4; //   O5*

//bk   template for building 1H5*
	ta(1,21, na_thy,1) =  5; //   C5*
	ta(2,21, na_thy,1) =  4; //   O5*
	ta(3,21, na_thy,1) =  6; //   C4*

//bk   template for building 2H5*
	ta(1,22, na_thy,1) =  5; //   C5*
	ta(2,22, na_thy,1) =  4; //   O5*
	ta(3,22, na_thy,1) =  6; //   C4*

//bk   template for building  H4*
	ta(1,23, na_thy,1) =  6; //   C4*
	ta(2,23, na_thy,1) =  7; //   O4*
	ta(3,23, na_thy,1) =  5; //   C5*

//bk   template for building  H3*
	ta(1,24, na_thy,1) =  8; //   C3*
	ta(2,24, na_thy,1) = 10; //   C2*
	ta(3,24, na_thy,1) =  6; //   C4*

//bk   template for building 1H2*
	ta(1,25, na_thy,1) = 10; //   C2*
	ta(2,25, na_thy,1) =  8; //   C3*
	ta(3,25, na_thy,1) = 11; //   C1*

//bk   template for building 2H2*
	ta(1,26, na_thy,1) = 10; //   C2*
	ta(2,26, na_thy,1) =  8; //   C3*
	ta(3,26, na_thy,1) = 11; //   C1*

//bk   template for building  H1*
	ta(1,27, na_thy,1) = 11; //   C1*
	ta(2,27, na_thy,1) = 10; //   C2*
	ta(3,27, na_thy,1) =  7; //   O4*

//bk   template for building  H3
	ta(1,28, na_thy,1) = 15; //   N3
	ta(2,28, na_thy,1) = 13; //   C2
	ta(3,28, na_thy,1) = 16; //   C4

//bk   template for building 1H5M
	ta(1,29, na_thy,1) = 19; //   C5M
	ta(2,29, na_thy,1) = 18; //   C5
	ta(3,29, na_thy,1) = 16; //   C4

//bk   template for building 2H5M
	ta(1,30, na_thy,1) = 19; //   C5M
	ta(2,30, na_thy,1) = 18; //   C5
	ta(3,30, na_thy,1) = 16; //   C4

//bk   template for building 3H5M
	ta(1,31, na_thy,1) = 19; //   C5M
	ta(2,31, na_thy,1) = 18; //   C5
	ta(3,31, na_thy,1) = 16; //   C4

//bk   template for building  H6
	ta(1,32, na_thy,1) = 20; //   C6
	ta(2,32, na_thy,1) = 18; //   C5
	ta(3,32, na_thy,1) = 16; //   C4

//bk   identify atoms that require chi angles to be built
//jjh  this is none for nucleic acids (for now)

//bk   hbonding information
//jjh set the next two to nonsense values
	HNpos( na_thy,1) = -1; // position of backbone HN
	HApos( na_thy,1) = -1; // position of backbone HA

	nH_polar( na_thy,1) =  1; // number of polar hydrogens
	nH_aromatic( na_thy,1) =  1; // number of aromatic hydrogens
	nH_apolar( na_thy,1) =  10; // number of apolar hydrogens
	nacceptors( na_thy,1) =  7; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1, na_thy,1) = 28; //  H3

//bk   atom numbers for aromatic hydrogens
	Hpos_aromatic( 1, na_thy,1) = 32; //  H6

//bk   atom numbers for apolar hydrogens
	Hpos_apolar( 1, na_thy,1) = 21; // 1H5*
	Hpos_apolar( 2, na_thy,1) = 22; // 2H5*
	Hpos_apolar( 3, na_thy,1) = 23; //  H4*
	Hpos_apolar( 4, na_thy,1) = 24; //  H3*
	Hpos_apolar( 5, na_thy,1) = 25; // 1H2*
	Hpos_apolar( 6, na_thy,1) = 26; // 2H2*
	Hpos_apolar( 7, na_thy,1) = 27; //  H1*
	Hpos_apolar( 8, na_thy,1) = 29; // 1H5M
	Hpos_apolar( 9, na_thy,1) = 30; // 2H5M
	Hpos_apolar(10, na_thy,1) = 31; // 3H5M

//bk   acceptor positions
	accpt_pos( 1, na_thy, 1) =  2; //  O1P
	accpt_pos( 2, na_thy, 1) =  3; //  O2P
	accpt_pos( 3, na_thy, 1) =  4; //  O5*
	accpt_pos( 4, na_thy, 1) =  9; //  O3*
	accpt_pos( 5, na_thy, 1) = 14; //  O2*
	accpt_pos( 6, na_thy, 1) = 17; //  O4
	accpt_pos( 7, na_thy, 1) =  7; //  O4*

//ja major groove specific hbonders
	major_groove_hb_atom[ na_thy ].assign( natoms(na_thy,1), false );
	major_groove_hb_atom[ na_thy ][ 17 ] = true;

//bk   hydrogens on each atom

	nhydrogens_on_atm( 5, na_thy, 1) =  2; //  C5*
	hydrogens_on_atm( 1, 5, na_thy, 1) = 21; // 1H5*
	hydrogens_on_atm( 2, 5, na_thy, 1) = 22; // 2H5*

	nhydrogens_on_atm( 6, na_thy, 1) =  1; //  C4*
	hydrogens_on_atm( 1, 6, na_thy, 1) =  23; //  H4*

	nhydrogens_on_atm( 8, na_thy, 1) =  1; //  C3*
	hydrogens_on_atm( 1, 8, na_thy, 1) =  24; //  H3*

	nhydrogens_on_atm(10, na_thy, 1) =  2; //  C2*
	hydrogens_on_atm( 1,10, na_thy, 1) =  25; // 1H2*
	hydrogens_on_atm( 2,10, na_thy, 1) =  26; // 2H2*

	nhydrogens_on_atm(11, na_thy, 1) =  1; //  C1*
	hydrogens_on_atm( 1,11, na_thy, 1) =  27; //  H1*

	nhydrogens_on_atm(15, na_thy, 1) =  1; //  N3
	hydrogens_on_atm( 1,15, na_thy, 1) =  28; //  H3

	nhydrogens_on_atm(19, na_thy, 1) =  1; //  C5M
	hydrogens_on_atm( 1,19, na_thy, 1) =  29; // 1H5M
	hydrogens_on_atm( 2,19, na_thy, 1) =  30; // 2H5M
	hydrogens_on_atm( 3,19, na_thy, 1) =  31; // 3H5M

	nhydrogens_on_atm(20, na_thy, 1) =  1; //  C6
	hydrogens_on_atm( 1,20, na_thy, 1) = 32; //  H6

//bk   template coordinates for the nucleic acid

///jjh See init_dna_gua() for references

	icoor( 1, 1, na_thy, 1) =    0.224; //   P
	icoor( 2, 1, na_thy, 1) =   -4.365; //   P
	icoor( 3, 1, na_thy, 1) =    2.383; //   P
	icoor( 1, 2, na_thy, 1) =    1.336; //   O1P
	icoor( 2, 2, na_thy, 1) =   -3.982; //   O1P
	icoor( 3, 2, na_thy, 1) =    3.290; //   O1P
	icoor( 1, 3, na_thy, 1) =    0.278; //   O2P
	icoor( 2, 3, na_thy, 1) =   -5.664; //   O2P
	icoor( 3, 3, na_thy, 1) =    1.666; //   O2P
	icoor( 1, 4, na_thy, 1) =    0.042; //   O5*
	icoor( 2, 4, na_thy, 1) =   -3.205; //   O5*
	icoor( 3, 4, na_thy, 1) =    1.307; //   O5*
	icoor( 1, 5, na_thy, 1) =   -1.014; //   C5*
	icoor( 2, 5, na_thy, 1) =   -3.256; //   C5*
	icoor( 3, 5, na_thy, 1) =    0.347; //   C5*
	icoor( 1, 6, na_thy, 1) =   -0.913; //   C4*
	icoor( 2, 6, na_thy, 1) =   -2.083; //   C4*
	icoor( 3, 6, na_thy, 1) =   -0.600; //   C4*
	icoor( 1, 7, na_thy, 1) =   -1.127; //   O4*
	icoor( 2, 7, na_thy, 1) =   -0.853; //   O4*
	icoor( 3, 7, na_thy, 1) =    0.133; //   O4*
	icoor( 1, 8, na_thy, 1) =    0.445; //   C3*
	icoor( 2, 8, na_thy, 1) =   -1.932; //   C3*
	icoor( 3, 8, na_thy, 1) =   -1.287; //   C3*
	icoor( 1, 9, na_thy, 1) =    0.272; //   O3*
	icoor( 2, 9, na_thy, 1) =   -1.450; //   O3*
	icoor( 3, 9, na_thy, 1) =   -2.624; //   O3*
	icoor( 1,10, na_thy, 1) =    1.149; //   C2*
	icoor( 2,10, na_thy, 1) =   -0.891; //   C2*
	icoor( 3,10, na_thy, 1) =   -0.438; //   C2*
	icoor( 1,11, na_thy, 1) =    0.000; //   C1*
	icoor( 2,11, na_thy, 1) =    0.000; //   C1*
	icoor( 3,11, na_thy, 1) =    0.000; //   C1*
	icoor( 1,12, na_thy, 1) =    0.214; //   N1
	icoor( 2,12, na_thy, 1) =    0.668; //   N1
	icoor( 3,12, na_thy, 1) =    1.296; //   N1
	icoor( 1,13, na_thy, 1) =    0.374; //   C2
	icoor( 2,13, na_thy, 1) =    2.035; //   C2
	icoor( 3,13, na_thy, 1) =    1.303; //   C2
	icoor( 1,14, na_thy, 1) =    0.416; //   O2
	icoor( 2,14, na_thy, 1) =    2.705; //   O2
	icoor( 3,14, na_thy, 1) =    0.284; //   O2
	icoor( 1,15, na_thy, 1) =    0.483; //   N3
	icoor( 2,15, na_thy, 1) =    2.592; //   N3
	icoor( 3,15, na_thy, 1) =    2.553; //   N3
	icoor( 1,16, na_thy, 1) =    0.449; //   C4
	icoor( 2,16, na_thy, 1) =    1.933; //   C4
	icoor( 3,16, na_thy, 1) =    3.767; //   C4
	icoor( 1,17, na_thy, 1) =    0.560; //   O4
	icoor( 2,17, na_thy, 1) =    2.568; //   O4
	icoor( 3,17, na_thy, 1) =    4.812; //   O4
	icoor( 1,18, na_thy, 1) =    0.279; //   C5
	icoor( 2,18, na_thy, 1) =    0.500; //   C5
	icoor( 3,18, na_thy, 1) =    3.685; //   C5
	icoor( 1,19, na_thy, 1) =    0.231; //   C5M
	icoor( 2,19, na_thy, 1) =   -0.299; //   C5M
	icoor( 3,19, na_thy, 1) =    4.949; //   C5M
	icoor( 1,20, na_thy, 1) =    0.171; //   C6
	icoor( 2,20, na_thy, 1) =   -0.052; //   C6
	icoor( 3,20, na_thy, 1) =    2.470; //   C6
	icoor( 1,21, na_thy, 1) =   -0.962; //  1H5*
	icoor( 2,21, na_thy, 1) =   -4.199; //  1H5*
	icoor( 3,21, na_thy, 1) =   -0.217; //  1H5*
	icoor( 1,22, na_thy, 1) =   -1.987; //  2H5*
	icoor( 2,22, na_thy, 1) =   -3.241; //  2H5*
	icoor( 3,22, na_thy, 1) =    0.859; //  2H5*
	icoor( 1,23, na_thy, 1) =   -1.673; //   H4*
	icoor( 2,23, na_thy, 1) =   -2.277; //   H4*
	icoor( 3,23, na_thy, 1) =   -1.372; //   H4*
	icoor( 1,24, na_thy, 1) =    1.002; //   H3*
	icoor( 2,24, na_thy, 1) =   -2.877; //   H3*
	icoor( 3,24, na_thy, 1) =   -1.364; //   H3*
	icoor( 1,25, na_thy, 1) =    1.904; //  1H2*
	icoor( 2,25, na_thy, 1) =   -0.335; //  1H2*
	icoor( 3,25, na_thy, 1) =   -1.013; //  1H2*
	icoor( 1,26, na_thy, 1) =    1.668; //  2H2*
	icoor( 2,26, na_thy, 1) =   -1.344; //  2H2*
	icoor( 3,26, na_thy, 1) =    0.420; //  2H2*
	icoor( 1,27, na_thy, 1) =   -0.119; //   H1*
	icoor( 2,27, na_thy, 1) =    0.800; //   H1*
	icoor( 3,27, na_thy, 1) =   -0.745; //   H1*
	icoor( 1,28, na_thy, 1) =    0.600; //   H3
	icoor( 2,28, na_thy, 1) =    3.584; //   H3
	icoor( 3,28, na_thy, 1) =    2.589; //   H3
	icoor( 1,29, na_thy, 1) =    0.098; //  1H5M
	icoor( 2,29, na_thy, 1) =   -1.363; //  1H5M
	icoor( 3,29, na_thy, 1) =    4.706; //  1H5M
	icoor( 1,30, na_thy, 1) =    1.171; //  2H5M
	icoor( 2,30, na_thy, 1) =   -0.165; //  2H5M
	icoor( 3,30, na_thy, 1) =    5.504; //  2H5M
	icoor( 1,31, na_thy, 1) =   -0.612; //  3H5M
	icoor( 2,31, na_thy, 1) =    0.043; //  3H5M
	icoor( 3,31, na_thy, 1) =    5.568; //  3H5M
	icoor( 1,32, na_thy, 1) =    0.041; //   H6
	icoor( 2,32, na_thy, 1) =   -1.142; //   H6
	icoor( 3,32, na_thy, 1) =    2.405; //   H6

	if ( design::hydrate_dna ) {
		create_dna_h2o_variants( na_thy );
	}

}

// start rna definitions - JJH 5/23/03
////////////////////////////////////////////////////////////////////////////////
/// @begin init_rna_gua
///
/// @brief arrays filled to define the proerties of the nucleic acid dGuanine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// na_rgu - index number for rna GUA
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_rna_gua()
{
	using namespace aaproperties_pack;
	using namespace param_aa;

//bk   Parameters for rGUA -----------------------------
	nvar(na_rgu) = 1; // starting starting number of variants for rGUA

//bk   variant1 ---------- rGUA

	natoms(na_rgu,1) = 34; // number of total atoms
	nheavyatoms(na_rgu,1) =  23; // number of heavy atoms
	nchi(na_rgu,1) =  1; // number of chi angles

//bk   atom names rGUA
	atom_name( 1, na_rgu,1) = " P  ";
	atom_name( 2, na_rgu,1) = " O1P";
	atom_name( 3, na_rgu,1) = " O2P";
	atom_name( 4, na_rgu,1) = " O5*";
	atom_name( 5, na_rgu,1) = " C5*";
	atom_name( 6, na_rgu,1) = " C4*";
	atom_name( 7, na_rgu,1) = " O4*";
	atom_name( 8, na_rgu,1) = " C3*";
	atom_name( 9, na_rgu,1) = " O3*";
	atom_name(10, na_rgu,1) = " C2*";
	atom_name(11, na_rgu,1) = " O2*";
	atom_name(12, na_rgu,1) = " C1*";
	atom_name(13, na_rgu,1) = " N1 ";
	atom_name(14, na_rgu,1) = " C2 ";
	atom_name(15, na_rgu,1) = " N2 ";
	atom_name(16, na_rgu,1) = " N3 ";
	atom_name(17, na_rgu,1) = " C4 ";
	atom_name(18, na_rgu,1) = " C5 ";
	atom_name(19, na_rgu,1) = " C6 ";
	atom_name(20, na_rgu,1) = " O6 ";
	atom_name(21, na_rgu,1) = " N7 ";
	atom_name(22, na_rgu,1) = " C8 ";
	atom_name(23, na_rgu,1) = " N9 ";
	atom_name(24, na_rgu,1) = "1H5*";
	atom_name(25, na_rgu,1) = "2H5*";
	atom_name(26, na_rgu,1) = " H4*";
	atom_name(27, na_rgu,1) = " H3*";
	atom_name(28, na_rgu,1) = "1H2*";
	atom_name(29, na_rgu,1) = " H1*"; // JSS WARNING: funky ordering for rgu only
	atom_name(30, na_rgu,1) = "2HO*";
	atom_name(31, na_rgu,1) = " H1 ";
	atom_name(32, na_rgu,1) = "1H2 ";
	atom_name(33, na_rgu,1) = "2H2 ";
	atom_name(34, na_rgu,1) = " H8 ";

//jjh intra residue bonding
	nbonded_neighbors( 1, na_rgu, 1) = 3; // P
		bonded_neighbor(1, 1, na_rgu, 1) = 2; // P--O1P
		bonded_neighbor(2, 1, na_rgu, 1) = 3; // P--O2P
		bonded_neighbor(3, 1, na_rgu, 1) = 4; // P--O5*
	nbonded_neighbors( 2, na_rgu, 1) = 1; // O1P
		bonded_neighbor(1, 2, na_rgu, 1) = 1; // O1P--P
	nbonded_neighbors( 3, na_rgu, 1) = 1; // O2P
		bonded_neighbor(1, 3, na_rgu, 1) = 1; // O2P--P
	nbonded_neighbors( 4, na_rgu, 1) = 2; // O5*
		bonded_neighbor(1, 4, na_rgu, 1) = 1; // O5*--P
		bonded_neighbor(2, 4, na_rgu, 1) = 5; // O5*--C5*
	nbonded_neighbors( 5, na_rgu, 1) = 4; // C5*
		bonded_neighbor(1, 5, na_rgu, 1) = 4; // C5*--O5*
		bonded_neighbor(2, 5, na_rgu, 1) = 6; // C5*--C4*
		bonded_neighbor(3, 5, na_rgu, 1) =24; // C5*--1H5*
		bonded_neighbor(4, 5, na_rgu, 1) =25; // C5*--2H5*
	nbonded_neighbors( 6, na_rgu, 1) = 4; // C4*
		bonded_neighbor(1, 6, na_rgu, 1) = 5; // C4*--C5*
		bonded_neighbor(2, 6, na_rgu, 1) = 7; // C4*--O4*
		bonded_neighbor(3, 6, na_rgu, 1) = 8; // C4*--C3*
		bonded_neighbor(4, 6, na_rgu, 1) =26; // C4*--H4*
	nbonded_neighbors( 7, na_rgu, 1) = 2; // O4*
		bonded_neighbor(1, 7, na_rgu, 1) = 6; // O4*--C4*
		bonded_neighbor(2, 7, na_rgu, 1) =12; // O4*--C1*
	nbonded_neighbors( 8, na_rgu, 1) = 4; // C3*
		bonded_neighbor(1, 8, na_rgu, 1) = 6; // C3*--C4*
		bonded_neighbor(2, 8, na_rgu, 1) = 9; // C3*--O3*
		bonded_neighbor(3, 8, na_rgu, 1) =10; // C3*--C2*
		bonded_neighbor(4, 8, na_rgu, 1) =27; // C3*--H3*
	nbonded_neighbors( 9, na_rgu, 1) = 1; // O3*
		bonded_neighbor(1, 9, na_rgu, 1) = 8; // O3*--C3*
	nbonded_neighbors( 10, na_rgu, 1) = 4; // C2*
		bonded_neighbor(1,10, na_rgu, 1) = 8; // C2*--C3*
		bonded_neighbor(2,10, na_rgu, 1) =11; // C2*--O2*
		bonded_neighbor(3,10, na_rgu, 1) =12; // C2*--C1*
		bonded_neighbor(4,10, na_rgu, 1) =28; // C2*--1H2*
	nbonded_neighbors( 11, na_rgu, 1) = 2; // O2*
		bonded_neighbor(1,11, na_rgu, 1) =10; // O2*--C2*
		bonded_neighbor(2,11, na_rgu, 1) =30; // O2*--2HO*
	nbonded_neighbors( 12, na_rgu, 1) = 4; // C1*
		bonded_neighbor(1,12, na_rgu, 1) = 7; // C1*--O4*
		bonded_neighbor(2,12, na_rgu, 1) =10; // C1*--C2*
		bonded_neighbor(3,12, na_rgu, 1) =23; // C1*--N9
		bonded_neighbor(4,12, na_rgu, 1) =29; // C1*--H1*
	nbonded_neighbors( 13, na_rgu, 1) = 3; // N1
		bonded_neighbor(1,13, na_rgu, 1) =14; // N1--C2
		bonded_neighbor(2,13, na_rgu, 1) =19; // N1--C6
		bonded_neighbor(3,13, na_rgu, 1) =31; // N1--H1
	nbonded_neighbors( 14, na_rgu, 1) = 3; // C2
		bonded_neighbor(1,14, na_rgu, 1) =13; // C2--N1
		bonded_neighbor(2,14, na_rgu, 1) =15; // C2--N2
		bonded_neighbor(3,14, na_rgu, 1) =16; // C2--N3
	nbonded_neighbors( 15, na_rgu, 1) = 3; // N2
		bonded_neighbor(1,15, na_rgu, 1) =14; // N2--C2
		bonded_neighbor(2,15, na_rgu, 1) =32; // N2--1H2
		bonded_neighbor(3,15, na_rgu, 1) =33; // N2--2H2
	nbonded_neighbors( 16, na_rgu, 1) = 2; // N3
		bonded_neighbor(1,16, na_rgu, 1) =14; // N3--C2
		bonded_neighbor(2,16, na_rgu, 1) =17; // N3--C4
	nbonded_neighbors( 17, na_rgu, 1) = 3; // C4
		bonded_neighbor(1,17, na_rgu, 1) =16; // C4--N3
		bonded_neighbor(2,17, na_rgu, 1) =18; // C4--C5
		bonded_neighbor(3,17, na_rgu, 1) =23; // C4--N9
	nbonded_neighbors( 18, na_rgu, 1) = 3; // C5
		bonded_neighbor(1,18, na_rgu, 1) =17; // C5--C4
		bonded_neighbor(2,18, na_rgu, 1) =19; // C5--C6
		bonded_neighbor(3,18, na_rgu, 1) =21; // C5--N7
	nbonded_neighbors( 19, na_rgu, 1) = 3; // C6
		bonded_neighbor(1,19, na_rgu, 1) =13; // C6--N1
		bonded_neighbor(2,19, na_rgu, 1) =18; // C6--C5
		bonded_neighbor(3,19, na_rgu, 1) =20; // C6--O6
	nbonded_neighbors( 20, na_rgu, 1) = 1; // O6
		bonded_neighbor(1,20, na_rgu, 1) =19; // O6--C6
	nbonded_neighbors( 21, na_rgu, 1) = 2; // N7
		bonded_neighbor(1,21, na_rgu, 1) =18; // N7--C5
		bonded_neighbor(2,21, na_rgu, 1) =22; // N7--C8
	nbonded_neighbors( 22, na_rgu, 1) = 3; // C8
		bonded_neighbor(1,22, na_rgu, 1) =21; // C8--N7
		bonded_neighbor(2,22, na_rgu, 1) =23; // C8--N9
		bonded_neighbor(3,22, na_rgu, 1) =34; // C8--H8
	nbonded_neighbors( 23, na_rgu, 1) = 3; // N9
		bonded_neighbor(1,23, na_rgu, 1) =12; // N9--C1*
		bonded_neighbor(2,23, na_rgu, 1) =17; // N9--C4
		bonded_neighbor(3,23, na_rgu, 1) =22; // N9--C8
	nbonded_neighbors( 24, na_rgu, 1) = 1; //1H5*
		bonded_neighbor(1,24, na_rgu, 1) = 5; //1H5*--C5*
	nbonded_neighbors( 25, na_rgu, 1) = 1; //2H5*
		bonded_neighbor(1,25, na_rgu, 1) = 5; //2H5*--C5*
	nbonded_neighbors( 26, na_rgu, 1) = 1; // H4*
		bonded_neighbor(1,26, na_rgu, 1) = 6; // H4*--C4*
	nbonded_neighbors( 27, na_rgu, 1) = 1; // H3*
		bonded_neighbor(1,27, na_rgu, 1) = 8; // H3*--C3*
	nbonded_neighbors( 28, na_rgu, 1) = 1; //1H2*
		bonded_neighbor(1,28, na_rgu, 1) =10; //1H2*--C2*
	nbonded_neighbors( 29, na_rgu, 1) = 1; // H1*
		bonded_neighbor(1,29, na_rgu, 1) =12; // H1*--C1*
	nbonded_neighbors( 30, na_rgu, 1) = 1; //2HO*
		bonded_neighbor(1,30, na_rgu, 1) =11; //2HO*--O2*
	nbonded_neighbors( 31, na_rgu, 1) = 1; // H1
		bonded_neighbor(1,31, na_rgu, 1) =13; // H1--N1
	nbonded_neighbors( 32, na_rgu, 1) = 1; //1H2
		bonded_neighbor(1,32, na_rgu, 1) =15; //!H2--N2
	nbonded_neighbors( 33, na_rgu, 1) = 1; //2H2
		bonded_neighbor(1,33, na_rgu, 1) =15; //2H2--N2
	nbonded_neighbors( 34, na_rgu, 1) = 1; // H8
		bonded_neighbor(1,34, na_rgu, 1) =22; // H8--C8

//jjh get the angles automatically
	get_angles_from_bonds(na_rgu, 1);

	atom_base( 1,na_rgu,1) =  2; // P    --> O1P
	atom_base( 2,na_rgu,1) =  1; // O1P  --> P
	atom_base( 3,na_rgu,1) =  1; // O2P  --> P
	atom_base( 4,na_rgu,1) =  1; // O5*  --> P
	abase2( 4, na_rgu, 1) =  5; //  C5* base for  O5*
	atom_base( 5,na_rgu,1) =  4; // C5*  --> O5*
	atom_base( 6,na_rgu,1) =  5; // C4*  --> C5*
	atom_base( 7,na_rgu,1) = 12; // O4*  --> C1* //abase=C4*
	abase2( 7, na_rgu, 1) =  6; //  C4* base for  O4*
	atom_base( 8,na_rgu,1) =  6; // C3*  --> C4*
	atom_base( 9,na_rgu,1) =  8; // O3*  --> C3*
	atom_base(10,na_rgu,1) =  8; // C2*  --> C3*
	atom_base(11,na_rgu,1) = 10; // O2*  --> C2*
	abase2(11, na_rgu, 1) = 30; // 2HO* base for  O2*
	atom_base(12,na_rgu,1) = 10; // C1*  --> C2*
	atom_base(13,na_rgu,1) = 14; // N1   --> C2
	atom_base(14,na_rgu,1) = 16; // C2   --> N3
	atom_base(15,na_rgu,1) = 14; // N2   --> C2
	atom_base(16,na_rgu,1) = 14; // N3   --> C2
	abase2(16, na_rgu, 1) = 17; //  C4  base for  N3
	atom_base(17,na_rgu,1) = 23; // C4   --> N9
	atom_base(18,na_rgu,1) = 21; // C5   --> N7
	atom_base(19,na_rgu,1) = 18; // C6   --> C5
	atom_base(20,na_rgu,1) = 19; // O6   --> C6
	atom_base(21,na_rgu,1) = 18; // N7   --> C5
	abase2(21, na_rgu, 1) = 22; //  C8  base for  N7
	atom_base(22,na_rgu,1) = 23; // C8   --> N9
	atom_base(23,na_rgu,1) = 12; // N9   --> C1*
	atom_base(24,na_rgu,1) =  5; // 1H5* --> C5*
	atom_base(25,na_rgu,1) =  5; // 2H5* --> C5*
	atom_base(26,na_rgu,1) =  6; // H4*  --> C4*
	atom_base(27,na_rgu,1) =  8; // H3*  --> C3*
	atom_base(28,na_rgu,1) = 10; // 1H2* --> C2*
	atom_base(29,na_rgu,1) = 12; // H1*  --> C1*
	atom_base(30,na_rgu,1) = 11; // 2HO* --> O2*
	atom_base(31,na_rgu,1) = 13; // H1   --> N1
	atom_base(32,na_rgu,1) = 15; // 1H2  --> N2
	atom_base(33,na_rgu,1) = 15; // 2H2  --> N2
	atom_base(34,na_rgu,1) = 22; // H8   --> C8

//vats New atom types for Nucleic Acid based on CHARMM 27
//vats Ref etable.h for more details
//vats   atom type numbers rGUA
//cmd  Reverted to standard rosetta atomtypes
	fullatom_type( 1,na_rgu,1) = 21; // Phos   P    P
	fullatom_type( 2,na_rgu,1) = 15; // 69; // OOC    O1P  ON3
	fullatom_type( 3,na_rgu,1) = 15; // 69; // OOC    O2P  ON3
	fullatom_type( 4,na_rgu,1) = 14; // 68; // ONH2   O5*  ON2
	fullatom_type( 5,na_rgu,1) =  4; // 61; // CH2    C5*  CN8 (CN8B)
	fullatom_type( 6,na_rgu,1) =  3; // 60; // CH1    C4*  CN7
	fullatom_type( 7,na_rgu,1) = 13; // 72; // OH     O4*  ON6 (ON6B)
	fullatom_type( 8,na_rgu,1) =  3; // 60; // CH1    C3*  CN7
	fullatom_type( 9,na_rgu,1) = 14; // 68; // ONH2   O3*  ON2
	fullatom_type(10,na_rgu,1) =  4; // 60; // CH2    C2*  CN7 (CN7B)
	fullatom_type(11,na_rgu,1) = 13; // 71; // OH     O2*  ON5
	fullatom_type(12,na_rgu,1) =  3; // 60; // CH1    C1*  CN7 (CN7B)
	fullatom_type(13,na_rgu,1) =  7; // 64; // Ntrp   N1   NN2 (NN2G)
	fullatom_type(14,na_rgu,1) =  6; // 56; // aroC   C2   CN2
	fullatom_type(15,na_rgu,1) =  9; // 63; // NH2O   N2   NN1
	fullatom_type(16,na_rgu,1) =  8; // 65; // Nhis   N3   NN3 (NN3G)
	fullatom_type(17,na_rgu,1) =  6; // 59; // aroC   C4   CN5
	fullatom_type(18,na_rgu,1) =  6; // 59; // aroC   C5   CN5 (CN5G)
	fullatom_type(19,na_rgu,1) = 19; // 55; // CObb   C6   CN1
	fullatom_type(20,na_rgu,1) = 20; // 67; // OCbb   O6   ON1
	fullatom_type(21,na_rgu,1) =  8; // 66; // Nhis   N7   NN4
	fullatom_type(22,na_rgu,1) =  6; // 58; // aroC   C8   CN4
	fullatom_type(23,na_rgu,1) =  8; // 64; // NHis   N9   NN2 (NN2B)
	fullatom_type(24,na_rgu,1) = 23; // 74; // Hapo  1H5*  HNP
	fullatom_type(25,na_rgu,1) = 23; // 74; // Hapo  2H5*  HNP
	fullatom_type(26,na_rgu,1) = 23; // 74; // Hapo   H4*  HNP
	fullatom_type(27,na_rgu,1) = 23; // 74; // Hapo   H3*  HNP
	fullatom_type(28,na_rgu,1) = 23; // 74; // Hapo  1H2*  HNP
	fullatom_type(29,na_rgu,1) = 23; // 74; // Hapo   H1*  HNP
	fullatom_type(30,na_rgu,1) = 22; // 73; // Hpol  2HO*  HP
	fullatom_type(31,na_rgu,1) = 22; // 73; // Hpol   H1   HP
	fullatom_type(32,na_rgu,1) = 22; // 73; // Hpol  1H2   HP
	fullatom_type(33,na_rgu,1) = 22; // 73; // Hpol  2H2   HP
	fullatom_type(34,na_rgu,1) = 24; // 74; // Haro   H8   HNP

//bk   atom partial charges rGUA
	atomic_charge( 1, na_rgu,1) = 1.500; // P
	atomic_charge( 2, na_rgu,1) = -.780; // O1P
	atomic_charge( 3, na_rgu,1) = -.780; // O2P
	atomic_charge( 4, na_rgu,1) = -.570; // O5*
	atomic_charge( 5, na_rgu,1) = -.080; // C5*
	atomic_charge( 6, na_rgu,1) =  .160; // C4*
	atomic_charge( 7, na_rgu,1) = -.500; // O4*
	atomic_charge( 8, na_rgu,1) =  .010; // C3*
	atomic_charge( 9, na_rgu,1) = -.570; // O3*
	atomic_charge(10, na_rgu,1) =  .140; // C2*
	atomic_charge(11, na_rgu,1) = -.660; // O2*
	atomic_charge(12, na_rgu,1) =  .160; // C1*
	atomic_charge(13, na_rgu,1) = -.340; // N1
	atomic_charge(14, na_rgu,1) =  .750; // C2
	atomic_charge(15, na_rgu,1) = -.680; // N2
	atomic_charge(16, na_rgu,1) = -.740; // N3
	atomic_charge(17, na_rgu,1) =  .260; // C4
	atomic_charge(18, na_rgu,1) =  .000; // C5
	atomic_charge(19, na_rgu,1) =  .540; // C6
	atomic_charge(20, na_rgu,1) = -.510; // O6
	atomic_charge(21, na_rgu,1) = -.600; // N7
	atomic_charge(22, na_rgu,1) =  .250; // C8
	atomic_charge(23, na_rgu,1) = -.020; // N9
	atomic_charge(24, na_rgu,1) =  .090; // 1H5*
	atomic_charge(25, na_rgu,1) =  .090; // 2H5*
	atomic_charge(26, na_rgu,1) =  .090; // H4*
	atomic_charge(27, na_rgu,1) =  .090; // H3*
	atomic_charge(28, na_rgu,1) =  .090; // 1H2*
	atomic_charge(29, na_rgu,1) =  .090; // H1*
	atomic_charge(30, na_rgu,1) =  .430; // 2HO*
	atomic_charge(31, na_rgu,1) =  .260; // H1
	atomic_charge(32, na_rgu,1) =  .320; // 1H2
	atomic_charge(33, na_rgu,1) =  .350; // 2H2
	atomic_charge(34, na_rgu,1) =  .160; // H8

	assert_total_charge( na_rgu, 1, -1.0);

//cmd  first sidechain atom
	first_scatom( na_rgu,1) = 13; //N1

//jjh anchor atoms for base mutations
	na_anchor(1, na_rgu, 1) = 12;
	na_anchor(2, na_rgu, 1) = 23;
	na_anchor(3, na_rgu, 1) = 17;

//bk   template atoms used for placing hydrogens

//bk   template for building  P
	ta(1, 1, na_rgu,1) =  4; //   O5*
	ta(2, 1, na_rgu,1) =  5; //   C5*
	ta(3, 1, na_rgu,1) =  6; //   C4*

//bk   template for building  O1P
	ta(1, 2, na_rgu,1) =  1; //   P
	ta(2, 2, na_rgu,1) =  5; //   C5*
	ta(3, 2, na_rgu,1) =  4; //   O5*

//bk   template for building  O2P
	ta(1, 3, na_rgu,1) =  1; //   P
	ta(2, 3, na_rgu,1) =  2; //   O1P
	ta(3, 3, na_rgu,1) =  4; //   O5*

//bk   template for building  O2*
	ta(1, 11, na_rgu,1) =  10; //   C2*
	ta(2, 11, na_rgu,1) =   8; //   C3*
	ta(3, 11, na_rgu,1) =  12; //   C1*

//bk   template for building 1H5*
	ta(1,24, na_rgu,1) =  5; //   C5*
	ta(2,24, na_rgu,1) =  4; //   O5*
	ta(3,24, na_rgu,1) =  6; //   C4*

//bk   template for building 2H5*
	ta(1,25, na_rgu,1) =  5; //   C5*
	ta(2,25, na_rgu,1) =  4; //   O5*
	ta(3,25, na_rgu,1) =  6; //   C4*

//bk   template for building  H4*
	ta(1,26, na_rgu,1) =  6; //   C4*
	ta(2,26, na_rgu,1) =  7; //   O4*
	ta(3,26, na_rgu,1) =  5; //   C5*

//bk   template for building  H3*
	ta(1,27, na_rgu,1) =  8; //   C3*
	ta(2,27, na_rgu,1) = 10; //   C2*
	ta(3,27, na_rgu,1) =  6; //   C4*

//bk   template for building 1H2*
	ta(1,28, na_rgu,1) = 10; //   C2*
	ta(2,28, na_rgu,1) =  8; //   C3*
	ta(3,28, na_rgu,1) = 12; //   C1*

//bk   template for building  H1*
	ta(1,29, na_rgu,1) = 12; //   C1*
	ta(2,29, na_rgu,1) = 10; //   C2*
	ta(3,29, na_rgu,1) =  7; //   O4*

//bk   template for building 2HO*
	ta(1,30, na_rgu,1) = 11; //   O2*
	ta(2,30, na_rgu,1) = 10; //   C2*
	ta(3,30, na_rgu,1) = 12; //   C1*

//bk   template for building  H1
	ta(1,31, na_rgu,1) = 13; //   N1
	ta(2,31, na_rgu,1) = 14; //   C2
	ta(3,31, na_rgu,1) = 19; //   C6

//bk   template for building 1H2
	ta(1,32, na_rgu,1) = 15; //   N2
	ta(2,32, na_rgu,1) = 14; //   C2
	ta(3,32, na_rgu,1) = 16; //   N3

//bk   template for building 2H2
	ta(1,33, na_rgu,1) = 15; //   N2
	ta(2,33, na_rgu,1) = 14; //   C2
	ta(3,33, na_rgu,1) = 16; //   N3

//bk   template for building  H8
	ta(1,34, na_rgu,1) = 22; //   C8
	ta(2,34, na_rgu,1) = 21; //   N7
	ta(3,34, na_rgu,1) = 23; //   N9

//bk   identify atoms that require chi angles to be built
//cy   chi angles needed for building 2HO*
	chi_required(1, 30,na_rgu,1) = true;

//cy   four atoms that define chi angle 1
	chi_atoms( 1, 1,na_rgu, 1) = 12; //   C1*
	chi_atoms( 2, 1,na_rgu, 1) = 10; //   C2*
	chi_atoms( 3, 1,na_rgu, 1) = 11; //   O2*
	chi_atoms( 4, 1,na_rgu, 1) = 30; //   2HO*

//bk   hbonding information
//jjh set the next two to nonsense values
	HNpos( na_rgu,1) = -1; // position of backbone HN
	HApos( na_rgu,1) = -1; // position of backbone HA

	nH_polar( na_rgu,1) =  4; // number of polar hydrogens
	nH_aromatic( na_rgu,1) =  1; // number of aromatic hydrogens
	nH_apolar( na_rgu,1) =  6; // number of apolar hydrogens
	nacceptors( na_rgu,1) =  9; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1, na_rgu,1) = 30; // 2HO*
	Hpos_polar( 2, na_rgu,1) = 31; //  H1
	Hpos_polar( 3, na_rgu,1) = 32; // 1H2
	Hpos_polar( 4, na_rgu,1) = 33; // 2H2

//bk   atom numbers for aromatic hydrogens
	Hpos_aromatic( 1, na_rgu,1) = 34; //  H8

//bk   atom numbers for apolar hydrogens
	Hpos_apolar( 1, na_rgu,1) = 24; //1H5*
	Hpos_apolar( 2, na_rgu,1) = 25; //2H5*
	Hpos_apolar( 3, na_rgu,1) = 26; // H4*
	Hpos_apolar( 4, na_rgu,1) = 27; // H3*
	Hpos_apolar( 5, na_rgu,1) = 28; //1H2*
	Hpos_apolar( 6, na_rgu,1) = 29; // H1*

//bk   acceptor positions
	accpt_pos( 1, na_rgu, 1) =  2; //  O1P
	accpt_pos( 2, na_rgu, 1) =  3; //  O2P
	accpt_pos( 3, na_rgu, 1) =  4; //  O5*
	accpt_pos( 4, na_rgu, 1) =  9; //  O3*
	accpt_pos( 5, na_rgu, 1) = 11; //  O2*
	accpt_pos( 6, na_rgu, 1) = 16; //  N3
	accpt_pos( 7, na_rgu, 1) = 20; //  O6
	accpt_pos( 8, na_rgu, 1) = 21; //  N7
	accpt_pos( 9, na_rgu, 1) =  7; //  O4*

//bk   hydrogens on each atom

	nhydrogens_on_atm( 5, na_rgu, 1) =  2; //  C5*
	hydrogens_on_atm( 1, 5, na_rgu, 1) = 24; // 1H5*
	hydrogens_on_atm( 2, 5, na_rgu, 1) = 25; // 2H5*

	nhydrogens_on_atm( 6, na_rgu, 1) =  1; //  C4*
	hydrogens_on_atm( 1, 6, na_rgu, 1) =  26; //  H4*

	nhydrogens_on_atm( 8, na_rgu, 1) =  1; //  C3*
	hydrogens_on_atm( 1, 8, na_rgu, 1) =  27; //  H3*

	nhydrogens_on_atm(10, na_rgu, 1) =  1; //  C2*
	hydrogens_on_atm( 1,10, na_rgu, 1) =  28; // 1H2*

	nhydrogens_on_atm(11, na_rgu, 1) =  1; //  O2*
	hydrogens_on_atm( 1,11, na_rgu, 1) =  30; // 2HO*

	nhydrogens_on_atm(12, na_rgu, 1) =  1; //  C1*
	hydrogens_on_atm( 1,12, na_rgu, 1) =  29; //  H1*

	nhydrogens_on_atm(13, na_rgu, 1) =  1; //  N1
	hydrogens_on_atm( 1,13, na_rgu, 1) =  31; //  H1

	nhydrogens_on_atm(15, na_rgu, 1) =  2; //  N2
	hydrogens_on_atm( 1,15, na_rgu, 1) =  32; // 1H2
	hydrogens_on_atm( 2,15, na_rgu, 1) =  33; // 2H2

	nhydrogens_on_atm(22, na_rgu, 1) =  1; //  C8
	hydrogens_on_atm( 1,22, na_rgu, 1) = 34; //  H8

//bk   template coordinates for the nucleic acid
//rhiju Fixed icoor, which had base poking out of
//rhiju sugar wrong. The takeoff angles from N1 read from 2INA.pdb,
//rhiju which uses ideal values used by  xplor and the NDB.
//rhiju Nov. 10, 2006.
  icoor( 1, 1, na_rgu, 1) =    4.691; //  P
  icoor( 2, 1, na_rgu, 1) =    0.327; //  P
  icoor( 3, 1, na_rgu, 1) =   -2.444; //  P
  icoor( 1, 2, na_rgu, 1) =    5.034; //  O1P
  icoor( 2, 2, na_rgu, 1) =    1.678; //  O1P
  icoor( 3, 2, na_rgu, 1) =   -1.932; //  O1P
  icoor( 1, 3, na_rgu, 1) =    4.718; //  O2P
  icoor( 2, 3, na_rgu, 1) =    0.068; //  O2P
  icoor( 3, 3, na_rgu, 1) =   -3.906; //  O2P
  icoor( 1, 4, na_rgu, 1) =    3.246; //  O5*
  icoor( 2, 4, na_rgu, 1) =   -0.057; //  O5*
  icoor( 3, 4, na_rgu, 1) =   -1.895; //  O5*
  icoor( 1, 5, na_rgu, 1) =    2.957; //  C5*
  icoor( 2, 5, na_rgu, 1) =   -1.393; //  C5*
  icoor( 3, 5, na_rgu, 1) =   -1.443; //  C5*
  icoor( 1, 6, na_rgu, 1) =    1.509; //  C4*
  icoor( 2, 6, na_rgu, 1) =   -1.478; //  C4*
  icoor( 3, 6, na_rgu, 1) =   -1.022; //  C4*
  icoor( 1, 7, na_rgu, 1) =    1.286; //  O4*
  icoor( 2, 7, na_rgu, 1) =   -0.587; //  O4*
  icoor( 3, 7, na_rgu, 1) =    0.103; //  O4*
  icoor( 1, 8, na_rgu, 1) =    0.499; //  C3*
  icoor( 2, 8, na_rgu, 1) =   -1.031; //  C3*
  icoor( 3, 8, na_rgu, 1) =   -2.067; //  C3*
  icoor( 1, 9, na_rgu, 1) =    0.178; //  O3*
  icoor( 2, 9, na_rgu, 1) =   -2.084; //  O3*
  icoor( 3, 9, na_rgu, 1) =   -2.981; //  O3*
  icoor( 1,10, na_rgu, 1) =   -0.694; //  C2*
  icoor( 2,10, na_rgu, 1) =   -0.627; //  C2*
  icoor( 3,10, na_rgu, 1) =   -1.210; //  C2*
  icoor( 1,11, na_rgu, 1) =   -1.474; //  O2*
  icoor( 2,11, na_rgu, 1) =   -1.731; //  O2*
  icoor( 3,11, na_rgu, 1) =   -0.795; //  O2*
  icoor( 1,12, na_rgu, 1) =    0.000; //  C1*
  icoor( 2,12, na_rgu, 1) =    0.000; //  C1*
  icoor( 3,12, na_rgu, 1) =    0.000; //  C1*
  icoor( 1,13, na_rgu, 1) =   -2.130; //  N1
  icoor( 2,13, na_rgu, 1) =    4.451; //  N1
  icoor( 3,13, na_rgu, 1) =    1.160; //  N1
  icoor( 1,14, na_rgu, 1) =   -2.615; //  C2
  icoor( 2,14, na_rgu, 1) =    3.177; //  C2
  icoor( 3,14, na_rgu, 1) =    1.331; //  C2
  icoor( 1,15, na_rgu, 1) =   -3.825; //  N2
  icoor( 2,15, na_rgu, 1) =    3.078; //  N2
  icoor( 3,15, na_rgu, 1) =    1.901; //  N2
  icoor( 1,16, na_rgu, 1) =   -1.963; //  N3
  icoor( 2,16, na_rgu, 1) =    2.084; //  N3
  icoor( 3,16, na_rgu, 1) =    0.972; //  N3
  icoor( 1,17, na_rgu, 1) =   -0.765; //  C4
  icoor( 2,17, na_rgu, 1) =    2.368; //  C4
  icoor( 3,17, na_rgu, 1) =    0.419; //  C4
  icoor( 1,18, na_rgu, 1) =   -0.190; //  C5
  icoor( 2,18, na_rgu, 1) =    3.602; //  C5
  icoor( 3,18, na_rgu, 1) =    0.202; //  C5
  icoor( 1,19, na_rgu, 1) =   -0.901; //  C6
  icoor( 2,19, na_rgu, 1) =    4.766; //  C6
  icoor( 3,19, na_rgu, 1) =    0.590; //  C6
  icoor( 1,20, na_rgu, 1) =   -0.558; //  O6
  icoor( 2,20, na_rgu, 1) =    5.951; //  O6
  icoor( 3,20, na_rgu, 1) =    0.482; //  O6
  icoor( 1,21, na_rgu, 1) =    1.056; //  N7
  icoor( 2,21, na_rgu, 1) =    3.466; //  N7
  icoor( 3,21, na_rgu, 1) =   -0.395; //  N7
  icoor( 1,22, na_rgu, 1) =    1.212; //  C8
  icoor( 2,22, na_rgu, 1) =    2.176; //  C8
  icoor( 3,22, na_rgu, 1) =   -0.528; //  C8
  icoor( 1,23, na_rgu, 1) =    0.144; //  N9
  icoor( 2,23, na_rgu, 1) =    1.453; //  N9
  icoor( 3,23, na_rgu, 1) =   -0.054; //  N9
  icoor( 1,24, na_rgu, 1) =    3.612; // 1H5*
  icoor( 2,24, na_rgu, 1) =   -1.657; // 1H5*
  icoor( 3,24, na_rgu, 1) =   -0.600; // 1H5*
  icoor( 1,25, na_rgu, 1) =    3.160; // 2H5*
  icoor( 2,25, na_rgu, 1) =   -2.116; // 2H5*
  icoor( 3,25, na_rgu, 1) =   -2.247; // 2H5*
  icoor( 1,26, na_rgu, 1) =    1.353; //  H4*
  icoor( 2,26, na_rgu, 1) =   -2.546; //  H4*
  icoor( 3,26, na_rgu, 1) =   -0.810; //  H4*
  icoor( 1,27, na_rgu, 1) =    0.863; //  H3*
  icoor( 2,27, na_rgu, 1) =   -0.220; //  H3*
  icoor( 3,27, na_rgu, 1) =   -2.715; //  H3*
  icoor( 1,28, na_rgu, 1) =   -1.398; // 1H2*
  icoor( 2,28, na_rgu, 1) =    0.026; // 1H2*
  icoor( 3,28, na_rgu, 1) =   -1.746; // 1H2*
  icoor( 1,29, na_rgu, 1) =   -0.639; //  H1*
  icoor( 2,29, na_rgu, 1) =   -0.197; //  H1*
  icoor( 3,29, na_rgu, 1) =    0.874; //  H1*
	icoor( 1,30, na_rgu, 1) =   -2.239; // 2HO*
	icoor( 2,30, na_rgu, 1) =   -1.409; // 2HO*
	icoor( 3,30, na_rgu, 1) =   -0.237; // 2HO*
  icoor( 1,31, na_rgu, 1) =   -2.704; //  H1
  icoor( 2,31, na_rgu, 1) =    5.209; //  H1
  icoor( 3,31, na_rgu, 1) =    1.469; //  H1
  icoor( 1,32, na_rgu, 1) =   -4.233; // 1H2
  icoor( 2,32, na_rgu, 1) =    2.178; // 1H2
  icoor( 3,32, na_rgu, 1) =    2.054; // 1H2
  icoor( 1,33, na_rgu, 1) =   -4.319; // 2H2
  icoor( 2,33, na_rgu, 1) =    3.904; // 2H2
  icoor( 3,33, na_rgu, 1) =    2.172; // 2H2
  icoor( 1,34, na_rgu, 1) =    2.106; //  H8
  icoor( 2,34, na_rgu, 1) =    1.716; //  H8
  icoor( 3,34, na_rgu, 1) =   -0.974; //  H8

//bk   create variants that have h2o connected to the backbone nh and carbonyl
//jjh  currently disabled.
//      if ( explicit_h2o ) {
//         create_backbone_h2o_variants(na_rgu);
//      }

}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_rna_ade
///
/// @brief arrays filled to define the proerties of the nucleic acid rAdenine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// na_rad - index number for rna ADE
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_rna_ade()
{
	using namespace aaproperties_pack;
	using namespace param_aa;

//bk   Parameters for rADE -----------------------------
	nvar(na_rad) = 1; // starting starting number of variants for rADE

//bk   variant1 ---------- rADE

	natoms(na_rad,1) = 33; // number of total atoms
	nheavyatoms(na_rad,1) = 22; // number of heavy atoms
	nchi(na_rad,1) = 1; // number of chi angles

//bk   atom names ADE
	atom_name( 1, na_rad,1) = " P  ";
	atom_name( 2, na_rad,1) = " O1P";
	atom_name( 3, na_rad,1) = " O2P";
	atom_name( 4, na_rad,1) = " O5*";
	atom_name( 5, na_rad,1) = " C5*";
	atom_name( 6, na_rad,1) = " C4*";
	atom_name( 7, na_rad,1) = " O4*";
	atom_name( 8, na_rad,1) = " C3*";
	atom_name( 9, na_rad,1) = " O3*";
	atom_name(10, na_rad,1) = " C2*";
	atom_name(11, na_rad,1) = " O2*";
	atom_name(12, na_rad,1) = " C1*";
	atom_name(13, na_rad,1) = " N1 ";
	atom_name(14, na_rad,1) = " C2 ";
	atom_name(15, na_rad,1) = " N3 ";
	atom_name(16, na_rad,1) = " C4 ";
	atom_name(17, na_rad,1) = " C5 ";
	atom_name(18, na_rad,1) = " C6 ";
	atom_name(19, na_rad,1) = " N6 ";
	atom_name(20, na_rad,1) = " N7 ";
	atom_name(21, na_rad,1) = " C8 ";
	atom_name(22, na_rad,1) = " N9 ";
	atom_name(23, na_rad,1) = "1H5*";
	atom_name(24, na_rad,1) = "2H5*";
	atom_name(25, na_rad,1) = " H4*";
	atom_name(26, na_rad,1) = " H3*";
	atom_name(27, na_rad,1) = "1H2*";
	atom_name(28, na_rad,1) = "2HO*";
	atom_name(29, na_rad,1) = " H1*";
	atom_name(30, na_rad,1) = " H2 ";
	atom_name(31, na_rad,1) = "1H6 ";
	atom_name(32, na_rad,1) = "2H6 ";
	atom_name(33, na_rad,1) = " H8 ";

//jjh intra residue bonding
	nbonded_neighbors( 1, na_rad, 1) = 3; // P
		bonded_neighbor(1, 1, na_rad, 1) = 2; // P--O1P
		bonded_neighbor(2, 1, na_rad, 1) = 3; // P--O2P
		bonded_neighbor(3, 1, na_rad, 1) = 4; // P--O5*
	nbonded_neighbors( 2, na_rad, 1) = 1; // O1P
		bonded_neighbor(1, 2, na_rad, 1) = 1; // O1P--P
	nbonded_neighbors( 3, na_rad, 1) = 1; // O2P
		bonded_neighbor(1, 3, na_rad, 1) = 1; // O2P--P
	nbonded_neighbors( 4, na_rad, 1) = 2; // O5*
		bonded_neighbor(1, 4, na_rad, 1) = 1; // O5*--P
		bonded_neighbor(2, 4, na_rad, 1) = 5; // O5*--C5*
	nbonded_neighbors( 5, na_rad, 1) = 4; // C5*
		bonded_neighbor(1, 5, na_rad, 1) = 4; // C5*--O5*
		bonded_neighbor(2, 5, na_rad, 1) = 6; // C5*--C4*
		bonded_neighbor(3, 5, na_rad, 1) =23; // C5*--1H5*
		bonded_neighbor(4, 5, na_rad, 1) =24; // C5*--2H5*
	nbonded_neighbors( 6, na_rad, 1) = 4; // C4*
		bonded_neighbor(1, 6, na_rad, 1) = 5; // C4*--C5*
		bonded_neighbor(2, 6, na_rad, 1) = 7; // C4*--O4*
		bonded_neighbor(3, 6, na_rad, 1) = 8; // C4*--C3*
		bonded_neighbor(4, 6, na_rad, 1) =25; // C4*--H4*
	nbonded_neighbors( 7, na_rad, 1) = 2; // O4*
		bonded_neighbor(1, 7, na_rad, 1) = 6; // O4*--C4*
		bonded_neighbor(2, 7, na_rad, 1) =12; // O4*--C1*
	nbonded_neighbors( 8, na_rad, 1) = 4; // C3*
		bonded_neighbor(1, 8, na_rad, 1) = 6; // C3*--C4*
		bonded_neighbor(2, 8, na_rad, 1) = 9; // C3*--O3*
		bonded_neighbor(3, 8, na_rad, 1) =10; // C3*--C2*
		bonded_neighbor(4, 8, na_rad, 1) =26; // C3*--H3*
	nbonded_neighbors( 9, na_rad, 1) = 1; // O3*
		bonded_neighbor(1, 9, na_rad, 1) = 8; // O3*--C3*
	nbonded_neighbors( 10, na_rad, 1) = 4; // C2*
		bonded_neighbor(1,10, na_rad, 1) = 8; // C2*--C3*
		bonded_neighbor(2,10, na_rad, 1) =11; // C2*--O2*
		bonded_neighbor(3,10, na_rad, 1) =12; // C2*--C1*
		bonded_neighbor(4,10, na_rad, 1) =27; // C2*--1H2*
	nbonded_neighbors( 11, na_rad, 1) = 2; // O2*
		bonded_neighbor(1,11, na_rad, 1) =10; // O2*--C2*
		bonded_neighbor(2,11, na_rad, 1) =28; // O2*--2HO*
	nbonded_neighbors( 12, na_rad, 1) = 4; // C1*
		bonded_neighbor(1,12, na_rad, 1) = 7; // C1*--O4*
		bonded_neighbor(2,12, na_rad, 1) =10; // C1*--C2*
		bonded_neighbor(3,12, na_rad, 1) =22; // C1*--N9
		bonded_neighbor(4,12, na_rad, 1) =29; // C1*--H1*
	nbonded_neighbors( 13, na_rad, 1) = 2; // N1
		bonded_neighbor(1,13, na_rad, 1) =14; // N1--C2
		bonded_neighbor(2,13, na_rad, 1) =18; // N1--C6
	nbonded_neighbors( 14, na_rad, 1) = 3; // C2
		bonded_neighbor(1,14, na_rad, 1) =13; // C2--N1
		bonded_neighbor(2,14, na_rad, 1) =15; // C2--N3
		bonded_neighbor(3,14, na_rad, 1) =30; // C2--H2
	nbonded_neighbors( 15, na_rad, 1) = 2; // N3
		bonded_neighbor(1,15, na_rad, 1) =14; // N3--C2
		bonded_neighbor(2,15, na_rad, 1) =16; // N3--C4
	nbonded_neighbors( 16, na_rad, 1) = 3; // C4
		bonded_neighbor(1,16, na_rad, 1) =15; // C4--N3
		bonded_neighbor(2,16, na_rad, 1) =17; // C4--C5
		bonded_neighbor(3,16, na_rad, 1) =22; // C4--N9
	nbonded_neighbors( 17, na_rad, 1) = 3; // C5
		bonded_neighbor(1,17, na_rad, 1) =16; // C5--C4
		bonded_neighbor(2,17, na_rad, 1) =18; // C5--C6
		bonded_neighbor(3,17, na_rad, 1) =20; // C5--N7
	nbonded_neighbors( 18, na_rad, 1) = 3; // C6
		bonded_neighbor(1,18, na_rad, 1) =13; // C6--N1
		bonded_neighbor(2,18, na_rad, 1) =17; // C6--C5
		bonded_neighbor(3,18, na_rad, 1) =19; // C6--N6
	nbonded_neighbors( 19, na_rad, 1) = 3; // N6
		bonded_neighbor(1,19, na_rad, 1) =18; // N6--C6
		bonded_neighbor(2,19, na_rad, 1) =31; // N6--1H6
		bonded_neighbor(3,19, na_rad, 1) =32; // N6--2H6
	nbonded_neighbors( 20, na_rad, 1) = 2; // N7
		bonded_neighbor(1,20, na_rad, 1) =17; // N7--C5
		bonded_neighbor(2,20, na_rad, 1) =21; // N7--C8
	nbonded_neighbors( 21, na_rad, 1) = 3; // C8
		bonded_neighbor(1,21, na_rad, 1) =20; // C8--N7
		bonded_neighbor(2,21, na_rad, 1) =22; // C8--N9
		bonded_neighbor(3,21, na_rad, 1) =33; // C8--H8
	nbonded_neighbors( 22, na_rad, 1) = 3; // N9
		bonded_neighbor(1,22, na_rad, 1) =12; // N9--C1*
		bonded_neighbor(2,22, na_rad, 1) =16; // N9--C4
		bonded_neighbor(3,22, na_rad, 1) =21; // N9--C8
	nbonded_neighbors( 23, na_rad, 1) = 1; //1H5*
		bonded_neighbor(1,23, na_rad, 1) = 5; //1H5*--C5*
	nbonded_neighbors( 24, na_rad, 1) = 1; //2H5*
		bonded_neighbor(1,24, na_rad, 1) = 5; //2H5*--C5*
	nbonded_neighbors( 25, na_rad, 1) = 1; // H4*
		bonded_neighbor(1,25, na_rad, 1) = 6; // H4*--C4*
	nbonded_neighbors( 26, na_rad, 1) = 1; // H3*
		bonded_neighbor(1,26, na_rad, 1) = 8; // H3*--C3*
	nbonded_neighbors( 27, na_rad, 1) = 1; //1H2*
		bonded_neighbor(1,27, na_rad, 1) =10; //1H2*--C2*
	nbonded_neighbors( 28, na_rad, 1) = 1; //2HO*
		bonded_neighbor(1,28, na_rad, 1) =11; //2HO*--O2*
	nbonded_neighbors( 29, na_rad, 1) = 1; // H1*
		bonded_neighbor(1,29, na_rad, 1) =12; // H1*--C1*
	nbonded_neighbors( 30, na_rad, 1) = 1; // H2
		bonded_neighbor(1,30, na_rad, 1) =14; // H2--C2
	nbonded_neighbors( 31, na_rad, 1) = 1; //1H6
		bonded_neighbor(1,31, na_rad, 1) =19; //1H6--N6
	nbonded_neighbors( 32, na_rad, 1) = 1; //2H6
		bonded_neighbor(1,32, na_rad, 1) =19; //2H6--N6
	nbonded_neighbors( 33, na_rad, 1) = 1; // H8
		bonded_neighbor(1,33, na_rad, 1) =21; // H8--C8

//jjh get the angles automatically
	get_angles_from_bonds(na_rad, 1);

	atom_base( 1,na_rad,1) =  2; //   P   --> O1P
	atom_base( 2,na_rad,1) =  1; //   O1P --> P
	atom_base( 3,na_rad,1) =  1; //   O2P --> P
	atom_base( 4,na_rad,1) =  1; //   O5* --> P
	abase2( 4, na_rad, 1) =  5; //  C5* base2 for  O5*
	atom_base( 5,na_rad,1) =  4; //   C5* --> O5*
	atom_base( 6,na_rad,1) =  5; //   C4* --> C5*
	atom_base( 7,na_rad,1) = 12; //   O4* --> C1* //abase=C4*
	abase2( 7, na_rad, 1) =  6; //  C4* base2 for  O4*
	atom_base( 8,na_rad,1) =  6; //   C3* --> C4*
	atom_base( 9,na_rad,1) =  8; //   O3* --> C3*
	atom_base(10,na_rad,1) =  8; //   C2* --> C3*
	atom_base(11,na_rad,1) = 10; //   O2* --> C2*
	abase2(11, na_rad, 1) = 28; // 2HO* base2 for  O2*
	atom_base(12,na_rad,1) = 10; //   C1* --> C2*
	atom_base(13,na_rad,1) = 18; //   N1  --> C6
	abase2(13, na_rad, 1) = 14; //  C2  base2 for  N1
	atom_base(14,na_rad,1) = 15; //   C2  --> N3
	atom_base(15,na_rad,1) = 14; //   N3  --> C2
	abase2(15, na_rad, 1) = 16; //  C4  base2 for  N3
	atom_base(16,na_rad,1) = 22; //   C4  --> N9
	atom_base(17,na_rad,1) = 20; //   C5  --> N7
	atom_base(18,na_rad,1) = 17; //   C6  --> C5
	atom_base(19,na_rad,1) = 18; //   N6  --> C6
	atom_base(20,na_rad,1) = 17; //   N7  --> C5
	abase2(20, na_rad, 1) = 21; //  C8  base2 for  N7
	atom_base(21,na_rad,1) = 22; //   C8  --> N9
	atom_base(22,na_rad,1) = 12; //   N9  --> C1*
	atom_base(23,na_rad,1) =  5; //  1H5* --> C5*
	atom_base(24,na_rad,1) =  5; //  2H5* --> C5*`
	atom_base(25,na_rad,1) =  6; //   H4* --> C4*
	atom_base(26,na_rad,1) =  8; //   H3* --> C3*
	atom_base(27,na_rad,1) = 10; //  1H2* --> C2*
	atom_base(28,na_rad,1) = 11; //  2HO* --> O2*
	atom_base(29,na_rad,1) = 12; //   H1* --> C1*
	atom_base(30,na_rad,1) = 14; //   H2  --> C2
	atom_base(31,na_rad,1) = 19; //  1H6  --> N6
	atom_base(32,na_rad,1) = 19; //  2H6  --> N6
	atom_base(33,na_rad,1) = 21; //   H8  --> C8

//vats   atom type numbers ADE (Vatsan Raman)
//vats New atom types for Nucleic Acid based on CHARMM 27
//vats Ref etable.h for more details
//cmd  Reverted to rosetta atom types
	fullatom_type( 1,na_rad,1) = 21; // Phos   P    P
	fullatom_type( 2,na_rad,1) = 15; // 69; // OOC    O1P  ON3
	fullatom_type( 3,na_rad,1) = 15; // 69; // OOC    O2P  ON3
	fullatom_type( 4,na_rad,1) = 14; // 68; // ONH2   O5*  ON2
	fullatom_type( 5,na_rad,1) =  4; // 61; // CH2    C5*  CN8 (CN8B)
	fullatom_type( 6,na_rad,1) =  3; // 60; // CH1    C4*  CN7
	fullatom_type( 7,na_rad,1) = 13; // 72; // OH     O4*  ON6 (ON6B)
	fullatom_type( 8,na_rad,1) =  3; // 60; // CH1    C3*  CN7
	fullatom_type( 9,na_rad,1) = 14; // 68; // ONH2   O3*  ON2
	fullatom_type(10,na_rad,1) =  4; // 60; // CH2    C2*  CN7 (CN7B)
	fullatom_type(11,na_rad,1) = 13; // 71; // OH     O2*  ON5
	fullatom_type(12,na_rad,1) =  3; // 60; // CH1    C1*  CN7 (CN7B)
	fullatom_type(13,na_rad,1) =  8; // 65; // Nhis   N1   NN3 (NN3A)
	fullatom_type(14,na_rad,1) =  6; // 58; // aroC   C2   CN4
	fullatom_type(15,na_rad,1) =  8; // 65; // Nhis   N3   NN3 (NN3A)
	fullatom_type(16,na_rad,1) =  6; // 59; // aroC   C4   CN5
	fullatom_type(17,na_rad,1) =  6; // 59; // aroC   C5   CN5
	fullatom_type(18,na_rad,1) =  6; // 56; // aroC   C6   CN2
	fullatom_type(19,na_rad,1) =  9; // 63; // NH2O   N6   NN1
	fullatom_type(20,na_rad,1) =  8; // 66; // Nhis   N7   NN4
	fullatom_type(21,na_rad,1) =  6; // 58; // aroC   C8   CN4
	fullatom_type(22,na_rad,1) =  8; // 64; // Nhis   N9   NN2
	fullatom_type(23,na_rad,1) = 23; // 74; // Hapo  1H5*  HNP
	fullatom_type(24,na_rad,1) = 23; // 74; // Hapo  2H5*  HNP
	fullatom_type(25,na_rad,1) = 23; // 74; // Hapo   H4*  HNP
	fullatom_type(26,na_rad,1) = 23; // 74; // Hapo   H3*  HNP
	fullatom_type(27,na_rad,1) = 23; // 74; // Hapo  1H2*  HNP
	fullatom_type(28,na_rad,1) = 23; // 73; // Hpol  2HO*  HP
	fullatom_type(29,na_rad,1) = 23; // 74; // Hapo   H1*  HNP
	fullatom_type(30,na_rad,1) = 24; // 74; // Haro   H2   HNP
	fullatom_type(31,na_rad,1) = 22; // 73; // Hpol  1H6   HP
	fullatom_type(32,na_rad,1) = 22; // 73; // Hpol  2H6   HP
	fullatom_type(33,na_rad,1) = 24; // 74; // Haro   H8   HNP

//bk   atom partial charges ADE
	atomic_charge( 1, na_rad,1) = 1.500; // P
	atomic_charge( 2, na_rad,1) = -.780; // O1P
	atomic_charge( 3, na_rad,1) = -.780; // O2P
	atomic_charge( 4, na_rad,1) = -.570; // O5*
	atomic_charge( 5, na_rad,1) = -.080; // C5*
	atomic_charge( 6, na_rad,1) =  .160; // C4*
	atomic_charge( 7, na_rad,1) = -.500; // O4*
	atomic_charge( 8, na_rad,1) =  .010; // C3*
	atomic_charge( 9, na_rad,1) = -.570; // O3*
	atomic_charge(10, na_rad,1) =  .140; // C2*
	atomic_charge(11, na_rad,1) = -.660; // O2*
	atomic_charge(12, na_rad,1) =  .160; // C1*
	atomic_charge(13, na_rad,1) = -.740; // N1
	atomic_charge(14, na_rad,1) =  .500; // C2
	atomic_charge(15, na_rad,1) = -.750; // N3
	atomic_charge(16, na_rad,1) =  .430; // C4
	atomic_charge(17, na_rad,1) =  .280; // C5
	atomic_charge(18, na_rad,1) =  .460; // C6
	atomic_charge(19, na_rad,1) = -.770; // N6
	atomic_charge(20, na_rad,1) = -.710; // N7
	atomic_charge(21, na_rad,1) =  .340; // C8
	atomic_charge(22, na_rad,1) = -.050; // N9
	atomic_charge(23, na_rad,1) =  .090; // 1H5*
	atomic_charge(24, na_rad,1) =  .090; // 2H5*
	atomic_charge(25, na_rad,1) =  .090; // H4*
	atomic_charge(26, na_rad,1) =  .090; // H3*
	atomic_charge(27, na_rad,1) =  .090; // 1H2*
	atomic_charge(28, na_rad,1) =  .430; // 2HO*
	atomic_charge(29, na_rad,1) =  .090; // H1*
	atomic_charge(30, na_rad,1) =  .130; // H2
	atomic_charge(31, na_rad,1) =  .380; // 1H6
	atomic_charge(32, na_rad,1) =  .380; // 2H6
	atomic_charge(33, na_rad,1) =  .120; // H8

	assert_total_charge( na_rad, 1, -1.0);

//cmd  first sidechain atom
	first_scatom( na_rad,1) = 13; //N1

//jjh anchor atoms for base mutations
	na_anchor(1, na_rad, 1) = 12;
	na_anchor(2, na_rad, 1) = 22;
	na_anchor(3, na_rad, 1) = 16;

//bk   template atoms used for placing hydrogens

//bk   template for building  P
	ta(1, 1, na_rad,1) =  4; //   O5*
	ta(2, 1, na_rad,1) =  5; //   C5*
	ta(3, 1, na_rad,1) =  6; //   C4*

//bk   template for building  O1P
	ta(1, 2, na_rad,1) =  1; //   P
	ta(2, 2, na_rad,1) =  5; //   C5*
	ta(3, 2, na_rad,1) =  4; //   O5*

//bk   template for building  O2P
	ta(1, 3, na_rad,1) =  1; //   P
	ta(2, 3, na_rad,1) =  2; //   O1P
	ta(3, 3, na_rad,1) =  4; //   O5*

//bk   template for building  O2*
	ta(1, 11, na_rad,1) =  10; //   C2*
	ta(2, 11, na_rad,1) =   8; //   C3*
	ta(3, 11, na_rad,1) =  12; //   C1*

//bk   template for building 1H5*
	ta(1,23, na_rad,1) =  5; //   C5*
	ta(2,23, na_rad,1) =  4; //   O5*
	ta(3,23, na_rad,1) =  6; //   C4*

//bk   template for building 2H5*
	ta(1,24, na_rad,1) =  5; //   C5*
	ta(2,24, na_rad,1) =  4; //   O5*
	ta(3,24, na_rad,1) =  6; //   C4*

//bk   template for building  H4*
	ta(1,25, na_rad,1) =  6; //   C4*
	ta(2,25, na_rad,1) =  7; //   O4*
	ta(3,25, na_rad,1) =  5; //   C5*

//bk   template for building  H3*
	ta(1,26, na_rad,1) =  8; //   C3*
	ta(2,26, na_rad,1) = 10; //   C2*
	ta(3,26, na_rad,1) =  6; //   C4*

//bk   template for building 1H2*
	ta(1,27, na_rad,1) = 10; //   C2*
	ta(2,27, na_rad,1) =  8; //   C3*
	ta(3,27, na_rad,1) = 12; //   C1*

//bk   template for building  2HO*
	ta(1,28, na_rad,1) = 11; //   O2*
	ta(2,28, na_rad,1) = 10; //   C2*
	ta(3,28, na_rad,1) = 12; //   C1*

//bk   template for building  H1*
	ta(1,29, na_rad,1) = 12; //   C1*
	ta(2,29, na_rad,1) = 10; //   C2*
	ta(3,29, na_rad,1) =  7; //   O4*

//bk   template for building  H2
	ta(1,30, na_rad,1) = 14; //   C2
	ta(2,30, na_rad,1) = 13; //   N1
	ta(3,30, na_rad,1) = 15; //   N3

//bk   template for building 1H6
	ta(1,31, na_rad,1) = 19; //   N6
	ta(2,31, na_rad,1) = 18; //   C6
	ta(3,31, na_rad,1) = 13; //   N1

//bk   template for building 2H6
	ta(1,32, na_rad,1) = 19; //   N6
	ta(2,32, na_rad,1) = 18; //   C6
	ta(3,32, na_rad,1) = 13; //   N1

//bk   template for building  H8
	ta(1,33, na_rad,1) = 21; //   C8
	ta(2,33, na_rad,1) = 20; //   N7
	ta(3,33, na_rad,1) = 22; //   N9

//bk   identify atoms that require chi angles to be built
//cy  chi anlges needed for building 2HO*
	chi_required(1, 28,na_rad,1) = true;

//cy   four atoms that define chi angle 1
	chi_atoms( 1, 1,na_rad, 1) = 12; //   C1*
	chi_atoms( 2, 1,na_rad, 1) = 10; //   C2*
	chi_atoms( 3, 1,na_rad, 1) = 11; //   O2*
	chi_atoms( 4, 1,na_rad, 1) = 28; //   2HO*

//bk   hbonding information
//jjh set the next two to nonsense values
	HNpos( na_rad,1) = -1; // position of backbone HN
	HApos( na_rad,1) = -1; // position of backbone HA

	nH_polar( na_rad,1) =  3; // number of polar hydrogens
	nH_aromatic( na_rad,1) =  2; // number of aromatic hydrogens
	nH_apolar( na_rad,1) =  6; // number of apolar hydrogens
	nacceptors( na_rad,1) =  9; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1, na_rad,1) = 28; // 2HO*
	Hpos_polar( 2, na_rad,1) = 31; // 1H6
	Hpos_polar( 3, na_rad,1) = 32; // 2H6

//bk   atom numbers for aromatic hydrogens
	Hpos_aromatic( 1, na_rad,1) = 30; //  H2
	Hpos_aromatic( 2, na_rad,1) = 33; //  H8

//bk   atom numbers for apolar hydrogens
	Hpos_apolar( 1, na_rad,1) = 23; //1H5*
	Hpos_apolar( 2, na_rad,1) = 24; //2H5*
	Hpos_apolar( 3, na_rad,1) = 25; // H4*
	Hpos_apolar( 4, na_rad,1) = 26; // H3*
	Hpos_apolar( 5, na_rad,1) = 27; //1H2*
	Hpos_apolar( 6, na_rad,1) = 29; // H1*

//bk   acceptor positions
	accpt_pos( 1, na_rad, 1) =  2; //  O1P
	accpt_pos( 2, na_rad, 1) =  3; //  O2P
	accpt_pos( 3, na_rad, 1) =  4; //  O5*
	accpt_pos( 4, na_rad, 1) =  9; //  O3*
	accpt_pos( 5, na_rad, 1) = 11; //  O2*
	accpt_pos( 6, na_rad, 1) = 15; //  N3
	accpt_pos( 7, na_rad, 1) = 20; //  N7
	accpt_pos( 8, na_rad, 1) =  7; //  O4*
	accpt_pos( 9, na_rad, 1) = 13; //  N1

//bk   hydrogens on each atom

	nhydrogens_on_atm( 5, na_rad, 1) =  2; //  C5*
	hydrogens_on_atm( 1, 5, na_rad, 1) = 23; // 1H5*
	hydrogens_on_atm( 2, 5, na_rad, 1) = 24; // 2H5*

	nhydrogens_on_atm( 6, na_rad, 1) =  1; //  C4*
	hydrogens_on_atm( 1, 6, na_rad, 1) =  25; //  H4*

	nhydrogens_on_atm( 8, na_rad, 1) =  1; //  C3*
	hydrogens_on_atm( 1, 8, na_rad, 1) =  26; //  H3*

	nhydrogens_on_atm(10, na_rad, 1) =  1; //  C2*
	hydrogens_on_atm( 1,10, na_rad, 1) =  27; // 1H2*

	nhydrogens_on_atm(11, na_rad, 1) =  1; //  O2*
	hydrogens_on_atm( 1,11, na_rad, 1) =  28; // 2HO*

	nhydrogens_on_atm(12, na_rad, 1) =  1; //  C1*
	hydrogens_on_atm( 1,12, na_rad, 1) =  29; //  H1*

	nhydrogens_on_atm(14, na_rad, 1) =  1; //  C2
	hydrogens_on_atm( 1,14, na_rad, 1) =  30; //  H2

	nhydrogens_on_atm(19, na_rad, 1) =  2; //  N6
	hydrogens_on_atm( 1,19, na_rad, 1) =  31; // 1H6
	hydrogens_on_atm( 2,19, na_rad, 1) =  32; // 2H6

	nhydrogens_on_atm(21, na_rad, 1) =  1; //  C8
	hydrogens_on_atm( 1,21, na_rad, 1) = 33; //  H8

//bk   template coordinates for the nucleic acid
//rhiju Fixed icoor, which had base poking out of
//rhiju sugar wrong. The takeoff angles from N1 read from 2INA.pdb,
//rhiju which uses ideal values used by  xplor and the NDB.
//rhiju Nov. 10, 2006.
	icoor( 1, 1, na_rad, 1) =    4.691; //  P
	icoor( 2, 1, na_rad, 1) =    0.327; //  P
	icoor( 3, 1, na_rad, 1) =   -2.444; //  P
	icoor( 1, 2, na_rad, 1) =    5.034; //  O1P
	icoor( 2, 2, na_rad, 1) =    1.678; //  O1P
	icoor( 3, 2, na_rad, 1) =   -1.932; //  O1P
	icoor( 1, 3, na_rad, 1) =    4.718; //  O2P
	icoor( 2, 3, na_rad, 1) =    0.068; //  O2P
	icoor( 3, 3, na_rad, 1) =   -3.906; //  O2P
	icoor( 1, 4, na_rad, 1) =    3.246; //  O5*
	icoor( 2, 4, na_rad, 1) =   -0.057; //  O5*
	icoor( 3, 4, na_rad, 1) =   -1.895; //  O5*
	icoor( 1, 5, na_rad, 1) =    2.957; //  C5*
	icoor( 2, 5, na_rad, 1) =   -1.393; //  C5*
	icoor( 3, 5, na_rad, 1) =   -1.443; //  C5*
	icoor( 1, 6, na_rad, 1) =    1.509; //  C4*
	icoor( 2, 6, na_rad, 1) =   -1.478; //  C4*
	icoor( 3, 6, na_rad, 1) =   -1.022; //  C4*
	icoor( 1, 7, na_rad, 1) =    1.286; //  O4*
	icoor( 2, 7, na_rad, 1) =   -0.587; //  O4*
	icoor( 3, 7, na_rad, 1) =    0.103; //  O4*
	icoor( 1, 8, na_rad, 1) =    0.499; //  C3*
	icoor( 2, 8, na_rad, 1) =   -1.031; //  C3*
	icoor( 3, 8, na_rad, 1) =   -2.067; //  C3*
	icoor( 1, 9, na_rad, 1) =    0.178; //  O3*
	icoor( 2, 9, na_rad, 1) =   -2.084; //  O3*
	icoor( 3, 9, na_rad, 1) =   -2.981; //  O3*
	icoor( 1,10, na_rad, 1) =   -0.694; //  C2*
	icoor( 2,10, na_rad, 1) =   -0.627; //  C2*
	icoor( 3,10, na_rad, 1) =   -1.210; //  C2*
	icoor( 1,11, na_rad, 1) =   -1.474; //  O2*
	icoor( 2,11, na_rad, 1) =   -1.731; //  O2*
	icoor( 3,11, na_rad, 1) =   -0.795; //  O2*
	icoor( 1,12, na_rad, 1) =    0.000; //  C1*
	icoor( 2,12, na_rad, 1) =    0.000; //  C1*
	icoor( 3,12, na_rad, 1) =    0.000; //  C1*
	icoor( 1,13, na_rad, 1) =   -2.115; //  N1
	icoor( 2,13, na_rad, 1) =    4.525; //  N1
	icoor( 3,13, na_rad, 1) =    1.186; //  N1
	icoor( 1,14, na_rad, 1) =   -2.540; //  C2
	icoor( 2,14, na_rad, 1) =    3.265; //  C2
	icoor( 3,14, na_rad, 1) =    1.332; //  C2
	icoor( 1,15, na_rad, 1) =   -1.942; //  N3
	icoor( 2,15, na_rad, 1) =    2.127; //  N3
	icoor( 3,15, na_rad, 1) =    0.987; //  N3
	icoor( 1,16, na_rad, 1) =   -0.749; //  C4
	icoor( 2,16, na_rad, 1) =    2.374; //  C4
	icoor( 3,16, na_rad, 1) =    0.420; //  C4
	icoor( 1,17, na_rad, 1) =   -0.176; //  C5
	icoor( 2,17, na_rad, 1) =    3.614; //  C5
	icoor( 3,17, na_rad, 1) =    0.201; //  C5
	icoor( 1,18, na_rad, 1) =   -0.911; //  C6
	icoor( 2,18, na_rad, 1) =    4.739; //  C6
	icoor( 3,18, na_rad, 1) =    0.612; //  C6
	icoor( 1,19, na_rad, 1) =   -0.486; //  N6
	icoor( 2,19, na_rad, 1) =    5.996; //  N6
	icoor( 3,19, na_rad, 1) =    0.465; //  N6
	icoor( 1,20, na_rad, 1) =    1.065; //  N7
	icoor( 2,20, na_rad, 1) =    3.481; //  N7
	icoor( 3,20, na_rad, 1) =   -0.407; //  N7
	icoor( 1,21, na_rad, 1) =    1.213; //  C8
	icoor( 2,21, na_rad, 1) =    2.186; //  C8
	icoor( 3,21, na_rad, 1) =   -0.541; //  C8
	icoor( 1,22, na_rad, 1) =    0.153; //  N9
	icoor( 2,22, na_rad, 1) =    1.455; //  N9
	icoor( 3,22, na_rad, 1) =   -0.061; //  N9
	icoor( 1,23, na_rad, 1) =    3.612; // 1H5*
	icoor( 2,23, na_rad, 1) =   -1.657; // 1H5*
	icoor( 3,23, na_rad, 1) =   -0.600; // 1H5*
	icoor( 1,24, na_rad, 1) =    3.160; // 2H5*
	icoor( 2,24, na_rad, 1) =   -2.116; // 2H5*
	icoor( 3,24, na_rad, 1) =   -2.247; // 2H5*
	icoor( 1,25, na_rad, 1) =    1.353; //  H4*
	icoor( 2,25, na_rad, 1) =   -2.546; //  H4*
	icoor( 3,25, na_rad, 1) =   -0.810; //  H4*
	icoor( 1,26, na_rad, 1) =    0.863; //  H3*
	icoor( 2,26, na_rad, 1) =   -0.220; //  H3*
	icoor( 3,26, na_rad, 1) =   -2.715; //  H3*
	icoor( 1,27, na_rad, 1) =   -1.398; // 1H2*
	icoor( 2,27, na_rad, 1) =    0.026; // 1H2*
	icoor( 3,27, na_rad, 1) =   -1.746; // 1H2*
	icoor( 1,28, na_rad, 1) =   -2.239; // 2HO*
	icoor( 2,28, na_rad, 1) =   -1.409; // 2HO*
	icoor( 3,28, na_rad, 1) =   -0.237; // 2HO*
	icoor( 1,29, na_rad, 1) =   -0.641; //  H1*
	icoor( 2,29, na_rad, 1) =   -0.195; //  H1*
	icoor( 3,29, na_rad, 1) =    0.873; //  H1*
	icoor( 1,30, na_rad, 1) =   -3.526; //  H2
	icoor( 2,30, na_rad, 1) =    3.154; //  H2
	icoor( 3,30, na_rad, 1) =    1.804; //  H2
	icoor( 1,31, na_rad, 1) =   -1.056; // 1H6
	icoor( 2,31, na_rad, 1) =    6.757; // 1H6
	icoor( 3,31, na_rad, 1) =    0.778; // 1H6
	icoor( 1,32, na_rad, 1) =    0.401; // 2H6
	icoor( 2,32, na_rad, 1) =    6.178; // 2H6
	icoor( 3,32, na_rad, 1) =    0.043; // 2H6
	icoor( 1,33, na_rad, 1) =    2.104; //  H8
	icoor( 2,33, na_rad, 1) =    1.726; //  H8
	icoor( 3,33, na_rad, 1) =   -0.995; //  H8

//bk   create variants that have h2o connected to the backbone nh and carbonyl
//jjh  currently disabled.
//      if ( explicit_h2o ) {
//         create_backbone_h2o_variants(na_rad);
//      }

}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_rna_cyt
///
/// @brief arrays filled to define the proerties of the nucleic acid rCytosine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// na_rcy - index number for rna CYT
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_rna_cyt()
{
	using namespace aaproperties_pack;
	using namespace param_aa;

//bk   Parameters for rCYT -----------------------------
	nvar(na_rcy) = 1; // starting starting number of variants for rCYT

//bk   variant1 ---------- rCYT

	natoms(na_rcy,1) = 31; // number of total atoms
	nheavyatoms(na_rcy,1) = 20; // number of heavy atoms
	nchi(na_rcy,1) = 1; // number of chi angles

//bk   atom names rCYT
	atom_name( 1, na_rcy,1) = " P  ";
	atom_name( 2, na_rcy,1) = " O1P";
	atom_name( 3, na_rcy,1) = " O2P";
	atom_name( 4, na_rcy,1) = " O5*";
	atom_name( 5, na_rcy,1) = " C5*";
	atom_name( 6, na_rcy,1) = " C4*";
	atom_name( 7, na_rcy,1) = " O4*";
	atom_name( 8, na_rcy,1) = " C3*";
	atom_name( 9, na_rcy,1) = " O3*";
	atom_name(10, na_rcy,1) = " C2*";
	atom_name(11, na_rcy,1) = " O2*";
	atom_name(12, na_rcy,1) = " C1*";
	atom_name(13, na_rcy,1) = " N1 ";
	atom_name(14, na_rcy,1) = " C2 ";
	atom_name(15, na_rcy,1) = " O2 ";
	atom_name(16, na_rcy,1) = " N3 ";
	atom_name(17, na_rcy,1) = " C4 ";
	atom_name(18, na_rcy,1) = " N4 ";
	atom_name(19, na_rcy,1) = " C5 ";
	atom_name(20, na_rcy,1) = " C6 ";
	atom_name(21, na_rcy,1) = "1H5*";
	atom_name(22, na_rcy,1) = "2H5*";
	atom_name(23, na_rcy,1) = " H4*";
	atom_name(24, na_rcy,1) = " H3*";
	atom_name(25, na_rcy,1) = "1H2*";
	atom_name(26, na_rcy,1) = "2HO*";
	atom_name(27, na_rcy,1) = " H1*";
	atom_name(28, na_rcy,1) = "1H4 ";
	atom_name(29, na_rcy,1) = "2H4 ";
	atom_name(30, na_rcy,1) = " H5 ";
	atom_name(31, na_rcy,1) = " H6 ";

//jjh intra residue bonding
	nbonded_neighbors( 1, na_rcy, 1) = 3; // P
		bonded_neighbor(1, 1, na_rcy, 1) = 2; // P--O1P
		bonded_neighbor(2, 1, na_rcy, 1) = 3; // P--O2P
		bonded_neighbor(3, 1, na_rcy, 1) = 4; // P--O5*
	nbonded_neighbors( 2, na_rcy, 1) = 1; // O1P
		bonded_neighbor(1, 2, na_rcy, 1) = 1; // O1P--P
	nbonded_neighbors( 3, na_rcy, 1) = 1; // O2P
		bonded_neighbor(1, 3, na_rcy, 1) = 1; // O2P--P
	nbonded_neighbors( 4, na_rcy, 1) = 2; // O5*
		bonded_neighbor(1, 4, na_rcy, 1) = 1; // O5*--P
		bonded_neighbor(2, 4, na_rcy, 1) = 5; // O5*--C5*
	nbonded_neighbors( 5, na_rcy, 1) = 4; // C5*
		bonded_neighbor(1, 5, na_rcy, 1) = 4; // C5*--O5*
		bonded_neighbor(2, 5, na_rcy, 1) = 6; // C5*--C4*
		bonded_neighbor(3, 5, na_rcy, 1) =21; // C5*--1H5*
		bonded_neighbor(4, 5, na_rcy, 1) =22; // C5*--2H5*
	nbonded_neighbors( 6, na_rcy, 1) = 4; // C4*
		bonded_neighbor(1, 6, na_rcy, 1) = 5; // C4*--C5*
		bonded_neighbor(2, 6, na_rcy, 1) = 7; // C4*--O4*
		bonded_neighbor(3, 6, na_rcy, 1) = 8; // C4*--C3*
		bonded_neighbor(4, 6, na_rcy, 1) =23; // C4*--H4*
	nbonded_neighbors( 7, na_rcy, 1) = 2; // O4*
		bonded_neighbor(1, 7, na_rcy, 1) = 6; // O4*--C4*
		bonded_neighbor(2, 7, na_rcy, 1) =12; // O4*--C1*
	nbonded_neighbors( 8, na_rcy, 1) = 4; // C3*
		bonded_neighbor(1, 8, na_rcy, 1) = 6; // C3*--C4*
		bonded_neighbor(2, 8, na_rcy, 1) = 9; // C3*--O3*
		bonded_neighbor(3, 8, na_rcy, 1) =10; // C3*--C2*
		bonded_neighbor(4, 8, na_rcy, 1) =24; // C3*--H3*
	nbonded_neighbors( 9, na_rcy, 1) = 1; // O3*
		bonded_neighbor(1, 9, na_rcy, 1) = 8; // O3*--C3*
	nbonded_neighbors( 10, na_rcy, 1) = 4; // C2*
		bonded_neighbor(1,10, na_rcy, 1) = 8; // C2*--C3*
		bonded_neighbor(2,10, na_rcy, 1) =11; // C2*--O2*
		bonded_neighbor(3,10, na_rcy, 1) =12; // C2*--C1*
		bonded_neighbor(4,10, na_rcy, 1) =25; // C2*--1H2*
	nbonded_neighbors( 11, na_rcy, 1) = 2; // O2*
		bonded_neighbor(1,11, na_rcy, 1) =10; // O2*--C2*
		bonded_neighbor(2,11, na_rcy, 1) =26; // O2*--2HO*
	nbonded_neighbors( 12, na_rcy, 1) = 4; // C1*
		bonded_neighbor(1,12, na_rcy, 1) = 7; // C1*--O4*
		bonded_neighbor(2,12, na_rcy, 1) =10; // C1*--C2*
		bonded_neighbor(3,12, na_rcy, 1) =13; // C1*--N1
		bonded_neighbor(4,12, na_rcy, 1) =27; // C1*--H1*
	nbonded_neighbors( 13, na_rcy, 1) = 3; // N1
		bonded_neighbor(1,13, na_rcy, 1) =12; // N1--C1*
		bonded_neighbor(2,13, na_rcy, 1) =14; // N1--C2
		bonded_neighbor(3,13, na_rcy, 1) =20; // N1--C6
	nbonded_neighbors( 14, na_rcy, 1) = 3; // C2
		bonded_neighbor(1,14, na_rcy, 1) =13; // C2--N1
		bonded_neighbor(2,14, na_rcy, 1) =15; // C2--O2
		bonded_neighbor(3,14, na_rcy, 1) =16; // C2--N3
	nbonded_neighbors( 15, na_rcy, 1) = 1; // O2
		bonded_neighbor(1,15, na_rcy, 1) =14; // O2--C2
	nbonded_neighbors( 16, na_rcy, 1) = 2; // N3
		bonded_neighbor(1,16, na_rcy, 1) =14; // N3--C2
		bonded_neighbor(2,16, na_rcy, 1) =17; // N3--C4
	nbonded_neighbors( 17, na_rcy, 1) = 3; // C4
		bonded_neighbor(1,17, na_rcy, 1) =16; // C4--N3
		bonded_neighbor(2,17, na_rcy, 1) =18; // C4--N4
		bonded_neighbor(3,17, na_rcy, 1) =19; // C4--C5
	nbonded_neighbors( 18, na_rcy, 1) = 3; // N4
		bonded_neighbor(1,18, na_rcy, 1) =17; // N4--C4
		bonded_neighbor(2,18, na_rcy, 1) =28; // N4--1H4
		bonded_neighbor(3,18, na_rcy, 1) =29; // N4--2H4
	nbonded_neighbors( 19, na_rcy, 1) = 3; // C5
		bonded_neighbor(1,19, na_rcy, 1) =17; // C5--C4
		bonded_neighbor(2,19, na_rcy, 1) =20; // C5--C6
		bonded_neighbor(3,19, na_rcy, 1) =30; // C5--H5
	nbonded_neighbors( 20, na_rcy, 1) = 3; // C6
		bonded_neighbor(1,20, na_rcy, 1) =13; // C6--N1
		bonded_neighbor(2,20, na_rcy, 1) =19; // C6--C5
		bonded_neighbor(3,20, na_rcy, 1) =31; // C6--H6
	nbonded_neighbors( 21, na_rcy, 1) = 1; //1H5*
		bonded_neighbor(1,21, na_rcy, 1) = 5; //1H5*--C5*
	nbonded_neighbors( 22, na_rcy, 1) = 1; //2H5*
		bonded_neighbor(1,22, na_rcy, 1) = 5; //2H5*--C5*
	nbonded_neighbors( 23, na_rcy, 1) = 1; // H4*
		bonded_neighbor(1,23, na_rcy, 1) = 6; // H4*--C4*
	nbonded_neighbors( 24, na_rcy, 1) = 1; // H3*
		bonded_neighbor(1,24, na_rcy, 1) = 8; // H3*--C3*
	nbonded_neighbors( 25, na_rcy, 1) = 1; //1H2*
		bonded_neighbor(1,25, na_rcy, 1) =10; //1H2*--C2*
	nbonded_neighbors( 26, na_rcy, 1) = 1; //2HO*
		bonded_neighbor(1,26, na_rcy, 1) =11; //2HO*--O2*
	nbonded_neighbors( 27, na_rcy, 1) = 1; // H1*
		bonded_neighbor(1,27, na_rcy, 1) =12; // H1*--C1*
	nbonded_neighbors( 28, na_rcy, 1) = 1; //1H4
		bonded_neighbor(1,28, na_rcy, 1) =18; //1H4--N4
	nbonded_neighbors( 29, na_rcy, 1) = 1; //2H4
		bonded_neighbor(1,29, na_rcy, 1) =18; //2H4--N4
	nbonded_neighbors( 30, na_rcy, 1) = 1; // H5
		bonded_neighbor(1,30, na_rcy, 1) =19; // H5--C5
	nbonded_neighbors( 31, na_rcy, 1) = 1; // H6
		bonded_neighbor(1,31, na_rcy, 1) =20; // H6--C6

//jjh get the angles automatically
	get_angles_from_bonds(na_rcy, 1);

	atom_base( 1,na_rcy,1) =  2; //   P   --> O1P
	atom_base( 2,na_rcy,1) =  1; //   O1P --> P
	atom_base( 3,na_rcy,1) =  1; //   O2P --> P
	atom_base( 4,na_rcy,1) =  1; //   O5* --> P
	abase2( 4, na_rcy, 1) =  5; //  C5* base2 for  O5*
	atom_base( 5,na_rcy,1) =  4; //   C5* --> O5*
	atom_base( 6,na_rcy,1) =  5; //   C4* --> C5*
	atom_base( 7,na_rcy,1) = 12; //   O4* --> C1* //abase=C4*
	abase2( 7, na_rcy, 1) =  6; //  C4* base2 for  O4*
	atom_base( 8,na_rcy,1) =  6; //   C3* --> C4*
	atom_base( 9,na_rcy,1) =  8; //   O3* --> C3*
	atom_base(10,na_rcy,1) =  8; //   C2* --> C3*
	atom_base(11,na_rcy,1) = 10; //   O2* --> C2*
	abase2(11, na_rcy, 1) = 26; // 2HO* base2 for  O2*
	atom_base(12,na_rcy,1) = 10; //   C1* --> C2*
	atom_base(13,na_rcy,1) = 12; //   N1  --> C1*
	atom_base(14,na_rcy,1) = 13; //   C2  --> N1
	atom_base(15,na_rcy,1) = 14; //   O2  --> C2
	abase2(16, na_rcy, 1) = 17; //  C4  base2 for  N3
	atom_base(16,na_rcy,1) = 14; //   N3  --> C2
	atom_base(17,na_rcy,1) = 19; //   C4  --> C5
	atom_base(18,na_rcy,1) = 17; //   N4  --> C4
	atom_base(19,na_rcy,1) = 20; //   C5  --> C6
	atom_base(20,na_rcy,1) = 13; //   C6  --> N1
	atom_base(21,na_rcy,1) =  5; //  1H5* --> C5*
	atom_base(22,na_rcy,1) =  5; //  2H5* --> C5*
	atom_base(23,na_rcy,1) =  6; //   H4* --> C4*
	atom_base(24,na_rcy,1) =  8; //   H3* --> C3*
	atom_base(25,na_rcy,1) = 10; //  1H2* --> C2*
	atom_base(26,na_rcy,1) = 11; //  2HO* --> O2*
	atom_base(27,na_rcy,1) = 12; //   H1* --> C1*
	atom_base(28,na_rcy,1) = 18; //  1H4  --> N4
	atom_base(29,na_rcy,1) = 18; //  2H4  --> N4
	atom_base(30,na_rcy,1) = 19; //   H5  --> C5
	atom_base(31,na_rcy,1) = 20; //   H6  --> C6

//vats   atom type numbers rCYT
//vats New atom types for Nucleic Acids (Vatsan Raman)
//vats Ref etable.h for details
//cmd  Reverted to rosetta atomtypes
	fullatom_type( 1,na_rcy,1) = 21; // Phos   P     P
	fullatom_type( 2,na_rcy,1) = 15; // 69; // OOC    O1P   ON3
	fullatom_type( 3,na_rcy,1) = 15; // 69; // OOC    O2P   ON3
	fullatom_type( 4,na_rcy,1) = 14; // 68; // ONH2   O5*   ON2
	fullatom_type( 5,na_rcy,1) =  4; // 61; // CH2    C5*   CN8 (CN8B)
	fullatom_type( 6,na_rcy,1) =  3; // 60; // CH1    C4*   CN7
	fullatom_type( 7,na_rcy,1) = 13; // 72; // OH     O4*   ON6 (ON6B)
	fullatom_type( 8,na_rcy,1) =  3; // 60; // CH1    C3*   CN7
	fullatom_type( 9,na_rcy,1) = 14; // 68; // ONH2   O3*   ON2
	fullatom_type(10,na_rcy,1) =  4; // 60; // CH2    C2*   CN7 (CN7B)
	fullatom_type(11,na_rcy,1) = 13; // 71; // OH     O2*   ON5
	fullatom_type(12,na_rcy,1) =  3; // 60; // CH1    C1*   CN7 (CN7B)
	fullatom_type(13,na_rcy,1) =  8; // 64; // Nhis   N1    NN2
	fullatom_type(14,na_rcy,1) = 19; // 55; // CObb   C2    CN1
	fullatom_type(15,na_rcy,1) = 20; // 67; // OCbb   O2    ON1 (ON1C)
	fullatom_type(16,na_rcy,1) =  8; // 65; // Nhis   N3    NN3
	fullatom_type(17,na_rcy,1) =  6; // 56; // aroC   C4    CN2
	fullatom_type(18,na_rcy,1) =  9; // 63; // NH2O   N4    NN1
	fullatom_type(19,na_rcy,1) =  6; // 57; // aroC   C5    CN3
	fullatom_type(20,na_rcy,1) =  6; // 57; // aroC   C6    CN3
	fullatom_type(21,na_rcy,1) = 23; // 74; // Hapo  1H5*   HNP
	fullatom_type(22,na_rcy,1) = 23; // 74; // Hapo  2H5*   HNP
	fullatom_type(23,na_rcy,1) = 23; // 74; // Hapo   H4*   HNP
	fullatom_type(24,na_rcy,1) = 23; // 74; // Hapo   H3*   HNP
	fullatom_type(25,na_rcy,1) = 23; // 74; // Hapo  1H2*   HNP
	fullatom_type(26,na_rcy,1) = 22; // 73; // Hpol  2HO*   HP
	fullatom_type(27,na_rcy,1) = 23; // 74; // Hapo   H1*   HNP
	fullatom_type(28,na_rcy,1) = 22; // 73; // Hpol  1H4    HP
	fullatom_type(29,na_rcy,1) = 22; // 73; // Hpol  2H4    HP
	fullatom_type(30,na_rcy,1) = 24; // 74; // Haro   H5    HNP
	fullatom_type(31,na_rcy,1) = 24; // 74; // Haro   H6    HNP

//bk   atom partial charges rCYT
	atomic_charge( 1, na_rcy,1) = 1.500; // P
	atomic_charge( 2, na_rcy,1) = -.780; // O1P
	atomic_charge( 3, na_rcy,1) = -.780; // O2P
	atomic_charge( 4, na_rcy,1) = -.570; // O5*
	atomic_charge( 5, na_rcy,1) = -.080; // C5*
	atomic_charge( 6, na_rcy,1) =  .160; // C4*
	atomic_charge( 7, na_rcy,1) = -.500; // O4*
	atomic_charge( 8, na_rcy,1) =  .010; // C3*
	atomic_charge( 9, na_rcy,1) = -.570; // O3*
	atomic_charge(10, na_rcy,1) =  .140; // C2*
	atomic_charge(11, na_rcy,1) = -.660; // O2*
	atomic_charge(12, na_rcy,1) =  .160; // C1*
	atomic_charge(13, na_rcy,1) = -.130; // N1
	atomic_charge(14, na_rcy,1) =  .520; // C2
	atomic_charge(15, na_rcy,1) = -.490; // O2
	atomic_charge(16, na_rcy,1) = -.660; // N3
	atomic_charge(17, na_rcy,1) =  .650; // C4
	atomic_charge(18, na_rcy,1) = -.750; // N4
	atomic_charge(19, na_rcy,1) = -.130; // C5
	atomic_charge(20, na_rcy,1) =  .050; // C6
	atomic_charge(21, na_rcy,1) =  .090; // 1H5*
	atomic_charge(22, na_rcy,1) =  .090; // 2H5*
	atomic_charge(23, na_rcy,1) =  .090; // H4*
	atomic_charge(24, na_rcy,1) =  .090; // H3*
	atomic_charge(25, na_rcy,1) =  .090; // 1H2*
	atomic_charge(26, na_rcy,1) =  .430; // 2HO*
	atomic_charge(27, na_rcy,1) =  .090; // H1*
	atomic_charge(28, na_rcy,1) =  .370; // 1H4
	atomic_charge(29, na_rcy,1) =  .330; // 2H4
	atomic_charge(30, na_rcy,1) =  .070; // H5
	atomic_charge(31, na_rcy,1) =  .170; // H6

	assert_total_charge( na_rcy, 1, -1.0);

//cmd  first sidechain atom
	first_scatom( na_rcy,1) = 13; //N1

//jjh anchor atoms for base mutations
	na_anchor(1, na_rcy, 1) = 12;
	na_anchor(2, na_rcy, 1) = 13;
	na_anchor(3, na_rcy, 1) = 14;

//bk   template atoms used for placing hydrogens

//bk   template for building  P
	ta(1, 1, na_rcy,1) =  4; //   O5*
	ta(2, 1, na_rcy,1) =  5; //   C5*
	ta(3, 1, na_rcy,1) =  6; //   C4*

//bk   template for building  O1P
	ta(1, 2, na_rcy,1) =  1; //   P
	ta(2, 2, na_rcy,1) =  5; //   C5*
	ta(3, 2, na_rcy,1) =  4; //   O5*

//bk   template for building  O2P
	ta(1, 3, na_rcy,1) =  1; //   P
	ta(2, 3, na_rcy,1) =  2; //   O1P
	ta(3, 3, na_rcy,1) =  4; //   O5*

//bk   template for building  O2*
	ta(1, 11, na_rcy,1) =  10; //   C2*
	ta(2, 11, na_rcy,1) =   8; //   C3*
	ta(3, 11, na_rcy,1) =  12; //   C1*

//bk   template for building 1H5*
	ta(1,21, na_rcy,1) =  5; //   C5*
	ta(2,21, na_rcy,1) =  4; //   O5*
	ta(3,21, na_rcy,1) =  6; //   C4*

//bk   template for building 2H5*
	ta(1,22, na_rcy,1) =  5; //   C5*
	ta(2,22, na_rcy,1) =  4; //   O5*
	ta(3,22, na_rcy,1) =  6; //   C4*

//bk   template for building  H4*
	ta(1,23, na_rcy,1) =  6; //   C4*
	ta(2,23, na_rcy,1) =  7; //   O4*
	ta(3,23, na_rcy,1) =  5; //   C5*

//bk   template for building  H3*
	ta(1,24, na_rcy,1) =  8; //   C3*
	ta(2,24, na_rcy,1) = 10; //   C2*
	ta(3,24, na_rcy,1) =  6; //   C4*

//bk   template for building 1H2*
	ta(1,25, na_rcy,1) = 10; //   C2*
	ta(2,25, na_rcy,1) =  8; //   C3*
	ta(3,25, na_rcy,1) = 12; //   C1*

//bk   template for building 2HO*
	ta(1,26, na_rcy,1) = 11; //   O2*
	ta(2,26, na_rcy,1) = 10; //   C2*
	ta(3,26, na_rcy,1) = 12; //   C1*

//bk   template for building  H1*
	ta(1,27, na_rcy,1) = 12; //   C1*
	ta(2,27, na_rcy,1) = 10; //   C2*
	ta(3,27, na_rcy,1) =  7; //   O4*

//bk   template for building 1H4
	ta(1,28, na_rcy,1) = 18; //   N4
	ta(2,28, na_rcy,1) = 17; //   C4
	ta(3,28, na_rcy,1) = 16; //   N3

//bk   template for building 2H4
	ta(1,29, na_rcy,1) = 18; //   N4
	ta(2,29, na_rcy,1) = 17; //   C4
	ta(3,29, na_rcy,1) = 16; //   N3

//bk   template for building  H5
	ta(1,30, na_rcy,1) = 19; //   C5
	ta(2,30, na_rcy,1) = 20; //   C6
	ta(3,30, na_rcy,1) = 13; //   N1

//bk   template for building  H6
	ta(1,31, na_rcy,1) = 20; //   C6
	ta(2,31, na_rcy,1) = 19; //   C5
	ta(3,31, na_rcy,1) = 17; //   C4

//bk   identify atoms that require chi angles to be built
//cy   chi angles needed for building 2HO*
	chi_required(1, 26,na_rcy,1) = true;

//cy   four atoms that define chi angle 1
	chi_atoms( 1, 1,na_rcy, 1) = 12; //   C1*
	chi_atoms( 2, 1,na_rcy, 1) = 10; //   C2*
	chi_atoms( 3, 1,na_rcy, 1) = 11; //   O2*
	chi_atoms( 4, 1,na_rcy, 1) = 26; //   2HO*

//bk   hbonding information
//jjh set the next two to nonsense values
	HNpos( na_rcy,1) = -1; // position of backbone HN
	HApos( na_rcy,1) = -1; // position of backbone HA

	nH_polar( na_rcy,1) =  3; // number of polar hydrogens
	nH_aromatic( na_rcy,1) =  2; // number of aromatic hydrogens
	nH_apolar( na_rcy,1) =  6; // number of apolar hydrogens
	nacceptors( na_rcy,1) =  8; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1, na_rcy,1) = 26; // 2HO*
	Hpos_polar( 2, na_rcy,1) = 28; // 1H4
	Hpos_polar( 3, na_rcy,1) = 29; // 2H4

//bk   atom numbers for aromatic hydrogens
	Hpos_aromatic( 1, na_rcy,1) = 30; //  H5
	Hpos_aromatic( 2, na_rcy,1) = 31; //  H6

//bk   atom numbers for apolar hydrogens
	Hpos_apolar( 1, na_rcy,1) = 21; // 1H5*
	Hpos_apolar( 2, na_rcy,1) = 22; // 2H5*
	Hpos_apolar( 3, na_rcy,1) = 23; //  H4*
	Hpos_apolar( 4, na_rcy,1) = 24; //  H3*
	Hpos_apolar( 5, na_rcy,1) = 25; // 1H2*
	Hpos_apolar( 6, na_rcy,1) = 27; //  H1*

//bk   acceptor positions
	accpt_pos( 1, na_rcy, 1) =  2; //  O1P
	accpt_pos( 2, na_rcy, 1) =  3; //  O2P
	accpt_pos( 3, na_rcy, 1) =  4; //  O5*
	accpt_pos( 4, na_rcy, 1) =  9; //  O3*
	accpt_pos( 5, na_rcy, 1) = 11; //  O2*
	accpt_pos( 6, na_rcy, 1) = 15; //  O2
	accpt_pos( 7, na_rcy, 1) = 16; //  N3
	accpt_pos( 8, na_rcy, 1) =  7; //  O4*

//bk   hydrogens on each atom

	nhydrogens_on_atm( 5, na_rcy, 1) =  2; //  C5*
	hydrogens_on_atm( 1, 5, na_rcy, 1) = 21; // 1H5*
	hydrogens_on_atm( 2, 5, na_rcy, 1) = 22; // 2H5*

	nhydrogens_on_atm( 6, na_rcy, 1) =  1; //  C4*
	hydrogens_on_atm( 1, 6, na_rcy, 1) =  23; //  H4*

	nhydrogens_on_atm( 8, na_rcy, 1) =  1; //  C3*
	hydrogens_on_atm( 1, 8, na_rcy, 1) =  24; //  H3*

	nhydrogens_on_atm(10, na_rcy, 1) =  1; //  C2*
	hydrogens_on_atm( 1,10, na_rcy, 1) =  25; // 1H2*

	nhydrogens_on_atm(11, na_rcy, 1) =  1; //  O2*
	hydrogens_on_atm( 1,11, na_rcy, 1) =  26; // 2HO*

	nhydrogens_on_atm(12, na_rcy, 1) =  1; //  C1*
	hydrogens_on_atm( 1,12, na_rcy, 1) =  27; //  H1*

	nhydrogens_on_atm(18, na_rcy, 1) =  2; //  N4
	hydrogens_on_atm( 1,18, na_rcy, 1) =  28; // 1H4
	hydrogens_on_atm( 2,18, na_rcy, 1) =  29; // 2H4

	nhydrogens_on_atm(19, na_rcy, 1) =  1; //  C5
	hydrogens_on_atm( 1,19, na_rcy, 1) =  30; //  H5

	nhydrogens_on_atm(20, na_rcy, 1) =  1; //  C6
	hydrogens_on_atm( 1,20, na_rcy, 1) = 31; //  H6

//bk   template coordinates for the nucleic acid
//rhiju Fixed icoor, which had base poking out of
//rhiju sugar wrong. The takeoff angles from N1 read from 2INA.pdb,
//rhiju which uses ideal values used by  xplor and the NDB.
//rhiju Nov. 10, 2006.
   icoor( 1, 1, na_rcy, 1) =    4.691; //  P
   icoor( 2, 1, na_rcy, 1) =    0.327; //  P
   icoor( 3, 1, na_rcy, 1) =   -2.444; //  P
   icoor( 1, 2, na_rcy, 1) =    5.034; //  O1P
   icoor( 2, 2, na_rcy, 1) =    1.678; //  O1P
   icoor( 3, 2, na_rcy, 1) =   -1.932; //  O1P
   icoor( 1, 3, na_rcy, 1) =    4.718; //  O2P
   icoor( 2, 3, na_rcy, 1) =    0.068; //  O2P
   icoor( 3, 3, na_rcy, 1) =   -3.906; //  O2P
   icoor( 1, 4, na_rcy, 1) =    3.246; //  O5*
   icoor( 2, 4, na_rcy, 1) =   -0.057; //  O5*
   icoor( 3, 4, na_rcy, 1) =   -1.895; //  O5*
   icoor( 1, 5, na_rcy, 1) =    2.957; //  C5*
   icoor( 2, 5, na_rcy, 1) =   -1.393; //  C5*
   icoor( 3, 5, na_rcy, 1) =   -1.443; //  C5*
   icoor( 1, 6, na_rcy, 1) =    1.509; //  C4*
   icoor( 2, 6, na_rcy, 1) =   -1.478; //  C4*
   icoor( 3, 6, na_rcy, 1) =   -1.022; //  C4*
   icoor( 1, 7, na_rcy, 1) =    1.286; //  O4*
   icoor( 2, 7, na_rcy, 1) =   -0.587; //  O4*
   icoor( 3, 7, na_rcy, 1) =    0.103; //  O4*
   icoor( 1, 8, na_rcy, 1) =    0.499; //  C3*
   icoor( 2, 8, na_rcy, 1) =   -1.031; //  C3*
   icoor( 3, 8, na_rcy, 1) =   -2.067; //  C3*
   icoor( 1, 9, na_rcy, 1) =    0.178; //  O3*
   icoor( 2, 9, na_rcy, 1) =   -2.084; //  O3*
   icoor( 3, 9, na_rcy, 1) =   -2.981; //  O3*
   icoor( 1,10, na_rcy, 1) =   -0.694; //  C2*
   icoor( 2,10, na_rcy, 1) =   -0.627; //  C2*
   icoor( 3,10, na_rcy, 1) =   -1.210; //  C2*
   icoor( 1,11, na_rcy, 1) =   -1.474; //  O2*
   icoor( 2,11, na_rcy, 1) =   -1.731; //  O2*
   icoor( 3,11, na_rcy, 1) =   -0.795; //  O2*
   icoor( 1,12, na_rcy, 1) =    0.000; //  C1*
   icoor( 2,12, na_rcy, 1) =    0.000; //  C1*
   icoor( 3,12, na_rcy, 1) =    0.000; //  C1*
   icoor( 1,13, na_rcy, 1) =    0.181; //  N1
   icoor( 2,13, na_rcy, 1) =    1.461; //  N1
   icoor( 3,13, na_rcy, 1) =   -0.075; //  N1
   icoor( 1,14, na_rcy, 1) =   -0.773; //  C2
   icoor( 2,14, na_rcy, 1) =    2.295; //  C2
   icoor( 3,14, na_rcy, 1) =    0.511; //  C2
   icoor( 1,15, na_rcy, 1) =   -1.769; //  O2
   icoor( 2,15, na_rcy, 1) =    1.785; //  O2
   icoor( 3,15, na_rcy, 1) =    1.044; //  O2
   icoor( 1,16, na_rcy, 1) =   -0.587; //  N3
   icoor( 2,16, na_rcy, 1) =    3.636; //  N3
   icoor( 3,16, na_rcy, 1) =    0.481; //  N3
   icoor( 1,17, na_rcy, 1) =    0.499; //  C4
   icoor( 2,17, na_rcy, 1) =    4.145; //  C4
   icoor( 3,17, na_rcy, 1) =   -0.104; //  C4
   icoor( 1,18, na_rcy, 1) =    0.640; //  N4
   icoor( 2,18, na_rcy, 1) =    5.473; //  N4
   icoor( 3,18, na_rcy, 1) =   -0.111; //  N4
   icoor( 1,19, na_rcy, 1) =    1.487; //  C5
   icoor( 2,19, na_rcy, 1) =    3.316; //  C5
   icoor( 3,19, na_rcy, 1) =   -0.709; //  C5
   icoor( 1,20, na_rcy, 1) =    1.291; //  C6
   icoor( 2,20, na_rcy, 1) =    1.993; //  C6
   icoor( 3,20, na_rcy, 1) =   -0.672; //  C6
   icoor( 1,21, na_rcy, 1) =    3.612; // 1H5*
   icoor( 2,21, na_rcy, 1) =   -1.657; // 1H5*
   icoor( 3,21, na_rcy, 1) =   -0.600; // 1H5*
   icoor( 1,22, na_rcy, 1) =    3.160; // 2H5*
   icoor( 2,22, na_rcy, 1) =   -2.116; // 2H5*
   icoor( 3,22, na_rcy, 1) =   -2.247; // 2H5*
   icoor( 1,23, na_rcy, 1) =    1.353; //  H4*
   icoor( 2,23, na_rcy, 1) =   -2.546; //  H4*
   icoor( 3,23, na_rcy, 1) =   -0.810; //  H4*
   icoor( 1,24, na_rcy, 1) =    0.863; //  H3*
   icoor( 2,24, na_rcy, 1) =   -0.220; //  H3*
   icoor( 3,24, na_rcy, 1) =   -2.715; //  H3*
   icoor( 1,25, na_rcy, 1) =   -1.398; // 1H2*
   icoor( 2,25, na_rcy, 1) =    0.026; // 1H2*
   icoor( 3,25, na_rcy, 1) =   -1.746; // 1H2*
	 icoor( 1,26, na_rcy, 1) =   -2.239; // 2HO*
	 icoor( 2,26, na_rcy, 1) =   -1.409; // 2HO*
	 icoor( 3,26, na_rcy, 1) =   -0.237; // 2HO*
   icoor( 1,27, na_rcy, 1) =   -0.650; //  H1*
   icoor( 2,27, na_rcy, 1) =   -0.188; //  H1*
   icoor( 3,27, na_rcy, 1) =    0.867; //  H1*
   icoor( 1,28, na_rcy, 1) =    1.442; // 1H4
   icoor( 2,28, na_rcy, 1) =    5.889; // 1H4
   icoor( 3,28, na_rcy, 1) =   -0.541; // 1H4
   icoor( 1,29, na_rcy, 1) =   -0.055; // 2H4
   icoor( 2,29, na_rcy, 1) =    6.052; // 2H4
   icoor( 3,29, na_rcy, 1) =    0.314; // 2H4
   icoor( 1,30, na_rcy, 1) =    2.377; //  H5
   icoor( 2,30, na_rcy, 1) =    3.749; //  H5
   icoor( 3,30, na_rcy, 1) =   -1.189; //  H5
   icoor( 1,31, na_rcy, 1) =    2.033; //  H6
   icoor( 2,31, na_rcy, 1) =    1.322; //  H6
   icoor( 3,31, na_rcy, 1) =   -1.129; //  H6

//bk   create variants that have h2o connected to the backbone nh and carbonyl
//jjh  currently disabled.
//      if ( explicit_h2o ) {
//         create_backbone_h2o_variants(na_rcy);
//      }

}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_rna_ura
///
/// @brief arrays filled to define the proerties of the nucleic acid rUracil
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// na_ura - index number for rna URA
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// atom_base - base heavy atom defined for each atom for graphics
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// //Hbase - id the base atom for each hydrogen (JSS replaced with atom_base Dec 2006)
/// accpt_pos - id the acceptor positions
/// //abase - id the base atom for each acceptor (JSS replaced with atom_base Dec 2006)
/// abase2 - second base - adj to acceptor(RING/SP3) or to abase(SP2)
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_rna_ura()
{
	using namespace aaproperties_pack;
	using namespace param_aa;

//bk   Parameters for rURA -----------------------------
	nvar(na_ura) = 1; // starting starting number of variants for rURA

//bk   variant1 ---------- rURA

	natoms(na_ura,1) = 30; // number of total atoms
	nheavyatoms(na_ura,1) = 20; // number of heavy atoms
	nchi(na_ura,1) = 1; // number of chi angles

//bk   atom names rURA
	atom_name( 1, na_ura,1) = " P  ";
	atom_name( 2, na_ura,1) = " O1P";
	atom_name( 3, na_ura,1) = " O2P";
	atom_name( 4, na_ura,1) = " O5*";
	atom_name( 5, na_ura,1) = " C5*";
	atom_name( 6, na_ura,1) = " C4*";
	atom_name( 7, na_ura,1) = " O4*";
	atom_name( 8, na_ura,1) = " C3*";
	atom_name( 9, na_ura,1) = " O3*";
	atom_name(10, na_ura,1) = " C2*";
	atom_name(11, na_ura,1) = " O2*";
	atom_name(12, na_ura,1) = " C1*";
	atom_name(13, na_ura,1) = " N1 ";
	atom_name(14, na_ura,1) = " C2 ";
	atom_name(15, na_ura,1) = " O2 ";
	atom_name(16, na_ura,1) = " N3 ";
	atom_name(17, na_ura,1) = " C4 ";
	atom_name(18, na_ura,1) = " O4 ";
	atom_name(19, na_ura,1) = " C5 ";
	atom_name(20, na_ura,1) = " C6 ";
	atom_name(21, na_ura,1) = "1H5*";
	atom_name(22, na_ura,1) = "2H5*";
	atom_name(23, na_ura,1) = " H4*";
	atom_name(24, na_ura,1) = " H3*";
	atom_name(25, na_ura,1) = "1H2*";
	atom_name(26, na_ura,1) = "2HO*";
	atom_name(27, na_ura,1) = " H1*";
	atom_name(28, na_ura,1) = " H3 ";
	atom_name(29, na_ura,1) = " H5 ";
	atom_name(30, na_ura,1) = " H6 ";

//jjh intra residue bonding
	nbonded_neighbors( 1, na_ura, 1) = 3; // P
		bonded_neighbor(1, 1, na_ura, 1) = 2; // P--O1P
		bonded_neighbor(2, 1, na_ura, 1) = 3; // P--O2P
		bonded_neighbor(3, 1, na_ura, 1) = 4; // P--O5*
	nbonded_neighbors( 2, na_ura, 1) = 1; // O1P
		bonded_neighbor(1, 2, na_ura, 1) = 1; // O1P--P
	nbonded_neighbors( 3, na_ura, 1) = 1; // O2P
		bonded_neighbor(1, 3, na_ura, 1) = 1; // O2P--P
	nbonded_neighbors( 4, na_ura, 1) = 2; // O5*
		bonded_neighbor(1, 4, na_ura, 1) = 1; // O5*--P
		bonded_neighbor(2, 4, na_ura, 1) = 5; // O5*--C5*
	nbonded_neighbors( 5, na_ura, 1) = 4; // C5*
		bonded_neighbor(1, 5, na_ura, 1) = 4; // C5*--O5*
		bonded_neighbor(2, 5, na_ura, 1) = 6; // C5*--C4*
		bonded_neighbor(3, 5, na_ura, 1) =21; // C5*--1H5*
		bonded_neighbor(4, 5, na_ura, 1) =22; // C5*--2H5*
	nbonded_neighbors( 6, na_ura, 1) = 4; // C4*
		bonded_neighbor(1, 6, na_ura, 1) = 5; // C4*--C5*
		bonded_neighbor(2, 6, na_ura, 1) = 7; // C4*--O4*
		bonded_neighbor(3, 6, na_ura, 1) = 8; // C4*--C3*
		bonded_neighbor(4, 6, na_ura, 1) =23; // C4*--H4*
	nbonded_neighbors( 7, na_ura, 1) = 2; // O4*
		bonded_neighbor(1, 7, na_ura, 1) = 6; // O4*--C4*
		bonded_neighbor(2, 7, na_ura, 1) =12; // O4*--C1*
	nbonded_neighbors( 8, na_ura, 1) = 4; // C3*
		bonded_neighbor(1, 8, na_ura, 1) = 6; // C3*--C4*
		bonded_neighbor(2, 8, na_ura, 1) = 9; // C3*--O3*
		bonded_neighbor(3, 8, na_ura, 1) =10; // C3*--C2*
		bonded_neighbor(4, 8, na_ura, 1) =24; // C3*--H3*
	nbonded_neighbors( 9, na_ura, 1) = 1; // O3*
		bonded_neighbor(1, 9, na_ura, 1) = 8; // O3*--C3*
	nbonded_neighbors( 10, na_ura, 1) = 4; // C2*
		bonded_neighbor(1,10, na_ura, 1) = 8; // C2*--C3*
		bonded_neighbor(2,10, na_ura, 1) =11; // C2*--O2*
		bonded_neighbor(3,10, na_ura, 1) =12; // C2*--C1*
		bonded_neighbor(4,10, na_ura, 1) =25; // C2*--1H2*
	nbonded_neighbors( 11, na_ura, 1) = 2; // O2*
		bonded_neighbor(1,11, na_ura, 1) =10; // O2*--C2*
		bonded_neighbor(2,11, na_ura, 1) =26; // O2*--2HO*
	nbonded_neighbors( 12, na_ura, 1) = 4; // C1*
		bonded_neighbor(1,12, na_ura, 1) = 7; // C1*--O4*
		bonded_neighbor(2,12, na_ura, 1) =10; // C1*--C2*
		bonded_neighbor(3,12, na_ura, 1) =13; // C1*--N1
		bonded_neighbor(4,12, na_ura, 1) =27; // C1*--H1*
	nbonded_neighbors( 13, na_ura, 1) = 3; // N1
		bonded_neighbor(1,13, na_ura, 1) =12; // N1--C1*
		bonded_neighbor(2,13, na_ura, 1) =14; // N1--C2
		bonded_neighbor(3,13, na_ura, 1) =20; // N1--C6
	nbonded_neighbors( 14, na_ura, 1) = 3; // C2
		bonded_neighbor(1,14, na_ura, 1) =13; // C2--N1
		bonded_neighbor(2,14, na_ura, 1) =15; // C2--O2
		bonded_neighbor(3,14, na_ura, 1) =16; // C2--N3
	nbonded_neighbors( 15, na_ura, 1) = 1; // O2
		bonded_neighbor(1,15, na_ura, 1) =14; // O2--C2
	nbonded_neighbors( 16, na_ura, 1) = 3; // N3
		bonded_neighbor(1,16, na_ura, 1) =14; // N3--C2
		bonded_neighbor(2,16, na_ura, 1) =17; // N3--C4
		bonded_neighbor(3,16, na_ura, 1) =28; // N3--H3
	nbonded_neighbors( 17, na_ura, 1) = 3; // C4
		bonded_neighbor(1,17, na_ura, 1) =16; // C4--N3
		bonded_neighbor(2,17, na_ura, 1) =18; // C4--O4
		bonded_neighbor(3,17, na_ura, 1) =19; // C4--C5
	nbonded_neighbors( 18, na_ura, 1) = 1; // O4
		bonded_neighbor(1,18, na_ura, 1) =17; // O4--C4
	nbonded_neighbors( 19, na_ura, 1) = 3; // C5
		bonded_neighbor(1,19, na_ura, 1) =17; // C5--C4
		bonded_neighbor(2,19, na_ura, 1) =20; // C5--C6
		bonded_neighbor(3,19, na_ura, 1) =29; // C5--H5
	nbonded_neighbors( 20, na_ura, 1) = 3; // C6
		bonded_neighbor(1,20, na_ura, 1) =13; // C6--N1
		bonded_neighbor(2,20, na_ura, 1) =19; // C6--C5
		bonded_neighbor(3,20, na_ura, 1) =30; // C6--H6
	nbonded_neighbors( 21, na_ura, 1) = 1; //1H5*
		bonded_neighbor(1,21, na_ura, 1) = 5; //1H5*--C5*
	nbonded_neighbors( 22, na_ura, 1) = 1; //2H5*
		bonded_neighbor(1,22, na_ura, 1) = 5; //2H5*--C5*
	nbonded_neighbors( 23, na_ura, 1) = 1; // H4*
		bonded_neighbor(1,23, na_ura, 1) = 6; // H4*--C4*
	nbonded_neighbors( 24, na_ura, 1) = 1; // H3*
		bonded_neighbor(1,24, na_ura, 1) = 8; // H3*--C3*
	nbonded_neighbors( 25, na_ura, 1) = 1; //1H2*
		bonded_neighbor(1,25, na_ura, 1) =10; //1H2*--C2*
	nbonded_neighbors( 26, na_ura, 1) = 1; //2HO*
		bonded_neighbor(1,26, na_ura, 1) =11; //2HO*--O2*
	nbonded_neighbors( 27, na_ura, 1) = 1; // H1*
		bonded_neighbor(1,27, na_ura, 1) =12; // H1*--C1*
	nbonded_neighbors( 28, na_ura, 1) = 1; // H3
		bonded_neighbor(1,28, na_ura, 1) =16; // H3--N3
	nbonded_neighbors( 29, na_ura, 1) = 1; // H5
		bonded_neighbor(1,29, na_ura, 1) =19; // H5--C5
	nbonded_neighbors( 30, na_ura, 1) = 1; // H6
		bonded_neighbor(1,30, na_ura, 1) =20; // H6--C6

//jjh get the angles automatically
	get_angles_from_bonds(na_ura, 1);

	atom_base( 1,na_ura,1) =  2; //   P   --> O1P
	atom_base( 2,na_ura,1) =  1; //   O1P --> P
	atom_base( 3,na_ura,1) =  1; //   O2P --> P
	atom_base( 4,na_ura,1) =  1; //   O5* --> P
	abase2( 4, na_ura, 1) =  5; //  C5* base2 for  O5*
	atom_base( 5,na_ura,1) =  4; //   C5* --> O5*
	atom_base( 6,na_ura,1) =  5; //   C4* --> C5*
	atom_base( 7,na_ura,1) = 12; //   O4* --> C1* //abase=C4*
	abase2( 7, na_ura, 1) =  6; //  C4* base2 for  O4*
	atom_base( 8,na_ura,1) =  6; //   C3* --> C4*
	atom_base( 9,na_ura,1) =  8; //   O3* --> C3*
	atom_base(10,na_ura,1) =  8; //   C2* --> C3*
	atom_base(11,na_ura,1) = 10; //   O2* --> C2*
	abase2(11, na_ura, 1) = 26; // 2HO* base2 for  O2*
	atom_base(12,na_ura,1) = 10; //   C1* --> C2*
	atom_base(13,na_ura,1) = 12; //   N1  --> C1*
	atom_base(14,na_ura,1) = 13; //   C2  --> N1
	atom_base(15,na_ura,1) = 14; //   O2  --> C2
	atom_base(16,na_ura,1) = 14; //   N3  --> C2
	atom_base(17,na_ura,1) = 19; //   C4  --> C5
	atom_base(18,na_ura,1) = 17; //   O4  --> C4
	atom_base(19,na_ura,1) = 20; //   C5  --> C6
	atom_base(20,na_ura,1) = 13; //   C6  --> N1
	atom_base(21,na_ura,1) =  5; //  1H5* --> C5*
	atom_base(22,na_ura,1) =  5; //  2H5* --> C5*
	atom_base(23,na_ura,1) =  6; //   H4* --> C4*
	atom_base(24,na_ura,1) =  8; //   H3* --> C3*
	atom_base(25,na_ura,1) = 10; //  1H2* --> C2*
	atom_base(26,na_ura,1) = 11; //  2HO* --> O2*
	atom_base(27,na_ura,1) = 12; //   H1* --> C1*
	atom_base(28,na_ura,1) = 16; //   H3  --> N3
	atom_base(29,na_ura,1) = 19; //   H5  --> C5
	atom_base(30,na_ura,1) = 20; //   H6  --> C6

//vats   atom type numbers rURA
//vats New atom types for Nucleic Acid from CHARMM 27
//vats Ref etable.h for details
	fullatom_type( 1,na_ura,1) = 21; // Phos   P     P
	fullatom_type( 2,na_ura,1) = 15; // 69; // OOC    O1P   ON3
	fullatom_type( 3,na_ura,1) = 15; // 69; // OOC    O2P   ON3
	fullatom_type( 4,na_ura,1) = 14; // 68; // ONH2   O5*   ON2
	fullatom_type( 5,na_ura,1) =  4; // 61; // CH2    C5*   CN8 (CN8B)
	fullatom_type( 6,na_ura,1) =  3; // 60; // CH1    C4*   CN7
	fullatom_type( 7,na_ura,1) = 13; // 72; // OH     O4*   ON6 (ON6B)
	fullatom_type( 8,na_ura,1) =  3; // 60; // CH1    C3*   CN7
	fullatom_type( 9,na_ura,1) = 14; // 68; // ONH2   O3*   ON2
	fullatom_type(10,na_ura,1) =  4; // 60; // CH2    C2*   CN7 (CN7B)
	fullatom_type(11,na_ura,1) = 13; // 67; // OH     O2*   ON1
	fullatom_type(12,na_ura,1) =  3; // 60; // CH1    C1*   CN7 (CN7B)
	fullatom_type(13,na_ura,1) =  8; // 64; // Nhis   N1    NN2 (NN2B)
	fullatom_type(14,na_ura,1) = 19; // 55; // CObb   C2    CN1 (CN1T)
	fullatom_type(15,na_ura,1) = 20; // 67; // OCbb   O2    ON1
	fullatom_type(16,na_ura,1) =  7; // 64; // Ntrp   N3    NN2 (NN2U)
	fullatom_type(17,na_ura,1) = 19; // 55; // CObb   C4    CN1
	fullatom_type(18,na_ura,1) = 20; // 67; // OCbb   O4    ON1
	fullatom_type(19,na_ura,1) =  6; // 57; // aroC   C5    CN3
	fullatom_type(20,na_ura,1) =  6; // 57; // aroC   C6    CN3
	fullatom_type(21,na_ura,1) = 23; // 74; // Hapo  1H5*   HNP
	fullatom_type(22,na_ura,1) = 23; // 74; // Hapo  2H5*   HNP
	fullatom_type(23,na_ura,1) = 23; // 74; // Hapo   H4*   HNP
	fullatom_type(24,na_ura,1) = 23; // 74; // Hapo   H3*   HNP
	fullatom_type(25,na_ura,1) = 23; // 74; // Hapo  1H2*   HNP
	fullatom_type(26,na_ura,1) = 22; // 73; // Hpol  2HO*   HP
	fullatom_type(27,na_ura,1) = 23; // 74; // Hapo   H1*   HNP
	fullatom_type(28,na_ura,1) = 22; // 73; // Hpol   H3    HP
	fullatom_type(29,na_ura,1) = 24; // 74; // Haro   H5    HNP
	fullatom_type(30,na_ura,1) = 24; // 74; // Haro   H6    HNP

//bk   atom partial charges rURA
	atomic_charge( 1, na_ura,1) = 1.500; // P
	atomic_charge( 2, na_ura,1) = -.780; // O1P
	atomic_charge( 3, na_ura,1) = -.780; // O2P
	atomic_charge( 4, na_ura,1) = -.570; // O5*
	atomic_charge( 5, na_ura,1) = -.080; // C5*
	atomic_charge( 6, na_ura,1) =  .160; // C4*
	atomic_charge( 7, na_ura,1) = -.500; // O4*
	atomic_charge( 8, na_ura,1) =  .010; // C3*
	atomic_charge( 9, na_ura,1) = -.570; // O3*
	atomic_charge(10, na_ura,1) =  .140; // C2*
	atomic_charge(11, na_ura,1) = -.660; // O2*
	atomic_charge(12, na_ura,1) =  .160; // C1*
	atomic_charge(13, na_ura,1) = -.340; // N1
	atomic_charge(14, na_ura,1) =  .550; // C2
	atomic_charge(15, na_ura,1) = -.450; // O2
	atomic_charge(16, na_ura,1) = -.460; // N3
	atomic_charge(17, na_ura,1) =  .530; // C4
	atomic_charge(18, na_ura,1) = -.480; // O4
	atomic_charge(19, na_ura,1) = -.150; // C5
	atomic_charge(20, na_ura,1) =  .200; // C6
	atomic_charge(21, na_ura,1) =  .090; // 1H5*
	atomic_charge(22, na_ura,1) =  .090; // 2H5*
	atomic_charge(23, na_ura,1) =  .090; // H4*
	atomic_charge(24, na_ura,1) =  .090; // H3*
	atomic_charge(25, na_ura,1) =  .090; // 1H2*
	atomic_charge(26, na_ura,1) =  .430; // 2HO*
	atomic_charge(27, na_ura,1) =  .090; // H1*
	atomic_charge(28, na_ura,1) =  .360; // H3
	atomic_charge(29, na_ura,1) =  .100; // H5
	atomic_charge(30, na_ura,1) =  .140; // H6

	assert_total_charge( na_ura, 1, -1.0);

//cmd  first sidechain atom
	first_scatom( na_ura,1) = 13; //N1

//jjh anchor atoms for base mutations
	na_anchor(1, na_ura, 1) = 12;
	na_anchor(2, na_ura, 1) = 13;
	na_anchor(3, na_ura, 1) = 14;

//bk   template atoms used for placing hydrogens

//bk   template for building  P
	ta(1, 1, na_ura,1) =  4; //   O5*
	ta(2, 1, na_ura,1) =  5; //   C5*
	ta(3, 1, na_ura,1) =  6; //   C4*

//bk   template for building  O1P
	ta(1, 2, na_ura,1) =  1; //   P
	ta(2, 2, na_ura,1) =  5; //   C5*
	ta(3, 2, na_ura,1) =  4; //   O5*

//bk   template for building  O2P
	ta(1, 3, na_ura,1) =  1; //   P
	ta(2, 3, na_ura,1) =  2; //   O1P
	ta(3, 3, na_ura,1) =  4; //   O5*

//bk   template for building  O2*
	ta(1, 11, na_ura,1) =  10; //   C2*
	ta(2, 11, na_ura,1) =   8; //   C3*
	ta(3, 11, na_ura,1) =  12; //   C1*

//bk   template for building 1H5*
	ta(1,21, na_ura,1) =  5; //   C5*
	ta(2,21, na_ura,1) =  4; //   O5*
	ta(3,21, na_ura,1) =  6; //   C4*

//bk   template for building 2H5*
	ta(1,22, na_ura,1) =  5; //   C5*
	ta(2,22, na_ura,1) =  4; //   O5*
	ta(3,22, na_ura,1) =  6; //   C4*

//bk   template for building  H4*
	ta(1,23, na_ura,1) =  6; //   C4*
	ta(2,23, na_ura,1) =  7; //   O4*
	ta(3,23, na_ura,1) =  5; //   C5*

//bk   template for building  H3*
	ta(1,24, na_ura,1) =  8; //   C3*
	ta(2,24, na_ura,1) = 10; //   C2*
	ta(3,24, na_ura,1) =  6; //   C4*

//bk   template for building 1H2*
	ta(1,25, na_ura,1) = 10; //   C2*
	ta(2,25, na_ura,1) =  8; //   C3*
	ta(3,25, na_ura,1) = 12; //   C1*

//bk   template for building 2HO*
	ta(1,26, na_ura,1) = 11; //   O2*
	ta(2,26, na_ura,1) = 10; //   C2*
	ta(3,26, na_ura,1) = 12; //   C1*

//bk   template for building  H1*
	ta(1,27, na_ura,1) = 12; //   C1*
	ta(2,27, na_ura,1) = 10; //   C2*
	ta(3,27, na_ura,1) =  7; //   O4*

//bk   template for building  H3
	ta(1,28, na_ura,1) = 16; //   N3
	ta(2,28, na_ura,1) = 14; //   C2
	ta(3,28, na_ura,1) = 17; //   C4

//bk   template for building  H5
	ta(1,29, na_ura,1) = 19; //   C5
	ta(2,29, na_ura,1) = 20; //   C6
	ta(3,29, na_ura,1) = 17; //   C4

//bk   template for building  H6
	ta(1,30, na_ura,1) = 20; //   C6
	ta(2,30, na_ura,1) = 19; //   C5
	ta(3,30, na_ura,1) = 17; //   C4

//bk   identify atoms that require chi angles to be built
//cy   chi angles needed for building 2HO*
	chi_required(1, 26,na_ura,1) = true;

//cy   four atoms that define chi angle 1
	chi_atoms( 1, 1,na_ura, 1) = 12; //   C1*
	chi_atoms( 2, 1,na_ura, 1) = 10; //   C2*
	chi_atoms( 3, 1,na_ura, 1) = 11; //   O2*
	chi_atoms( 4, 1,na_ura, 1) = 26; //   2HO*

//bk   hbonding information
//jjh set the next two to nonsense values
	HNpos( na_ura,1) = -1; // position of backbone HN
	HApos( na_ura,1) = -1; // position of backbone HA

	nH_polar( na_ura,1) =  2; // number of polar hydrogens
	nH_aromatic( na_ura,1) =  2; // number of aromatic hydrogens
	nH_apolar( na_ura,1) =  6; // number of apolar hydrogens
	nacceptors( na_ura,1) =  8; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1, na_ura,1) = 26; //  2HO*
	Hpos_polar( 2, na_ura,1) = 28; //  H3

//bk   atom numbers for aromatic hydrogens
	Hpos_aromatic( 1, na_ura,1) = 29; //  H5
	Hpos_aromatic( 2, na_ura,1) = 30; //  H6

//bk   atom numbers for apolar hydrogens
	Hpos_apolar( 1, na_ura,1) = 21; // 1H5*
	Hpos_apolar( 2, na_ura,1) = 22; // 2H5*
	Hpos_apolar( 3, na_ura,1) = 23; //  H4*
	Hpos_apolar( 4, na_ura,1) = 24; //  H3*
	Hpos_apolar( 5, na_ura,1) = 25; // 1H2*
	Hpos_apolar( 6, na_ura,1) = 27; //  H1*

//bk   acceptor positions
	accpt_pos( 1, na_ura, 1) =  2; //  O1P
	accpt_pos( 2, na_ura, 1) =  3; //  O2P
	accpt_pos( 3, na_ura, 1) =  4; //  O5*
	accpt_pos( 4, na_ura, 1) =  9; //  O3*
	accpt_pos( 5, na_ura, 1) = 11; //  O2*
	accpt_pos( 6, na_ura, 1) = 15; //  O2
	accpt_pos( 7, na_ura, 1) = 18; //  O4
	accpt_pos( 8, na_ura, 1) =  7; //  O4*

//bk   hydrogens on each atom

	nhydrogens_on_atm( 5, na_ura, 1) =  2; //  C5*
	hydrogens_on_atm( 1, 5, na_ura, 1) = 21; // 1H5*
	hydrogens_on_atm( 2, 5, na_ura, 1) = 22; // 2H5*

	nhydrogens_on_atm( 6, na_ura, 1) =  1; //  C4*
	hydrogens_on_atm( 1, 6, na_ura, 1) =  23; //  H4*

	nhydrogens_on_atm( 8, na_ura, 1) =  1; //  C3*
	hydrogens_on_atm( 1, 8, na_ura, 1) =  24; //  H3*

	nhydrogens_on_atm(10, na_ura, 1) =  1; //  C2*
	hydrogens_on_atm( 1,10, na_ura, 1) =  25; // 1H2*

	nhydrogens_on_atm(11, na_ura, 1) =  1; //  O2*
	hydrogens_on_atm( 1,11, na_ura, 1) =  26; // 2HO*

	nhydrogens_on_atm(12, na_ura, 1) =  1; //  C1*
	hydrogens_on_atm( 1,12, na_ura, 1) =  27; //  H1*

	nhydrogens_on_atm(16, na_ura, 1) =  1; //  N3
	hydrogens_on_atm( 1,16, na_ura, 1) =  28; //  H3

	nhydrogens_on_atm(19, na_ura, 1) =  1; //  C5
	hydrogens_on_atm( 1,19, na_ura, 1) =  29; //  H5

	nhydrogens_on_atm(20, na_ura, 1) =  1; //  C6
	hydrogens_on_atm( 1,20, na_ura, 1) = 30; //  H6

//bk   template coordinates for the nucleic acid
//rhiju Fixed icoor, which had base poking out of
//rhiju sugar wrong. The takeoff angles from N1 read from 2INA.pdb,
//rhiju which uses ideal values used by  xplor and the NDB.
//rhiju Nov. 10, 2006.
  icoor( 1, 1, na_ura, 1) =    4.691; //  P
  icoor( 2, 1, na_ura, 1) =    0.327; //  P
  icoor( 3, 1, na_ura, 1) =   -2.444; //  P
  icoor( 1, 2, na_ura, 1) =    5.034; //  O1P
  icoor( 2, 2, na_ura, 1) =    1.678; //  O1P
  icoor( 3, 2, na_ura, 1) =   -1.932; //  O1P
  icoor( 1, 3, na_ura, 1) =    4.718; //  O2P
  icoor( 2, 3, na_ura, 1) =    0.068; //  O2P
  icoor( 3, 3, na_ura, 1) =   -3.906; //  O2P
  icoor( 1, 4, na_ura, 1) =    3.246; //  O5*
  icoor( 2, 4, na_ura, 1) =   -0.057; //  O5*
  icoor( 3, 4, na_ura, 1) =   -1.895; //  O5*
  icoor( 1, 5, na_ura, 1) =    2.957; //  C5*
  icoor( 2, 5, na_ura, 1) =   -1.393; //  C5*
  icoor( 3, 5, na_ura, 1) =   -1.443; //  C5*
  icoor( 1, 6, na_ura, 1) =    1.509; //  C4*
  icoor( 2, 6, na_ura, 1) =   -1.478; //  C4*
  icoor( 3, 6, na_ura, 1) =   -1.022; //  C4*
  icoor( 1, 7, na_ura, 1) =    1.286; //  O4*
  icoor( 2, 7, na_ura, 1) =   -0.587; //  O4*
  icoor( 3, 7, na_ura, 1) =    0.103; //  O4*
  icoor( 1, 8, na_ura, 1) =    0.499; //  C3*
  icoor( 2, 8, na_ura, 1) =   -1.031; //  C3*
  icoor( 3, 8, na_ura, 1) =   -2.067; //  C3*
  icoor( 1, 9, na_ura, 1) =    0.178; //  O3*
  icoor( 2, 9, na_ura, 1) =   -2.084; //  O3*
  icoor( 3, 9, na_ura, 1) =   -2.981; //  O3*
  icoor( 1,10, na_ura, 1) =   -0.694; //  C2*
  icoor( 2,10, na_ura, 1) =   -0.627; //  C2*
  icoor( 3,10, na_ura, 1) =   -1.210; //  C2*
  icoor( 1,11, na_ura, 1) =   -1.474; //  O2*
  icoor( 2,11, na_ura, 1) =   -1.731; //  O2*
  icoor( 3,11, na_ura, 1) =   -0.795; //  O2*
  icoor( 1,12, na_ura, 1) =    0.000; //  C1*
  icoor( 2,12, na_ura, 1) =    0.000; //  C1*
  icoor( 3,12, na_ura, 1) =    0.000; //  C1*
  icoor( 1,13, na_ura, 1) =    0.169; //  N1
  icoor( 2,13, na_ura, 1) =    1.459; //  N1
  icoor( 3,13, na_ura, 1) =   -0.086; //  N1
  icoor( 1,14, na_ura, 1) =   -0.815; //  C2
  icoor( 2,14, na_ura, 1) =    2.259; //  C2
  icoor( 3,14, na_ura, 1) =    0.461; //  C2
  icoor( 1,15, na_ura, 1) =   -1.821; //  O2
  icoor( 2,15, na_ura, 1) =    1.807; //  O2
  icoor( 3,15, na_ura, 1) =    0.978; //  O2
  icoor( 1,16, na_ura, 1) =   -0.575; //  N3
  icoor( 2,16, na_ura, 1) =    3.608; //  N3
  icoor( 3,16, na_ura, 1) =    0.379; //  N3
  icoor( 1,17, na_ura, 1) =    0.525; //  C4
  icoor( 2,17, na_ura, 1) =    4.224; //  C4
  icoor( 3,17, na_ura, 1) =   -0.183; //  C4
  icoor( 1,18, na_ura, 1) =    0.598; //  O4
  icoor( 2,18, na_ura, 1) =    5.454; //  O4
  icoor( 3,18, na_ura, 1) =   -0.179; //  O4
  icoor( 1,19, na_ura, 1) =    1.498; //  C5
  icoor( 2,19, na_ura, 1) =    3.328; //  C5
  icoor( 3,19, na_ura, 1) =   -0.727; //  C5
  icoor( 1,20, na_ura, 1) =    1.292; //  C6
  icoor( 2,20, na_ura, 1) =    2.008; //  C6
  icoor( 3,20, na_ura, 1) =   -0.662; //  C6
  icoor( 1,21, na_ura, 1) =    3.612; // 1H5*
  icoor( 2,21, na_ura, 1) =   -1.657; // 1H5*
  icoor( 3,21, na_ura, 1) =   -0.600; // 1H5*
  icoor( 1,22, na_ura, 1) =    3.160; // 2H5*
  icoor( 2,22, na_ura, 1) =   -2.116; // 2H5*
  icoor( 3,22, na_ura, 1) =   -2.247; // 2H5*
  icoor( 1,23, na_ura, 1) =    1.353; //  H4*
  icoor( 2,23, na_ura, 1) =   -2.546; //  H4*
  icoor( 3,23, na_ura, 1) =   -0.810; //  H4*
  icoor( 1,24, na_ura, 1) =    0.863; //  H3*
  icoor( 2,24, na_ura, 1) =   -0.220; //  H3*
  icoor( 3,24, na_ura, 1) =   -2.715; //  H3*
  icoor( 1,25, na_ura, 1) =   -1.398; // 1H2*
  icoor( 2,25, na_ura, 1) =    0.026; // 1H2*
  icoor( 3,25, na_ura, 1) =   -1.746; // 1H2*
  icoor( 1,26, na_ura, 1) =   -2.239; // 2HO*
  icoor( 2,26, na_ura, 1) =   -1.409; // 2HO*
  icoor( 3,26, na_ura, 1) =   -0.237; // 2HO*
  icoor( 1,27, na_ura, 1) =   -0.640; //  H1*
  icoor( 2,27, na_ura, 1) =   -0.189; //  H1*
  icoor( 3,27, na_ura, 1) =    0.875; //  H1*
  icoor( 1,28, na_ura, 1) =   -1.273; //  H3
  icoor( 2,28, na_ura, 1) =    4.210; //  H3
  icoor( 3,28, na_ura, 1) =    0.767; //  H3
  icoor( 1,29, na_ura, 1) =    2.411; //  H5
  icoor( 2,29, na_ura, 1) =    3.722; //  H5
  icoor( 3,29, na_ura, 1) =   -1.198; //  H5
  icoor( 1,30, na_ura, 1) =    2.051; //  H6
  icoor( 2,30, na_ura, 1) =    1.334; //  H6
  icoor( 3,30, na_ura, 1) =   -1.085; //  H6

//bk   create variants that have h2o connected to the backbone nh and carbonyl
//jjh  currently disabled.
//      if ( explicit_h2o ) {
//         create_backbone_h2o_variants(na_ura);
//      }

}

////////////////////////////////////////////////////////////////////////////////
/// @begin zero_aaproperties
///
/// @brief initialize all properties of all amino acids
///
/// @detailed  set all variables to zero for all variants of all amino acids
///
/// @global_read
/// variables from "param.h"
///     MAX_AA          - max number of amino acid
///     MAX_AA_VARIANTS - max number of variants
///     MAX_CHI         - max number of chi angles
///     MAX_ATOM        - max number of atoms per amino acid
/// @global_write
/// variables from "aaproperties_pack.h"
///     most variables in this file will be set to zero, so ...
///     the comments in function "copy_aaproperties".
///
/// @remarks
///
/// @references
///
/// @authors Lin Jiang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
zero_aaproperties()
{
	using namespace aaproperties_pack;
	using namespace param;

	std::string const space4( 4, ' ' );

	// Note: These are whole-array assignments
	natoms = 0;
	nheavyatoms = 0;
	nchi = 0;
	HNpos = 0;
	HApos = 0;
	nH_polar = 0;
	nH_aromatic = 0;
	nH_apolar = 0;
	nacceptors = 0;
	nh2o = 0;
	chi_atoms = 0;
	atom_name = space4;
	fullatom_type = 0;
	atom_type_char = space4;
	Hpos_polar = 0;
	Hpos_apolar = 0;
	Hpos_aromatic = 0;
	h2opos = 0;
	accpt_pos = 0;
  atom_base = 0;
	abase2 = 0;
	nhydrogens_on_atm = 0;
	nside = 0;
	nh2o_on_atm = 0;
	heavy_atm_on_h2o = 0;
	hyd_atm_on_h2o = 0;
	template_atm_on_h2o = 0;
	hydrogens_on_atm = 0;
	ta = 0;
	icoor = 0.0;
	nbonded_neighbors = 0;
	bonded_neighbor = 0;
	nangle_neighbors = 0;
	angle_neighbor = 0;
	h2o_on_atm = 0;
	chi_required = false;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin create_new_variant
///
/// @brief create the new variant
///
/// @detailed  create the new variant,  copy the starting variant to the
///       new variant, and return the number of the new variant.
///
/// @param[in]   aa               - in  -  amino acid
/// @param[in]   starting_variant - in  -  the id of the starting variant
///                                   ready to copy
/// @param[in]    aav_type - in - type of the new variant.
///
/// @global_read
/// variables from "param.h"
///     MAX_AA_VARIANTS   -  max number of variants per aa
/// variables from "aaproperties_pack.h"
///     nvar              -  the current number of variants
/// @global_write
/// variables from "aaproperties_pack.h"
///     nvar              -  the current number of variants is incremented
///
/// @remarks
///
/// @references
///
/// @authors  Lin Jiang  08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
int
create_new_variant(
	int const aa,
	int const starting_variant,
	int const aav_type
)
{
//bk   intitalize arrays for new variant beginning with the starting_variant

	using namespace aaproperties_pack;
	using namespace param;

	int & nvar_aa( nvar(aa) );
	if ( nvar_aa == MAX_AA_VARIANTS() ) {
		std::cout << "the number of variants for aa " << aa << " "
		 << nvar_aa << " exceeds MAX_AA_VARIANTS" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	int new_variant = ++nvar_aa;
	copy_aaproperties(aa,starting_variant,new_variant);

	// aav_type of the newly created variant
	for ( int i = 1; i <= number_aav_type; ++i ){
		variant_type( i, aa, new_variant ) = false;
	}
	variant_type( aav_type, aa, new_variant ) = true;

	return new_variant; // number of the new variant we created
}


////////////////////////////////////////////////////////////////////////////////
/// @begin read_residue_paircutoffs
///
/// @brief read cutoff distances for pairs of rotamers
///
/// @detailed  read cutoff distances for pairs of rotamers from the file "paircutoffs"
///
/// @global_read
/// variables from "param.h"
///     MAX_AA      -  max number of amino acids
///
/// @global_write
/// variables from "aaproperties_pack.h"
///     paircutoff  -  maximum distance where two sidechains can have interactions
///
/// @remarks
///
/// @references
///
/// @authors Lin Jiang  08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
read_residue_paircutoffs()
{
	using namespace aaproperties_pack;
	using namespace param;
	using namespace param_aa;


	int aa1;
	int aa2;
	int cutoff;

//bk read in the magic cutoff distances for pairs of rotamers

	// Rosetta@home doesn't allow changes to rosetta_database files, for
	// silly reasons, so for these runs we need to look for paircutoffs_v2, which
	// is really just a copy of the recently updated paircutoffs file.

	std::string const paircutoff_file = stringafteroption( "paircutoffs", "paircutoffs" );
	utility::io::izstream & iunit( open_data_file( paircutoff_file ) );
	while ( iunit ) {
		iunit >> bite( 2, aa1 ) >> skip( 1 ) >> bite( 2, aa2 ) >> skip( 1 ) >>
		 bite( 2, cutoff ) >> skip;
		if ( iunit && aa1 <= MAX_AA() && aa2 <= MAX_AA() ) { // NO INPUT ERROR HANDLING
			paircutoff(aa1,aa2) = static_cast< float >( cutoff );
		}
	}
	iunit.close();
	iunit.clear();

//lin fill the cutoff distances for ligand
	if( get_enable_ligaa_flag() ) {
		int first_ligand_aa=int((*ligand_aa_vector.begin()));
		int last_ligand_aa=int((ligand_aa_vector.back()));
		for( int aa1=1; aa1<=last_ligand_aa; aa1++ ) {
			for( int aa2=first_ligand_aa; aa2<=last_ligand_aa; aa2++){
				paircutoff(aa1,aa2)=paircutoff(aa1,6); // 6=glycine
				paircutoff(aa2,aa1)=paircutoff(6,aa1); // 6=glycine
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin create_alternate_his_tautomer
///
/// @brief Creates a variant of histidine with the polar ring proton
///       moved from NE2 to ND1
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
/// Make a copy of all the information set up in init_his() in the
/// variant 2 set, then makes the required changes to move the proton,
/// reset charges, connectivity, etc.
///
/// @remarks
///
/// @references
///
/// @authors Jim Havranek  01/26/04
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
create_alternate_his_tautomer()
{
	using namespace aaproperties_pack;
	using namespace param_aa;

	int const aa = aa_his;

//jjh this is the number of the atom to twiddle
	int const move = 12;

//jjh make a copy of the variant 1 info in the variant 2 set
	int const new_variant = create_new_variant(aa, 1, aav_his_taut);

//jjh switch the polar ring proton to the other nitrogen
//jjh variant is attached to atom 10, move to 7
	atom_name(move, aa, new_variant) = " HD1";
	atom_base(move, aa, new_variant) = 7;

//jjh change the atom types of the nitrogens, since they switch
//jjh protonation state
	fullatom_type(10, aa, new_variant) = 8;
	fullatom_type( 7, aa, new_variant) = 7;

//jjh the charges according to charmm22 change for several

	atomic_charge( 5, aa,new_variant) = -0.09; //    CB
	atomic_charge( 6, aa,new_variant) = -0.05; //    CG
	atomic_charge( 7, aa,new_variant) = -0.36; //    ND1
	atomic_charge( 8, aa,new_variant) =  0.22; //    CD2
	atomic_charge( 9, aa,new_variant) =  0.25; //    CE1
	atomic_charge(10, aa,new_variant) = -0.70; //    NE2
	atomic_charge(12, aa,new_variant) =  0.32; //    HD1
	atomic_charge(17, aa,new_variant) =  0.10; //    HD2

//	assert_total_charge( aa_his, 2, 0.0 );

//jjh atoms to build with
	ta(1,move, aa, new_variant) = 7; // ND1
	ta(2,move, aa, new_variant) = 6; // CG
	ta(3,move, aa, new_variant) = 9; // CE1

//jjh change hbond info
	accpt_pos(2,aa,new_variant) = 10;
//	abase2(10,aa,new_variant) =  9; // CE1 was already set as abase2

	nhydrogens_on_atm(10,aa,new_variant) = 0;

	nhydrogens_on_atm(7,aa,new_variant) = 1;
	hydrogens_on_atm(1,7,aa,new_variant) = move;

//jjh change connectivity info
	nbonded_neighbors( 7, aa, new_variant) = 3; // ND1
		bonded_neighbor(1, 7, aa, new_variant) = 6; // ND1--CG
		bonded_neighbor(2, 7, aa, new_variant) = 9; // ND1--CE1
		bonded_neighbor(3, 7, aa, new_variant) =12; // NE2--HD1
	nbonded_neighbors( 10, aa, new_variant) = 2; // NE2
		bonded_neighbor(1,10, aa, new_variant) = 8; // NE2--CD2
		bonded_neighbor(2,10, aa, new_variant) = 9; // NE2--CE1
	nbonded_neighbors( 12, aa, new_variant) = 1; // HD1
		bonded_neighbor(1,12, aa, new_variant) = 7; // HD1__ND1

//jjh get the angles automatically
	get_angles_from_bonds(aa, new_variant);

//jjh new starting position
	icoor(1, move, aa, new_variant) = -3.092;
	icoor(2, move, aa, new_variant) = 16.198;
	icoor(3, move, aa, new_variant) = 23.000;

}

inline
int
insert_increment( int test, int insert )
{
  return ( test >= insert ? test + 1 : test );
}

inline
int
delete_decrement( int test, int remove )
{
  return ( test > remove ? test - 1 : test );
}


////////////////////////////////////////////////////////////////////////////////
/// @begin make_room_for_atom
///
/// @brief increments atoms numbers to make space for a new atom in the
///       middle of a variant definition
///
/// @detailed
///
///\PARAM
///
/// AA - the amino acid/nucleic acid of interest
/// AAV - the variant that is to be modified
/// INSERT - the position in the array vacated by this function
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///	Keep in mind that total atom counts and start indices must be
///	incremented separately - this just bumps up atom numbers.
///
/// @references
///
/// @authors Jim Havranek  08/31/04
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
make_room_for_atom(
	int const aa,
	int const aav,
	int const insert
)
{
	using namespace aaproperties_pack;
	using namespace param;
	using namespace param_aa;

	//jjh error check that no one is trying to add an atom to a
	//jjh variant definition that is already full
	if ( natoms( aa, aav ) == MAX_ATOM() )	{
		std::cout << "ERROR!!! make_room_for_atom() invoked on full variant!!!"
		<< std::endl << "aa: " << aa << " variant " << aav << std::endl;
	}

	HNpos( aa, aav ) = insert_increment( HNpos( aa, aav ), insert );
	HApos( aa, aav ) = insert_increment( HApos( aa, aav ), insert );

	//jjh some of this is trivial stuff - just copy all the information
	//jjh from atom x ( >= insert ) to x + 1.  However, in many cases
	//jjh the information itself is an atom number and may need to be
	//jjh incremented.

	for ( int i = MAX_ATOM()() - 1; i >= 1; --i ) {

		int dest = insert_increment( i, insert );

		//jjh This batch of information is not atom numbers
		atom_name(dest, aa, aav) = atom_name(i, aa, aav);
		fullatom_type(dest, aa, aav) = fullatom_type(i, aa, aav);
		atomic_charge(dest, aa, aav) = atomic_charge(i, aa, aav);
		for ( int k = 1; k <= MAX_CHI; ++k ) {
			chi_required(k, dest, aa, aav) = chi_required(k, i, aa, aav);
		}

		//jjh To deal with the atom number info, I use an inline helper function
		atom_base(dest, aa, aav) =
										insert_increment( atom_base(i, aa, aav), insert );
		abase2(dest, aa, aav) =
										insert_increment( abase2(i, aa, aav), insert );

		nhydrogens_on_atm(dest, aa, aav) =
		 nhydrogens_on_atm(i, aa, aav);

		for ( int k = 1; k <= 3; ++k ) {
			ta(k, dest, aa, aav) =
										insert_increment( ta(k, i, aa, aav), insert );
			hydrogens_on_atm(k, dest, aa, aav) =
										insert_increment( hydrogens_on_atm(k, i, aa, aav), insert );

			//jjh This one isn't atom number related
			icoor(k, dest, aa, aav) = icoor(k,i, aa, aav);
		}

		nbonded_neighbors(dest, aa, aav) = nbonded_neighbors(i, aa, aav);
		for ( int k = 1; k <= max_bonded_neighbors; ++k ) {
			bonded_neighbor(k, dest, aa, aav) =
										insert_increment( bonded_neighbor(k, i, aa, aav), insert );
		}

		nangle_neighbors(dest, aa, aav) = nangle_neighbors(i, aa, aav);
		for ( int k = 1; k <= max_angle_neighbors; ++k ) {
			angle_neighbor(k, dest, aa, aav) =
										insert_increment( angle_neighbor(k, i, aa, aav), insert );
		}

		//jjh Need to be changed, but only on the right hand side
		accpt_pos(i, aa, aav) =
								insert_increment( accpt_pos(i, aa, aav), insert );
		Hpos_polar(i, aa, aav) =
								insert_increment( Hpos_polar(i, aa, aav), insert );
		Hpos_apolar(i, aa, aav) =
								insert_increment( Hpos_apolar(i, aa, aav), insert );
		Hpos_aromatic(i, aa, aav) =
								insert_increment( Hpos_aromatic(i, aa, aav), insert );

	}

	//jjh Only change th rhs
	for ( int i = 1; i <= MAX_CHI; ++i ) {
		for ( int k = 1; k <= 4; ++k ) {
			chi_atoms(k, i, aa, aav) =
								insert_increment( chi_atoms(k, i, aa, aav), insert );
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin remove_atom
///
/// @brief decrements atoms numbers to remove an atom in the
///       middle of a variant definition
///
/// @detailed
///
///\PARAM
///
/// AA - the amino acid/nucleic acid of interest
/// AAV - the variant that is to be modified
/// REMOVE - the position in the array to be removed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///	Keep in mind that total atom counts and start indices must be
///	decremented separately - this just changes atom numbers.
///
/// @references
///
/// @authors Jim Havranek  08/31/04
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
remove_atom(
	int const aa,
	int const aav,
	int const remove
)
{
	using namespace aaproperties_pack;
	using namespace param;
	using namespace param_aa;

	//jjh error check that no one is trying to remove an atom from a
	//jjh variant definition that is already empty
	if ( natoms( aa, aav ) == 0 )	{
		std::cout << "ERROR!!! remove_atom() invoked on empty variant!!!"
		<< std::endl << "aa: " << aa << " variant " << aav << std::endl;
	}

	if ( HNpos( aa, aav ) == remove ) {
		std::cout << "ERROR!! remove_atom() invoked on peptide hydrogen!!!" << std::endl;
	} else {
		HNpos( aa, aav ) = delete_decrement( HNpos( aa, aav ), remove );
	}

	if ( HApos( aa, aav ) == remove ) {
		std::cout << "ERROR!! remove_atom() invoked on alpha hydrogen!!!" << std::endl;
	} else {
		HApos( aa, aav ) = delete_decrement( HApos( aa, aav ), remove );
	}

	//jjh some of this is trivial stuff - just copy all the information
	//jjh from atom x ( > remove ) to x - 1.  However, in many cases
	//jjh the information itself is an atom number and may need to be
	//jjh incremented.

	for ( int i = 1, e = MAX_ATOM(); i <= e; ++i ) {

		if ( i == remove ) continue;

		int dest = delete_decrement( i, remove );

		//jjh This batch of information is not atom numbers
		atom_name(dest, aa, aav) = atom_name(i, aa, aav);
		fullatom_type(dest, aa, aav) = fullatom_type(i, aa, aav);
		atomic_charge(dest, aa, aav) = atomic_charge(i, aa, aav);
		for ( int k = 1; k <= MAX_CHI; ++k ) {
			chi_required(k, dest, aa, aav) = chi_required(k, i, aa, aav);
		}

		//jjh To deal with the atom number info, I use an inline helper
		//jjh function
		atom_base(dest, aa, aav) =
										delete_decrement( atom_base(i, aa, aav), remove );
		abase2(dest, aa, aav) =
										delete_decrement( abase2(i, aa, aav), remove );

		nhydrogens_on_atm(dest, aa, aav) =
		 nhydrogens_on_atm(i, aa, aav);

		for ( int k = 1; k <= 3; ++k ) {
			ta(k, dest, aa, aav) =
										delete_decrement( ta(k, i, aa, aav), remove );
			hydrogens_on_atm(k, dest, aa, aav) =
										delete_decrement( hydrogens_on_atm(k, i, aa, aav), remove );

			//jjh This one isn't atom number related
			icoor(k, dest, aa, aav) = icoor(k,i, aa, aav);
		}

		nbonded_neighbors(dest, aa, aav) = nbonded_neighbors(i, aa, aav);
		for ( int k = 1; k <= max_bonded_neighbors; ++k ) {
			bonded_neighbor(k, dest, aa, aav) =
										delete_decrement( bonded_neighbor(k, i, aa, aav), remove );
		}

		nangle_neighbors(dest, aa, aav) = nangle_neighbors(i, aa, aav);
		for ( int k = 1; k <= max_angle_neighbors; ++k ) {
			angle_neighbor(k, dest, aa, aav) =
										delete_decrement( angle_neighbor(k, i, aa, aav), remove );
		}

		//jjh Need to be changed, but only on the right hand side
		accpt_pos(i, aa, aav) =
								delete_decrement( accpt_pos(i, aa, aav), remove );
		Hpos_polar(i, aa, aav) =
								delete_decrement( Hpos_polar(i, aa, aav), remove );
		Hpos_apolar(i, aa, aav) =
								delete_decrement( Hpos_apolar(i, aa, aav), remove );
		Hpos_aromatic(i, aa, aav) =
								delete_decrement( Hpos_aromatic(i, aa, aav), remove );

	}

	//jjh Only change th rhs
	for ( int i = 1; i <= MAX_CHI; ++i ) {
		for ( int k = 1; k <= 4; ++k ) {
			chi_atoms(k, i, aa, aav) =
								delete_decrement( chi_atoms(k, i, aa, aav), remove );
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_angles_from_bonds
///
/// @brief Uses the nonded_neighbor information to generate angle_neighbor
///       information
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Jim Havranek  04/16/04
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_angles_from_bonds(
	int const aa,
	int const aav
)
{
	using namespace aaproperties_pack;
	using namespace param;

	FArray1D_bool already_added( MAX_ATOM()() );

// jjh error check - make sure if i->j, then j->i also

	for ( int atm = 1, atme = natoms(aa, aav); atm <= atme; ++atm ) {
		for ( int bnd = 1, bnde = nbonded_neighbors(atm, aa, aav);
		 bnd <= bnde; ++bnd ) {
			int const atm2 = bonded_neighbor(bnd, atm, aa, aav);

			bool found = false;

			for ( int bnd2 = 1, bnd2e = nbonded_neighbors(atm2, aa, aav);
			 bnd2 <= bnd2e; ++bnd2 ) {
				int const atm3 = bonded_neighbor(bnd2, atm2, aa, aav);
				if ( atm3 == atm ) found = true;
			}

			if ( !found ) {
				std::cout << "BOND ERROR!!! atm, aa, aav: " << atm << ' ' << aa << ' ' << aav << std::endl;
			}
		}
	}

	for ( int atm = 1, atme = natoms(aa, aav); atm <= atme; ++atm ) {

//jjh Always reset
		int nangles = 0;

//jjh create a checklist of atoms in residue, to avoid double
//jjh counting angle partners
		for ( int atm2 = 1, atm2e = natoms(aa, aav); atm2 <= atm2e; ++atm2 ) {
			already_added(atm2) = false;
		}

//jjh run through all bonded neighbors, then add all of their
//jjh bonded neighbors
		for ( int bnd = 1, bnde = nbonded_neighbors(atm, aa, aav);
		 bnd <= bnde; ++bnd ) {
			int const atm2 = bonded_neighbor(bnd, atm, aa, aav);
			for ( int bnd2 = 1, bnd2e = nbonded_neighbors(atm2, aa, aav);
			 bnd2 <= bnd2e; ++bnd2 ) {
				int const atm3 = bonded_neighbor(bnd2, atm2, aa, aav);
				if ( ! already_added(atm2) ) {
					if ( nangles == max_angle_neighbors ) {
						std::cout << "Increase angle_neighbor array size!!!" << std::endl;
						utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
					}
					++nangles;
					angle_neighbor(nangles, atm, aa, aav) = atm3;
					already_added(atm3) = true;
				}
			}
		}
		nangle_neighbors(atm, aa, aav) = nangles;
	}

// Now do the dihedrals - Note, this actual gets the dihedrals, plus
// some extra bonded terms.  So when this is used, as in gb_elec.cc,
// make sure that assignments go dihedrals, angles, then bonds, so
// the bonded flags over-write the incorrect dihedrals

	for ( int atm = 1, atme = natoms(aa, aav); atm <= atme; ++atm ) {

//jjh Always reset
		int ndihes = 0;

//jjh create a checklist of atoms in residue, to avoid double
//jjh counting dihe partners
		for ( int atm2 = 1, atm2e = natoms(aa, aav); atm2 <= atm2e; ++atm2 ) {
			already_added(atm2) = false;
		}

//jjh run through all bonded neighbors, then add all of their
//jjh bonded neighbors
		for ( int ang = 1, ange = nangle_neighbors(atm, aa, aav);
		 ang <= ange; ++ang ) {
			int const atm2 = angle_neighbor(ang, atm, aa, aav);
			for ( int bnd = 1, bnde = nbonded_neighbors(atm2, aa, aav);
			 bnd <= bnde; ++bnd ) {
				int const atm3 = bonded_neighbor(bnd, atm2, aa, aav);
				if ( ! already_added(atm2) ) {
					if ( ndihes == max_dihe_neighbors ) {
						std::cout << "Increase max_dihe_neighbors array size!!!" << std::endl;
						utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
					}
					++ndihes;
					dihe_neighbor(ndihes, atm, aa, aav) = atm3;
					already_added(atm3) = true;
				}
			}
		}
		ndihe_neighbors(atm, aa, aav) = ndihes;
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin assert_total_charge
///
/// @brief Checks to make sure that the total charge of a residue is what
///       we think it is
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Jim Havranek  04/16/04
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
assert_total_charge(
	int const aa,
	int const aav,
	float const total_q
)
{
	using namespace aaproperties_pack;
	using namespace param;

		float running_sum = 0.0;

		for ( int atm = 1, atme = natoms(aa, aav); atm <= atme; ++atm ) {
			running_sum += atomic_charge(atm, aa, aav);
		}

		if ( std::abs( running_sum - total_q ) >= 1.0e-6 ) {
			std::cout << "Total charge check failed for aa, variant " << aa << " " << aav << std::endl;
			std::cout << "Expected " << total_q << " but got " << running_sum << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_aliphatic_proton
///
/// @brief selects aliphatic proton names according to -IUPAC command line option
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Jim Havranek  04/16/04
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_aliphatic_proton()
{
	using namespace aaproperties_pack;
	using namespace files_paths;
	using namespace param_aa;

	static bool init = false;

	if ( ! init ) IUPAC = truefalseoption( "IUPAC" );
	if ( IUPAC ) {

//  Adjust for CYS

		atom_name( 9, aa_cys,1) = "2HB ";
		atom_name(10, aa_cys,1) = "3HB ";
//  Adjust for ASP

		atom_name(11, aa_asp,1) = "2HB ";
		atom_name(12, aa_asp,1) = "3HB ";

//  Adjust for GLU

		atom_name(12, aa_glu,1) = "2HB ";
		atom_name(13, aa_glu,1) = "3HB ";
		atom_name(14, aa_glu,1) = "2HG ";
		atom_name(15, aa_glu,1) = "3HG ";

//  Adjust for PHE

		atom_name(19, aa_phe, 1) = "2HB ";
		atom_name(20, aa_phe, 1) = "3HB ";

//  Adjust for GLY

		atom_name( 6, aa_gly,1) = "2HA ";
		atom_name( 7, aa_gly,1) = "3HA ";

//  Adjust for HIS

		atom_name(14, aa_his,1) = "2HB ";
		atom_name(15, aa_his,1) = "3HB ";

//  Adjust for ILE

		atom_name(15, aa_ile,1) = "2HG1";
		atom_name(16, aa_ile,1) = "3HG1";

//  Adjust for LEU

		atom_name(11,aa_leu,1) = "2HB ";
		atom_name(12,aa_leu,1) = "3HB ";

//  Adjust for LYS

		atom_name(15, aa_lys,1) = "2HB ";
		atom_name(16, aa_lys,1) = "3HB ";
		atom_name(17, aa_lys,1) = "2HG ";
		atom_name(18, aa_lys,1) = "3HG ";
		atom_name(19, aa_lys,1) = "2HD ";
		atom_name(20, aa_lys,1) = "3HD ";
		atom_name(21, aa_lys,1) = "2HE ";
		atom_name(22, aa_lys,1) = "3HE ";

//  Adjust for MET

		atom_name(11,aa_met,1) = "2HB ";
		atom_name(12,aa_met,1) = "3HB ";
		atom_name(13,aa_met,1) = "2HG ";
		atom_name(14,aa_met,1) = "3HG ";

//  Adjust for ASN

		atom_name(13,aa_asn,1) = "2HB ";
		atom_name(14,aa_asn,1) = "3HB ";

//  Adjust for PRO

		atom_name( 8,aa_pro,1) = "2HD ";
		atom_name( 9,aa_pro,1) = "3HD ";
		atom_name(10,aa_pro,1) = "2HG ";
		atom_name(11,aa_pro,1) = "3HG ";
		atom_name(12,aa_pro,1) = "2HB ";
		atom_name(13,aa_pro,1) = "3HB ";

//  Adjust for GLN

		atom_name(14,aa_gln,1) = "2HB ";
		atom_name(15,aa_gln,1) = "3HB ";
		atom_name(16,aa_gln,1) = "2HG ";
		atom_name(17,aa_gln,1) = "3HG ";

//  Adjust for ARG

		atom_name(19,aa_arg,1) = "2HB ";
		atom_name(20,aa_arg,1) = "3HB ";
		atom_name(21,aa_arg,1) = "2HG ";
		atom_name(22,aa_arg,1) = "3HG ";
		atom_name(23,aa_arg,1) = "2HD ";
		atom_name(24,aa_arg,1) = "3HD ";

//  Adjust for SER

		atom_name(10,aa_ser,1) = "2HB ";
		atom_name(11,aa_ser,1) = "3HB ";

//  Adjust for TRP

		atom_name(23,aa_trp,1) = "2HB ";
		atom_name(24,aa_trp,1) = "3HB ";

//  Adjust for TYR

		atom_name(20,aa_tyr,1) = "2HB ";
		atom_name(21,aa_tyr,1) = "3HB ";

//KMa  Adjust for SEP
//matan_kob, changed 1HB and 2HB to IUPAC convention, for consistency with other aa,2008-02
		atom_name(13,aa_sep,1) = "2HB ";
		atom_name(14,aa_sep,1) = "3HB ";

	} else { //PDB convention

//  Adjust for CYS

		for (int aav = 1; aav <= nvar(aa_cys); ++aav){
			int pos9 = LookupByName(aa_cys, aav, "2HB ");
			int pos10= LookupByName(aa_cys, aav, "3HB ");
			atom_name( pos9, aa_cys,aav) = "1HB ";
			atom_name(pos10, aa_cys,aav) = "2HB ";
		}

//  Adjust for ASP

		for (int aav = 1; aav <= nvar(aa_asp); ++aav){
			int pos9 = LookupByName(aa_asp, aav, "2HB ");
			int pos10= LookupByName(aa_asp, aav, "3HB ");
			atom_name( pos9, aa_asp,aav) = "1HB ";
			atom_name(pos10, aa_asp,aav) = "2HB ";
		}

//  Adjust for GLU

		for (int aav = 1; aav <= nvar(aa_glu); ++aav){
			int pos9 = LookupByName(aa_glu, aav, "2HB ");
			int pos10= LookupByName(aa_glu, aav, "3HB ");
			atom_name( pos9, aa_glu,aav) = "1HB ";
			atom_name(pos10, aa_glu,aav) = "2HB ";
			pos9 = LookupByName(aa_glu, aav, "2HG ");
			pos10= LookupByName(aa_glu, aav, "3HG ");
			atom_name( pos9, aa_glu,aav) = "1HG ";
			atom_name(pos10, aa_glu,aav) = "2HG ";
		}

//  Adjust for PHE

		for (int aav = 1; aav <= nvar(aa_phe); ++aav){
			int pos9 = LookupByName(aa_phe, aav, "2HB ");
			int pos10= LookupByName(aa_phe, aav, "3HB ");
			atom_name( pos9, aa_phe,aav) = "1HB ";
			atom_name(pos10, aa_phe,aav) = "2HB ";
		}

//  Adjust for GLY

		for (int aav = 1; aav <= nvar(aa_gly); ++aav){
			int pos9 = LookupByName(aa_gly, aav, "2HA ");
			int pos10= LookupByName(aa_gly, aav, "3HA ");
			atom_name( pos9, aa_gly,aav) = "1HA ";
			atom_name(pos10, aa_gly,aav) = "2HA ";
		}

//  Adjust for HIS

		for (int aav = 1; aav <= nvar(aa_his); ++aav){
			int pos9 = LookupByName(aa_his, aav, "2HB ");
			int pos10= LookupByName(aa_his, aav, "3HB ");
			atom_name( pos9, aa_his,aav) = "1HB ";
			atom_name(pos10, aa_his,aav) = "2HB ";
		}

//  Adjust for ILE

		for (int aav = 1; aav <= nvar(aa_ile); ++aav){
			int pos9 = LookupByName(aa_ile, aav, "2HG1");
			int pos10= LookupByName(aa_ile, aav, "3HG1");
			atom_name( pos9, aa_ile,aav) = "1HG1";
			atom_name(pos10, aa_ile,aav) = "2HG1";
		}

//  Adjust for LEU

		for (int aav = 1; aav <= nvar(aa_leu); ++aav){
			int pos9 = LookupByName(aa_leu, aav, "2HB ");
			int pos10= LookupByName(aa_leu, aav, "3HB ");
			atom_name( pos9, aa_leu,aav) = "1HB ";
			atom_name(pos10, aa_leu,aav) = "2HB ";
		}


//  Adjust for LYS

		for (int aav = 1; aav <= nvar(aa_lys); ++aav){
			int pos9 = LookupByName(aa_lys, aav, "2HB ");
			int pos10= LookupByName(aa_lys, aav, "3HB ");
			atom_name( pos9, aa_lys,aav) = "1HB ";
			atom_name(pos10, aa_lys,aav) = "2HB ";

			pos9 = LookupByName(aa_lys, aav, "2HG ");
			pos10= LookupByName(aa_lys, aav, "3HG ");
			atom_name( pos9, aa_lys,aav) = "1HG ";
			atom_name(pos10, aa_lys,aav) = "2HG ";

			pos9 = LookupByName(aa_lys, aav, "2HD ");
			pos10= LookupByName(aa_lys, aav, "3HD ");
			atom_name( pos9, aa_lys,aav) = "1HD ";
			atom_name(pos10, aa_lys,aav) = "2HD ";

			pos9 = LookupByName(aa_lys, aav, "2HE ");
			pos10= LookupByName(aa_lys, aav, "3HE ");
			atom_name( pos9, aa_lys,aav) = "1HE ";
			atom_name(pos10, aa_lys,aav) = "2HE ";
		}

//  Adjust for MET

		for (int aav = 1; aav <= nvar(aa_met); ++aav){
			int pos9 = LookupByName(aa_met, aav, "2HB ");
			int pos10= LookupByName(aa_met, aav, "3HB ");
			atom_name( pos9, aa_met,aav) = "1HB ";
			atom_name(pos10, aa_met,aav) = "2HB ";

			pos9 = LookupByName(aa_met, aav, "2HG ");
			pos10= LookupByName(aa_met, aav, "3HG ");
			atom_name( pos9, aa_met,aav) = "1HG ";
			atom_name(pos10, aa_met,aav) = "2HG ";
		}

//  Adjust for ASN

		for (int aav = 1; aav <= nvar(aa_asn); ++aav){
			int pos9 = LookupByName(aa_asn, aav, "2HB ");
			int pos10= LookupByName(aa_asn, aav, "3HB ");
			atom_name( pos9, aa_asn,aav) = "1HB ";
			atom_name(pos10, aa_asn,aav) = "2HB ";
		}

//  Adjust for PRO

		for (int aav = 1; aav <= nvar(aa_pro); ++aav){
			int pos9 = LookupByName(aa_pro, aav, "2HD ");
			int pos10= LookupByName(aa_pro, aav, "3HD ");
			atom_name( pos9, aa_pro,aav) = "1HD ";
			atom_name(pos10, aa_pro,aav) = "2HD ";

			pos9 = LookupByName(aa_pro, aav, "2HG ");
			pos10= LookupByName(aa_pro, aav, "3HG ");
			atom_name( pos9, aa_pro,aav) = "1HG ";
			atom_name(pos10, aa_pro,aav) = "2HG ";

			pos9 = LookupByName(aa_pro, aav, "2HB ");
			pos10= LookupByName(aa_pro, aav, "3HB ");
			atom_name( pos9, aa_pro,aav) = "1HB ";
			atom_name(pos10, aa_pro,aav) = "2HB ";

		}

//  Adjust for GLN

		for (int aav = 1; aav <= nvar(aa_gln); ++aav){
			int pos9 = LookupByName(aa_gln, aav, "2HB ");
			int pos10= LookupByName(aa_gln, aav, "3HB ");
			atom_name( pos9, aa_gln,aav) = "1HB ";
			atom_name(pos10, aa_gln,aav) = "2HB ";

			pos9 = LookupByName(aa_gln, aav, "2HG ");
			pos10= LookupByName(aa_gln, aav, "3HG ");
			atom_name( pos9, aa_gln,aav) = "1HG ";
			atom_name(pos10, aa_gln,aav) = "2HG ";
		}

//  Adjust for ARG

		for (int aav = 1; aav <= nvar(aa_arg); ++aav){
			int pos9 = LookupByName(aa_arg, aav, "2HB ");
			int pos10= LookupByName(aa_arg, aav, "3HB ");
			atom_name( pos9, aa_arg,aav) = "1HB ";
			atom_name(pos10, aa_arg,aav) = "2HB ";

			pos9 = LookupByName(aa_arg, aav, "2HG ");
			pos10= LookupByName(aa_arg, aav, "3HG ");
			atom_name( pos9, aa_arg,aav) = "1HG ";
			atom_name(pos10, aa_arg,aav) = "2HG ";

			pos9 = LookupByName(aa_arg, aav, "2HD ");
			pos10= LookupByName(aa_arg, aav, "3HD ");
			atom_name( pos9, aa_arg,aav) = "1HD ";
			atom_name(pos10, aa_arg,aav) = "2HD ";
		}

//  Adjust for SER

		for (int aav = 1; aav <= nvar(aa_ser); ++aav){
			int pos9 = LookupByName(aa_ser, aav, "2HB ");
			int pos10= LookupByName(aa_ser, aav, "3HB ");
			atom_name( pos9, aa_ser,aav) = "1HB ";
			atom_name(pos10, aa_ser,aav) = "2HB ";
		}

//  Adjust for TRP

		for (int aav = 1; aav <= nvar(aa_trp); ++aav){
			int pos9 = LookupByName(aa_trp, aav, "2HB ");
			int pos10= LookupByName(aa_trp, aav, "3HB ");
			atom_name( pos9, aa_trp,aav) = "1HB ";
			atom_name(pos10, aa_trp,aav) = "2HB ";
		}

//  Adjust for TYR

		for (int aav = 1; aav <= nvar(aa_tyr); ++aav){
			int pos9 = LookupByName(aa_tyr, aav, "2HB ");
			int pos10= LookupByName(aa_tyr, aav, "3HB ");
			atom_name( pos9, aa_tyr,aav) = "1HB ";
			atom_name(pos10, aa_tyr,aav) = "2HB ";
		}

//  Adjust for SEP //KMa phosph_ser future
		/*	for (int aav = 1; aav <= nvar(aa_sep); ++aav){
			int pos9 = LookupByName(aa_sep, aav, "2HB ");
			int pos10= LookupByName(aa_sep, aav, "3HB ");
			atom_name( pos9, aa_sep,aav) = "1HB ";
			atom_name(pos10, aa_sep,aav) = "2HB ";
		}*/

	}

}



////////////////////////////////////////////////////////////////////////////////
/// @begin init_N_terminus
///
/// @brief initialize N terminus according to -Nterminus command line option
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Bin Qian  11/1/05
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_N_terminus()
{
	using namespace aaproperties_pack;
	using namespace files_paths;
	using namespace termini_ns;


	if ( !use_N_terminus ) return;

	Nterm_natoms = 6;

	Nterm_atom_name( 1 ) = " N  ";
	Nterm_atom_name( 2 ) = " CA ";
	Nterm_atom_name( 3 ) = " C  ";
	Nterm_atom_name( 4 ) = "1H  ";
	Nterm_atom_name( 5 ) = "2H  ";
	Nterm_atom_name( 6 ) = "3H  ";

//bk   template coordinates for the Nterm
	Nterm_icoor( 1, 1 ) =   -1.2830; //   N
	Nterm_icoor( 2, 1 ) =   -0.4010; //   N
	Nterm_icoor( 3, 1 ) =    1.2020; //   N
	Nterm_icoor( 1, 2 ) =   -0.6350; //   CA
	Nterm_icoor( 2, 2 ) =   -0.3940; //   CA
	Nterm_icoor( 3, 2 ) =   -0.1610; //   CA
	Nterm_icoor( 1, 3 ) =    0.8340; //   C
	Nterm_icoor( 2, 3 ) =   -0.0170; //   C
	Nterm_icoor( 3, 3 ) =    0.0740; //   C
	Nterm_icoor( 1, 4 ) =   -1.2330; //   1H
	Nterm_icoor( 2, 4 ) =    0.5110; //   1H
	Nterm_icoor( 3, 4 ) =    1.6270; //   1H
	Nterm_icoor( 1, 5 ) =   -0.7960; //   2H
	Nterm_icoor( 2, 5 ) =   -1.0180; //   2H
	Nterm_icoor( 3, 5 ) =    1.8320; //   2H
	Nterm_icoor( 1, 6 ) =   -2.2470; //   3H
	Nterm_icoor( 2, 6 ) =   -0.6810; //   3H
	Nterm_icoor( 3, 6 ) =    1.1580; //   3H

//bk   template coordinates for proline Nterminus
	Nterm_icoor_pro( 1, 1 ) =    0.5080; //   N
	Nterm_icoor_pro( 2, 1 ) =   -1.0530; //   N
	Nterm_icoor_pro( 3, 1 ) =   -0.3500; //   N
	Nterm_icoor_pro( 1, 2 ) =    1.5510; //   CA
	Nterm_icoor_pro( 2, 2 ) =   -0.3540; //   CA
	Nterm_icoor_pro( 3, 2 ) =    0.4860; //   CA
	Nterm_icoor_pro( 1, 3 ) =    1.7130; //   C
	Nterm_icoor_pro( 2, 3 ) =    1.0470; //   C
	Nterm_icoor_pro( 3, 3 ) =   -0.1280; //   C
	Nterm_icoor_pro( 1, 4 ) =    0.2970; //   1H
	Nterm_icoor_pro( 2, 4 ) =   -0.4120; //   1H
	Nterm_icoor_pro( 3, 4 ) =   -1.1070; //   1H
	Nterm_icoor_pro( 1, 5 ) =   -0.3450; //   2H
	Nterm_icoor_pro( 2, 5 ) =   -1.2070; //   2H
	Nterm_icoor_pro( 3, 5 ) =    0.1540; //   2H
//bq   dummy coords to be compatible with other res. Pro doesn't have 3H
	Nterm_icoor_pro( 1, 6 ) =   -2.2470; //   3H
	Nterm_icoor_pro( 2, 6 ) =   -0.6810; //   3H
	Nterm_icoor_pro( 3, 6 ) =    1.1580; //   3H

}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_C_terminus
///
/// @brief initialize C terminus according to -termini command line option
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Bin Qian  11/1/05
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_C_terminus()
{
	using namespace aaproperties_pack;
	using namespace files_paths;
	using namespace termini_ns;

	if ( !use_C_terminus ) return;

	Cterm_natoms = 5;

	Cterm_atom_name( 1 ) = " N  ";
	Cterm_atom_name( 2 ) = " CA ";
	Cterm_atom_name( 3 ) = " C  ";
	Cterm_atom_name( 4 ) = " O  ";
	Cterm_atom_name( 5 ) = " OXT";

//bk   template coordinates for the nucleic acid

	Cterm_icoor( 1, 1 ) =   -0.4460; //   N
	Cterm_icoor( 2, 1 ) =   -0.2090; //   N
	Cterm_icoor( 3, 1 ) =    0.4350; //   N
	Cterm_icoor( 1, 2 ) =    0.4680; //   CA
	Cterm_icoor( 2, 2 ) =    0.1960; //   CA
	Cterm_icoor( 3, 2 ) =   -0.6240; //   CA
	Cterm_icoor( 1, 3 ) =    1.6370; //   C
	Cterm_icoor( 2, 3 ) =    1.0080; //   C
	Cterm_icoor( 3, 3 ) =    0.0140; //   C
	Cterm_icoor( 1, 4 ) =    1.5110; //   1O
	Cterm_icoor( 2, 4 ) =    1.2850; //   1O
	Cterm_icoor( 3, 4 ) =    1.2110; //   1O
	Cterm_icoor( 1, 5 ) =    2.5430; //   2O
	Cterm_icoor( 2, 5 ) =    1.2880; //   2O
	Cterm_icoor( 3, 5 ) =   -0.7610; //   2O

}


////////////////////////////////////////////////////////////////////////////////
/// @begin init_sep
///
/// @brief arrays filled to define the proerties of the amino acid phospho-Serine
///
/// @detailed This and all other init_xxx functions are the intial assignments
///      of most atom types and properties for each amino acid; nearly
///      all assignments are hard-wired (i.e. not assigned through other variables)
///
/// @global_read
/// aa_sep - index number for SEP (21)
/// explicit_h2o - flag for including extra water rotamers
///
/// @global_write Most names self-explanatory
/// nvar - # of amino acid variants (used primarily for extra rotamers with H2Os)
/// natoms - total # of atoms for this aa
/// nheavyatoms - total # of non-hydrogen atoms
/// nchi - # of chi angles for side chain
/// atom_name - full PDB atom name for each atom
/// fullatom_type -  index for rosetta atom type
/// atomic_charge - partial atomic charges for electrostatics calculations
/// ta - 3 template atoms for building each hydrogen
/// chi_required - Which chi angles are required to build given atoms
/// chi_atoms - Which atoms define a given chi angle
/// HNpos - position of backbone HN
/// HApos - position of alpha hydrogen
/// nH_polar - # polar hydrogens
/// nH_aromatic - # aromatic hydrogens
/// nH_apolar - # non-polar hydrogens
/// nacceptors - # H-bond acceptors
/// Hpos_polar - id the polar hydrogens
/// Hpos_apolar - id the apolar hydrogens
/// atom_base - id the base atom for each atom
/// accpt_pos - id the acceptor positions
/// abase2 - id the base2 atoms for each acceptor
/// nhydrogens_on_atm - # hydrogens on each atom
/// nhydrogens_on_atm - id the hydrogens on each atom
/// icoor - xyz coordinates of template amino acids
///
/// @remarks
///
/// @references
///
/// @authors
/// Kosta Makrodimitris, Ph.D. (KMa)
///
/// @last_modified 2006-01
/////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////
void
init_sep()
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

//KMa   Parameters for SEP -----------------------------
	nvar(aa_sep) = 1; // starting number of variants for SEP
	variant_type( aav_base, aa_sep, 1 ) = true; // type of aa variant

//KMa   variant1 ---------- SEP
//KMa since the phosphate could be modeled in different pHs
//KMa  new variants and Hydrogens could add in the future
	natoms(aa_sep,1) = 14; // number of total atoms
	nheavyatoms (aa_sep,1) =  10 ; // number of heavy atoms
	nchi(aa_sep,1) =  3; // number of chi angles

//KMa   atom names SEP
//matan_kob, changed 1HB and 2HB to IUPAC convention, for consistency with other aa,2008-02
	atom_name( 1,aa_sep,1) = " N  ";
	atom_name( 2,aa_sep,1) = " CA ";
	atom_name( 3,aa_sep,1) = " C  ";
	atom_name( 4,aa_sep,1) = " O  ";
	atom_name( 5,aa_sep,1) = " CB ";
	atom_name( 6,aa_sep,1) = " OG ";
	atom_name( 7,aa_sep,1)  = " P  ";
	atom_name( 8,aa_sep,1)  = "O1P ";
	atom_name( 9,aa_sep,1)  = "O2P ";
	atom_name( 10,aa_sep,1) = "O3P ";
	atom_name( 11,aa_sep,1) = " H  ";
	atom_name( 12,aa_sep,1) = " HA ";
	atom_name( 13,aa_sep,1) = "2HB ";
	atom_name( 14,aa_sep,1) = "3HB ";

//KMa intra residue bonding
	nbonded_neighbors( 1, aa_sep, 1) = 2; // N
		bonded_neighbor(1, 1, aa_sep, 1) = 2; // N--CA
		bonded_neighbor(2, 1, aa_sep, 1) = 11; // N--H
	nbonded_neighbors( 2, aa_sep, 1) = 4; // CA
		bonded_neighbor(1, 2, aa_sep, 1) = 1; // CA--N
		bonded_neighbor(2, 2, aa_sep, 1) = 3; // CA--C
		bonded_neighbor(3, 2, aa_sep, 1) = 5; // CA--CB
		bonded_neighbor(4, 2, aa_sep, 1) = 12; // CA--HA
	nbonded_neighbors( 3, aa_sep, 1) = 2; // C
		bonded_neighbor(1, 3, aa_sep, 1) = 2; // C--CA
		bonded_neighbor(2, 3, aa_sep, 1) = 4; // C--O
	nbonded_neighbors( 4, aa_sep, 1) = 1; // O
		bonded_neighbor(1, 4, aa_sep, 1) = 3; // O--C
	nbonded_neighbors( 5, aa_sep, 1) = 4; // CB
		bonded_neighbor(1, 5, aa_sep, 1) = 2; // CB--CA
		bonded_neighbor(2, 5, aa_sep, 1) = 6; // CB--OG
		bonded_neighbor(3, 5, aa_sep, 1) =13; // CB--2HB
		bonded_neighbor(4, 5, aa_sep, 1) =14; // CB--3HB
	nbonded_neighbors( 6, aa_sep, 1) = 2; // OG
		bonded_neighbor(1, 6, aa_sep, 1) = 5; // OG--CB
		bonded_neighbor(2, 6, aa_sep, 1) = 7; // OG--Phos
	nbonded_neighbors( 7, aa_sep, 1) = 4; //Phos
		bonded_neighbor(1,7, aa_sep, 1) = 6; //Phos--OG
		bonded_neighbor(2,7, aa_sep, 1) = 8; //Phos--O1P
		bonded_neighbor(3,7, aa_sep, 1) = 9; //Phos--O2P
		bonded_neighbor(4,7, aa_sep, 1) = 10; //Phos--O3P
	nbonded_neighbors( 8, aa_sep, 1) = 1; //O1P
		bonded_neighbor(1,8, aa_sep, 1) = 7; //O1P--Phos
	nbonded_neighbors( 9, aa_sep, 1) = 1; //O2P
		bonded_neighbor(1,9, aa_sep, 1) = 7; //O2P-Phos
	nbonded_neighbors( 10, aa_sep, 1) = 1; //O3P
		bonded_neighbor(1,10, aa_sep, 1) = 7; //O3P-Phos
	nbonded_neighbors( 11, aa_sep, 1) = 1; // H
		bonded_neighbor(1, 11, aa_sep, 1) = 1; // H--N
	nbonded_neighbors( 12, aa_sep, 1) = 1; // HA
		bonded_neighbor(1, 12, aa_sep, 1) = 2; // HA--CA
		nbonded_neighbors( 13, aa_sep, 1) = 1; //1HB
		bonded_neighbor(1,13, aa_sep, 1) = 5; //1HB--CB
	nbonded_neighbors( 14, aa_sep, 1) = 1; //2HB
		bonded_neighbor(1,14, aa_sep, 1) = 5; //2HB--CB

//KMa get the angles automatically
	get_angles_from_bonds(aa_sep, 1);

//KMa   atom type numbers SEP
	fullatom_type( 1,aa_sep,1) = 17; // Nbb    N
	fullatom_type( 2,aa_sep,1) = 18; // CAbb   CA
	fullatom_type( 3,aa_sep,1) = 19; // CObb   C
	fullatom_type( 4,aa_sep,1) = 20; // OCbb   O
	fullatom_type( 5,aa_sep,1) =  4; // CH2    CB
	fullatom_type( 6,aa_sep,1) = 13; // OH     OG
	fullatom_type( 7,aa_sep,1) = 21; // Phos
	fullatom_type( 8,aa_sep,1) = 43; // O1P
	fullatom_type( 9,aa_sep,1) = 43; // O2P
	fullatom_type( 10,aa_sep,1) = 43; // O3P
	fullatom_type( 11,aa_sep,1) = 25; // HNbb   H
	fullatom_type( 12,aa_sep,1) = 23; // Hapo   HA
	fullatom_type( 13,aa_sep,1) = 23; // Hapo  1HB
	fullatom_type( 14,aa_sep,1) = 23; // Hapo  2HB

//KMa (CHARMM27 for DNA)
	atomic_charge( 1,aa_sep,1) = -0.47; //    N
	atomic_charge( 2,aa_sep,1) =  0.07; //    CA
	atomic_charge( 3,aa_sep,1) =  0.51; //    C
	atomic_charge( 4,aa_sep,1) = -0.51; //    O
	atomic_charge( 5,aa_sep,1) =  0.05; //    CB
	atomic_charge( 6,aa_sep,1) = -0.66; //    OG
	atomic_charge( 7,aa_sep,1) =  1.13; //   P
	atomic_charge( 8,aa_sep,1) = -0.90; //   OP
	atomic_charge( 9,aa_sep,1) = -0.90; //   OP
	atomic_charge(10,aa_sep,1) = -0.90; //   OP
	atomic_charge( 11,aa_sep,1) =  0.31; //    H
	atomic_charge( 12,aa_sep,1) =  0.09; //    HA
	atomic_charge( 13,aa_sep,1) =  0.09; //   1HB
	atomic_charge( 14,aa_sep,1) =  0.09; //   2HB

	assert_total_charge( aa_sep, 1, -2.0);

//KMa  first sidechain atom
	first_scatom( aa_sep,1) = 5; //CB

//KMa   template atoms used for placing atoms SEP
//KMa   template for building  HA
	ta(1, 12,aa_sep,1) =  2; //   CA
	ta(2, 12,aa_sep,1) =  1; //   N
	ta(3, 12,aa_sep,1) =  3; //   C

//KMa   template for building 2HB
	ta(1,13,aa_sep,1) =  5; //   CB
	ta(2,13,aa_sep,1) =  2; //   CA
	ta(3,13,aa_sep,1) =  6; //   OG

//KMa   template for building 3HB
	ta(1,14,aa_sep,1) =  5; //   CB//KOSTA
	ta(2,14,aa_sep,1) =  2; //   CA
	ta(3,14,aa_sep,1) =  6; //   OG

//KMa   template for building  P
	ta(1, 7, aa_sep,1) =  6; //   OG//KOSTA
	ta(2, 7, aa_sep,1) =  8; //   OP//KOSTA
	ta(3, 7, aa_sep,1) =  9; //   OP//KOSTA

//KMa   template for building  O1P
	ta(1, 8, aa_sep,1) =  7;  //   P
	ta(2, 8, aa_sep,1) =  9;  //   O1P
	ta(3, 8, aa_sep,1) =  10; //   O2P

//KMa   template for building  O2P
	ta(1, 9, aa_sep,1) =  7; //   P
	ta(2, 9, aa_sep,1) =  8; //   O1P
	ta(3, 9, aa_sep,1) =  10; //  O3P

//KMa   template for building  O3P
	ta(1, 10, aa_sep,1) =  7; //   P
	ta(2, 10, aa_sep,1) =  8; //   O1P
	ta(3, 10, aa_sep,1) =  9; //   O2P

//KMa   chi angles required to build atoms SEP
//KMa   chi angles needed for building  OG
	chi_required(1, 6,aa_sep,1) =  true;

//KMa   chi angles needed for building 1HB
	chi_required(1,13,aa_sep,1) =  true;

//KMa   chi angles needed for building 2HB
	chi_required(1,14,aa_sep,1) =  true;

//KMa   chi angles needed for building  P
	chi_required(1, 7,aa_sep,1) =  true;
	chi_required(2, 7,aa_sep,1) =  true;

//KMa   chi angles needed for building  O1P
	chi_required(1, 8,aa_sep,1) =  true;
	chi_required(2, 8,aa_sep,1) =  true;
	chi_required(3, 8,aa_sep,1) =  true;

//KMa   chi angles needed for building  O2P
	chi_required(1, 9,aa_sep,1) =  true;
	chi_required(2, 9,aa_sep,1) =  true;
	chi_required(3, 9,aa_sep,1) =  true;

//KMa   chi angles needed for building  O3P
	chi_required(1, 10,aa_sep,1) =  true;
	chi_required(2, 10,aa_sep,1) =  true;
	chi_required(3, 10,aa_sep,1) =  true;


//KMa   four atoms that define chi angle  1
	chi_atoms( 1, 1,aa_sep, 1) =  1; //   N
	chi_atoms( 2, 1,aa_sep, 1) =  2; //   CA
	chi_atoms( 3, 1,aa_sep, 1) =  5; //   CB
	chi_atoms( 4, 1,aa_sep, 1) =  6; //   OG
//KMa   four atoms that define chi angle  2
	chi_atoms( 1, 2,aa_sep, 1) =  2; //   CA
	chi_atoms( 2, 2,aa_sep, 1) =  5; //   CB
	chi_atoms( 3, 2,aa_sep, 1) =  6; //   OG
	chi_atoms( 4, 2,aa_sep, 1) =  7; //   P

//KMa   four atoms that define chi angle  3
	chi_atoms( 1, 3,aa_sep, 1) =  5; //   CB
	chi_atoms( 2, 3,aa_sep, 1) =  6; //   OG
	chi_atoms( 3, 3,aa_sep, 1) =  7; //   P
	chi_atoms( 4, 3,aa_sep, 1) =  8; //   O1P

//KMa   base atom numbers SEP
	atom_base( 1,aa_sep,1) = 2; //  N  -> CA
	atom_base( 2,aa_sep,1) = 1; //  CA -> N
	atom_base( 3,aa_sep,1) = 2; //  C  -> CA
	atom_base( 4,aa_sep,1) = 3; //  O  -> C
	atom_base( 5,aa_sep,1) = 2; //  CB -> CA
	atom_base( 6,aa_sep,1) = 5; //  OG  on  CB
 	abase2( 6, aa_sep, 1) =  7; //  P  base2 for  OG
	atom_base( 7,aa_sep,1) = 6; // P  on  OG
	atom_base( 8,aa_sep,1) = 7; // OP  on  P
	atom_base( 9,aa_sep,1) = 7; // OP  on  P
	atom_base(10,aa_sep,1) = 7; // OP  on  P
	atom_base(11,aa_sep,1) = 1; //  H   on  N
	atom_base(12,aa_sep,1) = 2; //  HA  on  CA
	atom_base(13,aa_sep,1) = 5; // 1HB  on  CB
	atom_base(14,aa_sep,1) = 5; // 2HB  on  CB

//KMa Hydrogens and Acceptors
	HNpos(aa_sep,1) =  11; // atom number for backbone HN
	HApos(aa_sep,1) =  12; // atom number for backbone HA
	nH_polar(aa_sep,1) =  1; // number of polar hydrogens
	nH_aromatic(aa_sep,1) =  0; // number of aromatic hydrogens
	nH_apolar(aa_sep,1) =  3; // number of apolar hydrogens
	nacceptors(aa_sep,1) =  5; // number of acceptors

//KMa   atom numbers for polar H
	Hpos_polar( 1,aa_sep,1) =  11; //  H

//KMa   atom number for apolar hydrogens
	Hpos_apolar( 1,aa_sep,1) = 12; //  HA
	Hpos_apolar( 2,aa_sep,1) = 13; // 1HB
	Hpos_apolar( 3,aa_sep,1) = 14; // 2HB

//KMa   atoms hydrogens are connected to

//KMa   acceptor positions
	accpt_pos( 1,aa_sep, 1) =  4; //  O
	accpt_pos( 2,aa_sep, 1) =  6; //  OG
	accpt_pos( 3,aa_sep, 1) =  8; //  O1P
	accpt_pos( 4,aa_sep, 1) =  9; //  O2P
	accpt_pos( 5,aa_sep, 1) =  10; // O3P

//KMa   hydrogens on each atom
	nhydrogens_on_atm( 1,aa_sep, 1) =  1; //  N
	hydrogens_on_atm( 1, 1,aa_sep, 1) =  11; //  H
	nhydrogens_on_atm( 2,aa_sep, 1) =  1; //  CA
	hydrogens_on_atm( 1, 2,aa_sep, 1) =  12; //  HA
	nhydrogens_on_atm( 5,aa_sep, 1) =  2; //  CB
	hydrogens_on_atm( 1, 5,aa_sep, 1) = 13; // 2HB
	hydrogens_on_atm( 2, 5,aa_sep, 1) = 14; // 3HB


//KMa   template coordinates for the amino acid
//KMa new template from Protein Data Bank eg 1BKX and existing SER
	icoor( 1, 1,aa_sep, 1) =  -0.363; //   N
	icoor( 2, 1,aa_sep, 1) =   1.683; //   N
	icoor( 3, 1,aa_sep, 1) =  -1.441; //   N

	icoor( 1, 2,aa_sep, 1) =  -0.426; //   CA
	icoor( 2, 2,aa_sep, 1) =   0.913; //   CA
	icoor( 3, 2,aa_sep, 1) =  -0.205; //   CA

	icoor( 1, 3,aa_sep, 1) =  -0.289; //   C
	icoor( 2, 3,aa_sep, 1) =   1.817; //   C
	icoor( 3, 3,aa_sep, 1) =   1.015; //   C

	icoor( 1, 4,aa_sep, 1) =  -0.149; //   O
	icoor( 2, 4,aa_sep, 1) =   3.033; //   O
	icoor( 3, 4,aa_sep, 1) =   0.885; //   O

	icoor( 1, 5,aa_sep, 1) =   0.783; //   CB
	icoor( 2, 5,aa_sep, 1) =  -0.020; //   CB
	icoor( 3, 5,aa_sep, 1) =  -0.318; //   CB

	icoor( 1, 6,aa_sep, 1) =   0.886; //   OG
	icoor( 2, 6,aa_sep, 1) =  -0.861; //   OG
	icoor( 3, 6,aa_sep, 1) =   0.816; //   OG

	icoor( 1,7,aa_sep, 1) =  -0.079; //  P
	icoor( 2,7,aa_sep, 1) =  -2.094; //  P
	icoor( 3,7,aa_sep, 1) =   0.631; //  P

	icoor( 1,8,aa_sep, 1) =  -1.006; //  OP
	icoor( 2,8,aa_sep, 1) =  -1.814; //  OP
	icoor( 3,8,aa_sep, 1) =  -0.481; //  OP

	icoor( 1,9,aa_sep, 1) =   0.724; //  OP
	icoor( 2,9,aa_sep, 1) =  -3.305; //  OP
	icoor( 3,9,aa_sep, 1) =   0.330; //  OP

	icoor( 1,10,aa_sep, 1) =  -0.840; //  OP
	icoor( 2,10,aa_sep, 1) =  -2.300; //  OP
	icoor( 3,10,aa_sep, 1) =   1.880; //  OP

	icoor( 1, 11,aa_sep, 1) =  -0.267; //   H
	icoor( 2, 11,aa_sep, 1) =   2.685; //   H
	icoor( 3, 11,aa_sep, 1) =  -1.377; //   H

	icoor( 1, 12,aa_sep, 1) =  -1.328; //   HA
	icoor( 2, 12,aa_sep, 1) =   0.306; //   HA
	icoor( 3, 12,aa_sep, 1) =  -0.118; //   HA

	icoor( 1,13,aa_sep, 1) =   0.669; //  1HB
	icoor( 2,13,aa_sep, 1) =  -0.629; //  1HB
	icoor( 3,13,aa_sep, 1) =  -1.215; //  1HB

	icoor( 1, 14,aa_sep, 1) =   1.685; //   2HB
	icoor( 2, 14,aa_sep, 1) =   0.586; //   2HB
	icoor( 3, 14,aa_sep, 1) =  -0.405; //   2HB

	// create N and Cterm variants
	create_termini_variants( aa_sep );

//KMa  create variants that have h2o connected to the backbone nh and carbonyl
	if ( explicit_h2o ) {
		create_backbone_h2o_variants(aa_sep);
		create_sidechain_h2o_variants(aa_sep);
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_aaproperties
///
/// @brief Initial call zeroes out amino acid arrays; subsequent calls fill
/// information arrays for each amino acid type
///
/// @detailed
///
/// @global_read - none
///
/// @global_write - none
///
/// @remarks
///  amino acid residues are initially named according to IUPAC conventions
///  (ie comments in init_cys etc refer to IUPAC names for aliphatic hydrogens
///      and atom names are using hydrogen conventions)
///  init_aliphatic_protons selects a naming convention to be used
///  current default (6/3/2004) is the PDB convention; IUPAC convention can be selected by
///  the command line option -IUPAC
///
/// @references
///
/// @authors
/// Chuck Duarte
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
initialize_aaproperties()
{
	//------------------------------------------------------------------------------
	zero_aaproperties();

	init_N_terminus();
	init_C_terminus();

	init_ala();
	init_cys();
	init_asp();
	init_glu();
	init_phe();
	init_gly();
	init_his();
	init_ile();
	init_lys();
	init_leu();
	init_met();
	init_asn();
	init_pro();
	init_gln();
	init_arg();
	init_ser();
	init_thr();
	init_val();
	init_trp();
	init_tyr();
	//KMa phospho_serine (SEP in PDBs)
	if ( add_pser() || dna_enabled() ) init_sep();
	if ( dna_enabled() ) {
		init_dna_gua();
		init_dna_ade();
		init_dna_cyt();
		init_dna_thy();
		init_rna_gua();
		init_rna_ade();
		init_rna_cyt();
		init_rna_ura();
	}
	init_aliphatic_proton();
	set_atom_mode( "initialize" );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin copy_aaproperties
///
/// @brief makes a complete copy of residue set-up information from one
///       variant set to another
///
/// @detailed
///
///\PARAM
///
/// AA - the amino acid/nucleic acid of interest
/// SRC_VAR - the variant that is the source of the information
/// DEST_VAR - the variant to recieve the information
///
/// @return
///
/// @global_read
///
/// @global_write
///
///     natoms            -  number of atoms per aa
///     nheavyatoms                -  number of heavy atoms (neither hydrogen nor water) per aa
///     nchi              -  number of needed chi angles per aa
///     HNpos                      -  backbone HN position
///     HApos                      -  hydrogen atttached to Ca atom
///     nH_polar          -  number of polar H
///     nH_aromatic       -  number of aromatic H
///     nH_apolar         -  number of apolar H
///     nh2o              -  number of water
///     atom_name         -  atom name
///     atom_base         -  base for atom (base is alwyas heavy atom for H)
///     fullatom_type     -  fullatom type
///     Hpos_polar        -  position of  polar H
///     Hpos_apolar       -  position of aromatic H
///     Hpos_aromatic     -  position of apolar H
///     h2opos            -  position of water
///     nh2o_on_atm                -  how many water are attached to each atom
///     accpt_pos                  -  hbond acceptor positions
///     abase2            - second base for acceptors that have them (sp3 & ring)
///     nhydrogens_on_atm -  how many hydrogens are attached to each atom
///     hydrogens_on_atm           -  what hydrogens are connected to each atom
///     ta                         -  template atoms for building each hdyrogen and water
///     icoor             -  template coordinates of each atom pre aa
///     h2o_on_atm                 -  what waters are connected to each atom
///     chi_required      -  chi angles needed for building each hdyrogen and water
///     chi_atoms                  -  four atoms which define each chi angle
///
/// @remarks
///
/// @references
///
/// @authors Jim Havranek  01/26/04
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
copy_aaproperties(
									int const aa,
									int const src_var,
									int const dest_var
									)
{
	using namespace aaproperties_pack;
	using namespace param;
	using namespace param_aa;

	natoms(aa,dest_var) = natoms(aa,src_var);
	first_scatom(aa,dest_var) = first_scatom(aa,src_var);
	nheavyatoms(aa,dest_var) = nheavyatoms(aa,src_var);
	nchi(aa,dest_var) = nchi(aa,src_var);
	nH_polar(aa,dest_var) = nH_polar(aa,src_var);
	nH_apolar(aa,dest_var) = nH_apolar(aa,src_var);
	nH_aromatic(aa,dest_var) = nH_aromatic(aa,src_var);
	nh2o(aa,dest_var) = nh2o(aa,src_var);
	nacceptors(aa,dest_var) = nacceptors(aa,src_var);
	HNpos(aa,dest_var) = HNpos(aa,src_var);
	HApos(aa,dest_var) = HApos(aa,src_var);

	for ( int i = 1; i <= number_aav_type; ++i ){
		variant_type(i, aa,dest_var) = variant_type(i, aa,src_var);
	}

	for ( int i = 1; i <= 3; ++i ){
		na_anchor(i, aa,dest_var) = na_anchor(i, aa,src_var);
	}

	for ( int i = 1, max_atom = MAX_ATOM(); i <= max_atom; ++i ) {
		atom_name(i,aa,dest_var) = atom_name(i,aa,src_var);
		atom_base(i,aa,dest_var) = atom_base(i,aa,src_var);
		abase2(i,aa,dest_var) = abase2(i,aa,src_var);
		accpt_pos(i,aa,dest_var) = accpt_pos(i,aa,src_var);
		Hpos_polar(i,aa,dest_var) = Hpos_polar(i,aa,src_var);
		Hpos_apolar(i,aa,dest_var) = Hpos_apolar(i,aa,src_var);
		Hpos_aromatic(i,aa,dest_var) = Hpos_aromatic(i,aa,src_var);
		fullatom_type(i,aa,dest_var) = fullatom_type(i,aa,src_var);
		h2opos(i,aa,dest_var) = h2opos(i,aa,src_var);
		atomic_charge(i,aa,dest_var) = atomic_charge(i,aa,src_var);
		nh2o_on_atm(i,aa,dest_var) = nh2o_on_atm(i,aa,src_var);
		template_atm_on_h2o(i,aa,dest_var) = template_atm_on_h2o(i,aa,src_var);
		heavy_atm_on_h2o(i,aa,dest_var) = heavy_atm_on_h2o(i,aa,src_var);
		hyd_atm_on_h2o(i,aa,dest_var) = hyd_atm_on_h2o(i,aa,src_var);
		nhydrogens_on_atm(i,aa,dest_var) = nhydrogens_on_atm(i,aa,src_var);
		nbonded_neighbors(i,aa,dest_var) = nbonded_neighbors(i,aa,src_var);
		nangle_neighbors(i,aa,dest_var) = nangle_neighbors(i,aa,src_var);
		for ( int k = 1; k <= 3; ++k ) {
			ta(k,i,aa,dest_var) = ta(k,i,aa,src_var);
			icoor(k,i,aa,dest_var) = icoor(k,i,aa,src_var);
			hydrogens_on_atm(k,i,aa,dest_var) = hydrogens_on_atm(k,i,aa,src_var);
		}

		for ( int k = 1; k <= 5; ++k ) {
			h2o_on_atm(k,i,aa,dest_var) = h2o_on_atm(k,i,aa,src_var);
		}
		for ( int k = 1; k <= max_bonded_neighbors; ++k ) {
			bonded_neighbor(k,i,aa,dest_var) = bonded_neighbor(k,i,aa,src_var);
		}
		for ( int k = 1; k <= max_angle_neighbors; ++k ) {
			angle_neighbor(k,i,aa,dest_var) = angle_neighbor(k,i,aa,src_var);
		}
		for ( int k = 1; k <= MAX_CHI; ++k ) {
			chi_required(k,i,aa,dest_var) = chi_required(k,i,aa,src_var);
		}
	}
	for ( int i = 1; i <= MAX_CHI; ++i ) {
		for ( int k = 1; k <= 4; ++k ) {
			chi_atoms(k,i,aa,dest_var) = chi_atoms(k,i,aa,src_var);
		}
	}

}

