// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 7630 $
//  $Date: 2006-03-10 09:37:52 -0800 (Fri, 10 Mar 2006) $
//  $Author: stuartm $


// Rosetta Headers
#include "rotamer_functions.h"
#include "angles.h"
#include "aaproperties_pack.h"
#include "current_pose.h"
#include "design.h"
#include "DesignMap.h"
#include "dna.h"
#include "dunbrack_pack.h"
#include "Dunbrack4D.h"
#include "Dunbrack5D.h"
#include "enzyme.h" //get_enable_lig_aa flag
#include "files_paths.h"
#include "jumping_util.h" // put_nh_pose
#include "ligand.h" //get_ligand_flag
#include "misc.h"
#include "pack.h"
#include "pack_geom_inline.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "pH_main.h"
#include "random_numbers.h"
#include "read_aaproperties.h"
#include "RotamerOptions.h"
#include "rotamer_explosion.h"
#include "termini.h"
#include "util_vector.h"
#include "water.h"
#include "water_ns.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Ds.hh>
#include <ObjexxFCL/FArray2Ds.hh>
#include <ObjexxFCL/FArray3Ds.hh>
#include <ObjexxFCL/FArray4Ds.hh>
#include <ObjexxFCL/FArray5Ds.hh>
#include <ObjexxFCL/formatted.io.hh>

// Numeric Headers
#include <numeric/conversions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>

#include <vector>
#include <algorithm> // std::difference

// Using
using namespace param;

namespace rotamer_minimization {
	// chu
	FArray1D_float sc_chi( MAX_CHI ); // all the chi angles for each rotamer tried.
	FArray1D_float start_chi( MAX_CHI );
	FArray2D_float sc_rotcoord( 3, MAX_ATOM() ); // the coord for this rotamer
	FArray1D_float sc_rot_born_radius( MAX_ATOM() );
	FArray1D_float sc_rotactcoord( 3 );
	FArray1D_int sc_rotid( MAX_CHI ); // all the rotamer id ...
	int sc_seqpos; // sequence position of the current residue with side-chain minimized
	int sc_aa; // residue type of ...
	int sc_aav; // variant type
	int sc_nfree; // num of freedom on this side-chain
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_extrachi
///
/// @brief
/// get extra chi angles for any single rotamer, where a single
/// rotamer is any unique sequence of rotamer numbers
///
/// @detailed
///
/// @param  aa - [in/out]? -
/// @param  aav - [in/out]? -
/// @param  base_chi - [in/out]? -
/// @param  base_sd - [in/out]? -
/// @param  neighbors - [in/out]? -
/// @param  total_chi_set - [in/out]? -
/// @param  total_rot_set - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void get_extrachi(
	int const aa,
	int const aav,
	FArray1Da_float base_chi,
	FArray1Da_float base_sd,
	int & neighbors,
	std::list< FArray1D_float > & total_chi_set,
	std::list< FArray1D_int > & total_rot_set,
	int const seqpos,
	const DesignMap & design_map
)
{

	using namespace design;
	using namespace param;

	base_chi.dimension( MAX_CHI );
	base_sd.dimension( MAX_CHI );

	bool is_buried = false;
	if ( neighbors >= active_rotamer_options.extrachi_cutoff ) is_buried = true;

	if ( active_rotamer_options.exrotset_from_db ) {
		// ctsa - create ex rots from an input db
		get_exrotset_from_db(aa,base_chi,base_sd,is_buried,
		 total_chi_set,total_rot_set);

	} else {
		// otherwise create ex rots from 'ex flags'
		calc_exrotset_from_exflags(aa,aav,base_chi,base_sd,is_buried,
		 total_chi_set,total_rot_set,seqpos,design_map);

	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_exrotset_from_db
///
/// @brief build extra rotamers from base rotamers using exdb values
///
/// @detailed
///
/// @param  aa - [in/out]? -
/// @param  base_chi - [in/out]? -
/// @param  base_sd - [in/out]? -
/// @param  is_buried - [in/out]? -
/// @param  total_chi_set - [in/out]? -
/// @param  total_rot_set - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void get_exrotset_from_db(
	int const aa,
	FArray1Da_float base_chi,
	FArray1Da_float base_sd,
	bool const is_buried,
	std::list< FArray1D_float > & total_chi_set,
	std::list< FArray1D_int > & total_rot_set
)
{

	using namespace dunbrack_pack;
	using namespace param;
	using namespace param_aa;

	base_chi.dimension( MAX_CHI );
	base_sd.dimension( MAX_CHI );

	//  ctsa - retrieve the extra rotamer set from a database
	//     of extra rots for each aa

	int total_tmp_hchi;
	FArray1D_float tmp_hchi( MAXANGLES_PER_CHI );

	total_chi_set.clear();
	total_rot_set.clear();

	// include base rotamer
	FArray1D_float new_chi_set( MAX_CHI, 0. );
	FArray1D_int new_rot_set( MAX_CHI, 0 );
	for ( int i = 1; i <= MAX_CHI; ++i ) {
		new_chi_set(i) = base_chi(i);
	}
	rotamer_from_chi( new_chi_set, aa, new_rot_set );
	total_chi_set.push_back(new_chi_set);
	total_rot_set.push_back(new_rot_set);

	if ( is_buried ) {
		//  add exdb rotamers for buried positions
		for( std::list<FArray1D_float>::iterator chiset_db = (total_chi_set_db[aa]).begin();
				 chiset_db != (total_chi_set_db[aa]).end(); ++chiset_db ) {
			for ( int j = 1; j <= MAX_CHI; ++j ) {
				new_chi_set(j) = base_chi(j) + (*chiset_db)(j) * base_sd(j);
			}
			// ctsa - add extra symmetry check plus rotamer reassignment based
			// on chu's finding that phe and tyr rot 2 step out of range @ 1 sd
			set_all_chi_to_periodic_range(new_chi_set,aa);
			rotamer_from_chi( new_chi_set, aa, new_rot_set );
			total_chi_set.push_back(new_chi_set);
			total_rot_set.push_back(new_rot_set);
		}
	}

	// ctsa - add 'extra' h rotamers for ser,thr & tyr

	// ctsa - for now, ignore previous extra rot system for h's
	// (previously, this was implemented as -ex2 = ser,thr and -ex3 = tyr)
	bool is_extra = false;

	if ( aa == aa_ser || aa == aa_thr ) {

		get_ser_thr_hchi(is_extra,total_tmp_hchi,tmp_hchi);

		new_chi_set = 0.;
		new_rot_set = 0;

		std::list<FArray1D_float>::iterator orig_chiset_end = total_chi_set.end();
		for( std::list<FArray1D_float>::iterator chiset = total_chi_set.begin();
				 chiset != orig_chiset_end; ++chiset ) {
			(*chiset)(2) = tmp_hchi(1);
		}
		std::list<FArray1D_int>::iterator rotset = total_rot_set.begin();
		for( std::list<FArray1D_float>::iterator chiset = total_chi_set.begin();
				 chiset != orig_chiset_end; ++chiset, ++rotset ) {
			new_chi_set(1) = (*chiset)(1);
			for ( int j = 2; j <= total_tmp_hchi; ++j ) {
				new_chi_set(2) = tmp_hchi(j);
				new_rot_set = *rotset;
				total_chi_set.push_back(new_chi_set);
				total_rot_set.push_back(new_rot_set);
			}
		}

	} else if ( aa == na_rgu || aa == na_rad || aa == na_rcy || aa == na_ura ) {

		get_rna_hchi(is_extra,total_tmp_hchi,tmp_hchi);

		new_chi_set = 0.;
		new_rot_set = 0;

		std::list<FArray1D_float>::iterator orig_chiset_end = total_chi_set.end();
		for( std::list<FArray1D_float>::iterator chiset = total_chi_set.begin();
				 chiset != orig_chiset_end; ++chiset ) {
			(*chiset)(1) = tmp_hchi(1);
		}
		std::list<FArray1D_int>::iterator rotset = total_rot_set.begin();
		for( std::list<FArray1D_float>::iterator chiset = total_chi_set.begin();
				 chiset != orig_chiset_end; ++chiset, ++rotset ) {
			for ( int j = 2; j <= total_tmp_hchi; ++j ) {
				new_chi_set(1) = tmp_hchi(j);
				new_rot_set = *rotset;
				total_chi_set.push_back(new_chi_set);
				total_rot_set.push_back(new_rot_set);
			}
		}

	} else if ( aa == aa_tyr ) {

		get_tyr_hchi(is_extra,total_tmp_hchi,tmp_hchi);

		new_chi_set = 0.;
		new_rot_set = 0;

		std::list<FArray1D_float>::iterator orig_chiset_end = total_chi_set.end();
		for( std::list<FArray1D_float>::iterator chiset = total_chi_set.begin();
				 chiset != orig_chiset_end; ++chiset ) {
			(*chiset)(3) = tmp_hchi(1);
		}
		std::list<FArray1D_int>::iterator rotset = total_rot_set.begin();
		for( std::list<FArray1D_float>::iterator chiset = total_chi_set.begin();
				 chiset != orig_chiset_end; ++chiset, ++rotset ) {
			new_chi_set(1) = (*chiset)(1);
			new_chi_set(2) = (*chiset)(2);
			for ( int j = 2; j <= total_tmp_hchi; ++j ) {
				new_chi_set(3) = tmp_hchi(j);
				new_rot_set = *rotset;
				total_chi_set.push_back(new_chi_set);
				total_rot_set.push_back(new_rot_set);
			}
		}

	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin calc_exrotset_from_exflags
///
/// @brief
/// calculate the extra rotamer set from the command line -ex flags,
/// assuming full combinatorics between the selections
///
/// @detailed
/// if you would like to add new chi sampling options, change three files:
/// namespace_exchi_flags.h namespace_exchi_flags.cc and rotamer_functions.cc
/// add new chi sampling options to the end of the enumeration list, or
/// the meaning of "-ex1 6" will change and that will upset people.
///
/// @authors
///
////////////////////////////////////////////////////////////////////////////////
void calc_exrotset_from_exflags(
	int const aa,
	int const aav,
	FArray1Da_float base_chi,
	FArray1Da_float base_sd,
	bool const is_buried,
	std::list< FArray1D_float > & total_chi_set,
	std::list< FArray1D_int > & total_rot_set,
	int const seqpos,
	DesignMap const & design_map
)
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace exchi_flags;
	using namespace param;
	using namespace param_aa;
	using namespace param_pack;

	base_chi.dimension( MAX_CHI );
	base_sd.dimension( MAX_CHI );

	// local
	// default value for total_chi_count is 1 for all chi angles to allow
	// the final permutation loop to operate correctly
	FArray1D_int total_chi_count( MAX_CHI, 1 );
	FArray2D_float total_chi( MAXANGLES_PER_CHI, MAX_CHI, 0.0 );
	FArray2D_bool screen_chi( MAXANGLES_PER_CHI, MAX_CHI, false );
	FArray1D_float tmp_hchi( MAXANGLES_PER_CHI );
	int total_tmp_hchi;
	bool rot_explosion_prescreen = true;

	// determine which extrachi set to use
	int const burial_code( is_buried ? 2 : 1 );

	// jk changes the values of base_sd in order to work in explicit angle space
	if ( ( active_rotamer_options.rot_pert && is_buried ) ) {
		base_sd(1) = active_rotamer_options.pert_size;
		base_sd(2) = active_rotamer_options.pert_size;
	}

	// setup default and extra chi angles
	for ( int chinum(1); chinum <= nchi(aa,aav); ++chinum ) {

		total_chi(1,chinum) = base_chi(chinum);

		// skip extra angles if sd is insignificant
		if ( base_sd(chinum) <= min_extrachi_sd ) continue;

		ExtraRotSample extrachi_type(
			active_rotamer_options.extrachi_flag(aa,chinum,burial_code)
		);

		std::vector<float> extra_steps;

		if ( extrachi_type != NO_EXTRA_CHI_SAMPLES ) {
			// default: use standard deviation factors based on command-line options
			extra_steps = active_rotamer_options.ex_step_set( extrachi_type );

		} else if ( active_rotamer_options.rot_pert &&
		            is_buried && ( chinum < 3 ) ) {
			// jk's rot_pert
			for ( int i(1); i >= -1; i-=2 ) {
				if ( active_rotamer_options.pert_acc_prob > ran3() ) {
					extra_steps.push_back( i );
				}
			}
		}
		// write standard extra angles to total_chi
		for ( std::vector<float>::const_iterator step( extra_steps.begin() );
		      step != extra_steps.end(); ++step ) {
			total_chi_count(chinum) += 1;
			total_chi( total_chi_count(chinum), chinum ) =
				base_chi(chinum) + (*step) * base_sd(chinum);
		}

		if ( active_rotamer_options.rot_explode ) {
			// add an additional 'exploded' set of angles that will be filtered below
			// uses the standard command-line option step sets
			ExtraRotSample explode_type(
				active_rotamer_options.explode_flag(aa,chinum,burial_code)
			);
			// full set of steps from option
			std::vector<float> allsteps(
				active_rotamer_options.ex_step_set( explode_type )
			);
			// storage for the unique set
			std::vector<float> explode_steps( allsteps.size() );
			// pare down exploded steps to those not already present in extra_steps
			// (return iterator points to the end of the unique set)
			std::sort( allsteps.begin(), allsteps.end() );
			std::sort( extra_steps.begin(), extra_steps.end() );
			std::vector<float>::const_iterator explend(
				std::set_difference(
				  allsteps.begin(), allsteps.end(),
				  extra_steps.begin(), extra_steps.end(),
				  explode_steps.begin() )
			);
			// write (any) new exploded angles
			for ( std::vector<float>::const_iterator step( explode_steps.begin() );
			      step != explend; ++step ) {
				total_chi_count(chinum) += 1;
				total_chi( total_chi_count(chinum), chinum ) =
				                        base_chi(chinum) + (*step) * base_sd(chinum);
				screen_chi( total_chi_count(chinum), chinum ) = true;
			}
		}

// preserved for debugging purposes
//		std::cout << seqpos << "-" << aa << " " << chinum << ": ";
//		for ( int j(1); j <= total_chi_count(chinum); ++j ) {
//			std::cout << total_chi_count(chinum) << " " <<
//			             total_chi(j,chinum) << ", ";
//		}
//    std::cout << std::endl;

	}

//cy extra angles for chi1, special for rna 2'-OH
	if ( aa == na_rgu || aa == na_rad || aa == na_rcy || aa == na_ura ) { //rna - 2OH*
		bool is_extra = false;
		if ( active_rotamer_options.extrachi_flag(aa,1,burial_code) != NO_EXTRA_CHI_SAMPLES ) is_extra = true;

		get_rna_hchi(is_extra,total_tmp_hchi,tmp_hchi);
		total_chi_count(1) = total_tmp_hchi;
		for ( int i = 1; i <= total_tmp_hchi; ++i ) {
			total_chi(i,1) = tmp_hchi(i);
		}
	}

//bk extra angles for chi2, special for some amino acids
	if ( aa == aa_ser || aa == aa_thr ) { // Ser,Thr - hydrogens
		bool is_extra = false;
		if ( active_rotamer_options.extrachi_flag(aa,2,burial_code) != NO_EXTRA_CHI_SAMPLES ) is_extra = true;

		get_ser_thr_hchi(is_extra,total_tmp_hchi,tmp_hchi);
		total_chi_count(2) = total_tmp_hchi;
		for ( int i = 1; i <= total_tmp_hchi; ++i ) {
			total_chi(i,2) = tmp_hchi(i);
		}
	}

	//bk extra rotamers for chi3, specific for different types of aa

	// apl -- the following comment seems disconnected from the code;
	// ctsa- old ex4 gave 6 angles for gln chi3, but now this has been
	//  reduced to 4 by default without any extra option yet.
	//  all other previously 'extra' chi3 and 4 angles should be
	//  integrated in by default
	if ( aa == aa_tyr ) { // Tyr,hydrogens
		bool is_extra = false;
		if ( active_rotamer_options.extrachi_flag(aa,3,burial_code) != NO_EXTRA_CHI_SAMPLES ) is_extra = true;

		get_tyr_hchi(is_extra,total_tmp_hchi,tmp_hchi);
		total_chi_count(3) = total_tmp_hchi;
		for ( int i = 1; i <= total_tmp_hchi; ++i ) {
			total_chi(i,3) = tmp_hchi(i);
		}
	}

	// rh - pH-sensitive rotamers
	// Extra Hydrogen rotamers for protonated variants of asp and glu
	// nchi(aa,aav) changes depending on protonation state.
	// For deprotonated ASP, nchi = 2. For protonated ASP, nchi = 3.
	// This is a temporary solution until an explicit variant
	// book keeping method is implimented (most likely with rosetta3)
	if ( aa == aa_asp && nchi(aa,aav) == 3 ) { // ASP, side chain OOH hydrogen
		bool is_extra = false;
		if ( active_rotamer_options.extrachi_flag(aa,3,burial_code) != NO_EXTRA_CHI_SAMPLES ) is_extra = true;

		get_ser_thr_hchi(is_extra,total_tmp_hchi,tmp_hchi);
		total_chi_count(3) = total_tmp_hchi;
		for ( int i = 1; i <= total_tmp_hchi; ++i ) {
			total_chi(i,3) = tmp_hchi(i);
		}
	}

	if ( aa == aa_glu && nchi(aa,aav) == 4 ) { // GLU, side chain OOH hydrogen
		bool is_extra = false;
		if ( active_rotamer_options.extrachi_flag(aa,4,burial_code) != NO_EXTRA_CHI_SAMPLES  ) is_extra = true;

		get_ser_thr_hchi(is_extra,total_tmp_hchi,tmp_hchi);
		total_chi_count(4) = total_tmp_hchi;
		for ( int i = 1; i <= total_tmp_hchi; ++i ) {
			total_chi(i,4) = tmp_hchi(i);
		}
	}

	if ( active_rotamer_options.rot_explode_accept_all ) screen_chi = false;

	//to make rotamer_explosion faster, prescreen the standard rotamer whether it is in the vicinity
	//of any ligand donor or acceptor
	if ( active_rotamer_options.rot_explode ) {

		rot_explosion_prescreen = prescreen_base_rotamer_ligand_contacts ( seqpos, aa, aav, base_chi );
	}

	// convert chi-encoded extra rotamer form to new flat format
	total_chi_set.clear();
	total_rot_set.clear();
	FArray1D_float new_chi_set( MAX_CHI, 0. );
	FArray1D_int new_rot_set( MAX_CHI, 0 );

	// build new rotamers by permutation over the extra angles
	for ( int chi1_type(1); chi1_type <= total_chi_count(1); ++chi1_type ) {
		for ( int chi2_type(1); chi2_type <= total_chi_count(2); ++chi2_type ) {
			for ( int chi3_type(1); chi3_type <= total_chi_count(3); ++chi3_type ) {
				for ( int chi4_type(1); chi4_type <= total_chi_count(4); ++chi4_type ) {

					new_chi_set(1) = total_chi(chi1_type,1);
					new_chi_set(2) = total_chi(chi2_type,2);
					new_chi_set(3) = total_chi(chi3_type,3);
					new_chi_set(4) = total_chi(chi4_type,4);

					if ( active_rotamer_options.rot_explode ) {
						// screen exploded rotamers
						if ( ( screen_chi(chi1_type,1) || screen_chi(chi2_type,2) ||
					         screen_chi(chi3_type,3) || screen_chi(chi4_type,4) ) ) {
							if ( !screen_exploded_rotamers( seqpos, aa, aav,
								                     new_chi_set, design_map, rot_explosion_prescreen ) ) continue;
						}
					}

					set_all_chi_to_periodic_range( new_chi_set, aa );
					rotamer_from_chi( new_chi_set, aa, new_rot_set );
					total_chi_set.push_back( new_chi_set );
					total_rot_set.push_back( new_rot_set );
				}
			}
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin screen_exploded_rotamers
///
/// @brief wrapper f
///
/// @detailed
///
/// @param[in]   aa - in - ...
/// @param[in]   chino - in - ...
///
/// @return true if aa-chino refers to the dihedral of a proton rotamer
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors flo! july 2007
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
screen_exploded_rotamers(
	int const seqpos,
	int const aa,
	int const aav,
	FArray1DB_float & new_chi_set,
	DesignMap const & design_map,
	bool const rot_explosion_prescreen
)
{

	bool accept = false;

	//screening dna contacts
	if ( dna_enabled() ) {
		accept = screen_exploded_rotamers_dna( seqpos, aa, aav,
																				 new_chi_set, design_map );

	}	else if ( (get_enable_ligaa_flag() || get_ligand_flag()) && !accept && rot_explosion_prescreen ) {
		// screening ligand contacts
		accept = screen_exploded_rotamers_ligand( seqpos, aa, aav,
																				 new_chi_set, design_map );
	}

	return accept;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin is_chi_proton_rotamer
///
/// @brief test to see if chi angle controls a proton rotamer
///
/// @detailed
///
/// @param[in]   aa - in - ...
/// @param[in]   chino - in - ...
///
/// @return true if aa-chino refers to the dihedral of a proton rotamer
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors ctsa 10-2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
is_chi_proton_rotamer(
	int aa,
	int aav,
	int chino
)
{
	using namespace aaproperties_pack;
	using namespace param_aa;

	bool is_chi_proton_rotamer = false; // Return value

	if ( chino == 2 && (aa == aa_ser || aa == aa_thr) ) {
		is_chi_proton_rotamer = true;
	} else if ( chino == 3 && aa == aa_tyr ) {
		is_chi_proton_rotamer = true;
	} else if ( chino == 1 &&
	 ( aa == na_rgu || aa == na_rad || aa == na_rcy || aa == na_ura ) ) {
		is_chi_proton_rotamer = true;
//rh If using extra pH rotamers - need to set the chi3 and chi4 of asp and glu
//rh to proton rotamer
	} else if ( get_pH_packing_flag() &&
	 ( (chino == 3 && aa == aa_asp && variant_type( aav_pH, aa, aav )) || (chino == 4 && aa == aa_glu && variant_type( aav_pH, aa, aav )))) {
		is_chi_proton_rotamer = true;
	}

	return is_chi_proton_rotamer;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin get_ser_thr_hchi
///
/// @brief specify hydrogen rotamers for ser and thr
///
/// @detailed Can also be used to set other sp2/sp3 hydrogen rotamers
///           Ex. Protonated Glutamate and Aspartate
///
/// returns the set of 'normal' and 'extra' angle values used for
/// the serine and threonine polar hydrogen dihedral
///
/// @param[in]   is_extra - in - return 'extra' angles if true
/// @param[out]  count - out - no of dihedral angles returned
/// @param[out]  chi - out - array of angles
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors ctsa 10-2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_ser_thr_hchi(
	bool const is_extra,
	int & count,
	FArray1Da_float chi
)
{
	using namespace param;

	chi.dimension( MAXANGLES_PER_CHI );

	// h pos on chi2 of ser/thr:
	// separated this b/c several functions need this outside
	// of get_extrachi

	if ( is_extra ) {
		count = 9;
		chi(1) = 60.0;
		chi(2) = 40.0;
		chi(3) = 80.0;
		chi(4) = 180.0;
		chi(5) = 160.0;
		chi(6) = 200.0;
		chi(7) = -60.0;
		chi(8) = -40.0;
		chi(9) = -80.0;
	} else {
		count = 3;
		chi(1) = 60.0;
		chi(2) = -60.0;
		chi(3) = 180.0;
	}

}

/////////////////////////////////////////////////////////////////////////
/// @begin get_rna_hchi
///
/// @brief specify hydrogen rotamers for na_rgu, na_rad, na_rcy, na_ura
/// by Yu Chen, Feb 2004
/// @detailed
///
/// returns the set of 'normal' and 'extra' angle values used for
/// the RNA 2'-OH polar hydrogen dihedral
/// J.M.B. 1997, v274, p54-63.
///
/// @param[in]   is_extra - in - return 'extra' angles if true
/// @param[out]  count - out - no of dihedral angles returned
/// @param[out]  chi - out - array of angles
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors 1-2004
///
/// @last_modified
//////////////////////////////////////////////////////////////////////////
void
get_rna_hchi(
	bool const is_extra,
	int & count,
	FArray1Da_float chi
)
{
	using namespace param;

	chi.dimension( MAXANGLES_PER_CHI );

	// h pos on chi1 of rna 2'-OH:
	// separated this b/c several functions need this outside
	// of get_extrachi
	//

	if ( is_extra ) {
		count = 9;
		chi(1) = 60.0;
		chi(2) = 40.0;
		chi(3) = 80.0;
		chi(4) = -130.0;
		chi(5) = -110.0;
		chi(6) = -150.0;
		chi(7) = -40.0;
		chi(8) = -20.0;
		chi(9) = -60.0;
	} else {
		count = 3;
		chi(1) = 60.0;
		chi(2) = -40.0;
		chi(3) = -130.0;
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin get_tyr_hchi
///
/// @brief specify hydrogen rotamers for tyr
///
/// @detailed
///
/// returns the set of 'normal' and 'extra' angle values used for
/// the tyrosine polar hydrogen dihedral
///
/// @param[in]   is_extra - in - return 'extra' angles if true
/// @param[out]  count - out - no of dihedral angles returned
/// @param[out]  chi - out - array of angles
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors ctsa 10-2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_tyr_hchi(
	bool const is_extra,
	int & count,
	FArray1Da_float chi
)
{
	using namespace param;

	chi.dimension( MAXANGLES_PER_CHI );

	// h pos on chi3 of tyr:
	// separated this b/c several functions need this outside
	// of get_extrachi

	if ( is_extra ) {
		count = 6;
		chi(1) = 0.0;
		chi(2) = 20.0;
		chi(3) = -20.0;
		chi(4) = 180.0;
		chi(5) = 200.0;
		chi(6) = 160.0;
	} else {
		count = 2;
		chi(1) = 0.0;
		chi(2) = 180.0;
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin get_rot_coord
///
/// @brief
///bk determines coordinates for rotamers superimposed on a pdb
///bk backbone
///
/// @detailed
///
/// @param  full_coord - [in/out]? - coordinates of backbone and existing sidechains
/// @param  res - [in/out]? - amino acid
/// @param  aav - [in/out]? -
/// @param  chi - [in/out]? - desired chi angles
/// @param  rotcoord - [in/out]? - rotamer coordinates
/// @param  seqpos - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_rot_coord(
	FArray3DB_float const & full_coord,
	int const aa,
	int const aav,
	FArray1Da_float chi,
	FArray2Da_float rotcoord,
	int const seqpos
)
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace param;
	using namespace param_aa;
	using numeric::conversions::radians;

	rotcoord.dimension( 3, MAX_ATOM() );

//bk local variables
	static FArray2D_float coor( 3, MAX_ATOM() );
	static FArray2D_float mat( 3, 3 );
	static FArray1D_float vec( 3 );
	static FArray2D_float rescoord( 3, MAX_ATOM() );
	static FArray1D_float nhcoord( 3 );
	static FArray2D_float Nterm_nhcoord( 3, 3 );
	static FArray2D_float Cterm_cocoord( 3, 2 );

	int atm1,atm2,atm3;
	float ichi, rot;
	float dis,rot_ang;
	int maxatom,c1,c2,c3,c4;
	int const chi_required_size1 = chi_required.size1();

	int & natoms_aa( natoms(aa,aav) );
	for ( int j = 1, lr = 0, lxyz = full_coord.index(1,j,seqpos); j <= natoms_aa; ++j ) {
		for ( int i = 1; i <= 3; ++i, ++lr, ++lxyz ) {
			rescoord[ lr ] = full_coord[ lxyz ]; // rescoord(i,j) = full_coord(i,j,seqpos);
		}
	}

//bk read in template coordinates for this amino acid
	for ( int j = 1, lc = 0, li = icoor.index(1,j,aa,aav); j <= natoms_aa; ++j ) {
		for ( int i = 1; i <= 3; ++i, ++lc, ++li ) {
			coor[ lc ] = icoor[ li ]; // coor(i,j) = icoor(i,j,aa,aav);
		}
	}

	// superimpose the rotamer on to the backbone
	// find the matrix/vector that lines up the NH and CA atoms
	// and apply this operation to all the atoms in the rotamer
	lineup_bk(coor(1,2),coor(1,1),rescoord(1,2),rescoord(1,1),mat,vec);
	maxatom = natoms(aa,aav);
	for ( int i = 1; i <= maxatom; ++i ) {
		move_bk(coor(1,i),mat,vec);
	}

	// find the matrix/vector that lines up the C = O carbons
	// and apply this operation to the rotamer
	align_bk(coor(1,1),coor(1,2),coor(1,3),rescoord(1,3),mat(1,1),vec(1));
	for ( int i = 3; i <= maxatom; ++i ) {
		move_bk(coor(1,i),mat,vec);
	}

	//ja if using command line option -use_input_cb, change template to include
	//ja native c-beta conformation.
	if ( active_rotamer_options.use_input_cb ) {
		if ( misc::res(seqpos) != aa_gly &&
				 misc::res(seqpos) != aa_pro &&
				 aa != aa_gly && aa != aa_pro ) {

			//ja aligns the bond in question
			align_atom( coor(1,5), rescoord(1,5), rescoord(1,2), mat, vec );

			//ja the first sidechain atom is 5 (c-beta)
			for ( int i = 5; i <= maxatom; ++i ){
				move_bk(coor(1,i),mat,vec);
			}
		}
	}

	int & nchi_aa( nchi(aa,aav) );
	for ( int ch = 1; ch <= MAX_CHI; ++ch ) { //bk change chi angle if it exists
		if ( nchi_aa >= ch ) {
			int l = chi_atoms.index(1,ch,aa,aav);
			c1 = chi_atoms[  l ]; // chi_atoms(1,ch,aa,aav);
			c2 = chi_atoms[ ++l ]; // chi_atoms(2,ch,aa,aav);
			c3 = chi_atoms[ ++l ]; // chi_atoms(3,ch,aa,aav);
			c4 = chi_atoms[ ++l ]; // chi_atoms(4,ch,aa,aav);
			// determine initial chi angle
			dihedral_bk(coor(1,c1),coor(1,c2),coor(1,c3),coor(1,c4),ichi);
			// rotate angle by new-initial degrees
			rot = radians( chi(ch) - ichi );
			// generate rotation vector and matrix
			getrot_bk(coor(1,c2), coor(1,c3), rot, mat, vec);
			// move atoms
			for ( int i = 1, lcr = chi_required.index(ch,i,aa,aav);
			 i <= natoms_aa; ++i, lcr += chi_required_size1 ) {
				if ( chi_required[ lcr ] ) { // chi_required(ch,i,aa,aav)
					move_bk(coor(1,i),mat,vec);
				}
			}
		}
	}

//  store coordinates in rotcoord
	int const HNpos_aa = HNpos(aa,aav);
	for ( int j = 1; j <= natoms_aa; ++j ) {
		std::string & atom_name_j( atom_name(j,aa,aav) );
		if ( j <= 4 ) {
			for ( int k = 1, l = rotcoord.index(k,j); k <= 3; ++k, ++l ) {
				rotcoord[ l ] = rescoord[ l ]; // rotcoord(k,j) = rescoord(k,j);
			}
		} else if ( HNpos_aa == j ) {
			if( atom_name_j == "1H  " && termini_ns::is_N_terminus(seqpos) ){ // n-term Hs

				assert( variant_type( aav_Nterm, aa, aav ) );

				build_N_terminus_Hs(full_coord,aa,seqpos,0.0,Nterm_nhcoord);
				// this has the desired side effect of slide( num_of_HN -1) residues
				int num_of_HN = 3;
				if ( aa == aa_pro ) num_of_HN = 2;
				for ( int nh = 1; nh <= num_of_HN; ++nh ){
					for ( int k = 1, l = rotcoord.index(k, j); k <= 3; ++k, ++l ) {
						rotcoord[ l ] = Nterm_nhcoord(k,nh); // rotcoord(k,j) = Nterm_nhcoord(k,nh);
					}
					if( nh < num_of_HN ) ++j;
				}
			} else {
				if ( score_check_current_pose() || pack_check_current_pose() ) {
					put_nh_current_pose( seqpos, full_coord, nhcoord );
				} else {
					put_nh_residue(full_coord,aa,seqpos,0.0,nhcoord);
				}
				for ( int k = 1, l = rotcoord.index(k,j); k <= 3; ++k, ++l ) {
					rotcoord[ l ] = nhcoord(k); // rotcoord(k,j) = nhcoord(k);
				}
			}

//bk        fix/rebuild the hydrogens on CD and CG of prolines
		} else if ( aa == 13 && ( atom_name_j == "2HD" || atom_name_j == "3HD" ||
		 atom_name_j == "2HG" || atom_name_j == "3HG" ) ) {
			place_atom(aa,aav,j,rotcoord);

//bk        need to build water attached to NH separately
//lin      using the new treatment for water building
		} else if ( atom_name_j == "1WN " ) {
			atm1 = HNpos_aa; // HN
			atm2 = 1; // NH
			place_atom_linear(rotcoord(1,atm1),rotcoord(1,atm2),rotcoord(1,j),1.95);

//bk        need to build water attached to OC separately
		} else if ( atom_name_j == "1WO " ) {
			atm1 = 4; // OC
			atm2 = 3; // C
			atm3 = 2; // CA
			dis = 2.8;
			rot_ang = 55.0;
			place_atom_linear(rotcoord(1,atm1),rotcoord(1,atm2),rotcoord(1,j),2.95);
			rotate_in_plane(rotcoord(1,atm1),rotcoord(1,j),rotcoord(1,atm1),
			 rotcoord(1,atm2),rotcoord(1,atm3),rot_ang);
		} else if ( atom_name_j == "2WO " ) {
			atm1 = 4; // OC
			atm2 = 3; // C
			atm3 = 2; // CA
			dis = 2.8;
			rot_ang = -55.0;
			place_atom_linear(rotcoord(1,atm1),rotcoord(1,atm2),rotcoord(1,j),dis);
			rotate_in_plane(rotcoord(1,atm1),rotcoord(1,j),rotcoord(1,atm1),
			 rotcoord(1,atm2),rotcoord(1,atm3),rot_ang);
		} else if ( atom_name_j == " O  " && termini_ns::is_C_terminus(seqpos) ){ // O terminus
			assert( variant_type( aav_Cterm, aa, aav ) );

			build_C_terminus_Os(full_coord,seqpos,0.0,Cterm_cocoord);
			// this have the desired side effect of slide( num_of_HN -1) residues
			for ( int k = 1, l = rotcoord.index(k, j); k <= 3; ++k, ++l ) {
				rotcoord[ l ] = Cterm_cocoord(k,1); // rotcoord(k,j) = Cterm_cocoord(k,co);
			}
		}	else if ( atom_name_j == " OXT" && termini_ns::is_C_terminus(seqpos) ){ // O terminus
			assert( variant_type( aav_Cterm, aa, aav ) );

			build_C_terminus_Os(full_coord,seqpos,0.0,Cterm_cocoord);
			// this have the desired side effect of slide( num_of_HN -1) residues
			for ( int k = 1, l = rotcoord.index(k, j); k <= 3; ++k, ++l )
				rotcoord[ l ] = Cterm_cocoord(k,2); // rotcoord(k,j) = Cterm_cocoord(k,co);
		} else {
			for ( int k = 1, l = rotcoord.index(k,j); k <= 3; ++k, ++l ) {
				rotcoord[ l ] = coor[ l ]; // rotcoord(k,j) = coor(k,j);
			}
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_rot_coord
///
/// @brief
///bk determines coordinates for rotamers superimposed on a pdb
///bk backbone
///am this version of get_rot_coord uses input rotamer coords rather ideal template coords when doing rotamer rebuild
/// @detailed
///am This function allows for "non-idealized" rotamers
/// @param  full_coord - [in/out]? - coordinates of backbone and existing sidechains
/// @param  aa - [in/out]? - amino acid
/// @param  aav - [in/out]? -
/// @param  chi - [in/out]? - desired chi angles
/// @param  rotcoord - [in/out]? - rotamer coordinates
/// @param  seqpos - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_rot_coord(
	FArray3DB_float & full_coord,
	int const aa,
	int const aav,
	FArray1Da_float chi,
	FArray2Da_float rotcoord,
	int const seqpos,
	FArray2Da_float template_coord
)
{
	using namespace aaproperties_pack;
	using namespace param;
	using numeric::conversions::radians;

	chi.dimension( MAX_CHI );
	rotcoord.dimension( 3, MAX_ATOM() );
	template_coord.dimension( 3, MAX_ATOM() );

//bk local variables
	static FArray2D_float coor( 3, MAX_ATOM() );
	static FArray2D_float mat( 3, 3 );
	static FArray1D_float vec( 3 );
	static FArray2D_float rescoord( 3, MAX_ATOM() );
	static FArray1D_float nhcoord( 3 );

	int atm1,atm2,atm3;
	float ichi, rot;
	float dis,rot_ang;
	int maxatom,c1,c2,c3,c4;
	int const chi_required_size1 = chi_required.size1();

	int & natoms_aa( natoms(aa,aav) );
	for ( int j = 1, lr = 0, lxyz = full_coord.index(1,j,seqpos); j <= natoms_aa; ++j ) {
		for ( int i = 1; i <= 3; ++i, ++lr, ++lxyz ) {
			rescoord[ lr ] = full_coord[ lxyz ]; // rescoord(i,j) = full_coord(i,j,seqpos);
		}
	}

//bk read in template coordinates for this amino acid
//	for ( int j = 1, lc = 0, li = icoor.index(1,j,aa,aav); j <= natoms_aa; ++j ) {
//		for ( int i = 1; i <= 3; ++i, ++lc, ++li ) {
//			coor[ lc ] = icoor[ li ]; // coor(i,j) = icoor(i,j,aa,aav);
//		}
//	}
//am read in user-provided template coordinates for this amino acid
	for ( int i = 1; i <= 3; ++i )
		for ( int j = 1; j <= natoms_aa; ++j )
			coor(i,j) = template_coord(i,j);

	int & nchi_aa( nchi(aa,aav) );
	for ( int ch = 1; ch <= MAX_CHI; ++ch ) { //bk change chi angle if it exists
		if ( nchi_aa >= ch ) {
			int l = chi_atoms.index(1,ch,aa,aav);
			c1 = chi_atoms[  l ]; // chi_atoms(1,ch,aa,aav);
			c2 = chi_atoms[ ++l ]; // chi_atoms(2,ch,aa,aav);
			c3 = chi_atoms[ ++l ]; // chi_atoms(3,ch,aa,aav);
			c4 = chi_atoms[ ++l ]; // chi_atoms(4,ch,aa,aav);
// determine initial chi angle
			dihedral_bk(coor(1,c1),coor(1,c2),coor(1,c3),coor(1,c4),ichi);
// rotate angle by new-initial degrees
			rot = radians( chi(ch) - ichi );
//   generate rotation vector and matrix
			getrot_bk(coor(1,c2), coor(1,c3), rot, mat, vec);
// move atoms
			for ( int i = 1, lcr = chi_required.index(ch,i,aa,aav);
			 i <= natoms_aa; ++i, lcr+=chi_required_size1 ) {
				if ( chi_required[ lcr ] ) { // chi_required(ch,i,aa,aav)
					move_bk(coor(1,i),mat,vec);
				}
			}
		}
	}

// superimpose the rotamer on to the backbone
//   find the matrix/vector that lines up the NH and CA atoms
//   and apply this operation to all the atoms in the rotamer

	lineup_bk(coor(1,2),coor(1,1),rescoord(1,2),rescoord(1,1),mat,vec);
	maxatom = natoms(aa,aav);
	for ( int i = 1; i <= maxatom; ++i ) {
		move_bk(coor(1,i),mat,vec);
	}

//   find the matrix/vector that lines up the C = O carbons
//   and apply this operation to the rotamer
	align_bk(coor(1,1),coor(1,2),coor(1,3),rescoord(1,3),mat(1,1),vec(1));
	for ( int i = 3; i <= maxatom; ++i ) {
		move_bk(coor(1,i),mat,vec);
	}

//   find the matrix/vector that lines up the C = O oxygens
//     and apply this operation to the oxygen
//        align_bk(coor(1,2),coor(1,3),coor(1,4),rescoord(1,4),
//         mat(1,1),vec(1));
//        move_bk(coor(1,4),mat,vec);

//  store coordinates in rotcoord
	int const HNpos_aa = HNpos(aa,aav);
	for ( int j = 1; j <= natoms_aa; ++j ) {
		std::string & atom_name_j( atom_name(j,aa,aav) );
		if ( j <= 4 ) {
			for ( int k = 1, l = rotcoord.index(k,j); k <= 3; ++k, ++l ) {
				rotcoord[ l ] = rescoord[ l ]; // rotcoord(k,j) = rescoord(k,j);
			}
		} else if ( HNpos_aa == j ) {
			if ( score_check_current_pose() || pack_check_current_pose() ) {
				put_nh_current_pose( seqpos, full_coord, nhcoord );
			} else {
				put_nh_residue(full_coord,aa,seqpos,0.0,nhcoord);
			}
			for ( int k = 1, l = rotcoord.index(k,j); k <= 3; ++k, ++l ) {
				rotcoord[ l ] = nhcoord(k); // rotcoord(k,j) = nhcoord(k);
			}

//bk        fix/rebuild the hydrogens on CD and CG of prolines
		} else if ( aa == 13 && ( atom_name_j == "2HD" || atom_name_j == "3HD" ||
		 atom_name_j == "2HG" || atom_name_j == "3HG" ) ) {
			place_atom(aa,aav,j,rotcoord);

//bk        need to build water attached to NH separately
//lin      using the new treatment for water building
		} else if ( atom_name_j == "1WN " ) {
			atm1 = HNpos_aa; // HN
			atm2 = 1; // NH
			place_atom_linear(rotcoord(1,atm1),rotcoord(1,atm2),rotcoord(1,j),1.95);

//bk        need to build water attached to OC separately
		} else if ( atom_name_j == "1WO " ) {
			atm1 = 4; // OC
			atm2 = 3; // C
			atm3 = 2; // CA
			dis = 2.8;
			rot_ang = 55.0;
			place_atom_linear(rotcoord(1,atm1),rotcoord(1,atm2),rotcoord(1,j),2.95);
			rotate_in_plane(rotcoord(1,atm1),rotcoord(1,j),rotcoord(1,atm1),
			 rotcoord(1,atm2),rotcoord(1,atm3),rot_ang);
		} else if ( atom_name_j == "2WO " ) {
			atm1 = 4; // OC
			atm2 = 3; // C
			atm3 = 2; // CA
			dis = 2.8;
			rot_ang = -55.0;
			place_atom_linear(rotcoord(1,atm1),rotcoord(1,atm2),rotcoord(1,j),dis);
			rotate_in_plane(rotcoord(1,atm1),rotcoord(1,j),rotcoord(1,atm1),
			 rotcoord(1,atm2),rotcoord(1,atm3),rot_ang);

		} else {
			for ( int k = 1, l = rotcoord.index(k,j); k <= 3; ++k, ++l ) {
				rotcoord[ l ] = coor[ l ]; // rotcoord(k,j) = coor(k,j);
			}
		}
	}

}


/////////////////////////////////////////////////////////////////////////
/// @begin get_coords_by_name_and_placement
///
/// @brief  Copy coords from one set to another, where the sets are for
///         two variants of the same amino acid.  To allow for different
///         ordering or insertions/deletions from one variant to another,
///         coords from atoms with identical names, and not array indices
///         are copied.  This involves a string comparison, so it is hardly
///         optimal.  Those atoms in the target set that have no corresponding
///         similarly-named counterpart in the source set are built using the
///         template definitions via the function place_atom.
///
/// @detailed
///
/// @param[in]  aa   input  amino acid type
/// @param[in]  src_aav  input   source amino acid variant
/// @param[in]  src_coord input  source coordinates for single residue
/// @param[in]  trg_aav  input   target amino acid variant
/// @param[in]  trg_coord input  target coordinates for single residue
///
///
/// @global_read  amino acid properties from aaproperties_pack.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  Jim Havranek 01/31/07
///
/// @last_modified
//////////////////////////////////////////////////////////////////////////

void
get_coords_by_name_and_placement(
  int const aa,
  int const src_aav,
  FArray2Da_float src_coords,
  int const trg_aav,
  FArray2Da_float trg_coords
)
{
	using namespace param;
	using namespace aaproperties_pack;

	src_coords.dimension( 3, MAX_ATOM() );
	trg_coords.dimension( 3, MAX_ATOM() );

	std::vector<int> missing;

// Loop thru new variants atoms, picking from src coords where
// possible, noting those that are missing
	for( int i = 1, ei = natoms( aa, trg_aav ) ; i <= ei ; ++ i ) {
		int src_i = LookupByName( aa, src_aav, atom_name( i, aa, trg_aav) );
		if( src_i > 0 ) {	// Failure in LookupByName indicated by -1 return value
			for( int k = 1 ; k <= 3 ; ++k ) {
				trg_coords( k, i ) = src_coords( k, src_i );
			}
		} else {	// Build this one below from template atoms
			missing.push_back( i );
		}
	}

// Place those that are missing
	for( int i = 0, ei = missing.size() ; i < ei ; ++i ) {
		int trg_i = missing[i];
		place_atom( aa, trg_aav, trg_i, trg_coords );
	}

return;
}


/////////////////////////////////////////////////////////////////////////
/// @begin coord_match_aa
///
/// @brief  check if a set of coordinates match the specified amino acid
///
/// @detailed
/// checks the distances between atoms to make sure they are
/// consistent with the specified amino acid
///
/// @param[in]  aa    input  amino acid
/// @param[in]  aav  input   amino acid variant
/// @param[in]  coord input  coordinates for single residue
///
///
/// @global_read  amino acid properties from aaproperties_pack.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  Brian Kuhlman 04/06/03
///
/// @last_modified
//////////////////////////////////////////////////////////////////////////
bool
coord_match_aa(
	int const aa,
	int const aav,
	FArray2Da_float coord
)
{
	using namespace aaproperties_pack;
	using namespace param;
	using namespace water;

	coord.dimension( 3, MAX_ATOM() );

	float dis1, dis2;

	for ( int atm1 = 5, atm1e = natoms( aa, aav ); atm1 <= atm1e; ++atm1 ) { // compare sidechain atoms
		// jk exclude waters (ie. matching all non-water atoms counts as an overall match)
		if ( fullatom_type(atm1,aa,aav) != type_h2o ) {
			int const atm2 = atom_base( atm1, aa, aav ); // atm2 and atm1 are bonded
			distance_bk( coord(1,atm1), coord(1,atm2), dis1 );
			distance_bk( icoor(1,atm1,aa,aav), icoor(1,atm2,aa,aav), dis2 );
			if ( std::abs( dis1 - dis2 ) > 0.1 ) {
				return false;
			}
		} //loop through atoms, excluding waters
	}

	return true;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin get_aa_rotno_to_packedrotno
///
/// @brief  protected accesss to the packedrotno structure
///
/// @detailed
///
/// Rotamers have to be looked up by a coded index based on
/// their amino acid and rotamer numbers for each rotatable
/// chi index. This function returns the coded index based
/// on the information provided and checks that the
/// requested rotamer number combination is possible and valid.
///
/// @param[in]   aa - in - amino acid of desired rotamer
/// @param[in]   rot - in - array of rot ids for each free chi angle
///
/// @return coded rotamer id
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// this coding map only exists becuase of fortran language
/// limitations and can/should be easily replaced with
/// modern language constructs when this becomes possible -ctsa
///
/// @references
///
/// @authors ctsa 10-2003
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
short
get_aa_rotno_to_packedrotno(
	int aa,
	int aav,
	FArray1Da_int rot
)
{
	using namespace aaproperties_pack;
	using namespace dunbrack_pack;
	using namespace param;
	using namespace param_aa;

	rot.dimension( MAX_CHI );

	// ctsa - this function is for real aa's only:
 	if ( !is_protein(aa) && !is_nonnatural(aa) ) {
		std::cout << "ABORT: bad to aa_rotno_to_packedrotno:" << std::endl;
		std::cout << "  attempt to extract non-amino acid rotamer from" << std::endl;
		std::cout << "  dunbrack library." << std::endl;
		std::cout << "aa,rot1/2/3/4: " << SS( aa ) << SS( rot(1) ) <<
		 SS( rot(2) ) << SS( rot(3) ) << SS( rot(4) ) << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	for ( int i = 1; i <= MAX_CHI; ++i ) {
		// for each aa, rotatable chi angles should have a rotamer number
		// from 1 to nrot and higher chi numbers should be set to 0.
		// proton rotamers chi angles should always be 0

		//rh Bug with -pH if nchi for aav1 differs from nchi for aav2.
		//rh Example: extra chi3 proton on aspartic acid causes exit failure
		//rh Fix in pH mode adding condition for aav2 nchi
		//rh Also included chi3 and chi4 of asp and glu as proton chi in:
		//rh "is_chi_proton_rotamer()"
		//rh std::cerr << "nchi(aa,1) " << nchi(aa,1) << std::endl;
		//rh std::cerr << "nchi(aa,2) " << nchi(aa,2) << std::endl;
		//rh std::cout << "AA " << aa << "\t AAV " << aav << "\t nchi(aa,aav) " << nchi(aa,aav) << "\t i " << i << "\t rot(i) " << rot(i) << "\t is_chi_proton_rotamer " << is_chi_proton_rotamer(aa,aav,i) << std::endl;

		if ( i <= nchi(aa,aav) ) { // i loops from 1--> MAX_CHI = 4
			if ( rot(i) > 0 || is_chi_proton_rotamer(aa,aav,i) ) continue;
		} else {
			if ( rot(i) == 0 ) continue;
		}

		if ( files_paths::antibody_modeler )
			continue;

		// failure
		std::cerr << "ABORT: bad to aa_rotno_to_packedrotno" << std::endl;
		std::cerr << "aa,rot1/2/3/4: " << aa_name3(aa) << '\t' << SS( aa )
							<< SS( rot(1) ) << SS( rot(2) ) << SS( rot(3) ) << SS( rot(4) )
							<< std::endl;
		std::cerr << "chi no " << i << "\t nchi " << nchi(aa,aav) << "\t aav "
							<< aav << "\t is_chi_proton_rotamer(aa,aav,i) "
							<< is_chi_proton_rotamer(aa,aav,i) << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

	}

	return aa_rotno_to_packedrotno(aa,rot(1),rot(2),rot(3),rot(4));
}


////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_from_chi
///
/// @brief
/// determines rotamer indices for chi angles
///  input arguments
///            chi: rotamer angles
///            res: amino acid type number
///  output: rot, rotamer indices
///
/// @detailed
///
/// @param  chi - [in/out]? -
/// @param  res - [in/out]? -
/// @param  rot - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
rotamer_from_chi(
	FArray1Da_float chi,
	int const res,
	FArray1Da_int rot
)
{
	using namespace param;
	using namespace param_aa;

	chi.dimension( MAX_CHI );
	rot.dimension( MAX_CHI );

	if ( !is_protein(res) && !is_RNA(res) && !is_nonnatural(res) ) {
		rot = 0;
		return;
	}

	// default to 0 for chi's greater than nchi, etc
	for ( int i = 1; i <= MAX_CHI; ++i ) {
		rot(i) = 0;

		if ( i == 1 ) { // chi 1 //KMa phospho_ser
			if ( res != aa_pro && res != aa_gly && res != aa_ala && res != aa_sep && ( !is_RNA(res) ) ) {
			 // chi1 of all residues except P,G,A,RNA
				if ( ( chi(i) >= 0.0 ) && ( chi(i) <= 120.0 ) ) rot(i) = 1;
				if ( std::abs(chi(i)) >= 120.0 ) rot(i) = 2;
				if ( ( chi(i) >= -120.0 ) && ( chi(i) <= 0.0 ) ) rot(i) = 3;
//mj      } else if ( res == aa_pro) { // chi1 of pro
//mj        if ( ( chi(i) >= 0.0 ) && ( chi(i) <= 90.0 ) ) rot(i) = 1;
//mj        if ( ( chi(i) >= -90.0 ) && ( chi(i) <= 0.0 ) ) rot(i) = 2;
//mj      }
			} else if ( res == aa_pro ) {  // chi1 of pro
				if ( chi(i) >= 0.0 ) rot(i) = 1;
				if ( chi(i) <= 0.0 ) rot(i) = 2;
			}
			//KMa phospho_ser 2006-01
			else if ( res == aa_sep )
			{
				if (  chi(i) >  -120.0    &&  chi(i) <= -10.0  )  rot(i) = 1;
				if (  chi(i) >   240.0    &&  chi(i) <= 350.0  )  rot(i) = 1;
				if (  chi(i) >  -10.0     &&  chi(i) <= 130.0  )  rot(i) = 2;
				if (  chi(i) >   350.0  )						  rot(i) = 2;
				if (  chi(i) <= -230.0  )						  rot(i) = 2;
				if (  chi(i) >   130.0    &&  chi(i) <=  240.0  ) rot(i) = 3;
				if (  chi(i) >  -230.0    &&  chi(i) <= -120.0  ) rot(i) = 3;
			}
			else if ( res == na_rgu || res == na_rad || res == na_rcy ||
			 res == na_ura ) {
				if ( ( chi(i) >= 0.0 ) && ( chi(i) <= 90.0 ) ) rot(i) = 1;
				if ( ( chi(i) >= -180.0 ) && ( chi(i) <= -90.0 ) ) rot(i) = 2;
				if ( ( chi(i) >= -90.0 ) && ( chi(i) <= 0.0 ) ) rot(i) = 3;
			}
		} else if ( i == 2 ) { // chi 2
			if ( res == aa_arg ) {
				if ( ( chi(i) >= 0.0 ) && ( chi(i) <= 120.0 ) ) rot(i) = 1;
//mj        if ( ( chi(i) >= 120.0 ) && ( chi(i) <= 240.0 ) ) rot(i) = 2;
//mj        if ( std::abs(chi(i)) > 120.0 ) rot(i) = 2;
				if ( std::abs(chi(i)) >= 120.0 ) rot(i) = 2;
				if ( ( chi(i) >= -120.0 ) && ( chi(i) <= 0.0 ) ) rot(i) = 3;
			} else if ( res == aa_glu || res == aa_his || res == aa_ile ||
			 res == aa_lys || res == aa_leu || res == aa_met || res == aa_gln ) {
				if ( ( chi(i) >= 0.0 ) && ( chi(i) <= 120.0 ) ) rot(i) = 1;
				if ( std::abs(chi(i)) > 120.0 ) rot(i) = 2;
				if ( ( chi(i) >= -120.0 ) && ( chi(i) <= 0.0 ) ) rot(i) = 3;
			} else if ( res == aa_asp ) {
				if ( ( ( chi(i) >= 30.0 ) && ( chi(i) <= 90.0 ) ) ||
				 ( ( chi(i) <= -90.0 ) && ( chi(i) >= -150.0 ) ) ) rot(i) = 1;
				if ( ( ( chi(i) >= -30.0 ) && ( chi(i) <= 30.0 ) ) ||
				 ( std::abs(chi(i)) >= 150.0 ) ) rot(i) = 2;
				if ( ( ( chi(i) >= -90.0 ) && ( chi(i) <= -30.0 ) ) ||
				 ( ( chi(i) >= 90.0 ) && ( chi(i) <= 150.0 ) ) ) rot(i) = 3;

			} else if ( ( res == aa_phe ) || ( res == aa_tyr ) ) {
				if ( ( ( chi(i) >= 30.0 ) && ( chi(i) <= 150.0 ) ) ||
				 ( ( chi(i) <= -30.0 ) && ( chi(i) >= -150.0 ) ) ) rot(i) = 1;
				if ( ( ( chi(i) >= -30.0 ) && ( chi(i) <= 30.0 ) ) ||
				 ( std::abs(chi(i)) >= 150.0 ) ) rot(i) = 2;
			} else if ( res == aa_trp ) {
				if ( ( chi(i) >= -180.0 ) && ( chi(i) <= -60.0 ) ) rot(i) = 1;
				if ( ( chi(i) >= -60.0 ) && ( chi(i) <= 60.0 ) ) rot(i) = 2;
				if ( ( chi(i) >= 60.0 ) && ( chi(i) <= 180.0 ) ) rot(i) = 3;
			} else if ( res == aa_asn ) { // chi2 of asn
				// ctsa - note this is a special case of the new dunbrack rotamer set
				if ( rot(1) == 1 ) {
					if ( chi(i) >= -150.0 && chi(i) < -90.0 ) rot(i) = 1;
					if ( chi(i) >= -90.0 && chi(i) < -30.0 ) rot(i) = 2;
					if ( chi(i) >= -30.0 && chi(i) < 30.0 ) rot(i) = 3;
					if ( chi(i) >=  30.0 && chi(i) < 90.0 ) rot(i) = 4;
					if ( chi(i) >=  90.0 && chi(i) < 150.0 ) rot(i) = 5;
					if ( std::abs(chi(i)) >= 150.0 ) rot(i) = 6;
				} else if ( rot(1) == 2 ) {
					if ( chi(i) >= -180.0 && chi(i) < -90.0 ) rot(i) = 1;
					if ( chi(i) >= -90.0 && chi(i) < -45.0 ) rot(i) = 2;
					if ( chi(i) >= -45.0 && chi(i) <  0.0 ) rot(i) = 3;
					if ( chi(i) >=   0.0 && chi(i) < 45.0 ) rot(i) = 4;
					if ( chi(i) >=  45.0 && chi(i) < 90.0 ) rot(i) = 5;
//mj          if ( chi(i) >=  90.0 && chi(i) < 180.0 ) rot(i) = 6;
					if ( chi(i) >=  90.0 && chi(i) <= 180.0 ) rot(i) = 6;
				} else {
					if ( chi(i) >= -180.0 && chi(i) < -105.0 ) rot(i) = 1;
					if ( chi(i) >= -105.0 && chi(i) < -45.0 ) rot(i) = 2;
					if ( chi(i) >= -45.0 && chi(i) <  15.0 ) rot(i) = 3;
					if ( chi(i) >=  15.0 && chi(i) <  60.0 ) rot(i) = 4;
					if ( chi(i) >=  60.0 && chi(i) < 120.0 ) rot(i) = 5;
//mj          if ( chi(i) >= 120.0 && chi(i) < 180.0 ) rot(i) = 6;
					if ( chi(i) >= 120.0 && chi(i) <= 180.0 ) rot(i) = 6;
				}
			}
			//KMa phospho_ser 2006-01
			else if ( res == aa_sep )
			{
			if (  chi(i)  >= -120.0  &&  chi(i) <    0.0   ) rot(i) = 1;
			if (  chi(i)  >=  240.0  )					     rot(i) = 1;
			if (  chi(i)  >=    0.0  &&  chi(i) <= 135.0   ) rot(i) = 2;
			if (  chi(i)  <  -225.0  )						 rot(i) = 2;
			if (  chi(i)  >=  135.0  &&  chi(i) <  240.0   ) rot(i) = 3;
			if (  chi(i)  >= -225.0  &&  chi(i) < -120.0   ) rot(i) = 3;
			}
		} else if ( i == 3 ) { // chi 3
			if ( res == aa_glu ) {
				if ( ( ( chi(i) >= 30.0 ) && ( chi(i) <= 90.0 ) ) ||
				 ( ( chi(i) <= -90.0 ) && ( chi(i) >= -150.0 ) ) ) rot(i) = 1;
				if ( ( ( chi(i) >= -30.0 ) && ( chi(i) <= 30.0 ) ) ||
				 ( std::abs(chi(i)) >= 150.0 ) ) rot(i) = 2;
				if ( ( ( chi(i) >= -90.0 ) && ( chi(i) <= -30.0 ) ) ||
				 ( ( chi(i) >= 90.0 ) && ( chi(i) <= 150.0 ) ) ) rot(i) = 3;
			} else if ( res == aa_arg || res == aa_lys || res == aa_met ) {
				if ( ( chi(i) >= 0.0 ) && ( chi(i) <= 120.0 ) ) rot(i) = 1;
				if ( std::abs(chi(i)) > 120.0 ) rot(i) = 2;
				if ( ( chi(i) >= -120.0 ) && ( chi(i) <= 0.0 ) ) rot(i) = 3;
			} else if ( res == aa_gln ) { // chi3 of gln
				// ctsa - note this is a special case of the new dunbrack rotamer set
				if ( rot(2) == 2 ) {
					if ( chi(i) >= 135.0 || chi(i) < -135.0 ) rot(i) = 1;
					if ( chi(i) >= -135.0 && chi(i) < -45.0 ) rot(i) = 2;
					if ( chi(i) >= -45.0 && chi(i) <  45.0 ) rot(i) = 3;
					if ( chi(i) >=  45.0 && chi(i) < 135.0 ) rot(i) = 4;
				} else {
					if ( chi(i) >= -180.0 && chi(i) < -90.0 ) rot(i) = 1;
					if ( chi(i) >= -90.0 && chi(i) <   0.0 ) rot(i) = 2;
					if ( chi(i) >=   0.0 && chi(i) <  90.0 ) rot(i) = 3;
//          if ( chi(i) >=  90.0 && chi(i) < 180.0 ) rot(i) = 4;
					if ( chi(i) >=  90.0 && chi(i) <= 180.0 ) rot(i) = 4;
				}
			}
			//KMa phospho_ser 2006-01
			else if ( res == aa_sep )
			{
				if ( ( chi(i) >= -120.0 ) && ( chi(i) < 0.0 ) ) rot(i) = 1;
				if ( (  chi(i) > 0.0 ) && ( chi(i) <= 120.0 ) ) rot(i) = 2;
				if ( std::abs(chi(i)) > 120.0 )					rot(i) = 3;
			}
		} else if ( i == 4 ) {  // chi 4
			if ( res == aa_arg || res == aa_lys ) {
				if ( ( chi(i) >= 0.0 ) && ( chi(i) <= 120.0 ) ) rot(i) = 1;
				if ( std::abs(chi(i)) > 120.0 ) rot(i) = 2;
				if ( ( chi(i) >= -120.0 ) && ( chi(i) <= 0.0 ) ) rot(i) = 3;
			}
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin residue_rmsd
///
/// @brief
/// Calculates rmsd between two residues (originally in select_closest_rotamer())
/// This function also checks for alternative symmetric sidechain conformations
///
/// @authors
/// ashworth, refactored from havranek
///
/////////////////////////////////////////////////////////////////////////////////
float sidechain_rmsd
(
	int const aa1,
	int const aav1,
	FArray2Da_float coord1,
	int const aa2,
	int const aav2,
	FArray2Da_float coord2
)
{
	using namespace aaproperties_pack;
	using namespace param;

	coord1.dimension( 3, MAX_ATOM() );
	coord2.dimension( 3, MAX_ATOM() );

	float accum(0.), dis;
	int firstatom( std::min( first_scatom(aa1,aav1), first_scatom(aa2,aav2) ) );
	int heavyatoms( std::min( nheavyatoms(aa1,aav1), nheavyatoms(aa2,aav2) ) );

	for ( int k(firstatom); k <= heavyatoms; ++k ) {
		distance_bk( coord1(1,k), coord2(1,k), dis );
		accum += dis * dis;
	}
	return std::sqrt( accum / ( float(heavyatoms) ) );
}
