// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 15424 $
//  $Date: 2007-06-12 21:26:56 +0300 (Tue, 12 Jun 2007) $
//  $Author: bcorreia $

// Rosetta Headers
#include "rotamer_trie_calc_energies.h"
#include "aaproperties_pack.h"
#include "angles.h"
#include "are_they_neighbors.h"
#include "constraints.h"
#include "design.h"
#include "design_structure.h"
#include "disulfides.h"
#include "dunbrack_pack.h"
#include "files_paths.h"
#include "fullatom_energies.h"
#include "fullatom_energy.h"
#include "gb_elec.h"
#include "gb_elec_ns.h"
#include "hbonds.h"
#include "hbonds_ns.h"
#include "input_pdb.h"
#include "InteractionGraphBase.h"
#include "ligand.h"
#include "make_pdb.h"
#include "minimize.h"
#include "misc.h"
#include "pack_bump_selector.h"
#include "pack.h"
#include "pack_geom_inline.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "param_rotamer_trie.h"
#include "param_torsion.h"
#include "PDInteractionGraph.h"
#include "pdbstatistics_pack.h"
#include "pH_main.h"
#include "pH_ns.h"
#include "pKa_mode.h"
#include "random_numbers.h"
#include "rotamer_functions.h"
#include "rotamer_trial_energies.h"
#include "rotamer_trials.h"
#include "RotamerOptions.h"
#include "RotamerSet.h"
#include "runlevel.h"
#include "scale_res_energy.h"
#include "score_ns.h"
#include "start.h"
#include "template_pack.h"
#include "util_basic.h"
#include "util_interpolate.h"
#include "water.h"
#include "water_ns.h"
#include "fast_pairenergy.h"
#include "PackerTask.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/ChunkVector.hh>

// C++ Headers
#include <ctime>
#include <iostream>
#include <vector>
#include <fstream>

//Utility Headers
#include <utility/basic_sys_util.hh>

namespace pack {
	bool read_interaction_graph_file = false;
	bool write_interaction_graph_file = false;
	std::string ig_file_name;
}

//int count_fpe_calls;
//int count_fpeh_calls;

using namespace pack;

////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie_calc_energies
///
/// @brief
/// apl replaces get_energies using fast trie_vs_trie algorithm
///
/// @detailed
/// apl computes values for energy1b and energy2b.  Does not handle explicit water
/// apl Uses rotamer_trie class for rapid
/// apl rotamer pair energy computation.  The largest and most important
/// apl difference between the energies the rotamer_tries compute and the energies
/// apl get_energies() computes is the inclusion of backbone/backbone pair energies.
/// apl I've incorportated backbone/backbone energies into energy2b
/// apl so that backbone motion as part of protein design will be readily integrated
/// apl into the code.
/// apl Order of operations:
/// apl 1) Construct Tries
/// apl 2) One body energy computation
/// apl 3) Rotamer Pair energy computation.
/// apl    - step 1 use Trie to calculate rotamer pair Lennard-Jones, solvation, and hydrogen bonding energies
/// apl    - step 2 correct cys/cys disulfide pair energies
/// apl    - step 3 add rotamer pair (NOE) constraint penalties
/// apl    - step 4 add generalized borne energies
/// apl    - step 5 add statistically derived electrostatic energies
///
/// @param  Task,        - [in ] - PackerTask object,
///   contains info on which residues have rotamers for placing. Call such residues "molten residues"
/// @param  ligenergy1b - [out] - the one body energies for each rotamer with the ligand
/// @param  neighbor_indexno  - [in ] - non-zero for any neighboring pair of residues
/// @param  ig - [in/out] - the interaction graph to load energies into
///
/// @global_read -
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors - apl kaufmann
///
/// @last_modified 5-16-2006
/////////////////////////////////////////////////////////////////////////////////
void
rotamer_trie_calc_energies(
	RotamerSet & rotamer_set,
	PackerTask const & Task,
	FArray1D_float & ligenergy1b,
	FArray2DB_short & neighbor_indexno,
	PrecomputedPairEnergiesInteractionGraph & ig
)
{

	using namespace design;
	using namespace misc;
	using namespace param;
	using namespace param_pack;
	using namespace scale_res_energy_ns;
	using namespace template_pack;

	if (rotamer_set.nrotamers() == 0) return;
	//std::cerr << "Calling rtce" << std::endl;

	// jk flag determines whether sc/bb energies are counted as 1-body or 2-body
	eval_sc_bbE_for_energy1b = false;

	//build_e2b_indexing_data(e2b_startindex, e2b_offsetindex, neighbor_indexno);

	//fill nrotoffest arrays....
	int const nmoltenres = ig.get_num_nodes();
	if ( nmoltenres == 0 ) return;

	assert ( nmoltenres == rotamer_set.nmoltenres() );

   //Phase 1: rotamer trie construction
   //std::cerr <<"//Phase 1: rotamer trie construction" << std::endl;

	int count_number_non_amino_acid_moltenres=0;
	for( int ii = 1; ii <= nmoltenres; ii++){
		if(rotamer_set.moltenres_2_resid(ii) < 0){
			++count_number_non_amino_acid_moltenres;
		}
	}
	//kwk we don't make tries of non_aminoacid components at the moment
	std::vector< rotamer_trie* > rt_vector(nmoltenres - count_number_non_amino_acid_moltenres + 1);

	for ( int ii = 1; ii <= nmoltenres; ++ii ) {
		//std::cerr << "Creating rotamer trie #: " << ii;
		if( rotamer_set.moltenres_2_resid(ii) < 0) continue; //kwk jump out for non amino acid molten residues
		create_rotamer_trie( rotamer_set, rotamer_set.nrotoffset(ii),
												 rotamer_set.num_states_for_moltenres(ii),
												 rotamer_set.moltenres_2_resid(ii), rt_vector[ii]);
		//std::cerr << " done." << std::endl;
	}

	//std::cerr << *rt_vector[ 2 ] << std::endl;

	//Phase 2:  calculate rotamer-internal- and rotamer-background energies.
	//std::cerr << "//Phase 2:  calculate rotamer-internal- and rotamer-background energies." << std::endl;


	//bs Count disulfides that satisfy centroid criteria.
	//bs After this point, disulfide connectivity is set,
	//bs so fullatom disulfide scoring below
	//bs will score these cys pairs only.
	//bs This must be done before first call to get_sc_scE.
	disulfides::BOUNDARY::evaluate_disulfides_centroid();

	//jjh generalized Born prep work, if needed
	if ( gen_born ) {
		std::cout << "jjh generalized Born prep work, if needed" << std::endl;
		gb_build_placeholders(total_residue, full_coord, res, res_variant, Task.get_designmap());
		gb_get_template_born_radii(total_residue, full_coord, born_radius,
			res, res_variant, Task.get_designmap());
		gb_get_rotamers_born_radii(total_residue, full_coord, res, res_variant, rotamer_set, Task.get_designmap());
	}

	//calculate self energies (dunbrack bb dependant rotamer library)
	//std::cerr << "calculate self energies (dunbrack bb dependant rotamer library)" << std::endl;

	FArray1D_float energy1b_base( rotamer_set.nrotamers(), 0. );
	for ( int ii = 1; ii <= nmoltenres; ++ii ) {
		int const rotres = rotamer_set.moltenres_2_resid(ii);
		if( rotres < 0){continue;}//kwk don't want to calculate for non aa molten residues
		for ( int jj = 1; jj <= rotamer_set.num_states_for_moltenres(ii); ++jj ) {
			int const rot = rotamer_set.nrotoffset(ii) + jj;
			int const aarot = rotamer_set.report_aa(rot);
			int const aavrot = rotamer_set.report_aav(rot);
			float const energy1b_this_rotamer = get_1body_energy( rotamer_set, rot, rotres, aarot,
				aavrot, rotamer_set.get_rotcoord(rot), energy1b_base, Task );
			ig.add_to_nodes_one_body_energy( ii, jj, energy1b_this_rotamer );
		}
	}

	if(!get_ligand_flexible_flag()){
	  //mj compute all ligand energies
	  //lin calculate the energy subset
	  //currently only energy between ligand and protein
	  get_ligenergies(ligenergy1b, rotamer_set);
	  //lin add the subset of ligand energy into energy1b;
	  ig.add_to_one_body_energies(ligenergy1b);
	}else{
	  //compute all ligand energies
	  compute_ligand_energies(&ig,Task.get_designmap(),rotamer_set,total_residue,res,res_variant,full_coord);
	}

	//Phase 3: rotamer pair energies
	//std::cerr << "//Phase 3: rotamer pair energies  " << std::endl;

	//apl make a copy of the original weights before we start scaling them
	float copy_Watr, copy_Wrep, copy_Wsol, copy_Whbond_bb, copy_Whbond_bb_sc, copy_Whbond_sc;
	save_weights(copy_Watr, copy_Wrep, copy_Wsol, copy_Whbond_bb, copy_Whbond_bb_sc, copy_Whbond_sc);

	for ( int ii = 1; ii <= nmoltenres; ++ii ) {
		int const iiresid = rotamer_set.moltenres_2_resid(ii);
		if( iiresid < 0 ){continue;}//kwk skip for non aa moltenres
		int const ii_num_states = rotamer_set.num_states_for_moltenres(ii);
		int const iirotindexoffset = rotamer_set.nrotoffset(ii);
		for ( int jj = ii+1; jj <= rotamer_set.nmoltenres(); ++jj ) {
			int const jjresid = rotamer_set.moltenres_2_resid(jj);
			if( jjresid < 0 ){continue;}//kwk skip for non aa moltenres
			int const jj_num_states = rotamer_set.num_states_for_moltenres(jj);
			int const jjrotindexoffset = rotamer_set.nrotoffset(jj);

			// modify weights before trie_v_trie called if we're going to
			// scale the interactions for this res/res pair
			if (get_scale_res_energy_flag() ) {
				if (scale_energy_matrix(iiresid,jjresid) ) {
					set_weights_to_scaled_values(copy_Watr, copy_Wsol, copy_Whbond_bb,
																			 copy_Whbond_bb_sc, copy_Whbond_sc);
				} else {
					set_weights_to_unscaled_values(copy_Watr, copy_Wsol,
																				 copy_Whbond_bb, copy_Whbond_bb_sc, copy_Whbond_sc);
				}
				if (scale_rep_energy_matrix(iiresid, jjresid) ) {
					set_rep_weight_to_scaled_value(copy_Wrep);
				} else {
					set_rep_weight_to_unscaled_value(copy_Wrep);
				}
			}
      param_pack::pack_wts.set_lock();
			get_trie_2body_energies( rotamer_set, neighbor_indexno(iiresid,jjresid), ig, Task,
															 ii, iiresid, ii_num_states, iirotindexoffset, rt_vector[ii],
															 jj, jjresid, jj_num_states, jjrotindexoffset, rt_vector[jj]);

			ig.declare_edge_energies_final( ii, jj );
      param_pack::pack_wts.release_lock();
		} //end for jj
	} //end for ii

	//return weights to their standard values
	if (get_scale_res_energy_flag() ) {
		set_weights_to_unscaled_values(copy_Watr, copy_Wsol, copy_Whbond_bb, copy_Whbond_bb_sc, copy_Whbond_sc);
		set_rep_weight_to_unscaled_value(copy_Wrep);
	}

	//apl deallocate rotamer tries.
	for ( int ii = 1; ii <= nmoltenres; ++ii ) {
		delete (rt_vector[ii]);
	}

	//std::cerr << "Num rotamer pair energies calculated in rotamer_trie_calc_energies: " << num_rpes_calculated << std::endl;

	//std::cerr << "Leaving rotamer_trie_calc_energies()" << std::endl;
	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin get_trie_2body_energies
///
/// @brief
/// jk get all the 2body energies for a residue pair, using trie
///
/// @detailed
///
/// @global_read -
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors - apl, modified by jk
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_trie_2body_energies(
	RotamerSet & rotamer_set,
	short const ij_neighbor_indexno,
	pack::PrecomputedPairEnergiesInteractionGraph & ig,
	PackerTask const & Task,
  int const ii,
  int const iiresid,
	int const ii_num_states,
	int const iirotindexoffset,
	rotamer_trie* & ii_rt,
  int const jj,
  int const jjresid,
	int const jj_num_states,
	int const jjrotindexoffset,
	rotamer_trie* & jj_rt
)
{

	using namespace hbonds;
	using namespace param;
	using namespace param_aa;
	using namespace param_pack;
	using namespace template_pack;

	float dis2, solvE, atrE, repE, elecE, pairE, tmp1;
	float plane_totalE;
	float schbE,srbbhbE,lrbbhbE,sc_bbhbE;
	bool  are_neighbors;
	bool const cst_exist = classical_constraints::BOUNDARY::get_constraints_exist();
	bool const disulf_flag = disulfides::BOUNDARY::get_disulf_flag(); // pack disulfides or not?

	int const max_nrotamers_per_res = max(ii_num_states,jj_num_states);
	FArray2D_float oversized_res_res_matrix1(max_nrotamers_per_res, max_nrotamers_per_res);
	FArray2D_float oversized_res_res_matrix2(max_nrotamers_per_res, max_nrotamers_per_res);
	FArray2D_bool aa_are_they_neighbors( MAX_AA(), MAX_AA(), false );

	FArray2Da_float oversized_res_res_matrix_proxy(oversized_res_res_matrix1);
	oversized_res_res_matrix_proxy.dimension(jj_num_states, ii_num_states);

	if ( ij_neighbor_indexno ) {
		aa_are_they_neighbors = false;
		bool found_aa_neighbors = false;
		for (int kk = 1; kk <= MAX_AA(); ++kk) {
			if (rotamer_set.first_rot_for_aa(kk,ii) == 0 ) continue;
			for (int ll = 1; ll <= MAX_AA(); ++ll) {
				if (rotamer_set.first_rot_for_aa(ll,jj) == 0 ) continue;
				are_they_neighbors(kk,ll,rotamer_set.get_first_rotcoord_for_aa(kk,ii),
													 rotamer_set.get_first_rotcoord_for_aa(ll,jj), dis2,are_neighbors);
				if (are_neighbors) found_aa_neighbors = true;
				aa_are_they_neighbors(ll, kk) = are_neighbors;
			}
		}

		if ( found_aa_neighbors ) {

			ig.add_edge(ii, jj);
			ig.set_sparse_aa_info_for_edge(ii, jj, aa_are_they_neighbors);

			//apl Invoke Trie vs Trie algorithm.  Varies, but is about 4x faster than get_sc_scE.
			(*ii_rt).trie_vs_trie( *jj_rt, oversized_res_res_matrix1, oversized_res_res_matrix2 );

			// jk Apply weights in Task, if desired
			if ( Task.get_residue_weight_map().check_for_reweighted_pair( iiresid, jjresid ) ) {
				for ( int kk = 1; kk <= ii_num_states; ++kk ) {
					int const kkrotindex = iirotindexoffset + kk;
					int const kkaa = rotamer_set.report_aa(kkrotindex);
					for ( int ll = 1; ll <= jj_num_states; ++ll ) {
						int const llrotindex = jjrotindexoffset + ll;
						int const llaa = rotamer_set.report_aa(llrotindex);
						oversized_res_res_matrix_proxy(ll, kk) *= Task.get_residue_weights(iiresid, kkaa, jjresid, llaa );
					}
				}
			}

			ig.add_to_two_body_energies_for_edge(ii, jj, oversized_res_res_matrix_proxy);

			//apl NOW go back and overwrite the disulfide pairs which need to be calculated using
			//apl the special-case rules described within "count_pair"
			//apl I revert to get_sc_scE calls; the extra time spent in this loop is slight.
			if (disulf_flag) {
				for ( int kk = 1; kk <= ii_num_states; ++kk ) {
					int const kkrotindex = iirotindexoffset + kk;
					int const kkaa = rotamer_set.report_aa(kkrotindex);
					int const kkaav = rotamer_set.report_aav(kkrotindex);
					if (kkaa != aa_cys) continue;

					//e2b_indexno = -1;

					for ( int ll = 1; ll <= jj_num_states; ++ll ) {
						int const llrotindex = jjrotindexoffset + ll;
						int const llaa = rotamer_set.report_aa(llrotindex);
						int const llaav = rotamer_set.report_aav(llrotindex);
						if (llaa != aa_cys) continue;
						float esum = 0;

						get_bb_bbE_P(iiresid, jjresid, kkaa, llaa, kkaav, llaav,
							rotamer_set.get_rotcoord(kkrotindex), rotamer_set.get_rotcoord(llrotindex), repE, atrE, solvE,elecE);
						esum += pack_wts.Wrep() * repE + pack_wts.Watr() * atrE + pack_wts.Wsol() * solvE;

						get_sc_bbE(kkaa, kkaav, llaa, llaav, rotamer_set.get_rotcoord(kkrotindex),
											 rotamer_set.get_rotcoord(llrotindex), iiresid, jjresid, solvE, atrE, repE,elecE);
						esum += pack_wts.Wrep() * repE + pack_wts.Watr() * atrE + pack_wts.Wsol() * solvE;

						get_sc_bbE(llaa, llaav, kkaa, kkaav, rotamer_set.get_rotcoord(llrotindex),
											 rotamer_set.get_rotcoord(kkrotindex), jjresid, iiresid, solvE, atrE, repE,elecE);
						esum += pack_wts.Wrep() * repE + pack_wts.Watr() * atrE + pack_wts.Wsol() * solvE;

						get_sc_scE(kkaa, kkaav, rotamer_set.get_rotcoord(kkrotindex), llaa, llaav,
											 rotamer_set.get_rotcoord(llrotindex), rotamer_set.get_rotactcoord(kkrotindex),
											 rotamer_set.get_rotactcoord(llrotindex), iiresid, jjresid, solvE, atrE,
							repE, pairE, plane_totalE, elecE);
						esum += pack_wts.Wrep() * repE + pack_wts.Watr() * atrE + pack_wts.Wsol() * solvE;

						get_hbE(true,kkaa,llaa,kkaav,llaav,iiresid,jjresid,
										neighbors(iiresid),neighbors(jjresid),rotamer_set.get_rotcoord(kkrotindex),
										rotamer_set.get_rotcoord(llrotindex),schbE,srbbhbE,lrbbhbE,sc_bbhbE);
						esum += pack_wts.Whbond_sc() * schbE + pack_wts.Whbond_bb() * (srbbhbE + lrbbhbE) + pack_wts.Whbond_bb_sc() *sc_bbhbE;

						esum += disulfides::BOUNDARY::getTotWtdScThisDisulf_pack(iiresid,rotamer_set.get_rotcoord(kkrotindex),
																							 jjresid,rotamer_set.get_rotcoord(llrotindex));

						esum *= Task.get_residue_weights(iiresid, kkaa, jjresid, llaa );

						//overwrite previous e2b contents
						//energy2b( e2b_indexno + e2b_startindex(kkrotindex) + rotindex(rotamer_number,llrotindex) ) = esum;
						ig.set_two_body_energy_for_edge(ii, jj, kk, ll, esum);
					}
				}
			}// end if ( disulf_flag )

			if ( gen_born && pack_wts.Wgb_elec() != 0) {
				oversized_res_res_matrix_proxy = 0;
				for ( int kk = 1; kk <= ii_num_states; ++kk ) {
					int const kkrotindex = iirotindexoffset + kk;
					int const kkaa = rotamer_set.report_aa(kkrotindex);
					int const kkaav = rotamer_set.report_aav(kkrotindex);

					for ( int ll = 1; ll <= jj_num_states; ++ll ) {
						int const llrotindex = jjrotindexoffset + ll;
						int const llaa = rotamer_set.report_aa(llrotindex);
						int const llaav = rotamer_set.report_aav(llrotindex);

						if (! aa_are_they_neighbors(llaa, kkaa) ) continue;

						float esum;
						//build eb2index
						//e2b_indexno = e2b_offsetindex(llaa, nnum, iiresid, kkaa );
						//if ( e2b_indexno < 0 ) continue; // these 2 aa's not neighbors

						esum = pack_wts.Wgb_elec() * gb_get_res_res_elecE( iiresid, kkaa, kkaav,
							rotamer_set.get_rotcoord(kkrotindex), rotamer_set.get_rot_born_radius(kkrotindex),
							jjresid, llaa, llaav, rotamer_set.get_rotcoord(llrotindex),
							rotamer_set.get_rot_born_radius(llrotindex));

						esum *= Task.get_residue_weights(iiresid, kkaa, jjresid, llaa );

						oversized_res_res_matrix_proxy( ll, kk ) = esum;

						//energy2b( e2b_indexno + e2b_startindex(kkrotindex) + rotindex(rotamer_number,llrotindex) ) += Wgb_elec * esum;
					}
				}
				ig.add_to_two_body_energies_for_edge(ii, jj, oversized_res_res_matrix_proxy);

			}//end if genborn

			oversized_res_res_matrix_proxy = 0;
			for ( int kk = 1; kk <= ii_num_states; ++kk ) {
				int const kkrotindex = iirotindexoffset + kk;
				int const kkaa = rotamer_set.report_aa(kkrotindex);

				for ( int ll = 1; ll <= jj_num_states; ++ll ) {
					int const llrotindex = jjrotindexoffset + ll;
					int const llaa = rotamer_set.report_aa(llrotindex);
					float esum;

					if (! aa_are_they_neighbors(llaa, kkaa) ) continue;

					//get statistical electrostatic interaction energy
					get_pairtermE(kkaa,llaa,iiresid,jjresid,rotamer_set.get_rotactcoord(kkrotindex),
												rotamer_set.get_rotactcoord(llrotindex),pairE,tmp1);
					esum = pack_wts.Wpair() * pairE;
					esum *= Task.get_residue_weights(iiresid, kkaa, jjresid, llaa );

					oversized_res_res_matrix_proxy(ll, kk) = esum;
				}
			}

			//std::cerr << "done" << std::endl;

			ig.add_to_two_body_energies_for_edge(ii, jj, oversized_res_res_matrix_proxy);

		} // end if found_aa_neighbors;
	} //end if non-zero neighbor indexno

	if (cst_exist && classical_constraints::BOUNDARY::residue_pair_has_constraints(iiresid, jjresid) ) {
		//FArray2Da_float oversized_res_res_matrix1proxy(oversized_res_res_matrix1);
		//oversized_res_res_matrix1proxy.dimension(jjnumrotamers, iinumrotamers);
		//oversized_res_res_matrix1proxy = 0;

		ig.force_all_aa_neighbors_for_edge(ii, jj);
		oversized_res_res_matrix_proxy = 0;

		for ( int kk = 1; kk <= ii_num_states; ++kk ) {
			int const kkrotindex = iirotindexoffset + kk;
			int const kkaa = rotamer_set.report_aa(kkrotindex);
			int const kkaav = rotamer_set.report_aav(kkrotindex);

			for ( int ll = 1; ll <= jj_num_states; ++ll ) {
				int const llrotindex = jjrotindexoffset + ll;
				int const llaa = rotamer_set.report_aa(llrotindex);
				int const llaav = rotamer_set.report_aav(llrotindex);

				float esum;

				esum = classical_constraints::BOUNDARY::get_res_res_cstE(iiresid,jjresid,kkaa,kkaav,llaa,
					llaav,rotamer_set.get_rotcoord(kkrotindex),rotamer_set.get_rotcoord(llrotindex),
					true,true,true,true);  //apl calculate all pairs (bb/bb,bb/sc...)
				//since pro otherwise improperly handled.
				esum *= Task.get_residue_weights(iiresid, kkaa, jjresid, llaa );

				oversized_res_res_matrix_proxy( ll, kk ) = esum;
			}
		}
		ig.add_to_two_body_energies_for_edge(ii, jj, oversized_res_res_matrix_proxy);
	}

	return;
}

float
get_sc_scE_trie(
	int aa_rot1,
	int aa_rot2,
	rotamer_trie const & rot1,
	rotamer_trie const & rot2,
	FArray1DB_float & rotactcoord1,
	FArray1DB_float & rotactcoord2
)
{
	using namespace param_pack;
	assert( rot1.get_num_total_rotamers() == 1 && rot2.get_num_total_rotamers() == 1 );

	float esum = rot1.path_vs_path( rot2 );
	float pairE(0.0f), tmp1( 0.0f );

	get_pairtermE(
		aa_rot1, aa_rot2,
		rot1.get_resid(),
		rot2.get_resid(),
		rotactcoord1,rotactcoord2,
		pairE,tmp1);
	esum += pack_wts.Wpair()*pairE;
	return esum;
}


char
read_pair_energy_file(
	PDInteractionGraph* ig,
	RotamerSet & rotamer_set,
	PackerTask const & Task,
	FArray2DB_short & neighbor_indexno,
	FArray1D_float & ligenergy1b
)
{
	if ( ig == 0 ) {
		std::cerr << "ERROR: Tried to read pair energy file with";
		std::cerr << " an interaction graph that does not support I/O" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	char energy_calculation_function;

	ig->prepare_to_read_energies_from_file();
	std::ifstream infile( pack::ig_file_name.c_str(), std::ios::in | std::ios::binary );
	infile.get( energy_calculation_function );

	//apl determine correspondence between rotamers in rotamers_pack and those
	//apl described by this input file.
	rotamer_set.read_rotamer_set_from_file( infile, *ig );

	//apl read the energies from input file and save those corresponding
	//apl to the rotamers contained in rotcoord
	ig->read_edge_energies_from_file( infile );
	infile.close();

	//apl compute one body energies and compute any pair energies
	//apl not contained in the input file; it's possible some rotamers
	//apl are absent from the input file that we must now compute energies for
	calculate_remaining_graph_energies( *ig, rotamer_set, energy_calculation_function,
		Task, neighbor_indexno, ligenergy1b );

	return energy_calculation_function;
}

void
write_pair_energy_file(
	PDInteractionGraph* ig,
	RotamerSet & rotamer_set,
	char energy_calculation_function
)
{
	if ( ig == 0 ) {
		std::cerr << "ERROR: Tried to output pair energy file with";
		std::cerr << " an interaction graph that does not support I/O" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	std::ofstream outfile( pack::ig_file_name.c_str(), std::ios::out | std::ios::binary );
	outfile.write( & energy_calculation_function, 1 );

	//apl describe the rotamers in rotamers_pack
	rotamer_set.write_rotamer_set_to_file( outfile, *ig );

	//apl write the rotamer pair energies to the file
	ig->write_edge_energies_to_file( outfile );
	outfile.close();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin calculate_remaining_graph_energies
///
/// @brief
/// apl Computes one body energies for rotamers and computes any
/// apl rotamer pair energies for rotamers that did not correspond to any
/// apl rotamer in the original interaction graph file.
///
/// @detailed
/// apl I fall back on get_sc_scE function calls for this code, but can
/// apl replace the rotamer pair energy calculations with a trie_vs_trie
/// apl variant if there is demand.  For docking and interface design, you'll
/// apl probably want that functionality, if I haven't yet figured out how to
/// apl keep rotamer pair energies in memory between invocations of
/// apl pack_rotamers.
///
/// @param  Task - [in ] - PackerTask object
///
/// @param  energy_calculation_function - [in] - 'r' for rotamer_trie_calc_energies,
///   'g' for get_energies.  rotamer_trie_calc_energies promotes sc/bb energies to
///    the two-body energy table, and includes backbone/backbone energies, whereas
///    get_energies stores bb/sc energies in one body energies, and doesn't include
///    backbone/backbone energies.  This function should calculate energies to match
///    however the original energy calculation function operated.
/// @param  ligenergy1b - [out] - the one body energies for each rotamer with the ligand
/// @param  neighbor_indexno  - [in ] - non-zero for any neighboring pair of residues
/// @param  ig - [in/out] - the interaction graph to load energies into
///
/// @global_read -
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors - apl
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calculate_remaining_graph_energies
(
	PDInteractionGraph & ig,
	RotamerSet & rotamer_set,
	char energy_calculation_function,
	PackerTask const & Task,
	FArray2DB_short & neighbor_indexno,
	FArray1D_float & ligenergy1b
)
{

	using namespace aaproperties_pack;
	using namespace design;
	using namespace hbonds;
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace param_pack;
	using namespace runlevel_ns;
	using namespace template_pack;
	using namespace water;

	//apl were energies in file calculated by get_energies() or
	//apl rotamer_trie_calc_energies()?
	eval_sc_bbE_for_energy1b = (energy_calculation_function == 'g');

//bs local disulfide variables
	bool disulf_flag = disulfides::BOUNDARY::get_disulf_flag(); // pack disulfides or not?
//bs Count disulfides that satisfy centroid criteria.
//bs After this point, disulfide connectivity is set,
//bs so fullatom disulfide scoring below
//bs will score these cys pairs only.
//bs This must be done before first call to get_sc_scE.
	disulfides::BOUNDARY::evaluate_disulfides_centroid();

	//jjh generalized Born prep work, if needed
	if ( gen_born ) {
		gb_build_placeholders(total_residue, full_coord, res, res_variant, Task.get_designmap());
		gb_get_template_born_radii(total_residue, full_coord, born_radius,
						res, res_variant, Task.get_designmap());
		gb_get_rotamers_born_radii(total_residue, full_coord, res, res_variant, rotamer_set, Task.get_designmap());
	}

	//fill nrotoffest arrays....
	int const nmoltenres = ig.get_num_nodes();
	//count the number molten residues

	if (nmoltenres == 0) return;

	//bk loop through rotamers, calculate self energies, rotamer-template energies
	std::cerr << "Calculating one body energies for rotamers" << std::endl;
	FArray1D_float energy1b_base( rotamer_set.nrotamers(), 0. );
	for ( int rot = 1; rot <= rotamer_set.nrotamers(); ++rot ) {
		int const rotres = rotamer_set.report_seqpos(rot);
		int const aarot = rotamer_set.report_aa(rot);
		int const aavrot = rotamer_set.report_aav(rot);
		float const energy1b_this_rotamer = get_1body_energy( rotamer_set, rot, rotres, aarot,
			aavrot, rotamer_set.get_rotcoord(rot), energy1b_base, Task );
		int const rot_molten_res = rotamer_set.resid_2_moltenres( rotres );
		int const rot_index_on_molten_res = rot - rotamer_set.nrotoffset( rot_molten_res );
		ig.add_to_nodes_one_body_energy( rot_molten_res, rot_index_on_molten_res, energy1b_this_rotamer );
	}

	//mj compute all ligand energies
	//lin calculate the energy subset
	//currently only energy between ligand and protein
	get_ligenergies(ligenergy1b, rotamer_set);
	//lin add the subset of ligand energy into energy1b;
	ig.add_to_one_body_energies(ligenergy1b);

	//apl now fill in missing rotamer pair energies
	//apl hopefully, very few rotamers are missing

	float schbE,sc_bbhbE,solvE,atrE,repE,elecE,pairE,plane_totalE,dis2,
	 h2oE,h2ohbE;
	float srbbhbE,lrbbhbE; // short and long range bb hbond energy
	float tmp1;

	//car
	bool const cst_exist = classical_constraints::BOUNDARY::get_constraints_exist();

	FArray1D_int num_rotamers_absent_from_file( nmoltenres );
	int total_absent = 0;
	for (int ii = 1; ii <= nmoltenres; ++ii) {
		total_absent += num_rotamers_absent_from_file(ii) =
			ig.get_num_rots_absent_from_file( ii );
		if (num_rotamers_absent_from_file(ii) != 0) {
			std::cerr << "rotamers absent from " << rotamer_set.moltenres_2_resid( ii ) << ": ";
			std::cerr << num_rotamers_absent_from_file(ii) << " (out of ";
			std::cerr << rotamer_set.num_states_for_moltenres( ii ) << ")" << std::endl;
		}
	}

	if (total_absent == 0) {
		ig.declare_finished_reading_from_file();
		return;
	}

	std::cerr << "Computing rotamer pair energies for rotamers missing from input interaction graph file." << std::endl;
	std::cerr << "Num Rotamers Missing: " << total_absent << " (out of ";
	std::cerr << rotamer_set.nrotamers() << " total rotamers). " << std::endl;

	int count_pair_energies_computed = 0;

	FArray1D< FArray1D< int > > rots_absent( nmoltenres );
	FArray1D< FArray1D< bool > > rots_present( nmoltenres );
	for (int ii = 1; ii <= nmoltenres; ++ii) {
		int ii_num_absent = num_rotamers_absent_from_file( ii );
		rots_absent(ii).dimension( ii_num_absent );
		rots_present(ii).dimension( rotamer_set.num_states_for_moltenres( ii ) );
		rots_present(ii) = true;
		ig.get_absent_rots( ii, rots_absent(ii) );
		for (int jj = 1; jj <= ii_num_absent; ++jj) {
			rots_present(ii)( rots_absent(ii)(jj) ) = false;
		}
	}

	for ( int ii = 1; ii <= nmoltenres; ++ii ) {
		int iiresid = rotamer_set.moltenres_2_resid( ii );
		int ii_num_absent = num_rotamers_absent_from_file( ii );
		int ii_num_states = rotamer_set.num_states_for_moltenres( ii );
		int iirotoffset = rotamer_set.rotindex_offsets( iiresid );
		bool ii_had_file_correspondence = ig.get_node_corresponded_to_file_node(ii);
		for ( int jj = ii + 1; jj <= rotamer_set.nmoltenres(); ++jj ) {
			int jjresid = rotamer_set.moltenres_2_resid( jj );
			int jj_num_absent = num_rotamers_absent_from_file( jj );
			int jj_num_states = rotamer_set.num_states_for_moltenres( jj );
			int jjrotoffset = rotamer_set.rotindex_offsets( jjresid );
			bool jj_had_file_correspondence = ig.get_node_corresponded_to_file_node(jj);

			if (ii_num_absent == 0 && jj_num_absent == 0) continue;

			//apl Assumption: no extra edges needed if both nodes represented in file
			if ( ii_had_file_correspondence && jj_had_file_correspondence &&
					 ! ig.get_edge_exists(ii, jj) ) {
				continue;
			}

			if ( ! ii_had_file_correspondence || ! jj_had_file_correspondence ) {
				if (neighbor_indexno(ii, jj) == 0 ) {
					continue;
				} else {
					//apl add the edge and (lazily) set all aa pairs as neighbors
					//apl we will shrink the energy table after filling all non-zero
					//apl entries
					ig.force_all_aa_neighbors_for_edge(ii, jj);
				}
			}

			for ( int kk = 1; kk <= ii_num_states; ++kk ) {
				bool kkpresent = rots_present(ii)(kk);
				if (kkpresent && jj_num_absent == 0 ) continue;

				int kkrotindex = iirotoffset + kk;
				int kkaa = rotamer_set.report_aa(kkrotindex );
				int kkaav = rotamer_set.report_aav( kkrotindex );

				for (int ll = 1; ll <= jj_num_states; ++ll ) {
					bool llpresent = rots_present(jj)(ll);
					if ( kkpresent && llpresent) continue;

					int llrotindex = jjrotoffset + ll;
					int llaa = rotamer_set.report_aa( llrotindex );
					int llaav = rotamer_set.report_aav( llrotindex );

					bool are_neighbors = false;
					bool aa_neighbors = ig.get_sparse_aa_info_for_edge( ii, jj, kkaa, llaa );
					if (! aa_neighbors ) {
						are_they_neighbors( kkaa, llaa, rotamer_set.get_rotcoord(kkrotindex),
																rotamer_set.get_rotcoord(llrotindex), dis2, are_neighbors);
					}

					if (are_neighbors && ! aa_neighbors ) {
						std::cerr << "forcing all aa neighbors:" << ii << " " << jj << std::endl;
						ig.force_all_aa_neighbors_for_edge(ii, jj);
					}

					if (!aa_neighbors && ! are_neighbors ) continue;

					++count_pair_energies_computed;

					float esum = 0;
					//calculate energies
					if ( eval_sc_bbE_for_energy1b ) {
						get_sc_scE(kkaa, kkaav, rotamer_set.get_rotcoord(kkrotindex),
											 llaa, llaav, rotamer_set.get_rotcoord(llrotindex),
											 rotamer_set.get_rotactcoord(kkrotindex), rotamer_set.get_rotactcoord(llrotindex),
									iiresid, jjresid, solvE, atrE, repE, pairE, plane_totalE, elecE);
						esum += scale_res_rep_energy(iiresid, jjresid, pack_wts.Wrep() * repE) +
							scale_res_energy(iiresid, jjresid, pack_wts.Watr() * atrE + pack_wts.Wsol() * solvE);
					} else {
						get_bb_bbE_P(iiresid, jjresid, kkaa, llaa, kkaav, llaav,
												 rotamer_set.get_rotcoord(kkrotindex), rotamer_set.get_rotcoord(llrotindex),
              repE, atrE, solvE,elecE);
						esum += scale_res_rep_energy(iiresid, jjresid, pack_wts.Wrep() * repE) +
							scale_res_energy(iiresid, jjresid, pack_wts.Watr() * atrE + pack_wts.Wsol() * solvE);

						get_sc_bbE(kkaa, kkaav, llaa, llaav, rotamer_set.get_rotcoord(kkrotindex),
											 rotamer_set.get_rotcoord(llrotindex), iiresid, jjresid, solvE, atrE, repE,elecE);
						esum += scale_res_rep_energy(iiresid, jjresid, pack_wts.Wrep() * repE) +
							scale_res_energy(iiresid, jjresid, pack_wts.Watr() * atrE + pack_wts.Wsol() * solvE);

						get_sc_bbE(llaa, llaav, kkaa, kkaav, rotamer_set.get_rotcoord(llrotindex),
											 rotamer_set.get_rotcoord(kkrotindex), jjresid, iiresid, solvE, atrE, repE,elecE);
						esum += scale_res_rep_energy(iiresid, jjresid, pack_wts.Wrep() * repE) +
							scale_res_energy(iiresid, jjresid, pack_wts.Watr() * atrE + pack_wts.Wsol() * solvE);

						get_sc_scE(kkaa, kkaav, rotamer_set.get_rotcoord(kkrotindex),
											 llaa, llaav, rotamer_set.get_rotcoord(llrotindex),
											 rotamer_set.get_rotactcoord(kkrotindex), rotamer_set.get_rotactcoord(llrotindex),
							iiresid, jjresid, solvE, atrE, repE, pairE, plane_totalE, elecE);

						esum += scale_res_rep_energy(iiresid, jjresid, pack_wts.Wrep() * repE) +
							scale_res_energy(iiresid, jjresid, pack_wts.Watr() * atrE + pack_wts.Wsol() * solvE);
					}

					get_hbE(true,kkaa,llaa,kkaav,llaav,iiresid,jjresid,
									neighbors(iiresid),neighbors(jjresid),rotamer_set.get_rotcoord(kkrotindex),
									rotamer_set.get_rotcoord(llrotindex),schbE,srbbhbE,lrbbhbE,sc_bbhbE);
					esum += scale_res_energy(iiresid, jjresid, pack_wts.Whbond_sc() * schbE +
						pack_wts.Whbond_bb() * (srbbhbE + lrbbhbE) + pack_wts.Whbond_bb_sc() *sc_bbhbE );

					get_sc_sc_h2oE(kkaa,kkaav,rotamer_set.get_rotcoord(kkrotindex),
												 llaa,llaav, rotamer_set.get_rotcoord(llrotindex),ii,jj,h2oE,h2ohbE);
					esum += pack_wts.Wh2o() * ( pack_wts.Wh2o_lj()*h2oE + h2ohbE );


					if ( gen_born && pack_wts.Wgb_elec() != 0) {
						esum += pack_wts.Wgb_elec() * gb_get_res_res_elecE( iiresid, kkaa, kkaav,
							rotamer_set.get_rotcoord(kkrotindex), rotamer_set.get_rot_born_radius(kkrotindex),
							jjresid, llaa, llaav, rotamer_set.get_rotcoord(llrotindex),
							rotamer_set.get_rot_born_radius(llrotindex));
					}

					if (cst_exist && classical_constraints::BOUNDARY::residue_pair_has_constraints(iiresid, jjresid) ) {
						bool calc_bb_cst;
						if (! eval_sc_bbE_for_energy1b ) {
							calc_bb_cst = true;
						} else {
							calc_bb_cst = false;
						}
						esum += classical_constraints::BOUNDARY::get_res_res_cstE(iiresid,jjresid,kkaa,kkaav,llaa,
							llaav,rotamer_set.get_rotcoord(kkrotindex),rotamer_set.get_rotcoord(llrotindex),
							calc_bb_cst,true,calc_bb_cst,true);
					}

					if (disulf_flag && kkaa == aa_cys &&
							llaa == aa_cys && disulfides::BOUNDARY::cys_pair_in_disulf(iiresid, jjresid) ) {
						esum += disulfides::BOUNDARY::getTotWtdScThisDisulf_pack(
							// tot. weighted fullatom disulf score, using wts in param_pack.h
							iiresid,rotamer_set.get_rotcoord(kkrotindex),jjresid,rotamer_set.get_rotcoord(llrotindex));
					}

					get_pairtermE(kkaa, llaa, iiresid, jjresid,
						rotamer_set.get_rotactcoord(kkrotindex),rotamer_set.get_rotactcoord(llrotindex),pairE,tmp1);
					esum += pack_wts.Wpair() * pairE;
					esum *= Task.get_residue_weights(iiresid, kkaa, jjresid, llaa );

					ig.set_two_body_energy_for_edge( ii, jj, kk, ll, esum );
				}
			}
			ig.declare_edge_energies_final(ii, jj);
		}
	}

	std::cerr << "Finished computing extra pair energies absent from input file" << std::endl;
	std::cerr << "Computed an extra " << count_pair_energies_computed << " pair energies" << std::endl;

	ig.declare_finished_reading_from_file();

}


////////////////////////////////////////////////////////////////////////////////
/// @begin create_rotamer_trie
///
/// @brief
/// apl creates a rotamer trie using rotamers within rotcoord;
///
/// @detailed
/// apl creates a vector of rotamer descriptors, sorts them
/// apl (total order defined in rotamer_descripor class) and
/// apl passes the sorted vector to the rotamer_trie constructor.
///
/// @param  rotamer_offset - [in ] - 1 before first rotamer for this residue
/// @param  num_rotamers   - [in ] - num rotamers for this residue
/// @param  resid          - [in ] - this residue
/// @param  rt             - [out] - pointer to a rotamer_trie
///
/// @global_read - rotindex, rotcoord
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors - apl
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
create_rotamer_trie(
  RotamerSet & rotamer_set,
	int  rotamer_offset,
	int  num_rotamers,
	int  resid,
	rotamer_trie* & rt
)
{

	using namespace aaproperties_pack;
	//using namespace count_pairenergy_calls;
	using namespace design;
	using namespace hbonds;
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace param_pack;
	using namespace pdbstatistics_pack;
	using namespace runlevel_ns;
	using namespace water;

	//put the rotamers into a vector of rotamer_descriptors
	//std::cerr << "put the rotamers into a vector of rotamer_descriptors" << std::endl;
	std::vector< rotamer_descriptor > resrotamers( num_rotamers + 1);
	for ( int ii = 1; ii <= num_rotamers; ++ii ) {
		int const index = rotamer_offset + ii;
   	create_rotamer_descriptor( resid, resrotamers[ii], rotamer_set.report_aa(index),
		  rotamer_set.report_aav(index), rotamer_set.get_rotcoord(index), ii);
	}

	//sort
	//std::cerr << "sort" << std::endl;
	std::vector<rotamer_descriptor>::iterator first_rotamer = resrotamers.begin();
	++first_rotamer;
	sort( first_rotamer, resrotamers.end() );

	//create rotamer trie
	//std::cerr << "create rotamer trie" << std::endl;
	rt = new rotamer_trie(resid, resrotamers);

	return;
}

void
create_rotamer_trie(
  RotamerSet & rotamer_set,
	int resid,
	rotamer_trie* & rt
)
{

	using namespace aaproperties_pack;
	//using namespace count_pairenergy_calls;
	using namespace design;
	using namespace hbonds;
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace param_pack;
	using namespace pdbstatistics_pack;
	using namespace runlevel_ns;
	using namespace water;

	assert( rotamer_set.nrotamers() > 0 );

	//put the rotamers into a vector of rotamer_descriptors
	//std::cerr << "put the rotamers into a vector of rotamer_descriptors" << std::endl;
	std::vector< rotamer_descriptor > resrotamers( rotamer_set.nrotamers() + 1);
	for ( int ii = 1; ii <= rotamer_set.nrotamers(); ++ii ) {
   	create_rotamer_descriptor( resid, resrotamers[ii], rotamer_set.report_aa(ii),
      rotamer_set.report_aav(ii), rotamer_set.get_rotcoord(ii), ii);
	}

	//sort
	//std::cerr << "sort" << std::endl;
	std::vector<rotamer_descriptor>::iterator first_rotamer = resrotamers.begin();
	++first_rotamer;
	sort( first_rotamer, resrotamers.end() );

	//create rotamer trie
	//std::cerr << "create rotamer trie" << std::endl;
	rt = new rotamer_trie(resid, resrotamers);

	return;
}


void
create_background_rotamer_trie(
	int  resid,
	rotamer_trie * & rt
)
{

	using namespace misc;

	std::vector< rotamer_descriptor > rdvect(2);
	create_rotamer_descriptor( resid, rdvect[1], res(resid),
		res_variant(resid), full_coord(1, 1, resid), 1);
	rt = new rotamer_trie(resid, rdvect);

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin create_rotamer_descriptor
///
/// @brief
/// apl creates a rotamer descriptor
///
/// @detailed
/// apl converts fullatom atom order to param_rot_trie atom order
/// apl and packs them into an array.  Will not handle solvated rotamers well.
///
/// @param  resid - [in] - the residue number; important for disulfides
/// @param  rot_descr  - [out] -
/// @param  aa         - [in ] -
/// @param  rot_coords - [in ] - slice of param_pack::rotcoord
/// @param  rotid      - [in ] - the index of this rotamer on this residue (rotindex index - nrotoffset[ resid ] )
///
/// @global_read
/// disulfides
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors - apl
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
create_rotamer_descriptor (
	int resid,
	rotamer_descriptor & rot_descr,
	int aa,
	int aav,
	FArray2Da_float rot_coords,
	int rotid
)
{

	using namespace aaproperties_pack;
	using namespace param;
	using namespace param_aa;
	using namespace param_rotamer_trie;

	rot_coords.dimension(3, MAX_ATOM());

	//apl at the moment, rotamer trie supports his taut variants
	//apl terminal variants and pH variants only
	assert( (variant_type(aav_base, aa, aav) && !(variant_type(aav_water, aa, aav)) )
		|| (aa == param_aa::aa_his && variant_type(aav_his_taut, aa, aav))
		|| variant_type( aav_Nterm, aa, aav) || variant_type( aav_Cterm, aa, aav)
		|| variant_type( aav_pH, aa, aav) );

	std::vector< atom_descriptor > atoms(rotamer_trie::MAX_ATS_PER_RES + 1);
	int aa_numatoms = natoms(aa, aav);

	for ( int ii = 1; ii <= aa_numatoms; ++ii ) {
		atoms[ii] = atom_descriptor(ii, aa, aav, rot_coords);
	}

	int variant_according_to_the_trie = get_variant_according_to_trie( aa, aav );

	FArray1Da_int fullatom_2_trie_order_this_aa( fullatom_2_trie_order(1, aa, variant_according_to_the_trie));
	fullatom_2_trie_order_this_aa.dimension(rotamer_trie::MAX_ATS_PER_RES);

	//apl if cys residue in disulfide bond, trim HG
	int temp_aa_numatoms( aa_numatoms );
  if ( aa == aa_cys && disulfides::BOUNDARY::cys_res_in_disulf( resid ) ) --temp_aa_numatoms;
	rot_descr = rotamer_descriptor( temp_aa_numatoms);
	rot_descr.set_rotamer_id(rotid);
	//std::cerr << "aa= " << aa << std::endl;
	for ( int ii = 1; ii <= aa_numatoms; ++ii ) {
		int ii_trie_order = fullatom_2_trie_order_this_aa( ii );

		//apl if cys residue in disulfide bond, trim HG -- the last atom
		//apl In param_rotamer_trie, HG must be last atom in cystine for all variants
		if ( aa == aa_cys && ii_trie_order == aa_numatoms && disulfides::BOUNDARY::cys_res_in_disulf( resid ) ) continue;

		rot_descr.set_atom( ii_trie_order, atoms[ ii ] );
	}
	//std::cerr << std::endl;
	rot_descr.set_aa_type(aa);
	rot_descr.set_aa_variant(variant_according_to_the_trie);
	return;

}


int get_variant_according_to_trie( int aa, int aav ) {

	using namespace aaproperties_pack;
	using namespace LookUp_aav_pH;
	using namespace param;
	using namespace param_aa;
	using namespace param_rotamer_trie;

	int variant_according_to_the_trie = 0;
	if ( variant_type(aav_base, aa, aav) &&
		! variant_type(aav_his_taut, aa, aav) &&
		! variant_type(aav_Nterm, aa, aav) &&
		! variant_type(aav_Cterm, aa, aav) &&
		! variant_type(aav_water, aa, aav) &&
		! variant_type(aav_pH, aa, aav)) {
		variant_according_to_the_trie = trie_variant_for_base;
	} else if ( aa == param_aa::aa_his && variant_type(aav_his_taut, aa, aav) ) {
		variant_according_to_the_trie = trie_variant_for_his_tautomer;
	} else if ( variant_type( aav_Nterm, aa, aav) && ! variant_type( aav_pH, aa, aav) ) {
		variant_according_to_the_trie = trie_variant_for_n_terminus;
	} else if ( variant_type( aav_Cterm, aa, aav) && ! variant_type( aav_pH, aa, aav) ) {
		variant_according_to_the_trie = trie_variant_for_c_terminus;
	} else if ( variant_type( aav_pH, aa, aav) &&
		! variant_type( aav_Cterm, aa, aav) &&
		! variant_type( aav_Nterm, aa, aav)) {
		//apl following code tightly coupled to the definitions in pH_main.cc
		if ( (aa == aa_asp || aa == aa_glu)) {
			if ( aav == asp_OD1protonated || aav == glu_OE1protonated ) {
				variant_according_to_the_trie = trie_variant_for_extra_protonation_1;
			} else if ( aav == asp_OD2protonated ||  aav == glu_OE2protonated ) {
				variant_according_to_the_trie = trie_variant_for_extra_protonation_2;
			}
		} else if ( (aa == aa_his) && (aav == his_protonated1 || aav == his_protonated2) ) {
			variant_according_to_the_trie = trie_variant_for_extra_protonation_1;
		} else if ( (aa == aa_lys && aav == lys_3HZdeprotonated) || (aa == aa_arg && aav == arg_1HH2deprotonated ) || (aa == aa_tyr && aav == tyr_HHdeprotonated) ) {
			variant_according_to_the_trie = trie_variant_for_deprotonation;
		}
	}

	if ( variant_according_to_the_trie == 0 ) {
		std::cerr << "FATAL ERROR: trie does not support the variant you've asked for: aa = " << aa << " aav = " << aav << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	return variant_according_to_the_trie;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin save_weights
///
/// @brief
/// apl  store current energy weights for use by the rotamer trie
///
/// @detailed
/// apl keeps copies of current energy weights so that the "scale_res_energy" functions
/// apl can be implemented more efficiently.  If the existant weights, (pack_wts.Watr() etc) are
/// apl scaled before trie_vs_trie begins, then we save one multiplication for every
/// apl energy we encounter
///
/// @param  copy_Watr,
/// @param  copy_Wrep,
/// @param  copy_Wsol,
/// @param  copy_Whbond_bb,
/// @param  copy_Whbond_bb_sc,
/// @param  copy_Whbond_sc
///
/// @global_read pack_wts.Watr(), pack_wts.Wrep(), pack_wts.Wsol(), pack_wts.Whbond_bb(), pack_wts.Whbond_bb_sc(), pack_wts.Whbond_sc()
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors - apl
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void save_weights(
	float & copy_Watr,
	float & copy_Wrep,
	float & copy_Wsol,
	float & copy_Whbond_bb,
	float & copy_Whbond_bb_sc,
	float & copy_Whbond_sc
)
{
	using namespace param_pack;
	copy_Watr = pack_wts.Watr();
	copy_Wrep = pack_wts.Wrep();
	copy_Wsol = pack_wts.Wsol();
	copy_Whbond_bb = pack_wts.Whbond_bb();
	copy_Whbond_bb_sc = pack_wts.Whbond_bb_sc();
	copy_Whbond_sc = pack_wts.Whbond_sc();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_weights_to_scaled_values
///
/// @brief
/// apl  before computing res/res energies with the rotamer trie, scale the existing
/// apl  energy weights (not pack_wts.Wrep()) by scale_res_energy_ns::scale_res_energy_weight
///
/// @detailed
///
/// @param  copy_Watr,
/// @param  copy_Wsol,
/// @param  copy_Whbond_bb,
/// @param  copy_Whbond_bb_sc,
/// @param  copy_Whbond_sc
///
/// @global_read
///
/// @global_write pack_wts.Watr(), pack_wts.Wsol(), pack_wts.Whbond_bb(), pack_wts.Whbond_bb_sc(), pack_wts.Whbond_sc()
///
/// @remarks
///
/// @references
///
/// @authors - apl
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void set_weights_to_scaled_values(
	float copy_Watr,
	float copy_Wsol,
	float copy_Whbond_bb,
	float copy_Whbond_bb_sc,
	float copy_Whbond_sc
)
{
	using namespace param_pack;
	using namespace scale_res_energy_ns;
	pack_wts.set_Watr(copy_Watr * scale_res_energy_weight);
	pack_wts.set_Wsol(copy_Wsol * scale_res_energy_weight);
	pack_wts.set_Whbond_bb(copy_Whbond_bb * scale_res_energy_weight);
	pack_wts.set_Whbond_bb_sc(copy_Whbond_bb_sc * scale_res_energy_weight);
	pack_wts.set_Whbond_sc(copy_Whbond_sc * scale_res_energy_weight);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_weights_to_unscaled_values
///
/// @brief
/// apl  before computing res/res energies with the rotamer trie, set the existing
/// apl  energy weights to their original values, held in the copy_W... variables.
///
/// @detailed
///
/// @param  copy_Watr,
/// @param  copy_Wsol,
/// @param  copy_Whbond_bb,
/// @param  copy_Whbond_bb_sc,
/// @param  copy_Whbond_sc
///
/// @global_read
///
/// @global_write pack_wts.Watr(), pack_wts.Wsol(), pack_wts.Whbond_bb(), pack_wts.Whbond_bb_sc(), pack_wts.Whbond_sc()
///
/// @remarks
///
/// @references
///
/// @authors - apl
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void set_weights_to_unscaled_values(
	float copy_Watr,
	float copy_Wsol,
	float copy_Whbond_bb,
	float copy_Whbond_bb_sc,
	float copy_Whbond_sc
)
{
	using namespace param_pack;

	pack_wts.set_Watr(copy_Watr);
	pack_wts.set_Wsol(copy_Wsol);
	pack_wts.set_Whbond_bb(copy_Whbond_bb);
	pack_wts.set_Whbond_bb_sc(copy_Whbond_bb_sc);
	pack_wts.set_Whbond_sc(copy_Whbond_sc);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_rep_weight_to_scaled_value
///
/// @brief
/// apl  before computing res/res energies with the rotamer trie, scale the
/// apl  pack_wts.Wrep() weight by scale_res_energy_ns::scale_res_rep_energy_weight.
///
/// @detailed
///
/// @param  copy_Wrep,
///
/// @global_read
///
/// @global_write pack_wts.Wrep()
///
/// @remarks
///
/// @references
///
/// @authors - apl
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
set_rep_weight_to_scaled_value(float copy_Wrep)
{
	using namespace scale_res_energy_ns;
	using namespace param_pack;
	pack_wts.set_Wrep(copy_Wrep * scale_res_rep_energy_weight);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_rep_weight_to_unscaled_value
///
/// @brief
/// apl  before computing res/res energies with the rotamer trie, set pack_wts.Wrep()
/// apl  to its original value, held in copy_Wrep.
///
/// @detailed
///
/// @param  copy_Wrep
///
/// @global_read
///
/// @global_write pack_wts.Wrep()
///
/// @remarks
///
/// @references
///
/// @authors - apl
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
set_rep_weight_to_unscaled_value(float copy_Wrep)
{  using namespace param_pack;
	pack_wts.set_Wrep(copy_Wrep);
}
