// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 10959 $
//  $Date: 2006-10-27 20:12:09 +0200 (Fri, 27 Oct 2006) $
//  $Author: ion $

#ifndef INCLUDED_scale_res_energy
#define INCLUDED_scale_res_energy


// Rosetta Headers
#include "misc.h"
#include "score_ns.h"

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray2D.hh>

////////////////////////////////////////////////////////////////////////////////
/// @begin get_scale_res_energy_flag
///
/// @brief
///
/// @detailed
///
/// @return get_scale_res_energy_flag
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Glenn Butterfoss
///
/// @last_modified 18 June 2004
////////////////////////////////////////////////////////////////////////////////
inline
bool
get_scale_res_energy_flag()
{
	using namespace scale_res_energy_ns;

	return scale_res_energy_flag;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin scale_rep_energy
///
/// @brief scale non-repulsion energies between residues by some weight
///
/// @detailed
/// only in effect if scale_res_energy_flag is true
///
/// @param[in]  residue1 - resnumber
/// @param[in]  residue2 - resnumber
/// @param[in]  energy - current energy for something
///
/// @return scaled energy
///
/// @global_read
/// namespace scale_res_energy in score_ns.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Glenn Butterfoss
///
/// @last_modified 18 June 2004
////////////////////////////////////////////////////////////////////////////////
inline
float
scale_res_energy(
	int const residue1,
	int const residue2,
	float const energy
)
{
	using namespace scale_res_energy_ns;

	if ( !get_scale_res_energy_flag() ) {
		return energy;
	}

	if ( !scale_energy_matrix(residue1,residue2) ) {
		return energy;
	} else {
		return scale_res_energy_weight * energy; // new energy
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin scale_res_rep_energy
///
/// @brief scale repulsion between residues
///
/// @detailed
/// only in effect if scale_res_energy_flag is true
///
/// @param[in]  residue1 - resnumber
/// @param[in]  residue2 - resnumber
/// @param[in]  energy - current Erep
///
/// @return scaled repulsion energy
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Glenn Butterfoss
///
/// @last_modified 18 June 2004
////////////////////////////////////////////////////////////////////////////////
inline
float
scale_res_rep_energy(
	int const residue1,
	int const residue2,
	float const energy
)
{
	using namespace scale_res_energy_ns;

	if ( !get_scale_res_energy_flag() ) {
		return energy;
	}

	if ( !scale_rep_energy_matrix(residue1,residue2) ) {
		return energy;
	} else {
		return scale_res_rep_energy_weight * energy; // new energy
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin set_scale_res_energy
///
/// @brief set res to be scaled
///
/// @detailed
///
/// @param[in]  residue1 - resnumber
/// @param[in]  residue2 - resnumber
/// @param[in]  scale_on_off - if true scale energies of this residue
///
/// @global_read
///   namespace scale_res_energy in score_ns.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Glenn Butterfoss
///
/// @last_modified 18 June 2004
////////////////////////////////////////////////////////////////////////////////
inline
void
set_scale_res_energy(
	int const residue1,
	int const residue2,
	bool const scale_on_off
)
{
	using namespace scale_res_energy_ns;

	if ( !get_scale_res_energy_flag() ) {
		return;
	}

	//force symmetry
	scale_energy_matrix(residue1,residue2) =
	 scale_energy_matrix(residue2,residue1) = scale_on_off;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin set_scale_res_rep_energy
///
/// @brief set res to be for which repulsive energy will be scaled
///
/// @detailed
///
/// @param[in]  residue1 - resnumber
/// @param[in]  residue2 - resnumber
/// @param[in]  scale_on_off - if true scale energies of this residue
///
/// @global_read
///   namespace scale_res_energy in score_ns.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Glenn Butterfoss
///
/// @last_modified 18 June 2004
////////////////////////////////////////////////////////////////////////////////
inline
void
set_scale_res_rep_energy(
	int const residue1,
	int const residue2,
	bool const scale_on_off
)
{
	using namespace scale_res_energy_ns;

	if ( !get_scale_res_energy_flag() ) {
		return;
	}

	// force symmetry
	scale_rep_energy_matrix(residue1,residue2) =
	 scale_rep_energy_matrix(residue2,residue1) = scale_on_off;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin set_scale_res_energy_flag
///
/// @brief
/// set array to scale residue
///
/// @detailed
/// input array or (res,bool)?
///
/// @param[in]  flag_on_off - T allow for energy scaling, F no scaling
/// @param[in]  init_to_zero - T fill @c scale_energy_matrix with zero value,
///                F leave array uninitialized.
///
/// @global_read
/// namespaces misc, scale_res_energy (in this file)
///
/// @global_write
/// namespace scale_res_energy in score_ns.h
///
/// @remarks
///
/// @references
///
/// @authors Glenn Butterfoss
///
/// @last_modified 18 June 2004
////////////////////////////////////////////////////////////////////////////////
inline
void
set_scale_res_energy_flag(
	bool const flag_on_off,
	bool const init_to_zero
)
{
	using namespace misc;
	using namespace scale_res_energy_ns;

	scale_res_energy_flag = flag_on_off;

	if ( init_to_zero ) { // init array to 0 (no scaling)
		for ( int j = 1; j <= total_residue; ++j ) {
			for ( int i = 1; i <= total_residue; ++i ) {
				scale_energy_matrix(i,j) =
				 scale_rep_energy_matrix(i,j) = 0;
			}
		}
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin set_scale_res_energy_weight
///
/// @brief
/// set weight for non repulsive energy scaling
///
/// @detailed
/// input array or (res,bool)?
///
/// @param[in]  weight - scale energies by this much
///
/// @global_read
/// namespaces misc, scale_res_energy
///
/// @global_write
/// namespaces  scale_res_energy in score_ns.h
///
/// @remarks
///
/// @references
///
/// @authors Glenn Butterfoss
///
/// @last_modified 18 June 2004
////////////////////////////////////////////////////////////////////////////////
inline
void
set_scale_res_energy_weight(
	float const weight
)
{
	using namespace misc;
	using namespace scale_res_energy_ns;

	if ( !get_scale_res_energy_flag() ) {
		return;
	} else {
		scale_res_energy_weight = weight;
	}
	//std::cout << "scale energy of selected residues by  " << scale_res_energy_weight << std::endl;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin set_scale_res_rep_energy_weight
///
/// @brief
/// set weight for repulsive energy scaling
///
/// @detailed
/// input array or (res,bool)?
///
/// @param[in]  weight - scale repulsion energies by this much
///
/// @global_read
/// namespaces misc, scale_res_energy
///
/// @global_write
/// namespaces  scale_res_energy in score_ns.h
///
/// @remarks
///
/// @references
///
/// @authors Glenn Buterfoss
///
/// @last_modified 18 June 2004
////////////////////////////////////////////////////////////////////////////////
inline
void
set_scale_res_rep_energy_weight(
	float const weight
)
{
	using namespace misc;
	using namespace scale_res_energy_ns;

	if ( !get_scale_res_energy_flag() ) {
		return;
	} else {
		scale_res_rep_energy_weight = weight;
	}
	//std::cout << "scale energy of selected residues by " << scale_res_energy_weight << std::endl;
}


#endif
