// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 7212 $
//  $Date: 2006-01-04 07:49:01 -0800 (Wed, 04 Jan 2006) $
//  $Author: pbradley $


// Rosetta Headers
#include "atom_tree_routines.h"
#include "after_opts.h"
#include "aaproperties_pack.h"
#include "disulfides.h"
#include "fullatom.h"
#include "jumping_util.h"
#include "minimize.h"
#include "misc.h" // pose_flag()
#include "monte_carlo.h"
#include "nblist.h"
#include "output_decoy.h"
#include "pose.h"
#include "pose_io.h"
#include "pose_design.h"
#include "pose_relax.h"
#include "pose_rotamer_trials.h"
#include "prof.h"
#include "relax_structure.h"
#include "read_aaproperties.h"
#include "score.h"
#include "smallmove.h"
#include "torsion_bbmove_trials.h"
#include "util_vector.h"

// Utility Headers
#include <numeric/xyz.functions.hh> // for dihedral()
#include <numeric/trig.functions.hh>


///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
typedef numeric::xyzVector_float Vec;
typedef numeric::xyzVector_double DVec;

///////////////////////////////////////////////////////////////////////////////
numeric::xyzVector_float
cb_local(
				 FArray2Da_float xyz
				 )
{
	xyz.dimension(3,param::MAX_ATOM()());

	Vec
		n ( &xyz(1,1)),
		ca( &xyz(1,2)),
		c ( &xyz(1,3)),
		cb( &xyz(1,5)),
		x,y,z;

	x = ( n - ca ).normalized();
	z = cross( x, c - ca).normalized();
	y = cross(z,x);

	cb -= ca;

	return Vec( dot( cb, x ),
							dot( cb, y ),
							dot( cb, z ) );
}

///////////////////////////////////////////////////////////////////////////////
numeric::xyzVector_float
build_ideal_cbeta(
									int const aa,
									int const aav,
									FArray2Da_float xyz
									)
{
	xyz.dimension(3,param::MAX_ATOM()());

	Vec
		n ( &xyz(1,1)),
		ca( &xyz(1,2)),
		c ( &xyz(1,3)),
		x,y,z;

	x = ( n - ca ).normalized();
	z = cross( x, c - ca).normalized();
	y = cross(z,x);

	Vec const ideal_cb = cb_local( aaproperties_pack::icoor(1,1,aa,aav) );

	return ca + ideal_cb(1) * x + ideal_cb(2) * y + ideal_cb(3) * z;
}

///////////////////////////////////////////////////////////////////////////////
float
cbeta_bond_angle_score(
	int const aa,
	int const aav,
	FArray2Da_float xyz
)
{
	static bool const verbose( truefalseoption("show_bond_angles") );
	if ( !param_aa::is_protein(aa) || aa == param_aa::aa_gly ) return 0.0;

	xyz.dimension(3,param::MAX_ATOM()());

	// get c-beta in local coord sys
	Vec
		cb         ( cb_local( xyz ) ),
		cb_template( cb_local( aaproperties_pack::icoor(1,1,aa,aav) ) );

	float const angle_dev
		( std::acos( numeric::sin_cos_range( dot( cb.normalized(),
																							cb_template.normalized() ) ) ) );

	if ( verbose ) {
		std::cout << "cbeta_dev:  " << param_aa::aa_name3( aa ) <<
			F(9,3,numeric::conversions::degrees( angle_dev )) << std::endl;
	}

	return angle_dev * angle_dev;
}

///////////////////////////////////////////////////////////////////////////////
float
cgamma_angle(
	int const aa,
	int const aav,
	FArray2Da_float xyz
)
{
	xyz.dimension(3,param::MAX_ATOM()());

	int const ca_atomno( LookupByName( aa, aav, " CA " ) );
	int const cb_atomno( LookupByName( aa, aav, " CB " ) );
	int const cg_atomno( LookupByName( aa, aav, " CG " ) );
	Vec
		ca( &xyz(1,ca_atomno)),
		cb( &xyz(1,cb_atomno)),
		cg( &xyz(1,cg_atomno));

	return std::acos( numeric::sin_cos_range( dot( ( cb-ca ).normalized(),
																								 ( cg-cb ).normalized() ) ) );

}

///////////////////////////////////////////////////////////////////////////////
float
cgamma_bond_angle_score(
	int const aa,
	int const aav,
	FArray2Da_float xyz
)
{
	static bool const verbose( truefalseoption("show_bond_angles") );

	int const atomno( LookupByName( aa,aav," CG "));
	if ( atomno < 1 ) return 0.0;

	float const current( cgamma_angle( aa,aav,xyz) );
	float const ideal( cgamma_angle( aa,aav,aaproperties_pack::icoor(1,1,aa,aav)));

	float const angle_dev( current - ideal );
	if ( verbose ) {
		std::cout << "cgamma_dev: "<< param_aa::aa_name3( aa ) <<
			F(9,3,numeric::conversions::degrees(angle_dev)) << std::endl;
	}

	return ( current - ideal ) * ( current - ideal );
}


///////////////////////////////////////////////////////////////////////////////
float
eval_bond_angle_score(
											pose_ns::Pose const & pose
											)
{
	static bool const verbose( truefalseoption("show_bond_angles") );
	float score(0.0);

	for ( int i=1; i<= pose.total_residue(); ++i ) {
		float rsd_score(0.0);
		rsd_score += cbeta_bond_angle_score( pose.res(i), pose.res_variant(i),
																				 pose.full_coord()(1,1,i) );
		rsd_score += cgamma_bond_angle_score( pose.res(i), pose.res_variant(i),
																					pose.full_coord()(1,1,i) );
		if ( verbose ) {
			std::cout << "rsd_bond_angle_score: " <<
				param_aa::aa_name3( pose.res(i) ) << F(12,6,rsd_score) << std::endl;
		}
		score += rsd_score;
	}
	if ( verbose ) {
		std::cout << "total_bond_angle_score/rsd: " <<
			F(12,6,score/pose.total_residue()) << std::endl;
	}

	return score;
}


///////////////////////////////////////////////////////////////////////////////
float
eval_bond_angle_deriv(
											kin::Torsion_id const & tor,
											pose_ns::Pose const & pose
											)
{
	using namespace kin;
	int const CBETA(5);

	Atom_id id( tor.atom_id() );
	Kin_torsion_type t( tor.type() );

	int const seqpos( id.rsd() );
	int const aa ( pose.res        (seqpos) );
	int const aav( pose.res_variant(seqpos) );

	int const CGAMMA( LookupByName( aa,aav," CG "));

	if ( id.atomno() == CBETA && ( t == PHI || t == THETA ) ) {


		//determine the internal coordinates of the ideal cbeta position
		Coords_FArray_const coords( pose.full_coord() );
		Stub stub( pose.get_atom_tree_atom( id )->get_input_stub( coords ) );

		DVec w( build_ideal_cbeta( aa, aav, pose.full_coord()(1,1,seqpos) ));

		/////////////////////////////////////////////////////////
		// this code stolen from kin_bonded_atom::update_torsions
		w-= stub.v;
		w.normalize();
		float const x( dot( w, stub.M.col_x() ) );
		float const y( dot( w, stub.M.col_y() ) );
		float const z( dot( w, stub.M.col_z() ) );

		float const ideal_theta = std::acos( x ); // DANGER
		float const ideal_phi  = std::atan2( z, y );
		//
		/////////////////////////////////////////////////////////

		float const current_val( pose.get_atom_tree_torsion( tor ) );



		if ( t == PHI ) {
			float const current_theta
				( pose.get_atom_tree_torsion( CBETA, seqpos, THETA ) );
			return 2.0 * std::sin( current_theta ) *
				std::sin( current_theta ) * ( current_val - ideal_phi );

		} else if ( t == THETA ) {
			return 2.0 * ( current_val - ideal_theta );

		}

	} else if ( id.atomno() == CGAMMA && t == THETA ) {

		float const current( cgamma_angle( aa,aav,pose.full_coord()(1,1,seqpos) ));
		float const ideal
			( cgamma_angle( aa,aav,aaproperties_pack::icoor(1,1,aa,aav)));

		float const current_theta
			( pose.get_atom_tree_torsion( CGAMMA, seqpos, THETA ) );

		assert( std::abs( current_theta - current ) < 1e-2 );

		return 2.0 * ( current - ideal );

	} else {
		return 0.0;
	}

	return 0.0;
}
