#ifndef INCLUDED_ObjexxFCL_FArray2Da_HH
#define INCLUDED_ObjexxFCL_FArray2Da_HH


// FArray2Da: Fortran-Compatible 2D Argument Array
//
// Project: Objexx Fortran Compatibility Library (ObjexxFCL)
//
// Version: 2.6.2
//
// Language: C++
//
// Copyright (c) 2007 Objexx Engineering, Inc. All Rights Reserved.
// Use of this source code or any derivative of it is restricted by license.
// Licensing is available from Objexx Engineering, Inc.:   http://objexx.com   Objexx@objexx.com


// ObjexxFCL Headers
#include <ObjexxFCL/FArray2Da.fwd.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray2Dp.hh>
#include <ObjexxFCL/StaticIndexRange.hh>


namespace ObjexxFCL {


/// @brief FArray2Da: Fortran-Compatible 2D Argument Array
template< typename T >
class FArray2Da :
	public FArray2DB< T >
{


private: // Types


	typedef  FArray2DB< T >  Super;
	typedef  typename Super::real_FArray  real_FArray;
	typedef  typename Super::proxy_FArray  proxy_FArray;
	typedef  typename Super::arg_FArray  arg_FArray;
	typedef  internal::ProxySentinel  ProxySentinel;


public: // Types


	typedef  typename Super::Base  Base;
	typedef  typename Base::Section  Section;
	typedef  typename Super::IR  SIR;
	typedef  StaticIndexRange  IR;

	// STL Style
	typedef  typename Base::value_type  value_type;
	typedef  typename Base::reference  reference;
	typedef  typename Base::const_reference  const_reference;
	typedef  typename Base::pointer  pointer;
	typedef  typename Base::const_pointer  const_pointer;
	typedef  typename Base::size_type  size_type;
	typedef  typename Base::difference_type  difference_type;

	// C++ Style
	typedef  typename Base::Value  Value;
	typedef  typename Base::Reference  Reference;
	typedef  typename Base::ConstReference  ConstReference;
	typedef  typename Base::Pointer  Pointer;
	typedef  typename Base::ConstPointer  ConstPointer;
	typedef  typename Base::Size  Size;
	typedef  typename Base::Difference  Difference;

	using Super::array_;
	using Super::array_size_;
	using Super::npos;
	using Super::sarray_;
	using Super::shift_;
	using Super::shift_set;
	using Super::size_set;
	using Super::s1_;


public: // Creation


	/// @brief Default Constructor
	inline
	FArray2Da() :
		Super( ProxySentinel() )
	{}


	/// @brief Copy Constructor
	inline
	FArray2Da( FArray2Da const & a ) :
		Super( a, ProxySentinel() ),
		I1_( a.I1_ ),
		I2_( a.I2_ )
	{
		shift_set( a.shift_ );
		s1_ = a.s1_;
	}


	/// @brief Real Constructor
	inline
	FArray2Da( real_FArray const & a ) :
		Super( a, ProxySentinel() ),
		I1_( a.I1_ ),
		I2_( a.I2_ )
	{
		shift_set( a.shift_ );
		s1_ = a.s1_;
	}


	/// @brief Proxy Constructor
	inline
	FArray2Da( proxy_FArray const & a ) :
		Super( a, ProxySentinel() ),
		I1_( a.I1_ ),
		I2_( a.I2_ )
	{
		shift_set( a.shift_ );
		s1_ = a.s1_;
	}


	/// @brief Super Constructor
	inline
	FArray2Da( Super const & a ) :
		Super( a, ProxySentinel() ),
		I1_( a.I1() ),
		I2_( a.I2() )
	{
		shift_set( a.shift_ );
		s1_ = a.s1_;
	}


	/// @brief Base Constructor
	inline
	FArray2Da( Base const & a ) :
		Super( a, ProxySentinel() ),
		I1_( 1 ),
		I2_( a.size() )
	{
		shift_set( 2 );
		s1_ = 1;
	}


	/// @brief Section Constructor
	inline
	FArray2Da( Section const & s ) :
		Super( s, ProxySentinel() ),
		I1_( 1 ),
		I2_( s.size() )
	{
		shift_set( 2 );
		s1_ = 1;
	}


	/// @brief Value Constructor
	inline
	FArray2Da( value_type const & t ) :
		Super( t, ProxySentinel() ),
		I1_( 1 ),
		I2_( star ) // Unbounded
	{
		shift_set( 2 );
		s1_ = 1;
	}


	/// @brief Copy + IndexRange Constructor
	inline
	FArray2Da( FArray2Da const & a, IR const & I1_a, IR const & I2_a ) :
		Super( a, ProxySentinel() ),
		I1_( I1_a ),
		I2_( I2_a )
	{
		dimension_argument();
	}


	/// @brief Super + IndexRange Constructor
	inline
	FArray2Da( Super const & a, IR const & I1_a, IR const & I2_a ) :
		Super( a, ProxySentinel() ),
		I1_( I1_a ),
		I2_( I2_a )
	{
		dimension_argument();
	}


	/// @brief Base + IndexRange Constructor
	inline
	FArray2Da( Base const & a, IR const & I1_a, IR const & I2_a ) :
		Super( a, ProxySentinel() ),
		I1_( I1_a ),
		I2_( I2_a )
	{
		dimension_argument();
	}


	/// @brief Section + IndexRange Constructor
	inline
	FArray2Da( Section const & s, IR const & I1_a, IR const & I2_a ) :
		Super( s, ProxySentinel() ),
		I1_( I1_a ),
		I2_( I2_a )
	{
		dimension_argument();
	}


	/// @brief Value + IndexRange Constructor
	inline
	FArray2Da( value_type const & t, IR const & I1_a, IR const & I2_a ) :
		Super( t, ProxySentinel() ),
		I1_( I1_a ),
		I2_( I2_a )
	{
		dimension_argument();
	}


	/// @brief Destructor
	inline
	virtual
	~FArray2Da()
	{}


public: // Assignment


	/// @brief Copy Assignment
	inline
	FArray2Da &
	operator =( FArray2Da const & a )
	{
		if ( this != &a ) {
			if ( ! equal_dimension( a ) ) dimension( a );
			Base::operator =( a );
		}
		return *this;
	}


	/// @brief Super Assignment
	inline
	FArray2Da &
	operator =( Super const & a )
	{
		if ( this != &a ) {
			if ( ! equal_dimension( a ) ) dimension( a );
			Base::operator =( a );
		}
		return *this;
	}


	/// @brief Super Assignment Template
	template< typename U >
	inline
	FArray2Da &
	operator =( FArray2DB< U > const & a )
	{
		if ( ! equal_dimension( a ) ) dimension( a );
		Base::operator =( a );
		return *this;
	}


	/// @brief += Array Template
	template< typename U >
	inline
	FArray2Da &
	operator +=( FArray2DB< U > const & a )
	{
		Super::operator +=( a );
		return *this;
	}


	/// @brief -= Array Template
	template< typename U >
	inline
	FArray2Da &
	operator -=( FArray2DB< U > const & a )
	{
		Super::operator -=( a );
		return *this;
	}


	/// @brief *= Array Template
	template< typename U >
	inline
	FArray2Da &
	operator *=( FArray2DB< U > const & a )
	{
		Super::operator *=( a );
		return *this;
	}


	/// @brief = Value
	inline
	FArray2Da &
	operator =( value_type const & t )
	{
		Super::operator =( t );
		return *this;
	}


	/// @brief += Value
	inline
	FArray2Da &
	operator +=( value_type const & t )
	{
		Super::operator +=( t );
		return *this;
	}


	/// @brief -= Value
	inline
	FArray2Da &
	operator -=( value_type const & t )
	{
		Super::operator -=( t );
		return *this;
	}


	/// @brief *= Value
	inline
	FArray2Da &
	operator *=( value_type const & t )
	{
		Super::operator *=( t );
		return *this;
	}


	/// @brief /= Value
	inline
	FArray2Da &
	operator /=( value_type const & t )
	{
		Super::operator /=( t );
		return *this;
	}


public: // Subscript


	/// @brief array( i1, i2 ) const
	inline
	value_type const &
	operator ()( int const i1, int const i2 ) const
	{
		assert( ( I1_.contains( i1 ) ) && ( I2_.contains( i2 ) ) );
		return sarray_[ ( i2 * s1_ ) + i1 ];
	}


	/// @brief array( i1, i2 )
	inline
	value_type &
	operator ()( int const i1, int const i2 )
	{
		assert( ( I1_.contains( i1 ) ) && ( I2_.contains( i2 ) ) );
		return sarray_[ ( i2 * s1_ ) + i1 ];
	}


	/// @brief Section Starting at array( i1, i2 )
	inline
	Section const
	a( int const i1, int const i2 ) const
	{
		assert( ( I1_.contains( i1 ) ) && ( I2_.contains( i2 ) ) );
		size_type const offset( ( ( i2 * s1_ ) + i1 ) - shift_ );
		return Section( ( array_size_ != npos ) ? array_size_ - offset : npos, array_ + offset );
	}


	/// @brief Linear Index
	inline
	size_type
	index( int const i1, int const i2 ) const
	{
		return ( ( ( i2 * s1_ ) + i1 ) - shift_ );
	}


public: // Predicate


	/// @brief Dimensions Initialized?
	inline
	bool
	dimensions_initialized() const
	{
		return true;
	}


	/// @brief Contains Indexed Element?
	inline
	bool
	contains( int const i1, int const i2 ) const
	{
		return ( ( I1_.contains( i1 ) ) && ( I2_.contains( i2 ) ) );
	}


	/// @brief Initializer Active?
	inline
	bool
	initializer_active() const
	{
		return false;
	}


public: // Inspector


	/// @brief IndexRange of Dimension 1
	inline
	IR const &
	I1() const
	{
		return I1_;
	}


	/// @brief Lower Index of Dimension 1
	inline
	int
	l1() const
	{
		return I1_.l();
	}


	/// @brief Upper Index of Dimension 1
	inline
	int
	u1() const
	{
		return I1_.u();
	}


	/// @brief IndexRange of Dimension 2
	inline
	IR const &
	I2() const
	{
		return I2_;
	}


	/// @brief Lower Index of Dimension 2
	inline
	int
	l2() const
	{
		return I2_.l();
	}


	/// @brief Upper Index of Dimension 2
	inline
	int
	u2() const
	{
		return I2_.u();
	}


	/// @brief Size of Dimension 2
	inline
	size_type
	size2() const
	{
		return I2_.size();
	}


public: // Modifier


	/// @brief Clear
	inline
	FArray2Da &
	clear()
	{
		Super::clear();
		I1_.clear();
		I2_.clear();
		return *this;
	}


	/// @brief Dimension by IndexRange
	inline
	FArray2Da &
	dimension( IR const & I1_a, IR const & I2_a )
	{
		I1_.assign_value_of( I1_a );
		I2_.assign_value_of( I2_a );
		dimension_argument();
		return *this;
	}


	/// @brief Dimension by Array
	template< typename U >
	inline
	FArray2Da &
	dimension( FArray2DB< U > const & a )
	{
		I1_.assign_value_of( a.I1() );
		I2_.assign_value_of( a.I2() );
		dimension_argument();
		return *this;
	}


	/// @brief Attach to Argument Array
	inline
	FArray2Da &
	attach( FArray2Da const & a )
	{
		Base::attach( a );
		s1_ = a.s1_;
		I1_.assign_value_of( a.I1_ );
		I2_.assign_value_of( a.I2_ );
		return *this;
	}


	/// @brief Attach to Real Array
	inline
	FArray2Da &
	attach( real_FArray const & a )
	{
		Base::attach( a );
		s1_ = a.s1_;
		I1_.assign_value_of( a.I1_ );
		I2_.assign_value_of( a.I2_ );
		return *this;
	}


	/// @brief Attach to Proxy Array
	inline
	FArray2Da &
	attach( proxy_FArray const & a )
	{
		Base::attach( a );
		s1_ = a.s1_;
		I1_.assign_value_of( a.I1_ );
		I2_.assign_value_of( a.I2_ );
		return *this;
	}


	/// @brief Attach to Super Array
	inline
	FArray2Da &
	attach( Super const & a )
	{
		Base::attach( a );
		s1_ = a.s1_;
		I1_.assign_value_of( a.I1() );
		I2_.assign_value_of( a.I2() );
		return *this;
	}


	/// @brief Attach to Base Array
	inline
	FArray2Da &
	attach( Base const & a )
	{
		Base::attach( a, 2 );
		s1_ = 1;
		I1_ = 1;
		I2_ = a.size();
		return *this;
	}


	/// @brief Attach to Section
	inline
	FArray2Da &
	attach( Section const & s )
	{
		Base::attach( s, 2 );
		s1_ = 1;
		I1_ = 1;
		I2_ = s.size();
		return *this;
	}


	/// @brief Attach to Value
	inline
	FArray2Da &
	attach( value_type const & t )
	{
		Base::attach( t, 2 );
		s1_ = 1;
		I1_ = 1;
		I2_ = star; // Unbounded
		return *this;
	}


	/// @brief Detach from Source Array
	inline
	FArray2Da &
	detach()
	{
		Base::detach();
		s1_ = 0;
		I1_.clear();
		I2_.clear();
		return *this;
	}


protected: // Functions


	/// @brief Dimension by IndexRange
	inline
	void
	dimension_assign( SIR const & I1_a, SIR const & I2_a )
	{
		I1_.assign_value_of( I1_a );
		I2_.assign_value_of( I2_a );
		dimension_argument();
	}


private: // Functions


	/// @brief Setup for Current IndexRange Dimensions
	inline
	void
	dimension_argument()
	{
		assert( I1_.bounded_value() );
		s1_ = I1_.size();
		if ( I2_.bounded_value() ) { // Bounded
			size_set( size_of( s1_, I2_.size() ) );
		} else if ( array_size_ != npos ) { // Unbounded with bounded data array
			if ( s1_ > 0 ) { // Infer upper index and size
				I2_.u( I2_.lz() + ( array_size_ / s1_ ) - 1 );
				size_set( size_of( s1_, I2_.size() ) );
			} else {
				size_set( array_size_ );
			}
		} else { // Unbounded with unbounded data array
			size_set( npos );
		}
		shift_set( ( I2_.lz() * s1_ ) + I1_.lz() );
	}


private: // Data


	/// @brief Dimension 1 index range
	IR I1_;

	/// @brief Dimension 2 index range
	IR I2_;


}; // FArray2Da


} // namespace ObjexxFCL


#endif // INCLUDED_ObjexxFCL_FArray2Da_HH
