#ifndef INCLUDED_ObjexxFCL_FArrayB_HH
#define INCLUDED_ObjexxFCL_FArrayB_HH


// FArrayB: Fortran-Compatible Array Hierarchy Abstract Base Class
//
// Project: Objexx Fortran Compatibility Library (ObjexxFCL)
//
// Version: 2.6.2
//
// Language: C++
//
// Copyright (c) 2007 Objexx Engineering, Inc. All Rights Reserved.
// Use of this source code or any derivative of it is restricted by license.
// Licensing is available from Objexx Engineering, Inc.:   http://objexx.com   Objexx@objexx.com


// ObjexxFCL Headers
#include <ObjexxFCL/FArrayB.fwd.hh>
#include <ObjexxFCL/FArraySection.hh>
#include <ObjexxFCL/FArrayTraits.hh>
#include <ObjexxFCL/IndexRange.hh>
#include <ObjexxFCL/internal/InitializerSentinel.hh>
#include <ObjexxFCL/internal/ProxySentinel.hh>

// C++ Standard Library Headers
#include <algorithm>
#include <cassert>
#include <cstddef>
#ifdef OBJEXXFCL_FARRAY_SIZE_REPORT
#include <iostream>
#include <typeinfo>
#endif // OBJEXXFCL_FARRAY_SIZE_REPORT

namespace ObjexxFCL {


/// @brief FArrayB: Fortran-Compatible Array Hierarchy Abstract Base Class
///
/// @remarks
///  @li Can hold numeric or non-numeric values: Numeric operations on non-numeric values won't compile
///  @li Any meaningful array index ranges can be specified as in Fortran
///  @li Column-major storage order used as in Fortran
///  @li Zero-sized arrays are supported but have no valid indices
///  @li Argument/proxy arrays can have unbounded/unknown size
///  @li For efficiency constructors without initializer function or value do not initialize the array
///  @li For efficiency bounds checking is only active via asserts in debug builds
template< typename T >
class FArrayB
{


private: // Friend


	template< typename > friend class FArrayB;


protected: // Types


	typedef  internal::InitializerSentinel  InitializerSentinel;
	typedef  internal::ProxySentinel  ProxySentinel;


public: // Types


	typedef  FArrayB< T >  Base;
	typedef  FArrayTraits< T >  Traits;
	typedef  FArraySection< T >  Section;
	typedef  IndexRange  IR;

	// STL style
	typedef  T  value_type;
	typedef  T &  reference;
	typedef  T const &  const_reference;
	typedef  T *  pointer;
	typedef  T const *  const_pointer;
	typedef  std::size_t  size_type;
	typedef  std::ptrdiff_t  difference_type;

	// C++ style
	typedef  T  Value;
	typedef  T &  Reference;
	typedef  T const &  ConstReference;
	typedef  T *  Pointer;
	typedef  T const *  ConstPointer;
	typedef  std::size_t  Size;
	typedef  std::ptrdiff_t  Difference;


protected: // Creation


	/// @brief Default Constructor
	inline
	FArrayB() :
		array_size_( 0 ),
		array_( 0 ),
		size_( 0 ),
		owner_( true ),
		shift_( 0 ),
		sarray_( 0 )
	{
#ifdef OBJEXXFCL_FARRAY_SIZE_REPORT
		size_report();
#endif // OBJEXXFCL_FARRAY_SIZE_REPORT
	}


	/// @brief Copy Constructor
	inline
	FArrayB( FArrayB const & a ) :
		array_size_( size_of( a.size_ ) ),
		array_( array_size_ > 0 ? new value_type[ array_size_ ] : 0 ),
		size_( array_size_ ),
		owner_( true ),
		shift_( a.shift_ ),
		sarray_( array_ - shift_ )
	{
		for ( size_type i = 0; i < size_; ++i ) {
			array_[ i ] = a[ i ];
		}
#ifdef OBJEXXFCL_FARRAY_SIZE_REPORT
		size_report();
#endif // OBJEXXFCL_FARRAY_SIZE_REPORT
	}


	/// @brief Copy Constructor Template
	template< typename U >
	inline
	explicit
	FArrayB( FArrayB< U > const & a ) :
		array_size_( size_of( a.size() ) ),
		array_( array_size_ > 0 ? new value_type[ array_size_ ] : 0 ),
		size_( array_size_ ),
		owner_( true ),
		shift_( a.shift_ ),
		sarray_( array_ - shift_ )
	{
		for ( size_type i = 0; i < size_; ++i ) {
			array_[ i ] = value_type( a[ i ] );
		}
#ifdef OBJEXXFCL_FARRAY_SIZE_REPORT
		size_report();
#endif // OBJEXXFCL_FARRAY_SIZE_REPORT
	}


	/// @brief Size Constructor
	inline
	explicit
	FArrayB( size_type const size_a ) :
		array_size_( size_of( size_a ) ),
		array_( array_size_ > 0 ? new value_type[ array_size_ ] : 0 ),
		size_( array_size_ ),
		owner_( true )
	{
#ifdef OBJEXXFCL_FARRAY_INIT
		std::fill_n( array_, size_, Traits::initial_value() );
#endif // OBJEXXFCL_FARRAY_INIT
#ifdef OBJEXXFCL_FARRAY_SIZE_REPORT
		size_report();
#endif // OBJEXXFCL_FARRAY_SIZE_REPORT
	}


	/// @brief Size + InitializerSentinel Constructor
	inline
	FArrayB( size_type const size_a, InitializerSentinel const & ) :
		array_size_( size_of( size_a ) ),
		array_( array_size_ > 0 ? new value_type[ array_size_ ] : 0 ),
		size_( array_size_ ),
		owner_( true )
	{
#ifdef OBJEXXFCL_FARRAY_SIZE_REPORT
		size_report();
#endif // OBJEXXFCL_FARRAY_SIZE_REPORT
	}


	/// @brief Default Proxy Constructor
	inline
	FArrayB( ProxySentinel const & ) :
		array_size_( 0 ),
		array_( 0 ),
		size_( 0 ),
		owner_( false ),
		shift_( 0 ),
		sarray_( 0 )
	{}


	/// @brief Array Proxy Constructor
	inline
	FArrayB( FArrayB const & a, ProxySentinel const & ) :
		array_size_( a.array_size_ ),
		array_( a.array_ ),
		size_( a.size_ ),
		owner_( false )
	{}


	/// @brief Section Proxy Constructor
	inline
	FArrayB( Section const & s, ProxySentinel const & ) :
		array_size_( s.size() ),
		array_( s.array() ),
		size_( array_size_ ),
		owner_( false )
	{}


	/// @brief Value Proxy Constructor
	inline
	FArrayB( value_type const & t, ProxySentinel const & ) :
		array_size_( npos ), // Unknown
		array_( const_cast< value_type * >( &t ) ),
		size_( npos ), // Unbounded
		owner_( false )
	{}


public: // Creation


	/// @brief Destructor
	inline
	virtual
	~FArrayB()
	{
		if ( owner_ ) delete[] array_;
	}


protected: // Assignment


	/// @brief Copy Assignment
	inline
	void
	operator =( FArrayB const & a )
	{
		assert( size_bounded() );
		assert( size_ == a.size_ );
		for ( size_type i = 0; i < size_; ++i ) {
			array_[ i ] = a[ i ];
		}
	}


	/// @brief Copy Assignment Template
	template< typename U >
	inline
	void
	operator =( FArrayB< U > const & a )
	{
		assert( size_bounded() );
		assert( size_ == a.size() );
		for ( size_type i = 0; i < size_; ++i ) {
			array_[ i ] = value_type( a[ i ] );
		}
	}


	/// @brief += Array Template
	template< typename U >
	inline
	void
	operator +=( FArrayB< U > const & a )
	{
		assert( size_bounded() );
		assert( size_ == a.size() );
		for ( size_type i = 0; i < size_; ++i ) {
			array_[ i ] += value_type( a[ i ] );
		}
	}


	/// @brief -= Array Template
	template< typename U >
	inline
	void
	operator -=( FArrayB< U > const & a )
	{
		assert( size_bounded() );
		assert( size_ == a.size() );
		for ( size_type i = 0; i < size_; ++i ) {
			array_[ i ] -= value_type( a[ i ] );
		}
	}


public: // Assignment


	/// @brief = Value
	inline
	FArrayB &
	operator =( value_type const & t )
	{
		assert( size_bounded() );
		// Fixing gcc profiler crashing: Revert back to revision 10273: 'for-loop' version.
		// std::fill_n( array_, size_, t );
		for ( size_type i = 0; i < size_; ++i ) { 
			array_[ i ] = t; 
		} 
		return *this;
	}


	/// @brief += Value
	inline
	FArrayB &
	operator +=( value_type const & t )
	{
		assert( size_bounded() );
		for ( size_type i = 0; i < size_; ++i ) {
			array_[ i ] += t;
		}
		return *this;
	}


	/// @brief -= Value
	inline
	FArrayB &
	operator -=( value_type const & t )
	{
		assert( size_bounded() );
		for ( size_type i = 0; i < size_; ++i ) {
			array_[ i ] -= t;
		}
		return *this;
	}


	/// @brief *= Value
	inline
	FArrayB &
	operator *=( value_type const & t )
	{
		assert( size_bounded() );
		for ( size_type i = 0; i < size_; ++i ) {
			array_[ i ] *= t;
		}
		return *this;
	}


	/// @brief /= Value
	inline
	FArrayB &
	operator /=( value_type const & t )
	{
		assert( size_bounded() );
		assert( t != value_type( 0 ) );
		for ( size_type i = 0; i < size_; ++i ) {
			array_[ i ] /= t;
		}
		return *this;
	}


public: // Subscript


	/// @brief array[ i ] const: Linear Subscript
	inline
	value_type const &
	operator []( size_type const i ) const
	{
		assert( ( i < size_ ) || ( size_ == npos ) );
		return array_[ i ];
	}


	/// @brief array[ i ]: Linear Subscript
	inline
	value_type &
	operator []( size_type const i )
	{
		assert( ( i < size_ ) || ( size_ == npos ) );
		return array_[ i ];
	}


public: // Predicate


	/// @brief Dimensions Initialized?
	virtual
	bool
	dimensions_initialized() const = 0;


	/// @brief Initializer Active?
	virtual
	bool
	initializer_active() const = 0;


	/// @brief Active?
	inline
	bool
	active() const
	{
		return ( array_ != 0 );
	}


	/// @brief Array Size Bounded?
	inline
	bool
	array_size_bounded() const
	{
		return ( array_size_ != npos );
	}


	/// @brief Array Size Unbounded?
	inline
	bool
	array_size_unbounded() const
	{
		return ( array_size_ == npos );
	}


	/// @brief Active Array Size Bounded?
	inline
	bool
	size_bounded() const
	{
		return ( size_ != npos );
	}


	/// @brief Active Array Size Unbounded?
	inline
	bool
	size_unbounded() const
	{
		return ( size_ == npos );
	}


	/// @brief Owner?
	inline
	bool
	owner() const
	{
		return owner_;
	}


	/// @brief All Elements Default Valued?
	inline
	bool
	is_default() const
	{
		for ( size_type i = 0; i < size_; ++i ) {
			if ( array_[ i ] != Traits::initial_value() ) return false;
		}
		return true;
	}


	/// @brief All Elements Zero?
	inline
	bool
	is_zero() const
	{
		for ( size_type i = 0; i < size_; ++i ) {
			if ( array_[ i ] != value_type( 0 ) ) return false;
		}
		return true;
	}


	/// @brief Uniform Valued?
	inline
	bool
	is_uniform() const
	{
		if ( size_ <= 1 ) return true;
		value_type const t( array_[ 0 ] );
		for ( size_type i = 1; i < size_; ++i ) {
			if ( array_[ i ] != t ) return false;
		}
		return true;
	}


	/// @brief Uniform Valued with Specified Value?
	inline
	bool
	is_uniform( value_type const & t ) const
	{
		for ( size_type i = 0; i < size_; ++i ) {
			if ( array_[ i ] != t ) return false;
		}
		return true;
	}


public: // Inspector


	/// @brief Array Size
	inline
	size_type
	array_size() const
	{
		return array_size_;
	}


	/// @brief Active Array Size
	inline
	size_type
	size() const
	{
		return size_;
	}


public: // Modifier


	/// @brief Clear
	inline
	virtual
	FArrayB &
	clear()
	{
		if ( owner_ ) {
			array_size_ = 0;
			delete[] array_; array_ = 0;
		}
		size_ = 0;
		shift_ = 0;
		sarray_ = array_;
		return *this;
	}


	/// @brief Assign Default Value to all Elements
	inline
	virtual
	FArrayB &
	to_default()
	{
		std::fill_n( array_, size_, Traits::initial_value() );
		return *this;
	}


	/// @brief Assign Zero to all Elements
	/// @note  Can't be virtual (for covariant return) or will try to instantiate for all value types
	inline
	void
	zero()
	{
		std::fill_n( array_, size_, value_type( 0 ) );
	}


	/// @brief Assign Zero to all Elements
	/// @note  Can't be virtual (for covariant return) or will try to instantiate for all value types
	inline
	void
	to_zero()
	{
		std::fill_n( array_, size_, value_type( 0 ) );
	}


//	/// @brief Set Elements to Zero Bytes
//	/// @note  Not meaningful/safe for non-POD value_type
//	inline
//	void
//	to_zero_bytes()
//	{
//		std::memset( array_, 0, size_ * sizeof( value_type ) );
//	}


protected: // Comparison


	/// @brief FArrayB == FArrayB
	friend
	inline
	bool
	operator ==( FArrayB const & a, FArrayB const & b )
	{
		if ( &a == &b ) { // Same objects
			return true;
		} else if ( a.size_ == b.size_ ) { // Sizes match
			for ( typename FArrayB::size_type i = 0, e = a.size_; i < e; ++i ) {
				if ( a[ i ] != b[ i ] ) return false;
			}
			return true;
		} else { // Sizes differ
			return false;
		}
	}


	/// @brief FArrayB != FArrayB
	friend
	inline
	bool
	operator !=( FArrayB const & a, FArrayB const & b )
	{
		return !( a == b );
	}


protected: // Functions


	/// @brief Array Size Product of Specified Bounded Dimensional Sizes
	inline
	static
	size_type
	size_of( size_type const s1 )
	{
		assert( s1 != npos );
		return s1;
	}


	/// @brief Array Size Product of Specified Bounded Dimensional Sizes
	inline
	static
	size_type
	size_of( size_type const s1, size_type const s2 )
	{
		assert( s1 != npos );
		assert( s2 != npos );
		assert( ( s2 == 0 ) || ( s1 <= max_size / s2 ) );
		return s1 * s2;
	}


	/// @brief Array Size Product of Specified Bounded Dimensional Sizes
	inline
	static
	size_type
	size_of( size_type const s1, size_type const s2, size_type const s3 )
	{
		return size_of( size_of( s1, s2 ), s3 );
	}


	/// @brief Array Size Product of Specified Bounded Dimensional Sizes
	inline
	static
	size_type
	size_of( size_type const s1, size_type const s2, size_type const s3, size_type const s4 )
	{
		return size_of( size_of( s1, s2 ), size_of( s3, s4 ) );
	}


	/// @brief Array Size Product of Specified Bounded Dimensional Sizes
	inline
	static
	size_type
	size_of( size_type const s1, size_type const s2, size_type const s3, size_type const s4, size_type const s5 )
	{
		return size_of( size_of( size_of( s1, s2 ), size_of( s3, s4 ) ), s5 );
	}


	/// @brief Array Size Product of Specified Bounded IndexRanges
	inline
	static
	size_type
	size_of( IR const & I1 )
	{
		return size_of( I1.size() );
	}


	/// @brief Array Size Product of Specified Bounded IndexRanges
	inline
	static
	size_type
	size_of( IR const & I1, IR const & I2 )
	{
		return size_of( I1.size(), I2.size() );
	}


	/// @brief Array Size Product of Specified Bounded IndexRanges
	inline
	static
	size_type
	size_of( IR const & I1, IR const & I2, IR const & I3 )
	{
		return size_of( I1.size(), I2.size(), I3.size() );
	}


	/// @brief Array Size Product of Specified Bounded IndexRanges
	inline
	static
	size_type
	size_of( IR const & I1, IR const & I2, IR const & I3, IR const & I4 )
	{
		return size_of( I1.size(), I2.size(), I3.size(), I4.size() );
	}


	/// @brief Array Size Product of Specified Bounded IndexRanges
	inline
	static
	size_type
	size_of( IR const & I1, IR const & I2, IR const & I3, IR const & I4, IR const & I5 )
	{
		return size_of( I1.size(), I2.size(), I3.size(), I4.size(), I5.size() );
	}


	/// @brief Pointer to Data Array
	inline
	value_type *
	array() const
	{
		return array_;
	}


	/// @brief Shift Setup
	inline
	void
	shift_set( int const shift_a )
	{
		shift_ = shift_a;
		sarray_ = array_ - shift_;
	}


	/// @brief Active Array Size Setup
	inline
	void
	size_set( size_type const size_a )
	{
		assert( size_a <= array_size_ );
		size_ = size_a;
	}


	/// @brief Resize a Real Array
	inline
	FArrayB &
	resize( size_type const size_a )
	{
		assert( owner_ );
		assert( size_a != npos );
		if ( array_size_ != size_a ) {
			array_size_ = size_a;
			delete[] array_; array_ = ( array_size_ > 0 ? new value_type[ array_size_ ] : 0 );
			size_ = size_a;
#ifdef OBJEXXFCL_FARRAY_SIZE_REPORT
			size_report();
#endif // OBJEXXFCL_FARRAY_SIZE_REPORT
		}
#ifdef OBJEXXFCL_FARRAY_INIT
		if ( ! initializer_active() ) {
			std::fill_n( array_, size_, Traits::initial_value() );
		}
#endif // OBJEXXFCL_FARRAY_INIT
		return *this;
	}


	/// @brief Attach Proxy/Argument Array to Array of Same Rank
	inline
	void
	attach( FArrayB const & a )
	{
		assert( ! owner_ );
		array_size_ = a.array_size_;
		array_ = a.array_;
		size_ = a.size_;
		shift_ = a.shift_;
		sarray_ = array_ - shift_;
	}


	/// @brief Attach Proxy/Argument Array to Array
	inline
	void
	attach( FArrayB const & a, int const shift_a )
	{
		assert( ! owner_ );
		array_size_ = a.array_size_;
		array_ = a.array_;
		size_ = a.size_;
		shift_ = shift_a;
		sarray_ = array_ - shift_;
	}


	/// @brief Attach Proxy/Argument Array to Section
	inline
	void
	attach( Section const & s, int const shift_a )
	{
		assert( ! owner_ );
		array_size_ = s.size();
		array_ = s.array();
		size_ = array_size_;
		shift_ = shift_a;
		sarray_ = array_ - shift_;
	}


	/// @brief Attach Proxy/Argument Array to Value
	inline
	void
	attach( value_type const & t, int const shift_a )
	{
		assert( ! owner_ );
		array_size_ = npos; // Unknown
		array_ = const_cast< value_type * >( &t );
		size_ = npos; // Unbounded
		shift_ = shift_a;
		sarray_ = array_ - shift_;
	}


	/// @brief Detach Proxy/Argument Array
	inline
	void
	detach()
	{
		assert( ! owner_ );
		array_size_ = 0;
		array_ = 0;
		size_ = 0;
		shift_ = 0;
		sarray_ = 0;
	}


	/// @brief Update Proxy Array Attachment to Array
	inline
	void
	update_to( FArrayB const & a )
	{
		assert( ! owner_ );
		array_size_ = a.array_size_;
		array_ = a.array_;
	}


	/// @brief Swap
	inline
	void
	swapB( FArrayB & v )
	{
		assert( owner_ );
		assert( v.owner_ );
		std::swap( array_size_, v.array_size_ );
		std::swap( array_, v.array_ );
		std::swap( size_, v.size_ );
		std::swap( shift_, v.shift_ );
		std::swap( sarray_, v.sarray_ );
	}


private: // Properties


#ifdef OBJEXXFCL_FARRAY_SIZE_REPORT
	/// @brief Report size if at least value defined for OBJEXXFCL_FARRAY_SIZE_REPORT
	/// @note  Size is based on sizeof( value_type ) so value_type-controlled heap memory is not counted
	inline
	void
	size_report() const
	{
		if ( size_ * sizeof( value_type ) >= OBJEXXFCL_FARRAY_SIZE_REPORT ) {
			std::cout << "FArray< " << typeid( value_type ).name() << " >"
			 << "  Size: " << size_ * sizeof( value_type ) << "  Elements: " << size_;
		}
	}
#endif // OBJEXXFCL_FARRAY_SIZE_REPORT


public: // Data


	/// @brief Unbounded "size"
	static size_type const npos = static_cast< size_type >( -1 );

	/// @brief Max array size
	static size_type const max_size = npos - static_cast< size_type >( 1 );


protected: // Data


	/// @brief Size of data array
	size_type array_size_;

	/// @brief Pointer to data array
	value_type * array_;

	/// @brief Size of active array
	size_type size_;

	/// @brief Owner of data array?
	bool const owner_;

	/// @brief Array shift
	int shift_;

	/// @brief Shifted pointer to data array
	value_type * sarray_;


}; // FArrayB


// Static Data Member Template Definitions

template< typename T > typename FArrayB< T >::size_type const FArrayB< T >::npos;

template< typename T > typename FArrayB< T >::size_type const FArrayB< T >::max_size;


} // namespace ObjexxFCL


#endif // INCLUDED_ObjexxFCL_FArrayB_HH
