#ifndef INCLUDED_ObjexxFCL_formatted_o_HH
#define INCLUDED_ObjexxFCL_formatted_o_HH


// Fortran-Compatible Formatted Output Functions
//
// Project: Objexx Fortran Compatibility Library (ObjexxFCL)
//
// Version: 2.6.2
//
// Language: C++
//
// Copyright (c) 2007 Objexx Engineering, Inc. All Rights Reserved.
// Use of this source code or any derivative of it is restricted by license.
// Licensing is available from Objexx Engineering, Inc.:   http://objexx.com   Objexx@objexx.com


// C++ Standard Library Headers
#include <algorithm>
#include <complex>
#include <iomanip>
#include <iosfwd>
#include <sstream>
#include <string>


namespace ObjexxFCL {


// Types
class byte;
class Fstring;


namespace fmt {


// Types
typedef  char       *    cstring;
typedef  char const *  c_cstring;


// General Formatting


/// @brief Single-Spaced Format
template< typename T >
std::string
SS( T const & t )
{
	std::ostringstream fmt_stream;
	fmt_stream << std::left << std::noshowpoint << std::uppercase << ' ' << t;
	return fmt_stream.str();
}


/// @brief Single-Spaced Format: bool Specialization
template<>
std::string
SS( bool const & t );


/// @brief Single-Spaced Format: float Specialization
template<>
std::string
SS( float const & t );


/// @brief Single-Spaced Format: double Specialization
template<>
std::string
SS( double const & t );


/// @brief Single-Spaced Format: long double Specialization
template<>
std::string
SS( long double const & t );


/// @brief Single-Spaced Format: complex< float > Specialization
template<>
std::string
SS( std::complex< float > const & t );


/// @brief Single-Spaced Format: complex< double > Specialization
template<>
std::string
SS( std::complex< double > const & t );


/// @brief Single-Spaced Format: complex< long double > Specialization
template<>
std::string
SS( std::complex< long double > const & t );


/// @brief Left-Justified Width-Specified Format
template< typename T >
std::string
LJ( int const w, T const & t )
{
	std::ostringstream fmt_stream;
	fmt_stream << std::left << std::setw( w ) << t;
	return fmt_stream.str();
}


/// @brief Right-Justified Width-Specified Format
template< typename T >
std::string
RJ( int const w, T const & t )
{
	std::ostringstream fmt_stream;
	fmt_stream << std::right << std::setw( w ) << t;
	return fmt_stream.str();
}


// String Formatting


/// @brief char Format
std::string
A( int const w, char const c );


/// @brief char Format (Width==1)
std::string
A( char const c );


/// @brief cstring Format
std::string
A( int const w, c_cstring const s );


/// @brief cstring Format (Width of cstring)
std::string
A( c_cstring const s );


/// @brief string Format
std::string
A( int const w, std::string const & s );


/// @brief string Format (Width of string)
std::string const &
A( std::string const & s );


/// @brief Fstring Format
std::string
A( int const w, Fstring const & s );


/// @brief Fstring Format (Width of Fstring)
std::string
A( Fstring const & s );


/// @brief Blank string
std::string
X( int const w );


/// @brief Blank string
std::string
space( int const w );


// Logical Formatting


/// @brief Logical Format
std::string
L( int const w, bool const & t );


/// @brief Logical Format (Width==1)
std::string
L( bool const & t );


// Integer Formatting


/// @brief Integer Format
template< typename T >
std::string
I( int const w, T const & t )
{
	std::ostringstream fmt_stream;
	fmt_stream << std::right << std::setw( w ) << t;
	return fmt_stream.str();
}


/// @brief Integer Format with Minimum Digits
template< typename T >
std::string
I( int const w, int const m, T const & t )
{
	std::ostringstream fmt_stream;
	fmt_stream << std::right << std::setfill( '0' ) << std::setw( std::min( m, w ) ) << t;
	std::string const str( fmt_stream.str() );
	return std::string( std::max( w - static_cast< int >( str.length() ), 0 ), ' ' ) + str;
}


// Floating Point Formatting


/// @brief Exponential Format: float
std::string
E( int const w, int const d, float const & t );


/// @brief Exponential Format: double
std::string
E( int const w, int const d, double const & t );


/// @brief Exponential Format: long double
std::string
E( int const w, int const d, long double const & t );


/// @brief Exponential Format: complex< float >
std::string
E( int const w, int const d, std::complex< float > const & t );


/// @brief Exponential Format: complex< double >
std::string
E( int const w, int const d, std::complex< double > const & t );


/// @brief Exponential Format: complex< long double >
std::string
E( int const w, int const d, std::complex< long double > const & t );


/// @brief Fixed Point Format: float
std::string
F( int const w, int const d, float const & t );


/// @brief Fixed Point Format: double
std::string
F( int const w, int const d, double const & t );


/// @brief Fixed Point Format: long double
std::string
F( int const w, int const d, long double const & t );


/// @brief Fixed Point Format: complex< float >
std::string
F( int const w, int const d, std::complex< float > const & t );


/// @brief Fixed Point Format: complex< double >
std::string
F( int const w, int const d, std::complex< double > const & t );


/// @brief Fixed Point Format: complex< long double >
std::string
F( int const w, int const d, std::complex< long double > const & t );


/// @brief General Format: float
std::string
G( int const w, int const d, float const & t );


/// @brief General Format: double
std::string
G( int const w, int const d, double const & t );


/// @brief General Format: long double
std::string
G( int const w, int const d, long double const & t );


/// @brief General Format: complex< float >
std::string
G( int const w, int const d, std::complex< float > const & t );


/// @brief General Format: complex< double >
std::string
G( int const w, int const d, std::complex< double > const & t );


/// @brief General Format: complex< long double >
std::string
G( int const w, int const d, std::complex< long double > const & t );


// Standard Formatting


/// @brief Standard Width Format: Default Implementation
template< typename T >
std::string
SW( T const & t )
{
	std::ostringstream fmt_stream;
	fmt_stream << std::left << std::noshowpoint << std::uppercase << t;
	return fmt_stream.str();
}


/// @brief Standard Width Format: bool Specialization
template<>
std::string
SW( bool const & t );


/// @brief Standard Width Format: byte Specialization
template<>
std::string
SW( byte const & t );


/// @brief Standard Width Format: short Specialization
template<>
std::string
SW( short int const & t );


/// @brief Standard Width Format: unsigned short Specialization
template<>
std::string
SW( unsigned short int const & t );


/// @brief Standard Width Format: int Specialization
template<>
std::string
SW( int const & t );


/// @brief Standard Width Format: unsigned int Specialization
template<>
std::string
SW( unsigned int const & t );


/// @brief Standard Width Format: long int Specialization
template<>
std::string
SW( long int const & t );


/// @brief Standard Width Format: unsigned long int Specialization
template<>
std::string
SW( unsigned long int const & t );


/// @brief Standard Width Format: float Specialization
template<>
std::string
SW( float const & t );


/// @brief Standard Width Format: double Specialization
template<>
std::string
SW( double const & t );


/// @brief Standard Width Format: long double Specialization
template<>
std::string
SW( long double const & t );


/// @brief Standard Width Format: complex< float > Specialization
template<>
std::string
SW( std::complex< float > const & t );


/// @brief Standard Width Format: complex< double > Specialization
template<>
std::string
SW( std::complex< double > const & t );


/// @brief Standard Width Format: complex< long double > Specialization
template<>
std::string
SW( std::complex< long double > const & t );


// Manipulators


/// @brief general: Manipulator to Turn Off scientific or fixed
std::ios_base &
general( std::ios_base & base );


} // namespace fmt
} // namespace ObjexxFCL


#endif // INCLUDED_ObjexxFCL_formatted_o_HH
