// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   AntibodyComplex.cc
/// @brief  Convenience class for manipulating antibody - antigen complexes.
/// @note   Manually manage Poses on the heap because Pose doesn't appear to have
/// @note   complete copy constructor and assignment capability.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)


// unit headers
#include <epigraft/AntibodyComplex.hh>

// package headers
#include <epigraft/epigraft_functions.hh>
#include <epigraft/ResidueRange.hh>

// rosetta headers
#include <aaproperties_pack.h>
#include <param.h>
#include <pose.h>
#include <pose_io.h>

// ObjexxFCL headers
#include <ObjexxFCL/FArray1D.hh>

// C++ headers
#include <string>

namespace epigraft {

////////////////////////////////
/* Public: Construct/Destruct */
////////////////////////////////

/// @brief construct from file
AntibodyComplex::AntibodyComplex(
	std::string filename,
	int const & nres_Ab,
	bool const & Ab_first,
	bool const & refold_sidechains_from_chi
)
{
	// memory management
	complex_ = new pose_ns::Pose;
	Ab_ = NULL;
	antigen_ = NULL;
	
	// initialize other
	initialize(); 
	
	// load Ab-antigen complex
	pose_from_pdb( *complex_, filename, true, false, true ); // for now we force the booleans: fullatom, ideal_pose, read_all_chains
	nres_Ab_ = nres_Ab;
	Ab_first_ = Ab_first;
	
	// see if Ab actually exists
	Ab_exists_ = ( nres_Ab_ != 0 );

	// flip symmetric sidechains, usually only used if non-rosetta structure (from pdb)
	if ( refold_sidechains_from_chi ) {
		Real const before_refold( (*complex_).score( score12_weights_ ) );
		(*complex_).refold_sidechains_from_chi();
		Real const after_refold( (*complex_).score( score12_weights_ ) );
		
		std::cout << "score-delta from sidechain flip (using score 12): " << ( after_refold - before_refold ) << std::endl;
	}
	
	// create Ab and antigen Poses
	refresh_components();
}

/// @brief construct from external Ab-antigen complex Pose
AntibodyComplex::AntibodyComplex(
	pose_ns::Pose const & complex,
	int const & nres_Ab,
	bool const & Ab_first
)
{
	// memory management
	complex_ = new pose_ns::Pose;
	Ab_ = NULL;
	antigen_ = NULL;
	
	// initialize other
	initialize();
	
	Ab_first_ = Ab_first;
	nres_Ab_ = nres_Ab;

	// see if Ab actually exists
	Ab_exists_ = ( nres_Ab_ != 0 );
	
	*complex_ = complex;
	
	refresh_components();
}

/// @brief construct from individual Ab and antigen Poses
AntibodyComplex::AntibodyComplex(
	pose_ns::Pose const & Ab,
	pose_ns::Pose const & antigen
)
{
	// memory management
	complex_ = NULL;
	Ab_ = new pose_ns::Pose;
	antigen_ = new pose_ns::Pose;
	
	// initialize other
	initialize();
	
	Ab_first_ = true;
	nres_Ab_ = Ab.total_residue();
	
	// Ab existence always true here
	Ab_exists_ = true;
	
	*Ab_ = Ab;
	*antigen_ = antigen;
	
	refresh_complex();
}


/////////////////////
/* Public: Scoring */
/////////////////////

/// @brief score complex and components
void
AntibodyComplex::score(
	pose_ns::Score_weight_map weight_map
)
{
	(*complex_).score( weight_map );
	(*Ab_).score( weight_map );
	(*antigen_).score( weight_map );
}


////////////////////////////////
/* Public: Ab/antigen cutting */
////////////////////////////////

/// @brief removes a segment of residue from antigen (scaffold)
/// @note  reverse function is trim_antigen()
void
AntibodyComplex::gapped_antigen(
	pose_ns::Pose & p,
	ResidueRange const & rr,
	bool const & remove_termini
) const
{
	ResidueRange gap_range( rr );
	if ( !remove_termini ) {
		gap_range = ResidueRange( rr.begin() + 1, rr.end() - 1 );
	}
	
	remove_segment_from_Pose( (*antigen_), gap_range, p );
}

/// @brief returns a segment of residues from antigen (scaffold)
/// @note  reverse function is gap_antigen()
void
AntibodyComplex::trimmed_antigen(
	pose_ns::Pose & p,
	ResidueRange const & rr
) const
{
	extract_segment_from_Pose( (*antigen_), rr, p );
}


///////////////////////////
/* Private: Manipulation */
///////////////////////////

/// @brief refresh Ab and antigen Poses from complex Pose
/// @note  checks Ab existence flag, if this is false, then only antigen and
/// @note  no Ab data is stored
void
AntibodyComplex::refresh_components()
{
	// memory management
	if ( Ab_ != NULL ) {
		delete Ab_;
	}
	if ( antigen_ != NULL ) {
		delete antigen_;
	}
	
	// data creation
	Ab_ = NULL;
	if ( Ab_exists_ ) {
		Ab_ = new pose_ns::Pose;
	}
	antigen_ = new pose_ns::Pose;
	
	ResidueRange Ab_range;
	ResidueRange antigen_range;
	
	if ( Ab_first_ ) { // Ab first
		
		if ( Ab_exists_ ) {
			Ab_range = ResidueRange( 1, nres_Ab_ );
		}
		antigen_range = ResidueRange( nres_Ab_ + 1, (*complex_).total_residue() );
		
	} else { // antigen first
		
		int const nres_antigen = (*complex_).total_residue() - nres_Ab_;
		antigen_range = ResidueRange( 1, nres_antigen );
		if ( Ab_exists_ ) {
			Ab_range = ResidueRange( nres_antigen + 1, (*complex_).total_residue() );
		}
		
	}
	
	// make Ab
	if ( Ab_exists_ ) {
		extract_segment_from_Pose( *complex_, Ab_range, *Ab_, true ); // boolean keep_pdb_info
	}
	
	// make antigen
	extract_segment_from_Pose( *complex_, antigen_range, *antigen_, true ); // boolean keep_pdb_info
}

/// @brief refresh complex from individual Ab and antigen Poses
void
AntibodyComplex::refresh_complex()
{
	// memory management
	if ( complex_ != NULL ) {
		delete complex_;
	}
	
	complex_ = new pose_ns::Pose;
	
	Integer const nres_complex = Ab_exists_ ? (*Ab_).total_residue() + (*antigen_).total_residue() : (*antigen_).total_residue();
	prepare_pose( *complex_, nres_complex );
	
	FArray3D_float complex_coords( 3, param::MAX_ATOM(), nres_complex );
	
	Integer counter = 1; // track where to start insertion
	
	// Ab first
	if ( Ab_exists_ ) {
		ResidueRange const ab_range( 1, (*Ab_).total_residue() );
		transfer_identity_info_between_Poses( *Ab_, ab_range, *complex_, counter, true ); // boolean: keep pdb info
		fill_full_coord_from_Pose( *Ab_, ab_range, complex_coords, counter ); // counter modified here	
	}
	
	// antigen next
	ResidueRange const antigen_range( 1, (*antigen_).total_residue() );
	transfer_identity_info_between_Poses( *antigen_, antigen_range, *complex_, counter, true ); // boolean: keep pdb info
	fill_full_coord_from_Pose( *antigen_, antigen_range, complex_coords, counter ); // counter modified here
	
	// make Eposition for complex
	FArray3D_float complex_epos( 3, param::MAX_POS, nres_complex ); // MAX_POS = 5
	full_coord_to_Eposition( nres_complex, complex_coords, complex_epos );
	
	// set coordinates for complex
	(*complex_).set_coords( false, complex_epos, complex_coords, false ); // booleans: ideal_pose, check_missing
}


////////////////////////////////////
/* Private: Internal Manipulation */
////////////////////////////////////

/// @brief repetitive pose preparation
void
AntibodyComplex::prepare_pose(
	pose_ns::Pose & p,
	int const & nres
) const
{
	p.simple_fold_tree( nres );
	p.set_fullatom_flag( true, false ); // force booleans: fullatom, repack
}


/////////////////////////////
/* Private: Initialization */
/////////////////////////////

/// @brief collected initialization
void
AntibodyComplex::initialize()
{	
	score12_weights_ = pose_ns::Score_weight_map( score12 );
}

} // namespace epigraft

