// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   AntibodyComplex.hh
/// @brief  Convenience class for manipulating antibody - antigen complexes.
/// @note   Manually manage Poses on the heap because Pose doesn't appear to have
/// @note   complete copy constructor and assignment capability.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef INCLUDED_epigraft_AntibodyComplex_HH_
#define INCLUDED_epigraft_AntibodyComplex_HH_


// package headers
#include <epigraft/epigraft_types.hh>
#include <epigraft/ResidueRange.hh>

// Rosetta headers
#include <pose.h>
#include <score.h>

// rootstock headers
#include <rootstock/rootstock_types.hh>

// C++ headers
#include <string>

namespace epigraft {

/// @brief  Convenience class for manipulating antibody - antigen complexes.
class AntibodyComplex {

	public: // types
		typedef rootstock::Real Score;

	public: // construct/destruct

		/// @brief default constructor
		inline
		AntibodyComplex()
		: complex_( NULL ),
		  Ab_( NULL ),
		  antigen_( NULL )
		{}

		/// @brief construct from file
		AntibodyComplex(
			std::string filename,
			int const & nres_Ab,
			bool const & Ab_first,
			bool const & refold_sidechains_from_chi = false
		);

		/// @brief construct from external Ab-antigen complex Pose
		AntibodyComplex(
			Pose const & complex,
			int const & nres_Ab,
			bool const & Ab_first
		);

		/// @brief construct from individual Ab and antigen Poses
		AntibodyComplex(
			Pose const & Ab,
			Pose const & antigen
		);

		/// @copy constructor
		AntibodyComplex(
			AntibodyComplex const & abc
		) : Ab_first_( abc.Ab_first_ ),
		    nres_Ab_( abc.nres_Ab_ ),
		    Ab_exists_( abc.Ab_exists_ )
		{
			// new data on heap
			complex_ = new Pose;
			Ab_ = new Pose;
			antigen_ = new Pose;

			// copy data
			*complex_ = *abc.complex_;
			*Ab_ = *abc.Ab_;
			*antigen_ = *abc.antigen_;
		}

		/// @brief default destructor
		inline
		~AntibodyComplex() {
			if ( complex_ ) {
				delete complex_;
			}
			if ( Ab_ ) {
				delete Ab_;
			}
			if ( antigen_ ) {
				delete antigen_;
			}
		}


	public: // assignment

		/// @brief copy assignment
		AntibodyComplex &
		operator =( AntibodyComplex const & abc )
		{
			if ( this != &abc ) {
				Ab_first_ = abc.Ab_first_;
				nres_Ab_ = abc.nres_Ab_;
				Ab_exists_ = abc.Ab_exists_;

				// delete original data
				if ( complex_ ) {
					delete complex_;
					complex_ = NULL;
				}
				if ( Ab_ ) {
					delete Ab_;
					Ab_ = NULL;
				}
				if ( antigen_ ) {
					delete antigen_;
					antigen_ = NULL;
				}

				// new data on heap and copy
				if ( abc.complex_ != NULL ) {
					complex_ = new Pose;
					*complex_ = *abc.complex_;
				}
				if ( abc.Ab_ != NULL ) {
					Ab_ = new Pose;
					*Ab_ = *abc.Ab_;
				}
				if ( abc.antigen_ != NULL ) {
					antigen_ = new Pose;
					*antigen_ = *abc.antigen_;
				}

			}
			return *this;
		}


	public: // accessors

		/// @brief is Ab first?
		inline
		bool const &
		Ab_first() const
		{
			return Ab_first_;
		}

		/// @brief is there an Ab in this complex?
		inline
		bool const &
		Ab_exists() const
		{
			return Ab_exists_;
		}

		/// @brief return Ab-antigen complex
		inline
		Pose &
		complex()
		{
			return *complex_;
		}

		/// @brief return Ab-antigen complex
		inline
		Pose const &
		complex() const
		{
			return *complex_;
		}

		/// @brief return Ab
		inline
		Pose &
		Ab()
		{
			return *Ab_;
		}

		/// @brief return Ab
		inline
		Pose const &
		Ab() const
		{
			return *Ab_;
		}

		/// @brief return antigen
		inline
		Pose &
		antigen()
		{
			return *antigen_;
		}

		/// @brief return antigen
		inline
		Pose const &
		antigen() const
		{
			return *antigen_;
		}


	public: // scoring

		/// @brief score complex and components
		void
		score(
			pose_ns::Score_weight_map weight_map
		);


	public: // Ab/antigen cutting

		/// @brief removes a segment of residues from antigen (scaffold)
		/// @note  default is to remove termini (begin and end of range)
		/// @note  reverse function is trim_antigen()
		void
		gapped_antigen(
			Pose &,
			ResidueRange const & rr,
			bool const & remove_termini = true
		) const;

		/// @brief returns a segment of residues from antigen (scaffold)
		/// @note  reverse function is gap_antigen()
		void
		trimmed_antigen(
			Pose &,
			ResidueRange const & rr
		) const;


	public: // manipulation

		/// @brief refresh individual Ab and antigen Poses from complex
		void
		refresh_components();

		/// @brief refresh complex from individual Ab and antigen Poses
		void
		refresh_complex();


	private: // internal manipulation

		/// @brief repetitive pose preparation
		void
		prepare_pose(
			Pose & p,
			int const & nres
		) const;


	private: // initialization

		/// @brief collected initialization
		void
		initialize();


	private: // data

		bool Ab_first_;
		int nres_Ab_;

		bool Ab_exists_;

		Pose * complex_;
		Pose * Ab_;
		Pose * antigen_;


	private: // for scoring

		pose_ns::Score_weight_map score12_weights_;

};

} // namespace epigraft

#endif /*INCLUDED_epigraft_AntibodyComplex_HH_*/
