// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   Checkpoint.cc
/// @brief  Class for managing checkpoint file.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)
/// @author Bill Schief (schief@u.washington.edu)

// unit headers
#include <epigraft/Checkpoint.hh>

// package headers
#include <epigraft/epigraft_types.hh>
#include <epigraft/epigraft_io.hh>

// ObjexxFCL headers
#include <ObjexxFCL/string.functions.hh>

// C++ headers
#include <iomanip>
#include <string>
#include <vector>


namespace epigraft {

/// @brief get last checkpoint id processed and store it internally
void
Checkpoint::recover_last_id_processed()
{
	std::fstream in( filename_.c_str(), std::ios::in );
	if ( in.fail() ) {
		in.close(); // cleanup (lack of) file
		current_id_ = 0; // no checkpoint file yet
		current_sub_id_ = 0;
		return;
	}

	current_id_ = 0;
	current_sub_id_ = 0;

	std::string line;
	std::vector< std::string > entry;

	while ( getline( in, line ) ) {

		split_string( line, entry );

		if ( entry[ 0 ] == "id" ) {
			std::istringstream( entry[ 1 ] ) >> current_id_;
		} else if ( entry[ 0 ] == "sub_id" ) {
			std::istringstream( entry[ 1 ] ) >> current_sub_id_;
		}

		entry.clear();
	}

	// close input file
	in.close();
}


/// @brief open file for checkpointing
void
Checkpoint::open()
{
	if ( checkfile_ ) {
		close();
		return;
	}
	checkfile_ = new std::fstream( filename_.c_str(), std::ios::out );
}


/// @brief open file for checkpointing
void
Checkpoint::open( std::string const & filename )
{
	filename_ = filename;
	open();
}


/// @brief close file checkpointing
void
Checkpoint::close()
{
	if ( !checkfile_ ) {
		return;
	}
	checkfile_->close();
	delete checkfile_;
	checkfile_ = NULL;
}


/// @brief update checkpointing file if open
void
Checkpoint::update()
{
	if ( checkfile_ ) {
		// move to beginning of file
		checkfile_->seekg( 0, std::ios::beg );

		// write id
		*checkfile_ << "id " << current_id_ << '\n';
		*checkfile_ << "sub_id " << current_sub_id_ << '\n';

		checkfile_->flush();
	}
}

} // namespace epigraft
