// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   Checkpoint.hh
/// @brief  Class for managing checkpoint file.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)
/// @author Bill Schief (schief@u.washington.edu)

#ifndef INCLUDED_epigraft_Checkpoint_HH_
#define INCLUDED_epigraft_Checkpoint_HH_

// package headers
#include <epigraft/epigraft_types.hh>

// C++ headers
#include <fstream>
#include <string>

namespace epigraft {


/// @brief  Class for managing checkpoint file.
class Checkpoint {

	public: // construct/destruct

		/// @brief default constructor
		Checkpoint()
		: checkfile_( NULL ),
		  current_id_( 0 ),
		  current_sub_id_( 0 )
		{}

		/// @brief constructor
		Checkpoint(
			std::string const & filename
		) : filename_( filename ),
		    checkfile_( NULL ),
		    current_id_( 0 ),
		    current_sub_id_( 0 )
		{}

		/// @brief default destructor
		~Checkpoint() {
			if ( checkfile_ ) {
				delete checkfile_;
			}
		}


	public: // accessors

		inline
		size_t const &
		current_id() const
		{
			return current_id_;
		}

		inline
		size_t const &
		current_sub_id() const
		{
			return current_sub_id_;
		}

		inline
		void
		set_current_id(
			size_t const & id
		)
		{
			current_id_ = id;
		}

		inline
		void
		set_current_sub_id(
			size_t const & id
		)
		{
			current_sub_id_ = id;
		}


	public: // methods

		// TODO: catch last_id_processed call if start() has been called
		/// @brief get last checkpoint id processed and store it interally
		void
		recover_last_id_processed();

		/// @brief open file for checkpointing
		void
		open();

		/// @brief open file for checkpointing
		void
		open( std::string const & filename );

		/// @brief close file for checkpointing
		void
		close();

		/// @brief update checkpointing file if open
		void
		update();


	private: // currently disallowed constructors and assignments

		/// @brief disallowed copy constructor
		Checkpoint( Checkpoint const & ) {}

		/// @brief disallowed copy assignment
//		Checkpoint &
//		operator =( Checkpoint const & ) {}


	private: // data

		std::string filename_;
		std::fstream * checkfile_;
		size_t current_id_;
		size_t current_sub_id_;
};

} // namespace epigraft

#endif /*INCLUDED_epigraft_Checkpoint_HH_*/
