// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   GraftOptions.cc
/// @brief  Simple epigraft options container, plain-data class, almost everything is public.
/// @brief  Not meant as a general purpose options parser, used for storing epigraft options in one object.
/// @note   This is supposed to named just 'Options.cc', but due to Mac filesystem being non-case sensitive
/// @note   and a flaw in Rosetta's 'make' setup, there's a linker conflict.  No time to change the make
/// @note   setup right now, so leave it named GraftOptions.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

// unit headers
#include <epigraft/GraftOptions.hh>

// package headers
#include <epigraft/epigraft_types.hh>

// rosetta headers
#include <after_opts.h>
#include <param_aa.h>

// C++ headers
#include <sstream>
#include <string>


namespace epigraft {

/// @brief grab options from command line
/// @note  conflict and bad options are checked for at the end of the procedure
void
GraftOptions::get_from_rosetta()
{
	// modes
	match_mode = truefalseoption( "match" );
	multigraft_mode = truefalseoption( "multigraft" );
	decimated_multigraft_mode = truefalseoption( "decimated_multigraft" );
	test_mode = truefalseoption( "test" );
	scarlet = truefalseoption( "scarlet" ); // enable experimental options

	// submodes
	rough_match = truefalseoption( "rough_match" );
	combi_match = truefalseoption( "combi_match" );
	superposition_minrepack_refine = truefalseoption( "superposition_minrepack_refine" );
	screen_with_repack = truefalseoption( "screen_with_repack" );

	// alignment systems
	skip_N2C_align = truefalseoption( "skip_N2C_align" );
	skip_C2N_align = truefalseoption( "skip_C2N_align" );
	use_E_align = truefalseoption( "E_align" );
	use_S_align = truefalseoption( "S_align" );
	use_SS_align = truefalseoption( "SS_align" );

	// input
	stringafteroption( "native_complex", native_complex_filename, native_complex_filename );
	stringafteroption( "loop_ranges", loop_ranges_filename, loop_ranges_filename );
	stringafteroption( "input_file", input_filename, input_filename );
	use_input_from_match = truefalseoption( "use_input_from_match" );
	if ( use_input_from_match && input_filename == "scaffold_list.txt" ) {
		input_filename = "graft_matches.in";
	}
	input_pdb_has_Ab = truefalseoption( "input_pdb_has_Ab" );

	// data specific
	intafteroption( "nres_Ab", nres_Ab, nres_Ab );
	scaffold_first = truefalseoption( "scaffold_first" );
	if ( scaffold_first ){
		Ab_first = truefalseoption( "Ab_first" ); // test so conflict can be caught below, otherwise there's no need to check for Ab_first by default
	}

	// range scanning options
	intafteroption( "termini_residue_skip", termini_residue_skip, termini_residue_skip );
	intafteroption( "min_match_width", min_match_width, min_match_width );
	intafteroption( "max_match_width_delta", max_match_width_delta, max_match_width_delta );
	intafteroption( "max_rough_match_width_delta", max_match_width_delta, max_rough_match_width_delta );//vds allow different
	intafteroption( "moveable_closure_residues", moveable_closure_residues, moveable_closure_residues );

	// filter
//	realafteroption( "match_distance_epsilon", match_distance_epsilon, match_distance_epsilon );
	realafteroption( "max_closure_rms", max_closure_rms, max_closure_rms );
	realafteroption( "max_rms_over_length", max_rms_over_length, max_rms_over_length );
	realafteroption( "max_intra_clash", max_intra_clash, max_intra_clash );
	realafteroption( "max_inter_clash", max_inter_clash, max_inter_clash );
	use_full_sidechain_inter_clash = truefalseoption( "use_full_sidechain_inter_clash" );
	use_epitope_sidechain_during_clash_check = truefalseoption( "use_epitope_sidechain_during_clash_check" );
	realafteroption( "rough_match_closure_rms", rough_match_closure_rms, rough_match_closure_rms );
	realafteroption( "rough_match_ca_distance", rough_match_closure_rms, rough_match_ca_distance ); // default is tied to rough_match_closure_rms
	realafteroption( "combi_match_ca_distance", combi_match_ca_distance, combi_match_ca_distance );

	// conformation
	fluidize_landing = truefalseoption( "fluidize_landing" );
	fluidize_takeoff = truefalseoption( "fluidize_takeoff" );
	rb_move = truefalseoption( "rb_move" );
	realafteroption( "recovery_rms_epsilon", recovery_rms_epsilon, recovery_rms_epsilon );
	realafteroption( "allowed_intra_clash_increase", allowed_intra_clash_increase, allowed_intra_clash_increase );
	realafteroption( "dihedral_deviation", dihedral_deviation, dihedral_deviation );
	realafteroption( "dihedral_step", dihedral_step, dihedral_step );
	realafteroption( "rb_cube_side_length", rb_cube_side_length, rb_cube_side_length );
	realafteroption( "rb_translation_step", rb_translation_step, rb_translation_step );
	realafteroption( "rb_angle_deviation", rb_angle_deviation, rb_angle_deviation );
	realafteroption( "rb_angle_step", rb_angle_step, rb_angle_step );

	// additional filters
	use_spatial_filter = truefalseoption( "spatial_filter" );
	stringafteroption( "spatial_filter", pdb_for_spatial_filter, pdb_for_spatial_filter );
	realafteroption( "spatial_filter_distance_cutoff", spatial_filter_distance_cutoff, spatial_filter_distance_cutoff );
	intafteroption( "spatial_filter_min_cbeta", spatial_filter_min_cbeta, spatial_filter_min_cbeta );
	intafteroption( "spatial_filter_max_cbeta", spatial_filter_max_cbeta, spatial_filter_max_cbeta );

	// additional statistics
	compute_cbeta_neighbors = truefalseoption( "compute_cbeta_neighbors" );
	realafteroption( "cbeta_neighbors_distance_cutoff", cbeta_neighbors_distance_cutoff, cbeta_neighbors_distance_cutoff );

	// output
	if ( truefalseoption( "multigraft" ) ) {
		output_filename = "multigraft_results.out";
	}
	stringafteroption( "output_file", output_filename, output_filename );
	output_single_matches = !truefalseoption( "no_single_match_output" );
	full_matches_only = truefalseoption( "no_partial_match_output" ); //vds
	output_aligned_loops = truefalseoption( "output_aligned_loops" );
	output_predesign_structure = truefalseoption( "output_predesign_structure" );
	output_predesign_structure_with_Ab = truefalseoption( "output_predesign_structure_with_Ab" );
	override_rosetta_pdb_output_path = truefalseoption( "override_pdb_output_path" );
	stringafteroption( "override_pdb_output_path", pdb_output_path, pdb_output_path );
	if ( override_rosetta_pdb_output_path ) { // catch directory delimiter
		if ( pdb_output_path.at( pdb_output_path.length() - 1 ) != '/' ) {
			pdb_output_path = pdb_output_path + "/";
		}
	}

	// checkpoint
	use_checkpoint = truefalseoption( "checkpoint" );
	if ( use_checkpoint ) {
		stringafteroption( "checkpoint", checkpoint_filename, checkpoint_filename );
		checkpoint_filename_inner = checkpoint_filename + ".inner";
	}
	if ( truefalseoption( "checkpoint_interval_minutes" ) ) {
		Real minutes = checkpoint_interval_seconds / 60.0;
		realafteroption( "checkpoint_interval_minutes", minutes, minutes );
		checkpoint_interval_seconds = static_cast< Integer >( minutes * 60.0 );
	}

	// multigraft specific
	stringafteroption( "vall", vall_filename, vall_filename ); // dummy variable, see fragments_pose.cc for actual setting
	use_keep_natro = truefalseoption( "keep_natro" );
	if ( use_keep_natro ) {
		stringafteroption ( "keep_natro", keep_natro_filename, keep_natro_filename );
	}
	use_batch_id = truefalseoption( "batch_id");
	stringafteroption( "batch_id", batch_id, batch_id );
	dump_predesign = truefalseoption( "dump_predesign" );
	stringafteroption ( "dump_predesign", predesign_filename, predesign_filename );
	dump_all_closure_attempt_structures = truefalseoption( "dump_all_closure_attempt_structures" );
	dump_closed = truefalseoption( "dump_closed" );
	dump_refined = truefalseoption( "dump_refined" );
	//vds or structures with at least one loop closed, in multigraft
	dump_someclosed =  truefalseoption( "dump_someclosed" );

	// multigraft, initial geometry
	idealize_loop_geometry = !truefalseoption( "no_idealize_loop_geometry" );
	epitope_rb_optimize = truefalseoption( "epitope_rb_optimize" );
	intafteroption( "epitope_rb_cycles", epitope_rb_cycles, epitope_rb_cycles );
	epitope_rb_min = truefalseoption( "epitope_rb_min" );
	Ab_epitope_optimize = truefalseoption( "Ab_epitope_optimize" );
	Ab_epitope_optimize_including_rb = truefalseoption( "Ab_epitope_optimize_including_rb" );
	if ( Ab_epitope_optimize_including_rb ) {
		Ab_epitope_optimize = true;
	}
	epitope_optimize = truefalseoption( "epitope_optimize" );

	// multigraft procedure
	micromanage_termini = !truefalseoption( "no_micromanage_termini" );
	randomize_moveable_phipsi = truefalseoption( "randomize_moveable_phipsi" );
	build_loops = truefalseoption( "build_loops" );
	refine_loops = truefalseoption( "refine_loops" );
	stop_on_build_failure = !truefalseoption( "ignore_build_failure" );
	stop_on_refine_failure = !truefalseoption( "ignore_refine_failure" );
	design_after_closure = truefalseoption( "design_after_closure" );
	refine_with_Ab_after_design = truefalseoption( "refine_with_Ab_after_design" );
	do_QC = truefalseoption( "do_QC" );
	store_top_design_per_backbone = truefalseoption( "store_top_design_per_backbone" );

	// multigraft closure/design attempts
	intafteroption( "closure_attempts", closure_attempts, closure_attempts );
	intafteroption( "closure_reset_period", closure_reset_period, closure_reset_period );
	intafteroption( "store_n_best_closures", store_n_best_closures, store_n_best_closures );
	store_n_best_closures = std::abs( store_n_best_closures ); // make sure nothing ridiculous happens
	intafteroption( "design_attempts", design_attempts, design_attempts );
	intafteroption( "store_n_best_designs", store_n_best_designs, store_n_best_designs );

	// multigraft closure options
	graft_with_Ab = truefalseoption( "graft_with_Ab" );
	if ( close_as_GLY = truefalseoption( "close_as_GLY" ) ) {
		closure_residue_type = param_aa::aa_gly;
	}
	grow_as_GLY = truefalseoption( "grow_as_GLY" );
	realafteroption( "rotation_perturb_magnitude", rotation_perturb_magnitude, rotation_perturb_magnitude );
	realafteroption( "translation_perturb_magnitude", translation_perturb_magnitude, translation_perturb_magnitude );
	use_fragment_insertion = !truefalseoption( "no_fragment_insertion" );
	use_sequence_biased_fragments = truefalseoption( "use_sequence_biased_fragments" );
	use_variable_length_fragments = truefalseoption( "variable_fragments" );
	allow_any_ss_during_fragment_insertion = truefalseoption( "allow_any_ss_during_fragment_insertion" );
	intafteroption( "nfrag", number_of_fragments, number_of_fragments );
	build_with_adaptive = truefalseoption( "build_with_adaptive" );
	build_with_arm = truefalseoption( "build_with_arm" );
	build_with_screen = truefalseoption( "build_with_screen" );
	intafteroption( "build_cycles", build_cycles, build_cycles );
	force_refine = truefalseoption( "force_refine" );
	refine_with_minrepack = truefalseoption( "refine_with_minrepack" );
	refine_with_classic = truefalseoption( "refine_with_classic" );
	refine_with_constraints = truefalseoption( "refine_with_constraints" );
	use_fast_refine = truefalseoption( "fast_refine" );
	intafteroption( "refine_cycles", refine_cycles, refine_cycles );

	// multigraft closure criteria
	realafteroption( "max_chainbreak_score", max_chainbreak_score, max_chainbreak_score );
	realafteroption( "max_local_rama", max_local_rama, max_local_rama );

	// multigraft design options
	repack_epitope = truefalseoption( "repack_epitope" );
	design_epitope = truefalseoption( "design_epitope" ); //vds usually use with keep_natro
	repack_Ab = truefalseoption( "repack_Ab" );
	allow_AA_at_inter_design = truefalseoption( "allow_AA_at_inter_design" );
	realafteroption( "intra_design_cutoff", intra_design_cutoff, intra_design_cutoff );
	realafteroption( "inter_design_cutoff", inter_design_cutoff, inter_design_cutoff );
	realafteroption( "Ab_repack_cutoff", Ab_repack_cutoff, Ab_repack_cutoff );

	// multigraft scan behavior
	scan_randomize_moveable = truefalseoption( "scan_randomize_moveable" );
	scan_randomize_cutpoints = truefalseoption( "scan_randomize_cutpoints" );
	scan_randomize_ss = truefalseoption( "scan_randomize_ss" );
	realafteroption( "scan_ss_minimum_content", scan_ss_minimum_content, scan_ss_minimum_content );
	if ( truefalseoption( "scan_ss_use_any" ) ) {
		scan_ss_type = 'D';
	}
	use_scan_behavior = scan_randomize_moveable || scan_randomize_cutpoints || scan_randomize_ss;

	// multigraft complementarity design
	complementarity_design = truefalseoption( "complementarity_design" );
	if ( complementarity_design ) {
		stringafteroption( "complementarity_design", complementarity_residue_file, complementarity_residue_file );
	}
	complementarity_rb = truefalseoption( "complementarity_rb" );
	realafteroption( "complementarity_design_cutoff", complementarity_design_cutoff, complementarity_design_cutoff );
	intafteroption( "complementarity_design_attempts", complementarity_design_attempts, complementarity_design_attempts );
	intafteroption( "complementarity_design_cycles", complementarity_design_cycles, complementarity_design_cycles );
	realafteroption( "complementarity_shell_cutoff", complementarity_shell_cutoff, complementarity_shell_cutoff );
	complementarity_shell_repack = truefalseoption( "complementarity_shell_repack" );
	complementarity_shell_redesign = truefalseoption( "complementarity_shell_redesign" );

	// multigraft export
	export_blueprint = truefalseoption( "export_blueprint" );
	export_resfile = truefalseoption( "export_resfile" );

	// options for handling deprecated features
	use_old_graft_info_format = truefalseoption( "use_old_graft_info_format" );

	// temporary multigraft
	do_restart = truefalseoption( "restart_from" );
	stringafteroption( "restart_from", restart_from, restart_from );
	post_modify_restart = truefalseoption( "post_modify_restart" );

	// decimated multigraft options
	intafteroption( "nstruct", nstruct, nstruct );
	realafteroption( "build_attempt_multiplier", build_attempt_multiplier, build_attempt_multiplier );
	intafteroption( "max_refine_attempts_per_round", max_refine_attempts_per_round, max_refine_attempts_per_round );

	test_water = truefalseoption( "water" );
	test_tea = truefalseoption( "tea" );
	test_coffee = truefalseoption( "coffee" );
	test_milk = truefalseoption( "milk" );
	test_port = truefalseoption( "port" );
	test_juice = truefalseoption( "juice" );
	test_cola = truefalseoption( "cola" );
	test_wine = truefalseoption( "wine" );
	test_beer = truefalseoption( "beer" );
	test_bubble_tea = truefalseoption( "bubble_tea" );
	test_mango_lassi = truefalseoption( "mango_lassi" );
	test_vodka = truefalseoption( "vodka" );
	test_tequila = truefalseoption( "tequila" );

	if ( test_water || test_tea || test_coffee || test_milk || test_port || test_juice || test_cola || test_wine || test_beer || test_bubble_tea
	     || test_mango_lassi || test_vodka || test_tequila ) {
		max_chainbreak_score = 0.2;
		design_after_closure = true;
		do_QC = false;
		store_n_best_closures = 5;
		store_n_best_designs = 1;
		design_attempts = 1;
		test_ccd_type = 0; // only for mango_lassi, vodka, and tequila
	}

	if ( test_water ) {
		build_loops = true;
		refine_loops = true;
		refine_with_minrepack = true;
	}

	if ( test_tea ) {
		build_loops = true;
		build_with_adaptive = true;
		refine_loops = true;
		refine_with_minrepack = true;
	}

	if ( test_coffee ) {
		build_loops = true;
		refine_loops = true;
	}

	if ( test_milk ) {
		build_loops = true;
		build_with_adaptive = true;
		refine_loops = true;
		force_refine = true;
		refine_with_minrepack = true;
	}

	if ( test_port ) {
		build_loops = true;
		build_with_adaptive = true;
		refine_loops = true;
		force_refine = true;
	}

	if ( test_juice ) {
		build_loops = true;
		refine_loops = true;
		force_refine = true;
		refine_with_minrepack = true;
	}

	if ( test_cola ) {
		build_loops = true;
		build_with_adaptive = true;
		refine_loops = true;
		refine_with_constraints = true;
	}

	if ( test_wine ) {
		build_loops = false;
		refine_loops = true;
		force_refine = true;
		refine_with_classic = true;
		refine_cycles = 3;
	}

	if ( test_beer ) {
		build_loops = false;
		refine_loops = true;
		force_refine = true;
		refine_with_minrepack = true;
		refine_cycles = 10;
	}

	if ( test_bubble_tea ) {
		build_loops = false;
		refine_loops = true;
		force_refine = true;
		refine_cycles = 10;
	}

	if ( test_mango_lassi ) {
		build_loops = true;
		build_with_fragments_only = true;
		refine_loops = false;
		test_ccd_type = 0;
	}

	if ( test_vodka ) {
		build_loops = true;
		build_with_fragments_only = true;
		refine_loops = false;
		test_ccd_type = 2;
	}

	if ( test_tequila ) {
		build_loops = true;
		build_with_fragments_only = true;
		refine_loops = false;
		test_ccd_type = 1;
	}

	// now check for conflicts
	if ( !( test_mode || scarlet ) ) {
		if ( nres_Ab < 0 ) {
			utility::exit( __FILE__, __LINE__, "epigraft::GraftOptions error: must set nres_Ab!" );
		}

		if ( Ab_first && scaffold_first ) {
			utility::exit( __FILE__, __LINE__, "epigraft::GraftOptions error: cannot have both Ab_first and scaffold_first!" );
		} else if ( (!Ab_first) && (!scaffold_first) ) {
			utility::exit( __FILE__, __LINE__, "epigraft::GraftOptions error: must have either Ab_first or scaffold_first!" );
		}

		if ( rough_match && combi_match ) {
			utility::exit( __FILE__, __LINE__, "epigraft::GraftOptions error: rough_match and combi_match are mutually exclusive!" );
		}
	}

	if ( use_checkpoint && ( checkpoint_filename.c_str()[ 0 ] == '-' || checkpoint_filename == "none" || checkpoint_filename.length() == 0 ) ) {
		utility::exit( __FILE__, __LINE__, "epigraft::GraftOptions error: -checkpoint must be followed by a filename!" );
	}

	if ( checkpoint_interval_seconds <= 0 ) {
		utility::exit( __FILE__, __LINE__, "epigraft::GraftOptions error: -checkpoint_interval_minutes must be positive!" );
	}

	if ( complementarity_design && ( complementarity_residue_file.c_str()[ 0 ] == '-' || checkpoint_filename == "none" || checkpoint_filename.length() == 0 ) ) {
		utility::exit( __FILE__, __LINE__, "epigraft::GraftOptions error: -complementarity_design must be followed by a filename!" );
	}

	if ( complementarity_shell_repack && complementarity_shell_redesign ) {
		utility::exit( __FILE__, __LINE__, "epigraft::GraftOptions error: both -complementarity_shell_repack and -complementarity_shell_design specified, ambiguous!" );
	}

	if ( scan_randomize_moveable && scan_randomize_cutpoints ) {
		utility::exit( __FILE__, __LINE__, "epigraft::GraftOptions error: -scan_randomize_moveable and -scan_randomize_cutpoints are incompatible with each other!" );
	}

}


/// @brief initialize with default values for all options data
void
GraftOptions::initialize()
{
	// modes
	match_mode = false;
	multigraft_mode = false;
	decimated_multigraft_mode = false;
	test_mode = false;
	scarlet = false; // experimental

	// submodes
	rough_match = false;
	combi_match = false;
	superposition_minrepack_refine = false;
	screen_with_repack = false;

	// alignment systems
	skip_N2C_align = false;
	skip_C2N_align = false;
	use_E_align = false;
	use_S_align = false;
	use_SS_align = false;

	// input
	native_complex_filename = "native_complex.pdb";
	loop_ranges_filename = "loop_ranges.txt"; // epitope loop ranges
	input_filename = "scaffold_list.txt";
	use_input_from_match = false;
	input_pdb_has_Ab = false;

	// data specific
	nres_Ab = -1;
	Ab_first = true;
	scaffold_first = false;

	// range scanning options
	termini_residue_skip = 4; // skip this number of residues from both the N and C termini
	min_match_width = 3; // minimum gap size
	max_match_width_delta = 10; // maximum difference between loop size and match gap size
	max_rough_match_width_delta = max_match_width_delta; // maximum difference between loop size and match gap size
	moveable_closure_residues = 3; // number of _scaffold_ residues adjacent to break that are allowed to move during closure

	// filter
//	match_distance_epsilon = 3.0; // epsilon added to CA-CA distance a loop when checking potential scaffold endpoints
	max_closure_rms = 2.0;
	max_rms_over_length = -1.0;
	max_intra_clash = 10000.0;
	max_inter_clash = 30000.0;
	use_full_sidechain_inter_clash = false;
	use_epitope_sidechain_during_clash_check = false;
	rough_match_closure_rms = 2.0; // max closure rms for rough match
	rough_match_ca_distance = 2.0; // max distance for CA-CA distance check during rough match, default is actually = rough_match_closure_rms
	combi_match_ca_distance = 1.0; // max distance for CA-CA distance check during combi match

	// conformation
	fluidize_landing = false;
	fluidize_takeoff = false;
	rb_move = false;
	recovery_rms_epsilon = 0.0;
	allowed_intra_clash_increase = 10.0;
	dihedral_deviation = 5.0;
	dihedral_step = 5.0;
	rb_cube_side_length = 1.0;
	rb_translation_step = 0.5;
	rb_angle_deviation = 5.0;
	rb_angle_step = 5.0;

	// additional constraints
	use_spatial_filter = false;
	std::string pdb_for_spatial_filter = "spatial_filter.pdb";
	spatial_filter_distance_cutoff = 5.5;
	spatial_filter_min_cbeta = 6;
	spatial_filter_max_cbeta = 10;

	// additional statistics
	compute_cbeta_neighbors = false;
	cbeta_neighbors_distance_cutoff = 5.5;

	// output
	output_filename = "graft_matches.out";
	output_single_matches = true;
	full_matches_only = false;
	output_aligned_loops = false;
	output_predesign_structure = false;
	output_predesign_structure_with_Ab = false;
	override_rosetta_pdb_output_path = false;
	pdb_output_path = "";

	// checkpoint
	use_checkpoint = false;
	checkpoint_filename = "none";
	checkpoint_filename_inner = "none";
	checkpoint_interval_seconds = 60 * 10; // ten minutes

	// multigraft specific
	vall_filename = "vall.dat.pc60_2002-12-27_v01.2"; // dummy variable, see fragments_pose.cc for actual setting
	use_keep_natro = false;
	keep_natro_filename = "none";
	use_batch_id = false;
	batch_id = "";
	dump_predesign = false;
	predesign_filename = "none";
	dump_all_closure_attempt_structures = false;
	dump_closed = false;
	dump_refined = false;
	dump_someclosed = false;

	// multigraft, initial geometry
	idealize_loop_geometry = true;
	epitope_rb_optimize = false;
	epitope_rb_cycles = 1;
	epitope_rb_min = false;
	Ab_epitope_optimize = false;
	Ab_epitope_optimize_including_rb = false;
	epitope_optimize = false;

	// multigraft procedure
	micromanage_termini = true;
	randomize_moveable_phipsi = false;
	build_loops = false;
	refine_loops = false;
	stop_on_build_failure = true;
	stop_on_refine_failure = true;
	design_after_closure = false;
	refine_with_Ab_after_design = false;
	do_QC = false;

	// multigraft closure/design attempts
	closure_attempts = 25;
	closure_reset_period = 3;
	store_n_best_closures = 5;
	design_attempts = 3;
	store_n_best_designs = 1;

	// multigraft closure options
	graft_with_Ab = false;
	close_as_GLY = false;
	closure_residue_type = param_aa::aa_ala;
	grow_as_GLY = false;
	rotation_perturb_magnitude = 1.0;
	translation_perturb_magnitude = 0.5;
	use_fragment_insertion = true;
	use_sequence_biased_fragments = false;
	use_variable_length_fragments = false;
	allow_any_ss_during_fragment_insertion = false;
	number_of_fragments = 200;
	build_with_adaptive = false;
	build_with_arm = false;
	build_with_screen = false;
	build_with_fragments_only = false;
	build_cycles = 10;
	force_refine = false;
	refine_with_minrepack = false;
	refine_with_classic = false;
	refine_with_constraints = false;
	use_fast_refine = false;
	refine_cycles = 3;

	// multigraft closure criteria
	max_chainbreak_score = 0.008;
	max_local_rama = 5.0;

	// multigraft design options
	repack_epitope = false;
	design_epitope = false;
	repack_Ab = false;
	allow_AA_at_inter_design = false;
	intra_design_cutoff = 4.0;
	inter_design_cutoff = 4.0;
	Ab_repack_cutoff = 4.5;

	// multigraft scan behavior
	use_scan_behavior = false;
	scan_randomize_moveable = false;
	scan_randomize_cutpoints = false;
	scan_randomize_ss = false;
	scan_ss_minimum_content = 0.5;
	scan_ss_type = 'L';

	// multigraft interaction design
	complementarity_design = false;
	complementarity_residue_file = "";
	complementarity_rb = false;
	complementarity_design_cutoff = 4.0;
	complementarity_design_attempts = 1;
	complementarity_design_cycles = 3;
	complementarity_shell_cutoff = 4.0;
	complementarity_shell_repack = false;
	complementarity_shell_redesign = false;

	// multigraft export
	export_blueprint = false;
	export_resfile = false;

	// temporary multigraft
	do_restart = false;
	restart_from = "";
	post_modify_restart = false;

	// options for handling deprecated features
	use_old_graft_info_format = false;

	// decimated multigraft options
	nstruct = 1;
	build_attempt_multiplier = 10.0;
	max_refine_attempts_per_round = 5;
}


/// @brief options and usage information
std::string
GraftOptions::usage()
{
	std::string tier1( "     " ); // tier1 spacing
	std::string tier2( "          " ); // tier2 spacing

	std::stringstream ss;

	ss << std::endl;
	ss << "Rosetta Epi-Graft mode ( -epi_graft ):" << std::endl;
	ss << std::endl;
	ss << "  By default all alignment systems except 'E' and 'S' are loaded." << std::endl;
	ss << "  Default values are indicated in brackets []." << std::endl;
	ss << "  Checkpointing is off if the given string is 'none'" << std::endl;
	ss << "  Flags with {} default values indicate boolean switches.  Setting the flag sets the option to true." << std::endl;
	ss << std::endl;
	ss << tier1 << "modes:" << std::endl;
	ss << tier2 << "-match" << bracket_boolean_value( match_mode ) << std::endl;
	ss << tier2 << "-multigraft" << bracket_boolean_value( multigraft_mode ) << std::endl;
	ss << std::endl;
	ss << tier1 << "submodes:" << std::endl;
	ss << tier2 << "-rough_match" << bracket_boolean_value( rough_match ) << std::endl;
	ss << tier2 << "-combi_match" << bracket_boolean_value( combi_match ) << std::endl;
	ss << tier2 << "-superposition_minrepack_refine" << bracket_boolean_value( superposition_minrepack_refine ) << std::endl;
	ss << tier2 << "-screen_with_repack" << bracket_boolean_value( screen_with_repack ) << std::endl;
	ss << std::endl;
	ss << tier1 << "additional alignment systems:" << std::endl;
	ss << tier2 << "-skip_N2C_align" << bracket_boolean_value( skip_N2C_align ) << std::endl;
	ss << tier2 << "-skip_C2N_align" << bracket_boolean_value( skip_C2N_align ) << std::endl;
	ss << tier2 << "-E_align" << bracket_boolean_value( use_E_align ) << std::endl;
	ss << tier2 << "-S_align" << bracket_boolean_value( use_S_align ) << std::endl;
	ss << tier2 << "-SS_align" << bracket_boolean_value( use_SS_align ) << std::endl;
	ss << std::endl;
	ss << tier1 << "required options:" << std::endl;
	ss << std::endl;
	ss << tier2 << "-nres_Ab <int>" << bracket_value( nres_Ab ) << std::endl;
	ss << std::endl;
	ss << tier1 << "input options:" << std::endl;
	ss << std::endl;
	ss << tier2 << "-native_complex <string>" << bracket_value( native_complex_filename ) << std::endl;
	ss << tier2 << "-loop_ranges <string>" << bracket_value( loop_ranges_filename ) << std::endl;
	ss << tier2 << "-input_file <string>" << bracket_value( input_filename ) << std::endl;
	ss << tier2 << "-use_input_from_match" << bracket_boolean_value( use_input_from_match ) << "     if true, this changes the default '-input_file' to: graft_matches.in" << std::endl;
	ss << tier2 << "-input_pdb_has_Ab" << bracket_boolean_value( input_pdb_has_Ab) << std::endl;
	ss << tier2 << "-Ab_first" << bracket_boolean_value( Ab_first ) << std::endl;
	ss << tier2 << "-scaffold_first" << bracket_boolean_value( scaffold_first ) << std::endl;
	ss << std::endl;
	ss << tier1 << "range scanning options" << std::endl;
	ss << tier2 << "-termini_residue_skip <int>" << bracket_value( termini_residue_skip ) << std::endl;
	ss << tier2 << "-min_match_width <int>" << bracket_value( min_match_width ) << std::endl;
	ss << tier2 << "-max_match_width_delta <int>" << bracket_value( max_match_width_delta ) << std::endl;
	ss << tier2 << "-max_rough_match_width_delta <int> (def: max_match_width_delta)" << bracket_value( max_match_width_delta ) << std::endl;
	ss << tier2 << "-moveable_closure_residues <int>" << bracket_value( moveable_closure_residues ) << std::endl;
	ss << std::endl;
	ss << tier1 << "filter options" << std::endl;
//	ss << tier2 << "-match_distance_epsilon <real>" << bracket_value( match_distance_epsilon ) << std::endl;
	ss << tier2 << "-max_closure_rms <real>" << bracket_value( max_closure_rms ) << std::endl;
	ss << tier2 << "-max_intra_clash <real>" << bracket_value( max_intra_clash ) << std::endl;
	ss << tier2 << "-max_inter_clash <real>" << bracket_value( max_inter_clash ) << std::endl;
	ss << tier2 << "-use_full_sidechain_inter_clash" << bracket_boolean_value( use_full_sidechain_inter_clash ) << std::endl;
	ss << tier2 << "-rough_match_closure_rms <real>" << bracket_value( rough_match_closure_rms ) << std::endl;
	ss << tier2 << "-rough_match_ca_distance <real>" << bracket_value( "= value of -rough_match_closure_rms" ) << std::endl;
	ss << tier2 << "-combi_match_ca_distance <real>" << bracket_value( combi_match_ca_distance ) << std::endl;
	ss << std::endl;
	ss << tier1 << "conformation options" << std::endl;
	ss << tier2 << "-fluidize_landing" << bracket_boolean_value( fluidize_landing ) << std::endl;
	ss << tier2 << "-fluidize_takeoff" << bracket_boolean_value( fluidize_takeoff ) << std::endl;
	ss << tier2 << "-rb_move" << bracket_boolean_value( rb_move ) << std::endl;
	ss << tier2 << "-recovery_rms_epsilon <real>" << bracket_value( recovery_rms_epsilon ) << std::endl;
	ss << tier2 << "-allowed_intra_clash_increase <real>" << bracket_value( allowed_intra_clash_increase ) << std::endl;
	ss << tier2 << "-dihedral_deviation <real>" << bracket_value( dihedral_deviation ) << std::endl;
	ss << tier2 << "-dihedral_step <real>" << bracket_value( dihedral_step ) << std::endl;
	ss << tier2 << "-rb_cube_side_length <real>" << bracket_value( rb_cube_side_length ) << std::endl;
	ss << tier2 << "-rb_translation_step <real>" << bracket_value( rb_translation_step ) << std::endl;
	ss << tier2 << "-rb_angle_deviation <real>" << bracket_value( rb_angle_deviation ) << std::endl;
	ss << tier2 << "-rb_angle_step <real>" << bracket_value( rb_angle_step ) << std::endl;
	ss << std::endl;
	ss << tier1 << "additional filters" << std::endl;
	ss << tier2 << "-spatial_filter <string>" << bracket_value( pdb_for_spatial_filter ) << std::endl;
	ss << tier2 << "-spatial_filter_distance_cutoff <real>" << bracket_value( spatial_filter_distance_cutoff ) << std::endl;
	ss << tier2 << "-spatial_filter_min_cbeta <int>" << bracket_value( spatial_filter_min_cbeta ) << std::endl;
	ss << tier2 << "-spatial_filter_max_cbeta <int>" << bracket_value( spatial_filter_max_cbeta ) << std::endl;
	ss << std::endl;
	ss << tier1 << "additional statistics" << std::endl;
	ss << tier2 << "-compute_cbeta_neighbors " << bracket_boolean_value( compute_cbeta_neighbors ) << std::endl;
	ss << tier2 << "-cbeta_neighbors_distance_cutoff <real> " << bracket_value( cbeta_neighbors_distance_cutoff ) << std::endl;
	ss << std::endl;
	ss << tier1 << "output options" << std::endl;
	ss << tier2 << "-output_file <string>" << bracket_value( output_filename ) << "     if -multigraft, this changes the default to: multigraft_results.out" << std::endl;
	ss << tier2 << "-no_single_match_output" << bracket_boolean_value( !output_single_matches ) << std::endl;
	ss << tier2 << "-output_full_matches_only" << bracket_boolean_value( !full_matches_only ) << std::endl;
	ss << tier2 << "-output_aligned_loops" << bracket_boolean_value( output_aligned_loops ) << std::endl;
	ss << tier2 << "-output_predesign_structure" << bracket_boolean_value( output_predesign_structure ) << std::endl;
	ss << tier2 << "-output_predesign_structure_with_Ab" << bracket_boolean_value( output_predesign_structure_with_Ab ) << std::endl;
	ss << tier2 << "-override_pdb_output_path <string> " << bracket_value( pdb_output_path ) << std::endl;
	ss << std::endl;
	ss << tier1 << "checkpointing" << std::endl;
	ss << tier2 << "-checkpoint <string>" << bracket_value( checkpoint_filename ) << std::endl;
	ss << tier2 << "-checkpoint_interval_minutes <real>" << bracket_value( checkpoint_interval_seconds / 60.0 ) << std::endl;
	ss << std::endl;
	ss << tier1 << "multigraft specific" << std::endl;
	ss << tier2 << "-vall <string>" << bracket_value( vall_filename ) <<std::endl;
	ss << tier2 << "-keep_natro <string>" << bracket_value( keep_natro_filename ) << std::endl;
	ss << tier2 << "-batch_id <string>" << bracket_value( batch_id ) << std::endl;
	ss << tier2 << "-dump_predesign <string>" << bracket_value( predesign_filename ) << std::endl;
	ss << tier2 << "-dump_closed <string>" << bracket_boolean_value( dump_closed ) << std::endl;
	ss << tier2 << "-dump_refined <string>" << bracket_boolean_value( dump_refined ) << std::endl;
	ss << tier2 << "-dump_someclosed <string>" << bracket_boolean_value( dump_someclosed ) << std::endl;
	ss << tier2 << "-dump_all_closure_attempt_structures" << bracket_boolean_value( dump_all_closure_attempt_structures ) << std::endl;
	ss << tier2 << "-use_old_graft_info_format" << bracket_boolean_value( use_old_graft_info_format ) << std::endl;
	ss << std::endl;
	ss << tier1 << "multigraft, initial geometry" << std::endl;
	ss << tier2 << "no_micromanage_termini" << bracket_boolean_value( !micromanage_termini ) << std::endl;
	ss << tier2 << "epitope_rb_optimize" << bracket_boolean_value( epitope_rb_optimize ) << std::endl;
	ss << tier2 << "epitope_rb_cycles" << bracket_value( epitope_rb_cycles ) << std::endl;
	ss << tier2 << "epitope_rb_min" << bracket_boolean_value( epitope_rb_min ) << std::endl;
	ss << tier2 << "Ab_epitope_optimize" << bracket_boolean_value( Ab_epitope_optimize ) << std::endl;
	ss << tier2 << "Ab_epitope_optimize_including_rb" << bracket_boolean_value( Ab_epitope_optimize_including_rb ) << std::endl;
	ss << tier2 << "epitope_optimize" << bracket_boolean_value( epitope_optimize ) << std::endl;
	ss << std::endl;
	ss << tier1 << "multigraft procedure" << std::endl;
	ss << tier2 << "-build_loops" << bracket_boolean_value( build_loops ) << std::endl;
	ss << tier2 << "-refine_loops" << bracket_boolean_value( refine_loops ) << std::endl;
	ss << tier2 << "-ignore_build_failure" << bracket_boolean_value( !stop_on_build_failure ) << std::endl;
	ss << tier2 << "-ignore_refine_failure" << bracket_boolean_value( !stop_on_refine_failure ) << std::endl;
	ss << tier2 << "-design_after_closure" << bracket_boolean_value( design_after_closure ) << std::endl;
	ss << tier2 << "-refine_with_Ab_after_design" << bracket_boolean_value( refine_with_Ab_after_design ) << std::endl;
	ss << tier2 << "-do_QC" << bracket_boolean_value( do_QC ) << std::endl;
	ss << std::endl;
	ss << tier1 << "multigraft closure/design attempts" << std::endl;
	ss << tier2 << "closure_attempts <int>" << bracket_value( closure_attempts ) << std::endl;
	ss << tier2 << "closure_reset_period <int>" << bracket_value( closure_reset_period ) << std::endl;
	ss << tier2 << "store_n_best_closures <int>" << bracket_value( store_n_best_closures ) << std::endl;
	ss << tier2 << "design_attempts <int>" << bracket_value( design_attempts ) << std::endl;
	ss << tier2 << "store_n_best_designs <int>" << bracket_value( store_n_best_designs ) << std::endl;
	ss << std::endl;
	ss << tier1 << "multigraft closure options" << std::endl;
	ss << tier2 << "-graft_with_Ab" << bracket_boolean_value( graft_with_Ab ) << std::endl;
	ss << tier2 << "-close_as_GLY" << bracket_boolean_value( close_as_GLY ) << std::endl;
	ss << tier2 << "-grow_as_GLY" << bracket_boolean_value( grow_as_GLY ) << std::endl;
	ss << tier2 << "-rotation_perturb_magnitude <real>" << bracket_value( rotation_perturb_magnitude ) << std::endl;
	ss << tier2 << "-translation_perturb_magnitude <real>" << bracket_value( translation_perturb_magnitude ) << std::endl;
	ss << tier2 << "-no_fragment_insertion" << bracket_boolean_value( !use_fragment_insertion ) << std::endl;
	ss << tier2 << "-use_sequence_biased_fragments" << bracket_boolean_value ( use_sequence_biased_fragments ) << std::endl;
	ss << tier2 << "-variable_fragments" << bracket_boolean_value( use_variable_length_fragments ) << std::endl;
	ss << tier2 << "-allow_any_ss_during_fragment_insertion" << bracket_boolean_value( allow_any_ss_during_fragment_insertion ) << std::endl;
	ss << tier2 << "-nfrag" << bracket_value( number_of_fragments ) << std::endl;
	ss << tier2 << "-build_with_adaptive" << bracket_boolean_value( build_with_adaptive ) << std::endl;
	ss << tier2 << "-build_cycles <int>" << bracket_value( build_cycles ) << std::endl;
	ss << tier2 << "-refine_with_minrepack" << bracket_value( refine_with_minrepack ) << std::endl;
	ss << tier2 << "-refine_with_classic" << bracket_value( refine_with_classic ) << std::endl;
	ss << tier2 << "-refine_with_constraints" << bracket_value( refine_with_constraints ) << std::endl;
	ss << tier2 << "-fast_refine" <<  bracket_boolean_value( use_fast_refine ) << std::endl;
	ss << tier2 << "-refine_cycles <int>" << bracket_value( refine_cycles ) << std::endl;
	ss << std::endl;
	ss << tier1 << "multigraft closure criteria" << std::endl;
	ss << tier2 << "-max_chainbreak_score <real>" << bracket_value( max_chainbreak_score ) << std::endl;
	ss << tier2 << "-max_local_rama <real>" << bracket_value( max_local_rama ) << std::endl;
	ss << std::endl;
	ss << tier1 << "multigraft design options" << std::endl;
	ss << tier2 << "-repack_epitope" << bracket_boolean_value( repack_epitope ) << std::endl;
	ss << tier2 << "-design_epitope" << bracket_boolean_value( design_epitope ) << std::endl;
	ss << tier2 << "-repack_Ab" << bracket_boolean_value( repack_Ab ) << std::endl;
	ss << tier2 << "-allow_AA_at_inter_design" << bracket_boolean_value( allow_AA_at_inter_design ) << std::endl;
	ss << tier2 << "-intra_design_cutoff <real>" << bracket_value( intra_design_cutoff ) << std::endl;
	ss << tier2 << "-inter_design_cutoff <real>" << bracket_value( inter_design_cutoff ) << std::endl;
	ss << tier2 << "-Ab_repack_cutoff <real>" << bracket_value( Ab_repack_cutoff ) << std::endl;
	ss << std::endl;
	ss << tier1 << "multigraft scan behavior" << std::endl;
	ss << tier2 << "-scan_randomize_moveable" << bracket_boolean_value( scan_randomize_moveable ) << std::endl;
	ss << tier2 << "-scan_randomize_cutpoints" << bracket_boolean_value( scan_randomize_cutpoints ) << std::endl;
	ss << tier2 << "-scan_randomize_ss" << bracket_boolean_value( scan_randomize_ss ) << std::endl;
	ss << tier2 << "-scan_ss_minimum_content <real>" << bracket_value( scan_ss_minimum_content ) << std::endl;
	ss << tier2 << "-scan_ss_use_any" << bracket_boolean_value( false ) << std::endl; // the actual variable is scan_ss_type
	ss << std::endl;
	ss << tier1 << "multigraft interaction design" << std::endl;
	ss << tier2 << "-complementarity_design <string>" << bracket_value( complementarity_residue_file ) << std::endl;
	ss << tier2 << "-complementarity_rb" << bracket_boolean_value( complementarity_rb ) << std::endl;
	ss << tier2 << "-complementarity_design_cutoff" << bracket_value( complementarity_design_cutoff ) << std::endl;
	ss << tier2 << "-complementarity_design_attempts" << bracket_value( complementarity_design_attempts ) << std::endl;
	ss << tier2 << "-complementarity_design_cycles" << bracket_value( complementarity_design_cycles ) << std::endl;
	ss << tier2 << "-complementarity_shell_repack" << bracket_boolean_value( complementarity_shell_repack ) << std::endl;
	ss << tier2 << "-complementarity_shell_redesign" << bracket_boolean_value( complementarity_shell_redesign ) << std::endl;
	ss << std::endl;
	ss << tier1 << "multigraft export" << std::endl;
	ss << tier2 << "-export_blueprint" << bracket_boolean_value( export_blueprint ) << std::endl;
	ss << tier2 << "-export_resfile" << bracket_boolean_value( export_resfile ) << std::endl;
	ss << std::endl;
	ss << tier1 << "temporary multigraft" << std::endl;
	ss << tier2 << "-restart_from <string>" << bracket_value( restart_from ) << std::endl;
	ss << tier2 << "-post_modify_restart" << bracket_boolean_value( post_modify_restart ) << std::endl;

	return ss.str();
}


/// @brief format value for usage() by adding space and brackets []
template< typename U >
std::string
GraftOptions::bracket_value(
	U const & val
) const
{
	std::stringstream ss;
	ss << "    [ " << val << " ]";
	return ss.str();
}

/// @brief format value for usage() by adding space and brackets {}
template< typename U >
std::string
GraftOptions::bracket_boolean_value(
	U const & val
) const
{
	std::stringstream ss;
	std::string val_string = "false";
	if ( val ) {
		val_string = "true";
	}
	ss << "    { " << val_string << " }";
	return ss.str();
}


} // namespace epigraft
