// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   GraftOptions.hh
/// @brief  Simple epigraft options container, plain-data objects, almost everything is public.
/// @brief  Not meant as a general purpose options parser, used for storing epigraft options in one object.
/// @note   This is supposed to named just 'Options.cc', but due to Mac filesystem being non-case sensitive
/// @note   and a flaw in Rosetta's 'make' setup, there's a linker conflict.  No time to change the make
/// @note   setup right now, so leave it named GraftOptions.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef INCLUDED_epigraft_GraftOptions_HH_
#define INCLUDED_epigraft_GraftOptions_HH_

// package headers
#include <epigraft/epigraft_types.hh>

// C++ headers
#include <string>


namespace epigraft {


/// @brief  Simple epigraft options container, plain-data objects, almost everything is public.
/// @brief  Not meant as a general purpose options parser, used for storing epigraft options in one object.
class GraftOptions {


	public: // construct/destruct

		/// @brief default constructor
		GraftOptions()
		{
			initialize();
		}

		/// @brief default destructor
		~GraftOptions() {}


	public: // process options

		/// @brief get options from rosetta command line using "afteropts"
		void
		get_from_rosetta();


	public: // initialization

		/// @brief initialize options with default values
		void
		initialize();


	public: // status and usage

		/// @brief options and usage information
		// TODO: fill in procedure
		std::string
		usage();

		/// @brief report status
		// TODO: fill in procedure
//		std::string
//		to_string();


	private: // output formatting

		/// @brief format value for usage() by adding space and brackets []
		template< typename U >
		std::string
		bracket_value(
			U const & val
		) const;

		/// @brief format value for usage() by adding space and brackets [] for true/false value
		template< typename U >
		std::string
		bracket_boolean_value(
			U const & val
		) const;


	public: // data

		// modes
		bool match_mode;
		bool multigraft_mode;
		bool decimated_multigraft_mode;
		bool test_mode;
		bool scarlet; // enable experimental options

		// submodes
		bool rough_match;
		bool combi_match;
		bool superposition_minrepack_refine;
		bool screen_with_repack;

		// alignment systems
		bool skip_N2C_align;
		bool skip_C2N_align;
		bool use_E_align;
		bool use_S_align;
		bool use_SS_align;

		// input
		std::string native_complex_filename;
		std::string loop_ranges_filename; // epitope loop ranges
		std::string input_filename;
		bool use_input_from_match;
		bool input_pdb_has_Ab;

		// data specific
		Integer nres_Ab;
		bool Ab_first;
		bool scaffold_first;

		// range scanning options
		Integer termini_residue_skip; // skip this number of residues from both the N and C termini
		Integer min_match_width; // minimum gap size
		Integer max_match_width_delta; // maximum difference between loop size and match gap size
		Integer max_rough_match_width_delta; // maximum difference between loop size and rough match gap size, def: max_match_width_delta
		Integer moveable_closure_residues; // number of _scaffold_ residues adjacent to break that are allowed to move during closure

		// filter
//		Real match_distance_epsilon; // epsilon added to CA-CA distance check between endpoints
		Real max_closure_rms;
		Real max_rms_over_length;
		Real max_intra_clash;
		Real max_inter_clash;
		bool use_full_sidechain_inter_clash;
		bool use_epitope_sidechain_during_clash_check;
		Real rough_match_closure_rms; // closure rms for rough match
		Real rough_match_ca_distance; // max distance for CA-CA distance check during rough match
		Real combi_match_ca_distance; // max distance for CA-CA distance check during combi match

		// conformation
		bool fluidize_landing;
		bool fluidize_takeoff;
		bool rb_move;
		Real recovery_rms_epsilon; // conformational recovery/optimization is attempted for matches that are within max_closure_rms + recovery_rms_epsilon
		Real allowed_intra_clash_increase; // allowed intra-clash increase during dihedral fluidization trials
		Real dihedral_deviation; // allowed dihedral angle deviation (+ and -) for conformation recovery/optimization
		Real dihedral_step; // angle step during dihedral changes
		Real rb_cube_side_length; // side length of cube for rb translation
		Real rb_translation_step; // rb translation step
		Real rb_angle_deviation; // allowed angle deviation (+ and - ) for rotation around each of the three coordinate axes during rb moves
		Real rb_angle_step; // angle step during rb rotation

		// additional filters
		bool use_spatial_filter;
		std::string pdb_for_spatial_filter;
		Real spatial_filter_distance_cutoff;
		Integer spatial_filter_min_cbeta; // minimum number of C-beta on (all-ala) scaffold required to keep match result
		Integer spatial_filter_max_cbeta; // maximum number of C-beta on (all-ala) scaffold required to keep match result

		// additional statistics
		bool compute_cbeta_neighbors;
		Real cbeta_neighbors_distance_cutoff;

		// output
		std::string output_filename;
		bool output_single_matches; // true by default
		bool full_matches_only; //vds false by default
		bool output_aligned_loops;
		bool output_predesign_structure;
		bool output_predesign_structure_with_Ab;
		bool override_rosetta_pdb_output_path;
		std::string pdb_output_path;

		// checkpoint
		bool use_checkpoint;
		std::string checkpoint_filename;
		std::string checkpoint_filename_inner;
		Integer checkpoint_interval_seconds; // currently only for multigraft

		// multigraft specific
		std::string vall_filename; // dummy variable, see fragments_pose.cc for actual setting
		bool use_keep_natro;
		std::string keep_natro_filename;
		bool use_batch_id;
		std::string batch_id;
		bool dump_predesign;
		std::string predesign_filename;
		bool dump_all_closure_attempt_structures;
		bool dump_closed; //vds just dump closed decoys (after build and after refine).
		bool dump_refined; //vds just dump closed decoys (after refine, but not the build ones).
		bool dump_someclosed; //vds or structures with at least one loop closed, in multigraft


		// multigraft, initial geometry
		bool idealize_loop_geometry; // only for testing
		bool epitope_rb_optimize;
		Integer epitope_rb_cycles;
		bool epitope_rb_min;
		bool Ab_epitope_optimize;
		bool Ab_epitope_optimize_including_rb;
		bool epitope_optimize;

		// multigraft procedure
		bool micromanage_termini; // true unless explicitly turned off
		bool randomize_moveable_phipsi;
		bool build_loops;
		bool refine_loops;
		bool stop_on_build_failure;
		bool stop_on_refine_failure;
		bool design_after_closure;
		bool refine_with_Ab_after_design;
		bool do_QC; // do quality control stage
		bool store_top_design_per_backbone; // only track top design per backbone

		// multigraft closure/design attempts
		Integer closure_attempts; // build + refine attempts
		/// @brief number of times to attempt to close a particular chainbreak before resetting
		/// @brief the moveable residues surrounding that chainbreak to the original backbone
		Integer closure_reset_period;
		Integer store_n_best_closures; // number of best closures to store based on score
		Integer design_attempts; // number of design attempts
		Integer store_n_best_designs; // number of best designs to store based on score

		// multigraft closure options
		bool graft_with_Ab;
		bool close_as_GLY;
		Integer closure_residue_type; // ala by default since 20/Mar/08
		bool grow_as_GLY;
		Real rotation_perturb_magnitude;
		Real translation_perturb_magnitude;
		bool use_fragment_insertion; // build loop option
		bool use_sequence_biased_fragments; // build loop option
		bool use_variable_length_fragments; // build loop option
		bool allow_any_ss_during_fragment_insertion; // build loop option
		Integer number_of_fragments; // build loop option
		bool build_with_adaptive; // build loops using adaptive protocol
		bool build_with_arm; // build loops using arm protocol (typically used with N2C/C2N w/ lever)
		bool build_with_screen; // build loops using screen protocol
		bool build_with_fragments_only; // build using fragments only, doesn't fully close (will depend on refine)
		Integer build_cycles; // build loop option
		bool force_refine; // refine loops even if loops aren't fully closed
		bool refine_with_minrepack; // refine loops using min-repack protocol
		bool refine_with_classic; // refine loops using classic pose loops style protocol
		bool refine_with_constraints; // refine loops using coordinate constraints protocol
		bool use_fast_refine; // refine loop option
		Integer refine_cycles; // refine loop option

		// multigraft closure criteria
		Real max_chainbreak_score;
		Real max_local_rama;

		// multigraft design options
		bool repack_epitope;
		bool design_epitope; //vds allow epitope to design use usually in conjunction with keep_natro file
		bool repack_Ab;
		bool allow_AA_at_inter_design;
		Real intra_design_cutoff;
		Real inter_design_cutoff;
		Real Ab_repack_cutoff;

		// multigraft scan behavior
		bool use_scan_behavior; // master switch for scan behavior
		bool scan_randomize_moveable;
		bool scan_randomize_cutpoints;
		bool scan_randomize_ss;
		Real scan_ss_minimum_content;
		char scan_ss_type;

		// multigraft interaction design
		bool complementarity_design;
		std::string complementarity_residue_file;
		bool complementarity_rb;
		Real complementarity_design_cutoff;
		Integer complementarity_design_attempts;
		Integer complementarity_design_cycles;
		Real complementarity_shell_cutoff;
		bool complementarity_shell_repack;
		bool complementarity_shell_redesign;

		// multigraft export
		bool export_blueprint;
		bool export_resfile;

		// temporary multigraft
		bool do_restart;
		std::string restart_from;
		bool post_modify_restart;

		// testing options
		bool test_water;
		bool test_tea;
		bool test_coffee;
		bool test_milk;
		bool test_port;
		bool test_juice;
		bool test_cola;
		bool test_wine;
		bool test_beer;
		bool test_bubble_tea;
		bool test_mango_lassi;
		bool test_vodka;
		bool test_tequila;
		Integer test_ccd_type;

		// options for handling deprecated features
		bool use_old_graft_info_format;

		// decimated multigraft options
		Integer nstruct;
		Real build_attempt_multiplier;
		Integer max_refine_attempts_per_round;
};

} // namespace epigraft


#endif /*INCLUDED_epigraft_GraftOptions_HH_*/
