// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   LoopInfo.hh
/// @brief  Tracks loop (epitope) info and ranges.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef INCLUDED_epigraft_LoopInfo_HH_
#define INCLUDED_epigraft_LoopInfo_HH_

// package headers
#include <epigraft/ResidueRange.hh>

// utility headers
#include <utility/vector1.hh>

namespace epigraft {

/// @brief  Tracks loop (epitope) info and ranges.
class LoopInfo {

	public: // construct/destruct

		/// @brief default constructor
		inline
		LoopInfo() {}

		/// @brief constructor
		inline
		LoopInfo(
			int id,
			ResidueRange full_native_range,
			bool is_primary = true
		) : id_( id ),
				is_primary_( is_primary ), //vds
				full_native_range_( full_native_range ),
		    full_internal_range_( convert_native_to_internal( full_native_range ) )
		{}

		/// @brief default destructor;
		inline
		~LoopInfo() {}


	public: // accessors

		/// @brief id of loop
		inline
		int const &
		id() const
		{
			return id_;
		}

		/// @brief number of residues in loop
		inline
		int
		nres() const
		{
			return full_native_range_.length();
		}

	  /// @brief whether this loop is a potential primary //vds
	  inline
    bool const &
		is_primary() const
	  {
			return is_primary_;
		}

		/// @brief get native loop subranges
		inline
		utility::vector1< ResidueRange > const &
		native_subranges() const
		{
			return native_subranges_;
		}

		/// @brief get internal subranges
		inline
		utility::vector1< ResidueRange > const &
		internal_subranges() const
		{
			return internal_subranges_;
		}

		/// @brief get full native range (this is the entire range of the loop on which
		/// @brief subranges are defined)
		inline
		ResidueRange const &
		full_native_range() const
		{
			return full_native_range_;
		}

		/// @brief get full internal range (this is the entire range of the loop on which
		/// @brief subranges are defined)
		inline
		ResidueRange const &
		full_internal_range() const
		{
			return full_internal_range_;
		}


	public: // methods

		/// @brief add epitope subrange
		void
		add_native_subrange(
			ResidueRange const & rr
		);

		/// @brief contains supplied native range?
		bool
		contains_native_subrange(
			ResidueRange const & rr
		) const;

		/// @brief convert native subrange to internal subrange
		inline
		ResidueRange
		convert_native_to_internal(
			ResidueRange const & native_subrange
		) const
		{
			return native_subrange - full_native_range_.begin() + 1;
		}


	public: // status

		/// @brief report object contents
		std::string
		to_string();


	private: // data

		int id_;
	  bool is_primary_;
		utility::vector1< ResidueRange > native_subranges_;
		utility::vector1< ResidueRange > internal_subranges_;
		ResidueRange full_native_range_;
		ResidueRange full_internal_range_;

};

} // namespace epigraft

#endif /*INCLUDED_epigraft_LoopInfo_HH_*/
