// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   DihedralInfo.hh
/// @brief  Stores residue + dihedral information.
/// @note   Uses enum and switch statement instead of multiple subclasses due to annoyances
/// @note   with disallowed abstract class instantiation and 'if' workaround.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef INCLUDED_epigraft_conformation_DihedralInfo_HH_
#define INCLUDED_epigraft_conformation_DihedralInfo_HH_

// package headers
#include <epigraft/conformation/conformation_types.hh>

// Rosetta headers
#include <pose.h>

// utility headers
#include <utility/pointer/ReferenceCount.hh>

// c++ headers
#include <sstream>


namespace epigraft {
namespace conformation {


/// @brief  Stores residue + dihedral information.
class DihedralInfo : public utility::pointer::ReferenceCount {

	public: // enum
	
		enum AngleType {
			PHI,
			PSI,
			OMEGA
		};


	private: // typedefs
	
		typedef utility::pointer::ReferenceCount Super;


	public: // construct/destruct
	
		/// @brief default constructor
		inline
		DihedralInfo()
		{}
		
		/// @brief constructor
		inline
		DihedralInfo(
			AngleType type,
			Integer residue,
			Real angle
		) : type_( type ),
		    residue_( residue ),
		    angle_( angle )
		{}
		
		/// @brief copy constructor
		inline
		DihedralInfo(
			DihedralInfo const & d
		) : ReferenceCount(),
		    type_( d.type_ ),
		    residue_( d.residue_ ),
		    angle_( d.angle_ )
		{}
		
		/// @brief default destructor
		inline
		~DihedralInfo() {}


	public: // assignment
	
		/// @brief copy assignment
		inline
		DihedralInfo &
		operator =( DihedralInfo const & d )
		{
			if ( this != &d ) {
				Super::operator =( d );
				type_ = d.type_;
				residue_ = d.residue_;
				angle_ = d.angle_;
			}
			return *this;
		}

	public: // operators
	
		/// @brief given 'a' and 'b', if a.residue < b.residue return true otherwise
		/// @brief compare dihedral index ( phi < psi < omega)
		friend
		inline
		bool
		operator <( DihedralInfo const & a, DihedralInfo const & b )
		{
			if ( a.residue_ < b.residue_ ) {
				return true;
			} else if ( ( a.residue_ == b.residue_ ) && ( a.type_ < b.type_ ) ) {
				return true;
			}
			return false;
		}
		
		/// @brief given 'a' and 'b', if a.residue > b.residue return true otherwise
		/// @brief compare dihedral index ( phi > psi > omega)
		friend
		inline
		bool
		operator >( DihedralInfo const & a, DihedralInfo const & b )
		{
			if ( a.residue_ > b.residue_ ) {
				return true;
			} else if ( ( a.residue_ == b.residue_ ) && ( a.type_ > b.type_ ) ) {
				return true;
			}
			return false;
		}
		
		/// @brief given 'a' and 'b', if a.residue == b.residue and dihedral indexes are the same
		/// @brief return true
		friend
		inline
		bool
		operator ==( DihedralInfo const & a, DihedralInfo const & b )
		{
			return ( a.residue_ == b.residue_ ) && ( a.type_ == b.type_ );
		}
		
		/// @brief given 'a' and 'b', if a.residue != b.residue or dihedral indices are different
		/// @brief return true
		friend
		inline
		bool
		operator !=( DihedralInfo const & a, DihedralInfo const & b )
		{
			return ( a.residue_ != b.residue_ ) || ( a.type_ != b.type_ );
		}


	public: // methods
	
		/// @brief given a Pose, set the corresponding dihedral angle
		void
		apply(
			Pose & p
		) const
		{
			switch( type_ ) {
				case PHI:   p.set_phi( residue_, angle_ );    break;
				case PSI:   p.set_psi( residue_, angle_ );    break;
				case OMEGA: p.set_omega( residue_, angle_ );  break;
			}
		}
	
	
	public: // accessors
	
		/// @brief return residue
		inline
		Integer const &
		residue() const
		{
			return residue_;
		}
		
		/// @brief return angle
		inline
		Real const &
		angle() const
		{
			return angle_;
		}
		
		/// @brief return type
		inline
		AngleType const &
		type() const
		{
			return type_;
		}
		
		/// @brief set residue
		inline
		void
		set_residue(
			Integer const & residue
		)
		{
			residue_ = residue;
		}
		
		/// @brief set angle
		inline
		void
		set_angle(
			Real const & angle
		)
		{
			angle_ = angle;
		}
		
		/// @brief set AngleType
		inline
		void
		set_type(
			AngleType const & type
		)
		{
			type_ = type;
		}
		
	
	public: // status
		
		/// @brief report status
		std::string
		to_string() const
		{
			std::stringstream ss;
			ss << "residue " << residue_ << " dihedral ";
			switch ( type_ ) {
				case PHI:   ss << "PHI";    break;
				case PSI:   ss << "PSI";    break;
				case OMEGA: ss << "OMEGA";  break;
			}
			ss << " angle " << angle_;
			return ss.str();
		}
	
	
	protected: // data
	
		AngleType type_; // provides ordering of phi -> psi -> omega for comparison, internal and not set-able!
	
		Integer residue_;
		Real angle_;
	
};

} // conformation
} // epigraft

#endif /*INCLUDED_epigraft_conformation_DihedralInfo_HH_*/
