// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   FluidLanding.cc
/// @brief  Class to fluidize and check dihedrals near landing endpoints, primarily for
/// @brief  either recovering lost positions (bad rms) or generating diversity.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

// unit headers
#include <epigraft/conformation/FluidLanding.hh>

// package headers
#include <epigraft/conformation/conformation_types.hh>
#include <epigraft/conformation/DihedralInfo.hh>
#include <epigraft/epigraft_functions.hh>
#include <epigraft/ResidueRange.hh>

// Rosetta headers
#include <fold_tree.h>
#include <pose.h>

// utility headers
#include <utility/pointer/access_ptr.hh>
#include <utility/pointer/owning_ptr.hh>
#include <utility/vector1.hh>


namespace epigraft {
namespace conformation {


/// @brief attempt fluidization of end and return angle states that satisfy rms_cutoff
void
FluidLanding::fluidize_n_terminus(
	utility::vector1< DihedralVector > const & trial_dihedrals,
	Real const & rms_cutoff,
	utility::vector1< FluidLandingResult > & results
)
{
	// construct initial state
	DihedralVectorTransient state;
	for ( Integer i = 1, ie = trial_dihedrals.size(); i <= ie; ++i ) { // C -> N
		state.push_back( DihedralInfoAP( trial_dihedrals[ i ][ 1 ].get() ) );
	}
	
	// recurse and check
	recursive_check( trial_dihedrals, state, 0, rms_cutoff, results ); // starts at level 0
}


/// @brief recursion for checking angles
void
FluidLanding::recursive_check(
	utility::vector1< DihedralVector > const & trial_dihedrals,
	DihedralVectorTransient & state,
	Integer level,
	Real const & rms_cutoff,
	utility::vector1< FluidLandingResult > & results
)
{
	// apply dihedral angles to loop
	for ( Integer i = 1, ie = state.size(); i <= ie; ++i ) {
		state[ i ]->apply( loop_context_ );
	}
	
	// force refold
	// TODO: is there a better way to refold?  why is Pose::refold() private?
	loop_context_.refold_sidechains_from_chi();

	// grab coordinates from moved landing residue for chainbreak computation
	FArray2D_float moved_landing_crd;
	extract_1bb_from_Pose( loop_context_, landing_residue_, moved_landing_crd );
	moved_landing_crd.redimension( 3, 5 );
	Integer idx = loop_context_.full_coord().index( 1, adjacent_landing_atom_, adjacent_landing_residue_ );
	moved_landing_crd[ 12 ] = loop_context_.full_coord()[ idx ];
	moved_landing_crd[ 13 ] = loop_context_.full_coord()[ idx + 1 ];
	moved_landing_crd[ 14 ] = loop_context_.full_coord()[ idx + 2 ];
	
	// compute chainbreak
	Real rms = five_point_chainbreak( moved_landing_crd, break_crd_ );

	if ( rms < rms_cutoff ) {
		// fill in additional information
		Real closure_angle = angle_of_degrees( &loop_context_.full_coord()( 1, 2, adjacent_landing_residue_ ), &moved_landing_crd[ 3 ], &break_crd_[ 15 ], &break_crd_[ 3 ] ); // angle between CA->CA vectors

		FluidLandingResult fr( rms, state, closure_angle );
		
		// alter residue numbering in result DihedralVector to correspond with input scaffold
		for ( utility::vector1< DihedralInfo >::iterator d = fr.dihedrals().begin(), de = fr.dihedrals().end(); d != de; ++d ) {
			DihedralInfo & dihedral = *d;
			dihedral.set_residue( context_to_original_residue[ dihedral.residue() ] );
		}
		
		results.push_back( fr ); // add to results
	}

	// branch to new states
	level++;
	for ( Integer l = level, le = state.size(); l <= le; ++l ) {
		DihedralVectorTransient new_state = state;
		for ( Integer d = 2, de = trial_dihedrals[ l ].size(); d <= de; ++d ) {
			new_state[ l ] = DihedralInfoAP( trial_dihedrals[ l ][ d ].get() );
			recursive_check( trial_dihedrals, new_state, l, rms_cutoff, results );
		}
	}
}


/// @brief creates both n- and c-terminal dihedral reference vectors containing the
/// @brief original dihedral angles in the loop context
void
FluidLanding::create_reference_dihedrals()
{
	// n-terminal reference vectors
	for ( Integer i = 2, ie = 3; i <= ie; ++i ) { // N -> C
		nt_reference_dihedrals_.push_back( DihedralInfoOP( new DihedralInfo( DihedralInfo::PHI, i, loop_context_.phi( i ) ) ) );
		nt_reference_dihedrals_.push_back( DihedralInfoOP( new DihedralInfo( DihedralInfo::PSI, i, loop_context_.psi( i ) ) ) );
	}
	
	// c-terminal reference vectors
	for ( Integer i = loop_context_.total_residue() - 1, ie = loop_context_.total_residue() - 2; i >= ie; --i ) { // C -> N
		ct_reference_dihedrals_.push_back( DihedralInfoOP( new DihedralInfo( DihedralInfo::PSI, i, loop_context_.psi( i ) ) ) );
		ct_reference_dihedrals_.push_back( DihedralInfoOP( new DihedralInfo( DihedralInfo::PHI, i, loop_context_.phi( i ) ) ) );
	}
}


/// @brief creates both n- and c-terminal trial dihedrals in a list-of-lists
void
FluidLanding::create_trial_dihedrals()
{
	// n-terminal trial dihedral list-of-lists, N -> C due to reference vector directionality
	for ( Integer i = 1, ie = nt_reference_dihedrals_.size(); i <= ie; ++i ) {
		DihedralInfo ref_dihedral = *nt_reference_dihedrals_[ i ];
		Real first_angle = ref_dihedral.angle() - angle_left_;
		Real last_angle = ref_dihedral.angle() + angle_right_;
		
		DihedralVector position; // dihedrals for single position
		
		for ( Real a = first_angle; a <= last_angle; a += angle_step_ ) {
			position.push_back( DihedralInfoOP( new DihedralInfo( ref_dihedral.type(), ref_dihedral.residue(), a ) ) );
		}
		
		nt_trial_dihedrals_.push_back( position );
	}

	// c-terminal trial dihedral list-of-lists, C- > N due to reference vector directionality
	for ( Integer i = 1, ie = ct_reference_dihedrals_.size(); i <= ie; ++i ) {
		DihedralInfo ref_dihedral = *ct_reference_dihedrals_[ i ];
		Real first_angle = ref_dihedral.angle() - angle_left_;
		Real last_angle = ref_dihedral.angle() + angle_right_;
		
		DihedralVector position; // dihedrals for single position
		
		for ( Real a = first_angle; a <= last_angle; a += angle_step_ ) {
			position.push_back( DihedralInfoOP( new DihedralInfo( ref_dihedral.type(), ref_dihedral.residue(), a ) ) );
		}
		
		ct_trial_dihedrals_.push_back( position );
	}
}


/// @brief creates "loop context" Pose consisting of four residues surrounding both match
/// @brief range endpoints and the loop itself
void
FluidLanding::create_loop_context(
	Pose const & scaffold,
	ResidueRange const & match_range,
	Pose const & loops,
	ResidueRange const & loop_subrange
)
{
	Integer total_residue = 8 + loop_subrange.length();
	prepare_pose( loop_context_, total_residue );
	
	FArray3D_float full_coord( 3, param::MAX_ATOM(), total_residue ); // loop_context full_coord
	
	Integer counter( 1 ); // residue counter, updated during fill_full_coord_from_Pose
	
	// transfer non-coordinate info for four residues around match_range.begin()  xxBx
	ResidueRange n_quad( match_range.begin() - 2, match_range.begin() + 1 );
	transfer_identity_info_between_Poses( scaffold, n_quad, loop_context_, counter );
	fill_full_coord_from_Pose( scaffold, n_quad, full_coord, counter );
	
	// transfer non-coordinate info for all loop residues
	transfer_identity_info_between_Poses( loops, loop_subrange, loop_context_, counter );
	fill_full_coord_from_Pose( loops, loop_subrange, full_coord, counter );
	
	// transfer non-coordinate info for four residues around match_range.end()  xExx
	ResidueRange c_quad( match_range.end() - 1, match_range.end() + 2 );
	transfer_identity_info_between_Poses( scaffold, c_quad, loop_context_, counter );
	fill_full_coord_from_Pose( scaffold, c_quad, full_coord, counter );
	
	assert( counter == total_residue + 1 );
	
	// make Eposition
	FArray3D_float Eposition( 3, param::MAX_POS, total_residue );
	full_coord_to_Eposition( total_residue, full_coord, Eposition );
	
	loop_context_.set_coords( false, Eposition, full_coord, false );

	// fill residue indices for handling loop context
	nt_match_residue_ = 3;
	ct_match_residue_ = loop_context_.total_residue() - 2;

	// make fold tree
	create_contextual_fold_tree();
	
	// fill map
	context_to_original_residue[ 1 ] = match_range.begin() - 2;
	context_to_original_residue[ 2 ] = match_range.begin() - 1;
	context_to_original_residue[ 3 ] = match_range.begin();
	context_to_original_residue[ 4 ] = match_range.begin() + 1;
	context_to_original_residue[ loop_context_.total_residue() - 3 ] = match_range.end() - 1;
	context_to_original_residue[ loop_context_.total_residue() - 2 ] = match_range.end();
	context_to_original_residue[ loop_context_.total_residue() - 1 ] = match_range.end() + 1;
	context_to_original_residue[ loop_context_.total_residue() ] = match_range.end() + 2; 
}


/// @brief creates and set fold tree for loop context
void
FluidLanding::create_contextual_fold_tree()
{
	using pose_ns::Fold_tree;

	Fold_tree f;
	f.add_edge( 5, loop_context_.total_residue() - 4, Fold_tree::PEPTIDE ); // fold the actual loop, directionality doesn't matter here because we won't change torsions
	f.add_edge( 5, 1, 1 ); // first jump
	f.add_edge( 1, 4, Fold_tree::PEPTIDE ); // fold scaffold n-terminus 1 -> 4
	f.add_edge( loop_context_.total_residue() - 4, loop_context_.total_residue(), 2 ); // second jump
	f.add_edge( loop_context_.total_residue(), loop_context_.total_residue() - 3, Fold_tree::PEPTIDE ); // fold scaffold c-terminus nres -> nres - 3

	f.reorder( 5 ); // start from n-terminus of loop

	loop_context_.set_fold_tree( f );
}


/// @brief setup break at n-terminus
void
FluidLanding::indicate_break_at_n_terminus()
{
	// indicate landing residue
	landing_residue_ = nt_match_residue_;

	// adjacent landing residue info
	adjacent_landing_residue_ = landing_residue_ + 1;
	adjacent_landing_atom_ = 1; // 'N'

	// indicate break information
	Integer break_residue = 5; // start of loop in loop_context_
	Integer adjacent_break_residue = break_residue + 1;
	Integer adjacent_break_atom = 1; // 'N'
	extract_1bb_from_Pose( loop_context_, break_residue, break_crd_ );
	
	// fill in 5th atom from adjacent break residue for 5 point chainbreak
	break_crd_.redimension( 3, 6 );
	Integer idx = loop_context_.full_coord().index( 1, adjacent_break_atom, adjacent_break_residue );
	break_crd_[ 12 ] = loop_context_.full_coord()[ idx ];
	break_crd_[ 13 ] = loop_context_.full_coord()[ idx + 1 ];
	break_crd_[ 14 ] = loop_context_.full_coord()[ idx + 2 ];
	
	// fill in 6th atom from adjacent break residue for closure angle
	idx = loop_context_.full_coord().index( 1, 2, adjacent_break_residue ); // 'CA'
	break_crd_[ 15 ] = loop_context_.full_coord()[ idx ];
	break_crd_[ 16 ] = loop_context_.full_coord()[ idx + 1 ];
	break_crd_[ 17 ] = loop_context_.full_coord()[ idx + 2 ];
}


/// @brief setup break at c-terminus
void
FluidLanding::indicate_break_at_c_terminus()
{
	// indicate landing residue
	landing_residue_ = ct_match_residue_;
	
	// adjacent landing residue info
	adjacent_landing_residue_ = landing_residue_ - 1;
	adjacent_landing_atom_ = 3; // 'C'
	
	// indicate break information
	Integer break_residue = loop_context_.total_residue() - 4; // end of loop in loop_context_
	Integer adjacent_break_residue = break_residue - 1;
	Integer adjacent_break_atom = 3; // 'C'
	extract_1bb_from_Pose( loop_context_, break_residue, break_crd_ );
	
	// fill in 5th atom from adjacent break residue for 5 point chainbreak
	break_crd_.redimension( 3, 6 );
	Integer idx = loop_context_.full_coord().index( 1, adjacent_break_atom, adjacent_break_residue );
	break_crd_[ 12 ] = loop_context_.full_coord()[ idx ];
	break_crd_[ 13 ] = loop_context_.full_coord()[ idx + 1 ];
	break_crd_[ 14 ] = loop_context_.full_coord()[ idx + 2 ];

	// fill in 6th atom from adjacent break residue for closure angle
	idx = loop_context_.full_coord().index( 1, 2, adjacent_break_residue ); // 'CA'
	break_crd_[ 15 ] = loop_context_.full_coord()[ idx ];
	break_crd_[ 16 ] = loop_context_.full_coord()[ idx + 1 ];
	break_crd_[ 17 ] = loop_context_.full_coord()[ idx + 2 ];
}


} // namespace conformation
} // namespace epigraft
