// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   FluidLandingResult.hh
/// @brief  Plain-data class for storing results from FluidLanding trial.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef INCLUDED_epigraft_conformation_FluidLandingResult_HH_
#define INCLUDED_epigraft_conformation_FluidLandingResult_HH_

// package headers
#include <epigraft/conformation/conformation_types.hh>
#include <epigraft/conformation/DihedralInfo.hh>

// ObjexxFCL headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray2D.hh>

// utility headers
#include <utility/pointer/access_ptr.hh>
#include <utility/pointer/owning_ptr.hh>
#include <utility/vector1.hh>

namespace epigraft {
namespace conformation {


/// @brief  Plain-data class for storing results from FluidLanding trial.
class FluidLandingResult {

	public: // constructor
	
		/// @brief default constructor
		inline
		FluidLandingResult() {}
		
		/// @brief DihedralVectorTransient constructor
		inline
		FluidLandingResult(
			Real const & rms,
			DihedralVectorTransient const & dihedrals,
			Real const & closure_angle
		) : rms_( rms ),
		    closure_angle_( closure_angle )
		{
			// need to make a copy of dihedrals here, otherwise modifications of e.g. residue numbering
			// will modify the original
			for ( Integer i = 1, ie = dihedrals.size(); i <= ie; ++i ) {
				dihedrals_.push_back( *dihedrals[ i ] );
			}
		}
		
		/// @brief DihedralVector constructor
		inline
		FluidLandingResult(
			Real const & rms,
			DihedralVector const & dihedrals,
			Real const & closure_angle
		) : rms_( rms ),
		    closure_angle_( closure_angle )
		{
			// need to make a copy of dihedrals here, otherwise modifications of e.g. residue numbering
			// will modify the original
			for ( Integer i = 1, ie = dihedrals.size(); i <= ie; ++i ) {
				dihedrals_.push_back( *dihedrals[ i ] );
			}
		}
		
		/// @brief copy constructor
		inline
		FluidLandingResult(
			FluidLandingResult const & f
		) : rms_( f.rms_ ),
		    dihedrals_( f.dihedrals_ ),
		    closure_angle_( f.closure_angle_ )
		{}
		
		/// @brief default destructor
		inline
		~FluidLandingResult() {}
		
	
	public: // assignment
	
		/// @brief copy assignment
		inline
		FluidLandingResult &
		operator =( FluidLandingResult const & f )
		{
			if ( this != &f ) {
				rms_ = f.rms_;
				dihedrals_ = f.dihedrals_;
				
				closure_angle_ = f.closure_angle_;
			}
			return *this;
		}


	public: // accessors

		/// @brief rms
		inline
		Real const &
		rms() const
		{
			return rms_;
		}
		
		/// @brief dihedrals vector
		inline
		utility::vector1< DihedralInfo > const &
		dihedrals() const
		{
			return dihedrals_;
		}
		
		/// @brief dihedrals vector
		inline
		utility::vector1< DihedralInfo > &
		dihedrals()
		{
			return dihedrals_;
		}

		/// @brief closure angle
		inline
		Real const &
		closure_angle() const
		{
			return closure_angle_;
		}

		
	private: // data
	
		Real rms_;
		utility::vector1< DihedralInfo > dihedrals_;
		
		Real closure_angle_;
		
};

} // namespace conformation
} // namespace epigraft

#endif /*INCLUDED_epigraft_conformation_FluidLandingResult_HH_*/
