// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   FluidTakeoff.hh
/// @brief  Class to fluidize and check dihedrals near closed endpoints (takeoff), primarily
/// @brief  for either recovering lost positions (bad rms) or generating diversity.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef INCLUDED_epigraft_conformation_FluidTakeoff_HH_
#define INCLUDED_epigraft_conformation_FluidTakeoff_HH_

// package headers
#include <epigraft/conformation/conformation_types.hh>
#include <epigraft/conformation/DihedralInfo.hh>
#include <epigraft/conformation/FluidTakeoffResult.hh>
#include <epigraft/ResidueRange.hh>
#include <epigraft/epigraft_functions.hh>

// Rosetta headers
#include <fold_tree.h>
#include <pose.h>

// utility headers
#include <utility/pointer/access_ptr.hh>
#include <utility/pointer/owning_ptr.hh>
#include <utility/vector1.hh>

// C++ headers
#include <map>


namespace epigraft {
namespace conformation {


/// @brief  Class to fluidize and check dihedrals near closed endpoints (takeoff), primarily
/// @brief  for either recovering lost positions (bad rms) or generating diversity.
class FluidTakeoff {

	public: // construct/destruct
	
		/// @brief default constructor
		inline
		FluidTakeoff() {}
		
		/// @brief constructor
		inline
		FluidTakeoff(
			Pose const & scaffold,
			ResidueRange const & match_range,
			Pose const & loop,
			Real angle_left,
			Real angle_right,
			Real angle_step
		) : angle_left_( angle_left ),
		    angle_right_( angle_right ),
		    angle_step_( angle_step )
		{
			create_loop_context( scaffold, match_range, loop );
			create_reference_dihedrals();
			create_trial_dihedrals();
		}
		
		/// @brief default destructor
		inline
		~FluidTakeoff() {}


	private: // disabled construct/assignment

		/// @brief copy constructor
		/// @note  disable this for now: Pose and Fold_tree constructor
		/// @note  is not 100% happy
		inline
		FluidTakeoff(
			FluidTakeoff const &
		)
		{}

		/// @brief copy assignment
		/// @note  disable this for now: Pose and Fold_tree assignment
		/// @note  is not 100% happy
		inline
		FluidTakeoff &
		operator =( FluidTakeoff const & )
		{
			return *this;
		}


	public: // methods

		/// @brief attempt fluidization of n-terminal end (N2C) and return angle states that
		/// @brief satisfy rms_cutoff
		inline
		void
		fluidize_n_terminus(
			Real const & rms_cutoff,
			utility::vector1< FluidTakeoffResult > & results
		)
		{
			indicate_break_at_c_terminus();
			fluidize_terminus( nt_trial_dihedrals_, rms_cutoff, results );
		}

		/// @brief attempt fluidization of c-terminal end (C2N) and return angle states that
		/// @brief satisfy rms_cutoff
		inline
		void
		fluidize_c_terminus(
			Real const & rms_cutoff,
			utility::vector1< FluidTakeoffResult > & results
		)
		{
			indicate_break_at_n_terminus();
			fluidize_terminus( ct_trial_dihedrals_, rms_cutoff, results );
		}

	
	public: // accessors
	
		/// @brief "loop context" Pose consisting of four residues surrounding both match
		/// @brief range endpoints and the loop itself
		inline
		Pose const &
		loop_context() const
		{
			return loop_context_;
		}


	private: // fluidize and recursion routines
	
		/// @brief attempt fluidization of end and return angle states that satisfy rms_cutoff
		void
		fluidize_terminus(
			utility::vector1< DihedralVector > const & trial_dihedrals,
			Real const & rms_cutoff,
			utility::vector1< FluidTakeoffResult > & results
		);
	
		/// @brief recursion for checking angles
		void
		recursive_check(
			utility::vector1< DihedralVector > const & trial_dihedrals,
			DihedralVectorTransient & state,
			Integer level,
			Real const & rms_cutoff,
			utility::vector1< FluidTakeoffResult > & results
		);

	
	private: // angle methods
	
		/// @brief creates both n- and c-terminal dihedral reference vectors containing the
		/// @brief original dihedral angles in the loop context
		void
		create_reference_dihedrals();

		/// @brief creates both n- and c-terminal trial dihedrals in a list-of-lists
		void
		create_trial_dihedrals();

	
	private: // Pose preparation methods
	
		/// @brief creates "loop context" Pose consisting of four residues surrounding both match
		/// @brief range endpoints and the loop itself
		void
		create_loop_context(
			Pose const & scaffold,
			ResidueRange const & match_range,
			Pose const & loop
		);
		
		/// @brief creates and set fold tree for loop context Pose
		void
		create_contextual_fold_tree();


	private: // set break type
	
		/// @brief setup as C2N type (broken at n-terminus)
		void
		indicate_break_at_n_terminus();
		
		/// @brief setup as N2C type (broken at c-terminus)
		void
		indicate_break_at_c_terminus();


	private: // data
	
		Pose loop_context_;
		
		std::map< Integer, Integer > context_to_original_residue; // filled during create_loop_context()
		
		Real angle_left_; // amount of deviation to the left (subtract) in degrees
		Real angle_right_; // amount of deviation to the right (add) in degress
		Real angle_step_; // size of angle step in degrees
		
		DihedralVector nt_reference_dihedrals_;
		DihedralVector ct_reference_dihedrals_;
		
		utility::vector1< DihedralVector > nt_trial_dihedrals_; // angles run from N -> C
		utility::vector1< DihedralVector > ct_trial_dihedrals_; // angles run from C -> N
		
		// residue indices for loop context, here just for clarity, can consider removing and stuffing elsewhere
		Integer nt_match_residue_;
		Integer ct_match_residue_;
		
		/* Data below is filled in indicate_break_at_*_terminus() methods */

		Integer takeoff_residue_; // the n- or c-terminal takeoff residue on the scaffold
		
		// cached original backbone residue coordinates for loop_context, used for finding transformation during
		// hinging
		FArray2D_float takeoff_crd_; // backbone coordinates of the n- or c-terminal takeoff residue on the scaffold
	
		// cached original backbone residue coordinates for loop_context, used for checking rms during hinging
		// contains +/- 1 additional 5th and 6th atom from adjacent residue
		FArray2D_float landing_crd_;
		
		// cached original backbone residue coordinates for loop_context, used for checking rms during hinging
		// contains +/- 1 additional 5th and 6th atom from adjacent residue
		FArray2D_float break_crd_;
};


} // namespace conformation
} // namespace epigraft


#endif /*INCLUDED_epigraft_conformation_FluidTakeoff_HH_*/
