// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   TransformGenerator.hh
/// @brief  Generates a set of rigid body transforms ( 4x4 homogeneous coordinates ) given an
/// @brief  origin and axis or axes.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef INCLUDED_epigraft_conformation_TransformGenerator_HH_
#define INCLUDED_epigraft_conformation_TransformGenerator_HH_

// numeric headers
#include <numeric/conversions.hh>
#include <numeric/xyzVector.hh>

// ObjexxFCL headers
#include <ObjexxFCL/FArray2D.hh>


namespace epigraft {
namespace conformation {


/// @brief  Generates a set of rigid body transforms ( 4x4 homogeneous coordinates ) given an
/// @brief  origin and axes.
template< typename T >
class TransformGenerator {
	
	public: // types
	
		typedef numeric::xyzVector< T > xyzVector;
		typedef ObjexxFCL::FArray2D< T > FArray2D;


	public: // construct/destruct
	
		/// @brief default constructor
		inline
		TransformGenerator() {}
		
		/// @brief three-axis constructor
		inline
		TransformGenerator(
			xyzVector const & origin,
			xyzVector const & x_axis,
			xyzVector const & y_axis,
			xyzVector const & z_axis
		) : origin_( origin )
		{
			// force normalization, if you know that input vectors are always normalized, can remove this
			x_axis.normalized( x_axis_ );
			y_axis.normalized( y_axis_ );
			z_axis.normalized( z_axis_ );
			
			initialize(); // cache transformation matrices for translation to/from the origin
		}
		
		/// @brief two-axis (x and z) constructor, third axis (y) found by cross product
		inline
		TransformGenerator(
			xyzVector const & origin,
			xyzVector const & x_axis,
			xyzVector const & z_axis
		) : origin_( origin )
		{
			// force normalization, if you know that input vectors are always normalized, can remove this
			x_axis.normalized( x_axis_ );
			z_axis.normalized( z_axis_ );
			
			y_axis_ = cross_product( x_axis_, z_axis_ );
			y_axis_.normalize(); // must normalized
			
			initialize(); // cache transformation matrices for translation to/from the origin
		}
		
		/// @brief one-axis (x) constructor, two other axes (y, z) inoperable
		inline
		TransformGenerator(
			xyzVector const & origin,
			xyzVector const & x_axis
		) : origin_( origin )
		{
			// force normalization, if you know that input vectors are always normalized, can remove this
			x_axis.normalized( x_axis_ );
			
			y_axis_ = xyzVector( 0, 0, 0 );
			z_axis_ = xyzVector( 0, 0, 0 );
			
			initialize(); // cache transformation matrices for translation to/from the origin
		}
		
		/// @brief copy constructor
		inline
		TransformGenerator(
			TransformGenerator const & g
		) : origin_( g.origin_ ),
		    translation_to_world_origin_( g.translation_to_world_origin_ ),
		    translation_to_local_origin_( g.translation_to_local_origin_ ),
		    x_axis_( g.x_axis_ ),
		    y_axis_( g.y_axis_ ),
		    z_axis_( g.z_axis_ )
		{}
		
		/// @brief default destructor
		inline
		~TransformGenerator() {}


	public: // assignment
	
		/// @brief copy assignment
		inline
		TransformGenerator &
		operator =( TransformGenerator const & g )
		{
			if ( this != &g ) {
				origin_ = g.origin_;
				translation_to_world_origin_ = g.translation_to_world_origin_;
				translation_to_local_origin_ = g.translation_to_local_origin_;
				
				x_axis_ = g.x_axis_;
				y_axis_ = g.y_axis_;
				z_axis_ = g.z_axis_;
			}
			return *this;
		}


	public: // accessors
	
		/// @brief origin
		inline
		xyzVector const &
		origin() const
		{
			return origin_;
		}
	
		/// @brief x-axis unit vector
		inline
		xyzVector const &
		x_axis() const
		{
			return x_axis_;
		}
		
		/// @brief y-axis unit vector
		inline
		xyzVector const &
		y_axis() const
		{
			return y_axis_;
		}
		
		/// @brief z-axis unit vector
		inline
		xyzVector const &
		z_axis() const
		{
			return z_axis_;
		}
	
	
	public: // identity
	
		/// @brief return 4x4 identity
		inline
		void
		identity(
			FArray2D & RT
		) const
		{
			// remember that FArrays are column-major:
			// | 0 4  8 12 |
			// | 1 5  9 13 |
			// | 2 6 10 14 |
			// | 3 7 11 15 |  ->  this last row is always | 0 0 0 1 |
			RT = 0;
			RT[ 0 ]  = 1;
			RT[ 5 ]  = 1;
			RT[ 10 ] = 1;
			RT[ 15 ] = 1;
		}
	
	
	public: // rotation
	
		/// @brief transformation matrix rotating about x in radians, output RT must be predimensioned to 4x4!
		inline
		void
		rotate_x_radians(
			T const & angle,
			FArray2D & RT
		) const
		{
			FArray2D q( 4, 4 ); // temporary storage
			
			rotate( x_axis_, angle, q );
			correct_for_local_origin( q, RT );
		}

		/// @brief transformation matrix rotating about y in radians, output RT must be predimensioned to 4x4!
		inline
		void
		rotate_y_radians(
			T const & angle,
			FArray2D & RT
		) const
		{
			FArray2D q( 4, 4 ); // temporary storage
			
			rotate( y_axis_, angle, q );
			correct_for_local_origin( q, RT );
		}

		/// @brief transformation matrix rotating about z in radians, output RT must be predimensioned to 4x4!
		inline
		void
		rotate_z_radians(
			T const & angle,
			FArray2D & RT
		) const
		{
			FArray2D q( 4, 4 ); // temporary storage
			
			rotate( z_axis_, angle, q );
			correct_for_local_origin( q, RT );
		}

		/// @brief transformation matrix rotating about x in degrees, output RT must be predimensioned to 4x4!
		inline
		void
		rotate_x_degrees(
			T const & angle,
			FArray2D & RT
		) const
		{
			rotate_x_radians( numeric::conversions::radians( angle ), RT );
		}

		/// @brief transformation matrix rotating about y in degrees, output RT must be predimensioned to 4x4!
		inline
		void
		rotate_y_degrees(
			T const & angle,
			FArray2D & RT
		) const
		{
			rotate_y_radians( numeric::conversions::radians( angle ), RT );
		}

		/// @brief transformation matrix rotating about z in degrees, output RT must be predimensioned to 4x4!
		inline
		void
		rotate_z_degrees(
			T const & angle,
			FArray2D & RT
		) const
		{
			rotate_z_radians( numeric::conversions::radians( angle ), RT );
		}


	public: // translation
	
		/// @brief transformation matrix translation in x, output RT must be predimensioned to 4x4!
		inline
		void
		translate_x(
			T const & distance,
			FArray2D & RT
		) const
		{
			translate( x_axis_, distance, RT );
		}
	
		/// @brief transformation matrix translation in y, output RT must be predimensioned to 4x4!
		inline
		void
		translate_y(
			T const & distance,
			FArray2D & RT
		) const
		{
			translate( y_axis_, distance, RT );
		}
	
		/// @brief transformation matrix translation in z, output RT must be predimensioned to 4x4!
		inline
		void
		translate_z(
			T const & distance,
			FArray2D & RT
		) const
		{
			translate( z_axis_, distance, RT );
		}
	
	
	public: // sequence of translations
	
		/// @brief generate translations for a cube given a side length and step size, these are meant to be
		/// @brief applied to a fixed frame; note that container 'U' requires the push_back() method
		template< typename U >
		inline
		void
		push_translate_cube(
			Real const & side,
			Real const & step,
			U & container
		) const
		{
			Real const half_side = side / 2;
			
			FArray2D RT( 4, 4 );
			
			for ( Real ix = -half_side; ix <= half_side; ix += step ) {
				for ( Real iy = -half_side; iy <= half_side; iy += step ) {
					for ( Real iz = -half_side; iz <= half_side; iz += step ) {
						
						xyzVector const p = ( ix * x_axis_ ) + ( iy * y_axis_ ) + ( iz * z_axis_ ); // point to translate to

						// remember that FArrays are column-major:
						// | 0 4  8 12 |
						// | 1 5  9 13 |
						// | 2 6 10 14 |
						// | 3 7 11 15 |  ->  this last row is always | 0 0 0 1 |
						RT = 0;
						RT[ 0 ]  = 1;
						RT[ 5 ]  = 1;
						RT[ 10 ] = 1;
						RT[ 12 ] = p.x();
						RT[ 13 ] = p.y();
						RT[ 14 ] = p.z();
						RT[ 15 ] = 1;
						
						container.push_back( RT );
					}
				}
			}
		}
	
	
	private: // methods

		/// @brief transformation matrix from axis-angle pair, angle must be in radians, and FArray2D must be 4x4!
		/// @note  Rodrigues' formula
		inline
		void
		rotate(
			xyzVector const & axis,
			T const & angle,
			FArray2D & RT
		) const
		{
			using std::cos;
			using std::sin;
			
			assert( RT.size1() == 4 && RT.size2() == 4 );
			
			// remember that FArrays are column-major:
			// | 0 4  8 12 |
			// | 1 5  9 13 |
			// | 2 6 10 14 |
			// | 3 7 11 15 |  ->  this last row is always | 0 0 0 1 |
			RT[ 0 ]  = cos( angle ) + axis.x() * axis.x() * ( 1 - cos( angle ) );
			RT[ 1 ]  = axis.z() * sin( angle ) + axis.x() * axis.y() * ( 1 - cos( angle ) );
			RT[ 2 ]  = -axis.y() * sin( angle ) + axis.x() * axis.z() * ( 1 - cos( angle ) );
			RT[ 3 ]  = 0.0;
			RT[ 4 ]  = axis.x() * axis.y() * ( 1 - cos( angle ) ) - axis.z() * sin( angle );
			RT[ 5 ]  = cos( angle ) + axis.y() * axis.y() * ( 1 - cos( angle ) );
			RT[ 6 ]  = axis.x() * sin( angle ) + axis.y() * axis.z() * ( 1 - cos( angle ) );
			RT[ 7 ]  = 0.0;
			RT[ 8 ]  = axis.y() * sin( angle ) + axis.x() * axis.z() * ( 1 - cos( angle ) );
			RT[ 9 ]  = -axis.x() * sin( angle ) + axis.y() * axis.z() * ( 1 - cos( angle ) );
			RT[ 10 ] = cos( angle ) + axis.z() * axis.z() * ( 1 - cos( angle ) );
			RT[ 11 ] = 0.0;
			RT[ 12 ] = 0.0;
			RT[ 13 ] = 0.0;
			RT[ 14 ] = 0.0;
			RT[ 15 ] = 1.0;
		}
		
		/// @brief translation -- note that internal axis is always unit vector; RT must be 4x4!
		inline
		void
		translate(
			xyzVector const & axis,
			T const & distance,
			FArray2D & RT
		) const
		{
			xyzVector v( distance * axis );
			RT = 0;
			RT[ 0 ] = 1;
			RT[ 5 ] = 1;
			RT[ 10 ] = 1;
			RT[ 12 ] = v.x();
			RT[ 13 ] = v.y();
			RT[ 14 ] = v.z();
			RT[ 15 ] = 1;
		}
		
		/// @brief correct rotation to rotate at internal/local origin
		inline
		void
		correct_for_local_origin(
			FArray2D const & q,
			FArray2D & RT
		) const
		{
			// temporary storage
			FArray2D s( 4, 4 );
			
			// T2 * Q * T1 * p
			// where 'Q' is the rotation around the world origin (0, 0, 0)
			// 'T1' is the translation from the local origin -> world origin
			// 'T2' is the translation back from the world origin -> local origin
			// and 'p' would be the point we are looking to transform
			compose( translation_to_local_origin_, q, s );
			compose( s, translation_to_world_origin_, RT );
		}
		
		/// @brief compose two transformation matrices to yield result: s * f
		/// @brief so that f is the first transform applied and s is the second
		inline
		void
		compose(
			FArray2D const & s,
			FArray2D const & f,
			FArray2D & result
		) const
		{
			// matrix multiply 's' * 'f' to get result
			// remember that FArrays are column-major:
			// | 0 4  8 12 |
			// | 1 5  9 13 |
			// | 2 6 10 14 |
			// | 3 7 11 15 |  ->  this last row is always | 0 0 0 1 | so simplifications are performed below
			result[ 0 ]  = s[ 0 ]*f[ 0 ]  + s[ 4 ]*f[ 1 ]  + s[ 8 ] *f[ 2 ]; //  + s[ 12 ]*f[ 3 ];
			result[ 1 ]  = s[ 1 ]*f[ 0 ]  + s[ 5 ]*f[ 1 ]  + s[ 9 ] *f[ 2 ]; //  + s[ 13 ]*f[ 3 ];
			result[ 2 ]  = s[ 2 ]*f[ 0 ]  + s[ 6 ]*f[ 1 ]  + s[ 10 ]*f[ 2 ]; //  + s[ 14 ]*f[ 3 ];
		//	result[ 3 ]  = s[ 3 ]*f[ 0 ]  + s[ 7 ]*f[ 1 ]  + s[ 11 ]*f[ 2 ]  + s[ 15 ]*f[ 3 ]; // always 0
			result[ 3 ]  = 0.0;
			result[ 4 ]  = s[ 0 ]*f[ 4 ]  + s[ 4 ]*f[ 5 ]  + s[ 8 ] *f[ 6 ]; //  + s[ 12 ]*f[ 7 ];
			result[ 5 ]  = s[ 1 ]*f[ 4 ]  + s[ 5 ]*f[ 5 ]  + s[ 9 ] *f[ 6 ]; //  + s[ 13 ]*f[ 7 ];
			result[ 6 ]  = s[ 2 ]*f[ 4 ]  + s[ 6 ]*f[ 5 ]  + s[ 10 ]*f[ 6 ]; //  + s[ 14 ]*f[ 7 ];
		//	result[ 7 ]  = s[ 3 ]*f[ 4 ]  + s[ 7 ]*f[ 5 ]  + s[ 11 ]*f[ 6 ]  + s[ 15 ]*f[ 7 ]; // always 0
			result[ 7 ]  = 0.0;
			result[ 8 ]  = s[ 0 ]*f[ 8 ]  + s[ 4 ]*f[ 9 ]  + s[ 8 ] *f[ 10 ]; // + s[ 12 ]*f[ 11 ];
			result[ 9 ]  = s[ 1 ]*f[ 8 ]  + s[ 5 ]*f[ 9 ]  + s[ 9 ] *f[ 10 ]; // + s[ 13 ]*f[ 11 ];
			result[ 10 ] = s[ 2 ]*f[ 8 ]  + s[ 6 ]*f[ 9 ]  + s[ 10 ]*f[ 10 ]; // + s[ 14 ]*f[ 11 ];
		//	result[ 11 ] = s[ 3 ]*f[ 8 ]  + s[ 7 ]*f[ 9 ]  + s[ 11 ]*f[ 10 ] + s[ 15 ]*f[ 11 ]; // always 0
			result[ 11 ]  = 0.0;
			result[ 12 ] = s[ 0 ]*f[ 12 ] + s[ 4 ]*f[ 13 ] + s[ 8 ] *f[ 14 ] + s[ 12 ]; // *f[ 15 ];
			result[ 13 ] = s[ 1 ]*f[ 12 ] + s[ 5 ]*f[ 13 ] + s[ 9 ] *f[ 14 ] + s[ 13 ]; // *f[ 15 ];
			result[ 14 ] = s[ 2 ]*f[ 12 ] + s[ 6 ]*f[ 13 ] + s[ 10 ]*f[ 14 ] + s[ 14 ]; // *f[ 15 ];
		//	result[ 15 ] = s[ 3 ]*f[ 12 ] + s[ 7 ]*f[ 13 ] + s[ 11 ]*f[ 14 ] + s[ 15 ]*f[ 15 ]; // always 1
			result[ 15 ] = 1.0;
		}

		
	private: // initialize
	
		inline
		void
		initialize()
		{
			// fill translation_to_*origin_ arrays
			// remember that FArrays are column-major:
			// | 0 4  8 12 |
			// | 1 5  9 13 |
			// | 2 6 10 14 |
			// | 3 7 11 15 |  ->  this last row is always | 0 0 0 1 |
			
			translation_to_world_origin_.dimension( 4, 4 );
			translation_to_world_origin_ = 0;
			translation_to_world_origin_[ 0 ] = 1;
			translation_to_world_origin_[ 5 ] = 1;
			translation_to_world_origin_[ 10 ] = 1;
			translation_to_world_origin_[ 12 ] = -origin_.x();
			translation_to_world_origin_[ 13 ] = -origin_.y();
			translation_to_world_origin_[ 14 ] = -origin_.z();
			translation_to_world_origin_[ 15 ] = 1;
			
			translation_to_local_origin_.dimension( 4, 4 );
			translation_to_local_origin_ = 0;
			translation_to_local_origin_[ 0 ] = 1;
			translation_to_local_origin_[ 5 ] = 1;
			translation_to_local_origin_[ 10 ] = 1;
			translation_to_local_origin_[ 12 ] = origin_.x();
			translation_to_local_origin_[ 13 ] = origin_.y();
			translation_to_local_origin_[ 14 ] = origin_.z();
			translation_to_local_origin_[ 15 ] = 1;
		}
	

	private: // data
	
		xyzVector origin_;
		FArray2D translation_to_world_origin_;
		FArray2D translation_to_local_origin_;
	
		xyzVector x_axis_;
		xyzVector y_axis_;
		xyzVector z_axis_;
		
};


} // conformation
} // epigraft

#endif /*INCLUDED_epigraft_conformation_TransformGenerator_HH_*/
