// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   ConnectionResult.hh
/// @brief  Stores resulting tracking information from a PoseAssembly::connect call.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef INCLUDED_epigraft_design_ConnectionResult_HH_
#define INCLUDED_epigraft_design_ConnectionResult_HH_

// package headers
#include <epigraft/design/design_types.hh>
#include <epigraft/ResidueRange.hh>

// C++ headers
#include <iostream>


namespace epigraft {
namespace design {


/// @file   ConnectionResult.hh
/// @brief  Stores resulting tracking information from a PoseAssembly::connect call.
class ConnectionResult {

	public: // construct/destruct

		/// @brief default constructor
		ConnectionResult()
		{}

		/// @brief constructor
		ConnectionResult(
			Integer const & jump_offset,
			Integer const & residue_offset,
			Integer const & jump_start,
			Integer const & jump_stop,
			Integer const & jump_label,
			ResidueRange const & segment_range
		) : jump_offset_( jump_offset ),
		    residue_offset_( residue_offset ),
		    jump_start_( jump_start ),
		    jump_stop_( jump_stop ),
		    jump_label_( jump_label ),
		    segment_range_( segment_range )
		{}

		/// @brief copy constructor
		ConnectionResult(
			ConnectionResult const & c
		) : jump_offset_( c.jump_offset_ ),
		    residue_offset_( c.residue_offset_ ),
		    jump_start_( c.jump_start_ ),
		    jump_stop_( c.jump_stop_ ),
		    jump_label_( c.jump_label_ ),
		    segment_range_( c.segment_range_ )
		{}

		/// @brief default constructor
		~ConnectionResult()
		{}


	public: // copy assignment

		/// @brief copy assignment
		ConnectionResult &
		operator =( ConnectionResult const & c )
		{
			if ( this != &c ) {
				jump_offset_ = c.jump_offset_;
				residue_offset_ = c.residue_offset_;
				jump_start_ = c.jump_start_;
				jump_stop_ = c.jump_stop_;
				jump_label_ = c.jump_label_;
				segment_range_ = c.segment_range_;
			}
			return *this;
		}


	public: // member operator

		/// @brief '<' by comparing segment ranges
		bool
		operator <( ConnectionResult const & rvalue )
		{
			return segment_range_ < rvalue.segment_range_;
		}


	public: // accessors

		/// @brief jump offset used during connection
		Integer const &
		jump_offset() const
		{
			return jump_offset_;
		}

		/// @brief residue offset used during connection
		Integer const &
		residue_offset() const
		{
			return residue_offset_;
		}

		/// @brief start residue of jump connection
		Integer const &
		jump_start() const
		{
			return jump_start_;
		}

		/// @brief stop residue of jump connection
		Integer const &
		jump_stop() const
		{
			return jump_stop_;
		}

		/// @brief label of jump connection
		Integer const &
		jump_label() const
		{
			return jump_label_;
		}

		/// @brief range of original Pose in resulting connected Pose
		ResidueRange const &
		segment_range() const
		{
			return segment_range_;
		}


	public: // update

		/// @brief update internal numbers due to disconnection of a Pose
		void
		disconnection_update(
			ConnectionResult const & d
		)
		{
			if ( segment_range_ > d.segment_range_ ) {
				residue_offset_ -= d.segment_range_.length();

				// check to see if jump start is within deleted region, if so this is
				// undefined behavior so set to -1
				if ( d.segment_range_.contains( jump_start_ ) ) { // undefined
					jump_start_ = -1;
					std::cerr << "ConnectionResult WARNING: you are removing a Pose that had other Poses connected to it.  This is undefined -- something will crash!." << std::endl;
				} // note that jump starts cannot be to the right of this connection's range due to the way connections are created

				// if d's segment range is not directly adjacent, then the jump start must be shifted as well
				if ( segment_range_.begin() != d.segment_range_.end() + 1 ) {
					jump_start_ -= d.segment_range_.length();
				}

				jump_stop_ -= d.segment_range_.length();
				segment_range_ = segment_range_ - d.segment_range_.length();

				// finally, update jump label
				if ( jump_label_ > d.jump_label() ) {
					--jump_label_;
				}
			}
		}


		/// @brief update internal numbers due to shift in the number of residues (insert/delete)
		/// @brief to the left of the Pose represented by this connection
		void
		shift(
			Integer const & shift_length
		)
		{
			residue_offset_ += shift_length;
			jump_start_ += shift_length;
			jump_stop_ += shift_length;
			segment_range_ = segment_range_ + shift_length;
		}


	private: // data

		/// @brief jump label offset used during connection
		Integer jump_offset_;

		/// @brief residue offset used during connection
		Integer residue_offset_;

		/// @brief start residue of jump connection ("left pose")
		Integer jump_start_;

		/// @brief stop residue of jump connection ("right pose")
		Integer jump_stop_;

		/// @brief label of jump connection
		Integer jump_label_;

		/// @brief range of original Pose in resulting connected Pose
		ResidueRange segment_range_;

};

} // namespace design
} // namespace epigraft


#endif /*INCLUDED_epigraft_design_ConnectionResult_HH_*/
