// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   DesignFileExport.cc
/// @brief  For export of design input files such as resfile and blueprint.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

// unit headers
#include <epigraft/design/DesignFileExport.hh>

// rosetta headers
#include <param_aa.h>

// c++ headers
#include <set>
#include <sstream>
#include <string>


namespace epigraft {
namespace design {

/// @brief return copy of resfile header
std::string
DesignFileExport::resfile_header()
{
	std::ostringstream ss;

	ss << " This file specifies which residues will be varied\n";
	ss << "                                                  \n";
	ss << " Column   2:  Chain                               \n";
	ss << " Column   4-7:  sequential residue number         \n";
	ss << " Column   9-12:  pdb residue number               \n";
	ss << " Column  14-18: id  (described below)             \n";
	ss << " Column  20-40: amino acids to be used            \n";
	ss << "                                                  \n";
	ss << " NATAA  => use native amino acid                  \n";
	ss << " ALLAA  => all amino acids                        \n";
	ss << " NATRO  => native amino acid and rotamer          \n";
	ss << " PIKAA  => select inividual amino acids           \n";
	ss << " POLAR  => polar amino acids                      \n";
	ss << " APOLA  => apolar amino acids                     \n";
	ss << "                                                  \n";
	ss << " The following demo lines are in the proper format\n";
	ss << "                                                  \n";
	ss << " A    1    3 ALLAA                                \n";
	ss << " A    2    4 ALLAA                                \n";
	ss << " A    3    6 NATRO                                \n";
	ss << " A    4    7 NATAA                                \n";
	ss << " B    5    1 PIKAA  DFLM                          \n";
	ss << " B    6    2 PIKAA  HIL                           \n";
	ss << " B    7    3 POLAR                                \n";
	ss << " -------------------------------------------------\n";
	ss << " start\n";

	return ss.str();
}


/// @brief identity action -> string
std::string
DesignFileExport::identity_action_str(
	IdentityAction const & action
) const
{

	switch ( action ) {
		case DEFAULT_IDENTITY_ACTION:
			return identity_action_str( default_identity_action_ );
		case NATRO:
			return "NATRO";
		case ALLAA:
			return "ALLAA";
		case NATAA:
			return "NATAA";
		case PIKAA:
			return "PIKAA";
		case NOTAA:
			return "NOTAA";
		case POLAR:
			return "POLAR";
		case APOLA:
			return "APOLA";
		default:
			return identity_action_str( default_identity_action_ );
	}
}


/// @brief structure action -> string
std::string
DesignFileExport::structure_action_str(
	StructureAction const & action
) const
{
	switch ( action ) {
		case NO_STRUCTURE_ACTION:
			return ".";
		case H:
			return "H";
		case L:
			return "L";
		case E:
			return "E";
		case D:
			return "D";
		default:
			return ".";
	}
}


/// @brief identity info -> string
std::string
DesignFileExport::identity_info_str(
	IdentitySet const & info,
	bool const & use_space_between_entries
) const
{
	std::ostringstream ss;

	if ( !info.empty() ) {
		// handle first entry
		IdentitySet::const_iterator i = info.begin();
		ss << *i;
		++i;

		// handle rest of entries
		for ( IdentitySet::const_iterator ie = info.end(); i != ie; ++i ) {
			if ( use_space_between_entries ) {
				ss << ' ';
			}
			ss << *i;
		}

	}

	return ss.str();
}


}
}