// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   DesignFileExport.hh
/// @brief  For export of design input files such as resfile and blueprint.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef INCLUDED_epigraft_design_DesignFileExport_HH_
#define INCLUDED_epigraft_design_DesignFileExport_HH_

// package headers
#include <epigraft/design/design_types.hh>

// rosetta headers
#include <pose.h>

// utility headers
#include <utility/vector1.hh>

// ObjexxFCL headers
#include <ObjexxFCL/string.functions.hh>

// C++ headers
#include <fstream>
#include <map>
#include <string>
#include <set>

namespace epigraft {
namespace design {

/// @brief Handles export of design input files such as resfile and blueprint.
class DesignFileExport {

	public: // typedef

		/// @brief stores allowed/disallowed amino acid types for PIKAA/NOTAA actions
		/// @details single character naming convention
		typedef std::set< char > IdentitySet;


	public: // enum

		/// @brief residue amino acid identity actions
		enum IdentityAction {
			DEFAULT_IDENTITY_ACTION,
			NATRO,
			ALLAA,
			NATAA,
			PIKAA,
			NOTAA, // currently incompatible with blueprint?
			POLAR,
			APOLA
		};

		/// @brief residue structure actions
		enum StructureAction {
			NO_STRUCTURE_ACTION,
			H,
			L,
			E,
			D
		};


	private: // internal structs

		struct ResidueSpecification {
			/// @brief default constructor
			ResidueSpecification()
			: identity_action( DEFAULT_IDENTITY_ACTION ),
			  structure_action( NO_STRUCTURE_ACTION )
			{}

			IdentityAction identity_action;
			StructureAction structure_action;
			IdentitySet identity_info;
		};


	private: // typedefs

		typedef utility::vector1< ResidueSpecification > Specifications;


	public: // construct/destruct

		/// @brief default constructor
		inline
		DesignFileExport()
		: default_identity_action_( NATRO )
		{}

		/// @brief number of residues constructor
		inline
		DesignFileExport(
			Size const & nres,
			IdentityAction const & default_identity_action = NATRO
		) : default_identity_action_( default_identity_action )
		{
			reset( nres );
		}

		/// @brief destructor
		inline
		~DesignFileExport()
		{}


	public: // assignment

		/// @brief copy assignment
		inline
		DesignFileExport &
		operator =( DesignFileExport const & dfe )
		{
			if ( this != &dfe ) {
				default_identity_action_ = dfe.default_identity_action_;
				specs_ = dfe.specs_;
			}
			return *this;
		}


	public: // accessors

		/// @brief get total number of residues
		inline
		Size
		nres() const
		{
			return specs_.size();
		}

		/// @brief get default identity action
		inline
		IdentityAction const &
		default_identity_action() const
		{
			return default_identity_action_;
		}

		/// @brief set total number of residues -- this will clear all actions and identities
		inline
		void
		set_nres(
			Size const & nres
		)
		{
			reset( nres );
		}

		/// @brief set default identity action
		inline
		void
		set_default_identity_action(
			IdentityAction const & action
		)
		{
			if ( action != DEFAULT_IDENTITY_ACTION ) { // prevents infinite loop when looking up identity action string
				default_identity_action_ = action;
			}
		}


	public: // identity setup

		/// @brief get identity action for residue
		inline
		IdentityAction const &
		identity_action(
			Size const & res
		) const
		{
			return specs_[ res ].identity_action;
		}

		/// @brief set identity action for residue
		inline
		void
		set_identity_action(
			Size const & res,
			IdentityAction const & action
		)
		{
			specs_[ res ].identity_action = action;
		}

		/// @brief get identity info for residue
		inline
		IdentitySet const &
		identity_info(
			Size const & res
		) const
		{
			return specs_[ res ].identity_info;
		}

		/// @brief add identity info for residue
		inline
		void
		add_identity_info(
			Size const & res,
			char const & one_letter_code
		)
		{
			specs_[ res ].identity_info.insert( one_letter_code );
		}

		/// @brief add identity info for residue
		template< typename OneLetterIterator >
		inline
		void
		add_identity_info(
			Size const & res,
			OneLetterIterator begin,
			OneLetterIterator end
		)
		{
			specs_[ res ].identity_info.insert( begin, end );
		}


	public: // structure setup

		/// @brief get structure action for residue
		inline
		StructureAction const &
		structure_action(
			Size const & res
		) const
		{
			return specs_[ res ].structure_action;
		}

		/// @brief set structure action for residue
		inline
		void
		set_structure_action(
			Size const & res,
			StructureAction const & action
		)
		{
			specs_[ res ].structure_action = action;
		}

		/// @brief set structure action for residue
		inline
		void
		set_structure_action(
			Size const & res,
			char const & ss_type
		)
		{
			switch ( ss_type ) {
				case 'H':
					specs_[ res ].structure_action = H;
					break;
				case 'L':
					specs_[ res ].structure_action = L;
					break;
				case 'E':
					specs_[ res ].structure_action = E;
					break;
				case 'D':
					specs_[ res ].structure_action = D;
					break;
				default:
					specs_[ res ].structure_action = NO_STRUCTURE_ACTION;
					break;
			}
		}


	public: // management

		/// @brief clear identity actions
		inline
		void
		clear_identity_actions()
		{
			for ( Specifications::iterator i = begin(), ie = end(); i != ie; ++i ) {
				i->identity_action = DEFAULT_IDENTITY_ACTION;
			}
		}

		/// @brief clear structure actions
		inline
		void
		clear_structure_actions()
		{
			for ( Specifications::iterator i = begin(), ie = end(); i != ie; ++i ) {
				i->structure_action = NO_STRUCTURE_ACTION;
			}
		}

		/// @brief clear identity info
		inline
		void
		clear_identity_info()
		{
			for ( Specifications::iterator i = begin(), ie = end(); i != ie; ++i ) {
				i->identity_info.clear();
			}
		}

		/// @brief clear everything
		inline
		void
		reset(
			Size const & nres = 0
		)
		{
			Size const size = nres ? nres : specs_.size();
			specs_.clear();
			specs_.resize( size );
		}


	public: // export

		/// @brief output resfile
		inline
		void
		resfile(
			std::string const & filename,
			Pose const & pose
		)
		{
			std::ofstream out( filename.c_str() );
			resfile( out, pose );
			out.close();
		}

		/// @brief output resfile
		template< typename Stream >
		inline
		void
		resfile(
			Stream & stream,
			Pose const & pose
		)
		{
			using ObjexxFCL::string_of;

			assert( nres() == static_cast< Size >( pose.total_residue() ) );

			// create rosetta numbering to pdb numbering and residue to chain
			std::map< Size, std::string > rosetta2pdb;
			std::map< Size, char > res2chain;
			if ( pose.pdb_info().use_pdb_numbering() ) {

				for ( Size i = 1, ie = specs_.size(); i <= ie; ++i ) {
					if ( pose.pdb_info().pdb_insert_let( i ) != ' ' ) {
						rosetta2pdb[ i ] = string_of( pose.pdb_info().pdb_res_num( i ) ) + pose.pdb_info().pdb_insert_let( i );
					} else {
						rosetta2pdb[ i ] = string_of( pose.pdb_info().pdb_res_num( i ) );
					}
					res2chain[ i ] = pose.pdb_info().res_chain( i );
				}

			}

			resfile( stream, rosetta2pdb, res2chain );
		}

		/// @brief output resfile to file specified by filename
		/// @param[in] filename  filename of output file
		/// @param[in] rosetta2pdb  map allowed to be empty, optional rosetta numbering to pdb numbering (string since there may be insertion code)
		inline
		void
		resfile(
			std::string const & filename,
			std::map< Size, std::string > const & rosetta2pdb,
			std::map< Size, char > const & res2chain
		) const
		{
			std::ofstream out( filename.c_str() );
			resfile( out, rosetta2pdb, res2chain );
			out.close();
		}

		/// @brief output resfile to stream
		/// @param[in] stream  output stream
		/// @param[in] rosetta2pdb  map allowed to be empty, optional rosetta numbering to pdb numbering (string since there may be insertion code)
		/// @param[in] res2chain  map allowed to be empty, optional rosetta numbering to chain id
		template< typename Stream >
		inline
		void
		resfile(
			Stream & stream,
			std::map< Size, std::string > const & rosetta2pdb,
			std::map< Size, char > const & res2chain
		) const
		{
			using ObjexxFCL::string_of;
			using ObjexxFCL::lpadded;

			// setup rosetta2pdb, copy since we may modify
			std::map< Size, std::string > r2pdb = rosetta2pdb;
			if ( r2pdb.empty() ) {
				for ( Size i = 1, ie = specs_.size(); i <= ie; ++i ) {
					r2pdb[ i ] = string_of( i );
				}
			}

			// setup res2chain, copy since we may modify
			std::map< Size, char > r2chain = res2chain;
			if ( r2chain.empty() ) {
				for ( Size i = 1, ie = specs_.size(); i <= ie; ++i ) {
					r2chain[ i ] = '0';
				}
			}

			// output header first
			stream << resfile_header();

			// now run through residue specifications and print
			for ( Size i = 1, ie = specs_.size(); i <= ie; ++i ) {
				// grab data
				ResidueSpecification const & rs = specs_[ i ];
				std::string const resid = string_of( i );
				std::string const & pdb_resid = r2pdb.find( i )->second;
				char const & chain = r2chain.find( i )->second;

				stream << ' ' << chain;
				stream << ' ' << lpadded( resid, 4 );
				stream << lpadded( pdb_resid, 5 );
				stream << ' ' << identity_action_str( rs.identity_action );
				if ( rs.identity_action == PIKAA || rs.identity_action == NOTAA ) {
					stream << "  " << identity_info_str( rs.identity_info );
				}
				stream << '\n';
			}
		}

		/// @brief output blueprint
		inline
		void
		blueprint(
			std::string const & filename,
			Pose const & pose
		)
		{
			std::ofstream out( filename.c_str() );
			blueprint( out, pose );
			out.close();
		}

		/// @brief output blueprint
		template< typename Stream >
		inline
		void
		blueprint(
			Stream & stream,
			Pose const & pose
		)
		{
			using ObjexxFCL::string_of;

			assert( nres() == static_cast< Size >( pose.total_residue() ) );

			// create rosetta numbering to original amino acid identity
			std::map< Size, char > res2aa;
			for ( Size i = 1, ie = specs_.size(); i <= ie; ++i ) {
				res2aa[ i ] = param_aa::aa_name1( pose.res( i ) );
			}

			blueprint( stream, res2aa );
		}

		/// @brief output blueprint to file specified by filename
		/// @param[in] res2aa  rosetta residue number to original amino acid (one letter convention)
		inline
		void
		blueprint(
			std::string const & filename,
			std::map< Size, char > const & res2aa
		) const
		{
			std::ofstream out( filename.c_str() );
			blueprint( out, res2aa );
			out.close();
		}

		/// @brief output blueprint to stream
		/// @param[in] res2aa  rosetta residue number to original amino acid (one letter convention)
		template< typename Stream >
		inline
		void
		blueprint(
			Stream & stream,
			std::map< Size, char > const & res2aa
		) const
		{
			// blueprint currently does not support 'NATAA' so first
			// run through identity actions and print error/refuse to output if
			// found
			for ( Size i = 1, ie = specs_.size(); i <= ie; ++i ) {
				// grab data
				ResidueSpecification const & rs = specs_[ i ];

				if ( rs.identity_action == NATAA ) {
					std::cerr << "ERROR: DesignFileExport skipping blueprint output due to presence of unsupported 'NATAA'." << std::endl;
					return;
				}
			}

			// run through residue specifications and print
			for ( Size i = 1, ie = specs_.size(); i <= ie; ++i ) {
				// grab data
				ResidueSpecification const & rs = specs_[ i ];

				stream << i;
				stream << ' ' << res2aa.find( i )->second;
				stream << ' ' << structure_action_str( rs.structure_action );
				stream << ' ' << identity_action_str( rs.identity_action );
				stream << ' ' << identity_info_str( rs.identity_info, true ); // boolean: use_space_between_entries
				stream << '\n';
			}
		}

		/// @brief return copy of resfile header
		static
		std::string
		resfile_header();


	private: // traversal

		/// @brief return iterator that points to beginning of residue specifications
		inline
		Specifications::const_iterator
		begin() const
		{
			return specs_.begin();
		}

		/// @brief return iterator that points to end of residue specifications
		inline
		Specifications::const_iterator
		end() const
		{
			return specs_.end();
		}

		/// @brief return iterator that points to beginning of residue specifications
		inline
		Specifications::iterator
		begin()
		{
			return specs_.begin();
		}

		/// @brief return iterator that points to end of residue specifications
		inline
		Specifications::iterator
		end()
		{
			return specs_.end();
		}

		/// @brief return reverse iterator that points to last element of residue specifications
		inline
		Specifications::const_reverse_iterator
		rbegin() const
		{
			return specs_.rbegin();
		}

		/// @brief return reverse iterator that points to last element of residue specifications
		inline
		Specifications::reverse_iterator
		rbegin()
		{
			return specs_.rbegin();
		}

		/// @brief return reverse iterator that points beyond first element of residue specifications
		inline
		Specifications::const_reverse_iterator
		rend() const
		{
			return specs_.rend();
		}

		/// @brief return reverse iterator that points beyond first element of residue specifications
		inline
		Specifications::reverse_iterator
		rend()
		{
			return specs_.rend();
		}


	private: // action -> string conversion

		/// @brief identity action -> string
		std::string
		identity_action_str(
			IdentityAction const & action
		) const;

		/// @brief structure action -> string
		std::string
		structure_action_str(
			StructureAction const & action
		) const;

		/// @brief identity info -> string
		std::string
		identity_info_str(
			IdentitySet const & info,
			bool const & use_space_between_entries = false
		) const;


	private: // data

		IdentityAction default_identity_action_;
		Specifications specs_;

};

}
}

#endif /*INCLUDED_epigraft_design_DesignFileExport_HH_*/
