// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   ESBundle.hh
/// @brief  epitope scaffold tracking for multigraft top-level protocol
/// @author Yih-En Andrew Ban (yab@u.washington.edu)


#ifndef INCLUDED_epigraft_design_ESBundle_HH_
#define INCLUDED_epigraft_design_ESBundle_HH_

// package headers
#include <epigraft/design/design_types.hh>
#include <epigraft/design/EpitopeScaffold.hh>
#include <epigraft/design/multigraft_checkpoint.hh>

// rosetta headers
#include <pose.h>

// utility headers
#include <utility/io/izstream.hh>

// ObjexxFCL headers
#include <ObjexxFCL/string.functions.hh>

// C++ headers
#include <sstream>
#include <string>
#include <vector>


namespace epigraft {
namespace design {


/// @brief struct for tracking epitope scaffolds as
/// @brief they move through the protocol, makes life easier instead of
/// @brief keeping maps
/// @note this was originally internal to multigraft.cc and has grown a bit
/// @note beyond it's original intent; at some point a re-wrapping into a
/// @note true class should be performed
/// @warning scaffolds stored here are NOT deleted off the heap when
/// @warning the ESBundle is destructed.  You MUST call destroy_contents()
/// @warning for this to happen.
struct ESBundle {

	/// @brief default constructor
	ESBundle()
	: closure_id( -1 ),
	  design_id( -1 ),
	  closed_es( NULL ),
	  designed_es( NULL ),
	  refined_es( NULL ),
	  qc_es( NULL )
	{}

	/// @brief constructor
	ESBundle(
		Integer const & c_id,
		Integer const & d_id,
		EpitopeScaffold * c_es = NULL,
		EpitopeScaffold * d_es = NULL,
		EpitopeScaffold * rAb_es = NULL,
		EpitopeScaffold * q_es = NULL
	) : closure_id( c_id ),
	    design_id( d_id ),
	    closed_es( c_es ),
	    designed_es( d_es ),
	    refined_es( rAb_es ),
	    qc_es( q_es )
	{}

	/// @brief copy constructor
	ESBundle(
		ESBundle const & b
	) : closure_id( b.closure_id ),
	    design_id( b.design_id ),
	    closed_es( b.closed_es ),
	    designed_es( b.designed_es ),
	    refined_es( b.refined_es ),
	    qc_es( b.qc_es)
	{}

	/// @brief default destructor
	~ESBundle()
	{}

	/// @brief copy assignment
	ESBundle &
	operator =(
		ESBundle const & b
	)
	{
		if ( this != &b ) {
			closure_id = b.closure_id;
			design_id = b.design_id;
			closed_es = b.closed_es;
			designed_es = b.designed_es;
			refined_es = b.refined_es;
			qc_es = b.qc_es;
		}
		return *this;
	}

	/// @brief destroy everything on the heap
	void
	destroy_contents()
	{
		if ( closed_es != NULL ) {
			delete closed_es;
		}
		if ( designed_es != NULL ) {
			delete designed_es;
		}
		if ( refined_es != NULL ) {
			delete refined_es;
		}
		if ( qc_es != NULL ) {
			delete qc_es;
		}
	}

	/// @brief checkpoint by writing checkpoint to line and saving pdbs
	void
	checkpoint_save(
		std::string const & pdb_prefix
	) const
	{
		if ( closed_es ) {
			closed_es->pose().dump_pdb( pdb_prefix + ".closed.pdb.gz" );
			closed_es->save_loop_checkpoint( pdb_prefix + ".closed.loops" );
		}

		if ( designed_es ) {
			designed_es->pose().dump_pdb( pdb_prefix + ".designed.pdb.gz" );
			designed_es->save_loop_checkpoint( pdb_prefix + ".designed.loops" );
		}

		if ( refined_es ) {
			refined_es->pose().dump_pdb( pdb_prefix + ".refined.pdb.gz" );
			refined_es->save_loop_checkpoint( pdb_prefix + ".refined.loops" );
		}

		if ( qc_es ) {
			qc_es->pose().dump_pdb( pdb_prefix + ".qc.pdb.gz" );
			qc_es->save_loop_checkpoint( pdb_prefix + ".qc.loops" );
		}
	}

	/// @brief checkpoint load
	void
	checkpoint_load(
		std::string const & pdb_prefix,
		Pose const & original_antibody,
		EpitopeScaffold const & base_es,
		bool const & also_replace_archive = false
	)
	{
		utility::io::izstream in;
		std::string filename;

		filename = pdb_prefix + ".closed.pdb.gz";
		in.open( filename );
		if ( !in.fail() ) {
			in.close();
			closed_es = new EpitopeScaffold( base_es );
			reload_epitope_scaffold( filename, original_antibody, *closed_es, also_replace_archive );
			closed_es->load_loop_checkpoint( pdb_prefix + ".closed.loops" );
		} else {
			in.close();
		}

		filename = pdb_prefix + ".designed.pdb.gz";
		in.open( filename );
		if ( !in.fail() ) {
			in.close();
			designed_es = new EpitopeScaffold( base_es );
			reload_epitope_scaffold( filename, original_antibody, *designed_es, also_replace_archive );
			designed_es->load_loop_checkpoint( pdb_prefix + ".designed.loops" );

			// presumably this scaffold has gone through design already
			designed_es->connect_Ab();
			designed_es->refresh_cached_design_positions();
			designed_es->disconnect_Ab();
		} else {
			in.close();
		}

		filename = pdb_prefix + ".refined.pdb.gz";
		in.open( filename );
		if ( !in.fail() ) {
			in.close();
			refined_es = new EpitopeScaffold( base_es );
			reload_epitope_scaffold( filename, original_antibody, *refined_es, also_replace_archive );
			refined_es->load_loop_checkpoint( pdb_prefix + ".refined.loops" );

			// presumably this scaffold has gone through design already
			refined_es->connect_Ab();
			refined_es->refresh_cached_design_positions();
			refined_es->disconnect_Ab();
		} else {
			in.close();
		}

		filename = pdb_prefix + ".qc.pdb.gz";
		in.open( filename );
		if ( !in.fail() ) {
			in.close();
			qc_es = new EpitopeScaffold( base_es );
			reload_epitope_scaffold( filename, original_antibody, *qc_es, also_replace_archive );
			qc_es->load_loop_checkpoint( pdb_prefix + ".qc.loops" );

			// presumably this scaffold has gone through design already
			qc_es->connect_Ab();
			qc_es->refresh_cached_design_positions();
			qc_es->disconnect_Ab();
		} else {
			in.close();
		}

	}


	/// @brief convenience scoring (es_score_string() for design/refine/qc)
	std::string
	score_string() const
	{
		using ObjexxFCL::fixed_string_of;
		using ObjexxFCL::lpadded;

		std::vector< Real > scores;
		std::ostringstream ss;

		// headers
		    ss << "#                              es_only       total       ddG\n";

		if ( designed_es ) {
			scores = es_score_vector( designed_es );
			ss << "# * Designed ES          :";
			for ( Size i = 0, ie = scores.size(); i < ie; ++i ) {
				ss << "  " << lpadded( fixed_string_of( scores[ i ], 0, 3 ), 10 );
			}
			ss << '\n';
		}
		if ( refined_es ) {
			scores = es_score_vector( refined_es );
			ss << "# * Refined ES (with Ab) :";
			for ( Size i = 0, ie = scores.size(); i < ie; ++i ) {
				ss << "  " << lpadded( fixed_string_of( scores[ i ], 0, 3 ), 10 );
			}
			ss << '\n';
		}
		if ( qc_es ) {
			scores = es_score_vector( qc_es );
			ss << "# * QC ES (without Ab)   :";
			for ( Size i = 0, ie = scores.size(); i < ie; ++i ) {
				ss << "  " << lpadded( fixed_string_of( scores[ i ], 0, 3 ), 10 );
			}
			ss << '\n';
		}

		return ss.str();
	}


	/// @brief convenience scoring: scaffold, total (w/Ab), and ddG
	/// @details w/ antibody scores are only present if Ab is connected
	std::vector< Real >
	es_score_vector(
		EpitopeScaffold const * es
	) const
	{
		std::vector< Real > scores;

		// compute scores
		scores.push_back( es->scaffold_fullatom_score_no_breaks() );

		if ( es->Ab_is_connected() ) {
			scores.push_back( es->fullatom_score_no_breaks() );
			scores.push_back( es->ddG() );
		}

		return scores;
	}


	/// @brief convenience rama scoring (closed->design->refined->qc)
	/// @warning this assume closed/design/refined/qc scaffolds are derived from
	/// @warning each other
	std::string
	closure_and_rama_string() const
	{
		using ObjexxFCL::fixed_string_of;
		using ObjexxFCL::lpadded;
		using ObjexxFCL::rpadded;
		using ObjexxFCL::string_of;

		// grab only epitope scaffolds that exist
		std::vector< EpitopeScaffold * > e_es;
		std::vector< std::string > tags;
		if ( closed_es ) {
			e_es.push_back( closed_es );
			tags.push_back( "Closed  " );
		}
		if ( designed_es ) {
			e_es.push_back( designed_es );
			tags.push_back( "Designed" );
		}
		if ( refined_es ) {
			e_es.push_back( refined_es );
			tags.push_back( "Refined " );
		}
		if ( qc_es ) {
			e_es.push_back( qc_es );
			tags.push_back( "QC      " );
		}

		if ( e_es.size() == 0 ) { // no epitope scaffolds
			return std::string();
		}

		std::ostringstream ss;

		// grab components, we use position 0 in the list as the "anchor" es to derive information
		std::set< ResidueRange > components = e_es[ 0 ]->all_components();
		for ( std::set< ResidueRange >::const_iterator l = components.begin(), le = components.end(); l != le; ++l ) {
			ResidueRange const & component = *l;

			// print epitope component range
			ss << "#\n";
			ss << "#   epitope native range = " << e_es[ 0 ]->new_to_native_epitope( component ).to_string() << "   new range = " << component.to_string() << "\n";

			// anchor es: get copy of closures for this component
			std::set< LoopClosureInfo > closures = e_es[ 0 ]->component_closures_to_attempt( component );

			// print closure info for each available epitope scaffold
			for ( std::set< LoopClosureInfo >::const_iterator c = closures.begin(), ce = closures.end(); c != ce; ++c ) {

				// cache closures once
				std::vector< LoopClosureInfo > cached_cl;
				for ( Size i = 0, ie = e_es.size(); i < ie; ++i ) {
					cached_cl.push_back( e_es[ i ]->closure_from_closure( *c ) );
				}

				 // overall info first
				for ( Size i = 0, ie = e_es.size(); i < ie; ++i ) {
					ss << "#      " << tags[ i ] << ":   " <<  e_es[ i ]->closed_during_trajectory( cached_cl[ i ] )  << " | " << cached_cl[ i ].to_string() << '\n';
				}

				ss << "#\n";

				// rama next
				ss << "#         Rama per residue:\n";
				ss << "#         " << "ss " << "res";
				for ( Size i = 0, ie = tags.size(); i < ie; ++i ) {
					ss << "    " << tags[ i ] << " ";
				}
				ss << '\n';
				for ( std::set< Integer >::const_iterator r = c->moveable_residues().begin(), re = c->moveable_residues().end(); r != re; ++r ) {
					ss << "#         " << e_es[ 0 ]->pose().secstruct( *r ) << "  " << lpadded( string_of( *r ), 3 );
					for ( Size j = 0, je = cached_cl.size(); j < je; ++j ) {
						ss << "    " << rpadded( fixed_string_of( cached_cl[ j ].rama_of_residue( *r ), 0, 3 ), 9 );
					}
					ss << '\n';
				}
			}

		}


		return ss.str();
	}


	// id counter for output tracking
	Integer closure_id;
	Integer design_id;

	// closure attempt Poses
	EpitopeScaffold * closed_es;

	// design attempt Poses
	EpitopeScaffold * designed_es;
	EpitopeScaffold * refined_es;

	// QC attempt Poses
	EpitopeScaffold * qc_es;
};

}
}

#endif /*INCLUDED_epigraft_design_multigraft_ESBundle_HH_*/
