// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   EpitopeScaffold.cc
/// @brief  Class encapsulating epitope-scaffold data and operations.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)
/// @author Bill Schief (schief@u.washington.edu)
/// @author Possu Huang (possu@u.washington.edu)
/// @author Bruno Correia (bcorreia@u.washinton.edu)
/// @author Vanita Sood (vanitas@u.washinton.edu)
/// @author Chris Carrico (ccarrico@gmail.com)

// unit headers
#include <epigraft/design/EpitopeScaffold.hh>

// package headers
#include <epigraft/design/design_constants.hh>
#include <epigraft/AtomPoint.hh>
#include <epigraft/epigraft_functions.hh>
#include <epigraft/conformation/DihedralInfo.hh>
#include <epigraft/design/ConnectionResult.hh>
#include <epigraft/design/DesignFileExport.hh>
#include <epigraft/design/LoopClosureInfo.hh>
#include <epigraft/design/PoseAssembly.hh>
#include <epigraft/design/design_functions.hh>
#include <epigraft/design/loop_functions.hh>
#include <epigraft/match/MatchComponent.hh>
#include <epigraft/match/MatchResult.hh>
#include <epigraft/match/align/AlignmentSystem.hh>
#include <epigraft/match/align/C2N_C_Align.hh>
#include <epigraft/match/align/C2N_CA_Align.hh>
#include <epigraft/match/align/C2N_N_Align.hh>
#include <epigraft/match/align/N2C_C_Align.hh>
#include <epigraft/match/align/N2C_CA_Align.hh>
#include <epigraft/match/align/N2C_N_Align.hh>

// rootstock headers
#include <rootstock/BoundingBox.hh>
#include <rootstock/Octree.hh>

// rosetta headers
#include <aa_name_conversion.h>
#include <cst_set.h>
#include <disulfides.h>
#include <fold_tree.h>
#include <InteractionGraphBase.h>
#include <jump_classes.h>
#include <minimize.h>
#include <misc.h>
#include <PackerTask.h>
#include <param.h>
#include <param_aa.h>
#include <pose.h>
#include <pose_constraints.h>
#include <pose_param.h>
#include <random_numbers.h>
#include <RotamerSet.h>
#include <score.h>
#include <score_data.h>
#include <score_ns.h>

// numeric headers
#include <numeric/xyzVector.hh>

// utility headers
#include <utility/vector1.hh>

// ObjexxFCL headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/string.functions.hh>

// C++ headers
#include <cmath>
#include <fstream>
#include <set>
#include <sstream>
#include <string>
#include <utility>


namespace epigraft {
namespace design {


/// @brief set Ab, saves a local copy of Ab, this will DISCONNECT any currently connected Ab
/// @warning remember to set transformation matrix to identity prior to this if you don't want
/// @warning antibody to be reoriented upon connection
/// @note  Jump connection from epitope-scaffold to Ab is currently from middle residue of primary component on
/// @note  epitope-scaffold to middle residue of Ab.
void
EpitopeScaffold::set_Ab(
	Pose const & Ab,
	bool const & connect
)
{
	// remove any prior connected Ab first
	if ( Ab_is_connected_ ) {
		disconnect_Ab();
	}

	// first copy Ab
	archived_Ab_ = Ab;

	// do connection if requested
	if ( connect ) {
		connect_Ab();
	}
}


/// @brief connect (archived) Ab, assume Ab has already been set via prior set_Ab call
bool
EpitopeScaffold::connect_Ab(
	bool const & only_connect_to_primary
)
{
	using numeric::xyzVector;

	if ( archived_Ab_.total_residue() != 0 && !Ab_is_connected_ ) {
		Pose aligned_Ab;
		aligned_Ab = archived_Ab_;
		aligned_Ab.transform_GL( Ab_transform_ );

		// new Pose
		Pose es_Ab;

		// gather all moveable residues
		std::set< Integer > moveable_residues;
		for ( std::set< LoopClosureInfo >::const_iterator c = closures_to_attempt_.begin(), ce = closures_to_attempt_.end(); c != ce; ++c ) {
			moveable_residues.insert( c->moveable_residues().begin(), c->moveable_residues().end() );
		}

		// connect, for now we connect two residues of Ab and epitope that are closest together by CA distance
		FArray3D_float const & es_Eposition = epitope_scaffold_.Eposition();
		FArray3D_float const & Ab_Eposition = aligned_Ab.Eposition();
		Integer closest_on_es = -1;
		Integer closest_on_Ab = -1;
		Real closest_distance_squared = DESIGN_INFINITY;

		// find residue on Ab closest to middle of specified component(s)
		std::set< ResidueRange > components;
		components.insert( primary_component() );
		if ( !only_connect_to_primary ) {
			std::set< ResidueRange > secondaries = secondary_components();

			// only add those secondary components that are not being closed as single break
			for ( std::set< ResidueRange >::const_iterator c = secondaries.begin(), ce = secondaries.end(); c != ce; ++c ) {
				ResidueRange const & component = *c;
				if ( !graft_info_.closing_as_single_break( epitope_to_old_gap( component ) ) ) {
					components.insert( component );
				}
			}
		}

		for ( std::set< ResidueRange >::const_iterator c = components.begin(), ce = components.end(); c != ce; ++c ) {
			ResidueRange const & component = *c;

			for ( Integer i = component.begin(), ie = component.end(); i <= ie; ++i ) {
				if ( moveable_residues.find( i ) != moveable_residues.end() ) {
					continue;
				}

				xyzVector< float > const es_crd( &es_Eposition( 1, 2, i ) );

				for ( Integer j = 1, je = aligned_Ab.total_residue(); j <= je; ++j ) {
					xyzVector< float > const Ab_crd( &Ab_Eposition( 1, 2, j ) );
					Real const d2 = distance_squared( es_crd, Ab_crd );
					if ( d2 < closest_distance_squared ) {
						closest_distance_squared = d2;
						closest_on_es = i;
						closest_on_Ab = j;
					}
				} // each Ab residue
			} // each epitope component residue
		}

		// connect
		Ab_connection_ = PoseAssembly::connect( epitope_scaffold_, aligned_Ab, closest_on_es, closest_on_Ab, es_Ab, idealize_loop_geometry_ );

		// set new es_Ab
		epitope_scaffold_ = es_Ab;
		epitope_scaffold_.pdb_info().set_use_pdb_numbering( true );

		// prevent bad score messages
		epitope_scaffold_.reset_score_data();

		Ab_is_connected_ = true;
	}

	return false; // no connection performed
}


/// @brief disconnect Ab
/// @details each time disconnect is called, the current antibody sidechain state is archived so that
/// @details state can be restored upon subsequence connect_Ab() call
/// @param[in] auto_archive_Ab_sidechains by default, archive Ab sidechains before disconnect
bool
EpitopeScaffold::disconnect_Ab(
		bool const & auto_archive_Ab_sidechains
)
{
	if ( Ab_is_connected_ ) {
		// we always want to maintain the Ab in fullatom, so copy the sidechains
		// from the newly saved state to archived state if possible
		if ( epitope_scaffold_.fullatom() && auto_archive_Ab_sidechains ) {
			// temporary Ab
			Pose Ab;
			extract_Ab( Ab );

			// save Ab sidechain state
			archived_Ab_.recover_sidechain( Ab );
		}

		// new Pose, just epitope-scaffold
		Pose es;

		// disconnect
		PoseAssembly::disconnect( epitope_scaffold_, Ab_connection_, es, idealize_loop_geometry_ );

		// set new es
		epitope_scaffold_ = es;

		// prevent bad score messages
		epitope_scaffold_.reset_score_data();

		Ab_is_connected_ = false;

		return true; // disconnect performed
	}

	return false; // no disconnect performed
}


/// @brief extract copy of current Ab (with current geometry) in epitope scaffold
bool
EpitopeScaffold::extract_Ab(
	Pose & output_Ab
) const
{
	if ( Ab_is_connected_ ) {
		extract_segment_from_Pose( epitope_scaffold_, Ab_connection_.segment_range(), output_Ab, true ); // boolean: keep pdb_info
	}

	output_Ab.pdb_info().set_use_pdb_numbering( true );
	return Ab_is_connected_;
}


/// @brief extract copy of current epitope scaffold (without Ab) in epitope scaffold
void
EpitopeScaffold::extract_epitope_scaffold(
	Pose & output_es
) const
{
	ResidueRange es_range( 1, epitope_scaffold_.total_residue() );

	if ( Ab_is_connected_ ) {
		es_range = ResidueRange( 1, Ab_connection_.segment_range().begin() - 1 );
	}

	extract_segment_from_Pose( epitope_scaffold_, es_range, output_es, true ); // boolean: keep pdb_info
	output_es.pdb_info().set_use_pdb_numbering( true );
}


/// @brief extract complex of current epitope scaffold + Ab
/// @details this is different from directly accessing the internal Pose in the EpitopeScaffold object,
/// @details as this function returns a complex whose chains are continuous (i.e. individual fold tree segments
/// @details are continuous), versus the internal Pose whose fold tree potentially contains multiple cuts
void
EpitopeScaffold::extract_complex(
	Pose & output_complex
) const
{
	using pose_ns::Fold_tree;

	if ( Ab_is_connected_ ) {
		Integer const es_end = epitope_scaffold_.total_residue() - Ab_connection_.segment_range().length();
		Fold_tree one_jump_fold_tree;
		one_jump_fold_tree.add_edge( 1, Ab_connection_.jump_start(), Fold_tree::PEPTIDE );
		one_jump_fold_tree.add_edge( Ab_connection_.jump_start(), es_end, Fold_tree::PEPTIDE );
		one_jump_fold_tree.add_edge( Ab_connection_.jump_start(), Ab_connection_.jump_stop(), 1 ); // unfortunately cannot use Ab_connection_.jump_label() here
		one_jump_fold_tree.add_edge( Ab_connection_.jump_stop(), Ab_connection_.segment_range().begin(), Fold_tree::PEPTIDE );
		one_jump_fold_tree.add_edge( Ab_connection_.jump_stop(), Ab_connection_.segment_range().end(), Fold_tree::PEPTIDE );
		one_jump_fold_tree.reorder( 1 );
		output_complex.set_fold_tree( one_jump_fold_tree );
	} else {
		Fold_tree simple_fold_tree;
		simple_fold_tree.simple_tree( epitope_scaffold_.total_residue() );
		simple_fold_tree.reorder( 1 );
		output_complex.set_fold_tree( simple_fold_tree );
	}

	output_complex.set_fullatom_flag( true, false ); // boolean: fullatom, repack
	transfer_identity_info_between_Poses( epitope_scaffold_, ResidueRange( 1, epitope_scaffold_.total_residue() ), output_complex, 1, true ); // boolean: keep_pdb_info
	output_complex.set_coords( false, epitope_scaffold_.Eposition(), epitope_scaffold_.full_coord(), false ); // boolean: ideal, check missing

	output_complex.pdb_info().set_use_pdb_numbering( true );
}


/// @brief flip epitope scaffold to all-ala (Ab, if connected, left intact)
/// @param[in] residue_type residue type from Rosetta's param_aa
/// @param[in] keep_gly do not convert any existing glycines?  default is true
void
EpitopeScaffold::convert_to_residue_type(
	Integer const & residue_type,
	bool const & keep_gly,
	bool const include_Ab
)
{
	// find correct range in Pose
	ResidueRange es_range( 1, epitope_scaffold_.total_residue() );
	if ( Ab_is_connected_ && !include_Ab ) {
		es_range = ResidueRange( 1, epitope_scaffold_.total_residue() - Ab_connection_.segment_range().length() );
	}

	mutate_range( epitope_scaffold_, es_range, residue_type, keep_gly );
}


/// @brief flip all sidechains (epitope scaffold + Ab, if present) back to state with original
/// @brief sidechains
/// @warning after this, internal epitope scaffold Pose is fullatom
void
EpitopeScaffold::recover_all_native_sidechains()
{
	// make pose fullatom if not fullatom, this destructively dimensions fullatom arrays inside Pose
	if ( !epitope_scaffold_.fullatom() ) {
		epitope_scaffold_.set_fullatom_flag( true, false ); // boolean: fullatom, repack
	}

	recover_native_epitope_scaffold_sidechains();
	recover_Ab_sidechains();
}


/// @brief flip epitope scaffold back to state with original sidechains (Ab, if connected, left intact)
/// @brief (native epitope and native scaffold)
/// @warning after this, internal epitope scaffold Pose is fullatom
void
EpitopeScaffold::recover_native_epitope_scaffold_sidechains()
{
	// make pose fullatom if not fullatom, this destructively dimensions fullatom arrays inside Pose
	if ( !epitope_scaffold_.fullatom() ) {
		epitope_scaffold_.set_fullatom_flag( true, false ); // boolean: fullatom, repack
	}

	// find correct range in Pose
	ResidueRange es_range( 1, epitope_scaffold_.total_residue() );
	if ( Ab_is_connected_ ) {
		es_range = ResidueRange( 1, epitope_scaffold_.total_residue() - Ab_connection_.segment_range().length() );
	}

	copy_sidechains( epitope_scaffold_, archived_epitope_scaffold_, es_range );
}


/// @brief flip just epitope sidechains back to state of native epitope sidechains
/// @note obeys keep natro if available
void
EpitopeScaffold::recover_native_epitope_sidechains()
{
	std::set< Integer > epitope_sidechains;

	// grab proper sidechains
	std::set< ResidueRange > components = all_components();
	for ( std::set< ResidueRange >::const_iterator c = components.begin(), ce = components.end(); c != ce; ++c ) {
		ResidueRange const & new_loop = *c;

		if ( do_graft_sc( new_loop ) ) {
			ResidueRange const & old_loop = new_to_native_epitope( new_loop );
			Integer const offset = new_loop.begin() - old_loop.begin();

			for ( Integer epitope_res = old_loop.begin(), last_epitope_res = old_loop.end(); epitope_res <= last_epitope_res; ++epitope_res ) {
				if ( keep_natro_.size() == 0 || keep_natro_.find( epitope_res ) != keep_natro_.end() ) {
					Integer const es_res = epitope_res + offset;

					epitope_sidechains.insert( es_res );
				}
			} // foreach epitope residue
		}

	}

	copy_sidechains( epitope_scaffold_, archived_epitope_scaffold_, epitope_sidechains.begin(), epitope_sidechains.end() );
}


/// @brief recover Ab sidechains
bool
EpitopeScaffold::recover_Ab_sidechains()
{
	if ( Ab_is_connected_ ) {
		// make pose fullatom if not fullatom, this destructively dimensions fullatom arrays inside Pose
		if ( !epitope_scaffold_.fullatom() ) {
			epitope_scaffold_.set_fullatom_flag( true, false ); // boolean: fullatom, repack
		}

		// manually recover antibody sidechains
		FArray3D_float const & ab_full_coord = archived_Ab_.full_coord(); // cache
		for ( Integer res = Ab_connection_.segment_range().begin(), last_res = Ab_connection_.segment_range().end(); res <= last_res; ++res ) {
			Integer const ab_res = res - Ab_connection_.residue_offset();
			epitope_scaffold_.copy_sidechain( res, archived_Ab_.res( ab_res ), archived_Ab_.res_variant( ab_res ), ab_full_coord( 1, 1, ab_res ) );
		}
	}

	return Ab_is_connected_;
}


/// @brief build all loops (centroid) simultaneously
/// @result success/failure according to closure criterion
/// @note  initial closure state checked here
/// @note  do_graft_bb honored here
bool
EpitopeScaffold::build_all_simul(
	bool const & ignore_closure_state,
	BuildType const & build_type
)
{
	using epigraft::match::align::AlignmentSystem;

	// gather closures that are not closed
	std::set< LoopClosureInfo > remaining_closures;
	for ( std::set< LoopClosureInfo >::const_iterator a = closures_to_attempt_.begin(), ae = closures_to_attempt_.end(); a != ae; ++a ) {
		LoopClosureInfo const & closure_info = *a;

		if ( do_graft_bb( closure_info ) &&
		     ( ( !closure_info.is_closed() && closure_info.moveable_residues().size() > 0 ) ||
		     ignore_closure_state ) &&
		     !closed_during_trajectory( closure_info ) ) { // do closure?
			remaining_closures.insert( closure_info );
		}
	}

	// check and make sure there's something to do
	if ( remaining_closures.size() < 1 ) { // nothing to do
		return true;
	}

	std::cout << "* es build_all_simul:" << std::endl;

	// disallow jump movements (here for safety, since procedure shouldn't modify jump)
	epitope_scaffold_.set_allow_jump_move( false );

	// setup bb move
	setup_proper_allow_bb_move( remaining_closures );

	// first store middle of primary component to use in updating Ab_transform
	FArray2D_float original_primary_middle_bb;
	extract_1bb_from_Pose( epitope_scaffold_, middle_of_range( primary_component() ), original_primary_middle_bb );

	// build loops for closures that remain
	switch ( build_type ) {
		case NORMAL_BUILD:
			epigraft::design::build_loops_simul( epitope_scaffold_, remaining_closures, build_cycles_, use_fragment_insertion_, chainbreak_criterion_, local_rama_criterion_, loop_closure_ss_string(), loop_closure_seq_string(), number_of_fragments(), use_variable_length_fragments_, repick_frag_ );
			break;
		case ARM_BUILD:
			epigraft::design::build_loops_simul_arm( epitope_scaffold_, remaining_closures, arm_residues_, build_cycles_, use_fragment_insertion_, chainbreak_criterion_, local_rama_criterion_, loop_closure_ss_string(), loop_closure_seq_string(), number_of_fragments(), use_variable_length_fragments_ );
			break;
		case ADAPTIVE_BUILD:
			epigraft::design::adaptive_build_loops_simul( epitope_scaffold_, remaining_closures, build_cycles_, use_fragment_insertion_, chainbreak_criterion_, local_rama_criterion_, loop_closure_ss_string(), loop_closure_seq_string(), number_of_fragments(), use_variable_length_fragments_ );
			break;
		case SCREEN_BUILD:
			epigraft::design::screen_build_loops_simul( epitope_scaffold_, remaining_closures, build_cycles_, chainbreak_criterion_, local_rama_criterion_, 10.0, loop_closure_ss_string(), loop_closure_seq_string(), number_of_fragments() );
			break;
		case FRAGMENT_FAST_CCD_BUILD:
			epigraft::design::build_loops_simul_with_fragments( epitope_scaffold_, remaining_closures, build_cycles_, chainbreak_criterion_, local_rama_criterion_, 3, loop_closure_ss_string(), loop_closure_seq_string(), number_of_fragments() );
			break;
		case FRAGMENT_TWO_TORSION_CCD_BUILD:
			epigraft::design::build_loops_simul_with_fragments( epitope_scaffold_, remaining_closures, build_cycles_, chainbreak_criterion_, local_rama_criterion_, 2, loop_closure_ss_string(), loop_closure_seq_string(), number_of_fragments() );
			break;
		case FRAGMENT_ONE_TORSION_CCD_BUILD:
			epigraft::design::build_loops_simul_with_fragments( epitope_scaffold_, remaining_closures, build_cycles_, chainbreak_criterion_, local_rama_criterion_, 1, loop_closure_ss_string(), loop_closure_seq_string(), number_of_fragments() );
			break;
		case FRAGMENT_NO_CCD_BUILD:
			epigraft::design::build_loops_simul_with_fragments( epitope_scaffold_, remaining_closures, build_cycles_, chainbreak_criterion_, local_rama_criterion_, 0, loop_closure_ss_string(), loop_closure_seq_string(), number_of_fragments() );
			break;
		default: // same as NORMAL_BUILD
			epigraft::design::build_loops_simul( epitope_scaffold_, remaining_closures, build_cycles_, use_fragment_insertion_, chainbreak_criterion_, local_rama_criterion_, loop_closure_ss_string(), loop_closure_seq_string(), number_of_fragments() );
			break;
	}


	// closure info updates
	bool all_closed = true;
	for ( std::set< LoopClosureInfo >::const_iterator a = remaining_closures.begin(), ae = remaining_closures.end(); a != ae; ++a ) {
		LoopClosureInfo const & closure_info = *a;

		// check closure criteria
		check_closure_criteria( closure_info );

		// update closure info
		update_closures( closure_info );

		// update trajectory info
		if ( !graft_info_.moving_lever_residues() ) {
			closed_during_trajectory_[ closure_info ] = closure_info.is_closed();
		}

		// total closure status
		all_closed = all_closed && closure_info.is_closed();
	}

	for ( std::set< LoopClosureInfo >::const_iterator a = closures_to_attempt_.begin(), ae = closures_to_attempt_.end(); a != ae; ++a ) {
		// show score
		std::cout << closed_during_trajectory_[ *a ] << " | " << a->to_string() << std::endl;
	}

	// update Ab transform
	FArray2D_float new_primary_middle_bb;
	extract_1bb_from_Pose( epitope_scaffold_, middle_of_range( primary_component() ), new_primary_middle_bb );
	update_Ab_transform( original_primary_middle_bb, new_primary_middle_bb );

	return all_closed;
}


/// @brief build primary loops (centroid)
/// @result success/failure according to closure criterion
/// @note  initial closure state checked here
/// @note  do_graft_bb honored here
bool
EpitopeScaffold::build_primary(
	bool const & ignore_closure_state,
	bool const & stop_on_fail
)
{
	bool all_passed = true;
	bool passed = false;

	// run over primary closures
	std::set< LoopClosureInfo > primary_closures = primary_closures_to_attempt();
	for ( std::set< LoopClosureInfo >::const_iterator a = primary_closures.begin(), ae = primary_closures.end(); a != ae; ++a ) {
		LoopClosureInfo const & closure_info = *a;

		// only build if not closed
		if ( do_graft_bb( closure_info ) && ( ignore_closure_state || !closure_info.is_closed() ) ) {
			passed = build_loop( closure_info );

			if ( stop_on_fail && !passed ) {
				return false;
			}

			all_passed = all_passed && passed;
		}
	}

	return all_passed;
}


/// @brief build secondary loops (centroid)
/// @result success/failure according to closure criterion
/// @note  initial closure state checked here
/// @note  do_graft_bb honored here
bool
EpitopeScaffold::build_secondary(
	bool const & ignore_closure_state,
	bool const & stop_on_fail
)
{
	bool all_passed = true;
	bool passed = false;

	std::set< LoopClosureInfo > secondary_closures = secondary_closures_to_attempt();
	for ( std::set< LoopClosureInfo >::const_iterator a = secondary_closures.begin(), ae = secondary_closures.end(); a != ae; ++a ) {
		LoopClosureInfo const & closure_info = *a;

		// only build if not closed
		if ( do_graft_bb( closure_info ) && ( ignore_closure_state || !closure_info.is_closed() ) ) {
			passed = build_loop( closure_info );

			if ( stop_on_fail && !passed ) {
				return false;
			}

			all_passed = all_passed && passed;
		}
	}

	return all_passed;
}


/// @brief build a component's loops (centroid)
/// @result success/failure according to closure criterion
/// @note  forcibly build, no check of initial closure state
bool
EpitopeScaffold::build_component(
	ResidueRange const & component,
	bool const & stop_on_fail
)
{
	bool all_passed = true;
	bool passed = false;

	std::set< LoopClosureInfo > closures = component_closures_to_attempt( component );
	for ( std::set< LoopClosureInfo >::const_iterator a = closures.begin(), ae = closures.end(); a != ae; ++a ) {
		LoopClosureInfo const & closure_info = *a;

		passed = build_loop( closure_info );

		if ( stop_on_fail && !passed ) {
			return false;
		}

		all_passed = all_passed && passed;
	}

	return all_passed;
}


/// @brief build a set of loops (centroid)
/// @result success/failure according to closure criterion
/// @note  forcibly build, no check of initial closure state
bool
EpitopeScaffold::build_loops(
	std::set< LoopClosureInfo > const & closures_to_attempt,
	bool const & stop_on_fail
)
{
	bool all_passed = true;
	bool passed = false;

	for ( std::set< LoopClosureInfo >::const_iterator a = closures_to_attempt.begin(), ae = closures_to_attempt.end(); a != ae; ++a ) {
		passed = build_loop( *a );

		if ( stop_on_fail && !passed ) {
			return false;
		}

		all_passed = all_passed && passed;
	}

	return all_passed;
}

/// @brief refine all loops (full atom) simultaneously
/// @result success/failure according to closure criterion
/// @note  initial closure state checked here
/// @note  do_graft_bb honored here
bool
EpitopeScaffold::refine_all_simul(
	bool const & ignore_closure_state,
	RefineType const & refine_type
)
{
	// gather closures that are not closed
	std::set< LoopClosureInfo > remaining_closures;
	for ( std::set< LoopClosureInfo >::const_iterator a = closures_to_attempt_.begin(), ae = closures_to_attempt_.end(); a != ae; ++a ) {
		LoopClosureInfo const & closure_info = *a;

		if ( do_graft_bb( closure_info ) &&
		     ( ( !closure_info.is_closed() && closure_info.moveable_residues().size() > 0 ) ||
		     ignore_closure_state ) &&
		     !closed_during_trajectory( closure_info ) ) { // do closure?
			remaining_closures.insert( closure_info );
		}
	}

	// check and make sure there's something to do
	if ( remaining_closures.size() < 1 ) { // nothing to do
		return true;
	}

	std::cout << "* es refine_all_simul:" << std::endl;

	// disallow jump movements
	epitope_scaffold_.set_allow_jump_move( false );

	// setup bb move
	setup_proper_allow_bb_move( remaining_closures );

	// first store middle of primary component to use in updating Ab_transform
	FArray2D_float original_primary_middle_bb;
	extract_1bb_from_Pose( epitope_scaffold_, middle_of_range( primary_component() ), original_primary_middle_bb );

	// build loops for closures that remain
	switch ( refine_type ) {
		case NORMAL_REFINE:
			epigraft::design::refine_loops_simul( epitope_scaffold_, remaining_closures, refine_cycles_, use_fast_refine_, get_fixed_residues() );
			break;
		case MIN_REPACK_REFINE:
			epigraft::design::minrepack_loops_simul( epitope_scaffold_, remaining_closures, refine_cycles_, get_fixed_residues(), 3.0 );
			break;
		case CLASSIC_REFINE:
			epigraft::design::refine_loops_simul_classic( epitope_scaffold_, remaining_closures, refine_cycles_, use_fast_refine_, get_fixed_residues() );
			break;
		case CLASSIC_REFINE_TWO:
			epigraft::design::refine_loops_simul_classic_two( epitope_scaffold_, remaining_closures, refine_cycles_, use_fast_refine_, get_fixed_residues() );
			break;
		case CONSTRAINED_REFINE: {
			// create sealed pose
			Pose sealed_pose;
			extract_complex( sealed_pose ); // not necessarily a complex if no Ab

			// mirror bb move
			for ( Integer r = 1, re = epitope_scaffold_.total_residue(); r <= re; ++r ) {
				sealed_pose.set_allow_bb_move( r, epitope_scaffold_.get_allow_bb_move( r ) );
			}

			// do refine
			epigraft::design::refine_loops_simul_constrained( sealed_pose, remaining_closures, refine_cycles_, get_fixed_residues() );

			// break fold tree and store as epitope scaffold
			sealed_pose.set_fold_tree( epitope_scaffold_.fold_tree() );
			epitope_scaffold_ = sealed_pose;

			// clear scores as pre-empt against bad score messages
			epitope_scaffold_.reset_score_data();

			break;
		}
		default: // same as NORMAL
			epigraft::design::refine_loops_simul( epitope_scaffold_, remaining_closures, refine_cycles_, use_fast_refine_, get_fixed_residues() );
			break;
	}

	// closure info updates
	bool all_closed = true;
	for ( std::set< LoopClosureInfo >::const_iterator a = remaining_closures.begin(), ae = remaining_closures.end(); a != ae; ++a ) {
		LoopClosureInfo const & closure_info = *a;

		// need to re-idealize if using constrained refine loops
		// this is necessary to be consistent with save/load/checkpointing
		// and may underscore a potential issue with the constrained refine
		// and very low ( < 0.2 ) "closed" chainbreak that could "re-break"
		// under the scheme
		if ( refine_type == CONSTRAINED_REFINE ) {
			idealize_cutpoint_retaining_psiphi( epitope_scaffold_, closure_info.cut(), epitope_scaffold_.psi( closure_info.cut() ), epitope_scaffold_.phi( closure_info.cut() + 1 ) );
		}

		// check closure criteria
		check_closure_criteria( closure_info );

		// update closure info
		update_closures( closure_info );

		// update trajectory info
		if ( !graft_info_.moving_lever_residues() ) {
			closed_during_trajectory_[ closure_info ] = closure_info.is_closed();
		}

		// total closure status
		all_closed = all_closed && closure_info.is_closed();
	}

	for ( std::set< LoopClosureInfo >::const_iterator a = closures_to_attempt_.begin(), ae = closures_to_attempt_.end(); a != ae; ++a ) {
		// show score
		std::cout << closed_during_trajectory_[ *a ] << " | " << a->to_string() << std::endl;
	}

	// update Ab transform
	FArray2D_float new_primary_middle_bb;
	extract_1bb_from_Pose( epitope_scaffold_, middle_of_range( primary_component() ), new_primary_middle_bb );
	update_Ab_transform( original_primary_middle_bb, new_primary_middle_bb );

	return all_closed;
}


/// @brief refine primary loops (full atom)
/// @result success/failure according to closure criterion
/// @note  initial closure state checked here
/// @note  do_graft_bb honored here
bool
EpitopeScaffold::refine_primary(
	bool const & ignore_closure_state,
	bool const & stop_on_fail
)
{
	bool all_passed = true;
	bool passed = false;

	// run over primary closures
	std::set< LoopClosureInfo > primary_closures = primary_closures_to_attempt();
	for ( std::set< LoopClosureInfo >::const_iterator a = primary_closures.begin(), ae = primary_closures.end(); a != ae; ++a ) {
		LoopClosureInfo const & closure_info = *a;

		// only build if not closed
		if ( do_graft_bb( closure_info ) && ( ignore_closure_state || !closure_info.is_closed() ) ) {
			passed = refine_loop( closure_info );

			if ( stop_on_fail && !passed ) {
				return false;
			}

			all_passed = all_passed && passed;
		}
	}

	return all_passed;
}


/// @brief refine secondary loops (full atom)
/// @result success/failure according to closure criterion
/// @note  initial closure state checked here
/// @note  do_graft_bb honored here
bool
EpitopeScaffold::refine_secondary(
	bool const & ignore_closure_state,
	bool const & stop_on_fail
)
{
	bool all_passed = true;
	bool passed = false;

	std::set< LoopClosureInfo > secondary_closures = secondary_closures_to_attempt();
	for ( std::set< LoopClosureInfo >::const_iterator a = secondary_closures.begin(), ae = secondary_closures.end(); a != ae; ++a ) {
		LoopClosureInfo const & closure_info = *a;

		// only build if not closed
		if ( do_graft_bb( closure_info ) && ( ignore_closure_state || !closure_info.is_closed() ) ) {
			passed = refine_loop( closure_info );

			if ( stop_on_fail && !passed ) {
				return false;
			}

			all_passed = all_passed && passed;
		}
	}

	return all_passed;
}


/// @brief refine a component's loops (full atom)
/// @result success/failure according to closure criterion
/// @note  forcibly refine, no check of initial closure state
bool
EpitopeScaffold::refine_component(
	ResidueRange const & component,
	bool const & stop_on_fail
)
{
	bool all_passed = true;
	bool passed = false;

	std::set< LoopClosureInfo > closures = component_closures_to_attempt( component );
	for ( std::set< LoopClosureInfo >::const_iterator a = closures.begin(), ae = closures.end(); a != ae; ++a ) {
		LoopClosureInfo const & closure_info = *a;

		passed = refine_loop( closure_info );

		if ( stop_on_fail && !passed ) {
			return false;
		}

		all_passed = all_passed && passed;
	}

	return all_passed;
}


/// @brief refine a set of loops (full atom)
/// @result success/failure according to closure criterion
/// @note  forcibly refine, no check of initial closure state
bool
EpitopeScaffold::refine_loops(
	std::set< LoopClosureInfo > const & closures_to_attempt,
	bool const & stop_on_fail
)
{
	bool all_passed = true;
	bool passed = false;

	for ( std::set< LoopClosureInfo >::const_iterator a = closures_to_attempt.begin(), ae = closures_to_attempt.end(); a != ae; ++a ) {
		passed = refine_loop( *a );

		if ( stop_on_fail && !passed ) {
			return false;
		}

		all_passed = all_passed && passed;
	}

	return all_passed;
}


/// @brief (centroid level) build loop
/// @result success/failure according to closure criterion
/// @note  internal closure info is updated here
/// @warning THIS ROUTINE IS OLD AND DOES NOT SUPPORT ALL BUILD TYPES
///          PLEASE USE build_all_simul() INSTEAD
bool
EpitopeScaffold::build_loop(
	LoopClosureInfo const & closure_info
)
{
	// check and make sure there's something to do
	if ( closure_info.moveable_residues().size() < 1 ) { // nothing to do
		return true;
	}

	std::cout << "* es build_loop:" << std::endl;

	// first store middle of primary component to use in updating Ab_transform
	FArray2D_float original_primary_middle_bb;
	extract_1bb_from_Pose( epitope_scaffold_, middle_of_range( primary_component() ), original_primary_middle_bb );

	// disallow jump movements (here for safety, since procedure shouldn't modify jump)
	epitope_scaffold_.set_allow_jump_move( false );

	// reset allow bb move
	epitope_scaffold_.set_allow_bb_move( false );

	// build loop
	closure_info.setup_allow_bb_move( epitope_scaffold_ );
	epigraft::design::build_loop( epitope_scaffold_, closure_info, build_cycles_, use_fragment_insertion_, loop_closure_ss_string(), loop_closure_seq_string(), number_of_fragments() );

	// show score
	std::cout << closure_info.to_string() << std::endl;

	// check closure criteria
	check_closure_criteria( closure_info );

	// update closure info
	update_closures( closure_info );

	// update Ab transform
	FArray2D_float new_primary_middle_bb;
	extract_1bb_from_Pose( epitope_scaffold_, middle_of_range( primary_component() ), new_primary_middle_bb );
	update_Ab_transform( original_primary_middle_bb, new_primary_middle_bb );

	return closure_info.is_closed();
}


/// @brief (full atom) refine loop
/// @result success/failure according to closure criterion
/// @note  internal closure info is updated here
bool
EpitopeScaffold::refine_loop(
	LoopClosureInfo const & closure_info
)
{
	// check and make sure there's something to do
	if ( closure_info.moveable_residues().size() < 1 ) { // nothing to do
		return true;
	}

	std::cout << "* es refine_loop:" << std::endl;

	// first store middle of primary component to use in updating Ab_transform
	FArray2D_float original_primary_middle_bb;
	extract_1bb_from_Pose( epitope_scaffold_, middle_of_range( primary_component() ), original_primary_middle_bb );

	// disallow jump movements
	epitope_scaffold_.set_allow_jump_move( false );

	// reset allow bb/chi moves
	epitope_scaffold_.set_allow_bb_move( false );
	epitope_scaffold_.set_allow_chi_move( false );

	// setup allow bb/chi moves
	closure_info.setup_allow_bb_move( epitope_scaffold_ );
	closure_info.setup_allow_chi_move( epitope_scaffold_ );
	// TODO: set allow_chi_move to false for epitope

	// refine loop
	epigraft::design::refine_loop( epitope_scaffold_, closure_info, refine_cycles_, use_fast_refine_, get_fixed_residues() );

	// show score
	std::cout << closure_info.to_string() << std::endl;

	// update closure info
	update_closures( closure_info );

	// check closure criteria
	check_closure_criteria( closure_info );

	// update Ab transform
	FArray2D_float new_primary_middle_bb;
	extract_1bb_from_Pose( epitope_scaffold_, middle_of_range( primary_component() ), new_primary_middle_bb );
	update_Ab_transform( original_primary_middle_bb, new_primary_middle_bb );

	return closure_info.is_closed();
}


/// @brief   design
///          In this procedure the es-foldtree is temporarily replaced with a
///          simple fold tree to score correctly during design.  The es-foldtree
///          is recovered at the end, at which point the scores for the internal
///          epitope-scaffold pose are reset.
void
EpitopeScaffold::design()
{
	if ( !epitope_scaffold_.fullatom() ) {
		std::cerr << "EpitopeScaffold::design() : design is being called with a non full-atom Pose, refusing to design!" << std::endl;
		return;
	}

	// make design matrix
	FArray2D< bool > design_matrix = make_design_info();

	// do design
	design( design_matrix );
}


/// @brief   design using design matrix
/// @details design using given design matrix only, no other information is taken
///          into account to allow user the ability to override anything they need
void
EpitopeScaffold::design(
	FArray2D< bool > & design_matrix
)
{
	PackerTask task( epitope_scaffold_ );
	pack::InteractionGraphBase * ig = NULL;
	RotamerSet rs;
	FArray1D_int current_rot_index( epitope_scaffold_.total_residue() );
	FArray1D_float ligenergy1b_old;

	design( design_matrix, &task, ig, &rs, &current_rot_index, &ligenergy1b_old );

	if ( ig != NULL ) {
		delete ig;
	}
}


/// @brief design returning cached information
/// @details implementation is kind of a mess here, it's setup the way it is due
///  to the way other things are hooked in, will cleanup later
void
EpitopeScaffold::design(
	FArray2D< bool > & design_matrix,
	PackerTask * task,
	pack::InteractionGraphBase * & ig,
	RotamerSet * rs,
	FArray1D_int * current_rot_index,
	FArray1D_float * ligenergy1b_old
)
{
	using pose_ns::Fold_tree;
	using pose_ns::Score_weight_map;

	std::cout << "* es design:" << std::endl;

	// first do error handling
	bool error = false;

	if ( !epitope_scaffold_.fullatom() ) {
		error = true;
		std::cerr << "EpitopeScaffold::design() : design is being called with a non full-atom Pose, refusing to design!" << std::endl;
	}

	if ( design_matrix.size2() != static_cast< Size >( epitope_scaffold_.total_residue() ) ) {
		error = true;
		std::cerr << "EpitopeScaffold::design( DesignMatrix ) : number of residues in DesignMatrix and internal Pose do not match, refusing to design!" << std::endl;
	}

	if ( error ) {
		return;
	}

	// make copy of original fold tree
	Fold_tree original_fold_tree = epitope_scaffold_.fold_tree();

	// use separate Pose to design -- if we don't do this there is a potential problem
	// with small shifts in large sections of the scaffold due to inexact loop closure
	// or floating point inconsistency
	Pose work_es;
	extract_complex( work_es ); // fold tree segments for chains are now continous, so rama etc. are scored properly

	// design
	epigraft::design::design( work_es, design_matrix, *task, ig, *rs, *current_rot_index, *ligenergy1b_old );

	// score once and show
	Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score12 );
	work_es.score( weight_map );
	work_es.show_scores( std::cout );
	std::cout << std::endl;

	// Copy design and recover original fold tree.
	work_es.set_fold_tree( original_fold_tree );

	// need to re-idealize cutpoints using prior angles to artificially break
	for ( std::set< LoopClosureInfo >::const_iterator a = closures_to_attempt_.begin(), ae = closures_to_attempt_.end(); a != ae; ++a ) {
		LoopClosureInfo const & closure_info = *a;

		// Idealize required cutpoints.
		if ( do_graft_bb( closure_info ) ) {
			idealize_cutpoint_retaining_psiphi( work_es,
			                                    closure_info.cut(),
			                                    epitope_scaffold_.psi( closure_info.cut() ),
			                                    epitope_scaffold_.phi( closure_info.cut() + 1 ) );
		}
	}

	// replace internal pose
	epitope_scaffold_ = work_es;

	// clear scores as pre-empt against bad score messages
	epitope_scaffold_.reset_score_data();
}


/// @brief design using cached information
/// @details implementation is kind of a mess here, it's setup the way it is due
///  to the way other things are hooked in, will cleanup later
void
EpitopeScaffold::design(
	PackerTask * task,
	pack::InteractionGraphBase * & ig,
	RotamerSet * rs,
	FArray1D_int * current_rot_index,
	FArray1D_float * ligenergy1b_old
)
{
	using pose_ns::Fold_tree;
	using pose_ns::Score_weight_map;

	std::cout << "* es design:" << std::endl;

	// first do error handling
	bool error = false;

	if ( !epitope_scaffold_.fullatom() ) {
		error = true;
		std::cerr << "EpitopeScaffold::design() : design is being called with a non full-atom Pose, refusing to design!" << std::endl;
	}

	if ( error ) {
		return;
	}

	// make copy of original fold tree
	Fold_tree original_fold_tree = epitope_scaffold_.fold_tree();

	// use separate Pose to design -- if we don't do this there is a potential problem
	// with small shifts in large sections of the scaffold due to inexact loop closure
	// or floating point inconsistency
	Pose work_es;
	extract_complex( work_es ); // fold tree segments for chains are now continous, so rama etc. are scored properly

	// design
	epigraft::design::design( work_es, *task, ig, *rs, *current_rot_index, *ligenergy1b_old );

	// score once and show
	Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score12 );
	work_es.score( weight_map );
	work_es.show_scores( std::cout );
	std::cout << std::endl;

	// Copy design and recover original fold tree.
	work_es.set_fold_tree( original_fold_tree );

	// need to re-idealize cutpoints using prior angles to artificially break
	for ( std::set< LoopClosureInfo >::const_iterator a = closures_to_attempt_.begin(), ae = closures_to_attempt_.end(); a != ae; ++a ) {
		LoopClosureInfo const & closure_info = *a;

		// Idealize required cutpoints.
		if ( do_graft_bb( closure_info ) ) {
			idealize_cutpoint_retaining_psiphi( work_es,
			                                    closure_info.cut(),
			                                    epitope_scaffold_.psi( closure_info.cut() ),
			                                    epitope_scaffold_.phi( closure_info.cut() + 1 ) );
		}
	}

	// replace internal pose
	epitope_scaffold_ = work_es;

	// clear scores as pre-empt against bad score messages
	epitope_scaffold_.reset_score_data();
}


/// @brief forcibly repack range in internal pose -- ignores internal allow repack flags
/// @warning can repack with cut multigrafting fold tree!
void
EpitopeScaffold::repack(
	ResidueRange const & range
)
{
	// set proper allow chi move
	// we need a secondary array for passing into Pose::repack() due to
	// some weird behavioral logic in that code
	FArray1D_bool allow_repack( epitope_scaffold_.total_residue(), false );
	epitope_scaffold_.set_allow_chi_move( false ); // lock all chi

	for ( Integer i = range.begin(), ie = range.end(); i <= ie; ++i ) {
		allow_repack( i ) = true;
		epitope_scaffold_.set_allow_chi_move( true );
	}

	// invoke repack
	epitope_scaffold_.repack( allow_repack, true ); // boolean: include current sidechain

	// safety, relock chi
	epitope_scaffold_.set_allow_chi_move( false );
}


/// @brief forcibly repack complex -- ignores internal allow repack flags
/// @warning can repack with cut multigrafting fold tree!
void
EpitopeScaffold::repack_complex()
{
	repack( ResidueRange( 1, epitope_scaffold_.total_residue() ) );
}


/// @brief forcibly repack antibody if present -- ignores internal allow repack flags
/// @warning can repack with cut multigrafting fold tree!
void
EpitopeScaffold::repack_Ab()
{
	if ( Ab_is_connected_ ) {
		repack( Ab_connection_.segment_range() );
	}
}


/// @brief forcibly repack epitope-scaffold -- ignores internal allow repack flags
/// @warning can repack with cut multigrafting fold tree!
void
EpitopeScaffold::repack_es()
{
	Integer last_es_residue = epitope_scaffold_.total_residue();
	if ( Ab_is_connected_ ) {
		last_es_residue = Ab_connection_.segment_range().begin() - 1;
	}

	repack( ResidueRange( 1, last_es_residue) );
}


/// @brief forcibly repack epitope-scaffold / antibody interface
/// @warning can repack with cut multigrafting fold tree!
void
EpitopeScaffold::repack_interface()
{
	if ( Ab_is_connected_ ) {
		std::set< Integer > interface_positions = identify_interface_residues();

		// set proper allow chi move
		// we need a secondary array for passing into Pose::repack() due to
		// some weird behavioral logic in that code
		FArray1D_bool allow_repack( epitope_scaffold_.total_residue(), false );
		epitope_scaffold_.set_allow_chi_move( false ); // lock all chi

		for ( std::set< Integer >::const_iterator i = interface_positions.begin(), ie = interface_positions.end(); i != ie; ++i ) {
			allow_repack( *i ) = true;
			epitope_scaffold_.set_allow_chi_move( *i, true );
		}

		// invoke repack
		epitope_scaffold_.repack( allow_repack, true ); // boolean: include current sidechain

		// safety, relock everything
		epitope_scaffold_.set_allow_chi_move( false );
	}
}


/// @brief optimize initial position of broken epitope by monte carlo rb-perturbation + minimization
/// @brief without Ab, mc rb-perturb attempted in 1 angstrom/1 degree gaussian spread
/// @return true if optimization was attempted, false if the procedure was unable to run due to input error
bool
EpitopeScaffold::epitope_rb_optimize(
	Integer const & cycles,
	Integer const & rb_perturbations,
	bool const & do_minimize,
	Real const & translation_scale,
	Real const & rotation_scale
)
{
	using epigraft::match::align::AlignmentSystem;
	using namespace pose_ns;

	// check to make sure alignment system type allows for rb_optimize
	if ( graft_info_.match_result().system_type == AlignmentSystem::N2C ||
	     graft_info_.match_result().system_type == AlignmentSystem::C2N ) {
		return false;
	}

	// first store middle of primary component to use in updating Ab_transform
	FArray2D_float original_primary_middle_bb;
	extract_1bb_from_Pose( epitope_scaffold_, middle_of_range( primary_component() ), original_primary_middle_bb );

	// weight map -- only vdw and chainbreak
	Score_weight_map weight_map;
	if ( epitope_scaffold_.fullatom() ) {
		weight_map.set_weight( FA_REP, 1.0 );
	} else {
		weight_map.set_weight( VDW, 1.0 );
	}
	weight_map.set_weight( CHAINBREAK, 1.0 );
	weight_map.set_weight( CHAINBREAK_OVERLAP, 1 );
	FArray1D_float cutpoint_weight( epitope_scaffold_.fold_tree().get_num_fold_tree_cutpoint(), 1.0 ); // score all cuts
	weight_map.set_1D_weight( CUT_WEIGHT, cutpoint_weight );

	// for handling setup if minimization requested
	if ( do_minimize ) { // force no bb/chi move, true rigid body only during minimization
		// set minimizer
		minimize_reset(); // only phi-psi and rb
		minimize_set_vary_phipsi( false ); // not even phi-psi

		// safety, lock angles
		epitope_scaffold_.set_allow_bb_move( false );
		epitope_scaffold_.set_allow_chi_move( false );
	}

	// store initial jump information for primary jump
	Integer const & jump_label = this->jump_label( primary_component() );
	Jump new_jump = epitope_scaffold_.get_jump( jump_label );

	// set up moveable jump
	epitope_scaffold_.set_allow_jump_move( false );
	epitope_scaffold_.set_allow_jump_move( jump_label, true );

	// monte carlo
	epitope_scaffold_.score( weight_map );
	Monte_carlo mc( epitope_scaffold_, weight_map, 1 ); // temperature = 1

	// run through cycles
	for ( Integer cycle = 1; cycle <= cycles; ++cycle ) {

		epitope_scaffold_ = mc.low_pose();

		// rb perturb with gaussian spread
		for ( Integer p = 1; p <= rb_perturbations; ++p ) {
			new_jump.gaussian_move( pose_param::n2c, translation_scale, rotation_scale ); // translation, rotation
			epitope_scaffold_.set_jump( jump_label, new_jump );
			mc.boltzmann( epitope_scaffold_ );
		}

		// minimize, if requested
		if ( do_minimize ) {
			epitope_scaffold_.main_minimize( weight_map, "dfpmin" );
			mc.boltzmann( epitope_scaffold_ );
		}

	}

	// recover low
	epitope_scaffold_ = mc.low_pose();

	// lock jump
	epitope_scaffold_.set_allow_jump_move( jump_label, false );

	// update Ab transform
	FArray2D_float new_primary_middle_bb;
	extract_1bb_from_Pose( epitope_scaffold_, middle_of_range( primary_component() ), new_primary_middle_bb );
	update_Ab_transform( original_primary_middle_bb, new_primary_middle_bb );

	// reset minimizer if used
	if ( do_minimize ) {
		minimize_reset();
	}

	return true;
}


/// @brief perturbs jump for epitope scaffold by gaussian spread
/// @param[in] epitope_scaffold_range  the epitope range in the current epitope scaffold (i.e. "new" range)
/// @param[in] translation_magnitude  translation magnitude in angstroms
/// @param[in] rotation_magnitude  rotation magnitude in degrees
/// @param[in] perturb_using_archived_jump  if true (default), perturb using the archived jump to limit drift from original position
/// @warning this may not be uniform
inline
bool
EpitopeScaffold::gaussian_perturb_epitope_rb_jump(
	ResidueRange const & epitope_scaffold_range,
	Real const & translation_magnitude,
	Real const & rotation_magnitude,
	bool const & perturb_using_archived_jump
)
{
	using namespace pose_ns;

	EpitopeTrack const & e = epitope_tracking_.find( epitope_scaffold_range )->second;

	if ( e.jump_label < 0 ) { // no jump present
		return false;
	}

	Jump jump = perturb_using_archived_jump ? e.archived_jump : epitope_scaffold_.get_jump( e.jump_label );
	jump.gaussian_move( pose_param::n2c, translation_magnitude, rotation_magnitude );

	epitope_scaffold_.set_jump( e.jump_label, jump );

	return true;
}


/// @brief perturb all non-closed secondary epitope components by gaussian spread with a forced clash check
/// @details Clash checks against antibody (if attached) and all residues of scaffold and epitope except for
/// @details loop moveable residues.  If there is a clash the proposed move is rejected and another is
/// @details attempted.  If an epitope component is "half-closed" (one side closed, one side open),
/// @details no perturbation is done.
void
EpitopeScaffold::gaussian_perturb_epitope_secondary_components(
	Real const & translation_magnitude,
	Real const & rotation_magnitude,
	bool const & perturb_using_archived_jump,
	bool const & use_graft_info_settings
)
{
	using namespace pose_ns;

	Size const attempt_cutoff = 50;

	// grab epitope secondary components in current epitope scaffold numbering
	std::set< ResidueRange > secondaries = secondary_components();

	for ( std::map< ResidueRange, EpitopeTrack >::const_iterator e = epitope_tracking_.begin(), ee = epitope_tracking_.end(); e != ee; ++e ) {

		if ( e->second.jump_label > -1 && ( secondaries.find( e->first ) != secondaries.end() ) ) {

			if ( use_graft_info_settings && !graft_info_.perturbing_rb( epitope_to_old_gap( e->first ) ) ) {
				continue;
			}

			// grab loop closures for this epitope component
			std::set< LoopClosureInfo > closures = component_closures_to_attempt( e->first );

			// check to see if epitope component's loops are fully closed
			bool fully_closed = true;
			for ( std::set< LoopClosureInfo >::const_iterator c = closures.begin(), ce = closures.end(); c != ce; ++c ) {
					fully_closed = fully_closed && c->is_closed();
			}
			// check for override by do_graft_bb
			for ( std::set< LoopClosureInfo >::const_iterator c = closures.begin(), ce = closures.end(); c != ce; ++c ) {
					if ( !do_graft_bb( *c ) ) {
						fully_closed = true;
						break;
					}
			}

			// only perform rb move if possible
			if ( !fully_closed ) {

				// mark residues excluded from clash check
				utility::vector1< bool > check_residue( epitope_scaffold_.total_residue(), true );
				for ( std::set< LoopClosureInfo >::const_iterator c = closures.begin(), ce = closures.end(); c != ce; ++c ) {
					for ( std::set< Integer >::const_iterator r = c->moveable_residues().begin(), re = c->moveable_residues().end(); r != re; ++r ) {
						check_residue[ *r ] = false; // mark moveable residues as false
					}
				}
				for ( Integer r = e->first.begin(), re = e->first.end(); r <= re; ++r ) {
					check_residue[ r ] = false; // remove epitope component residue
				}

				// make octree for clash checking
				BoundingBox bb = bounding_box( epitope_scaffold_ );
				Octree< AtomPoint > oc( 6.0, bb.lower_corner(), bb.upper_corner() );
				fill_octree( oc, epitope_scaffold_, check_residue );

				// save jump in case can't find a good move
				Jump original_jump = epitope_scaffold_.get_jump( e->second.jump_label );

				Real clash = DESIGN_INFINITY;
				Size attempts = 0; // cycle cutoff in case it gets stuck
				while ( clash > 0.0 && attempts < attempt_cutoff ) {
					gaussian_perturb_epitope_rb_jump( e->first, translation_magnitude, rotation_magnitude, perturb_using_archived_jump );
					clash = octree_inter_rep( oc, epitope_scaffold_, epitope_scaffold_, e->first );
					++attempts;
				}

				// reset the jump to the prior one if it couldn't find a good
				// move
				if ( clash > 0.0 && attempts == attempt_cutoff ) {
					epitope_scaffold_.set_jump( e->second.jump_label, original_jump );
				}

			}
		}

	}

	// finally, recheck closure criteria
	recheck_closure_criteria();
}


/// @brief optimize Ab->epitope interaction by minimization using score12
/// @brief with option to include rigid body minimization, primarily for use in
/// @brief superposition protocol scaffolding
/// @details to be consistent for, the same style of atom pair
/// @details constraints are used regardless of rigid body on/off
/// @param[in] interface_distance distance cutoff for atoms to be considered at interface, default 4.5
/// @warning procedure incomplete, currently no chi minimize
void
EpitopeScaffold::Ab_epitope_optimize(
	bool const & chi_minimize,
	bool const & rb_minimize,
	Real const & interface_distance
)
{
	using cst_set_ns::Cst_set;
	using namespace pose_ns;

	if ( !Ab_is_connected_ ) {
		std::cerr << "EpitopeScaffold::Ab_epitope_optimize() : no Ab to minimize, skipping!" << std::endl;
		return;
	}

	if ( !epitope_scaffold_.fullatom() ) {
		std::cerr << "EpitopeScaffold::Ab_epitope_optimize() : epitope scaffold currently not full atom, refusing!" << std::endl;
		return;
	}

	std::cout << "* es Ab_epitope_optimize:" << std::endl;
	std::cout << "ddG before = " << ddG() << std::endl;

	// store coordinates for Ab transform
	FArray2D_float original_one_bb;
	extract_1bb_from_Pose( epitope_scaffold_, Ab_connection_.jump_stop(), original_one_bb );

	// set minimizer
	minimize_reset(); // everything off except phi-psi and rb
	minimize_set_vary_phipsi( false ); // not even phi-psi
	minimize_set_vary_chi( chi_minimize );

	// lock all jumps
	epitope_scaffold_.set_allow_jump_move( false );

	// open jump between epitope-scaffold and Ab if requested
	if ( rb_minimize ) {
		epitope_scaffold_.set_allow_jump_move( Ab_connection_.jump_label(), true );
	}

	// safety, lock all angles
	epitope_scaffold_.set_allow_bb_move( false );
	epitope_scaffold_.set_allow_chi_move( false );

	// open chi angles if requested
	if ( chi_minimize ) {
		std::set< Integer > interface_positions = identify_interface_residues( interface_distance );

		for ( std::set< Integer >::const_iterator i = interface_positions.begin(), ie = interface_positions.end(); i != ie; ++i ) {
			epitope_scaffold_.set_allow_chi_move( *i, true );
		}
	}

	// setup constraints so things don't drift too far
	Cst_set constraints;
	fill_cst_set( constraints, epitope_scaffold_, true ); // boolean: constrain by heavy atoms only
	epitope_scaffold_.set_constraints( constraints );

	// cycles
	Integer const lj_ramp_cycles = 8;

	// repulsive weight modifiers, ramps up during cycles
	Real const initial_rep = 0.3, final_rep = 1.0;
	Real const rep_delta = std::exp( std::log( final_rep / initial_rep ) / ( lj_ramp_cycles - 1 ) );
	Real rep_weight = initial_rep / rep_delta;

	// constraint weight modifiers, ramps down during cycles
	Real const initial_cst = 20.0, final_cst = 2.0;
	Real const cst_delta = std::exp( std::log( final_cst / initial_cst ) / ( lj_ramp_cycles - 1 ) );
	Real cst_weight = initial_cst / cst_delta;

	// minimization tolerance
	Real const initial_min_tol = 0.01, final_min_tol = 0.0000001;
	Real const min_tol_delta = std::exp( std::log( final_min_tol / initial_min_tol ) / ( lj_ramp_cycles - 1 ) );
	Real min_tol = initial_min_tol / min_tol_delta;

	// base scoring function
	Score_weight_map weight_map( score12 );
	weight_map.set_weight( ATOMPAIR_CST, 500.0 * initial_cst / constraints.count_atompair_constraints() );

	// safety, initial score
	epitope_scaffold_.score( weight_map );

	// minimize cycles
	for ( Integer i = 1; i <= lj_ramp_cycles; ++i ) {
		// weight changes
		rep_weight *= rep_delta;
		weight_map.set_weight( FA_REP, rep_weight );
		cst_weight *= cst_delta;
		weight_map.set_weight( ATOMPAIR_CST, 500.0 * cst_weight / constraints.count_atompair_constraints() );

		// minimizer changes
		min_tol *= min_tol_delta;
		minimize_set_tolerance( min_tol );

		// minimize
		epitope_scaffold_.main_minimize( weight_map, "dfpmin" );

		// show scores
		std::cout << "   cycle " << i << " : ";
		epitope_scaffold_.show_scores( std::cout );
		std::cout << std::endl;
	}

	// final score
	std::cout << "   final : ";
	epitope_scaffold_.show_scores( std::cout );
	std::cout << std::endl;

	// clear constraints pointer in epitope-scaffold
	epitope_scaffold_.reset_constraints();

	// re-lock all jumps
	epitope_scaffold_.set_allow_jump_move( false );

	// clear scores
	epitope_scaffold_.reset_score_data();

	// reset minimizer
	minimize_reset();

	// update Ab transform
	FArray2D_float new_one_bb;
	extract_1bb_from_Pose( epitope_scaffold_, Ab_connection_.jump_stop(), new_one_bb );
	update_Ab_transform( original_one_bb, new_one_bb );

	std::cout << "ddG after = " << ddG() << std::endl;
}


/// @brief optimize epitope internals to de-clash, primarily for
///  use when no antibody is present during superposition scaffolding
void
EpitopeScaffold::epitope_optimize()
{
	using cst_set_ns::Cst_set;
	using namespace pose_ns;

	std::cout << "*es_epitope_optimize: begin" << std::endl;

	// set minimizer
	minimize_reset(); // everything off except phi-psi and rb
	minimize_set_vary_phipsi( false ); // not even phi-psi
	minimize_set_vary_chi( true );
	minimize_set_tolerance( 0.0000001 );

	// safety, lock all angles
	epitope_scaffold_.set_allow_bb_move( false );
	epitope_scaffold_.set_allow_chi_move( false );

	// lock all jumps
	epitope_scaffold_.set_allow_jump_move( false );

	// open chi angles
	for ( std::map< ResidueRange, ResidueRange >::const_iterator i = native_to_new_epitope_.begin(), ie = native_to_new_epitope_.end(); i != ie; ++i ) {
		ResidueRange const & es_loop = i->second;

		for ( Integer res = es_loop.begin(); res <= es_loop.end(); ++res ) {
			epitope_scaffold_.set_allow_chi_move( res, true );
		}
	}

	// base scoring function
	Score_weight_map weight_map( score12 );

	// safety, initial score
	epitope_scaffold_.score( weight_map );

	// final score
	std::cout << "   initial : ";
	epitope_scaffold_.show_scores( std::cout );

	// minimize
	epitope_scaffold_.main_minimize( weight_map, "dfpmin" );

	// final score
	std::cout << "   final : ";
	epitope_scaffold_.show_scores( std::cout );

	// clear scores
	epitope_scaffold_.reset_score_data();

	// reset minimizer
	minimize_reset();

	std::cout << "*es_epitope_optimize: done" << std::endl;
}


/// @brief mark complementarity design positions by checking against
///        Ab residues indicated by given Ab (pdb) residue string of
///        format 'CHAINRESNUMINSERTIONCODE'
/// @note  uses archived Ab as reference for residue number indexing
void
EpitopeScaffold::mark_complementarity_Ab_positions(
	std::set< String > const & pdb_residues
)
{
	// construct map from residue string to epitope_scaffold_ Pose string
	std::ostringstream ss;
	std::map< String, Integer > pdb_to_pose;
	for ( Integer i = 1, ie = archived_Ab_.total_residue(); i <= ie; ++i ) {

		// construct full resid
		if ( archived_Ab_.pdb_info().res_chain( i ) != ' ' ) {
			ss << archived_Ab_.pdb_info().res_chain( i );
		}
		ss << archived_Ab_.pdb_info().pdb_res_num( i );
		if ( archived_Ab_.pdb_info().pdb_insert_let( i ) != ' ' ) {
			ss << archived_Ab_.pdb_info().pdb_insert_let( i );
		}

		pdb_to_pose[ ss.str() ] = i;

		ss.str( "" ); // clear stringstream
	}

	assert( pdb_to_pose.size() == static_cast< Size >( archived_Ab_.total_residue() ) ); // make sure nothing weird happened

	// compute epitope scaffold offset
	Integer const offset = Ab_is_connected() ? epitope_scaffold_.total_residue() - archived_Ab_.total_residue() : epitope_scaffold_.total_residue();

	// construct ab residues
	std::set< Integer > ab_residues;
	for ( std::set< String >::const_iterator i = pdb_residues.begin(), ie = pdb_residues.end(); i != ie; ++i ) {
		ab_residues.insert( pdb_to_pose.find( *i )->second + offset );
	}

	// store
	complementarity_Ab_positions_ = ab_residues;
}


/// @brief  find complementarity design positions by distance check against
///         Ab residues
std::set< Integer >
EpitopeScaffold::find_complementarity_design_positions(
	std::set< Integer > const & ab_residues
)
{
	if ( !Ab_is_connected() ) {
		return std::set< Integer >();
	}

	// scaffold range
	ResidueRange es_range( 1, Ab_connection_.segment_range().begin() - 1 );

	// make octree containing epitope-scaffold sidechains
	Real max_distance_cutoff = std::max( static_cast< Real >( 6.0 ), complementarity_design_position_cutoff_ );
	BoundingBox bb = bounding_box( epitope_scaffold_, es_range );
	Octree< AtomPoint > es_octree_sc( max_distance_cutoff, bb.lower_corner(), bb.upper_corner() );
	fill_octree_sc( es_octree_sc, epitope_scaffold_, es_range );

	std::set< Integer > c_residues; // set of complementarity residues
	for ( std::set< Integer >::const_iterator i = ab_residues.begin(), ie = ab_residues.end(); i != ie; ++i ) {
		std::set< Integer > nr = epigraft::design::near_residues( complementarity_design_position_cutoff_, es_octree_sc, epitope_scaffold_, *i );

		c_residues.insert( nr.begin(), nr.end() );
	}

	std::set< Integer > complementarity_design_positions;
	for ( std::set< Integer >::const_iterator i = c_residues.begin(), ie = c_residues.end(); i != ie; ++i ) {
		complementarity_design_positions.insert( *i );
	}

	return complementarity_design_positions;
}


/// @brief  do complementarity design
/// @return returns false if Ab is not connected or if there is nothing to do,
///         otherwise true
bool
EpitopeScaffold::complementarity_design(
	bool const & allow_rb_min
)
{
	if ( !Ab_is_connected() || complementarity_Ab_positions_.size() == 0 ) {
		return false;
	}

	std::cout << "* es complementarity_design start" << std::endl;

	// clear design positions
	complementarity_design_positions_.clear();

	for ( Integer cycle = 1; cycle <= complementarity_design_cycles_; ++cycle ) {
		// find complementarity design positions
		std::set< Integer > cdp = find_complementarity_design_positions( complementarity_Ab_positions_ );

		// store in object's cache, we continually insert in case an rb move
		// somehow manages to shift enough so that a prior design position is
		// not found by distance cutoff later on
		complementarity_design_positions_.insert( cdp.begin(), cdp.end() );

		// make design matrix
		FArray2D< bool > design_matrix( param::MAX_AA(), epitope_scaffold_.total_residue(), false );

		// design matrix: scaffold positions
		for ( std::set< Integer >::const_iterator i = complementarity_design_positions_.begin(), ie = complementarity_design_positions_.end(); i != ie; ++i ) {
			for ( Integer a = 1; a <= 20; ++a ) {
				design_matrix( a, *i ) = true;
			}

			// no cys or pro
			design_matrix( param_aa::aa_cys, *i ) = false;
			design_matrix( param_aa::aa_pro, *i ) = false;
		}

		// design matrix: Ab positions, allow repack
		for ( std::set< Integer >::const_iterator i = complementarity_Ab_positions_.begin(), ie = complementarity_Ab_positions_.end(); i != ie; ++i ) {
			design_matrix( epitope_scaffold_.res( *i ), *i ) = true;
		}

		// do design
		design( design_matrix );

		// antibody-scaffold optimize scmin + rbmin
		Ab_epitope_optimize( true, allow_rb_min, std::max( complementarity_design_position_cutoff_, Ab_repack_cutoff_ ) ); // boolean: chi_minimize
	}

	std::cout << "* es complementarity_design end" << std::endl;

	return true;
}



/// @brief seal fold tree in working epitope scaffold
/// @warning use with care!  remember to unseal fold tree before attempting any subsequence
/// @warning loop closure operations or anything which requires a chainbreak
void
EpitopeScaffold::seal_fold_tree()
{
	// we don't directly set the fold tree because we don't want perturbations
	// due to refold or if, for example, broken chains still remain
	Pose es_complex;
	extract_complex( es_complex );
	epitope_scaffold_ = es_complex;
}


/// @brief pull fold tree from archived epitope scaffold for use in working epitope scaffold
/// @warning use with care!
void
EpitopeScaffold::unseal_fold_tree()
{
	Pose es;

	if ( Ab_is_connected_ ) {
		archive_Ab_sidechains();

		// make fake ConnectionResult with jump label = 1 for ease of disconnecting Ab
		ConnectionResult faked_connect( Ab_connection_.jump_offset(), Ab_connection_.residue_offset(),
		                                Ab_connection_.jump_start(), Ab_connection_.jump_stop(),
		                                1, Ab_connection_.segment_range() );

		// disconnect Ab
		PoseAssembly::disconnect( epitope_scaffold_, faked_connect, es, idealize_loop_geometry_ );
	} else {
		es = epitope_scaffold_;
	}

	// make epitope scaffold with unsealed fold tree pulled from archive
	Pose unsealed_es;
	unsealed_es.set_fold_tree( archived_epitope_scaffold_.fold_tree() );
	unsealed_es.set_fullatom_flag( true, false ); // force booleans: fullatom, repack

	transfer_identity_info_between_Poses( es, ResidueRange( 1, es.total_residue() ), unsealed_es, 1, true ); // boolean: keep_pdb_info
	unsealed_es.set_coords( false, es.Eposition(), es.full_coord(), false ); // boolean: ideal, check missing

	unsealed_es.pdb_info().set_use_pdb_numbering( true );

	epitope_scaffold_ = unsealed_es;

	// reconnect Ab
	if ( Ab_is_connected_ ) {
		Ab_is_connected_ = false;
		connect_Ab();
	}
}


/// @brief forcibly save state of Ab sidechains
bool
EpitopeScaffold::archive_Ab_sidechains()
{
	if ( Ab_is_connected_ ) {
		// manually archive antibody sidechains
		FArray3D_float const & es_full_coord = epitope_scaffold_.full_coord(); // cache
		for ( Integer res = Ab_connection_.segment_range().begin(), last_res = Ab_connection_.segment_range().end(); res <= last_res; ++res ) {
			archived_Ab_.copy_sidechain( res - Ab_connection_.residue_offset(), epitope_scaffold_.res( res ), epitope_scaffold_.res_variant( res ), es_full_coord( 1, 1, res ) );
		}
	}

	return Ab_is_connected_;
}


/// @brief force save state of epitope-scaffold sidechains that have same identity
/// @brief in both working and archive epitope scaffold
bool
EpitopeScaffold::archive_identity_equivalent_scaffold_sidechains()
{
	if ( epitope_scaffold_.fullatom() ) {
		ResidueRange es_range( 1, epitope_scaffold_.total_residue() );
		if ( Ab_is_connected_ ) {
			es_range = ResidueRange( 1, Ab_connection_.segment_range().begin() - 1 );
		}

		FArray3D_float const & es_full_coord = epitope_scaffold_.full_coord(); // cache
		for ( Integer res = es_range.begin(), last_res = es_range.end(); res <= last_res; ++res ) {
			if ( epitope_scaffold_.res( res ) == archived_epitope_scaffold_.res( res ) ) {
				archived_epitope_scaffold_.copy_sidechain( res, epitope_scaffold_.res( res ), epitope_scaffold_.res_variant( res ), es_full_coord( 1, 1, res ) );
			}
		}
	}

	return epitope_scaffold_.fullatom();
}


/// @brief force save state of epitope-scaffold and Ab sidechains that were used in
/// @brief complementarity design
bool
EpitopeScaffold::archive_all_complementarity_design_sidechains()
{
	if ( !epitope_scaffold_.fullatom() ){
		return false;
	}

	FArray3D_float const & es_full_coord = epitope_scaffold_.full_coord(); // cache

	for ( std::set< Integer >::const_iterator i = complementarity_design_positions_.begin(), ie = complementarity_design_positions_.end(); i != ie; ++i ) {
		Integer const & res = *i;
		archived_epitope_scaffold_.copy_sidechain( res, epitope_scaffold_.res( res ), epitope_scaffold_.res_variant( res ), es_full_coord( 1, 1, res ) );
	}

	if ( Ab_is_connected() ) {
		for ( std::set< Integer >::const_iterator i = complementarity_Ab_positions_.begin(), ie = complementarity_Ab_positions_.end(); i != ie; ++i ) {
			Integer const & res = *i;
			archived_Ab_.copy_sidechain( res - Ab_connection_.residue_offset(), epitope_scaffold_.res( res ), epitope_scaffold_.res_variant( res ), es_full_coord( 1, 1, res ) );
		}
	}

	return true;
}


/// @brief replace both working and archived epitope scaffold Pose
void
EpitopeScaffold::replace_structure(
	Pose const & p,
	bool const & also_replace_archive
)
{
	using pose_ns::Fold_tree;

//	Fold_tree es_f = epitope_scaffold_.fold_tree();
//	epitope_scaffold_ = p;
//	epitope_scaffold_.set_fold_tree( es_f );

	transfer_identity_info_between_Poses( p, ResidueRange( 1, epitope_scaffold_.total_residue() ), epitope_scaffold_, 1, true ); // boolean: keep_pdb_info
	epitope_scaffold_.set_coords( false, p.Eposition(), p.full_coord(), false ); // boolean: ideal, check missing


	// archived epitope scaffold needs to be handled completely separately
	// in case the fold tree happens to be different
	if ( also_replace_archive ) {
		Fold_tree a_es_f = archived_epitope_scaffold_.fold_tree();
		archived_epitope_scaffold_ = p;
		archived_epitope_scaffold_.set_fold_tree( a_es_f );
	}

	for ( std::set< LoopClosureInfo >::const_iterator c = closures_to_attempt_.begin(), ce = closures_to_attempt_.end(); c != ce; ++c ) {
		LoopClosureInfo const & closure_info = *c;

		if ( do_graft_bb( closure_info ) ) {
			Integer const & cut = closure_info.cut();
			Integer const cut_plus_one = cut + 1;

			// need to idealize peptide bond in case it's actually stretched and supposed to be broken
			idealize_cutpoint_retaining_psiphi( epitope_scaffold_, cut, epitope_scaffold_.psi( cut ), epitope_scaffold_.phi( cut_plus_one ) );
//			idealize_cutpoint_retaining_psiphi( epitope_scaffold_, cut, p.psi( cut ), p.phi( cut_plus_one ) );
//			epitope_scaffold_.set_phi( cut, p.phi( cut ) ); // safety
//			epitope_scaffold_.set_psi( cut_plus_one, p.psi( cut_plus_one ) ); // safety

			if ( also_replace_archive ) {
				// need to idealize peptide bond in case it's actually stretched and supposed to be broken
				idealize_cutpoint_retaining_psiphi( archived_epitope_scaffold_, cut, p.psi( cut ), p.phi( cut_plus_one ) );
				archived_epitope_scaffold_.set_phi( cut, p.phi( cut ) ); // safety
				archived_epitope_scaffold_.set_psi( cut_plus_one, p.psi( cut_plus_one ) ); // safety
			}
		}
	}

	recheck_closure_criteria(); // need to recheck closure criteria because coordinates have changed
}


/// @brief refresh cached design positions
/// @details requires antibody to be connected and epitope scaffold to be full atom, otherwise returns false
bool
EpitopeScaffold::refresh_cached_design_positions()
{
	if ( !Ab_is_connected() || !epitope_scaffold_.fullatom() ) {
		return false;
	}

	if ( complementarity_Ab_positions_.size() > 0 ) {
		complementarity_design_positions_ = find_complementarity_design_positions( complementarity_Ab_positions_ );
	}
	make_design_info();

	return true;
}


/// @brief alter moveable residues in closure sites to desired secondary structure
/// @param[in] ss_type secondary structure type, valid characters are 'H', 'E', 'L', and 'D'
void
EpitopeScaffold::alter_secondary_structure_of_moveable_residues(
	char const & ss_type
)
{
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt_.begin(), ie = closures_to_attempt_.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;

		for ( std::set< Integer >::const_iterator r = closure_info.moveable_residues().begin(), re = closure_info.moveable_residues().end(); r != re; ++r ) {
			epitope_scaffold_.set_secstruct( *r, ss_type );
		}
	}
}


/// @brief set proper moveable residues and grow/remove residues at closure sites
/// @brief based on tether/linker/moveable information in internal GraftInfo object
/// @note  If moving all epitope residues (see GraftInfo) then jump residue is always
/// @note  for consistency
/// @note  directly stores two copies of the epitope scaffold (working and archived)
void
EpitopeScaffold::alter_closure_sites()
{
	using epigraft::match::align::AlignmentSystem;

	// disconnect Ab first if present and reconnect later
	bool const Ab_connection_state = Ab_is_connected_;
	if ( Ab_is_connected_ ) {
		// disconnect
		disconnect_Ab();
	}

	// setup assembly object
	PoseAssembly assembly( epitope_scaffold_.total_residue() );

	// make set of (sorted) closures to run through
	std::set< LoopClosureInfo > closures = closures_to_attempt();
	ResidueRange p_component = primary_component(); // copy of primary component for detection later on
	AlignmentSystem::SystemType const & align_system = graft_info_.match_result().system_type;

	// run through all closures to setup PoseAssembly
	for ( std::set< LoopClosureInfo >::const_iterator c = closures.begin(), ce = closures.end(); c != ce; ++c ) {
		LoopClosureInfo const & closure = *c;

		// figure out which component we're looking at and the corresponding old scaffold gap range
		ResidueRange const component = component_from_closure( closure );
		ResidueRange const & old_scaffold_gap = epitope_to_old_gap( component ); // used for indexing with graft_info_

		if ( closure.cut() == component.begin() - 1 ) { // break at n-terminus

			Integer const & s_linker_n = graft_info_.scaffold_linker_residues_nterm( old_scaffold_gap );
			Integer const & e_linker_n = graft_info_.epitope_linker_residues_nterm( old_scaffold_gap );

			// instructions for assembly on scaffold side
			if ( s_linker_n > 0 ) { // insertion
				assembly.insert_right( closure.cut(), s_linker_n, false, grow_ss_type_, grow_residue_type_ ); // boolean: break_on_insertion
			} else if ( s_linker_n < 0 ) { // deletion
				assembly.remove( ResidueRange( closure.cut() + s_linker_n + 1, closure.cut() ) );
			}

			// instructions for assembly on epitope side
			assert( e_linker_n >= 0 );
			if ( e_linker_n > 0 ) { // insertion
				assembly.insert_left( closure.cut() + 1, e_linker_n, false, grow_ss_type_, grow_residue_type_ ); // boolean: break_on_insertion
			}

			// check to see if c-terminus of loop contains component (i.e. c-terminus is a takeoff)
			if ( closure.loop_range().contains( component.end() ) ) {
				Integer const & e_linker_c = graft_info_.epitope_linker_residues_cterm( old_scaffold_gap );
				Integer const & s_tether_c = graft_info_.scaffold_tether_residues_cterm( old_scaffold_gap );
				Integer const & s_linker_c = graft_info_.scaffold_linker_residues_cterm( old_scaffold_gap );

				// instructions for assembly on epitope side
				assert ( e_linker_c >= 0 );
				if ( e_linker_c > 0 ) { // insertion
					assembly.insert_right( closure.loop_range().end() - s_tether_c, e_linker_c, false, grow_ss_type_, grow_residue_type_ ); // boolean: break_on_insertion
				}

				// instructions for assembly on scaffold side
				if ( s_linker_c > 0 ) { // insertion
					assembly.insert_left( closure.loop_range().end() - s_tether_c + 1, s_linker_c, false, grow_ss_type_, grow_residue_type_ ); // boolean: break_on_insertion
				} else if ( s_linker_c < 0 ) { // deletion
					assembly.remove( ResidueRange( closure.loop_range().end() - s_tether_c + 1, closure.loop_range().end() - s_tether_c - s_linker_c ) );
				}
			}

		} else { // break at c-terminus ( closure.cut() == component.end() )

			// check to see if n-terminus of loop contains component (i.e. n-terminus is a takeoff)
			if ( closure.loop_range().contains( component.begin() ) ) {
				Integer const & s_tether_n = graft_info_.scaffold_tether_residues_nterm( old_scaffold_gap );
				Integer const & s_linker_n = graft_info_.scaffold_linker_residues_nterm( old_scaffold_gap );
				Integer const & e_linker_n = graft_info_.epitope_linker_residues_nterm( old_scaffold_gap );

				// instructions for assembly on scaffold side
				if ( s_linker_n > 0 ) { // insertion
					assembly.insert_right( closure.loop_range().begin() + s_tether_n - 1, s_linker_n, false, grow_ss_type_, grow_residue_type_ ); // boolean: break_on_insertion
				} else if ( s_linker_n < 0 ) { // deletion
					assembly.remove( ResidueRange( closure.loop_range().begin() + s_tether_n + s_linker_n, closure.loop_range().begin() + s_tether_n - 1 ) );
				}

				// instructions for assembly on epitope side
				assert( e_linker_n >= 0 );
				if ( e_linker_n > 0 ) { // insertion
					assembly.insert_left( closure.loop_range().begin() + s_tether_n, e_linker_n, false, grow_ss_type_, grow_residue_type_ ); // boolean: break_on_insertion
				}
			}

			Integer const & e_linker_c = graft_info_.epitope_linker_residues_cterm( old_scaffold_gap );
			Integer const & s_linker_c = graft_info_.scaffold_linker_residues_cterm( old_scaffold_gap );

			// instructions for assembly on epitope side
			assert( e_linker_c >= 0);
			if ( e_linker_c > 0 ) { // insertion
				assembly.insert_right( closure.cut(), e_linker_c, false, grow_ss_type_, grow_residue_type_ ); // boolean: break_on_insertion
			}

			// instructions for assembly on scaffold side
			if ( s_linker_c > 0 ) { // insertion
				assembly.insert_left( closure.cut() + 1, s_linker_c, false, grow_ss_type_, grow_residue_type_ ); // boolean: break_on_insertion
			} else if ( s_linker_c < 0 ) { // deletion
				assembly.remove( ResidueRange( closure.cut() + 1, closure.cut() - s_linker_c ) );
			}
		}

	} // end foreach closure

	// grab old to new residues map
	std::map< Integer, Integer > old_to_new = assembly.map_old_to_new();
	std::map< Integer, Integer > new_to_old = assembly.map_new_to_old();
	std::map< Integer, Integer > cut_old_to_new = assembly.map_cut_old_to_new( epitope_scaffold_.fold_tree() );

	// assemble new epitope scaffold
	Pose altered_epitope_scaffold;
	assembly.apply( epitope_scaffold_, altered_epitope_scaffold, idealize_loop_geometry_ );

	// set internal Poses
	epitope_scaffold_ = altered_epitope_scaffold;
	archived_epitope_scaffold_ = altered_epitope_scaffold;

	// mark grown residues
	grown_residues_.clear();
	for ( Integer i = 1, ie = epitope_scaffold_.total_residue(); i <= ie; ++i ) {
		assert( new_to_old.find( i ) != new_to_old.end() );
		Integer const & old_res = new_to_old.find( i )->second;
		if ( old_res == -1 ) {
			grown_residues_.insert( i );
		}
	}

	// here we need to modify the old -> new map so that deleted endpoints of the
	// loop closure range can be re-found and numbered properly
	for ( std::set< LoopClosureInfo >::const_iterator c = closures.begin(), ce = closures.end(); c != ce; ++c ) {
		LoopClosureInfo const & closure = *c;

		Integer begin = closure.loop_range().begin();
		Integer end = closure.loop_range().end();

		if ( old_to_new.find( begin )->second == -1 ) {
			while ( old_to_new.find( begin )->second == -1 ) {
				--begin;
			}

			old_to_new[ closure.loop_range().begin() ] = begin + 1;
		}

		if ( old_to_new.find( end )->second == -1 ) {
			while ( old_to_new.find( end )->second == -1 ) {
				++end;
			}

			old_to_new[ closure.loop_range().end() ] = end - 1;
		}
	}

	// new altered closure data
	std::set< LoopClosureInfo > altered_closures;
	std::set< Integer > altered_arm_residues;
	for ( std::set< LoopClosureInfo >::const_iterator c = closures.begin(), ce = closures.end(); c != ce; ++c ) {
		LoopClosureInfo const & closure = *c;

		// initialize altered closure
		LoopClosureInfo altered_closure = closure;

		// figure out which component we're looking at and the corresponding old scaffold gap range
		ResidueRange const component = component_from_closure( closure );
		ResidueRange const & old_scaffold_gap = epitope_to_old_gap( component ); // used for indexing with graft_info_
		Integer const & e_tether_n = graft_info_.epitope_tether_residues_nterm( old_scaffold_gap );
		Integer const & e_tether_c = graft_info_.epitope_tether_residues_cterm( old_scaffold_gap );

		if ( component == p_component
		     && ( align_system == AlignmentSystem::N2C || align_system == AlignmentSystem::C2N ) ) {

			// single break alignment system derived loop; there is only one case to handle
			altered_closure.set_loop_range( ResidueRange( old_to_new.find( closure.loop_range().begin() )->second,
			                                              old_to_new.find( closure.loop_range().end() )->second ) );

		} else { // double break derived loop

			// for the new loop, there are two primary cases:
			//  - loop length is squished to one due to some combination where no residues
			//    from the original scaffold are begin used (predesign only depends on scaffold tether)
			//  - loop length is > 0, in which case all indexing follows as it's supposed to
			if ( closure.loop_range().length() == 1 ) {
				if ( closure.cut() < closure.loop_range().begin() ) { // n-side relative to epitope component, so drag left endpoint
					altered_closure.set_loop_range( ResidueRange( old_to_new.find( closure.loop_range().begin() - 1 )->second,
					                                              old_to_new.find( closure.loop_range().end() + e_tether_n )->second ) );
				} else if ( closure.cut() == closure.loop_range().begin() ) { // c-side relative to epitope component, so drag right endpoint
					altered_closure.set_loop_range( ResidueRange( old_to_new.find( closure.loop_range().begin() - e_tether_c )->second,
					                                              old_to_new.find( closure.loop_range().end() + 1 )->second ) );
				}
			} else {
				if ( closure.loop_range().end() == closure.cut() + 1 ) { // n-side relative to epitope component
					altered_closure.set_loop_range( ResidueRange( old_to_new.find( closure.loop_range().begin() )->second,
					                                              old_to_new.find( closure.loop_range().end() + e_tether_n )->second ) );
				} else if ( closure.loop_range().begin() == closure.cut() ) { // c-side relative to epitope component
					altered_closure.set_loop_range( ResidueRange( old_to_new.find( closure.loop_range().begin() - e_tether_c )->second,
					                                              old_to_new.find( closure.loop_range().end() )->second ) );
				}
			}

		}

		altered_closure.set_cut( cut_old_to_new.find( closure.cut() )->second );

		// clear moveable residues for setup
		altered_closure.clear_moveable_residues();

		// insert proper moveable residues
		if ( closure.cut() == component.begin() - 1 ) { // break at n-terminus

			Integer const & s_moveable_n = graft_info_.scaffold_moveable_residues_nterm( old_scaffold_gap );
			Integer const & s_linker_n = graft_info_.scaffold_linker_residues_nterm( old_scaffold_gap );
			Integer const & e_moveable_n = graft_info_.epitope_moveable_residues_nterm( old_scaffold_gap );
			Integer const & e_linker_n = graft_info_.epitope_linker_residues_nterm( old_scaffold_gap );

			altered_closure.add_moveable_residues_adjacent_to_cut( s_moveable_n, e_moveable_n );

			// if residues have been grown, these residues must always be moveable
			if ( s_linker_n > 0 ) {
				altered_closure.set_always_moveable_adjacent_to_cut( s_linker_n, 0, true );
			}
			if ( e_linker_n > 0 ) {
				altered_closure.set_always_moveable_adjacent_to_cut( 0, e_linker_n, true );
			}

			// check to see if c-terminus of loop contains component (i.e. c-terminus is a takeoff)
			if ( closure.loop_range().contains( component.end() ) ) {
				Integer const & s_tether_c = graft_info_.scaffold_tether_residues_cterm( old_scaffold_gap );
				Integer const & s_linker_c = graft_info_.scaffold_linker_residues_cterm( old_scaffold_gap );
				Integer const & s_moveable_c = graft_info_.scaffold_moveable_residues_cterm( old_scaffold_gap );
				Integer const & e_tether_c = graft_info_.epitope_tether_residues_cterm( old_scaffold_gap );
				Integer const & e_linker_c = graft_info_.epitope_linker_residues_cterm( old_scaffold_gap );
				Integer const & e_moveable_c = graft_info_.epitope_moveable_residues_cterm( old_scaffold_gap );

				Integer const moveable_start = altered_closure.loop_range().end() - ( s_tether_c + s_linker_c + e_tether_c + e_linker_c ) + 1;
				if ( s_moveable_c + e_moveable_c > 0 ) {
					ResidueRange const lever = ResidueRange( moveable_start, moveable_start + s_moveable_c + e_moveable_c - 1 );
					altered_closure.add_moveable_residues( lever );
					for ( Integer l = lever.left(), le = lever.right(); l <= le; ++l ) {
						altered_arm_residues.insert( l );
					}
				}

				// if residues have been grown, these residues must always be moveable
				if ( e_linker_c > 0 ) {
					altered_closure.set_always_moveable( ResidueRange( moveable_start + e_tether_c, moveable_start + e_tether_c + e_linker_c - 1  ), true );
				}
				if ( s_linker_c > 0 ) {
					altered_closure.set_always_moveable( ResidueRange( moveable_start + e_tether_c + e_linker_c, moveable_start + e_tether_c + e_linker_c + s_linker_c - 1 ), true );
				}
			}

		} else { // break at c-terminus ( closure.cut() == component.end() )

			// check to see if n-terminus of loop contains component (i.e. n-terminus is a takeoff)
			if ( closure.loop_range().contains( component.begin() ) ) {
				Integer const & s_tether_n = graft_info_.scaffold_tether_residues_nterm( old_scaffold_gap );
				Integer const & s_linker_n = graft_info_.scaffold_linker_residues_nterm( old_scaffold_gap );
				Integer const & s_moveable_n = graft_info_.scaffold_moveable_residues_nterm( old_scaffold_gap );
				Integer const & e_tether_n = graft_info_.epitope_tether_residues_nterm( old_scaffold_gap );
				Integer const & e_linker_n = graft_info_.epitope_linker_residues_nterm( old_scaffold_gap );
				Integer const & e_moveable_n = graft_info_.epitope_moveable_residues_nterm( old_scaffold_gap );

				Integer const moveable_end = altered_closure.loop_range().begin() + ( s_tether_n + s_linker_n + e_tether_n + e_linker_n ) - 1;
				if ( s_moveable_n + e_moveable_n > 0 ) {
					ResidueRange const lever = ResidueRange( moveable_end - s_moveable_n - e_moveable_n + 1, moveable_end );
					altered_closure.add_moveable_residues( lever );
					for ( Integer l = lever.left(), le = lever.right(); l <= le; ++l ) {
						altered_arm_residues.insert( l );
					}
				}

				// if residues have been grown, these residues must always be moveable
				if ( e_linker_n > 0 ) {
					altered_closure.set_always_moveable( ResidueRange( moveable_end - e_tether_n - e_linker_n + 1, moveable_end - e_tether_n ), true );
				}
				if ( s_linker_n > 0 ) {
					altered_closure.set_always_moveable( ResidueRange( moveable_end - e_tether_n - e_linker_n - s_linker_n + 1, moveable_end - e_tether_n - e_linker_n ), true );
				}

			}

			Integer const & s_moveable_c = graft_info_.scaffold_moveable_residues_cterm( old_scaffold_gap );
			Integer const & s_linker_c = graft_info_.scaffold_linker_residues_cterm( old_scaffold_gap );
			Integer const & e_moveable_c = graft_info_.epitope_moveable_residues_cterm( old_scaffold_gap );
			Integer const & e_linker_c = graft_info_.epitope_linker_residues_cterm( old_scaffold_gap );
			altered_closure.add_moveable_residues_adjacent_to_cut( e_moveable_c, s_moveable_c );

			// if residues have been grown, these residues must always be moveable
			if ( e_linker_c > 0 ) {
				altered_closure.set_always_moveable_adjacent_to_cut( e_linker_c, 0, true );
			}
			if ( s_linker_c > 0 ) {
				altered_closure.set_always_moveable_adjacent_to_cut( 0, s_linker_c, true );
			}
		}

		// error checking
		if ( ( !altered_closure.loop_range().contains( altered_closure.cut() ) ) &&
		     ( ( altered_closure.cut() + 1 ) != altered_closure.loop_range().begin() ) ) {
			utility::exit( __FILE__, __LINE__, "ERROR: EpitopeScaffold::alter_closure_sites(), position of cut within altered loop range is invalid!\n" + altered_closure.to_string() );
		}

		// add new altered closure
		altered_closures.insert( altered_closure );
	}

	// vds: now remap "residues_originally_near_moveable_" to new/altered residue numbering
	std::set< Integer > renumbered_residues_originally_near_moveable;
	for ( std::set< Integer >::const_iterator i = residues_originally_near_moveable_.begin(), ie = residues_originally_near_moveable_.end(); i != ie; ++i ) {
		Integer const & new_number = old_to_new.find( *i )->second;
		if ( new_number > 0 ) {
			renumbered_residues_originally_near_moveable.insert( new_number );
		}
	}

	residues_originally_near_moveable_ = renumbered_residues_originally_near_moveable;

	// update all closure and map data
	alter_maps( old_to_new );
	closures_to_attempt_ = altered_closures;
	arm_residues_ = altered_arm_residues;

	// now go through and merge breaks if specified by graft info
	switch_secondaries_to_single_break();

	// override secondary structure of closure sites according to graft info
	override_closure_site_secondary_structure();

	// inactivate mobility of any grown residues according to graft info
	override_closure_site_grown_moveable_residues();

	// rescore closures _after_ updates are complete, because need to query do_graft_bb() function
	for ( std::set< LoopClosureInfo >::const_iterator c = closures_to_attempt_.begin(), ce = closures_to_attempt_.end(); c != ce; ++c ) {
		LoopClosureInfo const & closure_info = *c;

		// score and set closure status for potentially altered closure sites
		if ( do_graft_bb( closure_info ) ) {
			closure_info.set_chainbreak_score( score_cut_in_Pose_linear( epitope_scaffold_, closure_info.cut(), 1 ) ); // overlap = 1
			score_rama_for_moveable( epitope_scaffold_, closure_info );
			closure_info.set_is_closed( closure_info.chainbreak_score() <= chainbreak_criterion_ &&
					                    closure_info.ramas_are_not_above( local_rama_criterion_ ) );
		} else {
			closure_info.set_is_closed( true );
		}
	}

	// reconnect Ab if present
	if ( Ab_connection_state ) {
		connect_Ab();
	}
}


/// @brief randomize phi-psi of moveable residues around broken closure sites
/// @param[in] also_randomize_lever  if N2C/C2N, then also randomize residues
///  further from the cutpoint that induce lever moves
/// @param[in] also_randomize_archive induce same randomization to archive?
/// @param[in] force randomize all closure sites except for those marked
///   artificially closed
void
EpitopeScaffold::randomize_broken_closure_site_phipsi(
	bool const & also_randomize_lever,
	bool const & also_randomize_archive,
	bool const & force
)
{
	using epigraft::match::align::AlignmentSystem;

	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt_.begin(), ie = closures_to_attempt_.end(); i != ie; ++i ) {
		LoopClosureInfo c( *i );

		bool do_randomize = force ? !c.is_artificially_closed() : !c.is_closed();
		if ( closed_during_trajectory( c ) ) {
			do_randomize = false;
		}

		if ( do_randomize && do_graft_bb( c ) && c.moveable_residues().size() > 0 ) {
			// make a copy of the moveable residues for this closure, we'll need to
			// modify this if we're dealing with a primary component that's N2C or C2N
			std::set< Integer > moveable_residues = c.moveable_residues();

			// figure out which component we're looking at and the corresponding old scaffold gap range
			ResidueRange const component = component_from_closure( c );
			ResidueRange const & old_scaffold_gap = epitope_to_old_gap( component ); // used for indexing with graft_info_
			AlignmentSystem::SystemType const & align_system = graft_info_.match_result().system_type;

			// modify moveable residues if we're dealing with a primary component
			// and we're not randomizing lever residues
			if ( !also_randomize_lever ) {
				if ( component == primary_component() ) {
					if ( align_system == AlignmentSystem::N2C ) {
						Integer left_moveable = graft_info_.scaffold_moveable_residues_nterm( old_scaffold_gap ) +
						                         graft_info_.epitope_moveable_residues_nterm( old_scaffold_gap );

						if ( left_moveable > 0 ) { // remove residues on the left
							std::set< Integer >::iterator left_start = moveable_residues.begin();
							std::set< Integer >::iterator left_end = moveable_residues.begin();

							for ( Integer lm = 0; lm < left_moveable; ++lm ) {
								++left_end;
							}

							moveable_residues.erase( left_start, left_end );
						}
					} else if ( align_system == AlignmentSystem::C2N ) {
						Integer right_moveable = graft_info_.scaffold_moveable_residues_cterm( old_scaffold_gap ) +
						                         graft_info_.epitope_moveable_residues_cterm( old_scaffold_gap );

						if ( right_moveable > 0 ) { // remove residues on the right
							std::set< Integer >::iterator right_start = moveable_residues.end();
							std::set< Integer >::iterator right_end = moveable_residues.end();

							for ( Integer rm = 0; rm < right_moveable; ++rm ) {
								--right_start;
							}

							moveable_residues.erase( right_start, right_end );
						}
					}
				}
			}

			for ( std::set< Integer >::const_iterator r = moveable_residues.begin(), re = moveable_residues.end(); r != re; ++r ) {
				Real new_phi = ran3() * 360.0;
				Real new_psi = ran3() * 360.0;

				if ( new_phi > 180.0 ) {
					new_phi -= -360.0;
				}
				if ( new_psi > 180.0 ) {
					new_psi -= -360.0;
				}

				epitope_scaffold_.set_phi( *r, new_phi );
				epitope_scaffold_.set_psi( *r, new_psi );

				if ( also_randomize_archive ) {
					archived_epitope_scaffold_.set_phi( *r, new_phi );
					archived_epitope_scaffold_.set_psi( *r, new_psi );
				}
			}
		}
	}

	recheck_closure_criteria();

}


/// @brief randomize cutpoints of broken closure sites
/// @details For each closure site, takes into account the current moveable residues
/// @details and randomly place the cutpoint within them.  Fold tree is altered.
/// @warning using this with randomize_broken_closure_site_movable_residues() can have
/// @warning unintended consequences, since grown residues could end up being held fixed
void
EpitopeScaffold::randomize_broken_closure_site_cutpoints(
		bool const & also_randomize_archive
)
{
	using pose_ns::Fold_tree;
	using epigraft::match::align::AlignmentSystem;

	Fold_tree ft = epitope_scaffold_.fold_tree();
	Fold_tree aft = archived_epitope_scaffold_.fold_tree();

	std::set< LoopClosureInfo > altered_closures;
	ResidueRange old_left, old_right; // scratch variables
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt_.begin(), ie = closures_to_attempt_.end(); i != ie; ++i ) {
		LoopClosureInfo c( *i );

		if ( !closed_during_trajectory( c ) && !c.is_closed() && do_graft_bb( c ) && c.moveable_residues().size() > 0 ) {

			// make a copy of the moveable residues for this closure, we'll need to
			// modify this if we're dealing with a primary component that's N2C or C2N
			std::set< Integer > moveable_residues = c.moveable_residues();

			// figure out which component we're looking at and the corresponding old scaffold gap range
			ResidueRange const component = component_from_closure( c );
			ResidueRange const & old_scaffold_gap = epitope_to_old_gap( component ); // used for indexing with graft_info_
			AlignmentSystem::SystemType const & align_system = graft_info_.match_result().system_type;

			// modify moveable residues if we're dealing with a primary component
			if ( component == primary_component() ) {
				if ( align_system == AlignmentSystem::N2C ) {
					Integer left_moveable = graft_info_.scaffold_moveable_residues_nterm( old_scaffold_gap ) +
					                         graft_info_.epitope_moveable_residues_nterm( old_scaffold_gap );

					if ( left_moveable > 0 ) { // remove residues on the left
						std::set< Integer >::iterator left_start = moveable_residues.begin();
						std::set< Integer >::iterator left_end = moveable_residues.begin();

						for ( Integer lm = 0; lm < left_moveable; ++lm ) {
							++left_end;
						}

						moveable_residues.erase( left_start, left_end );
					}
				} else if ( align_system == AlignmentSystem::C2N ) {
					Integer right_moveable = graft_info_.scaffold_moveable_residues_cterm( old_scaffold_gap ) +
					                         graft_info_.epitope_moveable_residues_cterm( old_scaffold_gap );

					if ( right_moveable > 0 ) { // remove residues on the right
						std::set< Integer >::iterator right_start = moveable_residues.end();
						std::set< Integer >::iterator right_end = moveable_residues.end();

						for ( Integer rm = 0; rm < right_moveable; ++rm ) {
							--right_start;
						}

						moveable_residues.erase( right_start, right_end );
					}
				}
			}

			std::set< Integer >::const_iterator new_cut = moveable_residues.begin();

			// below we do c.moveable_residues().size() - 1 instead of just the size
			// to try and always place a cutpoint within the moveable residues to have
			// two moveable sections instead of one (this increases closure rate)
			for ( Size j = 0, je = ran3() * ( moveable_residues.size() - 1 ); j < je; ++j ) {
				assert( new_cut != moveable_residues.end() );
				++new_cut;
			}

			c.set_cut( *new_cut );

			// if new cutpoint was chosen, alter fold trees
			if ( c.cut() != i->cut() ) {

				// record all residues involved in jumps
				std::set< Integer > jump_residues;
				for ( Fold_tree::const_iterator e = ft.begin(), ee = ft.end(); e != ee; ++e ) {
					if ( e->label != Fold_tree::PEPTIDE ) {
						jump_residues.insert( e->start );
						jump_residues.insert( e->stop );
					}
				}

				std::vector< ResidueRange > new_edges;
				for ( Fold_tree::iterator e = ft.begin(), ee = ft.end(); e != ee; ++e ) {
					if ( e->label == Fold_tree::PEPTIDE ) {
						ResidueRange const edge = e->start < e->stop ? ResidueRange( e->start, e->stop ) : ResidueRange( e->stop, e->start );

						if ( edge.end() == i->cut() ) {

							if ( jump_residues.find( edge.end() ) != jump_residues.end() ) { // endpoint is a jump, we need to grow an extra edge
								new_edges.push_back( ResidueRange( edge.end(), c.cut() ) );
							} else { // regular case
								// delete old edge
								e = ft.delete_edge( e );
								--e; // need to back up after erase
								ee = ft.end(); // erasure invalidates iterator

								// track new edge
								new_edges.push_back( ResidueRange( edge.begin(), c.cut() ) );
							}

						} else if ( edge.begin() == ( i->cut() + 1 ) ) {

							if ( jump_residues.find( edge.begin() ) != jump_residues.end() ) { // endpoint is a jump, we need to grow an extra edge
								new_edges.push_back( ResidueRange( c.cut() + 1, edge.begin() ) );
							} else { // regular case
								// delete old edge
								e = ft.delete_edge( e );
								--e; // need to back up after erase
								ee = ft.end(); // erasure invalidates iterator

								// track new edge
								new_edges.push_back( ResidueRange( c.cut() + 1, edge.end() ) );
							}

						}
					}
				} // endfor ft edges

				// add new edges to tree
				for ( std::vector< ResidueRange >::const_iterator e = new_edges.begin(), ee = new_edges.end(); e != ee; ++e ) {
					ft.add_edge( e->begin(), e->end(), Fold_tree::PEPTIDE );
				}

				// handle archive if requested
				jump_residues.clear();
				new_edges.clear();
				if ( also_randomize_archive ) {
					for ( Fold_tree::const_iterator e = aft.begin(), ee = aft.end(); e != ee; ++e ) {
						if ( e->label != Fold_tree::PEPTIDE ) {
							jump_residues.insert( e->start );
							jump_residues.insert( e->stop );
						}
					}

					for ( Fold_tree::iterator e = aft.begin(), ee = aft.end(); e != ee; ++e ) {
						if ( e->label == Fold_tree::PEPTIDE ) {
							ResidueRange const edge = e->start < e->stop ? ResidueRange( e->start, e->stop ) : ResidueRange( e->stop, e->start );

							if ( edge.end() == i->cut() ) {

								if ( jump_residues.find( edge.end() ) != jump_residues.end() ) { // endpoint is a jump, we need to grow an extra edge
									new_edges.push_back( ResidueRange( edge.end(), c.cut() ) );
								} else {
									// delete old edge
									e = aft.delete_edge( e );
									--e; // need to back up after erase
									ee = aft.end(); // erasure invalidates iterator

									// track new edge
									new_edges.push_back( ResidueRange( edge.begin(), c.cut() ) );
								}

							} else if ( edge.begin() == ( i->cut() + 1 ) ) {

								if ( jump_residues.find( edge.begin() ) != jump_residues.end() ) { // endpoint is a jump, we need to grow an extra edge
									new_edges.push_back( ResidueRange( c.cut() + 1, edge.begin() ) );
								} else {
									// delete old edge
									e = aft.delete_edge( e );
									--e; // need to back up after erase
									ee = aft.end(); // erasure invalidates iterator

									// track new edge
									new_edges.push_back( ResidueRange( c.cut() + 1, edge.end() ) );
								}

							}
						}
					} // endfor aft edges

					// add new edges to tree
					for ( std::vector< ResidueRange >::const_iterator e = new_edges.begin(), ee = new_edges.end(); e != ee; ++e ) {
						aft.add_edge( e->begin(), e->end(), Fold_tree::PEPTIDE );
					}
				} // endif also_randomize_archive

			} // endif cutpoint change, alter fold trees

		}

		altered_closures.insert( c );
	}

	closures_to_attempt_ = altered_closures;

	// set fold trees
	ft.reorder( 1 );
	epitope_scaffold_.set_fold_tree( ft );
	if ( also_randomize_archive ) {
		aft.reorder( 1 );
		archived_epitope_scaffold_.set_fold_tree( aft );
	}

	recheck_closure_criteria();
}


/// @brief randomize number of moveable residues within broken closure sites
/// @details for each closure site, looks to the left of the cut and
/// @details picks a number of moveables residues, then looks to the right
/// @details of the cut and does the same
void
EpitopeScaffold::randomize_broken_closure_site_moveable_residues()
{
	std::set< LoopClosureInfo > altered_closures;

	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt_.begin(), ie = closures_to_attempt_.end(); i != ie; ++i ) {
		LoopClosureInfo c( *i );

		if ( !closed_during_trajectory( c ) && !c.is_closed() && do_graft_bb( c ) ) {
			// activate all moveable residues for counting
			c.activate_all_moveable_residues();
			Integer const random_left_moveable = c.count_moveable_residues_left_of_cut() > 0 ? ran3() * c.count_moveable_residues_left_of_cut() + 1 : 0;
			Integer const random_right_moveable = c.count_moveable_residues_right_of_cut() > 0 ? ran3() * c.count_moveable_residues_right_of_cut() + 1 : 0;

			// now only turn on the randomly selected left & right moveable residues
			c.inactivate_all_moveable_residues();
			c.activate_moveable_residues_adjacent_to_cut( random_left_moveable, random_right_moveable );
		}

		altered_closures.insert( c );
	}

	closures_to_attempt_ = altered_closures;
}


/// @brief randomize secondary structure of moveable residues within all closure sites
/// @details For each closure site, constructs the immediate *contiguous* interval of
/// @details moveable residues around the cut.  Then decides on the amount of
/// @details loop content w/ respect to a required minimum fraction and inserts
/// @details that amount of loop content with a randomly shifting window.  Other residues
/// @details are marked with the same structure to the immediate left & right of the merge
/// @details if there is at least three residues worth of the secondary structure content.
/// @param[in]  minimum_loop_content  minimum loop content (fraction, within interval [0, 1])
void
EpitopeScaffold::randomize_broken_closure_site_moveable_residue_secondary_structure(
	Real const & minimum_content,
	char const & ss_type
)
{
	assert( 0.0 <= minimum_content && minimum_content <= 1.0 );

	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt_.begin(), ie = closures_to_attempt_.end(); i != ie; ++i ) {
		LoopClosureInfo const & c = *i;

		if ( closed_during_trajectory( c ) || c.is_closed() || !do_graft_bb( c ) ) { // skip closed loops
			continue;
		}

		std::set< Integer > left_of_cut = c.moveable_residues_left_of_cut();
		std::set< Integer > right_of_cut = c.moveable_residues_right_of_cut();

		// variables
		Integer prior;
		std::set< Integer > contiguous;

		// find contiguous residues on left
		prior = c.cut() + 1;
		for ( std::set< Integer >::const_reverse_iterator j = left_of_cut.rbegin(), je = left_of_cut.rend(); j != je; ++j ) {
			Integer const & current = *j;
			if ( prior - current == 1 ) { // contiguous
				contiguous.insert( current );
				prior = current;
			} else { // found non-contiguous, break out of loop
				break;
			}
		}

		// find contiguous residues on right
		prior = c.cut();
		for ( std::set< Integer >::const_iterator j = right_of_cut.begin(), je = right_of_cut.end(); j != je; ++j ) {
			Integer const & current = *j;
			if ( current - prior == 1 ) { // contiguous
				contiguous.insert( current );
				prior = current;
			} else { // found non-contiguous, break out of loop
				break;
			}
		}

		// calculate loop content
		Real loop_content = ran3();
		if ( loop_content < minimum_content ) {
			loop_content = minimum_content;
		}

		// calculate placement of loop content
		Integer const total_moveable = contiguous.size();
		ResidueRange const range( *contiguous.begin(), *contiguous.rbegin() );
		Integer const n_loop_residues = std::ceil( loop_content * static_cast< Real >( range.length() ) );
		Integer const starting_position = range.begin() + std::floor( ran3() * ( total_moveable - n_loop_residues + 1 ) ); // choose starting residue
		Integer const ending_position = starting_position + n_loop_residues - 1;

		// mark desired secondary structure
		for ( Integer r = starting_position, re = ending_position; r <= re; ++r ) {
			epitope_scaffold_.set_secstruct( r, ss_type );
		}

		// mark secondary structure on left
		// if three contiguous residues of secondary structure at left merge point, use
		// that secondary structure, otherwise use loop
		assert( range.begin() > 3 );
		char left_ss = epitope_scaffold_.secstruct( range.begin() - 1 );
		for ( Integer r = range.begin() - 2, re = range.begin() - 3; r >= re; --r ) {
			if ( left_ss != epitope_scaffold_.secstruct( r ) ) {
				left_ss = 'L';
				break;
			}
		}
		for ( Integer r = range.begin(); r < starting_position; ++r ) {
			epitope_scaffold_.set_secstruct( r, left_ss );
		}

		// mark secondary structure on right
		// if three contiguous residues of secondary structure at right merge point, use
		// that secondary structure, otherwise use loop
		assert( range.end() < epitope_scaffold_.total_residue() - 3 + 1 );
		char right_ss = epitope_scaffold_.secstruct( range.end() + 1 );
		for ( Integer r = range.end() + 2, re = range.end() + 3; r <= re; ++r ) {
			if ( right_ss != epitope_scaffold_.secstruct( r ) ) {
				right_ss = 'L';
				break;
			}
		}
		for ( Integer r = ending_position + 1, re = range.end(); r <= re; ++r ) {
			epitope_scaffold_.set_secstruct( r, right_ss );
		}

	}
}


/// @brief resets the backbone of the residues in a closure site to the archived backbone
/// @details resets via set phi/psi/omega and not coordinate replacement
/// @note the closure_info is actually modified to store the new chainbreak & rama,
/// @note so make sure to save a prior copy if needed
void
EpitopeScaffold::reset_closure_site_backbone(
	LoopClosureInfo const & closure_info
)
{
	if ( do_graft_bb( closure_info ) ) {
		std::set< Integer > const & moveable = closure_info.moveable_residues();
		set_backbone_from_archive( moveable.begin(), moveable.end() );
		check_closure_criteria( closure_info );
	}
}


/// @brief resets the backbone of ALL closure sites
void
EpitopeScaffold::reset_all_closure_site_backbone()
{
	std::set< Integer > moveable;
	for ( std::set< LoopClosureInfo >::const_iterator c = closures_to_attempt_.begin(), ce = closures_to_attempt_.end(); c != ce; ++c ) {
		if ( do_graft_bb( *c ) ) {
			moveable.insert( c->moveable_residues().begin(), c->moveable_residues().end() );
		}
	}

	set_backbone_from_archive( moveable.begin(), moveable.end() );
	recheck_closure_criteria();
}


/// @brief resets the backbone of closures sites that are still broken (not closed)
void
EpitopeScaffold::reset_broken_closure_site_backbone()
{
	std::set< Integer > moveable;
	for ( std::set< LoopClosureInfo >::const_iterator c = closures_to_attempt_.begin(), ce = closures_to_attempt_.end(); c != ce; ++c ) {
		if ( !c->is_closed() && do_graft_bb( *c ) ) {
			moveable.insert( c->moveable_residues().begin(), c->moveable_residues().end() );
		}
	}

	set_backbone_from_archive( moveable.begin(), moveable.end() );
	recheck_closure_criteria();
}


/// @brief return total centroid score (vdw+rama+rg) assuming continuous epitope-scaffold (no chainbreaks)
/// @details temporarily sets up a continuous fold tree (if Ab attached, uses one jump tree) and scores
Real
EpitopeScaffold::centroid_score_no_breaks() const
{
	using namespace pose_ns;

	Score_weight_map weight_map;
	weight_map.set_weight( VDW, 1.0 );
	weight_map.set_weight( RAMACHANDRAN, 0.1 );
	weight_map.set_weight( RG, 1.0 );

	return score_no_breaks( weight_map );
}

/// @brief return total fullatom score (score12) assuming continuous epitope-scaffold (no chainbreaks)
/// @details temporarily sets up a continuous fold tree (if Ab attached, uses one jump tree) and scores
Real
EpitopeScaffold::fullatom_score_no_breaks() const
{
	using pose_ns::Score_weight_map;

	Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score12 );

	return score_no_breaks( weight_map );
}


/// @brief return score assuming continuous epitope-scaffold (no chainbreaks)
/// @details temporarily sets up a continuous fold tree (if Ab attached, uses one jump tree) and scores
Real
EpitopeScaffold::score_no_breaks(
	pose_ns::Score_weight_map weight_map
) const
{
	// use separate Pose to score -- if we don't do this there is a potential problem
	// with small shifts in large sections of the scaffold due to inexact loop closure
	// or floating point inconsistency
	Pose es_complex;
	extract_complex( es_complex );

	return es_complex.score( weight_map );
}


/// @brief return fullatom score of ONLY continuous epitope-scaffold (no chainbreaks)
Real
EpitopeScaffold::scaffold_centroid_score_no_breaks() const
{
	using namespace pose_ns;

	Score_weight_map weight_map;
	weight_map.set_weight( VDW, 1.0 );
	weight_map.set_weight( RAMACHANDRAN, 0.1 );
	weight_map.set_weight( RG, 1.0 );

	return scaffold_score_no_breaks( weight_map );
}


/// @brief return fullatom score of ONLY continuous epitope-scaffold (no chainbreaks)
Real
EpitopeScaffold::scaffold_fullatom_score_no_breaks() const
{
	using pose_ns::Score_weight_map;

	Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score12 );

	return scaffold_score_no_breaks( weight_map );
}


/// @brief return score of ONLY continuous epitope-scaffold (no chainbreaks)
Real
EpitopeScaffold::scaffold_score_no_breaks(
	pose_ns::Score_weight_map weight_map
) const
{
	Pose es;
	extract_epitope_scaffold( es );

	return es.score( weight_map );
}


/// @brief ddG computation
/// @details this assumes Ab is attached, please attach beforehand
Real
EpitopeScaffold::ddG() const
{
	using namespace pose_ns;

	Real ddg = DESIGN_INFINITY;

	if ( Ab_is_connected_ ) {
		// weight map
		Score_weight_map weight_map;
		setup_score_weight_map( weight_map, score12 );

		Pose Ab;
		extract_Ab( Ab );

		Pose es;
		extract_epitope_scaffold( es );

		ddg = fullatom_score_no_breaks() - ( Ab.score( weight_map ) + es.score( weight_map ) );
	}

	return ddg;
}


/// @brief export resfile
/// @details intra-positions are NOTAA  C
/// @details inter-positions are PIKAA  AGST or NOTAA  C (if allowed by option)
/// @details allowed to repacked positions are NATAA
/// @details interface design positions are PIKAA <residue type>
/// @details everything else is NATRO
/// @details To be relevant for design purposes, cached design data must be filled from a prior
/// @details EpitopeScaffold::design() call, otherwise call
/// @details EpitopeScaffold::refresh_cached_design_positions() to refresh artificially.
void
EpitopeScaffold::export_resfile(
	String const & filename
) const
{
	DesignFileExport dfe( epitope_scaffold_.total_residue() );

	// intra positions
	for ( std::set< Integer >::const_iterator i = intra_design_positions_.begin(), ie = intra_design_positions_.end(); i != ie; ++i ) {

		// we never design cys in disulfides
		if ( disulfides::BOUNDARY::cys_res_in_disulf( *i ) ) {
			continue;
		}

		dfe.set_identity_action( *i, DesignFileExport::ALLAA ); // ALLAA by default excludes C
	}

	// inter positions
	for ( std::set< Integer >::const_iterator i = inter_design_positions_.begin(), ie = inter_design_positions_.end(); i != ie; ++i ) {
		if ( allow_ALLAA_at_inter_design_positions_ ) {
			dfe.set_identity_action( *i, DesignFileExport::ALLAA ); // ALLAA by default excludes C
		} else {
			dfe.set_identity_action( *i, DesignFileExport::PIKAA );
			dfe.add_identity_info( *i, 'A' );
			dfe.add_identity_info( *i, 'G' );
			dfe.add_identity_info( *i, 'S' );
			dfe.add_identity_info( *i, 'T' );
		}
	}

	// epitope repack positions
	if ( allow_epitope_repack_) {
		for ( std::map< ResidueRange, ResidueRange >::const_iterator i = native_to_new_epitope_.begin(), ie = native_to_new_epitope_.end(); i != ie; ++i ) {
			ResidueRange const & epitope_range = i->second;

			for ( Integer res = epitope_range.begin(); res <= epitope_range.end(); ++res ) {
				dfe.set_identity_action( res, DesignFileExport::NATAA );
			}
		}
	}

	// Ab repack positions
	if ( allow_Ab_repack_ ) {
		for ( std::set< Integer >::const_iterator i = Ab_repack_positions_.begin(), ie = Ab_repack_positions_.end(); i != ie; ++i ) {
			dfe.set_identity_action( *i, DesignFileExport::NATAA );
		}
	}

	// complementarity positions
	for ( std::set< Integer >::const_iterator i = complementarity_design_positions_.begin(), ie = complementarity_design_positions_.end(); i != ie; ++i ) {
		dfe.set_identity_action( *i, DesignFileExport::PIKAA );
		dfe.add_identity_info( *i, param_aa::param_aa_data::aa_name1( epitope_scaffold_.res( *i ) ) );
	}

	dfe.resfile( filename, epitope_scaffold_ );
}

/// @brief export blueprint file
/// @details intra-positions are PIKAA <amino acids sans cys>
/// @details inter-positions are PIKAA  AGST or PIKAA <amino acids sans cys> (if allowed by option)
/// @details allowed to repacked positions are NATAA
/// @details interface design positions are PIKAA <residue type>
/// @details everything else is NATRO
/// @details moveable residues on loop closures are marked w/ their final secondary structure
/// @details To be relevant for design purposes, cached design data must be filled from a prior
/// @details EpitopeScaffold::design() call, otherwise call
/// @details EpitopeScaffold::refresh_cached_design_positions() to refresh artificially.
void
EpitopeScaffold::export_blueprint(
	String const & filename
) const
{
	// make temporary identity info set for NOTAA  C since blueprint doesn't support
	// NOTAA keyword
	std::set< char > notaa_C;
	notaa_C.insert( 'A' );
	notaa_C.insert( 'D' );
	notaa_C.insert( 'E' );
	notaa_C.insert( 'F' );
	notaa_C.insert( 'G' );
	notaa_C.insert( 'H' );
	notaa_C.insert( 'I' );
	notaa_C.insert( 'K' );
	notaa_C.insert( 'L' );
	notaa_C.insert( 'M' );
	notaa_C.insert( 'N' );
	notaa_C.insert( 'P' );
	notaa_C.insert( 'Q' );
	notaa_C.insert( 'R' );
	notaa_C.insert( 'S' );
	notaa_C.insert( 'T' );
	notaa_C.insert( 'V' );
	notaa_C.insert( 'W' );
	notaa_C.insert( 'Y' );

	DesignFileExport dfe( epitope_scaffold_.total_residue() );

	// intra positions
	for ( std::set< Integer >::const_iterator i = intra_design_positions_.begin(), ie = intra_design_positions_.end(); i != ie; ++i ) {

		// we never design cys in disulfides
		if ( disulfides::BOUNDARY::cys_res_in_disulf( *i ) ) {
			continue;
		}

		dfe.set_identity_action( *i, DesignFileExport::PIKAA );
		dfe.add_identity_info( *i, notaa_C.begin(), notaa_C.end() );
	}

	// inter positions
	for ( std::set< Integer >::const_iterator i = inter_design_positions_.begin(), ie = inter_design_positions_.end(); i != ie; ++i ) {
		if ( allow_ALLAA_at_inter_design_positions_ ) {
			dfe.set_identity_action( *i, DesignFileExport::PIKAA );
			dfe.add_identity_info( *i, notaa_C.begin(), notaa_C.end() );
		} else {
			dfe.set_identity_action( *i, DesignFileExport::PIKAA );
			dfe.add_identity_info( *i, 'A' );
			dfe.add_identity_info( *i, 'G' );
			dfe.add_identity_info( *i, 'S' );
			dfe.add_identity_info( *i, 'T' );
		}
	}

	// epitope repack positions
	if ( allow_epitope_repack_) {
		for ( std::map< ResidueRange, ResidueRange >::const_iterator i = native_to_new_epitope_.begin(), ie = native_to_new_epitope_.end(); i != ie; ++i ) {
			ResidueRange const & epitope_range = i->second;

			for ( Integer res = epitope_range.begin(); res <= epitope_range.end(); ++res ) {
				dfe.set_identity_action( res, DesignFileExport::PIKAA );
				dfe.add_identity_info( res, param_aa::param_aa_data::aa_name1( epitope_scaffold_.res( res ) ) );
			}
		}
	}

	// Ab repack positions
	if ( allow_Ab_repack_ ) {
		for ( std::set< Integer >::const_iterator i = Ab_repack_positions_.begin(), ie = Ab_repack_positions_.end(); i != ie; ++i ) {
			dfe.set_identity_action( *i, DesignFileExport::PIKAA );
			dfe.add_identity_info( *i, param_aa::param_aa_data::aa_name1( epitope_scaffold_.res( *i ) ) );
		}
	}

	// complementarity positions
	for ( std::set< Integer >::const_iterator i = complementarity_design_positions_.begin(), ie = complementarity_design_positions_.end(); i != ie; ++i ) {
		dfe.set_identity_action( *i, DesignFileExport::PIKAA );
		dfe.add_identity_info( *i, param_aa::param_aa_data::aa_name1( epitope_scaffold_.res( *i ) ) );
	}

	// secondary structure
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt_.begin(), ie = closures_to_attempt_.end(); i != ie; ++i ) {
		LoopClosureInfo const & c = *i;

		if ( do_graft_bb( c ) ) { // only mark allowed
			for ( std::set< Integer >::const_iterator m = c.moveable_residues().begin(), me = c.moveable_residues().end(); m != me; ++m ) {
				dfe.set_structure_action( *m, epitope_scaffold_.secstruct( *m ) ) ;
			}
		}
	}

	dfe.blueprint( filename, epitope_scaffold_ );
}


/// @brief return status (closure status, design status, etc. )
/// @details gives closure status, design status, runs a total score and runs a ddG score
String
EpitopeScaffold::to_string() const
{
	std::ostringstream ss;

	// grab all components (new epitope component ranges) and list their closure results
	ss << "# Closure Results:\n";
	std::set< ResidueRange > components = all_components(); // copy of all components
	for ( std::set< ResidueRange >::const_iterator l = components.begin(), le = components.end(); l != le; ++l ) {
		ResidueRange const & component = *l;

		// get copy of closures for this component
		std::set< LoopClosureInfo > closures = component_closures_to_attempt( component );

		// print epitope component range
		ss << "#\n";
		ss << "#   epitope native range = " << new_to_native_epitope( component ).to_string() << "   new range = " << component.to_string() << "\n";

		for ( std::set< LoopClosureInfo >::const_iterator c = closures.begin(), ce = closures.end(); c != ce; ++c ) {
			LoopClosureInfo const & closure_info = *c;
			ss << "#      " << closure_info.to_string() << "\n";
			ss << closure_info.to_string_rama_per_residue( "#         " );
		}
	}

	// spacer
	ss << "#\n";

	ss << design_positions_to_string();

	return ss.str();
}


/// @brief return residue information (epitope/intra/inter/complementarity positions)
String
EpitopeScaffold::design_positions_to_string() const
{
	std::ostringstream ss;

	std::set< ResidueRange > components = all_components(); // copy of all components

	// list epitope positions, with respect to keep_natro_ if available
	ss << "# Epitope positions:\n";
	for ( std::set< ResidueRange >::const_iterator l = components.begin(), le = components.end(); l != le; ++l ) {
		ResidueRange const & new_loop = *l;
		ResidueRange const & old_loop = new_to_native_epitope( new_loop );
		Integer const offset = new_loop.begin() - old_loop.begin();

		for ( Integer i = old_loop.begin(), ie = old_loop.end(); i <= ie; ++i ) {
			if ( keep_natro_.size() == 0 || keep_natro_.find( i ) != keep_natro_.end() ) {
				Integer const resid = i + offset;
				ss << "#    " << i << "  ->  " << resid <<  "  " << param_aa::param_aa_data::aa_name3( epitope_scaffold_.res( resid  ) ) << "\n";
			}
		}
	}

	// spacer
	ss << "#\n";

	// list design positions and results
	ss << "# Intra design positions\n";
	for ( std::set< Integer >::const_iterator r = intra_design_positions_.begin(), re = intra_design_positions_.end(); r != re; ++r ) {
		Integer const & resid = *r;
		ss << "#    " << resid << "  " << param_aa::param_aa_data::aa_name3( archived_epitope_scaffold_.res( resid  ) ) << "  ->  " << param_aa::param_aa_data::aa_name3( epitope_scaffold_.res( resid  ) ) << "\n";
	}
	ss << "#\n"; // spacer
	ss << "# Inter design positions:\n";
	for ( std::set< Integer >::const_iterator r = inter_design_positions_.begin(), re = inter_design_positions_.end(); r != re; ++r ) {
		Integer const & resid = *r;
		ss << "#    " << resid << "  " << param_aa::param_aa_data::aa_name3( archived_epitope_scaffold_.res( resid  ) ) << "  ->  " << param_aa::param_aa_data::aa_name3( epitope_scaffold_.res( resid  ) ) << "\n";
	}

	if ( complementarity_design_positions_.size() > 0 ) {
		ss << "#\n"; // spacer
		ss << "# Complementarity design positions:\n";

		for ( std::set< Integer >::const_iterator r = complementarity_design_positions_.begin(), re = complementarity_design_positions_.end(); r != re; ++r ) {
			Integer const & resid = *r;
			ss << "#    " << resid << "  " << param_aa::param_aa_data::aa_name3( epitope_scaffold_.res( resid  ) ) << "\n";
		}
	}

	if ( complementarity_shell_positions_.size() > 0 ) {
		ss << "#\n"; // spacer
		ss << "# Complementarity shell positions:\n";

		for ( std::set< Integer >::const_iterator r = complementarity_shell_positions_.begin(), re = complementarity_shell_positions_.end(); r != re; ++r ) {
			Integer const & resid = *r;
			ss << "#    " << resid << "  " << param_aa::param_aa_data::aa_name3( archived_epitope_scaffold_.res( resid  ) ) << "  ->  " << param_aa::param_aa_data::aa_name3( epitope_scaffold_.res( resid  ) ) << "\n";
		}
	}

	return ss.str();
}


/// @brief make predesign epitope scaffold
/// @note  directly stores two copies of the epitope-scaffold (working and archived)
void
EpitopeScaffold::make_predesign_epitope_scaffold(
	Pose const & original_scaffold,
	Pose const & epitope,
	GraftInfo const & graft_info,
	std::set< Integer > const & keep_natro
)
{
	using pose_ns::Fold_tree;
	using epigraft::conformation::DihedralInfo;
	using epigraft::match::MatchComponent;
	using namespace epigraft::match::align;

	// note that MatchResult is already privately typedef'd in this (EpitopeScaffold) class
	typedef GraftInfo::SortedMatchComponents SortedMatchComponents;

	// Storage for N2C/C2N moveable residues at the merge point to prevent
	// idealization of moveable residues that result in "lever" moves.
	// Without this the rigid body relationship between epitope and antibody
	// can be lost -- idealization can cause the epitope to swing during
	// construction, independent of the rigid body orientation of the antibody.
	std::set< Integer > non_idealizable_residues;

	// convenience
	MatchResult const & match_result = graft_info.match_result();
	MatchComponent const & primary_component = match_result.components[ 1 ];

	// first construct aligned epitope
	Pose aligned_epitope;
	aligned_epitope = epitope;
	aligned_epitope.transform_GL( match_result.transformation_matrix );

	// next build artifical scaffold with any takeoff optimized dihedral changes;
	// here we only want to change the takeoff optimization angles but keep the rest of the scaffold fixed
	Pose scaffold;
	scaffold = original_scaffold;
	if ( primary_component.dihedrals.size() > 0 && graft_info.do_graft_bb( primary_component ) ) {
		Fold_tree aft; // artificial fold tree that jumps from residue prior/after the takeoff change to just after/prior to the gap residue

		Integer first_optimization_residue = -1;
		switch ( match_result.system_type ) {
			case AlignmentSystem::N2C:
				first_optimization_residue = ( primary_component.dihedrals.begin() )->residue();
				if ( first_optimization_residue < primary_component.scaffold_gap_range.begin() ) {
					aft.add_edge( 1, first_optimization_residue - 1, Fold_tree::PEPTIDE );
					aft.add_edge( first_optimization_residue - 1, primary_component.scaffold_gap_range.begin(), Fold_tree::PEPTIDE );
					aft.add_edge( first_optimization_residue - 1, primary_component.scaffold_gap_range.begin() + 1, 1 ); // jump
					aft.add_edge( primary_component.scaffold_gap_range.begin() + 1, scaffold.total_residue(), Fold_tree::PEPTIDE );
					aft.reorder( 1 ); // fold forwards N->C
					scaffold.set_fold_tree( aft );

					// now set angles
					std::set< DihedralInfo >::const_iterator d = primary_component.dihedrals.begin(), de = primary_component.dihedrals.end();
					while ( d != de ) {
						if ( d->residue() < primary_component.scaffold_gap_range.begin() ) {
							d->apply( scaffold );
							++d;
						}
					}
				}
				break;
			case AlignmentSystem::C2N:
				first_optimization_residue = ( primary_component.dihedrals.rbegin() )->residue();
				if ( first_optimization_residue > primary_component.scaffold_gap_range.end() ) {
					aft.add_edge( scaffold.total_residue(), first_optimization_residue + 1, Fold_tree::PEPTIDE );
					aft.add_edge( first_optimization_residue + 1, primary_component.scaffold_gap_range.end(), Fold_tree::PEPTIDE );
					aft.add_edge( first_optimization_residue + 1, primary_component.scaffold_gap_range.end() - 1, 1 ); // jump
					aft.add_edge( primary_component.scaffold_gap_range.end() - 1, 1, Fold_tree::PEPTIDE );
					aft.reorder( scaffold.total_residue() ); // fold backwards C->N
					scaffold.set_fold_tree( aft );

					// now set angles
					std::set< DihedralInfo >::const_reverse_iterator d = primary_component.dihedrals.rbegin(), de = primary_component.dihedrals.rend();
					while ( d != de ) {
						if ( d->residue() > primary_component.scaffold_gap_range.begin() ) {
							d->apply( scaffold );
							++d;
						}
					}
				}
				break;
			default:
				break;
		}
	}

	// create set of components sorted on left endpoint of scaffold gap ranges
	SortedMatchComponents components = graft_info.sorted_match_components();

	// count total number of residues;
	Integer total_residue = scaffold.total_residue();
	for ( SortedMatchComponents::const_iterator c = components.begin(), ce = components.end(); c != ce; ++c ) {
		MatchComponent const & component = *c;
		if ( graft_info.do_graft_bb( component ) ) {
			total_residue = total_residue + component.loop_subrange.length() - component.scaffold_gap_range.length();
		}
	}

	// make fold tree for loop closure/design
	Fold_tree f;
	Integer start_vertex = 1; // initial vertex at N-terminus
	Integer offset = 0; // incrementally track differences after each gap/loop insertion
	Integer jump_label = 0;
	utility::vector1< ResidueRange > es_secondary_loops; // store for use in constructing parts of tree from epitope
	ResidueRange es_primary_loop;
	std::set< DihedralInfo > es_dihedrals; // as we set up the tree, track and renumber any dihedrals that need to be changed

	// additional tracking info transferred to object's local store at end of routine
	std::map< ResidueRange, ResidueRange > native_to_new_epitope; // maps native epitope loop range -> epitope-scaffold loop range
	std::map< ResidueRange, EpitopeTrack > epitope_tracking; // maps epitope-scaffold component range -> jump, native epitope, old scaffold gap
	std::set< LoopClosureInfo > primary_closures_to_attempt; // closure information for primary components
	std::set< LoopClosureInfo > secondary_closures_to_attempt; // closure information for secondary components

	// handle parts of tree that come from scaffold
	for ( SortedMatchComponents::const_iterator c = components.begin(), ce = components.end(); c != ce; ++c ) {
		MatchComponent const & component = *c;
		Integer const & TETHER_NTERM = graft_info.scaffold_tether_residues_nterm( component );
		Integer const & TETHER_CTERM = graft_info.scaffold_tether_residues_cterm( component );
		Integer const & MOVEABLE_NTERM = graft_info.scaffold_moveable_residues_nterm( component );
		Integer const & MOVEABLE_CTERM = graft_info.scaffold_moveable_residues_cterm( component );
		bool const & ALREADY_GRAFTED = !graft_info.do_graft_bb( component );
		Integer const LOOP_LENGTH = ALREADY_GRAFTED ? component.scaffold_gap_range.length() : component.loop_subrange.length();

		Integer const first_branch_vertex = component.scaffold_gap_range.begin() + offset - TETHER_NTERM - 1;
		Integer const second_branch_vertex = first_branch_vertex + TETHER_NTERM + LOOP_LENGTH
		                                     + TETHER_CTERM + 1;

		f.add_edge( start_vertex, first_branch_vertex, Fold_tree::PEPTIDE );
		f.add_edge( first_branch_vertex, first_branch_vertex + TETHER_NTERM, Fold_tree::PEPTIDE ); // moveable loop closure edge
		f.add_edge( first_branch_vertex, second_branch_vertex, ++jump_label );
		f.add_edge( second_branch_vertex, second_branch_vertex - TETHER_CTERM, Fold_tree::PEPTIDE ); // moveable loop closure edge

		// store new loop ranges for later use in constructing additional parts of tree from epitope
		// and store loop closure info
		if ( primary_component.scaffold_gap_range == component.scaffold_gap_range ) { // handle primary here
			es_primary_loop = ResidueRange( first_branch_vertex + TETHER_NTERM + 1, second_branch_vertex - TETHER_CTERM - 1 );
			native_to_new_epitope[ component.native_loop_subrange ] = es_primary_loop; // correspondence between native loop subrange and new epitope scaffold loop
			epitope_tracking[ es_primary_loop ].native_epitope = component.native_loop_subrange; // correspondence between new epitope scaffold loop and native loop subrange
			epitope_tracking[ es_primary_loop ].old_gap = component.scaffold_gap_range; // correspondence between new epitope-scaffold loop and original scaffold gap range

			// loop closure info
			switch ( match_result.system_type ) {
				case AlignmentSystem::N2C:
					{
						LoopClosureInfo closure_info( ResidueRange( first_branch_vertex + 1, second_branch_vertex - 1 ), second_branch_vertex - TETHER_CTERM - 1, 0, MOVEABLE_CTERM );  // right/c-side break
						if ( MOVEABLE_NTERM > 0 ) {
							ResidueRange const moveable_n_side_residues( es_primary_loop.begin() - MOVEABLE_NTERM, es_primary_loop.begin() - 1 );
							closure_info.add_moveable_residues( moveable_n_side_residues ); // left/n-side connected residues

							// don't idealize these residues due to lever arm effects
							for ( Integer r = moveable_n_side_residues.begin(), re = moveable_n_side_residues.end(); r <= re; ++r ) {
								non_idealizable_residues.insert( r );
								arm_residues_.insert( r );
							}
						}
						closure_info.set_is_closed( ALREADY_GRAFTED );
						closure_info.set_is_artificially_closed( graft_info.c_side_artificially_closed( component ) );
						primary_closures_to_attempt.insert( closure_info );
					}
					break;
				case AlignmentSystem::C2N:
					{
						LoopClosureInfo closure_info( ResidueRange( first_branch_vertex + 1, second_branch_vertex - 1 ), first_branch_vertex + TETHER_NTERM, MOVEABLE_NTERM, 0 );  // left/n-side break
						if ( MOVEABLE_CTERM > 0 ) {
							ResidueRange const moveable_c_side_residues( es_primary_loop.end() + 1, es_primary_loop.end() + MOVEABLE_CTERM );
							closure_info.add_moveable_residues( moveable_c_side_residues ); // right/c-side connected residues

							// don't idealize these residues due to lever arm effects
							for ( Integer r = moveable_c_side_residues.begin(), re = moveable_c_side_residues.end(); r <= re; ++r ) {
								non_idealizable_residues.insert( r );
								arm_residues_.insert( r );
							}
						}
						closure_info.set_is_closed( ALREADY_GRAFTED );
						closure_info.set_is_artificially_closed( graft_info.n_side_artificially_closed( component ) );
						primary_closures_to_attempt.insert( closure_info );
					}
					break;
				default:
					{
						LoopClosureInfo left_loop( ResidueRange( first_branch_vertex + 1, first_branch_vertex + TETHER_NTERM + 1 ), first_branch_vertex + TETHER_NTERM, MOVEABLE_NTERM, 0 );
						left_loop.set_is_closed( ALREADY_GRAFTED );
						left_loop.set_is_artificially_closed( graft_info.n_side_artificially_closed( component ) );
						primary_closures_to_attempt.insert( left_loop ); // left/n-side break

						LoopClosureInfo right_loop( ResidueRange( second_branch_vertex - TETHER_CTERM - 1, second_branch_vertex - 1 ), second_branch_vertex - TETHER_CTERM - 1, 0, MOVEABLE_CTERM );
						right_loop.set_is_closed( ALREADY_GRAFTED );
						right_loop.set_is_artificially_closed( graft_info.c_side_artificially_closed( component ) );
						primary_closures_to_attempt.insert( right_loop ); // right/c-side break
					}
					break;
			}
		} else { // handle secondary here
			ResidueRange es_secondary_loop = ResidueRange( first_branch_vertex + TETHER_NTERM + 1, second_branch_vertex - TETHER_CTERM - 1 );
			es_secondary_loops.push_back( es_secondary_loop );
			native_to_new_epitope[ component.native_loop_subrange ] = es_secondary_loop; // correspondence between native loop subrange and new epitope scaffold loop
			epitope_tracking[ es_secondary_loop ].native_epitope = component.native_loop_subrange; // correspondence between new epitope scaffold loop and native loop subrange
			epitope_tracking[ es_secondary_loop ].old_gap = component.scaffold_gap_range; // correspondence between new epitope-scaffold loop and original scaffold gap range

			// loop closure info
			LoopClosureInfo left_loop( ResidueRange( first_branch_vertex + 1, first_branch_vertex + TETHER_NTERM + 1 ), first_branch_vertex + TETHER_NTERM, TETHER_NTERM, 0 );
			left_loop.set_is_closed( ALREADY_GRAFTED );
			left_loop.set_is_artificially_closed( graft_info.n_side_artificially_closed( component ) );
			secondary_closures_to_attempt.insert( left_loop ); // left/n-side break

			LoopClosureInfo right_loop( ResidueRange( second_branch_vertex - TETHER_CTERM - 1, second_branch_vertex - 1 ), second_branch_vertex - TETHER_CTERM - 1, 0, TETHER_CTERM );
			right_loop.set_is_closed( ALREADY_GRAFTED );
			right_loop.set_is_artificially_closed( graft_info.c_side_artificially_closed( component ) );
			secondary_closures_to_attempt.insert( right_loop ); // right/c-side break
		}

		// dihedrals, only add for those components that need to be closed
		if ( !ALREADY_GRAFTED ) {
			if ( component.dihedrals.size() > 0 ) { // dihedrals exist in component, so track their new numbering for epitope-scaffold
				for ( std::set< DihedralInfo >::const_iterator d = component.dihedrals.begin(), de = component.dihedrals.end(); d != de; ++d ) {
					DihedralInfo dihedral( *d ); // will be re-numbered dihedral for epitope-scaffold

					// notice here that dihedrals for the gap/match residue are not applied as they are only used to improve rms
					if ( dihedral.residue() < component.scaffold_gap_range ) { // must be completely left of the loop

						dihedral.set_residue( dihedral.residue() + offset );
						es_dihedrals.insert( dihedral );

					} else if ( dihedral.residue() > component.scaffold_gap_range ) { // must be completely right of the loop

						dihedral.set_residue( dihedral.residue() + offset + component.loop_subrange.length() - component.scaffold_gap_range.length() ); // explicit loop_subrange.length() used instead of LOOP_LENGTH
						es_dihedrals.insert( dihedral );

					}
				}
			}
		}

		start_vertex = second_branch_vertex; // move start vertex
		offset += LOOP_LENGTH - component.scaffold_gap_range.length();
	}
	f.add_edge( start_vertex, total_residue, Fold_tree::PEPTIDE ); // last edge to end of epitope-scaffold

	// handle parts of tree that come from epitope by anchoring at primary component and connecting to secondaries
	Integer const middle_of_primary = middle_of_range( es_primary_loop ); // middle of primary loop
	switch ( match_result.system_type ) { // handle primary component here
		case AlignmentSystem::N2C: // connect at N
			{
				f.add_edge( es_primary_loop.begin() - 1, middle_of_primary, Fold_tree::PEPTIDE );
				f.add_edge( middle_of_primary, es_primary_loop.end(), Fold_tree::PEPTIDE );

				epitope_tracking[ es_primary_loop ].jump_label = -1; // no jump
			}
			break;
		case AlignmentSystem::C2N: // connect at C
			{
				f.add_edge( es_primary_loop.end() + 1, middle_of_primary, Fold_tree::PEPTIDE );
				f.add_edge( middle_of_primary, es_primary_loop.begin(), Fold_tree::PEPTIDE );

				epitope_tracking[ es_primary_loop ].jump_label = -1; // no jump
			}
			break;
		default: // double break alignment systems (S, E, SS) -- connect from branch vertex to middle of loop
			{
				Integer const & TETHER_NTERM = graft_info.scaffold_tether_residues_nterm( primary_component );;
				f.add_edge( es_primary_loop.begin() - TETHER_NTERM - 1, middle_of_primary, ++jump_label ); // scaffold -> primary loop
				f.add_edge( middle_of_primary, es_primary_loop.begin(), Fold_tree::PEPTIDE ); // grow to primary loop N-terminus
				f.add_edge( middle_of_primary, es_primary_loop.end(), Fold_tree::PEPTIDE ); // grow to primary loop C-terminus

				epitope_tracking[ es_primary_loop ].jump_label = jump_label; // store jump label/index
			}
			break;
	}
	for ( utility::vector1< ResidueRange >::const_iterator sl = es_secondary_loops.begin(), sle = es_secondary_loops.end(); sl != sle; ++sl ) { // now connect primary to secondaries
		ResidueRange const & es_secondary_loop = *sl;
		Integer const middle_of_secondary = middle_of_range( es_secondary_loop ); // middle of secondary loop

		f.add_edge( middle_of_primary, middle_of_secondary, ++jump_label ); // connect primary -> secondary
		f.add_edge( middle_of_secondary, es_secondary_loop.begin(), Fold_tree::PEPTIDE ); // grow to secondary loop N-terminus
		f.add_edge( middle_of_secondary, es_secondary_loop.end(), Fold_tree::PEPTIDE ); // grow to secondary loop C-terminus

		epitope_tracking[ es_secondary_loop ].jump_label = jump_label; // store jump label/index
	}

	// finalize tree
	f.reorder( 1 ); // start from N-terminus

	// new data for epitope-scaffold
	Pose epitope_scaffold;
//	if ( !idealize_loop_geometry_ ) {
//		// if not idealizing loop geometry we assume we
//		// are testing, and so set a simple fold tree first
//		// to fill in all bonds data, otherwise bond data
//		// around cutpoints will be set to nonsense.
//		Fold_tree sft;
//		sft.simple_tree( total_residue );
//		epitope_scaffold.set_fold_tree( sft );
//	}
	epitope_scaffold.set_fold_tree( f );
	epitope_scaffold.set_fullatom_flag( true, false ); // force booleans: fullatom, repack
	FArray3D_float full_coord( 3, param::MAX_ATOM(), total_residue );

	// counter variables for tracking progress
	Integer counter = 1; // keeps track of next residue to insert into
	Integer scaffold_counter = 1; // tracks position of next scaffold piece to transfer

	// track numbering for use in setting torsions adjacent to cutpoints down the line
	utility::vector1< Integer > new_to_old; // tracks epitope-scaffold numbering against original epitope (negative numbers) and original scaffold (positive numbers)

	// fill epitope scaffold
	for ( SortedMatchComponents::const_iterator c = components.begin(), ce = components.end(); c != ce; ++c ) {
		MatchComponent const & component = *c;

		// first handle scaffold piece
		ResidueRange scaffold_transfer_range( scaffold_counter, component.scaffold_gap_range.begin() - 1 );
		transfer_identity_info_between_Poses( scaffold, scaffold_transfer_range, epitope_scaffold, counter );
		fill_full_coord_from_Pose( scaffold, scaffold_transfer_range, full_coord, counter ); // counter modified here
		for ( Integer r = scaffold_transfer_range.begin(), re = scaffold_transfer_range.end(); r <= re; ++r ) { // track numbering
			new_to_old.push_back( r ); // positive number for scaffold
		}

		scaffold_counter = component.scaffold_gap_range.end() + 1; // shift to beginning of next scaffold piece for next iteration

		// next handle epitope piece
		if ( graft_info.do_graft_bb( component ) ) { // default, transfer from actual epitope
			transfer_identity_info_between_Poses( aligned_epitope, component.loop_subrange, epitope_scaffold, counter );
			fill_full_coord_from_Pose( aligned_epitope, component.loop_subrange, full_coord, counter ); // counter modified here
			for ( Integer r = component.loop_subrange.begin(), re = component.loop_subrange.end(); r <= re; ++r ) { // track numbering
				new_to_old.push_back( -r ); // negative number for epitope
			}
		} else { // epitope bb already present on scaffold, so for now get all coordinates from scaffold, sc handled later
			transfer_identity_info_between_Poses( scaffold, component.scaffold_gap_range, epitope_scaffold, counter );
			fill_full_coord_from_Pose( scaffold, component.scaffold_gap_range, full_coord, counter ); // counter modified here
			for ( Integer r = component.scaffold_gap_range.begin(), re = component.scaffold_gap_range.end(); r <= re; ++r ) { // track numbering
				new_to_old.push_back( r ); // positive number for scaffold
			}
		}

	}
	// transfer rest of scaffold, if present
	if ( counter < total_residue + 1 ) {
		ResidueRange scaffold_transfer_range( scaffold_counter, scaffold.total_residue() );
		transfer_identity_info_between_Poses( scaffold, scaffold_transfer_range, epitope_scaffold, counter );
		fill_full_coord_from_Pose( scaffold, scaffold_transfer_range, full_coord, counter ); // counter modified here
		for ( Integer r = scaffold_transfer_range.begin(), re = scaffold_transfer_range.end(); r <= re; ++r ) { // track numbering
			new_to_old.push_back( r ); // positive number for scaffold
		}
	}

	assert( counter == total_residue + 1 ); // make sure number of transfered residues is correct
	assert( new_to_old.size() == (Size)total_residue ); // make sure mapping was correct

	// micromanage N2C and C2N primary components, if grafting primary component
	if ( micromanage_termini_ && graft_info.do_graft_bb( primary_component ) ) {
		switch ( match_result.system_type ) {
			case AlignmentSystem::N2C:
				switch ( match_result.alignment_center ) {
					case AlignmentSystem::N:
						N2C_N_Align::micromanage_takeoff( scaffold.full_coord(), primary_component.scaffold_gap_range.begin(), full_coord, es_primary_loop.begin() );
						break;
					case AlignmentSystem::CA:
						N2C_CA_Align::micromanage_takeoff( scaffold.full_coord(), primary_component.scaffold_gap_range.begin(), full_coord, es_primary_loop.begin() );
						break;
					case AlignmentSystem::C:
						N2C_C_Align::micromanage_takeoff( scaffold.full_coord(), primary_component.scaffold_gap_range.begin(), full_coord, es_primary_loop.begin() );
						break;
					default:
						break;
				}
				break;

			case AlignmentSystem::C2N:
				switch ( match_result.alignment_center ) {
					case AlignmentSystem::N:
						C2N_N_Align::micromanage_takeoff( scaffold.full_coord(), primary_component.scaffold_gap_range.end(), full_coord, es_primary_loop.end() );
						break;
					case AlignmentSystem::CA:
						C2N_CA_Align::micromanage_takeoff( scaffold.full_coord(), primary_component.scaffold_gap_range.end(), full_coord, es_primary_loop.end() );
						break;
					case AlignmentSystem::C:
						C2N_C_Align::micromanage_takeoff( scaffold.full_coord(), primary_component.scaffold_gap_range.end(), full_coord, es_primary_loop.end() );
						break;
					default:
						break;
				}
				break;

			default:
				break;
		}
	}

	// make Eposition
	FArray3D_float Eposition( 3, param::MAX_POS, total_residue );
	full_coord_to_Eposition( total_residue, full_coord, Eposition );

	// set coordinates
	epitope_scaffold.set_coords( false, Eposition, full_coord, false ); // internally recomputes all angles

	// now change any dihedral angles indicated
	if ( es_dihedrals.size() > 0 ) {
		for ( std::set< DihedralInfo >::const_iterator d = es_dihedrals.begin(), de = es_dihedrals.end(); d != de; ++d ) {
			d->apply( epitope_scaffold ); // apply dihedral
		}
	}

	// Now handle the case where we want to transfer sidechains from the epitope.
	for ( SortedMatchComponents::const_iterator c = components.begin(), ce = components.end(); c != ce; ++c ) {
		MatchComponent const & component = *c;
		FArray3D_float aligned_epitope_full_coord = aligned_epitope.full_coord(); // cache

		if ( graft_info.do_graft_sc( component ) ) { // graft sidechains from epitope, taking into account keep_natro
			ResidueRange const & old_loop = component.native_loop_subrange;
			ResidueRange const & new_loop = native_to_new_epitope.find( old_loop )->second;
			Integer const offset = new_loop.begin() - old_loop.begin();
			Integer const epitope_loop_offset = component.loop_subrange.begin() - old_loop.begin();

			for ( Integer epitope_res = old_loop.begin(), last_epitope_res = old_loop.end(); epitope_res <= last_epitope_res; ++epitope_res ) {
				if ( keep_natro.size() == 0 || keep_natro.find( epitope_res ) != keep_natro.end() ) {
					Integer const es_res = epitope_res + offset;
					Integer const epitope_loop_res = epitope_res + epitope_loop_offset;
					epitope_scaffold.copy_sidechain( es_res,
					                                 aligned_epitope.res( epitope_loop_res ),
					                                 aligned_epitope.res_variant( epitope_loop_res ),
					                                 aligned_epitope_full_coord.a( 1, 1, epitope_loop_res ) );
				}
			}
		}
	}

	// correct pdb info data
	char const scaffold_chain_id = scaffold.pdb_info().res_chain( 1 );
	for ( Integer i = 1, ie = epitope_scaffold.total_residue(); i <= ie; ++i ) {
		epitope_scaffold.pdb_info().set_pdb_chain( i, scaffold_chain_id );
		epitope_scaffold.pdb_info().set_pdb_res( i, i );
		epitope_scaffold.pdb_info().set_pdb_insert_let( i, ' ' );
	}
	epitope_scaffold.pdb_info().set_use_pdb_numbering( true ); // use pdb info on output

	// save epitope tracking data
	native_to_new_epitope_ = native_to_new_epitope;
	epitope_tracking_ = epitope_tracking;

	// save internal closures to attempt
	closures_to_attempt_.clear();
	closures_to_attempt_.insert( primary_closures_to_attempt.begin(), primary_closures_to_attempt.end() );
	closures_to_attempt_.insert( secondary_closures_to_attempt.begin(), secondary_closures_to_attempt.end() );

	// create information for idealizing cutpoints
	std::map< Integer, Real > cut_to_torsion; // distinction between phi/psi is implicitly encoded due to constant n2c directionality of cutpoint
	for ( std::set< LoopClosureInfo >::const_iterator c = closures_to_attempt_.begin(), ce = closures_to_attempt_.end(); c != ce; ++c ) {
		LoopClosureInfo const & closure_info = *c;
		Integer const nside = closure_info.cut();
		Integer const cside = closure_info.cut() + 1;

		if ( !closure_info.is_closed() ) {
			std::set< DihedralInfo >::const_iterator di; // for lookup in changed dihedrals

			// query n-side psi
			if ( ( di = es_dihedrals.find( DihedralInfo( DihedralInfo::PSI, nside, 0.0 ) ) ) != es_dihedrals.end() ) { // check dihedrals first, in which case we take everything from epitope-scaffold
				cut_to_torsion[ nside ] = di->angle();
			} else { // take from either original epitope or original scaffold
				Integer const prior_residue_number = new_to_old[ nside ];
				assert( prior_residue_number != 0 );

				if ( prior_residue_number < 0 ) { // take from epitope
					cut_to_torsion[ nside ] = aligned_epitope.psi( -prior_residue_number );
				} else { // take from scaffold
					cut_to_torsion[ nside ] = scaffold.psi( prior_residue_number );
				}
			}

			// query c-side phi
			if ( ( di = es_dihedrals.find( DihedralInfo( DihedralInfo::PHI, cside, 0.0 ) ) ) != es_dihedrals.end() ) { // check dihedrals first, in which case we take everything from epitope-scaffold
				cut_to_torsion[ cside ] = di->angle();
			} else { // take from either original epitope or original scaffold
				Integer const prior_residue_number = new_to_old[ cside ];
				assert( prior_residue_number != 0 );

				if ( prior_residue_number < 0 ) { // take from epitope
					cut_to_torsion[ cside ] = aligned_epitope.phi( -prior_residue_number );
				} else { // take from scaffold
					cut_to_torsion[ cside ] = scaffold.phi( prior_residue_number );
				}
			}
		}
	}

	// score/handle cutpoints:
	// idealize cutpoints that are not closed, this also fixes the incorrect pose bonds data due to non-ideal pose
	for ( std::set< LoopClosureInfo >::const_iterator c = closures_to_attempt_.begin(), ce = closures_to_attempt_.end(); c != ce; ++c ) {
		LoopClosureInfo const & closure_info = *c;

		// need to idealize first
		if ( !closure_info.is_closed() ) {
			// idealize all moveable
			if ( idealize_loop_geometry_ ) {
				for ( std::set< Integer >::const_iterator m = closure_info.moveable_residues().begin(), me = closure_info.moveable_residues().end(); m != me; ++m ) {
					if ( non_idealizable_residues.find( *m ) == non_idealizable_residues.end() ) {
						epitope_scaffold.insert_ideal_bonds( *m, *m );
					}
				}
			}
			// force cutpoint to retain n-side psi and c-side phi
			idealize_cutpoint_retaining_psiphi( epitope_scaffold, closure_info.cut(), cut_to_torsion );
		}

		// score all
		closure_info.set_chainbreak_score( score_cut_in_Pose_linear( epitope_scaffold, closure_info.cut(), 1 ) ); // overlap = 1
		score_rama_for_moveable( epitope_scaffold, closure_info );

		// only mark closure for those loops that were not already marked as closed in above
		// procedure
		if ( !closure_info.is_closed() ) {
			closure_info.set_is_closed( closure_info.chainbreak_score() <= chainbreak_criterion_ &&
					                    closure_info.ramas_are_not_above( local_rama_criterion_ ) );
		}
	}

	// store Pose in local object data
	epitope_scaffold_ = epitope_scaffold;
	archived_epitope_scaffold_ = epitope_scaffold;

	// need to archive original jumps
	archive_epitope_rb_jumps();

} // end make_predesign_epitope_scaffold


/// @brief   make design matrix using Ab to find intra/inter design positions
/// @note    takes into account internal keep_natro_ set
// TODO: use single octree rather than three, or better yet, write NearestNeighbors class to handle
//       most generic inquiries, with the ability to turn off and on points
FArray2D< bool >
EpitopeScaffold::make_design_info()
{
	using pose_ns::Score_weight_map;

	// design matrix
	FArray2D< bool > design_matrix( param::MAX_AA(), epitope_scaffold_.total_residue(), false );

	// epitope scaffold range
	ResidueRange es_range( 1, epitope_scaffold_.total_residue() );
	if ( Ab_is_connected_ ) {
		es_range = ResidueRange( 1, Ab_connection_.segment_range().begin() - 1 );
	}

	// make octree containing epitope-scaffold for distance checks
	BoundingBox bb = bounding_box( epitope_scaffold_, es_range );
	Real max_distance_cutoff = std::max( static_cast< Real >( 6.0 ),
	                                    std::max( intra_design_position_cutoff_, inter_design_position_cutoff_ ) );
	max_distance_cutoff = std::max( max_distance_cutoff, Ab_repack_cutoff_ );
	Octree< AtomPoint > es_octree( max_distance_cutoff, bb.lower_corner(), bb.upper_corner() );
	fill_octree( es_octree, epitope_scaffold_, es_range );

	// make octree containing epitope-scaffold sidechains
	Octree< AtomPoint > es_octree_sc( max_distance_cutoff, bb.lower_corner(), bb.upper_corner() );
	fill_octree_sc( es_octree_sc, epitope_scaffold_, es_range );

	// make octree containing Ab
	BoundingBox bb_Ab;
	Octree< AtomPoint > Ab_octree;
	if ( Ab_is_connected_ ) {
		bb_Ab = bounding_box( epitope_scaffold_, Ab_connection_.segment_range() );
		Ab_octree = Octree< AtomPoint >( max_distance_cutoff, bb_Ab.lower_corner(), bb_Ab.upper_corner() );
		fill_octree( Ab_octree, epitope_scaffold_, Ab_connection_.segment_range() );
	}

	// design position data
	std::set< Integer > intra_design_positions;
	std::set< Integer > inter_design_positions;
	//vds add new set for moveable residues; these and their neighbours will be designed
	std::set< Integer > moved_design_positions;
	//vds set of residues that used to be near moveable, in original scaffold before moves; these were recorded earlier
	//vds and need to be redesigned in case moving residues create voids near these ones
	std::set< Integer >  originally_near_moveable_residues = residues_originally_near_moveable_;

	// gather inter design positions: run over epitope-scaffold residues and check against Ab
	if ( Ab_is_connected_ ) {
		for ( Integer res = es_range.begin(), last_res = es_range.end(); res <= last_res; ++res ) {
			std::set< Integer > nr = epigraft::design::near_residues_to_sc( inter_design_position_cutoff_, Ab_octree, epitope_scaffold_, res );
			if ( nr.size() > 0 ) {
				inter_design_positions.insert( res );
			}
		}
	}

	// gather intra design positions: run over epitope components and find nearby epitope-scaffold sidechains
	for ( std::map< ResidueRange, ResidueRange >::const_iterator i = native_to_new_epitope_.begin(), ie = native_to_new_epitope_.end(); i != ie; ++i ) {
		ResidueRange const & es_loop = i->second;

		for ( Integer res = es_loop.begin(); res <= es_loop.end(); ++res ) {
			std::set< Integer > nr = epigraft::design::near_residues_to_sc( intra_design_position_cutoff_, es_octree_sc, epitope_scaffold_, res );
			if ( keep_natro_.find( res ) != keep_natro_.end() ) {
				nr.erase( res ); // need to erase residue itself, which is picked up on by near neighbor routine
			}
			intra_design_positions.insert( nr.begin(), nr.end() );
		}
	}

	if ( allow_epitope_design_ ) { //vds ensure epi to be redesigned, except those in keep_natro (taken out below)
		for ( std::map< ResidueRange, ResidueRange >::const_iterator i = native_to_new_epitope_.begin(), ie = native_to_new_epitope_.end(); i != ie; ++i ) {
			ResidueRange const & es_loop = i->second;

			for ( Integer res = es_loop.begin(); res <= es_loop.end(); ++res ) {
				intra_design_positions.insert( res );
			}
		}
	}

	//vds gather more intra-design positions: run over moveable residues and add them to intra_design_positions
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt_.begin(), ie = closures_to_attempt_.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;

		intra_design_positions.insert( closure_info.moveable_residues().begin(), closure_info.moveable_residues().end() );
		moved_design_positions.insert( closure_info.moveable_residues().begin(), closure_info.moveable_residues().end() );
	}

	//vds gather more intra-design positions: run over moveable residues and find nearby epitope-scaffold sidechains
	for ( std::set< Integer >::const_iterator m = moved_design_positions.begin(), me = moved_design_positions.end(); m != me; ++m ) {
		std::set< Integer > nr = epigraft::design::near_residues_to_sc( intra_design_position_cutoff_, es_octree_sc, epitope_scaffold_, *m );
		if ( nr.size() > 0 ) {
			intra_design_positions.insert( nr.begin(), nr.end() );
		}
	}

	//vds gather more intra-design positions: residues that used to be near moveable residues in the original scaffold
	intra_design_positions.insert( originally_near_moveable_residues.begin(), originally_near_moveable_residues.end() );

	// remove inter- positions from intra- positions
	for ( std::set< Integer >::const_iterator i = inter_design_positions.begin(), ie = inter_design_positions.end(); i != ie; ++i ) {
		intra_design_positions.erase( *i );
	}

	// remove epitope positions from intra and inter design positions,
	// taking into account keep_natro_
	if ( keep_natro_.size() > 0 ) {
		erase_keep_natro_residues( intra_design_positions );
		erase_keep_natro_residues( inter_design_positions );
	} else {
		erase_epitope_residues( intra_design_positions );
		erase_epitope_residues( inter_design_positions );
	}

	// remove complementarity design positions from intra/inter design positions
	erase_complementarity_residues( intra_design_positions );
	erase_complementarity_residues( inter_design_positions );

	// begin setup of design matrix
	// first identify disulfides and make sure they aren't killed during design
	disulfides::options::find_disulf = true;
	Score_weight_map weight_map( score12 );
	epitope_scaffold_.score( weight_map );

	// remove cys-in-disulfide and pro from intra- design positions
	for ( std::set< Integer >::const_iterator r = intra_design_positions.begin(), re = intra_design_positions.end(); r != re; ++r ) {
		Integer const res = *r;

		// no design for cys or pro
		if ( disulfides::BOUNDARY::cys_res_in_disulf( res ) ||
		     ( epitope_scaffold_.res( res ) == param_aa::aa_pro &&
		       moved_design_positions.find( res ) == moved_design_positions.end() )
		) {
			intra_design_positions.erase( r );
		}
	}

	// inter- design positions, typically just AGST
	std::set< Integer > allowed_aa_inter;
	if ( allow_ALLAA_at_inter_design_positions_ ) {

		for ( Integer i = 1; i <= 20; ++i ) {
			allowed_aa_inter.insert( i );
		}
		allowed_aa_inter.erase( param_aa::aa_cys );

	} else { // default, AGST
		allowed_aa_inter.insert( param_aa::aa_ala );
		allowed_aa_inter.insert( param_aa::aa_gly );
		allowed_aa_inter.insert( param_aa::aa_ser );
		allowed_aa_inter.insert( param_aa::aa_thr );
	}
	for ( std::set< Integer >::const_iterator r = inter_design_positions.begin(), re = inter_design_positions.end(); r != re; ++r ) {
		Integer const & res = *r;
		for ( std::set< Integer >::const_iterator aa = allowed_aa_inter.begin(), aa_e = allowed_aa_inter.end(); aa != aa_e; ++aa ) {
			design_matrix( *aa, res ) = true;
		}
	}

	// intra- design positions, all allowed except for cys
	std::set< Integer > allowed_aa_intra;
	for ( Integer i = 1; i <= 20; ++i ) {
		allowed_aa_intra.insert( i );
	}
	allowed_aa_intra.erase( param_aa::aa_cys );
	for ( std::set< Integer >::const_iterator r = intra_design_positions.begin(), re = intra_design_positions.end(); r != re; ++r ) {
		Integer const & res = *r;

		for ( std::set< Integer >::const_iterator aa = allowed_aa_intra.begin(), aa_e = allowed_aa_intra.end(); aa != aa_e; ++aa ) {
			design_matrix( *aa, res ) = true;
		}
	}

	// Ab repack, if requested
	std::set< Integer > Ab_repack_positions;
	if ( Ab_is_connected_ && allow_Ab_repack_ ) {
		// gather Ab repack positions
		for ( Integer res = Ab_connection_.segment_range().begin(), last_res = Ab_connection_.segment_range().end(); res <= last_res; ++res ) {
			std::set< Integer > nr = epigraft::design::near_residues_to_sc( Ab_repack_cutoff_, es_octree, epitope_scaffold_, res );

			// no repack for cys
			if ( disulfides::BOUNDARY::cys_res_in_disulf( res ) ) {
				continue;
			}

			// no repack for pro
			if ( epitope_scaffold_.res( res ) == param_aa::aa_pro ) {
				continue;
			}

			if ( nr.size() > 0 ) {
				Ab_repack_positions.insert( res );
			}
		}

		// set Ab repack in design matrix
		for ( std::set< Integer >::const_iterator r = Ab_repack_positions.begin(), re = Ab_repack_positions.end(); r != re; ++r ) {
			Integer const & res = *r;

			design_matrix( epitope_scaffold_.res( res ), res ) = true;
		}
	}

	// override with specific aai settings in graft info
	override_design_info_using_graft_info( design_matrix );

	// override with complementarity design information
	override_design_info_using_complementarity_info( design_matrix );

	// epitope repack, if requested; set this last as it overrides other repack epitope settings
	if ( allow_epitope_repack_ ) {
		for ( std::map< ResidueRange, ResidueRange >::const_iterator i = native_to_new_epitope_.begin(), ie = native_to_new_epitope_.end(); i != ie; ++i ) {
			ResidueRange const & es_loop = i->second;

			for ( Integer res = es_loop.begin(); res <= es_loop.end(); ++res ) {
				// no repack for cys
				if ( disulfides::BOUNDARY::cys_res_in_disulf( res ) ) {
					continue;
				}

				// no repack for pro
				if ( epitope_scaffold_.res( res ) == param_aa::aa_pro ) {
					continue;
				}

				design_matrix( epitope_scaffold_.res( res ), res ) = true;
			}
		}
	}

	// record in local object data
	intra_design_positions_ = intra_design_positions;
	inter_design_positions_ = inter_design_positions;
	Ab_repack_positions_ = Ab_repack_positions;

	return design_matrix;
}


/// @brief save loop checkpoint file
void
EpitopeScaffold::save_loop_checkpoint( std::string const & filename ) const
{
	std::ofstream out( filename.c_str() );

	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt_.begin(), ie = closures_to_attempt_.end(); i != ie; ++i ) {
		out << i->loop_range().left() << " " << i->loop_range().right() << " " << i->cut() << " " << closed_during_trajectory( *i ) << '\n';
	}

	out.close();
}


/// @brief load loop checkpoint file
void
EpitopeScaffold::load_loop_checkpoint( std::string const & filename )
{
	std::ifstream in( filename.c_str() );

	Integer left = 0, right = 0, cut = 0;
	bool closed = false;
	std::string line;
	while ( getline( in, line ) ) {
		std::istringstream ss( line );
		ss >> left >> right >> cut >> closed;
		closed_during_trajectory_[ LoopClosureInfo( ResidueRange( left, right ), cut ) ] = closed;
	}

	in.close();
}


/// @brief override design info using graft info settings
void
EpitopeScaffold::override_design_info_using_graft_info(
	FArray2D< bool > & design_matrix
)
{
	std::set< ResidueRange> components = all_components();

	for ( std::set< ResidueRange >::const_iterator c = components.begin(), ce = components.end(); c != ce; ++c ) {
		ResidueRange const & old_scaffold_gap = epitope_to_old_gap( *c );
		String const aa_string = graft_info_.aa_string( old_scaffold_gap );

		Integer const & s_tether_n = graft_info_.scaffold_tether_residues_nterm( old_scaffold_gap );
		Integer const & s_linker_n = graft_info_.scaffold_linker_residues_nterm( old_scaffold_gap );
		Integer const & e_linker_n = graft_info_.epitope_linker_residues_nterm( old_scaffold_gap );

		assert( e_linker_n >= 0 ); // safety check

		Integer res = c->begin() - s_tether_n - s_linker_n - e_linker_n;

		for ( String::const_iterator i = aa_string.begin(), ie = aa_string.end(); i != ie; ++i, ++res ) {

			switch ( *i ) {

				case GraftInfo::FREEZE: {
					for ( Integer a = 1; a <= 20; ++a ) {
						design_matrix( a , res ) = false;
					}
					break;
				}

				case GraftInfo::DEFAULT: { // don't change the current settings
					break;
				}

				case GraftInfo::REPACK: {
					for ( Integer a = 1; a <= 20; ++a ) {
						design_matrix( a , res ) = false;
					}
					design_matrix( epitope_scaffold_.res( res ), res );
					break;
				}

				case GraftInfo::ANY_AA: {
					for ( Integer a = 1; a <= 20; ++a ) {
						design_matrix( a , res ) = true;
					}
					break;
				}

				case GraftInfo::NO_CP: {
					for ( Integer a = 1; a <= 20; ++a ) {
						design_matrix( a , res ) = true;
					}
					design_matrix( param_aa::aa_cys, res ) = false;
					design_matrix( param_aa::aa_pro, res ) = false;
					break;
				}

				case GraftInfo::NONPOLAR: {
					for ( Integer a = 1; a <= 20; ++a ) {
						design_matrix( a, res ) = param_aa::aa_is_nonpolar( a );
					}
					break;
				}

				case GraftInfo::AROMATIC: {
					for ( Integer a = 1; a <= 20; ++a ) {
						design_matrix( a, res ) = param_aa::aa_is_aromatic( a );
					}
					break;
				}

				case GraftInfo::POLAR: {
					for ( Integer a = 1; a <= 20; ++a ) {
						design_matrix( a, res ) = param_aa::aa_is_polar( a );
					}
					break;
				}

				case GraftInfo::CHARGED: {
					for ( Integer a = 1; a <= 20; ++a ) {
						design_matrix( a, res ) = param_aa::aa_is_charged( a );
					}
					break;
				}

				case GraftInfo::POSITIVE: {
					for ( Integer a = 1; a <= 20; ++a ) {
						design_matrix( a, res ) = param_aa::aa_is_positively_charged( a );
					}
					break;
				}

				case GraftInfo::NEGATIVE: {
					for ( Integer a = 1; a <= 20; ++a ) {
						design_matrix( a, res ) = param_aa::aa_is_negatively_charged( a );
					}
					break;
				}

				default: { // set from amino acid type
					for ( Integer a = 1; a <= 20; ++a ) {
						design_matrix( a , res ) = false;
					}
					Integer num;
					num_from_res1( *i, num );
					design_matrix( num, res ) = true;
					break;
				}

			} // end switch
		} // end for each residue and it's instructions
	}

	// deprecated epitope repack for individual components set up in graft info file
	for ( std::map< ResidueRange, EpitopeTrack >::const_iterator i = epitope_tracking_.begin(), ie = epitope_tracking_.end(); i != ie; ++i ) {
		ResidueRange const & component = i->first;
		if ( graft_info_.repacking_epitope_residues( epitope_to_old_gap( component ) ) ) {
			for ( Integer res = component.begin(); res <= component.end(); ++res ) {
				design_matrix( epitope_scaffold_.res( res ), res ) = true;
			}
		}
	}
}


/// @brief override design info using complementarity design and shell info
/// @param[out] design_matrix  modified design matrix only if Ab is attached
/// @return false if Ab is attached, true otherwise
bool
EpitopeScaffold::override_design_info_using_complementarity_info(
	FArray2D< bool > & design_matrix
)
{
	if ( !Ab_is_connected() ) {
		return false;
	}

	for ( std::set< Integer >::const_iterator i = complementarity_design_positions_.begin(), ie = complementarity_design_positions_.end(); i != ie; ++i ) {
		for ( Integer a = 1; a <= 20; ++a ) {
			design_matrix( a, *i ) = false;
		}

		design_matrix( epitope_scaffold_.res( *i ), *i ) = true;
	}

	// repack complementarity positions on the antibody
	for ( std::set< Integer >::const_iterator i = complementarity_Ab_positions_.begin(), ie = complementarity_Ab_positions_.end(); i != ie; ++i ) {
		design_matrix( epitope_scaffold_.res( *i ), *i ) = true;
	}

	if ( complementarity_shell_redesign() || complementarity_shell_repack() ) {

		// epitope scaffold range
		ResidueRange es_range( 1, Ab_connection_.segment_range().begin() - 1 );

		// make octree containing epitope-scaffold sidechains to find shell
		Real max_distance_cutoff = std::max( static_cast< Real >( 6.0 ), complementarity_shell_cutoff_ );
		BoundingBox bb = bounding_box( epitope_scaffold_, es_range );
		Octree< AtomPoint > es_octree_sc( max_distance_cutoff, bb.lower_corner(), bb.upper_corner() );
		fill_octree_sc( es_octree_sc, epitope_scaffold_, es_range );

		// find shell positions
		std::set< Integer > shell_positions;
		for ( std::set< Integer >::const_iterator i = complementarity_design_positions_.begin(), ie = complementarity_design_positions_.end(); i != ie; ++i ) {
			std::set< Integer > nr = epigraft::design::near_residues( complementarity_design_position_cutoff_, es_octree_sc, epitope_scaffold_, *i );
			shell_positions.insert( nr.begin(), nr.end() );
		}

		// remove design positions from shell positions
		for ( std::set< Integer >::const_iterator i = complementarity_design_positions_.begin(), ie = complementarity_design_positions_.end(); i != ie; ++i ) {
			shell_positions.erase( *i );
		}

		// either redesign or repack shell around scaffold
		for ( std::set< Integer >::const_iterator i = shell_positions.begin(), ie = shell_positions.end(); i != ie; ++i ) {
			if ( complementarity_shell_redesign() ) {

				for ( Integer a = 1; a <= 20; ++a ) {
					design_matrix( a, *i ) = true;
				}

				// no cys or pro
				design_matrix( param_aa::aa_cys, *i ) = false;
				design_matrix( param_aa::aa_pro, *i ) = false;

			} else if ( complementarity_shell_repack() ) {
				design_matrix( epitope_scaffold_.res( *i ), *i ) = true;
			}
		}

		// store positions
		complementarity_shell_positions_ = shell_positions;
	}

	return true;
}


/// @brief erase keep_natro_ residues (correctly offset-ed to current epitope scaffold indexing) from given set
void
EpitopeScaffold::erase_keep_natro_residues(
	std::set< Integer > & residues
)
{
	if ( keep_natro_.size() == 0 ) {
		// nothing in keep natro, return
		return;
	}

	for ( std::map< ResidueRange, ResidueRange >::const_iterator i = native_to_new_epitope_.begin(), ie = native_to_new_epitope_.end(); i != ie; ++i ) {
		ResidueRange const & es_loop = i->second;
		ResidueRange const & old_loop = new_to_native_epitope( es_loop );
		Integer const offset = old_loop.begin() - es_loop.begin();

		for ( Integer res = es_loop.begin(); res <= es_loop.end(); ++res ) {
			if ( keep_natro_.find( res + offset ) != keep_natro_.end() ) {
				residues.erase( res );
			}
		}
	}
}


/// @brief erase all epitope residues from given set
void
EpitopeScaffold::erase_epitope_residues(
	std::set< Integer > & residues
)
{
	for ( std::map< ResidueRange, ResidueRange >::const_iterator i = native_to_new_epitope_.begin(), ie = native_to_new_epitope_.end(); i != ie; ++i ) {
		ResidueRange const & es_loop = i->second;

		for ( Integer res = es_loop.begin(); res <= es_loop.end(); ++res ) {
				residues.erase( res );
		}
	}
}


/// @brief erase all complementarity design residues from given set
void
EpitopeScaffold::erase_complementarity_residues(
	std::set< Integer > & residues
)
{
	for ( std::set< Integer >::const_iterator i = complementarity_design_positions_.begin(), ie = complementarity_design_positions_.end(); i != ie; ++i ) {
		residues.erase( *i );
	}
}


/// @brief return fixed residues, this includes epitope/keep_natro and Ab if Ab is not allowed
/// @brief to repack
std::set< Integer >
EpitopeScaffold::get_fixed_residues()
{
	std::set< Integer > fixed_residues;

	for ( std::map< ResidueRange, ResidueRange >::const_iterator i = native_to_new_epitope_.begin(), ie = native_to_new_epitope_.end(); i != ie; ++i ) {
		ResidueRange const & es_loop = i->second;
		ResidueRange const & old_loop = new_to_native_epitope( es_loop );
		Integer const offset = old_loop.begin() - es_loop.begin();

		for ( Integer res = es_loop.begin(); res <= es_loop.end(); ++res ) {
			if ( keep_natro_.size() == 0 || keep_natro_.find( res + offset ) != keep_natro_.end() ) {
				fixed_residues.insert( res );
			}
		}
	}

	if ( Ab_is_connected_ && !allow_Ab_repack_ ) {
		for ( Integer i = Ab_connection_.segment_range().begin(), ie = Ab_connection_.segment_range().end(); i <= ie; ++i ) {
			fixed_residues.insert( i );
		}
	}

	return fixed_residues;
}

/// @brief find residues on scaffold near moveable loop residues dictated by intra design position cutoff
//vds
std::set< Integer >
EpitopeScaffold::find_residues_near_moveable(
	Pose const & scaffold,
	std::set< LoopClosureInfo > const & closures
)
{
	// make octree containing scaffold for distance checks
	BoundingBox bb = bounding_box( scaffold );
	Real max_distance_cutoff = std::max( static_cast< Real >( 6.0 ),
	                                    std::max( intra_design_position_cutoff_, inter_design_position_cutoff_ ) );
	max_distance_cutoff = std::max( max_distance_cutoff, Ab_repack_cutoff_ );

	// make octree containing scaffold sidechains
	Octree< AtomPoint > octree_sc( max_distance_cutoff, bb.lower_corner(), bb.upper_corner() );
	fill_octree_sc( octree_sc, scaffold );

	std::set< Integer > residues_near_moveable;
	for ( std::set< LoopClosureInfo >::const_iterator i = closures.begin(), ie = closures.end(); i != ie; ++i ) {
		std::set< Integer > const & moveable = i->moveable_residues();

		for ( std::set< Integer >::const_iterator m = moveable.begin(), me = moveable.end(); m != me; ++m ) {
			std::set< Integer > nr = epigraft::design::near_residues_to_sc( intra_design_position_cutoff_, octree_sc, scaffold, *m );
			if ( nr.size() > 0 ) {
				residues_near_moveable.insert( nr.begin(), nr.end() );
			}
		} // foreach moveable residue
	} // foreach loop

	return residues_near_moveable;
}


/// @brief identify scaffold - antibody interface residues given distance cutoff
/// @details computes interface residues by comparing sidechains of each component
/// @details against the full residues of the other
std::set< Integer >
EpitopeScaffold::identify_interface_residues(
	Real const & interface_distance
)
{
	if ( !Ab_is_connected_ ) {
		std::cerr << "WARNING EpitopeScaffold::identify_interface_residues() : called without antibody connection, returning empty set!" << std::endl;
		return std::set< Integer >();
	}

	Real max_distance_cutoff = std::max( interface_distance, static_cast< Real >( 6.0 ) );

	// for location of epitope scaffold interface sidechains
	ResidueRange es_range( 1, Ab_connection_.segment_range().begin() - 1 );
	BoundingBox bb = bounding_box( epitope_scaffold_, es_range );
	Octree< AtomPoint > es_octree_sc( max_distance_cutoff, bb.lower_corner(), bb.upper_corner() );
	fill_octree_sc( es_octree_sc, epitope_scaffold_, es_range );

	// for location of antibody interface sidechains
	BoundingBox bb_Ab = bounding_box( epitope_scaffold_, Ab_connection_.segment_range() );
	Octree< AtomPoint > ab_octree_sc( max_distance_cutoff, bb_Ab.lower_corner(), bb_Ab.upper_corner() );
	fill_octree_sc( ab_octree_sc, epitope_scaffold_, Ab_connection_.segment_range() );

	std::set< Integer > interface_positions;

	// epitope scaffold interface positions
	for ( Integer res = Ab_connection_.segment_range().begin(), last_res = Ab_connection_.segment_range().end(); res <= last_res; ++res ) {
		std::set< Integer > nr = near_residues( interface_distance, es_octree_sc, epitope_scaffold_, res );
		interface_positions.insert( nr.begin(), nr.end() );
	}

	// antibody interface positions
	for ( Integer res = 1, last_res = Ab_connection_.segment_range().begin() - 1; res <= last_res; ++res ) {
		std::set< Integer > nr = near_residues( interface_distance, ab_octree_sc, epitope_scaffold_, res );
		interface_positions.insert( nr.begin(), nr.end() );
	}

	return interface_positions;
}


/// @brief check to see if a loop's bb is to be grafted by querying internal GraftInfo
bool const &
EpitopeScaffold::do_graft_bb(
	LoopClosureInfo const & closure
) const
{
	ResidueRange component = component_from_closure( closure );
	return graft_info_.do_graft_bb( epitope_to_old_gap( component ) );
}


/// @brief check to see if a component's sc is to be grafted by querying internal GraftInfo
bool const &
EpitopeScaffold::do_graft_sc(
	ResidueRange const & component
) const
{
	return graft_info_.do_graft_sc( epitope_to_old_gap( component ) );
}


/// @brief setup proper allow bb move for set of loops
void
EpitopeScaffold::setup_proper_allow_bb_move(
	std::set< LoopClosureInfo > const & closures
)
{
	epigraft::design::setup_proper_allow_bb_move( epitope_scaffold_, closures );
}


/// @brief idealize cutpoint but retain psi on the n-side and phi on the c-side of the cut
/// @brief using given values if idealize_loop_geometry is active
/// @details if values don't exist in map, then regular idealization occurs
/// @note  this is useful when it's necessary to retain the existing torsions of the cut
void
EpitopeScaffold::idealize_cutpoint_retaining_psiphi(
	Pose & pose,
	Integer const & cut,
	std::map< Integer, Real > const & original_torsions
)
{
	if ( idealize_loop_geometry_ ) {
		epigraft::design::idealize_cutpoint_retaining_psiphi( pose, cut, original_torsions );
	}
}


/// @brief idealize cutpoint but retain psi on the n-side and phi on the c-side of the cut
/// @brief using given values if idealize_loop_geometry is active
/// @note  this is useful when it's necessary to retain the existing torsions of the cut
void
EpitopeScaffold::idealize_cutpoint_retaining_psiphi(
	Pose & pose,
	Integer const & cut,
	Real const & nside_psi,
	Real const & cside_phi
)
{
	if ( idealize_loop_geometry_ ) {
		epigraft::design::idealize_cutpoint_retaining_psiphi( pose, cut, nside_psi, cside_phi );
	}
}


/// @brief get proper secondary structure string for centroid level loop closure
String
EpitopeScaffold::loop_closure_ss_string() const
{
	std::ostringstream ss;

	for ( Integer r = 1, re = epitope_scaffold_.total_residue(); r <= re; ++ r ) {
		ss << epitope_scaffold_.secstruct( r );
	}

	return ss.str();
}


/// @brief  get proper amino acid identity string for centroid level loop closure
/// @return if using sequence biased fragments, returns the amino acid string for
///         centroid level loop closure
String
EpitopeScaffold::loop_closure_seq_string() const
{
	String seq_string( epitope_scaffold_.total_residue(), GraftInfo::ANY_AA );

	if ( using_sequence_biased_fragments() ) {

		std::set< ResidueRange> components = all_components();

		for ( std::set< ResidueRange >::const_iterator c = components.begin(), ce = components.end(); c != ce; ++c ) {
			ResidueRange const & old_scaffold_gap = epitope_to_old_gap( *c );
			String const aa_string = graft_info_.aa_string( old_scaffold_gap );

			Integer const & s_tether_n = graft_info_.scaffold_tether_residues_nterm( old_scaffold_gap );
			Integer const & s_linker_n = graft_info_.scaffold_linker_residues_nterm( old_scaffold_gap );
			Integer const & e_linker_n = graft_info_.epitope_linker_residues_nterm( old_scaffold_gap );

			assert( e_linker_n >= 0 ); // safety check

			Integer res = c->begin() - s_tether_n - s_linker_n - e_linker_n;

			for ( String::const_iterator i = aa_string.begin(), ie = aa_string.end(); i != ie; ++i, ++res ) {

				switch ( *i ) {

					case GraftInfo::FREEZE: {
						break;
					}

					case GraftInfo::DEFAULT: { // don't change the current settings
						break;
					}

					case GraftInfo::REPACK: {
						break;
					}

					default: { // set from instruction or amino acid type
						seq_string.at( res - 1 ) = *i;
						break;
					}

				} // end switch
			} // end for each residue and it's instructions
		}

	} // if using sequence biased fragment insertion

	return seq_string;
}


/// @brief update old epitope-scaffold component ranges to altered numbering
/// @param[in] map from residues with old epitope-scaffold component numbering to altered numbering
void
EpitopeScaffold::alter_maps(
	std::map< Integer, Integer > const & es_to_new_es
)
{
	std::map< ResidueRange, ResidueRange > old_to_new_epitope;
	std::map< ResidueRange, EpitopeTrack > epitope_tracking;

	for ( std::map< ResidueRange, EpitopeTrack >::const_iterator i = epitope_tracking_.begin(), ie = epitope_tracking_.end(); i != ie; ++i ) {
		ResidueRange original = i->first;
		ResidueRange altered( es_to_new_es.find( original.begin() )->second, es_to_new_es.find( original.end() )->second );

		old_to_new_epitope[ i->second.native_epitope ] = altered;
		epitope_tracking[ altered ].jump_label = i->second.jump_label;
		epitope_tracking[ altered ].native_epitope = i->second.native_epitope;
		epitope_tracking[ altered ].old_gap = i->second.old_gap;
		epitope_tracking[ altered ].archived_jump = i->second.archived_jump;
	}

	// set altered maps
	native_to_new_epitope_ = old_to_new_epitope;
	epitope_tracking_ = epitope_tracking;
}


/// @brief alter secondary component closures to single break instead of double break
/// @warning this will modify both working AND archive pose
/// @warning IMPORTANT: the way this is currently setup requires ALL possible moveable residues to be
/// @warning moveable
void
EpitopeScaffold::switch_secondaries_to_single_break(
	bool const & use_graft_info_settings
)
{
	using pose_ns::Fold_tree;
	typedef std::pair< Integer, Integer > JumpResidues;

	if ( !graft_info_.closing_any_secondary_as_single_break() ) {
		return;
	}

	Fold_tree altered_ft = epitope_scaffold_.fold_tree();
	std::set< Integer > residues_needing_idealization;

	// grab epitope secondary components in current epitope scaffold numbering
	ResidueRange const & primary = primary_component();
	std::set< ResidueRange > secondaries = secondary_components();

	for ( std::map< ResidueRange, EpitopeTrack >::iterator e = epitope_tracking_.begin(), ee = epitope_tracking_.end(); e != ee; ++e ) {

		ResidueRange const & component = e->first;
		EpitopeTrack & track = e->second;

		if ( track.jump_label > -1 && ( secondaries.find( component ) != secondaries.end() ) ) {

			if ( use_graft_info_settings && !graft_info_.closing_as_single_break( epitope_to_old_gap( component ) ) ) {
				continue;
			}

			// grab loop closures for this epitope component
			std::set< LoopClosureInfo > closures = component_closures_to_attempt( component );

			// check to see if epitope component's loops are fully closed
			bool fully_closed = false;
//			for ( std::set< LoopClosureInfo >::const_iterator c = closures.begin(), ce = closures.end(); c != ce; ++c ) {
//					fully_closed = fully_closed && c->is_closed();
//			}
			// check for override by do_graft_bb
			for ( std::set< LoopClosureInfo >::const_iterator c = closures.begin(), ce = closures.end(); c != ce; ++c ) {
					if ( !do_graft_bb( *c ) ) {
						fully_closed = true;
						break;
					}
			}

			// only merge if possible
			if ( !fully_closed ) {
				JumpResidues jump_res = jump_residues( track.jump_label );

				// figure out which vertex is within the primary component
				Integer on_primary, on_secondary;
				if ( primary.contains( jump_res.first ) ) {
					on_primary = jump_res.first;
					on_secondary = jump_res.second;
				} else {
					on_primary = jump_res.second;
					on_secondary = jump_res.first;
				}

				// grab moveable residues
				std::set< Integer > all_moveable;
				for ( std::set< LoopClosureInfo >::const_iterator c = closures.begin(), ce = closures.end(); c != ce; ++c ) {
					all_moveable.insert( c->moveable_residues().begin(), c->moveable_residues().end() );
					residues_needing_idealization.insert( c->moveable_residues().begin(), c->moveable_residues().end() );
				}

				// need to add the jump residue as well
				all_moveable.insert( on_secondary );
				residues_needing_idealization.insert( on_secondary );

				// setup new loop closure info
				// CUT is currently where the jump residue was on the secondary!
				ResidueRange const new_loop_range( *all_moveable.begin(), *all_moveable.rbegin() );
				ResidueRange const new_left_range( new_loop_range.begin(), on_secondary );
				ResidueRange const new_left_extent( new_loop_range.begin() - 1, on_secondary );
				ResidueRange const new_right_range( on_secondary + 1, new_loop_range.end() );
				ResidueRange const new_right_extent( on_secondary + 1, new_loop_range.end() + 1 );
				LoopClosureInfo new_closure( new_loop_range, on_secondary );
				new_closure.add_moveable_residues( all_moveable.begin(), all_moveable.end() );

				// remove old closures, add new
				for ( std::set< LoopClosureInfo >::const_iterator c = closures.begin(), ce = closures.end(); c != ce; ++c ) {
					closures_to_attempt_.erase( *c );
				}
				closures_to_attempt_.insert( new_closure );

				// modify new tree
				// CUT is currently where the jump residue was on the secondary!
				/// IMPORTANT: the way this is currently setup requires ALL possible moveable residues to be
				///            moveable
				altered_ft.delete_vertex( on_secondary ); // remove all edges incident to jump residue

				// remove peptide edges that overlap (jump edges remain intact)
				for ( Fold_tree::iterator i = altered_ft.begin(), ie = altered_ft.end(); i != ie; ++i ) {
					if ( i->label == Fold_tree::PEPTIDE ) {
						ResidueRange const edge = i->start < i->stop ? ResidueRange( i->start, i->stop ) : ResidueRange( i->stop, i->start );
						if ( edge.overlaps( new_left_range ) ) {
							i = altered_ft.delete_edge( i );
							--i; // need to back up after erase
							ie = altered_ft.end(); // erasure invalidates iterator
						} else if ( edge.overlaps( new_right_range ) ) {
							i = altered_ft.delete_edge( i );
							--i; // need to back up after erase
							ie = altered_ft.end(); // erasure invalidates iterator
						}
					}
				}

				// add edges
				altered_ft.add_edge( new_left_extent.begin(), new_left_extent.end(), Fold_tree::PEPTIDE ); // left
				altered_ft.add_edge( new_right_extent.end(), new_right_extent.begin(), Fold_tree::PEPTIDE ); // right

				// finally, modify jump in fold tree and graft tracking
				for ( Fold_tree::iterator i = altered_ft.begin(), ie = altered_ft.end(); i != ie; ++i ) {
					if ( i->label > track.jump_label ) {
						i->label = i->label - 1;
					}
				}
				for ( std::map< ResidueRange, EpitopeTrack >::iterator e2 = epitope_tracking_.begin(), e2e = epitope_tracking_.end(); e2 != e2e; ++e2 ) {
					if ( e2->second.jump_label > track.jump_label ) {
						e2->second.jump_label = e2->second.jump_label - 1;
					}
				}
				track.jump_label = -1; // no more jump
			}
		}
	}

	// finalize tree
	altered_ft.reorder( 1 );

	// set tree in both working and archive poses
	epitope_scaffold_.set_fold_tree( altered_ft );
	archived_epitope_scaffold_.set_fold_tree( altered_ft );

	// insert ideal bonds in moveable residues
	if ( idealize_loop_geometry_ ) {
		for ( std::set< Integer >::const_iterator i = residues_needing_idealization.begin(), ie = residues_needing_idealization.end(); i != ie; ++i ) {
			epitope_scaffold_.insert_ideal_bonds( *i, *i );
			archived_epitope_scaffold_.insert_ideal_bonds( *i, *i );
		}
	}

	// NOTE: closures are NOT rescored here because this method is called as a part of alter_closure_sites()
}


/// @brief   override secondary structure of closure sites according to graft info
/// @details secondary structure of both working and archived epitope scaffolds are
///          overridden following graft info settings
void
EpitopeScaffold::override_closure_site_secondary_structure()
{
	std::set< ResidueRange> components = all_components();

	// gather all residues for loops that are "immoveable" (no do_graft_bb or artificially closed)
	std::set< Integer > immoveable_residues;
	for ( std::set< LoopClosureInfo >::const_iterator c = closures_to_attempt_.begin(), ce = closures_to_attempt_.end(); c != ce; ++c ) {
		LoopClosureInfo const & closure_info = *c;

		// score and set closure status for potentially altered closure sites
		if ( ( !do_graft_bb( closure_info ) ) || closure_info.is_artificially_closed() ) {
			immoveable_residues.insert( closure_info.moveable_residues().begin(), closure_info.moveable_residues().end() );
		}
	}

	// figure out the proper uppercase character for GraftInfo::GROW_FROZEN
	std::string grow_frozen_str( 1, GraftInfo::GROW_FROZEN );
	ObjexxFCL::uppercase( grow_frozen_str );
	char const grow_frozen_char = grow_frozen_str.at( 0 );

	for ( std::set< ResidueRange >::const_iterator c = components.begin(), ce = components.end(); c != ce; ++c ) {
		ResidueRange const & old_scaffold_gap = epitope_to_old_gap( *c );
		String const ss_string = graft_info_.ss_string( old_scaffold_gap );

		Integer const & s_tether_n = graft_info_.scaffold_tether_residues_nterm( old_scaffold_gap );
		Integer const & s_linker_n = graft_info_.scaffold_linker_residues_nterm( old_scaffold_gap );
		Integer const & e_linker_n = graft_info_.epitope_linker_residues_nterm( old_scaffold_gap );

		assert( e_linker_n >= 0 ); // safety check

		Integer res = c->begin() - s_tether_n - s_linker_n - e_linker_n;

		for ( String::const_iterator i = ss_string.begin(), ie = ss_string.end(); i != ie; ++i, ++res ) {
			if ( *i != GraftInfo::KEEP_AND_MOVE  && *i != GraftInfo::FREEZE && *i != grow_frozen_char ) {
				if ( immoveable_residues.find( res ) == immoveable_residues.end() ) {
					epitope_scaffold_.set_secstruct( res, *i );
					archived_epitope_scaffold_.set_secstruct( res, *i );
				}
			}
		} // foreach ss instruction and residue
	} // foreach component
}


/// @brief   override mobility of grown residues according to graft info
/// @details looks for uppercase of GraftInfo::GROW_FROZEN and erases
///          those residues from the appropriate LoopClosureInfo
void
EpitopeScaffold::override_closure_site_grown_moveable_residues()
{
	std::set< ResidueRange> components = all_components();

	// figure out the proper uppercase character for GraftInfo::GROW_FROZEN
	std::string grow_frozen_str( 1, GraftInfo::GROW_FROZEN );
	ObjexxFCL::uppercase( grow_frozen_str );
	char const grow_frozen_char = grow_frozen_str.at( 0 );

	// find non-moveable grown residues
	std::set< Integer > immoveable;
	for ( std::set< ResidueRange >::const_iterator c = components.begin(), ce = components.end(); c != ce; ++c ) {
		ResidueRange const & old_scaffold_gap = epitope_to_old_gap( *c );
		String const ss_string = graft_info_.ss_string( old_scaffold_gap );

		Integer const & s_tether_n = graft_info_.scaffold_tether_residues_nterm( old_scaffold_gap );
		Integer const & s_linker_n = graft_info_.scaffold_linker_residues_nterm( old_scaffold_gap );
		Integer const & e_linker_n = graft_info_.epitope_linker_residues_nterm( old_scaffold_gap );

		assert( e_linker_n >= 0 ); // safety check

		Integer res = c->begin() - s_tether_n - s_linker_n - e_linker_n;

		for ( String::const_iterator i = ss_string.begin(), ie = ss_string.end(); i != ie; ++i, ++res ) {
			if ( *i == grow_frozen_char ) {
				// mark in LoopClosureInfo
				// this is really inefficient and it would be good to find
				// a better way to do this, but the number of times this
				// occurs is tiny, so doesn't really matter
				for ( std::set< LoopClosureInfo >::iterator l = closures_to_attempt_.begin(), le = closures_to_attempt_.end(); l != le; ++l ) {
					l->erase_moveable_residue( res );
				}
			}
		} // foreach ss instruction and residue
	} // foreach component

}


/// @brief return jump residues in the current epitope scaffold fold tree
std::pair< Integer, Integer >
EpitopeScaffold::jump_residues(
	ResidueRange const & component
) const
{
	return jump_residues( jump_label( component ) );
}


/// @brief return jump residues in the current epitope scaffold fold tree
std::pair< Integer, Integer >
EpitopeScaffold::jump_residues(
	Integer const & label
) const
{
	using pose_ns::Fold_tree;

	for ( Fold_tree::const_iterator i = epitope_scaffold_.fold_tree().begin(), ie = epitope_scaffold_.fold_tree().end(); i != ie; ++i ) {
		if ( i->label == label ) {
			return std::make_pair( i->start, i->stop );
		}
	}

	assert( false ); // should never be here!
	return std::make_pair( -32768, -32768 );
}


} // namespace design
} // namespace epigraft
