// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   EpitopeScaffold.hh
/// @brief  Class encapsulating epitope-scaffold data and operations.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)
/// @author Bill Schief (schief@u.washington.edu)
/// @author Possu Huang (possu@u.washington.edu)
/// @author Bruno Correia (bcorreia@u.washinton.edu)
/// @author Vanita Sood (vanitas@u.washington.edu)
/// @author Chris Carrico (ccarrico@gmail.com)

#ifndef INCLUDED_epigraft_design_EpitopeScaffold_HH_
#define INCLUDED_epigraft_design_EpitopeScaffold_HH_

// package headers
#include <epigraft/design/design_types.hh>
#include <epigraft/design/ConnectionResult.hh>
#include <epigraft/design/GraftInfo.hh>
#include <epigraft/design/LoopClosureInfo.hh>
#include <epigraft/design/loop_functions.hh>
#include <epigraft/match/MatchResult.hh>

// rosetta headers
#include <jump_classes.h>
#include <PackerTask.h>
#include <pack.h>
#include <pose.h>
#include <refold.h>

// ObjexxFCL Headers
#include <ObjexxFCL/FArray2D.hh>

// C++ headers
#include <cmath>
#include <set>
#include <sstream>
#include <string>
#include <utility>


namespace epigraft {
namespace design {


/// @brief  Class encapsulating epitope-scaffold data and operations.
class EpitopeScaffold {

	public: // enums

		enum BuildType {
			NORMAL_BUILD,
			ARM_BUILD,
			ADAPTIVE_BUILD,
			SCREEN_BUILD,
			FRAGMENT_FAST_CCD_BUILD,
			FRAGMENT_TWO_TORSION_CCD_BUILD,
			FRAGMENT_ONE_TORSION_CCD_BUILD,
			FRAGMENT_NO_CCD_BUILD
		};

		enum RefineType {
			NORMAL_REFINE,
			MIN_REPACK_REFINE,
			CLASSIC_REFINE,
			CONSTRAINED_REFINE,
			CLASSIC_REFINE_TWO
		};

	private: // typedefs

		typedef epigraft::match::MatchResult MatchResult;


	protected: // internal structs

		/// @brief tracks various epitope info, meant to be used in a map
		struct EpitopeTrack {
			/// @brief maps epitope-scaffold component -> jump_label -- primary component maps to jump_label from scaffold, secondary component maps to jump_label from primary
			Integer jump_label;
			/// @brief maps epitope-scaffold component range -> native epitope component range
			ResidueRange native_epitope;
			/// @brief maps epitope-scaffold component range -> old scaffold gap range
			ResidueRange old_gap;
			/// @brief maps epitope-scaffold component -> original jump -- primary component maps to original jump from scaffold, secondary component maps to original jump from primary
			pose_ns::Jump archived_jump;
		};


	public: // construct/destruct

		/// @brief default constructor
		inline
		EpitopeScaffold()
		{}

		/// @brief constructor w/ GraftInfo
		/// @note  constructor will attempt to lock internal copy of graft info to perform safety checks
		inline
		EpitopeScaffold(
			Pose const & original_scaffold,
			Pose const & epitope,
			GraftInfo const & graft_info,
			bool const & micromanage_termini = true,
			bool const & idealize_loop_geometry = true
		) : graft_info_( graft_info ),
		    Ab_is_connected_( false ),
		    Ab_transform_( graft_info.match_result().transformation_matrix ),
		    micromanage_termini_( micromanage_termini ),
		    idealize_loop_geometry_( idealize_loop_geometry )
		{
			if ( !graft_info_.locked() ) { // safety checks
				graft_info_.lock();
			}
			set_default_alteration_options();
			set_default_closure_options();
			set_default_design_options();
			make_predesign_epitope_scaffold( original_scaffold, epitope, graft_info_ );
			residues_originally_near_moveable_ = find_residues_near_moveable( epitope_scaffold_, closures_to_attempt_ ); //vds
		}

		/// @brief constructor w/ GraftInfo and "keep natro" information
		/// @note  constructor will attempt to lock internal copy of graft info to perform safety checks
		inline
		EpitopeScaffold(
			Pose const & original_scaffold,
			Pose const & epitope,
			GraftInfo const & graft_info,
			std::set< Integer > const & keep_natro,
			bool const & micromanage_termini = true,
			bool const & idealize_loop_geometry = true
		) : graft_info_( graft_info ),
		    keep_natro_( keep_natro ),
		    Ab_is_connected_( false ),
		    Ab_transform_( graft_info.match_result().transformation_matrix ),
		    micromanage_termini_( micromanage_termini ),
		    idealize_loop_geometry_( idealize_loop_geometry )
		{
			if ( !graft_info_.locked() ) { // safety checks
				graft_info_.lock();
			}
			set_default_alteration_options();
			set_default_closure_options();
			set_default_design_options();
			make_predesign_epitope_scaffold( original_scaffold, epitope, graft_info_, keep_natro_ );
			residues_originally_near_moveable_ = find_residues_near_moveable( epitope_scaffold_, closures_to_attempt_ ); //vds
		}

		/// @brief copy constructor
		inline
		EpitopeScaffold(
			EpitopeScaffold const & s
		) : graft_info_( s.graft_info_ ),
			keep_natro_( s.keep_natro_ ),
		    native_to_new_epitope_( s.native_to_new_epitope_ ),
		    epitope_tracking_( s.epitope_tracking_ ),
		    closures_to_attempt_( s.closures_to_attempt_ ),
		    arm_residues_( s.arm_residues_ ),
		    Ab_is_connected_( s.Ab_is_connected_ ),
		    Ab_connection_( s.Ab_connection_ ),
		    Ab_transform_( s.Ab_transform_ ),
		    micromanage_termini_( s.micromanage_termini_ ),
		    grow_ss_type_( s.grow_ss_type_ ),
		    grow_residue_type_( s.grow_residue_type_ ),
		    grown_residues_( s.grown_residues_ ),
		    chainbreak_criterion_( s.chainbreak_criterion_ ),
		    local_rama_criterion_( s.local_rama_criterion_ ),
		    use_fragment_insertion_( s.use_fragment_insertion_ ),
		    use_sequence_biased_fragments_( s.use_sequence_biased_fragments_ ),
		    use_variable_length_fragments_( s.use_variable_length_fragments_ ),
		    number_of_fragments_( s.number_of_fragments_ ),
		    build_cycles_( s.build_cycles_ ),
		    use_fast_refine_( s.use_fast_refine_ ),
		    refine_cycles_( s.refine_cycles_ ),
		    allow_epitope_repack_( s.allow_epitope_repack_ ),
		    allow_epitope_design_( s.allow_epitope_design_ ),
		    allow_Ab_repack_( s.allow_Ab_repack_ ),
		    allow_ALLAA_at_inter_design_positions_( s.allow_ALLAA_at_inter_design_positions_ ),
		    intra_design_position_cutoff_( s.intra_design_position_cutoff_ ),
		    inter_design_position_cutoff_( s.inter_design_position_cutoff_ ),
		    Ab_repack_cutoff_( s.Ab_repack_cutoff_ ),
		    intra_design_positions_( s.intra_design_positions_ ),
		    inter_design_positions_( s.inter_design_positions_ ),
		    Ab_repack_positions_( s.Ab_repack_positions_ ),
		    complementarity_design_cycles_( s.complementarity_design_cycles_ ),
		    complementarity_design_position_cutoff_( s.complementarity_design_position_cutoff_ ),
		    complementarity_design_positions_( s.complementarity_design_positions_ ),
		    complementarity_Ab_positions_( s.complementarity_Ab_positions_ ),
		    complementarity_shell_cutoff_( s.complementarity_shell_cutoff_ ),
		    complementarity_shell_repack_( s.complementarity_shell_repack_ ),
		    complementarity_shell_redesign_( s.complementarity_shell_redesign_ ),
		    complementarity_shell_positions_( s.complementarity_shell_positions_ ),
		    residues_originally_near_moveable_( s.residues_originally_near_moveable_ ), //vds
		    idealize_loop_geometry_( s.idealize_loop_geometry_ ),
		    repick_frag_( s.repick_frag_ ),
		    closed_during_trajectory_( s.closed_during_trajectory_ )
		{
			epitope_scaffold_ = s.epitope_scaffold_;
			archived_epitope_scaffold_ = s.archived_epitope_scaffold_;
			archived_Ab_ = s.archived_Ab_;
		}

		/// @brief default destructor
		inline
		~EpitopeScaffold()
		{}


	public: // assignment

		/// @brief copy assignment
		inline
		EpitopeScaffold &
		operator =( EpitopeScaffold const & s )
		{
			if ( this != &s ) {
				graft_info_ = s.graft_info_;
				keep_natro_ = s.keep_natro_;
				native_to_new_epitope_ = s.native_to_new_epitope_;
				epitope_tracking_ = s.epitope_tracking_;

				closures_to_attempt_ = s.closures_to_attempt_;
				arm_residues_ = s.arm_residues_;

				epitope_scaffold_ = s.epitope_scaffold_;
				archived_epitope_scaffold_ = s.archived_epitope_scaffold_;

				Ab_is_connected_ = s.Ab_is_connected_;
				Ab_connection_ = s.Ab_connection_;
				Ab_transform_ = s.Ab_transform_;
				archived_Ab_ = s.archived_Ab_;

				micromanage_termini_ = s.micromanage_termini_;
				grow_ss_type_ = s.grow_ss_type_;
				grow_residue_type_ = s.grow_residue_type_;
				grown_residues_ = s.grown_residues_;

				chainbreak_criterion_ = s.chainbreak_criterion_;
				local_rama_criterion_ = s.local_rama_criterion_;

				use_fragment_insertion_ = s.use_fragment_insertion_;
				use_sequence_biased_fragments_ = s.use_sequence_biased_fragments_;
				use_variable_length_fragments_ = s.use_variable_length_fragments_;
				number_of_fragments_ = s.number_of_fragments_;
				build_cycles_ = s.build_cycles_;

				use_fast_refine_ = use_fast_refine_;
				refine_cycles_ = s.refine_cycles_;

			    allow_epitope_repack_ = s.allow_epitope_repack_;
			    allow_epitope_design_ = s.allow_epitope_design_;
			    allow_Ab_repack_ = s.allow_Ab_repack_;
			    allow_ALLAA_at_inter_design_positions_ = s.allow_ALLAA_at_inter_design_positions_;

			    intra_design_position_cutoff_ = s.intra_design_position_cutoff_;
			    inter_design_position_cutoff_ = s.inter_design_position_cutoff_;
			    Ab_repack_cutoff_ = s.Ab_repack_cutoff_;

			    intra_design_positions_ = s.intra_design_positions_;
			    inter_design_positions_ = s.inter_design_positions_;
			    Ab_repack_positions_ = s.Ab_repack_positions_;

			    complementarity_design_cycles_ = s.complementarity_design_cycles_;
			    complementarity_design_position_cutoff_ = s.complementarity_design_position_cutoff_;
			    complementarity_design_positions_ = s.complementarity_design_positions_;
			    complementarity_Ab_positions_ = s.complementarity_Ab_positions_;
			    complementarity_shell_cutoff_ = s.complementarity_shell_cutoff_;
			    complementarity_shell_repack_ = s.complementarity_shell_repack_;
			    complementarity_shell_redesign_ = s.complementarity_shell_redesign_;
			    complementarity_shell_positions_ = s.complementarity_shell_positions_;

				residues_originally_near_moveable_ = s.residues_originally_near_moveable_; //vds

				idealize_loop_geometry_ = s.idealize_loop_geometry_;
				repick_frag_ = s.repick_frag_;

				closed_during_trajectory_ = s.closed_during_trajectory_;
			}
			return *this;
		}


	public: // Ab management

		/// @brief set Ab, saves a local copy of Ab, this will DISCONNECT any currently connected Ab
		/// @warning remember to set transformation matrix to identity prior to this if you don't want
		/// @warning antibody to be reoriented upon connection
		/// @note  Jump connection from epitope-scaffold to Ab is currently from middle residue of primary component on
		/// @note  epitope-scaffold to middle residue of Ab.
		void
		set_Ab(
			Pose const & Ab,
			bool const & connect = false
		);

		/// @brief connect (archived) Ab, assume Ab has already been set via prior set_Ab call
		/// @param[in] connect_to_primary  only connect to closest residue on primary component, otherwise finds closests residue on any component
		bool
		connect_Ab(
			bool const & only_connect_to_primary = false
		);

		/// @brief disconnect Ab
		/// @details each time disconnect is called, the current antibody sidechain state is archived so that
		/// @details state can be restored upon subsequence connect_Ab() call
		/// @param[in] auto_archive_Ab_sidechains by default, archive Ab sidechains before disconnect
		bool
		disconnect_Ab(
			bool const & auto_archive_Ab_sidechains = true
		);

		/// @brief get archived Ab
		Pose const &
		archived_Ab() const
		{
			return archived_Ab_;
		}

		/// @brief extract copy of current Ab (with current geometry) in epitope scaffold
		bool
		extract_Ab(
			Pose & output_Ab
		) const;

		/// @brief extract copy of current epitope scaffold (without Ab) in epitope scaffold
		void
		extract_epitope_scaffold(
			Pose & output_es
		) const;

		/// @brief extract complex of current epitope scaffold + Ab
		/// @details this is different from directly accessing the internal Pose in the EpitopeScaffold object,
		/// @details as this function returns a complex whose chains are continuous (i.e. individual fold tree segments
		/// @details are continuous), versus the internal Pose whose fold tree potentially contains multiple cuts
		void
		extract_complex(
			Pose & output_complex
		) const;

		/// @brief Ab connection status
		inline
		bool const &
		Ab_is_connected() const
		{
			return Ab_is_connected_;
		}

		/// @brief Ab connection info
		inline
		ConnectionResult const &
		Ab_connection() const
		{
			return Ab_connection_;
		}

		/// @brief current Ab transform
		/// @note  matrix can also be used to transform epitope, as the two are currently tied together
		inline
		FArray2D_float const &
		Ab_transform() const
		{
			return Ab_transform_;
		}

		/// @brief set current Ab transform
		/// @details does not disconnect any currently connected antibody
		inline
		void
		set_Ab_transform(
			FArray2D_float const & transform
		)
		{
			Ab_transform_ = transform;
		}


	public: // alteration parameters (grow/remove residues)

		/// @brief are termini being micromanaged when making predesign scaffold?
		/// @note there is currently no corresponding 'get' accessor because this is set and enforced
		/// @note at construction
		inline
		bool const &
		micromanage_termini()
		{
			return micromanage_termini_;
		}

		inline
		char const &
		grow_residue_type()
		{
			return grow_residue_type_;
		}

		inline
		char const &
		grow_ss_type()
		{
			return grow_ss_type_;
		}

		inline
		void
		set_grow_residue_type(
			char const & residue_type
		)
		{
			grow_residue_type_ = residue_type;
		}

		inline
		void
		set_grow_ss_type(
			char const & ss_type
		)
		{
			grow_ss_type_ = ss_type;
		}

		/// @brief   resets all secondary structure of grown residues to the "grow residue type"
		/// @details uses archived epitope scaffold as reference for grown residue ss type
		/// @param[in]
		inline
		void
		reset_grown_residue_ss(
			bool const & also_reset_grown_in_closed_loops = false
		)
		{
			if ( also_reset_grown_in_closed_loops ) {
				for ( std::set< Integer >::const_iterator i = grown_residues_.begin(), ie = grown_residues_.end(); i != ie; ++i ) {
					epitope_scaffold_.set_secstruct( *i, archived_epitope_scaffold_.secstruct( *i ) );
				}
			} else {
				for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt_.begin(), ie = closures_to_attempt_.end(); i != ie; ++i ) {
					if ( !i->is_closed() ) {
						std::set< Integer > const & moveable = i->moveable_residues();
						for ( std::set< Integer >::const_iterator r = moveable.begin(), re = moveable.end(); r != re; ++r ) {
							if ( grown_residues_.find( *r ) != grown_residues_.end() ) {
								epitope_scaffold_.set_secstruct( *r, archived_epitope_scaffold_.secstruct( *r ) );
							}
						}
					}
				}
			}

		}


	public: // loop closure criteria

		/// @brief linear chainbreak criterion
		inline
		Real const &
		chainbreak_criterion() const
		{
			return chainbreak_criterion_;
		}

		/// @brief set linear chainbreak criterion
		/// @details also updates status of internal closures based upon given criterion
		inline
		void
		set_chainbreak_criterion(
			Real const & cutoff
		)
		{
			chainbreak_criterion_ = cutoff;

			// update closures
			recheck_closure_criteria();
		}

		/// @brief local rama criterion for testing loop closure (rama over moveable residues in loop)
		inline
		Real const &
		local_rama_criterion() const
		{
			return local_rama_criterion_;
		}

		/// @brief set local rama criterion for testing loop closure (rama over moveable residues in loop)
		inline
		void
		set_local_rama_criterion(
			Real const & cutoff
		)
		{
			local_rama_criterion_ = cutoff;

			// update closures
			recheck_closure_criteria();
		}

		/// @brief re-test closure criteria and mark status for all loops
		/// @details honors graft info, so if do_graft_bb is false for a loop, the
		/// @details loop is marked as always closed
		inline
		void
		recheck_closure_criteria()
		{
			for ( std::set< LoopClosureInfo >::const_iterator c = closures_to_attempt_.begin(), ce = closures_to_attempt_.end(); c != ce; ++c ) {
				LoopClosureInfo const & closure_info = *c;
				if ( do_graft_bb( closure_info ) ) {
					check_closure_criteria( closure_info );
				} else {
					closure_info.set_is_closed( true );
				}
			}
		}

		/// @brief all loops closed?
		inline
		bool
		all_loops_closed() const
		{
			bool closed = true;

			for ( std::set< LoopClosureInfo >::const_iterator c = closures_to_attempt_.begin(), ce = closures_to_attempt_.end(); c != ce; ++c ) {
				LoopClosureInfo const & closure_info = *c;

				if ( do_graft_bb( closure_info ) ) {
					closed = closed && ( closure_info.is_closed() || closed_during_trajectory( closure_info ) );
				}
			}

			return closed;
		}

		/// @brief at least one loop closed?
		inline
		bool
		some_loops_closed() const
		{
			bool closed = false;

			for ( std::set< LoopClosureInfo >::const_iterator c = closures_to_attempt_.begin(), ce = closures_to_attempt_.end(); c != ce; ++c ) {
				LoopClosureInfo const & closure_info = *c;

				if ( do_graft_bb( closure_info ) ) {
					closed = closed || ( closure_info.is_closed() || closed_during_trajectory( closure_info ) );
				}
			}

			return closed;
		}

		/// @brief return the number of loops closed
		inline
		Size
		n_loops_closed() const
		{
			Size n_closed = 0;

			for ( std::set< LoopClosureInfo >::const_iterator c = closures_to_attempt_.begin(), ce = closures_to_attempt_.end(); c != ce; ++c ) {
				LoopClosureInfo const & closure_info = *c;

				if ( do_graft_bb( closure_info ) ) {
					if ( closure_info.is_closed() || closed_during_trajectory( closure_info ) ) {
						++n_closed;
					}
				}
			}

			return n_closed;
		}

		/// @brief return the number of loops remaining that are still open
		///  and need to be closed
		inline
		Size
		n_open_loops() const
		{
			Size n_open = 0;

			for ( std::set< LoopClosureInfo >::const_iterator c = closures_to_attempt_.begin(), ce = closures_to_attempt_.end(); c != ce; ++c ) {
				LoopClosureInfo const & closure_info = *c;

				if ( do_graft_bb( closure_info ) ) {
					if ( !closure_info.is_closed() && !closed_during_trajectory( closure_info ) ) {
						++n_open;
					}
				}
			}

			return n_open;
		}

		/// @brief return the number of loops that are requested to be closed
		inline
		Size
		n_requested_loop_closures() const {
			Size n_requested = 0;

			for ( std::set< LoopClosureInfo >::const_iterator c = closures_to_attempt_.begin(), ce = closures_to_attempt_.end(); c != ce; ++c ) {
				LoopClosureInfo const & closure_info = *c;

				if ( do_graft_bb( closure_info ) ) {
					++n_requested;
				}
			}

			return n_requested;
		}

		/// @brief string with loop status
		inline
		std::string
		loop_status_str() const {
			std::ostringstream out;
			for ( std::set< LoopClosureInfo >::const_iterator c = closures_to_attempt_.begin(), ce = closures_to_attempt_.end(); c != ce; ++c ) {
				out << closed_during_trajectory( *c ) << " | " << c->to_string() << '\n';
			}

			return out.str();
		}


	public: // loop build options

		/// @brief is using fragment insertion?
		inline
		bool const &
		using_fragment_insertion() const
		{
			return use_fragment_insertion_;
		}

		/// @brief turn on/off fragment insertion
		inline
		void
		set_use_fragment_insertion(
			bool const & flag
		)
		{
			use_fragment_insertion_ = flag;
		}

		/// @brief is using sequence biased fragment insertion?
		inline
		bool const &
		using_sequence_biased_fragments() const
		{
			return use_sequence_biased_fragments_;
		}

		/// @brief turn on/off sequence biased fragment insertion
		inline
		void
		set_use_sequence_biased_fragments(
			bool const & flag
		)
		{
			use_sequence_biased_fragments_ = flag;
		}

		/// @brief use variable length fragments (currently only loop relax and adaptive style)
		inline
		bool const &
		using_variable_length_fragments() const {
			return use_variable_length_fragments_;
		}

		/// @brief set usage of variable length fragments (currently only loop relax and adaptive style)
		inline
		void
		set_use_variable_length_fragments(
			bool const & flag
		)
		{
			use_variable_length_fragments_ = flag;
		}

		/// @brief get number of fragments used per position during fragment insertion
		inline
		Integer const &
		number_of_fragments() const
		{
			return number_of_fragments_;
		}

		/// @brief set number of fragments used per position during fragment insertion
		inline
		void
		set_number_of_fragments(
			Integer const & nfrag
		)
		{
			number_of_fragments_ = nfrag;
		}

		/// @brief repick fragments during build?
		inline
		bool
		repick_fragments() const {
			return repick_frag_;
		}

		/// @brief repick fragments during build?
		inline
		void
		set_repick_fragments(
			bool const flag
		)
		{
			repick_frag_ = flag;
		}

		/// @brief number of build loop cycles
		inline
		Integer const &
		build_cycles() const
		{
			return build_cycles_;
		}

		/// @brief set number of build loop cycles
		inline
		void
		set_build_cycles(
			Integer const & cycles
		)
		{
			build_cycles_ = cycles;
		}


	public: // loop refine options

		/// @brief use fast refine?
		inline
		bool const &
		using_fast_refine() const
		{
			return use_fast_refine_;
		}

		/// @brief set use fast refine
		inline
		void
		set_use_fast_refine(
			bool const & flag
		)
		{
			use_fast_refine_ = flag;
		}

		/// @brief number of refine loop cycles
		inline
		Integer const &
		refine_cycles() const
		{
			return refine_cycles_;
		}

		/// @brief set number of refine loop cycles
		inline
		void
		set_refine_cycles(
			Integer const & cycles
		)
		{
			refine_cycles_ = cycles;
		}


	public: // design options

		/// @brief allow epitope repack
		inline
		bool const &
		allow_epitope_repack() const
		{
			return allow_epitope_repack_;
		}

		/// @brief set allow epitope repack
		inline
		void
		set_allow_epitope_repack(
			bool const & flag
		)
		{
			allow_epitope_repack_ = flag;
		}

		/// @brief allow epitope design
		inline
		bool const &
		allow_epitope_design() const
		{
			return allow_epitope_design_;
		}

		/// @brief set allow epitope design
		inline
		void
		set_allow_epitope_design(
			bool const & flag
		)
		{
			allow_epitope_design_ = flag;
		}

		/// @brief allow Ab repack
		inline
		bool const &
		allow_Ab_repack() const
		{
			return allow_Ab_repack_;
		}

		/// @brief set allow Ab repack
		inline
		void
		set_allow_Ab_repack(
			bool const & flag
		)
		{
			allow_Ab_repack_ = flag;
		}

		/// @brief allow ALLAA at inter- design positions?
		inline
		bool const &
		allow_ALLAA_at_inter_design_positions() const
		{
			return allow_ALLAA_at_inter_design_positions_;
		}

		/// @brief set allow AA at all inter- design positions
		inline
		void
		set_allow_ALLAA_at_inter_design_positions(
			bool const & flag
		)
		{
			allow_ALLAA_at_inter_design_positions_ = flag;
		}

		/// @brief intra design position distance cutoff
		inline
		Real const &
		intra_design_position_cutoff() const
		{
			return intra_design_position_cutoff_;
		}

		/// @brief set intra design position distance cutoff
		inline
		void
		set_intra_design_position_cutoff(
			Real const & cutoff
		)
		{
			intra_design_position_cutoff_ = cutoff;
		}

		/// @brief inter design position distance cutoff
		inline
		Real const &
		inter_design_position_cutoff() const
		{
			return inter_design_position_cutoff_;
		}

		/// @brief set inter design position distance cutoff
		inline
		void
		set_inter_design_position_cutoff(
			Real const & cutoff
		)
		{
			inter_design_position_cutoff_ = cutoff;
		}

		/// @brief Ab repack distance cutoff
		inline
		Real const &
		Ab_repack_cutoff() const
		{
			return Ab_repack_cutoff_;
		}

		/// @brief set Ab repack distance cutoff
		inline
		void
		set_Ab_repack_cutoff(
			Real const & cutoff
		)
		{
			Ab_repack_cutoff_ = cutoff;
		}


	public: // complementarity design options

		/// @brief complementarity design cycles
		inline
		Integer const &
		complementarity_design_cycles() const
		{
			return complementarity_design_cycles_;
		}

		/// @brief set complementarity design cycles
		inline
		void
		set_complementarity_design_cycles(
			Integer const & cycles
		)
		{
			complementarity_design_cycles_ = cycles;
		}

		/// @brief complementarity design position distance cutoff
		inline
		Real const &
		complementarity_design_position_cutoff() const
		{
			return complementarity_design_position_cutoff_;
		}

		/// @brief set complementarity design position distance cutoff
		inline
		void
		set_complementarity_design_position_cutoff(
			Real const & cutoff
		)
		{
			complementarity_design_position_cutoff_ = cutoff;
		}

		/// @brief complementarity design shell distance cutoff for repack/redesign
		///        of shell on scaffold
		inline
		Real const &
		complementarity_shell_cutoff() const
		{
			return complementarity_shell_cutoff_;
		}

		/// @brief set complementarity design shell distance cutoff for repack/redesign
		///        of shell on scaffold
		inline
		void
		set_complementarity_shell_cutoff(
			Real const & cutoff
		)
		{
			complementarity_shell_cutoff_ = cutoff;
		}

		/// @brief doing repack for complementarity shell?
		inline
		bool const &
		complementarity_shell_repack() const
		{
			return complementarity_shell_repack_;
		}

		/// @brief set repack for complementarity shell
		inline
		void
		set_complementarity_shell_repack(
			bool const & flag
		)
		{
			complementarity_shell_repack_ = flag;
		}

		/// @brief doing redesign for complementarity shell?
		inline
		bool const &
		complementarity_shell_redesign() const
		{
			return complementarity_shell_redesign_;
		}

		/// @brief set redesign for complementarity shell
		inline
		void
		set_complementarity_shell_redesign(
			bool const & flag
		)
		{
			complementarity_shell_redesign_ = flag;
		}


	public: // sidechain identity

		/// @brief switch internal representation to full atom (repacks!)
		/// @warning repacks!  currently when fullatom flag is set for a pose,
		/// @warning the internal fullatom data is invalidated
		/// @warning after this function is called allow_chi_move will be set to false
		inline
		void
		switch_to_fullatom()
		{
			if ( !epitope_scaffold_.fullatom() ) {
				epitope_scaffold_.set_allow_chi_move( true );
				epitope_scaffold_.set_fullatom_flag( true, true ); // boolean: fullatom, repack
				epitope_scaffold_.set_allow_chi_move( false );
			}
		}

		/// @brief switch internal representation to centroid
		/// @warning fullatom data invalidated!
		inline
		void
		switch_to_centroid()
		{
			if ( epitope_scaffold_.fullatom() ) {
				epitope_scaffold_.set_fullatom_flag( false, false ); // boolean: fullatom, repack
			}
		}

		/// @brief flip epitope scaffold to all-X (Ab, if connected, left intact)
		/// @param[in] residue_type residue type from Rosetta's param_aa
		/// @param[in] keep_gly do not convert any existing glycines?  default is true
		void
		convert_to_residue_type(
			Integer const & residue_type,
			bool const & keep_gly = true,
			bool const include_Ab = false
		);

		/// @brief flip epitope scaffold to all-gly (Ab, if connected, left intact)
		inline
		void
		convert_to_gly()
		{
			convert_to_residue_type( param_aa::aa_gly, false ); // boolean: keep_gly
		}

		/// @brief flip epitope scaffold to all-ala (Ab, if connected, left intact)
		inline
		void
		convert_to_ala()
		{
			convert_to_residue_type( param_aa::aa_ala, false ); // boolean: keep_gly
		}

		/// @brief flip all sidechains (epitope scaffold + Ab, if present) back to state with original
		/// @brief sidechains
		/// @warning after this, internal epitope scaffold Pose is fullatom
		void
		recover_all_native_sidechains();

		/// @brief flip epitope scaffold back to state with original sidechains (Ab, if connected, left intact)
		/// @brief (native epitope and native scaffold)
		/// @warning after this, internal epitope scaffold Pose is fullatom
		void
		recover_native_epitope_scaffold_sidechains();

		/// @brief flip just epitope sidechains back to state original sidechains
		/// @note obeys keep natro if available
		void
		recover_native_epitope_sidechains();

		/// @brief recover Ab sidechains
		bool
		recover_Ab_sidechains();


	public: // scheduled closure

		/// @brief build all loops (centroid)
		/// @result success/failure according to closure criterion
		/// @note  initial closure state checked here
		/// @note  do_graft_bb honored here
		inline
		bool
		build_all(
			bool const & ignore_closure_state = false,
			bool const & stop_on_fail = true
		)
		{
			bool passed = build_primary( ignore_closure_state, stop_on_fail );
			if ( passed ) {
				passed = build_secondary( ignore_closure_state, stop_on_fail );
			}
			return passed;
		}

		/// @brief build all loops (centroid) simultaneously
		/// @result success/failure according to closure criterion
		/// @note  initial closure state checked here
		/// @note  do_graft_bb honored here
		bool
		build_all_simul(
			bool const & ignore_closure_state = false,
			BuildType const & build_type = NORMAL_BUILD
		);

		/// @brief build primary loops (centroid)
		/// @result success/failure according to closure criterion
		/// @note  initial closure state checked here
		/// @note  do_graft_bb honored here
		bool
		build_primary(
			bool const & ignore_closure_state = false,
			bool const & stop_on_fail = true
		);

		/// @brief build secondary loops (centroid)
		/// @result success/failure according to closure criterion
		/// @note  initial closure state checked here
		/// @note  do_graft_bb honored here
		bool
		build_secondary(
			bool const & ignore_closure_state = false,
			bool const & stop_on_fail = true
		);

		/// @brief build a component's loops (centroid)
		/// @result success/failure according to closure criterion
		/// @note  forcibly refine, no check of closure state
		bool
		build_component(
			ResidueRange const & component,
			bool const & stop_on_fail = true
		);

		/// @brief build a set of loops (centroid)
		/// @result success/failure according to closure criterion
		/// @note  forcibly build, no check of closure state
		bool
		build_loops(
			std::set< LoopClosureInfo > const & closures_to_attempt,
			bool const & stop_on_fail = true
		);

		/// @brief refine all loops (full atom)
		/// @result success/failure according to closure criterion
		/// @note  initial closure state checked here
		/// @note  do_graft_bb honored here
		inline
		bool
		refine_all(
			bool const & ignore_closure_state = false,
			bool const & stop_on_fail = true
		)
		{
			bool passed = refine_primary( ignore_closure_state, stop_on_fail );
			if ( passed ) {
				passed = refine_secondary( ignore_closure_state, stop_on_fail );
			}
			return passed;
		}

		/// @brief refine all loops (full atom) simultaneously
		/// @result success/failure according to closure criterion
		/// @note  initial closure state checked here
		/// @note  do_graft_bb honored here
		bool
		refine_all_simul(
			bool const & ignore_closure_state = false,
			RefineType const & refine_type = NORMAL_REFINE
		);

		/// @brief refine primary loops (full atom)
		/// @result success/failure according to closure criterion
		/// @note  initial closure state checked here
		/// @note  do_graft_bb honored here
		bool
		refine_primary(
			bool const & ignore_closure_state = false,
			bool const & stop_on_fail = true
		);

		/// @brief refine secondary loops (full atom)
		/// @result success/failure according to closure criterion
		/// @note  initial closure state checked here
		/// @note  do_graft_bb honored here
		bool
		refine_secondary(
			bool const & ignore_closure_state = false,
			bool const & stop_on_fail = true
		);

		/// @brief refine a component's loops (full atom)
		/// @result success/failure according to closure criterion
		/// @note  forcibly refine, no check of closure state
		bool
		refine_component(
			ResidueRange const & component,
			bool const & stop_on_fail = true
		);

		/// @brief refine a set of loops (full atom)
		/// @result success/failure according to closure criterion
		/// @note  forcibly refine, no check of closure state
		bool
		refine_loops(
			std::set< LoopClosureInfo > const & closures_to_attempt,
			bool const & stop_on_fail = true
		);


	public: // simple closure/design

		/// @brief (centroid level) build loop
		/// @result success/failure according to closure criterion
		/// @note  internal closure info is updated here
		/// @warning THIS ROUTINE IS OLD AND DOES NOT SUPPORT ALL BUILD TYPES
		///          PLEASE USE build_all_simul() INSTEAD
		bool
		build_loop(
			LoopClosureInfo const & closure_info
		);

		/// @brief (full atom) refine loop
		/// @result success/failure according to closure criterion
		/// @note  internal closure info is updated here
		bool
		refine_loop(
			LoopClosureInfo const & closure_info
		);

		/// @brief   design
		///          In this procedure the es-foldtree is temporarily replaced with a
		///          simple fold tree to score correctly during design.  The es-foldtree
		///          is recovered at the end, at which point the scores for the internal
		///          epitope-scaffold pose are reset.
		void
		design();

		/// @brief   design using design matrix
		/// @details design using given design matrix only, no other information is taken
		///          into account to allow user the ability to override anything they need
		void
		design(
			FArray2D< bool > & design_matrix
		);

		/// @brief design returning cached information
		/// @details implementation is kind of a mess here, it's setup the way it is due
		///  to the way other things are hooked in, will cleanup later
		void
		design(
			FArray2D< bool > & design_matrix,
			PackerTask * task,
			pack::InteractionGraphBase * & ig,
			RotamerSet * rs,
			FArray1D_int * current_rot_index,
			FArray1D_float * ligenergy1b_old
		);

		/// @brief design using cached information
		/// @details implementation is kind of a mess here, it's setup the way it is due
		///  to the way other things are hooked in, will cleanup later
		void
		design(
			PackerTask * task,
			pack::InteractionGraphBase * & ig,
			RotamerSet * rs,
			FArray1D_int * current_rot_index,
			FArray1D_float * ligenergy1b_old
		);


	public: // simple repack

		/// @brief forcibly repack range in internal pose -- ignores internal allow repack flags
		/// @warning can repack with cut multigrafting fold tree!
		void
		repack(
			ResidueRange const & range
		);

		/// @brief forcibly repack complex -- ignores internal allow repack flags
		/// @warning can repack with cut multigrafting fold tree!
		void
		repack_complex();

		/// @brief forcibly repack antibody if present -- ignores internal allow repack flags
		/// @warning can repack with cut multigrafting fold tree!
		void
		repack_Ab();

		/// @brief forcibly repack epitope-scaffold -- ignores internal allow repack flags
		/// @warning can repack with cut multigrafting fold tree!
		void
		repack_es();

		/// @brief forcibly repack epitope-scaffold / antibody interface
		/// @warning can repack with cut multigrafting fold tree!
		void
		repack_interface();


	public: // growing/removing residues around closure sites

		/// @brief alter moveable residues in closure sites to desired secondary structure
		/// @param[in] ss_type secondary structure type, valid characters are 'H', 'E', 'L', and 'D'
		void
		alter_secondary_structure_of_moveable_residues(
			char const & ss_type
		);

		/// @brief set proper moveable residues and grow/remove residues at closure sites
		/// @brief based on tether/linker/moveable information in internal GraftInfo object
		/// @note  directly stores two copies of the epitope scaffold (working and archived)
		void
		alter_closure_sites();


	public: // alteration of properties within closure site (moveable residues, secondary structure, etc)

		/// @brief randomize phi-psi of moveable residues around broken closure sites
		/// @param[in] also_randomize_lever  if N2C/C2N, then also randomize residues
		///  further from the cutpoint that induce lever moves
		/// @param[in] also_randomize_archive induce same randomization to archive?
		/// @param[in] force randomize all closure sites except for those marked
		///   artificially closed
		void
		randomize_broken_closure_site_phipsi(
			bool const & also_randomize_lever = false,
			bool const & also_randomize_archive = true,
			bool const & force = false
		);

		/// @brief randomize cutpoints of broken closure sites
		/// @details For each closure site, takes into account the current moveable residues
		/// @details and randomly place the cutpoint within them.  Fold tree is altered.
		/// @warning using this with randomize_broken_closure_site_movable_residues() can have
		/// @warning unintended consequences, since grown residues could end up being held fixed
		void
		randomize_broken_closure_site_cutpoints(
			bool const & also_randomize_archive = true
		);

		/// @brief randomize number of moveable residues within broken closure sites
		/// @details for each closure site, looks to the left of the cut and
		/// @details picks a number of moveables residues, then looks to the right
		/// @details of the cut and does the same.  # of actual moveable residues
		/// @details may be larger than picked by this procedure if there are any
		/// @details "always moveable" residues.
		void
		randomize_broken_closure_site_moveable_residues();

		/// @brief randomize secondary structure of moveable residues within all closure sites
		/// @details For each closure site, constructs the immediate *contiguous* interval of
		/// @details moveable residues around the cut.  Then decides on the amount of
		/// @details loop content w/ respect to a required minimum fraction and inserts
		/// @details that amount of loop content with a randomly shifting window.  Other residues
		/// @details are marked with the same structure to the immediate left & right of the merge
		/// @details if there is at least three residues worth of the secondary structure content.
		/// @param[in]  minimum_loop_content  minimum loop content (fraction, within interval [0, 1])
		void
		randomize_broken_closure_site_moveable_residue_secondary_structure(
			Real const & minimum_content,
			char const & ss_type = 'L'
		);

		/// @brief resets the backbone of the residues in a closure site to the archived backbone
		/// @details resets via set phi/psi/omega and not coordinate replacement
		/// @note the closure_info is actually modified to store the new chainbreak & rama,
		/// @note so make sure to save a prior copy if needed
		void
		reset_closure_site_backbone(
			LoopClosureInfo const & closure_info
		);

		/// @brief resets the backbone of ALL closure sites
		void
		reset_all_closure_site_backbone();

		/// @brief resets the backbone of closures sites that are still broken (not closed)
		void
		reset_broken_closure_site_backbone();


	public: // epitope rb position management

		/// @brief saves epitope rb jumps in current epitope_scaffold pose
		/// @note epitope rb jumps are managed separately from archiving of epitope scaffold pose
		/// @note because we typically want memory of the *original* jumps from the initial pre-design
		inline
		void
		archive_epitope_rb_jumps()
		{
			for ( std::map< ResidueRange, EpitopeTrack >::iterator i = epitope_tracking_.begin(), ie = epitope_tracking_.end(); i != ie; ++i ) {
				if ( i->second.jump_label > -1 ) { // -1 indicates no jump present
					i->second.archived_jump = epitope_scaffold_.get_jump( i->second.jump_label );
				}
			}
		}

		/// @brief reset epitope rb jumps in current epitope scaffold pose
		/// @details pulls the epitope rb jumps from archive, note that this is separate from the
		/// @details archived epitope scaffold pose
		inline
		void
		reset_epitope_rb_jumps()
		{
			for ( std::map< ResidueRange, EpitopeTrack >::iterator i = epitope_tracking_.begin(), ie = epitope_tracking_.end(); i != ie; ++i ) {
				if ( i->second.jump_label > -1 ) { // -1 indicates no jump present
					epitope_scaffold_.set_jump( i->second.jump_label, i->second.archived_jump );
				}
			}
		}

		/// @brief perturbs jump for epitope scaffold by gaussian spread
		/// @param[in] epitope_scaffold_range  the epitope range in the current epitope scaffold (i.e. "new" range)
		/// @param[in] translation_magnitude  translation magnitude in angstroms
		/// @param[in] rotation_magnitude  rotation magnitude in degrees
		/// @param[in] perturb_using_archived_jump  if true (default), perturb using the archived jump to limit drift from original position
		/// @warning this may not be uniform
		bool
		gaussian_perturb_epitope_rb_jump(
			ResidueRange const & epitope_scaffold_range,
			Real const & translation_magnitude = 0.5,
			Real const & rotation_magnitude = 1.0,
			bool const & recheck_closure = true
		);

		/// @brief perturb all non-closed secondary epitope components by gaussian spread with a forced clash check
		/// @details Clash checks against antibody (if attached) and all residues of scaffold and epitope except for
		/// @details loop moveable residues.  If there is a clash the proposed move is rejected and another is
		/// @details attempted.  If an epitope component is "half-closed" (one side closed, one side open),
		/// @details no perturbation is done.
		void
		gaussian_perturb_epitope_secondary_components(
			Real const & translation_magnitude = 0.5,
			Real const & rotation_magnitude = 1.0,
			bool const & perturb_using_archived_jump = true,
			bool const & use_graft_info_settings = true
		);


	public: // position optimization

		/// @brief optimize initial position of broken epitope by monte carlo rb-perturbation + minimization
		/// @brief without Ab, mc rb-perturb attempted in 1 angstrom/1 degree gaussian spread
		/// @return true if optimization was attempted, false if the procedure was unable to run due to input error
		bool
		epitope_rb_optimize(
			Integer const & cycles,
			Integer const & rb_pertubations,
			bool const & do_minimize = false,
			Real const & translation_scale = 0.5,
			Real const & rotation_scale = 0.5
		);

		/// @brief optimize Ab->epitope interaction by minimization using score12
		/// @brief with option to include rigid body minimization, primarily for use in
		/// @brief superposition protocol scaffolding
		/// @details to be consistent for, the same style of atom pair
		/// @details constraints are used regardless of rigid body on/off
		/// @param[in] interface_distance distance cutoff for atoms to be considered at interface, default 4.5
		/// @warning procedure incomplete, currently no chi minimize
		void
		Ab_epitope_optimize(
			bool const & chi_minimize = true,
			bool const & rb_minimize = false,
			Real const & interface_distance = 4.5
		);

		/// @brief optimize epitope internals to de-clash, primarily for
		///  use when no antibody is present during superposition scaffolding
		void
		epitope_optimize();


	public: // complementarity design

		/// @brief clear complementarity design positions on scaffold
		inline
		void
		clear_complementarity_design_positions()
		{
			complementarity_design_positions_.clear();
		}

		/// @brief clear complementarity Ab positions used to find complementarity design positions
		inline
		void
		clear_complementarity_Ab_positions()
		{
			complementarity_Ab_positions_.clear();
		}

		/// @brief clear complementarity shell positions found during a design() or override_* call
		inline
		void
		clear_complementarity_shell_positions()
		{
			complementarity_shell_positions_.clear();
		}

		/// @brief clear all complementarity design info (scaffold positions and Ab positions)
		inline
		void
		clear_complementarity_design_info()
		{
			clear_complementarity_design_positions();
			clear_complementarity_Ab_positions();
			clear_complementarity_shell_positions();
		}

		/// @brief mark complementarity design positions by checking against
		///        Ab residues indicated by given Ab (pdb) residue string of
		///        format 'CHAINRESNUMINSERTIONCODE'
		/// @note  uses archived Ab as reference for residue number indexing
		void
		mark_complementarity_Ab_positions(
			std::set< String > const & pdb_residues
		);

		/// @brief  find complementarity design positions by distance check against
		///         Ab residues
		std::set< Integer >
		find_complementarity_design_positions(
			std::set< Integer > const & ab_residues
		);

		/// @brief  do complementarity design
		/// @return returns false if Ab is not connected or if there is nothing to do,
		///         otherwise true
		bool
		complementarity_design(
			bool const & allow_rb_min
		);


	public: // fold tree mechanisms, use these with care!

		/// @brief seal fold tree in working epitope scaffold
		/// @warning use with care!  remember to unseal fold tree before attempting any subsequence
		/// @warning loop closure operations or anything which requires a chainbreak
		void
		seal_fold_tree();

		/// @brief pull fold tree from archived epitope scaffold for use in working epitope scaffold
		void
		unseal_fold_tree();


	public: // history mechanisms

		/// @brief copy archived epitope scaffold -> working epitope scaffold
		/// @warning if antibody was connected, wipes out antibody connection
		inline
		void
		pull_from_archive()
		{
			epitope_scaffold_ = archived_epitope_scaffold_;
			Ab_is_connected_ = false;
		}

		/// @brief force save state of Ab sidechains
		bool
		archive_Ab_sidechains();

		/// @brief force save state of epitope-scaffold sidechains that have same identity
		/// @brief in both working and archive epitope scaffold
		bool
		archive_identity_equivalent_scaffold_sidechains();

		/// @brief force save state of epitope-scaffold and Ab sidechains that were used in
		/// @brief complementarity design
		bool
		archive_all_complementarity_design_sidechains();

		/// @brief replace working and/or archived epitope scaffold Pose
		void
		replace_structure(
			Pose const & p,
			bool const & also_replace_archive = true
		);

		/// @brief refresh cached design positions
		/// @details requires antibody to be connected and epitope scaffold to be full atom, otherwise returns false
		bool
		refresh_cached_design_positions();


	public: // accessors

		/// @brief structure (Pose)
		inline
		Pose const &
		pose() const
		{
			return epitope_scaffold_;
		}

		/// @brief structure (Pose), modifiable
		inline
		Pose &
		pose()
		{
			return epitope_scaffold_;
		}

		/// @brief archived structure (Pose)
		inline
		Pose const &
		archived_pose() const
		{
			return archived_epitope_scaffold_;
		}

		/// @brief archived structure (Pose), modifiable
		inline
		Pose &
		archived_pose()
		{
			return archived_epitope_scaffold_;
		}

		/// @brief old to new epitope, maps native epitope loop range -> epitope-scaffold loop range
		inline
		std::map< ResidueRange, ResidueRange > const &
		native_to_new_epitope() const
		{
			return native_to_new_epitope_;
		}

		/// @brief old to new epitope, maps native epitope loop range -> epitope-scaffold loop range
		inline
		ResidueRange const &
		native_to_new_epitope(
			ResidueRange const & native_epitope_range
		) const
		{
			return native_to_new_epitope_.find( native_epitope_range )->second;
		}


		/// @brief new to old epitope, maps epitope-scaffold loop range -> native epitope loop range
		inline
		ResidueRange const &
		new_to_native_epitope(
			ResidueRange const & epitope_scaffold_range
		) const
		{
			return epitope_tracking_.find( epitope_scaffold_range )->second.native_epitope;
		}


		/// @brief maps epitope-scaffold loop range -> old scaffold gap range
		/// @note  use this to interface with internal GraftInfo
		inline
		ResidueRange const &
		epitope_to_old_gap(
			ResidueRange const & epitope_scaffold_range
		) const
		{
			return epitope_tracking_.find( epitope_scaffold_range )->second.old_gap;
		}

		/// @brief copy of primary loop closure info
		inline
		std::set< LoopClosureInfo >
		primary_closures_to_attempt() const
		{
			return component_closures_to_attempt( primary_component() );
		}

		/// @brief copy of secondary loop closure info
		inline
		std::set< LoopClosureInfo >
		secondary_closures_to_attempt() const
		{
			std::set< LoopClosureInfo > closures;

			std::set< ResidueRange > secondary = secondary_components();
			for ( std::set< ResidueRange >::const_iterator r = secondary.begin(), re = secondary.end(); r != re; ++r ) {

				std::set< LoopClosureInfo > component_closures = component_closures_to_attempt( *r );
				closures.insert( component_closures.begin(), component_closures.end() );

			}

			return closures;
		}

		/// @brief copy of component loop closure info
		inline
		std::set< LoopClosureInfo >
		component_closures_to_attempt(
			ResidueRange const & component
		) const
		{
			std::set< LoopClosureInfo > closures;

			for ( std::set< LoopClosureInfo >::const_iterator a = closures_to_attempt_.begin(), ae = closures_to_attempt_.end(); a != ae; ++a ) {
				LoopClosureInfo const & closure = *a;
				if ( closure.loop_range().overlaps( component ) ) {
					closures.insert( closure );
				}
			}

			return closures;
		}

		/// @brief get component that corresponds to (i.e. overlaps) loop closure info
		inline
		ResidueRange
		component_from_closure(
			LoopClosureInfo const & closure
		) const
		{
			std::set< ResidueRange > components( all_components() );
			for ( std::set< ResidueRange >::const_iterator l = components.begin(), le = components.end(); l != le; ++l ) {
				ResidueRange const & component = *l;
				if ( closure.loop_range().overlaps( component ) ) {
					return component;
				}
			}

			assert( false ); // dummy assert to indicate should never be here
			return ResidueRange( -1, -1 ); // shouldn't be here
		}

		///  @brief get copy of loop closure info that corresponds to loop range (not epitope range)
		///  @return returns LoopClosureInfo with range 0->0 and cut 0 if not found
		inline
		LoopClosureInfo
		closure_from_range(
			ResidueRange const & range
		) const
		{
			LoopClosureInfo scratch_closure( range, 0 );
			std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt_.find( scratch_closure );
			if ( i == closures_to_attempt_.end() ){
				return LoopClosureInfo( ResidueRange(0, 0 ), 0 );
			}

			return *i;
		}

		/// @brief get copy of loop closure info that corresponds to loop closure info
		/// @details this is more strict that closure_from_range() in that it requires the cut points
		/// @details to be the same
		///  @return returns LoopClosureInfo with range 0->0 and cut 0 if not found
		inline
		LoopClosureInfo
		closure_from_closure(
			LoopClosureInfo const & to_lookup
		) const
		{
			std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt_.find( to_lookup );
			if ( i == closures_to_attempt_.end() || i->cut() != to_lookup.cut() ) {
				return LoopClosureInfo( ResidueRange(0, 0 ), 0 );
			}

			return *i;
		}

		/// @brief copy of all loop closure info
		inline
		std::set< LoopClosureInfo >
		closures_to_attempt() const
		{
			return closures_to_attempt_;
		}

		/// @brief graft info
		inline
		GraftInfo const &
		graft_info() const
		{
			return graft_info_;
		}

		/// @brief keep natro (epitope positions that are kept during rotamer transfer and design)
		inline
		std::set< Integer > const &
		keep_natro() const
		{
			return keep_natro_;
		}

		/// @brief original match result
		inline
		MatchResult const &
		match_result() const
		{
			return graft_info_.match_result();
		}

		/// @brief return primary component range of current epitope-scaffold
		inline
		ResidueRange const &
		primary_component() const
		{
			return native_to_new_epitope_.find( graft_info_.primary_loop() )->second;
		}

		/// @brief return copy of secondary component ranges of current epitope-scaffold
		inline
		std::set< ResidueRange >
		secondary_components() const
		{
			std::set< ResidueRange > secondary_loops = graft_info_.secondary_loops(); // native epitope numbering

			std::set< ResidueRange > secondary_components;
			for ( std::set< ResidueRange >::const_iterator l = secondary_loops.begin(), le = secondary_loops.end(); l != le; ++l ) {
				secondary_components.insert( native_to_new_epitope_.find( *l )->second );
			}

			return secondary_components;
		}

		/// @brief return copy of all component ranges of current epitope-scaffold
		inline
		std::set< ResidueRange >
		all_components() const
		{
			std::set< ResidueRange > components;
			for ( std::map< ResidueRange, EpitopeTrack >::const_iterator i = epitope_tracking_.begin(), ie = epitope_tracking_.end(); i != ie; ++i ) {
				components.insert( i->first );
			}

			return components;
		}

	public: // convenience scoring

		/// @brief return total centroid score (vdw+rama+rg) assuming continuous epitope-scaffold (no chainbreaks)
		/// @details temporarily sets up a continuous fold tree (if Ab attached, uses one jump tree) and scores
		Real
		centroid_score_no_breaks() const;

		/// @brief return total fullatom score (score12) assuming continuous epitope-scaffold (no chainbreaks)
		/// @details temporarily sets up a continuous fold tree (if Ab attached, uses one jump tree) and scores
		Real
		fullatom_score_no_breaks() const;

		/// @brief return score assuming continuous epitope-scaffold (no chainbreaks)
		/// @details temporarily sets up a continuous fold tree (if Ab attached, uses one jump tree) and scores
		Real
		score_no_breaks(
			pose_ns::Score_weight_map weight_map
		) const;

		/// @brief return fullatom score of ONLY continuous epitope-scaffold (no chainbreaks)
		Real
		scaffold_centroid_score_no_breaks() const;

		/// @brief return fullatom score of ONLY continuous epitope-scaffold (no chainbreaks)
		Real
		scaffold_fullatom_score_no_breaks() const;

		/// @brief return score of ONLY continuous epitope-scaffold (no chainbreaks)
		Real
		scaffold_score_no_breaks(
			pose_ns::Score_weight_map weight_map
		) const;

		/// @brief ddG computation
		/// @details this assumes Ab is attached, please attach beforehand
		Real
		ddG() const;


	public: // convenience

		/// @brief return number of residues in scaffold
		/// @warning actually returns the number in the archived epitope scaffold,
		/// @warning so could be incorrect if you do any external grow/delete modifications
		/// @warning to the scaffold
		Integer
		n_res_scaffold() const
		{
			return archived_epitope_scaffold_.total_residue();
		}

		/// @brief return number of residues in antibody
		/// @warning actually returns the number in the archived antibody,
		/// @warning so could be incorrect if you do any external grow/delete modifications
		/// @warning to the antibody
		Integer
		n_res_Ab() const
		{
			return archived_Ab_.total_residue();
		}


	public: // export

		/// @brief export resfile
		/// @details intra-positions are NOTAA  C
		/// @details inter-positions are PIKAA  AGST or NOTAA  C (if allowed by option)
		/// @details allowed to repacked positions are NATAA
		/// @details interface design positions are PIKAA <residue type>
		/// @details everything else is NATRO
		/// @details To be relevant for design purposes, cached design data must be filled from a prior
		/// @details EpitopeScaffold::design() call, otherwise call
		/// @details EpitopeScaffold::refresh_cached_design_positions() to refresh artificially.
		void
		export_resfile(
			String const & filename
		) const;

		/// @brief export blueprint file
		/// @details intra-positions are PIKAA <amino acids sans cys>
		/// @details inter-positions are PIKAA  AGST or PIKAA <amino acids sans cys> (if allowed by option)
		/// @details allowed to repacked positions are NATAA
		/// @details interface design positions are PIKAA <residue type>
		/// @details everything else is NATRO
		/// @details moveable residues on loop closures are marked w/ their final secondary structure
		/// @details To be relevant for design purposes, cached design data must be filled from a prior
		/// @details EpitopeScaffold::design() call, otherwise call
		/// @details EpitopeScaffold::refresh_cached_design_positions() to refresh artificially.
		void
		export_blueprint(
			String const & filename
		) const;


	public: // status

		/// @brief return status (closure status, design status, etc. )
		String
		to_string() const;

		/// @brief return residue information (epitope/intra/inter/complementarity positions)
		String
		design_positions_to_string() const;


	public: // initialization

		/// @brief sets default alteration options
		/// @details sets the following options:
		///          @li grow_residue_type = gly
		inline
		void
		set_default_alteration_options()
		{
			grow_ss_type_ = 'D';
			set_grow_residue_type( 'A' );
		}

		/// @brief set default closure options
		/// @details sets the following options:
		///          @li chainbreak_criterion = 0.008
		///          @li local_rama_criterion = 5.0
		///          @li use_fragment_insertion = true
		///          @li build_cycles = 10
		///          @li use_fast_refine = false
		///          @li refine_cycles = 3
		inline
		void
		set_default_closure_options()
		{
			// closure criteria
			chainbreak_criterion_ = 0.008;
			local_rama_criterion_ = 5.0;

			// build options
			use_fragment_insertion_ = true;
			use_sequence_biased_fragments_ = false;
			use_variable_length_fragments_ = false;
			number_of_fragments_ = 200;
			build_cycles_ = 10;
			repick_frag_ = true;

			// refine options
			use_fast_refine_ = false;
			refine_cycles_ = 3;
		}

		/// @brief   set default design options
		/// @details sets the following options:
		///          @li allow_epitope_repack = false
		///          @li allow_Ab_repack = false
		///          @li allow_ALLAA_at_inter_design_positions = false;
		///          @li intra_design_cutoff = 4.0
		///          @li inter_design_cutoff = 4.0
		///          @li Ab_repack_cutoff = 4.5
		inline
		void
		set_default_design_options()
		{
			allow_epitope_repack_ = false;
			allow_epitope_design_ = false;
			allow_Ab_repack_ = false;
			allow_ALLAA_at_inter_design_positions_ = false;

			// distance cutoffs
			intra_design_position_cutoff_ = 4.0;
			inter_design_position_cutoff_ = 4.0;
			Ab_repack_cutoff_ = 4.5;

			// complementarity design
			complementarity_design_cycles_ = 3;
			complementarity_design_position_cutoff_ = 4.0;
			complementarity_shell_cutoff_ = 4.0;
			complementarity_shell_repack_ = false;
			complementarity_shell_redesign_ = false;
		}


	public: // methods

		/// @brief make design info with respect to antibody
		/// @note  makes design matrix and fills internal design info
		FArray2D< bool >
		make_design_info();


	public: // checkpointing interface -- HACK due to numerical problems in r++

		/// @brief loop closed during build simul trajectory?
		inline
		bool
		closed_during_trajectory( LoopClosureInfo const & loop ) const {
			std::map< LoopClosureInfo, bool >::const_iterator i = closed_during_trajectory_.find( loop );
			if ( i == closed_during_trajectory_.end() ) {
				return false;
			} else {
				return i->second;
			}
		}

		/// @brief clear checkpoint status
		inline
		void
		clear_closed_during_trajectory() {
			closed_during_trajectory_.clear();
		}

		/// @brief save loop checkpoint file
		void
		save_loop_checkpoint( std::string const & filename ) const;

		/// @brief load loop checkpoint file
		void
		load_loop_checkpoint( std::string const & filename );


	private: // methods

		/// @brief make predesign epitope scaffold
		void
		make_predesign_epitope_scaffold(
			Pose const & original_scaffold,
			Pose const & epitope,
			GraftInfo const & graft_info,
			std::set< Integer > const & keep_natro = std::set< Integer >()
		);

		/// @brief override design info using graft info settings
		void
		override_design_info_using_graft_info(
			FArray2D< bool > & design_matrix
		);

		/// @brief override design info using complementarity design and shell info
		/// @param[out] design_matrix  modified design matrix only if Ab is attached
		/// @return false if Ab is attached, true otherwise
		bool
		override_design_info_using_complementarity_info(
			FArray2D< bool > & design_matrix
		);

		/// @brief erase keep_natro_ residues (correctly offset-ed to current epitope scaffold indexing) from given set
		void
		erase_keep_natro_residues(
			std::set< Integer > & residues
		);

		/// @brief erase all epitope residues from given set
		void
		erase_epitope_residues(
			std::set< Integer > & residues
		);

		/// @brief erase all complementarity design residues from given set
		void
		erase_complementarity_residues(
			std::set< Integer > & residues
		);

		/// @brief return fixed residues, this includes epitope/keep_natro and Ab if Ab is not allowed
		/// @brief to repack
		std::set< Integer >
		get_fixed_residues();

		/// @brief find residues on scaffold near moveable loop residues dictated by intra design position cutoff
		// vds moveable residue tracking
		std::set< Integer >
		find_residues_near_moveable(
			Pose const & scaffold,
			std::set< LoopClosureInfo > const & closures
		);

		/// @brief identify scaffold - antibody interface residues given distance cutoff
		/// @details computes interface residues by comparing sidechains of each component
		/// @details against the full residues of the other
		std::set< Integer >
		identify_interface_residues(
			Real const & interface_distance = 4.5
		);


	private: // loop methods

		/// @brief check to see if a loop's bb is to be grafted by querying internal GraftInfo
		bool const &
		do_graft_bb(
			LoopClosureInfo const & closure
		) const;

		/// @brief check to see if a component's sc is to be grafted by querying internal GraftInfo
		bool const &
		do_graft_sc(
			ResidueRange const & component
		) const;

		/// @brief check closure criteria, modifies LoopClosureInfo 'is_closed' state
		inline
		void
		check_closure_criteria(
			LoopClosureInfo const & closure_info
		) const
		{
			closure_info.set_chainbreak_score( score_cut_in_Pose_linear( epitope_scaffold_, closure_info.cut(), 1 ) ); // overlap = 1
			score_rama_for_moveable( epitope_scaffold_, closure_info );
			closure_info.set_is_closed( closure_info.chainbreak_score() <= chainbreak_criterion_ &&
			                            closure_info.ramas_are_not_above( local_rama_criterion_ ) );
		}

		/// @brief setup proper allow bb move for set of loops
		void
		setup_proper_allow_bb_move(
			std::set< LoopClosureInfo > const & closures
		);

		/// @brief idealize cutpoint but retain psi on the n-side and phi on the c-side of the cut
		/// @brief using given values if idealize_loop_geometry is active
		/// @details if values don't exist in map, then regular idealization occurs
		/// @note  this is useful when it's necessary to retain the existing torsions of the cut
		void
		idealize_cutpoint_retaining_psiphi(
			Pose & pose,
			Integer const & cut,
			std::map< Integer, Real > const & original_torsions
		);


		/// @brief idealize cutpoint but retain psi on the n-side and phi on the c-side of the cut
		/// @brief using given values if idealize_loop_geometry is active
		/// @note  this is useful when it's necessary to retain the existing torsions of the cut
		void
		idealize_cutpoint_retaining_psiphi(
			Pose & pose,
			Integer const & cut,
			Real const & nside_psi,
			Real const & cside_phi
		);


	public: // TEMPORARY

		/// @brief get proper secondary structure string for centroid level loop closure
		String
		loop_closure_ss_string() const;

		/// @brief  get proper amino acid identity string for centroid level loop closure
		/// @return if using sequence biased fragments, returns the amino acid string for
		///         centroid level loop closure, otherwise returns ""
		String
		loop_closure_seq_string() const;


	private: // update info

		/// @brief update Ab transformation matrix
		inline
		void
		update_Ab_transform(
			FArray2D_float original_bb_crd,
			FArray2D_float new_bb_crd
		)
		{
			// calculate interim transform
			FArray2D_float interim_transform( 4, 4, 0.0 );
			get_GL_matrix( original_bb_crd.a( 1, 1 ), original_bb_crd.a( 1, 2 ), original_bb_crd.a( 1, 3 ),
			               new_bb_crd.a( 1, 1 ), new_bb_crd.a( 1, 2 ), new_bb_crd.a( 1, 3 ),
			               interim_transform );

			// calculate new Ab transform
			FArray2D_float original_transform = Ab_transform_;
			compose_GL( interim_transform, original_transform, Ab_transform_ );
		}

		/// @brief update old epitope-scaffold component ranges to altered numbering
		/// @param[in] map from residues with old epitope-scaffold component numbering to altered numbering
		void
		alter_maps(
			std::map< Integer, Integer > const & es_to_new_es
		);

		/// @brief alter secondary component closures to single break instead of double break
		/// @warning this will modify both working AND archive pose
		/// @warning IMPORTANT: the way this is currently setup requires ALL possible moveable residues to be
		/// @warning moveable
		void
		switch_secondaries_to_single_break(
			bool const & use_graft_info_settings = true
		);

		/// @brief   override secondary structure of closure sites according to graft info
		/// @details secondary structure of both working and archived epitope scaffolds are
		///          overridden following graft info settings
		void
		override_closure_site_secondary_structure();

		/// @brief   override mobility of grown residues according to graft info
		/// @details looks for uppercase of GraftInfo::GROW_FROZEN and erases
		///          those residues from the appropriate LoopClosureInfo
		void
		override_closure_site_grown_moveable_residues();

		/// @brief update closure information
		inline
		void
		update_closures(
			LoopClosureInfo const & i
		)
		{
			closures_to_attempt_.erase( i );
			closures_to_attempt_.insert( i );
		}


	private: // history mechanisms

		/// @brief replace backbone in indicated residues with archived backbone
		/// @details resets via set phi/psi/omega and not coordinate replacement
		/// @details also updates corresponding antibody transform
		template< typename MoveableResidueIterator >
		inline
		void
		set_backbone_from_archive(
			MoveableResidueIterator const & begin,
			MoveableResidueIterator const & end
		)
		{
			// first store middle of primary component to use in updating Ab_transform
			FArray2D_float original_primary_middle_bb;
			extract_1bb_from_Pose( epitope_scaffold_, middle_of_range( primary_component() ), original_primary_middle_bb );

			for ( MoveableResidueIterator i = begin; i != end; ++i ) {
				Integer const & res = *i;

				// backbone angles
				epitope_scaffold_.set_phi( res, archived_epitope_scaffold_.phi( res ) );
				epitope_scaffold_.set_psi( res, archived_epitope_scaffold_.psi( res ) );
				epitope_scaffold_.set_omega( res, archived_epitope_scaffold_.omega( res ) );

				// backbone identity
				epitope_scaffold_.set_secstruct( res, archived_epitope_scaffold_.secstruct( res ) );
			}

			// update Ab transform
			FArray2D_float new_primary_middle_bb;
			extract_1bb_from_Pose( epitope_scaffold_, middle_of_range( primary_component() ), new_primary_middle_bb );
			update_Ab_transform( original_primary_middle_bb, new_primary_middle_bb );
		}


	private: // convenience functions for jumps and fold tree setup

		/// @brief return jump index/label for component
		/// @return jump label/index in epitope-scaffold fold tree setup, -1 if no jump
		inline
		Integer const &
		jump_label(
			ResidueRange const & component
		) const
		{
			return epitope_tracking_.find( component )->second.jump_label;
		}

		/// @brief return jump residues in the current epitope scaffold fold tree
		std::pair< Integer, Integer >
		jump_residues(
			ResidueRange const & component
		) const;

		/// @brief return jump residues in the current epitope scaffold fold tree
		std::pair< Integer, Integer >
		jump_residues(
			Integer const & label
		) const;

		/// @brief return middle of range
		static
		inline
		Integer
		middle_of_range(
			ResidueRange const & range
		)
		{
			return std::ceil( static_cast< Real >( range.length() ) / 2.0 ) + range.begin() - 1;
		}


	private: // data

		GraftInfo graft_info_; // contains original match result info and additional closure instructions

		/// @brief contains epitope positions (native numbering) to be kept during transfer of epitope rotamers
		/// @brief and during design
		/// @note  If this variable is not filled then entire epitope component ranges will be transferred during
		/// @note  predesign structure creation.
		std::set< Integer > keep_natro_;

		// following are built by making predesign structure
		/// @brief maps native epitope component range -> epitope-scaffold component range
		std::map< ResidueRange, ResidueRange > native_to_new_epitope_;

		/// @brief maps epitope-scaffold component range -> various epitope info (e.g. jump, native epitope, old gap )
		/// @note (see struct definition at top of class for more information)
		std::map< ResidueRange, EpitopeTrack > epitope_tracking_;

		/// @brief closure information for all components
		/// @note use primary_closures_to_attempt() secondary_closures_to_attempt(), component_closures_to_attempt
		/// @note to get respective closure for components
		std::set< LoopClosureInfo > closures_to_attempt_;

		/// @brief only used in N2C/C2N, these are the "lever" arm residues
		std::set< Integer > arm_residues_;

		Pose epitope_scaffold_; // working epitope scaffold Pose
		Pose archived_epitope_scaffold_; // archived epitope scaffold for operations like rotamer transfer

		// Ab connections
		bool Ab_is_connected_;
		ConnectionResult Ab_connection_;
		/// @brief tracks Ab transform so if Ab is ever fully disconnected and needs to be reconnected
		/// @brief matrix can also be used to transform epitope, as the two are currently tied together
		FArray2D_float Ab_transform_;
		Pose archived_Ab_; // never in correct (tranformed) position, must always transform via Ab_transform_

		/// @brief when making predesign structure, take certain atoms from scaffold at the merge points?
		/// @brief default is true
		bool micromanage_termini_;

		// alteration parameters (grow/remove residues)
		char grow_ss_type_;
		char grow_residue_type_;
		std::set< Integer > grown_residues_; // tracks the grown residues so secondary structure can be reset

		// closure criteria
		Real chainbreak_criterion_;
		Real local_rama_criterion_;

		// loop build options
		bool use_fragment_insertion_;
		bool use_sequence_biased_fragments_;
		bool use_variable_length_fragments_;
		Integer number_of_fragments_;
		Integer build_cycles_;

		// loop refine options
		bool use_fast_refine_;
		Integer refine_cycles_;

		// design options
		bool allow_epitope_repack_;
		bool allow_epitope_design_;
		bool allow_Ab_repack_;
		bool allow_ALLAA_at_inter_design_positions_;
		Real intra_design_position_cutoff_; // distance cutoff when identifying intra design positions
		Real inter_design_position_cutoff_; // distance cutoff when identifying inter design positions
		Real Ab_repack_cutoff_; // distance cutoff when identifying positions when Ab is allowed to repack

		// built by make design info
		std::set< Integer > intra_design_positions_;
		std::set< Integer > inter_design_positions_;
		std::set< Integer > Ab_repack_positions_;

		// complementarity design info
		Integer complementarity_design_cycles_;
		/// @brief distance cutoff when identifying complementarity design positions
		Real complementarity_design_position_cutoff_;
		/// @brief list of scaffold design positions used during complementarity design
		std::set< Integer > complementarity_design_positions_;
		/// @brief list of Ab positions used to find complementarity design positions
		std::set< Integer > complementarity_Ab_positions_;
		/// @brief distance cutoff when identifying complementarity shell positions for repack/redesign
		Real complementarity_shell_cutoff_;
		bool complementarity_shell_repack_;
		bool complementarity_shell_redesign_;
		/// @brief list of scaffold shell positions surrounding complementarity design positions
		///        found during an override_design_info_using_complementarity_info() or design() call
		std::set< Integer > complementarity_shell_positions_;

		//vds additional tracking for design info
		//vds these are residues in the original scaffold that started off near the residues that are moving.
		std::set< Integer > residues_originally_near_moveable_;

		// only for testing/benchmarking
		bool idealize_loop_geometry_;
		bool repick_frag_;

		// checkpointing
		std::map< LoopClosureInfo, bool > closed_during_trajectory_;
};


} // namespace design
} // namespace epigraft

#endif /*INCLUDED_epigraft_design_EpitopeScaffold_HH_*/
