// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   GraftInfo.cc
/// @brief  Holds information for grafting: match result & components, closure options.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)


// unit headers
#include <epigraft/design/GraftInfo.hh>

// package headers
#include <epigraft/design/design_io.hh>
#include <epigraft/epigraft_io.hh>

// utility headers
#include <utility/io/izstream.hh>

// ObjexxFCL headers
#include <ObjexxFCL/string.functions.hh>

// C++ headers
#include <algorithm>
#include <cmath>
#include <iostream>
#include <sstream>
#include <string>
#include <vector>


namespace epigraft {
namespace design {


// what follows next is for static initialization of private constants

/// @brief no movement, keep current settings
char const GraftInfo::FREEZE;

/// @brief indicates start of amino acid instructions
char const GraftInfo::SSBM_START;

/// @brief indicates open break (cutpoint)
char const GraftInfo::OPEN_BREAK;

/// @brief indicates closed break (cutpoint)
char const GraftInfo::CLOSED_BREAK;

/// @brief indicates open break (cutpoint) and allowance of rigid body move
char const GraftInfo::OPEN_BREAK_RB;

/// @brief keep secondary structure but move backbone
char const GraftInfo::KEEP_AND_MOVE;

/// @brief delete residue at that position
char const GraftInfo::DELETE;

/// @brief grow a new residue as helix
char const GraftInfo::GROW_HELIX;

/// @brief grow a new residue as sheet
char const GraftInfo::GROW_SHEET;

/// @brief grow a new residue as loop
char const GraftInfo::GROW_LOOP;

/// @brief grow a new residue with any secondary structure ("degenerate")
char const GraftInfo::GROW_ANY;

/// @brief grow a new residue, but mark it as non-moveable; a special override for
///        when "history" is a problem
char const GraftInfo::GROW_FROZEN;

/// @brief use helix at existing residue position
char const GraftInfo::USE_HELIX;

/// @brief use sheet at existing residue position
char const GraftInfo::USE_SHEET;

/// @brief use loop at existing residue position
char const GraftInfo::USE_LOOP;

/// @brief use any secondary structure ("degenerate") at existing residue position
char const GraftInfo::USE_ANY;

/// @brief indicates start of amino acid instructions
char const GraftInfo::AAI_START;

/// @brief use protocol specific default behavior
char const GraftInfo::DEFAULT;

/// @brief indicates allow repack
char const GraftInfo::REPACK;

/// @brief any amino acid ("degenerate")
char const GraftInfo::ANY_AA;

/// @brief all amino acids except cys and pro ("restricted/regular degenerate")
char const GraftInfo::NO_CP;

/// @brief nonpolar residues
char const GraftInfo::NONPOLAR;

/// @brief aromatic residues
char const GraftInfo::AROMATIC;

/// @brief polar residues
char const GraftInfo::POLAR;

/// @brief charged residues
char const GraftInfo::CHARGED;

/// @brief positively charged residues
char const GraftInfo::POSITIVE;

/// @brief negatively charged residues
char const GraftInfo::NEGATIVE;


/// @brief   return secondary structure string
/// @details returns the secondary structure string constructed from
///          the ssbm by taking out break and deletion information and
///          capitalizing all secondary structure
String
GraftInfo::ss_string(
	ResidueRange const & scaffold_gap_range
) const
{
	String const & ssbm = ssbm_string( scaffold_gap_range );
	String const valid_ci = valid_ss_characters();

	std::ostringstream ss;

	for ( String::const_iterator i = ssbm.begin(), ie = ssbm.end(); i != ie; ++i ) {
		if ( valid_ci.find( *i ) != String::npos ) {
			ss << (*i);
		}
	}

	return ObjexxFCL::uppercased( ss.str() );
}


/// @brief return aa string
/// @details returns the amino acid string constructed from the aai
String
GraftInfo::aa_string(
	ResidueRange const & scaffold_gap_range
) const
{
	String const & aai = aai_string( scaffold_gap_range );
	String const valid_ci = valid_aa_characters();

	std::ostringstream ss;

	for ( String::const_iterator i = aai.begin(), ie = aai.end(); i != ie; ++i ) {
		if ( valid_ci.find( *i ) != String::npos ) {
			ss << (*i);
		}
	}

	return ss.str();
}


/// @brief rb perturbing any secondary component?
bool
GraftInfo::perturbing_any_secondary_component_rb() const
{
	bool perturb = false;

	for ( Integer i = 2, ie = match_result_.components.size(); i <= ie; ++i ) { // note that primary component is SKIPPED
		MatchComponent const & component = match_result_.components[ i ];

		perturb = perturb || graft_tracking_.find( component.scaffold_gap_range )->second.perturb_rb;
	}

	return perturb;
}

/// @brief closing any secondaries as single break?
bool
GraftInfo::closing_any_secondary_as_single_break()
{
	bool flag = false;

	for ( Integer i = 2, ie = match_result_.components.size(); i <= ie; ++i ) { // note that primary component is SKIPPED
		MatchComponent const & component = match_result_.components[ i ];

		flag = flag || graft_tracking_.find( component.scaffold_gap_range )->second.close_as_single_break;
	}

	return flag;
}


/// @brief returns help for loading external info from file
String
GraftInfo::external_graft_info_help()
{
	std::ostringstream ss;

	ss << '\n';
	ss << "*** THIS INFORMATION DESCRIBES THE OLD FORMAT AND IS DEPRECATED ***" << '\n';
	ss << "*** PLEASE SEE DOCUMENTATION FOR NEW GRAFT INFO FORMAT ***" << '\n';
	ss << "Column entries are delimited by whitespace." << std::endl;
	ss << "graft_bb/sc' columns are either 'true' or 'false' and 'graft_bb' will take" << std::endl;
	ss << "   precedence over 'graft_sc'.  i.e. If 'graft_bb' is true then 'graft_sc'" << std::endl;
	ss << "   will automatically be set to true." << std::endl;
	ss << "If 'graft_bb' is false, then no additional columns are necessary." << std::endl;
	ss << "Restriction: 'tether_*term' >= 0" << std::endl;
	ss << "Restriction: 'tether_*term' + 'linker_*term' >= 0" << std::endl;
	ss << "Restriction: 'tether_*term' + 'linker_*term' >= 'moveable_*term'" << std::endl;
	ss << "Restriction: epitope_tether_nterm < 1 + (length of epitope loop)/2" << std::endl;
	ss << "Restriction: epitope_tether_cterm < (length of epitope loop)/2" << std::endl;
	ss << "Restriction: epitope_linker_*term >= 0" << std::endl;
	ss << "Current file format example is as follows, note that residue ranges" << std::endl;
	ss << "are scaffold gap!  Remember to include the '|' separators." << std::endl;
	ss << "if 'move_all_epitope_residues' true then 'epitope_tether_*' and 'epitope_moveable_*' is overridden" << std::endl;
	ss << "'perturb_rb' and 'move_all_epitope_residues' for primary component are currently IGNORED" << std::endl;
	ss << "'repack_epitope_residues' is valid for ALL components" << std::endl;
	ss << "( scaffold_gap ) in example below is only for reference and not part of file format" << std::endl;
	ss << "(scaffold_gap_begin scaffold_gap_end) graft_bb graft_sc scaffold_tether_nterm scaffold_linker_nterm scaffold_moveable_nterm | scaffold_tether_cterm scaffold_linker_cterm scaffold_moveable_cterm | epitope_tether_nterm epitope_linker_nterm epitope_moveable_nterm | epitope_tether_cterm epitope_linker_cterm epitope_moveable_cterm    perturb_rb?    move_all_epitope_residues?    close_as_single_break?   repack_epitope_residues?" << std::endl;
	ss << "(362   379)   true     true    3  -1   3  |  4   2   2  |  0   0   0  |  0   0   0" << std::endl;
	ss << "(478   482)   true     true    4   0   4  |  4   0   4  |  0   3   3  |  1   0   1" << std::endl;
	ss << "(201   210)   false    true" << std::endl;
	ss << "( 97   111)   false    false" << std::endl;
	ss << "(  7    45)   true     true    6   3   3  |  6   4   4  |  2   0   2  |  1   1   2   true   false" << std::endl;
	ss << "(  7    45)   true     true    6   3   9  |  6   4   4  |  0   0   0  |  0   0   0   false   true   true   true" << std::endl;
	ss << "(     127  135 )  true     true    3   0   3  |  6   4   4  |  0   0   0  |  0   0   0    true  true" << std::endl;

	return ss.str();
}


/// @brief   set graft info instructions from character encoding
/// @details Line may have one or two "strings".  The first gives
///          secondary structure and move instructions, while the
///          second gives amino acid identity instructions.  The
///          strings may have arbitrary spaces for readability (these
///          will be removed by the program), but must be separated
///          by a '&' character, which indicates the start of the
///          a.a. identity instructions.
/// @return  false if given scaffold gap range is invalid, otherwise
///          return true, which indicates success
bool
GraftInfo::add_encoded_info_from_line(
	Integer const & scaffold_gap_begin,
	Integer const & scaffold_gap_end,
	String const & line
)
{
	if ( locked_ ) {
		access_denied();
		return false;
	}

	// current scaffold gap
	ResidueRange scaffold_gap( scaffold_gap_begin, scaffold_gap_end );

	// check to make sure scaffold gap is valid
	if ( graft_tracking_.find( scaffold_gap ) == graft_tracking_.end() ) { // range doesn't belong to this match
		return false;
	}

	// cache graft tracking
	GraftTrack & track = graft_tracking_[ scaffold_gap ];

	// remove all spaces from the string
	String const collapsed_line = collapsed_spaces( line );

	// check to see if instructions exist by looking for aa start code
	Size const ssbm_sep_index = collapsed_line.find( SSBM_START );
	Size const aai_sep_index = collapsed_line.find( AAI_START );

	// check to see if backbone and sidechains are being grafted
	utility::vector1< String > entries;
	split_string( line.substr( 0, line.find( SSBM_START ) ), entries );

	bool graft_bb = true;
	if ( entries[ 1 ] == "graft_bb" ) {
		graft_bb = true;
	} else if ( entries[ 1 ] == "keep_bb" ) {
		graft_bb = false;
	} else {
		utility::exit( __FILE__, __LINE__, "GraftInfo ERROR: unknown 'bb' setting in line: " + line );
	}

	bool graft_sc = true;
	if ( entries[ 2 ] == "graft_sc" ) {
		graft_sc = true;
	} else if ( entries[ 2 ] == "keep_sc" ) {
		graft_sc = false;
	} else {
		utility::exit( __FILE__, __LINE__, "GraftInfo ERROR: unknown 'sc' setting in line: " + line );
	}

	// check keep instructions
	if ( graft_bb && !graft_sc ) {
		graft_sc = true;
		std::ostringstream ss;
		ss << "GraftInfo WARNING: The line below indicates request to graft backbone but not" << std::endl;
		ss << "                   sidechain.  This is not possible, so setting graft sidechain" << std::endl;
		ss << "                   to true." << std::endl;
		ss << line << std::endl;
		std::cerr << ss.str() << std::endl;
	}

	// if grafting backbone check to make sure instructions exist
	if ( graft_bb && ( ssbm_sep_index == String::npos ) ) {
		std::ostringstream ss;
		ss << "GraftInfo ERROR: for scaffold gap range " << scaffold_gap_begin << ' ' << scaffold_gap_end;
		ss << " request for 'graft_bb' but missing secondary structure/break/moveable instructions in line: " << std::endl;
		ss << line << std::endl;
		utility::exit( __FILE__, __LINE__, ss.str() );
	}

	// set graft_bb/sc in tracking
	track.do_graft_bb = graft_bb;
	track.do_graft_sc = graft_sc;

	// split the line into the two possible instruction strings
	String const ssbm = ssbm_sep_index != String::npos ?
	                    collapsed_line.substr( ssbm_sep_index + 1, aai_sep_index - ssbm_sep_index - 1 ) : ""; // secondary structure, break, and moveable instructions
	String const aai  = aai_sep_index != String::npos ?
	                    collapsed_line.substr( aai_sep_index + 1, collapsed_line.length() - aai_sep_index - 1 ) : ""; // amino acid identity

	// do ssbm
	parse_and_store_ssbm( scaffold_gap, ssbm, line );

	// do aai
	parse_and_store_aai( scaffold_gap, aai, line );

	return true;
}


/// @brief add additional information about graft from a line (string)
/// @param[in] start_from  start from column entry (counting starting from 1)
/// @note  Column entries are delimited by whitespace.
/// @note  'graft_bb/sc' columns are either 'true' or 'false' and
/// @note  'graft_bb' will take precedence over 'graft_sc', i.e.
/// @note  If 'graft_bb' is true then 'graft_sc' will automatically
/// @note  be set to true.
/// @note  If 'graft_bb' is false, then no additional columns are necessary.
/// @note  Restriction: 'tether_*term' >= 0
/// @note  Restriction: 'tether_*term' + 'linker_*term' >= 0
/// @note  Restriction: 'tether_*term' + 'linker_*term' >= 'moveable_*term'
/// @note  Restriction: epitope_tether_nterm < 1 + (length of epitope loop)/2
/// @note  Restriction: epitope_tether_cterm < (length of epitope loop)/2
/// @note  Restriction: epitope_linker_*term >= 0
/// @note  Current file format example is as follows, note that residue
/// @note  ranges are scaffold gap!  Remember to include the '|' separators.
/// @note  if 'move_all_epitope_residues' true then 'epitope_tether_*' and 'epitope_moveable_*' is overridden
/// @note  'perturb_rb' and 'move_all_epitope_residues' for primary component are currently IGNORED
/// @note  'repack_epitope_residues' is valid for ALL components
/// @note  ( scaffold_gap ) in example below is only for reference and not part of file format
/// @note  (scaffold_gap_begin scaffold_gap_end) graft_bb graft_sc scaffold_tether_nterm scaffold_linker_nterm scaffold_moveable_nterm | scaffold_tether_cterm scaffold_linker_cterm scaffold_moveable_cterm | epitope_tether_nterm epitope_linker_nterm epitope_moveable_nterm | epitope_tether_cterm epitope_linker_cterm epitope_moveable_cterm   perturb_rb?   move_all_epitope_residues?   close_as_single_break  repack_epitope_residues?
/// @note  (    362   379 )  true     true    3  -1   3  |  4   2   2  |  0   0   0  |  0   0   0
/// @note  (    478   482 )  true     true    4   0   4  |  4   0   4  |  0   3   3  |  1   0   1
/// @note  (    201   210 )  false    true
/// @note  (     97   111 )  false    false
/// @note  (      7   45  )  true     true    6   3   3  |  6   4   4  |  2   0   2  |  1   1   2    true  false
/// @note  (     67   93  )  true     true    6   3   9  |  6   4   4  |  0   0   0  |  0   0   0    false  true  true  true
/// @note  (     127  135 )  true     true    3   0   3  |  6   4   4  |  0   0   0  |  0   0   0    true  true
/// @return  whether or not info successfully added from line
bool
GraftInfo::add_info_from_line(
	Integer const & scaffold_gap_begin,
	Integer const & scaffold_gap_end,
	String const & line,
	Size const & start_from
)
{
	using std::istringstream;

	if ( locked_ ) {
		access_denied();
		return false;
	}

	utility::vector1< String > entries;

	split_string( line, entries, start_from ); // grab all columns from input file

	if ( entries.size() < 2 ) { // something's missing, send error
		std::ostringstream ss;
		ss << "GraftInfo::load_additional_graft_info: missing graft info entries in line: " << std::endl;
		ss << line << std::endl;
		utility::exit( __FILE__, __LINE__, ss.str() );
	}

	// data
//	Integer scaffold_gap_begin, scaffold_gap_end;
	String graft_bb_str;
	String graft_sc_str;
	Integer scaffold_tether_nterm, scaffold_tether_cterm;
	Integer scaffold_linker_nterm, scaffold_linker_cterm;
	Integer scaffold_moveable_nterm, scaffold_moveable_cterm;
	Integer epitope_tether_nterm, epitope_tether_cterm;
	Integer epitope_linker_nterm, epitope_linker_cterm;
	Integer epitope_moveable_nterm, epitope_moveable_cterm;
	String bar1, bar2, bar3; // enforce use of '|' for input safety
	String perturb_rb_str;
	String move_all_epitope_residues_str;
	String close_as_single_break_str;
	String repack_epitope_residues_str;

//	istringstream( entries[ 1 ] ) >> scaffold_gap_begin;
//	istringstream( entries[ 2 ] ) >> scaffold_gap_end;
	graft_bb_str = entries[ 1 ];
	graft_sc_str = entries[ 2 ];

	// check just first character of graft_bb/sc strings,
	// this should be robust to typos
	bool graft_bb = ( graft_bb_str.c_str()[0] == 't' || graft_bb_str.c_str()[0] == 'T' );
	bool graft_sc = ( graft_sc_str.c_str()[0] == 't' || graft_sc_str.c_str()[0] == 'T' );

	// sanity check
	if ( !graft_bb && ( graft_bb_str.c_str()[0] != 'f' && graft_bb_str.c_str()[0] != 'F' ) ) {
		std::ostringstream ss;
		ss << "GraftInfo ERROR: 'graft_bb' setting incorrect in line: " << std::endl;
		ss << line << std::endl;
		utility::exit( __FILE__, __LINE__, ss.str() );
	}
	if ( !graft_sc && ( graft_sc_str.c_str()[0] != 'f' && graft_sc_str.c_str()[0] != 'F' ) ) {
		std::ostringstream ss;
		ss << "GraftInfo ERROR: 'graft_sc' setting incorrect in line: " << std::endl;
		ss << line << std::endl;
		utility::exit( __FILE__, __LINE__, ss.str() );
	}

	// check keep instructions
	if ( graft_bb && !graft_sc ) {
		graft_sc = true;
		std::ostringstream ss;
		ss << "GraftInfo WARNING: The line below indicates request to graft backbone but not" << std::endl;
		ss << "                   sidechain.  This is not possible, so setting graft sidechain" << std::endl;
		ss << "                   to true." << std::endl;
		ss << line << std::endl;
		std::cerr << ss.str() << std::endl;
	}

	if ( graft_bb && entries.size() < 17 ) {
		std::ostringstream ss;
		ss << "GraftInfo ERROR: 'graft_bb' = true but missing flanking residue entries in line: " << std::endl;
		ss << line << std::endl;
		utility::exit( __FILE__, __LINE__, ss.str() );
	}

	// current scaffold gap
	ResidueRange scaffold_gap( scaffold_gap_begin, scaffold_gap_end );

	// check to make sure scaffold gap is valid
	if ( graft_tracking_.find( scaffold_gap ) == graft_tracking_.end() ) { // range doesn't belong to this match
		return false;
	}

	// cache graft tracking
	GraftTrack & track = graft_tracking_[ scaffold_gap ];

	// mark whether or not to graft bb/sc
	track.do_graft_bb = graft_bb;
	track.do_graft_sc = graft_sc;

	// continue on to other columns if possible
	if ( graft_bb || entries.size() >= 17 ) {
		istringstream( entries[ 3 ] ) >> scaffold_tether_nterm;
		istringstream( entries[ 4 ] ) >> scaffold_linker_nterm;
		istringstream( entries[ 5 ] ) >> scaffold_moveable_nterm;
		bar1 = entries[ 6 ];
		istringstream( entries[ 7 ] ) >> scaffold_tether_cterm;
		istringstream( entries[ 8 ] ) >> scaffold_linker_cterm;
		istringstream( entries[ 9 ] ) >> scaffold_moveable_cterm;
		bar2 = entries[ 10 ];
		istringstream( entries[ 11 ] ) >> epitope_tether_nterm;
		istringstream( entries[ 12 ] ) >> epitope_linker_nterm;
		istringstream( entries[ 13 ] ) >> epitope_moveable_nterm;
		bar3 = entries[ 14 ];
		istringstream( entries[ 15 ] ) >> epitope_tether_cterm;
		istringstream( entries[ 16 ] ) >> epitope_linker_cterm;
		istringstream( entries[ 17 ] ) >> epitope_moveable_cterm;

		// first make sure all '|' separators are in place to
		// prevent any accidental input mistakes
		if ( bar1 != "|" || bar2 != "|" || bar3 != "|" ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: '|' separator missing for line: " << std::endl;
			ss << line << std::endl;
			utility::exit( __FILE__, __LINE__, ss.str() );
		}

		// tether*term must be positive
		scaffold_tether_nterm = std::abs( scaffold_tether_nterm );
		scaffold_tether_cterm = std::abs( scaffold_tether_cterm );
		epitope_tether_nterm = std::abs( epitope_tether_nterm );
		epitope_tether_cterm = std::abs( epitope_tether_cterm );

		// moveable*term must be positive
		scaffold_moveable_nterm = std::abs( scaffold_moveable_nterm );
		scaffold_moveable_cterm = std::abs( scaffold_moveable_cterm );
		epitope_moveable_nterm = std::abs( epitope_moveable_nterm );
		epitope_moveable_cterm = std::abs( epitope_moveable_cterm );

		// check and make sure tether*term >= 0 and
		//                     tether*term + linker*term > 0 and
		//                     tether*term + linker*term >= moveable*term
		// restriction on epitope tether terms versus epitope loop length will
		// be checked during locking
		// restriction on epitope linker >= 0 will be checked during locking
		if ( scaffold_tether_nterm < 0 || scaffold_tether_cterm < 0 ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: error make sure scaffold tether*term >= 0 in line:" << std::endl;
			ss << line << std::endl;
			utility::exit( __FILE__, __LINE__, ss.str() );
		}
		if ( scaffold_tether_nterm + scaffold_linker_nterm < 0 ||
		     scaffold_tether_nterm + scaffold_linker_nterm < scaffold_moveable_nterm ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: error make sure scaffold n-side tether*term + linker*term >= 0 and tether*term + linker*term >= moveable*term in line: " << std::endl;
			ss << line << std::endl;
			utility::exit( __FILE__, __LINE__, ss.str() );
		}
		if ( scaffold_tether_cterm + scaffold_linker_cterm < 0 ||
		     scaffold_tether_cterm + scaffold_linker_cterm < scaffold_moveable_cterm ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: error make sure scaffold c-side tether*term + linker*term >= 0 and tether*term + linker*term >= moveable*term in line: " << std::endl;
			ss << line << std::endl;
			utility::exit( __FILE__, __LINE__, ss.str() );
		}
		if ( epitope_tether_nterm < 0 || epitope_tether_cterm < 0 ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: error make sure epitope tether*term >= 0 in line:" << std::endl;
			ss << line << std::endl;
			utility::exit( __FILE__, __LINE__, ss.str() );
		}
		if ( epitope_tether_nterm + epitope_linker_nterm < 0 ||
		     epitope_tether_nterm + epitope_linker_nterm < epitope_moveable_nterm ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: error make sure epitope n-side tether*term + linker*term >= 0 and tether*term + linker*term >= moveable*term in line: " << std::endl;
			ss << line << std::endl;
			utility::exit( __FILE__, __LINE__, ss.str() );
		}
		if ( epitope_tether_cterm + epitope_linker_cterm < 0 ||
		     epitope_tether_cterm + epitope_linker_cterm < epitope_moveable_cterm ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: error make sure epitope c-side tether*term + linker*term >= 0 and tether*term + linker*term >= moveable*term in line: " << std::endl;
			ss << line << std::endl;
			utility::exit( __FILE__, __LINE__, ss.str() );
		}

		// store values
		track.scaffold_tether_residues = Flanking( scaffold_tether_nterm, scaffold_tether_cterm );
		track.scaffold_linker_residues = Flanking( scaffold_linker_nterm, scaffold_linker_cterm );
		track.scaffold_moveable_residues = Flanking( scaffold_moveable_nterm, scaffold_moveable_cterm );
		track.epitope_tether_residues = Flanking( epitope_tether_nterm, epitope_tether_cterm );
		track.epitope_linker_residues = Flanking( epitope_linker_nterm, epitope_linker_cterm );
		track.epitope_moveable_residues = Flanking( epitope_moveable_nterm, epitope_moveable_cterm );

		if ( entries.size() >= 18 ) {
			// check just first character of perturb_rb string
			perturb_rb_str = entries[ 18 ];
			track.perturb_rb = ( perturb_rb_str.c_str()[0] == 't' || perturb_rb_str.c_str()[0] == 'T' );
		} else {
			track.perturb_rb = false;
		}

		if ( entries.size() >= 19 ) {
			// check just first character of move_all string
			move_all_epitope_residues_str = entries[ 19 ];
			track.move_all_epitope_residues = ( move_all_epitope_residues_str.c_str()[0] == 't' || move_all_epitope_residues_str.c_str()[0] == 'T' );
		} else {
			track.move_all_epitope_residues = false;
		}

		if ( entries.size() >= 20 ) {
			close_as_single_break_str = entries[ 20 ];
			track.close_as_single_break = ( close_as_single_break_str.c_str()[0] == 't' || close_as_single_break_str.c_str()[0] == 'T' );
		} else {
			track.close_as_single_break = false;
		}

		if ( entries.size() >= 21 ) {
			repack_epitope_residues_str = entries[ 21 ];
			track.repack_epitope_residues = ( repack_epitope_residues_str.c_str()[0] == 't' || repack_epitope_residues_str.c_str()[0] == 'T' );
		} else {
			track.repack_epitope_residues = false;
		}

	} else {

		// no residue info, for safety we set total number of flanking residues to 3
		// user can specify otherwise in the external file
		track.scaffold_tether_residues = Flanking( 3, 3 );
		track.scaffold_linker_residues = Flanking( 0, 0 );
		track.scaffold_moveable_residues = Flanking( 0, 0 );
		track.epitope_tether_residues = Flanking( 0, 0 );
		track.epitope_linker_residues = Flanking( 0, 0 );
		track.epitope_moveable_residues = Flanking( 0, 0 );
		track.perturb_rb = false;
		track.move_all_epitope_residues = false;
		track.close_as_single_break = false;
		track.repack_epitope_residues = false;

	}

	return true;
}


/// @brief return status
String
GraftInfo::to_string(
	String const & scaffold_filename,
	String const & line_prefix,
	bool const & print_old_format
) const
{
	if ( print_old_format ) {
		return to_string_old_format( scaffold_filename, line_prefix );
	}

	std::ostringstream ss;

	for ( std::map< ResidueRange, GraftTrack >::const_iterator g = graft_tracking_.begin(), ge = graft_tracking_.end(); g != ge; ++g ) {
		ResidueRange const & scaffold_gap = g->first;
		GraftTrack const & track = g->second;

		ss << line_prefix << " " << scaffold_filename
		   << "   " << scaffold_gap.begin() << "   " << scaffold_gap.end();

		if ( track.do_graft_bb ) {
			ss << "   " << "graft_bb";
		} else {
			ss << "   " << "keep_bb ";
		}

		if ( track.do_graft_sc ) {
			ss << "   " << "graft_sc";
		} else {
			ss << "   " << "keep_sc ";
		}

		if ( track.ssbm_string.length() > 0 ) {
			ss << "   " << SSBM_START << "   " << track.ssbm_string;
		}

		if ( track.aai_string.length() > 0 ) {
			ss << "   " << AAI_START << "   " << track.aai_string;
		}

		ss << '\n';
	}

	return ss.str();
}


/// @brief return string for a given epitope component specified by scaffold gap range
String
GraftInfo::to_string(
	ResidueRange const & scaffold_gap_range,
	bool const & print_old_format
) const
{
	if ( print_old_format ) {
		return to_string_old_format( scaffold_gap_range );
	}

	GraftTrack const & track = graft_tracking_.find( scaffold_gap_range )->second;

	std::ostringstream ss;

	if ( track.do_graft_bb ) {
		ss << "   " << "graft_bb";
	} else {
		ss << "   " << "keep_bb";
	}

	if ( track.do_graft_sc ) {
		ss << "   " << "graft_sc";
	} else {
		ss << "   " << "keep_sc";
	}

	if ( track.ssbm_string.length() > 0 ) {
		ss << "   " << SSBM_START << "   " << track.ssbm_string;
	}

	if ( track.aai_string.length() > 0 ) {
		ss << "   " << AAI_START << "   " << track.aai_string;
	}

	return ss.str();
}


/// @brief return string for a given epitope component specified by MatchComponent
String
GraftInfo::to_string(
	MatchComponent const & component,
	bool const & print_old_format
) const
{
	return to_string( component.scaffold_gap_range, print_old_format );
}


/// @brief return status
String
GraftInfo::to_string_old_format(
	String const & scaffold_filename,
	String const & line_prefix
) const
{
	std::ostringstream ss;

	for ( std::map< ResidueRange, GraftTrack >::const_iterator g = graft_tracking_.begin(), ge = graft_tracking_.end(); g != ge; ++g ) {
		ResidueRange const & scaffold_gap = g->first;
		GraftTrack const & track = g->second;

		ss << line_prefix << " " << scaffold_filename
		   << "   " << scaffold_gap.begin() << "   " << scaffold_gap.end();

		if ( track.do_graft_bb ) {
			ss << "   " << "true";
		} else {
			ss << "   " << "false";
		}

		if ( track.do_graft_sc ) {
			ss << "   " << "true";
		} else {
			ss << "   " << "false";
		}

		// temporaries for clarity
		Integer const & s_tether_n = track.scaffold_tether_residues.first;
		Integer const & s_linker_n = track.scaffold_linker_residues.first;
		Integer const & s_move_n = track.scaffold_moveable_residues.first;
		Integer const & s_tether_c = track.scaffold_tether_residues.second;
		Integer const & s_linker_c = track.scaffold_linker_residues.second;
		Integer const & s_move_c = track.scaffold_moveable_residues.second;

		Integer const & e_tether_n = track.epitope_tether_residues.first;
		Integer const & e_linker_n = track.epitope_linker_residues.first;
		Integer const & e_move_n = track.epitope_moveable_residues.first;
		Integer const & e_tether_c = track.epitope_tether_residues.second;
		Integer const & e_linker_c = track.epitope_linker_residues.second;
		Integer const & e_move_c = track.epitope_moveable_residues.second;

		ss << "   " << s_tether_n << "   " << s_linker_n << "   " << s_move_n
		   << "  |  " << s_tether_c << "   " << s_linker_c << "   " << s_move_c
		   << "  |  " << e_tether_n << "   " << e_linker_n << "   " << e_move_n
		   << "  |  " << e_tether_c << "   " << e_linker_c << "   " << e_move_c;

		if ( track.perturb_rb ) {
			ss << "   " << "true";
		} else {
			ss << "   " << "false";
		}

		if ( track.move_all_epitope_residues ) {
			ss << "   " << "true";
		} else {
			ss << "   " << "false";
		}

		if ( track.close_as_single_break ) {
			ss << "   " << "true";
		} else {
			ss << "   " << "false";
		}

		if ( track.repack_epitope_residues ) {
			ss << "   " << "true";
		} else {
			ss << "   " << "false";
		}

		ss << '\n';
	}

	return ss.str();
}


/// @brief return string for a given epitope component specified by scaffold gap range
String
GraftInfo::to_string_old_format(
	ResidueRange const & scaffold_gap_range
) const
{
	GraftTrack const & track = graft_tracking_.find( scaffold_gap_range )->second;

	std::ostringstream ss;

	if ( track.do_graft_bb ) {
		ss << "   " << "true";
	} else {
		ss << "   " << "false";
	}

	if ( track.do_graft_sc ) {
		ss << "   " << "true";
	} else {
		ss << "   " << "false";
	}

	// temporaries for clarity
	Integer const & s_tether_n = track.scaffold_tether_residues.first;
	Integer const & s_linker_n = track.scaffold_linker_residues.first;
	Integer const & s_move_n = track.scaffold_moveable_residues.first;
	Integer const & s_tether_c = track.scaffold_tether_residues.second;
	Integer const & s_linker_c = track.scaffold_linker_residues.second;
	Integer const & s_move_c = track.scaffold_moveable_residues.second;

	Integer const & e_tether_n = track.epitope_tether_residues.first;
	Integer const & e_linker_n = track.epitope_linker_residues.first;
	Integer const & e_move_n = track.epitope_moveable_residues.first;
	Integer const & e_tether_c = track.epitope_tether_residues.second;
	Integer const & e_linker_c = track.epitope_linker_residues.second;
	Integer const & e_move_c = track.epitope_moveable_residues.second;

	ss << "   " << s_tether_n << "   " << s_linker_n << "   " << s_move_n
	   << "  |  " << s_tether_c << "   " << s_linker_c << "   " << s_move_c
	   << "  |  " << e_tether_n << "   " << e_linker_n << "   " << e_move_n
	   << "  |  " << e_tether_c << "   " << e_linker_c << "   " << e_move_c;

	if ( track.perturb_rb ) {
		ss << "   " << "true";
	} else {
		ss << "   " << "false";
	}

	if ( track.move_all_epitope_residues ) {
		ss << "   " << "true";
	} else {
		ss << "   " << "false";
	}

	if ( track.close_as_single_break ) {
		ss << "   " << "true";
	} else {
		ss << "   " << "false";
	}

	if ( track.repack_epitope_residues ) {
		ss << "   " << "true";
	} else {
		ss << "   " << "false";
	}

	return ss.str();
}


/// @brief return string for a given epitope component specified by MatchComponent
String
GraftInfo::to_string_old_format(
	MatchComponent const & component
) const
{
	return to_string_old_format( component.scaffold_gap_range );
}


/// @brief   count number of needed tether residues in string
/// @details this count "includes" deleted residues, since it is neccessary
/// @details to have a tether residue in order for a deletion to occur
Size
GraftInfo::count_tether(
	String const & s
) const
{
	Size count = std::count( s.begin(), s.end(), USE_HELIX );
	count += std::count( s.begin(), s.end(), USE_SHEET );
	count += std::count( s.begin(), s.end(), USE_LOOP );
	count += std::count( s.begin(), s.end(), USE_ANY );
	count += std::count( s.begin(), s.end(), KEEP_AND_MOVE );
	count += std::count( s.begin(), s.end(), FREEZE );

	// deleted residues also need to be added into the count, since
	// a tether residue is required for deletion
	count += std::count( s.begin(), s.end(), DELETE );

	return count;
}


/// @brief count number of linker residues in string
/// @return # of linker residues, can be negative if doing deletions
Integer
GraftInfo::count_linker(
	String const & s
) const
{
	// first see if there are any deleted residues
	Integer deleted = std::count( s.begin(), s.end(), DELETE );

	// next see if there are any grown residues
	Integer grown = std::count( s.begin(), s.end(), GROW_HELIX );
	grown += std::count( s.begin(), s.end(), GROW_SHEET );
	grown += std::count( s.begin(), s.end(), GROW_LOOP );
	grown += std::count( s.begin(), s.end(), GROW_ANY );
	grown += std::count( s.begin(), s.end(), GROW_FROZEN );

	if ( deleted && grown ) {
		utility::exit( __FILE__, __LINE__, "ERROR: cannot have both grown and deleted residues in linker!" );
	}

	Integer count = grown ? grown : -deleted;

	return count;
}


/// @brief count number of moveable residues in string
Size
GraftInfo::count_moveable(
	String const & s
) const
{
	return s.length() - count_frozen( s ) - std::count( s.begin(), s.end(), DELETE );
}


/// @brief count number if fixed residues in string
Size
GraftInfo::count_frozen(
	String const & s
) const
{
	return std::count( s.begin(), s.end(), FREEZE );
}


/// @brief find n-term tether/linker portion of epitope string
String
GraftInfo::left_nt_epitope(
	String const & s,
	bool const & include_tether
) const
{
	String ci = linker_characters(); // valid character instructions
	if ( include_tether ) {
		ci += epitope_tether_characters();
	}

	bool const has_tl = ( s.find_first_of( ci ) == 0 ); // actually has a tether/linker?

	std::ostringstream epi;

	if ( has_tl ) {
		for ( String::const_iterator i = s.begin(), ie = s.end(); i != ie; ++i ) {
			if ( ci.find( *i ) != String::npos ) {
				epi << (*i);
			} else {
				break;
			}
		}
	}

	return epi.str();
}


/// @brief find c-term tether/linker portion of epitope string
String
GraftInfo::right_ct_epitope(
	String const & s,
	bool const & include_tether
) const
{
	String ci = linker_characters(); // valid character instructions
	if ( include_tether ) {
		ci += epitope_tether_characters();
	}

	bool const has_tl = ( s.find_last_of( ci ) == ( s.length() - 1 ) ); // actually has a tether/linker?

	std::ostringstream epi;
	std::vector< char > v;

	if ( has_tl ) {
		for ( String::const_reverse_iterator i = s.rbegin(), ie = s.rend(); i != ie; ++i ) {
			if ( ci.find( *i ) != String::npos ) {
				v.push_back( *i );
			} else {
				break;
			}
		}

		std::reverse( v.begin(), v.end() ); // need to reverse the sequence to retain N->C order

		for ( std::vector< char >::const_iterator i = v.begin(), ie = v.end(); i != ie; ++i ) {
			epi << (*i);
		}
	}

	return epi.str();
}


/// @brief find non-moveable middle (core) of epitope sans tether/linker
String
GraftInfo::middle_core_epitope(
	String const & s,
	bool const & remove_tether
) const
{
	// we could pass in 'remove_tether' directly as the boolean, but it gets confusing
	// when reading the code off
	String left = remove_tether ? left_nt_epitope( s, true ) : left_nt_epitope( s, false );
	String right = remove_tether ? right_ct_epitope( s, true ) : right_ct_epitope( s, false );

	return s.substr( left.length(), s.length() - left.length() - right.length() );
}


/// @brief parse ssbm and store -- used inside add_encoded_info_from_line()
void
GraftInfo::parse_and_store_ssbm(
	ResidueRange const & scaffold_gap,
	String const & ssbm,
	String line
)
{
	if ( ssbm.length() == 0 ) {
		return;
	}

	if ( line == "" ) {
		line = ssbm;
	}

	// cache graft tracking
	GraftTrack & track = graft_tracking_[ scaffold_gap ];

	// check to see if ssbm contains any invalid characters
	if ( ssbm.find_first_not_of( valid_ssbm_characters() ) != String::npos ) {
		std::ostringstream oss;
		oss << "GraftInfo ERROR: for scaffold gap range " << scaffold_gap.to_string() << ' ';
		oss << " invalid character found in ssbm section.";
		oss << "       affected line is: " << line << '\n';
		utility::exit( __FILE__, __LINE__, oss.str() );
	}

	// error check: certain combinations of closed/open breaks are currently disallowed
	//   - any number of closed breaks and more than one open break
	//   - greater than two closed breaks
	//   - any number of open breaks w/ rb other than two
	//   - greater than two open breaks
	Size n_closed_break = std::count( ssbm.begin(), ssbm.end(), CLOSED_BREAK );
	Size n_open_break_rb = std::count( ssbm.begin(), ssbm.end(), OPEN_BREAK_RB );
	Size n_open_break = std::count( ssbm.begin(), ssbm.end(), OPEN_BREAK );
	if ( n_closed_break > 0 && n_open_break > 1 ) {
		std::ostringstream oss;
		oss << "GraftInfo ERROR: for scaffold gap range " << scaffold_gap.to_string() << ' ';
		oss << " cannot have more than one open break when using closed breaks";
		oss << "       affected line is: " << line << '\n';
		utility::exit( __FILE__, __LINE__, oss.str() );
	}
	if ( n_closed_break > 2 ) {
		std::ostringstream oss;
		oss << "GraftInfo ERROR: for scaffold gap range " << scaffold_gap.to_string() << ' ';
		oss << " cannot have more than two closed breaks";
		oss << "       affected line is: " << line << '\n';
		utility::exit( __FILE__, __LINE__, oss.str() );
	}
	if ( n_open_break_rb > 0 && n_open_break_rb != 2 ) { // both booleans needed here
		std::ostringstream oss;
		oss << "GraftInfo ERROR: for scaffold gap range " << scaffold_gap.to_string() << ' ';
		oss << " when indicating open break using rb, both sides must be marked";
		oss << "       affected line is: " << line << '\n';
		utility::exit( __FILE__, __LINE__, oss.str() );
	}
	if ( n_open_break > 2 ) {
		std::ostringstream oss;
		oss << "GraftInfo ERROR: for scaffold gap range " << scaffold_gap.to_string() << ' ';
		oss << " cannot have more than two open breaks";
		oss << "       affected line is: " << line << '\n';
		utility::exit( __FILE__, __LINE__, oss.str() );
	}

	// compile the indexes of left & right 'break' characters
	Size left_break_index = ssbm.find_first_of( break_characters() );
	Size right_break_index = ssbm.find_last_of( break_characters() );

	// error check: necessary breakpoint not found
	if ( left_break_index == String::npos || right_break_index == String::npos ) {
		std::ostringstream oss;
		oss << "GraftInfo ERROR: for scaffold gap range " << scaffold_gap.to_string() << ' ';
		oss << " missing a breakpoint in ssbm" << '\n';
		oss << "       affected line is: " << line << '\n';
		utility::exit( __FILE__, __LINE__, oss.str() );
	}

	// error check: indexing error
	if ( left_break_index == right_break_index ) {
		std::ostringstream oss;
		oss << "GraftInfo ERROR: for scaffold gap range " << scaffold_gap.to_string() << ' ';
		oss << " left and right breakpoints were found to have same index" << '\n';
		oss << "       This may be an internal code error, please file a bug report.\n";
		oss << "       affected line is: " << line << '\n';
		utility::exit( __FILE__, __LINE__, oss.str() );
	}

	// get breakpoint index of 'middle' break, which only exists for the
	// close_as_single_break case, where there are two closed breaks and
	// one open break
	bool const is_closing_as_single_break = ( n_closed_break == 2 ) && ( n_open_break == 1 );
//	Size const middle_break_index = is_closing_as_single_break ? ssbm.find( OPEN_BREAK ) : String::npos;

	// parse scaffold-side secondary structure, break, and moveable instructions
	String const s_n_ssbm = ssbm.substr( 0, left_break_index );
	String const s_c_ssbm = ssbm.substr( right_break_index + 1, ssbm.length() - right_break_index - 1 );
	track.scaffold_tether_residues = Flanking( count_tether( s_n_ssbm ), count_tether( s_c_ssbm ) );
	track.scaffold_linker_residues = Flanking( count_linker( s_n_ssbm ), count_linker( s_c_ssbm ) );
	track.scaffold_moveable_residues = Flanking( count_moveable( s_n_ssbm ), count_moveable( s_c_ssbm ) );

	// parse epitope-side secondary structure, break, and moveable residues
	String e_ssbm = ssbm.substr( left_break_index + 1, right_break_index - left_break_index - 1 );
	String e_n_ssbm = left_nt_epitope( e_ssbm );
	String e_c_ssbm = right_ct_epitope( e_ssbm );
	String e_core_ssbm = middle_core_epitope( e_ssbm );
	e_core_ssbm = collapsed_character( e_core_ssbm, OPEN_BREAK ); // removes any open break character in case of closing with single break

	if ( e_core_ssbm.length() == 0 ) { // special case, moveable will be overridden by 'move_all_epitope_residues'
		e_n_ssbm = left_nt_epitope( e_ssbm, false ); // boolean: include tether
		e_c_ssbm = right_ct_epitope( e_ssbm, false ); // boolean: include tether
		e_core_ssbm = middle_core_epitope( e_ssbm, false ); // boolean: remove tether
		e_core_ssbm = collapsed_character( e_core_ssbm, OPEN_BREAK ); // removes any open break character in case of closing with single break
		track.epitope_tether_residues = Flanking( 0, 0 );
		track.move_all_epitope_residues = true;
	} else {
		track.epitope_tether_residues = Flanking( count_tether( e_n_ssbm ), count_tether( e_c_ssbm ) );
		track.move_all_epitope_residues = false;
	}

	track.epitope_linker_residues = Flanking( count_linker( e_n_ssbm ), count_linker( e_c_ssbm ) );
	track.epitope_moveable_residues = Flanking( count_moveable( e_n_ssbm), count_moveable( e_c_ssbm ) );

	if ( track.epitope_linker_residues.first < 0 || track.epitope_linker_residues.second < 0 ) {
		std::ostringstream oss;
		oss << "GraftInfo ERROR: for scaffold gap range " << scaffold_gap.to_string() << ' ';
		oss << " deletions are not allowed for epitope linker residues" << '\n';
		oss << "       affected line is: " << line << '\n';
		utility::exit( __FILE__, __LINE__, oss.str() );
	}

	// check to make sure the number of residues indicated in the tether+core don't
	// exceed the number actually in the epitope
	expand_core_ssbm( scaffold_gap, e_core_ssbm ); // expand any shortcuts, depends on tether settings in "track"
	Integer n_original_epitope_instructions = track.epitope_tether_residues.first +
	                                          track.epitope_tether_residues.second +
	                                          e_core_ssbm.length();

	// the next two statements will not active in current scheme,
	// we leave them here for future modifications when epitope linkers are allowed to be negative
	if ( track.epitope_linker_residues.first < 0 ) {
		n_original_epitope_instructions += ( -track.epitope_linker_residues.first );
	}
	if ( track.epitope_linker_residues.second < 0 ) {
		n_original_epitope_instructions += ( -track.epitope_linker_residues.second );
	}

	if ( n_original_epitope_instructions != native_epitope_subrange( scaffold_gap ).length() ) {
		std::ostringstream oss;
		oss << "GraftInfo ERROR: for scaffold gap range " << scaffold_gap.to_string() << ' ' << '\n';
		oss << " total number of secondary structure/break/moveable instructions for deleted, tether, and fixed epitope residues incorrect" << '\n';
		oss << " found " << n_original_epitope_instructions << " expected " << native_epitope_subrange( scaffold_gap ).length() << '\n';
		oss << "       affected line is: " << line << '\n';
		utility::exit( __FILE__, __LINE__, oss.str() );
	}

	// handle special cases
	track.left_side_closed = ( ssbm.at( left_break_index ) == CLOSED_BREAK );
	track.right_side_closed = ( ssbm.at( right_break_index ) == CLOSED_BREAK );
	track.perturb_rb = ( n_open_break_rb == 2 );
	track.close_as_single_break = is_closing_as_single_break;

	// store reconstructed string
	expand_core_ssbm( scaffold_gap, e_ssbm ); // attempt any shortcut expansion in ssbm's core
	std::ostringstream expanded_ssbm;
	expanded_ssbm << s_n_ssbm;
	expanded_ssbm << ssbm.at( left_break_index ) << e_ssbm << ssbm.at( right_break_index );
	expanded_ssbm << s_c_ssbm;
	track.ssbm_string = expanded_ssbm.str();
}


/// @brief parse aai and store -- used inside add_encoded_info_from_line()
/// @param[in] line  original line; optional, just for error reporting
/// @warning routine depends on ssbm settings, so ssbm must be correct prior to
///          calling this
void
GraftInfo::parse_and_store_aai(
	ResidueRange const & scaffold_gap,
	String const & aai,
	String line
)
{
	if ( aai.length() == 0 ) {
		return;
	}

	if ( line == "" ) {
		line = aai;
	}

	// cache graft tracking
	GraftTrack & track = graft_tracking_[ scaffold_gap ];

	// check to see if aai contains any invalid characters
	if ( aai.find_first_not_of( valid_aai_characters() ) != String::npos ) {
		std::ostringstream oss;
		oss << "GraftInfo ERROR: for scaffold gap range " << scaffold_gap.to_string() << ' ';
		oss << " invalid character found in aai section.";
		oss << "       affected line is: " << line << '\n';
		utility::exit( __FILE__, __LINE__, oss.str() );
	}

	// compile the indexes of left & right 'break' characters
	Size left_break_index = aai.find_first_of( break_characters() );
	Size right_break_index = aai.find_last_of( break_characters() );

	// error check: necessary breakpoint not found
	if ( left_break_index == String::npos || right_break_index == String::npos ) {
		std::ostringstream oss;
		oss << "GraftInfo ERROR: for scaffold gap range " << scaffold_gap.to_string() << ' ';
		oss << " missing a breakpoint in aai instructions" << '\n';
		oss << "       affected line is: " << line << '\n';
		utility::exit( __FILE__, __LINE__, oss.str() );
	}

	// error check: indexing error
	if ( left_break_index == right_break_index ) {
		std::ostringstream oss;
		oss << "GraftInfo ERROR: for scaffold gap range " << scaffold_gap.to_string() << ' ';
		oss << " left and right breakpoints were found to have same index" << '\n';
		oss << "       This may be an internal code error, please file a bug report.\n";
		oss << "       affected line is: " << line << '\n';
		utility::exit( __FILE__, __LINE__, oss.str() );
	}

	// split string into parts
	String s_left_aai = aai.substr( 0, left_break_index );
	String s_right_aai = aai.substr( right_break_index + 1, aai.length() - right_break_index - 1 );
	String s_middle_aai = aai.substr( left_break_index + 1, right_break_index - left_break_index - 1);

	// check lengths, expand and fill in parameters if necessary
	// handle left
	Size const ll = track.scaffold_tether_residues.first + track.scaffold_linker_residues.first;
	String const stripped_s_left_aai = ObjexxFCL::rstripped( s_left_aai, String( 1, DELETE ) );
	if ( stripped_s_left_aai.length() < ll ) {
		s_left_aai = String( ll - stripped_s_left_aai.length(), DEFAULT ) + s_left_aai;
	} else if ( stripped_s_left_aai.length() > ll ) {
		std::ostringstream oss;
		oss << "GraftInfo ERROR: for scaffold gap range " << scaffold_gap.to_string();
		oss << " the length of instruction string for aai of scaffold n-side is too long!" << '\n';
		oss << " found = " << s_left_aai.length() << " expected = " << ll << '\n';
		oss << "       affected line is: " << line << '\n';
		utility::exit( __FILE__, __LINE__, oss.str() );
	}

	// handle right
	Size const rl = track.scaffold_tether_residues.second + track.scaffold_linker_residues.second;
	String const stripped_s_right_aai = ObjexxFCL::lstripped( s_right_aai, String( 1, DELETE ) );
	if ( stripped_s_right_aai.length() < rl ) {
		s_right_aai = s_right_aai + String( rl - stripped_s_right_aai.length(), DEFAULT );
	} else if ( stripped_s_right_aai.length() > rl ) {
		std::ostringstream oss;
		oss << "GraftInfo ERROR: for scaffold gap range " << scaffold_gap.to_string();
		oss << " the length of instruction string for aai of scaffold c-side is too long!" << '\n';
		oss << " found = " << s_right_aai.length() << " expected = " << rl << '\n';
		oss << "       affected line is: " << line << '\n';
		utility::exit( __FILE__, __LINE__, oss.str() );
	}

	// handle middle
	Size const middle_length = native_epitope_subrange( scaffold_gap ).length()
	                                + track.epitope_linker_residues.first + track.epitope_linker_residues.second;

	// single '.' alone is a special case that will expand to repack the existing epitope
	//   and allow default behavior for the linker between breaks
	// single '*' alone is a special case that will expand to freeze the existing epitope
	//   and allow default behavior for the linker between breaks
	if ( s_middle_aai == String( 1, REPACK ) ) {
		s_middle_aai = String( track.epitope_linker_residues.first, DEFAULT )
		                 + String( native_epitope_subrange( scaffold_gap ).length(), REPACK )
		                 + String( track.epitope_linker_residues.second, DEFAULT );
	} else if ( s_middle_aai == String( 1, FREEZE ) ) {
		s_middle_aai = String( track.epitope_linker_residues.first, DEFAULT )
		                 + String( native_epitope_subrange( scaffold_gap ).length(), FREEZE )
		                 + String( track.epitope_linker_residues.second, DEFAULT );
	}

	 // to track case where user put in an open break for keeping track of closing as single break
	String collapsed_s_middle_aai = collapsed_character( s_middle_aai, OPEN_BREAK );

	// single case '?' embedded within the string is a special case that will expand to
	// fill all necessary middle space
	if ( std::count( s_middle_aai.begin(), s_middle_aai.end(), DEFAULT ) >= 1 ) {
		s_middle_aai.replace( s_middle_aai.find( DEFAULT ), 1, String( middle_length - collapsed_s_middle_aai.length() + 1, DEFAULT ) );
	}

	// reset collapsed string using newly expanded string
	collapsed_s_middle_aai = collapsed_character( s_middle_aai, OPEN_BREAK );

	if ( collapsed_s_middle_aai.length() != middle_length ) {
		std::ostringstream oss;
		oss << "GraftInfo ERROR: for scaffold gap range " << scaffold_gap.to_string();
		oss << " the length of instruction string for aai of epitope is incorrect!" << '\n';
		oss << " found = " << collapsed_s_middle_aai.length() << " expected = " << middle_length << '\n';
		oss << "       affected line is: " << line << '\n';
		utility::exit( __FILE__, __LINE__, oss.str() );
	}

	// store reconstructed string
	std::ostringstream expanded_aai;
	expanded_aai << s_left_aai;
	expanded_aai << aai.at( left_break_index ) << s_middle_aai << aai.at( right_break_index );
	expanded_aai << s_right_aai;
	track.aai_string = expanded_aai.str();
}


/// @brief check to see if any epitope components are marked to move all epitope residues
/// @brief and change the tether/moveable accordingly
/// @note N2C/C2N or other primary component single breaks are skipped
void
GraftInfo::self_modify_if_moving_all_epitope_residues()
{
	for ( Integer i = 2, ie = match_result_.components.size(); i <= ie; ++i ) { // note that primary component is SKIPPED
		MatchComponent const & component = match_result_.components[ i ];

		// find corresponding GraftTrack entry
		GraftTrack & track = graft_tracking_.find( component.scaffold_gap_range )->second;

		if ( track.move_all_epitope_residues ) {

			// The following loop indexing is correct assuming this is a double break closure, since
			// the jump residues lives in the relative middle of the loop and you don't want it to be
			// moveable.  HOWEVER, if the secondary component is being switched from double break
			// to single break, the jump residue will need to be added to the list of moveable residues,
			// and is what is done in EpitopeScaffold::switch_secondaries_to_single_break().
			Integer const relative_middle_of_loop = relative_middle( component.native_loop_subrange );
			ResidueRange const left( 1, relative_middle_of_loop - 1 );
			ResidueRange const right( relative_middle_of_loop + 1, component.native_loop_subrange.length() );

			track.epitope_tether_residues = Flanking( left.length(), right.length() );
			track.epitope_moveable_residues = Flanking( track.epitope_tether_residues.first + track.epitope_linker_residues.first,
			                                            track.epitope_tether_residues.second + track.epitope_linker_residues.second );
		} // if track.move_all_epitope_residues
	}

}


/// @brief check to see if there is transfer consistency (e.g. ranges are same if not
/// @brief grafting bb
bool
GraftInfo::check_transfer_consistency() const
{
	bool check_ok = true;

	for ( utility::vector1< MatchComponent >::const_iterator c = match_result_.components.begin(), ce = match_result_.components.end(); c != ce; ++c ) {
		ResidueRange const & loop_range = c->loop_subrange;
		ResidueRange const & gap = c->scaffold_gap_range;

		if ( ( !do_graft_bb( gap ) ) && do_graft_sc( gap ) && ( gap.length() != loop_range.length() ) ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: at native scaffold range " << gap.to_string()
			   << " -- graft of sidechain while keeping native scaffold backbone was requested, however the lengths of the scaffold and epitope ranges do not match!" << std::endl;
			std::cerr << ss.str() << std::endl;
			check_ok = false;
		}
	}

	return check_ok;
}


/// @brief check to see if tether/linker/moveable are consistent with each other
/// @details tether >= 0
/// @details tether + linker >= 0
/// @details tether + linker >= moveable
/// @details epitope_tether_nterm < 1 + (length of epitope loop)/2
/// @details epitope_tether_cterm < (length of epitope loop)/2
/// @details epitope_linker_*term >= 0
bool
GraftInfo::check_flanking_consistency() const
{
	std::set< ResidueRange> scaffold_gaps = this->scaffold_gaps();

	// initialize
	bool check_ok = true;

	// first run through and see if all tether/linker/moveable combinations are reasonable
	for ( std::set< ResidueRange >::const_iterator g = scaffold_gaps.begin(), ge = scaffold_gaps.end(); g != ge; ++g ) {
		ResidueRange const & gap = *g;

		// scaffold side first
		Integer const & s_tether_n = scaffold_tether_residues_nterm( gap );
		Integer const & s_linker_n = scaffold_linker_residues_nterm( gap );
		Integer const & s_moveable_n = scaffold_moveable_residues_nterm( gap );

		Integer const & s_tether_c = scaffold_tether_residues_cterm( gap );
		Integer const & s_linker_c = scaffold_linker_residues_cterm( gap );
		Integer const & s_moveable_c = scaffold_moveable_residues_cterm( gap );

		if ( s_tether_n < 0 ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: scaffold tether_n of " << gap.to_string() << " not >= 0" << std::endl;
			std::cerr << ss.str() << std::endl;
			check_ok = false;
		}

		if ( s_tether_c < 0 ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: scaffold tether_c of " << gap.to_string() << " not >= 0" << std::endl;
			std::cerr << ss.str() << std::endl;
			check_ok = false;
		}

		if ( s_tether_n + s_linker_n < 0 ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: scaffold tether_n + linker_n of " << gap.to_string() << " not >= 0" << std::endl;
			std::cerr << ss.str() << std::endl;
			check_ok = false;
		}

		if ( s_tether_c + s_linker_c < 0 ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: scaffold tether_c + linker_c of " << gap.to_string() << " not >= 0" << std::endl;
			std::cerr << ss.str() << std::endl;
			check_ok = false;
		}

		Integer smn = s_linker_n < 0 ? s_tether_n : s_tether_n + s_linker_n;
		if ( smn < s_moveable_n ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: scaffold tether_n + linker_n of " << gap.to_string() << " not >= moveable_n" << std::endl;
			std::cerr << ss.str() << std::endl;
			check_ok = false;
		}

		Integer smc = s_linker_c < 0 ? s_tether_c : s_tether_c + s_linker_c;
		if ( smc < s_moveable_c ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: scaffold tether_c + linker_c of " << gap.to_string() << " not >= moveable_c" << std::endl;
			std::cerr << ss.str() << std::endl;
			check_ok = false;
		}

		// epitope side next
		Integer const & e_tether_n = epitope_tether_residues_nterm( gap );
		Integer const & e_linker_n = epitope_linker_residues_nterm( gap );
		Integer const & e_moveable_n = epitope_moveable_residues_nterm( gap );

		Integer const & e_tether_c = epitope_tether_residues_cterm( gap );
		Integer const & e_linker_c = epitope_linker_residues_cterm( gap );
		Integer const & e_moveable_c = epitope_moveable_residues_cterm( gap );

		if ( e_tether_n < 0 ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: epitope tether_n of " << gap.to_string() << " not >= 0" << std::endl;
			std::cerr << ss.str() << std::endl;
			check_ok = false;
		}

		if ( e_tether_c < 0 ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: epitope tether_c of " << gap.to_string() << " not >= 0" << std::endl;
			std::cerr << ss.str() << std::endl;
			check_ok = false;
		}

		if ( e_tether_n + e_linker_n < 0 ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: epitope tether_n + linker_n of " << gap.to_string() << " not >= 0" << std::endl;
			std::cerr << ss.str() << std::endl;
			check_ok = false;
		}

		if ( e_tether_c + e_linker_c < 0 ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: epitope tether_c + linker_c of " << gap.to_string() << " not >= 0" << std::endl;
			std::cerr << ss.str() << std::endl;
			check_ok = false;
		}

		if ( e_tether_n + e_linker_n < e_moveable_n ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: epitope tether_n + linker_n of " << gap.to_string() << " not >= moveable_n" << std::endl;
			std::cerr << ss.str() << std::endl;
			check_ok = false;
		}

		if ( e_tether_c + e_linker_c < e_moveable_c ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: epitope tether_c + linker_c of " << gap.to_string() << " not >= moveable_c" << std::endl;
			std::cerr << ss.str() << std::endl;
			check_ok = false;
		}

		if ( e_linker_n < 0 ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: epitope linker_n of " << gap.to_string() << " not >= 0" << std::endl;
			std::cerr << ss.str() << std::endl;
			check_ok = false;
		}

		if ( e_linker_c < 0 ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: epitope linker_c of " << gap.to_string() << " not >= 0" << std::endl;
			std::cerr << ss.str() << std::endl;
			check_ok = false;
		}
	}

	return check_ok;
}


/// @brief check to see if there's enough room to graft given the
/// @brief scaffold gap ranges and the number of linker/moveable flanking residues
bool
GraftInfo::check_scaffold_gaps_vs_flanking_residues() const
{
	std::set< ResidueRange> scaffold_gaps = this->scaffold_gaps();

	std::set< ResidueRange >::const_iterator prior = scaffold_gaps.begin();
	std::set< ResidueRange >::const_iterator current = scaffold_gaps.begin();
	++current; // advance iterator forward by one

	bool check_ok = true;
	while ( current != scaffold_gaps.end() ) {
		ResidueRange const & prior_gap = *prior;
		ResidueRange const & current_gap = *current;

		Integer const prior_linker = -scaffold_linker_residues_cterm( prior_gap ); // reversal for c-term
		Integer const current_linker = scaffold_linker_residues_nterm( current_gap );

		// note: cannot use ResidueRange here because it's possible start > stop due to human input error
		Integer const between_start = prior_gap.end() + 1 + prior_linker;
		Integer const between_stop = current_gap.begin() - 1 + current_linker;
		Integer const between_length = between_stop - between_start + 1;

		// check against total flanking, length of section between
		// gaps must be >= 1 + prior_flank_cterm + current_flank_nterm
		// the plus '1' is due to the need for one anchor residue to jump to/from
		Integer const prior_flank_cterm = scaffold_tether_residues_cterm( prior_gap );
		Integer const current_flank_nterm = scaffold_tether_residues_nterm( current_gap );
		if ( between_length < 1 + prior_flank_cterm + current_flank_nterm ) { // failure
			std::ostringstream ss;
			ss << "GraftInfo ERROR: length of section between scaffold gaps " << prior_gap.to_string() << std::endl;
			ss << "   and " << current_gap.to_string() << " = " << between_length << " (including both total and" << std::endl;
			ss << "   " << "linker residues) and is too small; to satisfy the flanking residues the in-between" << std::endl;
			ss << "   " << "length must be >= " << ( 1 + prior_flank_cterm + current_flank_nterm ) << std::endl;
			ss << "   " << "because flanks are (prior C, current N): " << prior_flank_cterm << ", " << current_flank_nterm << ".";
			std::cerr << ss.str() << std::endl;
			check_ok = false;
		}

		++prior;
		++current;
	}

	return check_ok;
}


/// @brief check to see if there's enough room to graft given the
/// @brief epitope loop lengths and the number of epitope-side tether/linker flanking residues
/// @details epitope_tether_nterm < | [ 1, sup( (length of epitope loop)/2 ) - 1 ] |
/// @details epitope_tether_cterm < | [ sup( (length of epitope loop)/2 ) + 1, length of epitope loop ] |
bool
GraftInfo::check_epitope_loop_lengths_vs_flanking_residues() const
{
	bool check_ok = true;

	for ( utility::vector1< MatchComponent >::const_iterator c = match_result_.components.begin(), ce = match_result_.components.end(); c != ce; ++c ) {
		MatchComponent const & component = *c;

		// gather tether lengths
		Integer const & e_tether_n = epitope_tether_residues_nterm( component.scaffold_gap_range );
		Integer const & e_tether_c = epitope_tether_residues_cterm( component.scaffold_gap_range );

		// EpitopeScaffold uses middle of loop to compute fixed jump points for fold tree connections
		Integer const relative_middle_of_loop = relative_middle( component.native_loop_subrange );

		ResidueRange const left( 1, relative_middle_of_loop - 1 );
		if ( e_tether_n > left.length() ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: for scaffold gap " << component.scaffold_gap_range.to_string()
			   << " and (native) epitope component " << component.native_loop_subrange.to_string()
			   << ", epitope tether_n is too large and encompasses middle of the epitope loop."
			   << "  Please make sure 'epitope tether_n < | [ 1, sup( (length of epitope loop)/2 ) - 1 ] |'" << std::endl;
			std::cerr << ss.str() << std::endl;
			check_ok = false;
		}

		ResidueRange const right( relative_middle_of_loop + 1, component.native_loop_subrange.length() );
		if ( e_tether_c > right.length() ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: for scaffold gap " << component.scaffold_gap_range.to_string()
			   << " and (native) epitope component " << component.native_loop_subrange.to_string()
			   << ", epitope tether_c is too large and encompasses middle of the epitope loop."
			   << "  Please make sure 'epitope tether_c < | [ sup( (length of epitope loop)/2 ) + 1, length of epitope loop ] |'" << std::endl;
			std::cerr << ss.str() << std::endl;
			check_ok = false;
		}
	}

	return check_ok;
}

/// @brief additional consistency checks
bool
GraftInfo::check_misc_consistency() const
{
	bool check_ok = true;

	for ( std::map< ResidueRange, GraftTrack >::const_iterator i = graft_tracking_.begin(), ie = graft_tracking_.end(); i != ie; ++i ) {
		if ( i->second.perturb_rb && i->second.close_as_single_break ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: for scaffold gap " << i->first.to_string()
			   << " : perturb_rb and close_as_single_break cannot both be true!" << std::endl;
			std::cerr << ss.str() << std::endl;
			check_ok = false;
		}

		if ( !i->second.move_all_epitope_residues && i->second.close_as_single_break ) {
			std::ostringstream ss;
			ss << "GraftInfo ERROR: for scaffold gap " << i->first.to_string()
			   << " : move_all_epitope_residues must be true if close_as_single_break is true!" << std::endl;
			std::cerr << ss.str() << std::endl;
			check_ok = false;
		}
	}

	return check_ok;
}

} // namespace design
} // namespace epigraft
