// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   GraftInfo.hh
/// @brief  Holds information for grafting: match result & components, closure options.
/// @note   By default for closure, scaffold flanking residues are 3 on each side, and
/// @note   scaffold pliant residues are 0 on each side.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)


#ifndef INCLUDED_epigraft_design_GraftInfo_HH_
#define INCLUDED_epigraft_design_GraftInfo_HH_

// package headers
#include <epigraft/design/design_types.hh>
#include <epigraft/ResidueRange.hh>
#include <epigraft/match/MatchResult.hh>
#include <epigraft/match/MatchComponent.hh>

// utility headers
#include <utility/vector1.hh>

// C++ headers
#include <set>
#include <string>
#include <utility>

namespace epigraft
{
namespace design
{

/// @brief    Holds information for grafting: match result & components, closure options.
/// @details  <b> Information in this class is indexed by scaffold gap residue!</b>
/// @details  By default for closure, scaffold tether residues are 3 on each side,
/// @details  scaffold linker residues are 0 on each side, epitope tether residues are 0
/// @details  on each side, and epitope linker residues are 0 on each side.
/// @details  Restriction: 'tether_*term' >= 0
/// @details  Restriction: 'tether_*term' + 'linker_*term' >= 0
/// @details  Restriction: 'tether_*term' + 'linker_*term' >= 'moveable_*term'
/// @details  Restriction: epitope_tether_nterm < 1 + (length of epitope loop)/2
/// @details  Restriction: epitope_tether_cterm < (length of epitope loop)/2
/// @details  Restriction: epitope_linker_*term >= 0
class GraftInfo
{

	private: // typedefs

		// convenience typedefs
		typedef epigraft::match::MatchResult MatchResult;
		typedef epigraft::match::MatchComponent MatchComponent;

		/// @brief alias to track number of tether/linker/moveable residues on n-term/c-term flanks, '.first' == n-term, '.second' == c-term
		typedef std::pair< Integer, Integer > Flanking;


	public: // typedefs

		/// @brief set of match components sorted by lexigraphic ordering of scaffold gap range
		typedef std::set< MatchComponent > SortedMatchComponents;


	public: // character alphabet shared by both structure/move and amino acid identity instructions

		/// @brief no movement, keep current settings
		static char const FREEZE        = '*';


	public: // character alphabet for one-line-string encoding of structure/move instructions

		/// @brief indicates start of amino acid instructions
		static char const SSBM_START      = '@';

		/// @brief indicates open break (cutpoint)
		static char const OPEN_BREAK    = '|';

		/// @brief indicates closed break (cutpoint)
		static char const CLOSED_BREAK  = '=';

		/// @brief indicates open break (cutpoint) and allowance of rigid body move
		static char const OPEN_BREAK_RB = '^';

		/// @brief keep secondary structure but move backbone
		static char const KEEP_AND_MOVE = '.';

		/// @brief delete residue at that position
		static char const DELETE        = 'x';

		/// @brief grow a new residue as helix
		static char const GROW_HELIX    = 'h';

		/// @brief grow a new residue as sheet
		static char const GROW_SHEET    = 'e';

		/// @brief grow a new residue as loop
		static char const GROW_LOOP     = 'l';

		/// @brief grow a new residue with any secondary structure ("degenerate")
		static char const GROW_ANY      = 'd';

		/// @brief grow a new residue, but mark it as non-moveable; a special override for
		///        when "history" is a problem
		static char const GROW_FROZEN   = 'g';

		/// @brief use helix at existing residue position
		static char const USE_HELIX     = 'H';

		/// @brief use sheet at existing residue position
		static char const USE_SHEET     = 'E';

		/// @brief use loop at existing residue position
		static char const USE_LOOP      = 'L';

		/// @brief use any secondary structure ("degenerate") at existing residue position
		static char const USE_ANY       = 'D';


	public: // character alphabet for one-line-string encoding of amino acid instructions

		/// @brief indicates start of amino acid instructions
		static char const AAI_START = '&';

		/// @brief use protocol specific default behavior
		static char const DEFAULT   = '?';

		/// @brief indicates allow repack
		static char const REPACK    = '.';

		/// @brief any amino acid ("degenerate")
		static char const ANY_AA    = 'd';

		/// @brief all amino acids except cys and pro ("restricted/regular degenerate")
		static char const NO_CP     = 'r';

		/// @brief nonpolar residues
		static char const NONPOLAR  = 'n';

		/// @brief aromatic residues
		static char const AROMATIC  = 'a';

		/// @brief polar residues
		static char const POLAR     = 'p';

		/// @brief charged residues
		static char const CHARGED   = 'c';

		/// @brief positively charged residues
		static char const POSITIVE  = '+';

		/// @brief negatively charged residues
		static char const NEGATIVE  = '-';


	protected: // internal structs

		/// @brief manage various graft info, meant to be used in a map from original scaffold
		/// @brief gap range -> data
		struct GraftTrack
		{
				/// @brief scaffold_gap_range -> bool : graft the bb of a particular match component?
				bool do_graft_bb;

				/// @brief scaffold_gap_range -> bool : graft the sc of a particular match component?
				bool do_graft_sc;

				/// @brief scaffold_gap_range -> n-/c-term tether residues to left/right of component
				Flanking scaffold_tether_residues;

				/// @brief scaffold_gap_range -> n-/c-term linker residues to left/right of component
				Flanking scaffold_linker_residues;

				/// @brief scaffold_gap_range -> n-/c-term moveable residues to left/right of component
				Flanking scaffold_moveable_residues;

				/// @brief scaffold_gap_range -> n-/c-term tether residues on epitope loop
				Flanking epitope_tether_residues;

				/// @brief scaffold_gap_range -> n-/c-term linker residues on epitope loop
				Flanking epitope_linker_residues;

				/// @brief scaffold_gap_range -> n-/c-term moveable residues on epitope loop
				Flanking epitope_moveable_residues;

				/// @brief left-side cut artificially closed?
				bool left_side_closed;

				/// @brief right-side cut artificially closed?
				bool right_side_closed;

				/// @brief rigid body perturb this epitope component prior to build loop attempt?
				bool perturb_rb;

				/// @brief disregard epitope moveable and move ALL epitope residues?
				bool move_all_epitope_residues;

				/// @brief close as single break instead of double?
				bool close_as_single_break;

				/// @brief repack epitope residues for this component?
				bool repack_epitope_residues;

				/// @brief the secondary structure/break/moveable instruction string
				String ssbm_string;

				/// @brief the amino acid identity instruction string
				String aai_string;
		};


	public: // construct/destruct

		/// @brief default constructor
		inline GraftInfo()
		{}

		/// @brief constructor
		inline GraftInfo(MatchResult const & match_result) :
			match_result_(match_result ), locked_( false)
		{
			initialize();
		}

		/// @brief copy constructor
		inline GraftInfo(GraftInfo const & g) :
			match_result_(g.match_result_), graft_tracking_(g.graft_tracking_),
					locked_(g.locked_)
		{}

		/// @brief default destructor
		inline ~GraftInfo()
		{}


	public: // copy assignment

		/// @brief copy assignment
		inline
		GraftInfo &operator =(GraftInfo const & g)
		{
			if ( this != &g )
			{
				match_result_ = g.match_result_;
				graft_tracking_ = g.graft_tracking_;
				locked_ = g.locked_;
			}
			return *this;
		}

	public: // safety mechanism

		/// @brief locked?
		inline
		bool const &locked() const
		{
			return locked_;
		}

		/// @brief unlock for modification
		inline
		void unlock()
		{
			locked_ = false;
		}

		/// @brief lock, check and see if ready for usage
		inline
		void lock()
		{
			self_modify_if_moving_all_epitope_residues();

			bool status_ok = check_transfer_consistency();
			status_ok = status_ok && check_flanking_consistency();
			status_ok = status_ok && check_scaffold_gaps_vs_flanking_residues();
			status_ok = status_ok && check_epitope_loop_lengths_vs_flanking_residues();
			status_ok = status_ok && check_misc_consistency();

			if (status_ok ) {
				locked_ = true;
			} else {
				utility::exit(__FILE__, __LINE__,"GraftInfo: checks failed during locking" );
			}
		}


	public: // presets

		/// @brief switch to superposition protocol defaults
		/// @details switches do_graft_bb to 'false' and do_graft_sc to 'true'
		/// @details for all graft ranges
		/// @note uses 3,0,3 and 0,0,0 on scaffold, and epitope sides, respectively
		inline
		void
		set_superposition_defaults()
		{
			for ( std::map< ResidueRange, GraftTrack >::iterator i = graft_tracking_.begin(), ie = graft_tracking_.end(); i != ie; ++i ) {
				i->second.do_graft_bb = false;
				i->second.do_graft_sc = true;
			}
		}


	public: // lookup

		/// @brief do bb graft of a particular component?
		inline
		bool const &
		do_graft_bb(
			MatchComponent const & component
		) const
		{
			return do_graft_bb( component.scaffold_gap_range );
		}

		/// @brief do sc graft of a particular component?
		inline
		bool const &
		do_graft_sc(
			MatchComponent const & component
		) const
		{
			return do_graft_sc( component.scaffold_gap_range );
		}

		/// @brief do bb graft of a component corresponding to give scaffold gap range?
		inline
		bool const &
		do_graft_bb(
			ResidueRange const & scaffold_gap_range
		) const
		{
			return graft_tracking_.find( scaffold_gap_range )->second.do_graft_bb;
		}

		/// @brief do sc graft of a component corresponding to give scaffold gap range?
		inline
		bool const &
		do_graft_sc(
			ResidueRange const & scaffold_gap_range
		) const
		{
			return graft_tracking_.find( scaffold_gap_range )->second.do_graft_sc;
		}

		/// @brief return number of scaffold tether n-terminal closure residues
		inline
		Integer const &
		scaffold_tether_residues_nterm(
			MatchComponent const & component
		) const
		{
			return scaffold_tether_residues_nterm( component.scaffold_gap_range );
		}

		/// @brief return number of scaffold tether n-terminal closure residues
		inline
		Integer const &
		scaffold_tether_residues_nterm(
			ResidueRange const & scaffold_gap_range
		) const
		{
			return graft_tracking_.find( scaffold_gap_range )->second.scaffold_tether_residues.first;
		}

		/// @brief return number of scaffold tether c-terminal closure residues
		inline
		Integer const &
		scaffold_tether_residues_cterm(
			MatchComponent const & component
		) const
		{
			return scaffold_tether_residues_cterm( component.scaffold_gap_range );
		}

		/// @brief return number of scaffold tether c-terminal closure residues
		inline
		Integer const &
		scaffold_tether_residues_cterm(
			ResidueRange const & scaffold_gap_range
		) const
		{
			return graft_tracking_.find( scaffold_gap_range )->second.scaffold_tether_residues.second;
		}

		/// @brief return number of scaffold linker n-terminal closure residues
		inline
		Integer const &
		scaffold_linker_residues_nterm(
			MatchComponent const & component
		) const
		{
			return scaffold_linker_residues_nterm( component.scaffold_gap_range );
		}

		/// @brief return number of scaffold linker n-terminal closure residues
		inline
		Integer const &
		scaffold_linker_residues_nterm(
			ResidueRange const & scaffold_gap_range
		) const
		{
			return graft_tracking_.find( scaffold_gap_range )->second.scaffold_linker_residues.first;
		}

		/// @brief return number of scaffold linker c-terminal closure residues
		inline
		Integer const &
		scaffold_linker_residues_cterm(
			MatchComponent const & component
		) const
		{
			return scaffold_linker_residues_cterm( component.scaffold_gap_range );
		}

		/// @brief return number of scaffold linker c-terminal closure residues
		inline
		Integer const &
		scaffold_linker_residues_cterm(
			ResidueRange const & scaffold_gap_range
		) const
		{
			return graft_tracking_.find( scaffold_gap_range )->second.scaffold_linker_residues.second;
		}

		/// @brief return number of scaffold moveable n-terminal closure residues
		inline
		Integer const &
		scaffold_moveable_residues_nterm(
			MatchComponent const & component
		) const
		{
			return scaffold_moveable_residues_nterm( component.scaffold_gap_range );
		}

		/// @brief return number of scaffold moveable n-terminal closure residues
		inline
		Integer const &
		scaffold_moveable_residues_nterm(
			ResidueRange const & scaffold_gap_range
		) const
		{
			return graft_tracking_.find( scaffold_gap_range )->second.scaffold_moveable_residues.first;
		}

		/// @brief return number of scaffold moveable c-terminal closure residues
		inline
		Integer const &
		scaffold_moveable_residues_cterm(
			MatchComponent const & component
		) const
		{
			return scaffold_moveable_residues_cterm( component.scaffold_gap_range );
		}

		/// @brief return number of scaffold moveable c-terminal closure residues
		inline
		Integer const &
		scaffold_moveable_residues_cterm(
			ResidueRange const & scaffold_gap_range
		) const
		{
			return graft_tracking_.find( scaffold_gap_range )->second.scaffold_moveable_residues.second;
		}

		/// @brief return number of epitope tether n-terminal closure residues
		inline
		Integer const &
		epitope_tether_residues_nterm(
			MatchComponent const & component
		) const
		{
			return epitope_tether_residues_nterm( component.scaffold_gap_range );
		}

		/// @brief return number of epitope tether n-terminal closure residues
		inline
		Integer const &
		epitope_tether_residues_nterm(
			ResidueRange const & scaffold_gap_range
		) const
		{
			return graft_tracking_.find( scaffold_gap_range )->second.epitope_tether_residues.first;
		}

		/// @brief return number of epitope tether c-terminal closure residues
		inline
		Integer const &
		epitope_tether_residues_cterm(
			MatchComponent const & component
		) const
		{
			return epitope_tether_residues_cterm( component.scaffold_gap_range );
		}

		/// @brief return number of epitope tether c-terminal closure residues
		inline
		Integer const &
		epitope_tether_residues_cterm(
			ResidueRange const & scaffold_gap_range
		) const
		{
			return graft_tracking_.find( scaffold_gap_range )->second.epitope_tether_residues.second;
		}

		/// @brief return number of epitope linker n-terminal closure residues
		inline
		Integer const &
		epitope_linker_residues_nterm(
			MatchComponent const & component
		) const
		{
			return epitope_linker_residues_nterm( component.scaffold_gap_range );
		}

		/// @brief return number of epitope linker n-terminal closure residues
		inline
		Integer const &
		epitope_linker_residues_nterm(
			ResidueRange const & scaffold_gap_range
		) const
		{
			return graft_tracking_.find( scaffold_gap_range )->second.epitope_linker_residues.first;
		}

		/// @brief return number of epitope linker c-terminal closure residues
		inline
		Integer const &
		epitope_linker_residues_cterm(
			MatchComponent const & component
		) const
		{
			return epitope_linker_residues_cterm( component.scaffold_gap_range );
		}

		/// @brief return number of epitope linker c-terminal closure residues
		inline
		Integer const &
		epitope_linker_residues_cterm(
			ResidueRange const & scaffold_gap_range
		) const
		{
			return graft_tracking_.find( scaffold_gap_range )->second.epitope_linker_residues.second;
		}

		/// @brief return number of epitope moveable n-terminal closure residues
		inline
		Integer const &
		epitope_moveable_residues_nterm(
			MatchComponent const & component
		) const
		{
			return epitope_moveable_residues_nterm( component.scaffold_gap_range );
		}

		/// @brief return number of epitope moveable n-terminal closure residues
		inline
		Integer const &
		epitope_moveable_residues_nterm(
			ResidueRange const & scaffold_gap_range
		) const
		{
			return graft_tracking_.find( scaffold_gap_range )->second.epitope_moveable_residues.first;
		}

		/// @brief return number of epitope moveable c-terminal closure residues
		inline
		Integer const &
		epitope_moveable_residues_cterm(
			MatchComponent const & component
		) const
		{
			return epitope_moveable_residues_cterm( component.scaffold_gap_range );
		}

		/// @brief return number of epitope moveable c-terminal closure residues
		inline
		Integer const &
		epitope_moveable_residues_cterm(
			ResidueRange const & scaffold_gap_range
		) const
		{
			return graft_tracking_.find( scaffold_gap_range )->second.epitope_moveable_residues.second;
		}

		/// @brief rb perturb this epitope component?
		inline
		bool const &
		perturbing_rb(
			MatchComponent const & component
		) const
		{
			return perturbing_rb( component.scaffold_gap_range );
		}

		/// @brief rb perturb this epitope component?
		inline
		bool const &
		perturbing_rb(
			ResidueRange const & scaffold_gap_range
		) const
		{
			return graft_tracking_.find( scaffold_gap_range )->second.perturb_rb;
		}

		/// @brief rb perturbing any secondary component?
		bool
		perturbing_any_secondary_component_rb() const;

		/// @brief move all residues of this epitope component?
		inline
		bool const &
		moving_all_epitope_residues(
				MatchComponent const & component
		) const
		{
			return moving_all_epitope_residues( component.scaffold_gap_range );
		}

		/// @brief move all residues of this epitope component?
		inline
		bool const &
		moving_all_epitope_residues(
			ResidueRange const & scaffold_gap_range
		) const
		{
			return graft_tracking_.find( scaffold_gap_range )->second.move_all_epitope_residues;
		}

		/// @brief close this part of the graft as single break instead of double?
		/// @details only valid for secondary components
		inline
		bool const &
		closing_as_single_break(
			MatchComponent const & component
		) const
		{
			return closing_as_single_break( component.scaffold_gap_range );
		}

		/// @brief close this part of the graft as single break instead of double?
		/// @details only valid for secondary components
		inline
		bool const &
		closing_as_single_break(
			ResidueRange const & scaffold_gap_range
		) const
		{
			return graft_tracking_.find( scaffold_gap_range )->second.close_as_single_break;
		}

		/// @brief closing any secondaries as single break?
		bool
		closing_any_secondary_as_single_break();

		/// @brief repacking epitope residues for this component?
		/// @details valid for ALL components
		inline
		bool const &
		repacking_epitope_residues(
			MatchComponent const & component
		) const
		{
			return repacking_epitope_residues( component.scaffold_gap_range );
		}

		/// @brief repacking epitope residues for this component?
		/// @details valid for ALL components
		inline
		bool const &
		repacking_epitope_residues(
			ResidueRange const & scaffold_gap_range
		) const
		{
			return graft_tracking_.find( scaffold_gap_range )->second.repack_epitope_residues;
		}

		/// @brief is n-side artificially closed due to no moveable residues on either
		/// @brief scaffold or epitope n-side?
		inline
		bool const &
		n_side_artificially_closed(
			MatchComponent const & component
		) const
		{
			return n_side_artificially_closed( component.scaffold_gap_range );
		}

		/// @brief is n-side artificially closed due to no moveable residues on either
		/// @brief scaffold or epitope n-side?
		inline
		bool const &
		n_side_artificially_closed(
			ResidueRange const & scaffold_gap_range
		) const
		{
			return graft_tracking_.find( scaffold_gap_range )->second.left_side_closed;
		}

		/// @brief is c-side artificially closed due to no moveable residues on either
		/// @brief scaffold or epitope c-side?
		inline
		bool
		c_side_artificially_closed(
			MatchComponent const & component
		) const
		{
			return c_side_artificially_closed( component.scaffold_gap_range );
		}

		/// @brief is c-side artificially closed due to no moveable residues on either
		/// @brief scaffold or epitope c-side?
		inline
		bool
		c_side_artificially_closed(
			ResidueRange const & scaffold_gap_range
		) const
		{
			return graft_tracking_.find( scaffold_gap_range )->second.right_side_closed;
		}

		/// @brief return ssbm string
		inline
		String const &
		ssbm_string(
			MatchComponent const & component
		)
		{
			return ssbm_string( component.scaffold_gap_range );
		}

		/// @brief return ssbm string
		inline
		String const &
		ssbm_string(
			ResidueRange const & scaffold_gap_range
		) const
		{
			return graft_tracking_.find( scaffold_gap_range )->second.ssbm_string;
		}

		/// @brief   return secondary structure string
		/// @details returns the secondary structure string constructed from
		///          the ssbm by taking out break and deletion information and
		///          capitalizing all secondary structure
		String
		ss_string(
			ResidueRange const & scaffold_gap_range
		) const;

		/// @brief return aai string
		inline
		String const &
		aai_string(
				MatchComponent const & component
		) const
		{
			return aai_string( component.scaffold_gap_range );
		}

		/// @brief return aai string
		inline
		String const &
		aai_string(
			ResidueRange const & scaffold_gap_range
		) const
		{
			return graft_tracking_.find( scaffold_gap_range )->second.aai_string;
		}

		/// @brief return aa string
		/// @details returns the amino acid string constructed from the aai
		String
		aa_string(
			ResidueRange const & scaffold_gap_range
		) const;


	public: // load info from file


		/// @brief returns help for loading external info from file
		static
		String
		external_graft_info_help();


	public: // load info from other methods

		/// @brief   set graft info instructions from character encoding
		/// @details Line may have one or two "strings".  The first gives
		///          secondary structure and move instructions, while the
		///          second gives amino acid identity instructions.  The
		///          strings may have arbitrary spaces for readability (these
		///          will be removed by the program), but must be separated
		///          by a '&' character, which indicates the start of the
		///          a.a. identity instructions.
		/// @return  whether or not info successfully added from line
		bool
		add_encoded_info_from_line(
			Integer const & scaffold_gap_begin,
			Integer const & scaffold_gap_end,
			String const & line
		);

		/// @brief add additional information about graft from a line (string)
		/// @param[in] start_from  start from column entry (counting starting from 1)
		/// @note  Column entries are delimited by whitespace.
		/// @note  'graft_bb/sc' columns are either 'true' or 'false' and
		/// @note  'graft_bb' will take precedence over 'graft_sc', i.e.
		/// @note  If 'graft_bb' is true then 'graft_sc' will automatically
		/// @note  be set to true.
		/// @note  If 'graft_bb' is false, then no additional columns are necessary.
		/// @note  Restriction: 'tether_*term' >= 0
		/// @note  Restriction: 'tether_*term' + 'linker_*term' >= 0
		/// @note  Restriction: 'tether_*term' + 'linker_*term' >= 'moveable_*term'
		/// @note  Restriction: epitope_tether_nterm < 1 + (length of epitope loop)/2
		/// @note  Restriction: epitope_tether_cterm < (length of epitope loop)/2
		/// @note  Restriction: epitope_linker_*term >= 0
		/// @note  Current file format example is as follows, note that residue
		/// @note  ranges are scaffold gap!  Remember to include the '|' separators.
		/// @note  if 'move_all_epitope_residues' true then 'epitope_tether_*' and 'epitope_moveable_*' is overridden
		/// @note  'perturb_rb' and 'move_all_epitope_residues' for primary component are currently IGNORED
		/// @note  'repack_epitope_residues' is valid for ALL components
		/// @note  ( scaffold_gap ) in example below is only for reference and not part of file format
		/// @note  (scaffold_gap_begin scaffold_gap_end) graft_bb graft_sc scaffold_tether_nterm scaffold_linker_nterm scaffold_moveable_nterm | scaffold_tether_cterm scaffold_linker_cterm scaffold_moveable_cterm | epitope_tether_nterm epitope_linker_nterm epitope_moveable_nterm | epitope_tether_cterm epitope_linker_cterm epitope_moveable_cterm   perturb_rb?   move_all_epitope_residues?  close_as_single_break?  repack_epitope_residues?
		/// @note  (    362   379 )  true     true    3  -1   3  |  4   2   2  |  0   0   0  |  0   0   0
		/// @note  (    478   482 )  true     true    4   0   4  |  4   0   4  |  0   3   3  |  1   0   1
		/// @note  (    201   210 )  false    true
		/// @note  (     97   111 )  false    false
		/// @note  (      7   45  )  true     true    6   3   3  |  6   4   4  |  2   0   2  |  1   1   2    true  false
		/// @note  (     67   93  )  true     true    6   3   9  |  6   4   4  |  0   0   0  |  0   0   0    false  true  true  true
		/// @note  (     127  135 )  true     true    3   0   3  |  6   4   4  |  0   0   0  |  0   0   0    true  true
		/// @return  whether or not info successfully added from line
		bool
		add_info_from_line(
			Integer const & scaffold_gap_begin,
			Integer const & scaffold_gap_end,
			String const & line,
			Size const & start_from = 1
		);


	public: // convenience

		/// @brief return a copy of match components sorted on left endpoint of scaffold gap ranges
		inline
		SortedMatchComponents
		sorted_match_components() const
		{
			// create set of components sorted on left endpoint of scaffold gap ranges
			SortedMatchComponents components;
			components.insert( match_result_.components.begin(), match_result_.components.end() );
			return components;
		}

		/// @brief primary scaffold gap range
		inline
		ResidueRange const &
		primary_scaffold_gap() const
		{
			return match_result_.components[ 1 ].scaffold_gap_range;
		}

		/// @brief secondary scaffold gap ranges
		inline
		std::set< ResidueRange >
		secondary_scaffold_gaps() const
		{
			std::set< ResidueRange > ranges;

			utility::vector1< MatchComponent >::const_iterator c = match_result_.components.begin(), ce = match_result_.components.end();
			++c; // increment past primary component to first secondary component
			for ( ; c != ce; ++c ) {
				ranges.insert( c->scaffold_gap_range );
			}

			return ranges;
		}

		/// @brief secondary scaffold gap ranges
		inline
		std::set< ResidueRange >
		scaffold_gaps() const
		{
			std::set< ResidueRange > ranges;

			for ( utility::vector1< MatchComponent >::const_iterator c = match_result_.components.begin(), ce = match_result_.components.end(); c != ce; ++c ) {
				ranges.insert( c->scaffold_gap_range );
			}

			return ranges;
		}

		/// @brief epitope primary loop native numbering
		inline
		ResidueRange const &
		primary_loop() const
		{
			return match_result_.components[ 1 ].native_loop_subrange;
		}

		/// @brief epitope secondary loop ranges, native numbering
		inline
		std::set< ResidueRange >
		secondary_loops() const
		{
			std::set< ResidueRange > ranges;

			utility::vector1< MatchComponent >::const_iterator c = match_result_.components.begin(), ce = match_result_.components.end();
			++c; // increment past primary component to first secondary component
			for ( ; c != ce; ++c ) {
				ranges.insert( c->native_loop_subrange );
			}

			return ranges;
		}

		/// @brief epitope secondary loop ranges, native numbering
		inline
		std::set< ResidueRange >
		loops() const
		{
			std::set< ResidueRange > ranges;

			for ( utility::vector1< MatchComponent >::const_iterator c = match_result_.components.begin(), ce = match_result_.components.end(); c != ce; ++c ) {
				ranges.insert( c->native_loop_subrange );
			}

			return ranges;
		}

		/// @brief requested move of lever residues?
		/// @return false if no lever residues or not moving lever residues (if N2C/C2N)
		inline
		bool
		moving_lever_residues()
		{
			using epigraft::match::align::AlignmentSystem;

			if ( match_result_.system_type == AlignmentSystem::N2C ) {
				Integer left_moveable = scaffold_moveable_residues_nterm( primary_scaffold_gap() ) +
				                        epitope_moveable_residues_nterm( primary_scaffold_gap() );
				return left_moveable > 0;
			} else if ( match_result_.system_type == AlignmentSystem::C2N ) {
				Integer right_moveable = scaffold_moveable_residues_cterm( primary_scaffold_gap() ) +
				                         epitope_moveable_residues_cterm( primary_scaffold_gap() );
				return right_moveable > 0;
			}

			return false;
		}


	public: // accessors

		/// @brief return internally held MatchResult object
		inline
		MatchResult const &
		match_result() const
		{
			return match_result_;
		}

		/// @brief return internally held MatchResult
		/// @details this is only a stopgap
		/// @warning take care to not modify ranges when using this
		// TODO: properly handle the access/change range issue
		inline
		MatchResult &
		match_result()
		{
			return match_result_;
		}


	public: // status

		/// @brief return status
		String
		to_string(
			String const & scaffold_filename = "",
			String const & line_prefix = "",
			bool const & print_old_format = false
		) const;

		/// @brief return string for a given epitope component specified by scaffold gap range
		String
		to_string(
			ResidueRange const & scaffold_gap_range,
			bool const & print_old_format = false
		) const;

		/// @brief return string for a given epitope component specified by MatchComponent
		String
		to_string(
			MatchComponent const & component,
			bool const & print_old_format = false
		) const;

		/// @brief return status
		String
		to_string_old_format(
			String const & scaffold_filename = "",
			String const & line_prefix = ""
		) const;

		/// @brief return string for a given epitope component specified by scaffold gap range
		String
		to_string_old_format(
			ResidueRange const & scaffold_gap_range
		) const;

		/// @brief return string for a given epitope component specified by MatchComponent
		String
		to_string_old_format(
			MatchComponent const & component
		) const;


	private: // secondary structure/break/moveable character instructions

		/// @brief is a linker instruction?
		bool
		is_linker_instruction(
			char const & i
		) const
		{
			return i == DELETE || i == GROW_HELIX || i == GROW_SHEET || i == GROW_LOOP || i == GROW_ANY || i == GROW_FROZEN;
		}

		/// @brief   count number of needed tether residues in string
		/// @details this count "includes" deleted residues, since it is neccessary
		/// @details to have a tether residue in order for a deletion to occur
		Size
		count_tether(
			String const & s
		) const;

		/// @brief  count number of linker residues in string
		/// @return # of linker residues, can be negative if doing deletions
		Integer
		count_linker(
			String const & s
		) const;

		/// @brief count number of moveable residues in string
		Size
		count_moveable(
			String const & s
		) const;

		/// @brief count number if fixed residues in string
		Size
		count_frozen(
			String const & s
		) const;

		/// @brief find n-term tether/linker portion of epitope string
		String
		left_nt_epitope(
			String const & s,
			bool const & include_tether = true
		) const;

		/// @brief find c-term tether/linker portion of epitope string
		String
		right_ct_epitope(
			String const & s,
			bool const & include_tether = true
		) const;

		/// @brief find middle (core) of epitope sans tether/linker
		String
		middle_core_epitope(
			String const & s,
			bool const & remove_tether = true
		) const;

		/// @brief return all characters used as instructions for breaks
		/// @details this operation is currently expensive, as it concatenates
		///          all the characters anew each time it's called
		inline
		String
		break_characters() const
		{
			std::ostringstream ss;

			ss << OPEN_BREAK << OPEN_BREAK_RB << CLOSED_BREAK;

			return ss.str();
		}

		/// @brief   return all characters used as instructions in ALL tethers
		/// @details The 'FREEZE' character is included in this set.
		///          This operation is currently expensive, as it concatenates
		///          all the characters anew each time it's called.
		inline
		String
		tether_characters() const
		{
			std::ostringstream ss;

			ss << FREEZE << KEEP_AND_MOVE << USE_HELIX << USE_SHEET << USE_LOOP << USE_ANY;

			return ss.str();
		}

		/// @brief   return all characters used as instructions in ONLY epitope tether
		/// @details this operation is currently expensive, as it concatenates
		///          all the characters anew each time it's called
		inline
		String
		epitope_tether_characters() const
		{
			std::ostringstream ss;

			ss << KEEP_AND_MOVE << USE_HELIX << USE_SHEET << USE_LOOP << USE_ANY;

			return ss.str();
		}

		/// @brief   return all characters used as instructions in linker
		/// @details this operation is currently expensive, as it concatenates
		///          all the characters anew each time it's called
		inline
		String
		linker_characters() const
		{
			std::ostringstream ss;

			ss << DELETE << GROW_HELIX << GROW_SHEET << GROW_LOOP << GROW_ANY << GROW_FROZEN;

			return ss.str();
		}

		/// @brief return all invalid secondary structure characters
		inline
		String
		invalid_ss_characters() const
		{
			std::ostringstream ss;

			ss << OPEN_BREAK << OPEN_BREAK_RB << CLOSED_BREAK << DELETE;

			return ss.str();
		}

		/// @brief return all valid secondary structure characters, includes KEEP_AND_MOVE
		inline
		String
		valid_ss_characters() const
		{
			std::ostringstream ss;

			ss << FREEZE << KEEP_AND_MOVE;
			ss << USE_HELIX << USE_SHEET << USE_LOOP << USE_ANY;
			ss << GROW_HELIX << GROW_SHEET << GROW_LOOP << GROW_ANY << GROW_FROZEN;

			return ss.str();
		}

		/// @brief return all valid ssbm characters
		inline
		String
		valid_ssbm_characters() const
		{
			std::ostringstream ss;

			ss << SSBM_START << invalid_ss_characters() << valid_ss_characters();

			return ss.str();
		}

		/// @brief   expand any shortcuts in the core ssbm to full
		/// @warning depends on epitope tether residues being set correctly
		inline
		void
		expand_core_ssbm(
			ResidueRange const & scaffold_gap,
			String & s
		)
		{
			Size const n_star = std::count( s.begin(), s.end(), FREEZE );
			if ( n_star >= 1 ) {
				Size const star_index = s.find( FREEZE );
				String const star_expansion( native_epitope_subrange( scaffold_gap ).length()
				                               - graft_tracking_[ scaffold_gap ].epitope_tether_residues.first
				                               - graft_tracking_[ scaffold_gap ].epitope_tether_residues.second
				                               - n_star + 1,
				                             FREEZE );
				s.replace( star_index, 1, star_expansion );
			}
		}

		/// @brief parse ssbm and store -- used inside add_encoded_info_from_line()
		/// @param[in] line  original line; optional, just for error reporting
		void
		parse_and_store_ssbm(
			ResidueRange const & scaffold_gap,
			String const & ssbm,
			String line = ""
		);


	private: // amino acid identity character instructions

		/// @brief return all valid aa characters
		inline
		String
		valid_aa_characters() const
		{
			std::ostringstream ss;

			// aai characters
			ss << FREEZE << DEFAULT << REPACK;
			ss << ANY_AA << NO_CP;
			ss << NONPOLAR << AROMATIC;
			ss << POLAR << CHARGED << POSITIVE << NEGATIVE;

			// actual amino acid identifiers
			ss << "ACDEFGHIKLMNPQRSTVWY";

			return ss.str();
		}

		/// @brief return all valid aai characters
		inline
		String
		valid_aai_characters() const
		{
			std::ostringstream ss;

			ss << AAI_START << valid_aa_characters();
			ss << invalid_ss_characters(); // these are dummy/ignored in aai

			return ss.str();
		}

		/// @brief parse aai and store -- used inside add_encoded_info_from_line()
		/// @param[in] line  original line; optional, just for error reporting
		/// @warning routine depends on ssbm settings, so ssbm must be correct prior to
		///          calling this
		void
		parse_and_store_aai(
			ResidueRange const & scaffold_gap,
			String const & aai,
			String line = ""
		);


	private: // convenience interface with match result

		/// @brief return the native loop subrange of the scaffold gap
		inline
		ResidueRange
		native_epitope_subrange(
			ResidueRange const & scaffold_gap
		) const
		{
			for ( utility::vector1< MatchComponent >::const_iterator c = match_result_.components.begin(), ce = match_result_.components.end(); c != ce; ++c ) {
				if ( scaffold_gap == c->scaffold_gap_range ) {
					return c->native_loop_subrange;
				}
			}

			// should never be here
			assert( false );
			return ResidueRange( 0, 0);
		}


	private: // automatic modifications (e.g. upon locking)

		/// @brief check to see if any epitope components are marked to move all epitope residues
		/// @brief and change the tether/moveable accordingly
		/// @note N2C/C2N or other primary component single breaks are skipped
		void
		self_modify_if_moving_all_epitope_residues();


	private: // sanity checks

		/// @brief check to see if there is transfer consistency (e.g. ranges are same if not
		/// @brief grafting bb
		bool
		check_transfer_consistency() const;

		/// @brief check to see if tether/linker/moveable are consistent with each other
		/// @details tether >= 0
		/// @details tether + linker >= 0
		/// @details tether + linker >= moveable
		/// @details epitope_tether_nterm < 1 + (length of epitope loop)/2
		/// @details epitope_tether_cterm < (length of epitope loop)/2
		/// @details epitope_linker_*term >= 0
		bool
		check_flanking_consistency() const;

		/// @brief check to see if there's enough room to graft given the
		/// @brief scaffold gap ranges and the number of linker/moveable flanking residues
		bool
		check_scaffold_gaps_vs_flanking_residues() const;

		/// @brief check to see if there's enough room to graft given the
		/// @brief epitope loop lengths and the number of epitope-side tether/linker flanking residues
		/// @details epitope_tether_nterm < sup( (length of epitope loop)/2 )
		/// @details epitope_tether_cterm < sup( (length of epitope loop)/2 )
		bool
		check_epitope_loop_lengths_vs_flanking_residues() const;

		/// @brief additional consistency checks
		bool
		check_misc_consistency() const;

		/// @brief print status message if modification attempted and GraftInfo is locked
		inline
		void
		access_denied()
		const
		{
			std::cerr << "GraftInfo: attempted modification of locked object, please unlock.  No modifications made." << std::endl;
		}


	private: // math

		static
		inline
		Integer
		relative_middle(
			ResidueRange const & range
		)
		{
			return std::ceil( static_cast< Real >( range.length() ) / 2.0 );
		}


	private: // initialize

		inline
		void
		initialize()
		{
			for ( utility::vector1< MatchComponent >::const_iterator c = match_result_.components.begin(), ce = match_result_.components.end(); c != ce; ++c ) {
				ResidueRange const & scaffold_gap_range = c->scaffold_gap_range;
				GraftTrack & track = graft_tracking_[ scaffold_gap_range ];

				track.do_graft_bb = true;
				track.do_graft_sc = true;
				track.scaffold_tether_residues = Flanking( 3, 3 );
				track.scaffold_linker_residues = Flanking( 0, 0 );
				track.scaffold_moveable_residues = Flanking( 0, 0 );
				track.epitope_tether_residues = Flanking( 0, 0 );
				track.epitope_linker_residues = Flanking( 0, 0 );
				track.epitope_moveable_residues = Flanking( 0, 0 );
				track.left_side_closed = false;
				track.right_side_closed = false;
				track.perturb_rb = false;
				track.move_all_epitope_residues = false;
				track.close_as_single_break = false;
				track.repack_epitope_residues = false;
				track.ssbm_string = "";
				track.aai_string = "";
			}
		}


	private: // data

		/// @brief internal copy of original match result
		MatchResult match_result_;

		/// @brief map for accessing information tied to a match component by indexing from scaffold_gap_range
		std::map< ResidueRange, GraftTrack > graft_tracking_;

		/// @brief is object safe for usage? controlled by lock() and unlock()
		bool locked_;

};

} // namespace design
} // namespace epigraft


#endif /*INCLUDED_epigraft_design_GraftInfo_HH_*/
