// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   LoopClosureInfo.hh
/// @brief  Class encapsulating loop closure information.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)


#ifndef INCLUDED_epigraft_design_LoopClosureInfo_HH_
#define INCLUDED_epigraft_design_LoopClosureInfo_HH_

// unit headers
#include <epigraft/design/LoopClosureInfo.fwd.hh>

// package headers
#include <epigraft/design/design_types.hh>
#include <epigraft/design/design_constants.hh>
#include <epigraft/ResidueRange.hh>

// rosetta headers
#include <pose.h>

// C++ headers
#include <algorithm>
#include <set>
#include <sstream>
#include <string>


namespace epigraft {
namespace design {


/// @brief  Class encapsulating loop closure information.
/// @note   Behavior of all methods adding moveable residues is to only add if residue
/// @note   falls within loop range.
class LoopClosureInfo {

	public: // construct/destruct

		/// @brief default constructor
		inline
		LoopClosureInfo()
		: is_closed_( false ),
		  is_artificially_closed_( false )
		{}

		/// @brief regular constructor
		inline
		LoopClosureInfo(
			ResidueRange const & loop_range,
			Integer const & cut
		) : is_closed_( false ),
		    is_artificially_closed_( false ),
		    loop_range_( loop_range ),
		    cut_( cut ),
		    cutpoint_weight_( 1.0 ),
		    chainbreak_score_( DESIGN_INFINITY )
		{}

		/// @brief constructor, can indicate moveable residues to left and right of cut
		inline
		LoopClosureInfo(
			ResidueRange const & loop_range,
			Integer const & cut,
			Integer const & moveable_residues_left_of_cut,
			Integer const & moveable_residues_right_of_cut
		) : is_closed_( false ),
		    is_artificially_closed_( false ),
		    loop_range_( loop_range ),
		    cut_( cut ),
		    cutpoint_weight_( 1.0 ),
		    chainbreak_score_( DESIGN_INFINITY )
		{
			add_moveable_residues_adjacent_to_cut( moveable_residues_left_of_cut, moveable_residues_right_of_cut );
		}

		/// @brief copy constructor
		inline
		LoopClosureInfo(
			LoopClosureInfo const & l
		) : is_closed_( l.is_closed_ ),
		    is_artificially_closed_( l.is_artificially_closed_ ),
		    loop_range_( l.loop_range_ ),
		    cut_( l.cut_ ),
		    moveable_residues_( l.moveable_residues_ ),
		    inactive_moveable_residues_( l.inactive_moveable_residues_ ),
		    always_moveable_( l.always_moveable_ ),
		    cutpoint_weight_( l.cutpoint_weight_ ),
		    chainbreak_score_( l.chainbreak_score_ ),
		    residue_to_rama_( l.residue_to_rama_ )
		{}

		/// @brief default destructor
		inline
		~LoopClosureInfo()
		{}


	public: // assignment

		/// @brief copy assignment
		inline
		LoopClosureInfo &
		operator =( LoopClosureInfo const & l )
		{
			if ( this != &l ) {
				is_closed_ = l.is_closed_;
				is_artificially_closed_ = l.is_artificially_closed_;
				loop_range_ = l.loop_range_;
				cut_ = l.cut_;
				moveable_residues_ = l.moveable_residues_;
				inactive_moveable_residues_ = l.inactive_moveable_residues_;
				always_moveable_ = l.always_moveable_;
				cutpoint_weight_ = l.cutpoint_weight_;
				chainbreak_score_ = l.chainbreak_score_;
				residue_to_rama_ = l.residue_to_rama_;
			}
			return *this;
		}


	public: // member operator

		/// @brief lexicographic order, comparision via loop ranges only
		inline
		bool
		operator <( LoopClosureInfo const & rvalue ) const
		{
			return loop_range_ < rvalue.loop_range_;
		}

		/// @brief '==', comparison via loop ranges and cut
		inline
		bool
		operator ==( LoopClosureInfo const & rvalue ) const
		{
			return loop_range_ == rvalue.loop_range_ && cut_ == rvalue.cut_;
		}


	public: // moveable residues

		/// @brief add moveable residue
		/// @details will only add moveable residue if it is contained within the loop range!
		inline
		void
		add_moveable_residue(
			Integer const & res
		) const
		{
			if ( loop_range_.contains( res ) ) {
				moveable_residues_.insert( res );
				residue_to_rama_[ res ] = DESIGN_INFINITY;
			}
		}

		/// @brief add moveable residues
		/// @details will only add moveable residue if it is contained within the loop range!
		inline
		void
		add_moveable_residues(
			ResidueRange const & range
		) const
		{
			for ( Integer i = range.begin(), ie = range.end(); i <= ie; ++i ) {
				add_moveable_residue( i );
			}
		}

		/// @brief add moveable residues
		/// @details will only add moveable residues if contained within the loop range!
		template< typename ResidueIterator >
		inline
		void
		add_moveable_residues(
			ResidueIterator begin,
			ResidueIterator end
		) const
		{
			for ( ResidueIterator i = begin; i != end; ++i ) {
				add_moveable_residue( *i );
			}
		}

		/// @brief moveable residues to left and right of cut
		/// @details will only add moveable residue if it is contained within the loop range!
		inline
		void
		add_moveable_residues_adjacent_to_cut(
			Integer const & moveable_residues_left_of_cut,
			Integer const & moveable_residues_right_of_cut
		) const
		{
			// set left moveable
			for ( Integer i = 1; i <= moveable_residues_left_of_cut; ++i ) {
				Integer res = cut_ + 1 - i;
				add_moveable_residue( res );
			}

			// set right moveable
			for ( Integer i = 1; i <= moveable_residues_right_of_cut; ++i ) {
				Integer res = cut_ + i;
				add_moveable_residue( res );
			}
		}

		/// @brief set always moveable residues to left and right of cut with respect
		/// @brief to already added moveable residues
		inline
		void
		set_always_moveable_adjacent_to_cut(
			Integer const & always_moveable_left_of_cut,
			Integer const & always_moveable_right_of_cut,
			bool const & flag
		) const
		{
			// set left always moveable
			Integer count = 0;
			std::set< Integer > left_of_cut = complete_moveable_residues_left_of_cut();
			for ( std::set< Integer >::const_reverse_iterator i = left_of_cut.rbegin(), ie = left_of_cut.rend(); i != ie && count < always_moveable_left_of_cut; ++i, ++count ) {
				set_always_moveable( *i, flag );
			}

			// set right always moveable
			count = 0;
			std::set< Integer > right_of_cut = complete_moveable_residues_right_of_cut();
			for ( std::set< Integer >::const_iterator i = right_of_cut.begin(), ie = right_of_cut.end(); i != ie && count < always_moveable_right_of_cut; ++i, ++count ) {
				set_always_moveable( *i, flag );
			}
		}

		/// @brief activate moveable residues to left and right of cut with respect
		/// @brief to already added moveable residues
		inline
		void
		activate_moveable_residues_adjacent_to_cut(
			Integer const & n_left_of_cut,
			Integer const & n_right_of_cut
		) const
		{
			// activate left
			Integer count = 0;
			std::set< Integer > left_of_cut = complete_moveable_residues_left_of_cut();
			for ( std::set< Integer >::const_reverse_iterator i = left_of_cut.rbegin(), ie = left_of_cut.rend(); i != ie && count < n_left_of_cut; ++i, ++count ) {
				activate_moveable_residue( *i );
			}

			// activate right
			count = 0;
			std::set< Integer > right_of_cut = complete_moveable_residues_right_of_cut();
			for ( std::set< Integer >::const_iterator i = right_of_cut.begin(), ie = right_of_cut.end(); i != ie && count < n_right_of_cut; ++i, ++count ) {
				activate_moveable_residue( *i );
			}
		}

		/// @brief remove moveable residue
		inline
		void
		erase_moveable_residue(
			Integer const & res
		) const
		{
			moveable_residues_.erase( res );
			inactive_moveable_residues_.erase( res );
			always_moveable_.erase( res );
			residue_to_rama_.erase( res );
		}

		/// @brief clear all moveable residue data
		inline
		void
		clear_moveable_residues() const
		{
			moveable_residues_.clear();
			inactive_moveable_residues_.clear();
			always_moveable_.clear();
			residue_to_rama_.clear();
		}

		/// @brief indicate whether or not residue is always moveable
		/// @details Residue must have already been added as moveable!
		/// @details If setting residue to true, will also automatically activate
		/// @details that residue.
		inline
		bool
		set_always_moveable(
			Integer const & res,
			bool const & flag
		) const
		{
			if ( is_moveable( res ) || is_inactive( res ) ) {

				if ( flag ) {
					always_moveable_.insert( res );
					activate_moveable_residue( res );
				} else {
					always_moveable_.erase( res );
				}
				return true;

			} else {
				return false;
			}
		}

		/// @brief indicate whether or not a range of residues is always moveable
		/// @details Residue must have already been added as moveable!
		/// @details If setting residue to true, will also automatically activate
		/// @details that residue.
		inline
		void
		set_always_moveable(
			ResidueRange const & range,
			bool const & flag
		) const
		{
			for ( Integer i = range.begin(), ie = range.end(); i <= ie; ++i ) {
				set_always_moveable( i, flag );
			}
		}

		/// @brief ask if a residue is moveable
		inline
		bool
		is_moveable(
			Integer const & res
		) const
		{
			return moveable_residues_.find( res ) != moveable_residues_.end();
		}

		/// @brief ask if residue is always moveable
		inline
		bool
		is_always_moveable(
			Integer const & res
		) const
		{
			return always_moveable_.find( res ) != always_moveable_.end();
		}

		/// @brief ask if a residue is inactive
		inline
		bool
		is_inactive(
			Integer const & res
		) const
		{
			return inactive_moveable_residues_.find( res ) != inactive_moveable_residues_.end();
		}

		/// @brief mark moveable residue as inactive
		bool
		inactivate_moveable_residue(
			Integer const & res
		) const
		{
			if ( is_always_moveable( res ) ) {
				return false;
			}

			std::set< Integer >::const_iterator i = moveable_residues_.find( res );
			if ( i == moveable_residues_.end() ) {
				return false;
			}

			moveable_residues_.erase( i );
			inactive_moveable_residues_.insert( res );

			return true;
		}

		/// @brief mark moveable residue as active
		bool
		activate_moveable_residue(
			Integer const & res
		) const
		{
			std::set< Integer >::const_iterator i = inactive_moveable_residues_.find( res );
			if ( i == inactive_moveable_residues_.end() ) {
				return false;
			}

			inactive_moveable_residues_.erase( i );
			moveable_residues_.insert( res );

			return true;
		}

		/// @brief re-activate any inactive moveable residues
		void
		activate_all_moveable_residues() const
		{
			std::set< Integer > mr = complete_moveable_residues();
			moveable_residues_.insert( mr.begin(), mr.end() );
			inactive_moveable_residues_.clear();
		}

		/// @brief inactivate all moveable residues that are not set as always moveable
		void
		inactivate_all_moveable_residues() const
		{
			std::set< Integer > mr = complete_moveable_residues(); // need a copy to work from
			for ( std::set< Integer >::const_iterator i = mr.begin(), ie = mr.end(); i != ie; ++i ) {
				inactivate_moveable_residue( *i );
			}
		}

		/// @brief set moveable bb residues within loop, does not touch state of other residues
		inline
		void
		setup_allow_bb_move(
			Pose & pose
		) const
		{
			for ( std::set< Integer >::const_iterator i = moveable_residues_.begin(), ie = moveable_residues_.end(); i != ie; ++i ) {
				pose.set_allow_bb_move( *i, true );
			}
		}

		/// @brief set moveable chi residues within loop, does not touch state of other residues
		inline
		void
		setup_allow_chi_move(
			Pose & pose
		) const
		{
			for ( std::set< Integer >::const_iterator i = moveable_residues_.begin(), ie = moveable_residues_.end(); i != ie; ++i ) {
				pose.set_allow_chi_move( *i, true );
			}
		}


	public: // for fragment insertion

		/// @brief generate secondary structure string for Vall fragment picking
		/// TODO: not in use, think about how to go about this...
		inline
		std::string
		secondary_structure_str() const
		{
			std::ostringstream oss;

			for ( Integer i = loop_range_.begin(), ie = loop_range_.end(); i <= ie; ++i ) {
				if ( moveable_residues_.find( i ) != moveable_residues_.end() ) {
					oss << 'D';
				} else {
					oss << '.';
				}
			}

			return oss.str();
		}


	public: // convenience

		/// @brief shift residue numbers left/right due to insertion/deletion left of cut
		/// @return new LoopClosureInfo with shifted internal residues
		LoopClosureInfo
		shifted_left_of_cut(
			Integer const & shift_length
		) const
		{
			LoopClosureInfo new_info( *this );

			// handle loop range
			new_info.loop_range_ = new_info.loop_range_ + shift_length;

			// handle cut
			new_info.cut_ = new_info.cut_ + shift_length;

			// handle moveable residues
			new_info.clear_moveable_residues();
			for ( std::set< Integer >::const_iterator i = moveable_residues_.begin(), ie = moveable_residues_.end(); i != ie; ++i ) {
				new_info.add_moveable_residue( *i + shift_length );
			}

			return new_info;
		}


	public: // accessors

		/// @brief return if loop is closed
		/// @details artificially closed implies closed
		inline
		bool
		is_closed() const
		{
			return is_closed_ || is_artificially_closed_;
		}

		/// @brief return if loop is artificially closed
		inline
		bool const &
		is_artificially_closed() const
		{
			return is_artificially_closed_;
		}

		/// @brief return loop range
		inline
		ResidueRange const &
		loop_range() const
		{
			return loop_range_;
		}

		/// @brief return loop cutpoint
		inline
		Integer const &
		cut() const
		{
			return cut_;
		}

		/// @brief return width of moveable residues (right - left + 1)
		inline
		Integer
		moveable_residues_width() const
		{
			std::set< Integer >::const_iterator begin = moveable_residues().begin();
			std::set< Integer >::const_iterator end = moveable_residues().end();

			return ( *std::max_element( begin, end ) ) - ( *std::min_element( begin, end ) ) + 1;
		}

		/// @brief return moveable residues
		inline
		std::set< Integer > const &
		moveable_residues() const
		{
			return moveable_residues_;
		}

		/// @brief return copy of both active and inactive moveable residues
		inline
		std::set< Integer >
		complete_moveable_residues() const
		{
			std::set< Integer > mr;
			mr.insert( moveable_residues_.begin(), moveable_residues_.end() );
			mr.insert( always_moveable_.begin(), always_moveable_.end() ); // safety, not necessary if everything is working right
			mr.insert( inactive_moveable_residues_.begin(), inactive_moveable_residues_.end() );
			return mr;
		}

		/// @brief return always moveable residues
		inline
		std::set< Integer > const &
		always_moveable_residues() const
		{
			return always_moveable_;
		}

		/// @brief return currently inactive moveable residues
		inline
		std::set< Integer > const &
		inactive_moveable_residues() const
		{
			return inactive_moveable_residues_;
		}

		/// @brief return number of moveable residues left of cut (including 'cut' residue)
		inline
		Integer
		count_moveable_residues_left_of_cut() const
		{
			Integer count = 0;
			for ( std::set< Integer >::const_iterator i = moveable_residues_.begin(), ie = moveable_residues_.end(); i != ie; ++i ) {
				if ( (*i) <= cut_ ) {
					++count;
				}
			}
			return count;
		}

		/// @brief return number of moveable residues right of cut (not including 'cut' residue)
		inline
		Integer
		count_moveable_residues_right_of_cut() const
		{
			Integer count = 0;
			for ( std::set< Integer >::const_iterator i = moveable_residues_.begin(), ie = moveable_residues_.end(); i != ie; ++i ) {
				if ( (*i) > cut_ ) {
					++count;
				}
			}
			return count;
		}

		/// @brief return copy of the moveable residues that are left of the cut (including 'cut' residue)
		inline
		std::set< Integer >
		moveable_residues_left_of_cut() const
		{
			std::set< Integer > left_of_cut;

			for ( std::set< Integer >::const_iterator i = moveable_residues_.begin(), ie = moveable_residues_.end(); i != ie; ++i ) {
				if ( (*i) <= cut_ ) {
					left_of_cut.insert( *i );
				}
			}

			return left_of_cut;
		}

		/// @brief return copy of the moveable residues that are right of the cut (not including 'cut' residue)
		inline
		std::set< Integer >
		moveable_residues_right_of_cut() const
		{
			std::set< Integer > right_of_cut;

			for ( std::set< Integer >::const_iterator i = moveable_residues_.begin(), ie = moveable_residues_.end(); i != ie; ++i ) {
				if ( (*i) > cut_ ) {
					right_of_cut.insert( *i );
				}
			}

			return right_of_cut;
		}

		/// @brief return copy of both active & inactive moveable residues that are left of the cut (including 'cut' residue)
		inline
		std::set< Integer >
		complete_moveable_residues_left_of_cut() const
		{
			std::set< Integer > left_of_cut;
			std::set< Integer > mr = complete_moveable_residues();

			for ( std::set< Integer >::const_iterator i = mr.begin(), ie = mr.end(); i != ie; ++i ) {
				if ( (*i) <= cut_ ) {
					left_of_cut.insert( *i );
				}
			}

			return left_of_cut;
		}

		/// @brief return copy of both active & inactive moveable residues that are right of the cut (not including 'cut' residue)
		inline
		std::set< Integer >
		complete_moveable_residues_right_of_cut() const
		{
			std::set< Integer > right_of_cut;
			std::set< Integer > mr = complete_moveable_residues();

			for ( std::set< Integer >::const_iterator i = mr.begin(), ie = mr.end(); i != ie; ++i ) {
				if ( (*i) > cut_ ) {
					right_of_cut.insert( *i );
				}
			}

			return right_of_cut;
		}

		/// @brief return cutpoint weight, useful in simultaneous loop build/refine type settings
		/// @note  This is not the chainbreak and chainbreak_overlap weight, both of which are (single) global weights
		/// @note  applicable to all loops.
		inline
		Real const &
		cutpoint_weight() const
		{
			return cutpoint_weight_;
		}

		/// @brief get chainbreak score
		inline
		Real const &
		chainbreak_score() const
		{
			return chainbreak_score_;
		}

		/// @brief return rama of residue
		/// @return  returns improbable value of 32768.0 if rama is not found
		inline
		Real
		rama_of_residue(
			Integer const & res
		) const
		{
			std::map< Integer, Real >::const_iterator i = residue_to_rama_.find( res );
			if ( i == residue_to_rama_.end() ) {
				return 32768.0;
			}
			return i->second;
		}

		/// @brief return residue -> local rama map
		inline
		std::map< Integer, Real > const &
		residue_to_rama() const
		{
			return residue_to_rama_;
		}

		/// @brief get total rama score for moveable residues
		inline
		Real
		total_rama_moveable() const
		{
			Real total_rama_moveable = 0;
			for ( std::map< Integer, Real >::const_iterator i = residue_to_rama_.begin(), ie = residue_to_rama_.end(); i != ie; ++i ) {
				total_rama_moveable += i->second;
			}
			return total_rama_moveable;
		}

		/// @brief returns whether or not ramas are above a particular threshold
		inline
		bool
		ramas_are_not_above(
			Real const & rama_limit
		) const
		{
			bool flag = true;
			for ( std::map< Integer, Real >::const_iterator i = residue_to_rama_.begin(), ie = residue_to_rama_.end(); i != ie; ++i ) {
				if ( i->second > rama_limit ) {
					flag = false;
					break;
				}
			}

			return flag;
		}


	public: // set

		/// @brief set loop range
		inline
		void
		set_loop_range(
			ResidueRange const & range
		)
		{
			loop_range_ = range;
		}

		/// @brief set loop cut point
		inline
		void
		set_cut(
			Integer const & res
		)
		{
			cut_ = res;
		}

		/// @brief indicate if loop is closed
		/// @note  call-able even when object is const, since it doesn't change the comparison sort order
		inline
		void
		set_is_closed(
			bool const & flag
		) const
		{
			is_closed_ = flag;
		}

		/// @brief indicate if loop is artificially closed
		inline
		void
		set_is_artificially_closed(
			bool const & flag
		)
		{
			is_artificially_closed_ = flag;
		}

		/// @brief set cutpoint weight, useful in simultaneous loop build/refine type settings
		/// @note  Call-able even when object is const, since it doesn't change the comparison sort order.
		/// @note  This is not the chainbreak and chainbreak_overlap weight, both of which are (single) global weights
		/// @note  applicable to all loops.
		inline
		void
		set_cutpoint_weight(
			Real const & weight
		) const
		{
			cutpoint_weight_ = weight;
		}

		/// @brief set chainbreak score
		/// @note  call-able even when object is const, since it doesn't change the comparison sort order
		inline
		void
		set_chainbreak_score(
			Real const & score
		) const
		{
			chainbreak_score_ = score;
		}

		/// @brief set rama score for moveable residues
		/// @note  call-able even when object is const, since it doesn't change the comparison sort order
		inline
		void
		set_rama_moveable(
			Integer const & residue,
			Real const & rama
		) const
		{
			residue_to_rama_[ residue ] = rama;
		}


	public: // report status

		/// @brief return status
		inline
		std::string
		to_string() const
		{
			std::ostringstream ss;
			ss << is_closed_ << " range = " << loop_range_.to_string() << "   cut = " << cut_;
			ss << "   moveable = [";
			for ( std::set< Integer >::const_iterator i = moveable_residues_.begin(), ie = moveable_residues_.end(); i != ie; ++i ) {
				ss << " " << *i;
			}
			ss << " ]";
			ss << "   cutpoint_weight = " << cutpoint_weight_;
			ss << "   chainbreak_score = " << chainbreak_score_;
			ss << "   total_rama_moveable = " << total_rama_moveable();
			ss << "   inactive_moveable = [";
			for ( std::set< Integer >::const_iterator i = inactive_moveable_residues_.begin(), ie = inactive_moveable_residues_.end(); i != ie; ++i ) {
				ss << " " << *i;
			}
			ss << " ]";
			return ss.str();
		}

		/// @brief return rama per residue (one entry per line)
		inline
		std::string
		to_string_rama_per_residue(
			std::string const & prefix
		) const
		{
			std::ostringstream ss;
			ss << prefix << "Rama per residue:\n";
			ss << prefix << "   res   rama\n";
			for ( std::map< Integer, Real >::const_iterator i = residue_to_rama_.begin(), ie = residue_to_rama_.end(); i != ie; ++i ) {
				ss << prefix << "   " << i->first << "   " << i->second << "\n";
			}
			return ss.str();
		}


	private: // data

		// loop info
		mutable bool is_closed_;
		bool is_artificially_closed_;
		ResidueRange loop_range_;
		Integer cut_;
		mutable std::set< Integer > moveable_residues_;

		// for switching on/off moveable residues
		mutable std::set< Integer > inactive_moveable_residues_;
		mutable std::set< Integer > always_moveable_;

		// score weights
		/// @brief cutpoint weight for this loop, useful in simultaneous loop build/refine type settings
		/// @note  by default this is 1.0
		/// @note  This is not the chainbreak and chainbreak_overlap weight, both of which are (single) global weights
		/// @note  applicable to all loops.
		mutable Real cutpoint_weight_;

		// resultant data
		mutable Real chainbreak_score_;
		mutable std::map< Integer, Real > residue_to_rama_; // we don't store this in moveable_residues_ because we want that set to be non-mutable

};

} // namespace design
} // namespace epigraft


#endif /*INCLUDED_epigraft_design_LoopClosureInfo_HH_*/
