// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   MultiGraftStats.cc
/// @brief  Tracks the operations and results of a multigraft run.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

// unit headers
#include <epigraft/design/MultiGraftStats.hh>


namespace epigraft {
namespace design {


/// @brief record closures in an EpitopeScaffold with a descriptor tag indicating where it happened in the
/// @brief protocol
void
MultiGraftStats::record_closure_attempt(
	std::string const & descriptor,
	EpitopeScaffold const & es
)
{
	// make sure descriptor exists
	if ( closure_categories_.find( descriptor ) == closure_categories_.end() ) {
		closure_categories_[ descriptor ] = std::set< ClosureStatistics >();
	}

	// grab correct closure statistics for category
	std::set< ClosureStatistics > & category_statistics = closure_categories_.find( descriptor )->second;

	// copy of loop closure info attempts
	std::set< LoopClosureInfo > closure_attempts = es.closures_to_attempt();

	// run through closure attempts, enforce existence, and update
	for ( std::set< LoopClosureInfo >::const_iterator ci = closure_attempts.begin(), ci_e = closure_attempts.end(); ci != ci_e; ++ci ) {
		LoopClosureInfo const & closure_attempt = *ci;
		ClosureStatistics lookup_cs( closure_attempt );

		// make if nonexistent
		if ( category_statistics.find( lookup_cs ) == category_statistics.end() ) {
			category_statistics.insert( lookup_cs );
		}

		// update
		category_statistics.find( lookup_cs )->update( closure_attempt, es );
	}

	// now update grouped statistics
	if ( grouped_.find( descriptor ) == grouped_.end() ) {
		grouped_[ descriptor ] = GroupedClosureStatistics();
	}
	grouped_.find( descriptor )->second.update( es );
}


/// @brief return status string
std::string
MultiGraftStats::to_string() const
{
	std::ostringstream ss;

	// grouped
	ss << "# Grouped Statistics:" << std::endl;
	for ( std::map< std::string, GroupedClosureStatistics >::const_iterator e = grouped_.begin(), ee = grouped_.end(); e != ee; ++e ) {
		std::string const & category = e->first;
		ss << "#   " << category << " -- " << std::endl;
		ss << "#      " << e->second.to_string() << std::endl;
	}

	// spacer
	ss << "#" << std::endl;

	// category
	ss << "# Individual Statistics:" << std::endl;
	for ( std::map< std::string, std::set< ClosureStatistics > >::const_iterator e = closure_categories_.begin(), ee = closure_categories_.end(); e != ee; ++e ) {
		std::string const & category = e->first;
		std::set< ClosureStatistics > const & category_statistics = e->second;

		ss << "#   " << category << " -- " << std::endl;
		for ( std::set< ClosureStatistics >::const_iterator s = category_statistics.begin(), se = category_statistics.end(); s != se; ++s ) {
			ss << "#      " << s->to_string() << std::endl;
		}
	}

	return ss.str();
}


} // namespace design
} // namespace epigraft
