// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   MultiGraftStats.hh
/// @brief  Tracks the operations and results of a multigraft run.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)


#ifndef INCLUDED_epigraft_design_MultiGraftStats_HH_
#define INCLUDED_epigraft_design_MultiGraftStats_HH_


// package headers
#include <epigraft/design/design_types.hh>
#include <epigraft/design/design_constants.hh>
#include <epigraft/design/EpitopeScaffold.hh>
#include <epigraft/design/LoopClosureInfo.hh>

// C++ header
#include <map>
#include <set>
#include <sstream>
#include <string>


namespace epigraft {
namespace design {


/// @brief Tracks the operations and results of a multigraft run.
class MultiGraftStats {

	private: // internal structs

		struct GroupedClosureStatistics {
			// default destructor
			inline
			GroupedClosureStatistics() :
				attempts( 0 ),
				successes( 0 )
			{}

			// update
			inline
			void
			update(
				EpitopeScaffold const & es
			)
			{
				++attempts;
				if ( es.all_loops_closed() ) {
					++successes;
				}
			}

			// status
			inline
			std::string
			to_string() const
			{
				std::ostringstream ss;
				ss << "successes/attempts = " << successes << "/" << attempts;
				return ss.str();
			}

			// data
			Integer attempts;
			Integer successes;
		};


		struct ClosureStatistics {
			// default constructor
			inline
			ClosureStatistics() :
				loop_range( -1, -1 ),
				attempts( 0 ),
				successes( 0 ),
				min_break( DESIGN_INFINITY ),
				max_break( -DESIGN_INFINITY ),
				total_local_rama_associated_with_min_break( DESIGN_INFINITY ),
			    total_local_rama_associated_with_max_break( DESIGN_INFINITY ),
				total_min_local_rama( DESIGN_INFINITY ),
				total_max_local_rama( -DESIGN_INFINITY )
			{}

			// LoopClosureInfo constructor
			inline
			ClosureStatistics(
				LoopClosureInfo const & closure
			) : loop_range( closure.loop_range() ),
			    cut( closure.cut() ),
			    attempts( 0 ),
			    successes ( 0 ),
			    min_break( DESIGN_INFINITY ),
			    max_break( -DESIGN_INFINITY ),
				total_local_rama_associated_with_min_break( DESIGN_INFINITY ),
			    total_local_rama_associated_with_max_break( DESIGN_INFINITY ),
			    total_min_local_rama( DESIGN_INFINITY ),
			    total_max_local_rama( -DESIGN_INFINITY )
			{}

			// copy constructor
			ClosureStatistics(
				ClosureStatistics const & s
			) : loop_range( s.loop_range ),
			    cut( s.cut ),
			    attempts( s.attempts ),
			    successes( s.successes ),
			    min_break( s.min_break ),
			    max_break( s.max_break ),
				total_local_rama_associated_with_min_break( s.total_local_rama_associated_with_min_break ),
			    total_local_rama_associated_with_max_break( s.total_local_rama_associated_with_max_break ),
			    total_min_local_rama( s.total_min_local_rama ),
			    total_max_local_rama( s.total_max_local_rama )
			{}

			// assignment
			inline
			ClosureStatistics &
			operator =( ClosureStatistics const & s )
			{
				if ( this != &s ) {
					loop_range = s.loop_range;
					cut = s.cut;
					attempts = s.attempts;
					successes = s.successes;
					min_break = s.min_break;
					max_break = s.max_break;
					total_local_rama_associated_with_min_break = s.total_local_rama_associated_with_min_break;
					total_local_rama_associated_with_max_break = s.total_local_rama_associated_with_max_break;
					total_min_local_rama = s.total_min_local_rama;
					total_max_local_rama = s.total_max_local_rama;
				}
				return *this;
			}

			// sort operator
			inline
			bool
			operator <( ClosureStatistics const & rvalue ) const
			{
				return loop_range < rvalue.loop_range;
			}

			// update (function as const ok, data mutable)
			inline
			void
			update(
				LoopClosureInfo const & c,
				EpitopeScaffold const & es
			) const
			{
				++attempts;
				if ( c.is_closed() || es.closed_during_trajectory( c ) ) {
					++successes;
				}

				// chainbreak
				min_break = std::min( min_break, c.chainbreak_score() );
				max_break = std::max( max_break, c.chainbreak_score() );

				// rama associated with min/max chainbreak scores
				if ( min_break == c.chainbreak_score() ) {
					total_local_rama_associated_with_min_break = c.total_rama_moveable();
				}
				if ( max_break == c.chainbreak_score() ) {
					total_local_rama_associated_with_max_break = c.total_rama_moveable();
				}

				// rama
				total_min_local_rama = std::min( total_min_local_rama, c.total_rama_moveable() );
				total_max_local_rama = std::max( total_max_local_rama, c.total_rama_moveable() );
			}

			// status
			inline
			std::string
			to_string() const
			{
				std::ostringstream ss;

				ss.setf( std::ios::fixed, std::ios::floatfield ); // formatting for reals, precision is adjusted on case-by-case basis
				ss.precision( 5 );

				ss << loop_range.to_string() << " cut = " << cut << "   successes/attempts = " << successes << "/" << attempts
				   << "   min/max_break = " << min_break << "/" << max_break
				   << "   break associated total min/max_local_rama = " << total_local_rama_associated_with_min_break << "/" << total_local_rama_associated_with_max_break
				   << "   total min/max_local_rama = " << total_min_local_rama << "/" << total_max_local_rama;

				return ss.str();
			}

			// data
			ResidueRange loop_range;
			Integer cut;

			// rest of data is mutable so can update within container
			mutable Integer attempts;
			mutable Integer successes;
			mutable Real min_break;
			mutable Real max_break;
			mutable Real total_local_rama_associated_with_min_break;
			mutable Real total_local_rama_associated_with_max_break;
			mutable Real total_min_local_rama;
			mutable Real total_max_local_rama;
		};


	public: // construct/destruct

		/// @brief default constructor
		inline
		MultiGraftStats()
		{}

		/// @brief default destructor
		inline
		~MultiGraftStats()
		{}


	public: // accessors

		/// @brief return number of closure attempts for a chainbreak
		Integer const &
		n_closure_attempts(
				std::string const & descriptor,
				LoopClosureInfo const & closure_info
		) const
		{
			ClosureStatistics lookup_cs( closure_info );
			return closure_categories_.find( descriptor )->second.find( lookup_cs )->attempts;
		}

		/// @brief return number of closure successes for a chainbreak
		Integer const &
		n_closure_successes(
			std::string const & descriptor,
			LoopClosureInfo const & closure_info
		) const
		{
			ClosureStatistics lookup_cs( closure_info );
			return closure_categories_.find( descriptor )->second.find( lookup_cs )->successes;
		}


	public: // update statistics

		/// @brief record closures in an EpitopeScaffold with a descriptor tag indicating where it happened in the
		/// @brief protocol
		void
		record_closure_attempt(
			std::string const & descriptor,
			EpitopeScaffold const & es
		);


	public: // status

		/// @brief return status string
		std::string
		to_string() const;

		/// @brief return grouped statistics string
		inline
		std::string
		grouped_status(
			std::string const & descriptor
		) const
		{
			std::ostringstream ss;

			std::map< std::string, GroupedClosureStatistics >::const_iterator i;
			if ( ( i = grouped_.find( descriptor ) ) != grouped_.end() ) {
				ss << i->second.to_string();
			}

			return ss.str();
		}

	private: // data

		// inividual statistics
		std::map< std::string, std::set< ClosureStatistics > > closure_categories_;

		// grouped statistics
		std::map< std::string, GroupedClosureStatistics > grouped_;
};


} // namespace design
} // namespace epigraft


#endif /*INCLUDED_epigraft_design_MultiGraftStats_HH_*/
